/*
 *
 *    Copyright (c) 2021-2022 Project CHIP Authors
 *    All rights reserved.
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

#include <lib/support/CodeUtils.h>
#include <lib/support/Span.h>

namespace chip {
namespace TestCerts {

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF1-8000-0000-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF1_8000_0000_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xe9, 0x30, 0x82, 0x01, 0x8f, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x1b, 0xd6, 0x38, 0x18, 0xf1, 0x44,
    0xa7, 0x97, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x31, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30,
    0x30, 0x30, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f,
    0x39, 0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b,
    0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41,
    0x43, 0x20, 0x30, 0x30, 0x30, 0x30, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02,
    0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x31, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c,
    0x02, 0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x30, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01,
    0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xfc, 0xa1, 0x38, 0xc9, 0x2b, 0x76, 0xe1,
    0xcd, 0x13, 0x87, 0x3c, 0x19, 0xea, 0x19, 0x72, 0x98, 0x48, 0x55, 0x78, 0xc1, 0x5d, 0xb7, 0xd2, 0x98, 0xd1, 0x84, 0x25, 0xfc,
    0x68, 0x43, 0xcd, 0x50, 0x80, 0x28, 0xef, 0x63, 0x74, 0xbb, 0x0b, 0xdd, 0x09, 0x3c, 0x2a, 0x0d, 0x43, 0xc6, 0xd6, 0xd2, 0xc6,
    0x7b, 0x10, 0x2b, 0xcf, 0x20, 0x70, 0x31, 0xda, 0x6c, 0xc0, 0xe3, 0x16, 0x4f, 0xe0, 0x35, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c,
    0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01,
    0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xe5, 0x70, 0x88,
    0xf4, 0x48, 0xe6, 0xff, 0x9f, 0xe2, 0x00, 0x97, 0x44, 0x52, 0x8b, 0xf9, 0xec, 0x08, 0x2f, 0xc1, 0x1f, 0x30, 0x1f, 0x06, 0x03,
    0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xaf, 0x42, 0xb7, 0x09, 0x4d, 0xeb, 0xd5, 0x15, 0xec, 0x6e, 0xcf, 0x33,
    0xb8, 0x11, 0x15, 0x22, 0x5f, 0x32, 0x52, 0x88, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03,
    0x48, 0x00, 0x30, 0x45, 0x02, 0x20, 0x05, 0x6e, 0xe3, 0x30, 0x5c, 0x1b, 0x2e, 0x80, 0xe2, 0x26, 0x0b, 0x84, 0xd4, 0x52, 0x8e,
    0x52, 0xda, 0xef, 0x0b, 0x80, 0x1b, 0xd4, 0x92, 0xe8, 0x3e, 0xf7, 0x86, 0x4f, 0xc5, 0x12, 0x8c, 0x4c, 0x02, 0x21, 0x00, 0xe9,
    0x87, 0xf4, 0x1b, 0xf8, 0xfa, 0xbe, 0x48, 0x10, 0x55, 0xdb, 0x9d, 0xda, 0xf8, 0xa7, 0x75, 0x55, 0x24, 0x2c, 0x38, 0xb5, 0xcd,
    0x8e, 0xbe, 0xd6, 0x73, 0x06, 0x8d, 0x75, 0x52, 0xb2, 0xfd,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0000_Cert = ByteSpan(sTestCert_DAC_FFF1_8000_0000_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF1_8000_0000_SKID_Array[] = {
    0xE5, 0x70, 0x88, 0xF4, 0x48, 0xE6, 0xFF, 0x9F, 0xE2, 0x00, 0x97, 0x44, 0x52, 0x8B, 0xF9, 0xEC, 0x08, 0x2F, 0xC1, 0x1F,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0000_SKID = ByteSpan(sTestCert_DAC_FFF1_8000_0000_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF1-8000-0000-Key.pem

constexpr uint8_t sTestCert_DAC_FFF1_8000_0000_PublicKey_Array[] = {
    0x04, 0xfc, 0xa1, 0x38, 0xc9, 0x2b, 0x76, 0xe1, 0xcd, 0x13, 0x87, 0x3c, 0x19, 0xea, 0x19, 0x72, 0x98,
    0x48, 0x55, 0x78, 0xc1, 0x5d, 0xb7, 0xd2, 0x98, 0xd1, 0x84, 0x25, 0xfc, 0x68, 0x43, 0xcd, 0x50, 0x80,
    0x28, 0xef, 0x63, 0x74, 0xbb, 0x0b, 0xdd, 0x09, 0x3c, 0x2a, 0x0d, 0x43, 0xc6, 0xd6, 0xd2, 0xc6, 0x7b,
    0x10, 0x2b, 0xcf, 0x20, 0x70, 0x31, 0xda, 0x6c, 0xc0, 0xe3, 0x16, 0x4f, 0xe0, 0x35,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0000_PublicKey = ByteSpan(sTestCert_DAC_FFF1_8000_0000_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF1_8000_0000_PrivateKey_Array[] = {
    0x21, 0xf2, 0xe3, 0xe4, 0x20, 0xc0, 0x70, 0x17, 0x34, 0x81, 0x04, 0x69, 0xb6, 0xba, 0xd1, 0x5c,
    0xf3, 0x06, 0x78, 0x22, 0xc9, 0xa4, 0xa5, 0x96, 0xc1, 0x86, 0xfa, 0x9b, 0xef, 0x15, 0x3f, 0xa1,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0000_PrivateKey = ByteSpan(sTestCert_DAC_FFF1_8000_0000_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF1-8000-0001-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF1_8000_0001_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xe9, 0x30, 0x82, 0x01, 0x8f, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x0e, 0x06, 0x3b, 0x74, 0x2b, 0xcf,
    0xbe, 0x5d, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x31, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30,
    0x30, 0x30, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f,
    0x39, 0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b,
    0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41,
    0x43, 0x20, 0x30, 0x30, 0x30, 0x31, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02,
    0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x31, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c,
    0x02, 0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x30, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01,
    0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xc2, 0x25, 0x83, 0x22, 0xc7, 0xdc, 0x72,
    0x73, 0x7c, 0x33, 0xbe, 0xed, 0x70, 0x73, 0x37, 0xaa, 0x24, 0x85, 0xbc, 0x46, 0x79, 0x3e, 0x4d, 0x5a, 0xc9, 0xa7, 0x5a, 0xd7,
    0x43, 0x52, 0x66, 0xc9, 0x0a, 0x02, 0x8e, 0xec, 0xaf, 0x26, 0x50, 0xfe, 0x70, 0x09, 0xef, 0xfc, 0xae, 0xcb, 0xea, 0xd1, 0xf2,
    0xc3, 0xd1, 0x24, 0x35, 0xde, 0xc2, 0xea, 0xd3, 0xd9, 0x92, 0x95, 0xbf, 0xce, 0xd6, 0xc3, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c,
    0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01,
    0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x96, 0xc2, 0xd9,
    0x24, 0x94, 0xea, 0x97, 0x85, 0xc0, 0xd1, 0x67, 0x08, 0xe3, 0x88, 0xf1, 0xc0, 0x91, 0xea, 0x0f, 0xd5, 0x30, 0x1f, 0x06, 0x03,
    0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xaf, 0x42, 0xb7, 0x09, 0x4d, 0xeb, 0xd5, 0x15, 0xec, 0x6e, 0xcf, 0x33,
    0xb8, 0x11, 0x15, 0x22, 0x5f, 0x32, 0x52, 0x88, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03,
    0x48, 0x00, 0x30, 0x45, 0x02, 0x20, 0x5f, 0xcb, 0x29, 0xa4, 0x0d, 0x3c, 0x35, 0xa6, 0xe8, 0xce, 0x60, 0x65, 0xc6, 0xd0, 0x9d,
    0xa6, 0x17, 0x3d, 0xc5, 0xb2, 0x45, 0xec, 0x32, 0x04, 0x91, 0xe3, 0xd3, 0x49, 0x32, 0xb7, 0x3e, 0x17, 0x02, 0x21, 0x00, 0xb4,
    0x56, 0x99, 0x0f, 0x52, 0x05, 0x10, 0x04, 0x5a, 0x38, 0x8f, 0x75, 0x4e, 0x77, 0x15, 0x40, 0xa0, 0x44, 0x97, 0x92, 0x31, 0x96,
    0x45, 0x5e, 0x44, 0x0d, 0x68, 0x25, 0xd9, 0x61, 0x03, 0x64,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0001_Cert = ByteSpan(sTestCert_DAC_FFF1_8000_0001_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF1_8000_0001_SKID_Array[] = {
    0x96, 0xC2, 0xD9, 0x24, 0x94, 0xEA, 0x97, 0x85, 0xC0, 0xD1, 0x67, 0x08, 0xE3, 0x88, 0xF1, 0xC0, 0x91, 0xEA, 0x0F, 0xD5,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0001_SKID = ByteSpan(sTestCert_DAC_FFF1_8000_0001_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF1-8000-0001-Key.pem

constexpr uint8_t sTestCert_DAC_FFF1_8000_0001_PublicKey_Array[] = {
    0x04, 0xc2, 0x25, 0x83, 0x22, 0xc7, 0xdc, 0x72, 0x73, 0x7c, 0x33, 0xbe, 0xed, 0x70, 0x73, 0x37, 0xaa,
    0x24, 0x85, 0xbc, 0x46, 0x79, 0x3e, 0x4d, 0x5a, 0xc9, 0xa7, 0x5a, 0xd7, 0x43, 0x52, 0x66, 0xc9, 0x0a,
    0x02, 0x8e, 0xec, 0xaf, 0x26, 0x50, 0xfe, 0x70, 0x09, 0xef, 0xfc, 0xae, 0xcb, 0xea, 0xd1, 0xf2, 0xc3,
    0xd1, 0x24, 0x35, 0xde, 0xc2, 0xea, 0xd3, 0xd9, 0x92, 0x95, 0xbf, 0xce, 0xd6, 0xc3,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0001_PublicKey = ByteSpan(sTestCert_DAC_FFF1_8000_0001_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF1_8000_0001_PrivateKey_Array[] = {
    0x7b, 0x5c, 0x5a, 0x9f, 0xb4, 0x69, 0x55, 0x47, 0xf8, 0x30, 0x10, 0xdf, 0xc8, 0x97, 0xa6, 0x98,
    0x6a, 0x99, 0x3e, 0xcb, 0x7c, 0xc9, 0xcc, 0x0f, 0x85, 0x7f, 0x90, 0x8a, 0xac, 0xcf, 0x07, 0xee,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0001_PrivateKey = ByteSpan(sTestCert_DAC_FFF1_8000_0001_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF1-8000-0002-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF1_8000_0002_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xe9, 0x30, 0x82, 0x01, 0x8f, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x6c, 0x7d, 0x32, 0xa1, 0x79, 0xfa,
    0x84, 0x13, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x31, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30,
    0x30, 0x30, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f,
    0x39, 0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b,
    0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41,
    0x43, 0x20, 0x30, 0x30, 0x30, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02,
    0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x31, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c,
    0x02, 0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x30, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01,
    0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x75, 0xd6, 0x0f, 0x47, 0x69, 0xba, 0xd7,
    0x91, 0x53, 0x56, 0x49, 0xc4, 0x4d, 0x5f, 0xb8, 0x5a, 0xdd, 0x57, 0x3a, 0x58, 0x45, 0x24, 0x67, 0xef, 0x1c, 0x58, 0xc8, 0x81,
    0x15, 0x52, 0x10, 0x36, 0xe6, 0xd7, 0xea, 0x43, 0x3d, 0xb3, 0xee, 0x57, 0x15, 0xa1, 0xb5, 0xf2, 0x84, 0x3b, 0xa3, 0x1e, 0x47,
    0x00, 0xe5, 0x7c, 0x4c, 0x7e, 0xf9, 0x06, 0x40, 0x8c, 0x9b, 0xf6, 0x94, 0x9e, 0x90, 0x9a, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c,
    0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01,
    0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x0d, 0xbc, 0x26,
    0x6e, 0x28, 0xe4, 0x09, 0xf5, 0x9b, 0xa8, 0x96, 0xf9, 0x8d, 0x7c, 0x9b, 0xa9, 0x6f, 0x19, 0xf5, 0xb1, 0x30, 0x1f, 0x06, 0x03,
    0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xaf, 0x42, 0xb7, 0x09, 0x4d, 0xeb, 0xd5, 0x15, 0xec, 0x6e, 0xcf, 0x33,
    0xb8, 0x11, 0x15, 0x22, 0x5f, 0x32, 0x52, 0x88, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03,
    0x48, 0x00, 0x30, 0x45, 0x02, 0x21, 0x00, 0xd4, 0xee, 0x6c, 0x8a, 0x87, 0xbd, 0x06, 0xf3, 0xe7, 0x8a, 0x85, 0x4b, 0x0c, 0x61,
    0x28, 0xa9, 0xb4, 0x0d, 0x7d, 0xf8, 0x6e, 0x2b, 0x4f, 0x34, 0x2a, 0xf8, 0x6c, 0xd3, 0x22, 0x05, 0x8a, 0xa5, 0x02, 0x20, 0x4b,
    0xa3, 0x16, 0x04, 0x6b, 0x0e, 0x65, 0x45, 0x33, 0x8e, 0xd1, 0x71, 0xdf, 0x69, 0x2b, 0x4f, 0xed, 0x8e, 0x71, 0x1e, 0x14, 0x79,
    0xc6, 0x8c, 0xf1, 0x2e, 0xd2, 0x56, 0x01, 0x97, 0xd4, 0xef,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0002_Cert = ByteSpan(sTestCert_DAC_FFF1_8000_0002_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF1_8000_0002_SKID_Array[] = {
    0x0D, 0xBC, 0x26, 0x6E, 0x28, 0xE4, 0x09, 0xF5, 0x9B, 0xA8, 0x96, 0xF9, 0x8D, 0x7C, 0x9B, 0xA9, 0x6F, 0x19, 0xF5, 0xB1,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0002_SKID = ByteSpan(sTestCert_DAC_FFF1_8000_0002_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF1-8000-0002-Key.pem

constexpr uint8_t sTestCert_DAC_FFF1_8000_0002_PublicKey_Array[] = {
    0x04, 0x75, 0xd6, 0x0f, 0x47, 0x69, 0xba, 0xd7, 0x91, 0x53, 0x56, 0x49, 0xc4, 0x4d, 0x5f, 0xb8, 0x5a,
    0xdd, 0x57, 0x3a, 0x58, 0x45, 0x24, 0x67, 0xef, 0x1c, 0x58, 0xc8, 0x81, 0x15, 0x52, 0x10, 0x36, 0xe6,
    0xd7, 0xea, 0x43, 0x3d, 0xb3, 0xee, 0x57, 0x15, 0xa1, 0xb5, 0xf2, 0x84, 0x3b, 0xa3, 0x1e, 0x47, 0x00,
    0xe5, 0x7c, 0x4c, 0x7e, 0xf9, 0x06, 0x40, 0x8c, 0x9b, 0xf6, 0x94, 0x9e, 0x90, 0x9a,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0002_PublicKey = ByteSpan(sTestCert_DAC_FFF1_8000_0002_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF1_8000_0002_PrivateKey_Array[] = {
    0x4f, 0x14, 0x1b, 0xfc, 0x27, 0x67, 0xd3, 0xe8, 0xf9, 0x24, 0x7b, 0xf9, 0x23, 0x1c, 0xf5, 0xa2,
    0x5c, 0x58, 0x9f, 0xca, 0x02, 0x31, 0xc2, 0x56, 0xff, 0x88, 0x96, 0x4c, 0x98, 0xef, 0x82, 0x3a,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0002_PrivateKey = ByteSpan(sTestCert_DAC_FFF1_8000_0002_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF1-8000-0003-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF1_8000_0003_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xe9, 0x30, 0x82, 0x01, 0x8f, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x36, 0x65, 0x7b, 0xde, 0x4d, 0xc2,
    0xc9, 0x86, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x31, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30,
    0x30, 0x30, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f,
    0x39, 0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b,
    0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41,
    0x43, 0x20, 0x30, 0x30, 0x30, 0x33, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02,
    0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x31, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c,
    0x02, 0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x30, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01,
    0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xff, 0xe4, 0x1e, 0x84, 0x7e, 0xbf, 0x14,
    0x11, 0x58, 0xfa, 0xe3, 0xd9, 0x7d, 0x76, 0x39, 0x13, 0x6b, 0x36, 0x4a, 0xbd, 0x9f, 0x77, 0x22, 0x52, 0x53, 0x6f, 0x1e, 0xd1,
    0xcf, 0x3e, 0x54, 0xd0, 0xec, 0x8d, 0x19, 0x3c, 0x4e, 0x2a, 0x48, 0x0c, 0x87, 0xd6, 0xc8, 0x94, 0xd6, 0x27, 0x0e, 0x69, 0x17,
    0xd9, 0x01, 0x63, 0xc5, 0xce, 0x9b, 0xf2, 0xdc, 0xbc, 0xf3, 0x34, 0xca, 0x5e, 0xa0, 0x9e, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c,
    0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01,
    0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x7a, 0x8a, 0x02,
    0x5e, 0x3b, 0xde, 0xa7, 0x84, 0x43, 0x3b, 0xae, 0x18, 0xe0, 0xec, 0x5e, 0xcd, 0x27, 0x71, 0xa4, 0xbe, 0x30, 0x1f, 0x06, 0x03,
    0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xaf, 0x42, 0xb7, 0x09, 0x4d, 0xeb, 0xd5, 0x15, 0xec, 0x6e, 0xcf, 0x33,
    0xb8, 0x11, 0x15, 0x22, 0x5f, 0x32, 0x52, 0x88, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03,
    0x48, 0x00, 0x30, 0x45, 0x02, 0x21, 0x00, 0x95, 0x0d, 0xff, 0x05, 0x7e, 0x44, 0x68, 0x06, 0x17, 0x62, 0x87, 0xc1, 0xa5, 0x6c,
    0x86, 0x1d, 0xe1, 0x5f, 0x44, 0x36, 0xf1, 0x14, 0x0b, 0x5e, 0xe7, 0x79, 0xaf, 0x8d, 0xb8, 0x46, 0x26, 0x01, 0x02, 0x20, 0x04,
    0x02, 0x76, 0xff, 0x31, 0x15, 0x32, 0xda, 0x8d, 0x1b, 0x65, 0xf9, 0x2e, 0xb8, 0xbf, 0xf9, 0xcd, 0x78, 0x56, 0xb7, 0xc4, 0x56,
    0x2f, 0x1b, 0xf5, 0x63, 0xce, 0x15, 0xdc, 0xe2, 0x51, 0xd7,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0003_Cert = ByteSpan(sTestCert_DAC_FFF1_8000_0003_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF1_8000_0003_SKID_Array[] = {
    0x7A, 0x8A, 0x02, 0x5E, 0x3B, 0xDE, 0xA7, 0x84, 0x43, 0x3B, 0xAE, 0x18, 0xE0, 0xEC, 0x5E, 0xCD, 0x27, 0x71, 0xA4, 0xBE,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0003_SKID = ByteSpan(sTestCert_DAC_FFF1_8000_0003_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF1-8000-0003-Key.pem

constexpr uint8_t sTestCert_DAC_FFF1_8000_0003_PublicKey_Array[] = {
    0x04, 0xff, 0xe4, 0x1e, 0x84, 0x7e, 0xbf, 0x14, 0x11, 0x58, 0xfa, 0xe3, 0xd9, 0x7d, 0x76, 0x39, 0x13,
    0x6b, 0x36, 0x4a, 0xbd, 0x9f, 0x77, 0x22, 0x52, 0x53, 0x6f, 0x1e, 0xd1, 0xcf, 0x3e, 0x54, 0xd0, 0xec,
    0x8d, 0x19, 0x3c, 0x4e, 0x2a, 0x48, 0x0c, 0x87, 0xd6, 0xc8, 0x94, 0xd6, 0x27, 0x0e, 0x69, 0x17, 0xd9,
    0x01, 0x63, 0xc5, 0xce, 0x9b, 0xf2, 0xdc, 0xbc, 0xf3, 0x34, 0xca, 0x5e, 0xa0, 0x9e,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0003_PublicKey = ByteSpan(sTestCert_DAC_FFF1_8000_0003_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF1_8000_0003_PrivateKey_Array[] = {
    0x9e, 0x5d, 0x2d, 0xc6, 0x2a, 0x06, 0x45, 0xfe, 0xaa, 0xcf, 0xd5, 0x75, 0x5f, 0x5a, 0x4e, 0xaf,
    0xd2, 0x0c, 0x6e, 0x67, 0xe9, 0xa0, 0xd9, 0xe9, 0x38, 0xfc, 0xb6, 0x89, 0x12, 0x34, 0x2e, 0x89,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0003_PrivateKey = ByteSpan(sTestCert_DAC_FFF1_8000_0003_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF1-8000-0004-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF1_8000_0004_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xea, 0x30, 0x82, 0x01, 0x8f, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x0c, 0x69, 0x4f, 0x7f, 0x86, 0x60,
    0x67, 0xb2, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x31, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30,
    0x30, 0x30, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f,
    0x39, 0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b,
    0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41,
    0x43, 0x20, 0x30, 0x30, 0x30, 0x34, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02,
    0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x31, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c,
    0x02, 0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x30, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01,
    0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x7a, 0x84, 0x58, 0xaf, 0xbb, 0x9b, 0xcd,
    0x15, 0xe1, 0x9a, 0xdc, 0xd2, 0x66, 0xf6, 0x6c, 0x9c, 0x2f, 0x60, 0x7c, 0x74, 0x74, 0x7a, 0x35, 0xf8, 0x0f, 0x37, 0xe1, 0x18,
    0x13, 0x3f, 0x80, 0xf1, 0x76, 0x01, 0x13, 0x27, 0x8f, 0x91, 0xf1, 0x5a, 0xa0, 0xf7, 0xf8, 0x79, 0x32, 0x09, 0x4f, 0xe6, 0x9f,
    0xb7, 0x28, 0x68, 0xa8, 0x1e, 0x26, 0x97, 0x9b, 0x36, 0x8b, 0x33, 0xb5, 0x54, 0x31, 0x03, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c,
    0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01,
    0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xd5, 0xad, 0xb2,
    0xb8, 0x83, 0x8e, 0xc8, 0x07, 0x3c, 0x47, 0x72, 0xdc, 0x7e, 0x87, 0x97, 0xfe, 0xbb, 0x23, 0xb3, 0xae, 0x30, 0x1f, 0x06, 0x03,
    0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xaf, 0x42, 0xb7, 0x09, 0x4d, 0xeb, 0xd5, 0x15, 0xec, 0x6e, 0xcf, 0x33,
    0xb8, 0x11, 0x15, 0x22, 0x5f, 0x32, 0x52, 0x88, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03,
    0x49, 0x00, 0x30, 0x46, 0x02, 0x21, 0x00, 0xf6, 0x47, 0x00, 0xa4, 0x17, 0x4e, 0xe2, 0xa5, 0x11, 0x71, 0x43, 0x13, 0x67, 0xeb,
    0x2c, 0x52, 0xbb, 0x78, 0xd3, 0xe0, 0xde, 0xea, 0x96, 0xe7, 0xcf, 0x6a, 0x36, 0x96, 0xf0, 0xe5, 0xe7, 0xe5, 0x02, 0x21, 0x00,
    0xbd, 0x56, 0x27, 0xec, 0x4e, 0xe5, 0xca, 0x14, 0x31, 0x78, 0x06, 0x28, 0xf3, 0x1d, 0xc1, 0xe3, 0xd7, 0x1e, 0xff, 0x25, 0x7f,
    0x87, 0xb6, 0xa0, 0x08, 0x7e, 0x71, 0x6c, 0xbb, 0x60, 0x61, 0xbf,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0004_Cert = ByteSpan(sTestCert_DAC_FFF1_8000_0004_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF1_8000_0004_SKID_Array[] = {
    0xD5, 0xAD, 0xB2, 0xB8, 0x83, 0x8E, 0xC8, 0x07, 0x3C, 0x47, 0x72, 0xDC, 0x7E, 0x87, 0x97, 0xFE, 0xBB, 0x23, 0xB3, 0xAE,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0004_SKID = ByteSpan(sTestCert_DAC_FFF1_8000_0004_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF1-8000-0004-Key.pem

constexpr uint8_t sTestCert_DAC_FFF1_8000_0004_PublicKey_Array[] = {
    0x04, 0x7a, 0x84, 0x58, 0xaf, 0xbb, 0x9b, 0xcd, 0x15, 0xe1, 0x9a, 0xdc, 0xd2, 0x66, 0xf6, 0x6c, 0x9c,
    0x2f, 0x60, 0x7c, 0x74, 0x74, 0x7a, 0x35, 0xf8, 0x0f, 0x37, 0xe1, 0x18, 0x13, 0x3f, 0x80, 0xf1, 0x76,
    0x01, 0x13, 0x27, 0x8f, 0x91, 0xf1, 0x5a, 0xa0, 0xf7, 0xf8, 0x79, 0x32, 0x09, 0x4f, 0xe6, 0x9f, 0xb7,
    0x28, 0x68, 0xa8, 0x1e, 0x26, 0x97, 0x9b, 0x36, 0x8b, 0x33, 0xb5, 0x54, 0x31, 0x03,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0004_PublicKey = ByteSpan(sTestCert_DAC_FFF1_8000_0004_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF1_8000_0004_PrivateKey_Array[] = {
    0x05, 0xc6, 0xc3, 0xa8, 0x4d, 0xc6, 0x05, 0xcc, 0x3c, 0xc8, 0x05, 0x80, 0x09, 0xb0, 0x1b, 0x32,
    0x9c, 0xf6, 0x0c, 0xf1, 0x59, 0x70, 0xc6, 0xa9, 0x0e, 0xad, 0xaa, 0xe2, 0xde, 0x49, 0x64, 0x9e,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0004_PrivateKey = ByteSpan(sTestCert_DAC_FFF1_8000_0004_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF1-8000-0005-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF1_8000_0005_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xea, 0x30, 0x82, 0x01, 0x8f, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x4e, 0x94, 0xeb, 0x65, 0xa3, 0xc2,
    0x66, 0xa0, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x31, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30,
    0x30, 0x30, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f,
    0x39, 0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b,
    0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41,
    0x43, 0x20, 0x30, 0x30, 0x30, 0x35, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02,
    0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x31, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c,
    0x02, 0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x30, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01,
    0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xae, 0x4f, 0x80, 0xc5, 0x46, 0x59, 0x6e,
    0x54, 0x91, 0x16, 0xe3, 0x24, 0x39, 0x5c, 0xa1, 0x2d, 0x52, 0xb9, 0x80, 0x7b, 0xce, 0xc3, 0xf7, 0x1d, 0xe2, 0x90, 0x05, 0x31,
    0x8c, 0xc6, 0x81, 0x56, 0xd3, 0xde, 0x4e, 0x8c, 0x46, 0x09, 0xdf, 0xda, 0xb6, 0xa5, 0x17, 0xd4, 0x40, 0x20, 0x5b, 0x7d, 0xbe,
    0x7e, 0x73, 0x76, 0x29, 0x9e, 0xc0, 0x32, 0x38, 0xdf, 0x11, 0x1f, 0x92, 0x1b, 0x40, 0xc1, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c,
    0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01,
    0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x7d, 0x1a, 0x37,
    0x72, 0xf6, 0x99, 0xaa, 0x54, 0xc6, 0xd9, 0xe2, 0x0a, 0x47, 0xba, 0xd7, 0xf1, 0x49, 0xd7, 0xc6, 0x35, 0x30, 0x1f, 0x06, 0x03,
    0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xaf, 0x42, 0xb7, 0x09, 0x4d, 0xeb, 0xd5, 0x15, 0xec, 0x6e, 0xcf, 0x33,
    0xb8, 0x11, 0x15, 0x22, 0x5f, 0x32, 0x52, 0x88, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03,
    0x49, 0x00, 0x30, 0x46, 0x02, 0x21, 0x00, 0x95, 0xc6, 0x79, 0x3c, 0x5c, 0x7b, 0x74, 0xcf, 0x28, 0x83, 0xcf, 0x45, 0x48, 0x8a,
    0x9f, 0xeb, 0x09, 0x84, 0x01, 0x5a, 0xdf, 0x5d, 0xf5, 0xb9, 0xd9, 0xcc, 0xc1, 0x75, 0x53, 0xb6, 0xed, 0x4a, 0x02, 0x21, 0x00,
    0xed, 0x4e, 0x8e, 0xb9, 0xd1, 0x86, 0x04, 0x72, 0xcf, 0xd7, 0x11, 0x9c, 0x51, 0x73, 0xa2, 0x33, 0x61, 0x2e, 0xe6, 0x81, 0xfa,
    0x10, 0x76, 0xee, 0xe7, 0x33, 0x03, 0xb5, 0x10, 0x9d, 0x56, 0xb1,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0005_Cert = ByteSpan(sTestCert_DAC_FFF1_8000_0005_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF1_8000_0005_SKID_Array[] = {
    0x7D, 0x1A, 0x37, 0x72, 0xF6, 0x99, 0xAA, 0x54, 0xC6, 0xD9, 0xE2, 0x0A, 0x47, 0xBA, 0xD7, 0xF1, 0x49, 0xD7, 0xC6, 0x35,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0005_SKID = ByteSpan(sTestCert_DAC_FFF1_8000_0005_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF1-8000-0005-Key.pem

constexpr uint8_t sTestCert_DAC_FFF1_8000_0005_PublicKey_Array[] = {
    0x04, 0xae, 0x4f, 0x80, 0xc5, 0x46, 0x59, 0x6e, 0x54, 0x91, 0x16, 0xe3, 0x24, 0x39, 0x5c, 0xa1, 0x2d,
    0x52, 0xb9, 0x80, 0x7b, 0xce, 0xc3, 0xf7, 0x1d, 0xe2, 0x90, 0x05, 0x31, 0x8c, 0xc6, 0x81, 0x56, 0xd3,
    0xde, 0x4e, 0x8c, 0x46, 0x09, 0xdf, 0xda, 0xb6, 0xa5, 0x17, 0xd4, 0x40, 0x20, 0x5b, 0x7d, 0xbe, 0x7e,
    0x73, 0x76, 0x29, 0x9e, 0xc0, 0x32, 0x38, 0xdf, 0x11, 0x1f, 0x92, 0x1b, 0x40, 0xc1,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0005_PublicKey = ByteSpan(sTestCert_DAC_FFF1_8000_0005_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF1_8000_0005_PrivateKey_Array[] = {
    0x64, 0x9f, 0xb7, 0xae, 0x06, 0xf2, 0x20, 0x02, 0xc9, 0xf5, 0x87, 0x52, 0x3e, 0x2f, 0x37, 0x95,
    0x04, 0x1c, 0x90, 0xe9, 0x0d, 0xa2, 0x06, 0x87, 0xce, 0x84, 0xdd, 0xee, 0xdc, 0x9c, 0x58, 0x1e,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0005_PrivateKey = ByteSpan(sTestCert_DAC_FFF1_8000_0005_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF1-8000-0006-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF1_8000_0006_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xea, 0x30, 0x82, 0x01, 0x8f, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x38, 0xf6, 0x94, 0x89, 0xc3, 0x08,
    0x95, 0xd1, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x31, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30,
    0x30, 0x30, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f,
    0x39, 0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b,
    0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41,
    0x43, 0x20, 0x30, 0x30, 0x30, 0x36, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02,
    0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x31, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c,
    0x02, 0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x30, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01,
    0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x81, 0x63, 0x0f, 0x80, 0xbe, 0x60, 0xf3,
    0xc2, 0x60, 0xde, 0xda, 0xba, 0xb3, 0xe3, 0xa6, 0xa7, 0xd8, 0xb5, 0x2b, 0x51, 0x0c, 0x69, 0xfe, 0x76, 0x9d, 0xe4, 0xf1, 0x19,
    0xd9, 0x8a, 0x51, 0x0a, 0x1e, 0x5d, 0x7c, 0x14, 0x79, 0x56, 0x90, 0x54, 0x95, 0xc7, 0x43, 0x7b, 0x0a, 0x1a, 0x66, 0x8f, 0x43,
    0xa8, 0xb0, 0x52, 0xeb, 0xe4, 0x2f, 0xc8, 0x28, 0x38, 0x4f, 0x2d, 0x4c, 0x38, 0x4d, 0x6a, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c,
    0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01,
    0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x26, 0xd1, 0x52,
    0x59, 0xbe, 0x31, 0xd8, 0x1c, 0x7e, 0xef, 0x5b, 0xa9, 0xf7, 0xf5, 0x30, 0xc6, 0x77, 0xe5, 0xc7, 0x66, 0x30, 0x1f, 0x06, 0x03,
    0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xaf, 0x42, 0xb7, 0x09, 0x4d, 0xeb, 0xd5, 0x15, 0xec, 0x6e, 0xcf, 0x33,
    0xb8, 0x11, 0x15, 0x22, 0x5f, 0x32, 0x52, 0x88, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03,
    0x49, 0x00, 0x30, 0x46, 0x02, 0x21, 0x00, 0xe0, 0xdd, 0x1c, 0xe8, 0x2e, 0x10, 0x7c, 0xea, 0xd0, 0xc1, 0x23, 0xf0, 0x55, 0x76,
    0x84, 0x0f, 0x4e, 0x3d, 0x58, 0x57, 0x85, 0xaa, 0x77, 0xcf, 0x56, 0x01, 0xe5, 0xee, 0x72, 0xef, 0x49, 0x34, 0x02, 0x21, 0x00,
    0xf9, 0x3f, 0x06, 0x2c, 0x39, 0x86, 0xa0, 0x74, 0x4c, 0xf1, 0xb9, 0xa4, 0x9c, 0xee, 0x28, 0x60, 0x91, 0xb8, 0x32, 0xc0, 0xc2,
    0x26, 0x20, 0x88, 0xb7, 0x83, 0xb8, 0x96, 0x38, 0x5f, 0x3a, 0xe5,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0006_Cert = ByteSpan(sTestCert_DAC_FFF1_8000_0006_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF1_8000_0006_SKID_Array[] = {
    0x26, 0xD1, 0x52, 0x59, 0xBE, 0x31, 0xD8, 0x1C, 0x7E, 0xEF, 0x5B, 0xA9, 0xF7, 0xF5, 0x30, 0xC6, 0x77, 0xE5, 0xC7, 0x66,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0006_SKID = ByteSpan(sTestCert_DAC_FFF1_8000_0006_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF1-8000-0006-Key.pem

constexpr uint8_t sTestCert_DAC_FFF1_8000_0006_PublicKey_Array[] = {
    0x04, 0x81, 0x63, 0x0f, 0x80, 0xbe, 0x60, 0xf3, 0xc2, 0x60, 0xde, 0xda, 0xba, 0xb3, 0xe3, 0xa6, 0xa7,
    0xd8, 0xb5, 0x2b, 0x51, 0x0c, 0x69, 0xfe, 0x76, 0x9d, 0xe4, 0xf1, 0x19, 0xd9, 0x8a, 0x51, 0x0a, 0x1e,
    0x5d, 0x7c, 0x14, 0x79, 0x56, 0x90, 0x54, 0x95, 0xc7, 0x43, 0x7b, 0x0a, 0x1a, 0x66, 0x8f, 0x43, 0xa8,
    0xb0, 0x52, 0xeb, 0xe4, 0x2f, 0xc8, 0x28, 0x38, 0x4f, 0x2d, 0x4c, 0x38, 0x4d, 0x6a,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0006_PublicKey = ByteSpan(sTestCert_DAC_FFF1_8000_0006_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF1_8000_0006_PrivateKey_Array[] = {
    0x18, 0xa5, 0xd5, 0x70, 0x1b, 0x03, 0x31, 0xd7, 0x0b, 0x78, 0x02, 0x4d, 0xfb, 0xc2, 0xb6, 0x80,
    0xf0, 0xe2, 0x4a, 0x29, 0x87, 0xb5, 0x69, 0x99, 0x0a, 0x7d, 0x41, 0x87, 0x16, 0x0d, 0x87, 0xe4,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0006_PrivateKey = ByteSpan(sTestCert_DAC_FFF1_8000_0006_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF1-8000-0007-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF1_8000_0007_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xe8, 0x30, 0x82, 0x01, 0x8f, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x14, 0x3c, 0x9d, 0x16, 0x89, 0xf4,
    0x98, 0xf0, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x31, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30,
    0x30, 0x30, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f,
    0x39, 0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b,
    0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41,
    0x43, 0x20, 0x30, 0x30, 0x30, 0x37, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02,
    0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x31, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c,
    0x02, 0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x30, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01,
    0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x62, 0xe2, 0xb6, 0xe1, 0xba, 0xff, 0x8d,
    0x74, 0xa6, 0xfd, 0x82, 0x16, 0xc4, 0xcb, 0x67, 0xa3, 0x36, 0x3a, 0x31, 0xe6, 0x91, 0x49, 0x27, 0x92, 0xe6, 0x1a, 0xee, 0x61,
    0x02, 0x61, 0x48, 0x13, 0x96, 0x72, 0x5e, 0xf9, 0x5e, 0x14, 0x26, 0x86, 0xba, 0x98, 0xf3, 0x39, 0xb0, 0xff, 0x65, 0xbc, 0x33,
    0x8b, 0xec, 0x7b, 0x9e, 0x8b, 0xe0, 0xbd, 0xf3, 0xb2, 0x77, 0x49, 0x82, 0x47, 0x62, 0x20, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c,
    0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01,
    0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xee, 0x95, 0xad,
    0x96, 0x98, 0x3a, 0x9e, 0xa9, 0x5b, 0xcd, 0x2b, 0x00, 0xdc, 0x5e, 0x67, 0x17, 0x27, 0x69, 0x03, 0x83, 0x30, 0x1f, 0x06, 0x03,
    0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xaf, 0x42, 0xb7, 0x09, 0x4d, 0xeb, 0xd5, 0x15, 0xec, 0x6e, 0xcf, 0x33,
    0xb8, 0x11, 0x15, 0x22, 0x5f, 0x32, 0x52, 0x88, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03,
    0x47, 0x00, 0x30, 0x44, 0x02, 0x20, 0x2f, 0x51, 0xcf, 0x53, 0xbf, 0x77, 0x77, 0xdf, 0x73, 0x18, 0x09, 0x4b, 0x9d, 0xb5, 0x95,
    0xee, 0xbf, 0x2f, 0xa8, 0x81, 0xc8, 0xc5, 0x72, 0x84, 0x7b, 0x1e, 0x68, 0x9e, 0xce, 0x65, 0x42, 0x64, 0x02, 0x20, 0x29, 0x78,
    0x27, 0x08, 0xee, 0x6b, 0x32, 0xc7, 0xf0, 0x8f, 0xf6, 0x3d, 0xbe, 0x61, 0x8e, 0x9a, 0x58, 0x0b, 0xb1, 0x4c, 0x18, 0x3b, 0xc2,
    0x88, 0x77, 0x7a, 0xdf, 0x9e, 0x2d, 0xcf, 0xf5, 0xe6,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0007_Cert = ByteSpan(sTestCert_DAC_FFF1_8000_0007_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF1_8000_0007_SKID_Array[] = {
    0xEE, 0x95, 0xAD, 0x96, 0x98, 0x3A, 0x9E, 0xA9, 0x5B, 0xCD, 0x2B, 0x00, 0xDC, 0x5E, 0x67, 0x17, 0x27, 0x69, 0x03, 0x83,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0007_SKID = ByteSpan(sTestCert_DAC_FFF1_8000_0007_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF1-8000-0007-Key.pem

constexpr uint8_t sTestCert_DAC_FFF1_8000_0007_PublicKey_Array[] = {
    0x04, 0x62, 0xe2, 0xb6, 0xe1, 0xba, 0xff, 0x8d, 0x74, 0xa6, 0xfd, 0x82, 0x16, 0xc4, 0xcb, 0x67, 0xa3,
    0x36, 0x3a, 0x31, 0xe6, 0x91, 0x49, 0x27, 0x92, 0xe6, 0x1a, 0xee, 0x61, 0x02, 0x61, 0x48, 0x13, 0x96,
    0x72, 0x5e, 0xf9, 0x5e, 0x14, 0x26, 0x86, 0xba, 0x98, 0xf3, 0x39, 0xb0, 0xff, 0x65, 0xbc, 0x33, 0x8b,
    0xec, 0x7b, 0x9e, 0x8b, 0xe0, 0xbd, 0xf3, 0xb2, 0x77, 0x49, 0x82, 0x47, 0x62, 0x20,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0007_PublicKey = ByteSpan(sTestCert_DAC_FFF1_8000_0007_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF1_8000_0007_PrivateKey_Array[] = {
    0x72, 0x7f, 0x10, 0x05, 0xcb, 0xa4, 0x7e, 0xd7, 0x82, 0x2a, 0x9d, 0x93, 0x09, 0x43, 0x62, 0x16,
    0x17, 0xcf, 0xd3, 0xb7, 0x9d, 0x9a, 0xf5, 0x28, 0xb8, 0x01, 0xec, 0xf9, 0xf1, 0x99, 0x22, 0x04,
};

extern const ByteSpan sTestCert_DAC_FFF1_8000_0007_PrivateKey = ByteSpan(sTestCert_DAC_FFF1_8000_0007_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8001-0008-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8001_0008_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xe9, 0x30, 0x82, 0x01, 0x8f, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x0a, 0x43, 0x91, 0x12, 0x6d, 0x01,
    0x49, 0xd9, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30,
    0x30, 0x31, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f,
    0x39, 0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b,
    0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41,
    0x43, 0x20, 0x30, 0x30, 0x30, 0x38, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02,
    0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c,
    0x02, 0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x31, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01,
    0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x7c, 0x39, 0xa7, 0xe1, 0x0f, 0x3d, 0x65,
    0x80, 0x31, 0x5e, 0xda, 0x3f, 0x00, 0x9f, 0xcf, 0x15, 0xce, 0x7d, 0x3c, 0x07, 0x2b, 0x8a, 0x4d, 0xa8, 0x65, 0x54, 0xa9, 0xc9,
    0xc7, 0xa8, 0xd9, 0x11, 0x20, 0x24, 0x50, 0xe0, 0xa1, 0x28, 0x87, 0x77, 0x54, 0x97, 0x86, 0xcb, 0x41, 0x94, 0x59, 0x8a, 0x27,
    0x23, 0x50, 0xcb, 0x00, 0x75, 0x60, 0xb0, 0x60, 0xfb, 0xfd, 0x97, 0xef, 0x01, 0xa1, 0xca, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c,
    0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01,
    0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xe2, 0x5b, 0x55,
    0x28, 0x31, 0x82, 0x75, 0x3a, 0xf7, 0x5c, 0x12, 0x13, 0xcd, 0x11, 0x0a, 0x54, 0x2a, 0xac, 0x99, 0x91, 0x30, 0x1f, 0x06, 0x03,
    0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xd0, 0x5a, 0x9b, 0x67, 0x71, 0xa1, 0xa3, 0x7a, 0x9b, 0xc1, 0x9d, 0x95,
    0xe8, 0xb5, 0xdb, 0x8f, 0x43, 0x00, 0xb6, 0x3a, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03,
    0x48, 0x00, 0x30, 0x45, 0x02, 0x21, 0x00, 0xcd, 0xe6, 0xcb, 0x2d, 0x93, 0x17, 0x4f, 0x99, 0x0e, 0x93, 0x23, 0x95, 0x77, 0xfe,
    0x94, 0x31, 0x6a, 0x0f, 0x5c, 0x94, 0xa6, 0x25, 0x33, 0x6b, 0x1b, 0x65, 0x4b, 0xef, 0x8c, 0x0a, 0xdf, 0x4f, 0x02, 0x20, 0x01,
    0x24, 0x8d, 0x87, 0xf8, 0xaf, 0xf7, 0xd2, 0xe0, 0xd5, 0xe6, 0x7f, 0x3c, 0x6e, 0xf6, 0xd0, 0xc2, 0x1f, 0xff, 0x0e, 0x73, 0xaa,
    0x9d, 0x60, 0x21, 0xa2, 0x45, 0x69, 0xed, 0xee, 0x5c, 0xa3,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_0008_Cert = ByteSpan(sTestCert_DAC_FFF2_8001_0008_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8001_0008_SKID_Array[] = {
    0xE2, 0x5B, 0x55, 0x28, 0x31, 0x82, 0x75, 0x3A, 0xF7, 0x5C, 0x12, 0x13, 0xCD, 0x11, 0x0A, 0x54, 0x2A, 0xAC, 0x99, 0x91,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_0008_SKID = ByteSpan(sTestCert_DAC_FFF2_8001_0008_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8001-0008-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8001_0008_PublicKey_Array[] = {
    0x04, 0x7c, 0x39, 0xa7, 0xe1, 0x0f, 0x3d, 0x65, 0x80, 0x31, 0x5e, 0xda, 0x3f, 0x00, 0x9f, 0xcf, 0x15,
    0xce, 0x7d, 0x3c, 0x07, 0x2b, 0x8a, 0x4d, 0xa8, 0x65, 0x54, 0xa9, 0xc9, 0xc7, 0xa8, 0xd9, 0x11, 0x20,
    0x24, 0x50, 0xe0, 0xa1, 0x28, 0x87, 0x77, 0x54, 0x97, 0x86, 0xcb, 0x41, 0x94, 0x59, 0x8a, 0x27, 0x23,
    0x50, 0xcb, 0x00, 0x75, 0x60, 0xb0, 0x60, 0xfb, 0xfd, 0x97, 0xef, 0x01, 0xa1, 0xca,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_0008_PublicKey = ByteSpan(sTestCert_DAC_FFF2_8001_0008_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8001_0008_PrivateKey_Array[] = {
    0xb6, 0x9a, 0x32, 0x05, 0x3f, 0xe3, 0xb3, 0x29, 0x72, 0xe7, 0x41, 0x41, 0x04, 0xea, 0x22, 0x0b,
    0xa2, 0x89, 0xda, 0x8a, 0x51, 0x62, 0x63, 0xc0, 0x30, 0x01, 0x27, 0xab, 0xd3, 0x58, 0xee, 0x7e,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_0008_PrivateKey = ByteSpan(sTestCert_DAC_FFF2_8001_0008_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8001-0009-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8001_0009_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xe8, 0x30, 0x82, 0x01, 0x8f, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x64, 0xfa, 0x18, 0x1f, 0xdb, 0xac,
    0x4c, 0x5f, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30,
    0x30, 0x31, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f,
    0x39, 0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b,
    0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41,
    0x43, 0x20, 0x30, 0x30, 0x30, 0x39, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02,
    0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c,
    0x02, 0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x31, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01,
    0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x0d, 0xc7, 0x2b, 0xd7, 0x48, 0xe0, 0x6f,
    0x4e, 0x19, 0x76, 0x1e, 0x70, 0xc0, 0x13, 0x2e, 0x9c, 0xa9, 0x98, 0x05, 0xb7, 0x20, 0x23, 0x18, 0x27, 0xce, 0x26, 0x7d, 0x19,
    0xe5, 0x64, 0xe9, 0xd9, 0xad, 0xb6, 0x98, 0x56, 0xe9, 0xb2, 0xe8, 0xb3, 0xf0, 0x71, 0x80, 0xcf, 0xbb, 0x8f, 0x64, 0xf6, 0xa0,
    0x5d, 0x1a, 0x64, 0xbd, 0xfa, 0x14, 0x8f, 0x72, 0x7b, 0x43, 0xd7, 0xb5, 0x75, 0xd8, 0x84, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c,
    0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01,
    0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x48, 0x36, 0xf5,
    0xad, 0x51, 0x96, 0x8d, 0x50, 0x4b, 0xa1, 0xb3, 0x76, 0xff, 0x60, 0x53, 0x17, 0x0b, 0xb2, 0xd5, 0x02, 0x30, 0x1f, 0x06, 0x03,
    0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xd0, 0x5a, 0x9b, 0x67, 0x71, 0xa1, 0xa3, 0x7a, 0x9b, 0xc1, 0x9d, 0x95,
    0xe8, 0xb5, 0xdb, 0x8f, 0x43, 0x00, 0xb6, 0x3a, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03,
    0x47, 0x00, 0x30, 0x44, 0x02, 0x20, 0x02, 0xe0, 0xce, 0x2f, 0x95, 0x4f, 0x03, 0x94, 0x71, 0x5e, 0x32, 0xdc, 0x5b, 0x98, 0xfc,
    0x29, 0xb3, 0x7f, 0x92, 0xa7, 0xf2, 0x87, 0x23, 0x69, 0x71, 0xa0, 0xb0, 0x24, 0x7a, 0x2b, 0x4d, 0x7b, 0x02, 0x20, 0x15, 0x59,
    0xd3, 0x02, 0xa4, 0xae, 0x5d, 0x75, 0x28, 0x1c, 0x39, 0xd6, 0x45, 0xb3, 0x24, 0x47, 0x12, 0xb2, 0xcc, 0x3d, 0xa5, 0xe7, 0x53,
    0xe2, 0x7c, 0x7e, 0x90, 0x6b, 0xf5, 0xf8, 0xe0, 0x91,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_0009_Cert = ByteSpan(sTestCert_DAC_FFF2_8001_0009_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8001_0009_SKID_Array[] = {
    0x48, 0x36, 0xF5, 0xAD, 0x51, 0x96, 0x8D, 0x50, 0x4B, 0xA1, 0xB3, 0x76, 0xFF, 0x60, 0x53, 0x17, 0x0B, 0xB2, 0xD5, 0x02,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_0009_SKID = ByteSpan(sTestCert_DAC_FFF2_8001_0009_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8001-0009-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8001_0009_PublicKey_Array[] = {
    0x04, 0x0d, 0xc7, 0x2b, 0xd7, 0x48, 0xe0, 0x6f, 0x4e, 0x19, 0x76, 0x1e, 0x70, 0xc0, 0x13, 0x2e, 0x9c,
    0xa9, 0x98, 0x05, 0xb7, 0x20, 0x23, 0x18, 0x27, 0xce, 0x26, 0x7d, 0x19, 0xe5, 0x64, 0xe9, 0xd9, 0xad,
    0xb6, 0x98, 0x56, 0xe9, 0xb2, 0xe8, 0xb3, 0xf0, 0x71, 0x80, 0xcf, 0xbb, 0x8f, 0x64, 0xf6, 0xa0, 0x5d,
    0x1a, 0x64, 0xbd, 0xfa, 0x14, 0x8f, 0x72, 0x7b, 0x43, 0xd7, 0xb5, 0x75, 0xd8, 0x84,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_0009_PublicKey = ByteSpan(sTestCert_DAC_FFF2_8001_0009_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8001_0009_PrivateKey_Array[] = {
    0x2d, 0x70, 0x20, 0x39, 0xf7, 0xa6, 0x25, 0x65, 0xe4, 0x7c, 0xf1, 0x40, 0xa9, 0x54, 0x3e, 0x55,
    0xd9, 0x0e, 0x41, 0x72, 0x2b, 0x01, 0x8d, 0xa2, 0x9f, 0x7e, 0x23, 0x85, 0x47, 0x93, 0xb3, 0x52,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_0009_PrivateKey = ByteSpan(sTestCert_DAC_FFF2_8001_0009_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8001-000A-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8001_000A_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xe8, 0x30, 0x82, 0x01, 0x8f, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x2d, 0xf6, 0x98, 0x7d, 0x26, 0x35,
    0x4a, 0xbe, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30,
    0x30, 0x31, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f,
    0x39, 0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b,
    0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41,
    0x43, 0x20, 0x30, 0x30, 0x30, 0x41, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02,
    0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c,
    0x02, 0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x31, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01,
    0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x53, 0xda, 0xdd, 0x85, 0xa3, 0xa9, 0x05,
    0xcb, 0x1d, 0x87, 0xb3, 0x5a, 0xb2, 0xe2, 0x8d, 0x43, 0xcf, 0x2a, 0x2d, 0x69, 0x60, 0xf0, 0x25, 0x63, 0xbd, 0x54, 0xc2, 0xd1,
    0x05, 0xc8, 0x15, 0xd5, 0x6b, 0x58, 0x42, 0x71, 0x26, 0x47, 0xc6, 0x76, 0x19, 0x69, 0x22, 0xbd, 0x7c, 0xa2, 0x90, 0x07, 0xba,
    0x54, 0x39, 0x60, 0xe3, 0xb7, 0xb3, 0x5c, 0x26, 0x6a, 0x19, 0xdf, 0xc2, 0xf2, 0xf3, 0xae, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c,
    0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01,
    0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x32, 0x05, 0x94,
    0xd8, 0xf0, 0xed, 0x8f, 0x62, 0x15, 0x22, 0xef, 0x73, 0x9c, 0xdc, 0x4a, 0xac, 0x18, 0x50, 0xe9, 0x73, 0x30, 0x1f, 0x06, 0x03,
    0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xd0, 0x5a, 0x9b, 0x67, 0x71, 0xa1, 0xa3, 0x7a, 0x9b, 0xc1, 0x9d, 0x95,
    0xe8, 0xb5, 0xdb, 0x8f, 0x43, 0x00, 0xb6, 0x3a, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03,
    0x47, 0x00, 0x30, 0x44, 0x02, 0x20, 0x61, 0x56, 0xff, 0x87, 0x09, 0x93, 0x2a, 0x4c, 0x6b, 0xe8, 0x63, 0x12, 0xd0, 0xc3, 0x12,
    0xfc, 0x00, 0xf9, 0x18, 0xd3, 0x04, 0x15, 0xa8, 0xca, 0xc7, 0xba, 0xcd, 0xd8, 0x11, 0x56, 0xc1, 0x61, 0x02, 0x20, 0x68, 0x8c,
    0xd0, 0x9d, 0x36, 0xba, 0x34, 0x75, 0x5d, 0xc4, 0x87, 0xcc, 0x3e, 0x4b, 0xd1, 0xab, 0x7e, 0xe1, 0xe1, 0xfa, 0x1e, 0x5d, 0xe3,
    0xb5, 0x51, 0xc4, 0x5e, 0xaf, 0x5a, 0x28, 0x61, 0xb8,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_000A_Cert = ByteSpan(sTestCert_DAC_FFF2_8001_000A_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8001_000A_SKID_Array[] = {
    0x32, 0x05, 0x94, 0xD8, 0xF0, 0xED, 0x8F, 0x62, 0x15, 0x22, 0xEF, 0x73, 0x9C, 0xDC, 0x4A, 0xAC, 0x18, 0x50, 0xE9, 0x73,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_000A_SKID = ByteSpan(sTestCert_DAC_FFF2_8001_000A_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8001-000A-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8001_000A_PublicKey_Array[] = {
    0x04, 0x53, 0xda, 0xdd, 0x85, 0xa3, 0xa9, 0x05, 0xcb, 0x1d, 0x87, 0xb3, 0x5a, 0xb2, 0xe2, 0x8d, 0x43,
    0xcf, 0x2a, 0x2d, 0x69, 0x60, 0xf0, 0x25, 0x63, 0xbd, 0x54, 0xc2, 0xd1, 0x05, 0xc8, 0x15, 0xd5, 0x6b,
    0x58, 0x42, 0x71, 0x26, 0x47, 0xc6, 0x76, 0x19, 0x69, 0x22, 0xbd, 0x7c, 0xa2, 0x90, 0x07, 0xba, 0x54,
    0x39, 0x60, 0xe3, 0xb7, 0xb3, 0x5c, 0x26, 0x6a, 0x19, 0xdf, 0xc2, 0xf2, 0xf3, 0xae,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_000A_PublicKey = ByteSpan(sTestCert_DAC_FFF2_8001_000A_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8001_000A_PrivateKey_Array[] = {
    0xa2, 0xce, 0x69, 0xb0, 0x67, 0x20, 0xa2, 0xe1, 0x24, 0x63, 0xa4, 0xd6, 0xd7, 0x3c, 0xc6, 0xd6,
    0xa5, 0x30, 0x97, 0x7d, 0xb6, 0x7b, 0x4c, 0x5d, 0x63, 0xd1, 0x27, 0x25, 0x22, 0xb9, 0xf4, 0x76,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_000A_PrivateKey = ByteSpan(sTestCert_DAC_FFF2_8001_000A_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8001-000B-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8001_000B_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xea, 0x30, 0x82, 0x01, 0x8f, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x14, 0xe3, 0xf1, 0x0d, 0xde, 0x3f,
    0x94, 0x61, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30,
    0x30, 0x31, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f,
    0x39, 0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b,
    0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41,
    0x43, 0x20, 0x30, 0x30, 0x30, 0x42, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02,
    0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c,
    0x02, 0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x31, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01,
    0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xb0, 0x3d, 0x22, 0x15, 0xcf, 0xd0, 0x2a,
    0xb6, 0x2a, 0x18, 0xc3, 0xcb, 0x98, 0x58, 0x2f, 0xc6, 0x3a, 0xe5, 0x14, 0x63, 0xf7, 0xd1, 0xf7, 0x15, 0x3d, 0x26, 0x1b, 0x07,
    0x16, 0x7a, 0x9e, 0xb2, 0xe5, 0x8b, 0x8c, 0xc2, 0x51, 0x31, 0xca, 0xb5, 0xf9, 0x88, 0x6c, 0x77, 0x77, 0xb3, 0x77, 0x9a, 0x5d,
    0x74, 0x54, 0xa8, 0x99, 0x45, 0x28, 0x77, 0x2b, 0x02, 0xc2, 0x65, 0xe8, 0x21, 0xe4, 0xdd, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c,
    0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01,
    0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x7e, 0x35, 0xf7,
    0x08, 0xed, 0x58, 0x5d, 0xd0, 0x53, 0xd8, 0x56, 0xe4, 0x13, 0xd7, 0xd1, 0x25, 0x13, 0x33, 0x9f, 0x17, 0x30, 0x1f, 0x06, 0x03,
    0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xd0, 0x5a, 0x9b, 0x67, 0x71, 0xa1, 0xa3, 0x7a, 0x9b, 0xc1, 0x9d, 0x95,
    0xe8, 0xb5, 0xdb, 0x8f, 0x43, 0x00, 0xb6, 0x3a, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03,
    0x49, 0x00, 0x30, 0x46, 0x02, 0x21, 0x00, 0xd9, 0xe9, 0xd5, 0x18, 0x2a, 0x85, 0x3b, 0xf1, 0x53, 0x87, 0x2b, 0x12, 0x89, 0xc9,
    0x8f, 0x8b, 0x32, 0x55, 0x6c, 0xe4, 0x0a, 0xe6, 0xfa, 0x9b, 0x0d, 0x37, 0x0b, 0x8f, 0x08, 0xd8, 0x3c, 0x0e, 0x02, 0x21, 0x00,
    0xb6, 0xc8, 0x8a, 0xaf, 0xf1, 0xbc, 0xaa, 0xba, 0x40, 0x53, 0x6c, 0x4a, 0x05, 0xdf, 0x1f, 0x73, 0xbd, 0x80, 0x1e, 0xc9, 0xdd,
    0xbb, 0xd6, 0x2b, 0x44, 0xf9, 0x2a, 0x45, 0xfd, 0xa6, 0x44, 0x36,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_000B_Cert = ByteSpan(sTestCert_DAC_FFF2_8001_000B_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8001_000B_SKID_Array[] = {
    0x7E, 0x35, 0xF7, 0x08, 0xED, 0x58, 0x5D, 0xD0, 0x53, 0xD8, 0x56, 0xE4, 0x13, 0xD7, 0xD1, 0x25, 0x13, 0x33, 0x9F, 0x17,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_000B_SKID = ByteSpan(sTestCert_DAC_FFF2_8001_000B_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8001-000B-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8001_000B_PublicKey_Array[] = {
    0x04, 0xb0, 0x3d, 0x22, 0x15, 0xcf, 0xd0, 0x2a, 0xb6, 0x2a, 0x18, 0xc3, 0xcb, 0x98, 0x58, 0x2f, 0xc6,
    0x3a, 0xe5, 0x14, 0x63, 0xf7, 0xd1, 0xf7, 0x15, 0x3d, 0x26, 0x1b, 0x07, 0x16, 0x7a, 0x9e, 0xb2, 0xe5,
    0x8b, 0x8c, 0xc2, 0x51, 0x31, 0xca, 0xb5, 0xf9, 0x88, 0x6c, 0x77, 0x77, 0xb3, 0x77, 0x9a, 0x5d, 0x74,
    0x54, 0xa8, 0x99, 0x45, 0x28, 0x77, 0x2b, 0x02, 0xc2, 0x65, 0xe8, 0x21, 0xe4, 0xdd,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_000B_PublicKey = ByteSpan(sTestCert_DAC_FFF2_8001_000B_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8001_000B_PrivateKey_Array[] = {
    0xb2, 0x7f, 0x21, 0xaf, 0x10, 0xec, 0x13, 0xb4, 0x56, 0xab, 0x2a, 0x6d, 0x6e, 0x3d, 0x99, 0xdc,
    0xbd, 0xa0, 0x5b, 0xab, 0x33, 0x99, 0xb7, 0x87, 0xfc, 0x39, 0x6d, 0xf9, 0xdb, 0xad, 0x67, 0x3c,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_000B_PrivateKey = ByteSpan(sTestCert_DAC_FFF2_8001_000B_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8001-000C-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8001_000C_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xe9, 0x30, 0x82, 0x01, 0x8f, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x7c, 0xa3, 0x8e, 0x50, 0x7a, 0xb3,
    0x98, 0x7f, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30,
    0x30, 0x31, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f,
    0x39, 0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b,
    0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41,
    0x43, 0x20, 0x30, 0x30, 0x30, 0x43, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02,
    0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c,
    0x02, 0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x31, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01,
    0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x77, 0xc8, 0x6b, 0x3d, 0x4d, 0x5d, 0xd4,
    0xd7, 0xdf, 0x8b, 0x0b, 0x99, 0x83, 0x30, 0xaf, 0xf1, 0x2f, 0x10, 0xf3, 0xdf, 0x91, 0x60, 0x22, 0xee, 0x88, 0xa5, 0x82, 0x51,
    0x52, 0xf9, 0x9f, 0x85, 0x2a, 0x31, 0x0d, 0x5f, 0xab, 0x80, 0xa2, 0xb0, 0xe6, 0x75, 0x38, 0x31, 0xb5, 0xe3, 0xda, 0x2a, 0xad,
    0x6c, 0x18, 0x43, 0x4e, 0x5f, 0x65, 0xa0, 0x91, 0x16, 0x23, 0x75, 0x24, 0x40, 0xdb, 0x7a, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c,
    0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01,
    0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xdb, 0xa9, 0xab,
    0x53, 0x5a, 0xdc, 0x0b, 0x70, 0x3b, 0x2b, 0x6f, 0x14, 0x84, 0x9c, 0xd5, 0xce, 0x88, 0x18, 0xa6, 0x6a, 0x30, 0x1f, 0x06, 0x03,
    0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xd0, 0x5a, 0x9b, 0x67, 0x71, 0xa1, 0xa3, 0x7a, 0x9b, 0xc1, 0x9d, 0x95,
    0xe8, 0xb5, 0xdb, 0x8f, 0x43, 0x00, 0xb6, 0x3a, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03,
    0x48, 0x00, 0x30, 0x45, 0x02, 0x20, 0x67, 0x4c, 0x32, 0x06, 0xc7, 0xde, 0x79, 0x77, 0x2e, 0x4a, 0x6a, 0x9c, 0x1f, 0xdd, 0xa7,
    0x35, 0x16, 0x80, 0xca, 0x16, 0x06, 0x62, 0x6d, 0x15, 0x26, 0xae, 0xcd, 0x7d, 0x51, 0xc4, 0x21, 0x32, 0x02, 0x21, 0x00, 0xce,
    0xc5, 0x6e, 0xfb, 0x85, 0x62, 0xe6, 0xc7, 0x03, 0xcf, 0x5e, 0x61, 0x26, 0xda, 0x26, 0xaa, 0x53, 0x90, 0x17, 0xad, 0x4b, 0x9e,
    0xad, 0xb6, 0x40, 0x58, 0x62, 0x4b, 0xa6, 0xf4, 0x97, 0x48,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_000C_Cert = ByteSpan(sTestCert_DAC_FFF2_8001_000C_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8001_000C_SKID_Array[] = {
    0xDB, 0xA9, 0xAB, 0x53, 0x5A, 0xDC, 0x0B, 0x70, 0x3B, 0x2B, 0x6F, 0x14, 0x84, 0x9C, 0xD5, 0xCE, 0x88, 0x18, 0xA6, 0x6A,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_000C_SKID = ByteSpan(sTestCert_DAC_FFF2_8001_000C_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8001-000C-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8001_000C_PublicKey_Array[] = {
    0x04, 0x77, 0xc8, 0x6b, 0x3d, 0x4d, 0x5d, 0xd4, 0xd7, 0xdf, 0x8b, 0x0b, 0x99, 0x83, 0x30, 0xaf, 0xf1,
    0x2f, 0x10, 0xf3, 0xdf, 0x91, 0x60, 0x22, 0xee, 0x88, 0xa5, 0x82, 0x51, 0x52, 0xf9, 0x9f, 0x85, 0x2a,
    0x31, 0x0d, 0x5f, 0xab, 0x80, 0xa2, 0xb0, 0xe6, 0x75, 0x38, 0x31, 0xb5, 0xe3, 0xda, 0x2a, 0xad, 0x6c,
    0x18, 0x43, 0x4e, 0x5f, 0x65, 0xa0, 0x91, 0x16, 0x23, 0x75, 0x24, 0x40, 0xdb, 0x7a,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_000C_PublicKey = ByteSpan(sTestCert_DAC_FFF2_8001_000C_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8001_000C_PrivateKey_Array[] = {
    0x85, 0x39, 0x19, 0x2d, 0x1f, 0x36, 0x1f, 0x06, 0x82, 0x8e, 0xfa, 0x3d, 0xd7, 0xd9, 0x0f, 0xb4,
    0xff, 0xc5, 0xf8, 0xae, 0x74, 0x92, 0x18, 0x2b, 0x91, 0x70, 0xf5, 0xf6, 0xb8, 0x89, 0xa2, 0x91,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_000C_PrivateKey = ByteSpan(sTestCert_DAC_FFF2_8001_000C_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8001-000D-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8001_000D_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xe9, 0x30, 0x82, 0x01, 0x8f, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x48, 0x1d, 0x07, 0xab, 0xae, 0x70,
    0x5e, 0xf8, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30,
    0x30, 0x31, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f,
    0x39, 0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b,
    0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41,
    0x43, 0x20, 0x30, 0x30, 0x30, 0x44, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02,
    0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c,
    0x02, 0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x31, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01,
    0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xb5, 0x7d, 0xf1, 0xa9, 0x14, 0x34, 0x5c,
    0xf8, 0x47, 0x20, 0x28, 0x62, 0xd2, 0xcd, 0x7b, 0x35, 0xda, 0xb4, 0x25, 0x99, 0x76, 0xc1, 0x7c, 0x3a, 0xdf, 0xc4, 0xf5, 0xbb,
    0xd4, 0x3e, 0xd4, 0xc7, 0x78, 0x97, 0x82, 0x77, 0xec, 0x0c, 0xde, 0xeb, 0x4c, 0x1c, 0xac, 0x43, 0xa4, 0x9f, 0x60, 0x68, 0xf4,
    0xd4, 0xb9, 0xd3, 0xaa, 0x08, 0x36, 0x77, 0x8b, 0x2c, 0xb2, 0xee, 0x55, 0x35, 0xc7, 0x82, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c,
    0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01,
    0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x6c, 0xf4, 0x38,
    0x5a, 0x9d, 0x75, 0xdf, 0xb7, 0x4d, 0x8f, 0xfa, 0xb3, 0x9d, 0xb6, 0xff, 0x90, 0xe8, 0x90, 0x21, 0x12, 0x30, 0x1f, 0x06, 0x03,
    0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xd0, 0x5a, 0x9b, 0x67, 0x71, 0xa1, 0xa3, 0x7a, 0x9b, 0xc1, 0x9d, 0x95,
    0xe8, 0xb5, 0xdb, 0x8f, 0x43, 0x00, 0xb6, 0x3a, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03,
    0x48, 0x00, 0x30, 0x45, 0x02, 0x20, 0x6a, 0xd8, 0x59, 0xd1, 0xa5, 0x3d, 0x75, 0x44, 0x16, 0x2e, 0xc1, 0x01, 0x3b, 0x93, 0xa0,
    0x03, 0x5d, 0xa5, 0xa9, 0x10, 0xc8, 0x7b, 0x3c, 0xd9, 0x6b, 0x82, 0x86, 0xca, 0xcf, 0x8a, 0x17, 0x2e, 0x02, 0x21, 0x00, 0xda,
    0x6d, 0xbe, 0xd8, 0xe1, 0xef, 0x46, 0xcd, 0x82, 0x5e, 0xd2, 0xab, 0xfd, 0x2e, 0xaf, 0x98, 0xf3, 0x8e, 0x09, 0x54, 0x32, 0x07,
    0x2c, 0x8b, 0x29, 0x1b, 0xcd, 0xe8, 0xa4, 0x01, 0x4d, 0xe4,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_000D_Cert = ByteSpan(sTestCert_DAC_FFF2_8001_000D_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8001_000D_SKID_Array[] = {
    0x6C, 0xF4, 0x38, 0x5A, 0x9D, 0x75, 0xDF, 0xB7, 0x4D, 0x8F, 0xFA, 0xB3, 0x9D, 0xB6, 0xFF, 0x90, 0xE8, 0x90, 0x21, 0x12,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_000D_SKID = ByteSpan(sTestCert_DAC_FFF2_8001_000D_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8001-000D-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8001_000D_PublicKey_Array[] = {
    0x04, 0xb5, 0x7d, 0xf1, 0xa9, 0x14, 0x34, 0x5c, 0xf8, 0x47, 0x20, 0x28, 0x62, 0xd2, 0xcd, 0x7b, 0x35,
    0xda, 0xb4, 0x25, 0x99, 0x76, 0xc1, 0x7c, 0x3a, 0xdf, 0xc4, 0xf5, 0xbb, 0xd4, 0x3e, 0xd4, 0xc7, 0x78,
    0x97, 0x82, 0x77, 0xec, 0x0c, 0xde, 0xeb, 0x4c, 0x1c, 0xac, 0x43, 0xa4, 0x9f, 0x60, 0x68, 0xf4, 0xd4,
    0xb9, 0xd3, 0xaa, 0x08, 0x36, 0x77, 0x8b, 0x2c, 0xb2, 0xee, 0x55, 0x35, 0xc7, 0x82,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_000D_PublicKey = ByteSpan(sTestCert_DAC_FFF2_8001_000D_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8001_000D_PrivateKey_Array[] = {
    0x57, 0x02, 0x9f, 0xf9, 0xee, 0x08, 0xfa, 0x2e, 0xe2, 0x1b, 0x8c, 0xe1, 0xde, 0x9a, 0x12, 0x26,
    0x20, 0x4c, 0x87, 0x20, 0xb4, 0xd2, 0x55, 0x27, 0xdf, 0x7c, 0x66, 0x27, 0xc1, 0x00, 0x4c, 0x35,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_000D_PrivateKey = ByteSpan(sTestCert_DAC_FFF2_8001_000D_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8001-000E-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8001_000E_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xea, 0x30, 0x82, 0x01, 0x8f, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x66, 0xb4, 0x2e, 0x55, 0xe3, 0x14,
    0x34, 0xdb, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30,
    0x30, 0x31, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f,
    0x39, 0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b,
    0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41,
    0x43, 0x20, 0x30, 0x30, 0x30, 0x45, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02,
    0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c,
    0x02, 0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x31, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01,
    0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xf1, 0x19, 0xf9, 0xf2, 0x9e, 0xad, 0xe0,
    0xeb, 0xcc, 0xc2, 0x05, 0x54, 0xe8, 0xd4, 0x53, 0x7e, 0x12, 0xdb, 0xe9, 0x73, 0xaf, 0xbc, 0xf2, 0xba, 0xed, 0xb6, 0xde, 0xc1,
    0x15, 0x62, 0xc1, 0xda, 0x50, 0xbe, 0x34, 0x37, 0x8d, 0xe5, 0x04, 0x41, 0x4e, 0x30, 0x21, 0x24, 0x79, 0x23, 0xe9, 0x73, 0xfa,
    0xd4, 0x55, 0x28, 0x5b, 0xcd, 0xce, 0x55, 0x59, 0x9f, 0x29, 0x57, 0x6f, 0xb2, 0x9a, 0xaf, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c,
    0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01,
    0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x40, 0x9a, 0x35,
    0xf2, 0x03, 0x52, 0x38, 0x1d, 0xb2, 0xe9, 0x2e, 0x40, 0xfd, 0xd2, 0x5b, 0x6c, 0x03, 0x3d, 0x5a, 0xce, 0x30, 0x1f, 0x06, 0x03,
    0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xd0, 0x5a, 0x9b, 0x67, 0x71, 0xa1, 0xa3, 0x7a, 0x9b, 0xc1, 0x9d, 0x95,
    0xe8, 0xb5, 0xdb, 0x8f, 0x43, 0x00, 0xb6, 0x3a, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03,
    0x49, 0x00, 0x30, 0x46, 0x02, 0x21, 0x00, 0x83, 0x9f, 0x0a, 0x41, 0x8d, 0x5d, 0x48, 0x22, 0x4a, 0x29, 0x96, 0xda, 0xb4, 0x9b,
    0xcc, 0xc4, 0x43, 0x41, 0xb5, 0xf8, 0x2c, 0xb3, 0xd4, 0x86, 0xe7, 0x76, 0xba, 0xca, 0x6a, 0x3c, 0x03, 0x6c, 0x02, 0x21, 0x00,
    0xd1, 0x2b, 0x44, 0xa2, 0xe6, 0xf4, 0xac, 0xb9, 0x9b, 0x33, 0x43, 0x07, 0xde, 0x5f, 0x12, 0x93, 0x2b, 0x11, 0xc6, 0xd2, 0x7a,
    0x08, 0x56, 0x79, 0xa1, 0x67, 0x03, 0x25, 0x5a, 0xd1, 0xc7, 0xbf,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_000E_Cert = ByteSpan(sTestCert_DAC_FFF2_8001_000E_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8001_000E_SKID_Array[] = {
    0x40, 0x9A, 0x35, 0xF2, 0x03, 0x52, 0x38, 0x1D, 0xB2, 0xE9, 0x2E, 0x40, 0xFD, 0xD2, 0x5B, 0x6C, 0x03, 0x3D, 0x5A, 0xCE,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_000E_SKID = ByteSpan(sTestCert_DAC_FFF2_8001_000E_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8001-000E-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8001_000E_PublicKey_Array[] = {
    0x04, 0xf1, 0x19, 0xf9, 0xf2, 0x9e, 0xad, 0xe0, 0xeb, 0xcc, 0xc2, 0x05, 0x54, 0xe8, 0xd4, 0x53, 0x7e,
    0x12, 0xdb, 0xe9, 0x73, 0xaf, 0xbc, 0xf2, 0xba, 0xed, 0xb6, 0xde, 0xc1, 0x15, 0x62, 0xc1, 0xda, 0x50,
    0xbe, 0x34, 0x37, 0x8d, 0xe5, 0x04, 0x41, 0x4e, 0x30, 0x21, 0x24, 0x79, 0x23, 0xe9, 0x73, 0xfa, 0xd4,
    0x55, 0x28, 0x5b, 0xcd, 0xce, 0x55, 0x59, 0x9f, 0x29, 0x57, 0x6f, 0xb2, 0x9a, 0xaf,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_000E_PublicKey = ByteSpan(sTestCert_DAC_FFF2_8001_000E_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8001_000E_PrivateKey_Array[] = {
    0xd8, 0x5c, 0xb5, 0x46, 0x4d, 0x83, 0xa5, 0xd2, 0x10, 0x9f, 0xd8, 0x7e, 0x8e, 0x7f, 0xd3, 0xff,
    0x1b, 0x72, 0x1b, 0x0d, 0xea, 0xf3, 0x23, 0x53, 0x5a, 0xa4, 0x72, 0x4a, 0x99, 0x9d, 0x39, 0xce,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_000E_PrivateKey = ByteSpan(sTestCert_DAC_FFF2_8001_000E_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8001-000F-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8001_000F_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xe8, 0x30, 0x82, 0x01, 0x8f, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x18, 0xeb, 0xf9, 0x61, 0x48, 0x3e,
    0xf9, 0xda, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30,
    0x30, 0x31, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f,
    0x39, 0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b,
    0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41,
    0x43, 0x20, 0x30, 0x30, 0x30, 0x46, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02,
    0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c,
    0x02, 0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x31, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01,
    0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x80, 0x0e, 0x36, 0x99, 0xa8, 0x44, 0x08,
    0x08, 0xa3, 0xdb, 0x59, 0x64, 0x46, 0x56, 0x29, 0x10, 0xc5, 0x88, 0x15, 0x96, 0x31, 0x94, 0x93, 0x9a, 0xb9, 0xe0, 0x00, 0x37,
    0x6a, 0x6a, 0xb9, 0x55, 0x1e, 0x5c, 0x3b, 0xbd, 0x1b, 0x68, 0xf1, 0x31, 0x9e, 0xa6, 0x62, 0xba, 0x4c, 0xf1, 0x0b, 0xd0, 0x14,
    0x11, 0xd9, 0xda, 0xda, 0x1c, 0xe4, 0x61, 0xbe, 0x61, 0x22, 0xa0, 0x6c, 0x0a, 0x8c, 0xc6, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c,
    0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01,
    0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xa3, 0xd5, 0x0f,
    0x46, 0x6b, 0x03, 0xf0, 0x6b, 0xf6, 0x9c, 0x41, 0x48, 0x9e, 0x4b, 0x2b, 0xb4, 0x38, 0x0f, 0xfc, 0x10, 0x30, 0x1f, 0x06, 0x03,
    0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xd0, 0x5a, 0x9b, 0x67, 0x71, 0xa1, 0xa3, 0x7a, 0x9b, 0xc1, 0x9d, 0x95,
    0xe8, 0xb5, 0xdb, 0x8f, 0x43, 0x00, 0xb6, 0x3a, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03,
    0x47, 0x00, 0x30, 0x44, 0x02, 0x20, 0x57, 0xa1, 0xca, 0x58, 0xd8, 0x72, 0xe3, 0x12, 0x1c, 0xfc, 0x69, 0x5e, 0x64, 0x47, 0xf1,
    0x80, 0xf5, 0x4f, 0x44, 0x12, 0x5f, 0xc3, 0xf7, 0x1a, 0x7d, 0x7a, 0x5c, 0xcc, 0xac, 0x4d, 0x12, 0xf1, 0x02, 0x20, 0x41, 0x4a,
    0xa1, 0x89, 0x8f, 0x7c, 0xfd, 0x1c, 0xa7, 0xa6, 0xd2, 0xe5, 0x4e, 0xde, 0xb4, 0x38, 0xdd, 0x13, 0x53, 0x1a, 0x2b, 0xb2, 0x69,
    0x85, 0xdb, 0x4f, 0x3a, 0x9a, 0xb4, 0x38, 0x03, 0xeb,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_000F_Cert = ByteSpan(sTestCert_DAC_FFF2_8001_000F_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8001_000F_SKID_Array[] = {
    0xA3, 0xD5, 0x0F, 0x46, 0x6B, 0x03, 0xF0, 0x6B, 0xF6, 0x9C, 0x41, 0x48, 0x9E, 0x4B, 0x2B, 0xB4, 0x38, 0x0F, 0xFC, 0x10,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_000F_SKID = ByteSpan(sTestCert_DAC_FFF2_8001_000F_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8001-000F-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8001_000F_PublicKey_Array[] = {
    0x04, 0x80, 0x0e, 0x36, 0x99, 0xa8, 0x44, 0x08, 0x08, 0xa3, 0xdb, 0x59, 0x64, 0x46, 0x56, 0x29, 0x10,
    0xc5, 0x88, 0x15, 0x96, 0x31, 0x94, 0x93, 0x9a, 0xb9, 0xe0, 0x00, 0x37, 0x6a, 0x6a, 0xb9, 0x55, 0x1e,
    0x5c, 0x3b, 0xbd, 0x1b, 0x68, 0xf1, 0x31, 0x9e, 0xa6, 0x62, 0xba, 0x4c, 0xf1, 0x0b, 0xd0, 0x14, 0x11,
    0xd9, 0xda, 0xda, 0x1c, 0xe4, 0x61, 0xbe, 0x61, 0x22, 0xa0, 0x6c, 0x0a, 0x8c, 0xc6,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_000F_PublicKey = ByteSpan(sTestCert_DAC_FFF2_8001_000F_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8001_000F_PrivateKey_Array[] = {
    0x1c, 0x15, 0x18, 0x4f, 0x51, 0xc8, 0x24, 0x10, 0x03, 0x3f, 0xb0, 0x85, 0x33, 0xfc, 0xbd, 0xe3,
    0x5b, 0xc2, 0x5a, 0x2b, 0x1d, 0x16, 0x8b, 0x92, 0xce, 0x33, 0x97, 0xf4, 0x6f, 0xc9, 0xe9, 0x9c,
};

extern const ByteSpan sTestCert_DAC_FFF2_8001_000F_PrivateKey = ByteSpan(sTestCert_DAC_FFF2_8001_000F_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8002-0010-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8002_0010_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xd4, 0x30, 0x82, 0x01, 0x79, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x52, 0x1a, 0xa9, 0xf7, 0xed, 0xbb,
    0xaa, 0xac, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x30, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x32, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x39,
    0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41, 0x43,
    0x20, 0x30, 0x30, 0x31, 0x30, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01,
    0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02,
    0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x32, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06,
    0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x03, 0x6a, 0xf4, 0x53, 0x44, 0xcd, 0xb7, 0x03,
    0x47, 0xb7, 0x8e, 0x8c, 0xa4, 0x19, 0x15, 0x46, 0x61, 0xb3, 0x37, 0x10, 0x2d, 0x9e, 0x74, 0xfc, 0xe2, 0xf9, 0xee, 0x29, 0xe7,
    0x71, 0x44, 0x6c, 0x57, 0x9b, 0xb7, 0x84, 0xa0, 0x2f, 0x7a, 0x77, 0x1c, 0x5e, 0xa2, 0x52, 0x2d, 0xa9, 0xa4, 0x37, 0x55, 0xd5,
    0xbe, 0x86, 0x4c, 0xa9, 0x55, 0x6e, 0x6e, 0xaf, 0x0b, 0xcd, 0xef, 0x2e, 0x36, 0x3c, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c, 0x06,
    0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff,
    0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x34, 0x5f, 0x0c, 0xe9,
    0xf7, 0x06, 0x5c, 0x77, 0x82, 0xf8, 0xdf, 0x87, 0xb0, 0x12, 0x64, 0x36, 0x1f, 0xed, 0x9b, 0xd8, 0x30, 0x1f, 0x06, 0x03, 0x55,
    0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x61, 0x3d, 0xd0, 0x87, 0x35, 0x5e, 0xf0, 0x8b, 0xae, 0x01, 0xe4, 0xc6, 0x9a,
    0x8f, 0xc7, 0x3d, 0xac, 0x8c, 0x7d, 0xfd, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x49,
    0x00, 0x30, 0x46, 0x02, 0x21, 0x00, 0xd4, 0x57, 0xbc, 0xe3, 0xae, 0xb3, 0xa9, 0xb0, 0x51, 0xde, 0xf5, 0x1d, 0x1f, 0x3b, 0x30,
    0x90, 0x2c, 0xb5, 0xd4, 0x9f, 0x9d, 0xb1, 0x41, 0x89, 0x0d, 0x4b, 0x3c, 0xdd, 0xfc, 0xb9, 0xd8, 0xa1, 0x02, 0x21, 0x00, 0xc0,
    0x9f, 0xad, 0x98, 0xfe, 0x95, 0x74, 0x90, 0xb8, 0x17, 0x71, 0xb6, 0x3d, 0xd5, 0x66, 0x29, 0x67, 0x57, 0xdc, 0x86, 0x9a, 0x78,
    0x44, 0xbc, 0xd8, 0x97, 0x05, 0xb3, 0xb8, 0x34, 0xb5, 0x02,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0010_Cert = ByteSpan(sTestCert_DAC_FFF2_8002_0010_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8002_0010_SKID_Array[] = {
    0x34, 0x5F, 0x0C, 0xE9, 0xF7, 0x06, 0x5C, 0x77, 0x82, 0xF8, 0xDF, 0x87, 0xB0, 0x12, 0x64, 0x36, 0x1F, 0xED, 0x9B, 0xD8,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0010_SKID = ByteSpan(sTestCert_DAC_FFF2_8002_0010_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8002-0010-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8002_0010_PublicKey_Array[] = {
    0x04, 0x03, 0x6a, 0xf4, 0x53, 0x44, 0xcd, 0xb7, 0x03, 0x47, 0xb7, 0x8e, 0x8c, 0xa4, 0x19, 0x15, 0x46,
    0x61, 0xb3, 0x37, 0x10, 0x2d, 0x9e, 0x74, 0xfc, 0xe2, 0xf9, 0xee, 0x29, 0xe7, 0x71, 0x44, 0x6c, 0x57,
    0x9b, 0xb7, 0x84, 0xa0, 0x2f, 0x7a, 0x77, 0x1c, 0x5e, 0xa2, 0x52, 0x2d, 0xa9, 0xa4, 0x37, 0x55, 0xd5,
    0xbe, 0x86, 0x4c, 0xa9, 0x55, 0x6e, 0x6e, 0xaf, 0x0b, 0xcd, 0xef, 0x2e, 0x36, 0x3c,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0010_PublicKey = ByteSpan(sTestCert_DAC_FFF2_8002_0010_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8002_0010_PrivateKey_Array[] = {
    0xff, 0x00, 0x86, 0x1d, 0x2d, 0x9e, 0x6e, 0x5d, 0xba, 0xae, 0xaa, 0xbb, 0x32, 0xec, 0x35, 0x11,
    0xc4, 0xc4, 0x60, 0xda, 0x2a, 0x08, 0x61, 0x68, 0xef, 0x4a, 0x4b, 0x9d, 0x50, 0x7c, 0x6a, 0x99,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0010_PrivateKey = ByteSpan(sTestCert_DAC_FFF2_8002_0010_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8002-0011-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8002_0011_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xd3, 0x30, 0x82, 0x01, 0x79, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x35, 0xe2, 0xe4, 0xc9, 0x50, 0x63,
    0x55, 0x85, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x30, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x32, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x39,
    0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41, 0x43,
    0x20, 0x30, 0x30, 0x31, 0x31, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01,
    0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02,
    0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x32, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06,
    0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xde, 0x24, 0xc8, 0xd0, 0x5c, 0x82, 0xb2, 0x56,
    0xde, 0x06, 0xd5, 0x4c, 0x7d, 0x8f, 0x6e, 0x42, 0x68, 0x01, 0x39, 0x84, 0x38, 0x09, 0xf5, 0x53, 0x04, 0x4c, 0xfc, 0xd9, 0xd0,
    0xb3, 0x70, 0x54, 0x6b, 0xb0, 0xd9, 0x04, 0x67, 0xb8, 0xaa, 0xa0, 0x90, 0xd2, 0x1c, 0x29, 0x55, 0x78, 0x97, 0x70, 0x41, 0xe9,
    0x43, 0xb2, 0xd1, 0x74, 0xe0, 0x54, 0x5c, 0x9e, 0x0d, 0x20, 0xc9, 0x49, 0xff, 0x47, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c, 0x06,
    0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff,
    0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xd6, 0x1c, 0x56, 0x66,
    0x02, 0xce, 0xad, 0xf9, 0x2b, 0xdf, 0x48, 0x12, 0xe3, 0x91, 0xac, 0x10, 0x6f, 0x2e, 0x1e, 0xc9, 0x30, 0x1f, 0x06, 0x03, 0x55,
    0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x61, 0x3d, 0xd0, 0x87, 0x35, 0x5e, 0xf0, 0x8b, 0xae, 0x01, 0xe4, 0xc6, 0x9a,
    0x8f, 0xc7, 0x3d, 0xac, 0x8c, 0x7d, 0xfd, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x48,
    0x00, 0x30, 0x45, 0x02, 0x20, 0x76, 0x7d, 0x4f, 0x2a, 0x8f, 0xa1, 0xc4, 0x24, 0x7e, 0x80, 0x38, 0x81, 0xe5, 0x41, 0x22, 0x93,
    0x25, 0x67, 0x6b, 0x11, 0xe7, 0x8e, 0xd8, 0x50, 0xbe, 0x00, 0x1f, 0x90, 0x0d, 0x4f, 0x27, 0xfb, 0x02, 0x21, 0x00, 0x80, 0x5a,
    0x32, 0xfc, 0xb0, 0xe4, 0xf9, 0x07, 0xf1, 0x03, 0x9b, 0x46, 0x46, 0x37, 0xf0, 0xa7, 0x85, 0x28, 0xb1, 0x51, 0xb1, 0x98, 0xcb,
    0x0e, 0x97, 0x0e, 0xbb, 0xe0, 0x88, 0x2a, 0xb9, 0xe3,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0011_Cert = ByteSpan(sTestCert_DAC_FFF2_8002_0011_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8002_0011_SKID_Array[] = {
    0xD6, 0x1C, 0x56, 0x66, 0x02, 0xCE, 0xAD, 0xF9, 0x2B, 0xDF, 0x48, 0x12, 0xE3, 0x91, 0xAC, 0x10, 0x6F, 0x2E, 0x1E, 0xC9,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0011_SKID = ByteSpan(sTestCert_DAC_FFF2_8002_0011_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8002-0011-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8002_0011_PublicKey_Array[] = {
    0x04, 0xde, 0x24, 0xc8, 0xd0, 0x5c, 0x82, 0xb2, 0x56, 0xde, 0x06, 0xd5, 0x4c, 0x7d, 0x8f, 0x6e, 0x42,
    0x68, 0x01, 0x39, 0x84, 0x38, 0x09, 0xf5, 0x53, 0x04, 0x4c, 0xfc, 0xd9, 0xd0, 0xb3, 0x70, 0x54, 0x6b,
    0xb0, 0xd9, 0x04, 0x67, 0xb8, 0xaa, 0xa0, 0x90, 0xd2, 0x1c, 0x29, 0x55, 0x78, 0x97, 0x70, 0x41, 0xe9,
    0x43, 0xb2, 0xd1, 0x74, 0xe0, 0x54, 0x5c, 0x9e, 0x0d, 0x20, 0xc9, 0x49, 0xff, 0x47,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0011_PublicKey = ByteSpan(sTestCert_DAC_FFF2_8002_0011_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8002_0011_PrivateKey_Array[] = {
    0x7d, 0xfd, 0x85, 0x68, 0x97, 0x5b, 0x3f, 0x22, 0x1b, 0x1d, 0x19, 0x6d, 0x49, 0x41, 0xd9, 0xd0,
    0x06, 0xa4, 0xa0, 0x78, 0x55, 0xce, 0x9d, 0xce, 0x79, 0x43, 0x0e, 0xdc, 0xb9, 0x32, 0x2e, 0xa2,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0011_PrivateKey = ByteSpan(sTestCert_DAC_FFF2_8002_0011_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8002-0012-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8002_0012_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xd4, 0x30, 0x82, 0x01, 0x79, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x14, 0xc0, 0xab, 0xb2, 0xae, 0x41,
    0xc1, 0x3e, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x30, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x32, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x39,
    0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41, 0x43,
    0x20, 0x30, 0x30, 0x31, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01,
    0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02,
    0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x32, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06,
    0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xb9, 0xca, 0x7b, 0xd8, 0x5a, 0x31, 0x20, 0x93,
    0xcc, 0x5d, 0xe8, 0x15, 0xc1, 0x54, 0x99, 0x3b, 0x04, 0xf8, 0x73, 0xc8, 0x2e, 0x50, 0xff, 0xc9, 0x1d, 0x4c, 0x35, 0x34, 0xf7,
    0x23, 0x16, 0x7b, 0x3a, 0x3f, 0xca, 0xe8, 0xf9, 0x33, 0x15, 0xb0, 0xc2, 0x81, 0x5b, 0x64, 0x9b, 0x52, 0xdb, 0xae, 0x3d, 0xd7,
    0xca, 0x54, 0xd1, 0xf2, 0xab, 0x1a, 0xc4, 0xed, 0xbe, 0x22, 0x17, 0x62, 0xa4, 0xd2, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c, 0x06,
    0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff,
    0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xe1, 0xa1, 0xc4, 0xd1,
    0x41, 0xca, 0x84, 0x41, 0x15, 0xf4, 0x2a, 0xf8, 0xe3, 0x23, 0xe8, 0xd4, 0x6a, 0xda, 0xc7, 0x6d, 0x30, 0x1f, 0x06, 0x03, 0x55,
    0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x61, 0x3d, 0xd0, 0x87, 0x35, 0x5e, 0xf0, 0x8b, 0xae, 0x01, 0xe4, 0xc6, 0x9a,
    0x8f, 0xc7, 0x3d, 0xac, 0x8c, 0x7d, 0xfd, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x49,
    0x00, 0x30, 0x46, 0x02, 0x21, 0x00, 0xab, 0x57, 0x78, 0x83, 0xf3, 0xd4, 0x6c, 0xda, 0xad, 0x8a, 0xef, 0xa3, 0x5c, 0x8d, 0x65,
    0xc1, 0xb8, 0xc3, 0x7f, 0x16, 0xe1, 0x9b, 0xf3, 0xdb, 0xf9, 0x44, 0x02, 0x3a, 0x45, 0xf6, 0x5e, 0x92, 0x02, 0x21, 0x00, 0x86,
    0x96, 0x80, 0xe5, 0x18, 0xc4, 0x06, 0x5c, 0x30, 0x7a, 0x7f, 0x13, 0x31, 0xd0, 0x65, 0xdc, 0x19, 0x3d, 0xb3, 0xcd, 0xf6, 0xaa,
    0x43, 0x5e, 0x70, 0x87, 0xa0, 0xcc, 0x82, 0x63, 0x25, 0x2d,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0012_Cert = ByteSpan(sTestCert_DAC_FFF2_8002_0012_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8002_0012_SKID_Array[] = {
    0xE1, 0xA1, 0xC4, 0xD1, 0x41, 0xCA, 0x84, 0x41, 0x15, 0xF4, 0x2A, 0xF8, 0xE3, 0x23, 0xE8, 0xD4, 0x6A, 0xDA, 0xC7, 0x6D,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0012_SKID = ByteSpan(sTestCert_DAC_FFF2_8002_0012_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8002-0012-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8002_0012_PublicKey_Array[] = {
    0x04, 0xb9, 0xca, 0x7b, 0xd8, 0x5a, 0x31, 0x20, 0x93, 0xcc, 0x5d, 0xe8, 0x15, 0xc1, 0x54, 0x99, 0x3b,
    0x04, 0xf8, 0x73, 0xc8, 0x2e, 0x50, 0xff, 0xc9, 0x1d, 0x4c, 0x35, 0x34, 0xf7, 0x23, 0x16, 0x7b, 0x3a,
    0x3f, 0xca, 0xe8, 0xf9, 0x33, 0x15, 0xb0, 0xc2, 0x81, 0x5b, 0x64, 0x9b, 0x52, 0xdb, 0xae, 0x3d, 0xd7,
    0xca, 0x54, 0xd1, 0xf2, 0xab, 0x1a, 0xc4, 0xed, 0xbe, 0x22, 0x17, 0x62, 0xa4, 0xd2,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0012_PublicKey = ByteSpan(sTestCert_DAC_FFF2_8002_0012_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8002_0012_PrivateKey_Array[] = {
    0x8b, 0x6c, 0xb3, 0x96, 0xcf, 0x5b, 0xf5, 0x18, 0x1f, 0x5c, 0x49, 0xfa, 0xd1, 0x64, 0x3a, 0xae,
    0xac, 0xdf, 0x79, 0xdf, 0x9a, 0x65, 0x3e, 0xd2, 0xe4, 0x22, 0x89, 0x7c, 0xd0, 0x8a, 0xbc, 0x93,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0012_PrivateKey = ByteSpan(sTestCert_DAC_FFF2_8002_0012_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8002-0013-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8002_0013_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xd4, 0x30, 0x82, 0x01, 0x79, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x24, 0x27, 0x9d, 0x72, 0x28, 0x41,
    0x29, 0xb5, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x30, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x32, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x39,
    0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41, 0x43,
    0x20, 0x30, 0x30, 0x31, 0x33, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01,
    0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02,
    0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x32, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06,
    0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x6c, 0xe7, 0x6b, 0x7b, 0x8c, 0x70, 0xdc, 0x9d,
    0x2f, 0x51, 0x07, 0xad, 0x31, 0xf2, 0xf6, 0x2b, 0x99, 0xa3, 0x1b, 0x4f, 0xc6, 0xa9, 0x01, 0xcd, 0x6a, 0x0d, 0x20, 0x5e, 0x5f,
    0xcc, 0x67, 0x9a, 0xbf, 0x0a, 0x78, 0x30, 0x36, 0x6c, 0x18, 0x2a, 0xeb, 0x79, 0x35, 0xe1, 0x42, 0x4a, 0xf2, 0x46, 0x04, 0x0b,
    0xfd, 0x45, 0x02, 0x03, 0x87, 0xc3, 0x0e, 0xc5, 0xbe, 0x2d, 0xc9, 0x57, 0xb3, 0x08, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c, 0x06,
    0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff,
    0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x77, 0xf2, 0x48, 0xde,
    0x17, 0x8c, 0x07, 0xd1, 0xe4, 0xbc, 0x13, 0x62, 0x9d, 0xc6, 0x6d, 0xea, 0xb2, 0xf7, 0xac, 0x06, 0x30, 0x1f, 0x06, 0x03, 0x55,
    0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x61, 0x3d, 0xd0, 0x87, 0x35, 0x5e, 0xf0, 0x8b, 0xae, 0x01, 0xe4, 0xc6, 0x9a,
    0x8f, 0xc7, 0x3d, 0xac, 0x8c, 0x7d, 0xfd, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x49,
    0x00, 0x30, 0x46, 0x02, 0x21, 0x00, 0xdd, 0x75, 0xf1, 0x2e, 0x2e, 0x40, 0x0b, 0x58, 0xcf, 0xe9, 0x34, 0x5b, 0x4d, 0x2c, 0x02,
    0x3f, 0xac, 0x66, 0x8c, 0x64, 0x03, 0x0a, 0x13, 0x37, 0x6b, 0x72, 0x4f, 0x88, 0xf3, 0xc4, 0x0d, 0xa7, 0x02, 0x21, 0x00, 0xf5,
    0x99, 0xab, 0x20, 0xd4, 0x8c, 0xd2, 0x73, 0xf2, 0x3f, 0xba, 0xc7, 0x30, 0x37, 0xa9, 0x1d, 0xb8, 0x73, 0xb5, 0xcd, 0x3f, 0x59,
    0xb9, 0x58, 0xd4, 0xbe, 0xf2, 0x68, 0x7a, 0x46, 0x97, 0x34,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0013_Cert = ByteSpan(sTestCert_DAC_FFF2_8002_0013_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8002_0013_SKID_Array[] = {
    0x77, 0xF2, 0x48, 0xDE, 0x17, 0x8C, 0x07, 0xD1, 0xE4, 0xBC, 0x13, 0x62, 0x9D, 0xC6, 0x6D, 0xEA, 0xB2, 0xF7, 0xAC, 0x06,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0013_SKID = ByteSpan(sTestCert_DAC_FFF2_8002_0013_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8002-0013-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8002_0013_PublicKey_Array[] = {
    0x04, 0x6c, 0xe7, 0x6b, 0x7b, 0x8c, 0x70, 0xdc, 0x9d, 0x2f, 0x51, 0x07, 0xad, 0x31, 0xf2, 0xf6, 0x2b,
    0x99, 0xa3, 0x1b, 0x4f, 0xc6, 0xa9, 0x01, 0xcd, 0x6a, 0x0d, 0x20, 0x5e, 0x5f, 0xcc, 0x67, 0x9a, 0xbf,
    0x0a, 0x78, 0x30, 0x36, 0x6c, 0x18, 0x2a, 0xeb, 0x79, 0x35, 0xe1, 0x42, 0x4a, 0xf2, 0x46, 0x04, 0x0b,
    0xfd, 0x45, 0x02, 0x03, 0x87, 0xc3, 0x0e, 0xc5, 0xbe, 0x2d, 0xc9, 0x57, 0xb3, 0x08,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0013_PublicKey = ByteSpan(sTestCert_DAC_FFF2_8002_0013_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8002_0013_PrivateKey_Array[] = {
    0xd7, 0x31, 0x94, 0x4c, 0xa9, 0xa8, 0x07, 0xd1, 0x89, 0xf9, 0xaa, 0xd9, 0xde, 0x63, 0x72, 0x0e,
    0x72, 0x81, 0x84, 0xf9, 0x88, 0x23, 0xea, 0xec, 0x4c, 0x52, 0x3a, 0xa9, 0x58, 0x1a, 0xa7, 0x78,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0013_PrivateKey = ByteSpan(sTestCert_DAC_FFF2_8002_0013_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8002-0014-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8002_0014_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xd2, 0x30, 0x82, 0x01, 0x79, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x21, 0x04, 0x7b, 0x7a, 0xda, 0x0c,
    0x8c, 0x3d, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x30, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x32, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x39,
    0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41, 0x43,
    0x20, 0x30, 0x30, 0x31, 0x34, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01,
    0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02,
    0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x32, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06,
    0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xba, 0x88, 0x8f, 0xad, 0xeb, 0x72, 0x58, 0x3c,
    0x15, 0xa8, 0x9c, 0x65, 0x45, 0x41, 0x1a, 0xa6, 0x00, 0xb9, 0x37, 0x5f, 0x7e, 0x74, 0xb9, 0x4f, 0xf8, 0xa4, 0x13, 0xef, 0xde,
    0x9d, 0x5c, 0xe2, 0x78, 0x4b, 0x19, 0x5b, 0xc6, 0x6f, 0xc5, 0x4c, 0xc5, 0x2a, 0xef, 0x5b, 0xf4, 0xb9, 0x84, 0x6f, 0xad, 0x19,
    0x01, 0xef, 0x68, 0x98, 0x5d, 0xc9, 0x66, 0x3e, 0x12, 0x05, 0xe9, 0x9c, 0xed, 0xff, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c, 0x06,
    0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff,
    0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xd5, 0x5b, 0xe9, 0x31,
    0x0b, 0x65, 0x43, 0x18, 0x53, 0xcb, 0xf9, 0xc2, 0xdd, 0x49, 0x7a, 0x55, 0xf4, 0x9f, 0x5e, 0x2a, 0x30, 0x1f, 0x06, 0x03, 0x55,
    0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x61, 0x3d, 0xd0, 0x87, 0x35, 0x5e, 0xf0, 0x8b, 0xae, 0x01, 0xe4, 0xc6, 0x9a,
    0x8f, 0xc7, 0x3d, 0xac, 0x8c, 0x7d, 0xfd, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x47,
    0x00, 0x30, 0x44, 0x02, 0x20, 0x0c, 0xf0, 0xf9, 0xa0, 0x01, 0x97, 0x7b, 0x88, 0xe3, 0x2f, 0xfb, 0xab, 0x99, 0xc3, 0x76, 0xb8,
    0x73, 0x97, 0xe8, 0xf9, 0x64, 0x1e, 0x88, 0x7d, 0x23, 0xb4, 0xbf, 0x80, 0x69, 0xd1, 0x63, 0x94, 0x02, 0x20, 0x71, 0x33, 0xde,
    0x48, 0xdf, 0x2a, 0x74, 0x8f, 0xf3, 0xb9, 0xae, 0xc4, 0x2b, 0x25, 0x35, 0x8b, 0x60, 0x1e, 0xdd, 0xb3, 0x9a, 0x06, 0x3a, 0x82,
    0x14, 0xdc, 0xaf, 0x59, 0xf3, 0x6e, 0x06, 0x25,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0014_Cert = ByteSpan(sTestCert_DAC_FFF2_8002_0014_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8002_0014_SKID_Array[] = {
    0xD5, 0x5B, 0xE9, 0x31, 0x0B, 0x65, 0x43, 0x18, 0x53, 0xCB, 0xF9, 0xC2, 0xDD, 0x49, 0x7A, 0x55, 0xF4, 0x9F, 0x5E, 0x2A,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0014_SKID = ByteSpan(sTestCert_DAC_FFF2_8002_0014_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8002-0014-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8002_0014_PublicKey_Array[] = {
    0x04, 0xba, 0x88, 0x8f, 0xad, 0xeb, 0x72, 0x58, 0x3c, 0x15, 0xa8, 0x9c, 0x65, 0x45, 0x41, 0x1a, 0xa6,
    0x00, 0xb9, 0x37, 0x5f, 0x7e, 0x74, 0xb9, 0x4f, 0xf8, 0xa4, 0x13, 0xef, 0xde, 0x9d, 0x5c, 0xe2, 0x78,
    0x4b, 0x19, 0x5b, 0xc6, 0x6f, 0xc5, 0x4c, 0xc5, 0x2a, 0xef, 0x5b, 0xf4, 0xb9, 0x84, 0x6f, 0xad, 0x19,
    0x01, 0xef, 0x68, 0x98, 0x5d, 0xc9, 0x66, 0x3e, 0x12, 0x05, 0xe9, 0x9c, 0xed, 0xff,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0014_PublicKey = ByteSpan(sTestCert_DAC_FFF2_8002_0014_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8002_0014_PrivateKey_Array[] = {
    0x53, 0x7e, 0x8c, 0x7f, 0x71, 0xe8, 0xb1, 0x8b, 0x24, 0xe2, 0x28, 0x3d, 0x87, 0xe8, 0x1f, 0x8b,
    0x99, 0x7f, 0xf5, 0xbe, 0x35, 0xe7, 0x5f, 0x27, 0xe3, 0xde, 0x63, 0x1d, 0x22, 0x29, 0x51, 0x87,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0014_PrivateKey = ByteSpan(sTestCert_DAC_FFF2_8002_0014_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8002-0015-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8002_0015_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xd4, 0x30, 0x82, 0x01, 0x79, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x3a, 0x35, 0x65, 0xf3, 0x0a, 0x66,
    0x32, 0x4b, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x30, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x32, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x39,
    0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41, 0x43,
    0x20, 0x30, 0x30, 0x31, 0x35, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01,
    0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02,
    0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x32, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06,
    0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xdd, 0x4e, 0x6b, 0xe4, 0x15, 0xd5, 0x3f, 0x6a,
    0x38, 0x84, 0xd9, 0xfd, 0x68, 0xa1, 0xc4, 0x3c, 0x67, 0xa0, 0x48, 0x7b, 0xe9, 0x1b, 0x3a, 0xe3, 0xed, 0x61, 0x2b, 0x72, 0x92,
    0xe9, 0xea, 0x41, 0xd1, 0xbe, 0x20, 0x3d, 0x1f, 0x75, 0xbf, 0xe8, 0xb1, 0xa0, 0xbc, 0x2c, 0x02, 0xb9, 0xbb, 0x5f, 0x01, 0xfa,
    0x53, 0x53, 0x9a, 0x4a, 0x73, 0xde, 0x29, 0xe2, 0x11, 0xe9, 0x80, 0xa4, 0x98, 0x3c, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c, 0x06,
    0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff,
    0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xc1, 0xf4, 0x48, 0xf4,
    0x97, 0xba, 0x7f, 0xa9, 0xd6, 0x6e, 0xf2, 0x53, 0xca, 0xe7, 0xa4, 0xf1, 0xee, 0x35, 0xb3, 0x12, 0x30, 0x1f, 0x06, 0x03, 0x55,
    0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x61, 0x3d, 0xd0, 0x87, 0x35, 0x5e, 0xf0, 0x8b, 0xae, 0x01, 0xe4, 0xc6, 0x9a,
    0x8f, 0xc7, 0x3d, 0xac, 0x8c, 0x7d, 0xfd, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x49,
    0x00, 0x30, 0x46, 0x02, 0x21, 0x00, 0x93, 0x0d, 0x0b, 0xef, 0x78, 0x5d, 0xad, 0x46, 0x79, 0xca, 0x18, 0xee, 0x91, 0xc7, 0x37,
    0x86, 0x32, 0xb9, 0x36, 0xde, 0x75, 0x20, 0x45, 0x89, 0x9f, 0xa1, 0x1c, 0x96, 0x7d, 0x2b, 0xb1, 0x01, 0x02, 0x21, 0x00, 0x82,
    0xb0, 0x0f, 0x09, 0x78, 0x2c, 0x65, 0x5c, 0xfe, 0x57, 0x24, 0x94, 0xfb, 0x85, 0x09, 0xe4, 0x02, 0x77, 0x6e, 0xa2, 0x11, 0xb4,
    0x6d, 0x43, 0x50, 0x5b, 0xc8, 0x4e, 0x55, 0x16, 0x22, 0x90,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0015_Cert = ByteSpan(sTestCert_DAC_FFF2_8002_0015_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8002_0015_SKID_Array[] = {
    0xC1, 0xF4, 0x48, 0xF4, 0x97, 0xBA, 0x7F, 0xA9, 0xD6, 0x6E, 0xF2, 0x53, 0xCA, 0xE7, 0xA4, 0xF1, 0xEE, 0x35, 0xB3, 0x12,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0015_SKID = ByteSpan(sTestCert_DAC_FFF2_8002_0015_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8002-0015-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8002_0015_PublicKey_Array[] = {
    0x04, 0xdd, 0x4e, 0x6b, 0xe4, 0x15, 0xd5, 0x3f, 0x6a, 0x38, 0x84, 0xd9, 0xfd, 0x68, 0xa1, 0xc4, 0x3c,
    0x67, 0xa0, 0x48, 0x7b, 0xe9, 0x1b, 0x3a, 0xe3, 0xed, 0x61, 0x2b, 0x72, 0x92, 0xe9, 0xea, 0x41, 0xd1,
    0xbe, 0x20, 0x3d, 0x1f, 0x75, 0xbf, 0xe8, 0xb1, 0xa0, 0xbc, 0x2c, 0x02, 0xb9, 0xbb, 0x5f, 0x01, 0xfa,
    0x53, 0x53, 0x9a, 0x4a, 0x73, 0xde, 0x29, 0xe2, 0x11, 0xe9, 0x80, 0xa4, 0x98, 0x3c,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0015_PublicKey = ByteSpan(sTestCert_DAC_FFF2_8002_0015_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8002_0015_PrivateKey_Array[] = {
    0x34, 0x69, 0x91, 0x54, 0x61, 0x90, 0x48, 0xf0, 0xb4, 0x90, 0xee, 0xdb, 0x01, 0x14, 0x03, 0x8e,
    0x41, 0x42, 0xb7, 0xfa, 0xb8, 0xc3, 0x6e, 0xbf, 0xc9, 0x56, 0xff, 0x62, 0xa2, 0x2c, 0xe3, 0xbb,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0015_PrivateKey = ByteSpan(sTestCert_DAC_FFF2_8002_0015_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8002-0016-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8002_0016_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xd4, 0x30, 0x82, 0x01, 0x79, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x27, 0x0c, 0x80, 0xb4, 0x6c, 0xf9,
    0x52, 0x57, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x30, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x32, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x39,
    0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41, 0x43,
    0x20, 0x30, 0x30, 0x31, 0x36, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01,
    0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02,
    0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x32, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06,
    0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x14, 0xf6, 0x2c, 0x09, 0xec, 0x12, 0xbb, 0x78,
    0xa4, 0x8e, 0xfc, 0x68, 0x94, 0x67, 0xec, 0x29, 0x9e, 0xb7, 0x5d, 0x4c, 0x16, 0x48, 0x1f, 0x37, 0x6e, 0xbb, 0x50, 0x98, 0xed,
    0x19, 0x8b, 0x9b, 0xf9, 0x84, 0xf9, 0xf8, 0xa2, 0xf9, 0x31, 0xc4, 0x6d, 0x3b, 0xa2, 0xd2, 0xa5, 0xb0, 0x72, 0x9f, 0x62, 0x4e,
    0xee, 0xc6, 0x7d, 0xc2, 0xe2, 0x30, 0x91, 0xb8, 0x6a, 0x08, 0x6f, 0x1e, 0x1d, 0x8e, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c, 0x06,
    0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff,
    0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x71, 0x72, 0x0a, 0x62,
    0x72, 0x8f, 0xca, 0xb3, 0xef, 0xf0, 0x62, 0x85, 0xbc, 0x7c, 0x34, 0x55, 0x38, 0x01, 0x92, 0x12, 0x30, 0x1f, 0x06, 0x03, 0x55,
    0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x61, 0x3d, 0xd0, 0x87, 0x35, 0x5e, 0xf0, 0x8b, 0xae, 0x01, 0xe4, 0xc6, 0x9a,
    0x8f, 0xc7, 0x3d, 0xac, 0x8c, 0x7d, 0xfd, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x49,
    0x00, 0x30, 0x46, 0x02, 0x21, 0x00, 0xec, 0x72, 0x19, 0x98, 0x79, 0xd8, 0xdf, 0xc7, 0x43, 0x98, 0x8e, 0xf8, 0xa5, 0x86, 0x90,
    0x22, 0xb5, 0x88, 0xf9, 0xc0, 0x1b, 0x51, 0x86, 0xa6, 0xa2, 0xc5, 0x67, 0xdb, 0x4e, 0xb0, 0x4c, 0x3b, 0x02, 0x21, 0x00, 0xa2,
    0x1c, 0x5f, 0x93, 0x0a, 0xcd, 0x0b, 0x9a, 0xd5, 0xff, 0xcc, 0x1b, 0x69, 0xc0, 0xf9, 0x06, 0x75, 0x48, 0x1f, 0x7d, 0x07, 0xb1,
    0x7a, 0x7b, 0x5c, 0x74, 0x4d, 0xf2, 0xe1, 0x2c, 0x30, 0x85,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0016_Cert = ByteSpan(sTestCert_DAC_FFF2_8002_0016_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8002_0016_SKID_Array[] = {
    0x71, 0x72, 0x0A, 0x62, 0x72, 0x8F, 0xCA, 0xB3, 0xEF, 0xF0, 0x62, 0x85, 0xBC, 0x7C, 0x34, 0x55, 0x38, 0x01, 0x92, 0x12,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0016_SKID = ByteSpan(sTestCert_DAC_FFF2_8002_0016_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8002-0016-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8002_0016_PublicKey_Array[] = {
    0x04, 0x14, 0xf6, 0x2c, 0x09, 0xec, 0x12, 0xbb, 0x78, 0xa4, 0x8e, 0xfc, 0x68, 0x94, 0x67, 0xec, 0x29,
    0x9e, 0xb7, 0x5d, 0x4c, 0x16, 0x48, 0x1f, 0x37, 0x6e, 0xbb, 0x50, 0x98, 0xed, 0x19, 0x8b, 0x9b, 0xf9,
    0x84, 0xf9, 0xf8, 0xa2, 0xf9, 0x31, 0xc4, 0x6d, 0x3b, 0xa2, 0xd2, 0xa5, 0xb0, 0x72, 0x9f, 0x62, 0x4e,
    0xee, 0xc6, 0x7d, 0xc2, 0xe2, 0x30, 0x91, 0xb8, 0x6a, 0x08, 0x6f, 0x1e, 0x1d, 0x8e,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0016_PublicKey = ByteSpan(sTestCert_DAC_FFF2_8002_0016_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8002_0016_PrivateKey_Array[] = {
    0x26, 0xab, 0xde, 0x44, 0xa7, 0x80, 0x4d, 0xe4, 0x0f, 0xed, 0x59, 0x26, 0x99, 0x9a, 0x23, 0x51,
    0xf2, 0x04, 0x5a, 0x2d, 0x82, 0xff, 0xe5, 0x23, 0x58, 0x1e, 0x1f, 0xbd, 0x43, 0x2b, 0x07, 0x74,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0016_PrivateKey = ByteSpan(sTestCert_DAC_FFF2_8002_0016_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8002-0017-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8002_0017_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xd2, 0x30, 0x82, 0x01, 0x79, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x4b, 0x8c, 0x51, 0xf5, 0xf9, 0x73,
    0xca, 0xe3, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x30, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x32, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x39,
    0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41, 0x43,
    0x20, 0x30, 0x30, 0x31, 0x37, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01,
    0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02,
    0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x32, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06,
    0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x31, 0xd7, 0x7a, 0x0a, 0x5c, 0x13, 0x6f, 0x49,
    0xcf, 0xba, 0x2e, 0x2d, 0x7d, 0xbb, 0x8f, 0x79, 0x16, 0xcc, 0x60, 0xe8, 0x9b, 0x89, 0x3f, 0xbe, 0x30, 0x24, 0x2c, 0x8f, 0xf8,
    0x8d, 0x87, 0xa6, 0x01, 0x7a, 0x17, 0x06, 0xc7, 0xde, 0xbd, 0xa8, 0x83, 0x27, 0xd7, 0x2e, 0x59, 0xa0, 0xe4, 0x4b, 0xb8, 0xf4,
    0x57, 0x29, 0xd7, 0x61, 0xd7, 0xe6, 0x88, 0x3e, 0x40, 0xd0, 0x53, 0xd9, 0xbd, 0xe8, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c, 0x06,
    0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff,
    0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x39, 0x33, 0xe2, 0x48,
    0xb1, 0x76, 0x97, 0x57, 0x15, 0x8b, 0x0e, 0x39, 0x9f, 0x5e, 0x5c, 0x2e, 0xee, 0x60, 0xbc, 0x0a, 0x30, 0x1f, 0x06, 0x03, 0x55,
    0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x61, 0x3d, 0xd0, 0x87, 0x35, 0x5e, 0xf0, 0x8b, 0xae, 0x01, 0xe4, 0xc6, 0x9a,
    0x8f, 0xc7, 0x3d, 0xac, 0x8c, 0x7d, 0xfd, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x47,
    0x00, 0x30, 0x44, 0x02, 0x20, 0x14, 0xe4, 0x21, 0x8c, 0x4d, 0xaf, 0x03, 0x23, 0x30, 0xff, 0x04, 0xdb, 0xa9, 0x29, 0x81, 0x5c,
    0xdf, 0xef, 0x3f, 0x0e, 0x7c, 0x4d, 0x82, 0x97, 0x1b, 0x24, 0xbd, 0x5f, 0xe9, 0x3e, 0xbf, 0x78, 0x02, 0x20, 0x5d, 0xed, 0xc4,
    0x19, 0x8c, 0xfe, 0xe8, 0xb3, 0x1b, 0x25, 0x85, 0xe9, 0x82, 0x1b, 0x7c, 0x3b, 0x11, 0xa2, 0xf8, 0x18, 0x49, 0xcf, 0xee, 0x23,
    0x2b, 0x99, 0x5b, 0x30, 0xa6, 0x09, 0x1f, 0xc0,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0017_Cert = ByteSpan(sTestCert_DAC_FFF2_8002_0017_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8002_0017_SKID_Array[] = {
    0x39, 0x33, 0xE2, 0x48, 0xB1, 0x76, 0x97, 0x57, 0x15, 0x8B, 0x0E, 0x39, 0x9F, 0x5E, 0x5C, 0x2E, 0xEE, 0x60, 0xBC, 0x0A,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0017_SKID = ByteSpan(sTestCert_DAC_FFF2_8002_0017_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8002-0017-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8002_0017_PublicKey_Array[] = {
    0x04, 0x31, 0xd7, 0x7a, 0x0a, 0x5c, 0x13, 0x6f, 0x49, 0xcf, 0xba, 0x2e, 0x2d, 0x7d, 0xbb, 0x8f, 0x79,
    0x16, 0xcc, 0x60, 0xe8, 0x9b, 0x89, 0x3f, 0xbe, 0x30, 0x24, 0x2c, 0x8f, 0xf8, 0x8d, 0x87, 0xa6, 0x01,
    0x7a, 0x17, 0x06, 0xc7, 0xde, 0xbd, 0xa8, 0x83, 0x27, 0xd7, 0x2e, 0x59, 0xa0, 0xe4, 0x4b, 0xb8, 0xf4,
    0x57, 0x29, 0xd7, 0x61, 0xd7, 0xe6, 0x88, 0x3e, 0x40, 0xd0, 0x53, 0xd9, 0xbd, 0xe8,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0017_PublicKey = ByteSpan(sTestCert_DAC_FFF2_8002_0017_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8002_0017_PrivateKey_Array[] = {
    0x4f, 0x55, 0xd1, 0xff, 0x27, 0xff, 0x69, 0xb3, 0x39, 0x5b, 0x69, 0xac, 0x53, 0x1e, 0x00, 0x83,
    0x49, 0xb1, 0x61, 0x40, 0x51, 0xa2, 0x6c, 0xb9, 0x92, 0x4f, 0xf2, 0xd7, 0x15, 0xe4, 0x32, 0x40,
};

extern const ByteSpan sTestCert_DAC_FFF2_8002_0017_PrivateKey = ByteSpan(sTestCert_DAC_FFF2_8002_0017_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8003-0018-FB-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8003_0018_FB_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xaf, 0x30, 0x82, 0x01, 0x55, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x5a, 0x16, 0x82, 0x9b, 0x0d, 0xaf,
    0x00, 0x8a, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x24, 0x31, 0x22, 0x30, 0x20, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x19, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x20, 0x4d, 0x76, 0x69, 0x64, 0x3a, 0x46, 0x46, 0x46, 0x32, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31,
    0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x39, 0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35,
    0x39, 0x5a, 0x30, 0x33, 0x31, 0x31, 0x30, 0x2f, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x28, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72,
    0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41, 0x43, 0x20, 0x30, 0x30, 0x31, 0x38, 0x20, 0x4d, 0x76, 0x69, 0x64, 0x3a, 0x46,
    0x46, 0x46, 0x32, 0x20, 0x4d, 0x70, 0x69, 0x64, 0x3a, 0x38, 0x30, 0x30, 0x33, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86,
    0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xba, 0x9f,
    0xaf, 0x3d, 0x77, 0xf5, 0x93, 0x5d, 0xe8, 0x69, 0xfd, 0x1f, 0xc3, 0x20, 0x28, 0xc8, 0x0d, 0x24, 0xe7, 0x98, 0x6a, 0x0c, 0x49,
    0x90, 0x88, 0xce, 0x1f, 0x28, 0x17, 0x30, 0x83, 0x90, 0xf0, 0x1d, 0x97, 0x79, 0xbd, 0x03, 0xb2, 0xf7, 0xe8, 0x95, 0x4a, 0x86,
    0xe7, 0x28, 0x29, 0xa2, 0xc4, 0xcc, 0x97, 0x08, 0x55, 0xb8, 0x8c, 0x5d, 0xec, 0x31, 0x13, 0xd9, 0x8f, 0x55, 0x04, 0x36, 0xa3,
    0x60, 0x30, 0x5e, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03,
    0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16,
    0x04, 0x14, 0xb9, 0x59, 0xdc, 0xed, 0xb2, 0x52, 0x93, 0xe4, 0xa8, 0xf2, 0xb7, 0x56, 0x71, 0x38, 0x5f, 0x62, 0x63, 0x84, 0xf1,
    0xd6, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x19, 0xfa, 0x94, 0x74, 0x93, 0x29, 0xda,
    0x74, 0x0f, 0x55, 0x6a, 0x33, 0x3f, 0xb5, 0x13, 0x73, 0xd7, 0x08, 0x17, 0x56, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce,
    0x3d, 0x04, 0x03, 0x02, 0x03, 0x48, 0x00, 0x30, 0x45, 0x02, 0x20, 0x5d, 0x47, 0x64, 0xfa, 0xe1, 0xcc, 0x21, 0x15, 0x20, 0x12,
    0x8f, 0xd2, 0x52, 0x78, 0xe1, 0xc8, 0xda, 0x8e, 0xd4, 0xc6, 0x0a, 0x71, 0x30, 0x4f, 0x42, 0x10, 0xea, 0x3f, 0x1d, 0x49, 0x11,
    0x9b, 0x02, 0x21, 0x00, 0xb6, 0xa3, 0xce, 0xd8, 0xc1, 0x12, 0xf4, 0x8e, 0x1e, 0x1a, 0x98, 0x0e, 0xfe, 0xd2, 0x32, 0xd4, 0x32,
    0xca, 0x4d, 0xe9, 0xc5, 0xe5, 0x20, 0x19, 0x22, 0x8e, 0xbf, 0x8a, 0x99, 0x2e, 0xf1, 0x4a,
};

extern const ByteSpan sTestCert_DAC_FFF2_8003_0018_FB_Cert = ByteSpan(sTestCert_DAC_FFF2_8003_0018_FB_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8003_0018_FB_SKID_Array[] = {
    0xB9, 0x59, 0xDC, 0xED, 0xB2, 0x52, 0x93, 0xE4, 0xA8, 0xF2, 0xB7, 0x56, 0x71, 0x38, 0x5F, 0x62, 0x63, 0x84, 0xF1, 0xD6,
};

extern const ByteSpan sTestCert_DAC_FFF2_8003_0018_FB_SKID = ByteSpan(sTestCert_DAC_FFF2_8003_0018_FB_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8003-0018-FB-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8003_0018_FB_PublicKey_Array[] = {
    0x04, 0xba, 0x9f, 0xaf, 0x3d, 0x77, 0xf5, 0x93, 0x5d, 0xe8, 0x69, 0xfd, 0x1f, 0xc3, 0x20, 0x28, 0xc8,
    0x0d, 0x24, 0xe7, 0x98, 0x6a, 0x0c, 0x49, 0x90, 0x88, 0xce, 0x1f, 0x28, 0x17, 0x30, 0x83, 0x90, 0xf0,
    0x1d, 0x97, 0x79, 0xbd, 0x03, 0xb2, 0xf7, 0xe8, 0x95, 0x4a, 0x86, 0xe7, 0x28, 0x29, 0xa2, 0xc4, 0xcc,
    0x97, 0x08, 0x55, 0xb8, 0x8c, 0x5d, 0xec, 0x31, 0x13, 0xd9, 0x8f, 0x55, 0x04, 0x36,
};

extern const ByteSpan sTestCert_DAC_FFF2_8003_0018_FB_PublicKey = ByteSpan(sTestCert_DAC_FFF2_8003_0018_FB_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8003_0018_FB_PrivateKey_Array[] = {
    0xb7, 0x50, 0x27, 0x5c, 0x30, 0x5a, 0xe8, 0xc7, 0xbf, 0xf8, 0xa0, 0xc7, 0x46, 0x24, 0x1b, 0xda,
    0x98, 0xde, 0xf5, 0x21, 0x6f, 0xcf, 0xba, 0x61, 0x26, 0x78, 0x3a, 0xb3, 0xae, 0xcb, 0x41, 0x68,
};

extern const ByteSpan sTestCert_DAC_FFF2_8003_0018_FB_PrivateKey = ByteSpan(sTestCert_DAC_FFF2_8003_0018_FB_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8003-0019-FB-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8003_0019_FB_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xaf, 0x30, 0x82, 0x01, 0x55, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x3d, 0x8c, 0x8b, 0x82, 0xc8, 0x8f,
    0x46, 0x22, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x24, 0x31, 0x22, 0x30, 0x20, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x19, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x20, 0x4d, 0x76, 0x69, 0x64, 0x3a, 0x46, 0x46, 0x46, 0x32, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31,
    0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x39, 0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35,
    0x39, 0x5a, 0x30, 0x33, 0x31, 0x31, 0x30, 0x2f, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x28, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72,
    0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41, 0x43, 0x20, 0x30, 0x30, 0x31, 0x39, 0x20, 0x4d, 0x76, 0x69, 0x64, 0x3a, 0x46,
    0x46, 0x46, 0x32, 0x20, 0x4d, 0x70, 0x69, 0x64, 0x3a, 0x38, 0x30, 0x30, 0x33, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86,
    0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x74, 0x3f,
    0x87, 0xb9, 0xe7, 0xac, 0x98, 0xc3, 0x20, 0x01, 0x54, 0x72, 0xd7, 0xf5, 0x04, 0xe6, 0x63, 0xda, 0x7b, 0xe8, 0x76, 0x14, 0x54,
    0x8c, 0xa6, 0xf1, 0x25, 0x23, 0x62, 0x51, 0xbf, 0xe2, 0xaf, 0xb2, 0x41, 0x01, 0x69, 0x7c, 0x14, 0xb2, 0x19, 0xb2, 0x85, 0xc2,
    0x82, 0xf4, 0xab, 0x3e, 0x74, 0x7c, 0x53, 0x59, 0x1a, 0xb3, 0x7c, 0xe8, 0x86, 0xdd, 0x86, 0x45, 0xfb, 0x09, 0x12, 0x29, 0xa3,
    0x60, 0x30, 0x5e, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03,
    0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16,
    0x04, 0x14, 0x4c, 0x4f, 0x73, 0xb5, 0x17, 0x7e, 0xed, 0xb4, 0x86, 0x26, 0x80, 0xd6, 0x94, 0x08, 0x8a, 0x35, 0x1a, 0x45, 0x2c,
    0x05, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x19, 0xfa, 0x94, 0x74, 0x93, 0x29, 0xda,
    0x74, 0x0f, 0x55, 0x6a, 0x33, 0x3f, 0xb5, 0x13, 0x73, 0xd7, 0x08, 0x17, 0x56, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce,
    0x3d, 0x04, 0x03, 0x02, 0x03, 0x48, 0x00, 0x30, 0x45, 0x02, 0x21, 0x00, 0x80, 0x12, 0x73, 0xaa, 0x8f, 0x9f, 0x4f, 0xbf, 0x6d,
    0x7f, 0xb6, 0xc7, 0x0b, 0x07, 0x53, 0x9c, 0x07, 0xbc, 0x12, 0x1b, 0xe2, 0x98, 0x02, 0x56, 0x73, 0xbb, 0x0b, 0x50, 0x58, 0xe8,
    0xa5, 0xea, 0x02, 0x20, 0x73, 0xf2, 0x3a, 0xd5, 0x37, 0xd7, 0xa1, 0xfc, 0x2e, 0x1b, 0x81, 0x17, 0x4d, 0x85, 0x13, 0x47, 0x50,
    0x8d, 0x6e, 0x9c, 0xf2, 0xb4, 0x66, 0xec, 0x88, 0x59, 0x3f, 0x86, 0x46, 0xfd, 0xe5, 0xb7,
};

extern const ByteSpan sTestCert_DAC_FFF2_8003_0019_FB_Cert = ByteSpan(sTestCert_DAC_FFF2_8003_0019_FB_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8003_0019_FB_SKID_Array[] = {
    0x4C, 0x4F, 0x73, 0xB5, 0x17, 0x7E, 0xED, 0xB4, 0x86, 0x26, 0x80, 0xD6, 0x94, 0x08, 0x8A, 0x35, 0x1A, 0x45, 0x2C, 0x05,
};

extern const ByteSpan sTestCert_DAC_FFF2_8003_0019_FB_SKID = ByteSpan(sTestCert_DAC_FFF2_8003_0019_FB_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8003-0019-FB-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8003_0019_FB_PublicKey_Array[] = {
    0x04, 0x74, 0x3f, 0x87, 0xb9, 0xe7, 0xac, 0x98, 0xc3, 0x20, 0x01, 0x54, 0x72, 0xd7, 0xf5, 0x04, 0xe6,
    0x63, 0xda, 0x7b, 0xe8, 0x76, 0x14, 0x54, 0x8c, 0xa6, 0xf1, 0x25, 0x23, 0x62, 0x51, 0xbf, 0xe2, 0xaf,
    0xb2, 0x41, 0x01, 0x69, 0x7c, 0x14, 0xb2, 0x19, 0xb2, 0x85, 0xc2, 0x82, 0xf4, 0xab, 0x3e, 0x74, 0x7c,
    0x53, 0x59, 0x1a, 0xb3, 0x7c, 0xe8, 0x86, 0xdd, 0x86, 0x45, 0xfb, 0x09, 0x12, 0x29,
};

extern const ByteSpan sTestCert_DAC_FFF2_8003_0019_FB_PublicKey = ByteSpan(sTestCert_DAC_FFF2_8003_0019_FB_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8003_0019_FB_PrivateKey_Array[] = {
    0xd5, 0x30, 0x2b, 0xe9, 0x1e, 0x11, 0x11, 0x1d, 0x05, 0x79, 0x04, 0x1b, 0xac, 0x66, 0x17, 0x21,
    0xc5, 0xab, 0x51, 0x6d, 0xcb, 0x41, 0x18, 0x9f, 0xa3, 0x50, 0xa3, 0x0b, 0xc2, 0x6f, 0x13, 0xb9,
};

extern const ByteSpan sTestCert_DAC_FFF2_8003_0019_FB_PrivateKey = ByteSpan(sTestCert_DAC_FFF2_8003_0019_FB_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8003-001A-FB-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8003_001A_FB_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xaf, 0x30, 0x82, 0x01, 0x55, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x0d, 0xde, 0x73, 0x14, 0xb3, 0x6e,
    0xe4, 0x2a, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x24, 0x31, 0x22, 0x30, 0x20, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x19, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x20, 0x4d, 0x76, 0x69, 0x64, 0x3a, 0x46, 0x46, 0x46, 0x32, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31,
    0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x39, 0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35,
    0x39, 0x5a, 0x30, 0x33, 0x31, 0x31, 0x30, 0x2f, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x28, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72,
    0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41, 0x43, 0x20, 0x30, 0x30, 0x31, 0x41, 0x20, 0x4d, 0x76, 0x69, 0x64, 0x3a, 0x46,
    0x46, 0x46, 0x32, 0x20, 0x4d, 0x70, 0x69, 0x64, 0x3a, 0x38, 0x30, 0x30, 0x33, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86,
    0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x37, 0xb4,
    0xe0, 0x3d, 0x45, 0x95, 0x22, 0xfa, 0x2f, 0xa7, 0x76, 0x17, 0x39, 0xcb, 0x2e, 0xdb, 0xc1, 0x14, 0x85, 0x50, 0x1b, 0x40, 0xdb,
    0xb0, 0x82, 0x6f, 0x62, 0x4e, 0x54, 0xe0, 0x62, 0x72, 0x63, 0x94, 0x7f, 0x4f, 0xf2, 0x12, 0xf9, 0xb6, 0x99, 0x30, 0x0a, 0x2c,
    0x6a, 0xc5, 0x3e, 0xa8, 0x69, 0xa4, 0x9c, 0x15, 0xe2, 0x16, 0x65, 0xe5, 0x30, 0xdb, 0x56, 0x0a, 0x87, 0x51, 0x4d, 0x71, 0xa3,
    0x60, 0x30, 0x5e, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03,
    0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16,
    0x04, 0x14, 0x55, 0xb3, 0xfc, 0x16, 0x29, 0x6d, 0x5a, 0x9d, 0xfa, 0xe5, 0xb3, 0x96, 0x68, 0x72, 0x34, 0x2a, 0x30, 0x81, 0x44,
    0x99, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x19, 0xfa, 0x94, 0x74, 0x93, 0x29, 0xda,
    0x74, 0x0f, 0x55, 0x6a, 0x33, 0x3f, 0xb5, 0x13, 0x73, 0xd7, 0x08, 0x17, 0x56, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce,
    0x3d, 0x04, 0x03, 0x02, 0x03, 0x48, 0x00, 0x30, 0x45, 0x02, 0x20, 0x57, 0xa8, 0x2f, 0x74, 0x9f, 0xc5, 0x30, 0x30, 0x74, 0x5c,
    0x23, 0x02, 0xee, 0xdd, 0x81, 0xb3, 0x2a, 0x78, 0x12, 0x49, 0x57, 0x0e, 0x52, 0x61, 0xc0, 0xe7, 0xde, 0x5d, 0xf7, 0x9d, 0x49,
    0xde, 0x02, 0x21, 0x00, 0xf2, 0xa6, 0xe0, 0xc7, 0x1a, 0xcb, 0xa5, 0xf3, 0x07, 0xbb, 0x77, 0x51, 0xa0, 0x8f, 0xe6, 0xbf, 0xaa,
    0x58, 0x11, 0x9b, 0x9d, 0x27, 0x09, 0xc6, 0x24, 0x6d, 0xf4, 0x9b, 0xe9, 0x6b, 0x62, 0x89,
};

extern const ByteSpan sTestCert_DAC_FFF2_8003_001A_FB_Cert = ByteSpan(sTestCert_DAC_FFF2_8003_001A_FB_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8003_001A_FB_SKID_Array[] = {
    0x55, 0xB3, 0xFC, 0x16, 0x29, 0x6D, 0x5A, 0x9D, 0xFA, 0xE5, 0xB3, 0x96, 0x68, 0x72, 0x34, 0x2A, 0x30, 0x81, 0x44, 0x99,
};

extern const ByteSpan sTestCert_DAC_FFF2_8003_001A_FB_SKID = ByteSpan(sTestCert_DAC_FFF2_8003_001A_FB_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8003-001A-FB-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8003_001A_FB_PublicKey_Array[] = {
    0x04, 0x37, 0xb4, 0xe0, 0x3d, 0x45, 0x95, 0x22, 0xfa, 0x2f, 0xa7, 0x76, 0x17, 0x39, 0xcb, 0x2e, 0xdb,
    0xc1, 0x14, 0x85, 0x50, 0x1b, 0x40, 0xdb, 0xb0, 0x82, 0x6f, 0x62, 0x4e, 0x54, 0xe0, 0x62, 0x72, 0x63,
    0x94, 0x7f, 0x4f, 0xf2, 0x12, 0xf9, 0xb6, 0x99, 0x30, 0x0a, 0x2c, 0x6a, 0xc5, 0x3e, 0xa8, 0x69, 0xa4,
    0x9c, 0x15, 0xe2, 0x16, 0x65, 0xe5, 0x30, 0xdb, 0x56, 0x0a, 0x87, 0x51, 0x4d, 0x71,
};

extern const ByteSpan sTestCert_DAC_FFF2_8003_001A_FB_PublicKey = ByteSpan(sTestCert_DAC_FFF2_8003_001A_FB_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8003_001A_FB_PrivateKey_Array[] = {
    0x1a, 0x94, 0x70, 0x1e, 0x70, 0x5f, 0xb3, 0x83, 0x15, 0x5a, 0x4e, 0xdf, 0x84, 0xad, 0x67, 0xbf,
    0x9f, 0x93, 0x48, 0x76, 0x0d, 0x28, 0x46, 0xba, 0xdc, 0x75, 0xc4, 0xf8, 0x63, 0x5c, 0xcb, 0x0b,
};

extern const ByteSpan sTestCert_DAC_FFF2_8003_001A_FB_PrivateKey = ByteSpan(sTestCert_DAC_FFF2_8003_001A_FB_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8003-001B-FB-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8003_001B_FB_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xb0, 0x30, 0x82, 0x01, 0x55, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x38, 0x52, 0x74, 0x71, 0x96, 0xfa,
    0x6a, 0xf1, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x24, 0x31, 0x22, 0x30, 0x20, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x19, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x20, 0x4d, 0x76, 0x69, 0x64, 0x3a, 0x46, 0x46, 0x46, 0x32, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31,
    0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x39, 0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35,
    0x39, 0x5a, 0x30, 0x33, 0x31, 0x31, 0x30, 0x2f, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x28, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72,
    0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41, 0x43, 0x20, 0x30, 0x30, 0x31, 0x42, 0x20, 0x4d, 0x76, 0x69, 0x64, 0x3a, 0x46,
    0x46, 0x46, 0x32, 0x20, 0x4d, 0x70, 0x69, 0x64, 0x3a, 0x38, 0x30, 0x30, 0x33, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86,
    0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xc7, 0x8e,
    0xe5, 0x70, 0xc3, 0x2c, 0x8b, 0x31, 0xb1, 0x2d, 0xbb, 0x26, 0xb5, 0x58, 0xee, 0xdb, 0x2c, 0xfe, 0x7e, 0xcb, 0x7c, 0x61, 0x65,
    0xa5, 0x2e, 0x2a, 0x84, 0x18, 0xbe, 0x2e, 0xca, 0x7e, 0x28, 0x1f, 0x6d, 0x54, 0x27, 0x54, 0xe6, 0xc2, 0xfa, 0x07, 0x6c, 0xe9,
    0xc1, 0x7b, 0xf9, 0x4d, 0xe1, 0x0e, 0x2a, 0xea, 0x4f, 0x89, 0xed, 0x70, 0xe2, 0x6e, 0x85, 0x0f, 0x49, 0xa9, 0xb5, 0x7e, 0xa3,
    0x60, 0x30, 0x5e, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03,
    0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16,
    0x04, 0x14, 0x11, 0xf1, 0x0f, 0x09, 0x08, 0x78, 0xb5, 0xb8, 0x29, 0x32, 0x45, 0xce, 0x27, 0xe7, 0x8c, 0x5d, 0x3e, 0x97, 0x2f,
    0xd9, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x19, 0xfa, 0x94, 0x74, 0x93, 0x29, 0xda,
    0x74, 0x0f, 0x55, 0x6a, 0x33, 0x3f, 0xb5, 0x13, 0x73, 0xd7, 0x08, 0x17, 0x56, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce,
    0x3d, 0x04, 0x03, 0x02, 0x03, 0x49, 0x00, 0x30, 0x46, 0x02, 0x21, 0x00, 0xbf, 0x77, 0x77, 0xe7, 0x6c, 0x18, 0x4e, 0xdb, 0xb2,
    0x03, 0x71, 0xe8, 0xf3, 0x72, 0xec, 0xd1, 0xdc, 0x9f, 0x81, 0x2a, 0x44, 0xd2, 0xdb, 0x43, 0x68, 0x89, 0x1f, 0x1b, 0x59, 0xab,
    0xa4, 0x7b, 0x02, 0x21, 0x00, 0xce, 0x48, 0x82, 0xbd, 0x52, 0x92, 0x0b, 0x39, 0x04, 0xcb, 0x4d, 0x3e, 0x84, 0xf7, 0x78, 0xcb,
    0x15, 0xd2, 0x50, 0x74, 0xc8, 0xad, 0x98, 0x2c, 0x5e, 0xcf, 0xa9, 0xc4, 0xab, 0x40, 0xbb, 0x83,
};

extern const ByteSpan sTestCert_DAC_FFF2_8003_001B_FB_Cert = ByteSpan(sTestCert_DAC_FFF2_8003_001B_FB_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8003_001B_FB_SKID_Array[] = {
    0x11, 0xF1, 0x0F, 0x09, 0x08, 0x78, 0xB5, 0xB8, 0x29, 0x32, 0x45, 0xCE, 0x27, 0xE7, 0x8C, 0x5D, 0x3E, 0x97, 0x2F, 0xD9,
};

extern const ByteSpan sTestCert_DAC_FFF2_8003_001B_FB_SKID = ByteSpan(sTestCert_DAC_FFF2_8003_001B_FB_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8003-001B-FB-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8003_001B_FB_PublicKey_Array[] = {
    0x04, 0xc7, 0x8e, 0xe5, 0x70, 0xc3, 0x2c, 0x8b, 0x31, 0xb1, 0x2d, 0xbb, 0x26, 0xb5, 0x58, 0xee, 0xdb,
    0x2c, 0xfe, 0x7e, 0xcb, 0x7c, 0x61, 0x65, 0xa5, 0x2e, 0x2a, 0x84, 0x18, 0xbe, 0x2e, 0xca, 0x7e, 0x28,
    0x1f, 0x6d, 0x54, 0x27, 0x54, 0xe6, 0xc2, 0xfa, 0x07, 0x6c, 0xe9, 0xc1, 0x7b, 0xf9, 0x4d, 0xe1, 0x0e,
    0x2a, 0xea, 0x4f, 0x89, 0xed, 0x70, 0xe2, 0x6e, 0x85, 0x0f, 0x49, 0xa9, 0xb5, 0x7e,
};

extern const ByteSpan sTestCert_DAC_FFF2_8003_001B_FB_PublicKey = ByteSpan(sTestCert_DAC_FFF2_8003_001B_FB_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8003_001B_FB_PrivateKey_Array[] = {
    0xb9, 0xf1, 0x7e, 0xf4, 0xff, 0xc0, 0x26, 0xaf, 0x96, 0xf6, 0xb6, 0x3b, 0x96, 0x3d, 0x52, 0x97,
    0xe1, 0xc4, 0x24, 0xe3, 0xcd, 0xcf, 0x2e, 0x1b, 0xe1, 0x39, 0xc7, 0x2a, 0xc0, 0x68, 0x88, 0x1a,
};

extern const ByteSpan sTestCert_DAC_FFF2_8003_001B_FB_PrivateKey = ByteSpan(sTestCert_DAC_FFF2_8003_001B_FB_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8004-001C-FB-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8004_001C_FB_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xb9, 0x30, 0x82, 0x01, 0x5f, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x1a, 0xad, 0x64, 0xa6, 0x95, 0x07,
    0xcd, 0xcf, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x2e, 0x31, 0x2c, 0x30, 0x2a, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x23, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x20, 0x4d, 0x76, 0x69, 0x64, 0x3a, 0x46, 0x46, 0x46, 0x32, 0x20, 0x4d, 0x70, 0x69, 0x64, 0x3a, 0x38, 0x30, 0x30, 0x34, 0x30,
    0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x39, 0x39, 0x39,
    0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x33, 0x31, 0x31, 0x30, 0x2f, 0x06, 0x03, 0x55,
    0x04, 0x03, 0x0c, 0x28, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41, 0x43, 0x20, 0x30,
    0x30, 0x31, 0x43, 0x20, 0x4d, 0x76, 0x69, 0x64, 0x3a, 0x46, 0x46, 0x46, 0x32, 0x20, 0x4d, 0x70, 0x69, 0x64, 0x3a, 0x38, 0x30,
    0x30, 0x34, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce,
    0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xd9, 0x00, 0x0e, 0xe4, 0x63, 0xcd, 0x36, 0x84, 0x71, 0x06, 0x4c, 0x3d, 0x28,
    0xfc, 0x34, 0x15, 0x68, 0x8d, 0xaf, 0x2e, 0x5a, 0xc4, 0xe4, 0xd8, 0x28, 0xd8, 0xd4, 0xb2, 0x11, 0xd8, 0xe6, 0xd3, 0x59, 0xb9,
    0x59, 0x64, 0x47, 0xda, 0x57, 0x80, 0xb0, 0x62, 0xb5, 0xc3, 0xa5, 0x00, 0xcc, 0xfc, 0xe5, 0x43, 0x23, 0x79, 0xe9, 0x2f, 0xcc,
    0x2c, 0x89, 0x27, 0x4d, 0xf1, 0x19, 0x23, 0xe7, 0x32, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01,
    0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x07,
    0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xc0, 0x32, 0x51, 0x0a, 0x09, 0xe4, 0x76, 0xdb, 0x55,
    0x23, 0x0c, 0xa4, 0x7b, 0xd6, 0x5a, 0x6b, 0xa5, 0x12, 0x34, 0x79, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30,
    0x16, 0x80, 0x14, 0x15, 0x6e, 0xcd, 0x8c, 0x14, 0xbe, 0x80, 0xb4, 0x10, 0xae, 0xe4, 0x02, 0xa3, 0xf2, 0x33, 0x17, 0xe7, 0x34,
    0x06, 0x74, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x48, 0x00, 0x30, 0x45, 0x02, 0x21,
    0x00, 0xba, 0xd9, 0x9d, 0x34, 0x1d, 0x38, 0x4d, 0x9e, 0x8d, 0x2e, 0xa0, 0x6d, 0x09, 0xe6, 0x37, 0x88, 0xa7, 0x04, 0x90, 0xf3,
    0x98, 0xfc, 0xd8, 0xc4, 0x7c, 0x2b, 0xfa, 0xb7, 0x7b, 0x8b, 0xd4, 0xfb, 0x02, 0x20, 0x1d, 0x8d, 0x40, 0x9a, 0x3e, 0x93, 0x7e,
    0x16, 0x11, 0x11, 0x6a, 0xed, 0xd0, 0xc9, 0xe2, 0x9a, 0x71, 0x91, 0x4a, 0x63, 0x0b, 0x0c, 0x32, 0xdb, 0xda, 0x45, 0xf6, 0x1b,
    0x26, 0x73, 0x6e, 0x7d,
};

extern const ByteSpan sTestCert_DAC_FFF2_8004_001C_FB_Cert = ByteSpan(sTestCert_DAC_FFF2_8004_001C_FB_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8004_001C_FB_SKID_Array[] = {
    0xC0, 0x32, 0x51, 0x0A, 0x09, 0xE4, 0x76, 0xDB, 0x55, 0x23, 0x0C, 0xA4, 0x7B, 0xD6, 0x5A, 0x6B, 0xA5, 0x12, 0x34, 0x79,
};

extern const ByteSpan sTestCert_DAC_FFF2_8004_001C_FB_SKID = ByteSpan(sTestCert_DAC_FFF2_8004_001C_FB_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8004-001C-FB-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8004_001C_FB_PublicKey_Array[] = {
    0x04, 0xd9, 0x00, 0x0e, 0xe4, 0x63, 0xcd, 0x36, 0x84, 0x71, 0x06, 0x4c, 0x3d, 0x28, 0xfc, 0x34, 0x15,
    0x68, 0x8d, 0xaf, 0x2e, 0x5a, 0xc4, 0xe4, 0xd8, 0x28, 0xd8, 0xd4, 0xb2, 0x11, 0xd8, 0xe6, 0xd3, 0x59,
    0xb9, 0x59, 0x64, 0x47, 0xda, 0x57, 0x80, 0xb0, 0x62, 0xb5, 0xc3, 0xa5, 0x00, 0xcc, 0xfc, 0xe5, 0x43,
    0x23, 0x79, 0xe9, 0x2f, 0xcc, 0x2c, 0x89, 0x27, 0x4d, 0xf1, 0x19, 0x23, 0xe7, 0x32,
};

extern const ByteSpan sTestCert_DAC_FFF2_8004_001C_FB_PublicKey = ByteSpan(sTestCert_DAC_FFF2_8004_001C_FB_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8004_001C_FB_PrivateKey_Array[] = {
    0xf4, 0xd8, 0xd0, 0xb2, 0x50, 0x85, 0xa4, 0x8d, 0x96, 0xa6, 0x36, 0xda, 0xe5, 0x65, 0x36, 0xab,
    0x75, 0x10, 0xb8, 0xc0, 0x64, 0xd9, 0x4b, 0x5d, 0x0b, 0xcb, 0x6f, 0xa9, 0x00, 0x12, 0xeb, 0x80,
};

extern const ByteSpan sTestCert_DAC_FFF2_8004_001C_FB_PrivateKey = ByteSpan(sTestCert_DAC_FFF2_8004_001C_FB_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8004-001D-FB-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8004_001D_FB_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xb9, 0x30, 0x82, 0x01, 0x5f, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x42, 0xaf, 0xcf, 0xa9, 0x11, 0xb7,
    0x6f, 0xa5, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x2e, 0x31, 0x2c, 0x30, 0x2a, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x23, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x20, 0x4d, 0x76, 0x69, 0x64, 0x3a, 0x46, 0x46, 0x46, 0x32, 0x20, 0x4d, 0x70, 0x69, 0x64, 0x3a, 0x38, 0x30, 0x30, 0x34, 0x30,
    0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x39, 0x39, 0x39,
    0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x33, 0x31, 0x31, 0x30, 0x2f, 0x06, 0x03, 0x55,
    0x04, 0x03, 0x0c, 0x28, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41, 0x43, 0x20, 0x30,
    0x30, 0x31, 0x44, 0x20, 0x4d, 0x76, 0x69, 0x64, 0x3a, 0x46, 0x46, 0x46, 0x32, 0x20, 0x4d, 0x70, 0x69, 0x64, 0x3a, 0x38, 0x30,
    0x30, 0x34, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce,
    0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xd3, 0x4a, 0x2c, 0x59, 0xa2, 0xc6, 0xec, 0x5c, 0x3b, 0xd3, 0xea, 0x39, 0x00,
    0x66, 0x34, 0x05, 0xd9, 0x05, 0x5c, 0x22, 0xdf, 0xb2, 0x4e, 0x10, 0xce, 0xa2, 0x8c, 0x6b, 0x61, 0x8e, 0x03, 0x7e, 0x95, 0x3e,
    0x19, 0x74, 0x14, 0xa2, 0x23, 0x66, 0x79, 0x77, 0x96, 0x5c, 0x64, 0xcb, 0x90, 0x02, 0x18, 0xe5, 0xd5, 0x1b, 0x12, 0x9f, 0x19,
    0x46, 0x4e, 0xe4, 0x97, 0x2f, 0x24, 0xfa, 0x87, 0xd2, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01,
    0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x07,
    0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x21, 0xf9, 0x47, 0x53, 0x37, 0x61, 0xa5, 0xd9, 0x3e,
    0xbf, 0x88, 0x34, 0x9c, 0xa8, 0x1f, 0xff, 0x77, 0x0b, 0x21, 0xb3, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30,
    0x16, 0x80, 0x14, 0x15, 0x6e, 0xcd, 0x8c, 0x14, 0xbe, 0x80, 0xb4, 0x10, 0xae, 0xe4, 0x02, 0xa3, 0xf2, 0x33, 0x17, 0xe7, 0x34,
    0x06, 0x74, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x48, 0x00, 0x30, 0x45, 0x02, 0x21,
    0x00, 0xea, 0x33, 0x22, 0x48, 0x5e, 0x83, 0x9f, 0x89, 0x38, 0x88, 0x3c, 0x61, 0x04, 0x6b, 0x01, 0x03, 0x7e, 0xcf, 0x92, 0x96,
    0x9f, 0x9b, 0x53, 0x8d, 0x57, 0x37, 0xab, 0x8a, 0xb6, 0x65, 0x49, 0x74, 0x02, 0x20, 0x52, 0x2b, 0xd3, 0x77, 0xa1, 0xb1, 0xb8,
    0xb5, 0x26, 0xc4, 0x8f, 0xc1, 0x6b, 0xc6, 0x6d, 0x20, 0x14, 0xa5, 0x41, 0x06, 0xb1, 0x83, 0x1c, 0x8b, 0x23, 0xd0, 0xb1, 0x7b,
    0x92, 0xcf, 0xb8, 0xae,
};

extern const ByteSpan sTestCert_DAC_FFF2_8004_001D_FB_Cert = ByteSpan(sTestCert_DAC_FFF2_8004_001D_FB_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8004_001D_FB_SKID_Array[] = {
    0x21, 0xF9, 0x47, 0x53, 0x37, 0x61, 0xA5, 0xD9, 0x3E, 0xBF, 0x88, 0x34, 0x9C, 0xA8, 0x1F, 0xFF, 0x77, 0x0B, 0x21, 0xB3,
};

extern const ByteSpan sTestCert_DAC_FFF2_8004_001D_FB_SKID = ByteSpan(sTestCert_DAC_FFF2_8004_001D_FB_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8004-001D-FB-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8004_001D_FB_PublicKey_Array[] = {
    0x04, 0xd3, 0x4a, 0x2c, 0x59, 0xa2, 0xc6, 0xec, 0x5c, 0x3b, 0xd3, 0xea, 0x39, 0x00, 0x66, 0x34, 0x05,
    0xd9, 0x05, 0x5c, 0x22, 0xdf, 0xb2, 0x4e, 0x10, 0xce, 0xa2, 0x8c, 0x6b, 0x61, 0x8e, 0x03, 0x7e, 0x95,
    0x3e, 0x19, 0x74, 0x14, 0xa2, 0x23, 0x66, 0x79, 0x77, 0x96, 0x5c, 0x64, 0xcb, 0x90, 0x02, 0x18, 0xe5,
    0xd5, 0x1b, 0x12, 0x9f, 0x19, 0x46, 0x4e, 0xe4, 0x97, 0x2f, 0x24, 0xfa, 0x87, 0xd2,
};

extern const ByteSpan sTestCert_DAC_FFF2_8004_001D_FB_PublicKey = ByteSpan(sTestCert_DAC_FFF2_8004_001D_FB_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8004_001D_FB_PrivateKey_Array[] = {
    0xeb, 0x02, 0x7e, 0x67, 0x6c, 0xb0, 0x6c, 0xa9, 0xf0, 0xb6, 0xee, 0x2a, 0x5b, 0xfb, 0x9c, 0x8c,
    0xb0, 0x9a, 0xc5, 0x47, 0x73, 0xd9, 0x11, 0x40, 0xd1, 0x1e, 0x45, 0x2a, 0x42, 0xaa, 0xa6, 0x0c,
};

extern const ByteSpan sTestCert_DAC_FFF2_8004_001D_FB_PrivateKey = ByteSpan(sTestCert_DAC_FFF2_8004_001D_FB_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8004-001E-FB-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8004_001E_FB_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xb9, 0x30, 0x82, 0x01, 0x5f, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x38, 0x58, 0x3c, 0x1d, 0x11, 0xa8,
    0x2b, 0xc6, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x2e, 0x31, 0x2c, 0x30, 0x2a, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x23, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x20, 0x4d, 0x76, 0x69, 0x64, 0x3a, 0x46, 0x46, 0x46, 0x32, 0x20, 0x4d, 0x70, 0x69, 0x64, 0x3a, 0x38, 0x30, 0x30, 0x34, 0x30,
    0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x39, 0x39, 0x39,
    0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x33, 0x31, 0x31, 0x30, 0x2f, 0x06, 0x03, 0x55,
    0x04, 0x03, 0x0c, 0x28, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41, 0x43, 0x20, 0x30,
    0x30, 0x31, 0x45, 0x20, 0x4d, 0x76, 0x69, 0x64, 0x3a, 0x46, 0x46, 0x46, 0x32, 0x20, 0x4d, 0x70, 0x69, 0x64, 0x3a, 0x38, 0x30,
    0x30, 0x34, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce,
    0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x6e, 0x98, 0x0d, 0x08, 0x26, 0x6b, 0x93, 0x7c, 0x46, 0x41, 0x60, 0xe1, 0x2b,
    0x25, 0xb2, 0x7b, 0x8f, 0xbd, 0xfb, 0x4f, 0xca, 0xe6, 0xc0, 0x2e, 0x7f, 0x60, 0x92, 0xc1, 0xf6, 0xf4, 0x3b, 0x56, 0xa6, 0x0e,
    0xa1, 0x62, 0x98, 0x17, 0x83, 0xe6, 0xdd, 0xc5, 0xdd, 0xe7, 0xe4, 0x74, 0x3a, 0xab, 0x5c, 0xd1, 0xfc, 0x92, 0x79, 0x9d, 0xb3,
    0x1a, 0xde, 0xc0, 0x8d, 0xb1, 0x58, 0x84, 0x01, 0xd0, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01,
    0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x07,
    0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xf8, 0x92, 0x5a, 0xb3, 0x60, 0x86, 0x28, 0x94, 0x3f,
    0x3d, 0xdb, 0x88, 0xf1, 0xe3, 0xe8, 0x1f, 0xef, 0x35, 0xc7, 0x48, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30,
    0x16, 0x80, 0x14, 0x15, 0x6e, 0xcd, 0x8c, 0x14, 0xbe, 0x80, 0xb4, 0x10, 0xae, 0xe4, 0x02, 0xa3, 0xf2, 0x33, 0x17, 0xe7, 0x34,
    0x06, 0x74, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x48, 0x00, 0x30, 0x45, 0x02, 0x20,
    0x1b, 0x5f, 0xdd, 0xad, 0x5d, 0x4b, 0x14, 0x5c, 0xcd, 0x6e, 0x51, 0xf3, 0x9f, 0xa0, 0x14, 0xd8, 0x83, 0xa5, 0xa3, 0xf0, 0x0b,
    0xd7, 0x39, 0x47, 0x16, 0xad, 0x21, 0xd6, 0x57, 0x7a, 0x08, 0x93, 0x02, 0x21, 0x00, 0x9c, 0x6f, 0xe5, 0x3e, 0x91, 0x56, 0xf2,
    0xcb, 0x85, 0x22, 0x67, 0xb6, 0x6b, 0x9d, 0x7d, 0x42, 0xe7, 0xa4, 0x88, 0x47, 0x18, 0x4c, 0x2f, 0xc8, 0xb4, 0xc4, 0xf0, 0x92,
    0xea, 0x7e, 0x1e, 0x46,
};

extern const ByteSpan sTestCert_DAC_FFF2_8004_001E_FB_Cert = ByteSpan(sTestCert_DAC_FFF2_8004_001E_FB_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8004_001E_FB_SKID_Array[] = {
    0xF8, 0x92, 0x5A, 0xB3, 0x60, 0x86, 0x28, 0x94, 0x3F, 0x3D, 0xDB, 0x88, 0xF1, 0xE3, 0xE8, 0x1F, 0xEF, 0x35, 0xC7, 0x48,
};

extern const ByteSpan sTestCert_DAC_FFF2_8004_001E_FB_SKID = ByteSpan(sTestCert_DAC_FFF2_8004_001E_FB_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8004-001E-FB-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8004_001E_FB_PublicKey_Array[] = {
    0x04, 0x6e, 0x98, 0x0d, 0x08, 0x26, 0x6b, 0x93, 0x7c, 0x46, 0x41, 0x60, 0xe1, 0x2b, 0x25, 0xb2, 0x7b,
    0x8f, 0xbd, 0xfb, 0x4f, 0xca, 0xe6, 0xc0, 0x2e, 0x7f, 0x60, 0x92, 0xc1, 0xf6, 0xf4, 0x3b, 0x56, 0xa6,
    0x0e, 0xa1, 0x62, 0x98, 0x17, 0x83, 0xe6, 0xdd, 0xc5, 0xdd, 0xe7, 0xe4, 0x74, 0x3a, 0xab, 0x5c, 0xd1,
    0xfc, 0x92, 0x79, 0x9d, 0xb3, 0x1a, 0xde, 0xc0, 0x8d, 0xb1, 0x58, 0x84, 0x01, 0xd0,
};

extern const ByteSpan sTestCert_DAC_FFF2_8004_001E_FB_PublicKey = ByteSpan(sTestCert_DAC_FFF2_8004_001E_FB_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8004_001E_FB_PrivateKey_Array[] = {
    0xaf, 0xba, 0xf0, 0x31, 0xe9, 0x4c, 0x9b, 0x75, 0x69, 0x4b, 0x52, 0x0e, 0x37, 0x1a, 0x41, 0xe4,
    0xb7, 0xf0, 0xd3, 0x83, 0x90, 0x10, 0xa1, 0xe7, 0x26, 0x72, 0x34, 0xe2, 0x07, 0x22, 0x48, 0x2f,
};

extern const ByteSpan sTestCert_DAC_FFF2_8004_001E_FB_PrivateKey = ByteSpan(sTestCert_DAC_FFF2_8004_001E_FB_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8004-001F-FB-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8004_001F_FB_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xb8, 0x30, 0x82, 0x01, 0x5f, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x7e, 0xf9, 0x2e, 0xfb, 0x56, 0xe3,
    0x60, 0x2d, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x2e, 0x31, 0x2c, 0x30, 0x2a, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x23, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x20, 0x4d, 0x76, 0x69, 0x64, 0x3a, 0x46, 0x46, 0x46, 0x32, 0x20, 0x4d, 0x70, 0x69, 0x64, 0x3a, 0x38, 0x30, 0x30, 0x34, 0x30,
    0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x39, 0x39, 0x39,
    0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x33, 0x31, 0x31, 0x30, 0x2f, 0x06, 0x03, 0x55,
    0x04, 0x03, 0x0c, 0x28, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41, 0x43, 0x20, 0x30,
    0x30, 0x31, 0x46, 0x20, 0x4d, 0x76, 0x69, 0x64, 0x3a, 0x46, 0x46, 0x46, 0x32, 0x20, 0x4d, 0x70, 0x69, 0x64, 0x3a, 0x38, 0x30,
    0x30, 0x34, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce,
    0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x20, 0x87, 0x30, 0x7b, 0x3e, 0x63, 0x65, 0xeb, 0x99, 0x5a, 0xb6, 0x86, 0xe9,
    0x80, 0x82, 0xb3, 0x5a, 0x4a, 0x03, 0xb9, 0x9a, 0xf0, 0x13, 0xed, 0x1e, 0x31, 0xba, 0xb1, 0xb5, 0xf3, 0xe3, 0x88, 0x92, 0x24,
    0x53, 0xad, 0x2a, 0x2d, 0x07, 0xfb, 0x3b, 0x78, 0x20, 0xc5, 0x4a, 0xaf, 0xda, 0xfa, 0xb5, 0x07, 0x3f, 0xa7, 0x91, 0x1f, 0x39,
    0x84, 0x86, 0x8e, 0x2b, 0xe7, 0x70, 0x88, 0x70, 0xb3, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01,
    0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x07,
    0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x3f, 0x6a, 0xd7, 0x91, 0x88, 0xd2, 0xe2, 0x7a, 0xef,
    0x11, 0x1b, 0x51, 0xf3, 0x89, 0xb4, 0xf6, 0x91, 0x16, 0x59, 0xe3, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30,
    0x16, 0x80, 0x14, 0x15, 0x6e, 0xcd, 0x8c, 0x14, 0xbe, 0x80, 0xb4, 0x10, 0xae, 0xe4, 0x02, 0xa3, 0xf2, 0x33, 0x17, 0xe7, 0x34,
    0x06, 0x74, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x47, 0x00, 0x30, 0x44, 0x02, 0x20,
    0x51, 0xe8, 0xec, 0xe9, 0xf9, 0x47, 0xc2, 0xb2, 0xaf, 0x30, 0xf9, 0x9d, 0x64, 0x7a, 0x85, 0xf2, 0xc2, 0xc1, 0x58, 0x5c, 0x48,
    0xdb, 0x22, 0x13, 0x1b, 0x94, 0x86, 0xad, 0xdf, 0x13, 0x09, 0xfb, 0x02, 0x20, 0x0b, 0x49, 0xb8, 0xb7, 0x6a, 0x92, 0xc6, 0xcb,
    0x9b, 0x13, 0xea, 0xd9, 0xcd, 0x6e, 0x72, 0xc5, 0xfe, 0x4c, 0xce, 0x33, 0x51, 0x9e, 0xde, 0x83, 0xe2, 0xac, 0x9f, 0x05, 0xac,
    0x06, 0x7a, 0xf7,
};

extern const ByteSpan sTestCert_DAC_FFF2_8004_001F_FB_Cert = ByteSpan(sTestCert_DAC_FFF2_8004_001F_FB_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8004_001F_FB_SKID_Array[] = {
    0x3F, 0x6A, 0xD7, 0x91, 0x88, 0xD2, 0xE2, 0x7A, 0xEF, 0x11, 0x1B, 0x51, 0xF3, 0x89, 0xB4, 0xF6, 0x91, 0x16, 0x59, 0xE3,
};

extern const ByteSpan sTestCert_DAC_FFF2_8004_001F_FB_SKID = ByteSpan(sTestCert_DAC_FFF2_8004_001F_FB_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8004-001F-FB-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8004_001F_FB_PublicKey_Array[] = {
    0x04, 0x20, 0x87, 0x30, 0x7b, 0x3e, 0x63, 0x65, 0xeb, 0x99, 0x5a, 0xb6, 0x86, 0xe9, 0x80, 0x82, 0xb3,
    0x5a, 0x4a, 0x03, 0xb9, 0x9a, 0xf0, 0x13, 0xed, 0x1e, 0x31, 0xba, 0xb1, 0xb5, 0xf3, 0xe3, 0x88, 0x92,
    0x24, 0x53, 0xad, 0x2a, 0x2d, 0x07, 0xfb, 0x3b, 0x78, 0x20, 0xc5, 0x4a, 0xaf, 0xda, 0xfa, 0xb5, 0x07,
    0x3f, 0xa7, 0x91, 0x1f, 0x39, 0x84, 0x86, 0x8e, 0x2b, 0xe7, 0x70, 0x88, 0x70, 0xb3,
};

extern const ByteSpan sTestCert_DAC_FFF2_8004_001F_FB_PublicKey = ByteSpan(sTestCert_DAC_FFF2_8004_001F_FB_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8004_001F_FB_PrivateKey_Array[] = {
    0x2e, 0x29, 0xb7, 0x00, 0x5b, 0x46, 0x4e, 0x3a, 0x3d, 0xd8, 0x8b, 0x5a, 0xea, 0xcf, 0x7d, 0x16,
    0x27, 0x67, 0x1d, 0xa0, 0x32, 0x37, 0x7f, 0x21, 0xc2, 0xb6, 0xf6, 0xd9, 0x1a, 0xf8, 0xdd, 0x1b,
};

extern const ByteSpan sTestCert_DAC_FFF2_8004_001F_FB_PrivateKey = ByteSpan(sTestCert_DAC_FFF2_8004_001F_FB_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8004-0020-ValInPast-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8004_0020_ValInPast_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xcf, 0x30, 0x82, 0x01, 0x75, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x1f, 0xdd, 0x40, 0x7e, 0xc6, 0xbf,
    0xca, 0xc1, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x2e, 0x31, 0x2c, 0x30, 0x2a, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x23, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x20, 0x4d, 0x76, 0x69, 0x64, 0x3a, 0x46, 0x46, 0x46, 0x32, 0x20, 0x4d, 0x70, 0x69, 0x64, 0x3a, 0x38, 0x30, 0x30, 0x34, 0x30,
    0x1e, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x17, 0x0d, 0x32, 0x32, 0x30,
    0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x32, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b, 0x06, 0x03, 0x55, 0x04, 0x03,
    0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41, 0x43, 0x20, 0x30, 0x30, 0x32,
    0x30, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46,
    0x46, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38,
    0x30, 0x30, 0x34, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48,
    0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x15, 0x2e, 0xfe, 0x67, 0xbc, 0xbb, 0xdb, 0x8c, 0x3f, 0xa4, 0xb4, 0x57,
    0xfa, 0x3e, 0xfd, 0xc5, 0xfa, 0x44, 0xd0, 0xc2, 0x17, 0x96, 0xe7, 0x4b, 0xa5, 0x85, 0xd1, 0x31, 0x18, 0xc2, 0x46, 0xdb, 0xed,
    0x0a, 0xc0, 0x58, 0x7a, 0x83, 0x48, 0x7c, 0x4e, 0x48, 0x65, 0x0e, 0xe8, 0x6d, 0xea, 0xb3, 0x81, 0x80, 0x80, 0x91, 0x9a, 0x3a,
    0xe2, 0x5a, 0x4f, 0xa3, 0xac, 0x80, 0x8e, 0xaf, 0x11, 0x4b, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13,
    0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02,
    0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x9c, 0xbf, 0x0b, 0xd8, 0x3b, 0xc6, 0x5e, 0x4f,
    0x40, 0x2f, 0x15, 0xa1, 0xf7, 0x46, 0xfd, 0x17, 0x7d, 0x43, 0xe7, 0xa4, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18,
    0x30, 0x16, 0x80, 0x14, 0x15, 0x6e, 0xcd, 0x8c, 0x14, 0xbe, 0x80, 0xb4, 0x10, 0xae, 0xe4, 0x02, 0xa3, 0xf2, 0x33, 0x17, 0xe7,
    0x34, 0x06, 0x74, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x48, 0x00, 0x30, 0x45, 0x02,
    0x21, 0x00, 0xa6, 0xc1, 0xde, 0x4e, 0x25, 0x0d, 0xa1, 0x1a, 0xcb, 0xf7, 0x34, 0xfa, 0x76, 0x9e, 0x50, 0x8b, 0xd1, 0x3e, 0xf6,
    0x1e, 0xef, 0x17, 0x0f, 0xf8, 0x85, 0x5c, 0x57, 0x23, 0x4d, 0x28, 0x60, 0x89, 0x02, 0x20, 0x65, 0xe0, 0xb2, 0x18, 0x06, 0xc0,
    0xdd, 0xdb, 0xd5, 0x98, 0x7f, 0xed, 0x44, 0xc8, 0xee, 0x70, 0x89, 0xb2, 0x47, 0x15, 0xcc, 0xdd, 0xd8, 0xa6, 0x38, 0x3e, 0xf6,
    0x2f, 0xb7, 0xb4, 0xdc, 0xcd,
};

extern const ByteSpan sTestCert_DAC_FFF2_8004_0020_ValInPast_Cert = ByteSpan(sTestCert_DAC_FFF2_8004_0020_ValInPast_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8004_0020_ValInPast_SKID_Array[] = {
    0x9C, 0xBF, 0x0B, 0xD8, 0x3B, 0xC6, 0x5E, 0x4F, 0x40, 0x2F, 0x15, 0xA1, 0xF7, 0x46, 0xFD, 0x17, 0x7D, 0x43, 0xE7, 0xA4,
};

extern const ByteSpan sTestCert_DAC_FFF2_8004_0020_ValInPast_SKID = ByteSpan(sTestCert_DAC_FFF2_8004_0020_ValInPast_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8004-0020-ValInPast-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8004_0020_ValInPast_PublicKey_Array[] = {
    0x04, 0x15, 0x2e, 0xfe, 0x67, 0xbc, 0xbb, 0xdb, 0x8c, 0x3f, 0xa4, 0xb4, 0x57, 0xfa, 0x3e, 0xfd, 0xc5,
    0xfa, 0x44, 0xd0, 0xc2, 0x17, 0x96, 0xe7, 0x4b, 0xa5, 0x85, 0xd1, 0x31, 0x18, 0xc2, 0x46, 0xdb, 0xed,
    0x0a, 0xc0, 0x58, 0x7a, 0x83, 0x48, 0x7c, 0x4e, 0x48, 0x65, 0x0e, 0xe8, 0x6d, 0xea, 0xb3, 0x81, 0x80,
    0x80, 0x91, 0x9a, 0x3a, 0xe2, 0x5a, 0x4f, 0xa3, 0xac, 0x80, 0x8e, 0xaf, 0x11, 0x4b,
};

extern const ByteSpan sTestCert_DAC_FFF2_8004_0020_ValInPast_PublicKey =
    ByteSpan(sTestCert_DAC_FFF2_8004_0020_ValInPast_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8004_0020_ValInPast_PrivateKey_Array[] = {
    0x04, 0x80, 0x12, 0x84, 0x16, 0x1e, 0x02, 0xef, 0x05, 0x43, 0xbd, 0x2c, 0xea, 0x6a, 0x5a, 0xf2,
    0x3c, 0x97, 0x31, 0x50, 0x4a, 0xeb, 0x8a, 0x89, 0x79, 0x2c, 0x01, 0xfa, 0xec, 0x8b, 0xfe, 0x06,
};

extern const ByteSpan sTestCert_DAC_FFF2_8004_0020_ValInPast_PrivateKey =
    ByteSpan(sTestCert_DAC_FFF2_8004_0020_ValInPast_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8004-0021-ValInFuture-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8004_0021_ValInFuture_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xd2, 0x30, 0x82, 0x01, 0x77, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x76, 0x7c, 0x6e, 0x56, 0xf6, 0xe2,
    0x60, 0x41, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x2e, 0x31, 0x2c, 0x30, 0x2a, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x23, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x20, 0x4d, 0x76, 0x69, 0x64, 0x3a, 0x46, 0x46, 0x46, 0x32, 0x20, 0x4d, 0x70, 0x69, 0x64, 0x3a, 0x38, 0x30, 0x30, 0x34, 0x30,
    0x20, 0x17, 0x0d, 0x33, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x39, 0x39, 0x39,
    0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b, 0x06, 0x03, 0x55,
    0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41, 0x43, 0x20, 0x30,
    0x30, 0x32, 0x31, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04,
    0x46, 0x46, 0x46, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c,
    0x04, 0x38, 0x30, 0x30, 0x34, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a,
    0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xb5, 0xd3, 0x71, 0x26, 0xf5, 0x2e, 0x5c, 0x85, 0xf6, 0x82,
    0x64, 0x5f, 0x7f, 0x3d, 0xd9, 0x6f, 0x4c, 0xed, 0xa7, 0xef, 0x67, 0xd9, 0x3c, 0x0a, 0x06, 0x83, 0x4e, 0xa3, 0xe7, 0x9e, 0xf3,
    0x6e, 0x25, 0x0f, 0xd3, 0x09, 0x76, 0xb3, 0xd1, 0x19, 0xbb, 0x09, 0xad, 0x46, 0x25, 0x43, 0x88, 0xb7, 0x84, 0xae, 0x05, 0xf3,
    0x31, 0x00, 0xa1, 0x72, 0x60, 0xd7, 0x27, 0x8a, 0xe7, 0xc2, 0xe0, 0x98, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c, 0x06, 0x03, 0x55,
    0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04,
    0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x08, 0x84, 0xff, 0x0d, 0x25, 0x23,
    0x97, 0x7f, 0xd9, 0xb6, 0x80, 0x2f, 0x9e, 0x7d, 0x7c, 0xd7, 0x40, 0x1f, 0x5b, 0xf1, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23,
    0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x15, 0x6e, 0xcd, 0x8c, 0x14, 0xbe, 0x80, 0xb4, 0x10, 0xae, 0xe4, 0x02, 0xa3, 0xf2, 0x33,
    0x17, 0xe7, 0x34, 0x06, 0x74, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x49, 0x00, 0x30,
    0x46, 0x02, 0x21, 0x00, 0xde, 0x03, 0x54, 0x3d, 0x06, 0x45, 0xe1, 0xa2, 0x7e, 0x05, 0xbb, 0xe5, 0x8b, 0x35, 0x01, 0xcf, 0x06,
    0xa7, 0xc9, 0xa2, 0x08, 0x03, 0xc9, 0xa1, 0x7b, 0x7e, 0xc2, 0xe3, 0x56, 0x07, 0xa6, 0xab, 0x02, 0x21, 0x00, 0xe1, 0xc0, 0x94,
    0x9e, 0x6a, 0x65, 0xf6, 0x00, 0x38, 0x8f, 0x4e, 0x08, 0xbc, 0x45, 0xde, 0xaa, 0x32, 0x86, 0x75, 0x30, 0xdc, 0xd5, 0x8f, 0x0c,
    0xd0, 0xe5, 0xc4, 0xe7, 0x39, 0xbc, 0x32, 0xe8,
};

extern const ByteSpan sTestCert_DAC_FFF2_8004_0021_ValInFuture_Cert = ByteSpan(sTestCert_DAC_FFF2_8004_0021_ValInFuture_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8004_0021_ValInFuture_SKID_Array[] = {
    0x08, 0x84, 0xFF, 0x0D, 0x25, 0x23, 0x97, 0x7F, 0xD9, 0xB6, 0x80, 0x2F, 0x9E, 0x7D, 0x7C, 0xD7, 0x40, 0x1F, 0x5B, 0xF1,
};

extern const ByteSpan sTestCert_DAC_FFF2_8004_0021_ValInFuture_SKID = ByteSpan(sTestCert_DAC_FFF2_8004_0021_ValInFuture_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8004-0021-ValInFuture-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8004_0021_ValInFuture_PublicKey_Array[] = {
    0x04, 0xb5, 0xd3, 0x71, 0x26, 0xf5, 0x2e, 0x5c, 0x85, 0xf6, 0x82, 0x64, 0x5f, 0x7f, 0x3d, 0xd9, 0x6f,
    0x4c, 0xed, 0xa7, 0xef, 0x67, 0xd9, 0x3c, 0x0a, 0x06, 0x83, 0x4e, 0xa3, 0xe7, 0x9e, 0xf3, 0x6e, 0x25,
    0x0f, 0xd3, 0x09, 0x76, 0xb3, 0xd1, 0x19, 0xbb, 0x09, 0xad, 0x46, 0x25, 0x43, 0x88, 0xb7, 0x84, 0xae,
    0x05, 0xf3, 0x31, 0x00, 0xa1, 0x72, 0x60, 0xd7, 0x27, 0x8a, 0xe7, 0xc2, 0xe0, 0x98,
};

extern const ByteSpan sTestCert_DAC_FFF2_8004_0021_ValInFuture_PublicKey =
    ByteSpan(sTestCert_DAC_FFF2_8004_0021_ValInFuture_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8004_0021_ValInFuture_PrivateKey_Array[] = {
    0xf3, 0x5b, 0x53, 0xe2, 0x75, 0x00, 0x10, 0xa9, 0x74, 0xd5, 0x23, 0xd4, 0x2a, 0x41, 0xe9, 0xbb,
    0xb9, 0x27, 0x8c, 0x82, 0x3d, 0xdf, 0x95, 0xbc, 0x01, 0xc3, 0xb2, 0xb0, 0x33, 0xab, 0x38, 0x86,
};

extern const ByteSpan sTestCert_DAC_FFF2_8004_0021_ValInFuture_PrivateKey =
    ByteSpan(sTestCert_DAC_FFF2_8004_0021_ValInFuture_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8004-0030-Val1SecBefore-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8004_0030_Val1SecBefore_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xce, 0x30, 0x82, 0x01, 0x75, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x70, 0x95, 0x31, 0xc8, 0x6b, 0x69,
    0x8d, 0x5a, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x2e, 0x31, 0x2c, 0x30, 0x2a, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x23, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x20, 0x4d, 0x76, 0x69, 0x64, 0x3a, 0x46, 0x46, 0x46, 0x32, 0x20, 0x4d, 0x70, 0x69, 0x64, 0x3a, 0x38, 0x30, 0x30, 0x34, 0x30,
    0x1e, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x32, 0x5a, 0x17, 0x0d, 0x32, 0x32, 0x30,
    0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x31, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b, 0x06, 0x03, 0x55, 0x04, 0x03,
    0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41, 0x43, 0x20, 0x30, 0x30, 0x33,
    0x30, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46,
    0x46, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38,
    0x30, 0x30, 0x34, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48,
    0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x3a, 0x95, 0x20, 0x98, 0xbf, 0x5c, 0x27, 0xf1, 0x9a, 0x3d, 0x23, 0xf7,
    0xb8, 0x48, 0xba, 0x3a, 0x23, 0x17, 0xef, 0xde, 0x47, 0x58, 0xa5, 0x83, 0xd2, 0xaf, 0x8f, 0x02, 0x33, 0x67, 0x3f, 0x12, 0xad,
    0x17, 0x9f, 0x65, 0xc6, 0xc8, 0xba, 0x76, 0x8a, 0x3b, 0x2c, 0x4d, 0xc3, 0x20, 0x90, 0x0b, 0x51, 0xab, 0x90, 0x8d, 0x8a, 0xc3,
    0xec, 0xf0, 0x73, 0x5b, 0x98, 0xfa, 0xcf, 0x55, 0xb2, 0xc9, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13,
    0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02,
    0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xb4, 0x7e, 0xa8, 0xff, 0x2e, 0xaa, 0x9e, 0x1e,
    0xcd, 0x96, 0x35, 0xa7, 0xeb, 0xd7, 0x64, 0xd7, 0x09, 0xf0, 0x4a, 0x60, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18,
    0x30, 0x16, 0x80, 0x14, 0x15, 0x6e, 0xcd, 0x8c, 0x14, 0xbe, 0x80, 0xb4, 0x10, 0xae, 0xe4, 0x02, 0xa3, 0xf2, 0x33, 0x17, 0xe7,
    0x34, 0x06, 0x74, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x47, 0x00, 0x30, 0x44, 0x02,
    0x20, 0x21, 0x57, 0x9a, 0xc9, 0x23, 0x61, 0xab, 0xe8, 0x5a, 0x24, 0x96, 0x23, 0xae, 0x2c, 0x0a, 0x85, 0xd9, 0x25, 0xdd, 0xd8,
    0x0a, 0xff, 0x80, 0x2a, 0xc9, 0x18, 0x0f, 0xa5, 0xa6, 0xb3, 0x01, 0x29, 0x02, 0x20, 0x72, 0x42, 0x25, 0x4e, 0xb6, 0x42, 0xdd,
    0xa6, 0x42, 0x60, 0xdf, 0x5b, 0x2e, 0xa9, 0x22, 0x11, 0xbd, 0x84, 0x1b, 0x8b, 0xf3, 0x51, 0x3d, 0xe6, 0x00, 0x63, 0xa1, 0x52,
    0x58, 0x48, 0xca, 0x3d,
};

extern const ByteSpan sTestCert_DAC_FFF2_8004_0030_Val1SecBefore_Cert =
    ByteSpan(sTestCert_DAC_FFF2_8004_0030_Val1SecBefore_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8004_0030_Val1SecBefore_SKID_Array[] = {
    0xB4, 0x7E, 0xA8, 0xFF, 0x2E, 0xAA, 0x9E, 0x1E, 0xCD, 0x96, 0x35, 0xA7, 0xEB, 0xD7, 0x64, 0xD7, 0x09, 0xF0, 0x4A, 0x60,
};

extern const ByteSpan sTestCert_DAC_FFF2_8004_0030_Val1SecBefore_SKID =
    ByteSpan(sTestCert_DAC_FFF2_8004_0030_Val1SecBefore_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8004-0030-Val1SecBefore-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8004_0030_Val1SecBefore_PublicKey_Array[] = {
    0x04, 0x3a, 0x95, 0x20, 0x98, 0xbf, 0x5c, 0x27, 0xf1, 0x9a, 0x3d, 0x23, 0xf7, 0xb8, 0x48, 0xba, 0x3a,
    0x23, 0x17, 0xef, 0xde, 0x47, 0x58, 0xa5, 0x83, 0xd2, 0xaf, 0x8f, 0x02, 0x33, 0x67, 0x3f, 0x12, 0xad,
    0x17, 0x9f, 0x65, 0xc6, 0xc8, 0xba, 0x76, 0x8a, 0x3b, 0x2c, 0x4d, 0xc3, 0x20, 0x90, 0x0b, 0x51, 0xab,
    0x90, 0x8d, 0x8a, 0xc3, 0xec, 0xf0, 0x73, 0x5b, 0x98, 0xfa, 0xcf, 0x55, 0xb2, 0xc9,
};

extern const ByteSpan sTestCert_DAC_FFF2_8004_0030_Val1SecBefore_PublicKey =
    ByteSpan(sTestCert_DAC_FFF2_8004_0030_Val1SecBefore_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8004_0030_Val1SecBefore_PrivateKey_Array[] = {
    0x16, 0x2f, 0x5d, 0x2f, 0x12, 0x33, 0x20, 0x37, 0xcd, 0x82, 0xad, 0x83, 0xcd, 0x22, 0xbb, 0x87,
    0x80, 0x92, 0xab, 0x1d, 0xad, 0x95, 0x87, 0xad, 0xdf, 0xc1, 0x0a, 0xba, 0x32, 0x0a, 0x6d, 0xa3,
};

extern const ByteSpan sTestCert_DAC_FFF2_8004_0030_Val1SecBefore_PrivateKey =
    ByteSpan(sTestCert_DAC_FFF2_8004_0030_Val1SecBefore_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8005-0022-ValInPast-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8005_0022_ValInPast_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xe8, 0x30, 0x82, 0x01, 0x8d, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x05, 0x29, 0x02, 0xb9, 0x51, 0x66,
    0xb6, 0x38, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30,
    0x30, 0x35, 0x30, 0x1e, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x17, 0x0d,
    0x32, 0x32, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x32, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b, 0x06, 0x03,
    0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41, 0x43, 0x20,
    0x30, 0x30, 0x32, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c,
    0x04, 0x46, 0x46, 0x46, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02,
    0x0c, 0x04, 0x38, 0x30, 0x30, 0x35, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08,
    0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x5b, 0xc9, 0xe3, 0xea, 0xee, 0x45, 0x1f, 0xc8, 0x18,
    0xc8, 0xd2, 0x4a, 0x71, 0xab, 0x39, 0x4a, 0xbd, 0x39, 0x4b, 0xe1, 0xdf, 0x5b, 0x8e, 0xda, 0xe0, 0x72, 0x94, 0x18, 0x94, 0x57,
    0x26, 0xcc, 0x0f, 0x0c, 0x74, 0x2f, 0xf4, 0xcb, 0x01, 0xa6, 0xdb, 0x9c, 0x11, 0x0d, 0xe6, 0xed, 0x16, 0xbc, 0xe1, 0x11, 0x92,
    0xad, 0x55, 0x7c, 0xdc, 0xb1, 0x76, 0xf8, 0xae, 0x9b, 0x52, 0x43, 0x13, 0x7d, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c, 0x06, 0x03,
    0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04,
    0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x53, 0xd6, 0xdb, 0xef, 0xce,
    0xc8, 0x53, 0xaf, 0x5d, 0x58, 0x49, 0xec, 0x92, 0xe8, 0x29, 0x6a, 0x85, 0xa5, 0xca, 0x20, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d,
    0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xda, 0xd4, 0x67, 0x6f, 0x50, 0x6d, 0x45, 0x43, 0xdb, 0x94, 0x76, 0xd6, 0x18, 0x69,
    0x1c, 0x67, 0x49, 0x72, 0x8d, 0x08, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x49, 0x00,
    0x30, 0x46, 0x02, 0x21, 0x00, 0xd1, 0xe9, 0x73, 0xff, 0xfa, 0x3a, 0xfa, 0xb5, 0x2e, 0xb6, 0x68, 0x32, 0x51, 0x74, 0x2e, 0x77,
    0x34, 0x49, 0xaa, 0x1f, 0xf8, 0xd7, 0xbd, 0xd0, 0x08, 0x3b, 0x80, 0x35, 0xd8, 0x91, 0x44, 0xf4, 0x02, 0x21, 0x00, 0x94, 0xf3,
    0x9d, 0x0a, 0x57, 0x59, 0xb5, 0x33, 0x57, 0x2f, 0xeb, 0x44, 0x1d, 0xba, 0x56, 0x9f, 0x53, 0xb5, 0x2b, 0xa3, 0x99, 0x00, 0x6c,
    0xab, 0xad, 0x14, 0x6f, 0x4e, 0x0f, 0x95, 0xa6, 0x19,
};

extern const ByteSpan sTestCert_DAC_FFF2_8005_0022_ValInPast_Cert = ByteSpan(sTestCert_DAC_FFF2_8005_0022_ValInPast_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8005_0022_ValInPast_SKID_Array[] = {
    0x53, 0xD6, 0xDB, 0xEF, 0xCE, 0xC8, 0x53, 0xAF, 0x5D, 0x58, 0x49, 0xEC, 0x92, 0xE8, 0x29, 0x6A, 0x85, 0xA5, 0xCA, 0x20,
};

extern const ByteSpan sTestCert_DAC_FFF2_8005_0022_ValInPast_SKID = ByteSpan(sTestCert_DAC_FFF2_8005_0022_ValInPast_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8005-0022-ValInPast-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8005_0022_ValInPast_PublicKey_Array[] = {
    0x04, 0x5b, 0xc9, 0xe3, 0xea, 0xee, 0x45, 0x1f, 0xc8, 0x18, 0xc8, 0xd2, 0x4a, 0x71, 0xab, 0x39, 0x4a,
    0xbd, 0x39, 0x4b, 0xe1, 0xdf, 0x5b, 0x8e, 0xda, 0xe0, 0x72, 0x94, 0x18, 0x94, 0x57, 0x26, 0xcc, 0x0f,
    0x0c, 0x74, 0x2f, 0xf4, 0xcb, 0x01, 0xa6, 0xdb, 0x9c, 0x11, 0x0d, 0xe6, 0xed, 0x16, 0xbc, 0xe1, 0x11,
    0x92, 0xad, 0x55, 0x7c, 0xdc, 0xb1, 0x76, 0xf8, 0xae, 0x9b, 0x52, 0x43, 0x13, 0x7d,
};

extern const ByteSpan sTestCert_DAC_FFF2_8005_0022_ValInPast_PublicKey =
    ByteSpan(sTestCert_DAC_FFF2_8005_0022_ValInPast_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8005_0022_ValInPast_PrivateKey_Array[] = {
    0xd0, 0xef, 0xa1, 0x44, 0x72, 0x87, 0x05, 0x7b, 0x07, 0x6d, 0x49, 0xe4, 0xcb, 0x16, 0x51, 0x52,
    0x6d, 0x4d, 0x83, 0x0b, 0x34, 0xf4, 0xa1, 0xed, 0x7b, 0x80, 0x0d, 0x3a, 0xf9, 0xf2, 0x94, 0xc5,
};

extern const ByteSpan sTestCert_DAC_FFF2_8005_0022_ValInPast_PrivateKey =
    ByteSpan(sTestCert_DAC_FFF2_8005_0022_ValInPast_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8005-0023-ValInFuture-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8005_0023_ValInFuture_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xea, 0x30, 0x82, 0x01, 0x8f, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x58, 0xa7, 0x95, 0x53, 0x86, 0x2b,
    0x0d, 0x74, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30,
    0x30, 0x35, 0x30, 0x20, 0x17, 0x0d, 0x33, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f,
    0x39, 0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b,
    0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41,
    0x43, 0x20, 0x30, 0x30, 0x32, 0x33, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02,
    0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c,
    0x02, 0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x35, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01,
    0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x61, 0xdb, 0xae, 0x67, 0x1e, 0x63, 0x81,
    0x42, 0xd3, 0xb5, 0x4d, 0x05, 0xc8, 0x87, 0x06, 0x85, 0x8c, 0xd4, 0x7f, 0xef, 0x4b, 0x05, 0x91, 0xa7, 0xc1, 0xfd, 0x1d, 0x76,
    0x99, 0xcf, 0xe0, 0xdd, 0x6b, 0x23, 0xcf, 0x68, 0x5d, 0xcb, 0x6a, 0xc6, 0x3f, 0x3d, 0x6b, 0x9d, 0x7b, 0xfe, 0x5a, 0x5e, 0x5c,
    0x8a, 0xef, 0x42, 0x00, 0xc4, 0x2a, 0x51, 0xe3, 0x88, 0x31, 0x7d, 0xad, 0x95, 0xd4, 0x6e, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c,
    0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01,
    0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xc7, 0x64, 0xa5,
    0x6a, 0x84, 0xa0, 0xa8, 0x16, 0x88, 0x01, 0x16, 0x7c, 0x6a, 0x95, 0xd0, 0x9c, 0xa2, 0x53, 0x6c, 0x76, 0x30, 0x1f, 0x06, 0x03,
    0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x29, 0x38, 0xba, 0xc7, 0xf6, 0x23, 0xe1, 0x83, 0x26, 0x43, 0xa5, 0xeb,
    0x76, 0x2b, 0xaf, 0xd4, 0x0a, 0x62, 0xc0, 0x21, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03,
    0x49, 0x00, 0x30, 0x46, 0x02, 0x21, 0x00, 0xf6, 0xc5, 0x37, 0x9f, 0x4f, 0xdd, 0x73, 0x19, 0x2a, 0xc3, 0x05, 0xf1, 0x00, 0x58,
    0xc7, 0xa3, 0x20, 0xbb, 0xe9, 0xca, 0x60, 0xb3, 0x91, 0xfb, 0xa6, 0x00, 0xc3, 0xf2, 0x24, 0x6b, 0x8a, 0x7c, 0x02, 0x21, 0x00,
    0xc5, 0xbb, 0x40, 0x5e, 0x5d, 0x36, 0xce, 0xac, 0x84, 0x45, 0x3e, 0x1f, 0xcb, 0x76, 0xe7, 0x34, 0x0f, 0x01, 0xe4, 0x72, 0x08,
    0x6a, 0x98, 0x01, 0x33, 0x05, 0x0d, 0xa7, 0x19, 0x57, 0xd4, 0xc7,
};

extern const ByteSpan sTestCert_DAC_FFF2_8005_0023_ValInFuture_Cert = ByteSpan(sTestCert_DAC_FFF2_8005_0023_ValInFuture_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8005_0023_ValInFuture_SKID_Array[] = {
    0xC7, 0x64, 0xA5, 0x6A, 0x84, 0xA0, 0xA8, 0x16, 0x88, 0x01, 0x16, 0x7C, 0x6A, 0x95, 0xD0, 0x9C, 0xA2, 0x53, 0x6C, 0x76,
};

extern const ByteSpan sTestCert_DAC_FFF2_8005_0023_ValInFuture_SKID = ByteSpan(sTestCert_DAC_FFF2_8005_0023_ValInFuture_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8005-0023-ValInFuture-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8005_0023_ValInFuture_PublicKey_Array[] = {
    0x04, 0x61, 0xdb, 0xae, 0x67, 0x1e, 0x63, 0x81, 0x42, 0xd3, 0xb5, 0x4d, 0x05, 0xc8, 0x87, 0x06, 0x85,
    0x8c, 0xd4, 0x7f, 0xef, 0x4b, 0x05, 0x91, 0xa7, 0xc1, 0xfd, 0x1d, 0x76, 0x99, 0xcf, 0xe0, 0xdd, 0x6b,
    0x23, 0xcf, 0x68, 0x5d, 0xcb, 0x6a, 0xc6, 0x3f, 0x3d, 0x6b, 0x9d, 0x7b, 0xfe, 0x5a, 0x5e, 0x5c, 0x8a,
    0xef, 0x42, 0x00, 0xc4, 0x2a, 0x51, 0xe3, 0x88, 0x31, 0x7d, 0xad, 0x95, 0xd4, 0x6e,
};

extern const ByteSpan sTestCert_DAC_FFF2_8005_0023_ValInFuture_PublicKey =
    ByteSpan(sTestCert_DAC_FFF2_8005_0023_ValInFuture_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8005_0023_ValInFuture_PrivateKey_Array[] = {
    0x5d, 0x79, 0x0b, 0x36, 0x3a, 0x00, 0x77, 0x82, 0x35, 0x11, 0x2a, 0xe5, 0xd5, 0x92, 0xeb, 0x11,
    0xaa, 0xaa, 0x5c, 0xaa, 0xce, 0x61, 0xf0, 0xc2, 0xf4, 0x7e, 0xa3, 0x3b, 0x87, 0x49, 0x0e, 0xbf,
};

extern const ByteSpan sTestCert_DAC_FFF2_8005_0023_ValInFuture_PrivateKey =
    ByteSpan(sTestCert_DAC_FFF2_8005_0023_ValInFuture_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8005-0032-Val1SecBefore-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8005_0032_Val1SecBefore_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xe7, 0x30, 0x82, 0x01, 0x8d, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x2a, 0x5c, 0xaa, 0xf0, 0x5f, 0x55,
    0xc8, 0xfd, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30,
    0x30, 0x35, 0x30, 0x1e, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x32, 0x5a, 0x17, 0x0d,
    0x32, 0x32, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x31, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b, 0x06, 0x03,
    0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41, 0x43, 0x20,
    0x30, 0x30, 0x33, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c,
    0x04, 0x46, 0x46, 0x46, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02,
    0x0c, 0x04, 0x38, 0x30, 0x30, 0x35, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08,
    0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x3a, 0x1c, 0xde, 0x2a, 0x2f, 0x0e, 0x20, 0xc6, 0xcf,
    0xb8, 0x3b, 0x40, 0x16, 0xa6, 0x42, 0xa2, 0x19, 0x5c, 0x4d, 0x1d, 0xc3, 0xce, 0xe2, 0x79, 0xbc, 0x51, 0xca, 0x7a, 0x40, 0x77,
    0xf5, 0x64, 0x5f, 0x53, 0x45, 0xcd, 0x48, 0x97, 0x51, 0xf1, 0x85, 0x80, 0x84, 0x64, 0xe7, 0xee, 0x29, 0x65, 0xb0, 0x67, 0x72,
    0x70, 0x2c, 0x1a, 0x6d, 0x26, 0x56, 0x17, 0x6c, 0xc8, 0x3b, 0xb4, 0xde, 0x2e, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c, 0x06, 0x03,
    0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04,
    0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x37, 0x6a, 0xcc, 0x5c, 0x8d,
    0x7a, 0x75, 0xe0, 0x9f, 0xef, 0xe9, 0xd8, 0xf3, 0xf5, 0xe5, 0xbb, 0x79, 0xc8, 0x87, 0x01, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d,
    0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xb3, 0x0e, 0x90, 0x53, 0x42, 0x6f, 0x02, 0x0d, 0x13, 0x91, 0x1b, 0x03, 0xcb, 0x57,
    0x8d, 0xcb, 0x4a, 0xab, 0x9a, 0x52, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x48, 0x00,
    0x30, 0x45, 0x02, 0x20, 0x6c, 0x04, 0xe4, 0xf2, 0x91, 0x78, 0xf6, 0xa8, 0xb8, 0x88, 0xf4, 0x40, 0x50, 0x48, 0xc6, 0xc2, 0x5e,
    0x37, 0xf5, 0x42, 0x49, 0x98, 0x14, 0x18, 0x7f, 0xb4, 0xed, 0x30, 0x5c, 0x7b, 0xfd, 0x6c, 0x02, 0x21, 0x00, 0x89, 0x92, 0xd9,
    0xe5, 0x91, 0xf8, 0x1e, 0x5b, 0xfe, 0x20, 0x10, 0x7e, 0xb0, 0xa3, 0xf4, 0xe9, 0x74, 0x8f, 0xd3, 0x82, 0x02, 0xa2, 0xc5, 0x98,
    0x98, 0xe3, 0x11, 0x05, 0xfd, 0xcd, 0x0b, 0xc1,
};

extern const ByteSpan sTestCert_DAC_FFF2_8005_0032_Val1SecBefore_Cert =
    ByteSpan(sTestCert_DAC_FFF2_8005_0032_Val1SecBefore_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8005_0032_Val1SecBefore_SKID_Array[] = {
    0x37, 0x6A, 0xCC, 0x5C, 0x8D, 0x7A, 0x75, 0xE0, 0x9F, 0xEF, 0xE9, 0xD8, 0xF3, 0xF5, 0xE5, 0xBB, 0x79, 0xC8, 0x87, 0x01,
};

extern const ByteSpan sTestCert_DAC_FFF2_8005_0032_Val1SecBefore_SKID =
    ByteSpan(sTestCert_DAC_FFF2_8005_0032_Val1SecBefore_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8005-0032-Val1SecBefore-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8005_0032_Val1SecBefore_PublicKey_Array[] = {
    0x04, 0x3a, 0x1c, 0xde, 0x2a, 0x2f, 0x0e, 0x20, 0xc6, 0xcf, 0xb8, 0x3b, 0x40, 0x16, 0xa6, 0x42, 0xa2,
    0x19, 0x5c, 0x4d, 0x1d, 0xc3, 0xce, 0xe2, 0x79, 0xbc, 0x51, 0xca, 0x7a, 0x40, 0x77, 0xf5, 0x64, 0x5f,
    0x53, 0x45, 0xcd, 0x48, 0x97, 0x51, 0xf1, 0x85, 0x80, 0x84, 0x64, 0xe7, 0xee, 0x29, 0x65, 0xb0, 0x67,
    0x72, 0x70, 0x2c, 0x1a, 0x6d, 0x26, 0x56, 0x17, 0x6c, 0xc8, 0x3b, 0xb4, 0xde, 0x2e,
};

extern const ByteSpan sTestCert_DAC_FFF2_8005_0032_Val1SecBefore_PublicKey =
    ByteSpan(sTestCert_DAC_FFF2_8005_0032_Val1SecBefore_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8005_0032_Val1SecBefore_PrivateKey_Array[] = {
    0x85, 0xfc, 0x34, 0x74, 0xcd, 0xcf, 0xe2, 0xa9, 0xa7, 0x95, 0x38, 0x15, 0xbc, 0x0f, 0x4e, 0x47,
    0x46, 0x3c, 0x2e, 0xc4, 0x0a, 0x90, 0x20, 0x73, 0x67, 0x4e, 0x94, 0x1a, 0x18, 0xdb, 0x69, 0x99,
};

extern const ByteSpan sTestCert_DAC_FFF2_8005_0032_Val1SecBefore_PrivateKey =
    ByteSpan(sTestCert_DAC_FFF2_8005_0032_Val1SecBefore_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8005-0033-Val1SecBefore-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8005_0033_Val1SecBefore_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xe8, 0x30, 0x82, 0x01, 0x8f, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x01, 0x21, 0xad, 0x5a, 0x24, 0x56,
    0x42, 0xd4, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30,
    0x30, 0x35, 0x30, 0x20, 0x17, 0x0d, 0x33, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x32, 0x5a, 0x18, 0x0f,
    0x39, 0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b,
    0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41,
    0x43, 0x20, 0x30, 0x30, 0x33, 0x33, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02,
    0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c,
    0x02, 0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x35, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01,
    0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x44, 0xf3, 0xc4, 0xf5, 0x82, 0x5f, 0x74,
    0x35, 0x77, 0x83, 0x85, 0x89, 0x2a, 0xe5, 0xd0, 0x9d, 0x68, 0x6f, 0x4a, 0x5e, 0xd9, 0x2a, 0x72, 0x74, 0xea, 0x2b, 0xa7, 0xc9,
    0x5f, 0x08, 0xaf, 0xc9, 0x40, 0xe6, 0x92, 0xf2, 0x6d, 0x87, 0x23, 0x5b, 0x12, 0x7a, 0xb7, 0x2b, 0x4f, 0x1b, 0xa6, 0xc6, 0xde,
    0x90, 0x72, 0x2e, 0x0c, 0xd0, 0xb6, 0xd7, 0x69, 0xd3, 0xc9, 0xe5, 0x59, 0xbb, 0x3d, 0x2e, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c,
    0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01,
    0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xa6, 0xd2, 0x19,
    0x27, 0xd5, 0x62, 0x73, 0xb3, 0x19, 0xca, 0x04, 0x6d, 0x87, 0x82, 0x7c, 0x06, 0x31, 0x64, 0xc4, 0x9b, 0x30, 0x1f, 0x06, 0x03,
    0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x29, 0x38, 0xba, 0xc7, 0xf6, 0x23, 0xe1, 0x83, 0x26, 0x43, 0xa5, 0xeb,
    0x76, 0x2b, 0xaf, 0xd4, 0x0a, 0x62, 0xc0, 0x21, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03,
    0x47, 0x00, 0x30, 0x44, 0x02, 0x20, 0x3b, 0x3a, 0x0f, 0xb5, 0x19, 0x5a, 0xbe, 0xcf, 0x2d, 0x0e, 0xb3, 0xf5, 0x76, 0xb1, 0x98,
    0xd2, 0x3a, 0x1c, 0x65, 0x6f, 0x87, 0x72, 0xb7, 0x8f, 0x67, 0x3b, 0x3f, 0xe3, 0x4c, 0xcc, 0xc2, 0xb1, 0x02, 0x20, 0x28, 0x12,
    0x19, 0x6a, 0x4b, 0x2e, 0x0d, 0xa2, 0x8d, 0xa4, 0xd9, 0x10, 0x5a, 0xe8, 0x0c, 0x42, 0x4b, 0x44, 0x02, 0xa6, 0xda, 0xaa, 0x83,
    0x1b, 0xa5, 0x29, 0x94, 0x37, 0xda, 0x75, 0x2f, 0x88,
};

extern const ByteSpan sTestCert_DAC_FFF2_8005_0033_Val1SecBefore_Cert =
    ByteSpan(sTestCert_DAC_FFF2_8005_0033_Val1SecBefore_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8005_0033_Val1SecBefore_SKID_Array[] = {
    0xA6, 0xD2, 0x19, 0x27, 0xD5, 0x62, 0x73, 0xB3, 0x19, 0xCA, 0x04, 0x6D, 0x87, 0x82, 0x7C, 0x06, 0x31, 0x64, 0xC4, 0x9B,
};

extern const ByteSpan sTestCert_DAC_FFF2_8005_0033_Val1SecBefore_SKID =
    ByteSpan(sTestCert_DAC_FFF2_8005_0033_Val1SecBefore_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8005-0033-Val1SecBefore-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8005_0033_Val1SecBefore_PublicKey_Array[] = {
    0x04, 0x44, 0xf3, 0xc4, 0xf5, 0x82, 0x5f, 0x74, 0x35, 0x77, 0x83, 0x85, 0x89, 0x2a, 0xe5, 0xd0, 0x9d,
    0x68, 0x6f, 0x4a, 0x5e, 0xd9, 0x2a, 0x72, 0x74, 0xea, 0x2b, 0xa7, 0xc9, 0x5f, 0x08, 0xaf, 0xc9, 0x40,
    0xe6, 0x92, 0xf2, 0x6d, 0x87, 0x23, 0x5b, 0x12, 0x7a, 0xb7, 0x2b, 0x4f, 0x1b, 0xa6, 0xc6, 0xde, 0x90,
    0x72, 0x2e, 0x0c, 0xd0, 0xb6, 0xd7, 0x69, 0xd3, 0xc9, 0xe5, 0x59, 0xbb, 0x3d, 0x2e,
};

extern const ByteSpan sTestCert_DAC_FFF2_8005_0033_Val1SecBefore_PublicKey =
    ByteSpan(sTestCert_DAC_FFF2_8005_0033_Val1SecBefore_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8005_0033_Val1SecBefore_PrivateKey_Array[] = {
    0x17, 0xd6, 0x33, 0xba, 0x54, 0xe1, 0x3f, 0x3d, 0x40, 0x0c, 0x37, 0xd7, 0x68, 0x8f, 0xe3, 0xc6,
    0x44, 0x52, 0x29, 0xbe, 0xc7, 0xee, 0xef, 0x02, 0x9e, 0x17, 0xc2, 0xdc, 0x3a, 0xfc, 0xd3, 0xd1,
};

extern const ByteSpan sTestCert_DAC_FFF2_8005_0033_Val1SecBefore_PrivateKey =
    ByteSpan(sTestCert_DAC_FFF2_8005_0033_Val1SecBefore_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8006-0024-ValInPast-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8006_0024_ValInPast_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xe8, 0x30, 0x82, 0x01, 0x8f, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x25, 0xfb, 0x9c, 0xa6, 0x64, 0xe1,
    0x55, 0xe5, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30,
    0x30, 0x36, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f,
    0x39, 0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b,
    0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41,
    0x43, 0x20, 0x30, 0x30, 0x32, 0x34, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02,
    0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c,
    0x02, 0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x36, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01,
    0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xee, 0xc0, 0xef, 0x90, 0x73, 0x92, 0xe8,
    0x08, 0xec, 0xb5, 0x52, 0xcd, 0xb9, 0xf7, 0x4c, 0x03, 0x9c, 0x14, 0x7b, 0x4f, 0x0e, 0x66, 0x0e, 0xd1, 0x9e, 0x87, 0x57, 0x98,
    0x6f, 0x46, 0x45, 0xb1, 0x49, 0xb9, 0xb6, 0x8b, 0x92, 0x98, 0xac, 0xbd, 0x92, 0x43, 0xf4, 0xba, 0x6a, 0x4c, 0xbc, 0x8b, 0xd9,
    0x2a, 0x77, 0xbd, 0x83, 0x9f, 0x09, 0xf7, 0x0f, 0x11, 0x57, 0xc8, 0x24, 0x86, 0xf3, 0x55, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c,
    0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01,
    0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x3f, 0x04, 0x17,
    0xa5, 0x5f, 0xe1, 0xc8, 0xa0, 0x44, 0xc9, 0xa4, 0x03, 0x5d, 0xbb, 0x30, 0x8e, 0x35, 0xe2, 0xa1, 0x5a, 0x30, 0x1f, 0x06, 0x03,
    0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x0a, 0x6f, 0xab, 0xbd, 0xe2, 0x77, 0xfd, 0x19, 0xf6, 0x46, 0x51, 0x25,
    0x62, 0x91, 0x58, 0xac, 0x3a, 0xb3, 0xa4, 0xe2, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03,
    0x47, 0x00, 0x30, 0x44, 0x02, 0x20, 0x0d, 0x85, 0xc7, 0xaf, 0xa1, 0xbb, 0xd6, 0x75, 0x43, 0xb8, 0x5d, 0xc1, 0xc8, 0xca, 0x67,
    0x4f, 0xfa, 0x6c, 0x65, 0x89, 0x71, 0x91, 0x7d, 0x7d, 0x72, 0xff, 0x8d, 0x29, 0x27, 0x11, 0x99, 0xa2, 0x02, 0x20, 0x72, 0x26,
    0x1a, 0xf6, 0x4e, 0x04, 0x8a, 0x0d, 0x58, 0x29, 0x6f, 0xab, 0xc2, 0x22, 0x4e, 0xaa, 0x53, 0xc9, 0x65, 0xa7, 0xba, 0x9e, 0x68,
    0x76, 0x10, 0xc0, 0x95, 0xcf, 0x60, 0x5c, 0x37, 0x7f,
};

extern const ByteSpan sTestCert_DAC_FFF2_8006_0024_ValInPast_Cert = ByteSpan(sTestCert_DAC_FFF2_8006_0024_ValInPast_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8006_0024_ValInPast_SKID_Array[] = {
    0x3F, 0x04, 0x17, 0xA5, 0x5F, 0xE1, 0xC8, 0xA0, 0x44, 0xC9, 0xA4, 0x03, 0x5D, 0xBB, 0x30, 0x8E, 0x35, 0xE2, 0xA1, 0x5A,
};

extern const ByteSpan sTestCert_DAC_FFF2_8006_0024_ValInPast_SKID = ByteSpan(sTestCert_DAC_FFF2_8006_0024_ValInPast_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8006-0024-ValInPast-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8006_0024_ValInPast_PublicKey_Array[] = {
    0x04, 0xee, 0xc0, 0xef, 0x90, 0x73, 0x92, 0xe8, 0x08, 0xec, 0xb5, 0x52, 0xcd, 0xb9, 0xf7, 0x4c, 0x03,
    0x9c, 0x14, 0x7b, 0x4f, 0x0e, 0x66, 0x0e, 0xd1, 0x9e, 0x87, 0x57, 0x98, 0x6f, 0x46, 0x45, 0xb1, 0x49,
    0xb9, 0xb6, 0x8b, 0x92, 0x98, 0xac, 0xbd, 0x92, 0x43, 0xf4, 0xba, 0x6a, 0x4c, 0xbc, 0x8b, 0xd9, 0x2a,
    0x77, 0xbd, 0x83, 0x9f, 0x09, 0xf7, 0x0f, 0x11, 0x57, 0xc8, 0x24, 0x86, 0xf3, 0x55,
};

extern const ByteSpan sTestCert_DAC_FFF2_8006_0024_ValInPast_PublicKey =
    ByteSpan(sTestCert_DAC_FFF2_8006_0024_ValInPast_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8006_0024_ValInPast_PrivateKey_Array[] = {
    0xb3, 0x3b, 0x1c, 0x1c, 0x6f, 0x29, 0xda, 0x89, 0x50, 0x42, 0x26, 0xb2, 0x3c, 0xc3, 0x6c, 0x9c,
    0x46, 0x0c, 0xc4, 0x53, 0x28, 0xfa, 0x19, 0xd7, 0x10, 0x88, 0x21, 0xc2, 0xc4, 0x96, 0x8e, 0xc9,
};

extern const ByteSpan sTestCert_DAC_FFF2_8006_0024_ValInPast_PrivateKey =
    ByteSpan(sTestCert_DAC_FFF2_8006_0024_ValInPast_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8006-0025-ValInFuture-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8006_0025_ValInFuture_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xe9, 0x30, 0x82, 0x01, 0x8f, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x50, 0x02, 0xdf, 0x32, 0xe7, 0x54,
    0x8a, 0xac, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30,
    0x30, 0x36, 0x30, 0x20, 0x17, 0x0d, 0x33, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f,
    0x39, 0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b,
    0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41,
    0x43, 0x20, 0x30, 0x30, 0x32, 0x35, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02,
    0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c,
    0x02, 0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x36, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01,
    0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x7d, 0x5c, 0x18, 0x83, 0x66, 0xc4, 0xf7,
    0xab, 0xbe, 0xd2, 0x32, 0xb7, 0x38, 0xfa, 0x16, 0xda, 0xf4, 0xd0, 0xfe, 0xee, 0x85, 0x54, 0x16, 0xcd, 0x82, 0xde, 0x89, 0x78,
    0x18, 0xa1, 0x24, 0xbd, 0x52, 0x0e, 0x2d, 0xad, 0xa7, 0xfc, 0x26, 0x2a, 0xd7, 0xd0, 0x96, 0x8c, 0x33, 0x8f, 0xb3, 0x55, 0x4e,
    0x58, 0x9a, 0xec, 0x70, 0x54, 0x6c, 0xce, 0xe0, 0x09, 0x6d, 0x3c, 0x91, 0x1f, 0xfa, 0xbd, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c,
    0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01,
    0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xd0, 0x1a, 0xc5,
    0xe6, 0x1c, 0x40, 0x03, 0x32, 0x64, 0xad, 0x45, 0xe2, 0x42, 0x8c, 0x55, 0xfe, 0xdd, 0x00, 0xa7, 0xf3, 0x30, 0x1f, 0x06, 0x03,
    0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x3c, 0x28, 0x86, 0xd0, 0x28, 0x54, 0x87, 0x00, 0x50, 0x2f, 0x89, 0x4b,
    0xda, 0x09, 0x82, 0xb9, 0xd9, 0x70, 0x5e, 0xb0, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03,
    0x48, 0x00, 0x30, 0x45, 0x02, 0x21, 0x00, 0x99, 0x16, 0x23, 0x29, 0x9c, 0x86, 0x63, 0x04, 0x7b, 0x69, 0x16, 0xfe, 0xaa, 0xc7,
    0x0a, 0x96, 0x66, 0x7e, 0x9e, 0x69, 0xaf, 0xe4, 0x18, 0x07, 0x0a, 0x5b, 0xa6, 0x72, 0x4d, 0x03, 0x14, 0xa7, 0x02, 0x20, 0x39,
    0x06, 0xc8, 0x0e, 0x66, 0x92, 0xf0, 0xe7, 0x94, 0xac, 0x7c, 0xd6, 0x8f, 0xdb, 0xad, 0x49, 0xb3, 0xde, 0xa6, 0x0a, 0x2e, 0xa6,
    0x1f, 0x9b, 0xf4, 0x02, 0x72, 0x86, 0x25, 0x3e, 0x9c, 0x6d,
};

extern const ByteSpan sTestCert_DAC_FFF2_8006_0025_ValInFuture_Cert = ByteSpan(sTestCert_DAC_FFF2_8006_0025_ValInFuture_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8006_0025_ValInFuture_SKID_Array[] = {
    0xD0, 0x1A, 0xC5, 0xE6, 0x1C, 0x40, 0x03, 0x32, 0x64, 0xAD, 0x45, 0xE2, 0x42, 0x8C, 0x55, 0xFE, 0xDD, 0x00, 0xA7, 0xF3,
};

extern const ByteSpan sTestCert_DAC_FFF2_8006_0025_ValInFuture_SKID = ByteSpan(sTestCert_DAC_FFF2_8006_0025_ValInFuture_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8006-0025-ValInFuture-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8006_0025_ValInFuture_PublicKey_Array[] = {
    0x04, 0x7d, 0x5c, 0x18, 0x83, 0x66, 0xc4, 0xf7, 0xab, 0xbe, 0xd2, 0x32, 0xb7, 0x38, 0xfa, 0x16, 0xda,
    0xf4, 0xd0, 0xfe, 0xee, 0x85, 0x54, 0x16, 0xcd, 0x82, 0xde, 0x89, 0x78, 0x18, 0xa1, 0x24, 0xbd, 0x52,
    0x0e, 0x2d, 0xad, 0xa7, 0xfc, 0x26, 0x2a, 0xd7, 0xd0, 0x96, 0x8c, 0x33, 0x8f, 0xb3, 0x55, 0x4e, 0x58,
    0x9a, 0xec, 0x70, 0x54, 0x6c, 0xce, 0xe0, 0x09, 0x6d, 0x3c, 0x91, 0x1f, 0xfa, 0xbd,
};

extern const ByteSpan sTestCert_DAC_FFF2_8006_0025_ValInFuture_PublicKey =
    ByteSpan(sTestCert_DAC_FFF2_8006_0025_ValInFuture_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8006_0025_ValInFuture_PrivateKey_Array[] = {
    0x58, 0xe8, 0x1e, 0xd5, 0xe0, 0xa0, 0x73, 0x49, 0xc0, 0xb9, 0x84, 0xfa, 0x51, 0x8d, 0x17, 0xe6,
    0xf6, 0xfa, 0xcb, 0xf7, 0x75, 0x9e, 0x58, 0x49, 0xf6, 0x8f, 0xa3, 0x4c, 0xf7, 0xc1, 0x98, 0xb1,
};

extern const ByteSpan sTestCert_DAC_FFF2_8006_0025_ValInFuture_PrivateKey =
    ByteSpan(sTestCert_DAC_FFF2_8006_0025_ValInFuture_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8006-0034-ValInFuture-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8006_0034_ValInFuture_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xea, 0x30, 0x82, 0x01, 0x8f, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x53, 0x31, 0xb9, 0x16, 0x85, 0x31,
    0x38, 0x00, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30,
    0x30, 0x36, 0x30, 0x20, 0x17, 0x0d, 0x33, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f,
    0x39, 0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b,
    0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41,
    0x43, 0x20, 0x30, 0x30, 0x33, 0x34, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02,
    0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c,
    0x02, 0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x36, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01,
    0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x4a, 0xca, 0xfd, 0x49, 0x0f, 0x13, 0x96,
    0xc7, 0x00, 0x63, 0x63, 0xdc, 0x55, 0xe1, 0x9a, 0x9e, 0xd5, 0xa8, 0x09, 0xc1, 0x99, 0xd9, 0x76, 0x66, 0x8d, 0x61, 0x24, 0x37,
    0x92, 0x1a, 0x2b, 0x39, 0x18, 0x89, 0xc7, 0x1e, 0xc2, 0x42, 0xd6, 0x65, 0xa0, 0x78, 0x8f, 0xca, 0xad, 0x1b, 0xfb, 0x81, 0x1d,
    0x2f, 0x1c, 0x9f, 0xf1, 0xff, 0xd6, 0x16, 0x93, 0x94, 0x0a, 0xd7, 0x80, 0xf4, 0x67, 0x37, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c,
    0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01,
    0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x50, 0x35, 0x1f,
    0xba, 0x3f, 0x92, 0xdb, 0x07, 0x26, 0xd7, 0x5d, 0xe4, 0x13, 0x00, 0xe9, 0x99, 0xde, 0x4d, 0x3a, 0x7a, 0x30, 0x1f, 0x06, 0x03,
    0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x0a, 0x6f, 0xab, 0xbd, 0xe2, 0x77, 0xfd, 0x19, 0xf6, 0x46, 0x51, 0x25,
    0x62, 0x91, 0x58, 0xac, 0x3a, 0xb3, 0xa4, 0xe2, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03,
    0x49, 0x00, 0x30, 0x46, 0x02, 0x21, 0x00, 0xc9, 0xb0, 0x28, 0xdc, 0x5f, 0xf4, 0x3b, 0x60, 0xac, 0x8e, 0xa7, 0xab, 0x9b, 0x44,
    0x25, 0xf5, 0x98, 0xa3, 0x07, 0x07, 0x8e, 0x65, 0xd2, 0xb0, 0x02, 0x0e, 0xe1, 0x0c, 0x20, 0xcd, 0x02, 0x84, 0x02, 0x21, 0x00,
    0xba, 0xb6, 0x31, 0x6f, 0xd4, 0xa8, 0x7e, 0x31, 0x18, 0xd6, 0x95, 0x95, 0x69, 0xbf, 0x82, 0x4d, 0xfd, 0x87, 0x3d, 0x6e, 0xd7,
    0x12, 0xf4, 0x61, 0x3a, 0x6f, 0xa6, 0xa0, 0xe5, 0xc1, 0xeb, 0x3d,
};

extern const ByteSpan sTestCert_DAC_FFF2_8006_0034_ValInFuture_Cert = ByteSpan(sTestCert_DAC_FFF2_8006_0034_ValInFuture_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8006_0034_ValInFuture_SKID_Array[] = {
    0x50, 0x35, 0x1F, 0xBA, 0x3F, 0x92, 0xDB, 0x07, 0x26, 0xD7, 0x5D, 0xE4, 0x13, 0x00, 0xE9, 0x99, 0xDE, 0x4D, 0x3A, 0x7A,
};

extern const ByteSpan sTestCert_DAC_FFF2_8006_0034_ValInFuture_SKID = ByteSpan(sTestCert_DAC_FFF2_8006_0034_ValInFuture_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8006-0034-ValInFuture-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8006_0034_ValInFuture_PublicKey_Array[] = {
    0x04, 0x4a, 0xca, 0xfd, 0x49, 0x0f, 0x13, 0x96, 0xc7, 0x00, 0x63, 0x63, 0xdc, 0x55, 0xe1, 0x9a, 0x9e,
    0xd5, 0xa8, 0x09, 0xc1, 0x99, 0xd9, 0x76, 0x66, 0x8d, 0x61, 0x24, 0x37, 0x92, 0x1a, 0x2b, 0x39, 0x18,
    0x89, 0xc7, 0x1e, 0xc2, 0x42, 0xd6, 0x65, 0xa0, 0x78, 0x8f, 0xca, 0xad, 0x1b, 0xfb, 0x81, 0x1d, 0x2f,
    0x1c, 0x9f, 0xf1, 0xff, 0xd6, 0x16, 0x93, 0x94, 0x0a, 0xd7, 0x80, 0xf4, 0x67, 0x37,
};

extern const ByteSpan sTestCert_DAC_FFF2_8006_0034_ValInFuture_PublicKey =
    ByteSpan(sTestCert_DAC_FFF2_8006_0034_ValInFuture_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8006_0034_ValInFuture_PrivateKey_Array[] = {
    0xa0, 0x7c, 0xe4, 0x07, 0xf3, 0x4c, 0xa2, 0x2f, 0x3f, 0x64, 0x90, 0x66, 0x76, 0xfe, 0xaf, 0x9c,
    0xc2, 0xa6, 0xb1, 0x1a, 0x4a, 0x4f, 0x81, 0x2c, 0xc9, 0x33, 0xbf, 0xc1, 0x0a, 0x22, 0x1a, 0x2b,
};

extern const ByteSpan sTestCert_DAC_FFF2_8006_0034_ValInFuture_PrivateKey =
    ByteSpan(sTestCert_DAC_FFF2_8006_0034_ValInFuture_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8006-0035-Val1SecBefore-Cert.pem

constexpr uint8_t sTestCert_DAC_FFF2_8006_0035_Val1SecBefore_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xe8, 0x30, 0x82, 0x01, 0x8d, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x49, 0x76, 0xa7, 0x39, 0x0c, 0x76,
    0xe2, 0x76, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30,
    0x30, 0x36, 0x30, 0x1e, 0x17, 0x0d, 0x33, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x32, 0x5a, 0x17, 0x0d,
    0x33, 0x32, 0x30, 0x36, 0x32, 0x37, 0x31, 0x34, 0x32, 0x33, 0x34, 0x31, 0x5a, 0x30, 0x4b, 0x31, 0x1d, 0x30, 0x1b, 0x06, 0x03,
    0x55, 0x04, 0x03, 0x0c, 0x14, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x44, 0x41, 0x43, 0x20,
    0x30, 0x30, 0x33, 0x35, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c,
    0x04, 0x46, 0x46, 0x46, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02,
    0x0c, 0x04, 0x38, 0x30, 0x30, 0x36, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08,
    0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x44, 0x96, 0x1c, 0xae, 0x99, 0x99, 0x20, 0x6c, 0xda,
    0x46, 0xa0, 0xcf, 0x24, 0x09, 0x79, 0x97, 0x32, 0xbc, 0xa8, 0xd3, 0x23, 0x15, 0x77, 0x6c, 0xdd, 0x0e, 0xdf, 0x31, 0x58, 0xd2,
    0x19, 0x65, 0xb7, 0x93, 0xc5, 0xf1, 0xce, 0xc4, 0x89, 0x71, 0xa1, 0x0a, 0xa6, 0x4c, 0x4e, 0x07, 0x40, 0x58, 0x62, 0x8a, 0x69,
    0x5f, 0xe6, 0x6e, 0x7e, 0x3f, 0xdf, 0x38, 0x9f, 0x90, 0x62, 0xb7, 0x1d, 0x80, 0xa3, 0x60, 0x30, 0x5e, 0x30, 0x0c, 0x06, 0x03,
    0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04,
    0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xf2, 0xc0, 0xb6, 0x87, 0x73,
    0x43, 0x40, 0x85, 0x2b, 0xeb, 0x19, 0xa2, 0x7b, 0x93, 0xc4, 0xa2, 0x60, 0x5a, 0x4d, 0x34, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d,
    0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x3c, 0x28, 0x86, 0xd0, 0x28, 0x54, 0x87, 0x00, 0x50, 0x2f, 0x89, 0x4b, 0xda, 0x09,
    0x82, 0xb9, 0xd9, 0x70, 0x5e, 0xb0, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x49, 0x00,
    0x30, 0x46, 0x02, 0x21, 0x00, 0xf5, 0xd7, 0xfe, 0x94, 0x00, 0x93, 0xcc, 0x1b, 0xf4, 0x68, 0x36, 0x6d, 0x9d, 0x33, 0x20, 0x4a,
    0x1e, 0x54, 0xc0, 0x10, 0x69, 0xb3, 0x8e, 0x80, 0x57, 0x5d, 0x02, 0x2a, 0x8e, 0x55, 0xec, 0x16, 0x02, 0x21, 0x00, 0xfb, 0x13,
    0xe6, 0xd6, 0x5b, 0x2a, 0x13, 0x19, 0x38, 0xef, 0x0e, 0x76, 0xd9, 0x6a, 0x31, 0x28, 0x62, 0x55, 0xaf, 0x79, 0x26, 0xb4, 0x5a,
    0x5e, 0x7d, 0x62, 0x1f, 0xbe, 0x22, 0x2d, 0x8b, 0xb9,
};

extern const ByteSpan sTestCert_DAC_FFF2_8006_0035_Val1SecBefore_Cert =
    ByteSpan(sTestCert_DAC_FFF2_8006_0035_Val1SecBefore_Cert_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8006_0035_Val1SecBefore_SKID_Array[] = {
    0xF2, 0xC0, 0xB6, 0x87, 0x73, 0x43, 0x40, 0x85, 0x2B, 0xEB, 0x19, 0xA2, 0x7B, 0x93, 0xC4, 0xA2, 0x60, 0x5A, 0x4D, 0x34,
};

extern const ByteSpan sTestCert_DAC_FFF2_8006_0035_Val1SecBefore_SKID =
    ByteSpan(sTestCert_DAC_FFF2_8006_0035_Val1SecBefore_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-DAC-FFF2-8006-0035-Val1SecBefore-Key.pem

constexpr uint8_t sTestCert_DAC_FFF2_8006_0035_Val1SecBefore_PublicKey_Array[] = {
    0x04, 0x44, 0x96, 0x1c, 0xae, 0x99, 0x99, 0x20, 0x6c, 0xda, 0x46, 0xa0, 0xcf, 0x24, 0x09, 0x79, 0x97,
    0x32, 0xbc, 0xa8, 0xd3, 0x23, 0x15, 0x77, 0x6c, 0xdd, 0x0e, 0xdf, 0x31, 0x58, 0xd2, 0x19, 0x65, 0xb7,
    0x93, 0xc5, 0xf1, 0xce, 0xc4, 0x89, 0x71, 0xa1, 0x0a, 0xa6, 0x4c, 0x4e, 0x07, 0x40, 0x58, 0x62, 0x8a,
    0x69, 0x5f, 0xe6, 0x6e, 0x7e, 0x3f, 0xdf, 0x38, 0x9f, 0x90, 0x62, 0xb7, 0x1d, 0x80,
};

extern const ByteSpan sTestCert_DAC_FFF2_8006_0035_Val1SecBefore_PublicKey =
    ByteSpan(sTestCert_DAC_FFF2_8006_0035_Val1SecBefore_PublicKey_Array);

constexpr uint8_t sTestCert_DAC_FFF2_8006_0035_Val1SecBefore_PrivateKey_Array[] = {
    0xb6, 0x2f, 0x61, 0x4d, 0x69, 0xd4, 0x79, 0xc4, 0xfc, 0x49, 0x4a, 0x85, 0x5f, 0xf1, 0xb3, 0x3b,
    0xa9, 0x38, 0x51, 0x24, 0xd4, 0xe9, 0x55, 0xbe, 0x64, 0x00, 0xc9, 0x73, 0xca, 0x60, 0x1d, 0x98,
};

extern const ByteSpan sTestCert_DAC_FFF2_8006_0035_Val1SecBefore_PrivateKey =
    ByteSpan(sTestCert_DAC_FFF2_8006_0035_Val1SecBefore_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAA-FFF1-Cert.pem

constexpr uint8_t sTestCert_PAA_FFF1_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xbd, 0x30, 0x82, 0x01, 0x64, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x4e, 0xa8, 0xe8, 0x31, 0x82, 0xd4,
    0x1c, 0x1c, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x30, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x41,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x31, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x39,
    0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x30, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x41,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x31, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d,
    0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xb6, 0xcb, 0x63, 0x72, 0x88, 0x7f, 0x29, 0x28, 0xf5, 0xba, 0xc8, 0x1a, 0xa9, 0xd9,
    0x3a, 0xe2, 0x43, 0x1c, 0xad, 0xa9, 0xd7, 0x9e, 0x24, 0x2f, 0x65, 0x17, 0x7e, 0xf9, 0xce, 0xd9, 0x32, 0xa2, 0x8e, 0xcd, 0x03,
    0xba, 0xaf, 0x6a, 0x8f, 0xca, 0x18, 0x4a, 0x1a, 0x50, 0x35, 0x42, 0x96, 0x0d, 0x45, 0x3f, 0x30, 0x3f, 0x1f, 0x19, 0x42, 0x1d,
    0x75, 0x1e, 0x8f, 0x8f, 0x1a, 0x9a, 0x9b, 0x75, 0xa3, 0x66, 0x30, 0x64, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01,
    0xff, 0x04, 0x08, 0x30, 0x06, 0x01, 0x01, 0xff, 0x02, 0x01, 0x01, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff,
    0x04, 0x04, 0x03, 0x02, 0x01, 0x06, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x6a, 0xfd, 0x22, 0x77,
    0x1f, 0x51, 0x1f, 0xec, 0xbf, 0x16, 0x41, 0x97, 0x67, 0x10, 0xdc, 0xdc, 0x31, 0xa1, 0x71, 0x7e, 0x30, 0x1f, 0x06, 0x03, 0x55,
    0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x6a, 0xfd, 0x22, 0x77, 0x1f, 0x51, 0x1f, 0xec, 0xbf, 0x16, 0x41, 0x97, 0x67,
    0x10, 0xdc, 0xdc, 0x31, 0xa1, 0x71, 0x7e, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x47,
    0x00, 0x30, 0x44, 0x02, 0x20, 0x50, 0xaa, 0x80, 0x02, 0xf4, 0xd9, 0x32, 0xa9, 0xa0, 0x05, 0x38, 0xf6, 0x53, 0x68, 0xad, 0x0f,
    0xff, 0xc8, 0xef, 0xbb, 0xc9, 0xbe, 0xb7, 0xda, 0x56, 0x98, 0x35, 0xcf, 0x9a, 0xa7, 0x51, 0x0e, 0x02, 0x20, 0x23, 0xba, 0xc8,
    0xfe, 0x0f, 0x23, 0xe7, 0x54, 0x45, 0xb6, 0x53, 0x39, 0x08, 0x1a, 0x47, 0x99, 0x49, 0x29, 0xc7, 0x2a, 0xaf, 0x0a, 0x15, 0x48,
    0xd4, 0x0d, 0x03, 0x4d, 0x51, 0x4b, 0x25, 0xde,
};

extern const ByteSpan sTestCert_PAA_FFF1_Cert = ByteSpan(sTestCert_PAA_FFF1_Cert_Array);

constexpr uint8_t sTestCert_PAA_FFF1_SKID_Array[] = {
    0x6A, 0xFD, 0x22, 0x77, 0x1F, 0x51, 0x1F, 0xEC, 0xBF, 0x16, 0x41, 0x97, 0x67, 0x10, 0xDC, 0xDC, 0x31, 0xA1, 0x71, 0x7E,
};

extern const ByteSpan sTestCert_PAA_FFF1_SKID = ByteSpan(sTestCert_PAA_FFF1_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAA-FFF1-Key.pem

constexpr uint8_t sTestCert_PAA_FFF1_PublicKey_Array[] = {
    0x04, 0xb6, 0xcb, 0x63, 0x72, 0x88, 0x7f, 0x29, 0x28, 0xf5, 0xba, 0xc8, 0x1a, 0xa9, 0xd9, 0x3a, 0xe2,
    0x43, 0x1c, 0xad, 0xa9, 0xd7, 0x9e, 0x24, 0x2f, 0x65, 0x17, 0x7e, 0xf9, 0xce, 0xd9, 0x32, 0xa2, 0x8e,
    0xcd, 0x03, 0xba, 0xaf, 0x6a, 0x8f, 0xca, 0x18, 0x4a, 0x1a, 0x50, 0x35, 0x42, 0x96, 0x0d, 0x45, 0x3f,
    0x30, 0x3f, 0x1f, 0x19, 0x42, 0x1d, 0x75, 0x1e, 0x8f, 0x8f, 0x1a, 0x9a, 0x9b, 0x75,
};

extern const ByteSpan sTestCert_PAA_FFF1_PublicKey = ByteSpan(sTestCert_PAA_FFF1_PublicKey_Array);

constexpr uint8_t sTestCert_PAA_FFF1_PrivateKey_Array[] = {
    0x65, 0x12, 0xca, 0xec, 0xae, 0xcf, 0xc5, 0x43, 0xd6, 0x06, 0x23, 0x16, 0x15, 0x97, 0x16, 0x2f,
    0x01, 0x46, 0x84, 0xc5, 0x65, 0xa1, 0x29, 0xb6, 0x2f, 0xd2, 0x8c, 0x27, 0xab, 0x1c, 0xcc, 0x50,
};

extern const ByteSpan sTestCert_PAA_FFF1_PrivateKey = ByteSpan(sTestCert_PAA_FFF1_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAA-FFF2-ValInFuture-Cert.pem

constexpr uint8_t sTestCert_PAA_FFF2_ValInFuture_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xbc, 0x30, 0x82, 0x01, 0x62, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x51, 0x4d, 0xf5, 0x4f, 0x81, 0x7f,
    0x6f, 0x27, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x30, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x41,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x32, 0x30, 0x1e, 0x17, 0x0d, 0x33, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x17, 0x0d, 0x33,
    0x32, 0x30, 0x36, 0x32, 0x37, 0x31, 0x34, 0x32, 0x33, 0x34, 0x32, 0x5a, 0x30, 0x30, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55,
    0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x41, 0x31, 0x14,
    0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x30,
    0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01,
    0x07, 0x03, 0x42, 0x00, 0x04, 0x23, 0x62, 0xd9, 0x13, 0x5b, 0xb1, 0x3b, 0xc3, 0x6e, 0xa6, 0x36, 0x7d, 0x4b, 0x4c, 0xb3, 0x7f,
    0xf7, 0x72, 0x19, 0xa7, 0xfc, 0x60, 0x3a, 0xb0, 0x06, 0x3f, 0xdc, 0x9e, 0xff, 0xc6, 0xcf, 0x0b, 0x5e, 0xba, 0x13, 0x20, 0xa6,
    0x6e, 0xf4, 0xe3, 0x48, 0x1b, 0xca, 0xbb, 0xe1, 0xcc, 0x52, 0x55, 0x26, 0xb7, 0x6a, 0xfb, 0xf5, 0xb9, 0x1b, 0xf5, 0x2c, 0x33,
    0x86, 0x0d, 0xc6, 0x5f, 0x10, 0x39, 0xa3, 0x66, 0x30, 0x64, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04,
    0x08, 0x30, 0x06, 0x01, 0x01, 0xff, 0x02, 0x01, 0x01, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04,
    0x03, 0x02, 0x01, 0x06, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x7f, 0x1d, 0xaa, 0xf2, 0x44, 0x98,
    0xb9, 0x86, 0x68, 0x0e, 0xa0, 0x8f, 0xc1, 0x89, 0x21, 0xe8, 0x48, 0x48, 0x9d, 0x17, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23,
    0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x7f, 0x1d, 0xaa, 0xf2, 0x44, 0x98, 0xb9, 0x86, 0x68, 0x0e, 0xa0, 0x8f, 0xc1, 0x89, 0x21,
    0xe8, 0x48, 0x48, 0x9d, 0x17, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x48, 0x00, 0x30,
    0x45, 0x02, 0x20, 0x6c, 0x13, 0x8c, 0xe5, 0x70, 0xf3, 0xac, 0x0d, 0x5c, 0x58, 0xe4, 0xc7, 0x8a, 0x04, 0x78, 0x82, 0x8f, 0x7a,
    0x6a, 0x8b, 0x38, 0xf9, 0x6d, 0x69, 0x03, 0xc2, 0x2b, 0x96, 0x3d, 0x8b, 0xc5, 0x17, 0x02, 0x21, 0x00, 0xe3, 0x7b, 0x7f, 0xb6,
    0x1a, 0xfc, 0xfb, 0x51, 0x6f, 0x6b, 0x46, 0x47, 0xfc, 0xa7, 0xf4, 0x86, 0x35, 0xb2, 0xac, 0xb8, 0xb8, 0x8d, 0x9b, 0xc2, 0x39,
    0x89, 0xd7, 0x2d, 0x42, 0x0d, 0xde, 0x6b,
};

extern const ByteSpan sTestCert_PAA_FFF2_ValInFuture_Cert = ByteSpan(sTestCert_PAA_FFF2_ValInFuture_Cert_Array);

constexpr uint8_t sTestCert_PAA_FFF2_ValInFuture_SKID_Array[] = {
    0x7F, 0x1D, 0xAA, 0xF2, 0x44, 0x98, 0xB9, 0x86, 0x68, 0x0E, 0xA0, 0x8F, 0xC1, 0x89, 0x21, 0xE8, 0x48, 0x48, 0x9D, 0x17,
};

extern const ByteSpan sTestCert_PAA_FFF2_ValInFuture_SKID = ByteSpan(sTestCert_PAA_FFF2_ValInFuture_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAA-FFF2-ValInFuture-Key.pem

constexpr uint8_t sTestCert_PAA_FFF2_ValInFuture_PublicKey_Array[] = {
    0x04, 0x23, 0x62, 0xd9, 0x13, 0x5b, 0xb1, 0x3b, 0xc3, 0x6e, 0xa6, 0x36, 0x7d, 0x4b, 0x4c, 0xb3, 0x7f,
    0xf7, 0x72, 0x19, 0xa7, 0xfc, 0x60, 0x3a, 0xb0, 0x06, 0x3f, 0xdc, 0x9e, 0xff, 0xc6, 0xcf, 0x0b, 0x5e,
    0xba, 0x13, 0x20, 0xa6, 0x6e, 0xf4, 0xe3, 0x48, 0x1b, 0xca, 0xbb, 0xe1, 0xcc, 0x52, 0x55, 0x26, 0xb7,
    0x6a, 0xfb, 0xf5, 0xb9, 0x1b, 0xf5, 0x2c, 0x33, 0x86, 0x0d, 0xc6, 0x5f, 0x10, 0x39,
};

extern const ByteSpan sTestCert_PAA_FFF2_ValInFuture_PublicKey = ByteSpan(sTestCert_PAA_FFF2_ValInFuture_PublicKey_Array);

constexpr uint8_t sTestCert_PAA_FFF2_ValInFuture_PrivateKey_Array[] = {
    0x97, 0xa0, 0x75, 0x64, 0xb3, 0x0f, 0x54, 0x62, 0x82, 0x78, 0x78, 0x9e, 0x23, 0x9a, 0x36, 0x57,
    0x70, 0x3c, 0x4a, 0xf2, 0x00, 0x5e, 0x84, 0x02, 0x8a, 0x9a, 0xfe, 0xa1, 0xa6, 0xb4, 0x83, 0xb4,
};

extern const ByteSpan sTestCert_PAA_FFF2_ValInFuture_PrivateKey = ByteSpan(sTestCert_PAA_FFF2_ValInFuture_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAA-FFF2-ValInPast-Cert.pem

constexpr uint8_t sTestCert_PAA_FFF2_ValInPast_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xbc, 0x30, 0x82, 0x01, 0x62, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x03, 0x01, 0x4c, 0x5d, 0x09, 0x1b,
    0xb9, 0x24, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x30, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x41,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x32, 0x30, 0x1e, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x17, 0x0d, 0x32,
    0x32, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x32, 0x5a, 0x30, 0x30, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55,
    0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x41, 0x31, 0x14,
    0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x30,
    0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01,
    0x07, 0x03, 0x42, 0x00, 0x04, 0xb7, 0x0a, 0x16, 0x96, 0x6d, 0xe4, 0x6d, 0xbf, 0xb8, 0x38, 0xa0, 0x80, 0xab, 0x20, 0xff, 0x76,
    0x90, 0x97, 0xc7, 0xa8, 0xa0, 0x21, 0xdf, 0xb4, 0x4a, 0xb6, 0x87, 0xb2, 0x04, 0xe0, 0x11, 0x2d, 0x3b, 0x10, 0x7d, 0xfc, 0x6a,
    0x70, 0xf7, 0x53, 0x28, 0x81, 0xdb, 0x76, 0x56, 0xd4, 0x90, 0x24, 0xe2, 0x10, 0x26, 0xc5, 0x55, 0xfb, 0x92, 0x2d, 0x89, 0x09,
    0xcc, 0x6f, 0x0c, 0x05, 0xb1, 0xca, 0xa3, 0x66, 0x30, 0x64, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04,
    0x08, 0x30, 0x06, 0x01, 0x01, 0xff, 0x02, 0x01, 0x01, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04,
    0x03, 0x02, 0x01, 0x06, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x9b, 0xb2, 0x99, 0xbb, 0xa6, 0x49,
    0x7d, 0xb5, 0xa0, 0x4e, 0x1e, 0x18, 0xc7, 0xc2, 0x0e, 0xae, 0xc5, 0x4c, 0x42, 0xe9, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23,
    0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x9b, 0xb2, 0x99, 0xbb, 0xa6, 0x49, 0x7d, 0xb5, 0xa0, 0x4e, 0x1e, 0x18, 0xc7, 0xc2, 0x0e,
    0xae, 0xc5, 0x4c, 0x42, 0xe9, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x48, 0x00, 0x30,
    0x45, 0x02, 0x20, 0x10, 0x3d, 0x5f, 0x9d, 0xe1, 0xfc, 0x20, 0x75, 0xc8, 0x88, 0x68, 0x8d, 0x7f, 0xf9, 0x27, 0x93, 0x4f, 0x4c,
    0x5e, 0x1e, 0x37, 0x0a, 0xa9, 0xa3, 0xf2, 0xd0, 0x16, 0x24, 0x63, 0x6a, 0xb4, 0x61, 0x02, 0x21, 0x00, 0xa3, 0xa9, 0x96, 0xf9,
    0xa7, 0xab, 0x65, 0xe7, 0x32, 0xc6, 0xaf, 0x87, 0x7b, 0xd2, 0xc5, 0x4e, 0xd8, 0x62, 0xf9, 0x45, 0xc1, 0xca, 0xa2, 0x1c, 0x7e,
    0xa8, 0xbd, 0x28, 0xe7, 0xb7, 0x40, 0xb9,
};

extern const ByteSpan sTestCert_PAA_FFF2_ValInPast_Cert = ByteSpan(sTestCert_PAA_FFF2_ValInPast_Cert_Array);

constexpr uint8_t sTestCert_PAA_FFF2_ValInPast_SKID_Array[] = {
    0x9B, 0xB2, 0x99, 0xBB, 0xA6, 0x49, 0x7D, 0xB5, 0xA0, 0x4E, 0x1E, 0x18, 0xC7, 0xC2, 0x0E, 0xAE, 0xC5, 0x4C, 0x42, 0xE9,
};

extern const ByteSpan sTestCert_PAA_FFF2_ValInPast_SKID = ByteSpan(sTestCert_PAA_FFF2_ValInPast_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAA-FFF2-ValInPast-Key.pem

constexpr uint8_t sTestCert_PAA_FFF2_ValInPast_PublicKey_Array[] = {
    0x04, 0xb7, 0x0a, 0x16, 0x96, 0x6d, 0xe4, 0x6d, 0xbf, 0xb8, 0x38, 0xa0, 0x80, 0xab, 0x20, 0xff, 0x76,
    0x90, 0x97, 0xc7, 0xa8, 0xa0, 0x21, 0xdf, 0xb4, 0x4a, 0xb6, 0x87, 0xb2, 0x04, 0xe0, 0x11, 0x2d, 0x3b,
    0x10, 0x7d, 0xfc, 0x6a, 0x70, 0xf7, 0x53, 0x28, 0x81, 0xdb, 0x76, 0x56, 0xd4, 0x90, 0x24, 0xe2, 0x10,
    0x26, 0xc5, 0x55, 0xfb, 0x92, 0x2d, 0x89, 0x09, 0xcc, 0x6f, 0x0c, 0x05, 0xb1, 0xca,
};

extern const ByteSpan sTestCert_PAA_FFF2_ValInPast_PublicKey = ByteSpan(sTestCert_PAA_FFF2_ValInPast_PublicKey_Array);

constexpr uint8_t sTestCert_PAA_FFF2_ValInPast_PrivateKey_Array[] = {
    0x88, 0x8c, 0x1f, 0x57, 0x37, 0x4f, 0x03, 0xe7, 0x51, 0x8c, 0x17, 0x1a, 0x41, 0xc4, 0xa6, 0x74,
    0xd1, 0xa9, 0x71, 0x6c, 0x55, 0x4a, 0x0a, 0x76, 0x26, 0xc6, 0x0e, 0xf5, 0x56, 0x70, 0x22, 0x9a,
};

extern const ByteSpan sTestCert_PAA_FFF2_ValInPast_PrivateKey = ByteSpan(sTestCert_PAA_FFF2_ValInPast_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAA-NoVID-Cert.pem

constexpr uint8_t sTestCert_PAA_NoVID_Cert_Array[] = {
    0x30, 0x82, 0x01, 0x91, 0x30, 0x82, 0x01, 0x37, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x07, 0x0b, 0x8f, 0xba, 0xa8, 0xdd, 0x86,
    0xee, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x1a, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03,
    0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x41, 0x30,
    0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x39, 0x39, 0x39,
    0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x1a, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55,
    0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x41, 0x30, 0x59,
    0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07,
    0x03, 0x42, 0x00, 0x04, 0x10, 0xef, 0x02, 0xa8, 0x1a, 0x87, 0xb6, 0x81, 0x21, 0xfb, 0xa8, 0xd3, 0x19, 0x78, 0xf8, 0x07, 0xa3,
    0x17, 0xe5, 0x0a, 0xa8, 0xa8, 0x28, 0x44, 0x68, 0x28, 0x91, 0x4b, 0x93, 0x3d, 0xe8, 0xed, 0xd4, 0xa5, 0xc3, 0x9c, 0x9f, 0xf7,
    0x1a, 0x4c, 0xe3, 0x64, 0x7f, 0xd7, 0xf6, 0x26, 0x53, 0xb7, 0xd2, 0x49, 0x5f, 0xcb, 0xa4, 0xc0, 0xf4, 0x7f, 0x87, 0x68, 0x80,
    0x03, 0x9e, 0x07, 0x20, 0x4a, 0xa3, 0x66, 0x30, 0x64, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x08,
    0x30, 0x06, 0x01, 0x01, 0xff, 0x02, 0x01, 0x01, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03,
    0x02, 0x01, 0x06, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x78, 0x5c, 0xe7, 0x05, 0xb8, 0x6b, 0x8f,
    0x4e, 0x6f, 0xc7, 0x93, 0xaa, 0x60, 0xcb, 0x43, 0xea, 0x69, 0x68, 0x82, 0xd5, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04,
    0x18, 0x30, 0x16, 0x80, 0x14, 0x78, 0x5c, 0xe7, 0x05, 0xb8, 0x6b, 0x8f, 0x4e, 0x6f, 0xc7, 0x93, 0xaa, 0x60, 0xcb, 0x43, 0xea,
    0x69, 0x68, 0x82, 0xd5, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x48, 0x00, 0x30, 0x45,
    0x02, 0x21, 0x00, 0xb9, 0xef, 0xdb, 0x3e, 0xa0, 0x6a, 0x52, 0xec, 0x0b, 0xf0, 0x1e, 0x61, 0xda, 0xed, 0x2c, 0x2d, 0x15, 0x6d,
    0xdb, 0x6c, 0xf0, 0x14, 0x10, 0x1d, 0xab, 0x79, 0x8f, 0xac, 0x05, 0xfa, 0x47, 0xe5, 0x02, 0x20, 0x60, 0x06, 0x1d, 0x3e, 0x35,
    0xd6, 0x0d, 0x9d, 0x4b, 0x0d, 0x44, 0x8d, 0xad, 0x76, 0x12, 0xf7, 0xe8, 0x5c, 0x58, 0x2e, 0x3f, 0xc3, 0x12, 0xdc, 0x18, 0x79,
    0x4d, 0xd3, 0x73, 0x71, 0x5e, 0x5d,
};

extern const ByteSpan sTestCert_PAA_NoVID_Cert = ByteSpan(sTestCert_PAA_NoVID_Cert_Array);

constexpr uint8_t sTestCert_PAA_NoVID_SKID_Array[] = {
    0x78, 0x5C, 0xE7, 0x05, 0xB8, 0x6B, 0x8F, 0x4E, 0x6F, 0xC7, 0x93, 0xAA, 0x60, 0xCB, 0x43, 0xEA, 0x69, 0x68, 0x82, 0xD5,
};

extern const ByteSpan sTestCert_PAA_NoVID_SKID = ByteSpan(sTestCert_PAA_NoVID_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAA-NoVID-Key.pem

constexpr uint8_t sTestCert_PAA_NoVID_PublicKey_Array[] = {
    0x04, 0x10, 0xef, 0x02, 0xa8, 0x1a, 0x87, 0xb6, 0x81, 0x21, 0xfb, 0xa8, 0xd3, 0x19, 0x78, 0xf8, 0x07,
    0xa3, 0x17, 0xe5, 0x0a, 0xa8, 0xa8, 0x28, 0x44, 0x68, 0x28, 0x91, 0x4b, 0x93, 0x3d, 0xe8, 0xed, 0xd4,
    0xa5, 0xc3, 0x9c, 0x9f, 0xf7, 0x1a, 0x4c, 0xe3, 0x64, 0x7f, 0xd7, 0xf6, 0x26, 0x53, 0xb7, 0xd2, 0x49,
    0x5f, 0xcb, 0xa4, 0xc0, 0xf4, 0x7f, 0x87, 0x68, 0x80, 0x03, 0x9e, 0x07, 0x20, 0x4a,
};

extern const ByteSpan sTestCert_PAA_NoVID_PublicKey = ByteSpan(sTestCert_PAA_NoVID_PublicKey_Array);

constexpr uint8_t sTestCert_PAA_NoVID_PrivateKey_Array[] = {
    0xe1, 0xf0, 0x73, 0xc9, 0x34, 0x85, 0x3b, 0xaf, 0xfb, 0x38, 0xbf, 0x7e, 0x8b, 0xda, 0xd7, 0xa0,
    0xa6, 0x74, 0x10, 0x7c, 0x77, 0x69, 0x89, 0x2a, 0x0f, 0xf2, 0xe0, 0x6c, 0x1a, 0x2e, 0xf7, 0xa7,
};

extern const ByteSpan sTestCert_PAA_NoVID_PrivateKey = ByteSpan(sTestCert_PAA_NoVID_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAA-NoVID-ToResignPAIs-Cert.pem

constexpr uint8_t sTestCert_PAA_NoVID_ToResignPAIs_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xb0, 0x30, 0x82, 0x01, 0x56, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x7d, 0x00, 0x0a, 0x17, 0x23, 0xbe,
    0xdf, 0x06, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x29, 0x31, 0x27, 0x30, 0x25, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x1e, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x41,
    0x20, 0x54, 0x6f, 0x20, 0x52, 0x65, 0x73, 0x69, 0x67, 0x6e, 0x20, 0x50, 0x41, 0x49, 0x73, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31,
    0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x39, 0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31,
    0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x29, 0x31, 0x27, 0x30, 0x25, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x1e, 0x4d,
    0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x41, 0x20, 0x54, 0x6f, 0x20, 0x52, 0x65, 0x73,
    0x69, 0x67, 0x6e, 0x20, 0x50, 0x41, 0x49, 0x73, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01,
    0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x10, 0xef, 0x02, 0xa8, 0x1a, 0x87, 0xb6,
    0x81, 0x21, 0xfb, 0xa8, 0xd3, 0x19, 0x78, 0xf8, 0x07, 0xa3, 0x17, 0xe5, 0x0a, 0xa8, 0xa8, 0x28, 0x44, 0x68, 0x28, 0x91, 0x4b,
    0x93, 0x3d, 0xe8, 0xed, 0xd4, 0xa5, 0xc3, 0x9c, 0x9f, 0xf7, 0x1a, 0x4c, 0xe3, 0x64, 0x7f, 0xd7, 0xf6, 0x26, 0x53, 0xb7, 0xd2,
    0x49, 0x5f, 0xcb, 0xa4, 0xc0, 0xf4, 0x7f, 0x87, 0x68, 0x80, 0x03, 0x9e, 0x07, 0x20, 0x4a, 0xa3, 0x66, 0x30, 0x64, 0x30, 0x12,
    0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x08, 0x30, 0x06, 0x01, 0x01, 0xff, 0x02, 0x01, 0x01, 0x30, 0x0e, 0x06,
    0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x01, 0x06, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04,
    0x16, 0x04, 0x14, 0x78, 0x5c, 0xe7, 0x05, 0xb8, 0x6b, 0x8f, 0x4e, 0x6f, 0xc7, 0x93, 0xaa, 0x60, 0xcb, 0x43, 0xea, 0x69, 0x68,
    0x82, 0xd5, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x78, 0x5c, 0xe7, 0x05, 0xb8, 0x6b,
    0x8f, 0x4e, 0x6f, 0xc7, 0x93, 0xaa, 0x60, 0xcb, 0x43, 0xea, 0x69, 0x68, 0x82, 0xd5, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48,
    0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x48, 0x00, 0x30, 0x45, 0x02, 0x20, 0x7b, 0x63, 0xf5, 0xb2, 0xd8, 0x87, 0x97, 0xd2, 0x63,
    0xb0, 0x3f, 0xad, 0x87, 0xbd, 0x3b, 0x8c, 0x86, 0xbc, 0xf0, 0x0f, 0x99, 0x6f, 0xa8, 0x73, 0xe8, 0xfd, 0x2f, 0x80, 0x60, 0xc2,
    0x1d, 0x6f, 0x02, 0x21, 0x00, 0x92, 0xe1, 0x4c, 0xb3, 0xf6, 0x97, 0xc1, 0x8f, 0x8c, 0x85, 0x4d, 0x1f, 0x70, 0x54, 0xbe, 0x6f,
    0x1a, 0xab, 0xce, 0xcc, 0xf3, 0xd5, 0x9e, 0x0c, 0xd2, 0xc2, 0x72, 0x0c, 0xb2, 0xf0, 0xc8, 0x76,
};

extern const ByteSpan sTestCert_PAA_NoVID_ToResignPAIs_Cert = ByteSpan(sTestCert_PAA_NoVID_ToResignPAIs_Cert_Array);

constexpr uint8_t sTestCert_PAA_NoVID_ToResignPAIs_SKID_Array[] = {
    0x78, 0x5C, 0xE7, 0x05, 0xB8, 0x6B, 0x8F, 0x4E, 0x6F, 0xC7, 0x93, 0xAA, 0x60, 0xCB, 0x43, 0xEA, 0x69, 0x68, 0x82, 0xD5,
};

extern const ByteSpan sTestCert_PAA_NoVID_ToResignPAIs_SKID = ByteSpan(sTestCert_PAA_NoVID_ToResignPAIs_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAI-FFF1-8000-Cert.pem

constexpr uint8_t sTestCert_PAI_FFF1_8000_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xd4, 0x30, 0x82, 0x01, 0x7a, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x3e, 0x6c, 0xe6, 0x50, 0x9a, 0xd8,
    0x40, 0xcd, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x30, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x41,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x31, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x39,
    0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x31, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30,
    0x30, 0x30, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce,
    0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x80, 0xdd, 0xf1, 0x1b, 0x22, 0x8f, 0x3e, 0x31, 0xf6, 0x3b, 0xcf, 0x57, 0x98,
    0xda, 0x14, 0x62, 0x3a, 0xeb, 0xbd, 0xe8, 0x2e, 0xf3, 0x78, 0xee, 0xad, 0xbf, 0xb1, 0x8f, 0xe1, 0xab, 0xce, 0x31, 0xd0, 0x8e,
    0xd4, 0xb2, 0x06, 0x04, 0xb6, 0xcc, 0xc6, 0xd9, 0xb5, 0xfa, 0xb6, 0x4e, 0x7d, 0xe1, 0x0c, 0xb7, 0x4b, 0xe0, 0x17, 0xc9, 0xec,
    0x15, 0x16, 0x05, 0x6d, 0x70, 0xf2, 0xcd, 0x0b, 0x22, 0xa3, 0x66, 0x30, 0x64, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01,
    0x01, 0xff, 0x04, 0x08, 0x30, 0x06, 0x01, 0x01, 0xff, 0x02, 0x01, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01,
    0xff, 0x04, 0x04, 0x03, 0x02, 0x01, 0x06, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xaf, 0x42, 0xb7,
    0x09, 0x4d, 0xeb, 0xd5, 0x15, 0xec, 0x6e, 0xcf, 0x33, 0xb8, 0x11, 0x15, 0x22, 0x5f, 0x32, 0x52, 0x88, 0x30, 0x1f, 0x06, 0x03,
    0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x6a, 0xfd, 0x22, 0x77, 0x1f, 0x51, 0x1f, 0xec, 0xbf, 0x16, 0x41, 0x97,
    0x67, 0x10, 0xdc, 0xdc, 0x31, 0xa1, 0x71, 0x7e, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03,
    0x48, 0x00, 0x30, 0x45, 0x02, 0x21, 0x00, 0x96, 0xc9, 0xc8, 0xcf, 0x2e, 0x01, 0x88, 0x60, 0x05, 0xd8, 0xf5, 0xbc, 0x72, 0xc0,
    0x7b, 0x75, 0xfd, 0x9a, 0x57, 0x69, 0x5a, 0xc4, 0x91, 0x11, 0x31, 0x13, 0x8b, 0xea, 0x03, 0x3c, 0xe5, 0x03, 0x02, 0x20, 0x25,
    0x54, 0x94, 0x3b, 0xe5, 0x7d, 0x53, 0xd6, 0xc4, 0x75, 0xf7, 0xd2, 0x3e, 0xbf, 0xcf, 0xc2, 0x03, 0x6c, 0xd2, 0x9b, 0xa6, 0x39,
    0x3e, 0xc7, 0xef, 0xad, 0x87, 0x14, 0xab, 0x71, 0x82, 0x19,
};

extern const ByteSpan sTestCert_PAI_FFF1_8000_Cert = ByteSpan(sTestCert_PAI_FFF1_8000_Cert_Array);

constexpr uint8_t sTestCert_PAI_FFF1_8000_SKID_Array[] = {
    0xAF, 0x42, 0xB7, 0x09, 0x4D, 0xEB, 0xD5, 0x15, 0xEC, 0x6E, 0xCF, 0x33, 0xB8, 0x11, 0x15, 0x22, 0x5F, 0x32, 0x52, 0x88,
};

extern const ByteSpan sTestCert_PAI_FFF1_8000_SKID = ByteSpan(sTestCert_PAI_FFF1_8000_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAI-FFF1-8000-Key.pem

constexpr uint8_t sTestCert_PAI_FFF1_8000_PublicKey_Array[] = {
    0x04, 0x80, 0xdd, 0xf1, 0x1b, 0x22, 0x8f, 0x3e, 0x31, 0xf6, 0x3b, 0xcf, 0x57, 0x98, 0xda, 0x14, 0x62,
    0x3a, 0xeb, 0xbd, 0xe8, 0x2e, 0xf3, 0x78, 0xee, 0xad, 0xbf, 0xb1, 0x8f, 0xe1, 0xab, 0xce, 0x31, 0xd0,
    0x8e, 0xd4, 0xb2, 0x06, 0x04, 0xb6, 0xcc, 0xc6, 0xd9, 0xb5, 0xfa, 0xb6, 0x4e, 0x7d, 0xe1, 0x0c, 0xb7,
    0x4b, 0xe0, 0x17, 0xc9, 0xec, 0x15, 0x16, 0x05, 0x6d, 0x70, 0xf2, 0xcd, 0x0b, 0x22,
};

extern const ByteSpan sTestCert_PAI_FFF1_8000_PublicKey = ByteSpan(sTestCert_PAI_FFF1_8000_PublicKey_Array);

constexpr uint8_t sTestCert_PAI_FFF1_8000_PrivateKey_Array[] = {
    0x46, 0x7b, 0x2d, 0x8a, 0x69, 0xb3, 0x5c, 0xfe, 0x6b, 0xdb, 0x0f, 0xc3, 0x6a, 0x81, 0xa7, 0xd8,
    0xe3, 0xe4, 0x5d, 0x06, 0x2d, 0xf7, 0xcd, 0xb8, 0x22, 0x58, 0xf3, 0xc8, 0x40, 0x1d, 0xc3, 0x18,
};

extern const ByteSpan sTestCert_PAI_FFF1_8000_PrivateKey = ByteSpan(sTestCert_PAI_FFF1_8000_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAI-FFF2-8001-Cert.pem

constexpr uint8_t sTestCert_PAI_FFF2_8001_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xbd, 0x30, 0x82, 0x01, 0x64, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x65, 0x3a, 0x88, 0x7c, 0x1b, 0xfe,
    0x16, 0x2e, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x1a, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x41,
    0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x39, 0x39,
    0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03,
    0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49, 0x31,
    0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x32,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30, 0x30,
    0x31, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d,
    0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x2c, 0x06, 0x3c, 0x20, 0x8b, 0xb7, 0xcf, 0x3f, 0x89, 0xd8, 0x9c, 0x70, 0xb2, 0x68,
    0xde, 0xc2, 0xf0, 0xad, 0x89, 0x19, 0x2f, 0x02, 0x2f, 0x95, 0x47, 0xad, 0xcc, 0x72, 0x2a, 0x06, 0x80, 0x69, 0xca, 0x72, 0x35,
    0x60, 0x83, 0xb7, 0x24, 0x77, 0x1b, 0x21, 0x76, 0x4c, 0xb2, 0x4c, 0x57, 0x11, 0xb5, 0x3e, 0xfe, 0x87, 0xfe, 0x4f, 0xe0, 0xb4,
    0x73, 0x14, 0x1b, 0xcc, 0xa1, 0x04, 0xb2, 0x8d, 0xa3, 0x66, 0x30, 0x64, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01,
    0xff, 0x04, 0x08, 0x30, 0x06, 0x01, 0x01, 0xff, 0x02, 0x01, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff,
    0x04, 0x04, 0x03, 0x02, 0x01, 0x06, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xd0, 0x5a, 0x9b, 0x67,
    0x71, 0xa1, 0xa3, 0x7a, 0x9b, 0xc1, 0x9d, 0x95, 0xe8, 0xb5, 0xdb, 0x8f, 0x43, 0x00, 0xb6, 0x3a, 0x30, 0x1f, 0x06, 0x03, 0x55,
    0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x78, 0x5c, 0xe7, 0x05, 0xb8, 0x6b, 0x8f, 0x4e, 0x6f, 0xc7, 0x93, 0xaa, 0x60,
    0xcb, 0x43, 0xea, 0x69, 0x68, 0x82, 0xd5, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x47,
    0x00, 0x30, 0x44, 0x02, 0x20, 0x60, 0x5e, 0x97, 0xdc, 0x65, 0xe0, 0xc5, 0x73, 0xf9, 0xf8, 0xf6, 0xa5, 0xb9, 0x68, 0x1c, 0xdb,
    0x9b, 0x6b, 0xee, 0x1e, 0x9f, 0xfd, 0x58, 0x96, 0x73, 0xaa, 0x03, 0xd3, 0x61, 0x5e, 0xfd, 0x60, 0x02, 0x20, 0x26, 0xe8, 0x90,
    0xb7, 0x1e, 0x1f, 0x9d, 0x42, 0xe7, 0x93, 0xa4, 0x8e, 0xcc, 0xdb, 0xc8, 0xf8, 0x08, 0x18, 0x07, 0xf2, 0xfb, 0x37, 0x0a, 0x3d,
    0x24, 0x97, 0xaf, 0x37, 0xdc, 0x4a, 0xa5, 0x33,
};

extern const ByteSpan sTestCert_PAI_FFF2_8001_Cert = ByteSpan(sTestCert_PAI_FFF2_8001_Cert_Array);

constexpr uint8_t sTestCert_PAI_FFF2_8001_SKID_Array[] = {
    0xD0, 0x5A, 0x9B, 0x67, 0x71, 0xA1, 0xA3, 0x7A, 0x9B, 0xC1, 0x9D, 0x95, 0xE8, 0xB5, 0xDB, 0x8F, 0x43, 0x00, 0xB6, 0x3A,
};

extern const ByteSpan sTestCert_PAI_FFF2_8001_SKID = ByteSpan(sTestCert_PAI_FFF2_8001_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAI-FFF2-8001-Key.pem

constexpr uint8_t sTestCert_PAI_FFF2_8001_PublicKey_Array[] = {
    0x04, 0x2c, 0x06, 0x3c, 0x20, 0x8b, 0xb7, 0xcf, 0x3f, 0x89, 0xd8, 0x9c, 0x70, 0xb2, 0x68, 0xde, 0xc2,
    0xf0, 0xad, 0x89, 0x19, 0x2f, 0x02, 0x2f, 0x95, 0x47, 0xad, 0xcc, 0x72, 0x2a, 0x06, 0x80, 0x69, 0xca,
    0x72, 0x35, 0x60, 0x83, 0xb7, 0x24, 0x77, 0x1b, 0x21, 0x76, 0x4c, 0xb2, 0x4c, 0x57, 0x11, 0xb5, 0x3e,
    0xfe, 0x87, 0xfe, 0x4f, 0xe0, 0xb4, 0x73, 0x14, 0x1b, 0xcc, 0xa1, 0x04, 0xb2, 0x8d,
};

extern const ByteSpan sTestCert_PAI_FFF2_8001_PublicKey = ByteSpan(sTestCert_PAI_FFF2_8001_PublicKey_Array);

constexpr uint8_t sTestCert_PAI_FFF2_8001_PrivateKey_Array[] = {
    0xec, 0x56, 0xfc, 0x11, 0x59, 0xba, 0xc5, 0xa9, 0x00, 0xb4, 0x53, 0x7e, 0xd4, 0xfa, 0x0e, 0xd4,
    0x54, 0xa3, 0xfd, 0x45, 0xea, 0xfe, 0xc6, 0xc3, 0xf6, 0x5f, 0x61, 0x8f, 0x7a, 0x2e, 0x1a, 0xe1,
};

extern const ByteSpan sTestCert_PAI_FFF2_8001_PrivateKey = ByteSpan(sTestCert_PAI_FFF2_8001_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAI-FFF2-8001-Resigned-Cert.pem

constexpr uint8_t sTestCert_PAI_FFF2_8001_Resigned_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xcd, 0x30, 0x82, 0x01, 0x73, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x62, 0x72, 0xc7, 0xaa, 0x0a, 0x13,
    0x20, 0x9d, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x29, 0x31, 0x27, 0x30, 0x25, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x1e, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x41,
    0x20, 0x54, 0x6f, 0x20, 0x52, 0x65, 0x73, 0x69, 0x67, 0x6e, 0x20, 0x50, 0x41, 0x49, 0x73, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31,
    0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x39, 0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31,
    0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d,
    0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b,
    0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a,
    0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x31, 0x30, 0x59, 0x30, 0x13, 0x06,
    0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00,
    0x04, 0x2c, 0x06, 0x3c, 0x20, 0x8b, 0xb7, 0xcf, 0x3f, 0x89, 0xd8, 0x9c, 0x70, 0xb2, 0x68, 0xde, 0xc2, 0xf0, 0xad, 0x89, 0x19,
    0x2f, 0x02, 0x2f, 0x95, 0x47, 0xad, 0xcc, 0x72, 0x2a, 0x06, 0x80, 0x69, 0xca, 0x72, 0x35, 0x60, 0x83, 0xb7, 0x24, 0x77, 0x1b,
    0x21, 0x76, 0x4c, 0xb2, 0x4c, 0x57, 0x11, 0xb5, 0x3e, 0xfe, 0x87, 0xfe, 0x4f, 0xe0, 0xb4, 0x73, 0x14, 0x1b, 0xcc, 0xa1, 0x04,
    0xb2, 0x8d, 0xa3, 0x66, 0x30, 0x64, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x08, 0x30, 0x06, 0x01,
    0x01, 0xff, 0x02, 0x01, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x01, 0x06,
    0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xd0, 0x5a, 0x9b, 0x67, 0x71, 0xa1, 0xa3, 0x7a, 0x9b, 0xc1,
    0x9d, 0x95, 0xe8, 0xb5, 0xdb, 0x8f, 0x43, 0x00, 0xb6, 0x3a, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16,
    0x80, 0x14, 0x78, 0x5c, 0xe7, 0x05, 0xb8, 0x6b, 0x8f, 0x4e, 0x6f, 0xc7, 0x93, 0xaa, 0x60, 0xcb, 0x43, 0xea, 0x69, 0x68, 0x82,
    0xd5, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x48, 0x00, 0x30, 0x45, 0x02, 0x21, 0x00,
    0x9f, 0x0d, 0x54, 0x37, 0xe9, 0x09, 0x9f, 0x85, 0x38, 0xca, 0x53, 0x0e, 0xe8, 0x9d, 0x2a, 0xb3, 0x64, 0x24, 0x5c, 0xe9, 0x2b,
    0x21, 0xc9, 0xca, 0xbe, 0xc7, 0x9b, 0xd5, 0x4d, 0x18, 0xe6, 0x1c, 0x02, 0x20, 0x76, 0xd0, 0xe7, 0x1f, 0xf5, 0xe8, 0x50, 0xea,
    0xdd, 0x07, 0xd4, 0x83, 0x1e, 0x75, 0x46, 0xd1, 0x79, 0xd6, 0xe2, 0xb6, 0xd8, 0xe1, 0xc6, 0x17, 0xac, 0x56, 0xa5, 0xec, 0xa1,
    0x54, 0x02, 0x38,
};

extern const ByteSpan sTestCert_PAI_FFF2_8001_Resigned_Cert = ByteSpan(sTestCert_PAI_FFF2_8001_Resigned_Cert_Array);

constexpr uint8_t sTestCert_PAI_FFF2_8001_Resigned_SKID_Array[] = {
    0xD0, 0x5A, 0x9B, 0x67, 0x71, 0xA1, 0xA3, 0x7A, 0x9B, 0xC1, 0x9D, 0x95, 0xE8, 0xB5, 0xDB, 0x8F, 0x43, 0x00, 0xB6, 0x3A,
};

extern const ByteSpan sTestCert_PAI_FFF2_8001_Resigned_SKID = ByteSpan(sTestCert_PAI_FFF2_8001_Resigned_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAI-FFF2-8001-ResignedSKIDDiff-Cert.pem

constexpr uint8_t sTestCert_PAI_FFF2_8001_ResignedSKIDDiff_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xcc, 0x30, 0x82, 0x01, 0x73, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x4f, 0x3c, 0xc2, 0xdd, 0xb1, 0x1d,
    0xc4, 0xcb, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x29, 0x31, 0x27, 0x30, 0x25, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x1e, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x41,
    0x20, 0x54, 0x6f, 0x20, 0x52, 0x65, 0x73, 0x69, 0x67, 0x6e, 0x20, 0x50, 0x41, 0x49, 0x73, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31,
    0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x39, 0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31,
    0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d,
    0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b,
    0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a,
    0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x31, 0x30, 0x59, 0x30, 0x13, 0x06,
    0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00,
    0x04, 0xa1, 0xfa, 0x48, 0x2a, 0x3c, 0x92, 0xf3, 0xa4, 0x66, 0x63, 0x9c, 0xe3, 0x35, 0x71, 0x38, 0xb6, 0xa4, 0xc9, 0xdd, 0x44,
    0x76, 0x16, 0x8e, 0xc7, 0xec, 0x5c, 0x45, 0x08, 0x2b, 0xdc, 0x1b, 0xc0, 0x6d, 0x9b, 0x94, 0xd3, 0x4c, 0x79, 0xdf, 0x05, 0xdf,
    0xc4, 0x0b, 0x27, 0x10, 0x80, 0x44, 0x0b, 0x7a, 0x88, 0xb4, 0x36, 0x38, 0xac, 0x4c, 0x9e, 0xe4, 0x39, 0xc7, 0x41, 0x9c, 0x14,
    0xdd, 0xfa, 0xa3, 0x66, 0x30, 0x64, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x08, 0x30, 0x06, 0x01,
    0x01, 0xff, 0x02, 0x01, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x01, 0x06,
    0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xdb, 0x77, 0xd2, 0x5b, 0x63, 0x6d, 0x04, 0x38, 0xa1, 0x92,
    0x1c, 0x41, 0x52, 0x10, 0x3e, 0xea, 0xe2, 0xc4, 0x3b, 0x07, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16,
    0x80, 0x14, 0x78, 0x5c, 0xe7, 0x05, 0xb8, 0x6b, 0x8f, 0x4e, 0x6f, 0xc7, 0x93, 0xaa, 0x60, 0xcb, 0x43, 0xea, 0x69, 0x68, 0x82,
    0xd5, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x47, 0x00, 0x30, 0x44, 0x02, 0x20, 0x05,
    0x1f, 0x30, 0x34, 0x56, 0xfd, 0xeb, 0x01, 0xff, 0xfd, 0xe4, 0x79, 0x9f, 0xfb, 0xb4, 0x3d, 0x6d, 0xd1, 0x48, 0x65, 0x34, 0x8a,
    0xd6, 0x3f, 0x92, 0xba, 0xcb, 0xca, 0x60, 0xc8, 0x1c, 0x29, 0x02, 0x20, 0x11, 0x12, 0xc3, 0x0a, 0xfb, 0x2a, 0x92, 0xf6, 0x00,
    0x2e, 0x14, 0x78, 0x39, 0x20, 0x3b, 0x45, 0xdb, 0xb8, 0x25, 0x51, 0x61, 0xb0, 0xfd, 0x0c, 0xd0, 0x5e, 0x0c, 0xa4, 0xc4, 0x70,
    0xc8, 0xf1,
};

extern const ByteSpan sTestCert_PAI_FFF2_8001_ResignedSKIDDiff_Cert = ByteSpan(sTestCert_PAI_FFF2_8001_ResignedSKIDDiff_Cert_Array);

constexpr uint8_t sTestCert_PAI_FFF2_8001_ResignedSKIDDiff_SKID_Array[] = {
    0xDB, 0x77, 0xD2, 0x5B, 0x63, 0x6D, 0x04, 0x38, 0xA1, 0x92, 0x1C, 0x41, 0x52, 0x10, 0x3E, 0xEA, 0xE2, 0xC4, 0x3B, 0x07,
};

extern const ByteSpan sTestCert_PAI_FFF2_8001_ResignedSKIDDiff_SKID = ByteSpan(sTestCert_PAI_FFF2_8001_ResignedSKIDDiff_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAI-FFF2-8001-ResignedSKIDDiff-Key.pem

constexpr uint8_t sTestCert_PAI_FFF2_8001_ResignedSKIDDiff_PublicKey_Array[] = {
    0x04, 0xa1, 0xfa, 0x48, 0x2a, 0x3c, 0x92, 0xf3, 0xa4, 0x66, 0x63, 0x9c, 0xe3, 0x35, 0x71, 0x38, 0xb6,
    0xa4, 0xc9, 0xdd, 0x44, 0x76, 0x16, 0x8e, 0xc7, 0xec, 0x5c, 0x45, 0x08, 0x2b, 0xdc, 0x1b, 0xc0, 0x6d,
    0x9b, 0x94, 0xd3, 0x4c, 0x79, 0xdf, 0x05, 0xdf, 0xc4, 0x0b, 0x27, 0x10, 0x80, 0x44, 0x0b, 0x7a, 0x88,
    0xb4, 0x36, 0x38, 0xac, 0x4c, 0x9e, 0xe4, 0x39, 0xc7, 0x41, 0x9c, 0x14, 0xdd, 0xfa,
};

extern const ByteSpan sTestCert_PAI_FFF2_8001_ResignedSKIDDiff_PublicKey =
    ByteSpan(sTestCert_PAI_FFF2_8001_ResignedSKIDDiff_PublicKey_Array);

constexpr uint8_t sTestCert_PAI_FFF2_8001_ResignedSKIDDiff_PrivateKey_Array[] = {
    0xd1, 0xae, 0x62, 0xfb, 0x4d, 0xb3, 0x05, 0xfe, 0xa3, 0x8c, 0xa3, 0xe9, 0x44, 0x74, 0x4d, 0x1b,
    0xc0, 0xbf, 0xf9, 0x94, 0xa8, 0x74, 0xdc, 0xc5, 0xf5, 0x7a, 0x17, 0xf8, 0xb5, 0x18, 0x51, 0x35,
};

extern const ByteSpan sTestCert_PAI_FFF2_8001_ResignedSKIDDiff_PrivateKey =
    ByteSpan(sTestCert_PAI_FFF2_8001_ResignedSKIDDiff_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAI-FFF2-8001-ResignedSubjectDiff-Cert.pem

constexpr uint8_t sTestCert_PAI_FFF2_8001_ResignedSubjectDiff_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xd6, 0x30, 0x82, 0x01, 0x7c, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x47, 0xed, 0x63, 0x68, 0x21, 0x97,
    0x69, 0x31, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x29, 0x31, 0x27, 0x30, 0x25, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x1e, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x41,
    0x20, 0x54, 0x6f, 0x20, 0x52, 0x65, 0x73, 0x69, 0x67, 0x6e, 0x20, 0x50, 0x41, 0x49, 0x73, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31,
    0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x39, 0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31,
    0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x4f, 0x31, 0x21, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x18, 0x4d,
    0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49, 0x20, 0x52, 0x65, 0x73, 0x69, 0x67, 0x6e,
    0x65, 0x64, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46,
    0x46, 0x46, 0x32, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04,
    0x38, 0x30, 0x30, 0x31, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86,
    0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x2c, 0x06, 0x3c, 0x20, 0x8b, 0xb7, 0xcf, 0x3f, 0x89, 0xd8, 0x9c,
    0x70, 0xb2, 0x68, 0xde, 0xc2, 0xf0, 0xad, 0x89, 0x19, 0x2f, 0x02, 0x2f, 0x95, 0x47, 0xad, 0xcc, 0x72, 0x2a, 0x06, 0x80, 0x69,
    0xca, 0x72, 0x35, 0x60, 0x83, 0xb7, 0x24, 0x77, 0x1b, 0x21, 0x76, 0x4c, 0xb2, 0x4c, 0x57, 0x11, 0xb5, 0x3e, 0xfe, 0x87, 0xfe,
    0x4f, 0xe0, 0xb4, 0x73, 0x14, 0x1b, 0xcc, 0xa1, 0x04, 0xb2, 0x8d, 0xa3, 0x66, 0x30, 0x64, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1d,
    0x13, 0x01, 0x01, 0xff, 0x04, 0x08, 0x30, 0x06, 0x01, 0x01, 0xff, 0x02, 0x01, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f,
    0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x01, 0x06, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xd0,
    0x5a, 0x9b, 0x67, 0x71, 0xa1, 0xa3, 0x7a, 0x9b, 0xc1, 0x9d, 0x95, 0xe8, 0xb5, 0xdb, 0x8f, 0x43, 0x00, 0xb6, 0x3a, 0x30, 0x1f,
    0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x78, 0x5c, 0xe7, 0x05, 0xb8, 0x6b, 0x8f, 0x4e, 0x6f, 0xc7,
    0x93, 0xaa, 0x60, 0xcb, 0x43, 0xea, 0x69, 0x68, 0x82, 0xd5, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03,
    0x02, 0x03, 0x48, 0x00, 0x30, 0x45, 0x02, 0x21, 0x00, 0xe8, 0xb2, 0xc6, 0x11, 0x70, 0x4a, 0x54, 0x06, 0x79, 0x0c, 0xc5, 0xc8,
    0x87, 0x38, 0x38, 0x89, 0x03, 0x9c, 0x18, 0xed, 0xf2, 0x3d, 0x5e, 0xe2, 0x55, 0x15, 0x0e, 0x0f, 0xf4, 0x75, 0xef, 0xc9, 0x02,
    0x20, 0x7f, 0x0d, 0x94, 0x91, 0xe6, 0xe6, 0x91, 0xc0, 0x55, 0xcb, 0xa0, 0xb0, 0x4a, 0x72, 0x90, 0x76, 0xa2, 0x58, 0x2f, 0xb3,
    0xb9, 0xe3, 0x0a, 0x15, 0xbe, 0xc3, 0xd0, 0x5c, 0xe9, 0x11, 0x2f, 0xfc,
};

extern const ByteSpan sTestCert_PAI_FFF2_8001_ResignedSubjectDiff_Cert =
    ByteSpan(sTestCert_PAI_FFF2_8001_ResignedSubjectDiff_Cert_Array);

constexpr uint8_t sTestCert_PAI_FFF2_8001_ResignedSubjectDiff_SKID_Array[] = {
    0xD0, 0x5A, 0x9B, 0x67, 0x71, 0xA1, 0xA3, 0x7A, 0x9B, 0xC1, 0x9D, 0x95, 0xE8, 0xB5, 0xDB, 0x8F, 0x43, 0x00, 0xB6, 0x3A,
};

extern const ByteSpan sTestCert_PAI_FFF2_8001_ResignedSubjectDiff_SKID =
    ByteSpan(sTestCert_PAI_FFF2_8001_ResignedSubjectDiff_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAI-FFF2-8004-FB-Cert.pem

constexpr uint8_t sTestCert_PAI_FFF2_8004_FB_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xa6, 0x30, 0x82, 0x01, 0x4c, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x55, 0x40, 0xe9, 0xa2, 0xdc, 0x98,
    0x93, 0x3b, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x1a, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x41,
    0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x39, 0x39,
    0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x2e, 0x31, 0x2c, 0x30, 0x2a, 0x06, 0x03,
    0x55, 0x04, 0x03, 0x0c, 0x23, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49, 0x20,
    0x4d, 0x76, 0x69, 0x64, 0x3a, 0x46, 0x46, 0x46, 0x32, 0x20, 0x4d, 0x70, 0x69, 0x64, 0x3a, 0x38, 0x30, 0x30, 0x34, 0x30, 0x59,
    0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07,
    0x03, 0x42, 0x00, 0x04, 0xd3, 0xf3, 0x9b, 0xc2, 0xe3, 0x36, 0xe7, 0x6c, 0x88, 0xd2, 0x31, 0x38, 0x08, 0x9b, 0xc9, 0x63, 0x90,
    0xe8, 0x2b, 0x07, 0x71, 0x49, 0x42, 0x42, 0x53, 0x9b, 0xf5, 0x7c, 0xda, 0x3c, 0xce, 0x2b, 0x99, 0x1d, 0x63, 0x8d, 0xea, 0x52,
    0xbb, 0xdc, 0x68, 0x30, 0x83, 0xb6, 0xdc, 0x7c, 0xc4, 0x38, 0x87, 0xc1, 0xc9, 0x43, 0xf5, 0xf3, 0xea, 0xf4, 0x2f, 0x54, 0x2f,
    0x12, 0x7e, 0xa6, 0x37, 0x0c, 0xa3, 0x66, 0x30, 0x64, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x08,
    0x30, 0x06, 0x01, 0x01, 0xff, 0x02, 0x01, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03,
    0x02, 0x01, 0x06, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x15, 0x6e, 0xcd, 0x8c, 0x14, 0xbe, 0x80,
    0xb4, 0x10, 0xae, 0xe4, 0x02, 0xa3, 0xf2, 0x33, 0x17, 0xe7, 0x34, 0x06, 0x74, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04,
    0x18, 0x30, 0x16, 0x80, 0x14, 0x78, 0x5c, 0xe7, 0x05, 0xb8, 0x6b, 0x8f, 0x4e, 0x6f, 0xc7, 0x93, 0xaa, 0x60, 0xcb, 0x43, 0xea,
    0x69, 0x68, 0x82, 0xd5, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x48, 0x00, 0x30, 0x45,
    0x02, 0x21, 0x00, 0xef, 0xe5, 0x8e, 0xfd, 0x49, 0x15, 0x67, 0x80, 0xc6, 0x50, 0xb3, 0x93, 0x2c, 0x82, 0x2d, 0x56, 0x11, 0xbb,
    0xac, 0x2f, 0xa6, 0x9e, 0xa2, 0x48, 0x7c, 0x0b, 0xb0, 0x89, 0xbf, 0x64, 0x09, 0x10, 0x02, 0x20, 0x56, 0x8c, 0x09, 0xd0, 0xe8,
    0xfe, 0x7f, 0x84, 0xfa, 0x13, 0x81, 0x5c, 0x35, 0xd5, 0x7a, 0x69, 0x5b, 0xcf, 0x4c, 0xe7, 0x31, 0xa2, 0xe7, 0xfc, 0xa7, 0x2a,
    0xba, 0x8c, 0x43, 0x21, 0xf6, 0xfb,
};

extern const ByteSpan sTestCert_PAI_FFF2_8004_FB_Cert = ByteSpan(sTestCert_PAI_FFF2_8004_FB_Cert_Array);

constexpr uint8_t sTestCert_PAI_FFF2_8004_FB_SKID_Array[] = {
    0x15, 0x6E, 0xCD, 0x8C, 0x14, 0xBE, 0x80, 0xB4, 0x10, 0xAE, 0xE4, 0x02, 0xA3, 0xF2, 0x33, 0x17, 0xE7, 0x34, 0x06, 0x74,
};

extern const ByteSpan sTestCert_PAI_FFF2_8004_FB_SKID = ByteSpan(sTestCert_PAI_FFF2_8004_FB_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAI-FFF2-8004-FB-Key.pem

constexpr uint8_t sTestCert_PAI_FFF2_8004_FB_PublicKey_Array[] = {
    0x04, 0xd3, 0xf3, 0x9b, 0xc2, 0xe3, 0x36, 0xe7, 0x6c, 0x88, 0xd2, 0x31, 0x38, 0x08, 0x9b, 0xc9, 0x63,
    0x90, 0xe8, 0x2b, 0x07, 0x71, 0x49, 0x42, 0x42, 0x53, 0x9b, 0xf5, 0x7c, 0xda, 0x3c, 0xce, 0x2b, 0x99,
    0x1d, 0x63, 0x8d, 0xea, 0x52, 0xbb, 0xdc, 0x68, 0x30, 0x83, 0xb6, 0xdc, 0x7c, 0xc4, 0x38, 0x87, 0xc1,
    0xc9, 0x43, 0xf5, 0xf3, 0xea, 0xf4, 0x2f, 0x54, 0x2f, 0x12, 0x7e, 0xa6, 0x37, 0x0c,
};

extern const ByteSpan sTestCert_PAI_FFF2_8004_FB_PublicKey = ByteSpan(sTestCert_PAI_FFF2_8004_FB_PublicKey_Array);

constexpr uint8_t sTestCert_PAI_FFF2_8004_FB_PrivateKey_Array[] = {
    0x61, 0xd1, 0x9e, 0x11, 0xf2, 0x1a, 0xd9, 0x78, 0x2c, 0x79, 0x45, 0x69, 0xcd, 0x35, 0x75, 0xf3,
    0x94, 0x1d, 0x27, 0x82, 0xd8, 0x9d, 0xf5, 0xa7, 0x1b, 0xdf, 0xbb, 0x6a, 0xa8, 0xb4, 0xd1, 0x69,
};

extern const ByteSpan sTestCert_PAI_FFF2_8004_FB_PrivateKey = ByteSpan(sTestCert_PAI_FFF2_8004_FB_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAI-FFF2-8005-Val1SecBefore-Cert.pem

constexpr uint8_t sTestCert_PAI_FFF2_8005_Val1SecBefore_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xbd, 0x30, 0x82, 0x01, 0x62, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x1c, 0x18, 0x43, 0x52, 0x67, 0x55,
    0x39, 0x1c, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x1a, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x41,
    0x30, 0x1e, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x32, 0x5a, 0x17, 0x0d, 0x32, 0x32,
    0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x31, 0x5a, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04,
    0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49, 0x31, 0x14, 0x30,
    0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x31, 0x14,
    0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x35, 0x30,
    0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01,
    0x07, 0x03, 0x42, 0x00, 0x04, 0x13, 0x3f, 0xdf, 0x53, 0x64, 0x46, 0x1f, 0xf2, 0xc0, 0x2b, 0xed, 0x7f, 0x88, 0x20, 0x58, 0x1d,
    0xb2, 0xc5, 0xf5, 0x68, 0x08, 0x3c, 0x8c, 0xa3, 0x62, 0x36, 0x1c, 0x2f, 0xf7, 0x77, 0xd1, 0x3f, 0xe3, 0x80, 0x1f, 0xe7, 0xe1,
    0x6b, 0xb8, 0xe8, 0xe6, 0xa0, 0x97, 0x27, 0xb0, 0x5d, 0x0f, 0x56, 0x0c, 0x59, 0xb8, 0x46, 0xe6, 0x2d, 0xc4, 0xe8, 0xf2, 0x9b,
    0x87, 0x26, 0x41, 0xf2, 0x0c, 0xce, 0xa3, 0x66, 0x30, 0x64, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04,
    0x08, 0x30, 0x06, 0x01, 0x01, 0xff, 0x02, 0x01, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04,
    0x03, 0x02, 0x01, 0x06, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xb3, 0x0e, 0x90, 0x53, 0x42, 0x6f,
    0x02, 0x0d, 0x13, 0x91, 0x1b, 0x03, 0xcb, 0x57, 0x8d, 0xcb, 0x4a, 0xab, 0x9a, 0x52, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23,
    0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x78, 0x5c, 0xe7, 0x05, 0xb8, 0x6b, 0x8f, 0x4e, 0x6f, 0xc7, 0x93, 0xaa, 0x60, 0xcb, 0x43,
    0xea, 0x69, 0x68, 0x82, 0xd5, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x49, 0x00, 0x30,
    0x46, 0x02, 0x21, 0x00, 0xcb, 0x8c, 0x99, 0x3b, 0x6e, 0x97, 0x90, 0x17, 0x1d, 0x47, 0x75, 0x89, 0x39, 0x51, 0xaa, 0xea, 0xcd,
    0x8e, 0x64, 0xce, 0x94, 0xae, 0xd8, 0x08, 0x2e, 0x48, 0x45, 0x4a, 0xb5, 0x90, 0xb3, 0x78, 0x02, 0x21, 0x00, 0xd4, 0xcd, 0xf2,
    0x94, 0x3c, 0xfc, 0x28, 0x98, 0x01, 0x24, 0xbc, 0x67, 0x20, 0xd7, 0x25, 0xf4, 0x2e, 0x0c, 0x6f, 0xcb, 0x47, 0x7d, 0x95, 0xc7,
    0xdd, 0x8c, 0xfe, 0x20, 0x7d, 0x3a, 0x40, 0x71,
};

extern const ByteSpan sTestCert_PAI_FFF2_8005_Val1SecBefore_Cert = ByteSpan(sTestCert_PAI_FFF2_8005_Val1SecBefore_Cert_Array);

constexpr uint8_t sTestCert_PAI_FFF2_8005_Val1SecBefore_SKID_Array[] = {
    0xB3, 0x0E, 0x90, 0x53, 0x42, 0x6F, 0x02, 0x0D, 0x13, 0x91, 0x1B, 0x03, 0xCB, 0x57, 0x8D, 0xCB, 0x4A, 0xAB, 0x9A, 0x52,
};

extern const ByteSpan sTestCert_PAI_FFF2_8005_Val1SecBefore_SKID = ByteSpan(sTestCert_PAI_FFF2_8005_Val1SecBefore_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAI-FFF2-8005-Val1SecBefore-Key.pem

constexpr uint8_t sTestCert_PAI_FFF2_8005_Val1SecBefore_PublicKey_Array[] = {
    0x04, 0x13, 0x3f, 0xdf, 0x53, 0x64, 0x46, 0x1f, 0xf2, 0xc0, 0x2b, 0xed, 0x7f, 0x88, 0x20, 0x58, 0x1d,
    0xb2, 0xc5, 0xf5, 0x68, 0x08, 0x3c, 0x8c, 0xa3, 0x62, 0x36, 0x1c, 0x2f, 0xf7, 0x77, 0xd1, 0x3f, 0xe3,
    0x80, 0x1f, 0xe7, 0xe1, 0x6b, 0xb8, 0xe8, 0xe6, 0xa0, 0x97, 0x27, 0xb0, 0x5d, 0x0f, 0x56, 0x0c, 0x59,
    0xb8, 0x46, 0xe6, 0x2d, 0xc4, 0xe8, 0xf2, 0x9b, 0x87, 0x26, 0x41, 0xf2, 0x0c, 0xce,
};

extern const ByteSpan sTestCert_PAI_FFF2_8005_Val1SecBefore_PublicKey =
    ByteSpan(sTestCert_PAI_FFF2_8005_Val1SecBefore_PublicKey_Array);

constexpr uint8_t sTestCert_PAI_FFF2_8005_Val1SecBefore_PrivateKey_Array[] = {
    0xa6, 0xc3, 0x7a, 0x3f, 0xbd, 0x2b, 0xdc, 0x89, 0x43, 0xdf, 0x51, 0x7f, 0x76, 0x6b, 0x99, 0x82,
    0x3a, 0x91, 0x84, 0xb9, 0x7e, 0xdc, 0x0b, 0x6a, 0x23, 0x11, 0x8a, 0x00, 0xd0, 0x99, 0x90, 0x72,
};

extern const ByteSpan sTestCert_PAI_FFF2_8005_Val1SecBefore_PrivateKey =
    ByteSpan(sTestCert_PAI_FFF2_8005_Val1SecBefore_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAI-FFF2-8005-ValInFuture-Cert.pem

constexpr uint8_t sTestCert_PAI_FFF2_8005_ValInFuture_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xbc, 0x30, 0x82, 0x01, 0x62, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x4d, 0x5d, 0xa9, 0xcb, 0xe0, 0x65,
    0x19, 0x9b, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x1a, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x41,
    0x30, 0x1e, 0x17, 0x0d, 0x33, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x17, 0x0d, 0x33, 0x32,
    0x30, 0x36, 0x32, 0x37, 0x31, 0x34, 0x32, 0x33, 0x34, 0x32, 0x5a, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04,
    0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49, 0x31, 0x14, 0x30,
    0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x31, 0x14,
    0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x35, 0x30,
    0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01,
    0x07, 0x03, 0x42, 0x00, 0x04, 0x19, 0x98, 0xe9, 0x1c, 0xaf, 0x72, 0xfe, 0x63, 0xf3, 0x07, 0xfd, 0xe1, 0x31, 0xbd, 0x66, 0x89,
    0xaa, 0xe3, 0x49, 0x01, 0x68, 0xe8, 0xc2, 0x55, 0xeb, 0x1c, 0xe9, 0x2c, 0x8c, 0xc5, 0x07, 0xf8, 0x35, 0x05, 0x93, 0x16, 0x33,
    0x3c, 0x58, 0xec, 0x87, 0x66, 0x31, 0xc7, 0x82, 0xd6, 0x6e, 0xcc, 0x0b, 0xc3, 0x44, 0x31, 0x8c, 0x32, 0x25, 0x2b, 0xe8, 0x8f,
    0x0e, 0xc7, 0x73, 0xdb, 0xd4, 0xa8, 0xa3, 0x66, 0x30, 0x64, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04,
    0x08, 0x30, 0x06, 0x01, 0x01, 0xff, 0x02, 0x01, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04,
    0x03, 0x02, 0x01, 0x06, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x29, 0x38, 0xba, 0xc7, 0xf6, 0x23,
    0xe1, 0x83, 0x26, 0x43, 0xa5, 0xeb, 0x76, 0x2b, 0xaf, 0xd4, 0x0a, 0x62, 0xc0, 0x21, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23,
    0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x78, 0x5c, 0xe7, 0x05, 0xb8, 0x6b, 0x8f, 0x4e, 0x6f, 0xc7, 0x93, 0xaa, 0x60, 0xcb, 0x43,
    0xea, 0x69, 0x68, 0x82, 0xd5, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x48, 0x00, 0x30,
    0x45, 0x02, 0x21, 0x00, 0xe9, 0x97, 0x32, 0x71, 0x1d, 0xf0, 0xe5, 0x15, 0x3e, 0x20, 0xc2, 0xa6, 0x6e, 0xa0, 0xde, 0x40, 0x15,
    0xd5, 0x9e, 0x28, 0x44, 0xe4, 0x78, 0xc6, 0xab, 0x11, 0x08, 0x70, 0xf1, 0x40, 0xcf, 0xdb, 0x02, 0x20, 0x0c, 0x2d, 0x04, 0x23,
    0x48, 0x7a, 0x99, 0x41, 0x62, 0x46, 0xb6, 0x16, 0xf6, 0xe0, 0x5a, 0x3b, 0x73, 0x51, 0xe2, 0xe9, 0x19, 0xa7, 0x92, 0x52, 0xed,
    0xbb, 0x36, 0xbd, 0xfc, 0xc2, 0x74, 0xa5,
};

extern const ByteSpan sTestCert_PAI_FFF2_8005_ValInFuture_Cert = ByteSpan(sTestCert_PAI_FFF2_8005_ValInFuture_Cert_Array);

constexpr uint8_t sTestCert_PAI_FFF2_8005_ValInFuture_SKID_Array[] = {
    0x29, 0x38, 0xBA, 0xC7, 0xF6, 0x23, 0xE1, 0x83, 0x26, 0x43, 0xA5, 0xEB, 0x76, 0x2B, 0xAF, 0xD4, 0x0A, 0x62, 0xC0, 0x21,
};

extern const ByteSpan sTestCert_PAI_FFF2_8005_ValInFuture_SKID = ByteSpan(sTestCert_PAI_FFF2_8005_ValInFuture_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAI-FFF2-8005-ValInFuture-Key.pem

constexpr uint8_t sTestCert_PAI_FFF2_8005_ValInFuture_PublicKey_Array[] = {
    0x04, 0x19, 0x98, 0xe9, 0x1c, 0xaf, 0x72, 0xfe, 0x63, 0xf3, 0x07, 0xfd, 0xe1, 0x31, 0xbd, 0x66, 0x89,
    0xaa, 0xe3, 0x49, 0x01, 0x68, 0xe8, 0xc2, 0x55, 0xeb, 0x1c, 0xe9, 0x2c, 0x8c, 0xc5, 0x07, 0xf8, 0x35,
    0x05, 0x93, 0x16, 0x33, 0x3c, 0x58, 0xec, 0x87, 0x66, 0x31, 0xc7, 0x82, 0xd6, 0x6e, 0xcc, 0x0b, 0xc3,
    0x44, 0x31, 0x8c, 0x32, 0x25, 0x2b, 0xe8, 0x8f, 0x0e, 0xc7, 0x73, 0xdb, 0xd4, 0xa8,
};

extern const ByteSpan sTestCert_PAI_FFF2_8005_ValInFuture_PublicKey = ByteSpan(sTestCert_PAI_FFF2_8005_ValInFuture_PublicKey_Array);

constexpr uint8_t sTestCert_PAI_FFF2_8005_ValInFuture_PrivateKey_Array[] = {
    0x56, 0xda, 0x15, 0x5b, 0x02, 0x9b, 0xbb, 0x4c, 0xb3, 0xe9, 0xb5, 0xc1, 0xa9, 0x90, 0x2d, 0xd2,
    0xf9, 0x9c, 0x94, 0x33, 0x4c, 0x6e, 0xaa, 0x6b, 0x94, 0xe8, 0xf1, 0x19, 0x23, 0x6d, 0x2d, 0x93,
};

extern const ByteSpan sTestCert_PAI_FFF2_8005_ValInFuture_PrivateKey =
    ByteSpan(sTestCert_PAI_FFF2_8005_ValInFuture_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAI-FFF2-8005-ValInPast-Cert.pem

constexpr uint8_t sTestCert_PAI_FFF2_8005_ValInPast_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xbb, 0x30, 0x82, 0x01, 0x62, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x12, 0x5d, 0xe4, 0xfb, 0x2c, 0xe4,
    0x42, 0xea, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x1a, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x41,
    0x30, 0x1e, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x17, 0x0d, 0x32, 0x32,
    0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x32, 0x5a, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04,
    0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49, 0x31, 0x14, 0x30,
    0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x31, 0x14,
    0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x35, 0x30,
    0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01,
    0x07, 0x03, 0x42, 0x00, 0x04, 0x2c, 0xa3, 0x72, 0x51, 0x49, 0x16, 0xda, 0x99, 0x4b, 0x95, 0x93, 0x80, 0xb6, 0x6b, 0x27, 0x01,
    0xf4, 0xcf, 0x38, 0x08, 0xf3, 0x93, 0xf7, 0x16, 0xb1, 0x23, 0x89, 0xc2, 0xf9, 0x5a, 0x8a, 0xf2, 0xf4, 0x93, 0x26, 0x2a, 0x65,
    0xa0, 0xa3, 0xeb, 0xf8, 0x1b, 0xe0, 0x6a, 0x71, 0x07, 0x32, 0xa0, 0xaa, 0xa0, 0x2f, 0xf5, 0x28, 0xef, 0x95, 0x8b, 0xfd, 0x20,
    0x32, 0x63, 0x2f, 0xbd, 0x88, 0x5f, 0xa3, 0x66, 0x30, 0x64, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04,
    0x08, 0x30, 0x06, 0x01, 0x01, 0xff, 0x02, 0x01, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04,
    0x03, 0x02, 0x01, 0x06, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xda, 0xd4, 0x67, 0x6f, 0x50, 0x6d,
    0x45, 0x43, 0xdb, 0x94, 0x76, 0xd6, 0x18, 0x69, 0x1c, 0x67, 0x49, 0x72, 0x8d, 0x08, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23,
    0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x78, 0x5c, 0xe7, 0x05, 0xb8, 0x6b, 0x8f, 0x4e, 0x6f, 0xc7, 0x93, 0xaa, 0x60, 0xcb, 0x43,
    0xea, 0x69, 0x68, 0x82, 0xd5, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x47, 0x00, 0x30,
    0x44, 0x02, 0x20, 0x62, 0xb7, 0x97, 0x73, 0x53, 0x72, 0xfb, 0x91, 0xd9, 0x45, 0x2c, 0x6a, 0x4f, 0x88, 0x0f, 0x54, 0xfe, 0x73,
    0xe5, 0x99, 0x21, 0x65, 0x75, 0x12, 0x61, 0x6e, 0xc6, 0x1d, 0xaa, 0xac, 0xf1, 0x50, 0x02, 0x20, 0x3f, 0xfc, 0x88, 0x81, 0xac,
    0x85, 0xd7, 0x9d, 0x72, 0x30, 0x1b, 0x23, 0xab, 0x10, 0x3b, 0xe0, 0x2d, 0x7a, 0xf7, 0x0e, 0xa8, 0xbc, 0xa9, 0xc6, 0x78, 0x46,
    0x56, 0x52, 0xf6, 0x81, 0x09, 0xdf,
};

extern const ByteSpan sTestCert_PAI_FFF2_8005_ValInPast_Cert = ByteSpan(sTestCert_PAI_FFF2_8005_ValInPast_Cert_Array);

constexpr uint8_t sTestCert_PAI_FFF2_8005_ValInPast_SKID_Array[] = {
    0xDA, 0xD4, 0x67, 0x6F, 0x50, 0x6D, 0x45, 0x43, 0xDB, 0x94, 0x76, 0xD6, 0x18, 0x69, 0x1C, 0x67, 0x49, 0x72, 0x8D, 0x08,
};

extern const ByteSpan sTestCert_PAI_FFF2_8005_ValInPast_SKID = ByteSpan(sTestCert_PAI_FFF2_8005_ValInPast_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAI-FFF2-8005-ValInPast-Key.pem

constexpr uint8_t sTestCert_PAI_FFF2_8005_ValInPast_PublicKey_Array[] = {
    0x04, 0x2c, 0xa3, 0x72, 0x51, 0x49, 0x16, 0xda, 0x99, 0x4b, 0x95, 0x93, 0x80, 0xb6, 0x6b, 0x27, 0x01,
    0xf4, 0xcf, 0x38, 0x08, 0xf3, 0x93, 0xf7, 0x16, 0xb1, 0x23, 0x89, 0xc2, 0xf9, 0x5a, 0x8a, 0xf2, 0xf4,
    0x93, 0x26, 0x2a, 0x65, 0xa0, 0xa3, 0xeb, 0xf8, 0x1b, 0xe0, 0x6a, 0x71, 0x07, 0x32, 0xa0, 0xaa, 0xa0,
    0x2f, 0xf5, 0x28, 0xef, 0x95, 0x8b, 0xfd, 0x20, 0x32, 0x63, 0x2f, 0xbd, 0x88, 0x5f,
};

extern const ByteSpan sTestCert_PAI_FFF2_8005_ValInPast_PublicKey = ByteSpan(sTestCert_PAI_FFF2_8005_ValInPast_PublicKey_Array);

constexpr uint8_t sTestCert_PAI_FFF2_8005_ValInPast_PrivateKey_Array[] = {
    0x6b, 0x27, 0x54, 0x62, 0x8f, 0x6a, 0x9b, 0xb4, 0xf2, 0x4f, 0xac, 0xee, 0xe2, 0x09, 0x79, 0x58,
    0x1f, 0xc7, 0x7a, 0x3c, 0x92, 0x08, 0xe4, 0x04, 0x5d, 0x51, 0xb7, 0xa4, 0x65, 0xbd, 0x00, 0x3d,
};

extern const ByteSpan sTestCert_PAI_FFF2_8005_ValInPast_PrivateKey = ByteSpan(sTestCert_PAI_FFF2_8005_ValInPast_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAI-FFF2-8006-ValInFuture-Cert.pem

constexpr uint8_t sTestCert_PAI_FFF2_8006_ValInFuture_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xd2, 0x30, 0x82, 0x01, 0x78, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x49, 0x7f, 0xc6, 0x94, 0xe3, 0x1e,
    0xbf, 0x59, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x30, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x41,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x32, 0x30, 0x1e, 0x17, 0x0d, 0x33, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x17, 0x0d, 0x33,
    0x32, 0x30, 0x36, 0x32, 0x37, 0x31, 0x34, 0x32, 0x33, 0x34, 0x32, 0x5a, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55,
    0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49, 0x31, 0x14,
    0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x31,
    0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x36,
    0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03,
    0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x8b, 0xa1, 0x97, 0xdd, 0x28, 0x2c, 0x26, 0xd0, 0xf0, 0x9f, 0x97, 0xfd, 0x86, 0x24, 0xd3,
    0x4a, 0x7e, 0xf2, 0xef, 0x5f, 0x2f, 0x00, 0x59, 0x75, 0x3b, 0xc3, 0x91, 0x7b, 0x9e, 0x24, 0xdb, 0xea, 0x8c, 0x1f, 0xdd, 0x17,
    0x15, 0xda, 0x4f, 0xa9, 0xf9, 0xaa, 0x81, 0xac, 0xf1, 0x34, 0xa6, 0x40, 0x00, 0x11, 0x40, 0x04, 0xb5, 0xb4, 0x9f, 0x8e, 0x54,
    0x38, 0x6f, 0x2f, 0x9c, 0xcb, 0xa0, 0x57, 0xa3, 0x66, 0x30, 0x64, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff,
    0x04, 0x08, 0x30, 0x06, 0x01, 0x01, 0xff, 0x02, 0x01, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04,
    0x04, 0x03, 0x02, 0x01, 0x06, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x3c, 0x28, 0x86, 0xd0, 0x28,
    0x54, 0x87, 0x00, 0x50, 0x2f, 0x89, 0x4b, 0xda, 0x09, 0x82, 0xb9, 0xd9, 0x70, 0x5e, 0xb0, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d,
    0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x7f, 0x1d, 0xaa, 0xf2, 0x44, 0x98, 0xb9, 0x86, 0x68, 0x0e, 0xa0, 0x8f, 0xc1, 0x89,
    0x21, 0xe8, 0x48, 0x48, 0x9d, 0x17, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x48, 0x00,
    0x30, 0x45, 0x02, 0x21, 0x00, 0x96, 0xb0, 0xc0, 0x54, 0x3a, 0x8e, 0x95, 0xbd, 0x5d, 0xb9, 0x50, 0x79, 0xa8, 0x69, 0x65, 0x9f,
    0xf7, 0x24, 0x62, 0xe5, 0xa8, 0xb8, 0x34, 0x20, 0x76, 0x53, 0xb5, 0x5d, 0xdc, 0x6c, 0x36, 0xe4, 0x02, 0x20, 0x66, 0x61, 0x80,
    0x8d, 0x01, 0x12, 0x4e, 0x8b, 0x6d, 0xb6, 0x4d, 0x27, 0x9a, 0xe0, 0xbf, 0xfa, 0x6b, 0x45, 0x2a, 0x96, 0x4c, 0xfc, 0x66, 0x3c,
    0xfa, 0x18, 0x97, 0x2c, 0x6e, 0xad, 0x68, 0xe0,
};

extern const ByteSpan sTestCert_PAI_FFF2_8006_ValInFuture_Cert = ByteSpan(sTestCert_PAI_FFF2_8006_ValInFuture_Cert_Array);

constexpr uint8_t sTestCert_PAI_FFF2_8006_ValInFuture_SKID_Array[] = {
    0x3C, 0x28, 0x86, 0xD0, 0x28, 0x54, 0x87, 0x00, 0x50, 0x2F, 0x89, 0x4B, 0xDA, 0x09, 0x82, 0xB9, 0xD9, 0x70, 0x5E, 0xB0,
};

extern const ByteSpan sTestCert_PAI_FFF2_8006_ValInFuture_SKID = ByteSpan(sTestCert_PAI_FFF2_8006_ValInFuture_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAI-FFF2-8006-ValInFuture-Key.pem

constexpr uint8_t sTestCert_PAI_FFF2_8006_ValInFuture_PublicKey_Array[] = {
    0x04, 0x8b, 0xa1, 0x97, 0xdd, 0x28, 0x2c, 0x26, 0xd0, 0xf0, 0x9f, 0x97, 0xfd, 0x86, 0x24, 0xd3, 0x4a,
    0x7e, 0xf2, 0xef, 0x5f, 0x2f, 0x00, 0x59, 0x75, 0x3b, 0xc3, 0x91, 0x7b, 0x9e, 0x24, 0xdb, 0xea, 0x8c,
    0x1f, 0xdd, 0x17, 0x15, 0xda, 0x4f, 0xa9, 0xf9, 0xaa, 0x81, 0xac, 0xf1, 0x34, 0xa6, 0x40, 0x00, 0x11,
    0x40, 0x04, 0xb5, 0xb4, 0x9f, 0x8e, 0x54, 0x38, 0x6f, 0x2f, 0x9c, 0xcb, 0xa0, 0x57,
};

extern const ByteSpan sTestCert_PAI_FFF2_8006_ValInFuture_PublicKey = ByteSpan(sTestCert_PAI_FFF2_8006_ValInFuture_PublicKey_Array);

constexpr uint8_t sTestCert_PAI_FFF2_8006_ValInFuture_PrivateKey_Array[] = {
    0x30, 0x44, 0x66, 0xdf, 0x85, 0xf2, 0xe1, 0x49, 0xcf, 0x15, 0x7b, 0x1c, 0x79, 0xc1, 0x9a, 0x48,
    0x60, 0xf4, 0x28, 0x85, 0x2a, 0xd6, 0xa7, 0x84, 0xcc, 0x29, 0xf2, 0x43, 0xb5, 0xf7, 0x4f, 0xa3,
};

extern const ByteSpan sTestCert_PAI_FFF2_8006_ValInFuture_PrivateKey =
    ByteSpan(sTestCert_PAI_FFF2_8006_ValInFuture_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAI-FFF2-8006-ValInPast-Cert.pem

constexpr uint8_t sTestCert_PAI_FFF2_8006_ValInPast_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xd2, 0x30, 0x82, 0x01, 0x78, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x79, 0x08, 0xed, 0x7b, 0x59, 0x76,
    0x15, 0xc7, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x30, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x41,
    0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46,
    0x32, 0x30, 0x1e, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x17, 0x0d, 0x32,
    0x32, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x32, 0x5a, 0x30, 0x46, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55,
    0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49, 0x31, 0x14,
    0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x31,
    0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x02, 0x0c, 0x04, 0x38, 0x30, 0x30, 0x36,
    0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03,
    0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xda, 0x8b, 0x5e, 0xc1, 0x00, 0x44, 0x1c, 0x2c, 0x0c, 0xf8, 0x7a, 0x7a, 0xa3, 0x89, 0xb2,
    0x60, 0x1c, 0xda, 0x79, 0x68, 0xe2, 0xa4, 0xa2, 0xfb, 0xbe, 0xc8, 0x14, 0x79, 0x75, 0x8a, 0xc2, 0xe8, 0x4f, 0x68, 0x3e, 0x61,
    0x8a, 0xa6, 0x0f, 0xf7, 0x51, 0xed, 0xac, 0x69, 0x3a, 0xb1, 0x73, 0xba, 0x93, 0xfc, 0x52, 0xe2, 0xfb, 0x88, 0x92, 0x69, 0x93,
    0xae, 0xa2, 0xe6, 0xb6, 0x8c, 0xbe, 0x05, 0xa3, 0x66, 0x30, 0x64, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff,
    0x04, 0x08, 0x30, 0x06, 0x01, 0x01, 0xff, 0x02, 0x01, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04,
    0x04, 0x03, 0x02, 0x01, 0x06, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x0a, 0x6f, 0xab, 0xbd, 0xe2,
    0x77, 0xfd, 0x19, 0xf6, 0x46, 0x51, 0x25, 0x62, 0x91, 0x58, 0xac, 0x3a, 0xb3, 0xa4, 0xe2, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d,
    0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x9b, 0xb2, 0x99, 0xbb, 0xa6, 0x49, 0x7d, 0xb5, 0xa0, 0x4e, 0x1e, 0x18, 0xc7, 0xc2,
    0x0e, 0xae, 0xc5, 0x4c, 0x42, 0xe9, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x48, 0x00,
    0x30, 0x45, 0x02, 0x20, 0x39, 0x8f, 0x59, 0x7a, 0x7c, 0xfc, 0xeb, 0x37, 0x41, 0x19, 0x6a, 0x64, 0x37, 0x0e, 0xd6, 0x00, 0x4c,
    0x5b, 0x7d, 0xd5, 0xd2, 0x10, 0xf3, 0x2c, 0x8a, 0x59, 0x56, 0xa0, 0xb7, 0x2b, 0xd9, 0xb5, 0x02, 0x21, 0x00, 0x8e, 0x0c, 0x62,
    0xa7, 0xe4, 0x98, 0x81, 0xfc, 0xbc, 0x60, 0x17, 0xa1, 0xc5, 0x37, 0x31, 0x44, 0xef, 0x56, 0xff, 0x1a, 0x87, 0x61, 0x25, 0xad,
    0x1e, 0xa2, 0x7b, 0x02, 0xe7, 0x88, 0x87, 0xa6,
};

extern const ByteSpan sTestCert_PAI_FFF2_8006_ValInPast_Cert = ByteSpan(sTestCert_PAI_FFF2_8006_ValInPast_Cert_Array);

constexpr uint8_t sTestCert_PAI_FFF2_8006_ValInPast_SKID_Array[] = {
    0x0A, 0x6F, 0xAB, 0xBD, 0xE2, 0x77, 0xFD, 0x19, 0xF6, 0x46, 0x51, 0x25, 0x62, 0x91, 0x58, 0xAC, 0x3A, 0xB3, 0xA4, 0xE2,
};

extern const ByteSpan sTestCert_PAI_FFF2_8006_ValInPast_SKID = ByteSpan(sTestCert_PAI_FFF2_8006_ValInPast_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAI-FFF2-8006-ValInPast-Key.pem

constexpr uint8_t sTestCert_PAI_FFF2_8006_ValInPast_PublicKey_Array[] = {
    0x04, 0xda, 0x8b, 0x5e, 0xc1, 0x00, 0x44, 0x1c, 0x2c, 0x0c, 0xf8, 0x7a, 0x7a, 0xa3, 0x89, 0xb2, 0x60,
    0x1c, 0xda, 0x79, 0x68, 0xe2, 0xa4, 0xa2, 0xfb, 0xbe, 0xc8, 0x14, 0x79, 0x75, 0x8a, 0xc2, 0xe8, 0x4f,
    0x68, 0x3e, 0x61, 0x8a, 0xa6, 0x0f, 0xf7, 0x51, 0xed, 0xac, 0x69, 0x3a, 0xb1, 0x73, 0xba, 0x93, 0xfc,
    0x52, 0xe2, 0xfb, 0x88, 0x92, 0x69, 0x93, 0xae, 0xa2, 0xe6, 0xb6, 0x8c, 0xbe, 0x05,
};

extern const ByteSpan sTestCert_PAI_FFF2_8006_ValInPast_PublicKey = ByteSpan(sTestCert_PAI_FFF2_8006_ValInPast_PublicKey_Array);

constexpr uint8_t sTestCert_PAI_FFF2_8006_ValInPast_PrivateKey_Array[] = {
    0xf3, 0x1e, 0x22, 0xd1, 0x3e, 0x33, 0x9d, 0xaa, 0xd9, 0xb6, 0x01, 0x4b, 0x08, 0x43, 0x5e, 0x9f,
    0x65, 0x47, 0xbc, 0x00, 0xf7, 0x93, 0xca, 0x9f, 0xe5, 0x26, 0xaa, 0x33, 0x8a, 0x40, 0x1d, 0x77,
};

extern const ByteSpan sTestCert_PAI_FFF2_8006_ValInPast_PrivateKey = ByteSpan(sTestCert_PAI_FFF2_8006_ValInPast_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAI-FFF2-NoPID-Cert.pem

constexpr uint8_t sTestCert_PAI_FFF2_NoPID_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xa8, 0x30, 0x82, 0x01, 0x4e, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x3d, 0x74, 0xbb, 0x56, 0x59, 0x71,
    0x10, 0x1c, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x1a, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x41,
    0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x39, 0x39,
    0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x30, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03,
    0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49, 0x31,
    0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x32,
    0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03,
    0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xd8, 0xbf, 0x93, 0x47, 0x92, 0xcf, 0x8e, 0xae, 0xda, 0xc2, 0x4f, 0xfc, 0x96, 0x6c, 0x91,
    0x76, 0x20, 0xfb, 0x97, 0x2f, 0xba, 0xb9, 0x8e, 0xc6, 0xd5, 0x13, 0x14, 0xa0, 0x7a, 0xe9, 0x8e, 0x1a, 0x03, 0xfb, 0x41, 0x91,
    0xd2, 0x6e, 0x2d, 0x12, 0x7c, 0xb9, 0x52, 0x76, 0x21, 0xc3, 0x6e, 0x97, 0x3a, 0x18, 0x6c, 0x56, 0xd0, 0xca, 0xd9, 0x99, 0xb0,
    0x41, 0xc2, 0x70, 0xa9, 0xb1, 0xcb, 0x3f, 0xa3, 0x66, 0x30, 0x64, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff,
    0x04, 0x08, 0x30, 0x06, 0x01, 0x01, 0xff, 0x02, 0x01, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04,
    0x04, 0x03, 0x02, 0x01, 0x06, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x61, 0x3d, 0xd0, 0x87, 0x35,
    0x5e, 0xf0, 0x8b, 0xae, 0x01, 0xe4, 0xc6, 0x9a, 0x8f, 0xc7, 0x3d, 0xac, 0x8c, 0x7d, 0xfd, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d,
    0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x78, 0x5c, 0xe7, 0x05, 0xb8, 0x6b, 0x8f, 0x4e, 0x6f, 0xc7, 0x93, 0xaa, 0x60, 0xcb,
    0x43, 0xea, 0x69, 0x68, 0x82, 0xd5, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x48, 0x00,
    0x30, 0x45, 0x02, 0x21, 0x00, 0xdf, 0xc0, 0x9d, 0xe8, 0x4b, 0xfa, 0x95, 0x6d, 0x60, 0xd6, 0xfe, 0xf4, 0xe2, 0xad, 0x11, 0x0b,
    0xde, 0x22, 0x2a, 0x32, 0x88, 0x30, 0xa6, 0x7e, 0x3f, 0x17, 0xb9, 0x1e, 0x51, 0x85, 0x1a, 0x78, 0x02, 0x20, 0x71, 0xed, 0x97,
    0x18, 0xe4, 0x88, 0xb2, 0xb8, 0xcf, 0x95, 0x44, 0x35, 0xce, 0x3f, 0xf3, 0x2e, 0xa5, 0x05, 0x56, 0x14, 0x3c, 0x4f, 0x2c, 0x9c,
    0x05, 0xa2, 0x0a, 0xf3, 0x3e, 0xd4, 0xca, 0xd9,
};

extern const ByteSpan sTestCert_PAI_FFF2_NoPID_Cert = ByteSpan(sTestCert_PAI_FFF2_NoPID_Cert_Array);

constexpr uint8_t sTestCert_PAI_FFF2_NoPID_SKID_Array[] = {
    0x61, 0x3D, 0xD0, 0x87, 0x35, 0x5E, 0xF0, 0x8B, 0xAE, 0x01, 0xE4, 0xC6, 0x9A, 0x8F, 0xC7, 0x3D, 0xAC, 0x8C, 0x7D, 0xFD,
};

extern const ByteSpan sTestCert_PAI_FFF2_NoPID_SKID = ByteSpan(sTestCert_PAI_FFF2_NoPID_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAI-FFF2-NoPID-Key.pem

constexpr uint8_t sTestCert_PAI_FFF2_NoPID_PublicKey_Array[] = {
    0x04, 0xd8, 0xbf, 0x93, 0x47, 0x92, 0xcf, 0x8e, 0xae, 0xda, 0xc2, 0x4f, 0xfc, 0x96, 0x6c, 0x91, 0x76,
    0x20, 0xfb, 0x97, 0x2f, 0xba, 0xb9, 0x8e, 0xc6, 0xd5, 0x13, 0x14, 0xa0, 0x7a, 0xe9, 0x8e, 0x1a, 0x03,
    0xfb, 0x41, 0x91, 0xd2, 0x6e, 0x2d, 0x12, 0x7c, 0xb9, 0x52, 0x76, 0x21, 0xc3, 0x6e, 0x97, 0x3a, 0x18,
    0x6c, 0x56, 0xd0, 0xca, 0xd9, 0x99, 0xb0, 0x41, 0xc2, 0x70, 0xa9, 0xb1, 0xcb, 0x3f,
};

extern const ByteSpan sTestCert_PAI_FFF2_NoPID_PublicKey = ByteSpan(sTestCert_PAI_FFF2_NoPID_PublicKey_Array);

constexpr uint8_t sTestCert_PAI_FFF2_NoPID_PrivateKey_Array[] = {
    0x6d, 0x25, 0x55, 0xa6, 0x85, 0x07, 0x3a, 0xc5, 0xfa, 0x44, 0xe5, 0x0f, 0x78, 0x0d, 0x8b, 0xb2,
    0xc3, 0x40, 0x4c, 0x55, 0xf0, 0xf8, 0x85, 0x26, 0x4c, 0xf5, 0x39, 0x2a, 0x44, 0xf4, 0x4c, 0x21,
};

extern const ByteSpan sTestCert_PAI_FFF2_NoPID_PrivateKey = ByteSpan(sTestCert_PAI_FFF2_NoPID_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAI-FFF2-NoPID-FB-Cert.pem

constexpr uint8_t sTestCert_PAI_FFF2_NoPID_FB_Cert_Array[] = {
    0x30, 0x82, 0x01, 0x9b, 0x30, 0x82, 0x01, 0x42, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x22, 0xdf, 0x09, 0x71, 0x27, 0x86,
    0x6a, 0xa3, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x1a, 0x31, 0x18, 0x30, 0x16, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x41,
    0x30, 0x20, 0x17, 0x0d, 0x32, 0x31, 0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x39, 0x39,
    0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x24, 0x31, 0x22, 0x30, 0x20, 0x06, 0x03,
    0x55, 0x04, 0x03, 0x0c, 0x19, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49, 0x20,
    0x4d, 0x76, 0x69, 0x64, 0x3a, 0x46, 0x46, 0x46, 0x32, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02,
    0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x6a, 0x7b, 0xf1, 0xfb, 0x23, 0xfa,
    0xb3, 0x7f, 0xbc, 0xa3, 0x58, 0xe6, 0x3d, 0x92, 0x04, 0xb7, 0x03, 0x57, 0x06, 0xbe, 0xad, 0x7e, 0xd3, 0xe0, 0x4b, 0x0b, 0x1b,
    0xfe, 0x2a, 0x59, 0x56, 0x75, 0x1f, 0xea, 0x15, 0x6f, 0x94, 0xd3, 0x46, 0xce, 0x28, 0x61, 0xf4, 0xeb, 0xaf, 0xd8, 0x22, 0x93,
    0xc2, 0xe7, 0x64, 0x5c, 0xcc, 0x7a, 0x5a, 0xac, 0xbf, 0xb1, 0x7e, 0x41, 0xe5, 0xe2, 0x44, 0x81, 0xa3, 0x66, 0x30, 0x64, 0x30,
    0x12, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x08, 0x30, 0x06, 0x01, 0x01, 0xff, 0x02, 0x01, 0x00, 0x30, 0x0e,
    0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x01, 0x06, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e,
    0x04, 0x16, 0x04, 0x14, 0x19, 0xfa, 0x94, 0x74, 0x93, 0x29, 0xda, 0x74, 0x0f, 0x55, 0x6a, 0x33, 0x3f, 0xb5, 0x13, 0x73, 0xd7,
    0x08, 0x17, 0x56, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x78, 0x5c, 0xe7, 0x05, 0xb8,
    0x6b, 0x8f, 0x4e, 0x6f, 0xc7, 0x93, 0xaa, 0x60, 0xcb, 0x43, 0xea, 0x69, 0x68, 0x82, 0xd5, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86,
    0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x47, 0x00, 0x30, 0x44, 0x02, 0x20, 0x76, 0x50, 0xe1, 0x8b, 0xfc, 0x04, 0x98, 0x6a,
    0xeb, 0x0d, 0xc6, 0x8e, 0x1b, 0xbc, 0x01, 0x99, 0x6f, 0x44, 0xb4, 0xb0, 0x34, 0xb0, 0xd8, 0x0c, 0x8d, 0x94, 0xa6, 0x8b, 0x61,
    0x06, 0x32, 0x03, 0x02, 0x20, 0x78, 0xfe, 0xbc, 0x9c, 0xb7, 0xf4, 0x04, 0xcc, 0xef, 0x19, 0x5c, 0xcf, 0x1e, 0xcb, 0x09, 0x54,
    0x43, 0x8b, 0xe5, 0xa5, 0x9d, 0x74, 0x8a, 0xe9, 0xae, 0x31, 0x39, 0x0d, 0x9b, 0x12, 0x5b, 0xfe,
};

extern const ByteSpan sTestCert_PAI_FFF2_NoPID_FB_Cert = ByteSpan(sTestCert_PAI_FFF2_NoPID_FB_Cert_Array);

constexpr uint8_t sTestCert_PAI_FFF2_NoPID_FB_SKID_Array[] = {
    0x19, 0xFA, 0x94, 0x74, 0x93, 0x29, 0xDA, 0x74, 0x0F, 0x55, 0x6A, 0x33, 0x3F, 0xB5, 0x13, 0x73, 0xD7, 0x08, 0x17, 0x56,
};

extern const ByteSpan sTestCert_PAI_FFF2_NoPID_FB_SKID = ByteSpan(sTestCert_PAI_FFF2_NoPID_FB_SKID_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAI-FFF2-NoPID-FB-Key.pem

constexpr uint8_t sTestCert_PAI_FFF2_NoPID_FB_PublicKey_Array[] = {
    0x04, 0x6a, 0x7b, 0xf1, 0xfb, 0x23, 0xfa, 0xb3, 0x7f, 0xbc, 0xa3, 0x58, 0xe6, 0x3d, 0x92, 0x04, 0xb7,
    0x03, 0x57, 0x06, 0xbe, 0xad, 0x7e, 0xd3, 0xe0, 0x4b, 0x0b, 0x1b, 0xfe, 0x2a, 0x59, 0x56, 0x75, 0x1f,
    0xea, 0x15, 0x6f, 0x94, 0xd3, 0x46, 0xce, 0x28, 0x61, 0xf4, 0xeb, 0xaf, 0xd8, 0x22, 0x93, 0xc2, 0xe7,
    0x64, 0x5c, 0xcc, 0x7a, 0x5a, 0xac, 0xbf, 0xb1, 0x7e, 0x41, 0xe5, 0xe2, 0x44, 0x81,
};

extern const ByteSpan sTestCert_PAI_FFF2_NoPID_FB_PublicKey = ByteSpan(sTestCert_PAI_FFF2_NoPID_FB_PublicKey_Array);

constexpr uint8_t sTestCert_PAI_FFF2_NoPID_FB_PrivateKey_Array[] = {
    0x22, 0x78, 0xe1, 0xaa, 0x00, 0x59, 0x58, 0x1e, 0xcf, 0x2d, 0xbf, 0x68, 0xc3, 0xf2, 0x51, 0xf6,
    0x76, 0xfc, 0xde, 0xdd, 0x88, 0xb6, 0xe7, 0x50, 0x50, 0xdf, 0x6a, 0x72, 0x41, 0xc7, 0x65, 0x56,
};

extern const ByteSpan sTestCert_PAI_FFF2_NoPID_FB_PrivateKey = ByteSpan(sTestCert_PAI_FFF2_NoPID_FB_PrivateKey_Array);

// ${chip_root}/credentials/test/attestation/Chip-Test-PAI-FFF2-NoPID-Resigned-Cert.pem

constexpr uint8_t sTestCert_PAI_FFF2_NoPID_Resigned_Cert_Array[] = {
    0x30, 0x82, 0x01, 0xb7, 0x30, 0x82, 0x01, 0x5d, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x5d, 0x0a, 0x59, 0xc0, 0x17, 0x4b,
    0x3b, 0x65, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x29, 0x31, 0x27, 0x30, 0x25, 0x06,
    0x03, 0x55, 0x04, 0x03, 0x0c, 0x1e, 0x4d, 0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x41,
    0x20, 0x54, 0x6f, 0x20, 0x52, 0x65, 0x73, 0x69, 0x67, 0x6e, 0x20, 0x50, 0x41, 0x49, 0x73, 0x30, 0x20, 0x17, 0x0d, 0x32, 0x31,
    0x30, 0x36, 0x32, 0x38, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x39, 0x39, 0x39, 0x39, 0x31, 0x32, 0x33, 0x31,
    0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x30, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0f, 0x4d,
    0x61, 0x74, 0x74, 0x65, 0x72, 0x20, 0x54, 0x65, 0x73, 0x74, 0x20, 0x50, 0x41, 0x49, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x2b,
    0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x02, 0x01, 0x0c, 0x04, 0x46, 0x46, 0x46, 0x32, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07,
    0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04,
    0xd8, 0xbf, 0x93, 0x47, 0x92, 0xcf, 0x8e, 0xae, 0xda, 0xc2, 0x4f, 0xfc, 0x96, 0x6c, 0x91, 0x76, 0x20, 0xfb, 0x97, 0x2f, 0xba,
    0xb9, 0x8e, 0xc6, 0xd5, 0x13, 0x14, 0xa0, 0x7a, 0xe9, 0x8e, 0x1a, 0x03, 0xfb, 0x41, 0x91, 0xd2, 0x6e, 0x2d, 0x12, 0x7c, 0xb9,
    0x52, 0x76, 0x21, 0xc3, 0x6e, 0x97, 0x3a, 0x18, 0x6c, 0x56, 0xd0, 0xca, 0xd9, 0x99, 0xb0, 0x41, 0xc2, 0x70, 0xa9, 0xb1, 0xcb,
    0x3f, 0xa3, 0x66, 0x30, 0x64, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x08, 0x30, 0x06, 0x01, 0x01,
    0xff, 0x02, 0x01, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x01, 0x06, 0x30,
    0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x61, 0x3d, 0xd0, 0x87, 0x35, 0x5e, 0xf0, 0x8b, 0xae, 0x01, 0xe4,
    0xc6, 0x9a, 0x8f, 0xc7, 0x3d, 0xac, 0x8c, 0x7d, 0xfd, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80,
    0x14, 0x78, 0x5c, 0xe7, 0x05, 0xb8, 0x6b, 0x8f, 0x4e, 0x6f, 0xc7, 0x93, 0xaa, 0x60, 0xcb, 0x43, 0xea, 0x69, 0x68, 0x82, 0xd5,
    0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x48, 0x00, 0x30, 0x45, 0x02, 0x20, 0x7e, 0x5f,
    0x9f, 0x5f, 0xff, 0x06, 0xeb, 0x39, 0xff, 0x13, 0x4f, 0x5a, 0xed, 0x7f, 0x3f, 0x96, 0x61, 0x3f, 0xe2, 0xf1, 0x7f, 0x4f, 0x7b,
    0xbf, 0x51, 0x24, 0x3e, 0x2a, 0x73, 0x53, 0x70, 0xf8, 0x02, 0x21, 0x00, 0xec, 0x54, 0x7d, 0x78, 0x73, 0xdb, 0x45, 0x80, 0xf1,
    0xe8, 0xd8, 0xe9, 0x2e, 0xca, 0xec, 0x24, 0x5d, 0x88, 0x0a, 0x88, 0x17, 0xfc, 0xd8, 0x1c, 0x67, 0xa8, 0xfa, 0xe4, 0x95, 0x8e,
    0xda, 0x82,
};

extern const ByteSpan sTestCert_PAI_FFF2_NoPID_Resigned_Cert = ByteSpan(sTestCert_PAI_FFF2_NoPID_Resigned_Cert_Array);

constexpr uint8_t sTestCert_PAI_FFF2_NoPID_Resigned_SKID_Array[] = {
    0x61, 0x3D, 0xD0, 0x87, 0x35, 0x5E, 0xF0, 0x8B, 0xAE, 0x01, 0xE4, 0xC6, 0x9A, 0x8F, 0xC7, 0x3D, 0xAC, 0x8C, 0x7D, 0xFD,
};

extern const ByteSpan sTestCert_PAI_FFF2_NoPID_Resigned_SKID = ByteSpan(sTestCert_PAI_FFF2_NoPID_Resigned_SKID_Array);

} // namespace TestCerts
} // namespace chip
