/*
 *
 *    Copyright (c) 2020-2021 Project CHIP Authors
 *    Copyright (c) 2019 Google LLC.
 *    Copyright (c) 2016-2017 Nest Labs, Inc.
 *    All rights reserved.
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

/**
 *    @file
 *      This file contains definitions of test certificates used by various unit tests.
 *
 */

#include <credentials/CHIPCert.h>

#include "CHIPCert_test_vectors.h"

namespace chip {
namespace TestCerts {

// clang-format off
extern const uint8_t gTestCerts[] = {
    TestCert::kRoot01,
    TestCert::kRoot02,
    TestCert::kRoot03,
    TestCert::kICA01,
    TestCert::kICA02,
    TestCert::kICA01_1,
    TestCert::kFWSign01,
    TestCert::kNode01_01,
    TestCert::kNode01_02,
    TestCert::kNode02_01,
    TestCert::kNode02_02,
    TestCert::kNode02_03,
    TestCert::kNode02_04,
    TestCert::kNode02_05,
    TestCert::kNode02_06,
    TestCert::kNode02_07,
    TestCert::kNode02_08,
};
// clang-format on

extern const size_t gNumTestCerts = ArraySize(gTestCerts);

CHIP_ERROR GetTestCert(uint8_t certType, BitFlags<TestCertLoadFlags> certLoadFlags, ByteSpan & cert)
{
    CHIP_ERROR err;
    bool derForm = certLoadFlags.Has(TestCertLoadFlags::kDERForm);

#define SELECT_CERT(NAME)                                                                                                          \
    do                                                                                                                             \
    {                                                                                                                              \
        if (certType == TestCert::k##NAME)                                                                                         \
        {                                                                                                                          \
            if (derForm)                                                                                                           \
            {                                                                                                                      \
                cert = ByteSpan(sTestCert_##NAME##_DER, sTestCert_##NAME##_DER_Len);                                               \
            }                                                                                                                      \
            else                                                                                                                   \
            {                                                                                                                      \
                cert = ByteSpan(sTestCert_##NAME##_Chip, sTestCert_##NAME##_Chip_Len);                                             \
            }                                                                                                                      \
            ExitNow(err = CHIP_NO_ERROR);                                                                                          \
        }                                                                                                                          \
    } while (0)

    SELECT_CERT(Root01);
    SELECT_CERT(Root02);
    SELECT_CERT(Root03);
    SELECT_CERT(ICA01);
    SELECT_CERT(ICA02);
    SELECT_CERT(ICA01_1);
    SELECT_CERT(FWSign01);
    SELECT_CERT(Node01_01);
    SELECT_CERT(Node01_02);
    SELECT_CERT(Node02_01);
    SELECT_CERT(Node02_02);
    SELECT_CERT(Node02_03);
    SELECT_CERT(Node02_04);
    SELECT_CERT(Node02_05);
    SELECT_CERT(Node02_06);
    SELECT_CERT(Node02_07);
    SELECT_CERT(Node02_08);

#undef SELECT_CERT

    err = CHIP_ERROR_CA_CERT_NOT_FOUND;

exit:
    return err;
}

const char * GetTestCertName(uint8_t certType)
{
#define NAME_CERT(NAME)                                                                                                            \
    do                                                                                                                             \
    {                                                                                                                              \
        if (certType == TestCert::k##NAME)                                                                                         \
        {                                                                                                                          \
            return #NAME;                                                                                                          \
        }                                                                                                                          \
    } while (0)

    NAME_CERT(Root01);
    NAME_CERT(Root02);
    NAME_CERT(Root03);
    NAME_CERT(ICA01);
    NAME_CERT(ICA02);
    NAME_CERT(ICA01_1);
    NAME_CERT(FWSign01);
    NAME_CERT(Node01_01);
    NAME_CERT(Node01_02);
    NAME_CERT(Node02_01);
    NAME_CERT(Node02_02);
    NAME_CERT(Node02_03);
    NAME_CERT(Node02_04);
    NAME_CERT(Node02_05);
    NAME_CERT(Node02_06);
    NAME_CERT(Node02_07);
    NAME_CERT(Node02_08);

#undef NAME_CERT

    return nullptr;
}

CHIP_ERROR GetTestCertPubkey(uint8_t certType, ByteSpan & pubkey)
{
    CHIP_ERROR err;

#define SELECT_PUBKEY(NAME)                                                                                                        \
    do                                                                                                                             \
    {                                                                                                                              \
        if (certType == TestCert::k##NAME)                                                                                         \
        {                                                                                                                          \
            pubkey = ByteSpan(sTestCert_##NAME##_PublicKey, sTestCert_##NAME##_PublicKey_Len);                                     \
            ExitNow(err = CHIP_NO_ERROR);                                                                                          \
        }                                                                                                                          \
    } while (0)

    SELECT_PUBKEY(Root01);
    SELECT_PUBKEY(Root02);
    SELECT_PUBKEY(Root03);
    SELECT_PUBKEY(ICA01);
    SELECT_PUBKEY(ICA02);
    SELECT_PUBKEY(ICA01_1);
    SELECT_PUBKEY(FWSign01);
    SELECT_PUBKEY(Node01_01);
    SELECT_PUBKEY(Node01_02);
    SELECT_PUBKEY(Node02_01);
    SELECT_PUBKEY(Node02_02);
    SELECT_PUBKEY(Node02_03);
    SELECT_PUBKEY(Node02_04);
    SELECT_PUBKEY(Node02_05);
    SELECT_PUBKEY(Node02_06);
    SELECT_PUBKEY(Node02_07);
    SELECT_PUBKEY(Node02_08);

#undef SELECT_PUBKEY

    err = CHIP_ERROR_CA_CERT_NOT_FOUND;

exit:
    return err;
}

CHIP_ERROR GetTestCertSKID(uint8_t certType, ByteSpan & skid)
{
    CHIP_ERROR err;

#define SELECT_SKID(NAME)                                                                                                          \
    do                                                                                                                             \
    {                                                                                                                              \
        if (certType == TestCert::k##NAME)                                                                                         \
        {                                                                                                                          \
            skid = ByteSpan(sTestCert_##NAME##_SubjectKeyId, sTestCert_##NAME##_SubjectKeyId_Len);                                 \
            ExitNow(err = CHIP_NO_ERROR);                                                                                          \
        }                                                                                                                          \
    } while (0)

    SELECT_SKID(Root01);
    SELECT_SKID(Root02);
    SELECT_SKID(Root03);
    SELECT_SKID(ICA01);
    SELECT_SKID(ICA02);
    SELECT_SKID(ICA01_1);
    SELECT_SKID(FWSign01);
    SELECT_SKID(Node01_01);
    SELECT_SKID(Node01_02);
    SELECT_SKID(Node02_01);
    SELECT_SKID(Node02_02);
    SELECT_SKID(Node02_03);
    SELECT_SKID(Node02_04);
    SELECT_SKID(Node02_05);
    SELECT_SKID(Node02_06);
    SELECT_SKID(Node02_07);
    SELECT_SKID(Node02_08);

#undef SELECT_SKID

    err = CHIP_ERROR_CA_CERT_NOT_FOUND;

exit:
    return err;
}

CHIP_ERROR GetTestCertAKID(uint8_t certType, ByteSpan & akid)
{
    CHIP_ERROR err;

#define SELECT_AKID(NAME)                                                                                                          \
    do                                                                                                                             \
    {                                                                                                                              \
        if (certType == TestCert::k##NAME)                                                                                         \
        {                                                                                                                          \
            akid = ByteSpan(sTestCert_##NAME##_AuthorityKeyId, sTestCert_##NAME##_AuthorityKeyId_Len);                             \
            ExitNow(err = CHIP_NO_ERROR);                                                                                          \
        }                                                                                                                          \
    } while (0)

    SELECT_AKID(Root01);
    SELECT_AKID(Root02);
    SELECT_AKID(Root03);
    SELECT_AKID(ICA01);
    SELECT_AKID(ICA02);
    SELECT_AKID(ICA01_1);
    SELECT_AKID(FWSign01);
    SELECT_AKID(Node01_01);
    SELECT_AKID(Node01_02);
    SELECT_AKID(Node02_01);
    SELECT_AKID(Node02_02);
    SELECT_AKID(Node02_03);
    SELECT_AKID(Node02_04);
    SELECT_AKID(Node02_05);
    SELECT_AKID(Node02_06);
    SELECT_AKID(Node02_07);
    SELECT_AKID(Node02_08);

#undef SELECT_AKID

    err = CHIP_ERROR_CA_CERT_NOT_FOUND;

exit:
    return err;
}

CHIP_ERROR LoadTestCert(ChipCertificateSet & certSet, uint8_t certType, BitFlags<TestCertLoadFlags> certLoadFlags,
                        BitFlags<CertDecodeFlags> decodeFlags)
{
    CHIP_ERROR err;
    ChipCertificateData * certData;
    ByteSpan cert;

    // Get the requested certificate data.
    err = GetTestCert(certType, certLoadFlags, cert);
    SuccessOrExit(err);

    // Load it into the certificate set.
    err = certSet.LoadCert(cert, decodeFlags);
    SuccessOrExit(err);

    // Get loaded certificate data.
    certData = const_cast<ChipCertificateData *>(certSet.GetLastCert());
    VerifyOrExit(certData != nullptr, err = CHIP_ERROR_INVALID_ARGUMENT);

    // Apply load flags.
    if (certLoadFlags.Has(TestCertLoadFlags::kSuppressIsCA))
    {
        certData->mCertFlags.Clear(CertFlags::kIsCA);
    }
    if (certLoadFlags.Has(TestCertLoadFlags::kSuppressKeyUsage))
    {
        certData->mCertFlags.Clear(CertFlags::kExtPresent_KeyUsage);
    }
    if (certLoadFlags.Has(TestCertLoadFlags::kSuppressKeyCertSign))
    {
        certData->mKeyUsageFlags.Clear(KeyUsageFlags::kKeyCertSign);
    }
    if (certLoadFlags.Has(TestCertLoadFlags::kSetPathLenConstZero))
    {
        certData->mCertFlags.Set(CertFlags::kPathLenConstraintPresent);
        certData->mPathLenConstraint = 0;
    }

exit:
    return err;
}

// ------------------------------ TEST CERTIFICATES ----------------------------------------

/**************  Test Root01 Certificate  **************
Certificate:
    Data:
        Version: 3 (0x2)
        Serial Number: 6002248829961909524 (0x534c458273623514)
        Signature Algorithm: ecdsa-with-SHA256
        Issuer: 1.3.6.1.4.1.37244.1.4 = CACACACA00000001
        Validity
            Not Before: Oct 15 14:23:43 2020 GMT
            Not After : Oct 15 14:23:42 2040 GMT
        Subject: 1.3.6.1.4.1.37244.1.4 = CACACACA00000001
        Subject Public Key Info:
            Public Key Algorithm: id-ecPublicKey
                Public-Key: (256 bit)
                pub:
                    04:3b:88:46:0e:c9:68:7a:5d:0f:3b:4b:3b:13:fc:
                    d2:99:c2:f6:d5:05:1d:00:3e:e4:9c:99:24:cf:98:
                    f4:f7:80:eb:20:fd:37:c8:d3:58:34:7f:5f:87:d0:
                    8c:32:13:e5:40:af:11:ba:b9:13:7e:49:35:4f:0c:
                    5b:63:43:de:63
                ASN1 OID: prime256v1
                NIST CURVE: P-256
        X509v3 extensions:
            X509v3 Basic Constraints: critical
                CA:TRUE
            X509v3 Key Usage: critical
                Certificate Sign, CRL Sign
            X509v3 Subject Key Identifier:
                CC:13:08:AF:82:CF:EE:50:5E:B2:3B:57:BF:E8:6A:31:16:65:53:5F
            X509v3 Authority Key Identifier:
                keyid:CC:13:08:AF:82:CF:EE:50:5E:B2:3B:57:BF:E8:6A:31:16:65:53:5F

    Signature Algorithm: ecdsa-with-SHA256
         30:46:02:21:00:f7:f0:09:26:90:49:4e:46:c8:b1:c5:cb:d1:
         a5:08:5e:1e:65:d4:36:0f:98:e9:6c:4e:8e:49:5d:c5:e2:16:
         d0:02:21:00:bf:a2:3d:8f:57:47:0d:89:fd:da:f0:3f:04:64:
         b0:ae:8e:1f:95:6d:6f:67:a3:11:24:38:58:24:68:97:80:a9

-----BEGIN CERTIFICATE-----
MIIBnjCCAUOgAwIBAgIIU0xFgnNiNRQwCgYIKoZIzj0EAwIwIjEgMB4GCisGAQQB
gqJ8AQQMEENBQ0FDQUNBMDAwMDAwMDEwHhcNMjAxMDE1MTQyMzQzWhcNNDAxMDE1
MTQyMzQyWjAiMSAwHgYKKwYBBAGConwBBAwQQ0FDQUNBQ0EwMDAwMDAwMTBZMBMG
ByqGSM49AgEGCCqGSM49AwEHA0IABDuIRg7JaHpdDztLOxP80pnC9tUFHQA+5JyZ
JM+Y9PeA6yD9N8jTWDR/X4fQjDIT5UCvEbq5E35JNU8MW2ND3mOjYzBhMA8GA1Ud
EwEB/wQFMAMBAf8wDgYDVR0PAQH/BAQDAgEGMB0GA1UdDgQWBBTMEwivgs/uUF6y
O1e/6GoxFmVTXzAfBgNVHSMEGDAWgBTMEwivgs/uUF6yO1e/6GoxFmVTXzAKBggq
hkjOPQQDAgNJADBGAiEA9/AJJpBJTkbIscXL0aUIXh5l1DYPmOlsTo5JXcXiFtAC
IQC/oj2PV0cNif3a8D8EZLCujh+VbW9noxEkOFgkaJeAqQ==
-----END CERTIFICATE-----

-----BEGIN EC PRIVATE KEY-----
MHcCAQEEIPzd/ZESsw0kD2rm3ieiAp57tudDd+O3l1120j3jw2gUoAoGCCqGSM49
AwEHoUQDQgAEO4hGDsloel0PO0s7E/zSmcL21QUdAD7knJkkz5j094DrIP03yNNY
NH9fh9CMMhPlQK8RurkTfkk1TwxbY0PeYw==
-----END EC PRIVATE KEY-----
*/

extern const uint8_t sTestCert_Root01_Chip[] = {
    0x15, 0x30, 0x01, 0x08, 0x53, 0x4c, 0x45, 0x82, 0x73, 0x62, 0x35, 0x14, 0x24, 0x02, 0x01, 0x37, 0x03, 0x27, 0x14, 0x01, 0x00,
    0x00, 0x00, 0xca, 0xca, 0xca, 0xca, 0x18, 0x26, 0x04, 0xef, 0x17, 0x1b, 0x27, 0x26, 0x05, 0x6e, 0xb5, 0xb9, 0x4c, 0x37, 0x06,
    0x27, 0x14, 0x01, 0x00, 0x00, 0x00, 0xca, 0xca, 0xca, 0xca, 0x18, 0x24, 0x07, 0x01, 0x24, 0x08, 0x01, 0x30, 0x09, 0x41, 0x04,
    0x3b, 0x88, 0x46, 0x0e, 0xc9, 0x68, 0x7a, 0x5d, 0x0f, 0x3b, 0x4b, 0x3b, 0x13, 0xfc, 0xd2, 0x99, 0xc2, 0xf6, 0xd5, 0x05, 0x1d,
    0x00, 0x3e, 0xe4, 0x9c, 0x99, 0x24, 0xcf, 0x98, 0xf4, 0xf7, 0x80, 0xeb, 0x20, 0xfd, 0x37, 0xc8, 0xd3, 0x58, 0x34, 0x7f, 0x5f,
    0x87, 0xd0, 0x8c, 0x32, 0x13, 0xe5, 0x40, 0xaf, 0x11, 0xba, 0xb9, 0x13, 0x7e, 0x49, 0x35, 0x4f, 0x0c, 0x5b, 0x63, 0x43, 0xde,
    0x63, 0x37, 0x0a, 0x35, 0x01, 0x29, 0x01, 0x18, 0x24, 0x02, 0x60, 0x30, 0x04, 0x14, 0xcc, 0x13, 0x08, 0xaf, 0x82, 0xcf, 0xee,
    0x50, 0x5e, 0xb2, 0x3b, 0x57, 0xbf, 0xe8, 0x6a, 0x31, 0x16, 0x65, 0x53, 0x5f, 0x30, 0x05, 0x14, 0xcc, 0x13, 0x08, 0xaf, 0x82,
    0xcf, 0xee, 0x50, 0x5e, 0xb2, 0x3b, 0x57, 0xbf, 0xe8, 0x6a, 0x31, 0x16, 0x65, 0x53, 0x5f, 0x18, 0x30, 0x0b, 0x40, 0xf7, 0xf0,
    0x09, 0x26, 0x90, 0x49, 0x4e, 0x46, 0xc8, 0xb1, 0xc5, 0xcb, 0xd1, 0xa5, 0x08, 0x5e, 0x1e, 0x65, 0xd4, 0x36, 0x0f, 0x98, 0xe9,
    0x6c, 0x4e, 0x8e, 0x49, 0x5d, 0xc5, 0xe2, 0x16, 0xd0, 0xbf, 0xa2, 0x3d, 0x8f, 0x57, 0x47, 0x0d, 0x89, 0xfd, 0xda, 0xf0, 0x3f,
    0x04, 0x64, 0xb0, 0xae, 0x8e, 0x1f, 0x95, 0x6d, 0x6f, 0x67, 0xa3, 0x11, 0x24, 0x38, 0x58, 0x24, 0x68, 0x97, 0x80, 0xa9, 0x18,
};

extern const size_t sTestCert_Root01_Chip_Len = sizeof(sTestCert_Root01_Chip);

extern const uint8_t sTestCert_Root01_DER[] = {
    0x30, 0x82, 0x01, 0x9e, 0x30, 0x82, 0x01, 0x43, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x53, 0x4c, 0x45, 0x82, 0x73, 0x62,
    0x35, 0x14, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x22, 0x31, 0x20, 0x30, 0x1e, 0x06,
    0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x04, 0x0c, 0x10, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41,
    0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x31, 0x30, 0x1e, 0x17, 0x0d, 0x32, 0x30, 0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32,
    0x33, 0x34, 0x33, 0x5a, 0x17, 0x0d, 0x34, 0x30, 0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32, 0x33, 0x34, 0x32, 0x5a, 0x30, 0x22,
    0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x04, 0x0c, 0x10, 0x43, 0x41, 0x43,
    0x41, 0x43, 0x41, 0x43, 0x41, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x31, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86,
    0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x3b, 0x88,
    0x46, 0x0e, 0xc9, 0x68, 0x7a, 0x5d, 0x0f, 0x3b, 0x4b, 0x3b, 0x13, 0xfc, 0xd2, 0x99, 0xc2, 0xf6, 0xd5, 0x05, 0x1d, 0x00, 0x3e,
    0xe4, 0x9c, 0x99, 0x24, 0xcf, 0x98, 0xf4, 0xf7, 0x80, 0xeb, 0x20, 0xfd, 0x37, 0xc8, 0xd3, 0x58, 0x34, 0x7f, 0x5f, 0x87, 0xd0,
    0x8c, 0x32, 0x13, 0xe5, 0x40, 0xaf, 0x11, 0xba, 0xb9, 0x13, 0x7e, 0x49, 0x35, 0x4f, 0x0c, 0x5b, 0x63, 0x43, 0xde, 0x63, 0xa3,
    0x63, 0x30, 0x61, 0x30, 0x0f, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xff, 0x30,
    0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x01, 0x06, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d,
    0x0e, 0x04, 0x16, 0x04, 0x14, 0xcc, 0x13, 0x08, 0xaf, 0x82, 0xcf, 0xee, 0x50, 0x5e, 0xb2, 0x3b, 0x57, 0xbf, 0xe8, 0x6a, 0x31,
    0x16, 0x65, 0x53, 0x5f, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xcc, 0x13, 0x08, 0xaf,
    0x82, 0xcf, 0xee, 0x50, 0x5e, 0xb2, 0x3b, 0x57, 0xbf, 0xe8, 0x6a, 0x31, 0x16, 0x65, 0x53, 0x5f, 0x30, 0x0a, 0x06, 0x08, 0x2a,
    0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x49, 0x00, 0x30, 0x46, 0x02, 0x21, 0x00, 0xf7, 0xf0, 0x09, 0x26, 0x90, 0x49,
    0x4e, 0x46, 0xc8, 0xb1, 0xc5, 0xcb, 0xd1, 0xa5, 0x08, 0x5e, 0x1e, 0x65, 0xd4, 0x36, 0x0f, 0x98, 0xe9, 0x6c, 0x4e, 0x8e, 0x49,
    0x5d, 0xc5, 0xe2, 0x16, 0xd0, 0x02, 0x21, 0x00, 0xbf, 0xa2, 0x3d, 0x8f, 0x57, 0x47, 0x0d, 0x89, 0xfd, 0xda, 0xf0, 0x3f, 0x04,
    0x64, 0xb0, 0xae, 0x8e, 0x1f, 0x95, 0x6d, 0x6f, 0x67, 0xa3, 0x11, 0x24, 0x38, 0x58, 0x24, 0x68, 0x97, 0x80, 0xa9,
};

extern const size_t sTestCert_Root01_DER_Len = sizeof(sTestCert_Root01_DER);

extern const uint8_t sTestCert_Root01_PublicKey[] = {
    0x04, 0x3b, 0x88, 0x46, 0x0e, 0xc9, 0x68, 0x7a, 0x5d, 0x0f, 0x3b, 0x4b, 0x3b, 0x13, 0xfc, 0xd2, 0x99,
    0xc2, 0xf6, 0xd5, 0x05, 0x1d, 0x00, 0x3e, 0xe4, 0x9c, 0x99, 0x24, 0xcf, 0x98, 0xf4, 0xf7, 0x80, 0xeb,
    0x20, 0xfd, 0x37, 0xc8, 0xd3, 0x58, 0x34, 0x7f, 0x5f, 0x87, 0xd0, 0x8c, 0x32, 0x13, 0xe5, 0x40, 0xaf,
    0x11, 0xba, 0xb9, 0x13, 0x7e, 0x49, 0x35, 0x4f, 0x0c, 0x5b, 0x63, 0x43, 0xde, 0x63,
};

extern const size_t sTestCert_Root01_PublicKey_Len = sizeof(sTestCert_Root01_PublicKey);

extern const uint8_t sTestCert_Root01_PrivateKey[] = {
    0xfc, 0xdd, 0xfd, 0x91, 0x12, 0xb3, 0x0d, 0x24, 0x0f, 0x6a, 0xe6, 0xde, 0x27, 0xa2, 0x02, 0x9e,
    0x7b, 0xb6, 0xe7, 0x43, 0x77, 0xe3, 0xb7, 0x97, 0x5d, 0x76, 0xd2, 0x3d, 0xe3, 0xc3, 0x68, 0x14,
};

extern const size_t sTestCert_Root01_PrivateKey_Len = sizeof(sTestCert_Root01_PrivateKey);

extern const uint8_t sTestCert_Root01_SubjectKeyId[] = {
    0xCC, 0x13, 0x08, 0xAF, 0x82, 0xCF, 0xEE, 0x50, 0x5E, 0xB2, 0x3B, 0x57, 0xBF, 0xE8, 0x6A, 0x31, 0x16, 0x65, 0x53, 0x5F,
};

extern const size_t sTestCert_Root01_SubjectKeyId_Len = sizeof(sTestCert_Root01_SubjectKeyId);

extern const uint8_t sTestCert_Root01_AuthorityKeyId[] = {
    0xCC, 0x13, 0x08, 0xAF, 0x82, 0xCF, 0xEE, 0x50, 0x5E, 0xB2, 0x3B, 0x57, 0xBF, 0xE8, 0x6A, 0x31, 0x16, 0x65, 0x53, 0x5F,
};

extern const size_t sTestCert_Root01_AuthorityKeyId_Len = sizeof(sTestCert_Root01_AuthorityKeyId);

/**************  Test Root02 Certificate  **************
Certificate:
    Data:
        Version: 3 (0x2)
        Serial Number: 6921238115407929496 (0x600d2d654ac8a098)
        Signature Algorithm: ecdsa-with-SHA256
        Issuer: 1.3.6.1.4.1.37244.1.4 = CACACACA00000002, 1.3.6.1.4.1.37244.1.5 = FAB000000000001D
        Validity
            Not Before: Oct 15 14:23:43 2020 GMT
            Not After : Oct 15 14:23:42 2040 GMT
        Subject: 1.3.6.1.4.1.37244.1.4 = CACACACA00000002, 1.3.6.1.4.1.37244.1.5 = FAB000000000001D
        Subject Public Key Info:
            Public Key Algorithm: id-ecPublicKey
                Public-Key: (256 bit)
                pub:
                    04:27:50:0b:20:60:52:ce:33:77:6c:63:08:3f:1c:
                    f1:03:6e:a4:cc:7f:fd:61:7c:17:6d:4c:ad:f5:51:
                    bb:b4:b0:d9:97:ca:e5:55:db:f9:bc:a8:56:e4:cc:
                    7a:8e:de:91:e0:a7:33:e1:67:c0:41:67:a6:c2:c9:
                    fa:48:f1:4f:0b
                ASN1 OID: prime256v1
                NIST CURVE: P-256
        X509v3 extensions:
            X509v3 Basic Constraints: critical
                CA:TRUE, pathlen:1
            X509v3 Key Usage: critical
                Certificate Sign, CRL Sign
            X509v3 Subject Key Identifier:
                62:BE:B9:67:1C:91:C3:55:C8:6F:06:FA:6C:08:80:14:51:E1:A0:EA
            X509v3 Authority Key Identifier:
                keyid:62:BE:B9:67:1C:91:C3:55:C8:6F:06:FA:6C:08:80:14:51:E1:A0:EA

    Signature Algorithm: ecdsa-with-SHA256
         30:46:02:21:00:b7:e7:6c:d3:95:be:d1:21:51:2f:10:f7:2a:
         7a:b1:5a:91:60:b7:f0:38:6c:e8:19:36:d1:5a:cd:19:c8:c0:
         4a:02:21:00:d1:9d:3e:b5:c7:ea:f1:a5:cb:06:43:bb:67:68:
         54:02:e2:ff:1e:65:80:bc:c5:2c:2b:03:a3:b6:a4:92:00:5e

-----BEGIN CERTIFICATE-----
MIIB5TCCAYqgAwIBAgIIYA0tZUrIoJgwCgYIKoZIzj0EAwIwRDEgMB4GCisGAQQB
gqJ8AQQMEENBQ0FDQUNBMDAwMDAwMDIxIDAeBgorBgEEAYKifAEFDBBGQUIwMDAw
MDAwMDAwMDFEMB4XDTIwMTAxNTE0MjM0M1oXDTQwMTAxNTE0MjM0MlowRDEgMB4G
CisGAQQBgqJ8AQQMEENBQ0FDQUNBMDAwMDAwMDIxIDAeBgorBgEEAYKifAEFDBBG
QUIwMDAwMDAwMDAwMDFEMFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAEJ1ALIGBS
zjN3bGMIPxzxA26kzH/9YXwXbUyt9VG7tLDZl8rlVdv5vKhW5Mx6jt6R4Kcz4WfA
QWemwsn6SPFPC6NmMGQwEgYDVR0TAQH/BAgwBgEB/wIBATAOBgNVHQ8BAf8EBAMC
AQYwHQYDVR0OBBYEFGK+uWcckcNVyG8G+mwIgBRR4aDqMB8GA1UdIwQYMBaAFGK+
uWcckcNVyG8G+mwIgBRR4aDqMAoGCCqGSM49BAMCA0kAMEYCIQC352zTlb7RIVEv
EPcqerFakWC38Dhs6Bk20VrNGcjASgIhANGdPrXH6vGlywZDu2doVALi/x5lgLzF
LCsDo7akkgBe
-----END CERTIFICATE-----

-----BEGIN EC PRIVATE KEY-----
MHcCAQEEIPm3IofWpI3VEJPEEuYbQ6/hIhUolnG/M5b/l/a2IbKsoAoGCCqGSM49
AwEHoUQDQgAEJ1ALIGBSzjN3bGMIPxzxA26kzH/9YXwXbUyt9VG7tLDZl8rlVdv5
vKhW5Mx6jt6R4Kcz4WfAQWemwsn6SPFPCw==
-----END EC PRIVATE KEY-----
*/

extern const uint8_t sTestCert_Root02_Chip[] = {
    0x15, 0x30, 0x01, 0x08, 0x60, 0x0d, 0x2d, 0x65, 0x4a, 0xc8, 0xa0, 0x98, 0x24, 0x02, 0x01, 0x37, 0x03, 0x27, 0x14, 0x02,
    0x00, 0x00, 0x00, 0xca, 0xca, 0xca, 0xca, 0x27, 0x15, 0x1d, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0, 0xfa, 0x18, 0x26, 0x04,
    0xef, 0x17, 0x1b, 0x27, 0x26, 0x05, 0x6e, 0xb5, 0xb9, 0x4c, 0x37, 0x06, 0x27, 0x14, 0x02, 0x00, 0x00, 0x00, 0xca, 0xca,
    0xca, 0xca, 0x27, 0x15, 0x1d, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0, 0xfa, 0x18, 0x24, 0x07, 0x01, 0x24, 0x08, 0x01, 0x30,
    0x09, 0x41, 0x04, 0x27, 0x50, 0x0b, 0x20, 0x60, 0x52, 0xce, 0x33, 0x77, 0x6c, 0x63, 0x08, 0x3f, 0x1c, 0xf1, 0x03, 0x6e,
    0xa4, 0xcc, 0x7f, 0xfd, 0x61, 0x7c, 0x17, 0x6d, 0x4c, 0xad, 0xf5, 0x51, 0xbb, 0xb4, 0xb0, 0xd9, 0x97, 0xca, 0xe5, 0x55,
    0xdb, 0xf9, 0xbc, 0xa8, 0x56, 0xe4, 0xcc, 0x7a, 0x8e, 0xde, 0x91, 0xe0, 0xa7, 0x33, 0xe1, 0x67, 0xc0, 0x41, 0x67, 0xa6,
    0xc2, 0xc9, 0xfa, 0x48, 0xf1, 0x4f, 0x0b, 0x37, 0x0a, 0x35, 0x01, 0x29, 0x01, 0x24, 0x02, 0x01, 0x18, 0x24, 0x02, 0x60,
    0x30, 0x04, 0x14, 0x62, 0xbe, 0xb9, 0x67, 0x1c, 0x91, 0xc3, 0x55, 0xc8, 0x6f, 0x06, 0xfa, 0x6c, 0x08, 0x80, 0x14, 0x51,
    0xe1, 0xa0, 0xea, 0x30, 0x05, 0x14, 0x62, 0xbe, 0xb9, 0x67, 0x1c, 0x91, 0xc3, 0x55, 0xc8, 0x6f, 0x06, 0xfa, 0x6c, 0x08,
    0x80, 0x14, 0x51, 0xe1, 0xa0, 0xea, 0x18, 0x30, 0x0b, 0x40, 0xb7, 0xe7, 0x6c, 0xd3, 0x95, 0xbe, 0xd1, 0x21, 0x51, 0x2f,
    0x10, 0xf7, 0x2a, 0x7a, 0xb1, 0x5a, 0x91, 0x60, 0xb7, 0xf0, 0x38, 0x6c, 0xe8, 0x19, 0x36, 0xd1, 0x5a, 0xcd, 0x19, 0xc8,
    0xc0, 0x4a, 0xd1, 0x9d, 0x3e, 0xb5, 0xc7, 0xea, 0xf1, 0xa5, 0xcb, 0x06, 0x43, 0xbb, 0x67, 0x68, 0x54, 0x02, 0xe2, 0xff,
    0x1e, 0x65, 0x80, 0xbc, 0xc5, 0x2c, 0x2b, 0x03, 0xa3, 0xb6, 0xa4, 0x92, 0x00, 0x5e, 0x18,
};

extern const size_t sTestCert_Root02_Chip_Len = sizeof(sTestCert_Root02_Chip);

extern const uint8_t sTestCert_Root02_DER[] = {
    0x30, 0x82, 0x01, 0xe5, 0x30, 0x82, 0x01, 0x8a, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x60, 0x0d, 0x2d, 0x65, 0x4a, 0xc8,
    0xa0, 0x98, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x44, 0x31, 0x20, 0x30, 0x1e, 0x06,
    0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x04, 0x0c, 0x10, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41,
    0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x32, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2,
    0x7c, 0x01, 0x05, 0x0c, 0x10, 0x46, 0x41, 0x42, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x31, 0x44,
    0x30, 0x1e, 0x17, 0x0d, 0x32, 0x30, 0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x17, 0x0d, 0x34, 0x30,
    0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32, 0x33, 0x34, 0x32, 0x5a, 0x30, 0x44, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06,
    0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x04, 0x0c, 0x10, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41, 0x30, 0x30, 0x30,
    0x30, 0x30, 0x30, 0x30, 0x32, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x05,
    0x0c, 0x10, 0x46, 0x41, 0x42, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x31, 0x44, 0x30, 0x59, 0x30,
    0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03,
    0x42, 0x00, 0x04, 0x27, 0x50, 0x0b, 0x20, 0x60, 0x52, 0xce, 0x33, 0x77, 0x6c, 0x63, 0x08, 0x3f, 0x1c, 0xf1, 0x03, 0x6e, 0xa4,
    0xcc, 0x7f, 0xfd, 0x61, 0x7c, 0x17, 0x6d, 0x4c, 0xad, 0xf5, 0x51, 0xbb, 0xb4, 0xb0, 0xd9, 0x97, 0xca, 0xe5, 0x55, 0xdb, 0xf9,
    0xbc, 0xa8, 0x56, 0xe4, 0xcc, 0x7a, 0x8e, 0xde, 0x91, 0xe0, 0xa7, 0x33, 0xe1, 0x67, 0xc0, 0x41, 0x67, 0xa6, 0xc2, 0xc9, 0xfa,
    0x48, 0xf1, 0x4f, 0x0b, 0xa3, 0x66, 0x30, 0x64, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x08, 0x30,
    0x06, 0x01, 0x01, 0xff, 0x02, 0x01, 0x01, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02,
    0x01, 0x06, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x62, 0xbe, 0xb9, 0x67, 0x1c, 0x91, 0xc3, 0x55,
    0xc8, 0x6f, 0x06, 0xfa, 0x6c, 0x08, 0x80, 0x14, 0x51, 0xe1, 0xa0, 0xea, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18,
    0x30, 0x16, 0x80, 0x14, 0x62, 0xbe, 0xb9, 0x67, 0x1c, 0x91, 0xc3, 0x55, 0xc8, 0x6f, 0x06, 0xfa, 0x6c, 0x08, 0x80, 0x14, 0x51,
    0xe1, 0xa0, 0xea, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x49, 0x00, 0x30, 0x46, 0x02,
    0x21, 0x00, 0xb7, 0xe7, 0x6c, 0xd3, 0x95, 0xbe, 0xd1, 0x21, 0x51, 0x2f, 0x10, 0xf7, 0x2a, 0x7a, 0xb1, 0x5a, 0x91, 0x60, 0xb7,
    0xf0, 0x38, 0x6c, 0xe8, 0x19, 0x36, 0xd1, 0x5a, 0xcd, 0x19, 0xc8, 0xc0, 0x4a, 0x02, 0x21, 0x00, 0xd1, 0x9d, 0x3e, 0xb5, 0xc7,
    0xea, 0xf1, 0xa5, 0xcb, 0x06, 0x43, 0xbb, 0x67, 0x68, 0x54, 0x02, 0xe2, 0xff, 0x1e, 0x65, 0x80, 0xbc, 0xc5, 0x2c, 0x2b, 0x03,
    0xa3, 0xb6, 0xa4, 0x92, 0x00, 0x5e,
};

extern const size_t sTestCert_Root02_DER_Len = sizeof(sTestCert_Root02_DER);

extern const uint8_t sTestCert_Root02_PublicKey[] = {
    0x04, 0x27, 0x50, 0x0b, 0x20, 0x60, 0x52, 0xce, 0x33, 0x77, 0x6c, 0x63, 0x08, 0x3f, 0x1c, 0xf1, 0x03,
    0x6e, 0xa4, 0xcc, 0x7f, 0xfd, 0x61, 0x7c, 0x17, 0x6d, 0x4c, 0xad, 0xf5, 0x51, 0xbb, 0xb4, 0xb0, 0xd9,
    0x97, 0xca, 0xe5, 0x55, 0xdb, 0xf9, 0xbc, 0xa8, 0x56, 0xe4, 0xcc, 0x7a, 0x8e, 0xde, 0x91, 0xe0, 0xa7,
    0x33, 0xe1, 0x67, 0xc0, 0x41, 0x67, 0xa6, 0xc2, 0xc9, 0xfa, 0x48, 0xf1, 0x4f, 0x0b,
};

extern const size_t sTestCert_Root02_PublicKey_Len = sizeof(sTestCert_Root02_PublicKey);

extern const uint8_t sTestCert_Root02_PrivateKey[] = {
    0xf9, 0xb7, 0x22, 0x87, 0xd6, 0xa4, 0x8d, 0xd5, 0x10, 0x93, 0xc4, 0x12, 0xe6, 0x1b, 0x43, 0xaf,
    0xe1, 0x22, 0x15, 0x28, 0x96, 0x71, 0xbf, 0x33, 0x96, 0xff, 0x97, 0xf6, 0xb6, 0x21, 0xb2, 0xac,
};

extern const size_t sTestCert_Root02_PrivateKey_Len = sizeof(sTestCert_Root02_PrivateKey);

extern const uint8_t sTestCert_Root02_SubjectKeyId[] = {
    0x62, 0xBE, 0xB9, 0x67, 0x1C, 0x91, 0xC3, 0x55, 0xC8, 0x6F, 0x06, 0xFA, 0x6C, 0x08, 0x80, 0x14, 0x51, 0xE1, 0xA0, 0xEA,
};

extern const size_t sTestCert_Root02_SubjectKeyId_Len = sizeof(sTestCert_Root02_SubjectKeyId);

extern const uint8_t sTestCert_Root02_AuthorityKeyId[] = {
    0x62, 0xBE, 0xB9, 0x67, 0x1C, 0x91, 0xC3, 0x55, 0xC8, 0x6F, 0x06, 0xFA, 0x6C, 0x08, 0x80, 0x14, 0x51, 0xE1, 0xA0, 0xEA,
};

extern const size_t sTestCert_Root02_AuthorityKeyId_Len = sizeof(sTestCert_Root02_AuthorityKeyId);

/**************  Test Root03 Certificate  **************
Certificate:
    Data:
        Version: 3 (0x2)
        Serial Number: 7744074172075392897 (0x6b787a6dfcd4bf81)
        Signature Algorithm: ecdsa-with-SHA256
        Issuer: 1.3.6.1.4.1.37244.1.4 = CACACACA00000002, 1.3.6.1.4.1.37244.1.5 = FAB000000000001D
        Validity
            Not Before: Oct 15 14:23:43 2020 GMT
            Not After : Dec 31 23:59:59 9999 GMT
        Subject: 1.3.6.1.4.1.37244.1.4 = CACACACA00000002, 1.3.6.1.4.1.37244.1.5 = FAB000000000001D
        Subject Public Key Info:
            Public Key Algorithm: id-ecPublicKey
                Public-Key: (256 bit)
                pub:
                    04:71:05:40:8a:85:a9:d9:a0:8a:f8:b7:70:77:db:
                    38:8b:7e:a4:38:97:dc:df:d3:16:f2:4f:0a:7e:71:
                    de:69:a5:0c:44:55:0c:0c:9d:a2:36:1c:d8:29:e5:
                    5f:dd:63:cc:f5:79:ed:44:e0:22:08:b4:f8:25:fc:
                    d6:f0:68:2c:02
                ASN1 OID: prime256v1
                NIST CURVE: P-256
        X509v3 extensions:
            X509v3 Basic Constraints: critical
                CA:TRUE, pathlen:1
            X509v3 Key Usage: critical
                Certificate Sign, CRL Sign
            X509v3 Subject Key Identifier:
                FF:87:F3:CD:D2:06:9A:EF:8D:5D:32:EB:A3:16:3B:9E:B0:0A:00:29
            X509v3 Authority Key Identifier:
                keyid:FF:87:F3:CD:D2:06:9A:EF:8D:5D:32:EB:A3:16:3B:9E:B0:0A:00:29

    Signature Algorithm: ecdsa-with-SHA256
         30:44:02:20:75:f6:70:eb:35:45:b9:97:86:46:67:99:eb:ea:
         40:51:a3:41:af:2a:9b:67:7f:d7:1b:4c:3d:4e:68:09:3b:66:
         02:20:5f:28:42:cb:7e:d3:19:9c:9a:d7:c9:62:79:47:a1:8a:
         92:16:8a:fc:d6:5f:3e:9c:af:6e:ed:fa:9e:60:c5:2f

-----BEGIN CERTIFICATE-----
MIIB5TCCAYygAwIBAgIIa3h6bfzUv4EwCgYIKoZIzj0EAwIwRDEgMB4GCisGAQQB
gqJ8AQQMEENBQ0FDQUNBMDAwMDAwMDIxIDAeBgorBgEEAYKifAEFDBBGQUIwMDAw
MDAwMDAwMDFEMCAXDTIwMTAxNTE0MjM0M1oYDzk5OTkxMjMxMjM1OTU5WjBEMSAw
HgYKKwYBBAGConwBBAwQQ0FDQUNBQ0EwMDAwMDAwMjEgMB4GCisGAQQBgqJ8AQUM
EEZBQjAwMDAwMDAwMDAwMUQwWTATBgcqhkjOPQIBBggqhkjOPQMBBwNCAARxBUCK
hanZoIr4t3B32ziLfqQ4l9zf0xbyTwp+cd5ppQxEVQwMnaI2HNgp5V/dY8z1ee1E
4CIItPgl/NbwaCwCo2YwZDASBgNVHRMBAf8ECDAGAQH/AgEBMA4GA1UdDwEB/wQE
AwIBBjAdBgNVHQ4EFgQU/4fzzdIGmu+NXTLroxY7nrAKACkwHwYDVR0jBBgwFoAU
/4fzzdIGmu+NXTLroxY7nrAKACkwCgYIKoZIzj0EAwIDRwAwRAIgdfZw6zVFuZeG
RmeZ6+pAUaNBryqbZ3/XG0w9TmgJO2YCIF8oQst+0xmcmtfJYnlHoYqSFor81l8+
nK9u7fqeYMUv
-----END CERTIFICATE-----

-----BEGIN EC PRIVATE KEY-----
MHcCAQEEIHDnbQApKIHktaW8tOzQL/XkdbwUid5uw+jWPFo/LlqHoAoGCCqGSM49
AwEHoUQDQgAEcQVAioWp2aCK+Ldwd9s4i36kOJfc39MW8k8KfnHeaaUMRFUMDJ2i
NhzYKeVf3WPM9XntROAiCLT4JfzW8GgsAg==
-----END EC PRIVATE KEY-----
*/

extern const uint8_t sTestCert_Root03_Chip[] = {
    0x15, 0x30, 0x01, 0x08, 0x6b, 0x78, 0x7a, 0x6d, 0xfc, 0xd4, 0xbf, 0x81, 0x24, 0x02, 0x01, 0x37, 0x03, 0x27, 0x14, 0x02, 0x00,
    0x00, 0x00, 0xca, 0xca, 0xca, 0xca, 0x27, 0x15, 0x1d, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0, 0xfa, 0x18, 0x26, 0x04, 0xef, 0x17,
    0x1b, 0x27, 0x24, 0x05, 0x00, 0x37, 0x06, 0x27, 0x14, 0x02, 0x00, 0x00, 0x00, 0xca, 0xca, 0xca, 0xca, 0x27, 0x15, 0x1d, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xb0, 0xfa, 0x18, 0x24, 0x07, 0x01, 0x24, 0x08, 0x01, 0x30, 0x09, 0x41, 0x04, 0x71, 0x05, 0x40, 0x8a,
    0x85, 0xa9, 0xd9, 0xa0, 0x8a, 0xf8, 0xb7, 0x70, 0x77, 0xdb, 0x38, 0x8b, 0x7e, 0xa4, 0x38, 0x97, 0xdc, 0xdf, 0xd3, 0x16, 0xf2,
    0x4f, 0x0a, 0x7e, 0x71, 0xde, 0x69, 0xa5, 0x0c, 0x44, 0x55, 0x0c, 0x0c, 0x9d, 0xa2, 0x36, 0x1c, 0xd8, 0x29, 0xe5, 0x5f, 0xdd,
    0x63, 0xcc, 0xf5, 0x79, 0xed, 0x44, 0xe0, 0x22, 0x08, 0xb4, 0xf8, 0x25, 0xfc, 0xd6, 0xf0, 0x68, 0x2c, 0x02, 0x37, 0x0a, 0x35,
    0x01, 0x29, 0x01, 0x24, 0x02, 0x01, 0x18, 0x24, 0x02, 0x60, 0x30, 0x04, 0x14, 0xff, 0x87, 0xf3, 0xcd, 0xd2, 0x06, 0x9a, 0xef,
    0x8d, 0x5d, 0x32, 0xeb, 0xa3, 0x16, 0x3b, 0x9e, 0xb0, 0x0a, 0x00, 0x29, 0x30, 0x05, 0x14, 0xff, 0x87, 0xf3, 0xcd, 0xd2, 0x06,
    0x9a, 0xef, 0x8d, 0x5d, 0x32, 0xeb, 0xa3, 0x16, 0x3b, 0x9e, 0xb0, 0x0a, 0x00, 0x29, 0x18, 0x30, 0x0b, 0x40, 0x75, 0xf6, 0x70,
    0xeb, 0x35, 0x45, 0xb9, 0x97, 0x86, 0x46, 0x67, 0x99, 0xeb, 0xea, 0x40, 0x51, 0xa3, 0x41, 0xaf, 0x2a, 0x9b, 0x67, 0x7f, 0xd7,
    0x1b, 0x4c, 0x3d, 0x4e, 0x68, 0x09, 0x3b, 0x66, 0x5f, 0x28, 0x42, 0xcb, 0x7e, 0xd3, 0x19, 0x9c, 0x9a, 0xd7, 0xc9, 0x62, 0x79,
    0x47, 0xa1, 0x8a, 0x92, 0x16, 0x8a, 0xfc, 0xd6, 0x5f, 0x3e, 0x9c, 0xaf, 0x6e, 0xed, 0xfa, 0x9e, 0x60, 0xc5, 0x2f, 0x18,
};

extern const size_t sTestCert_Root03_Chip_Len = sizeof(sTestCert_Root03_Chip);

extern const uint8_t sTestCert_Root03_DER[] = {
    0x30, 0x82, 0x01, 0xe5, 0x30, 0x82, 0x01, 0x8c, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x6b, 0x78, 0x7a, 0x6d, 0xfc, 0xd4,
    0xbf, 0x81, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x44, 0x31, 0x20, 0x30, 0x1e, 0x06,
    0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x04, 0x0c, 0x10, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41,
    0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x32, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2,
    0x7c, 0x01, 0x05, 0x0c, 0x10, 0x46, 0x41, 0x42, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x31, 0x44,
    0x30, 0x20, 0x17, 0x0d, 0x32, 0x30, 0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x18, 0x0f, 0x39, 0x39,
    0x39, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x44, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a,
    0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x04, 0x0c, 0x10, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41, 0x30,
    0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x32, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c,
    0x01, 0x05, 0x0c, 0x10, 0x46, 0x41, 0x42, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x31, 0x44, 0x30,
    0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01,
    0x07, 0x03, 0x42, 0x00, 0x04, 0x71, 0x05, 0x40, 0x8a, 0x85, 0xa9, 0xd9, 0xa0, 0x8a, 0xf8, 0xb7, 0x70, 0x77, 0xdb, 0x38, 0x8b,
    0x7e, 0xa4, 0x38, 0x97, 0xdc, 0xdf, 0xd3, 0x16, 0xf2, 0x4f, 0x0a, 0x7e, 0x71, 0xde, 0x69, 0xa5, 0x0c, 0x44, 0x55, 0x0c, 0x0c,
    0x9d, 0xa2, 0x36, 0x1c, 0xd8, 0x29, 0xe5, 0x5f, 0xdd, 0x63, 0xcc, 0xf5, 0x79, 0xed, 0x44, 0xe0, 0x22, 0x08, 0xb4, 0xf8, 0x25,
    0xfc, 0xd6, 0xf0, 0x68, 0x2c, 0x02, 0xa3, 0x66, 0x30, 0x64, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04,
    0x08, 0x30, 0x06, 0x01, 0x01, 0xff, 0x02, 0x01, 0x01, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04,
    0x03, 0x02, 0x01, 0x06, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xff, 0x87, 0xf3, 0xcd, 0xd2, 0x06,
    0x9a, 0xef, 0x8d, 0x5d, 0x32, 0xeb, 0xa3, 0x16, 0x3b, 0x9e, 0xb0, 0x0a, 0x00, 0x29, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23,
    0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xff, 0x87, 0xf3, 0xcd, 0xd2, 0x06, 0x9a, 0xef, 0x8d, 0x5d, 0x32, 0xeb, 0xa3, 0x16, 0x3b,
    0x9e, 0xb0, 0x0a, 0x00, 0x29, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x47, 0x00, 0x30,
    0x44, 0x02, 0x20, 0x75, 0xf6, 0x70, 0xeb, 0x35, 0x45, 0xb9, 0x97, 0x86, 0x46, 0x67, 0x99, 0xeb, 0xea, 0x40, 0x51, 0xa3, 0x41,
    0xaf, 0x2a, 0x9b, 0x67, 0x7f, 0xd7, 0x1b, 0x4c, 0x3d, 0x4e, 0x68, 0x09, 0x3b, 0x66, 0x02, 0x20, 0x5f, 0x28, 0x42, 0xcb, 0x7e,
    0xd3, 0x19, 0x9c, 0x9a, 0xd7, 0xc9, 0x62, 0x79, 0x47, 0xa1, 0x8a, 0x92, 0x16, 0x8a, 0xfc, 0xd6, 0x5f, 0x3e, 0x9c, 0xaf, 0x6e,
    0xed, 0xfa, 0x9e, 0x60, 0xc5, 0x2f,
};

extern const size_t sTestCert_Root03_DER_Len = sizeof(sTestCert_Root03_DER);

extern const uint8_t sTestCert_Root03_PublicKey[] = {
    0x04, 0x71, 0x05, 0x40, 0x8a, 0x85, 0xa9, 0xd9, 0xa0, 0x8a, 0xf8, 0xb7, 0x70, 0x77, 0xdb, 0x38, 0x8b,
    0x7e, 0xa4, 0x38, 0x97, 0xdc, 0xdf, 0xd3, 0x16, 0xf2, 0x4f, 0x0a, 0x7e, 0x71, 0xde, 0x69, 0xa5, 0x0c,
    0x44, 0x55, 0x0c, 0x0c, 0x9d, 0xa2, 0x36, 0x1c, 0xd8, 0x29, 0xe5, 0x5f, 0xdd, 0x63, 0xcc, 0xf5, 0x79,
    0xed, 0x44, 0xe0, 0x22, 0x08, 0xb4, 0xf8, 0x25, 0xfc, 0xd6, 0xf0, 0x68, 0x2c, 0x02,
};

extern const size_t sTestCert_Root03_PublicKey_Len = sizeof(sTestCert_Root03_PublicKey);

extern const uint8_t sTestCert_Root03_PrivateKey[] = {
    0x70, 0xe7, 0x6d, 0x00, 0x29, 0x28, 0x81, 0xe4, 0xb5, 0xa5, 0xbc, 0xb4, 0xec, 0xd0, 0x2f, 0xf5,
    0xe4, 0x75, 0xbc, 0x14, 0x89, 0xde, 0x6e, 0xc3, 0xe8, 0xd6, 0x3c, 0x5a, 0x3f, 0x2e, 0x5a, 0x87,
};

extern const size_t sTestCert_Root03_PrivateKey_Len = sizeof(sTestCert_Root03_PrivateKey);

extern const uint8_t sTestCert_Root03_SubjectKeyId[] = {
    0xFF, 0x87, 0xF3, 0xCD, 0xD2, 0x06, 0x9A, 0xEF, 0x8D, 0x5D, 0x32, 0xEB, 0xA3, 0x16, 0x3B, 0x9E, 0xB0, 0x0A, 0x00, 0x29,
};

extern const size_t sTestCert_Root03_SubjectKeyId_Len = sizeof(sTestCert_Root03_SubjectKeyId);

extern const uint8_t sTestCert_Root03_AuthorityKeyId[] = {
    0xFF, 0x87, 0xF3, 0xCD, 0xD2, 0x06, 0x9A, 0xEF, 0x8D, 0x5D, 0x32, 0xEB, 0xA3, 0x16, 0x3B, 0x9E, 0xB0, 0x0A, 0x00, 0x29,
};

extern const size_t sTestCert_Root03_AuthorityKeyId_Len = sizeof(sTestCert_Root03_AuthorityKeyId);

/**************  Test ICA01 Certificate  **************
Certificate:
    Data:
        Version: 3 (0x2)
        Serial Number: 7626963124938903389 (0x69d86a8d80fc8f5d)
        Signature Algorithm: ecdsa-with-SHA256
        Issuer: 1.3.6.1.4.1.37244.1.4 = CACACACA00000001
        Validity
            Not Before: Oct 15 14:23:43 2020 GMT
            Not After : Oct 15 14:23:42 2040 GMT
        Subject: 1.3.6.1.4.1.37244.1.3 = CACACACA00000003
        Subject Public Key Info:
            Public Key Algorithm: id-ecPublicKey
                Public-Key: (256 bit)
                pub:
                    04:5f:94:f5:7e:0b:13:c9:cf:cf:96:df:e1:fc:e7:
                    88:8d:56:4c:c2:09:c5:5c:45:08:e4:4d:cf:16:ba:
                    2e:09:66:2f:9e:ec:f1:9f:40:b0:e8:8a:0b:28:15:
                    da:9e:e1:0a:3a:17:7c:25:1f:43:4f:5b:0f:26:3c:
                    e7:de:62:78:c6
                ASN1 OID: prime256v1
                NIST CURVE: P-256
        X509v3 extensions:
            X509v3 Basic Constraints: critical
                CA:TRUE
            X509v3 Key Usage: critical
                Certificate Sign, CRL Sign
            X509v3 Subject Key Identifier:
                44:0C:C6:92:31:C4:CB:5B:37:94:24:26:F8:1B:BE:24:B7:EF:34:5C
            X509v3 Authority Key Identifier:
                keyid:CC:13:08:AF:82:CF:EE:50:5E:B2:3B:57:BF:E8:6A:31:16:65:53:5F

    Signature Algorithm: ecdsa-with-SHA256
         30:45:02:21:00:ad:b8:5b:5d:68:cb:fd:36:14:0d:8c:9d:12:
         90:14:c4:5f:a7:ca:19:1f:34:d9:af:24:1d:b7:17:36:e6:0f:
         44:02:20:19:9b:c0:7c:7f:79:5b:ed:81:a2:e7:7d:c5:34:25:
         76:f6:a0:d1:41:98:f4:6b:91:07:49:42:7c:2e:ed:65:9c

-----BEGIN CERTIFICATE-----
MIIBnTCCAUOgAwIBAgIIadhqjYD8j10wCgYIKoZIzj0EAwIwIjEgMB4GCisGAQQB
gqJ8AQQMEENBQ0FDQUNBMDAwMDAwMDEwHhcNMjAxMDE1MTQyMzQzWhcNNDAxMDE1
MTQyMzQyWjAiMSAwHgYKKwYBBAGConwBAwwQQ0FDQUNBQ0EwMDAwMDAwMzBZMBMG
ByqGSM49AgEGCCqGSM49AwEHA0IABF+U9X4LE8nPz5bf4fzniI1WTMIJxVxFCORN
zxa6LglmL57s8Z9AsOiKCygV2p7hCjoXfCUfQ09bDyY8595ieMajYzBhMA8GA1Ud
EwEB/wQFMAMBAf8wDgYDVR0PAQH/BAQDAgEGMB0GA1UdDgQWBBREDMaSMcTLWzeU
JCb4G74kt+80XDAfBgNVHSMEGDAWgBTMEwivgs/uUF6yO1e/6GoxFmVTXzAKBggq
hkjOPQQDAgNIADBFAiEArbhbXWjL/TYUDYydEpAUxF+nyhkfNNmvJB23FzbmD0QC
IBmbwHx/eVvtgaLnfcU0JXb2oNFBmPRrkQdJQnwu7WWc
-----END CERTIFICATE-----

-----BEGIN EC PRIVATE KEY-----
MHcCAQEEIJ4WvYoXsvHCFCEPUWB+l5bLWWBBU1fTirhsxNWWIS6XoAoGCCqGSM49
AwEHoUQDQgAEX5T1fgsTyc/Plt/h/OeIjVZMwgnFXEUI5E3PFrouCWYvnuzxn0Cw
6IoLKBXanuEKOhd8JR9DT1sPJjzn3mJ4xg==
-----END EC PRIVATE KEY-----
*/

extern const uint8_t sTestCert_ICA01_Chip[] = {
    0x15, 0x30, 0x01, 0x08, 0x69, 0xd8, 0x6a, 0x8d, 0x80, 0xfc, 0x8f, 0x5d, 0x24, 0x02, 0x01, 0x37, 0x03, 0x27, 0x14, 0x01, 0x00,
    0x00, 0x00, 0xca, 0xca, 0xca, 0xca, 0x18, 0x26, 0x04, 0xef, 0x17, 0x1b, 0x27, 0x26, 0x05, 0x6e, 0xb5, 0xb9, 0x4c, 0x37, 0x06,
    0x27, 0x13, 0x03, 0x00, 0x00, 0x00, 0xca, 0xca, 0xca, 0xca, 0x18, 0x24, 0x07, 0x01, 0x24, 0x08, 0x01, 0x30, 0x09, 0x41, 0x04,
    0x5f, 0x94, 0xf5, 0x7e, 0x0b, 0x13, 0xc9, 0xcf, 0xcf, 0x96, 0xdf, 0xe1, 0xfc, 0xe7, 0x88, 0x8d, 0x56, 0x4c, 0xc2, 0x09, 0xc5,
    0x5c, 0x45, 0x08, 0xe4, 0x4d, 0xcf, 0x16, 0xba, 0x2e, 0x09, 0x66, 0x2f, 0x9e, 0xec, 0xf1, 0x9f, 0x40, 0xb0, 0xe8, 0x8a, 0x0b,
    0x28, 0x15, 0xda, 0x9e, 0xe1, 0x0a, 0x3a, 0x17, 0x7c, 0x25, 0x1f, 0x43, 0x4f, 0x5b, 0x0f, 0x26, 0x3c, 0xe7, 0xde, 0x62, 0x78,
    0xc6, 0x37, 0x0a, 0x35, 0x01, 0x29, 0x01, 0x18, 0x24, 0x02, 0x60, 0x30, 0x04, 0x14, 0x44, 0x0c, 0xc6, 0x92, 0x31, 0xc4, 0xcb,
    0x5b, 0x37, 0x94, 0x24, 0x26, 0xf8, 0x1b, 0xbe, 0x24, 0xb7, 0xef, 0x34, 0x5c, 0x30, 0x05, 0x14, 0xcc, 0x13, 0x08, 0xaf, 0x82,
    0xcf, 0xee, 0x50, 0x5e, 0xb2, 0x3b, 0x57, 0xbf, 0xe8, 0x6a, 0x31, 0x16, 0x65, 0x53, 0x5f, 0x18, 0x30, 0x0b, 0x40, 0xad, 0xb8,
    0x5b, 0x5d, 0x68, 0xcb, 0xfd, 0x36, 0x14, 0x0d, 0x8c, 0x9d, 0x12, 0x90, 0x14, 0xc4, 0x5f, 0xa7, 0xca, 0x19, 0x1f, 0x34, 0xd9,
    0xaf, 0x24, 0x1d, 0xb7, 0x17, 0x36, 0xe6, 0x0f, 0x44, 0x19, 0x9b, 0xc0, 0x7c, 0x7f, 0x79, 0x5b, 0xed, 0x81, 0xa2, 0xe7, 0x7d,
    0xc5, 0x34, 0x25, 0x76, 0xf6, 0xa0, 0xd1, 0x41, 0x98, 0xf4, 0x6b, 0x91, 0x07, 0x49, 0x42, 0x7c, 0x2e, 0xed, 0x65, 0x9c, 0x18,
};

extern const size_t sTestCert_ICA01_Chip_Len = sizeof(sTestCert_ICA01_Chip);

extern const uint8_t sTestCert_ICA01_DER[] = {
    0x30, 0x82, 0x01, 0x9d, 0x30, 0x82, 0x01, 0x43, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x69, 0xd8, 0x6a, 0x8d, 0x80, 0xfc,
    0x8f, 0x5d, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x22, 0x31, 0x20, 0x30, 0x1e, 0x06,
    0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x04, 0x0c, 0x10, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41,
    0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x31, 0x30, 0x1e, 0x17, 0x0d, 0x32, 0x30, 0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32,
    0x33, 0x34, 0x33, 0x5a, 0x17, 0x0d, 0x34, 0x30, 0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32, 0x33, 0x34, 0x32, 0x5a, 0x30, 0x22,
    0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x03, 0x0c, 0x10, 0x43, 0x41, 0x43,
    0x41, 0x43, 0x41, 0x43, 0x41, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x33, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86,
    0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x5f, 0x94,
    0xf5, 0x7e, 0x0b, 0x13, 0xc9, 0xcf, 0xcf, 0x96, 0xdf, 0xe1, 0xfc, 0xe7, 0x88, 0x8d, 0x56, 0x4c, 0xc2, 0x09, 0xc5, 0x5c, 0x45,
    0x08, 0xe4, 0x4d, 0xcf, 0x16, 0xba, 0x2e, 0x09, 0x66, 0x2f, 0x9e, 0xec, 0xf1, 0x9f, 0x40, 0xb0, 0xe8, 0x8a, 0x0b, 0x28, 0x15,
    0xda, 0x9e, 0xe1, 0x0a, 0x3a, 0x17, 0x7c, 0x25, 0x1f, 0x43, 0x4f, 0x5b, 0x0f, 0x26, 0x3c, 0xe7, 0xde, 0x62, 0x78, 0xc6, 0xa3,
    0x63, 0x30, 0x61, 0x30, 0x0f, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xff, 0x30,
    0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x01, 0x06, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d,
    0x0e, 0x04, 0x16, 0x04, 0x14, 0x44, 0x0c, 0xc6, 0x92, 0x31, 0xc4, 0xcb, 0x5b, 0x37, 0x94, 0x24, 0x26, 0xf8, 0x1b, 0xbe, 0x24,
    0xb7, 0xef, 0x34, 0x5c, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xcc, 0x13, 0x08, 0xaf,
    0x82, 0xcf, 0xee, 0x50, 0x5e, 0xb2, 0x3b, 0x57, 0xbf, 0xe8, 0x6a, 0x31, 0x16, 0x65, 0x53, 0x5f, 0x30, 0x0a, 0x06, 0x08, 0x2a,
    0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x48, 0x00, 0x30, 0x45, 0x02, 0x21, 0x00, 0xad, 0xb8, 0x5b, 0x5d, 0x68, 0xcb,
    0xfd, 0x36, 0x14, 0x0d, 0x8c, 0x9d, 0x12, 0x90, 0x14, 0xc4, 0x5f, 0xa7, 0xca, 0x19, 0x1f, 0x34, 0xd9, 0xaf, 0x24, 0x1d, 0xb7,
    0x17, 0x36, 0xe6, 0x0f, 0x44, 0x02, 0x20, 0x19, 0x9b, 0xc0, 0x7c, 0x7f, 0x79, 0x5b, 0xed, 0x81, 0xa2, 0xe7, 0x7d, 0xc5, 0x34,
    0x25, 0x76, 0xf6, 0xa0, 0xd1, 0x41, 0x98, 0xf4, 0x6b, 0x91, 0x07, 0x49, 0x42, 0x7c, 0x2e, 0xed, 0x65, 0x9c,
};

extern const size_t sTestCert_ICA01_DER_Len = sizeof(sTestCert_ICA01_DER);

extern const uint8_t sTestCert_ICA01_PublicKey[] = {
    0x04, 0x5f, 0x94, 0xf5, 0x7e, 0x0b, 0x13, 0xc9, 0xcf, 0xcf, 0x96, 0xdf, 0xe1, 0xfc, 0xe7, 0x88, 0x8d,
    0x56, 0x4c, 0xc2, 0x09, 0xc5, 0x5c, 0x45, 0x08, 0xe4, 0x4d, 0xcf, 0x16, 0xba, 0x2e, 0x09, 0x66, 0x2f,
    0x9e, 0xec, 0xf1, 0x9f, 0x40, 0xb0, 0xe8, 0x8a, 0x0b, 0x28, 0x15, 0xda, 0x9e, 0xe1, 0x0a, 0x3a, 0x17,
    0x7c, 0x25, 0x1f, 0x43, 0x4f, 0x5b, 0x0f, 0x26, 0x3c, 0xe7, 0xde, 0x62, 0x78, 0xc6,
};

extern const size_t sTestCert_ICA01_PublicKey_Len = sizeof(sTestCert_ICA01_PublicKey);

extern const uint8_t sTestCert_ICA01_PrivateKey[] = {
    0x9e, 0x16, 0xbd, 0x8a, 0x17, 0xb2, 0xf1, 0xc2, 0x14, 0x21, 0x0f, 0x51, 0x60, 0x7e, 0x97, 0x96,
    0xcb, 0x59, 0x60, 0x41, 0x53, 0x57, 0xd3, 0x8a, 0xb8, 0x6c, 0xc4, 0xd5, 0x96, 0x21, 0x2e, 0x97,
};

extern const size_t sTestCert_ICA01_PrivateKey_Len = sizeof(sTestCert_ICA01_PrivateKey);

extern const uint8_t sTestCert_ICA01_SubjectKeyId[] = {
    0x44, 0x0C, 0xC6, 0x92, 0x31, 0xC4, 0xCB, 0x5B, 0x37, 0x94, 0x24, 0x26, 0xF8, 0x1B, 0xBE, 0x24, 0xB7, 0xEF, 0x34, 0x5C,
};

extern const size_t sTestCert_ICA01_SubjectKeyId_Len = sizeof(sTestCert_ICA01_SubjectKeyId);

extern const uint8_t sTestCert_ICA01_AuthorityKeyId[] = {
    0xCC, 0x13, 0x08, 0xAF, 0x82, 0xCF, 0xEE, 0x50, 0x5E, 0xB2, 0x3B, 0x57, 0xBF, 0xE8, 0x6A, 0x31, 0x16, 0x65, 0x53, 0x5F,
};

extern const size_t sTestCert_ICA01_AuthorityKeyId_Len = sizeof(sTestCert_ICA01_AuthorityKeyId);

/**************  Test ICA02 Certificate  **************
Certificate:
    Data:
        Version: 3 (0x2)
        Serial Number: 2073221257978332490 (0x1cc58fbfee96454a)
        Signature Algorithm: ecdsa-with-SHA256
        Issuer: 1.3.6.1.4.1.37244.1.4 = CACACACA00000002, 1.3.6.1.4.1.37244.1.5 = FAB000000000001D
        Validity
            Not Before: Oct 15 14:23:43 2020 GMT
            Not After : Oct 15 14:23:42 2040 GMT
        Subject: 1.3.6.1.4.1.37244.1.3 = CACACACA00000004, 1.3.6.1.4.1.37244.1.5 = FAB000000000001D
        Subject Public Key Info:
            Public Key Algorithm: id-ecPublicKey
                Public-Key: (256 bit)
                pub:
                    04:fa:9f:d2:8c:b7:6a:77:ef:0e:39:30:d5:9e:41:
                    2b:d1:8e:b2:0e:ff:d5:19:7e:f4:71:39:37:93:90:
                    37:8a:48:04:32:48:18:9c:c4:a9:74:17:75:7e:0e:
                    7b:76:72:34:d9:cb:03:dc:75:28:9a:99:74:be:3d:
                    f7:61:a7:56:be
                ASN1 OID: prime256v1
                NIST CURVE: P-256
        X509v3 extensions:
            X509v3 Basic Constraints: critical
                CA:TRUE, pathlen:0
            X509v3 Key Usage: critical
                Certificate Sign, CRL Sign
            X509v3 Subject Key Identifier:
                E1:E7:6E:67:77:85:1D:D7:74:16:BD:DD:35:EC:3C:13:7C:47:29:DC
            X509v3 Authority Key Identifier:
                keyid:62:BE:B9:67:1C:91:C3:55:C8:6F:06:FA:6C:08:80:14:51:E1:A0:EA

    Signature Algorithm: ecdsa-with-SHA256
         30:45:02:20:6f:fa:73:c0:42:6c:9c:ab:2d:e6:20:76:eb:e2:
         4b:5e:79:e2:bb:98:f2:14:ab:6e:e8:f0:43:7c:55:7d:e7:43:
         02:21:00:c7:53:da:e7:dd:f3:b0:27:c2:f1:3e:23:ce:3a:3a:
         b9:d4:34:fc:7e:a1:5b:b9:d6:77:c3:6c:9c:8b:55:15:42

-----BEGIN CERTIFICATE-----
MIIB5DCCAYqgAwIBAgIIHMWPv+6WRUowCgYIKoZIzj0EAwIwRDEgMB4GCisGAQQB
gqJ8AQQMEENBQ0FDQUNBMDAwMDAwMDIxIDAeBgorBgEEAYKifAEFDBBGQUIwMDAw
MDAwMDAwMDFEMB4XDTIwMTAxNTE0MjM0M1oXDTQwMTAxNTE0MjM0MlowRDEgMB4G
CisGAQQBgqJ8AQMMEENBQ0FDQUNBMDAwMDAwMDQxIDAeBgorBgEEAYKifAEFDBBG
QUIwMDAwMDAwMDAwMDFEMFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAE+p/SjLdq
d+8OOTDVnkEr0Y6yDv/VGX70cTk3k5A3ikgEMkgYnMSpdBd1fg57dnI02csD3HUo
mpl0vj33YadWvqNmMGQwEgYDVR0TAQH/BAgwBgEB/wIBADAOBgNVHQ8BAf8EBAMC
AQYwHQYDVR0OBBYEFOHnbmd3hR3XdBa93TXsPBN8RyncMB8GA1UdIwQYMBaAFGK+
uWcckcNVyG8G+mwIgBRR4aDqMAoGCCqGSM49BAMCA0gAMEUCIG/6c8BCbJyrLeYg
duviS1554ruY8hSrbujwQ3xVfedDAiEAx1Pa593zsCfC8T4jzjo6udQ0/H6hW7nW
d8NsnItVFUI=
-----END CERTIFICATE-----

-----BEGIN EC PRIVATE KEY-----
MHcCAQEEILp2etVZ4xX7pwGY3Y/JdiCDBZT6Xrta8JD1jv8bnWCloAoGCCqGSM49
AwEHoUQDQgAE+p/SjLdqd+8OOTDVnkEr0Y6yDv/VGX70cTk3k5A3ikgEMkgYnMSp
dBd1fg57dnI02csD3HUompl0vj33YadWvg==
-----END EC PRIVATE KEY-----
*/

extern const uint8_t sTestCert_ICA02_Chip[] = {
    0x15, 0x30, 0x01, 0x08, 0x1c, 0xc5, 0x8f, 0xbf, 0xee, 0x96, 0x45, 0x4a, 0x24, 0x02, 0x01, 0x37, 0x03, 0x27, 0x14, 0x02,
    0x00, 0x00, 0x00, 0xca, 0xca, 0xca, 0xca, 0x27, 0x15, 0x1d, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0, 0xfa, 0x18, 0x26, 0x04,
    0xef, 0x17, 0x1b, 0x27, 0x26, 0x05, 0x6e, 0xb5, 0xb9, 0x4c, 0x37, 0x06, 0x27, 0x13, 0x04, 0x00, 0x00, 0x00, 0xca, 0xca,
    0xca, 0xca, 0x27, 0x15, 0x1d, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0, 0xfa, 0x18, 0x24, 0x07, 0x01, 0x24, 0x08, 0x01, 0x30,
    0x09, 0x41, 0x04, 0xfa, 0x9f, 0xd2, 0x8c, 0xb7, 0x6a, 0x77, 0xef, 0x0e, 0x39, 0x30, 0xd5, 0x9e, 0x41, 0x2b, 0xd1, 0x8e,
    0xb2, 0x0e, 0xff, 0xd5, 0x19, 0x7e, 0xf4, 0x71, 0x39, 0x37, 0x93, 0x90, 0x37, 0x8a, 0x48, 0x04, 0x32, 0x48, 0x18, 0x9c,
    0xc4, 0xa9, 0x74, 0x17, 0x75, 0x7e, 0x0e, 0x7b, 0x76, 0x72, 0x34, 0xd9, 0xcb, 0x03, 0xdc, 0x75, 0x28, 0x9a, 0x99, 0x74,
    0xbe, 0x3d, 0xf7, 0x61, 0xa7, 0x56, 0xbe, 0x37, 0x0a, 0x35, 0x01, 0x29, 0x01, 0x24, 0x02, 0x00, 0x18, 0x24, 0x02, 0x60,
    0x30, 0x04, 0x14, 0xe1, 0xe7, 0x6e, 0x67, 0x77, 0x85, 0x1d, 0xd7, 0x74, 0x16, 0xbd, 0xdd, 0x35, 0xec, 0x3c, 0x13, 0x7c,
    0x47, 0x29, 0xdc, 0x30, 0x05, 0x14, 0x62, 0xbe, 0xb9, 0x67, 0x1c, 0x91, 0xc3, 0x55, 0xc8, 0x6f, 0x06, 0xfa, 0x6c, 0x08,
    0x80, 0x14, 0x51, 0xe1, 0xa0, 0xea, 0x18, 0x30, 0x0b, 0x40, 0x6f, 0xfa, 0x73, 0xc0, 0x42, 0x6c, 0x9c, 0xab, 0x2d, 0xe6,
    0x20, 0x76, 0xeb, 0xe2, 0x4b, 0x5e, 0x79, 0xe2, 0xbb, 0x98, 0xf2, 0x14, 0xab, 0x6e, 0xe8, 0xf0, 0x43, 0x7c, 0x55, 0x7d,
    0xe7, 0x43, 0xc7, 0x53, 0xda, 0xe7, 0xdd, 0xf3, 0xb0, 0x27, 0xc2, 0xf1, 0x3e, 0x23, 0xce, 0x3a, 0x3a, 0xb9, 0xd4, 0x34,
    0xfc, 0x7e, 0xa1, 0x5b, 0xb9, 0xd6, 0x77, 0xc3, 0x6c, 0x9c, 0x8b, 0x55, 0x15, 0x42, 0x18,
};

extern const size_t sTestCert_ICA02_Chip_Len = sizeof(sTestCert_ICA02_Chip);

extern const uint8_t sTestCert_ICA02_DER[] = {
    0x30, 0x82, 0x01, 0xe4, 0x30, 0x82, 0x01, 0x8a, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x1c, 0xc5, 0x8f, 0xbf, 0xee, 0x96,
    0x45, 0x4a, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x44, 0x31, 0x20, 0x30, 0x1e, 0x06,
    0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x04, 0x0c, 0x10, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41,
    0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x32, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2,
    0x7c, 0x01, 0x05, 0x0c, 0x10, 0x46, 0x41, 0x42, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x31, 0x44,
    0x30, 0x1e, 0x17, 0x0d, 0x32, 0x30, 0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x17, 0x0d, 0x34, 0x30,
    0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32, 0x33, 0x34, 0x32, 0x5a, 0x30, 0x44, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06,
    0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x03, 0x0c, 0x10, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41, 0x30, 0x30, 0x30,
    0x30, 0x30, 0x30, 0x30, 0x34, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x05,
    0x0c, 0x10, 0x46, 0x41, 0x42, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x31, 0x44, 0x30, 0x59, 0x30,
    0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03,
    0x42, 0x00, 0x04, 0xfa, 0x9f, 0xd2, 0x8c, 0xb7, 0x6a, 0x77, 0xef, 0x0e, 0x39, 0x30, 0xd5, 0x9e, 0x41, 0x2b, 0xd1, 0x8e, 0xb2,
    0x0e, 0xff, 0xd5, 0x19, 0x7e, 0xf4, 0x71, 0x39, 0x37, 0x93, 0x90, 0x37, 0x8a, 0x48, 0x04, 0x32, 0x48, 0x18, 0x9c, 0xc4, 0xa9,
    0x74, 0x17, 0x75, 0x7e, 0x0e, 0x7b, 0x76, 0x72, 0x34, 0xd9, 0xcb, 0x03, 0xdc, 0x75, 0x28, 0x9a, 0x99, 0x74, 0xbe, 0x3d, 0xf7,
    0x61, 0xa7, 0x56, 0xbe, 0xa3, 0x66, 0x30, 0x64, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x08, 0x30,
    0x06, 0x01, 0x01, 0xff, 0x02, 0x01, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02,
    0x01, 0x06, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xe1, 0xe7, 0x6e, 0x67, 0x77, 0x85, 0x1d, 0xd7,
    0x74, 0x16, 0xbd, 0xdd, 0x35, 0xec, 0x3c, 0x13, 0x7c, 0x47, 0x29, 0xdc, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18,
    0x30, 0x16, 0x80, 0x14, 0x62, 0xbe, 0xb9, 0x67, 0x1c, 0x91, 0xc3, 0x55, 0xc8, 0x6f, 0x06, 0xfa, 0x6c, 0x08, 0x80, 0x14, 0x51,
    0xe1, 0xa0, 0xea, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x48, 0x00, 0x30, 0x45, 0x02,
    0x20, 0x6f, 0xfa, 0x73, 0xc0, 0x42, 0x6c, 0x9c, 0xab, 0x2d, 0xe6, 0x20, 0x76, 0xeb, 0xe2, 0x4b, 0x5e, 0x79, 0xe2, 0xbb, 0x98,
    0xf2, 0x14, 0xab, 0x6e, 0xe8, 0xf0, 0x43, 0x7c, 0x55, 0x7d, 0xe7, 0x43, 0x02, 0x21, 0x00, 0xc7, 0x53, 0xda, 0xe7, 0xdd, 0xf3,
    0xb0, 0x27, 0xc2, 0xf1, 0x3e, 0x23, 0xce, 0x3a, 0x3a, 0xb9, 0xd4, 0x34, 0xfc, 0x7e, 0xa1, 0x5b, 0xb9, 0xd6, 0x77, 0xc3, 0x6c,
    0x9c, 0x8b, 0x55, 0x15, 0x42,
};

extern const size_t sTestCert_ICA02_DER_Len = sizeof(sTestCert_ICA02_DER);

extern const uint8_t sTestCert_ICA02_PublicKey[] = {
    0x04, 0xfa, 0x9f, 0xd2, 0x8c, 0xb7, 0x6a, 0x77, 0xef, 0x0e, 0x39, 0x30, 0xd5, 0x9e, 0x41, 0x2b, 0xd1,
    0x8e, 0xb2, 0x0e, 0xff, 0xd5, 0x19, 0x7e, 0xf4, 0x71, 0x39, 0x37, 0x93, 0x90, 0x37, 0x8a, 0x48, 0x04,
    0x32, 0x48, 0x18, 0x9c, 0xc4, 0xa9, 0x74, 0x17, 0x75, 0x7e, 0x0e, 0x7b, 0x76, 0x72, 0x34, 0xd9, 0xcb,
    0x03, 0xdc, 0x75, 0x28, 0x9a, 0x99, 0x74, 0xbe, 0x3d, 0xf7, 0x61, 0xa7, 0x56, 0xbe,
};

extern const size_t sTestCert_ICA02_PublicKey_Len = sizeof(sTestCert_ICA02_PublicKey);

extern const uint8_t sTestCert_ICA02_PrivateKey[] = {
    0xba, 0x76, 0x7a, 0xd5, 0x59, 0xe3, 0x15, 0xfb, 0xa7, 0x01, 0x98, 0xdd, 0x8f, 0xc9, 0x76, 0x20,
    0x83, 0x05, 0x94, 0xfa, 0x5e, 0xbb, 0x5a, 0xf0, 0x90, 0xf5, 0x8e, 0xff, 0x1b, 0x9d, 0x60, 0xa5,
};

extern const size_t sTestCert_ICA02_PrivateKey_Len = sizeof(sTestCert_ICA02_PrivateKey);

extern const uint8_t sTestCert_ICA02_SubjectKeyId[] = {
    0xE1, 0xE7, 0x6E, 0x67, 0x77, 0x85, 0x1D, 0xD7, 0x74, 0x16, 0xBD, 0xDD, 0x35, 0xEC, 0x3C, 0x13, 0x7C, 0x47, 0x29, 0xDC,
};

extern const size_t sTestCert_ICA02_SubjectKeyId_Len = sizeof(sTestCert_ICA02_SubjectKeyId);

extern const uint8_t sTestCert_ICA02_AuthorityKeyId[] = {
    0x62, 0xBE, 0xB9, 0x67, 0x1C, 0x91, 0xC3, 0x55, 0xC8, 0x6F, 0x06, 0xFA, 0x6C, 0x08, 0x80, 0x14, 0x51, 0xE1, 0xA0, 0xEA,
};

extern const size_t sTestCert_ICA02_AuthorityKeyId_Len = sizeof(sTestCert_ICA02_AuthorityKeyId);

/**************  Test ICA01_1 Certificate  **************
Certificate:
    Data:
        Version: 3 (0x2)
        Serial Number: 3957838079739582840 (0x36ed10c395645978)
        Signature Algorithm: ecdsa-with-SHA256
        Issuer: 1.3.6.1.4.1.37244.1.4 = CACACACA00000001
        Validity
            Not Before: Oct 15 14:23:43 2020 GMT
            Not After : Oct 15 14:23:42 2040 GMT
        Subject: 1.3.6.1.4.1.37244.1.3 = CACACACA00000005
        Subject Public Key Info:
            Public Key Algorithm: id-ecPublicKey
                Public-Key: (256 bit)
                pub:
                    04:e1:d5:3a:9c:25:7a:e3:2b:ab:05:77:89:06:03:
                    13:91:b5:8a:df:9f:8a:b2:69:38:ad:eb:f1:39:6c:
                    f1:2c:4a:41:45:ba:d8:5e:92:eb:7f:3b:37:d5:da:
                    69:d0:cf:60:4c:41:c4:96:4f:ad:2d:70:45:68:d6:
                    be:8c:af:da:ee
                ASN1 OID: prime256v1
                NIST CURVE: P-256
        X509v3 extensions:
            X509v3 Basic Constraints: critical
                CA:TRUE
            X509v3 Key Usage: critical
                Certificate Sign, CRL Sign
            X509v3 Subject Key Identifier:
                50:6F:C3:FC:B7:94:15:88:EA:73:6C:20:65:2E:5F:1B:11:70:1C:6A
            X509v3 Authority Key Identifier:
                keyid:CC:13:08:AF:82:CF:EE:50:5E:B2:3B:57:BF:E8:6A:31:16:65:53:5F

    Signature Algorithm: ecdsa-with-SHA256
         30:45:02:21:00:e9:3b:4d:4b:8e:e1:66:9f:49:85:97:04:83:
         e4:00:00:9e:d0:46:b3:4e:dc:11:8a:cd:b4:ed:ea:b6:21:71:
         be:02:20:7e:80:d5:c2:98:5d:6c:6a:b2:55:c3:c4:ad:55:4c:
         72:70:04:8c:26:d6:e6:d3:e9:c2:e7:1f:b2:0a:f2:f0:fe

-----BEGIN CERTIFICATE-----
MIIBnTCCAUOgAwIBAgIINu0Qw5VkWXgwCgYIKoZIzj0EAwIwIjEgMB4GCisGAQQB
gqJ8AQQMEENBQ0FDQUNBMDAwMDAwMDEwHhcNMjAxMDE1MTQyMzQzWhcNNDAxMDE1
MTQyMzQyWjAiMSAwHgYKKwYBBAGConwBAwwQQ0FDQUNBQ0EwMDAwMDAwNTBZMBMG
ByqGSM49AgEGCCqGSM49AwEHA0IABOHVOpwleuMrqwV3iQYDE5G1it+firJpOK3r
8Tls8SxKQUW62F6S6387N9XaadDPYExBxJZPrS1wRWjWvoyv2u6jYzBhMA8GA1Ud
EwEB/wQFMAMBAf8wDgYDVR0PAQH/BAQDAgEGMB0GA1UdDgQWBBRQb8P8t5QViOpz
bCBlLl8bEXAcajAfBgNVHSMEGDAWgBTMEwivgs/uUF6yO1e/6GoxFmVTXzAKBggq
hkjOPQQDAgNIADBFAiEA6TtNS47hZp9JhZcEg+QAAJ7QRrNO3BGKzbTt6rYhcb4C
IH6A1cKYXWxqslXDxK1VTHJwBIwm1ubT6cLnH7IK8vD+
-----END CERTIFICATE-----

-----BEGIN EC PRIVATE KEY-----
MHcCAQEEIJPbzmpr+Ns8SpbdKC9MuPBrymFvuQCe1kpJD7Blg6+NoAoGCCqGSM49
AwEHoUQDQgAE4dU6nCV64yurBXeJBgMTkbWK35+Ksmk4revxOWzxLEpBRbrYXpLr
fzs31dpp0M9gTEHElk+tLXBFaNa+jK/a7g==
-----END EC PRIVATE KEY-----
*/

extern const uint8_t sTestCert_ICA01_1_Chip[] = {
    0x15, 0x30, 0x01, 0x08, 0x36, 0xed, 0x10, 0xc3, 0x95, 0x64, 0x59, 0x78, 0x24, 0x02, 0x01, 0x37, 0x03, 0x27, 0x14, 0x01, 0x00,
    0x00, 0x00, 0xca, 0xca, 0xca, 0xca, 0x18, 0x26, 0x04, 0xef, 0x17, 0x1b, 0x27, 0x26, 0x05, 0x6e, 0xb5, 0xb9, 0x4c, 0x37, 0x06,
    0x27, 0x13, 0x05, 0x00, 0x00, 0x00, 0xca, 0xca, 0xca, 0xca, 0x18, 0x24, 0x07, 0x01, 0x24, 0x08, 0x01, 0x30, 0x09, 0x41, 0x04,
    0xe1, 0xd5, 0x3a, 0x9c, 0x25, 0x7a, 0xe3, 0x2b, 0xab, 0x05, 0x77, 0x89, 0x06, 0x03, 0x13, 0x91, 0xb5, 0x8a, 0xdf, 0x9f, 0x8a,
    0xb2, 0x69, 0x38, 0xad, 0xeb, 0xf1, 0x39, 0x6c, 0xf1, 0x2c, 0x4a, 0x41, 0x45, 0xba, 0xd8, 0x5e, 0x92, 0xeb, 0x7f, 0x3b, 0x37,
    0xd5, 0xda, 0x69, 0xd0, 0xcf, 0x60, 0x4c, 0x41, 0xc4, 0x96, 0x4f, 0xad, 0x2d, 0x70, 0x45, 0x68, 0xd6, 0xbe, 0x8c, 0xaf, 0xda,
    0xee, 0x37, 0x0a, 0x35, 0x01, 0x29, 0x01, 0x18, 0x24, 0x02, 0x60, 0x30, 0x04, 0x14, 0x50, 0x6f, 0xc3, 0xfc, 0xb7, 0x94, 0x15,
    0x88, 0xea, 0x73, 0x6c, 0x20, 0x65, 0x2e, 0x5f, 0x1b, 0x11, 0x70, 0x1c, 0x6a, 0x30, 0x05, 0x14, 0xcc, 0x13, 0x08, 0xaf, 0x82,
    0xcf, 0xee, 0x50, 0x5e, 0xb2, 0x3b, 0x57, 0xbf, 0xe8, 0x6a, 0x31, 0x16, 0x65, 0x53, 0x5f, 0x18, 0x30, 0x0b, 0x40, 0xe9, 0x3b,
    0x4d, 0x4b, 0x8e, 0xe1, 0x66, 0x9f, 0x49, 0x85, 0x97, 0x04, 0x83, 0xe4, 0x00, 0x00, 0x9e, 0xd0, 0x46, 0xb3, 0x4e, 0xdc, 0x11,
    0x8a, 0xcd, 0xb4, 0xed, 0xea, 0xb6, 0x21, 0x71, 0xbe, 0x7e, 0x80, 0xd5, 0xc2, 0x98, 0x5d, 0x6c, 0x6a, 0xb2, 0x55, 0xc3, 0xc4,
    0xad, 0x55, 0x4c, 0x72, 0x70, 0x04, 0x8c, 0x26, 0xd6, 0xe6, 0xd3, 0xe9, 0xc2, 0xe7, 0x1f, 0xb2, 0x0a, 0xf2, 0xf0, 0xfe, 0x18,
};

extern const size_t sTestCert_ICA01_1_Chip_Len = sizeof(sTestCert_ICA01_1_Chip);

extern const uint8_t sTestCert_ICA01_1_DER[] = {
    0x30, 0x82, 0x01, 0x9d, 0x30, 0x82, 0x01, 0x43, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x36, 0xed, 0x10, 0xc3, 0x95, 0x64,
    0x59, 0x78, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x22, 0x31, 0x20, 0x30, 0x1e, 0x06,
    0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x04, 0x0c, 0x10, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41,
    0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x31, 0x30, 0x1e, 0x17, 0x0d, 0x32, 0x30, 0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32,
    0x33, 0x34, 0x33, 0x5a, 0x17, 0x0d, 0x34, 0x30, 0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32, 0x33, 0x34, 0x32, 0x5a, 0x30, 0x22,
    0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x03, 0x0c, 0x10, 0x43, 0x41, 0x43,
    0x41, 0x43, 0x41, 0x43, 0x41, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x35, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86,
    0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xe1, 0xd5,
    0x3a, 0x9c, 0x25, 0x7a, 0xe3, 0x2b, 0xab, 0x05, 0x77, 0x89, 0x06, 0x03, 0x13, 0x91, 0xb5, 0x8a, 0xdf, 0x9f, 0x8a, 0xb2, 0x69,
    0x38, 0xad, 0xeb, 0xf1, 0x39, 0x6c, 0xf1, 0x2c, 0x4a, 0x41, 0x45, 0xba, 0xd8, 0x5e, 0x92, 0xeb, 0x7f, 0x3b, 0x37, 0xd5, 0xda,
    0x69, 0xd0, 0xcf, 0x60, 0x4c, 0x41, 0xc4, 0x96, 0x4f, 0xad, 0x2d, 0x70, 0x45, 0x68, 0xd6, 0xbe, 0x8c, 0xaf, 0xda, 0xee, 0xa3,
    0x63, 0x30, 0x61, 0x30, 0x0f, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xff, 0x30,
    0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x01, 0x06, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d,
    0x0e, 0x04, 0x16, 0x04, 0x14, 0x50, 0x6f, 0xc3, 0xfc, 0xb7, 0x94, 0x15, 0x88, 0xea, 0x73, 0x6c, 0x20, 0x65, 0x2e, 0x5f, 0x1b,
    0x11, 0x70, 0x1c, 0x6a, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xcc, 0x13, 0x08, 0xaf,
    0x82, 0xcf, 0xee, 0x50, 0x5e, 0xb2, 0x3b, 0x57, 0xbf, 0xe8, 0x6a, 0x31, 0x16, 0x65, 0x53, 0x5f, 0x30, 0x0a, 0x06, 0x08, 0x2a,
    0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x48, 0x00, 0x30, 0x45, 0x02, 0x21, 0x00, 0xe9, 0x3b, 0x4d, 0x4b, 0x8e, 0xe1,
    0x66, 0x9f, 0x49, 0x85, 0x97, 0x04, 0x83, 0xe4, 0x00, 0x00, 0x9e, 0xd0, 0x46, 0xb3, 0x4e, 0xdc, 0x11, 0x8a, 0xcd, 0xb4, 0xed,
    0xea, 0xb6, 0x21, 0x71, 0xbe, 0x02, 0x20, 0x7e, 0x80, 0xd5, 0xc2, 0x98, 0x5d, 0x6c, 0x6a, 0xb2, 0x55, 0xc3, 0xc4, 0xad, 0x55,
    0x4c, 0x72, 0x70, 0x04, 0x8c, 0x26, 0xd6, 0xe6, 0xd3, 0xe9, 0xc2, 0xe7, 0x1f, 0xb2, 0x0a, 0xf2, 0xf0, 0xfe,
};

extern const size_t sTestCert_ICA01_1_DER_Len = sizeof(sTestCert_ICA01_1_DER);

extern const uint8_t sTestCert_ICA01_1_PublicKey[] = {
    0x04, 0xe1, 0xd5, 0x3a, 0x9c, 0x25, 0x7a, 0xe3, 0x2b, 0xab, 0x05, 0x77, 0x89, 0x06, 0x03, 0x13, 0x91,
    0xb5, 0x8a, 0xdf, 0x9f, 0x8a, 0xb2, 0x69, 0x38, 0xad, 0xeb, 0xf1, 0x39, 0x6c, 0xf1, 0x2c, 0x4a, 0x41,
    0x45, 0xba, 0xd8, 0x5e, 0x92, 0xeb, 0x7f, 0x3b, 0x37, 0xd5, 0xda, 0x69, 0xd0, 0xcf, 0x60, 0x4c, 0x41,
    0xc4, 0x96, 0x4f, 0xad, 0x2d, 0x70, 0x45, 0x68, 0xd6, 0xbe, 0x8c, 0xaf, 0xda, 0xee,
};

extern const size_t sTestCert_ICA01_1_PublicKey_Len = sizeof(sTestCert_ICA01_1_PublicKey);

extern const uint8_t sTestCert_ICA01_1_PrivateKey[] = {
    0x93, 0xdb, 0xce, 0x6a, 0x6b, 0xf8, 0xdb, 0x3c, 0x4a, 0x96, 0xdd, 0x28, 0x2f, 0x4c, 0xb8, 0xf0,
    0x6b, 0xca, 0x61, 0x6f, 0xb9, 0x00, 0x9e, 0xd6, 0x4a, 0x49, 0x0f, 0xb0, 0x65, 0x83, 0xaf, 0x8d,
};

extern const size_t sTestCert_ICA01_1_PrivateKey_Len = sizeof(sTestCert_ICA01_1_PrivateKey);

extern const uint8_t sTestCert_ICA01_1_SubjectKeyId[] = {
    0x50, 0x6F, 0xC3, 0xFC, 0xB7, 0x94, 0x15, 0x88, 0xEA, 0x73, 0x6C, 0x20, 0x65, 0x2E, 0x5F, 0x1B, 0x11, 0x70, 0x1C, 0x6A,
};

extern const size_t sTestCert_ICA01_1_SubjectKeyId_Len = sizeof(sTestCert_ICA01_1_SubjectKeyId);

extern const uint8_t sTestCert_ICA01_1_AuthorityKeyId[] = {
    0xCC, 0x13, 0x08, 0xAF, 0x82, 0xCF, 0xEE, 0x50, 0x5E, 0xB2, 0x3B, 0x57, 0xBF, 0xE8, 0x6A, 0x31, 0x16, 0x65, 0x53, 0x5F,
};

extern const size_t sTestCert_ICA01_1_AuthorityKeyId_Len = sizeof(sTestCert_ICA01_1_AuthorityKeyId);

/**************  Test FWSign01 Certificate  **************
Certificate:
    Data:
        Version: 3 (0x2)
        Serial Number: 8818095388640854520 (0x7a602b092372c1f8)
        Signature Algorithm: ecdsa-with-SHA256
        Issuer: 1.3.6.1.4.1.37244.1.3 = CACACACA00000005
        Validity
            Not Before: Oct 15 14:23:43 2020 GMT
            Not After : Oct 15 14:23:42 2040 GMT
        Subject: 1.3.6.1.4.1.37244.1.2 = FFFFFFFF00000001, CN = FW SIGN CERT 01
        Subject Public Key Info:
            Public Key Algorithm: id-ecPublicKey
                Public-Key: (256 bit)
                pub:
                    04:fd:2a:e4:0a:74:7c:6d:5a:c6:c8:dc:79:61:2b:
                    9a:85:3c:3e:0f:b4:1c:94:ff:08:5b:4e:1d:02:24:
                    85:98:65:ea:6c:7d:67:bb:88:84:34:a6:0d:42:0f:
                    d2:b9:9e:94:2b:6a:f8:a5:0f:12:76:53:5c:4f:f0:
                    32:61:79:dd:78
                ASN1 OID: prime256v1
                NIST CURVE: P-256
        X509v3 extensions:
            X509v3 Basic Constraints: critical
                CA:FALSE
            X509v3 Key Usage: critical
                Digital Signature
            X509v3 Extended Key Usage: critical
                Code Signing
            X509v3 Subject Key Identifier:
                52:25:D5:83:FB:71:8A:7C:64:51:FD:AF:92:2A:CD:2D:36:10:05:C7
            X509v3 Authority Key Identifier:
                keyid:50:6F:C3:FC:B7:94:15:88:EA:73:6C:20:65:2E:5F:1B:11:70:1C:6A

    Signature Algorithm: ecdsa-with-SHA256
         30:46:02:21:00:ed:3e:a4:16:ed:19:69:0d:e5:2b:38:66:6d:
         5a:7b:e6:e6:62:09:af:29:4e:a4:63:b4:95:1c:ad:d7:c4:6f:
         5a:02:21:00:a7:4a:4d:56:0f:89:7d:53:65:a8:9f:de:38:f3:
         f9:11:72:48:39:1c:f5:bc:76:16:f6:99:08:59:33:d5:63:c9

-----BEGIN CERTIFICATE-----
MIIBzTCCAXKgAwIBAgIIemArCSNywfgwCgYIKoZIzj0EAwIwIjEgMB4GCisGAQQB
gqJ8AQMMEENBQ0FDQUNBMDAwMDAwMDUwHhcNMjAxMDE1MTQyMzQzWhcNNDAxMDE1
MTQyMzQyWjA8MSAwHgYKKwYBBAGConwBAgwQRkZGRkZGRkYwMDAwMDAwMTEYMBYG
A1UEAwwPRlcgU0lHTiBDRVJUIDAxMFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAE
/SrkCnR8bVrGyNx5YSuahTw+D7QclP8IW04dAiSFmGXqbH1nu4iENKYNQg/SuZ6U
K2r4pQ8SdlNcT/AyYXndeKN4MHYwDAYDVR0TAQH/BAIwADAOBgNVHQ8BAf8EBAMC
B4AwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwMwHQYDVR0OBBYEFFIl1YP7cYp8ZFH9
r5IqzS02EAXHMB8GA1UdIwQYMBaAFFBvw/y3lBWI6nNsIGUuXxsRcBxqMAoGCCqG
SM49BAMCA0kAMEYCIQDtPqQW7RlpDeUrOGZtWnvm5mIJrylOpGO0lRyt18RvWgIh
AKdKTVYPiX1TZaif3jjz+RFySDkc9bx2FvaZCFkz1WPJ
-----END CERTIFICATE-----

-----BEGIN EC PRIVATE KEY-----
MHcCAQEEIGHzoGacIS0jn3F7zq4ipXOVSm+SFTejhUInbbswiS3goAoGCCqGSM49
AwEHoUQDQgAE/SrkCnR8bVrGyNx5YSuahTw+D7QclP8IW04dAiSFmGXqbH1nu4iE
NKYNQg/SuZ6UK2r4pQ8SdlNcT/AyYXndeA==
-----END EC PRIVATE KEY-----
*/

extern const uint8_t sTestCert_FWSign01_Chip[] = {
    0x15, 0x30, 0x01, 0x08, 0x7a, 0x60, 0x2b, 0x09, 0x23, 0x72, 0xc1, 0xf8, 0x24, 0x02, 0x01, 0x37, 0x03, 0x27, 0x13, 0x05,
    0x00, 0x00, 0x00, 0xca, 0xca, 0xca, 0xca, 0x18, 0x26, 0x04, 0xef, 0x17, 0x1b, 0x27, 0x26, 0x05, 0x6e, 0xb5, 0xb9, 0x4c,
    0x37, 0x06, 0x27, 0x12, 0x01, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x2c, 0x01, 0x0f, 0x46, 0x57, 0x20, 0x53, 0x49,
    0x47, 0x4e, 0x20, 0x43, 0x45, 0x52, 0x54, 0x20, 0x30, 0x31, 0x18, 0x24, 0x07, 0x01, 0x24, 0x08, 0x01, 0x30, 0x09, 0x41,
    0x04, 0xfd, 0x2a, 0xe4, 0x0a, 0x74, 0x7c, 0x6d, 0x5a, 0xc6, 0xc8, 0xdc, 0x79, 0x61, 0x2b, 0x9a, 0x85, 0x3c, 0x3e, 0x0f,
    0xb4, 0x1c, 0x94, 0xff, 0x08, 0x5b, 0x4e, 0x1d, 0x02, 0x24, 0x85, 0x98, 0x65, 0xea, 0x6c, 0x7d, 0x67, 0xbb, 0x88, 0x84,
    0x34, 0xa6, 0x0d, 0x42, 0x0f, 0xd2, 0xb9, 0x9e, 0x94, 0x2b, 0x6a, 0xf8, 0xa5, 0x0f, 0x12, 0x76, 0x53, 0x5c, 0x4f, 0xf0,
    0x32, 0x61, 0x79, 0xdd, 0x78, 0x37, 0x0a, 0x35, 0x01, 0x28, 0x01, 0x18, 0x24, 0x02, 0x01, 0x36, 0x03, 0x04, 0x03, 0x18,
    0x30, 0x04, 0x14, 0x52, 0x25, 0xd5, 0x83, 0xfb, 0x71, 0x8a, 0x7c, 0x64, 0x51, 0xfd, 0xaf, 0x92, 0x2a, 0xcd, 0x2d, 0x36,
    0x10, 0x05, 0xc7, 0x30, 0x05, 0x14, 0x50, 0x6f, 0xc3, 0xfc, 0xb7, 0x94, 0x15, 0x88, 0xea, 0x73, 0x6c, 0x20, 0x65, 0x2e,
    0x5f, 0x1b, 0x11, 0x70, 0x1c, 0x6a, 0x18, 0x30, 0x0b, 0x40, 0xed, 0x3e, 0xa4, 0x16, 0xed, 0x19, 0x69, 0x0d, 0xe5, 0x2b,
    0x38, 0x66, 0x6d, 0x5a, 0x7b, 0xe6, 0xe6, 0x62, 0x09, 0xaf, 0x29, 0x4e, 0xa4, 0x63, 0xb4, 0x95, 0x1c, 0xad, 0xd7, 0xc4,
    0x6f, 0x5a, 0xa7, 0x4a, 0x4d, 0x56, 0x0f, 0x89, 0x7d, 0x53, 0x65, 0xa8, 0x9f, 0xde, 0x38, 0xf3, 0xf9, 0x11, 0x72, 0x48,
    0x39, 0x1c, 0xf5, 0xbc, 0x76, 0x16, 0xf6, 0x99, 0x08, 0x59, 0x33, 0xd5, 0x63, 0xc9, 0x18,
};

extern const size_t sTestCert_FWSign01_Chip_Len = sizeof(sTestCert_FWSign01_Chip);

extern const uint8_t sTestCert_FWSign01_DER[] = {
    0x30, 0x82, 0x01, 0xcd, 0x30, 0x82, 0x01, 0x72, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x7a, 0x60, 0x2b, 0x09, 0x23, 0x72,
    0xc1, 0xf8, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x22, 0x31, 0x20, 0x30, 0x1e, 0x06,
    0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x03, 0x0c, 0x10, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41,
    0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x35, 0x30, 0x1e, 0x17, 0x0d, 0x32, 0x30, 0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32,
    0x33, 0x34, 0x33, 0x5a, 0x17, 0x0d, 0x34, 0x30, 0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32, 0x33, 0x34, 0x32, 0x5a, 0x30, 0x3c,
    0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x02, 0x0c, 0x10, 0x46, 0x46, 0x46,
    0x46, 0x46, 0x46, 0x46, 0x46, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x31, 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04,
    0x03, 0x0c, 0x0f, 0x46, 0x57, 0x20, 0x53, 0x49, 0x47, 0x4e, 0x20, 0x43, 0x45, 0x52, 0x54, 0x20, 0x30, 0x31, 0x30, 0x59, 0x30,
    0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03,
    0x42, 0x00, 0x04, 0xfd, 0x2a, 0xe4, 0x0a, 0x74, 0x7c, 0x6d, 0x5a, 0xc6, 0xc8, 0xdc, 0x79, 0x61, 0x2b, 0x9a, 0x85, 0x3c, 0x3e,
    0x0f, 0xb4, 0x1c, 0x94, 0xff, 0x08, 0x5b, 0x4e, 0x1d, 0x02, 0x24, 0x85, 0x98, 0x65, 0xea, 0x6c, 0x7d, 0x67, 0xbb, 0x88, 0x84,
    0x34, 0xa6, 0x0d, 0x42, 0x0f, 0xd2, 0xb9, 0x9e, 0x94, 0x2b, 0x6a, 0xf8, 0xa5, 0x0f, 0x12, 0x76, 0x53, 0x5c, 0x4f, 0xf0, 0x32,
    0x61, 0x79, 0xdd, 0x78, 0xa3, 0x78, 0x30, 0x76, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30,
    0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x16, 0x06, 0x03,
    0x55, 0x1d, 0x25, 0x01, 0x01, 0xff, 0x04, 0x0c, 0x30, 0x0a, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x03, 0x30,
    0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x52, 0x25, 0xd5, 0x83, 0xfb, 0x71, 0x8a, 0x7c, 0x64, 0x51, 0xfd,
    0xaf, 0x92, 0x2a, 0xcd, 0x2d, 0x36, 0x10, 0x05, 0xc7, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80,
    0x14, 0x50, 0x6f, 0xc3, 0xfc, 0xb7, 0x94, 0x15, 0x88, 0xea, 0x73, 0x6c, 0x20, 0x65, 0x2e, 0x5f, 0x1b, 0x11, 0x70, 0x1c, 0x6a,
    0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x49, 0x00, 0x30, 0x46, 0x02, 0x21, 0x00, 0xed,
    0x3e, 0xa4, 0x16, 0xed, 0x19, 0x69, 0x0d, 0xe5, 0x2b, 0x38, 0x66, 0x6d, 0x5a, 0x7b, 0xe6, 0xe6, 0x62, 0x09, 0xaf, 0x29, 0x4e,
    0xa4, 0x63, 0xb4, 0x95, 0x1c, 0xad, 0xd7, 0xc4, 0x6f, 0x5a, 0x02, 0x21, 0x00, 0xa7, 0x4a, 0x4d, 0x56, 0x0f, 0x89, 0x7d, 0x53,
    0x65, 0xa8, 0x9f, 0xde, 0x38, 0xf3, 0xf9, 0x11, 0x72, 0x48, 0x39, 0x1c, 0xf5, 0xbc, 0x76, 0x16, 0xf6, 0x99, 0x08, 0x59, 0x33,
    0xd5, 0x63, 0xc9,
};

extern const size_t sTestCert_FWSign01_DER_Len = sizeof(sTestCert_FWSign01_DER);

extern const uint8_t sTestCert_FWSign01_PublicKey[] = {
    0x04, 0xfd, 0x2a, 0xe4, 0x0a, 0x74, 0x7c, 0x6d, 0x5a, 0xc6, 0xc8, 0xdc, 0x79, 0x61, 0x2b, 0x9a, 0x85,
    0x3c, 0x3e, 0x0f, 0xb4, 0x1c, 0x94, 0xff, 0x08, 0x5b, 0x4e, 0x1d, 0x02, 0x24, 0x85, 0x98, 0x65, 0xea,
    0x6c, 0x7d, 0x67, 0xbb, 0x88, 0x84, 0x34, 0xa6, 0x0d, 0x42, 0x0f, 0xd2, 0xb9, 0x9e, 0x94, 0x2b, 0x6a,
    0xf8, 0xa5, 0x0f, 0x12, 0x76, 0x53, 0x5c, 0x4f, 0xf0, 0x32, 0x61, 0x79, 0xdd, 0x78,
};

extern const size_t sTestCert_FWSign01_PublicKey_Len = sizeof(sTestCert_FWSign01_PublicKey);

extern const uint8_t sTestCert_FWSign01_PrivateKey[] = {
    0x61, 0xf3, 0xa0, 0x66, 0x9c, 0x21, 0x2d, 0x23, 0x9f, 0x71, 0x7b, 0xce, 0xae, 0x22, 0xa5, 0x73,
    0x95, 0x4a, 0x6f, 0x92, 0x15, 0x37, 0xa3, 0x85, 0x42, 0x27, 0x6d, 0xbb, 0x30, 0x89, 0x2d, 0xe0,
};

extern const size_t sTestCert_FWSign01_PrivateKey_Len = sizeof(sTestCert_FWSign01_PrivateKey);

extern const uint8_t sTestCert_FWSign01_SubjectKeyId[] = {
    0x52, 0x25, 0xD5, 0x83, 0xFB, 0x71, 0x8A, 0x7C, 0x64, 0x51, 0xFD, 0xAF, 0x92, 0x2A, 0xCD, 0x2D, 0x36, 0x10, 0x05, 0xC7,
};

extern const size_t sTestCert_FWSign01_SubjectKeyId_Len = sizeof(sTestCert_FWSign01_SubjectKeyId);

extern const uint8_t sTestCert_FWSign01_AuthorityKeyId[] = {
    0x50, 0x6F, 0xC3, 0xFC, 0xB7, 0x94, 0x15, 0x88, 0xEA, 0x73, 0x6C, 0x20, 0x65, 0x2E, 0x5F, 0x1B, 0x11, 0x70, 0x1C, 0x6A,
};

extern const size_t sTestCert_FWSign01_AuthorityKeyId_Len = sizeof(sTestCert_FWSign01_AuthorityKeyId);

/**************  Test Node01_01 Certificate  **************
Certificate:
    Data:
        Version: 3 (0x2)
        Serial Number: 1795082174304132643 (0x18e969ba0e089e23)
        Signature Algorithm: ecdsa-with-SHA256
        Issuer: 1.3.6.1.4.1.37244.1.3 = CACACACA00000003
        Validity
            Not Before: Oct 15 14:23:43 2020 GMT
            Not After : Oct 15 14:23:42 2040 GMT
        Subject: 1.3.6.1.4.1.37244.1.1 = DEDEDEDE00010001, 1.3.6.1.4.1.37244.1.5 = FAB000000000001D
        Subject Public Key Info:
            Public Key Algorithm: id-ecPublicKey
                Public-Key: (256 bit)
                pub:
                    04:bc:f6:58:0d:2d:71:e1:44:16:65:1f:7c:31:1b:
                    5e:fc:f9:ae:c0:a8:c1:0a:f8:09:27:84:4c:24:0f:
                    51:a8:eb:23:fa:07:44:13:88:87:ac:1e:73:cb:72:
                    a0:54:b6:a0:db:06:22:aa:80:70:71:01:63:13:b1:
                    59:6c:85:52:cf
                ASN1 OID: prime256v1
                NIST CURVE: P-256
        X509v3 extensions:
            X509v3 Basic Constraints: critical
                CA:FALSE
            X509v3 Key Usage: critical
                Digital Signature
            X509v3 Extended Key Usage: critical
                TLS Web Client Authentication, TLS Web Server Authentication
            X509v3 Subject Key Identifier:
                69:67:C9:12:F8:A3:E6:89:55:6F:89:9B:65:D7:6F:53:FA:65:C7:B6
            X509v3 Authority Key Identifier:
                keyid:44:0C:C6:92:31:C4:CB:5B:37:94:24:26:F8:1B:BE:24:B7:EF:34:5C

    Signature Algorithm: ecdsa-with-SHA256
         30:46:02:21:00:ce:6e:f3:93:cb:bc:94:f8:0e:e2:90:cb:3c:
         3d:37:33:35:ba:b9:59:07:73:4d:99:d3:84:a6:2a:37:3b:84:
         84:02:21:00:e1:d4:1a:04:c3:14:0f:aa:19:e8:a2:b9:9b:0c:
         61:e3:3c:27:ea:91:39:73:e4:5b:5b:c6:e3:9c:27:0d:ac:53

-----BEGIN CERTIFICATE-----
MIIB4TCCAYagAwIBAgIIGOlpug4IniMwCgYIKoZIzj0EAwIwIjEgMB4GCisGAQQB
gqJ8AQMMEENBQ0FDQUNBMDAwMDAwMDMwHhcNMjAxMDE1MTQyMzQzWhcNNDAxMDE1
MTQyMzQyWjBEMSAwHgYKKwYBBAGConwBAQwQREVERURFREUwMDAxMDAwMTEgMB4G
CisGAQQBgqJ8AQUMEEZBQjAwMDAwMDAwMDAwMUQwWTATBgcqhkjOPQIBBggqhkjO
PQMBBwNCAAS89lgNLXHhRBZlH3wxG178+a7AqMEK+AknhEwkD1Go6yP6B0QTiIes
HnPLcqBUtqDbBiKqgHBxAWMTsVlshVLPo4GDMIGAMAwGA1UdEwEB/wQCMAAwDgYD
VR0PAQH/BAQDAgeAMCAGA1UdJQEB/wQWMBQGCCsGAQUFBwMCBggrBgEFBQcDATAd
BgNVHQ4EFgQUaWfJEvij5olVb4mbZddvU/plx7YwHwYDVR0jBBgwFoAURAzGkjHE
y1s3lCQm+Bu+JLfvNFwwCgYIKoZIzj0EAwIDSQAwRgIhAM5u85PLvJT4DuKQyzw9
NzM1urlZB3NNmdOEpio3O4SEAiEA4dQaBMMUD6oZ6KK5mwxh4zwn6pE5c+RbW8bj
nCcNrFM=
-----END CERTIFICATE-----

-----BEGIN EC PRIVATE KEY-----
MHcCAQEEIEkuf5eD7hX9qnGopnsfMfPI1Xt1CUpGE7m7PvRvpg2JoAoGCCqGSM49
AwEHoUQDQgAEvPZYDS1x4UQWZR98MRte/PmuwKjBCvgJJ4RMJA9RqOsj+gdEE4iH
rB5zy3KgVLag2wYiqoBwcQFjE7FZbIVSzw==
-----END EC PRIVATE KEY-----
*/

extern const uint8_t sTestCert_Node01_01_Chip[] = {
    0x15, 0x30, 0x01, 0x08, 0x18, 0xe9, 0x69, 0xba, 0x0e, 0x08, 0x9e, 0x23, 0x24, 0x02, 0x01, 0x37, 0x03, 0x27, 0x13, 0x03, 0x00,
    0x00, 0x00, 0xca, 0xca, 0xca, 0xca, 0x18, 0x26, 0x04, 0xef, 0x17, 0x1b, 0x27, 0x26, 0x05, 0x6e, 0xb5, 0xb9, 0x4c, 0x37, 0x06,
    0x27, 0x11, 0x01, 0x00, 0x01, 0x00, 0xde, 0xde, 0xde, 0xde, 0x27, 0x15, 0x1d, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0, 0xfa, 0x18,
    0x24, 0x07, 0x01, 0x24, 0x08, 0x01, 0x30, 0x09, 0x41, 0x04, 0xbc, 0xf6, 0x58, 0x0d, 0x2d, 0x71, 0xe1, 0x44, 0x16, 0x65, 0x1f,
    0x7c, 0x31, 0x1b, 0x5e, 0xfc, 0xf9, 0xae, 0xc0, 0xa8, 0xc1, 0x0a, 0xf8, 0x09, 0x27, 0x84, 0x4c, 0x24, 0x0f, 0x51, 0xa8, 0xeb,
    0x23, 0xfa, 0x07, 0x44, 0x13, 0x88, 0x87, 0xac, 0x1e, 0x73, 0xcb, 0x72, 0xa0, 0x54, 0xb6, 0xa0, 0xdb, 0x06, 0x22, 0xaa, 0x80,
    0x70, 0x71, 0x01, 0x63, 0x13, 0xb1, 0x59, 0x6c, 0x85, 0x52, 0xcf, 0x37, 0x0a, 0x35, 0x01, 0x28, 0x01, 0x18, 0x24, 0x02, 0x01,
    0x36, 0x03, 0x04, 0x02, 0x04, 0x01, 0x18, 0x30, 0x04, 0x14, 0x69, 0x67, 0xc9, 0x12, 0xf8, 0xa3, 0xe6, 0x89, 0x55, 0x6f, 0x89,
    0x9b, 0x65, 0xd7, 0x6f, 0x53, 0xfa, 0x65, 0xc7, 0xb6, 0x30, 0x05, 0x14, 0x44, 0x0c, 0xc6, 0x92, 0x31, 0xc4, 0xcb, 0x5b, 0x37,
    0x94, 0x24, 0x26, 0xf8, 0x1b, 0xbe, 0x24, 0xb7, 0xef, 0x34, 0x5c, 0x18, 0x30, 0x0b, 0x40, 0xce, 0x6e, 0xf3, 0x93, 0xcb, 0xbc,
    0x94, 0xf8, 0x0e, 0xe2, 0x90, 0xcb, 0x3c, 0x3d, 0x37, 0x33, 0x35, 0xba, 0xb9, 0x59, 0x07, 0x73, 0x4d, 0x99, 0xd3, 0x84, 0xa6,
    0x2a, 0x37, 0x3b, 0x84, 0x84, 0xe1, 0xd4, 0x1a, 0x04, 0xc3, 0x14, 0x0f, 0xaa, 0x19, 0xe8, 0xa2, 0xb9, 0x9b, 0x0c, 0x61, 0xe3,
    0x3c, 0x27, 0xea, 0x91, 0x39, 0x73, 0xe4, 0x5b, 0x5b, 0xc6, 0xe3, 0x9c, 0x27, 0x0d, 0xac, 0x53, 0x18,
};

extern const size_t sTestCert_Node01_01_Chip_Len = sizeof(sTestCert_Node01_01_Chip);

// Error Testing 01: Manually updated Node01_01 CHIP TLV encoded certificate.
// Updated Tag of the Subject Key Identifier Extension from ContextTag to CommonProfile_2Bytes:
//     0x30, 0x04  --> 0x50, 0xee, 0x04
// The CHIP_ERROR_INVALID_TLV_TAG error is expected when this certificate is loaded/decoded
extern const uint8_t sTestCert_Node01_01_Err01_Chip[] = {
    0x15, 0x30, 0x01, 0x08, 0x18, 0xe9, 0x69, 0xba, 0x0e, 0x08, 0x9e, 0x23, 0x24, 0x02, 0x01, 0x37, 0x03, 0x27, 0x13, 0x03, 0x00,
    0x00, 0x00, 0xca, 0xca, 0xca, 0xca, 0x18, 0x26, 0x04, 0xef, 0x17, 0x1b, 0x27, 0x26, 0x05, 0x6e, 0xb5, 0xb9, 0x4c, 0x37, 0x06,
    0x27, 0x11, 0x01, 0x00, 0x01, 0x00, 0xde, 0xde, 0xde, 0xde, 0x27, 0x15, 0x1d, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0, 0xfa, 0x18,
    0x24, 0x07, 0x01, 0x24, 0x08, 0x01, 0x30, 0x09, 0x41, 0x04, 0xbc, 0xf6, 0x58, 0x0d, 0x2d, 0x71, 0xe1, 0x44, 0x16, 0x65, 0x1f,
    0x7c, 0x31, 0x1b, 0x5e, 0xfc, 0xf9, 0xae, 0xc0, 0xa8, 0xc1, 0x0a, 0xf8, 0x09, 0x27, 0x84, 0x4c, 0x24, 0x0f, 0x51, 0xa8, 0xeb,
    0x23, 0xfa, 0x07, 0x44, 0x13, 0x88, 0x87, 0xac, 0x1e, 0x73, 0xcb, 0x72, 0xa0, 0x54, 0xb6, 0xa0, 0xdb, 0x06, 0x22, 0xaa, 0x80,
    0x70, 0x71, 0x01, 0x63, 0x13, 0xb1, 0x59, 0x6c, 0x85, 0x52, 0xcf, 0x37, 0x0a, 0x35, 0x01, 0x28, 0x01, 0x18, 0x24, 0x02, 0x01,
    0x36, 0x03, 0x04, 0x02, 0x04, 0x01, 0x18, 0x50, 0xee, 0x04, 0x14, 0x69, 0x67, 0xc9, 0x12, 0xf8, 0xa3, 0xe6, 0x89, 0x55, 0x6f,
    0x89, 0x9b, 0x65, 0xd7, 0x6f, 0x53, 0xfa, 0x65, 0xc7, 0xb6, 0x30, 0x05, 0x14, 0x44, 0x0c, 0xc6, 0x92, 0x31, 0xc4, 0xcb, 0x5b,
    0x37, 0x94, 0x24, 0x26, 0xf8, 0x1b, 0xbe, 0x24, 0xb7, 0xef, 0x34, 0x5c, 0x18, 0x30, 0x0b, 0x40, 0xce, 0x6e, 0xf3, 0x93, 0xcb,
    0xbc, 0x94, 0xf8, 0x0e, 0xe2, 0x90, 0xcb, 0x3c, 0x3d, 0x37, 0x33, 0x35, 0xba, 0xb9, 0x59, 0x07, 0x73, 0x4d, 0x99, 0xd3, 0x84,
    0xa6, 0x2a, 0x37, 0x3b, 0x84, 0x84, 0xe1, 0xd4, 0x1a, 0x04, 0xc3, 0x14, 0x0f, 0xaa, 0x19, 0xe8, 0xa2, 0xb9, 0x9b, 0x0c, 0x61,
    0xe3, 0x3c, 0x27, 0xea, 0x91, 0x39, 0x73, 0xe4, 0x5b, 0x5b, 0xc6, 0xe3, 0x9c, 0x27, 0x0d, 0xac, 0x53, 0x18,
};

extern const size_t sTestCert_Node01_01_Err01_Chip_Len = sizeof(sTestCert_Node01_01_Err01_Chip);

extern const uint8_t sTestCert_Node01_01_DER[] = {
    0x30, 0x82, 0x01, 0xe1, 0x30, 0x82, 0x01, 0x86, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x18, 0xe9, 0x69, 0xba, 0x0e, 0x08,
    0x9e, 0x23, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x22, 0x31, 0x20, 0x30, 0x1e, 0x06,
    0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x03, 0x0c, 0x10, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41,
    0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x33, 0x30, 0x1e, 0x17, 0x0d, 0x32, 0x30, 0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32,
    0x33, 0x34, 0x33, 0x5a, 0x17, 0x0d, 0x34, 0x30, 0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32, 0x33, 0x34, 0x32, 0x5a, 0x30, 0x44,
    0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x01, 0x0c, 0x10, 0x44, 0x45, 0x44,
    0x45, 0x44, 0x45, 0x44, 0x45, 0x30, 0x30, 0x30, 0x31, 0x30, 0x30, 0x30, 0x31, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06,
    0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x05, 0x0c, 0x10, 0x46, 0x41, 0x42, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
    0x30, 0x30, 0x30, 0x31, 0x44, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a,
    0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xbc, 0xf6, 0x58, 0x0d, 0x2d, 0x71, 0xe1, 0x44, 0x16, 0x65,
    0x1f, 0x7c, 0x31, 0x1b, 0x5e, 0xfc, 0xf9, 0xae, 0xc0, 0xa8, 0xc1, 0x0a, 0xf8, 0x09, 0x27, 0x84, 0x4c, 0x24, 0x0f, 0x51, 0xa8,
    0xeb, 0x23, 0xfa, 0x07, 0x44, 0x13, 0x88, 0x87, 0xac, 0x1e, 0x73, 0xcb, 0x72, 0xa0, 0x54, 0xb6, 0xa0, 0xdb, 0x06, 0x22, 0xaa,
    0x80, 0x70, 0x71, 0x01, 0x63, 0x13, 0xb1, 0x59, 0x6c, 0x85, 0x52, 0xcf, 0xa3, 0x81, 0x83, 0x30, 0x81, 0x80, 0x30, 0x0c, 0x06,
    0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff,
    0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x20, 0x06, 0x03, 0x55, 0x1d, 0x25, 0x01, 0x01, 0xff, 0x04, 0x16, 0x30, 0x14, 0x06,
    0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x02, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x01, 0x30, 0x1d,
    0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x69, 0x67, 0xc9, 0x12, 0xf8, 0xa3, 0xe6, 0x89, 0x55, 0x6f, 0x89, 0x9b,
    0x65, 0xd7, 0x6f, 0x53, 0xfa, 0x65, 0xc7, 0xb6, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14,
    0x44, 0x0c, 0xc6, 0x92, 0x31, 0xc4, 0xcb, 0x5b, 0x37, 0x94, 0x24, 0x26, 0xf8, 0x1b, 0xbe, 0x24, 0xb7, 0xef, 0x34, 0x5c, 0x30,
    0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x49, 0x00, 0x30, 0x46, 0x02, 0x21, 0x00, 0xce, 0x6e,
    0xf3, 0x93, 0xcb, 0xbc, 0x94, 0xf8, 0x0e, 0xe2, 0x90, 0xcb, 0x3c, 0x3d, 0x37, 0x33, 0x35, 0xba, 0xb9, 0x59, 0x07, 0x73, 0x4d,
    0x99, 0xd3, 0x84, 0xa6, 0x2a, 0x37, 0x3b, 0x84, 0x84, 0x02, 0x21, 0x00, 0xe1, 0xd4, 0x1a, 0x04, 0xc3, 0x14, 0x0f, 0xaa, 0x19,
    0xe8, 0xa2, 0xb9, 0x9b, 0x0c, 0x61, 0xe3, 0x3c, 0x27, 0xea, 0x91, 0x39, 0x73, 0xe4, 0x5b, 0x5b, 0xc6, 0xe3, 0x9c, 0x27, 0x0d,
    0xac, 0x53,
};

extern const size_t sTestCert_Node01_01_DER_Len = sizeof(sTestCert_Node01_01_DER);

extern const uint8_t sTestCert_Node01_01_PublicKey[] = {
    0x04, 0xbc, 0xf6, 0x58, 0x0d, 0x2d, 0x71, 0xe1, 0x44, 0x16, 0x65, 0x1f, 0x7c, 0x31, 0x1b, 0x5e, 0xfc,
    0xf9, 0xae, 0xc0, 0xa8, 0xc1, 0x0a, 0xf8, 0x09, 0x27, 0x84, 0x4c, 0x24, 0x0f, 0x51, 0xa8, 0xeb, 0x23,
    0xfa, 0x07, 0x44, 0x13, 0x88, 0x87, 0xac, 0x1e, 0x73, 0xcb, 0x72, 0xa0, 0x54, 0xb6, 0xa0, 0xdb, 0x06,
    0x22, 0xaa, 0x80, 0x70, 0x71, 0x01, 0x63, 0x13, 0xb1, 0x59, 0x6c, 0x85, 0x52, 0xcf,
};

extern const size_t sTestCert_Node01_01_PublicKey_Len = sizeof(sTestCert_Node01_01_PublicKey);

extern const uint8_t sTestCert_Node01_01_PrivateKey[] = {
    0x49, 0x2e, 0x7f, 0x97, 0x83, 0xee, 0x15, 0xfd, 0xaa, 0x71, 0xa8, 0xa6, 0x7b, 0x1f, 0x31, 0xf3,
    0xc8, 0xd5, 0x7b, 0x75, 0x09, 0x4a, 0x46, 0x13, 0xb9, 0xbb, 0x3e, 0xf4, 0x6f, 0xa6, 0x0d, 0x89,
};

extern const size_t sTestCert_Node01_01_PrivateKey_Len = sizeof(sTestCert_Node01_01_PrivateKey);

extern const uint8_t sTestCert_Node01_01_SubjectKeyId[] = {
    0x69, 0x67, 0xC9, 0x12, 0xF8, 0xA3, 0xE6, 0x89, 0x55, 0x6F, 0x89, 0x9B, 0x65, 0xD7, 0x6F, 0x53, 0xFA, 0x65, 0xC7, 0xB6,
};

extern const size_t sTestCert_Node01_01_SubjectKeyId_Len = sizeof(sTestCert_Node01_01_SubjectKeyId);

extern const uint8_t sTestCert_Node01_01_AuthorityKeyId[] = {
    0x44, 0x0C, 0xC6, 0x92, 0x31, 0xC4, 0xCB, 0x5B, 0x37, 0x94, 0x24, 0x26, 0xF8, 0x1B, 0xBE, 0x24, 0xB7, 0xEF, 0x34, 0x5C,
};

extern const size_t sTestCert_Node01_01_AuthorityKeyId_Len = sizeof(sTestCert_Node01_01_AuthorityKeyId);

/**************  Test Node01_02 Certificate  **************
Certificate:
    Data:
        Version: 3 (0x2)
        Serial Number: 977443105016929468 (0xd90935346b05cbc)
        Signature Algorithm: ecdsa-with-SHA256
        Issuer: 1.3.6.1.4.1.37244.1.4 = CACACACA00000001
        Validity
            Not Before: Oct 15 14:23:43 2020 GMT
            Not After : Oct 15 14:23:42 2040 GMT
        Subject: 1.3.6.1.4.1.37244.1.1 = DEDEDEDE00010002, 1.3.6.1.4.1.37244.1.5 = FAB000000000001D
        Subject Public Key Info:
            Public Key Algorithm: id-ecPublicKey
                Public-Key: (256 bit)
                pub:
                    04:96:5f:78:c5:37:ec:e1:b8:c3:4a:7b:98:b9:aa:
                    45:f1:35:63:a5:02:b1:97:9a:60:7b:d0:c4:19:88:
                    bd:d0:f0:bb:b8:98:16:c2:07:e3:b5:15:d9:26:41:
                    59:f7:8b:d0:97:8e:32:d7:4c:6d:05:5a:14:9e:8e:
                    9d:ba:40:19:bf
                ASN1 OID: prime256v1
                NIST CURVE: P-256
        X509v3 extensions:
            X509v3 Basic Constraints: critical
                CA:FALSE
            X509v3 Key Usage: critical
                Digital Signature
            X509v3 Extended Key Usage: critical
                TLS Web Client Authentication, TLS Web Server Authentication
            X509v3 Subject Key Identifier:
                56:7B:4F:20:E4:B9:C7:BD:27:B2:9B:3D:CE:6A:76:F7:CD:8E:CC:B6
            X509v3 Authority Key Identifier:
                keyid:CC:13:08:AF:82:CF:EE:50:5E:B2:3B:57:BF:E8:6A:31:16:65:53:5F

    Signature Algorithm: ecdsa-with-SHA256
         30:44:02:20:60:58:11:4b:a7:21:82:fc:f6:30:1f:7a:08:1b:
         ca:5a:84:82:02:43:1a:52:fd:bf:f4:97:d8:dd:6f:9a:59:59:
         02:20:7b:ad:cc:d6:a5:6d:70:ef:d8:c9:7c:49:6e:ba:7e:28:
         01:d7:33:7d:cf:f7:4d:78:e4:6e:cd:3a:08:cc:ba:e3

-----BEGIN CERTIFICATE-----
MIIB3zCCAYagAwIBAgIIDZCTU0awXLwwCgYIKoZIzj0EAwIwIjEgMB4GCisGAQQB
gqJ8AQQMEENBQ0FDQUNBMDAwMDAwMDEwHhcNMjAxMDE1MTQyMzQzWhcNNDAxMDE1
MTQyMzQyWjBEMSAwHgYKKwYBBAGConwBAQwQREVERURFREUwMDAxMDAwMjEgMB4G
CisGAQQBgqJ8AQUMEEZBQjAwMDAwMDAwMDAwMUQwWTATBgcqhkjOPQIBBggqhkjO
PQMBBwNCAASWX3jFN+zhuMNKe5i5qkXxNWOlArGXmmB70MQZiL3Q8Lu4mBbCB+O1
FdkmQVn3i9CXjjLXTG0FWhSejp26QBm/o4GDMIGAMAwGA1UdEwEB/wQCMAAwDgYD
VR0PAQH/BAQDAgeAMCAGA1UdJQEB/wQWMBQGCCsGAQUFBwMCBggrBgEFBQcDATAd
BgNVHQ4EFgQUVntPIOS5x70nsps9zmp2982OzLYwHwYDVR0jBBgwFoAUzBMIr4LP
7lBesjtXv+hqMRZlU18wCgYIKoZIzj0EAwIDRwAwRAIgYFgRS6chgvz2MB96CBvK
WoSCAkMaUv2/9JfY3W+aWVkCIHutzNalbXDv2Ml8SW66figB1zN9z/dNeORuzToI
zLrj
-----END CERTIFICATE-----

-----BEGIN EC PRIVATE KEY-----
MHcCAQEEIBiymGFRQud62w0dw8UXgNN4yj5jqR/Y3KYL53easvcNoAoGCCqGSM49
AwEHoUQDQgAEll94xTfs4bjDSnuYuapF8TVjpQKxl5pge9DEGYi90PC7uJgWwgfj
tRXZJkFZ94vQl44y10xtBVoUno6dukAZvw==
-----END EC PRIVATE KEY-----
*/

extern const uint8_t sTestCert_Node01_02_Chip[] = {
    0x15, 0x30, 0x01, 0x08, 0x0d, 0x90, 0x93, 0x53, 0x46, 0xb0, 0x5c, 0xbc, 0x24, 0x02, 0x01, 0x37, 0x03, 0x27, 0x14, 0x01, 0x00,
    0x00, 0x00, 0xca, 0xca, 0xca, 0xca, 0x18, 0x26, 0x04, 0xef, 0x17, 0x1b, 0x27, 0x26, 0x05, 0x6e, 0xb5, 0xb9, 0x4c, 0x37, 0x06,
    0x27, 0x11, 0x02, 0x00, 0x01, 0x00, 0xde, 0xde, 0xde, 0xde, 0x27, 0x15, 0x1d, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0, 0xfa, 0x18,
    0x24, 0x07, 0x01, 0x24, 0x08, 0x01, 0x30, 0x09, 0x41, 0x04, 0x96, 0x5f, 0x78, 0xc5, 0x37, 0xec, 0xe1, 0xb8, 0xc3, 0x4a, 0x7b,
    0x98, 0xb9, 0xaa, 0x45, 0xf1, 0x35, 0x63, 0xa5, 0x02, 0xb1, 0x97, 0x9a, 0x60, 0x7b, 0xd0, 0xc4, 0x19, 0x88, 0xbd, 0xd0, 0xf0,
    0xbb, 0xb8, 0x98, 0x16, 0xc2, 0x07, 0xe3, 0xb5, 0x15, 0xd9, 0x26, 0x41, 0x59, 0xf7, 0x8b, 0xd0, 0x97, 0x8e, 0x32, 0xd7, 0x4c,
    0x6d, 0x05, 0x5a, 0x14, 0x9e, 0x8e, 0x9d, 0xba, 0x40, 0x19, 0xbf, 0x37, 0x0a, 0x35, 0x01, 0x28, 0x01, 0x18, 0x24, 0x02, 0x01,
    0x36, 0x03, 0x04, 0x02, 0x04, 0x01, 0x18, 0x30, 0x04, 0x14, 0x56, 0x7b, 0x4f, 0x20, 0xe4, 0xb9, 0xc7, 0xbd, 0x27, 0xb2, 0x9b,
    0x3d, 0xce, 0x6a, 0x76, 0xf7, 0xcd, 0x8e, 0xcc, 0xb6, 0x30, 0x05, 0x14, 0xcc, 0x13, 0x08, 0xaf, 0x82, 0xcf, 0xee, 0x50, 0x5e,
    0xb2, 0x3b, 0x57, 0xbf, 0xe8, 0x6a, 0x31, 0x16, 0x65, 0x53, 0x5f, 0x18, 0x30, 0x0b, 0x40, 0x60, 0x58, 0x11, 0x4b, 0xa7, 0x21,
    0x82, 0xfc, 0xf6, 0x30, 0x1f, 0x7a, 0x08, 0x1b, 0xca, 0x5a, 0x84, 0x82, 0x02, 0x43, 0x1a, 0x52, 0xfd, 0xbf, 0xf4, 0x97, 0xd8,
    0xdd, 0x6f, 0x9a, 0x59, 0x59, 0x7b, 0xad, 0xcc, 0xd6, 0xa5, 0x6d, 0x70, 0xef, 0xd8, 0xc9, 0x7c, 0x49, 0x6e, 0xba, 0x7e, 0x28,
    0x01, 0xd7, 0x33, 0x7d, 0xcf, 0xf7, 0x4d, 0x78, 0xe4, 0x6e, 0xcd, 0x3a, 0x08, 0xcc, 0xba, 0xe3, 0x18,
};

extern const size_t sTestCert_Node01_02_Chip_Len = sizeof(sTestCert_Node01_02_Chip);

extern const uint8_t sTestCert_Node01_02_DER[] = {
    0x30, 0x82, 0x01, 0xdf, 0x30, 0x82, 0x01, 0x86, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x0d, 0x90, 0x93, 0x53, 0x46, 0xb0,
    0x5c, 0xbc, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x22, 0x31, 0x20, 0x30, 0x1e, 0x06,
    0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x04, 0x0c, 0x10, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41,
    0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x31, 0x30, 0x1e, 0x17, 0x0d, 0x32, 0x30, 0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32,
    0x33, 0x34, 0x33, 0x5a, 0x17, 0x0d, 0x34, 0x30, 0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32, 0x33, 0x34, 0x32, 0x5a, 0x30, 0x44,
    0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x01, 0x0c, 0x10, 0x44, 0x45, 0x44,
    0x45, 0x44, 0x45, 0x44, 0x45, 0x30, 0x30, 0x30, 0x31, 0x30, 0x30, 0x30, 0x32, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06,
    0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x05, 0x0c, 0x10, 0x46, 0x41, 0x42, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
    0x30, 0x30, 0x30, 0x31, 0x44, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a,
    0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x96, 0x5f, 0x78, 0xc5, 0x37, 0xec, 0xe1, 0xb8, 0xc3, 0x4a,
    0x7b, 0x98, 0xb9, 0xaa, 0x45, 0xf1, 0x35, 0x63, 0xa5, 0x02, 0xb1, 0x97, 0x9a, 0x60, 0x7b, 0xd0, 0xc4, 0x19, 0x88, 0xbd, 0xd0,
    0xf0, 0xbb, 0xb8, 0x98, 0x16, 0xc2, 0x07, 0xe3, 0xb5, 0x15, 0xd9, 0x26, 0x41, 0x59, 0xf7, 0x8b, 0xd0, 0x97, 0x8e, 0x32, 0xd7,
    0x4c, 0x6d, 0x05, 0x5a, 0x14, 0x9e, 0x8e, 0x9d, 0xba, 0x40, 0x19, 0xbf, 0xa3, 0x81, 0x83, 0x30, 0x81, 0x80, 0x30, 0x0c, 0x06,
    0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff,
    0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x20, 0x06, 0x03, 0x55, 0x1d, 0x25, 0x01, 0x01, 0xff, 0x04, 0x16, 0x30, 0x14, 0x06,
    0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x02, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x01, 0x30, 0x1d,
    0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x56, 0x7b, 0x4f, 0x20, 0xe4, 0xb9, 0xc7, 0xbd, 0x27, 0xb2, 0x9b, 0x3d,
    0xce, 0x6a, 0x76, 0xf7, 0xcd, 0x8e, 0xcc, 0xb6, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14,
    0xcc, 0x13, 0x08, 0xaf, 0x82, 0xcf, 0xee, 0x50, 0x5e, 0xb2, 0x3b, 0x57, 0xbf, 0xe8, 0x6a, 0x31, 0x16, 0x65, 0x53, 0x5f, 0x30,
    0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x47, 0x00, 0x30, 0x44, 0x02, 0x20, 0x60, 0x58, 0x11,
    0x4b, 0xa7, 0x21, 0x82, 0xfc, 0xf6, 0x30, 0x1f, 0x7a, 0x08, 0x1b, 0xca, 0x5a, 0x84, 0x82, 0x02, 0x43, 0x1a, 0x52, 0xfd, 0xbf,
    0xf4, 0x97, 0xd8, 0xdd, 0x6f, 0x9a, 0x59, 0x59, 0x02, 0x20, 0x7b, 0xad, 0xcc, 0xd6, 0xa5, 0x6d, 0x70, 0xef, 0xd8, 0xc9, 0x7c,
    0x49, 0x6e, 0xba, 0x7e, 0x28, 0x01, 0xd7, 0x33, 0x7d, 0xcf, 0xf7, 0x4d, 0x78, 0xe4, 0x6e, 0xcd, 0x3a, 0x08, 0xcc, 0xba, 0xe3,
};

extern const size_t sTestCert_Node01_02_DER_Len = sizeof(sTestCert_Node01_02_DER);

extern const uint8_t sTestCert_Node01_02_PublicKey[] = {
    0x04, 0x96, 0x5f, 0x78, 0xc5, 0x37, 0xec, 0xe1, 0xb8, 0xc3, 0x4a, 0x7b, 0x98, 0xb9, 0xaa, 0x45, 0xf1,
    0x35, 0x63, 0xa5, 0x02, 0xb1, 0x97, 0x9a, 0x60, 0x7b, 0xd0, 0xc4, 0x19, 0x88, 0xbd, 0xd0, 0xf0, 0xbb,
    0xb8, 0x98, 0x16, 0xc2, 0x07, 0xe3, 0xb5, 0x15, 0xd9, 0x26, 0x41, 0x59, 0xf7, 0x8b, 0xd0, 0x97, 0x8e,
    0x32, 0xd7, 0x4c, 0x6d, 0x05, 0x5a, 0x14, 0x9e, 0x8e, 0x9d, 0xba, 0x40, 0x19, 0xbf,
};

extern const size_t sTestCert_Node01_02_PublicKey_Len = sizeof(sTestCert_Node01_02_PublicKey);

extern const uint8_t sTestCert_Node01_02_PrivateKey[] = {
    0x18, 0xb2, 0x98, 0x61, 0x51, 0x42, 0xe7, 0x7a, 0xdb, 0x0d, 0x1d, 0xc3, 0xc5, 0x17, 0x80, 0xd3,
    0x78, 0xca, 0x3e, 0x63, 0xa9, 0x1f, 0xd8, 0xdc, 0xa6, 0x0b, 0xe7, 0x77, 0x9a, 0xb2, 0xf7, 0x0d,
};

extern const size_t sTestCert_Node01_02_PrivateKey_Len = sizeof(sTestCert_Node01_02_PrivateKey);

extern const uint8_t sTestCert_Node01_02_SubjectKeyId[] = {
    0x56, 0x7B, 0x4F, 0x20, 0xE4, 0xB9, 0xC7, 0xBD, 0x27, 0xB2, 0x9B, 0x3D, 0xCE, 0x6A, 0x76, 0xF7, 0xCD, 0x8E, 0xCC, 0xB6,
};

extern const size_t sTestCert_Node01_02_SubjectKeyId_Len = sizeof(sTestCert_Node01_02_SubjectKeyId);

extern const uint8_t sTestCert_Node01_02_AuthorityKeyId[] = {
    0xCC, 0x13, 0x08, 0xAF, 0x82, 0xCF, 0xEE, 0x50, 0x5E, 0xB2, 0x3B, 0x57, 0xBF, 0xE8, 0x6A, 0x31, 0x16, 0x65, 0x53, 0x5F,
};

extern const size_t sTestCert_Node01_02_AuthorityKeyId_Len = sizeof(sTestCert_Node01_02_AuthorityKeyId);

/**************  Test Node02_01 Certificate  **************
Certificate:
    Data:
        Version: 3 (0x2)
        Serial Number: 280707040284312631 (0x3e54590de190437)
        Signature Algorithm: ecdsa-with-SHA256
        Issuer: 1.3.6.1.4.1.37244.1.3 = CACACACA00000004, 1.3.6.1.4.1.37244.1.5 = FAB000000000001D
        Validity
            Not Before: Oct 15 14:23:43 2020 GMT
            Not After : Oct 15 14:23:42 2040 GMT
        Subject: 1.3.6.1.4.1.37244.1.1 = DEDEDEDE00020001, 1.3.6.1.4.1.37244.1.5 = FAB000000000001D
        Subject Public Key Info:
            Public Key Algorithm: id-ecPublicKey
                Public-Key: (256 bit)
                pub:
                    04:a5:70:10:42:aa:69:14:20:f5:ce:f5:bb:65:2e:
                    de:d7:43:cc:4d:6f:00:0e:11:35:f0:ef:69:98:2e:
                    52:8d:bf:9d:b2:2b:90:4d:97:05:9d:51:32:87:53:
                    84:31:1a:f4:07:94:c3:55:75:39:b6:a0:da:de:78:
                    8b:0c:1f:f6:49
                ASN1 OID: prime256v1
                NIST CURVE: P-256
        X509v3 extensions:
            X509v3 Basic Constraints: critical
                CA:FALSE
            X509v3 Key Usage: critical
                Digital Signature
            X509v3 Extended Key Usage: critical
                TLS Web Client Authentication, TLS Web Server Authentication
            X509v3 Subject Key Identifier:
                B4:67:76:E1:CC:0A:F7:81:54:C6:2B:5A:17:98:42:64:BA:F5:50:86
            X509v3 Authority Key Identifier:
                keyid:E1:E7:6E:67:77:85:1D:D7:74:16:BD:DD:35:EC:3C:13:7C:47:29:DC

    Signature Algorithm: ecdsa-with-SHA256
         30:45:02:20:26:46:9a:64:2d:9d:cd:c8:37:fb:ee:58:25:be:
         2f:f3:b9:6a:10:e9:b0:33:73:4a:d5:fd:19:98:6d:2c:90:9c:
         02:21:00:81:86:fc:9c:5b:3a:47:c3:35:ca:16:d0:ae:d1:aa:
         c8:9a:20:37:08:50:b4:15:eb:62:3a:73:c3:38:19:33:74

-----BEGIN CERTIFICATE-----
MIICAjCCAaigAwIBAgIIA+VFkN4ZBDcwCgYIKoZIzj0EAwIwRDEgMB4GCisGAQQB
gqJ8AQMMEENBQ0FDQUNBMDAwMDAwMDQxIDAeBgorBgEEAYKifAEFDBBGQUIwMDAw
MDAwMDAwMDFEMB4XDTIwMTAxNTE0MjM0M1oXDTQwMTAxNTE0MjM0MlowRDEgMB4G
CisGAQQBgqJ8AQEMEERFREVERURFMDAwMjAwMDExIDAeBgorBgEEAYKifAEFDBBG
QUIwMDAwMDAwMDAwMDFEMFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAEpXAQQqpp
FCD1zvW7ZS7e10PMTW8ADhE18O9pmC5Sjb+dsiuQTZcFnVEyh1OEMRr0B5TDVXU5
tqDa3niLDB/2SaOBgzCBgDAMBgNVHRMBAf8EAjAAMA4GA1UdDwEB/wQEAwIHgDAg
BgNVHSUBAf8EFjAUBggrBgEFBQcDAgYIKwYBBQUHAwEwHQYDVR0OBBYEFLRnduHM
CveBVMYrWheYQmS69VCGMB8GA1UdIwQYMBaAFOHnbmd3hR3XdBa93TXsPBN8Rync
MAoGCCqGSM49BAMCA0gAMEUCICZGmmQtnc3IN/vuWCW+L/O5ahDpsDNzStX9GZht
LJCcAiEAgYb8nFs6R8M1yhbQrtGqyJogNwhQtBXrYjpzwzgZM3Q=
-----END CERTIFICATE-----

-----BEGIN EC PRIVATE KEY-----
MHcCAQEEILSfDjcknEjCW6MMevxMlE+IrDwDxbKhk2vYZxWte+kkoAoGCCqGSM49
AwEHoUQDQgAEpXAQQqppFCD1zvW7ZS7e10PMTW8ADhE18O9pmC5Sjb+dsiuQTZcF
nVEyh1OEMRr0B5TDVXU5tqDa3niLDB/2SQ==
-----END EC PRIVATE KEY-----
*/

extern const uint8_t sTestCert_Node02_01_Chip[] = {
    0x15, 0x30, 0x01, 0x08, 0x03, 0xe5, 0x45, 0x90, 0xde, 0x19, 0x04, 0x37, 0x24, 0x02, 0x01, 0x37, 0x03, 0x27, 0x13, 0x04,
    0x00, 0x00, 0x00, 0xca, 0xca, 0xca, 0xca, 0x27, 0x15, 0x1d, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0, 0xfa, 0x18, 0x26, 0x04,
    0xef, 0x17, 0x1b, 0x27, 0x26, 0x05, 0x6e, 0xb5, 0xb9, 0x4c, 0x37, 0x06, 0x27, 0x11, 0x01, 0x00, 0x02, 0x00, 0xde, 0xde,
    0xde, 0xde, 0x27, 0x15, 0x1d, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0, 0xfa, 0x18, 0x24, 0x07, 0x01, 0x24, 0x08, 0x01, 0x30,
    0x09, 0x41, 0x04, 0xa5, 0x70, 0x10, 0x42, 0xaa, 0x69, 0x14, 0x20, 0xf5, 0xce, 0xf5, 0xbb, 0x65, 0x2e, 0xde, 0xd7, 0x43,
    0xcc, 0x4d, 0x6f, 0x00, 0x0e, 0x11, 0x35, 0xf0, 0xef, 0x69, 0x98, 0x2e, 0x52, 0x8d, 0xbf, 0x9d, 0xb2, 0x2b, 0x90, 0x4d,
    0x97, 0x05, 0x9d, 0x51, 0x32, 0x87, 0x53, 0x84, 0x31, 0x1a, 0xf4, 0x07, 0x94, 0xc3, 0x55, 0x75, 0x39, 0xb6, 0xa0, 0xda,
    0xde, 0x78, 0x8b, 0x0c, 0x1f, 0xf6, 0x49, 0x37, 0x0a, 0x35, 0x01, 0x28, 0x01, 0x18, 0x24, 0x02, 0x01, 0x36, 0x03, 0x04,
    0x02, 0x04, 0x01, 0x18, 0x30, 0x04, 0x14, 0xb4, 0x67, 0x76, 0xe1, 0xcc, 0x0a, 0xf7, 0x81, 0x54, 0xc6, 0x2b, 0x5a, 0x17,
    0x98, 0x42, 0x64, 0xba, 0xf5, 0x50, 0x86, 0x30, 0x05, 0x14, 0xe1, 0xe7, 0x6e, 0x67, 0x77, 0x85, 0x1d, 0xd7, 0x74, 0x16,
    0xbd, 0xdd, 0x35, 0xec, 0x3c, 0x13, 0x7c, 0x47, 0x29, 0xdc, 0x18, 0x30, 0x0b, 0x40, 0x26, 0x46, 0x9a, 0x64, 0x2d, 0x9d,
    0xcd, 0xc8, 0x37, 0xfb, 0xee, 0x58, 0x25, 0xbe, 0x2f, 0xf3, 0xb9, 0x6a, 0x10, 0xe9, 0xb0, 0x33, 0x73, 0x4a, 0xd5, 0xfd,
    0x19, 0x98, 0x6d, 0x2c, 0x90, 0x9c, 0x81, 0x86, 0xfc, 0x9c, 0x5b, 0x3a, 0x47, 0xc3, 0x35, 0xca, 0x16, 0xd0, 0xae, 0xd1,
    0xaa, 0xc8, 0x9a, 0x20, 0x37, 0x08, 0x50, 0xb4, 0x15, 0xeb, 0x62, 0x3a, 0x73, 0xc3, 0x38, 0x19, 0x33, 0x74, 0x18,
};

extern const size_t sTestCert_Node02_01_Chip_Len = sizeof(sTestCert_Node02_01_Chip);

extern const uint8_t sTestCert_Node02_01_DER[] = {
    0x30, 0x82, 0x02, 0x02, 0x30, 0x82, 0x01, 0xa8, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x03, 0xe5, 0x45, 0x90, 0xde, 0x19,
    0x04, 0x37, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x44, 0x31, 0x20, 0x30, 0x1e, 0x06,
    0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x03, 0x0c, 0x10, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41,
    0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x34, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2,
    0x7c, 0x01, 0x05, 0x0c, 0x10, 0x46, 0x41, 0x42, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x31, 0x44,
    0x30, 0x1e, 0x17, 0x0d, 0x32, 0x30, 0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x17, 0x0d, 0x34, 0x30,
    0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32, 0x33, 0x34, 0x32, 0x5a, 0x30, 0x44, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06,
    0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x01, 0x0c, 0x10, 0x44, 0x45, 0x44, 0x45, 0x44, 0x45, 0x44, 0x45, 0x30, 0x30, 0x30,
    0x32, 0x30, 0x30, 0x30, 0x31, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x05,
    0x0c, 0x10, 0x46, 0x41, 0x42, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x31, 0x44, 0x30, 0x59, 0x30,
    0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03,
    0x42, 0x00, 0x04, 0xa5, 0x70, 0x10, 0x42, 0xaa, 0x69, 0x14, 0x20, 0xf5, 0xce, 0xf5, 0xbb, 0x65, 0x2e, 0xde, 0xd7, 0x43, 0xcc,
    0x4d, 0x6f, 0x00, 0x0e, 0x11, 0x35, 0xf0, 0xef, 0x69, 0x98, 0x2e, 0x52, 0x8d, 0xbf, 0x9d, 0xb2, 0x2b, 0x90, 0x4d, 0x97, 0x05,
    0x9d, 0x51, 0x32, 0x87, 0x53, 0x84, 0x31, 0x1a, 0xf4, 0x07, 0x94, 0xc3, 0x55, 0x75, 0x39, 0xb6, 0xa0, 0xda, 0xde, 0x78, 0x8b,
    0x0c, 0x1f, 0xf6, 0x49, 0xa3, 0x81, 0x83, 0x30, 0x81, 0x80, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04,
    0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x20,
    0x06, 0x03, 0x55, 0x1d, 0x25, 0x01, 0x01, 0xff, 0x04, 0x16, 0x30, 0x14, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03,
    0x02, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x01, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04,
    0x14, 0xb4, 0x67, 0x76, 0xe1, 0xcc, 0x0a, 0xf7, 0x81, 0x54, 0xc6, 0x2b, 0x5a, 0x17, 0x98, 0x42, 0x64, 0xba, 0xf5, 0x50, 0x86,
    0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xe1, 0xe7, 0x6e, 0x67, 0x77, 0x85, 0x1d, 0xd7,
    0x74, 0x16, 0xbd, 0xdd, 0x35, 0xec, 0x3c, 0x13, 0x7c, 0x47, 0x29, 0xdc, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d,
    0x04, 0x03, 0x02, 0x03, 0x48, 0x00, 0x30, 0x45, 0x02, 0x20, 0x26, 0x46, 0x9a, 0x64, 0x2d, 0x9d, 0xcd, 0xc8, 0x37, 0xfb, 0xee,
    0x58, 0x25, 0xbe, 0x2f, 0xf3, 0xb9, 0x6a, 0x10, 0xe9, 0xb0, 0x33, 0x73, 0x4a, 0xd5, 0xfd, 0x19, 0x98, 0x6d, 0x2c, 0x90, 0x9c,
    0x02, 0x21, 0x00, 0x81, 0x86, 0xfc, 0x9c, 0x5b, 0x3a, 0x47, 0xc3, 0x35, 0xca, 0x16, 0xd0, 0xae, 0xd1, 0xaa, 0xc8, 0x9a, 0x20,
    0x37, 0x08, 0x50, 0xb4, 0x15, 0xeb, 0x62, 0x3a, 0x73, 0xc3, 0x38, 0x19, 0x33, 0x74,
};

extern const size_t sTestCert_Node02_01_DER_Len = sizeof(sTestCert_Node02_01_DER);

extern const uint8_t sTestCert_Node02_01_PublicKey[] = {
    0x04, 0xa5, 0x70, 0x10, 0x42, 0xaa, 0x69, 0x14, 0x20, 0xf5, 0xce, 0xf5, 0xbb, 0x65, 0x2e, 0xde, 0xd7,
    0x43, 0xcc, 0x4d, 0x6f, 0x00, 0x0e, 0x11, 0x35, 0xf0, 0xef, 0x69, 0x98, 0x2e, 0x52, 0x8d, 0xbf, 0x9d,
    0xb2, 0x2b, 0x90, 0x4d, 0x97, 0x05, 0x9d, 0x51, 0x32, 0x87, 0x53, 0x84, 0x31, 0x1a, 0xf4, 0x07, 0x94,
    0xc3, 0x55, 0x75, 0x39, 0xb6, 0xa0, 0xda, 0xde, 0x78, 0x8b, 0x0c, 0x1f, 0xf6, 0x49,
};

extern const size_t sTestCert_Node02_01_PublicKey_Len = sizeof(sTestCert_Node02_01_PublicKey);

extern const uint8_t sTestCert_Node02_01_PrivateKey[] = {
    0xb4, 0x9f, 0x0e, 0x37, 0x24, 0x9c, 0x48, 0xc2, 0x5b, 0xa3, 0x0c, 0x7a, 0xfc, 0x4c, 0x94, 0x4f,
    0x88, 0xac, 0x3c, 0x03, 0xc5, 0xb2, 0xa1, 0x93, 0x6b, 0xd8, 0x67, 0x15, 0xad, 0x7b, 0xe9, 0x24,
};

extern const size_t sTestCert_Node02_01_PrivateKey_Len = sizeof(sTestCert_Node02_01_PrivateKey);

extern const uint8_t sTestCert_Node02_01_SubjectKeyId[] = {
    0xB4, 0x67, 0x76, 0xE1, 0xCC, 0x0A, 0xF7, 0x81, 0x54, 0xC6, 0x2B, 0x5A, 0x17, 0x98, 0x42, 0x64, 0xBA, 0xF5, 0x50, 0x86,
};

extern const size_t sTestCert_Node02_01_SubjectKeyId_Len = sizeof(sTestCert_Node02_01_SubjectKeyId);

extern const uint8_t sTestCert_Node02_01_AuthorityKeyId[] = {
    0xE1, 0xE7, 0x6E, 0x67, 0x77, 0x85, 0x1D, 0xD7, 0x74, 0x16, 0xBD, 0xDD, 0x35, 0xEC, 0x3C, 0x13, 0x7C, 0x47, 0x29, 0xDC,
};

extern const size_t sTestCert_Node02_01_AuthorityKeyId_Len = sizeof(sTestCert_Node02_01_AuthorityKeyId);

/**************  Test Node02_02 Certificate  **************
Certificate:
    Data:
        Version: 3 (0x2)
        Serial Number: 1052152040695922986 (0xe99feb4db62012a)
        Signature Algorithm: ecdsa-with-SHA256
        Issuer: 1.3.6.1.4.1.37244.1.3 = CACACACA00000004, 1.3.6.1.4.1.37244.1.5 = FAB000000000001D
        Validity
            Not Before: Oct 15 14:23:43 2020 GMT
            Not After : Oct 15 14:23:42 2040 GMT
        Subject: 1.3.6.1.4.1.37244.1.1 = DEDEDEDE00020002, 1.3.6.1.4.1.37244.1.5 = FAB000000000001D, CN = TEST CERT COMMON NAME Attr
for Node02_02 Subject Public Key Info: Public Key Algorithm: id-ecPublicKey Public-Key: (256 bit) pub:
                    04:f6:a0:95:8f:b8:fc:32:53:4a:7a:4d:44:87:4b:
                    51:0f:30:9a:df:e6:51:42:2b:35:70:63:05:88:11:
                    b7:d5:bc:56:36:66:d5:48:db:2f:35:23:4d:d2:22:
                    1b:e4:04:2f:ce:c4:74:0e:10:5e:08:b7:89:68:a7:
                    94:a1:84:16:d7
                ASN1 OID: prime256v1
                NIST CURVE: P-256
        X509v3 extensions:
            X509v3 Basic Constraints: critical
                CA:FALSE
            X509v3 Key Usage: critical
                Digital Signature
            X509v3 Extended Key Usage: critical
                TLS Web Client Authentication, TLS Web Server Authentication
            X509v3 Subject Key Identifier:
                21:F4:13:AA:28:D3:89:AF:4E:EE:25:71:1E:DF:A1:98:F7:71:88:85
            X509v3 Authority Key Identifier:
                keyid:E1:E7:6E:67:77:85:1D:D7:74:16:BD:DD:35:EC:3C:13:7C:47:29:DC

    Signature Algorithm: ecdsa-with-SHA256
         30:45:02:20:27:1b:0d:b2:11:95:14:d5:08:02:31:76:34:52:
         25:c4:a8:ca:9e:c2:10:f6:ae:dc:0a:e1:03:73:77:f6:f8:48:
         02:21:00:86:bb:6a:20:27:42:da:75:c9:2d:43:1d:f8:ed:ba:
         cb:8c:30:73:b8:10:0a:a1:9d:46:9e:c8:e4:92:1e:b8:13

-----BEGIN CERTIFICATE-----
MIICNTCCAdugAwIBAgIIDpn+tNtiASowCgYIKoZIzj0EAwIwRDEgMB4GCisGAQQB
gqJ8AQMMEENBQ0FDQUNBMDAwMDAwMDQxIDAeBgorBgEEAYKifAEFDBBGQUIwMDAw
MDAwMDAwMDFEMB4XDTIwMTAxNTE0MjM0M1oXDTQwMTAxNTE0MjM0MlowdzEgMB4G
CisGAQQBgqJ8AQEMEERFREVERURFMDAwMjAwMDIxIDAeBgorBgEEAYKifAEFDBBG
QUIwMDAwMDAwMDAwMDFEMTEwLwYDVQQDDChURVNUIENFUlQgQ09NTU9OIE5BTUUg
QXR0ciBmb3IgTm9kZTAyXzAyMFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAE9qCV
j7j8MlNKek1Eh0tRDzCa3+ZRQis1cGMFiBG31bxWNmbVSNsvNSNN0iIb5AQvzsR0
DhBeCLeJaKeUoYQW16OBgzCBgDAMBgNVHRMBAf8EAjAAMA4GA1UdDwEB/wQEAwIH
gDAgBgNVHSUBAf8EFjAUBggrBgEFBQcDAgYIKwYBBQUHAwEwHQYDVR0OBBYEFCH0
E6oo04mvTu4lcR7foZj3cYiFMB8GA1UdIwQYMBaAFOHnbmd3hR3XdBa93TXsPBN8
RyncMAoGCCqGSM49BAMCA0gAMEUCICcbDbIRlRTVCAIxdjRSJcSoyp7CEPau3Arh
A3N39vhIAiEAhrtqICdC2nXJLUMd+O26y4wwc7gQCqGdRp7I5JIeuBM=
-----END CERTIFICATE-----

-----BEGIN EC PRIVATE KEY-----
MHcCAQEEIOd1sYI17TVC8waIZL6dpiSIeqjoLCmMS1ogbQa1sIVzoAoGCCqGSM49
AwEHoUQDQgAE9qCVj7j8MlNKek1Eh0tRDzCa3+ZRQis1cGMFiBG31bxWNmbVSNsv
NSNN0iIb5AQvzsR0DhBeCLeJaKeUoYQW1w==
-----END EC PRIVATE KEY-----
*/

extern const uint8_t sTestCert_Node02_02_Chip[] = {
    0x15, 0x30, 0x01, 0x08, 0x0e, 0x99, 0xfe, 0xb4, 0xdb, 0x62, 0x01, 0x2a, 0x24, 0x02, 0x01, 0x37, 0x03, 0x27, 0x13, 0x04, 0x00,
    0x00, 0x00, 0xca, 0xca, 0xca, 0xca, 0x27, 0x15, 0x1d, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0, 0xfa, 0x18, 0x26, 0x04, 0xef, 0x17,
    0x1b, 0x27, 0x26, 0x05, 0x6e, 0xb5, 0xb9, 0x4c, 0x37, 0x06, 0x27, 0x11, 0x02, 0x00, 0x02, 0x00, 0xde, 0xde, 0xde, 0xde, 0x27,
    0x15, 0x1d, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0, 0xfa, 0x2c, 0x01, 0x28, 0x54, 0x45, 0x53, 0x54, 0x20, 0x43, 0x45, 0x52, 0x54,
    0x20, 0x43, 0x4f, 0x4d, 0x4d, 0x4f, 0x4e, 0x20, 0x4e, 0x41, 0x4d, 0x45, 0x20, 0x41, 0x74, 0x74, 0x72, 0x20, 0x66, 0x6f, 0x72,
    0x20, 0x4e, 0x6f, 0x64, 0x65, 0x30, 0x32, 0x5f, 0x30, 0x32, 0x18, 0x24, 0x07, 0x01, 0x24, 0x08, 0x01, 0x30, 0x09, 0x41, 0x04,
    0xf6, 0xa0, 0x95, 0x8f, 0xb8, 0xfc, 0x32, 0x53, 0x4a, 0x7a, 0x4d, 0x44, 0x87, 0x4b, 0x51, 0x0f, 0x30, 0x9a, 0xdf, 0xe6, 0x51,
    0x42, 0x2b, 0x35, 0x70, 0x63, 0x05, 0x88, 0x11, 0xb7, 0xd5, 0xbc, 0x56, 0x36, 0x66, 0xd5, 0x48, 0xdb, 0x2f, 0x35, 0x23, 0x4d,
    0xd2, 0x22, 0x1b, 0xe4, 0x04, 0x2f, 0xce, 0xc4, 0x74, 0x0e, 0x10, 0x5e, 0x08, 0xb7, 0x89, 0x68, 0xa7, 0x94, 0xa1, 0x84, 0x16,
    0xd7, 0x37, 0x0a, 0x35, 0x01, 0x28, 0x01, 0x18, 0x24, 0x02, 0x01, 0x36, 0x03, 0x04, 0x02, 0x04, 0x01, 0x18, 0x30, 0x04, 0x14,
    0x21, 0xf4, 0x13, 0xaa, 0x28, 0xd3, 0x89, 0xaf, 0x4e, 0xee, 0x25, 0x71, 0x1e, 0xdf, 0xa1, 0x98, 0xf7, 0x71, 0x88, 0x85, 0x30,
    0x05, 0x14, 0xe1, 0xe7, 0x6e, 0x67, 0x77, 0x85, 0x1d, 0xd7, 0x74, 0x16, 0xbd, 0xdd, 0x35, 0xec, 0x3c, 0x13, 0x7c, 0x47, 0x29,
    0xdc, 0x18, 0x30, 0x0b, 0x40, 0x27, 0x1b, 0x0d, 0xb2, 0x11, 0x95, 0x14, 0xd5, 0x08, 0x02, 0x31, 0x76, 0x34, 0x52, 0x25, 0xc4,
    0xa8, 0xca, 0x9e, 0xc2, 0x10, 0xf6, 0xae, 0xdc, 0x0a, 0xe1, 0x03, 0x73, 0x77, 0xf6, 0xf8, 0x48, 0x86, 0xbb, 0x6a, 0x20, 0x27,
    0x42, 0xda, 0x75, 0xc9, 0x2d, 0x43, 0x1d, 0xf8, 0xed, 0xba, 0xcb, 0x8c, 0x30, 0x73, 0xb8, 0x10, 0x0a, 0xa1, 0x9d, 0x46, 0x9e,
    0xc8, 0xe4, 0x92, 0x1e, 0xb8, 0x13, 0x18,
};

extern const size_t sTestCert_Node02_02_Chip_Len = sizeof(sTestCert_Node02_02_Chip);

extern const uint8_t sTestCert_Node02_02_DER[] = {
    0x30, 0x82, 0x02, 0x35, 0x30, 0x82, 0x01, 0xdb, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x0e, 0x99, 0xfe, 0xb4, 0xdb, 0x62,
    0x01, 0x2a, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x44, 0x31, 0x20, 0x30, 0x1e, 0x06,
    0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x03, 0x0c, 0x10, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41,
    0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x34, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2,
    0x7c, 0x01, 0x05, 0x0c, 0x10, 0x46, 0x41, 0x42, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x31, 0x44,
    0x30, 0x1e, 0x17, 0x0d, 0x32, 0x30, 0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x17, 0x0d, 0x34, 0x30,
    0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32, 0x33, 0x34, 0x32, 0x5a, 0x30, 0x77, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06,
    0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x01, 0x0c, 0x10, 0x44, 0x45, 0x44, 0x45, 0x44, 0x45, 0x44, 0x45, 0x30, 0x30, 0x30,
    0x32, 0x30, 0x30, 0x30, 0x32, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x05,
    0x0c, 0x10, 0x46, 0x41, 0x42, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x31, 0x44, 0x31, 0x31, 0x30,
    0x2f, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x28, 0x54, 0x45, 0x53, 0x54, 0x20, 0x43, 0x45, 0x52, 0x54, 0x20, 0x43, 0x4f, 0x4d,
    0x4d, 0x4f, 0x4e, 0x20, 0x4e, 0x41, 0x4d, 0x45, 0x20, 0x41, 0x74, 0x74, 0x72, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x4e, 0x6f, 0x64,
    0x65, 0x30, 0x32, 0x5f, 0x30, 0x32, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08,
    0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xf6, 0xa0, 0x95, 0x8f, 0xb8, 0xfc, 0x32, 0x53, 0x4a,
    0x7a, 0x4d, 0x44, 0x87, 0x4b, 0x51, 0x0f, 0x30, 0x9a, 0xdf, 0xe6, 0x51, 0x42, 0x2b, 0x35, 0x70, 0x63, 0x05, 0x88, 0x11, 0xb7,
    0xd5, 0xbc, 0x56, 0x36, 0x66, 0xd5, 0x48, 0xdb, 0x2f, 0x35, 0x23, 0x4d, 0xd2, 0x22, 0x1b, 0xe4, 0x04, 0x2f, 0xce, 0xc4, 0x74,
    0x0e, 0x10, 0x5e, 0x08, 0xb7, 0x89, 0x68, 0xa7, 0x94, 0xa1, 0x84, 0x16, 0xd7, 0xa3, 0x81, 0x83, 0x30, 0x81, 0x80, 0x30, 0x0c,
    0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01,
    0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x20, 0x06, 0x03, 0x55, 0x1d, 0x25, 0x01, 0x01, 0xff, 0x04, 0x16, 0x30, 0x14,
    0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x02, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x01, 0x30,
    0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x21, 0xf4, 0x13, 0xaa, 0x28, 0xd3, 0x89, 0xaf, 0x4e, 0xee, 0x25,
    0x71, 0x1e, 0xdf, 0xa1, 0x98, 0xf7, 0x71, 0x88, 0x85, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80,
    0x14, 0xe1, 0xe7, 0x6e, 0x67, 0x77, 0x85, 0x1d, 0xd7, 0x74, 0x16, 0xbd, 0xdd, 0x35, 0xec, 0x3c, 0x13, 0x7c, 0x47, 0x29, 0xdc,
    0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x48, 0x00, 0x30, 0x45, 0x02, 0x20, 0x27, 0x1b,
    0x0d, 0xb2, 0x11, 0x95, 0x14, 0xd5, 0x08, 0x02, 0x31, 0x76, 0x34, 0x52, 0x25, 0xc4, 0xa8, 0xca, 0x9e, 0xc2, 0x10, 0xf6, 0xae,
    0xdc, 0x0a, 0xe1, 0x03, 0x73, 0x77, 0xf6, 0xf8, 0x48, 0x02, 0x21, 0x00, 0x86, 0xbb, 0x6a, 0x20, 0x27, 0x42, 0xda, 0x75, 0xc9,
    0x2d, 0x43, 0x1d, 0xf8, 0xed, 0xba, 0xcb, 0x8c, 0x30, 0x73, 0xb8, 0x10, 0x0a, 0xa1, 0x9d, 0x46, 0x9e, 0xc8, 0xe4, 0x92, 0x1e,
    0xb8, 0x13,
};

extern const size_t sTestCert_Node02_02_DER_Len = sizeof(sTestCert_Node02_02_DER);

extern const uint8_t sTestCert_Node02_02_PublicKey[] = {
    0x04, 0xf6, 0xa0, 0x95, 0x8f, 0xb8, 0xfc, 0x32, 0x53, 0x4a, 0x7a, 0x4d, 0x44, 0x87, 0x4b, 0x51, 0x0f,
    0x30, 0x9a, 0xdf, 0xe6, 0x51, 0x42, 0x2b, 0x35, 0x70, 0x63, 0x05, 0x88, 0x11, 0xb7, 0xd5, 0xbc, 0x56,
    0x36, 0x66, 0xd5, 0x48, 0xdb, 0x2f, 0x35, 0x23, 0x4d, 0xd2, 0x22, 0x1b, 0xe4, 0x04, 0x2f, 0xce, 0xc4,
    0x74, 0x0e, 0x10, 0x5e, 0x08, 0xb7, 0x89, 0x68, 0xa7, 0x94, 0xa1, 0x84, 0x16, 0xd7,
};

extern const size_t sTestCert_Node02_02_PublicKey_Len = sizeof(sTestCert_Node02_02_PublicKey);

extern const uint8_t sTestCert_Node02_02_PrivateKey[] = {
    0xe7, 0x75, 0xb1, 0x82, 0x35, 0xed, 0x35, 0x42, 0xf3, 0x06, 0x88, 0x64, 0xbe, 0x9d, 0xa6, 0x24,
    0x88, 0x7a, 0xa8, 0xe8, 0x2c, 0x29, 0x8c, 0x4b, 0x5a, 0x20, 0x6d, 0x06, 0xb5, 0xb0, 0x85, 0x73,
};

extern const size_t sTestCert_Node02_02_PrivateKey_Len = sizeof(sTestCert_Node02_02_PrivateKey);

extern const uint8_t sTestCert_Node02_02_SubjectKeyId[] = {
    0x21, 0xF4, 0x13, 0xAA, 0x28, 0xD3, 0x89, 0xAF, 0x4E, 0xEE, 0x25, 0x71, 0x1E, 0xDF, 0xA1, 0x98, 0xF7, 0x71, 0x88, 0x85,
};

extern const size_t sTestCert_Node02_02_SubjectKeyId_Len = sizeof(sTestCert_Node02_02_SubjectKeyId);

extern const uint8_t sTestCert_Node02_02_AuthorityKeyId[] = {
    0xE1, 0xE7, 0x6E, 0x67, 0x77, 0x85, 0x1D, 0xD7, 0x74, 0x16, 0xBD, 0xDD, 0x35, 0xEC, 0x3C, 0x13, 0x7C, 0x47, 0x29, 0xDC,
};

extern const size_t sTestCert_Node02_02_AuthorityKeyId_Len = sizeof(sTestCert_Node02_02_AuthorityKeyId);

/**************  Test Node02_03 Certificate  **************
Certificate:
    Data:
        Version: 3 (0x2)
        Serial Number: 1067793285220423275 (0xed19055e31c566b)
        Signature Algorithm: ecdsa-with-SHA256
        Issuer: 1.3.6.1.4.1.37244.1.3 = CACACACA00000004, 1.3.6.1.4.1.37244.1.5 = FAB000000000001D
        Validity
            Not Before: Oct 15 14:23:43 2020 GMT
            Not After : Oct 15 14:23:42 2040 GMT
        Subject: 1.3.6.1.4.1.37244.1.1 = DEDEDEDE00020003, 1.3.6.1.4.1.37244.1.5 = FAB000000000001D, CN =
TestCert02_03, 1.3.6.1.4.1.37244.1.6 = ABCD0001 Subject Public Key Info: Public Key Algorithm: id-ecPublicKey Public-Key: (256 bit)
                pub:
                    04:ab:9b:9c:a9:e1:93:76:c4:f5:7d:de:c4:38:6a:
                    00:fc:3e:8a:4b:38:8c:d1:8a:b4:ff:cc:da:45:62:
                    08:0c:08:b5:e7:16:e6:b2:ce:c5:7d:22:d6:01:87:
                    a9:20:fb:a0:b4:57:b9:ac:8c:29:e1:9a:36:7e:2f:
                    f0:1b:5c:7a:59
                ASN1 OID: prime256v1
                NIST CURVE: P-256
        X509v3 extensions:
            X509v3 Basic Constraints: critical
                CA:FALSE
            X509v3 Key Usage: critical
                Digital Signature
            X509v3 Extended Key Usage: critical
                TLS Web Client Authentication, TLS Web Server Authentication
            X509v3 Subject Key Identifier:
                E5:7F:7A:B3:74:B0:2F:53:EF:A0:A5:B6:52:F2:21:1C:1A:AF:05:CF
            X509v3 Authority Key Identifier:
                keyid:E1:E7:6E:67:77:85:1D:D7:74:16:BD:DD:35:EC:3C:13:7C:47:29:DC

    Signature Algorithm: ecdsa-with-SHA256
         30:46:02:21:00:8f:32:e6:83:2d:dd:ed:55:3e:02:02:d2:61:
         86:db:08:86:6e:2e:ac:3b:ec:3d:1f:11:d1:74:00:97:ee:5d:
         ea:02:21:00:cc:22:75:f7:00:0c:a1:e5:e4:9b:ab:45:19:c6:
         70:f7:c2:50:9c:ce:18:97:ce:95:8a:e8:f4:a9:49:9f:33:ea

-----BEGIN CERTIFICATE-----
MIICNTCCAdqgAwIBAgIIDtGQVeMcVmswCgYIKoZIzj0EAwIwRDEgMB4GCisGAQQB
gqJ8AQMMEENBQ0FDQUNBMDAwMDAwMDQxIDAeBgorBgEEAYKifAEFDBBGQUIwMDAw
MDAwMDAwMDFEMB4XDTIwMTAxNTE0MjM0M1oXDTQwMTAxNTE0MjM0MlowdjEgMB4G
CisGAQQBgqJ8AQEMEERFREVERURFMDAwMjAwMDMxIDAeBgorBgEEAYKifAEFDBBG
QUIwMDAwMDAwMDAwMDFEMRYwFAYDVQQDDA1UZXN0Q2VydDAyXzAzMRgwFgYKKwYB
BAGConwBBgwIQUJDRDAwMDEwWTATBgcqhkjOPQIBBggqhkjOPQMBBwNCAASrm5yp
4ZN2xPV93sQ4agD8PopLOIzRirT/zNpFYggMCLXnFuayzsV9ItYBh6kg+6C0V7ms
jCnhmjZ+L/AbXHpZo4GDMIGAMAwGA1UdEwEB/wQCMAAwDgYDVR0PAQH/BAQDAgeA
MCAGA1UdJQEB/wQWMBQGCCsGAQUFBwMCBggrBgEFBQcDATAdBgNVHQ4EFgQU5X96
s3SwL1PvoKW2UvIhHBqvBc8wHwYDVR0jBBgwFoAU4eduZ3eFHdd0Fr3dNew8E3xH
KdwwCgYIKoZIzj0EAwIDSQAwRgIhAI8y5oMt3e1VPgIC0mGG2wiGbi6sO+w9HxHR
dACX7l3qAiEAzCJ19wAMoeXkm6tFGcZw98JQnM4Yl86Viuj0qUmfM+o=
-----END CERTIFICATE-----

-----BEGIN EC PRIVATE KEY-----
MHcCAQEEIAoS9pwV9C8RDuP063f4r+8FDAl18wZiFOWKMEP7IN66oAoGCCqGSM49
AwEHoUQDQgAEq5ucqeGTdsT1fd7EOGoA/D6KSziM0Yq0/8zaRWIIDAi15xbmss7F
fSLWAYepIPugtFe5rIwp4Zo2fi/wG1x6WQ==
-----END EC PRIVATE KEY-----
*/

extern const uint8_t sTestCert_Node02_03_Chip[] = {
    0x15, 0x30, 0x01, 0x08, 0x0e, 0xd1, 0x90, 0x55, 0xe3, 0x1c, 0x56, 0x6b, 0x24, 0x02, 0x01, 0x37, 0x03, 0x27, 0x13, 0x04, 0x00,
    0x00, 0x00, 0xca, 0xca, 0xca, 0xca, 0x27, 0x15, 0x1d, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0, 0xfa, 0x18, 0x26, 0x04, 0xef, 0x17,
    0x1b, 0x27, 0x26, 0x05, 0x6e, 0xb5, 0xb9, 0x4c, 0x37, 0x06, 0x27, 0x11, 0x03, 0x00, 0x02, 0x00, 0xde, 0xde, 0xde, 0xde, 0x27,
    0x15, 0x1d, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0, 0xfa, 0x2c, 0x01, 0x0d, 0x54, 0x65, 0x73, 0x74, 0x43, 0x65, 0x72, 0x74, 0x30,
    0x32, 0x5f, 0x30, 0x33, 0x26, 0x16, 0x01, 0x00, 0xcd, 0xab, 0x18, 0x24, 0x07, 0x01, 0x24, 0x08, 0x01, 0x30, 0x09, 0x41, 0x04,
    0xab, 0x9b, 0x9c, 0xa9, 0xe1, 0x93, 0x76, 0xc4, 0xf5, 0x7d, 0xde, 0xc4, 0x38, 0x6a, 0x00, 0xfc, 0x3e, 0x8a, 0x4b, 0x38, 0x8c,
    0xd1, 0x8a, 0xb4, 0xff, 0xcc, 0xda, 0x45, 0x62, 0x08, 0x0c, 0x08, 0xb5, 0xe7, 0x16, 0xe6, 0xb2, 0xce, 0xc5, 0x7d, 0x22, 0xd6,
    0x01, 0x87, 0xa9, 0x20, 0xfb, 0xa0, 0xb4, 0x57, 0xb9, 0xac, 0x8c, 0x29, 0xe1, 0x9a, 0x36, 0x7e, 0x2f, 0xf0, 0x1b, 0x5c, 0x7a,
    0x59, 0x37, 0x0a, 0x35, 0x01, 0x28, 0x01, 0x18, 0x24, 0x02, 0x01, 0x36, 0x03, 0x04, 0x02, 0x04, 0x01, 0x18, 0x30, 0x04, 0x14,
    0xe5, 0x7f, 0x7a, 0xb3, 0x74, 0xb0, 0x2f, 0x53, 0xef, 0xa0, 0xa5, 0xb6, 0x52, 0xf2, 0x21, 0x1c, 0x1a, 0xaf, 0x05, 0xcf, 0x30,
    0x05, 0x14, 0xe1, 0xe7, 0x6e, 0x67, 0x77, 0x85, 0x1d, 0xd7, 0x74, 0x16, 0xbd, 0xdd, 0x35, 0xec, 0x3c, 0x13, 0x7c, 0x47, 0x29,
    0xdc, 0x18, 0x30, 0x0b, 0x40, 0x8f, 0x32, 0xe6, 0x83, 0x2d, 0xdd, 0xed, 0x55, 0x3e, 0x02, 0x02, 0xd2, 0x61, 0x86, 0xdb, 0x08,
    0x86, 0x6e, 0x2e, 0xac, 0x3b, 0xec, 0x3d, 0x1f, 0x11, 0xd1, 0x74, 0x00, 0x97, 0xee, 0x5d, 0xea, 0xcc, 0x22, 0x75, 0xf7, 0x00,
    0x0c, 0xa1, 0xe5, 0xe4, 0x9b, 0xab, 0x45, 0x19, 0xc6, 0x70, 0xf7, 0xc2, 0x50, 0x9c, 0xce, 0x18, 0x97, 0xce, 0x95, 0x8a, 0xe8,
    0xf4, 0xa9, 0x49, 0x9f, 0x33, 0xea, 0x18,
};

extern const size_t sTestCert_Node02_03_Chip_Len = sizeof(sTestCert_Node02_03_Chip);

extern const uint8_t sTestCert_Node02_03_DER[] = {
    0x30, 0x82, 0x02, 0x35, 0x30, 0x82, 0x01, 0xda, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x0e, 0xd1, 0x90, 0x55, 0xe3, 0x1c,
    0x56, 0x6b, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x44, 0x31, 0x20, 0x30, 0x1e, 0x06,
    0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x03, 0x0c, 0x10, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41,
    0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x34, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2,
    0x7c, 0x01, 0x05, 0x0c, 0x10, 0x46, 0x41, 0x42, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x31, 0x44,
    0x30, 0x1e, 0x17, 0x0d, 0x32, 0x30, 0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x17, 0x0d, 0x34, 0x30,
    0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32, 0x33, 0x34, 0x32, 0x5a, 0x30, 0x76, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06,
    0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x01, 0x0c, 0x10, 0x44, 0x45, 0x44, 0x45, 0x44, 0x45, 0x44, 0x45, 0x30, 0x30, 0x30,
    0x32, 0x30, 0x30, 0x30, 0x33, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x05,
    0x0c, 0x10, 0x46, 0x41, 0x42, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x31, 0x44, 0x31, 0x16, 0x30,
    0x14, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0d, 0x54, 0x65, 0x73, 0x74, 0x43, 0x65, 0x72, 0x74, 0x30, 0x32, 0x5f, 0x30, 0x33,
    0x31, 0x18, 0x30, 0x16, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x06, 0x0c, 0x08, 0x41, 0x42, 0x43,
    0x44, 0x30, 0x30, 0x30, 0x31, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a,
    0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xab, 0x9b, 0x9c, 0xa9, 0xe1, 0x93, 0x76, 0xc4, 0xf5, 0x7d,
    0xde, 0xc4, 0x38, 0x6a, 0x00, 0xfc, 0x3e, 0x8a, 0x4b, 0x38, 0x8c, 0xd1, 0x8a, 0xb4, 0xff, 0xcc, 0xda, 0x45, 0x62, 0x08, 0x0c,
    0x08, 0xb5, 0xe7, 0x16, 0xe6, 0xb2, 0xce, 0xc5, 0x7d, 0x22, 0xd6, 0x01, 0x87, 0xa9, 0x20, 0xfb, 0xa0, 0xb4, 0x57, 0xb9, 0xac,
    0x8c, 0x29, 0xe1, 0x9a, 0x36, 0x7e, 0x2f, 0xf0, 0x1b, 0x5c, 0x7a, 0x59, 0xa3, 0x81, 0x83, 0x30, 0x81, 0x80, 0x30, 0x0c, 0x06,
    0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff,
    0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x20, 0x06, 0x03, 0x55, 0x1d, 0x25, 0x01, 0x01, 0xff, 0x04, 0x16, 0x30, 0x14, 0x06,
    0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x02, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x01, 0x30, 0x1d,
    0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xe5, 0x7f, 0x7a, 0xb3, 0x74, 0xb0, 0x2f, 0x53, 0xef, 0xa0, 0xa5, 0xb6,
    0x52, 0xf2, 0x21, 0x1c, 0x1a, 0xaf, 0x05, 0xcf, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14,
    0xe1, 0xe7, 0x6e, 0x67, 0x77, 0x85, 0x1d, 0xd7, 0x74, 0x16, 0xbd, 0xdd, 0x35, 0xec, 0x3c, 0x13, 0x7c, 0x47, 0x29, 0xdc, 0x30,
    0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x49, 0x00, 0x30, 0x46, 0x02, 0x21, 0x00, 0x8f, 0x32,
    0xe6, 0x83, 0x2d, 0xdd, 0xed, 0x55, 0x3e, 0x02, 0x02, 0xd2, 0x61, 0x86, 0xdb, 0x08, 0x86, 0x6e, 0x2e, 0xac, 0x3b, 0xec, 0x3d,
    0x1f, 0x11, 0xd1, 0x74, 0x00, 0x97, 0xee, 0x5d, 0xea, 0x02, 0x21, 0x00, 0xcc, 0x22, 0x75, 0xf7, 0x00, 0x0c, 0xa1, 0xe5, 0xe4,
    0x9b, 0xab, 0x45, 0x19, 0xc6, 0x70, 0xf7, 0xc2, 0x50, 0x9c, 0xce, 0x18, 0x97, 0xce, 0x95, 0x8a, 0xe8, 0xf4, 0xa9, 0x49, 0x9f,
    0x33, 0xea,
};

extern const size_t sTestCert_Node02_03_DER_Len = sizeof(sTestCert_Node02_03_DER);

extern const uint8_t sTestCert_Node02_03_PublicKey[] = {
    0x04, 0xab, 0x9b, 0x9c, 0xa9, 0xe1, 0x93, 0x76, 0xc4, 0xf5, 0x7d, 0xde, 0xc4, 0x38, 0x6a, 0x00, 0xfc,
    0x3e, 0x8a, 0x4b, 0x38, 0x8c, 0xd1, 0x8a, 0xb4, 0xff, 0xcc, 0xda, 0x45, 0x62, 0x08, 0x0c, 0x08, 0xb5,
    0xe7, 0x16, 0xe6, 0xb2, 0xce, 0xc5, 0x7d, 0x22, 0xd6, 0x01, 0x87, 0xa9, 0x20, 0xfb, 0xa0, 0xb4, 0x57,
    0xb9, 0xac, 0x8c, 0x29, 0xe1, 0x9a, 0x36, 0x7e, 0x2f, 0xf0, 0x1b, 0x5c, 0x7a, 0x59,
};

extern const size_t sTestCert_Node02_03_PublicKey_Len = sizeof(sTestCert_Node02_03_PublicKey);

extern const uint8_t sTestCert_Node02_03_PrivateKey[] = {
    0x0a, 0x12, 0xf6, 0x9c, 0x15, 0xf4, 0x2f, 0x11, 0x0e, 0xe3, 0xf4, 0xeb, 0x77, 0xf8, 0xaf, 0xef,
    0x05, 0x0c, 0x09, 0x75, 0xf3, 0x06, 0x62, 0x14, 0xe5, 0x8a, 0x30, 0x43, 0xfb, 0x20, 0xde, 0xba,
};

extern const size_t sTestCert_Node02_03_PrivateKey_Len = sizeof(sTestCert_Node02_03_PrivateKey);

extern const uint8_t sTestCert_Node02_03_SubjectKeyId[] = {
    0xE5, 0x7F, 0x7A, 0xB3, 0x74, 0xB0, 0x2F, 0x53, 0xEF, 0xA0, 0xA5, 0xB6, 0x52, 0xF2, 0x21, 0x1C, 0x1A, 0xAF, 0x05, 0xCF,
};

extern const size_t sTestCert_Node02_03_SubjectKeyId_Len = sizeof(sTestCert_Node02_03_SubjectKeyId);

extern const uint8_t sTestCert_Node02_03_AuthorityKeyId[] = {
    0xE1, 0xE7, 0x6E, 0x67, 0x77, 0x85, 0x1D, 0xD7, 0x74, 0x16, 0xBD, 0xDD, 0x35, 0xEC, 0x3C, 0x13, 0x7C, 0x47, 0x29, 0xDC,
};

extern const size_t sTestCert_Node02_03_AuthorityKeyId_Len = sizeof(sTestCert_Node02_03_AuthorityKeyId);

/**************  Test Node02_04 Certificate  **************
Certificate:
    Data:
        Version: 3 (0x2)
        Serial Number: 8650176850154750578 (0x780b9a03bdc80272)
        Signature Algorithm: ecdsa-with-SHA256
        Issuer: 1.3.6.1.4.1.37244.1.3 = CACACACA00000004, 1.3.6.1.4.1.37244.1.5 = FAB000000000001D
        Validity
            Not Before: Oct 15 14:23:43 2020 GMT
            Not After : Oct 15 14:23:42 2040 GMT
        Subject: 1.3.6.1.4.1.37244.1.6 = ABCE1002, CN = TestCert02_04, 1.3.6.1.4.1.37244.1.5 =
FAB000000000001D, 1.3.6.1.4.1.37244.1.6 = ABCD0003, 1.3.6.1.4.1.37244.1.1 = DEDEDEDE00020004 Subject Public Key Info: Public Key
Algorithm: id-ecPublicKey Public-Key: (256 bit) pub: 04:ef:70:00:d0:0e:73:2e:02:9e:e6:c1:15:9a:b6:
                    6c:a0:a2:e6:13:74:c9:2b:6f:45:b7:99:89:66:15:
                    49:2b:7d:d5:aa:9d:87:fc:56:df:90:f8:0d:88:4b:
                    3f:9f:79:3e:5b:a8:50:0c:be:85:87:a4:41:c2:21:
                    5b:87:d7:1e:4a
                ASN1 OID: prime256v1
                NIST CURVE: P-256
        X509v3 extensions:
            X509v3 Basic Constraints: critical
                CA:FALSE
            X509v3 Key Usage: critical
                Digital Signature
            X509v3 Extended Key Usage: critical
                TLS Web Client Authentication, TLS Web Server Authentication
            X509v3 Subject Key Identifier:
                B4:49:16:80:10:4F:C3:8D:9C:FA:DA:69:EA:30:85:14:4C:92:7D:2A
            X509v3 Authority Key Identifier:
                keyid:E1:E7:6E:67:77:85:1D:D7:74:16:BD:DD:35:EC:3C:13:7C:47:29:DC

    Signature Algorithm: ecdsa-with-SHA256
         30:45:02:20:2d:13:c2:ac:58:01:83:77:58:4b:a6:2d:0d:d1:
         cf:bc:d1:a0:da:20:ff:e7:61:79:e5:f2:f6:83:9a:50:90:70:
         02:21:00:c7:5b:07:89:74:26:73:f7:e5:02:64:cc:9c:ed:c0:
         01:c9:3c:9e:9c:05:02:90:3f:d7:3d:25:e3:1e:42:ad:24

-----BEGIN CERTIFICATE-----
MIICTzCCAfWgAwIBAgIIeAuaA73IAnIwCgYIKoZIzj0EAwIwRDEgMB4GCisGAQQB
gqJ8AQMMEENBQ0FDQUNBMDAwMDAwMDQxIDAeBgorBgEEAYKifAEFDBBGQUIwMDAw
MDAwMDAwMDFEMB4XDTIwMTAxNTE0MjM0M1oXDTQwMTAxNTE0MjM0MlowgZAxGDAW
BgorBgEEAYKifAEGDAhBQkNFMTAwMjEWMBQGA1UEAwwNVGVzdENlcnQwMl8wNDEg
MB4GCisGAQQBgqJ8AQUMEEZBQjAwMDAwMDAwMDAwMUQxGDAWBgorBgEEAYKifAEG
DAhBQkNEMDAwMzEgMB4GCisGAQQBgqJ8AQEMEERFREVERURFMDAwMjAwMDQwWTAT
BgcqhkjOPQIBBggqhkjOPQMBBwNCAATvcADQDnMuAp7mwRWatmygouYTdMkrb0W3
mYlmFUkrfdWqnYf8Vt+Q+A2ISz+feT5bqFAMvoWHpEHCIVuH1x5Ko4GDMIGAMAwG
A1UdEwEB/wQCMAAwDgYDVR0PAQH/BAQDAgeAMCAGA1UdJQEB/wQWMBQGCCsGAQUF
BwMCBggrBgEFBQcDATAdBgNVHQ4EFgQUtEkWgBBPw42c+tpp6jCFFEySfSowHwYD
VR0jBBgwFoAU4eduZ3eFHdd0Fr3dNew8E3xHKdwwCgYIKoZIzj0EAwIDSAAwRQIg
LRPCrFgBg3dYS6YtDdHPvNGg2iD/52F55fL2g5pQkHACIQDHWweJdCZz9+UCZMyc
7cAByTyenAUCkD/XPSXjHkKtJA==
-----END CERTIFICATE-----

-----BEGIN EC PRIVATE KEY-----
MHcCAQEEIOGVhbY4gyNL3AJ8VJzNemfINBwBcuAW7MJd9uuchFq2oAoGCCqGSM49
AwEHoUQDQgAE73AA0A5zLgKe5sEVmrZsoKLmE3TJK29Ft5mJZhVJK33Vqp2H/Fbf
kPgNiEs/n3k+W6hQDL6Fh6RBwiFbh9ceSg==
-----END EC PRIVATE KEY-----
*/

extern const uint8_t sTestCert_Node02_04_Chip[] = {
    0x15, 0x30, 0x01, 0x08, 0x78, 0x0b, 0x9a, 0x03, 0xbd, 0xc8, 0x02, 0x72, 0x24, 0x02, 0x01, 0x37, 0x03, 0x27, 0x13, 0x04, 0x00,
    0x00, 0x00, 0xca, 0xca, 0xca, 0xca, 0x27, 0x15, 0x1d, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0, 0xfa, 0x18, 0x26, 0x04, 0xef, 0x17,
    0x1b, 0x27, 0x26, 0x05, 0x6e, 0xb5, 0xb9, 0x4c, 0x37, 0x06, 0x26, 0x16, 0x02, 0x10, 0xce, 0xab, 0x2c, 0x01, 0x0d, 0x54, 0x65,
    0x73, 0x74, 0x43, 0x65, 0x72, 0x74, 0x30, 0x32, 0x5f, 0x30, 0x34, 0x27, 0x15, 0x1d, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0, 0xfa,
    0x26, 0x16, 0x03, 0x00, 0xcd, 0xab, 0x27, 0x11, 0x04, 0x00, 0x02, 0x00, 0xde, 0xde, 0xde, 0xde, 0x18, 0x24, 0x07, 0x01, 0x24,
    0x08, 0x01, 0x30, 0x09, 0x41, 0x04, 0xef, 0x70, 0x00, 0xd0, 0x0e, 0x73, 0x2e, 0x02, 0x9e, 0xe6, 0xc1, 0x15, 0x9a, 0xb6, 0x6c,
    0xa0, 0xa2, 0xe6, 0x13, 0x74, 0xc9, 0x2b, 0x6f, 0x45, 0xb7, 0x99, 0x89, 0x66, 0x15, 0x49, 0x2b, 0x7d, 0xd5, 0xaa, 0x9d, 0x87,
    0xfc, 0x56, 0xdf, 0x90, 0xf8, 0x0d, 0x88, 0x4b, 0x3f, 0x9f, 0x79, 0x3e, 0x5b, 0xa8, 0x50, 0x0c, 0xbe, 0x85, 0x87, 0xa4, 0x41,
    0xc2, 0x21, 0x5b, 0x87, 0xd7, 0x1e, 0x4a, 0x37, 0x0a, 0x35, 0x01, 0x28, 0x01, 0x18, 0x24, 0x02, 0x01, 0x36, 0x03, 0x04, 0x02,
    0x04, 0x01, 0x18, 0x30, 0x04, 0x14, 0xb4, 0x49, 0x16, 0x80, 0x10, 0x4f, 0xc3, 0x8d, 0x9c, 0xfa, 0xda, 0x69, 0xea, 0x30, 0x85,
    0x14, 0x4c, 0x92, 0x7d, 0x2a, 0x30, 0x05, 0x14, 0xe1, 0xe7, 0x6e, 0x67, 0x77, 0x85, 0x1d, 0xd7, 0x74, 0x16, 0xbd, 0xdd, 0x35,
    0xec, 0x3c, 0x13, 0x7c, 0x47, 0x29, 0xdc, 0x18, 0x30, 0x0b, 0x40, 0x2d, 0x13, 0xc2, 0xac, 0x58, 0x01, 0x83, 0x77, 0x58, 0x4b,
    0xa6, 0x2d, 0x0d, 0xd1, 0xcf, 0xbc, 0xd1, 0xa0, 0xda, 0x20, 0xff, 0xe7, 0x61, 0x79, 0xe5, 0xf2, 0xf6, 0x83, 0x9a, 0x50, 0x90,
    0x70, 0xc7, 0x5b, 0x07, 0x89, 0x74, 0x26, 0x73, 0xf7, 0xe5, 0x02, 0x64, 0xcc, 0x9c, 0xed, 0xc0, 0x01, 0xc9, 0x3c, 0x9e, 0x9c,
    0x05, 0x02, 0x90, 0x3f, 0xd7, 0x3d, 0x25, 0xe3, 0x1e, 0x42, 0xad, 0x24, 0x18,
};

extern const size_t sTestCert_Node02_04_Chip_Len = sizeof(sTestCert_Node02_04_Chip);

extern const uint8_t sTestCert_Node02_04_DER[] = {
    0x30, 0x82, 0x02, 0x4f, 0x30, 0x82, 0x01, 0xf5, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x78, 0x0b, 0x9a, 0x03, 0xbd, 0xc8,
    0x02, 0x72, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x44, 0x31, 0x20, 0x30, 0x1e, 0x06,
    0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x03, 0x0c, 0x10, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41,
    0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x34, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2,
    0x7c, 0x01, 0x05, 0x0c, 0x10, 0x46, 0x41, 0x42, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x31, 0x44,
    0x30, 0x1e, 0x17, 0x0d, 0x32, 0x30, 0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x17, 0x0d, 0x34, 0x30,
    0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32, 0x33, 0x34, 0x32, 0x5a, 0x30, 0x81, 0x90, 0x31, 0x18, 0x30, 0x16, 0x06, 0x0a, 0x2b,
    0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x06, 0x0c, 0x08, 0x41, 0x42, 0x43, 0x45, 0x31, 0x30, 0x30, 0x32, 0x31, 0x16,
    0x30, 0x14, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x0d, 0x54, 0x65, 0x73, 0x74, 0x43, 0x65, 0x72, 0x74, 0x30, 0x32, 0x5f, 0x30,
    0x34, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x05, 0x0c, 0x10, 0x46, 0x41,
    0x42, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x31, 0x44, 0x31, 0x18, 0x30, 0x16, 0x06, 0x0a, 0x2b,
    0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x06, 0x0c, 0x08, 0x41, 0x42, 0x43, 0x44, 0x30, 0x30, 0x30, 0x33, 0x31, 0x20,
    0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x01, 0x0c, 0x10, 0x44, 0x45, 0x44, 0x45, 0x44,
    0x45, 0x44, 0x45, 0x30, 0x30, 0x30, 0x32, 0x30, 0x30, 0x30, 0x34, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce,
    0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xef, 0x70, 0x00, 0xd0,
    0x0e, 0x73, 0x2e, 0x02, 0x9e, 0xe6, 0xc1, 0x15, 0x9a, 0xb6, 0x6c, 0xa0, 0xa2, 0xe6, 0x13, 0x74, 0xc9, 0x2b, 0x6f, 0x45, 0xb7,
    0x99, 0x89, 0x66, 0x15, 0x49, 0x2b, 0x7d, 0xd5, 0xaa, 0x9d, 0x87, 0xfc, 0x56, 0xdf, 0x90, 0xf8, 0x0d, 0x88, 0x4b, 0x3f, 0x9f,
    0x79, 0x3e, 0x5b, 0xa8, 0x50, 0x0c, 0xbe, 0x85, 0x87, 0xa4, 0x41, 0xc2, 0x21, 0x5b, 0x87, 0xd7, 0x1e, 0x4a, 0xa3, 0x81, 0x83,
    0x30, 0x81, 0x80, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03,
    0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x20, 0x06, 0x03, 0x55, 0x1d, 0x25, 0x01, 0x01,
    0xff, 0x04, 0x16, 0x30, 0x14, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x02, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05,
    0x05, 0x07, 0x03, 0x01, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xb4, 0x49, 0x16, 0x80, 0x10, 0x4f,
    0xc3, 0x8d, 0x9c, 0xfa, 0xda, 0x69, 0xea, 0x30, 0x85, 0x14, 0x4c, 0x92, 0x7d, 0x2a, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23,
    0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xe1, 0xe7, 0x6e, 0x67, 0x77, 0x85, 0x1d, 0xd7, 0x74, 0x16, 0xbd, 0xdd, 0x35, 0xec, 0x3c,
    0x13, 0x7c, 0x47, 0x29, 0xdc, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x48, 0x00, 0x30,
    0x45, 0x02, 0x20, 0x2d, 0x13, 0xc2, 0xac, 0x58, 0x01, 0x83, 0x77, 0x58, 0x4b, 0xa6, 0x2d, 0x0d, 0xd1, 0xcf, 0xbc, 0xd1, 0xa0,
    0xda, 0x20, 0xff, 0xe7, 0x61, 0x79, 0xe5, 0xf2, 0xf6, 0x83, 0x9a, 0x50, 0x90, 0x70, 0x02, 0x21, 0x00, 0xc7, 0x5b, 0x07, 0x89,
    0x74, 0x26, 0x73, 0xf7, 0xe5, 0x02, 0x64, 0xcc, 0x9c, 0xed, 0xc0, 0x01, 0xc9, 0x3c, 0x9e, 0x9c, 0x05, 0x02, 0x90, 0x3f, 0xd7,
    0x3d, 0x25, 0xe3, 0x1e, 0x42, 0xad, 0x24,
};

extern const size_t sTestCert_Node02_04_DER_Len = sizeof(sTestCert_Node02_04_DER);

extern const uint8_t sTestCert_Node02_04_PublicKey[] = {
    0x04, 0xef, 0x70, 0x00, 0xd0, 0x0e, 0x73, 0x2e, 0x02, 0x9e, 0xe6, 0xc1, 0x15, 0x9a, 0xb6, 0x6c, 0xa0,
    0xa2, 0xe6, 0x13, 0x74, 0xc9, 0x2b, 0x6f, 0x45, 0xb7, 0x99, 0x89, 0x66, 0x15, 0x49, 0x2b, 0x7d, 0xd5,
    0xaa, 0x9d, 0x87, 0xfc, 0x56, 0xdf, 0x90, 0xf8, 0x0d, 0x88, 0x4b, 0x3f, 0x9f, 0x79, 0x3e, 0x5b, 0xa8,
    0x50, 0x0c, 0xbe, 0x85, 0x87, 0xa4, 0x41, 0xc2, 0x21, 0x5b, 0x87, 0xd7, 0x1e, 0x4a,
};

extern const size_t sTestCert_Node02_04_PublicKey_Len = sizeof(sTestCert_Node02_04_PublicKey);

extern const uint8_t sTestCert_Node02_04_PrivateKey[] = {
    0xe1, 0x95, 0x85, 0xb6, 0x38, 0x83, 0x23, 0x4b, 0xdc, 0x02, 0x7c, 0x54, 0x9c, 0xcd, 0x7a, 0x67,
    0xc8, 0x34, 0x1c, 0x01, 0x72, 0xe0, 0x16, 0xec, 0xc2, 0x5d, 0xf6, 0xeb, 0x9c, 0x84, 0x5a, 0xb6,
};

extern const size_t sTestCert_Node02_04_PrivateKey_Len = sizeof(sTestCert_Node02_04_PrivateKey);

extern const uint8_t sTestCert_Node02_04_SubjectKeyId[] = {
    0xB4, 0x49, 0x16, 0x80, 0x10, 0x4F, 0xC3, 0x8D, 0x9C, 0xFA, 0xDA, 0x69, 0xEA, 0x30, 0x85, 0x14, 0x4C, 0x92, 0x7D, 0x2A,
};

extern const size_t sTestCert_Node02_04_SubjectKeyId_Len = sizeof(sTestCert_Node02_04_SubjectKeyId);

extern const uint8_t sTestCert_Node02_04_AuthorityKeyId[] = {
    0xE1, 0xE7, 0x6E, 0x67, 0x77, 0x85, 0x1D, 0xD7, 0x74, 0x16, 0xBD, 0xDD, 0x35, 0xEC, 0x3C, 0x13, 0x7C, 0x47, 0x29, 0xDC,
};

extern const size_t sTestCert_Node02_04_AuthorityKeyId_Len = sizeof(sTestCert_Node02_04_AuthorityKeyId);

/**************  Test Node02_05 Certificate  **************
Certificate:
    Data:
        Version: 3 (0x2)
        Serial Number: 7874366387595048762 (0x6d475e8263e0f73a)
        Signature Algorithm: ecdsa-with-SHA256
        Issuer: 1.3.6.1.4.1.37244.1.3 = CACACACA00000004, 1.3.6.1.4.1.37244.1.5 = FAB000000000001D
        Validity
            Not Before: Oct 15 14:23:43 2020 GMT
            Not After : Oct 15 14:23:42 2040 GMT
        Subject: 1.3.6.1.4.1.37244.1.6 = ABCD0010, 1.3.6.1.4.1.37244.1.5 = FAB000000000001D, 1.3.6.1.4.1.37244.1.6 =
ABCE1008, 1.3.6.1.4.1.37244.1.1 = DEDEDEDE00020005 Subject Public Key Info: Public Key Algorithm: id-ecPublicKey Public-Key: (256
bit) pub: 04:fc:8a:ff:06:3a:d0:e0:bf:df:24:dd:9d:84:13: 0f:74:49:3c:95:a4:0d:b3:f4:0a:af:42:cf:2c:b1:
                    15:8b:a1:aa:1d:61:de:38:3c:9b:51:03:ca:f7:96:
                    43:1e:0a:4e:9c:5a:c2:d7:f0:e9:1c:c7:0e:7d:f0:
                    9d:b3:2c:f8:63
                ASN1 OID: prime256v1
                NIST CURVE: P-256
        X509v3 extensions:
            X509v3 Basic Constraints: critical
                CA:FALSE
            X509v3 Key Usage: critical
                Digital Signature
            X509v3 Extended Key Usage: critical
                TLS Web Client Authentication, TLS Web Server Authentication
            X509v3 Subject Key Identifier:
                4B:46:42:9C:69:3D:0E:28:79:8C:8F:76:26:A5:01:20:05:96:AD:2D
            X509v3 Authority Key Identifier:
                keyid:E1:E7:6E:67:77:85:1D:D7:74:16:BD:DD:35:EC:3C:13:7C:47:29:DC

            X509v3 Subject Alternative Name:
                email:test@chip.org
    Signature Algorithm: ecdsa-with-SHA256
         30:46:02:21:00:ff:06:6b:df:4a:4a:93:0c:3b:a8:ef:82:8c:
         5f:93:fe:7e:bb:dd:5b:b4:ec:54:45:bd:d7:78:62:d5:a2:1d:
         43:02:21:00:9e:3c:3b:ec:bc:b3:f1:aa:bd:97:86:10:19:a2:
         e9:45:c9:94:d2:73:60:4f:60:f0:2d:ed:3b:6a:c5:be:07:5d

-----BEGIN CERTIFICATE-----
MIICUTCCAfagAwIBAgIIbUdegmPg9zowCgYIKoZIzj0EAwIwRDEgMB4GCisGAQQB
gqJ8AQMMEENBQ0FDQUNBMDAwMDAwMDQxIDAeBgorBgEEAYKifAEFDBBGQUIwMDAw
MDAwMDAwMDFEMB4XDTIwMTAxNTE0MjM0M1oXDTQwMTAxNTE0MjM0MloweDEYMBYG
CisGAQQBgqJ8AQYMCEFCQ0QwMDEwMSAwHgYKKwYBBAGConwBBQwQRkFCMDAwMDAw
MDAwMDAxRDEYMBYGCisGAQQBgqJ8AQYMCEFCQ0UxMDA4MSAwHgYKKwYBBAGConwB
AQwQREVERURFREUwMDAyMDAwNTBZMBMGByqGSM49AgEGCCqGSM49AwEHA0IABPyK
/wY60OC/3yTdnYQTD3RJPJWkDbP0Cq9CzyyxFYuhqh1h3jg8m1EDyveWQx4KTpxa
wtfw6RzHDn3wnbMs+GOjgZ0wgZowDAYDVR0TAQH/BAIwADAOBgNVHQ8BAf8EBAMC
B4AwIAYDVR0lAQH/BBYwFAYIKwYBBQUHAwIGCCsGAQUFBwMBMB0GA1UdDgQWBBRL
RkKcaT0OKHmMj3YmpQEgBZatLTAfBgNVHSMEGDAWgBTh525nd4Ud13QWvd017DwT
fEcp3DAYBgNVHREEETAPgQ10ZXN0QGNoaXAub3JnMAoGCCqGSM49BAMCA0kAMEYC
IQD/BmvfSkqTDDuo74KMX5P+frvdW7TsVEW913hi1aIdQwIhAJ48O+y8s/GqvZeG
EBmi6UXJlNJzYE9g8C3tO2rFvgdd
-----END CERTIFICATE-----

-----BEGIN EC PRIVATE KEY-----
MHcCAQEEIGbQQJBxdmC6r8zcRkdtqR2tZuitOSg9sX0HedrQzWXCoAoGCCqGSM49
AwEHoUQDQgAE/Ir/BjrQ4L/fJN2dhBMPdEk8laQNs/QKr0LPLLEVi6GqHWHeODyb
UQPK95ZDHgpOnFrC1/DpHMcOffCdsyz4Yw==
-----END EC PRIVATE KEY-----
*/

extern const uint8_t sTestCert_Node02_05_Chip[] = {
    0x15, 0x30, 0x01, 0x08, 0x6d, 0x47, 0x5e, 0x82, 0x63, 0xe0, 0xf7, 0x3a, 0x24, 0x02, 0x01, 0x37, 0x03, 0x27, 0x13, 0x04,
    0x00, 0x00, 0x00, 0xca, 0xca, 0xca, 0xca, 0x27, 0x15, 0x1d, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0, 0xfa, 0x18, 0x26, 0x04,
    0xef, 0x17, 0x1b, 0x27, 0x26, 0x05, 0x6e, 0xb5, 0xb9, 0x4c, 0x37, 0x06, 0x26, 0x16, 0x10, 0x00, 0xcd, 0xab, 0x27, 0x15,
    0x1d, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0, 0xfa, 0x26, 0x16, 0x08, 0x10, 0xce, 0xab, 0x27, 0x11, 0x05, 0x00, 0x02, 0x00,
    0xde, 0xde, 0xde, 0xde, 0x18, 0x24, 0x07, 0x01, 0x24, 0x08, 0x01, 0x30, 0x09, 0x41, 0x04, 0xfc, 0x8a, 0xff, 0x06, 0x3a,
    0xd0, 0xe0, 0xbf, 0xdf, 0x24, 0xdd, 0x9d, 0x84, 0x13, 0x0f, 0x74, 0x49, 0x3c, 0x95, 0xa4, 0x0d, 0xb3, 0xf4, 0x0a, 0xaf,
    0x42, 0xcf, 0x2c, 0xb1, 0x15, 0x8b, 0xa1, 0xaa, 0x1d, 0x61, 0xde, 0x38, 0x3c, 0x9b, 0x51, 0x03, 0xca, 0xf7, 0x96, 0x43,
    0x1e, 0x0a, 0x4e, 0x9c, 0x5a, 0xc2, 0xd7, 0xf0, 0xe9, 0x1c, 0xc7, 0x0e, 0x7d, 0xf0, 0x9d, 0xb3, 0x2c, 0xf8, 0x63, 0x37,
    0x0a, 0x35, 0x01, 0x28, 0x01, 0x18, 0x24, 0x02, 0x01, 0x36, 0x03, 0x04, 0x02, 0x04, 0x01, 0x18, 0x30, 0x04, 0x14, 0x4b,
    0x46, 0x42, 0x9c, 0x69, 0x3d, 0x0e, 0x28, 0x79, 0x8c, 0x8f, 0x76, 0x26, 0xa5, 0x01, 0x20, 0x05, 0x96, 0xad, 0x2d, 0x30,
    0x05, 0x14, 0xe1, 0xe7, 0x6e, 0x67, 0x77, 0x85, 0x1d, 0xd7, 0x74, 0x16, 0xbd, 0xdd, 0x35, 0xec, 0x3c, 0x13, 0x7c, 0x47,
    0x29, 0xdc, 0x30, 0x06, 0x1a, 0x30, 0x18, 0x06, 0x03, 0x55, 0x1d, 0x11, 0x04, 0x11, 0x30, 0x0f, 0x81, 0x0d, 0x74, 0x65,
    0x73, 0x74, 0x40, 0x63, 0x68, 0x69, 0x70, 0x2e, 0x6f, 0x72, 0x67, 0x18, 0x30, 0x0b, 0x40, 0xff, 0x06, 0x6b, 0xdf, 0x4a,
    0x4a, 0x93, 0x0c, 0x3b, 0xa8, 0xef, 0x82, 0x8c, 0x5f, 0x93, 0xfe, 0x7e, 0xbb, 0xdd, 0x5b, 0xb4, 0xec, 0x54, 0x45, 0xbd,
    0xd7, 0x78, 0x62, 0xd5, 0xa2, 0x1d, 0x43, 0x9e, 0x3c, 0x3b, 0xec, 0xbc, 0xb3, 0xf1, 0xaa, 0xbd, 0x97, 0x86, 0x10, 0x19,
    0xa2, 0xe9, 0x45, 0xc9, 0x94, 0xd2, 0x73, 0x60, 0x4f, 0x60, 0xf0, 0x2d, 0xed, 0x3b, 0x6a, 0xc5, 0xbe, 0x07, 0x5d, 0x18,
};

extern const size_t sTestCert_Node02_05_Chip_Len = sizeof(sTestCert_Node02_05_Chip);

extern const uint8_t sTestCert_Node02_05_DER[] = {
    0x30, 0x82, 0x02, 0x51, 0x30, 0x82, 0x01, 0xf6, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x6d, 0x47, 0x5e, 0x82, 0x63, 0xe0,
    0xf7, 0x3a, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x44, 0x31, 0x20, 0x30, 0x1e, 0x06,
    0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x03, 0x0c, 0x10, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41,
    0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x34, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2,
    0x7c, 0x01, 0x05, 0x0c, 0x10, 0x46, 0x41, 0x42, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x31, 0x44,
    0x30, 0x1e, 0x17, 0x0d, 0x32, 0x30, 0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x17, 0x0d, 0x34, 0x30,
    0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32, 0x33, 0x34, 0x32, 0x5a, 0x30, 0x78, 0x31, 0x18, 0x30, 0x16, 0x06, 0x0a, 0x2b, 0x06,
    0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x06, 0x0c, 0x08, 0x41, 0x42, 0x43, 0x44, 0x30, 0x30, 0x31, 0x30, 0x31, 0x20, 0x30,
    0x1e, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x05, 0x0c, 0x10, 0x46, 0x41, 0x42, 0x30, 0x30, 0x30,
    0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x31, 0x44, 0x31, 0x18, 0x30, 0x16, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01,
    0x82, 0xa2, 0x7c, 0x01, 0x06, 0x0c, 0x08, 0x41, 0x42, 0x43, 0x45, 0x31, 0x30, 0x30, 0x38, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a,
    0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x01, 0x0c, 0x10, 0x44, 0x45, 0x44, 0x45, 0x44, 0x45, 0x44, 0x45, 0x30,
    0x30, 0x30, 0x32, 0x30, 0x30, 0x30, 0x35, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06,
    0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0xfc, 0x8a, 0xff, 0x06, 0x3a, 0xd0, 0xe0, 0xbf,
    0xdf, 0x24, 0xdd, 0x9d, 0x84, 0x13, 0x0f, 0x74, 0x49, 0x3c, 0x95, 0xa4, 0x0d, 0xb3, 0xf4, 0x0a, 0xaf, 0x42, 0xcf, 0x2c, 0xb1,
    0x15, 0x8b, 0xa1, 0xaa, 0x1d, 0x61, 0xde, 0x38, 0x3c, 0x9b, 0x51, 0x03, 0xca, 0xf7, 0x96, 0x43, 0x1e, 0x0a, 0x4e, 0x9c, 0x5a,
    0xc2, 0xd7, 0xf0, 0xe9, 0x1c, 0xc7, 0x0e, 0x7d, 0xf0, 0x9d, 0xb3, 0x2c, 0xf8, 0x63, 0xa3, 0x81, 0x9d, 0x30, 0x81, 0x9a, 0x30,
    0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01,
    0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x20, 0x06, 0x03, 0x55, 0x1d, 0x25, 0x01, 0x01, 0xff, 0x04, 0x16, 0x30,
    0x14, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x02, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x01,
    0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x4b, 0x46, 0x42, 0x9c, 0x69, 0x3d, 0x0e, 0x28, 0x79, 0x8c,
    0x8f, 0x76, 0x26, 0xa5, 0x01, 0x20, 0x05, 0x96, 0xad, 0x2d, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16,
    0x80, 0x14, 0xe1, 0xe7, 0x6e, 0x67, 0x77, 0x85, 0x1d, 0xd7, 0x74, 0x16, 0xbd, 0xdd, 0x35, 0xec, 0x3c, 0x13, 0x7c, 0x47, 0x29,
    0xdc, 0x30, 0x18, 0x06, 0x03, 0x55, 0x1d, 0x11, 0x04, 0x11, 0x30, 0x0f, 0x81, 0x0d, 0x74, 0x65, 0x73, 0x74, 0x40, 0x63, 0x68,
    0x69, 0x70, 0x2e, 0x6f, 0x72, 0x67, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x49, 0x00,
    0x30, 0x46, 0x02, 0x21, 0x00, 0xff, 0x06, 0x6b, 0xdf, 0x4a, 0x4a, 0x93, 0x0c, 0x3b, 0xa8, 0xef, 0x82, 0x8c, 0x5f, 0x93, 0xfe,
    0x7e, 0xbb, 0xdd, 0x5b, 0xb4, 0xec, 0x54, 0x45, 0xbd, 0xd7, 0x78, 0x62, 0xd5, 0xa2, 0x1d, 0x43, 0x02, 0x21, 0x00, 0x9e, 0x3c,
    0x3b, 0xec, 0xbc, 0xb3, 0xf1, 0xaa, 0xbd, 0x97, 0x86, 0x10, 0x19, 0xa2, 0xe9, 0x45, 0xc9, 0x94, 0xd2, 0x73, 0x60, 0x4f, 0x60,
    0xf0, 0x2d, 0xed, 0x3b, 0x6a, 0xc5, 0xbe, 0x07, 0x5d,
};

extern const size_t sTestCert_Node02_05_DER_Len = sizeof(sTestCert_Node02_05_DER);

extern const uint8_t sTestCert_Node02_05_PublicKey[] = {
    0x04, 0xfc, 0x8a, 0xff, 0x06, 0x3a, 0xd0, 0xe0, 0xbf, 0xdf, 0x24, 0xdd, 0x9d, 0x84, 0x13, 0x0f, 0x74,
    0x49, 0x3c, 0x95, 0xa4, 0x0d, 0xb3, 0xf4, 0x0a, 0xaf, 0x42, 0xcf, 0x2c, 0xb1, 0x15, 0x8b, 0xa1, 0xaa,
    0x1d, 0x61, 0xde, 0x38, 0x3c, 0x9b, 0x51, 0x03, 0xca, 0xf7, 0x96, 0x43, 0x1e, 0x0a, 0x4e, 0x9c, 0x5a,
    0xc2, 0xd7, 0xf0, 0xe9, 0x1c, 0xc7, 0x0e, 0x7d, 0xf0, 0x9d, 0xb3, 0x2c, 0xf8, 0x63,
};

extern const size_t sTestCert_Node02_05_PublicKey_Len = sizeof(sTestCert_Node02_05_PublicKey);

extern const uint8_t sTestCert_Node02_05_PrivateKey[] = {
    0x66, 0xd0, 0x40, 0x90, 0x71, 0x76, 0x60, 0xba, 0xaf, 0xcc, 0xdc, 0x46, 0x47, 0x6d, 0xa9, 0x1d,
    0xad, 0x66, 0xe8, 0xad, 0x39, 0x28, 0x3d, 0xb1, 0x7d, 0x07, 0x79, 0xda, 0xd0, 0xcd, 0x65, 0xc2,
};

extern const size_t sTestCert_Node02_05_PrivateKey_Len = sizeof(sTestCert_Node02_05_PrivateKey);

extern const uint8_t sTestCert_Node02_05_SubjectKeyId[] = {
    0x4B, 0x46, 0x42, 0x9C, 0x69, 0x3D, 0x0E, 0x28, 0x79, 0x8C, 0x8F, 0x76, 0x26, 0xA5, 0x01, 0x20, 0x05, 0x96, 0xAD, 0x2D,
};

extern const size_t sTestCert_Node02_05_SubjectKeyId_Len = sizeof(sTestCert_Node02_05_SubjectKeyId);

extern const uint8_t sTestCert_Node02_05_AuthorityKeyId[] = {
    0xE1, 0xE7, 0x6E, 0x67, 0x77, 0x85, 0x1D, 0xD7, 0x74, 0x16, 0xBD, 0xDD, 0x35, 0xEC, 0x3C, 0x13, 0x7C, 0x47, 0x29, 0xDC,
};

extern const size_t sTestCert_Node02_05_AuthorityKeyId_Len = sizeof(sTestCert_Node02_05_AuthorityKeyId);

/**************  Test Node02_06 Certificate  **************
Certificate:
    Data:
        Version: 3 (0x2)
        Serial Number: 1159553127949419850 (0x10178f6d219e454a)
        Signature Algorithm: ecdsa-with-SHA256
        Issuer: 1.3.6.1.4.1.37244.1.3 = CACACACA00000004, 1.3.6.1.4.1.37244.1.5 = FAB000000000001D
        Validity
            Not Before: Oct 15 14:23:43 2020 GMT
            Not After : Oct 15 14:23:42 2040 GMT
        Subject: 1.3.6.1.4.1.37244.1.1 = DEDEDEDE00020006, 1.3.6.1.4.1.37244.1.5 = FAB000000000001D
        Subject Public Key Info:
            Public Key Algorithm: id-ecPublicKey
                Public-Key: (256 bit)
                pub:
                    04:bb:a4:15:69:52:63:e1:d9:dc:17:cc:32:2d:39:
                    5e:32:0d:a6:f1:7d:56:f2:02:18:16:0c:68:53:e6:
                    21:0f:41:d0:10:19:63:fe:e3:91:5b:9a:8c:71:76:
                    24:df:34:2d:13:86:6d:dd:17:5b:42:8b:c3:8d:21:
                    3d:6b:f5:c7:d3
                ASN1 OID: prime256v1
                NIST CURVE: P-256
        X509v3 extensions:
            X509v3 Basic Constraints: critical
                CA:FALSE
            X509v3 Key Usage: critical
                Digital Signature
            X509v3 Extended Key Usage: critical
                TLS Web Client Authentication, TLS Web Server Authentication
            X509v3 Subject Key Identifier:
                4B:04:1B:79:7E:B8:1B:32:61:9A:AD:60:0F:6C:FD:78:CE:04:74:28
            X509v3 Authority Key Identifier:
                keyid:E1:E7:6E:67:77:85:1D:D7:74:16:BD:DD:35:EC:3C:13:7C:47:29:DC

            X509v3 Subject Alternative Name:
                email:test@chip.org
            Authority Information Access:
                OCSP - URI:test

    Signature Algorithm: ecdsa-with-SHA256
         30:45:02:20:1f:e0:bd:fd:10:8b:88:3d:1a:70:12:9e:31:85:
         a3:7a:9e:96:01:df:62:63:f2:0a:0a:8c:fe:e3:14:04:f6:79:
         02:21:00:d1:85:b0:21:dd:f2:b7:a1:6c:d6:af:6c:38:ee:92:
         04:03:ec:b4:d6:16:05:f3:c2:f7:f1:7b:d6:16:78:42:93

-----BEGIN CERTIFICATE-----
MIICPjCCAeSgAwIBAgIIEBePbSGeRUowCgYIKoZIzj0EAwIwRDEgMB4GCisGAQQB
gqJ8AQMMEENBQ0FDQUNBMDAwMDAwMDQxIDAeBgorBgEEAYKifAEFDBBGQUIwMDAw
MDAwMDAwMDFEMB4XDTIwMTAxNTE0MjM0M1oXDTQwMTAxNTE0MjM0MlowRDEgMB4G
CisGAQQBgqJ8AQEMEERFREVERURFMDAwMjAwMDYxIDAeBgorBgEEAYKifAEFDBBG
QUIwMDAwMDAwMDAwMDFEMFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAEu6QVaVJj
4dncF8wyLTleMg2m8X1W8gIYFgxoU+YhD0HQEBlj/uORW5qMcXYk3zQtE4Zt3Rdb
QovDjSE9a/XH06OBvzCBvDAMBgNVHRMBAf8EAjAAMA4GA1UdDwEB/wQEAwIHgDAg
BgNVHSUBAf8EFjAUBggrBgEFBQcDAgYIKwYBBQUHAwEwHQYDVR0OBBYEFEsEG3l+
uBsyYZqtYA9s/XjOBHQoMB8GA1UdIwQYMBaAFOHnbmd3hR3XdBa93TXsPBN8Rync
MBgGA1UdEQQRMA+BDXRlc3RAY2hpcC5vcmcwIAYIKwYBBQUHAQEEFDASMBAGCCsG
AQUFBzABhgR0ZXN0MAoGCCqGSM49BAMCA0gAMEUCIB/gvf0Qi4g9GnASnjGFo3qe
lgHfYmPyCgqM/uMUBPZ5AiEA0YWwId3yt6Fs1q9sOO6SBAPstNYWBfPC9/F71hZ4
QpM=
-----END CERTIFICATE-----

-----BEGIN EC PRIVATE KEY-----
MHcCAQEEIMCX/fl2O3KOkYpURsST8zO/HW8zpqgDrxHagcX17lWOoAoGCCqGSM49
AwEHoUQDQgAEu6QVaVJj4dncF8wyLTleMg2m8X1W8gIYFgxoU+YhD0HQEBlj/uOR
W5qMcXYk3zQtE4Zt3RdbQovDjSE9a/XH0w==
-----END EC PRIVATE KEY-----
*/

extern const uint8_t sTestCert_Node02_06_Chip[] = {
    0x15, 0x30, 0x01, 0x08, 0x10, 0x17, 0x8f, 0x6d, 0x21, 0x9e, 0x45, 0x4a, 0x24, 0x02, 0x01, 0x37, 0x03, 0x27, 0x13, 0x04, 0x00,
    0x00, 0x00, 0xca, 0xca, 0xca, 0xca, 0x27, 0x15, 0x1d, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0, 0xfa, 0x18, 0x26, 0x04, 0xef, 0x17,
    0x1b, 0x27, 0x26, 0x05, 0x6e, 0xb5, 0xb9, 0x4c, 0x37, 0x06, 0x27, 0x11, 0x06, 0x00, 0x02, 0x00, 0xde, 0xde, 0xde, 0xde, 0x27,
    0x15, 0x1d, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0, 0xfa, 0x18, 0x24, 0x07, 0x01, 0x24, 0x08, 0x01, 0x30, 0x09, 0x41, 0x04, 0xbb,
    0xa4, 0x15, 0x69, 0x52, 0x63, 0xe1, 0xd9, 0xdc, 0x17, 0xcc, 0x32, 0x2d, 0x39, 0x5e, 0x32, 0x0d, 0xa6, 0xf1, 0x7d, 0x56, 0xf2,
    0x02, 0x18, 0x16, 0x0c, 0x68, 0x53, 0xe6, 0x21, 0x0f, 0x41, 0xd0, 0x10, 0x19, 0x63, 0xfe, 0xe3, 0x91, 0x5b, 0x9a, 0x8c, 0x71,
    0x76, 0x24, 0xdf, 0x34, 0x2d, 0x13, 0x86, 0x6d, 0xdd, 0x17, 0x5b, 0x42, 0x8b, 0xc3, 0x8d, 0x21, 0x3d, 0x6b, 0xf5, 0xc7, 0xd3,
    0x37, 0x0a, 0x35, 0x01, 0x28, 0x01, 0x18, 0x24, 0x02, 0x01, 0x36, 0x03, 0x04, 0x02, 0x04, 0x01, 0x18, 0x30, 0x04, 0x14, 0x4b,
    0x04, 0x1b, 0x79, 0x7e, 0xb8, 0x1b, 0x32, 0x61, 0x9a, 0xad, 0x60, 0x0f, 0x6c, 0xfd, 0x78, 0xce, 0x04, 0x74, 0x28, 0x30, 0x05,
    0x14, 0xe1, 0xe7, 0x6e, 0x67, 0x77, 0x85, 0x1d, 0xd7, 0x74, 0x16, 0xbd, 0xdd, 0x35, 0xec, 0x3c, 0x13, 0x7c, 0x47, 0x29, 0xdc,
    0x30, 0x06, 0x1a, 0x30, 0x18, 0x06, 0x03, 0x55, 0x1d, 0x11, 0x04, 0x11, 0x30, 0x0f, 0x81, 0x0d, 0x74, 0x65, 0x73, 0x74, 0x40,
    0x63, 0x68, 0x69, 0x70, 0x2e, 0x6f, 0x72, 0x67, 0x30, 0x06, 0x22, 0x30, 0x20, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07,
    0x01, 0x01, 0x04, 0x14, 0x30, 0x12, 0x30, 0x10, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x01, 0x86, 0x04, 0x74,
    0x65, 0x73, 0x74, 0x18, 0x30, 0x0b, 0x40, 0x1f, 0xe0, 0xbd, 0xfd, 0x10, 0x8b, 0x88, 0x3d, 0x1a, 0x70, 0x12, 0x9e, 0x31, 0x85,
    0xa3, 0x7a, 0x9e, 0x96, 0x01, 0xdf, 0x62, 0x63, 0xf2, 0x0a, 0x0a, 0x8c, 0xfe, 0xe3, 0x14, 0x04, 0xf6, 0x79, 0xd1, 0x85, 0xb0,
    0x21, 0xdd, 0xf2, 0xb7, 0xa1, 0x6c, 0xd6, 0xaf, 0x6c, 0x38, 0xee, 0x92, 0x04, 0x03, 0xec, 0xb4, 0xd6, 0x16, 0x05, 0xf3, 0xc2,
    0xf7, 0xf1, 0x7b, 0xd6, 0x16, 0x78, 0x42, 0x93, 0x18,
};

extern const size_t sTestCert_Node02_06_Chip_Len = sizeof(sTestCert_Node02_06_Chip);

extern const uint8_t sTestCert_Node02_06_DER[] = {
    0x30, 0x82, 0x02, 0x3e, 0x30, 0x82, 0x01, 0xe4, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x10, 0x17, 0x8f, 0x6d, 0x21, 0x9e,
    0x45, 0x4a, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x44, 0x31, 0x20, 0x30, 0x1e, 0x06,
    0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x03, 0x0c, 0x10, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41,
    0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x34, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2,
    0x7c, 0x01, 0x05, 0x0c, 0x10, 0x46, 0x41, 0x42, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x31, 0x44,
    0x30, 0x1e, 0x17, 0x0d, 0x32, 0x30, 0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x17, 0x0d, 0x34, 0x30,
    0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32, 0x33, 0x34, 0x32, 0x5a, 0x30, 0x44, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06,
    0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x01, 0x0c, 0x10, 0x44, 0x45, 0x44, 0x45, 0x44, 0x45, 0x44, 0x45, 0x30, 0x30, 0x30,
    0x32, 0x30, 0x30, 0x30, 0x36, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x05,
    0x0c, 0x10, 0x46, 0x41, 0x42, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x31, 0x44, 0x30, 0x59, 0x30,
    0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03,
    0x42, 0x00, 0x04, 0xbb, 0xa4, 0x15, 0x69, 0x52, 0x63, 0xe1, 0xd9, 0xdc, 0x17, 0xcc, 0x32, 0x2d, 0x39, 0x5e, 0x32, 0x0d, 0xa6,
    0xf1, 0x7d, 0x56, 0xf2, 0x02, 0x18, 0x16, 0x0c, 0x68, 0x53, 0xe6, 0x21, 0x0f, 0x41, 0xd0, 0x10, 0x19, 0x63, 0xfe, 0xe3, 0x91,
    0x5b, 0x9a, 0x8c, 0x71, 0x76, 0x24, 0xdf, 0x34, 0x2d, 0x13, 0x86, 0x6d, 0xdd, 0x17, 0x5b, 0x42, 0x8b, 0xc3, 0x8d, 0x21, 0x3d,
    0x6b, 0xf5, 0xc7, 0xd3, 0xa3, 0x81, 0xbf, 0x30, 0x81, 0xbc, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04,
    0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x20,
    0x06, 0x03, 0x55, 0x1d, 0x25, 0x01, 0x01, 0xff, 0x04, 0x16, 0x30, 0x14, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03,
    0x02, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x01, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04,
    0x14, 0x4b, 0x04, 0x1b, 0x79, 0x7e, 0xb8, 0x1b, 0x32, 0x61, 0x9a, 0xad, 0x60, 0x0f, 0x6c, 0xfd, 0x78, 0xce, 0x04, 0x74, 0x28,
    0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xe1, 0xe7, 0x6e, 0x67, 0x77, 0x85, 0x1d, 0xd7,
    0x74, 0x16, 0xbd, 0xdd, 0x35, 0xec, 0x3c, 0x13, 0x7c, 0x47, 0x29, 0xdc, 0x30, 0x18, 0x06, 0x03, 0x55, 0x1d, 0x11, 0x04, 0x11,
    0x30, 0x0f, 0x81, 0x0d, 0x74, 0x65, 0x73, 0x74, 0x40, 0x63, 0x68, 0x69, 0x70, 0x2e, 0x6f, 0x72, 0x67, 0x30, 0x20, 0x06, 0x08,
    0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x01, 0x01, 0x04, 0x14, 0x30, 0x12, 0x30, 0x10, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05,
    0x07, 0x30, 0x01, 0x86, 0x04, 0x74, 0x65, 0x73, 0x74, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02,
    0x03, 0x48, 0x00, 0x30, 0x45, 0x02, 0x20, 0x1f, 0xe0, 0xbd, 0xfd, 0x10, 0x8b, 0x88, 0x3d, 0x1a, 0x70, 0x12, 0x9e, 0x31, 0x85,
    0xa3, 0x7a, 0x9e, 0x96, 0x01, 0xdf, 0x62, 0x63, 0xf2, 0x0a, 0x0a, 0x8c, 0xfe, 0xe3, 0x14, 0x04, 0xf6, 0x79, 0x02, 0x21, 0x00,
    0xd1, 0x85, 0xb0, 0x21, 0xdd, 0xf2, 0xb7, 0xa1, 0x6c, 0xd6, 0xaf, 0x6c, 0x38, 0xee, 0x92, 0x04, 0x03, 0xec, 0xb4, 0xd6, 0x16,
    0x05, 0xf3, 0xc2, 0xf7, 0xf1, 0x7b, 0xd6, 0x16, 0x78, 0x42, 0x93,
};

extern const size_t sTestCert_Node02_06_DER_Len = sizeof(sTestCert_Node02_06_DER);

extern const uint8_t sTestCert_Node02_06_PublicKey[] = {
    0x04, 0xbb, 0xa4, 0x15, 0x69, 0x52, 0x63, 0xe1, 0xd9, 0xdc, 0x17, 0xcc, 0x32, 0x2d, 0x39, 0x5e, 0x32,
    0x0d, 0xa6, 0xf1, 0x7d, 0x56, 0xf2, 0x02, 0x18, 0x16, 0x0c, 0x68, 0x53, 0xe6, 0x21, 0x0f, 0x41, 0xd0,
    0x10, 0x19, 0x63, 0xfe, 0xe3, 0x91, 0x5b, 0x9a, 0x8c, 0x71, 0x76, 0x24, 0xdf, 0x34, 0x2d, 0x13, 0x86,
    0x6d, 0xdd, 0x17, 0x5b, 0x42, 0x8b, 0xc3, 0x8d, 0x21, 0x3d, 0x6b, 0xf5, 0xc7, 0xd3,
};

extern const size_t sTestCert_Node02_06_PublicKey_Len = sizeof(sTestCert_Node02_06_PublicKey);

extern const uint8_t sTestCert_Node02_06_PrivateKey[] = {
    0xc0, 0x97, 0xfd, 0xf9, 0x76, 0x3b, 0x72, 0x8e, 0x91, 0x8a, 0x54, 0x46, 0xc4, 0x93, 0xf3, 0x33,
    0xbf, 0x1d, 0x6f, 0x33, 0xa6, 0xa8, 0x03, 0xaf, 0x11, 0xda, 0x81, 0xc5, 0xf5, 0xee, 0x55, 0x8e,
};

extern const size_t sTestCert_Node02_06_PrivateKey_Len = sizeof(sTestCert_Node02_06_PrivateKey);

extern const uint8_t sTestCert_Node02_06_SubjectKeyId[] = {
    0x4B, 0x04, 0x1B, 0x79, 0x7E, 0xB8, 0x1B, 0x32, 0x61, 0x9A, 0xAD, 0x60, 0x0F, 0x6C, 0xFD, 0x78, 0xCE, 0x04, 0x74, 0x28,
};

extern const size_t sTestCert_Node02_06_SubjectKeyId_Len = sizeof(sTestCert_Node02_06_SubjectKeyId);

extern const uint8_t sTestCert_Node02_06_AuthorityKeyId[] = {
    0xE1, 0xE7, 0x6E, 0x67, 0x77, 0x85, 0x1D, 0xD7, 0x74, 0x16, 0xBD, 0xDD, 0x35, 0xEC, 0x3C, 0x13, 0x7C, 0x47, 0x29, 0xDC,
};

extern const size_t sTestCert_Node02_06_AuthorityKeyId_Len = sizeof(sTestCert_Node02_06_AuthorityKeyId);

/**************  Test Node02_07 Certificate  **************
Certificate:
    Data:
        Version: 3 (0x2)
        Serial Number: 6951342034215031142 (0x607820c015a0d166)
        Signature Algorithm: ecdsa-with-SHA256
        Issuer: 1.3.6.1.4.1.37244.1.3 = CACACACA00000004, 1.3.6.1.4.1.37244.1.5 = FAB000000000001D
        Validity
            Not Before: Oct 15 14:23:43 2020 GMT
            Not After : Oct 15 14:23:42 2040 GMT
        Subject: 1.3.6.1.4.1.37244.1.1 = DEDEDEDE00020007, 1.3.6.1.4.1.37244.1.5 = FAB000000000001D
        Subject Public Key Info:
            Public Key Algorithm: id-ecPublicKey
                Public-Key: (256 bit)
                pub:
                    04:f7:3c:7f:e5:75:d8:b9:06:d4:75:ff:f7:91:2e:
                    e0:e1:b3:cd:23:6c:32:46:cf:2d:85:3d:e3:39:84:
                    c1:f4:18:17:b1:c5:b5:28:01:8f:90:e6:26:16:ea:
                    a1:87:7f:47:14:b9:6f:c8:78:6f:91:b3:03:e4:46:
                    a9:65:09:d3:61
                ASN1 OID: prime256v1
                NIST CURVE: P-256
        X509v3 extensions:
            X509v3 Basic Constraints: critical
                CA:FALSE
            X509v3 Key Usage: critical
                Digital Signature
            X509v3 Extended Key Usage: critical
                TLS Web Client Authentication, TLS Web Server Authentication
            X509v3 Subject Key Identifier:
                77:7C:77:B6:35:65:DC:51:F3:02:04:59:63:C5:CE:FC:E7:09:2A:1E
            X509v3 Authority Key Identifier:
                keyid:E1:E7:6E:67:77:85:1D:D7:74:16:BD:DD:35:EC:3C:13:7C:47:29:DC

            X509v3 Subject Alternative Name: critical
                email:test@chip.org
            Authority Information Access:
                OCSP - URI:test

    Signature Algorithm: ecdsa-with-SHA256
         30:45:02:21:00:a7:c5:93:9d:db:18:9c:d8:4c:1e:85:48:bc:
         40:c2:c3:af:cd:98:19:54:8c:f8:e5:c2:49:d2:7d:aa:5e:d8:
         86:02:20:19:70:46:95:b7:10:50:fe:70:1a:5a:67:5a:49:55:
         98:1c:28:7f:25:f6:73:93:8b:43:e7:71:73:54:f6:c8:2e

-----BEGIN CERTIFICATE-----
MIICQTCCAeegAwIBAgIIYHggwBWg0WYwCgYIKoZIzj0EAwIwRDEgMB4GCisGAQQB
gqJ8AQMMEENBQ0FDQUNBMDAwMDAwMDQxIDAeBgorBgEEAYKifAEFDBBGQUIwMDAw
MDAwMDAwMDFEMB4XDTIwMTAxNTE0MjM0M1oXDTQwMTAxNTE0MjM0MlowRDEgMB4G
CisGAQQBgqJ8AQEMEERFREVERURFMDAwMjAwMDcxIDAeBgorBgEEAYKifAEFDBBG
QUIwMDAwMDAwMDAwMDFEMFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAE9zx/5XXY
uQbUdf/3kS7g4bPNI2wyRs8thT3jOYTB9BgXscW1KAGPkOYmFuqhh39HFLlvyHhv
kbMD5EapZQnTYaOBwjCBvzAMBgNVHRMBAf8EAjAAMA4GA1UdDwEB/wQEAwIHgDAg
BgNVHSUBAf8EFjAUBggrBgEFBQcDAgYIKwYBBQUHAwEwHQYDVR0OBBYEFHd8d7Y1
ZdxR8wIEWWPFzvznCSoeMB8GA1UdIwQYMBaAFOHnbmd3hR3XdBa93TXsPBN8Rync
MBsGA1UdEQEB/wQRMA+BDXRlc3RAY2hpcC5vcmcwIAYIKwYBBQUHAQEEFDASMBAG
CCsGAQUFBzABhgR0ZXN0MAoGCCqGSM49BAMCA0gAMEUCIQCnxZOd2xic2EwehUi8
QMLDr82YGVSM+OXCSdJ9ql7YhgIgGXBGlbcQUP5wGlpnWklVmBwofyX2c5OLQ+dx
c1T2yC4=
-----END CERTIFICATE-----

-----BEGIN EC PRIVATE KEY-----
MHcCAQEEIJWoN3/bUlY4BMAQbRFLUnQFaTeczcj1doNHPb28RSVAoAoGCCqGSM49
AwEHoUQDQgAE9zx/5XXYuQbUdf/3kS7g4bPNI2wyRs8thT3jOYTB9BgXscW1KAGP
kOYmFuqhh39HFLlvyHhvkbMD5EapZQnTYQ==
-----END EC PRIVATE KEY-----
*/

extern const uint8_t sTestCert_Node02_07_Chip[] = {
    0x15, 0x30, 0x01, 0x08, 0x60, 0x78, 0x20, 0xc0, 0x15, 0xa0, 0xd1, 0x66, 0x24, 0x02, 0x01, 0x37, 0x03, 0x27, 0x13, 0x04, 0x00,
    0x00, 0x00, 0xca, 0xca, 0xca, 0xca, 0x27, 0x15, 0x1d, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0, 0xfa, 0x18, 0x26, 0x04, 0xef, 0x17,
    0x1b, 0x27, 0x26, 0x05, 0x6e, 0xb5, 0xb9, 0x4c, 0x37, 0x06, 0x27, 0x11, 0x07, 0x00, 0x02, 0x00, 0xde, 0xde, 0xde, 0xde, 0x27,
    0x15, 0x1d, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0, 0xfa, 0x18, 0x24, 0x07, 0x01, 0x24, 0x08, 0x01, 0x30, 0x09, 0x41, 0x04, 0xf7,
    0x3c, 0x7f, 0xe5, 0x75, 0xd8, 0xb9, 0x06, 0xd4, 0x75, 0xff, 0xf7, 0x91, 0x2e, 0xe0, 0xe1, 0xb3, 0xcd, 0x23, 0x6c, 0x32, 0x46,
    0xcf, 0x2d, 0x85, 0x3d, 0xe3, 0x39, 0x84, 0xc1, 0xf4, 0x18, 0x17, 0xb1, 0xc5, 0xb5, 0x28, 0x01, 0x8f, 0x90, 0xe6, 0x26, 0x16,
    0xea, 0xa1, 0x87, 0x7f, 0x47, 0x14, 0xb9, 0x6f, 0xc8, 0x78, 0x6f, 0x91, 0xb3, 0x03, 0xe4, 0x46, 0xa9, 0x65, 0x09, 0xd3, 0x61,
    0x37, 0x0a, 0x35, 0x01, 0x28, 0x01, 0x18, 0x24, 0x02, 0x01, 0x36, 0x03, 0x04, 0x02, 0x04, 0x01, 0x18, 0x30, 0x04, 0x14, 0x77,
    0x7c, 0x77, 0xb6, 0x35, 0x65, 0xdc, 0x51, 0xf3, 0x02, 0x04, 0x59, 0x63, 0xc5, 0xce, 0xfc, 0xe7, 0x09, 0x2a, 0x1e, 0x30, 0x05,
    0x14, 0xe1, 0xe7, 0x6e, 0x67, 0x77, 0x85, 0x1d, 0xd7, 0x74, 0x16, 0xbd, 0xdd, 0x35, 0xec, 0x3c, 0x13, 0x7c, 0x47, 0x29, 0xdc,
    0x30, 0x06, 0x1d, 0x30, 0x1b, 0x06, 0x03, 0x55, 0x1d, 0x11, 0x01, 0x01, 0xff, 0x04, 0x11, 0x30, 0x0f, 0x81, 0x0d, 0x74, 0x65,
    0x73, 0x74, 0x40, 0x63, 0x68, 0x69, 0x70, 0x2e, 0x6f, 0x72, 0x67, 0x30, 0x06, 0x22, 0x30, 0x20, 0x06, 0x08, 0x2b, 0x06, 0x01,
    0x05, 0x05, 0x07, 0x01, 0x01, 0x04, 0x14, 0x30, 0x12, 0x30, 0x10, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x01,
    0x86, 0x04, 0x74, 0x65, 0x73, 0x74, 0x18, 0x30, 0x0b, 0x40, 0xa7, 0xc5, 0x93, 0x9d, 0xdb, 0x18, 0x9c, 0xd8, 0x4c, 0x1e, 0x85,
    0x48, 0xbc, 0x40, 0xc2, 0xc3, 0xaf, 0xcd, 0x98, 0x19, 0x54, 0x8c, 0xf8, 0xe5, 0xc2, 0x49, 0xd2, 0x7d, 0xaa, 0x5e, 0xd8, 0x86,
    0x19, 0x70, 0x46, 0x95, 0xb7, 0x10, 0x50, 0xfe, 0x70, 0x1a, 0x5a, 0x67, 0x5a, 0x49, 0x55, 0x98, 0x1c, 0x28, 0x7f, 0x25, 0xf6,
    0x73, 0x93, 0x8b, 0x43, 0xe7, 0x71, 0x73, 0x54, 0xf6, 0xc8, 0x2e, 0x18,
};

extern const size_t sTestCert_Node02_07_Chip_Len = sizeof(sTestCert_Node02_07_Chip);

extern const uint8_t sTestCert_Node02_07_DER[] = {
    0x30, 0x82, 0x02, 0x41, 0x30, 0x82, 0x01, 0xe7, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x60, 0x78, 0x20, 0xc0, 0x15, 0xa0,
    0xd1, 0x66, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x44, 0x31, 0x20, 0x30, 0x1e, 0x06,
    0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x03, 0x0c, 0x10, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41,
    0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x34, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2,
    0x7c, 0x01, 0x05, 0x0c, 0x10, 0x46, 0x41, 0x42, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x31, 0x44,
    0x30, 0x1e, 0x17, 0x0d, 0x32, 0x30, 0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x17, 0x0d, 0x34, 0x30,
    0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32, 0x33, 0x34, 0x32, 0x5a, 0x30, 0x44, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06,
    0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x01, 0x0c, 0x10, 0x44, 0x45, 0x44, 0x45, 0x44, 0x45, 0x44, 0x45, 0x30, 0x30, 0x30,
    0x32, 0x30, 0x30, 0x30, 0x37, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x05,
    0x0c, 0x10, 0x46, 0x41, 0x42, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x31, 0x44, 0x30, 0x59, 0x30,
    0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03,
    0x42, 0x00, 0x04, 0xf7, 0x3c, 0x7f, 0xe5, 0x75, 0xd8, 0xb9, 0x06, 0xd4, 0x75, 0xff, 0xf7, 0x91, 0x2e, 0xe0, 0xe1, 0xb3, 0xcd,
    0x23, 0x6c, 0x32, 0x46, 0xcf, 0x2d, 0x85, 0x3d, 0xe3, 0x39, 0x84, 0xc1, 0xf4, 0x18, 0x17, 0xb1, 0xc5, 0xb5, 0x28, 0x01, 0x8f,
    0x90, 0xe6, 0x26, 0x16, 0xea, 0xa1, 0x87, 0x7f, 0x47, 0x14, 0xb9, 0x6f, 0xc8, 0x78, 0x6f, 0x91, 0xb3, 0x03, 0xe4, 0x46, 0xa9,
    0x65, 0x09, 0xd3, 0x61, 0xa3, 0x81, 0xc2, 0x30, 0x81, 0xbf, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04,
    0x02, 0x30, 0x00, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x20,
    0x06, 0x03, 0x55, 0x1d, 0x25, 0x01, 0x01, 0xff, 0x04, 0x16, 0x30, 0x14, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03,
    0x02, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x01, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04,
    0x14, 0x77, 0x7c, 0x77, 0xb6, 0x35, 0x65, 0xdc, 0x51, 0xf3, 0x02, 0x04, 0x59, 0x63, 0xc5, 0xce, 0xfc, 0xe7, 0x09, 0x2a, 0x1e,
    0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xe1, 0xe7, 0x6e, 0x67, 0x77, 0x85, 0x1d, 0xd7,
    0x74, 0x16, 0xbd, 0xdd, 0x35, 0xec, 0x3c, 0x13, 0x7c, 0x47, 0x29, 0xdc, 0x30, 0x1b, 0x06, 0x03, 0x55, 0x1d, 0x11, 0x01, 0x01,
    0xff, 0x04, 0x11, 0x30, 0x0f, 0x81, 0x0d, 0x74, 0x65, 0x73, 0x74, 0x40, 0x63, 0x68, 0x69, 0x70, 0x2e, 0x6f, 0x72, 0x67, 0x30,
    0x20, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x01, 0x01, 0x04, 0x14, 0x30, 0x12, 0x30, 0x10, 0x06, 0x08, 0x2b, 0x06,
    0x01, 0x05, 0x05, 0x07, 0x30, 0x01, 0x86, 0x04, 0x74, 0x65, 0x73, 0x74, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d,
    0x04, 0x03, 0x02, 0x03, 0x48, 0x00, 0x30, 0x45, 0x02, 0x21, 0x00, 0xa7, 0xc5, 0x93, 0x9d, 0xdb, 0x18, 0x9c, 0xd8, 0x4c, 0x1e,
    0x85, 0x48, 0xbc, 0x40, 0xc2, 0xc3, 0xaf, 0xcd, 0x98, 0x19, 0x54, 0x8c, 0xf8, 0xe5, 0xc2, 0x49, 0xd2, 0x7d, 0xaa, 0x5e, 0xd8,
    0x86, 0x02, 0x20, 0x19, 0x70, 0x46, 0x95, 0xb7, 0x10, 0x50, 0xfe, 0x70, 0x1a, 0x5a, 0x67, 0x5a, 0x49, 0x55, 0x98, 0x1c, 0x28,
    0x7f, 0x25, 0xf6, 0x73, 0x93, 0x8b, 0x43, 0xe7, 0x71, 0x73, 0x54, 0xf6, 0xc8, 0x2e,
};

extern const size_t sTestCert_Node02_07_DER_Len = sizeof(sTestCert_Node02_07_DER);

extern const uint8_t sTestCert_Node02_07_PublicKey[] = {
    0x04, 0xf7, 0x3c, 0x7f, 0xe5, 0x75, 0xd8, 0xb9, 0x06, 0xd4, 0x75, 0xff, 0xf7, 0x91, 0x2e, 0xe0, 0xe1,
    0xb3, 0xcd, 0x23, 0x6c, 0x32, 0x46, 0xcf, 0x2d, 0x85, 0x3d, 0xe3, 0x39, 0x84, 0xc1, 0xf4, 0x18, 0x17,
    0xb1, 0xc5, 0xb5, 0x28, 0x01, 0x8f, 0x90, 0xe6, 0x26, 0x16, 0xea, 0xa1, 0x87, 0x7f, 0x47, 0x14, 0xb9,
    0x6f, 0xc8, 0x78, 0x6f, 0x91, 0xb3, 0x03, 0xe4, 0x46, 0xa9, 0x65, 0x09, 0xd3, 0x61,
};

extern const size_t sTestCert_Node02_07_PublicKey_Len = sizeof(sTestCert_Node02_07_PublicKey);

extern const uint8_t sTestCert_Node02_07_PrivateKey[] = {
    0x95, 0xa8, 0x37, 0x7f, 0xdb, 0x52, 0x56, 0x38, 0x04, 0xc0, 0x10, 0x6d, 0x11, 0x4b, 0x52, 0x74,
    0x05, 0x69, 0x37, 0x9c, 0xcd, 0xc8, 0xf5, 0x76, 0x83, 0x47, 0x3d, 0xbd, 0xbc, 0x45, 0x25, 0x40,
};

extern const size_t sTestCert_Node02_07_PrivateKey_Len = sizeof(sTestCert_Node02_07_PrivateKey);

extern const uint8_t sTestCert_Node02_07_SubjectKeyId[] = {
    0x77, 0x7C, 0x77, 0xB6, 0x35, 0x65, 0xDC, 0x51, 0xF3, 0x02, 0x04, 0x59, 0x63, 0xC5, 0xCE, 0xFC, 0xE7, 0x09, 0x2A, 0x1E,
};

extern const size_t sTestCert_Node02_07_SubjectKeyId_Len = sizeof(sTestCert_Node02_07_SubjectKeyId);

extern const uint8_t sTestCert_Node02_07_AuthorityKeyId[] = {
    0xE1, 0xE7, 0x6E, 0x67, 0x77, 0x85, 0x1D, 0xD7, 0x74, 0x16, 0xBD, 0xDD, 0x35, 0xEC, 0x3C, 0x13, 0x7C, 0x47, 0x29, 0xDC,
};

extern const size_t sTestCert_Node02_07_AuthorityKeyId_Len = sizeof(sTestCert_Node02_07_AuthorityKeyId);

/**************  Test Node02_08 Certificate  **************
Certificate:
    Data:
        Version: 3 (0x2)
        Serial Number: 4496725963772311573 (0x3e6794707aecb815)
        Signature Algorithm: ecdsa-with-SHA256
        Issuer: 1.3.6.1.4.1.37244.1.3 = CACACACA00000004, 1.3.6.1.4.1.37244.1.5 = FAB000000000001D
        Validity
            Not Before: Oct 15 14:23:43 2020 GMT
            Not After : Oct 15 14:23:42 2040 GMT
        Subject: 1.3.6.1.4.1.37244.1.6 = ABCF00A0, 1.3.6.1.4.1.37244.1.1 = DEDEDEDE00020008, 1.3.6.1.4.1.37244.1.6 =
ABCD0020, 1.3.6.1.4.1.37244.1.5 = FAB000000000001D, 1.3.6.1.4.1.37244.1.6 = ABCE0100 Subject Public Key Info: Public Key Algorithm:
id-ecPublicKey Public-Key: (256 bit) pub: 04:98:ca:97:34:da:af:f7:33:98:33:6d:c0:a4:de:
                    89:2d:e6:2a:1f:96:90:23:e0:33:70:86:00:85:dc:
                    dc:07:2b:23:72:60:79:37:ba:3a:34:4d:94:55:46:
                    b4:14:f2:23:d2:72:31:c9:a8:3d:71:b3:97:b4:32:
                    06:62:c0:f5:cb
                ASN1 OID: prime256v1
                NIST CURVE: P-256
        X509v3 extensions:
            X509v3 Basic Constraints: critical
                CA:FALSE
            X509v3 Key Usage: critical
                Digital Signature
            X509v3 Extended Key Usage: critical
                TLS Web Client Authentication, TLS Web Server Authentication
            X509v3 Subject Key Identifier:
                A9:C5:FD:6C:BD:38:BB:18:37:0F:8E:80:64:16:6A:FB:1E:C8:39:73
            X509v3 Authority Key Identifier:
                keyid:E1:E7:6E:67:77:85:1D:D7:74:16:BD:DD:35:EC:3C:13:7C:47:29:DC

    Signature Algorithm: ecdsa-with-SHA256
         30:46:02:21:00:bc:e8:c9:50:5a:b1:e4:b4:d1:d8:f2:c5:e5:
         26:37:a5:3e:f4:05:4a:9f:18:ef:45:94:3d:3d:23:58:fd:7e:
         b9:02:21:00:cb:12:15:04:a3:dc:1d:e4:2d:c0:ae:e4:f5:11:
         c0:05:67:7c:11:18:8b:44:98:78:bd:7d:69:3e:37:82:2c:47

-----BEGIN CERTIFICATE-----
MIICUjCCAfegAwIBAgIIPmeUcHrsuBUwCgYIKoZIzj0EAwIwRDEgMB4GCisGAQQB
gqJ8AQMMEENBQ0FDQUNBMDAwMDAwMDQxIDAeBgorBgEEAYKifAEFDBBGQUIwMDAw
MDAwMDAwMDFEMB4XDTIwMTAxNTE0MjM0M1oXDTQwMTAxNTE0MjM0MlowgZIxGDAW
BgorBgEEAYKifAEGDAhBQkNGMDBBMDEgMB4GCisGAQQBgqJ8AQEMEERFREVERURF
MDAwMjAwMDgxGDAWBgorBgEEAYKifAEGDAhBQkNEMDAyMDEgMB4GCisGAQQBgqJ8
AQUMEEZBQjAwMDAwMDAwMDAwMUQxGDAWBgorBgEEAYKifAEGDAhBQkNFMDEwMDBZ
MBMGByqGSM49AgEGCCqGSM49AwEHA0IABJjKlzTar/czmDNtwKTeiS3mKh+WkCPg
M3CGAIXc3AcrI3JgeTe6OjRNlFVGtBTyI9JyMcmoPXGzl7QyBmLA9cujgYMwgYAw
DAYDVR0TAQH/BAIwADAOBgNVHQ8BAf8EBAMCB4AwIAYDVR0lAQH/BBYwFAYIKwYB
BQUHAwIGCCsGAQUFBwMBMB0GA1UdDgQWBBSpxf1svTi7GDcPjoBkFmr7Hsg5czAf
BgNVHSMEGDAWgBTh525nd4Ud13QWvd017DwTfEcp3DAKBggqhkjOPQQDAgNJADBG
AiEAvOjJUFqx5LTR2PLF5SY3pT70BUqfGO9FlD09I1j9frkCIQDLEhUEo9wd5C3A
ruT1EcAFZ3wRGItEmHi9fWk+N4IsRw==
-----END CERTIFICATE-----

-----BEGIN EC PRIVATE KEY-----
MHcCAQEEIN+LEDQYW7Dtm8w5ta/A7JvVb0p2XemfaZuwmvbPOvdvoAoGCCqGSM49
AwEHoUQDQgAEmMqXNNqv9zOYM23ApN6JLeYqH5aQI+AzcIYAhdzcBysjcmB5N7o6
NE2UVUa0FPIj0nIxyag9cbOXtDIGYsD1yw==
-----END EC PRIVATE KEY-----
*/

extern const uint8_t sTestCert_Node02_08_Chip[] = {
    0x15, 0x30, 0x01, 0x08, 0x3e, 0x67, 0x94, 0x70, 0x7a, 0xec, 0xb8, 0x15, 0x24, 0x02, 0x01, 0x37, 0x03, 0x27, 0x13, 0x04,
    0x00, 0x00, 0x00, 0xca, 0xca, 0xca, 0xca, 0x27, 0x15, 0x1d, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0, 0xfa, 0x18, 0x26, 0x04,
    0xef, 0x17, 0x1b, 0x27, 0x26, 0x05, 0x6e, 0xb5, 0xb9, 0x4c, 0x37, 0x06, 0x26, 0x16, 0xa0, 0x00, 0xcf, 0xab, 0x27, 0x11,
    0x08, 0x00, 0x02, 0x00, 0xde, 0xde, 0xde, 0xde, 0x26, 0x16, 0x20, 0x00, 0xcd, 0xab, 0x27, 0x15, 0x1d, 0x00, 0x00, 0x00,
    0x00, 0x00, 0xb0, 0xfa, 0x26, 0x16, 0x00, 0x01, 0xce, 0xab, 0x18, 0x24, 0x07, 0x01, 0x24, 0x08, 0x01, 0x30, 0x09, 0x41,
    0x04, 0x98, 0xca, 0x97, 0x34, 0xda, 0xaf, 0xf7, 0x33, 0x98, 0x33, 0x6d, 0xc0, 0xa4, 0xde, 0x89, 0x2d, 0xe6, 0x2a, 0x1f,
    0x96, 0x90, 0x23, 0xe0, 0x33, 0x70, 0x86, 0x00, 0x85, 0xdc, 0xdc, 0x07, 0x2b, 0x23, 0x72, 0x60, 0x79, 0x37, 0xba, 0x3a,
    0x34, 0x4d, 0x94, 0x55, 0x46, 0xb4, 0x14, 0xf2, 0x23, 0xd2, 0x72, 0x31, 0xc9, 0xa8, 0x3d, 0x71, 0xb3, 0x97, 0xb4, 0x32,
    0x06, 0x62, 0xc0, 0xf5, 0xcb, 0x37, 0x0a, 0x35, 0x01, 0x28, 0x01, 0x18, 0x24, 0x02, 0x01, 0x36, 0x03, 0x04, 0x02, 0x04,
    0x01, 0x18, 0x30, 0x04, 0x14, 0xa9, 0xc5, 0xfd, 0x6c, 0xbd, 0x38, 0xbb, 0x18, 0x37, 0x0f, 0x8e, 0x80, 0x64, 0x16, 0x6a,
    0xfb, 0x1e, 0xc8, 0x39, 0x73, 0x30, 0x05, 0x14, 0xe1, 0xe7, 0x6e, 0x67, 0x77, 0x85, 0x1d, 0xd7, 0x74, 0x16, 0xbd, 0xdd,
    0x35, 0xec, 0x3c, 0x13, 0x7c, 0x47, 0x29, 0xdc, 0x18, 0x30, 0x0b, 0x40, 0xbc, 0xe8, 0xc9, 0x50, 0x5a, 0xb1, 0xe4, 0xb4,
    0xd1, 0xd8, 0xf2, 0xc5, 0xe5, 0x26, 0x37, 0xa5, 0x3e, 0xf4, 0x05, 0x4a, 0x9f, 0x18, 0xef, 0x45, 0x94, 0x3d, 0x3d, 0x23,
    0x58, 0xfd, 0x7e, 0xb9, 0xcb, 0x12, 0x15, 0x04, 0xa3, 0xdc, 0x1d, 0xe4, 0x2d, 0xc0, 0xae, 0xe4, 0xf5, 0x11, 0xc0, 0x05,
    0x67, 0x7c, 0x11, 0x18, 0x8b, 0x44, 0x98, 0x78, 0xbd, 0x7d, 0x69, 0x3e, 0x37, 0x82, 0x2c, 0x47, 0x18,
};

extern const size_t sTestCert_Node02_08_Chip_Len = sizeof(sTestCert_Node02_08_Chip);

extern const uint8_t sTestCert_Node02_08_DER[] = {
    0x30, 0x82, 0x02, 0x52, 0x30, 0x82, 0x01, 0xf7, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x08, 0x3e, 0x67, 0x94, 0x70, 0x7a, 0xec,
    0xb8, 0x15, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x30, 0x44, 0x31, 0x20, 0x30, 0x1e, 0x06,
    0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x03, 0x0c, 0x10, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41, 0x43, 0x41,
    0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x34, 0x31, 0x20, 0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2,
    0x7c, 0x01, 0x05, 0x0c, 0x10, 0x46, 0x41, 0x42, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x31, 0x44,
    0x30, 0x1e, 0x17, 0x0d, 0x32, 0x30, 0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32, 0x33, 0x34, 0x33, 0x5a, 0x17, 0x0d, 0x34, 0x30,
    0x31, 0x30, 0x31, 0x35, 0x31, 0x34, 0x32, 0x33, 0x34, 0x32, 0x5a, 0x30, 0x81, 0x92, 0x31, 0x18, 0x30, 0x16, 0x06, 0x0a, 0x2b,
    0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x06, 0x0c, 0x08, 0x41, 0x42, 0x43, 0x46, 0x30, 0x30, 0x41, 0x30, 0x31, 0x20,
    0x30, 0x1e, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x01, 0x0c, 0x10, 0x44, 0x45, 0x44, 0x45, 0x44,
    0x45, 0x44, 0x45, 0x30, 0x30, 0x30, 0x32, 0x30, 0x30, 0x30, 0x38, 0x31, 0x18, 0x30, 0x16, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04,
    0x01, 0x82, 0xa2, 0x7c, 0x01, 0x06, 0x0c, 0x08, 0x41, 0x42, 0x43, 0x44, 0x30, 0x30, 0x32, 0x30, 0x31, 0x20, 0x30, 0x1e, 0x06,
    0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2, 0x7c, 0x01, 0x05, 0x0c, 0x10, 0x46, 0x41, 0x42, 0x30, 0x30, 0x30, 0x30, 0x30,
    0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x31, 0x44, 0x31, 0x18, 0x30, 0x16, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0xa2,
    0x7c, 0x01, 0x06, 0x0c, 0x08, 0x41, 0x42, 0x43, 0x45, 0x30, 0x31, 0x30, 0x30, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86,
    0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x98, 0xca,
    0x97, 0x34, 0xda, 0xaf, 0xf7, 0x33, 0x98, 0x33, 0x6d, 0xc0, 0xa4, 0xde, 0x89, 0x2d, 0xe6, 0x2a, 0x1f, 0x96, 0x90, 0x23, 0xe0,
    0x33, 0x70, 0x86, 0x00, 0x85, 0xdc, 0xdc, 0x07, 0x2b, 0x23, 0x72, 0x60, 0x79, 0x37, 0xba, 0x3a, 0x34, 0x4d, 0x94, 0x55, 0x46,
    0xb4, 0x14, 0xf2, 0x23, 0xd2, 0x72, 0x31, 0xc9, 0xa8, 0x3d, 0x71, 0xb3, 0x97, 0xb4, 0x32, 0x06, 0x62, 0xc0, 0xf5, 0xcb, 0xa3,
    0x81, 0x83, 0x30, 0x81, 0x80, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0e,
    0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x20, 0x06, 0x03, 0x55, 0x1d, 0x25,
    0x01, 0x01, 0xff, 0x04, 0x16, 0x30, 0x14, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x02, 0x06, 0x08, 0x2b, 0x06,
    0x01, 0x05, 0x05, 0x07, 0x03, 0x01, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xa9, 0xc5, 0xfd, 0x6c,
    0xbd, 0x38, 0xbb, 0x18, 0x37, 0x0f, 0x8e, 0x80, 0x64, 0x16, 0x6a, 0xfb, 0x1e, 0xc8, 0x39, 0x73, 0x30, 0x1f, 0x06, 0x03, 0x55,
    0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xe1, 0xe7, 0x6e, 0x67, 0x77, 0x85, 0x1d, 0xd7, 0x74, 0x16, 0xbd, 0xdd, 0x35,
    0xec, 0x3c, 0x13, 0x7c, 0x47, 0x29, 0xdc, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x49,
    0x00, 0x30, 0x46, 0x02, 0x21, 0x00, 0xbc, 0xe8, 0xc9, 0x50, 0x5a, 0xb1, 0xe4, 0xb4, 0xd1, 0xd8, 0xf2, 0xc5, 0xe5, 0x26, 0x37,
    0xa5, 0x3e, 0xf4, 0x05, 0x4a, 0x9f, 0x18, 0xef, 0x45, 0x94, 0x3d, 0x3d, 0x23, 0x58, 0xfd, 0x7e, 0xb9, 0x02, 0x21, 0x00, 0xcb,
    0x12, 0x15, 0x04, 0xa3, 0xdc, 0x1d, 0xe4, 0x2d, 0xc0, 0xae, 0xe4, 0xf5, 0x11, 0xc0, 0x05, 0x67, 0x7c, 0x11, 0x18, 0x8b, 0x44,
    0x98, 0x78, 0xbd, 0x7d, 0x69, 0x3e, 0x37, 0x82, 0x2c, 0x47,
};

extern const size_t sTestCert_Node02_08_DER_Len = sizeof(sTestCert_Node02_08_DER);

extern const uint8_t sTestCert_Node02_08_PublicKey[] = {
    0x04, 0x98, 0xca, 0x97, 0x34, 0xda, 0xaf, 0xf7, 0x33, 0x98, 0x33, 0x6d, 0xc0, 0xa4, 0xde, 0x89, 0x2d,
    0xe6, 0x2a, 0x1f, 0x96, 0x90, 0x23, 0xe0, 0x33, 0x70, 0x86, 0x00, 0x85, 0xdc, 0xdc, 0x07, 0x2b, 0x23,
    0x72, 0x60, 0x79, 0x37, 0xba, 0x3a, 0x34, 0x4d, 0x94, 0x55, 0x46, 0xb4, 0x14, 0xf2, 0x23, 0xd2, 0x72,
    0x31, 0xc9, 0xa8, 0x3d, 0x71, 0xb3, 0x97, 0xb4, 0x32, 0x06, 0x62, 0xc0, 0xf5, 0xcb,
};

extern const size_t sTestCert_Node02_08_PublicKey_Len = sizeof(sTestCert_Node02_08_PublicKey);

extern const uint8_t sTestCert_Node02_08_PrivateKey[] = {
    0xdf, 0x8b, 0x10, 0x34, 0x18, 0x5b, 0xb0, 0xed, 0x9b, 0xcc, 0x39, 0xb5, 0xaf, 0xc0, 0xec, 0x9b,
    0xd5, 0x6f, 0x4a, 0x76, 0x5d, 0xe9, 0x9f, 0x69, 0x9b, 0xb0, 0x9a, 0xf6, 0xcf, 0x3a, 0xf7, 0x6f,
};

extern const size_t sTestCert_Node02_08_PrivateKey_Len = sizeof(sTestCert_Node02_08_PrivateKey);

extern const uint8_t sTestCert_Node02_08_SubjectKeyId[] = {
    0xA9, 0xC5, 0xFD, 0x6C, 0xBD, 0x38, 0xBB, 0x18, 0x37, 0x0F, 0x8E, 0x80, 0x64, 0x16, 0x6A, 0xFB, 0x1E, 0xC8, 0x39, 0x73,
};

extern const size_t sTestCert_Node02_08_SubjectKeyId_Len = sizeof(sTestCert_Node02_08_SubjectKeyId);

extern const uint8_t sTestCert_Node02_08_AuthorityKeyId[] = {
    0xE1, 0xE7, 0x6E, 0x67, 0x77, 0x85, 0x1D, 0xD7, 0x74, 0x16, 0xBD, 0xDD, 0x35, 0xEC, 0x3C, 0x13, 0x7C, 0x47, 0x29, 0xDC,
};

extern const size_t sTestCert_Node02_08_AuthorityKeyId_Len = sizeof(sTestCert_Node02_08_AuthorityKeyId);

} // namespace TestCerts
} // namespace chip
