/*
 *
 *    Copyright (c) 2021 Project CHIP Authors
 *    All rights reserved.
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */
#include <credentials/DeviceAttestationConstructor.h>
#include <credentials/DeviceAttestationVendorReserved.h>
#include <lib/core/CHIPError.h>
#include <lib/core/TLV.h>
#include <lib/support/CodeUtils.h>
#include <lib/support/ScopedBuffer.h>
#include <lib/support/Span.h>
#include <lib/support/UnitTestRegistration.h>

#include <cstdio>
#include <nlunit-test.h>

using namespace chip;
using namespace chip::Credentials;

static void TestAttestationElements_Roundtrip(nlTestSuite * inSuite, void * inContext)
{
    CHIP_ERROR err = CHIP_NO_ERROR;
    chip::Platform::ScopedMemoryBuffer<uint8_t> attestationElements;
    size_t attestationElementsLen;

    static constexpr uint8_t certificationDeclaration[] = {
        0xd2, 0x84, 0x4b, 0xa2, 0x01, 0x26, 0x04, 0x46, 0x63, 0x73, 0x61, 0x63, 0x64, 0x30, 0xa0, 0x58, 0x1d, 0x15, 0x25,
        0x01, 0x88, 0x99, 0x25, 0x02, 0xfe, 0xff, 0x25, 0x03, 0xd2, 0x04, 0x25, 0x04, 0x2e, 0x16, 0x24, 0x05, 0xaa, 0x25,
        0x06, 0xde, 0xc0, 0x25, 0x07, 0x94, 0x26, 0x18, 0x58, 0x40, 0x96, 0x57, 0x2d, 0xd6, 0x3c, 0x03, 0x64, 0x0b, 0x28,
        0x67, 0x02, 0xbd, 0x6b, 0xba, 0x48, 0xac, 0x7c, 0x83, 0x54, 0x9b, 0x68, 0x73, 0x29, 0x47, 0x48, 0xb9, 0x51, 0xd5,
        0xab, 0x66, 0x62, 0x2e, 0x9d, 0x26, 0x10, 0x41, 0xf8, 0x0e, 0x97, 0x49, 0xfe, 0xff, 0x78, 0x10, 0x02, 0x49, 0x67,
        0xae, 0xdf, 0x41, 0x38, 0x36, 0x5b, 0x0a, 0x22, 0x57, 0x14, 0x9c, 0x9a, 0x12, 0x3e, 0x0d, 0x30, 0xaa
    };
    static constexpr uint8_t attestationNonce[] = {
        0xe0, 0x42, 0x1b, 0x91, 0xc6, 0xfd, 0xcd, 0xb4, 0x0e, 0x2a, 0x4d, 0x2c, 0xf3, 0x1d, 0xb2, 0xb4,
        0xe1, 0x8b, 0x41, 0x1b, 0x1d, 0x3a, 0xd4, 0xd1, 0x2a, 0x9d, 0x90, 0xaa, 0x8e, 0x52, 0xfa, 0xe2,
    };
    static constexpr uint32_t timestamp        = 677103357;
    static constexpr uint8_t vendorReserved1[] = {
        0x73, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x5f, 0x76, 0x65, 0x6e, 0x64, 0x6f,
        0x72, 0x5f, 0x72, 0x65, 0x73, 0x65, 0x72, 0x76, 0x65, 0x64, 0x31,
    };
    static constexpr uint8_t vendorReserved3[] = {
        0x76, 0x65, 0x6e, 0x64, 0x6f, 0x72, 0x5f, 0x72, 0x65, 0x73, 0x65, 0x72,
        0x76, 0x65, 0x64, 0x33, 0x5f, 0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65,
    };
    static constexpr uint16_t vendorId   = 0xbeef;
    static constexpr uint16_t profileNum = 0xdead;
    VendorReservedElement vendorReservedArray[2];
    DeviceAttestationVendorReservedConstructor vendorReservedConstructor(vendorReservedArray, 2);

    attestationElementsLen = sizeof(certificationDeclaration) + sizeof(attestationNonce) + sizeof(timestamp) +
        sizeof(vendorReserved1) + sizeof(vendorReserved3) + sizeof(uint64_t) * 5;
    attestationElements.Alloc(attestationElementsLen);
    vendorReservedConstructor.addVendorReservedElement(vendorId, profileNum, 1, ByteSpan(vendorReserved1));
    vendorReservedConstructor.addVendorReservedElement(vendorId, profileNum, 3, ByteSpan(vendorReserved3));
    NL_TEST_ASSERT(inSuite, attestationElements);

    {
        MutableByteSpan attestationElementsSpan(attestationElements.Get(), attestationElementsLen);

        // Test wrong size nonce
        err = ConstructAttestationElements(ByteSpan(certificationDeclaration),
                                           ByteSpan(attestationNonce, sizeof(attestationNonce) - 1), timestamp, ByteSpan(),
                                           vendorReservedConstructor, attestationElementsSpan);
        NL_TEST_ASSERT(inSuite, err == CHIP_ERROR_INVALID_ARGUMENT);

        // Test with missing mandatory TLV entries
        err = ConstructAttestationElements(ByteSpan(), ByteSpan(attestationNonce), timestamp, ByteSpan(), vendorReservedConstructor,
                                           attestationElementsSpan);
        NL_TEST_ASSERT(inSuite, err == CHIP_ERROR_INVALID_ARGUMENT);

        // Test with missing mandatory TLV entries
        err = ConstructAttestationElements(ByteSpan(certificationDeclaration), ByteSpan(), timestamp, ByteSpan(),
                                           vendorReservedConstructor, attestationElementsSpan);
        NL_TEST_ASSERT(inSuite, err == CHIP_ERROR_INVALID_ARGUMENT);

        // Test for success with entirely valid arguments
        err = ConstructAttestationElements(ByteSpan(certificationDeclaration), ByteSpan(attestationNonce), timestamp, ByteSpan(),
                                           vendorReservedConstructor, attestationElementsSpan);
        NL_TEST_ASSERT(inSuite, err == CHIP_NO_ERROR);
        attestationElementsLen = attestationElementsSpan.size();
    }

    ByteSpan certificationDeclarationSpan;
    ByteSpan attestationNonceSpan;
    uint32_t timestampDeconstructed;
    ByteSpan firmwareInfoSpan;
    DeviceAttestationVendorReservedDeconstructor vendorReservedDeconstructor;

    err =
        DeconstructAttestationElements(ByteSpan(attestationElements.Get(), attestationElementsLen), certificationDeclarationSpan,
                                       attestationNonceSpan, timestampDeconstructed, firmwareInfoSpan, vendorReservedDeconstructor);
    NL_TEST_ASSERT(inSuite, err == CHIP_NO_ERROR);
    NL_TEST_ASSERT(inSuite, certificationDeclarationSpan.data_equal(ByteSpan(certificationDeclaration)));
    NL_TEST_ASSERT(inSuite, attestationNonceSpan.data_equal(ByteSpan(attestationNonce)));
    NL_TEST_ASSERT(inSuite, timestamp == timestampDeconstructed);
    NL_TEST_ASSERT(inSuite, firmwareInfoSpan.empty());

    NL_TEST_ASSERT(inSuite, vendorReservedConstructor.GetNumberOfElements() == vendorReservedDeconstructor.GetNumberOfElements());

    const VendorReservedElement * constructionElement = vendorReservedConstructor.cbegin();
    VendorReservedElement deconstructionElement;

    while ((constructionElement = vendorReservedConstructor.Next()) != nullptr &&
           vendorReservedDeconstructor.GetNextVendorReservedElement(deconstructionElement) == CHIP_NO_ERROR)
    {
        NL_TEST_ASSERT(inSuite, constructionElement->vendorId == deconstructionElement.vendorId);
        NL_TEST_ASSERT(inSuite, constructionElement->profileNum == deconstructionElement.profileNum);
        NL_TEST_ASSERT(inSuite, constructionElement->vendorReservedData.data_equal(deconstructionElement.vendorReservedData));
    }
}

static void TestAttestationElements_Construction(nlTestSuite * inSuite, void * inContext)
{
    CHIP_ERROR err = CHIP_NO_ERROR;
    Platform::ScopedMemoryBuffer<uint8_t> attestationElements;
    size_t attestationElementsLen;

    static constexpr uint8_t attestationElementsTestVector[] = {
        0x15, 0x30, 0x01, 0x70, 0xd2, 0x84, 0x4b, 0xa2, 0x01, 0x26, 0x04, 0x46, 0x63, 0x73, 0x61, 0x63, 0x64, 0x30, 0xa0,
        0x58, 0x1d, 0x15, 0x25, 0x01, 0x88, 0x99, 0x25, 0x02, 0xfe, 0xff, 0x25, 0x03, 0xd2, 0x04, 0x25, 0x04, 0x2e, 0x16,
        0x24, 0x05, 0xaa, 0x25, 0x06, 0xde, 0xc0, 0x25, 0x07, 0x94, 0x26, 0x18, 0x58, 0x40, 0x96, 0x57, 0x2d, 0xd6, 0x3c,
        0x03, 0x64, 0x0b, 0x28, 0x67, 0x02, 0xbd, 0x6b, 0xba, 0x48, 0xac, 0x7c, 0x83, 0x54, 0x9b, 0x68, 0x73, 0x29, 0x47,
        0x48, 0xb9, 0x51, 0xd5, 0xab, 0x66, 0x62, 0x2e, 0x9d, 0x26, 0x10, 0x41, 0xf8, 0x0e, 0x97, 0x49, 0xfe, 0xff, 0x78,
        0x10, 0x02, 0x49, 0x67, 0xae, 0xdf, 0x41, 0x38, 0x36, 0x5b, 0x0a, 0x22, 0x57, 0x14, 0x9c, 0x9a, 0x12, 0x3e, 0x0d,
        0x30, 0xaa, 0x30, 0x02, 0x20, 0xe0, 0x42, 0x1b, 0x91, 0xc6, 0xfd, 0xcd, 0xb4, 0x0e, 0x2a, 0x4d, 0x2c, 0xf3, 0x1d,
        0xb2, 0xb4, 0xe1, 0x8b, 0x41, 0x1b, 0x1d, 0x3a, 0xd4, 0xd1, 0x2a, 0x9d, 0x90, 0xaa, 0x8e, 0x52, 0xfa, 0xe2, 0x26,
        0x03, 0xfd, 0xc6, 0x5b, 0x28, 0xd0, 0xf1, 0xff, 0x3e, 0x00, 0x01, 0x00, 0x17, 0x73, 0x61, 0x6d, 0x70, 0x6c, 0x65,
        0x5f, 0x76, 0x65, 0x6e, 0x64, 0x6f, 0x72, 0x5f, 0x72, 0x65, 0x73, 0x65, 0x72, 0x76, 0x65, 0x64, 0x31, 0xd0, 0xf1,
        0xff, 0x3e, 0x00, 0x03, 0x00, 0x18, 0x76, 0x65, 0x6e, 0x64, 0x6f, 0x72, 0x5f, 0x72, 0x65, 0x73, 0x65, 0x72, 0x76,
        0x65, 0x64, 0x33, 0x5f, 0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x18
    };
    static constexpr uint8_t certificationDeclaration[] = {
        0xd2, 0x84, 0x4b, 0xa2, 0x01, 0x26, 0x04, 0x46, 0x63, 0x73, 0x61, 0x63, 0x64, 0x30, 0xa0, 0x58, 0x1d, 0x15, 0x25,
        0x01, 0x88, 0x99, 0x25, 0x02, 0xfe, 0xff, 0x25, 0x03, 0xd2, 0x04, 0x25, 0x04, 0x2e, 0x16, 0x24, 0x05, 0xaa, 0x25,
        0x06, 0xde, 0xc0, 0x25, 0x07, 0x94, 0x26, 0x18, 0x58, 0x40, 0x96, 0x57, 0x2d, 0xd6, 0x3c, 0x03, 0x64, 0x0b, 0x28,
        0x67, 0x02, 0xbd, 0x6b, 0xba, 0x48, 0xac, 0x7c, 0x83, 0x54, 0x9b, 0x68, 0x73, 0x29, 0x47, 0x48, 0xb9, 0x51, 0xd5,
        0xab, 0x66, 0x62, 0x2e, 0x9d, 0x26, 0x10, 0x41, 0xf8, 0x0e, 0x97, 0x49, 0xfe, 0xff, 0x78, 0x10, 0x02, 0x49, 0x67,
        0xae, 0xdf, 0x41, 0x38, 0x36, 0x5b, 0x0a, 0x22, 0x57, 0x14, 0x9c, 0x9a, 0x12, 0x3e, 0x0d, 0x30, 0xaa
    };
    static constexpr uint8_t attestationNonce[] = {
        0xe0, 0x42, 0x1b, 0x91, 0xc6, 0xfd, 0xcd, 0xb4, 0x0e, 0x2a, 0x4d, 0x2c, 0xf3, 0x1d, 0xb2, 0xb4,
        0xe1, 0x8b, 0x41, 0x1b, 0x1d, 0x3a, 0xd4, 0xd1, 0x2a, 0x9d, 0x90, 0xaa, 0x8e, 0x52, 0xfa, 0xe2,
    };
    static constexpr uint32_t timestamp        = 677103357;
    static constexpr uint8_t vendorReserved1[] = {
        0x73, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x5f, 0x76, 0x65, 0x6e, 0x64, 0x6f,
        0x72, 0x5f, 0x72, 0x65, 0x73, 0x65, 0x72, 0x76, 0x65, 0x64, 0x31,
    };
    static constexpr uint8_t vendorReserved3[] = {
        0x76, 0x65, 0x6e, 0x64, 0x6f, 0x72, 0x5f, 0x72, 0x65, 0x73, 0x65, 0x72,
        0x76, 0x65, 0x64, 0x33, 0x5f, 0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65,
    };
    static constexpr uint16_t vendorId   = 0xFFF1;
    static constexpr uint16_t profileNum = 0x003E;
    VendorReservedElement vendorReservedArray[2];
    DeviceAttestationVendorReservedConstructor vendorReserved(vendorReservedArray, 2);
    vendorReserved.addVendorReservedElement(vendorId, profileNum, 1, ByteSpan(vendorReserved1));
    vendorReserved.addVendorReservedElement(vendorId, profileNum, 3, ByteSpan(vendorReserved3));

    attestationElementsLen = sizeof(certificationDeclaration) + sizeof(attestationNonce) + sizeof(timestamp) +
        sizeof(vendorReserved1) + sizeof(vendorReserved3) + sizeof(uint64_t) * 5;
    attestationElements.Alloc(attestationElementsLen);
    NL_TEST_ASSERT(inSuite, attestationElements);

    {
        MutableByteSpan attestationElementsSpan(attestationElements.Get(), attestationElementsLen);

        err = ConstructAttestationElements(ByteSpan(certificationDeclaration), ByteSpan(attestationNonce), timestamp, ByteSpan(),
                                           vendorReserved, attestationElementsSpan);
        NL_TEST_ASSERT(inSuite, err == CHIP_NO_ERROR);
        attestationElementsLen = attestationElementsSpan.size();

        NL_TEST_ASSERT(inSuite, attestationElementsSpan.data_equal(ByteSpan(attestationElementsTestVector)));
    }
}

static void TestAttestationElements_Deconstruction(nlTestSuite * inSuite, void * inContext)
{
    // This is a test case with only the known TLV tags fields
    constexpr uint8_t attestationElementsTestVectorOnlyKnownTags[] = {
        0x15, 0x30, 0x01, 0x70, 0xd2, 0x84, 0x4b, 0xa2, 0x01, 0x26, 0x04, 0x46, 0x63, 0x73, 0x61, 0x63, 0x64, 0x30, 0xa0,
        0x58, 0x1d, 0x15, 0x25, 0x01, 0x88, 0x99, 0x25, 0x02, 0xfe, 0xff, 0x25, 0x03, 0xd2, 0x04, 0x25, 0x04, 0x2e, 0x16,
        0x24, 0x05, 0xaa, 0x25, 0x06, 0xde, 0xc0, 0x25, 0x07, 0x94, 0x26, 0x18, 0x58, 0x40, 0x96, 0x57, 0x2d, 0xd6, 0x3c,
        0x03, 0x64, 0x0b, 0x28, 0x67, 0x02, 0xbd, 0x6b, 0xba, 0x48, 0xac, 0x7c, 0x83, 0x54, 0x9b, 0x68, 0x73, 0x29, 0x47,
        0x48, 0xb9, 0x51, 0xd5, 0xab, 0x66, 0x62, 0x2e, 0x9d, 0x26, 0x10, 0x41, 0xf8, 0x0e, 0x97, 0x49, 0xfe, 0xff, 0x78,
        0x10, 0x02, 0x49, 0x67, 0xae, 0xdf, 0x41, 0x38, 0x36, 0x5b, 0x0a, 0x22, 0x57, 0x14, 0x9c, 0x9a, 0x12, 0x3e, 0x0d,
        0x30, 0xaa, 0x30, 0x02, 0x20, 0xe0, 0x42, 0x1b, 0x91, 0xc6, 0xfd, 0xcd, 0xb4, 0x0e, 0x2a, 0x4d, 0x2c, 0xf3, 0x1d,
        0xb2, 0xb4, 0xe1, 0x8b, 0x41, 0x1b, 0x1d, 0x3a, 0xd4, 0xd1, 0x2a, 0x9d, 0x90, 0xaa, 0x8e, 0x52, 0xfa, 0xe2, 0x26,
        0x03, 0xfd, 0xc6, 0x5b, 0x28, 0xd0, 0xf1, 0xff, 0x3e, 0x00, 0x01, 0x00, 0x17, 0x73, 0x61, 0x6d, 0x70, 0x6c, 0x65,
        0x5f, 0x76, 0x65, 0x6e, 0x64, 0x6f, 0x72, 0x5f, 0x72, 0x65, 0x73, 0x65, 0x72, 0x76, 0x65, 0x64, 0x31, 0xd0, 0xf1,
        0xff, 0x3e, 0x00, 0x03, 0x00, 0x18, 0x76, 0x65, 0x6e, 0x64, 0x6f, 0x72, 0x5f, 0x72, 0x65, 0x73, 0x65, 0x72, 0x76,
        0x65, 0x64, 0x33, 0x5f, 0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x18
    };

    // The following case has an extra top-level context-specific tag (254) with value 0xDEADBEEF, in the right order,
    // that should not cause a failure and merely be ignored.
    constexpr uint8_t attestationElementsTestVectorWithOneUnknownTag[] = {
        0x15, 0x30, 0x01, 0x70, 0xd2, 0x84, 0x4b, 0xa2, 0x01, 0x26, 0x04, 0x46, 0x63, 0x73, 0x61, 0x63, 0x64, 0x30, 0xa0,
        0x58, 0x1d, 0x15, 0x25, 0x01, 0x88, 0x99, 0x25, 0x02, 0xfe, 0xff, 0x25, 0x03, 0xd2, 0x04, 0x25, 0x04, 0x2e, 0x16,
        0x24, 0x05, 0xaa, 0x25, 0x06, 0xde, 0xc0, 0x25, 0x07, 0x94, 0x26, 0x18, 0x58, 0x40, 0x96, 0x57, 0x2d, 0xd6, 0x3c,
        0x03, 0x64, 0x0b, 0x28, 0x67, 0x02, 0xbd, 0x6b, 0xba, 0x48, 0xac, 0x7c, 0x83, 0x54, 0x9b, 0x68, 0x73, 0x29, 0x47,
        0x48, 0xb9, 0x51, 0xd5, 0xab, 0x66, 0x62, 0x2e, 0x9d, 0x26, 0x10, 0x41, 0xf8, 0x0e, 0x97, 0x49, 0xfe, 0xff, 0x78,
        0x10, 0x02, 0x49, 0x67, 0xae, 0xdf, 0x41, 0x38, 0x36, 0x5b, 0x0a, 0x22, 0x57, 0x14, 0x9c, 0x9a, 0x12, 0x3e, 0x0d,
        0x30, 0xaa, 0x30, 0x02, 0x20, 0xe0, 0x42, 0x1b, 0x91, 0xc6, 0xfd, 0xcd, 0xb4, 0x0e, 0x2a, 0x4d, 0x2c, 0xf3, 0x1d,
        0xb2, 0xb4, 0xe1, 0x8b, 0x41, 0x1b, 0x1d, 0x3a, 0xd4, 0xd1, 0x2a, 0x9d, 0x90, 0xaa, 0x8e, 0x52, 0xfa, 0xe2, 0x26,
        0x03, 0xfd, 0xc6, 0x5b, 0x28, 0x26, 0xfe, 0xef, 0xbe, 0xad, 0xde, 0xd0, 0xf1, 0xff, 0x3e, 0x00, 0x01, 0x00, 0x17,
        0x73, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x5f, 0x76, 0x65, 0x6e, 0x64, 0x6f, 0x72, 0x5f, 0x72, 0x65, 0x73, 0x65, 0x72,
        0x76, 0x65, 0x64, 0x31, 0xd0, 0xf1, 0xff, 0x3e, 0x00, 0x03, 0x00, 0x18, 0x76, 0x65, 0x6e, 0x64, 0x6f, 0x72, 0x5f,
        0x72, 0x65, 0x73, 0x65, 0x72, 0x76, 0x65, 0x64, 0x33, 0x5f, 0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x18,
    };

    constexpr uint8_t certificationDeclarationTestVector[] = {
        0xd2, 0x84, 0x4b, 0xa2, 0x01, 0x26, 0x04, 0x46, 0x63, 0x73, 0x61, 0x63, 0x64, 0x30, 0xa0, 0x58, 0x1d, 0x15, 0x25,
        0x01, 0x88, 0x99, 0x25, 0x02, 0xfe, 0xff, 0x25, 0x03, 0xd2, 0x04, 0x25, 0x04, 0x2e, 0x16, 0x24, 0x05, 0xaa, 0x25,
        0x06, 0xde, 0xc0, 0x25, 0x07, 0x94, 0x26, 0x18, 0x58, 0x40, 0x96, 0x57, 0x2d, 0xd6, 0x3c, 0x03, 0x64, 0x0b, 0x28,
        0x67, 0x02, 0xbd, 0x6b, 0xba, 0x48, 0xac, 0x7c, 0x83, 0x54, 0x9b, 0x68, 0x73, 0x29, 0x47, 0x48, 0xb9, 0x51, 0xd5,
        0xab, 0x66, 0x62, 0x2e, 0x9d, 0x26, 0x10, 0x41, 0xf8, 0x0e, 0x97, 0x49, 0xfe, 0xff, 0x78, 0x10, 0x02, 0x49, 0x67,
        0xae, 0xdf, 0x41, 0x38, 0x36, 0x5b, 0x0a, 0x22, 0x57, 0x14, 0x9c, 0x9a, 0x12, 0x3e, 0x0d, 0x30, 0xaa
    };
    constexpr uint8_t attestationNonceTestVector[] = {
        0xe0, 0x42, 0x1b, 0x91, 0xc6, 0xfd, 0xcd, 0xb4, 0x0e, 0x2a, 0x4d, 0x2c, 0xf3, 0x1d, 0xb2, 0xb4,
        0xe1, 0x8b, 0x41, 0x1b, 0x1d, 0x3a, 0xd4, 0xd1, 0x2a, 0x9d, 0x90, 0xaa, 0x8e, 0x52, 0xfa, 0xe2,
    };
    constexpr uint32_t timestampTestVector        = 677103357;
    constexpr uint8_t vendorReserved1TestVector[] = {
        0x73, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x5f, 0x76, 0x65, 0x6e, 0x64, 0x6f,
        0x72, 0x5f, 0x72, 0x65, 0x73, 0x65, 0x72, 0x76, 0x65, 0x64, 0x31,
    };
    constexpr uint8_t vendorReserved3TestVector[] = {
        0x76, 0x65, 0x6e, 0x64, 0x6f, 0x72, 0x5f, 0x72, 0x65, 0x73, 0x65, 0x72,
        0x76, 0x65, 0x64, 0x33, 0x5f, 0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65,
    };
    const ByteSpan vendorReservedArrayTestVector[] = { ByteSpan{ vendorReserved1TestVector },
                                                       ByteSpan{ vendorReserved3TestVector } };
    constexpr uint16_t vendorIdTestVector          = 0xFFF1;
    constexpr uint16_t profileNumTestVector        = 0x003E;

    const ByteSpan kTestCases[] = { ByteSpan{ attestationElementsTestVectorOnlyKnownTags },
                                    ByteSpan{ attestationElementsTestVectorWithOneUnknownTag } };

    // Try all test cases and see they are all OK
    for (const ByteSpan & attestationElementsTestCase : kTestCases)
    {
        CHIP_ERROR err = CHIP_NO_ERROR;
        ByteSpan certificationDeclarationDeconstructed;
        ByteSpan attestationNonceDeconstructed;
        uint32_t timestampDeconstructed;
        ByteSpan firmwareInfoDeconstructed;
        DeviceAttestationVendorReservedDeconstructor vendorReserved;

        err = DeconstructAttestationElements(attestationElementsTestCase, certificationDeclarationDeconstructed,
                                             attestationNonceDeconstructed, timestampDeconstructed, firmwareInfoDeconstructed,
                                             vendorReserved);
        NL_TEST_ASSERT(inSuite, err == CHIP_NO_ERROR);

        NL_TEST_ASSERT(inSuite, certificationDeclarationDeconstructed.data_equal(ByteSpan(certificationDeclarationTestVector)));
        NL_TEST_ASSERT(inSuite, attestationNonceDeconstructed.data_equal(ByteSpan(attestationNonceTestVector)));
        NL_TEST_ASSERT(inSuite, timestampTestVector == timestampDeconstructed);
        NL_TEST_ASSERT(inSuite, firmwareInfoDeconstructed.empty());
        NL_TEST_ASSERT(inSuite, ArraySize(vendorReservedArrayTestVector) == vendorReserved.GetNumberOfElements());
        struct VendorReservedElement element;

        while (vendorReserved.GetNextVendorReservedElement(element) == CHIP_NO_ERROR)
        {
            NL_TEST_ASSERT(inSuite, vendorIdTestVector == element.vendorId);
            NL_TEST_ASSERT(inSuite, profileNumTestVector == element.profileNum);
            switch (element.tagNum)
            {
            case 1:
                NL_TEST_ASSERT(inSuite, element.vendorReservedData.data_equal(vendorReservedArrayTestVector[0]));
                break;
            case 3:
                NL_TEST_ASSERT(inSuite, element.vendorReservedData.data_equal(vendorReservedArrayTestVector[1]));
                break;
            default:
                NL_TEST_ASSERT(inSuite, 0);
                break;
            }
        }
    }
}

static void TestAttestationElements_DeconstructionNoVendorReserved(nlTestSuite * inSuite, void * inContext)
{
    // This is a test case with only the known TLV tags fields
    constexpr uint8_t attestationElementsTestVectorNoVendor[] = {
        0x15, 0x30, 0x01, 0x70, 0xd2, 0x84, 0x4b, 0xa2, 0x01, 0x26, 0x04, 0x46, 0x63, 0x73, 0x61, 0x63, 0x64, 0x30, 0xa0, 0x58,
        0x1d, 0x15, 0x25, 0x01, 0x88, 0x99, 0x25, 0x02, 0xfe, 0xff, 0x25, 0x03, 0xd2, 0x04, 0x25, 0x04, 0x2e, 0x16, 0x24, 0x05,
        0xaa, 0x25, 0x06, 0xde, 0xc0, 0x25, 0x07, 0x94, 0x26, 0x18, 0x58, 0x40, 0x96, 0x57, 0x2d, 0xd6, 0x3c, 0x03, 0x64, 0x0b,
        0x28, 0x67, 0x02, 0xbd, 0x6b, 0xba, 0x48, 0xac, 0x7c, 0x83, 0x54, 0x9b, 0x68, 0x73, 0x29, 0x47, 0x48, 0xb9, 0x51, 0xd5,
        0xab, 0x66, 0x62, 0x2e, 0x9d, 0x26, 0x10, 0x41, 0xf8, 0x0e, 0x97, 0x49, 0xfe, 0xff, 0x78, 0x10, 0x02, 0x49, 0x67, 0xae,
        0xdf, 0x41, 0x38, 0x36, 0x5b, 0x0a, 0x22, 0x57, 0x14, 0x9c, 0x9a, 0x12, 0x3e, 0x0d, 0x30, 0xaa, 0x30, 0x02, 0x20, 0xe0,
        0x42, 0x1b, 0x91, 0xc6, 0xfd, 0xcd, 0xb4, 0x0e, 0x2a, 0x4d, 0x2c, 0xf3, 0x1d, 0xb2, 0xb4, 0xe1, 0x8b, 0x41, 0x1b, 0x1d,
        0x3a, 0xd4, 0xd1, 0x2a, 0x9d, 0x90, 0xaa, 0x8e, 0x52, 0xfa, 0xe2, 0x26, 0x03, 0xfd, 0xc6, 0x5b, 0x28, 0x18
    };

    constexpr uint8_t certificationDeclarationTestVector[] = {
        0xd2, 0x84, 0x4b, 0xa2, 0x01, 0x26, 0x04, 0x46, 0x63, 0x73, 0x61, 0x63, 0x64, 0x30, 0xa0, 0x58, 0x1d, 0x15, 0x25,
        0x01, 0x88, 0x99, 0x25, 0x02, 0xfe, 0xff, 0x25, 0x03, 0xd2, 0x04, 0x25, 0x04, 0x2e, 0x16, 0x24, 0x05, 0xaa, 0x25,
        0x06, 0xde, 0xc0, 0x25, 0x07, 0x94, 0x26, 0x18, 0x58, 0x40, 0x96, 0x57, 0x2d, 0xd6, 0x3c, 0x03, 0x64, 0x0b, 0x28,
        0x67, 0x02, 0xbd, 0x6b, 0xba, 0x48, 0xac, 0x7c, 0x83, 0x54, 0x9b, 0x68, 0x73, 0x29, 0x47, 0x48, 0xb9, 0x51, 0xd5,
        0xab, 0x66, 0x62, 0x2e, 0x9d, 0x26, 0x10, 0x41, 0xf8, 0x0e, 0x97, 0x49, 0xfe, 0xff, 0x78, 0x10, 0x02, 0x49, 0x67,
        0xae, 0xdf, 0x41, 0x38, 0x36, 0x5b, 0x0a, 0x22, 0x57, 0x14, 0x9c, 0x9a, 0x12, 0x3e, 0x0d, 0x30, 0xaa
    };
    constexpr uint8_t attestationNonceTestVector[] = {
        0xe0, 0x42, 0x1b, 0x91, 0xc6, 0xfd, 0xcd, 0xb4, 0x0e, 0x2a, 0x4d, 0x2c, 0xf3, 0x1d, 0xb2, 0xb4,
        0xe1, 0x8b, 0x41, 0x1b, 0x1d, 0x3a, 0xd4, 0xd1, 0x2a, 0x9d, 0x90, 0xaa, 0x8e, 0x52, 0xfa, 0xe2,
    };
    constexpr uint32_t timestampTestVector = 677103357;

    CHIP_ERROR err = CHIP_NO_ERROR;
    ByteSpan certificationDeclarationDeconstructed;
    ByteSpan attestationNonceDeconstructed;
    uint32_t timestampDeconstructed;
    ByteSpan firmwareInfoDeconstructed;
    DeviceAttestationVendorReservedDeconstructor vendorReserved;

    err = DeconstructAttestationElements(ByteSpan{ attestationElementsTestVectorNoVendor }, certificationDeclarationDeconstructed,
                                         attestationNonceDeconstructed, timestampDeconstructed, firmwareInfoDeconstructed,
                                         vendorReserved);
    NL_TEST_ASSERT(inSuite, err == CHIP_NO_ERROR);

    NL_TEST_ASSERT(inSuite, certificationDeclarationDeconstructed.data_equal(ByteSpan(certificationDeclarationTestVector)));
    NL_TEST_ASSERT(inSuite, attestationNonceDeconstructed.data_equal(ByteSpan(attestationNonceTestVector)));
    NL_TEST_ASSERT(inSuite, timestampTestVector == timestampDeconstructed);
    NL_TEST_ASSERT(inSuite, firmwareInfoDeconstructed.empty());
    NL_TEST_ASSERT(inSuite, 0 == vendorReserved.GetNumberOfElements());
    struct VendorReservedElement element;

    NL_TEST_ASSERT(inSuite, vendorReserved.GetNextVendorReservedElement(element) == CHIP_END_OF_TLV);
}

static void TestVendorReservedData(nlTestSuite * inSuite, void * inContext)
{

    struct VendorReservedElement inputArray[] = { { 3000, 100, 10 }, { 2999, 99, 10 },  { 10, 20, 100 },
                                                  { 100, 50, 200 },  { 3000, 100, 11 }, { 3000, 100, 9 } };

    VendorReservedElement vendorReservedArray[6];
    DeviceAttestationVendorReservedConstructor vendorReserved(vendorReservedArray, 6);
    size_t i;
    uint8_t strings[6][50];
    for (i = 0; i < ArraySize(inputArray); i++)
    {
        snprintf(reinterpret_cast<char *>(strings[i]), sizeof(strings[i]), "Vendor Reserved Data #%d",
                 (int) i); // for debugging use
        CHIP_ERROR err =
            vendorReserved.addVendorReservedElement(inputArray[i].vendorId, inputArray[i].profileNum, inputArray[i].tagNum,
                                                    ByteSpan(strings[i], strlen(reinterpret_cast<char *>(strings[i]))));

        NL_TEST_ASSERT(inSuite, err == CHIP_NO_ERROR);
    }

    // manually figure out the order these should be read out in when sorted
    struct VendorReservedElement * desiredOrder[] = {
        &inputArray[2], &inputArray[3], &inputArray[1], &inputArray[5], &inputArray[0], &inputArray[4],
    };

    const struct VendorReservedElement * element = vendorReserved.cbegin();
    NL_TEST_ASSERT(inSuite, element);
    NL_TEST_ASSERT(inSuite, element = vendorReserved.Next());

    for (i = 0; element && i < ArraySize(desiredOrder); element = vendorReserved.Next(), i++)
    {
        NL_TEST_ASSERT(inSuite,
                       element->vendorId == desiredOrder[i]->vendorId && element->profileNum == desiredOrder[i]->profileNum &&
                           element->tagNum == desiredOrder[i]->tagNum);
    }
    NL_TEST_ASSERT(inSuite, i == ArraySize(desiredOrder)); // check if previous loop matched for every array entry.

    // add another element, it should fail
    uint8_t testByteSpan[] = { 0x1, 0x2, 0x3 };
    CHIP_ERROR err         = vendorReserved.addVendorReservedElement(5, 10, 20, ByteSpan(testByteSpan));
    NL_TEST_ASSERT(inSuite, err == CHIP_ERROR_NO_MEMORY);
}

static void TestAttestationElements_DeconstructionWithFirmwareInfo(nlTestSuite * inSuite, void * inContext)
{
    CHIP_ERROR err = CHIP_NO_ERROR;

    static constexpr uint8_t attestationElementsTestVectorWithFirmwareInfo[] = {
        0x15, 0x30, 0x01, 0x70, 0xd2, 0x84, 0x4b, 0xa2, 0x01, 0x26, 0x04, 0x46, 0x63, 0x73, 0x61, 0x63, 0x64, 0x30, 0xa0, 0x58,
        0x1d, 0x15, 0x25, 0x01, 0x88, 0x99, 0x25, 0x02, 0xfe, 0xff, 0x25, 0x03, 0xd2, 0x04, 0x25, 0x04, 0x2e, 0x16, 0x24, 0x05,
        0xaa, 0x25, 0x06, 0xde, 0xc0, 0x25, 0x07, 0x94, 0x26, 0x18, 0x58, 0x40, 0x96, 0x57, 0x2d, 0xd6, 0x3c, 0x03, 0x64, 0x0b,
        0x28, 0x67, 0x02, 0xbd, 0x6b, 0xba, 0x48, 0xac, 0x7c, 0x83, 0x54, 0x9b, 0x68, 0x73, 0x29, 0x47, 0x48, 0xb9, 0x51, 0xd5,
        0xab, 0x66, 0x62, 0x2e, 0x9d, 0x26, 0x10, 0x41, 0xf8, 0x0e, 0x97, 0x49, 0xfe, 0xff, 0x78, 0x10, 0x02, 0x49, 0x67, 0xae,
        0xdf, 0x41, 0x38, 0x36, 0x5b, 0x0a, 0x22, 0x57, 0x14, 0x9c, 0x9a, 0x12, 0x3e, 0x0d, 0x30, 0xaa, 0x30, 0x02, 0x20, 0xe0,
        0x42, 0x1b, 0x91, 0xc6, 0xfd, 0xcd, 0xb4, 0x0e, 0x2a, 0x4d, 0x2c, 0xf3, 0x1d, 0xb2, 0xb4, 0xe1, 0x8b, 0x41, 0x1b, 0x1d,
        0x3a, 0xd4, 0xd1, 0x2a, 0x9d, 0x90, 0xaa, 0x8e, 0x52, 0xfa, 0xe2, 0x26, 0x03, 0xfd, 0xc6, 0x5b, 0x28, 0x30, 0x04, 0x69,
        0x30, 0x67, 0x80, 0x01, 0x05, 0xA1, 0x62, 0x30, 0x2F, 0xA0, 0x0B, 0x80, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04,
        0x02, 0x01, 0x81, 0x20, 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF,
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF, 0x30, 0x2F, 0xA0, 0x0B,
        0x80, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x81, 0x20, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16,
        0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A,
        0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0xd0, 0xf1, 0xff, 0x3e, 0x00, 0x01, 0x00, 0x17, 0x73, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x5f,
        0x76, 0x65, 0x6e, 0x64, 0x6f, 0x72, 0x5f, 0x72, 0x65, 0x73, 0x65, 0x72, 0x76, 0x65, 0x64, 0x31, 0xd0, 0xf1, 0xff, 0x3e,
        0x00, 0x03, 0x00, 0x18, 0x76, 0x65, 0x6e, 0x64, 0x6f, 0x72, 0x5f, 0x72, 0x65, 0x73, 0x65, 0x72, 0x76, 0x65, 0x64, 0x33,
        0x5f, 0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x18
    };
    static constexpr uint8_t certificationDeclarationTestVector[] = {
        0xd2, 0x84, 0x4b, 0xa2, 0x01, 0x26, 0x04, 0x46, 0x63, 0x73, 0x61, 0x63, 0x64, 0x30, 0xa0, 0x58, 0x1d, 0x15, 0x25,
        0x01, 0x88, 0x99, 0x25, 0x02, 0xfe, 0xff, 0x25, 0x03, 0xd2, 0x04, 0x25, 0x04, 0x2e, 0x16, 0x24, 0x05, 0xaa, 0x25,
        0x06, 0xde, 0xc0, 0x25, 0x07, 0x94, 0x26, 0x18, 0x58, 0x40, 0x96, 0x57, 0x2d, 0xd6, 0x3c, 0x03, 0x64, 0x0b, 0x28,
        0x67, 0x02, 0xbd, 0x6b, 0xba, 0x48, 0xac, 0x7c, 0x83, 0x54, 0x9b, 0x68, 0x73, 0x29, 0x47, 0x48, 0xb9, 0x51, 0xd5,
        0xab, 0x66, 0x62, 0x2e, 0x9d, 0x26, 0x10, 0x41, 0xf8, 0x0e, 0x97, 0x49, 0xfe, 0xff, 0x78, 0x10, 0x02, 0x49, 0x67,
        0xae, 0xdf, 0x41, 0x38, 0x36, 0x5b, 0x0a, 0x22, 0x57, 0x14, 0x9c, 0x9a, 0x12, 0x3e, 0x0d, 0x30, 0xaa
    };
    static constexpr uint8_t attestationNonceTestVector[] = {
        0xe0, 0x42, 0x1b, 0x91, 0xc6, 0xfd, 0xcd, 0xb4, 0x0e, 0x2a, 0x4d, 0x2c, 0xf3, 0x1d, 0xb2, 0xb4,
        0xe1, 0x8b, 0x41, 0x1b, 0x1d, 0x3a, 0xd4, 0xd1, 0x2a, 0x9d, 0x90, 0xaa, 0x8e, 0x52, 0xfa, 0xe2,
    };
    static constexpr uint32_t timestampTestVector     = 677103357;
    static constexpr uint8_t firmwareInfoTestVector[] = {
        0x30, 0x67, 0x80, 0x01, 0x05, 0xA1, 0x62, 0x30, 0x2F, 0xA0, 0x0B, 0x80, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65,
        0x03, 0x04, 0x02, 0x01, 0x81, 0x20, 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xAA, 0xBB,
        0xCC, 0xDD, 0xEE, 0xFF, 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD,
        0xEE, 0xFF, 0x30, 0x2F, 0xA0, 0x0B, 0x80, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x81,
        0x20, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x10,
        0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F
    };
    static constexpr uint8_t vendorReserved1TestVector[] = {
        0x73, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x5f, 0x76, 0x65, 0x6e, 0x64, 0x6f,
        0x72, 0x5f, 0x72, 0x65, 0x73, 0x65, 0x72, 0x76, 0x65, 0x64, 0x31,
    };
    static constexpr uint8_t vendorReserved3TestVector[] = {
        0x76, 0x65, 0x6e, 0x64, 0x6f, 0x72, 0x5f, 0x72, 0x65, 0x73, 0x65, 0x72,
        0x76, 0x65, 0x64, 0x33, 0x5f, 0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65
    };
    static constexpr ByteSpan vendorReservedArrayTestVector[] = { ByteSpan(vendorReserved1TestVector),
                                                                  ByteSpan(vendorReserved3TestVector) };
    static constexpr uint16_t vendorIdTestVector              = 0xFFF1;
    static constexpr uint16_t profileNumTestVector            = 0x003E;

    ByteSpan certificationDeclarationDeconstructed;
    ByteSpan attestationNonceDeconstructed;
    uint32_t timestampDeconstructed;
    ByteSpan firmwareInfoDeconstructed;

    DeviceAttestationVendorReservedDeconstructor vendorReserved;

    err = DeconstructAttestationElements(ByteSpan(attestationElementsTestVectorWithFirmwareInfo),
                                         certificationDeclarationDeconstructed, attestationNonceDeconstructed,
                                         timestampDeconstructed, firmwareInfoDeconstructed, vendorReserved);
    NL_TEST_ASSERT(inSuite, err == CHIP_NO_ERROR);

    NL_TEST_ASSERT(inSuite, certificationDeclarationDeconstructed.data_equal(ByteSpan(certificationDeclarationTestVector)));
    NL_TEST_ASSERT(inSuite, attestationNonceDeconstructed.data_equal(ByteSpan(attestationNonceTestVector)));
    NL_TEST_ASSERT(inSuite, timestampTestVector == timestampDeconstructed);
    NL_TEST_ASSERT(inSuite, firmwareInfoDeconstructed.data_equal(ByteSpan(firmwareInfoTestVector)));
    NL_TEST_ASSERT(inSuite, ArraySize(vendorReservedArrayTestVector) == vendorReserved.GetNumberOfElements());
    struct VendorReservedElement element;
    size_t elementsSeen = 0;

    while (vendorReserved.GetNextVendorReservedElement(element) == CHIP_NO_ERROR)
    {
        NL_TEST_ASSERT(inSuite, vendorIdTestVector == element.vendorId);
        NL_TEST_ASSERT(inSuite, profileNumTestVector == element.profileNum);
        switch (element.tagNum)
        {
        case 1:
            NL_TEST_ASSERT(inSuite, element.vendorReservedData.data_equal(vendorReservedArrayTestVector[0]));
            elementsSeen++;
            break;
        case 3:
            NL_TEST_ASSERT(inSuite, element.vendorReservedData.data_equal(vendorReservedArrayTestVector[1]));
            elementsSeen++;
            break;
        default:
            NL_TEST_ASSERT(inSuite, 0);
            break;
        }
    }
    NL_TEST_ASSERT(inSuite, elementsSeen == ArraySize(vendorReservedArrayTestVector));
}

static void TestAttestationElements_DeconstructionUnordered(nlTestSuite * inSuite, void * inContext)
{
    CHIP_ERROR err = CHIP_NO_ERROR;

    static constexpr uint8_t attestationElementsUnorderedTestVector[] = {
        0x15, 0x30, 0x02, 0x20, 0xe0, 0x42, 0x1b, 0x91, 0xc6, 0xfd, 0xcd, 0xb4, 0x0e, 0x2a, 0x4d, 0x2c, 0xf3, 0x1d, 0xb2,
        0xb4, 0xe1, 0x8b, 0x41, 0x1b, 0x1d, 0x3a, 0xd4, 0xd1, 0x2a, 0x9d, 0x90, 0xaa, 0x8e, 0x52, 0xfa, 0xe2, 0x30, 0x01,
        0x70, 0xd2, 0x84, 0x4b, 0xa2, 0x01, 0x26, 0x04, 0x46, 0x63, 0x73, 0x61, 0x63, 0x64, 0x30, 0xa0, 0x58, 0x1d, 0x15,
        0x25, 0x01, 0x88, 0x99, 0x25, 0x02, 0xfe, 0xff, 0x25, 0x03, 0xd2, 0x04, 0x25, 0x04, 0x2e, 0x16, 0x24, 0x05, 0xaa,
        0x25, 0x06, 0xde, 0xc0, 0x25, 0x07, 0x94, 0x26, 0x18, 0x58, 0x40, 0x96, 0x57, 0x2d, 0xd6, 0x3c, 0x03, 0x64, 0x0b,
        0x28, 0x67, 0x02, 0xbd, 0x6b, 0xba, 0x48, 0xac, 0x7c, 0x83, 0x54, 0x9b, 0x68, 0x73, 0x29, 0x47, 0x48, 0xb9, 0x51,
        0xd5, 0xab, 0x66, 0x62, 0x2e, 0x9d, 0x26, 0x10, 0x41, 0xf8, 0x0e, 0x97, 0x49, 0xfe, 0xff, 0x78, 0x10, 0x02, 0x49,
        0x67, 0xae, 0xdf, 0x41, 0x38, 0x36, 0x5b, 0x0a, 0x22, 0x57, 0x14, 0x9c, 0x9a, 0x12, 0x3e, 0x0d, 0x30, 0xaa, 0x26,
        0x03, 0xfd, 0xc6, 0x5b, 0x28, 0xd0, 0xf1, 0xff, 0x3e, 0x00, 0x01, 0x00, 0x17, 0x73, 0x61, 0x6d, 0x70, 0x6c, 0x65,
        0x5f, 0x76, 0x65, 0x6e, 0x64, 0x6f, 0x72, 0x5f, 0x72, 0x65, 0x73, 0x65, 0x72, 0x76, 0x65, 0x64, 0x31, 0xd0, 0xf1,
        0xff, 0x3e, 0x00, 0x03, 0x00, 0x18, 0x76, 0x65, 0x6e, 0x64, 0x6f, 0x72, 0x5f, 0x72, 0x65, 0x73, 0x65, 0x72, 0x76,
        0x65, 0x64, 0x33, 0x5f, 0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x18
    };

    ByteSpan certificationDeclarationDeconstructed;
    ByteSpan attestationNonceDeconstructed;
    uint32_t timestampDeconstructed;
    ByteSpan firmwareInfoDeconstructed;

    DeviceAttestationVendorReservedDeconstructor vendorReserved;
    err = DeconstructAttestationElements(ByteSpan(attestationElementsUnorderedTestVector), certificationDeclarationDeconstructed,
                                         attestationNonceDeconstructed, timestampDeconstructed, firmwareInfoDeconstructed,
                                         vendorReserved);
    NL_TEST_ASSERT(inSuite, err == CHIP_ERROR_UNEXPECTED_TLV_ELEMENT);
}

static void TestAttestationElements_DeconstructionCorruptedTLV(nlTestSuite * inSuite, void * inContext)
{
    CHIP_ERROR err = CHIP_NO_ERROR;
    using chip::FormatCHIPError;

    // last element in the TLV 0x30 0x10 0x37 is engineered to generate a TLV
    // underrun error -- the element claims to be an octet string with a length
    // of 0x37 bytes but no data is actually present
    constexpr uint8_t attestationElementsCorruptedTLVTestVector[] = {
        0x15, 0x30, 0x01, 0x70, 0xd2, 0x84, 0x4b, 0xa2, 0x01, 0x26, 0x04, 0x46, 0x63, 0x73, 0x61, 0x63, 0x64, 0x30, 0xa0, 0x58,
        0x1d, 0x15, 0x25, 0x01, 0x88, 0x99, 0x25, 0x02, 0xfe, 0xff, 0x25, 0x03, 0xd2, 0x04, 0x25, 0x04, 0x2e, 0x16, 0x24, 0x05,
        0xaa, 0x25, 0x06, 0xde, 0xc0, 0x25, 0x07, 0x94, 0x26, 0x18, 0x58, 0x40, 0x96, 0x57, 0x2d, 0xd6, 0x3c, 0x03, 0x64, 0x0b,
        0x28, 0x67, 0x02, 0xbd, 0x6b, 0xba, 0x48, 0xac, 0x7c, 0x83, 0x54, 0x9b, 0x68, 0x73, 0x29, 0x47, 0x48, 0xb9, 0x51, 0xd5,
        0xab, 0x66, 0x62, 0x2e, 0x9d, 0x26, 0x10, 0x41, 0xf8, 0x0e, 0x97, 0x49, 0xfe, 0xff, 0x78, 0x10, 0x02, 0x49, 0x67, 0xae,
        0xdf, 0x41, 0x38, 0x36, 0x5b, 0x0a, 0x22, 0x57, 0x14, 0x9c, 0x9a, 0x12, 0x3e, 0x0d, 0x30, 0xaa, 0x30, 0x02, 0x20, 0xe0,
        0x42, 0x1b, 0x91, 0xc6, 0xfd, 0xcd, 0xb4, 0x0e, 0x2a, 0x4d, 0x2c, 0xf3, 0x1d, 0xb2, 0xb4, 0xe1, 0x8b, 0x41, 0x1b, 0x1d,
        0x3a, 0xd4, 0xd1, 0x2a, 0x9d, 0x90, 0xaa, 0x8e, 0x52, 0xfa, 0xe2, 0x26, 0x03, 0xfd, 0xc6, 0x5b, 0x28, 0x30, 0x10, 0x37
    };

    DeviceAttestationVendorReservedDeconstructor vendorReserved;
    size_t count = 2;
    err          = vendorReserved.PrepareToReadVendorReservedElements(ByteSpan(attestationElementsCorruptedTLVTestVector), count);
    NL_TEST_ASSERT(inSuite, err == CHIP_ERROR_TLV_UNDERRUN);
}

static void TestNocsrElements_Construction(nlTestSuite * inSuite, void * inContext)
{
    static constexpr uint8_t kNocsrNonce[] = {
        0x81, 0x4a, 0x4d, 0x4c, 0x1c, 0x4a, 0x8e, 0xbb, 0xea, 0xdb, 0x0a, 0xe2, 0x82, 0xf9, 0x91, 0xeb,
        0x13, 0xac, 0x5f, 0x9f, 0xce, 0x94, 0x30, 0x93, 0x19, 0xaa, 0x94, 0x09, 0x6c, 0x8c, 0xd4, 0xb8,
    };

    static constexpr uint8_t kExampleCsr[] = {
        0x30, 0x81, 0xda, 0x30, 0x81, 0x81, 0x02, 0x01, 0x00, 0x30, 0x0e, 0x31, 0x0c, 0x30, 0x0a, 0x06, 0x03, 0x55, 0x04,
        0x0a, 0x0c, 0x03, 0x43, 0x53, 0x41, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01,
        0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x5c, 0xa2, 0x79, 0xe3, 0x66,
        0x82, 0xc2, 0xd4, 0x6c, 0xe7, 0xd4, 0xcf, 0x89, 0x67, 0x84, 0x67, 0x08, 0xb5, 0xb9, 0xf8, 0x5b, 0x9c, 0xda, 0xfd,
        0x8c, 0xa8, 0x85, 0x26, 0x12, 0xcb, 0x0f, 0x0c, 0x7a, 0x71, 0x31, 0x4e, 0xc8, 0xdc, 0x9c, 0x96, 0x34, 0xdd, 0xee,
        0xfe, 0xe9, 0xf6, 0x3f, 0x0e, 0x8b, 0xd7, 0xda, 0xcf, 0xc3, 0xb6, 0xa4, 0x53, 0x2a, 0xad, 0xd8, 0x9a, 0x96, 0x51,
        0xcd, 0x6e, 0xa0, 0x11, 0x30, 0x0f, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x0e, 0x31, 0x02,
        0x30, 0x00, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x48, 0x00, 0x30, 0x45,
        0x02, 0x20, 0x0e, 0x67, 0x5e, 0xe1, 0xb3, 0xbb, 0xfe, 0x15, 0x2a, 0x17, 0x4a, 0xf5, 0x35, 0xe2, 0x2d, 0x55, 0xce,
        0x10, 0xc1, 0x50, 0xca, 0xc0, 0x1b, 0x31, 0x18, 0xde, 0x05, 0xe8, 0xfd, 0x9f, 0x10, 0x48, 0x02, 0x21, 0x00, 0xd8,
        0x8c, 0x57, 0xcc, 0x6e, 0x74, 0xf0, 0xe5, 0x48, 0x8a, 0x26, 0x16, 0x7a, 0x07, 0xfd, 0x6d, 0xbe, 0xf1, 0xaa, 0xad,
        0x72, 0x1c, 0x58, 0x0b, 0x6e, 0xae, 0x21, 0xbe, 0x5e, 0x6d, 0x0c, 0x72,
    };

    const uint8_t kVendorReserved1[] = {
        0x73, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x5f, 0x76, 0x65, 0x6e, 0x64, 0x6f,
        0x72, 0x5f, 0x72, 0x65, 0x73, 0x65, 0x72, 0x76, 0x65, 0x64, 0x31,
    };

    const uint8_t kVendorReserved3[] = {
        0x76, 0x65, 0x6e, 0x64, 0x6f, 0x72, 0x5f, 0x72, 0x65, 0x73, 0x65, 0x72,
        0x76, 0x65, 0x64, 0x33, 0x5f, 0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65,
    };

    const uint8_t kNoCsrElementsVector[] = {
        0x15, 0x30, 0x01, 0xdd, 0x30, 0x81, 0xda, 0x30, 0x81, 0x81, 0x02, 0x01, 0x00, 0x30, 0x0e, 0x31, 0x0c, 0x30, 0x0a, 0x06,
        0x03, 0x55, 0x04, 0x0a, 0x0c, 0x03, 0x43, 0x53, 0x41, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d,
        0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x5c, 0xa2, 0x79, 0xe3,
        0x66, 0x82, 0xc2, 0xd4, 0x6c, 0xe7, 0xd4, 0xcf, 0x89, 0x67, 0x84, 0x67, 0x08, 0xb5, 0xb9, 0xf8, 0x5b, 0x9c, 0xda, 0xfd,
        0x8c, 0xa8, 0x85, 0x26, 0x12, 0xcb, 0x0f, 0x0c, 0x7a, 0x71, 0x31, 0x4e, 0xc8, 0xdc, 0x9c, 0x96, 0x34, 0xdd, 0xee, 0xfe,
        0xe9, 0xf6, 0x3f, 0x0e, 0x8b, 0xd7, 0xda, 0xcf, 0xc3, 0xb6, 0xa4, 0x53, 0x2a, 0xad, 0xd8, 0x9a, 0x96, 0x51, 0xcd, 0x6e,
        0xa0, 0x11, 0x30, 0x0f, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x0e, 0x31, 0x02, 0x30, 0x00, 0x30,
        0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x48, 0x00, 0x30, 0x45, 0x02, 0x20, 0x0e, 0x67,
        0x5e, 0xe1, 0xb3, 0xbb, 0xfe, 0x15, 0x2a, 0x17, 0x4a, 0xf5, 0x35, 0xe2, 0x2d, 0x55, 0xce, 0x10, 0xc1, 0x50, 0xca, 0xc0,
        0x1b, 0x31, 0x18, 0xde, 0x05, 0xe8, 0xfd, 0x9f, 0x10, 0x48, 0x02, 0x21, 0x00, 0xd8, 0x8c, 0x57, 0xcc, 0x6e, 0x74, 0xf0,
        0xe5, 0x48, 0x8a, 0x26, 0x16, 0x7a, 0x07, 0xfd, 0x6d, 0xbe, 0xf1, 0xaa, 0xad, 0x72, 0x1c, 0x58, 0x0b, 0x6e, 0xae, 0x21,
        0xbe, 0x5e, 0x6d, 0x0c, 0x72, 0x30, 0x02, 0x20, 0x81, 0x4a, 0x4d, 0x4c, 0x1c, 0x4a, 0x8e, 0xbb, 0xea, 0xdb, 0x0a, 0xe2,
        0x82, 0xf9, 0x91, 0xeb, 0x13, 0xac, 0x5f, 0x9f, 0xce, 0x94, 0x30, 0x93, 0x19, 0xaa, 0x94, 0x09, 0x6c, 0x8c, 0xd4, 0xb8,
        0x30, 0x03, 0x17, 0x73, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x5f, 0x76, 0x65, 0x6e, 0x64, 0x6f, 0x72, 0x5f, 0x72, 0x65, 0x73,
        0x65, 0x72, 0x76, 0x65, 0x64, 0x31, 0x30, 0x05, 0x18, 0x76, 0x65, 0x6e, 0x64, 0x6f, 0x72, 0x5f, 0x72, 0x65, 0x73, 0x65,
        0x72, 0x76, 0x65, 0x64, 0x33, 0x5f, 0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x18,
    };

    Platform::ScopedMemoryBuffer<uint8_t> nocsrElements;
    size_t nocsrElementsLen =
        TLV::EstimateStructOverhead(sizeof(kExampleCsr), sizeof(kNocsrNonce), sizeof(kVendorReserved1), sizeof(kVendorReserved3));

    nocsrElements.Alloc(nocsrElementsLen);
    NL_TEST_ASSERT(inSuite, nocsrElements.Get() != nullptr);

    MutableByteSpan nocsrElementsSpan(nocsrElements.Get(), nocsrElementsLen);

    CHIP_ERROR err = ConstructNOCSRElements(ByteSpan(kExampleCsr), ByteSpan(kNocsrNonce), ByteSpan{ kVendorReserved1 }, ByteSpan{},
                                            ByteSpan{ kVendorReserved3 }, nocsrElementsSpan);
    NL_TEST_ASSERT(inSuite, err == CHIP_NO_ERROR);
    NL_TEST_ASSERT(inSuite, nocsrElementsSpan.size() <= nocsrElementsLen);
    NL_TEST_ASSERT(inSuite, nocsrElementsSpan.data_equal(ByteSpan(kNoCsrElementsVector)));
}

static void TestNocsrElements_Deconstruction(nlTestSuite * inSuite, void * inContext)
{
    static constexpr uint8_t kNocsrNonce[] = {
        0x81, 0x4a, 0x4d, 0x4c, 0x1c, 0x4a, 0x8e, 0xbb, 0xea, 0xdb, 0x0a, 0xe2, 0x82, 0xf9, 0x91, 0xeb,
        0x13, 0xac, 0x5f, 0x9f, 0xce, 0x94, 0x30, 0x93, 0x19, 0xaa, 0x94, 0x09, 0x6c, 0x8c, 0xd4, 0xb8,
    };

    static constexpr uint8_t kExampleCsr[] = {
        0x30, 0x81, 0xda, 0x30, 0x81, 0x81, 0x02, 0x01, 0x00, 0x30, 0x0e, 0x31, 0x0c, 0x30, 0x0a, 0x06, 0x03, 0x55, 0x04,
        0x0a, 0x0c, 0x03, 0x43, 0x53, 0x41, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01,
        0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x5c, 0xa2, 0x79, 0xe3, 0x66,
        0x82, 0xc2, 0xd4, 0x6c, 0xe7, 0xd4, 0xcf, 0x89, 0x67, 0x84, 0x67, 0x08, 0xb5, 0xb9, 0xf8, 0x5b, 0x9c, 0xda, 0xfd,
        0x8c, 0xa8, 0x85, 0x26, 0x12, 0xcb, 0x0f, 0x0c, 0x7a, 0x71, 0x31, 0x4e, 0xc8, 0xdc, 0x9c, 0x96, 0x34, 0xdd, 0xee,
        0xfe, 0xe9, 0xf6, 0x3f, 0x0e, 0x8b, 0xd7, 0xda, 0xcf, 0xc3, 0xb6, 0xa4, 0x53, 0x2a, 0xad, 0xd8, 0x9a, 0x96, 0x51,
        0xcd, 0x6e, 0xa0, 0x11, 0x30, 0x0f, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x0e, 0x31, 0x02,
        0x30, 0x00, 0x30, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x48, 0x00, 0x30, 0x45,
        0x02, 0x20, 0x0e, 0x67, 0x5e, 0xe1, 0xb3, 0xbb, 0xfe, 0x15, 0x2a, 0x17, 0x4a, 0xf5, 0x35, 0xe2, 0x2d, 0x55, 0xce,
        0x10, 0xc1, 0x50, 0xca, 0xc0, 0x1b, 0x31, 0x18, 0xde, 0x05, 0xe8, 0xfd, 0x9f, 0x10, 0x48, 0x02, 0x21, 0x00, 0xd8,
        0x8c, 0x57, 0xcc, 0x6e, 0x74, 0xf0, 0xe5, 0x48, 0x8a, 0x26, 0x16, 0x7a, 0x07, 0xfd, 0x6d, 0xbe, 0xf1, 0xaa, 0xad,
        0x72, 0x1c, 0x58, 0x0b, 0x6e, 0xae, 0x21, 0xbe, 0x5e, 0x6d, 0x0c, 0x72,
    };

    const uint8_t kVendorReserved1[] = {
        0x73, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x5f, 0x76, 0x65, 0x6e, 0x64, 0x6f,
        0x72, 0x5f, 0x72, 0x65, 0x73, 0x65, 0x72, 0x76, 0x65, 0x64, 0x31,
    };

    const uint8_t kVendorReserved3[] = {
        0x76, 0x65, 0x6e, 0x64, 0x6f, 0x72, 0x5f, 0x72, 0x65, 0x73, 0x65, 0x72,
        0x76, 0x65, 0x64, 0x33, 0x5f, 0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65,
    };

    const uint8_t kNoCsrElementsVector[] = {
        0x15, 0x30, 0x01, 0xdd, 0x30, 0x81, 0xda, 0x30, 0x81, 0x81, 0x02, 0x01, 0x00, 0x30, 0x0e, 0x31, 0x0c, 0x30, 0x0a, 0x06,
        0x03, 0x55, 0x04, 0x0a, 0x0c, 0x03, 0x43, 0x53, 0x41, 0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d,
        0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x5c, 0xa2, 0x79, 0xe3,
        0x66, 0x82, 0xc2, 0xd4, 0x6c, 0xe7, 0xd4, 0xcf, 0x89, 0x67, 0x84, 0x67, 0x08, 0xb5, 0xb9, 0xf8, 0x5b, 0x9c, 0xda, 0xfd,
        0x8c, 0xa8, 0x85, 0x26, 0x12, 0xcb, 0x0f, 0x0c, 0x7a, 0x71, 0x31, 0x4e, 0xc8, 0xdc, 0x9c, 0x96, 0x34, 0xdd, 0xee, 0xfe,
        0xe9, 0xf6, 0x3f, 0x0e, 0x8b, 0xd7, 0xda, 0xcf, 0xc3, 0xb6, 0xa4, 0x53, 0x2a, 0xad, 0xd8, 0x9a, 0x96, 0x51, 0xcd, 0x6e,
        0xa0, 0x11, 0x30, 0x0f, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x0e, 0x31, 0x02, 0x30, 0x00, 0x30,
        0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x04, 0x03, 0x02, 0x03, 0x48, 0x00, 0x30, 0x45, 0x02, 0x20, 0x0e, 0x67,
        0x5e, 0xe1, 0xb3, 0xbb, 0xfe, 0x15, 0x2a, 0x17, 0x4a, 0xf5, 0x35, 0xe2, 0x2d, 0x55, 0xce, 0x10, 0xc1, 0x50, 0xca, 0xc0,
        0x1b, 0x31, 0x18, 0xde, 0x05, 0xe8, 0xfd, 0x9f, 0x10, 0x48, 0x02, 0x21, 0x00, 0xd8, 0x8c, 0x57, 0xcc, 0x6e, 0x74, 0xf0,
        0xe5, 0x48, 0x8a, 0x26, 0x16, 0x7a, 0x07, 0xfd, 0x6d, 0xbe, 0xf1, 0xaa, 0xad, 0x72, 0x1c, 0x58, 0x0b, 0x6e, 0xae, 0x21,
        0xbe, 0x5e, 0x6d, 0x0c, 0x72, 0x30, 0x02, 0x20, 0x81, 0x4a, 0x4d, 0x4c, 0x1c, 0x4a, 0x8e, 0xbb, 0xea, 0xdb, 0x0a, 0xe2,
        0x82, 0xf9, 0x91, 0xeb, 0x13, 0xac, 0x5f, 0x9f, 0xce, 0x94, 0x30, 0x93, 0x19, 0xaa, 0x94, 0x09, 0x6c, 0x8c, 0xd4, 0xb8,
        0x30, 0x03, 0x17, 0x73, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x5f, 0x76, 0x65, 0x6e, 0x64, 0x6f, 0x72, 0x5f, 0x72, 0x65, 0x73,
        0x65, 0x72, 0x76, 0x65, 0x64, 0x31, 0x30, 0x05, 0x18, 0x76, 0x65, 0x6e, 0x64, 0x6f, 0x72, 0x5f, 0x72, 0x65, 0x73, 0x65,
        0x72, 0x76, 0x65, 0x64, 0x33, 0x5f, 0x65, 0x78, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x18,
    };

    ByteSpan csrSpan;
    ByteSpan csrNonceSpan;
    ByteSpan vendorReserved1Span;
    ByteSpan vendorReserved2Span;
    ByteSpan vendorReserved3Span;
    CHIP_ERROR err = DeconstructNOCSRElements(ByteSpan(kNoCsrElementsVector), csrSpan, csrNonceSpan, vendorReserved1Span,
                                              vendorReserved2Span, vendorReserved3Span);
    NL_TEST_ASSERT(inSuite, err == CHIP_NO_ERROR);
    NL_TEST_ASSERT(inSuite, csrSpan.data_equal(ByteSpan(kExampleCsr)));
    NL_TEST_ASSERT(inSuite, csrNonceSpan.data_equal(ByteSpan(kNocsrNonce)));
    NL_TEST_ASSERT(inSuite, vendorReserved1Span.data_equal(ByteSpan(kVendorReserved1)));
    NL_TEST_ASSERT(inSuite, vendorReserved2Span.empty());
    NL_TEST_ASSERT(inSuite, vendorReserved3Span.data_equal(ByteSpan(kVendorReserved3)));
}

/**
 *   Test Suite. It lists all the test functions.
 */
// clang-format off
static const nlTest sTests[] = {
    NL_TEST_DEF("Test Device Attestation Elements Roundtrip", TestAttestationElements_Roundtrip),
    NL_TEST_DEF("Test Device Attestation Elements Construction", TestAttestationElements_Construction),
    NL_TEST_DEF("Test Device Attestation Elements Deconstruction", TestAttestationElements_Deconstruction),
    NL_TEST_DEF("Test Vendor Reserved Data Ordering", TestVendorReservedData),
    NL_TEST_DEF("Test Device Attestation Elements Deconstruction with Firmware Information", TestAttestationElements_DeconstructionWithFirmwareInfo),
    NL_TEST_DEF("Test Device Attestation Elements Deconstruction - Corrupted/Out of Order TLV", TestAttestationElements_DeconstructionUnordered),
    NL_TEST_DEF("Test Device Attestation Elements Deconstruction - Corrupted TLV -- vendor reserved elements", TestAttestationElements_DeconstructionCorruptedTLV),
    NL_TEST_DEF("Test Device Attestation Elements Deconstruction - No vendor reserved", TestAttestationElements_DeconstructionNoVendorReserved),
    NL_TEST_DEF("Test Device NOCSR Elements Construction", TestNocsrElements_Construction),
    NL_TEST_DEF("Test Device NOCSR Elements Deconstruction", TestNocsrElements_Deconstruction),
    NL_TEST_SENTINEL()
};
// clang-format on

/**
 *  Set up the test suite.
 */
int TestDeviceAttestationElementsConstruction_Setup(void * inContext)
{
    CHIP_ERROR error = chip::Platform::MemoryInit();
    if (error != CHIP_NO_ERROR)
        return FAILURE;
    return SUCCESS;
}

/**
 *  Tear down the test suite.
 */
int TestDeviceAttestationElementsConstruction_Teardown(void * inContext)
{
    chip::Platform::MemoryShutdown();
    return SUCCESS;
}

int TestDeviceAttestationElementsConstruction()
{
    // clang-format off
    nlTestSuite theSuite =
    {
        "Device Attestation Elements Construction",
        &sTests[0],
        TestDeviceAttestationElementsConstruction_Setup,
        TestDeviceAttestationElementsConstruction_Teardown
    };
    // clang-format on
    nlTestRunner(&theSuite, nullptr);
    return (nlTestRunnerStats(&theSuite));
}

CHIP_REGISTER_TEST_SUITE(TestDeviceAttestationElementsConstruction);
