/*
 *
 *    Copyright (c) 2020 Project CHIP Authors
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

/**
 * @file - This file contains elliptic curve point multiplication & addition test vectors.
 */

#pragma once

namespace chip {
namespace Crypto {

struct spake2p_point_muladd_tv
{
    const uint8_t * point1;
    size_t point1_len;
    const uint8_t * scalar1;
    size_t scalar1_len;
    const uint8_t * point2;
    size_t point2_len;
    const uint8_t * scalar2;
    size_t scalar2_len;
    const uint8_t * out_point;
    size_t out_point_len;
    int valid;
};

static const uint8_t chiptest_fcf59b3113ef_point1_1[]    = { 0x04, 0x5d, 0xe7, 0x3c, 0x3b, 0x9c, 0x5a, 0x6f, 0x62, 0x83, 0x01,
                                                          0xdc, 0xd3, 0x84, 0x9f, 0x1b, 0x58, 0xa2, 0xd5, 0x44, 0x27, 0x58,
                                                          0x4f, 0xb3, 0xa2, 0xdf, 0xb8, 0x18, 0x25, 0x88, 0x17, 0x97, 0xa3,
                                                          0xf8, 0xbc, 0x49, 0xdd, 0xd3, 0xee, 0xd6, 0x55, 0x86, 0xec, 0xa4,
                                                          0x58, 0x93, 0x06, 0x9b, 0x79, 0x66, 0x07, 0xc1, 0x85, 0x46, 0x0b,
                                                          0xce, 0x6d, 0xe5, 0x42, 0x69, 0x70, 0x02, 0x12, 0x3b, 0x77 };
static const uint8_t chiptest_fcf59b3113ef_scalar1_2[]   = { 0xb1, 0x4b, 0x5e, 0x03, 0xfc, 0xbf, 0xd5, 0x07, 0x98, 0x96, 0x9c,
                                                           0x24, 0xc8, 0x0c, 0x77, 0xb6, 0x78, 0xa7, 0xc6, 0x6b, 0x9c, 0xce,
                                                           0x16, 0xef, 0x46, 0xe1, 0x1e, 0xbf, 0x6d, 0xd5, 0x2c, 0x00 };
static const uint8_t chiptest_fcf59b3113ef_point2_3[]    = { 0x04, 0xe5, 0x6d, 0xed, 0x91, 0x01, 0x0a, 0x92, 0x45, 0x0c, 0x5d,
                                                          0xef, 0x8a, 0xac, 0xd3, 0x3c, 0x09, 0xdd, 0x42, 0xb0, 0x2b, 0x4e,
                                                          0xc2, 0xb6, 0xf9, 0x25, 0x90, 0xd6, 0x19, 0x78, 0xd3, 0x84, 0x79,
                                                          0x76, 0x61, 0xc2, 0x72, 0x70, 0x90, 0xf3, 0x17, 0x49, 0x8e, 0xb9,
                                                          0xd7, 0xe4, 0x65, 0x3f, 0xbd, 0xc6, 0xef, 0x53, 0xdb, 0x0e, 0x5c,
                                                          0xd1, 0xe9, 0x17, 0xb4, 0x6e, 0x5f, 0x6c, 0xd3, 0x2e, 0x64 };
static const uint8_t chiptest_fcf59b3113ef_scalar2_4[]   = { 0x98, 0x97, 0xb6, 0x17, 0x69, 0xae, 0xa7, 0xfe, 0x82, 0x67, 0xe2,
                                                           0xbf, 0x19, 0x95, 0xfd, 0x09, 0xae, 0xe0, 0x02, 0x64, 0xd7, 0x22,
                                                           0x0d, 0x55, 0x06, 0x60, 0x7b, 0xcc, 0x20, 0xe5, 0xa0, 0xf1 };
static const uint8_t chiptest_fcf59b3113ef_out_point_5[] = { 0x04, 0x1b, 0x4a, 0xdb, 0x11, 0x30, 0xec, 0x0a, 0x72, 0xda, 0x84,
                                                             0x70, 0x93, 0xa1, 0x76, 0x35, 0x35, 0xa5, 0xe9, 0xe6, 0x92, 0x12,
                                                             0xfc, 0x5f, 0xee, 0xc6, 0x97, 0x27, 0xb2, 0x8c, 0x7c, 0x83, 0x77,
                                                             0x8b, 0xdb, 0x6f, 0x6c, 0xa1, 0xc5, 0x8f, 0x89, 0xc1, 0x6d, 0xfd,
                                                             0xf0, 0x20, 0xb3, 0x62, 0x97, 0xae, 0x6f, 0xc1, 0xf2, 0xa4, 0x16,
                                                             0xfe, 0x9d, 0xe8, 0x5a, 0x30, 0x1b, 0x89, 0x31, 0x00, 0x47 };
static const struct spake2p_point_muladd_tv chiptest_fcf59b3113ef_test_vector_6 = {
    .point1        = chiptest_fcf59b3113ef_point1_1,
    .point1_len    = 65,
    .scalar1       = chiptest_fcf59b3113ef_scalar1_2,
    .scalar1_len   = 32,
    .point2        = chiptest_fcf59b3113ef_point2_3,
    .point2_len    = 65,
    .scalar2       = chiptest_fcf59b3113ef_scalar2_4,
    .scalar2_len   = 32,
    .out_point     = chiptest_fcf59b3113ef_out_point_5,
    .out_point_len = 65,
};
static const uint8_t chiptest_fcf59b3113ef_point1_7[]     = { 0x04, 0x8e, 0xa4, 0x1c, 0xaa, 0x08, 0x06, 0x29, 0x8b, 0x95, 0x69,
                                                          0xc1, 0xc5, 0x80, 0x7f, 0x58, 0x17, 0x15, 0xa3, 0x66, 0x18, 0x8a,
                                                          0x67, 0xf8, 0x43, 0xa8, 0xcc, 0x93, 0x64, 0x14, 0x5f, 0x00, 0x4b,
                                                          0x7a, 0x9f, 0xef, 0xaf, 0xa2, 0x72, 0xe6, 0x8b, 0x08, 0x79, 0xa3,
                                                          0xc1, 0x93, 0xda, 0x96, 0xec, 0x48, 0xfe, 0xad, 0xe2, 0xd8, 0x6e,
                                                          0x50, 0x9a, 0x47, 0x92, 0xea, 0xde, 0x33, 0x83, 0xc9, 0xe9 };
static const uint8_t chiptest_fcf59b3113ef_scalar1_8[]    = { 0x9f, 0xbb, 0x0e, 0xc3, 0x4d, 0x11, 0xed, 0x91, 0x2a, 0x51, 0x72,
                                                           0x0e, 0x71, 0x30, 0x22, 0x32, 0xda, 0x76, 0x18, 0x9d, 0x21, 0x0c,
                                                           0xcb, 0xc1, 0x85, 0x38, 0x9d, 0x6d, 0xaa, 0x94, 0x76, 0x5f };
static const uint8_t chiptest_fcf59b3113ef_point2_9[]     = { 0x04, 0xe6, 0x45, 0xf8, 0xba, 0xbd, 0xb3, 0x3f, 0xd2, 0xe9, 0x55,
                                                          0x06, 0xcf, 0xd4, 0x5a, 0xee, 0x5a, 0x70, 0x37, 0xe1, 0x61, 0x90,
                                                          0xfb, 0x98, 0x3c, 0x33, 0x8a, 0x99, 0x79, 0x5f, 0xeb, 0x5e, 0x47,
                                                          0xbb, 0x0b, 0xfe, 0xc4, 0xac, 0x34, 0x7e, 0x82, 0xa4, 0x77, 0x9a,
                                                          0xdf, 0x98, 0x7e, 0xa1, 0x6a, 0x3d, 0xae, 0x73, 0xc8, 0xcf, 0x68,
                                                          0x06, 0x56, 0xeb, 0xec, 0x30, 0x30, 0x50, 0x3f, 0xfe, 0xa7 };
static const uint8_t chiptest_fcf59b3113ef_scalar2_10[]   = { 0x87, 0x8d, 0x31, 0xb4, 0x82, 0xf5, 0x6a, 0x32, 0x20, 0x21, 0xb7,
                                                            0xbb, 0x57, 0x76, 0xc0, 0xca, 0xa0, 0x34, 0x3f, 0x8b, 0x02, 0x89,
                                                            0xbe, 0xeb, 0xb9, 0x64, 0x4d, 0x45, 0x0c, 0x6a, 0x06 };
static const uint8_t chiptest_fcf59b3113ef_out_point_11[] = { 0x04, 0x5c, 0x60, 0xf0, 0x8c, 0x47, 0x57, 0x07, 0x23, 0xf5, 0x74,
                                                              0xcc, 0x01, 0x7b, 0x2b, 0x50, 0xd8, 0x66, 0xfe, 0x5d, 0x84, 0x46,
                                                              0xd2, 0x37, 0xde, 0xa2, 0x42, 0x33, 0x5c, 0xfe, 0x8d, 0xda, 0xcd,
                                                              0x63, 0xb6, 0xa8, 0x1f, 0x59, 0x5b, 0x9c, 0xbe, 0x4d, 0xd1, 0xa8,
                                                              0x54, 0xe9, 0x69, 0x15, 0xda, 0x1f, 0xa5, 0x7d, 0x35, 0x7e, 0xf6,
                                                              0x13, 0x48, 0x17, 0xcf, 0xc5, 0x37, 0x3c, 0x7c, 0x94, 0xbb };
static const struct spake2p_point_muladd_tv chiptest_fcf59b3113ef_test_vector_12 = {
    .point1        = chiptest_fcf59b3113ef_point1_7,
    .point1_len    = 65,
    .scalar1       = chiptest_fcf59b3113ef_scalar1_8,
    .scalar1_len   = 32,
    .point2        = chiptest_fcf59b3113ef_point2_9,
    .point2_len    = 65,
    .scalar2       = chiptest_fcf59b3113ef_scalar2_10,
    .scalar2_len   = 31,
    .out_point     = chiptest_fcf59b3113ef_out_point_11,
    .out_point_len = 65,
};
static const uint8_t chiptest_fcf59b3113ef_point1_13[]    = { 0x04, 0x90, 0x8a, 0xe0, 0x65, 0x97, 0xb4, 0x17, 0x02, 0xca, 0x2e,
                                                           0xc9, 0x7f, 0xbd, 0x5b, 0x63, 0xae, 0x15, 0x82, 0x46, 0x4a, 0x55,
                                                           0xc7, 0xe6, 0x5c, 0x76, 0xaf, 0x3f, 0xf6, 0xf3, 0xb5, 0x7f, 0x56,
                                                           0xeb, 0xa2, 0x64, 0x03, 0x4c, 0x98, 0x8e, 0x1c, 0xdd, 0x2c, 0x8f,
                                                           0xe2, 0x98, 0x44, 0x8c, 0xcc, 0x12, 0x0c, 0xcf, 0x12, 0xb1, 0x97,
                                                           0x6d, 0x55, 0x92, 0xa5, 0x2c, 0x03, 0x3a, 0xd1, 0x0f, 0xa0 };
static const uint8_t chiptest_fcf59b3113ef_scalar1_14[]   = { 0x2e, 0x56, 0x81, 0xc3, 0xe0, 0x63, 0x4d, 0xfb, 0x15, 0x3a, 0xdc,
                                                            0x38, 0xb2, 0x6e, 0x96, 0x69, 0x44, 0x9c, 0x45, 0x11, 0x3d, 0x0c,
                                                            0xf1, 0x9f, 0x5c, 0x23, 0x35, 0x6c, 0x18, 0xb5, 0xc0, 0x31 };
static const uint8_t chiptest_fcf59b3113ef_point2_15[]    = { 0x04, 0x78, 0xdc, 0xaa, 0x28, 0x27, 0x02, 0xcf, 0x4a, 0x0f, 0x89,
                                                           0x19, 0xf0, 0x72, 0x9c, 0x19, 0xce, 0xee, 0x6f, 0xa9, 0x00, 0x84,
                                                           0x0a, 0xd4, 0x88, 0x0d, 0x35, 0x28, 0xf0, 0x80, 0x12, 0x7c, 0x8e,
                                                           0xbe, 0x0f, 0xad, 0xf7, 0x9f, 0x0a, 0xc4, 0x02, 0x84, 0x21, 0xf8,
                                                           0x38, 0x8f, 0xdd, 0x39, 0x48, 0x56, 0x7b, 0x07, 0xd9, 0x15, 0xc2,
                                                           0x19, 0xec, 0x95, 0x60, 0x5e, 0xbb, 0x2c, 0x22, 0x57, 0x0d };
static const uint8_t chiptest_fcf59b3113ef_scalar2_16[]   = { 0x17, 0x52, 0xb9, 0x70, 0xa6, 0x7a, 0x43, 0xaf, 0x9e, 0x3f, 0x5d,
                                                            0x68, 0xe3, 0xa4, 0x7c, 0x29, 0x83, 0x28, 0x81, 0x6c, 0xaa, 0x3b,
                                                            0xa4, 0xcf, 0x1c, 0xdf, 0x45, 0x64, 0xf5, 0x08, 0x04, 0x5e };
static const uint8_t chiptest_fcf59b3113ef_out_point_17[] = { 0x04, 0xf6, 0xcd, 0xf5, 0xbd, 0xc0, 0x70, 0xd0, 0xd7, 0x92, 0xa4,
                                                              0x9b, 0x6a, 0x09, 0x64, 0xd3, 0xaf, 0x26, 0x78, 0x28, 0xa4, 0xb4,
                                                              0xbf, 0x47, 0x6e, 0x95, 0xe4, 0x5b, 0x4e, 0x3e, 0x93, 0x85, 0xf4,
                                                              0x90, 0xc8, 0xf8, 0xea, 0x19, 0xff, 0x11, 0x70, 0x6f, 0xb9, 0x89,
                                                              0x45, 0x1f, 0xa6, 0x5d, 0x97, 0xd4, 0x5c, 0x2d, 0x19, 0x87, 0x73,
                                                              0x73, 0xd4, 0x01, 0x8d, 0xcc, 0x51, 0xee, 0xa0, 0xf7, 0xae };
static const struct spake2p_point_muladd_tv chiptest_fcf59b3113ef_test_vector_18 = {
    .point1        = chiptest_fcf59b3113ef_point1_13,
    .point1_len    = 65,
    .scalar1       = chiptest_fcf59b3113ef_scalar1_14,
    .scalar1_len   = 32,
    .point2        = chiptest_fcf59b3113ef_point2_15,
    .point2_len    = 65,
    .scalar2       = chiptest_fcf59b3113ef_scalar2_16,
    .scalar2_len   = 32,
    .out_point     = chiptest_fcf59b3113ef_out_point_17,
    .out_point_len = 65,
};
static const uint8_t chiptest_fcf59b3113ef_point1_19[]    = { 0x04, 0x04, 0x05, 0x2b, 0x2c, 0xc8, 0xef, 0xb8, 0xc2, 0xad, 0xfd,
                                                           0x12, 0x67, 0xc6, 0xf1, 0xc1, 0xcb, 0x34, 0x49, 0xae, 0x6e, 0x09,
                                                           0x5e, 0xf2, 0x54, 0x5c, 0x91, 0x48, 0x66, 0xe8, 0x55, 0xf0, 0x9d,
                                                           0x7c, 0xdb, 0xa3, 0x2d, 0x86, 0x32, 0x03, 0x13, 0x34, 0x2d, 0xbb,
                                                           0x38, 0x0d, 0x0a, 0x16, 0xcd, 0x65, 0xa5, 0xdd, 0x17, 0xc6, 0xa0,
                                                           0x46, 0x77, 0xd1, 0xa9, 0x9b, 0xaf, 0x32, 0x3b, 0x45, 0x60 };
static const uint8_t chiptest_fcf59b3113ef_scalar1_20[]   = { 0x39, 0x95, 0xe8, 0x7e, 0xd1, 0x30, 0x5d, 0x34, 0x95, 0x4c, 0x00,
                                                            0x1b, 0xcd, 0x68, 0xd9, 0xb0, 0x5f, 0xae, 0x71, 0x82, 0x19, 0xf9,
                                                            0xa5, 0x6a, 0x11, 0x96, 0x01, 0xac, 0x18, 0xa6, 0x1a, 0xf4 };
static const uint8_t chiptest_fcf59b3113ef_point2_21[]    = { 0x04, 0x16, 0x33, 0xcd, 0xc0, 0xd3, 0x03, 0x7e, 0xa0, 0x7d, 0xb6,
                                                           0x75, 0x02, 0x44, 0xfd, 0x56, 0xa1, 0x7f, 0x1f, 0xb0, 0xe4, 0x2a,
                                                           0xf5, 0xb5, 0xb2, 0x86, 0xf2, 0x92, 0xe9, 0xcf, 0xd5, 0xb9, 0x19,
                                                           0x66, 0x23, 0x8b, 0x12, 0xf3, 0x1e, 0x4c, 0x07, 0x7a, 0x26, 0x2c,
                                                           0xa6, 0x92, 0x86, 0x9d, 0x8e, 0x2d, 0x49, 0x31, 0x77, 0x13, 0x37,
                                                           0x09, 0x57, 0xe5, 0x70, 0x25, 0xa3, 0x2d, 0xc6, 0x71, 0xb9 };
static const uint8_t chiptest_fcf59b3113ef_scalar2_22[]   = { 0x94, 0xcd, 0x9d, 0xe5, 0x38, 0xf2, 0xac, 0xff, 0x85, 0x44, 0xa1,
                                                            0x75, 0xd7, 0xed, 0xad, 0x1c, 0x5a, 0xca, 0xa4, 0xa1, 0x55, 0x89,
                                                            0xf5, 0x6f, 0x9d, 0xe2, 0xb0, 0xae, 0xef, 0x3c, 0xaf, 0x44 };
static const uint8_t chiptest_fcf59b3113ef_out_point_23[] = { 0x04, 0x0d, 0xce, 0x4a, 0x77, 0x28, 0xfa, 0xfb, 0x9e, 0xa1, 0x89,
                                                              0x08, 0xcc, 0xae, 0x5b, 0x1c, 0x9d, 0x68, 0x17, 0x48, 0x30, 0x8b,
                                                              0x6b, 0xf2, 0x68, 0xc2, 0x65, 0xdd, 0xe6, 0x84, 0x03, 0x82, 0x8d,
                                                              0x86, 0xb4, 0x08, 0x3e, 0x6d, 0xb2, 0xdf, 0x42, 0xee, 0xd9, 0x3d,
                                                              0xd1, 0x27, 0x26, 0x5b, 0xd6, 0x7a, 0x28, 0x69, 0x5e, 0xb8, 0x3a,
                                                              0xaf, 0xd1, 0x78, 0xe6, 0xa0, 0x3a, 0xb4, 0xea, 0xd0, 0x0a };
static const struct spake2p_point_muladd_tv chiptest_fcf59b3113ef_test_vector_24 = {
    .point1        = chiptest_fcf59b3113ef_point1_19,
    .point1_len    = 65,
    .scalar1       = chiptest_fcf59b3113ef_scalar1_20,
    .scalar1_len   = 32,
    .point2        = chiptest_fcf59b3113ef_point2_21,
    .point2_len    = 65,
    .scalar2       = chiptest_fcf59b3113ef_scalar2_22,
    .scalar2_len   = 32,
    .out_point     = chiptest_fcf59b3113ef_out_point_23,
    .out_point_len = 65,
};
static const uint8_t chiptest_fcf59b3113ef_point1_25[]    = { 0x04, 0xa1, 0xe5, 0x1c, 0xcf, 0x92, 0x59, 0x79, 0x0d, 0xe7, 0xce,
                                                           0x64, 0x18, 0x53, 0x9c, 0x2f, 0x43, 0x30, 0x54, 0x68, 0x8d, 0x5b,
                                                           0x61, 0x97, 0xe8, 0x15, 0x5e, 0x0a, 0x28, 0xad, 0x50, 0xb4, 0xa3,
                                                           0x41, 0x0e, 0xe4, 0x65, 0xe0, 0xad, 0x74, 0xc3, 0x20, 0xf7, 0xdd,
                                                           0x5e, 0xbf, 0xea, 0xa4, 0xac, 0x6d, 0x0c, 0xb4, 0xf3, 0x25, 0x82,
                                                           0xaf, 0x99, 0x55, 0x6a, 0x0f, 0xcc, 0x0d, 0x9c, 0x38, 0xb2 };
static const uint8_t chiptest_fcf59b3113ef_scalar1_26[]   = { 0xe5, 0x9f, 0xca, 0x67, 0x6c, 0x9e, 0x6d, 0x89, 0xa4, 0xa8, 0x95,
                                                            0xe1, 0x4a, 0x6a, 0x36, 0x77, 0xbc, 0x25, 0x8a, 0x4e, 0xf0, 0x96,
                                                            0x6a, 0x80, 0xf3, 0x27, 0xe3, 0x08, 0x70, 0x6c, 0xf7, 0xc0 };
static const uint8_t chiptest_fcf59b3113ef_point2_27[]    = { 0x04, 0xe2, 0x69, 0x7d, 0x01, 0x24, 0xec, 0x02, 0xaf, 0x50, 0x80,
                                                           0x52, 0xc1, 0x83, 0x2b, 0xad, 0x8c, 0x2e, 0x44, 0xba, 0x2a, 0xc2,
                                                           0x49, 0x1f, 0x43, 0x1f, 0xae, 0x4f, 0x67, 0xfe, 0xeb, 0xa5, 0x1b,
                                                           0x62, 0x84, 0x84, 0x41, 0x94, 0x79, 0x4d, 0x11, 0xc8, 0xd8, 0xe1,
                                                           0x2c, 0xc2, 0xa6, 0x50, 0x23, 0xed, 0x6e, 0x44, 0xc5, 0xc9, 0xa7,
                                                           0x4a, 0xda, 0x0f, 0x29, 0x61, 0x4a, 0x72, 0x3d, 0x6a, 0xf3 };
static const uint8_t chiptest_fcf59b3113ef_scalar2_28[]   = { 0x7f, 0x97, 0xa2, 0x28, 0xac, 0x84, 0xf8, 0x4f, 0x86, 0x4f, 0x5e,
                                                            0x11, 0x81, 0x02, 0x40, 0xdf, 0x2e, 0xa1, 0xd0, 0x6b, 0xb9, 0xce,
                                                            0x36, 0x2f, 0x75, 0xc8, 0x56, 0x5c, 0x38, 0xb2, 0x87, 0x18 };
static const uint8_t chiptest_fcf59b3113ef_out_point_29[] = { 0x04, 0x40, 0xa6, 0xc1, 0xc2, 0x00, 0x05, 0x4c, 0x39, 0x6f, 0xbc,
                                                              0xee, 0xab, 0x78, 0x44, 0x77, 0xb0, 0x79, 0x79, 0x95, 0x61, 0x6d,
                                                              0xfe, 0x63, 0xba, 0xfe, 0x80, 0x22, 0x24, 0xbe, 0x09, 0x1a, 0xec,
                                                              0xe0, 0x39, 0xd5, 0x5c, 0x0f, 0xfd, 0x04, 0x59, 0xa7, 0x94, 0x1d,
                                                              0xc2, 0x10, 0x66, 0x89, 0xb6, 0xc7, 0x1b, 0x99, 0xa9, 0x83, 0xcb,
                                                              0x65, 0x85, 0x67, 0x35, 0x88, 0x6e, 0x80, 0x57, 0x61, 0xc8 };
static const struct spake2p_point_muladd_tv chiptest_fcf59b3113ef_test_vector_30 = {
    .point1        = chiptest_fcf59b3113ef_point1_25,
    .point1_len    = 65,
    .scalar1       = chiptest_fcf59b3113ef_scalar1_26,
    .scalar1_len   = 32,
    .point2        = chiptest_fcf59b3113ef_point2_27,
    .point2_len    = 65,
    .scalar2       = chiptest_fcf59b3113ef_scalar2_28,
    .scalar2_len   = 32,
    .out_point     = chiptest_fcf59b3113ef_out_point_29,
    .out_point_len = 65,
};
static const uint8_t chiptest_fcf59b3113ef_point1_31[]    = { 0x04, 0x61, 0xf0, 0x79, 0x37, 0x77, 0x2f, 0xda, 0x0d, 0x5e, 0xe9,
                                                           0xd2, 0x05, 0x61, 0x54, 0x61, 0x7b, 0x12, 0xb4, 0x81, 0xad, 0xe9,
                                                           0x3c, 0x68, 0x6e, 0x13, 0x33, 0xf9, 0xb3, 0x69, 0x36, 0xcd, 0xb1,
                                                           0xc2, 0x20, 0xe0, 0x80, 0x4d, 0xd3, 0x7c, 0x4e, 0x68, 0x36, 0x25,
                                                           0x71, 0xca, 0x9f, 0x52, 0xc1, 0x05, 0xd4, 0x0f, 0x13, 0x92, 0x80,
                                                           0x3b, 0x55, 0xc9, 0xc8, 0x55, 0x26, 0xd7, 0xeb, 0x68, 0xd5 };
static const uint8_t chiptest_fcf59b3113ef_scalar1_32[]   = { 0x4a, 0x2d, 0x7f, 0x72, 0xc4, 0x2f, 0x2e, 0xe5, 0xa4, 0x67, 0x1e,
                                                            0xb2, 0xbf, 0x90, 0x6f, 0x56, 0x79, 0x06, 0x8c, 0x58, 0x10, 0x16,
                                                            0x46, 0xbb, 0x02, 0xa5, 0xff, 0x36, 0xf2, 0x9f, 0x6d, 0x70 };
static const uint8_t chiptest_fcf59b3113ef_point2_33[]    = { 0x04, 0x56, 0xb2, 0xd9, 0xa7, 0x4d, 0x0b, 0x74, 0x4a, 0xe7, 0x79,
                                                           0x3a, 0xb6, 0x25, 0xcf, 0x39, 0xf1, 0xdc, 0x2b, 0xc0, 0x00, 0xe4,
                                                           0x39, 0xdd, 0x35, 0x37, 0x96, 0x8e, 0x30, 0xa6, 0xea, 0xba, 0x65,
                                                           0x65, 0x58, 0x9d, 0x63, 0xcf, 0xe2, 0x86, 0x0a, 0x59, 0x9a, 0x8c,
                                                           0xce, 0x18, 0x3d, 0x59, 0x8b, 0x9b, 0x2c, 0x3a, 0xbc, 0xb4, 0xc9,
                                                           0x1e, 0xed, 0x2c, 0x26, 0x63, 0xee, 0x77, 0x37, 0x86, 0x7f };
static const uint8_t chiptest_fcf59b3113ef_scalar2_34[]   = { 0x37, 0x84, 0x30, 0xa3, 0xae, 0x9f, 0x0e, 0xdf, 0x78, 0xdd, 0x43,
                                                            0x08, 0x02, 0x60, 0xae, 0x91, 0xa4, 0x58, 0xca, 0x08, 0x04, 0x61,
                                                            0x0b, 0x7c, 0x46, 0x61, 0x20, 0x6e, 0x5d, 0xb3, 0x98, 0xbb };
static const uint8_t chiptest_fcf59b3113ef_out_point_35[] = { 0x04, 0xd0, 0x3f, 0xe0, 0x2f, 0x58, 0x4e, 0x32, 0x09, 0x2d, 0x91,
                                                              0x4c, 0x60, 0xba, 0xe3, 0x6a, 0xaf, 0xac, 0x6e, 0x62, 0x09, 0x41,
                                                              0x9c, 0x9f, 0xea, 0xab, 0xe7, 0x69, 0x03, 0xa4, 0x86, 0x7e, 0xd6,
                                                              0x8f, 0xab, 0xc2, 0x38, 0x3b, 0xa7, 0x29, 0xe8, 0xe2, 0xe8, 0x1f,
                                                              0x75, 0x17, 0xd6, 0xf8, 0x35, 0x98, 0x4d, 0xdd, 0xf1, 0x79, 0x27,
                                                              0x23, 0xae, 0x16, 0xa0, 0xa3, 0x7f, 0x6a, 0x16, 0x1b, 0xa3 };
static const struct spake2p_point_muladd_tv chiptest_fcf59b3113ef_test_vector_36 = {
    .point1        = chiptest_fcf59b3113ef_point1_31,
    .point1_len    = 65,
    .scalar1       = chiptest_fcf59b3113ef_scalar1_32,
    .scalar1_len   = 32,
    .point2        = chiptest_fcf59b3113ef_point2_33,
    .point2_len    = 65,
    .scalar2       = chiptest_fcf59b3113ef_scalar2_34,
    .scalar2_len   = 32,
    .out_point     = chiptest_fcf59b3113ef_out_point_35,
    .out_point_len = 65,
};
static const uint8_t chiptest_fcf59b3113ef_point1_37[]    = { 0x04, 0x19, 0x5c, 0xe1, 0x33, 0x2f, 0x34, 0xbb, 0x82, 0x45, 0x43,
                                                           0xef, 0x9e, 0x99, 0x41, 0x9f, 0xd2, 0xde, 0xd7, 0xd9, 0xd5, 0x4b,
                                                           0x5f, 0x55, 0x98, 0xeb, 0xaa, 0xe2, 0xee, 0x60, 0x01, 0x3b, 0xef,
                                                           0xe2, 0x08, 0xf8, 0xae, 0x81, 0x31, 0x11, 0x1f, 0xe6, 0xad, 0x86,
                                                           0x10, 0x45, 0x12, 0x5b, 0x84, 0x50, 0x49, 0xbc, 0x34, 0x19, 0xf7,
                                                           0x39, 0xec, 0x4c, 0x17, 0xbe, 0x13, 0x2c, 0xc8, 0xbc, 0x2c };
static const uint8_t chiptest_fcf59b3113ef_scalar1_38[]   = { 0xc1, 0x9d, 0xa9, 0x8f, 0x1a, 0x9c, 0xd3, 0x3f, 0x7b, 0xfe, 0xd5,
                                                            0xb8, 0xfa, 0x1b, 0x57, 0x3e, 0xc0, 0x4c, 0x60, 0x35, 0xbd, 0xb5,
                                                            0x66, 0xb6, 0x34, 0xd7, 0xa7, 0xe5, 0xe9, 0xd0, 0xbe, 0x85 };
static const uint8_t chiptest_fcf59b3113ef_point2_39[]    = { 0x04, 0x3e, 0xae, 0x00, 0xab, 0x5c, 0xe8, 0x14, 0x33, 0x12, 0x47,
                                                           0xff, 0xc9, 0xab, 0x97, 0x51, 0xd1, 0x7d, 0xdc, 0xdb, 0xd7, 0xd8,
                                                           0x26, 0x2c, 0xb7, 0xf8, 0x1f, 0xd1, 0xbd, 0x59, 0x4e, 0x21, 0x4e,
                                                           0x20, 0x39, 0xa9, 0x2e, 0xeb, 0xbb, 0xd1, 0x57, 0x12, 0xfd, 0x91,
                                                           0xa9, 0x4f, 0xe6, 0xc3, 0x24, 0x2c, 0x4d, 0xe5, 0xa8, 0x1a, 0x2a,
                                                           0x6d, 0x45, 0xee, 0x04, 0xb1, 0x31, 0xdc, 0xe7, 0xa7, 0xdf };
static const uint8_t chiptest_fcf59b3113ef_scalar2_40[]   = { 0x5b, 0x64, 0x97, 0x39, 0x8f, 0x94, 0x83, 0xd0, 0xac, 0x97, 0xb4,
                                                            0xaa, 0x04, 0x49, 0xd1, 0x2f, 0x51, 0xef, 0x42, 0x32, 0x7b, 0x1a,
                                                            0x22, 0xf7, 0x88, 0xec, 0x1c, 0x1d, 0xd3, 0xb1, 0x6e, 0xce };
static const uint8_t chiptest_fcf59b3113ef_out_point_41[] = { 0x04, 0xa9, 0x05, 0xa7, 0x08, 0x8a, 0xeb, 0x82, 0xcd, 0xe9, 0xe8,
                                                              0x54, 0x65, 0x43, 0x5e, 0xa7, 0xe6, 0x95, 0x7b, 0x7a, 0x90, 0x11,
                                                              0xe5, 0x90, 0xe9, 0x96, 0xae, 0x70, 0x48, 0x68, 0xef, 0xc1, 0x81,
                                                              0x0b, 0xfb, 0x2e, 0x50, 0x51, 0xbc, 0xb1, 0x41, 0x0b, 0x4c, 0xd1,
                                                              0xac, 0x99, 0x1f, 0xac, 0x0b, 0x09, 0x05, 0x61, 0x28, 0xbf, 0x8f,
                                                              0x3d, 0x34, 0x01, 0x99, 0x30, 0x60, 0xa5, 0x11, 0x2b, 0x94 };
static const struct spake2p_point_muladd_tv chiptest_fcf59b3113ef_test_vector_42 = {
    .point1        = chiptest_fcf59b3113ef_point1_37,
    .point1_len    = 65,
    .scalar1       = chiptest_fcf59b3113ef_scalar1_38,
    .scalar1_len   = 32,
    .point2        = chiptest_fcf59b3113ef_point2_39,
    .point2_len    = 65,
    .scalar2       = chiptest_fcf59b3113ef_scalar2_40,
    .scalar2_len   = 32,
    .out_point     = chiptest_fcf59b3113ef_out_point_41,
    .out_point_len = 65,
};
static const uint8_t chiptest_fcf59b3113ef_point1_43[]    = { 0x04, 0x1d, 0xe7, 0x29, 0x26, 0x0b, 0x7a, 0x6e, 0x8f, 0x5d, 0x54,
                                                           0x91, 0x92, 0x92, 0x12, 0x82, 0xbb, 0xd0, 0xc2, 0xbb, 0x1d, 0xf3,
                                                           0x62, 0xed, 0x85, 0x2a, 0xd1, 0x1e, 0xcf, 0x0e, 0x34, 0x06, 0x91,
                                                           0x2d, 0xb7, 0xe0, 0x2d, 0x61, 0x8d, 0x1f, 0xbd, 0x57, 0x4c, 0xfa,
                                                           0x01, 0x27, 0x28, 0x32, 0x40, 0x70, 0xce, 0xf6, 0x11, 0xc5, 0x3e,
                                                           0xac, 0x14, 0x07, 0xba, 0x1a, 0x46, 0x56, 0xa7, 0x07, 0x5a };
static const uint8_t chiptest_fcf59b3113ef_scalar1_44[]   = { 0xd2, 0x97, 0x77, 0x2d, 0x0c, 0x72, 0x50, 0x47, 0xf6, 0x5d, 0xe6,
                                                            0x3a, 0x78, 0x43, 0x8d, 0xef, 0x33, 0x8f, 0x3e, 0x87, 0xb7, 0x4b,
                                                            0x01, 0x22, 0xcf, 0x72, 0x8f, 0x88, 0x56, 0x63, 0x70, 0xe0 };
static const uint8_t chiptest_fcf59b3113ef_point2_45[]    = { 0x04, 0xd2, 0xcd, 0x9d, 0xae, 0xd6, 0x93, 0xcf, 0xf6, 0xcd, 0x61,
                                                           0xbc, 0xd2, 0x1c, 0xbe, 0x1e, 0xaa, 0x51, 0x3a, 0x33, 0x22, 0x9f,
                                                           0x58, 0x3d, 0x7e, 0xca, 0x53, 0x95, 0xd9, 0x80, 0xdb, 0x42, 0x9e,
                                                           0xf6, 0x35, 0x66, 0xe9, 0x06, 0xb3, 0x68, 0x5a, 0xca, 0xcd, 0xab,
                                                           0x56, 0x87, 0xd0, 0xd9, 0xb3, 0xcf, 0x60, 0xf1, 0x8d, 0x38, 0x2c,
                                                           0x52, 0x1a, 0x71, 0x69, 0xb5, 0x83, 0x60, 0xa1, 0xbc, 0x36 };
static const uint8_t chiptest_fcf59b3113ef_scalar2_46[]   = { 0xbb, 0xf4, 0xf2, 0x78, 0x67, 0x24, 0xc1, 0xed, 0x8c, 0x47, 0x6c,
                                                            0x34, 0x17, 0xff, 0x4c, 0xcd, 0x83, 0x97, 0x6f, 0xbd, 0x77, 0xc2,
                                                            0xde, 0x5f, 0x60, 0x24, 0x85, 0x88, 0x20, 0x77, 0xf0, 0xa9 };
static const uint8_t chiptest_fcf59b3113ef_out_point_47[] = { 0x04, 0x05, 0xd5, 0x25, 0x05, 0xec, 0x51, 0xf9, 0x65, 0xb9, 0xcf,
                                                              0x9f, 0x73, 0x7d, 0x9e, 0x6a, 0xfd, 0x5b, 0x4d, 0x50, 0xfa, 0xd8,
                                                              0xb0, 0xc8, 0xba, 0xbb, 0x96, 0xd0, 0x52, 0x95, 0x14, 0xf6, 0x7a,
                                                              0xfe, 0x25, 0x94, 0x81, 0x29, 0x7d, 0x09, 0xe7, 0xa9, 0xc4, 0x44,
                                                              0xb6, 0x0c, 0x56, 0xbf, 0x41, 0xc6, 0x96, 0x1f, 0x62, 0xce, 0x32,
                                                              0x44, 0xee, 0x08, 0xfd, 0xcc, 0x91, 0x4c, 0xe5, 0x33, 0x19 };
static const struct spake2p_point_muladd_tv chiptest_fcf59b3113ef_test_vector_48 = {
    .point1        = chiptest_fcf59b3113ef_point1_43,
    .point1_len    = 65,
    .scalar1       = chiptest_fcf59b3113ef_scalar1_44,
    .scalar1_len   = 32,
    .point2        = chiptest_fcf59b3113ef_point2_45,
    .point2_len    = 65,
    .scalar2       = chiptest_fcf59b3113ef_scalar2_46,
    .scalar2_len   = 32,
    .out_point     = chiptest_fcf59b3113ef_out_point_47,
    .out_point_len = 65,
};
static const uint8_t chiptest_fcf59b3113ef_point1_49[]    = { 0x04, 0xe4, 0x96, 0x23, 0x9b, 0xa5, 0x1a, 0x04, 0x6d, 0xf4, 0xad,
                                                           0x40, 0x8e, 0x68, 0x7e, 0x56, 0x51, 0x5f, 0xd3, 0xd5, 0x0d, 0x26,
                                                           0xa4, 0x99, 0x12, 0x84, 0x7e, 0xb4, 0x3a, 0x71, 0xe2, 0xfe, 0xb3,
                                                           0x74, 0x24, 0xbd, 0x48, 0x1d, 0x46, 0x15, 0x1f, 0x25, 0x61, 0xc1,
                                                           0x52, 0xd1, 0x44, 0x4b, 0xcf, 0xc5, 0xd6, 0xe7, 0x00, 0x10, 0x48,
                                                           0xe5, 0x0d, 0x00, 0x8d, 0xc8, 0xd8, 0x6f, 0x00, 0xe6, 0xd4 };
static const uint8_t chiptest_fcf59b3113ef_scalar1_50[]   = { 0xce, 0x6b, 0xea, 0xe7, 0xa6, 0xc6, 0xe9, 0xb9, 0xb7, 0x19, 0x8d,
                                                            0xa8, 0xee, 0x40, 0x2a, 0x8a, 0xce, 0xd0, 0x81, 0xce, 0xbe, 0x21,
                                                            0x5b, 0xfb, 0x73, 0x02, 0x73, 0xb8, 0x4d, 0x94, 0xd2, 0x9f };
static const uint8_t chiptest_fcf59b3113ef_point2_51[]    = { 0x04, 0xfd, 0x81, 0x98, 0x3d, 0xed, 0xf6, 0xa9, 0xa5, 0xcd, 0x7c,
                                                           0x5f, 0x87, 0x5f, 0x42, 0x1f, 0xd6, 0x67, 0x2a, 0xfe, 0x82, 0xa5,
                                                           0xa2, 0x44, 0x04, 0xfb, 0xdb, 0xd5, 0xb1, 0x75, 0x76, 0xbc, 0xb5,
                                                           0x66, 0xef, 0x05, 0xe6, 0x3f, 0x5e, 0x74, 0x11, 0xfe, 0x09, 0x67,
                                                           0x38, 0xc4, 0xbd, 0x22, 0xe7, 0xf9, 0x3e, 0xbb, 0x72, 0x3a, 0x99,
                                                           0xfd, 0xbe, 0x65, 0x32, 0x96, 0xdc, 0x2e, 0xc6, 0xa6, 0xfd };
static const uint8_t chiptest_fcf59b3113ef_scalar2_52[]   = { 0xe6, 0xdc, 0x4c, 0x0e, 0x62, 0xf5, 0x4e, 0x8a, 0xa0, 0xc3, 0x56,
                                                            0xc9, 0x53, 0xd1, 0x6e, 0x54, 0x60, 0x58, 0x99, 0xe6, 0xfc, 0x58,
                                                            0xc3, 0x5f, 0x59, 0xf8, 0x1b, 0x72, 0x64, 0xb1, 0xdb, 0x9a };
static const uint8_t chiptest_fcf59b3113ef_out_point_53[] = { 0x04, 0xdc, 0x2b, 0x7d, 0xdb, 0xa5, 0x47, 0xe7, 0xf5, 0x2c, 0xa0,
                                                              0x13, 0x64, 0x52, 0xcc, 0x0f, 0x42, 0x6b, 0x51, 0x01, 0xd4, 0xff,
                                                              0xac, 0x31, 0xf3, 0x72, 0x1d, 0x2c, 0x37, 0x6f, 0xba, 0xcf, 0x35,
                                                              0xcf, 0x15, 0x00, 0x53, 0xea, 0xdf, 0x88, 0xbb, 0xc2, 0x02, 0xee,
                                                              0xcd, 0x2d, 0xaf, 0xe7, 0xef, 0x9f, 0x56, 0xd2, 0x2f, 0x12, 0x5e,
                                                              0xb0, 0xa1, 0x98, 0x4b, 0xdf, 0xef, 0x86, 0x1d, 0x3c, 0x40 };
static const struct spake2p_point_muladd_tv chiptest_fcf59b3113ef_test_vector_54 = {
    .point1        = chiptest_fcf59b3113ef_point1_49,
    .point1_len    = 65,
    .scalar1       = chiptest_fcf59b3113ef_scalar1_50,
    .scalar1_len   = 32,
    .point2        = chiptest_fcf59b3113ef_point2_51,
    .point2_len    = 65,
    .scalar2       = chiptest_fcf59b3113ef_scalar2_52,
    .scalar2_len   = 32,
    .out_point     = chiptest_fcf59b3113ef_out_point_53,
    .out_point_len = 65,
};
static const uint8_t chiptest_fcf59b3113ef_point1_55[]    = { 0x04, 0x31, 0x34, 0x05, 0x2a, 0x59, 0x08, 0xdd, 0xba, 0x38, 0xf5,
                                                           0xae, 0x22, 0x58, 0x94, 0xd4, 0x58, 0x30, 0xa4, 0x25, 0x3d, 0xe5,
                                                           0xcc, 0xa1, 0x37, 0x48, 0xb1, 0x6e, 0x27, 0x96, 0x34, 0x5c, 0x46,
                                                           0x5d, 0x49, 0xdd, 0x55, 0xe6, 0x98, 0xb0, 0x7e, 0xdb, 0x55, 0xee,
                                                           0x98, 0xfa, 0x49, 0x31, 0x82, 0xd9, 0x83, 0x00, 0x65, 0xe9, 0x90,
                                                           0xb1, 0x04, 0x1a, 0xde, 0xcf, 0xef, 0x0a, 0x4a, 0x0d, 0xe9 };
static const uint8_t chiptest_fcf59b3113ef_scalar1_56[]   = { 0x9b, 0xb5, 0xbb, 0x1a, 0xb7, 0x47, 0xcf, 0x16, 0x77, 0xfe, 0xcd,
                                                            0x56, 0xae, 0xac, 0xd9, 0x6d, 0x2a, 0xd6, 0x4e, 0xc3, 0xc8, 0xc0,
                                                            0x32, 0xdf, 0x17, 0x7e, 0xfc, 0x86, 0x1c, 0x96, 0x48, 0x40 };
static const uint8_t chiptest_fcf59b3113ef_point2_57[]    = { 0x04, 0x7a, 0x8b, 0xd8, 0xe8, 0x3d, 0xa1, 0x78, 0xb8, 0x44, 0x1f,
                                                           0x5e, 0x65, 0xfe, 0x1e, 0x4c, 0x24, 0x6f, 0x13, 0xcc, 0x82, 0x12,
                                                           0x73, 0xda, 0x7c, 0x6e, 0xf7, 0x76, 0xe0, 0x45, 0x0a, 0x65, 0xbf,
                                                           0xab, 0xcb, 0x25, 0x2b, 0xda, 0xac, 0xbc, 0xc5, 0xd9, 0x41, 0x5e,
                                                           0xdf, 0xee, 0x07, 0x1a, 0x1f, 0x9e, 0xcf, 0x24, 0x0b, 0x77, 0xe6,
                                                           0x70, 0x08, 0x80, 0x33, 0x29, 0x2c, 0xba, 0x93, 0x4a, 0x60 };
static const uint8_t chiptest_fcf59b3113ef_scalar2_58[]   = { 0x33, 0xb9, 0x4d, 0xf8, 0xaf, 0x32, 0xef, 0x7b, 0x2c, 0x83, 0x98,
                                                            0xbf, 0x0f, 0x93, 0x41, 0x77, 0x31, 0xf9, 0x15, 0xcc, 0xbb, 0x40,
                                                            0x6c, 0xad, 0x96, 0xf7, 0x95, 0x1a, 0x19, 0x7f, 0x11, 0x90 };
static const uint8_t chiptest_fcf59b3113ef_out_point_59[] = { 0x04, 0x6d, 0xb3, 0xe3, 0x7c, 0x6f, 0x44, 0x31, 0x27, 0x9c, 0x73,
                                                              0xdf, 0x67, 0x08, 0xb8, 0x6b, 0x2f, 0xfa, 0x77, 0xf9, 0x00, 0x7b,
                                                              0x4a, 0xbc, 0xc8, 0xe5, 0xeb, 0xe4, 0x57, 0x26, 0x93, 0xcb, 0x23,
                                                              0x7d, 0x52, 0x0f, 0xd8, 0x59, 0x41, 0x91, 0x8f, 0xaa, 0x47, 0xb0,
                                                              0xa9, 0xf7, 0x6a, 0x65, 0x34, 0xd4, 0x63, 0x42, 0xc7, 0xf9, 0x7d,
                                                              0x1a, 0xef, 0xaa, 0xaa, 0xb9, 0xf8, 0x49, 0x3b, 0xc6, 0x40 };
static const struct spake2p_point_muladd_tv chiptest_fcf59b3113ef_test_vector_60 = {
    .point1        = chiptest_fcf59b3113ef_point1_55,
    .point1_len    = 65,
    .scalar1       = chiptest_fcf59b3113ef_scalar1_56,
    .scalar1_len   = 32,
    .point2        = chiptest_fcf59b3113ef_point2_57,
    .point2_len    = 65,
    .scalar2       = chiptest_fcf59b3113ef_scalar2_58,
    .scalar2_len   = 32,
    .out_point     = chiptest_fcf59b3113ef_out_point_59,
    .out_point_len = 65,
};
static const uint8_t chiptest_fcf59b3113ef_point1_61[]    = { 0x04, 0xa0, 0x00, 0x94, 0x24, 0x8c, 0x85, 0xb3, 0xb7, 0xbe, 0xa0,
                                                           0x98, 0x15, 0x25, 0xdb, 0xd2, 0xd8, 0xc9, 0x49, 0x17, 0x1e, 0x75,
                                                           0x8d, 0x0e, 0x2b, 0x37, 0x02, 0x58, 0x54, 0xc2, 0xe5, 0x33, 0x3e,
                                                           0xfc, 0xf4, 0x04, 0x5e, 0xdc, 0x3d, 0x21, 0x79, 0xbb, 0x63, 0xb4,
                                                           0x3a, 0x83, 0xc7, 0x3b, 0xdb, 0x40, 0xe1, 0x67, 0x37, 0x57, 0x8c,
                                                           0x86, 0xbe, 0x49, 0x2c, 0x49, 0xcf, 0x31, 0xfb, 0x54, 0x36 };
static const uint8_t chiptest_fcf59b3113ef_scalar1_62[]   = { 0x5a, 0x8d, 0x81, 0xe1, 0xf9, 0x62, 0xeb, 0x8a, 0x3b, 0x2c, 0x5f,
                                                            0x4d, 0x9e, 0x08, 0x03, 0x92, 0xe4, 0xe0, 0x58, 0x03, 0x91, 0x62,
                                                            0x8a, 0xe2, 0x59, 0xd0, 0x89, 0xee, 0x10, 0xfc, 0xa9, 0x21 };
static const uint8_t chiptest_fcf59b3113ef_point2_63[]    = { 0x04, 0x64, 0xcf, 0x03, 0xf3, 0x06, 0x36, 0x40, 0x46, 0xed, 0xe1,
                                                           0xa7, 0x1c, 0x1f, 0x3a, 0xb0, 0xf3, 0xa7, 0xe0, 0x18, 0xf3, 0xb7,
                                                           0xb1, 0xee, 0xc1, 0x78, 0x63, 0x2e, 0x55, 0xf0, 0x04, 0xae, 0x33,
                                                           0x0c, 0x9d, 0x1e, 0x33, 0xc6, 0xc2, 0x7f, 0x7f, 0xdd, 0xba, 0xf1,
                                                           0x5c, 0x30, 0x61, 0x8d, 0xb0, 0x09, 0x1a, 0x69, 0xd2, 0x0f, 0x3b,
                                                           0xf4, 0xa0, 0x18, 0xbe, 0x8a, 0x8c, 0x72, 0x84, 0xad, 0x6c };
static const uint8_t chiptest_fcf59b3113ef_scalar2_64[]   = { 0xa7, 0xd9, 0xa6, 0xe1, 0x82, 0xbe, 0x17, 0xd4, 0xc8, 0x1f, 0x32,
                                                            0x92, 0xa4, 0xa8, 0x2e, 0x80, 0x14, 0xa9, 0x31, 0x96, 0x0c, 0xcc,
                                                            0xf0, 0x95, 0xde, 0xcd, 0x56, 0xd1, 0x98, 0xd6, 0x72, 0xd4 };
static const uint8_t chiptest_fcf59b3113ef_out_point_65[] = { 0x04, 0xe5, 0xa5, 0x2f, 0x8f, 0xc5, 0xff, 0x51, 0xef, 0xcd, 0x61,
                                                              0x4f, 0x48, 0x5f, 0xc7, 0xf0, 0x39, 0x83, 0x27, 0x2b, 0x26, 0xd0,
                                                              0x27, 0x8f, 0x1f, 0x23, 0xf2, 0xd6, 0xe9, 0x69, 0x8a, 0xc6, 0xb3,
                                                              0xc6, 0x26, 0xa2, 0x3a, 0x64, 0x64, 0xc0, 0xf4, 0x19, 0x21, 0xb5,
                                                              0xe3, 0x22, 0x4d, 0x47, 0xec, 0x00, 0x6a, 0x42, 0xf8, 0x5c, 0x21,
                                                              0x80, 0x73, 0x0a, 0x39, 0x1e, 0xd7, 0xe8, 0xbb, 0x0f, 0x7a };
static const struct spake2p_point_muladd_tv chiptest_fcf59b3113ef_test_vector_66 = {
    .point1        = chiptest_fcf59b3113ef_point1_61,
    .point1_len    = 65,
    .scalar1       = chiptest_fcf59b3113ef_scalar1_62,
    .scalar1_len   = 32,
    .point2        = chiptest_fcf59b3113ef_point2_63,
    .point2_len    = 65,
    .scalar2       = chiptest_fcf59b3113ef_scalar2_64,
    .scalar2_len   = 32,
    .out_point     = chiptest_fcf59b3113ef_out_point_65,
    .out_point_len = 65,
};
static const uint8_t chiptest_fcf59b3113ef_point1_67[]    = { 0x04, 0xa0, 0x24, 0x27, 0xc8, 0x30, 0x58, 0x5a, 0xca, 0x4f, 0xaa,
                                                           0x49, 0xf7, 0x1d, 0x30, 0xbb, 0x28, 0x01, 0x0b, 0x41, 0x41, 0xf2,
                                                           0xfa, 0xda, 0xa5, 0x2b, 0x1e, 0x86, 0x60, 0x61, 0xd5, 0xfa, 0xf5,
                                                           0x2c, 0x7a, 0x59, 0x92, 0x70, 0x87, 0xbd, 0x71, 0xbc, 0xd8, 0x5a,
                                                           0xd7, 0x18, 0xff, 0x66, 0xf3, 0x59, 0x16, 0x56, 0xe8, 0x08, 0xe4,
                                                           0x57, 0xa7, 0xfc, 0xae, 0x55, 0x6c, 0x0a, 0xc0, 0xfb, 0x2f };
static const uint8_t chiptest_fcf59b3113ef_scalar1_68[]   = { 0x45, 0xb2, 0x65, 0xd4, 0xbd, 0xcd, 0x74, 0x5f, 0x71, 0x0e, 0xe7,
                                                            0xc2, 0xec, 0x7a, 0xdd, 0x59, 0x52, 0x9f, 0x16, 0xc7, 0x34, 0xf3,
                                                            0x2b, 0x06, 0x25, 0x5b, 0x6c, 0x07, 0x2b, 0x86, 0x7e, 0xcc };
static const uint8_t chiptest_fcf59b3113ef_point2_69[]    = { 0x04, 0x70, 0x64, 0x21, 0x3e, 0x56, 0x25, 0x93, 0x78, 0xdf, 0x55,
                                                           0xc6, 0x37, 0xeb, 0x02, 0x1d, 0x29, 0x93, 0xe8, 0xe7, 0x5a, 0xc5,
                                                           0x11, 0x9b, 0x71, 0x9e, 0xd5, 0x5a, 0x26, 0x4a, 0x90, 0x4c, 0x15,
                                                           0xe0, 0xc5, 0x95, 0x9e, 0x73, 0x89, 0xbe, 0xaa, 0xfe, 0xd2, 0x6d,
                                                           0xbd, 0xe0, 0x88, 0x01, 0x5d, 0xcc, 0xcc, 0x9f, 0x2a, 0x0d, 0x61,
                                                           0x5d, 0x2e, 0x36, 0xc8, 0xc1, 0xb8, 0xe8, 0x76, 0xa1, 0xc6 };
static const uint8_t chiptest_fcf59b3113ef_scalar2_70[]   = { 0x6b, 0xcc, 0x8f, 0x0f, 0x5e, 0x4b, 0x2b, 0xa9, 0x4f, 0xd1, 0xfa,
                                                            0xcf, 0x39, 0xd3, 0x78, 0x69, 0x6a, 0xde, 0xee, 0xe7, 0x4c, 0x89,
                                                            0x1b, 0x6d, 0xc2, 0xe9, 0x96, 0x05, 0x77, 0x59, 0x0d, 0x86 };
static const uint8_t chiptest_fcf59b3113ef_out_point_71[] = { 0x04, 0x28, 0xf8, 0x4d, 0x78, 0x05, 0xbc, 0xb9, 0x9f, 0xd4, 0x22,
                                                              0x71, 0xaf, 0x8b, 0x5f, 0x92, 0x9f, 0x5c, 0x0c, 0x51, 0xa5, 0x5e,
                                                              0xac, 0x33, 0x51, 0xfe, 0x68, 0x2c, 0x03, 0x5d, 0x1b, 0xa4, 0x25,
                                                              0x17, 0x58, 0x11, 0xab, 0xb4, 0xfd, 0xb5, 0x57, 0xb5, 0xaa, 0x55,
                                                              0x72, 0x56, 0x4e, 0xc9, 0x5d, 0x58, 0x76, 0xeb, 0x0c, 0xd8, 0x5d,
                                                              0x69, 0x1a, 0x1a, 0xe9, 0xf4, 0x56, 0x31, 0x23, 0x34, 0x96 };
static const struct spake2p_point_muladd_tv chiptest_fcf59b3113ef_test_vector_72 = {
    .point1        = chiptest_fcf59b3113ef_point1_67,
    .point1_len    = 65,
    .scalar1       = chiptest_fcf59b3113ef_scalar1_68,
    .scalar1_len   = 32,
    .point2        = chiptest_fcf59b3113ef_point2_69,
    .point2_len    = 65,
    .scalar2       = chiptest_fcf59b3113ef_scalar2_70,
    .scalar2_len   = 32,
    .out_point     = chiptest_fcf59b3113ef_out_point_71,
    .out_point_len = 65,
};
static const uint8_t chiptest_fcf59b3113ef_point1_73[]    = { 0x04, 0xd9, 0x3d, 0x3f, 0x70, 0xc3, 0xf6, 0x49, 0xaf, 0x56, 0x53,
                                                           0xef, 0x96, 0xe9, 0xc2, 0xec, 0x2d, 0x8d, 0x66, 0x45, 0x56, 0x97,
                                                           0x2f, 0x2d, 0x01, 0x59, 0x9b, 0x8f, 0x68, 0x08, 0xf2, 0xe2, 0x12,
                                                           0x61, 0x2c, 0x58, 0x80, 0x06, 0x7c, 0x30, 0xee, 0x55, 0xc8, 0x74,
                                                           0x58, 0x75, 0xbf, 0xb8, 0x45, 0x7f, 0x6a, 0x91, 0xf1, 0x80, 0xe7,
                                                           0xd6, 0xfc, 0x6c, 0xd6, 0xda, 0x91, 0xeb, 0xa3, 0x03, 0xd4 };
static const uint8_t chiptest_fcf59b3113ef_scalar1_74[]   = { 0xb4, 0x3b, 0x48, 0x6f, 0x81, 0xf2, 0x6f, 0xed, 0x7f, 0xde, 0xa7,
                                                            0x11, 0x06, 0x87, 0xe9, 0xc4, 0x72, 0xd1, 0x91, 0xe5, 0x44, 0x2c,
                                                            0xc2, 0x31, 0xc1, 0xbf, 0x8e, 0x66, 0xd7, 0x69, 0x74, 0x4b };
static const uint8_t chiptest_fcf59b3113ef_point2_75[]    = { 0x04, 0x01, 0xbf, 0x20, 0x58, 0x20, 0x2e, 0x60, 0x94, 0x40, 0xc2,
                                                           0xd9, 0xc9, 0x68, 0x3b, 0x86, 0x78, 0xa8, 0x60, 0x0d, 0x3c, 0xb3,
                                                           0xcb, 0x09, 0x31, 0x73, 0xc3, 0x13, 0x3f, 0xdc, 0x4a, 0x58, 0x49,
                                                           0x90, 0x5b, 0xb0, 0x1d, 0x8e, 0x70, 0x63, 0x21, 0x3a, 0xaa, 0x03,
                                                           0x4e, 0xc2, 0xc5, 0xfa, 0x86, 0x54, 0x84, 0xf3, 0xfc, 0xec, 0xdc,
                                                           0x5b, 0xdc, 0xee, 0x25, 0x0d, 0x42, 0x84, 0x26, 0x28, 0x8b };
static const uint8_t chiptest_fcf59b3113ef_scalar2_76[]   = { 0x13, 0x7f, 0x0e, 0x3e, 0xec, 0x85, 0x18, 0xf0, 0x95, 0x6d, 0x86,
                                                            0xf4, 0xc7, 0x86, 0xe7, 0x7c, 0x72, 0x10, 0xcc, 0x93, 0x05, 0xb3,
                                                            0x6c, 0xb6, 0xeb, 0x6f, 0x36, 0x94, 0xf1, 0x3e, 0xb4, 0x0b };
static const uint8_t chiptest_fcf59b3113ef_out_point_77[] = { 0x04, 0x12, 0xa6, 0x09, 0x10, 0x47, 0x32, 0xa6, 0xa0, 0x23, 0x27,
                                                              0xe5, 0x52, 0x35, 0x2c, 0x6a, 0x1c, 0x28, 0xd4, 0x5d, 0xfe, 0x6a,
                                                              0x54, 0x3d, 0x41, 0x93, 0x0b, 0xa3, 0x93, 0x64, 0xb7, 0x56, 0xe9,
                                                              0x3f, 0xab, 0x33, 0x6f, 0xb2, 0x4b, 0xd1, 0xdb, 0x2b, 0x52, 0xb7,
                                                              0x8b, 0x4c, 0xea, 0xe0, 0x4f, 0xca, 0x70, 0x53, 0x15, 0x61, 0xf8,
                                                              0x63, 0xa2, 0x5a, 0x14, 0x93, 0x5b, 0x03, 0xcf, 0x73, 0x9f };
static const struct spake2p_point_muladd_tv chiptest_fcf59b3113ef_test_vector_78 = {
    .point1        = chiptest_fcf59b3113ef_point1_73,
    .point1_len    = 65,
    .scalar1       = chiptest_fcf59b3113ef_scalar1_74,
    .scalar1_len   = 32,
    .point2        = chiptest_fcf59b3113ef_point2_75,
    .point2_len    = 65,
    .scalar2       = chiptest_fcf59b3113ef_scalar2_76,
    .scalar2_len   = 32,
    .out_point     = chiptest_fcf59b3113ef_out_point_77,
    .out_point_len = 65,
};
static const uint8_t chiptest_fcf59b3113ef_point1_79[]    = { 0x04, 0xe3, 0xf1, 0x0e, 0x4c, 0xda, 0x25, 0x69, 0xcc, 0x9a, 0xa6,
                                                           0xf7, 0x8b, 0xfe, 0xec, 0x27, 0x90, 0xf7, 0x80, 0x81, 0x07, 0x2c,
                                                           0x30, 0xd3, 0xbf, 0x83, 0x62, 0x66, 0xf3, 0xe7, 0x31, 0x70, 0x71,
                                                           0x8f, 0x63, 0x94, 0x3d, 0x88, 0x57, 0xd2, 0x3e, 0x78, 0x97, 0xd2,
                                                           0x89, 0x97, 0xaf, 0xf2, 0xf8, 0x5a, 0x6c, 0x8b, 0xba, 0x4b, 0x4c,
                                                           0xc2, 0xb6, 0x33, 0xc3, 0x49, 0xcd, 0x58, 0xe7, 0x12, 0x4c };
static const uint8_t chiptest_fcf59b3113ef_scalar1_80[]   = { 0x45, 0xc0, 0x3c, 0x6a, 0x85, 0xc6, 0x2d, 0xfb, 0x37, 0x23, 0x20,
                                                            0xda, 0xf8, 0xd2, 0xb1, 0x1f, 0x86, 0x24, 0xa0, 0x63, 0xc3, 0x3b,
                                                            0x28, 0x28, 0x6c, 0xe4, 0x76, 0x6b, 0x49, 0xb3, 0xd2, 0x39 };
static const uint8_t chiptest_fcf59b3113ef_point2_81[]    = { 0x04, 0x46, 0xfb, 0x35, 0xcd, 0x58, 0x9f, 0xb9, 0x99, 0xfa, 0x7f,
                                                           0xa6, 0x6c, 0xc9, 0x29, 0x4e, 0xbb, 0x6c, 0xa0, 0x2d, 0xd6, 0x28,
                                                           0xb2, 0x8e, 0xcd, 0xa6, 0x3c, 0x3b, 0xef, 0x5a, 0xc2, 0xf1, 0xaf,
                                                           0x84, 0xbb, 0xbc, 0x3b, 0x30, 0xac, 0xf0, 0x02, 0xf8, 0xa4, 0x92,
                                                           0x0d, 0xbf, 0xb5, 0x7e, 0xe5, 0x3c, 0x0e, 0xa1, 0xe9, 0xb9, 0x1c,
                                                           0x39, 0x3a, 0xeb, 0x10, 0x3b, 0xbe, 0xc6, 0x57, 0x47, 0x67 };
static const uint8_t chiptest_fcf59b3113ef_scalar2_82[]   = { 0xfa, 0xad, 0x85, 0xd1, 0xe7, 0x1a, 0x2d, 0x1b, 0x32, 0x9e, 0x3b,
                                                            0x30, 0x9e, 0x83, 0x3a, 0x8b, 0xbf, 0xe8, 0xd4, 0x2f, 0x46, 0x2f,
                                                            0x3f, 0x77, 0x40, 0x89, 0x91, 0xb4, 0x14, 0x65, 0x2d, 0x34 };
static const uint8_t chiptest_fcf59b3113ef_out_point_83[] = { 0x04, 0xc9, 0x7b, 0xa6, 0x19, 0xd8, 0xb2, 0xf6, 0x5c, 0x62, 0x44,
                                                              0xfb, 0xa0, 0x78, 0xed, 0x43, 0xd7, 0xd5, 0xcc, 0x75, 0x93, 0x4f,
                                                              0x02, 0x9d, 0x39, 0x43, 0x46, 0x40, 0x08, 0xff, 0xa6, 0x65, 0xfc,
                                                              0x38, 0xd6, 0x19, 0xed, 0x26, 0x9f, 0xad, 0x5a, 0x1c, 0x5f, 0x2d,
                                                              0xa9, 0xae, 0xbc, 0x56, 0xdf, 0xc0, 0x2b, 0xbc, 0x0b, 0x7f, 0xa9,
                                                              0xb7, 0x4e, 0x5b, 0xe2, 0x22, 0xab, 0xe5, 0x38, 0xda, 0xed };
static const struct spake2p_point_muladd_tv chiptest_fcf59b3113ef_test_vector_84 = {
    .point1        = chiptest_fcf59b3113ef_point1_79,
    .point1_len    = 65,
    .scalar1       = chiptest_fcf59b3113ef_scalar1_80,
    .scalar1_len   = 32,
    .point2        = chiptest_fcf59b3113ef_point2_81,
    .point2_len    = 65,
    .scalar2       = chiptest_fcf59b3113ef_scalar2_82,
    .scalar2_len   = 32,
    .out_point     = chiptest_fcf59b3113ef_out_point_83,
    .out_point_len = 65,
};
static const uint8_t chiptest_fcf59b3113ef_point1_85[]    = { 0x04, 0x56, 0xf8, 0xed, 0xe1, 0x8e, 0xf1, 0x4e, 0xad, 0xa8, 0x56,
                                                           0xcd, 0x8e, 0xbb, 0x77, 0x4f, 0x19, 0xfe, 0xe4, 0x96, 0x38, 0x19,
                                                           0xae, 0x88, 0x86, 0xea, 0xed, 0xa9, 0x70, 0xc9, 0x41, 0x03, 0x74,
                                                           0x74, 0x42, 0xf2, 0x36, 0x0f, 0xfb, 0x02, 0xa7, 0x46, 0x85, 0xdb,
                                                           0xce, 0x7d, 0x6e, 0x65, 0xa2, 0xe3, 0xba, 0x5e, 0x67, 0x3f, 0x3d,
                                                           0x76, 0x21, 0x4c, 0xeb, 0xf8, 0x5b, 0x33, 0x98, 0xc4, 0xa9 };
static const uint8_t chiptest_fcf59b3113ef_scalar1_86[]   = { 0xb1, 0xf4, 0x5c, 0xad, 0x7f, 0xfe, 0xb6, 0x04, 0x61, 0x7e, 0xf3,
                                                            0xea, 0xf3, 0x35, 0x01, 0x35, 0x16, 0x10, 0x86, 0xd5, 0x0d, 0xc3,
                                                            0x21, 0xae, 0x7f, 0x9b, 0x95, 0xc8, 0xeb, 0xb2, 0x4f, 0x0d };
static const uint8_t chiptest_fcf59b3113ef_point2_87[]    = { 0x04, 0x95, 0x3c, 0x8e, 0xc3, 0x23, 0x34, 0x74, 0x4f, 0xd0, 0x99,
                                                           0x2c, 0x1e, 0x14, 0xf4, 0x5f, 0x74, 0x97, 0x21, 0xa8, 0xe9, 0xfd,
                                                           0xcd, 0x32, 0xd3, 0x40, 0x0f, 0xc6, 0xbb, 0x50, 0xc9, 0x5b, 0x50,
                                                           0x14, 0x04, 0x5d, 0x74, 0xa9, 0x33, 0x4c, 0x3b, 0xc7, 0x12, 0x0c,
                                                           0x8f, 0x8f, 0x90, 0x54, 0x93, 0x98, 0x61, 0xb1, 0xe8, 0x53, 0x44,
                                                           0x36, 0x2f, 0xc5, 0x48, 0x31, 0xc6, 0x70, 0x41, 0xa1, 0xa3 };
static const uint8_t chiptest_fcf59b3113ef_scalar2_88[]   = { 0x27, 0x30, 0xdc, 0xa7, 0xa7, 0xf4, 0x31, 0x43, 0x12, 0x87, 0x82,
                                                            0x84, 0xa2, 0xd2, 0xe8, 0xa1, 0x0e, 0x3b, 0x94, 0x56, 0xb5, 0x09,
                                                            0xc2, 0xde, 0xd4, 0x54, 0x81, 0xce, 0x33, 0x0e, 0x65, 0x19 };
static const uint8_t chiptest_fcf59b3113ef_out_point_89[] = { 0x04, 0x14, 0x3d, 0xe8, 0xb9, 0xac, 0x3b, 0x6a, 0x88, 0x17, 0x1b,
                                                              0xc2, 0x2e, 0x76, 0x32, 0x29, 0x62, 0x9b, 0x44, 0x82, 0x2d, 0x94,
                                                              0x54, 0x91, 0x68, 0x86, 0xdc, 0x8e, 0x74, 0x4b, 0x33, 0x40, 0xd5,
                                                              0x1e, 0xc0, 0xd0, 0x0b, 0xb2, 0x45, 0x8f, 0x64, 0x07, 0x21, 0x77,
                                                              0xfd, 0xea, 0x1a, 0x68, 0xd1, 0x4e, 0x96, 0x38, 0x52, 0xe5, 0x29,
                                                              0xd9, 0x80, 0x9d, 0xdf, 0xf4, 0x59, 0xef, 0xaa, 0x33, 0xcd };
static const struct spake2p_point_muladd_tv chiptest_fcf59b3113ef_test_vector_90 = {
    .point1        = chiptest_fcf59b3113ef_point1_85,
    .point1_len    = 65,
    .scalar1       = chiptest_fcf59b3113ef_scalar1_86,
    .scalar1_len   = 32,
    .point2        = chiptest_fcf59b3113ef_point2_87,
    .point2_len    = 65,
    .scalar2       = chiptest_fcf59b3113ef_scalar2_88,
    .scalar2_len   = 32,
    .out_point     = chiptest_fcf59b3113ef_out_point_89,
    .out_point_len = 65,
};
static const uint8_t chiptest_fcf59b3113ef_point1_91[]    = { 0x04, 0xb0, 0x14, 0x84, 0xd2, 0x82, 0xf3, 0x3e, 0x48, 0xbd, 0xb4,
                                                           0x17, 0x5b, 0x67, 0xcd, 0xda, 0xa9, 0xfb, 0x35, 0x3c, 0x6e, 0x2b,
                                                           0x64, 0xae, 0x39, 0x59, 0xdf, 0x49, 0xed, 0xbb, 0x82, 0x94, 0x8a,
                                                           0xf3, 0xfa, 0x5f, 0xd1, 0xc7, 0x99, 0x1e, 0xbf, 0xf0, 0xd4, 0x43,
                                                           0x09, 0x04, 0x3e, 0xc8, 0xfa, 0x64, 0xda, 0xe1, 0xcd, 0xe2, 0x1e,
                                                           0x49, 0x7d, 0xc0, 0xe7, 0x90, 0x24, 0x44, 0xea, 0xc7, 0xe6 };
static const uint8_t chiptest_fcf59b3113ef_scalar1_92[]   = { 0xfc, 0x8e, 0x98, 0x95, 0x8b, 0xa3, 0x50, 0xb0, 0x3f, 0xba, 0x98,
                                                            0xf9, 0xf4, 0x57, 0x37, 0xce, 0xb8, 0xfe, 0x8c, 0x58, 0x9d, 0xdf,
                                                            0x5c, 0x2f, 0x63, 0xc5, 0x3a, 0x43, 0xbc, 0x87, 0xeb, 0x4e };
static const uint8_t chiptest_fcf59b3113ef_point2_93[]    = { 0x04, 0x3c, 0x8a, 0x6c, 0xb6, 0x66, 0x2c, 0x07, 0x3b, 0xb6, 0x20,
                                                           0x28, 0xb0, 0x12, 0x31, 0x26, 0x92, 0x2f, 0x00, 0xe1, 0x90, 0x0b,
                                                           0x66, 0x17, 0x5f, 0xcc, 0x1e, 0x4f, 0xdb, 0xf7, 0xe7, 0xfb, 0x65,
                                                           0x99, 0x81, 0xb8, 0x8b, 0xab, 0x1e, 0xda, 0x74, 0x62, 0x22, 0xa3,
                                                           0xd9, 0x46, 0xa5, 0x44, 0xa0, 0x6b, 0xc9, 0x05, 0xeb, 0xa7, 0x02,
                                                           0xf9, 0xf7, 0x2e, 0x7b, 0x55, 0xb3, 0x28, 0xbf, 0x22, 0xc4 };
static const uint8_t chiptest_fcf59b3113ef_scalar2_94[]   = { 0x19, 0xd7, 0x60, 0xf1, 0x64, 0x90, 0x1e, 0x83, 0x5b, 0x2e, 0xc1,
                                                            0xf9, 0xc6, 0x1d, 0xf8, 0xf7, 0x10, 0x1c, 0x8a, 0x01, 0xdc, 0xd0,
                                                            0x70, 0xfc, 0x82, 0x00, 0x2c, 0x06, 0x6f, 0xfb, 0x20, 0xb3 };
static const uint8_t chiptest_fcf59b3113ef_out_point_95[] = { 0x04, 0x22, 0x11, 0xed, 0x30, 0xee, 0x65, 0xe4, 0x45, 0xde, 0x8d,
                                                              0xd7, 0xfe, 0xa9, 0x47, 0x8c, 0x70, 0x35, 0x80, 0x64, 0x35, 0xa7,
                                                              0x5d, 0x96, 0xea, 0xc2, 0xe3, 0x3c, 0x73, 0xcf, 0x5e, 0xbb, 0xf5,
                                                              0x48, 0xb8, 0x06, 0x00, 0x06, 0x47, 0xc0, 0x27, 0xd9, 0x79, 0x13,
                                                              0x26, 0x4f, 0x62, 0x5c, 0xbc, 0x3b, 0x17, 0x7d, 0xe8, 0xb9, 0xc4,
                                                              0x04, 0x86, 0x08, 0xcd, 0x0a, 0x83, 0xe7, 0x27, 0x30, 0x24 };
static const struct spake2p_point_muladd_tv chiptest_fcf59b3113ef_test_vector_96 = {
    .point1        = chiptest_fcf59b3113ef_point1_91,
    .point1_len    = 65,
    .scalar1       = chiptest_fcf59b3113ef_scalar1_92,
    .scalar1_len   = 32,
    .point2        = chiptest_fcf59b3113ef_point2_93,
    .point2_len    = 65,
    .scalar2       = chiptest_fcf59b3113ef_scalar2_94,
    .scalar2_len   = 32,
    .out_point     = chiptest_fcf59b3113ef_out_point_95,
    .out_point_len = 65,
};
static const uint8_t chiptest_fcf59b3113ef_point1_97[]     = { 0x04, 0x30, 0x51, 0x75, 0x4e, 0x6e, 0xf1, 0x96, 0xe1, 0x18, 0x48,
                                                           0x1c, 0x03, 0x55, 0x93, 0xfd, 0xa5, 0xd5, 0x11, 0x62, 0x8d, 0x9f,
                                                           0xbd, 0x8f, 0x28, 0xb7, 0xa3, 0x05, 0x7f, 0xc2, 0xc8, 0xb1, 0x02,
                                                           0xce, 0x46, 0x08, 0x28, 0x7b, 0x86, 0x1f, 0xdc, 0x04, 0xd6, 0x73,
                                                           0xb8, 0x21, 0x69, 0x9d, 0x0f, 0x2b, 0x8b, 0x32, 0xc5, 0xc3, 0xb6,
                                                           0x54, 0x9e, 0xd0, 0x3e, 0x71, 0x5c, 0xbf, 0x37, 0xff, 0x0b };
static const uint8_t chiptest_fcf59b3113ef_scalar1_98[]    = { 0x72, 0xc6, 0x32, 0x24, 0x28, 0x90, 0x3c, 0x1c, 0xee, 0xdd, 0x75,
                                                            0xe4, 0xdc, 0xfc, 0xc1, 0xb0, 0x14, 0xbf, 0xc9, 0x72, 0x25, 0x0f,
                                                            0xcd, 0xbe, 0x0c, 0x70, 0x03, 0x14, 0xdd, 0x8a, 0xf9, 0xa2 };
static const uint8_t chiptest_fcf59b3113ef_point2_99[]     = { 0x04, 0xe2, 0x0b, 0x0f, 0xa2, 0x93, 0x13, 0xf0, 0x5a, 0x6e, 0x71,
                                                           0x39, 0x24, 0x5a, 0x82, 0xcd, 0xbf, 0x1c, 0x11, 0xb5, 0xe2, 0x30,
                                                           0x74, 0xe6, 0x6e, 0xe7, 0x1d, 0x18, 0x66, 0x30, 0xc1, 0x5f, 0x23,
                                                           0x6b, 0x12, 0xc9, 0xc9, 0x53, 0x66, 0xe8, 0xd1, 0xc0, 0x9f, 0x39,
                                                           0x03, 0x74, 0x24, 0x35, 0x03, 0x93, 0xa9, 0x15, 0x47, 0xcf, 0x88,
                                                           0x2a, 0xb2, 0x0f, 0xe1, 0x1c, 0x7c, 0x06, 0x9e, 0x95, 0x74 };
static const uint8_t chiptest_fcf59b3113ef_scalar2_100[]   = { 0x3c, 0x64, 0x29, 0xf0, 0x93, 0xf8, 0xd9, 0x4d, 0x83, 0x02, 0x3c,
                                                             0xf3, 0xe9, 0x9e, 0x1f, 0xe6, 0x99, 0xf7, 0x2c, 0xc0, 0x19, 0xbe,
                                                             0x23, 0xea, 0x0d, 0x54, 0x2b, 0x11, 0xc1, 0x7c, 0x22 };
static const uint8_t chiptest_fcf59b3113ef_out_point_101[] = { 0x04, 0x72, 0x40, 0xd8, 0x36, 0xff, 0x7b, 0x00, 0xb1, 0x8b, 0x34,
                                                               0x06, 0xda, 0x5c, 0xa4, 0x5e, 0x8e, 0x15, 0xf5, 0x70, 0x9b, 0x4f,
                                                               0x09, 0x73, 0x45, 0x1d, 0x8b, 0xc0, 0xc8, 0x17, 0x60, 0xdc, 0x01,
                                                               0x96, 0x1e, 0x7b, 0xb0, 0xeb, 0x2e, 0x30, 0xe5, 0x7f, 0x12, 0x28,
                                                               0x66, 0xca, 0xd1, 0x74, 0xde, 0x77, 0x1a, 0x19, 0x9b, 0x42, 0x13,
                                                               0xbb, 0x76, 0x37, 0xd8, 0x47, 0x37, 0xc6, 0x54, 0x1f, 0x69 };
static const struct spake2p_point_muladd_tv chiptest_fcf59b3113ef_test_vector_102 = {
    .point1        = chiptest_fcf59b3113ef_point1_97,
    .point1_len    = 65,
    .scalar1       = chiptest_fcf59b3113ef_scalar1_98,
    .scalar1_len   = 32,
    .point2        = chiptest_fcf59b3113ef_point2_99,
    .point2_len    = 65,
    .scalar2       = chiptest_fcf59b3113ef_scalar2_100,
    .scalar2_len   = 31,
    .out_point     = chiptest_fcf59b3113ef_out_point_101,
    .out_point_len = 65,
};
static const uint8_t chiptest_fcf59b3113ef_point1_103[]    = { 0x04, 0xe4, 0xa2, 0xce, 0x58, 0xd2, 0xe9, 0xe5, 0x6e, 0xa8, 0x7d,
                                                            0xc8, 0xd8, 0x95, 0x8e, 0xca, 0x58, 0x1b, 0x0f, 0x92, 0x3f, 0xe9,
                                                            0xca, 0xb9, 0xc2, 0x79, 0x48, 0x7b, 0xb8, 0x04, 0x3e, 0x47, 0x02,
                                                            0xd4, 0x17, 0x73, 0x03, 0xcc, 0x61, 0x72, 0x2b, 0xec, 0x0e, 0xee,
                                                            0x15, 0xe9, 0xcb, 0x7d, 0x76, 0x51, 0xe0, 0x2d, 0x5b, 0x2f, 0xbd,
                                                            0x5e, 0xe5, 0x03, 0xd6, 0x53, 0x86, 0x26, 0xe7, 0x98, 0xc0 };
static const uint8_t chiptest_fcf59b3113ef_scalar1_104[]   = { 0x3a, 0x78, 0x23, 0xeb, 0xf9, 0xa6, 0x0a, 0xb8, 0x42, 0x1d, 0x04,
                                                             0x23, 0xb9, 0x3e, 0x0b, 0xaa, 0x4d, 0x6a, 0x8c, 0xb4, 0x5c, 0xc9,
                                                             0x68, 0x27, 0x3a, 0x04, 0x55, 0xc1, 0x3b, 0xc5, 0x9e, 0x1a };
static const uint8_t chiptest_fcf59b3113ef_point2_105[]    = { 0x04, 0xaf, 0x52, 0x6b, 0x71, 0x16, 0xb7, 0x11, 0xcd, 0xbb, 0x15,
                                                            0xbe, 0x85, 0x95, 0xb3, 0x11, 0x93, 0x4c, 0xa3, 0x3e, 0x19, 0xb7,
                                                            0xca, 0x89, 0xc7, 0x81, 0xf8, 0xac, 0x47, 0xf7, 0xa2, 0x80, 0x48,
                                                            0xa1, 0x93, 0x68, 0x7c, 0x68, 0x1d, 0x1f, 0xae, 0xbe, 0x32, 0x3f,
                                                            0xdd, 0x23, 0xde, 0x10, 0x81, 0x08, 0xf5, 0x33, 0xa2, 0x17, 0xd2,
                                                            0x3b, 0xde, 0x53, 0x6e, 0xb7, 0x47, 0x49, 0x5f, 0x20, 0x40 };
static const uint8_t chiptest_fcf59b3113ef_scalar2_106[]   = { 0xf1, 0x4c, 0xfd, 0x9f, 0x84, 0xcf, 0xab, 0xcb, 0xab, 0x88, 0x77,
                                                             0x0f, 0xee, 0x35, 0x3d, 0xca, 0xfa, 0x42, 0x83, 0x5b, 0x03, 0x2a,
                                                             0x48, 0x95, 0x63, 0x46, 0x69, 0x60, 0xda, 0x7d, 0x93, 0x71 };
static const uint8_t chiptest_fcf59b3113ef_out_point_107[] = { 0x04, 0xc3, 0xc6, 0xff, 0x52, 0xbb, 0x8a, 0x11, 0x6b, 0xf4, 0x9c,
                                                               0x7a, 0x00, 0x9c, 0xd1, 0xf7, 0x1c, 0xb4, 0xc9, 0xbe, 0xfc, 0x9c,
                                                               0x05, 0x19, 0x39, 0xc8, 0x08, 0xa8, 0x7d, 0x15, 0x5a, 0x0e, 0x5d,
                                                               0x56, 0x6f, 0x6f, 0xf0, 0xe3, 0x89, 0x3e, 0x59, 0xe3, 0x4e, 0x78,
                                                               0x57, 0x1f, 0x1c, 0xab, 0x2e, 0x76, 0x49, 0x09, 0x8a, 0x1a, 0xd8,
                                                               0xa8, 0x1d, 0xf9, 0x7b, 0x58, 0x80, 0xa9, 0x07, 0x1b, 0x10 };
static const struct spake2p_point_muladd_tv chiptest_fcf59b3113ef_test_vector_108 = {
    .point1        = chiptest_fcf59b3113ef_point1_103,
    .point1_len    = 65,
    .scalar1       = chiptest_fcf59b3113ef_scalar1_104,
    .scalar1_len   = 32,
    .point2        = chiptest_fcf59b3113ef_point2_105,
    .point2_len    = 65,
    .scalar2       = chiptest_fcf59b3113ef_scalar2_106,
    .scalar2_len   = 32,
    .out_point     = chiptest_fcf59b3113ef_out_point_107,
    .out_point_len = 65,
};
static const uint8_t chiptest_fcf59b3113ef_point1_109[]    = { 0x04, 0xea, 0x43, 0xef, 0x47, 0x57, 0xf9, 0xff, 0x04, 0xfd, 0xac,
                                                            0x1a, 0xf8, 0x89, 0x12, 0xda, 0x2d, 0xf4, 0xdc, 0x40, 0x08, 0x21,
                                                            0x87, 0x62, 0xbc, 0x94, 0x51, 0x0f, 0xbc, 0x3e, 0xd7, 0x05, 0xc6,
                                                            0x97, 0x2c, 0x27, 0x04, 0x94, 0x76, 0x4e, 0xb7, 0xf7, 0x80, 0xe5,
                                                            0x57, 0x4e, 0x14, 0x15, 0x9e, 0x49, 0x7e, 0xbb, 0xa3, 0x2c, 0xac,
                                                            0x08, 0x69, 0x34, 0xc2, 0xd9, 0x08, 0x13, 0x76, 0xf7, 0xdc };
static const uint8_t chiptest_fcf59b3113ef_scalar1_110[]   = { 0x47, 0xb3, 0x81, 0x34, 0x60, 0x0e, 0xb7, 0xb7, 0x7b, 0xb7, 0xf7,
                                                             0xe9, 0xfc, 0x8c, 0x00, 0x56, 0x01, 0x1f, 0xd6, 0xc2, 0x95, 0xa4,
                                                             0x91, 0xc5, 0x90, 0x34, 0x69, 0x21, 0x45, 0xaf, 0xdc, 0x6f };
static const uint8_t chiptest_fcf59b3113ef_point2_111[]    = { 0x04, 0x42, 0xc2, 0x2a, 0x1d, 0x88, 0x7c, 0xd0, 0x3e, 0x1b, 0xd9,
                                                            0xaa, 0x8a, 0x54, 0xf7, 0x9c, 0x23, 0xb5, 0xfc, 0x27, 0x40, 0x76,
                                                            0x1c, 0x78, 0x7e, 0x40, 0x1b, 0x21, 0xd8, 0x22, 0x38, 0x07, 0x8d,
                                                            0xb4, 0xc0, 0x53, 0xde, 0x18, 0x8c, 0xc4, 0x3a, 0xca, 0xe2, 0x93,
                                                            0x58, 0x20, 0xfd, 0x5e, 0x50, 0xc0, 0x72, 0x8c, 0x3f, 0x60, 0x16,
                                                            0xb1, 0x93, 0x5c, 0xaf, 0x61, 0xed, 0x78, 0xd9, 0x43, 0xf9 };
static const uint8_t chiptest_fcf59b3113ef_scalar2_112[]   = { 0x79, 0x85, 0xce, 0x7d, 0x70, 0x9f, 0x3b, 0xa8, 0xd9, 0xfb, 0x20,
                                                             0x71, 0x60, 0xd4, 0x48, 0x08, 0x40, 0xac, 0xfb, 0x9b, 0x05, 0xcb,
                                                             0x18, 0xd7, 0xac, 0xb0, 0x27, 0xe8, 0x85, 0x97, 0xe0, 0xaf };
static const uint8_t chiptest_fcf59b3113ef_out_point_113[] = { 0x04, 0xff, 0x7e, 0x7e, 0xde, 0x7c, 0x30, 0xb4, 0x24, 0x29, 0x0e,
                                                               0xa6, 0xa8, 0x1a, 0x6d, 0xa8, 0x22, 0xcc, 0x87, 0x12, 0xd5, 0x56,
                                                               0x83, 0x3a, 0x22, 0x0b, 0x08, 0xdf, 0x2f, 0x00, 0x29, 0xb2, 0x5a,
                                                               0xb7, 0xab, 0x90, 0x90, 0x78, 0x72, 0x30, 0xe6, 0x4f, 0xda, 0x64,
                                                               0x58, 0x97, 0x52, 0xa4, 0x22, 0x5b, 0x84, 0xd3, 0x5c, 0x50, 0x94,
                                                               0x5a, 0x4c, 0xfb, 0x8b, 0x80, 0x65, 0xee, 0x0c, 0xec, 0xc2 };
static const struct spake2p_point_muladd_tv chiptest_fcf59b3113ef_test_vector_114 = {
    .point1        = chiptest_fcf59b3113ef_point1_109,
    .point1_len    = 65,
    .scalar1       = chiptest_fcf59b3113ef_scalar1_110,
    .scalar1_len   = 32,
    .point2        = chiptest_fcf59b3113ef_point2_111,
    .point2_len    = 65,
    .scalar2       = chiptest_fcf59b3113ef_scalar2_112,
    .scalar2_len   = 32,
    .out_point     = chiptest_fcf59b3113ef_out_point_113,
    .out_point_len = 65,
};
static const uint8_t chiptest_fcf59b3113ef_point1_115[]    = { 0x04, 0xfa, 0xa1, 0x3a, 0xf2, 0x55, 0x2c, 0x6b, 0x5a, 0x6c, 0x47,
                                                            0xbf, 0x59, 0x22, 0x19, 0x7e, 0x7a, 0xd4, 0xd6, 0x9c, 0x68, 0x4e,
                                                            0x28, 0xca, 0x0a, 0x91, 0x43, 0xa9, 0x72, 0x84, 0x5f, 0x48, 0x7e,
                                                            0x16, 0x45, 0x97, 0x96, 0x0b, 0xab, 0x98, 0x16, 0xde, 0xb3, 0x53,
                                                            0x02, 0x1e, 0x1f, 0x99, 0xde, 0x7f, 0x04, 0xa6, 0x63, 0x52, 0x63,
                                                            0xbe, 0xf8, 0xd1, 0x50, 0x52, 0xfa, 0xed, 0xef, 0x3e, 0xf4 };
static const uint8_t chiptest_fcf59b3113ef_scalar1_116[]   = { 0x1b, 0x1b, 0x04, 0xf9, 0x37, 0x74, 0x8f, 0xd7, 0xf9, 0x6f, 0x61,
                                                             0x3f, 0x8b, 0xbf, 0xf1, 0xa5, 0x0d, 0x69, 0x3c, 0x99, 0x58, 0x28,
                                                             0xcf, 0xca, 0x4e, 0xfd, 0x3a, 0x51, 0x15, 0x49, 0x1c, 0x61 };
static const uint8_t chiptest_fcf59b3113ef_point2_117[]    = { 0x04, 0x2e, 0xf0, 0xaf, 0x10, 0xa1, 0x6c, 0xbf, 0x87, 0xa2, 0x3e,
                                                            0x57, 0x3c, 0x33, 0xa6, 0x22, 0x51, 0x02, 0xc9, 0x34, 0x25, 0x8c,
                                                            0xbe, 0x5b, 0x42, 0xac, 0x7d, 0x39, 0x3f, 0x9e, 0x40, 0x8b, 0x21,
                                                            0xf1, 0x86, 0x97, 0x79, 0xc9, 0x66, 0x3d, 0xd5, 0xab, 0xaa, 0x1b,
                                                            0x18, 0xac, 0x87, 0x27, 0x7a, 0xec, 0x19, 0x9b, 0x24, 0x07, 0x1d,
                                                            0x6f, 0x41, 0x34, 0x80, 0x57, 0x2d, 0x11, 0x94, 0xf8, 0x28 };
static const uint8_t chiptest_fcf59b3113ef_scalar2_118[]   = { 0xfc, 0x90, 0x80, 0x01, 0xad, 0x4e, 0x8a, 0xfb, 0x17, 0xf7, 0x7a,
                                                             0x51, 0x14, 0xfd, 0x19, 0x41, 0xf9, 0x7a, 0x16, 0x87, 0x78, 0xe8,
                                                             0x24, 0x5c, 0x48, 0xf7, 0x4b, 0xdb, 0x31, 0xe6, 0x45, 0x2a };
static const uint8_t chiptest_fcf59b3113ef_out_point_119[] = { 0x04, 0x85, 0x16, 0xf2, 0x66, 0xd4, 0x10, 0x63, 0xed, 0xb8, 0xe1,
                                                               0x2c, 0x1f, 0x31, 0x13, 0xcb, 0xa0, 0x6f, 0xa9, 0xbd, 0x4e, 0x0b,
                                                               0x11, 0x94, 0xab, 0x88, 0x5a, 0x53, 0xc4, 0xdb, 0x7b, 0x35, 0x72,
                                                               0x6d, 0xaa, 0x9b, 0x81, 0xc3, 0x3d, 0x34, 0x4d, 0x67, 0x2b, 0xfc,
                                                               0x6d, 0x0d, 0x75, 0xb2, 0x1c, 0x7f, 0x00, 0x13, 0x1a, 0xe7, 0xcd,
                                                               0xdf, 0x36, 0x63, 0xdd, 0xc2, 0x76, 0xaa, 0xfa, 0x72, 0x5c };
static const struct spake2p_point_muladd_tv chiptest_fcf59b3113ef_test_vector_120 = {
    .point1        = chiptest_fcf59b3113ef_point1_115,
    .point1_len    = 65,
    .scalar1       = chiptest_fcf59b3113ef_scalar1_116,
    .scalar1_len   = 32,
    .point2        = chiptest_fcf59b3113ef_point2_117,
    .point2_len    = 65,
    .scalar2       = chiptest_fcf59b3113ef_scalar2_118,
    .scalar2_len   = 32,
    .out_point     = chiptest_fcf59b3113ef_out_point_119,
    .out_point_len = 65,
};
static const struct spake2p_point_muladd_tv * point_muladd_tvs[] = {
    &chiptest_fcf59b3113ef_test_vector_6,   &chiptest_fcf59b3113ef_test_vector_12,  &chiptest_fcf59b3113ef_test_vector_18,
    &chiptest_fcf59b3113ef_test_vector_24,  &chiptest_fcf59b3113ef_test_vector_30,  &chiptest_fcf59b3113ef_test_vector_36,
    &chiptest_fcf59b3113ef_test_vector_42,  &chiptest_fcf59b3113ef_test_vector_48,  &chiptest_fcf59b3113ef_test_vector_54,
    &chiptest_fcf59b3113ef_test_vector_60,  &chiptest_fcf59b3113ef_test_vector_66,  &chiptest_fcf59b3113ef_test_vector_72,
    &chiptest_fcf59b3113ef_test_vector_78,  &chiptest_fcf59b3113ef_test_vector_84,  &chiptest_fcf59b3113ef_test_vector_90,
    &chiptest_fcf59b3113ef_test_vector_96,  &chiptest_fcf59b3113ef_test_vector_102, &chiptest_fcf59b3113ef_test_vector_108,
    &chiptest_fcf59b3113ef_test_vector_114, &chiptest_fcf59b3113ef_test_vector_120
};

} // namespace Crypto
} // namespace chip
