/*
 *
 *    Copyright (c) 2020 Project CHIP Authors
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

/**
 * @file - This file contains elliptic curve point multipication test vectors.
 */

#pragma once

namespace chip {
namespace Crypto {

struct spake2p_point_mul_tv
{
    const uint8_t * point;
    size_t point_len;
    const uint8_t * scalar;
    size_t scalar_len;
    const uint8_t * out_point;
    size_t out_point_len;
    int valid;
};

static const uint8_t chiptest_0fd26a42aa63_point_1[]     = { 0x04, 0x11, 0xc6, 0x0f, 0x3c, 0x30, 0x72, 0x47, 0x4f, 0xec, 0x46,
                                                         0x86, 0x68, 0xe6, 0x92, 0xc2, 0xfe, 0xd2, 0x32, 0x75, 0x57, 0x03,
                                                         0x9b, 0x74, 0xc1, 0x03, 0x56, 0xff, 0x4a, 0x4f, 0xa2, 0x09, 0x88,
                                                         0x7e, 0xca, 0xfa, 0xd1, 0xe7, 0x05, 0x10, 0xb8, 0x0e, 0x89, 0x30,
                                                         0xb9, 0x8b, 0x09, 0x69, 0x02, 0x09, 0x6c, 0x5a, 0x16, 0x50, 0x7b,
                                                         0xf7, 0x02, 0x24, 0x5c, 0x8f, 0x56, 0xef, 0xcf, 0xdf, 0xf3 };
static const uint8_t chiptest_0fd26a42aa63_scalar_2[]    = { 0xa4, 0xce, 0x40, 0x79, 0x0c, 0x1a, 0xf0, 0xcd, 0xa9, 0x67, 0xac,
                                                          0xb6, 0x66, 0x28, 0xa6, 0x65, 0xd3, 0x4e, 0x2f, 0x1c, 0xd4, 0xe8,
                                                          0x06, 0xe4, 0x8f, 0x6d, 0xc9, 0x30, 0x0b, 0x3b, 0xc3, 0xb6 };
static const uint8_t chiptest_0fd26a42aa63_out_point_3[] = { 0x04, 0xc4, 0x1e, 0x45, 0x5d, 0x9b, 0xf8, 0xe2, 0x71, 0x40, 0xe4,
                                                             0x4d, 0x86, 0xd3, 0xa3, 0xc3, 0xa2, 0x84, 0xda, 0x30, 0xd1, 0xae,
                                                             0x41, 0x48, 0x2c, 0x1d, 0x6c, 0xa4, 0x96, 0x02, 0x81, 0xf1, 0xb8,
                                                             0x7b, 0xe6, 0x6d, 0xe3, 0xd3, 0xf0, 0x99, 0x44, 0x45, 0xa4, 0x5d,
                                                             0xba, 0xae, 0x0a, 0xb1, 0xa1, 0x62, 0x61, 0x4c, 0xd5, 0x42, 0x35,
                                                             0x89, 0xff, 0x25, 0xe1, 0x10, 0x7f, 0x23, 0xad, 0x4b, 0x2b };
static const struct spake2p_point_mul_tv chiptest_0fd26a42aa63_test_vector_4 = {
    .point         = chiptest_0fd26a42aa63_point_1,
    .point_len     = 65,
    .scalar        = chiptest_0fd26a42aa63_scalar_2,
    .scalar_len    = 32,
    .out_point     = chiptest_0fd26a42aa63_out_point_3,
    .out_point_len = 65,
};
static const uint8_t chiptest_0fd26a42aa63_point_5[]     = { 0x04, 0xed, 0x09, 0x15, 0x95, 0x7b, 0x1a, 0x2b, 0x71, 0x31, 0x54,
                                                         0x9c, 0xa5, 0x23, 0x5c, 0x9b, 0x0e, 0x00, 0x4d, 0x84, 0x7d, 0xa1,
                                                         0x52, 0xfb, 0x3d, 0xc3, 0x58, 0xa3, 0xfd, 0x53, 0x5a, 0x31, 0x28,
                                                         0x67, 0x10, 0xf7, 0x69, 0x10, 0xab, 0x7b, 0xfa, 0xfa, 0x2f, 0xce,
                                                         0x55, 0x6d, 0x1e, 0x1e, 0x09, 0x17, 0x49, 0x46, 0x0e, 0xcb, 0x67,
                                                         0xb4, 0x44, 0x83, 0x08, 0x98, 0x4b, 0xe5, 0xc9, 0xc4, 0x28 };
static const uint8_t chiptest_0fd26a42aa63_scalar_6[]    = { 0x14, 0x96, 0x87, 0x7b, 0xed, 0x29, 0x11, 0xa8, 0x58, 0xf2, 0xd8,
                                                          0x5f, 0x45, 0xcd, 0xff, 0x9c, 0xfd, 0xcf, 0x8e, 0xc4, 0x8e, 0xd2,
                                                          0x86, 0xa1, 0xd1, 0xb2, 0x13, 0x52, 0x0f, 0x59, 0xdb, 0x68 };
static const uint8_t chiptest_0fd26a42aa63_out_point_7[] = { 0x04, 0x09, 0x4f, 0x3f, 0xf6, 0xc4, 0x88, 0x74, 0x15, 0x87, 0x67,
                                                             0xf1, 0x0e, 0x8e, 0x51, 0x7c, 0x46, 0x79, 0x86, 0xe7, 0x17, 0x19,
                                                             0x1f, 0x06, 0xec, 0xa2, 0x66, 0x6d, 0x45, 0x3a, 0xb5, 0xf9, 0xce,
                                                             0xce, 0xc3, 0xac, 0xdb, 0x91, 0xe8, 0x8c, 0x12, 0x63, 0xd9, 0x46,
                                                             0x60, 0x2b, 0x0b, 0x92, 0xe6, 0xf2, 0x0f, 0x0e, 0x89, 0x51, 0x8f,
                                                             0xae, 0xbf, 0x08, 0xcc, 0x4e, 0x37, 0x86, 0x00, 0xe4, 0x4a };
static const struct spake2p_point_mul_tv chiptest_0fd26a42aa63_test_vector_8 = {
    .point         = chiptest_0fd26a42aa63_point_5,
    .point_len     = 65,
    .scalar        = chiptest_0fd26a42aa63_scalar_6,
    .scalar_len    = 32,
    .out_point     = chiptest_0fd26a42aa63_out_point_7,
    .out_point_len = 65,
};
static const uint8_t chiptest_0fd26a42aa63_point_9[]      = { 0x04, 0xc5, 0xf5, 0x1e, 0x3c, 0x06, 0x56, 0x19, 0x9d, 0xae, 0xcf,
                                                         0x7f, 0xad, 0x11, 0x1b, 0x77, 0xa0, 0xa5, 0xd2, 0xb4, 0x1a, 0x7e,
                                                         0xea, 0xc6, 0xf3, 0x0e, 0x96, 0xb8, 0x2a, 0x90, 0xd7, 0x51, 0x73,
                                                         0x9e, 0x9d, 0x1d, 0x75, 0xf3, 0xbb, 0xa2, 0xe9, 0xff, 0x53, 0x18,
                                                         0xb1, 0x3c, 0xa8, 0x1d, 0xea, 0x52, 0x33, 0x59, 0x3f, 0x43, 0x65,
                                                         0xea, 0x88, 0x9c, 0x2a, 0x02, 0xe3, 0x2e, 0xb7, 0x2a, 0xd8 };
static const uint8_t chiptest_0fd26a42aa63_scalar_10[]    = { 0x24, 0x6c, 0xf5, 0xef, 0x60, 0xfa, 0x9f, 0x72, 0xef, 0xce, 0x30,
                                                           0x8e, 0xaa, 0x04, 0x3e, 0x07, 0x69, 0x49, 0xff, 0xfc, 0x94, 0x0a,
                                                           0xf5, 0xe3, 0xcd, 0x24, 0x66, 0x5e, 0x7b, 0x8d, 0x3a, 0x3b };
static const uint8_t chiptest_0fd26a42aa63_out_point_11[] = { 0x04, 0x01, 0x1b, 0xb4, 0xf0, 0x69, 0xbf, 0x57, 0xee, 0x07, 0x48,
                                                              0xb9, 0x4a, 0x39, 0x2d, 0x15, 0xb7, 0x16, 0x99, 0xa4, 0xce, 0x15,
                                                              0x75, 0xcf, 0x7b, 0xd0, 0xd6, 0x63, 0xde, 0x06, 0xc0, 0x2d, 0x4e,
                                                              0x12, 0xaa, 0x2b, 0x03, 0x0c, 0xf8, 0xed, 0x9d, 0x41, 0xb6, 0x87,
                                                              0xdf, 0xa8, 0x6f, 0x3c, 0xd9, 0x3d, 0x3b, 0x35, 0xf7, 0xa3, 0x06,
                                                              0xcd, 0x14, 0xda, 0x3c, 0x0e, 0x15, 0x57, 0x22, 0xfb, 0xba };
static const struct spake2p_point_mul_tv chiptest_0fd26a42aa63_test_vector_12 = {
    .point         = chiptest_0fd26a42aa63_point_9,
    .point_len     = 65,
    .scalar        = chiptest_0fd26a42aa63_scalar_10,
    .scalar_len    = 32,
    .out_point     = chiptest_0fd26a42aa63_out_point_11,
    .out_point_len = 65,
};
static const uint8_t chiptest_0fd26a42aa63_point_13[]     = { 0x04, 0xb3, 0x54, 0xc6, 0x21, 0x42, 0xa2, 0xbb, 0xd8, 0x8e, 0x53,
                                                          0xb1, 0x11, 0x62, 0x7b, 0x60, 0xcd, 0xff, 0xc5, 0xf1, 0x98, 0x61,
                                                          0x26, 0x7b, 0x01, 0x8a, 0x64, 0x3b, 0x0b, 0x69, 0x77, 0x88, 0x90,
                                                          0x50, 0x3b, 0x73, 0xef, 0x3a, 0xb8, 0x05, 0xb4, 0x48, 0x6b, 0xc5,
                                                          0x28, 0xb9, 0xff, 0x37, 0xef, 0xc6, 0xa8, 0xc2, 0x77, 0x62, 0xbf,
                                                          0x8d, 0x2c, 0x3d, 0x40, 0x83, 0x13, 0x01, 0xeb, 0x75, 0x65 };
static const uint8_t chiptest_0fd26a42aa63_scalar_14[]    = { 0x56, 0x14, 0xeb, 0xa3, 0xe4, 0x92, 0x02, 0x34, 0xd0, 0x07, 0x7d,
                                                           0x79, 0x7c, 0xe7, 0xec, 0xec, 0x06, 0x7f, 0x11, 0x89, 0xb6, 0x38,
                                                           0x0b, 0xbb, 0x1b, 0x70, 0xed, 0xed, 0xf2, 0x65, 0xd7, 0x03 };
static const uint8_t chiptest_0fd26a42aa63_out_point_15[] = { 0x04, 0xf6, 0x7a, 0x2f, 0x66, 0x9e, 0xa9, 0xb5, 0xbc, 0xf4, 0x42,
                                                              0x80, 0x9c, 0xc5, 0xf2, 0x08, 0xa4, 0xe2, 0x05, 0x64, 0xdd, 0x23,
                                                              0x8a, 0x3e, 0x60, 0xaa, 0x66, 0xd6, 0xe6, 0x4e, 0x11, 0x51, 0x32,
                                                              0xd3, 0x6d, 0xa4, 0x13, 0xac, 0x50, 0xb2, 0x26, 0x8a, 0x50, 0xe4,
                                                              0x4e, 0xfa, 0xb7, 0xe0, 0x18, 0x5c, 0x91, 0x51, 0xd3, 0x75, 0xf2,
                                                              0xfe, 0x6f, 0x7d, 0xf6, 0xe5, 0x32, 0x19, 0xaa, 0x60, 0x71 };
static const struct spake2p_point_mul_tv chiptest_0fd26a42aa63_test_vector_16 = {
    .point         = chiptest_0fd26a42aa63_point_13,
    .point_len     = 65,
    .scalar        = chiptest_0fd26a42aa63_scalar_14,
    .scalar_len    = 32,
    .out_point     = chiptest_0fd26a42aa63_out_point_15,
    .out_point_len = 65,
};
static const uint8_t chiptest_0fd26a42aa63_point_17[]     = { 0x04, 0xf9, 0xf7, 0x75, 0xbb, 0xe4, 0x94, 0x87, 0xa4, 0x53, 0x0b,
                                                          0xe8, 0xff, 0x52, 0xfd, 0x14, 0x6d, 0x06, 0xe8, 0x6a, 0xae, 0x7e,
                                                          0xa4, 0x97, 0xf6, 0xfb, 0x71, 0x15, 0x16, 0x69, 0x68, 0x56, 0xab,
                                                          0xce, 0xb1, 0xa6, 0x76, 0x54, 0xc3, 0xd0, 0xeb, 0x54, 0x04, 0x6c,
                                                          0x87, 0x95, 0x92, 0xd3, 0x6d, 0xa2, 0x9a, 0xba, 0xbc, 0x9f, 0x2f,
                                                          0x0d, 0xc0, 0xc7, 0xcc, 0x5f, 0xeb, 0x2b, 0x35, 0x24, 0x22 };
static const uint8_t chiptest_0fd26a42aa63_scalar_18[]    = { 0x4c, 0xc3, 0xec, 0x10, 0x2d, 0x90, 0x9b, 0xdf, 0xf8, 0xb5, 0x1b,
                                                           0xa3, 0x78, 0x77, 0x93, 0x90, 0xb9, 0xcf, 0x89, 0x4f, 0xcd, 0x00,
                                                           0x3b, 0xe0, 0x12, 0x78, 0xe0, 0xb8, 0xd6, 0x75, 0x53, 0xe7 };
static const uint8_t chiptest_0fd26a42aa63_out_point_19[] = { 0x04, 0x91, 0xdb, 0xa2, 0x99, 0xe1, 0xbc, 0xb4, 0x6b, 0xfb, 0xde,
                                                              0x5b, 0x39, 0x86, 0xdf, 0x8c, 0xd6, 0x55, 0xbe, 0x57, 0x9e, 0xd6,
                                                              0xf8, 0x21, 0x8f, 0xcb, 0x62, 0x3a, 0x6b, 0x3c, 0x1f, 0xaf, 0xaa,
                                                              0xd7, 0xd7, 0x43, 0x22, 0x2a, 0xbf, 0xfa, 0x94, 0x10, 0xd9, 0x37,
                                                              0xab, 0x7a, 0x2a, 0xc0, 0x3a, 0x11, 0xae, 0xc8, 0x38, 0xf8, 0x92,
                                                              0xdb, 0xf5, 0x39, 0x77, 0x66, 0x15, 0x29, 0x2f, 0x44, 0x2d };
static const struct spake2p_point_mul_tv chiptest_0fd26a42aa63_test_vector_20 = {
    .point         = chiptest_0fd26a42aa63_point_17,
    .point_len     = 65,
    .scalar        = chiptest_0fd26a42aa63_scalar_18,
    .scalar_len    = 32,
    .out_point     = chiptest_0fd26a42aa63_out_point_19,
    .out_point_len = 65,
};
static const uint8_t chiptest_0fd26a42aa63_point_21[]     = { 0x04, 0xf5, 0xb1, 0xbe, 0x11, 0xd3, 0xa8, 0x01, 0xac, 0xc6, 0xf3,
                                                          0x53, 0xe1, 0xf9, 0x6e, 0xce, 0x70, 0xe2, 0x42, 0x0a, 0xa4, 0x1f,
                                                          0xf4, 0x55, 0xc7, 0xff, 0xfc, 0xef, 0xa5, 0xb0, 0x96, 0x15, 0x2e,
                                                          0x7f, 0x2f, 0x81, 0x44, 0xe1, 0xa0, 0xb6, 0xac, 0x03, 0xb3, 0x39,
                                                          0x83, 0x5b, 0x94, 0xfd, 0x82, 0x52, 0x89, 0x93, 0xe3, 0xf7, 0x42,
                                                          0xb0, 0xad, 0xe8, 0x6b, 0x95, 0xdf, 0xc5, 0x1c, 0x48, 0xc2 };
static const uint8_t chiptest_0fd26a42aa63_scalar_22[]    = { 0x14, 0x88, 0x89, 0x3b, 0x0d, 0x4e, 0xe0, 0x83, 0x58, 0xae, 0x99,
                                                           0x20, 0x66, 0x92, 0x79, 0x5f, 0xb8, 0x05, 0x44, 0x77, 0xbc, 0x88,
                                                           0xfa, 0x81, 0xae, 0xfe, 0xd2, 0x17, 0x91, 0xa9, 0xfe, 0xbf };
static const uint8_t chiptest_0fd26a42aa63_out_point_23[] = { 0x04, 0x7a, 0x85, 0x1b, 0xcf, 0x5f, 0xc1, 0x1a, 0xa5, 0x93, 0x9d,
                                                              0xda, 0xd4, 0xc1, 0xa3, 0x9c, 0x94, 0xad, 0xe2, 0x0c, 0x0a, 0x96,
                                                              0x5c, 0xd8, 0xde, 0x5d, 0x15, 0xae, 0x8b, 0x2d, 0x28, 0xa4, 0xe8,
                                                              0xb8, 0x51, 0xdb, 0xe6, 0xe2, 0x28, 0x9f, 0xa9, 0x31, 0xb2, 0x1a,
                                                              0x43, 0x62, 0x76, 0xdd, 0x21, 0x41, 0x32, 0xe3, 0x91, 0x50, 0x3d,
                                                              0xdc, 0x69, 0x4f, 0x3b, 0xee, 0xe9, 0x23, 0x0e, 0xb8, 0xd8 };
static const struct spake2p_point_mul_tv chiptest_0fd26a42aa63_test_vector_24 = {
    .point         = chiptest_0fd26a42aa63_point_21,
    .point_len     = 65,
    .scalar        = chiptest_0fd26a42aa63_scalar_22,
    .scalar_len    = 32,
    .out_point     = chiptest_0fd26a42aa63_out_point_23,
    .out_point_len = 65,
};
static const uint8_t chiptest_0fd26a42aa63_point_25[]     = { 0x04, 0x7f, 0x34, 0x22, 0x9b, 0xbd, 0x59, 0xa1, 0xff, 0xf3, 0x37,
                                                          0xd8, 0x5d, 0xe5, 0xf3, 0x57, 0xe9, 0xed, 0xdf, 0xf4, 0x81, 0xbe,
                                                          0xfa, 0xd2, 0x65, 0x3e, 0xfd, 0x62, 0xac, 0x9c, 0x56, 0x22, 0x1b,
                                                          0x14, 0xb7, 0x36, 0xf8, 0x9a, 0x0a, 0x53, 0x10, 0xc0, 0x7a, 0x3f,
                                                          0x3a, 0xf0, 0xf0, 0x24, 0x2b, 0xef, 0xa5, 0x0c, 0xfa, 0xbe, 0xf4,
                                                          0xc3, 0x21, 0x97, 0x35, 0xf5, 0x4c, 0xd3, 0x09, 0x31, 0x75 };
static const uint8_t chiptest_0fd26a42aa63_scalar_26[]    = { 0x73, 0x8b, 0xe1, 0x07, 0xb1, 0x92, 0xba, 0x31, 0x15, 0x4e, 0xde,
                                                           0x86, 0xe9, 0xa3, 0x4c, 0x03, 0xb1, 0x86, 0x21, 0x1c, 0xaa, 0x6b,
                                                           0xfc, 0xcd, 0x7c, 0xc0, 0xc9, 0x1d, 0xf4, 0xb6, 0x07, 0x97 };
static const uint8_t chiptest_0fd26a42aa63_out_point_27[] = { 0x04, 0x01, 0x46, 0xd9, 0x17, 0xe7, 0xdd, 0x36, 0x4e, 0xc3, 0x2a,
                                                              0x77, 0xa9, 0x38, 0xa2, 0xd7, 0xcc, 0xda, 0x2a, 0x3d, 0xba, 0x33,
                                                              0xd0, 0x85, 0x28, 0xc7, 0x9f, 0x39, 0xb7, 0xbc, 0xaf, 0x4c, 0x1f,
                                                              0x1d, 0x7b, 0xa4, 0xf4, 0x50, 0x8b, 0x87, 0xaa, 0x24, 0xbd, 0x2f,
                                                              0x75, 0x78, 0x3e, 0xa7, 0x82, 0xce, 0xb9, 0x2e, 0x88, 0x7a, 0xf2,
                                                              0x20, 0x65, 0x6d, 0x95, 0xb8, 0x63, 0x98, 0x03, 0xe3, 0x51 };
static const struct spake2p_point_mul_tv chiptest_0fd26a42aa63_test_vector_28 = {
    .point         = chiptest_0fd26a42aa63_point_25,
    .point_len     = 65,
    .scalar        = chiptest_0fd26a42aa63_scalar_26,
    .scalar_len    = 32,
    .out_point     = chiptest_0fd26a42aa63_out_point_27,
    .out_point_len = 65,
};
static const uint8_t chiptest_0fd26a42aa63_point_29[]     = { 0x04, 0xd6, 0x55, 0x49, 0x55, 0xcd, 0x8c, 0x16, 0xc6, 0x67, 0xb2,
                                                          0x44, 0xb0, 0xc0, 0x81, 0xb0, 0x98, 0x38, 0xc4, 0x39, 0x47, 0x9f,
                                                          0x7e, 0x37, 0x55, 0x32, 0x5d, 0x81, 0x47, 0x0a, 0xdd, 0x89, 0x1b,
                                                          0x0d, 0x78, 0x69, 0x7d, 0xc2, 0xb7, 0xa7, 0x90, 0x96, 0xf2, 0x14,
                                                          0xbe, 0x6d, 0x51, 0x3f, 0xab, 0xc8, 0x7b, 0x90, 0x10, 0xe0, 0x4d,
                                                          0x59, 0xf3, 0xb7, 0x44, 0x89, 0x0b, 0x46, 0x8a, 0x73, 0x0d };
static const uint8_t chiptest_0fd26a42aa63_scalar_30[]    = { 0x18, 0xe4, 0x86, 0x61, 0xb8, 0x9c, 0x90, 0x5d, 0x79, 0x82, 0x63,
                                                           0x49, 0xfe, 0x40, 0xca, 0xff, 0xda, 0x53, 0xcd, 0x24, 0xee, 0x6f,
                                                           0xa2, 0xbe, 0x9a, 0xb6, 0x73, 0xd3, 0x74, 0x63, 0xb4, 0xf1 };
static const uint8_t chiptest_0fd26a42aa63_out_point_31[] = { 0x04, 0xa3, 0xd5, 0x6e, 0xa9, 0x72, 0x53, 0xae, 0x05, 0x8a, 0xb5,
                                                              0xa2, 0xac, 0x86, 0x36, 0x80, 0x53, 0xd8, 0x93, 0xca, 0xfe, 0xec,
                                                              0x14, 0x76, 0x8a, 0x97, 0x34, 0x1a, 0xe2, 0x82, 0xc6, 0x20, 0x78,
                                                              0x0f, 0x01, 0x62, 0x40, 0x40, 0x23, 0xe7, 0x51, 0xda, 0xb4, 0x41,
                                                              0xaa, 0x9c, 0x4a, 0x1d, 0xef, 0x92, 0x8a, 0xf8, 0xd7, 0x8d, 0x97,
                                                              0x0e, 0xaf, 0xc8, 0x02, 0x5a, 0x9c, 0x8e, 0x7a, 0x86, 0xe9 };
static const struct spake2p_point_mul_tv chiptest_0fd26a42aa63_test_vector_32 = {
    .point         = chiptest_0fd26a42aa63_point_29,
    .point_len     = 65,
    .scalar        = chiptest_0fd26a42aa63_scalar_30,
    .scalar_len    = 32,
    .out_point     = chiptest_0fd26a42aa63_out_point_31,
    .out_point_len = 65,
};
static const uint8_t chiptest_0fd26a42aa63_point_33[]     = { 0x04, 0x0d, 0x49, 0xad, 0x9f, 0xb0, 0x45, 0x7b, 0x61, 0x85, 0x41,
                                                          0x32, 0x19, 0xc0, 0x51, 0x4d, 0x90, 0x23, 0x78, 0x71, 0x23, 0xde,
                                                          0x56, 0x22, 0x2b, 0xc2, 0x02, 0xa1, 0x72, 0x5c, 0x74, 0x51, 0x08,
                                                          0x5c, 0x51, 0x99, 0x93, 0x60, 0x83, 0x9a, 0x4c, 0x05, 0x66, 0x1f,
                                                          0x6f, 0x04, 0x50, 0xd3, 0x27, 0x92, 0x44, 0x08, 0x3d, 0x8d, 0x8d,
                                                          0xee, 0xd7, 0x29, 0x1b, 0xb5, 0x82, 0xd1, 0x37, 0x9f, 0xb6 };
static const uint8_t chiptest_0fd26a42aa63_scalar_34[]    = { 0x1b, 0x86, 0x6d, 0x87, 0x31, 0x79, 0x68, 0x37, 0x4e, 0xa1, 0xb2,
                                                           0xa4, 0x61, 0x36, 0x71, 0x24, 0x2a, 0xc7, 0x0d, 0x2a, 0x36, 0x79,
                                                           0x46, 0x4e, 0x42, 0x89, 0x9b, 0x89, 0x76, 0xc2, 0x90, 0xa3 };
static const uint8_t chiptest_0fd26a42aa63_out_point_35[] = { 0x04, 0xc0, 0xd3, 0x40, 0x8f, 0x67, 0x35, 0xab, 0x47, 0xb5, 0x82,
                                                              0x3c, 0xb7, 0x08, 0x92, 0x7a, 0xe2, 0xb3, 0x46, 0x6d, 0x4e, 0x77,
                                                              0xca, 0xee, 0xb8, 0x62, 0x79, 0x2e, 0xc4, 0xa1, 0xc2, 0x99, 0xcb,
                                                              0xe9, 0xa8, 0x7e, 0x73, 0x8a, 0xe3, 0x44, 0x84, 0x70, 0xe1, 0x5e,
                                                              0x56, 0xdb, 0x8e, 0xa2, 0x78, 0x50, 0x08, 0xef, 0x26, 0xe6, 0x72,
                                                              0xec, 0x00, 0x0c, 0xc1, 0x89, 0xdb, 0x4b, 0x7a, 0xf3, 0x4c };
static const struct spake2p_point_mul_tv chiptest_0fd26a42aa63_test_vector_36 = {
    .point         = chiptest_0fd26a42aa63_point_33,
    .point_len     = 65,
    .scalar        = chiptest_0fd26a42aa63_scalar_34,
    .scalar_len    = 32,
    .out_point     = chiptest_0fd26a42aa63_out_point_35,
    .out_point_len = 65,
};
static const uint8_t chiptest_0fd26a42aa63_point_37[]     = { 0x04, 0x38, 0x44, 0x33, 0x8d, 0x85, 0x5a, 0xe4, 0x6c, 0xfc, 0xc8,
                                                          0x0f, 0x07, 0xc9, 0x26, 0xb5, 0xd6, 0x42, 0x72, 0x38, 0x8e, 0xcb,
                                                          0x4c, 0xee, 0x2f, 0x43, 0xe9, 0x83, 0x74, 0xdc, 0xdb, 0x8a, 0x4d,
                                                          0x9b, 0xbc, 0xd2, 0x8e, 0xeb, 0x3a, 0xea, 0x62, 0x09, 0x4c, 0x9d,
                                                          0x98, 0xcc, 0xab, 0x78, 0xfc, 0xa8, 0xa1, 0x7b, 0x35, 0xf4, 0x7a,
                                                          0x63, 0x7c, 0x9f, 0x3e, 0xd3, 0x19, 0xb3, 0x15, 0x3b, 0x87 };
static const uint8_t chiptest_0fd26a42aa63_scalar_38[]    = { 0xd0, 0x41, 0x88, 0x24, 0xfe, 0x36, 0x62, 0x81, 0xd7, 0xaa, 0xde,
                                                           0xfb, 0x34, 0x41, 0x98, 0x7f, 0x13, 0x2f, 0x8b, 0x15, 0x94, 0xbc,
                                                           0x11, 0xc3, 0x41, 0xff, 0xa0, 0xd6, 0x31, 0x7f, 0x22, 0x23 };
static const uint8_t chiptest_0fd26a42aa63_out_point_39[] = { 0x04, 0x9e, 0xee, 0x22, 0xab, 0x31, 0x46, 0x78, 0xaf, 0xeb, 0xb6,
                                                              0x70, 0x10, 0xdd, 0x02, 0x12, 0x75, 0x4a, 0x86, 0xad, 0x3f, 0x32,
                                                              0xe9, 0xb0, 0xd1, 0x2f, 0x65, 0x25, 0x0e, 0x75, 0xbd, 0x51, 0x20,
                                                              0x95, 0xc0, 0x67, 0x3c, 0x58, 0x7b, 0x52, 0xed, 0x2a, 0x1b, 0x32,
                                                              0x19, 0x16, 0xc3, 0xe0, 0x6f, 0xa8, 0xc5, 0xc6, 0x48, 0x4d, 0x61,
                                                              0xaa, 0x3f, 0x29, 0x7d, 0x3a, 0x1d, 0xde, 0x84, 0xab, 0xf2 };
static const struct spake2p_point_mul_tv chiptest_0fd26a42aa63_test_vector_40 = {
    .point         = chiptest_0fd26a42aa63_point_37,
    .point_len     = 65,
    .scalar        = chiptest_0fd26a42aa63_scalar_38,
    .scalar_len    = 32,
    .out_point     = chiptest_0fd26a42aa63_out_point_39,
    .out_point_len = 65,
};
static const uint8_t chiptest_0fd26a42aa63_point_41[]     = { 0x04, 0x47, 0x90, 0x7a, 0xea, 0xc9, 0x01, 0xd6, 0x9f, 0xc3, 0xeb,
                                                          0x67, 0x03, 0x4f, 0x1a, 0x79, 0x13, 0xde, 0x15, 0x69, 0x24, 0x04,
                                                          0x9c, 0x4d, 0x9b, 0x78, 0x58, 0x9d, 0x31, 0x88, 0xbb, 0x90, 0x43,
                                                          0x28, 0x4c, 0x35, 0x58, 0xa6, 0x46, 0xf1, 0x4d, 0xb0, 0x6b, 0xc4,
                                                          0xf2, 0x43, 0x93, 0x8d, 0xfc, 0x74, 0x12, 0xaa, 0x53, 0x39, 0x75,
                                                          0x97, 0xee, 0x9e, 0xe8, 0xe4, 0x45, 0x20, 0x68, 0x7c, 0x2f };
static const uint8_t chiptest_0fd26a42aa63_scalar_42[]    = { 0x86, 0xb9, 0xb0, 0xfd, 0x6b, 0x95, 0x63, 0xed, 0x69, 0xf8, 0x64,
                                                           0x7e, 0x42, 0xa1, 0xf5, 0x32, 0x67, 0x1c, 0x84, 0xbd, 0x19, 0x05,
                                                           0xa9, 0xa2, 0x57, 0x88, 0xa1, 0x6e, 0x56, 0x5a, 0xf6, 0xe1 };
static const uint8_t chiptest_0fd26a42aa63_out_point_43[] = { 0x04, 0x17, 0x34, 0x86, 0xf5, 0xc9, 0x20, 0xbf, 0x0f, 0x3e, 0x99,
                                                              0x6e, 0x65, 0xd4, 0xe4, 0xf3, 0xbe, 0x6e, 0xf2, 0x5a, 0x72, 0x27,
                                                              0xff, 0xfd, 0x09, 0xf8, 0x0b, 0x11, 0x04, 0xad, 0x35, 0x7a, 0x1c,
                                                              0xf4, 0x1b, 0xee, 0x1e, 0x17, 0xf1, 0xe5, 0x09, 0x9d, 0x1c, 0x43,
                                                              0xbc, 0xd3, 0x6f, 0x53, 0x92, 0x6d, 0x78, 0x07, 0x8f, 0x73, 0xec,
                                                              0xf6, 0xeb, 0x06, 0x58, 0x1b, 0xac, 0xba, 0x56, 0x80, 0xba };
static const struct spake2p_point_mul_tv chiptest_0fd26a42aa63_test_vector_44 = {
    .point         = chiptest_0fd26a42aa63_point_41,
    .point_len     = 65,
    .scalar        = chiptest_0fd26a42aa63_scalar_42,
    .scalar_len    = 32,
    .out_point     = chiptest_0fd26a42aa63_out_point_43,
    .out_point_len = 65,
};
static const uint8_t chiptest_0fd26a42aa63_point_45[]     = { 0x04, 0x5b, 0x87, 0x49, 0x93, 0x82, 0x19, 0x18, 0x75, 0xc7, 0x98,
                                                          0x29, 0x78, 0x75, 0x4c, 0x96, 0x2e, 0xf7, 0xbe, 0x02, 0x16, 0x70,
                                                          0xfc, 0xd2, 0xf0, 0x19, 0x2f, 0xd0, 0x5b, 0xc7, 0xe5, 0xde, 0xbc,
                                                          0x94, 0xbb, 0x48, 0x68, 0x4b, 0x22, 0xe5, 0xc9, 0xec, 0x75, 0xef,
                                                          0xa5, 0x2d, 0xab, 0x72, 0xfd, 0xbe, 0xb9, 0x6e, 0xb2, 0xe4, 0x18,
                                                          0xf8, 0x41, 0x94, 0xb1, 0x28, 0x0c, 0x00, 0xcc, 0x82, 0x40 };
static const uint8_t chiptest_0fd26a42aa63_scalar_46[]    = { 0xf3, 0x6b, 0x85, 0x87, 0xbb, 0xc1, 0xab, 0x8d, 0x8a, 0xee, 0x77,
                                                           0xdf, 0xac, 0x70, 0x49, 0xf5, 0xf8, 0xaa, 0x9d, 0x8c, 0x65, 0x2a,
                                                           0x08, 0x4c, 0xab, 0x74, 0x6e, 0xdd, 0xfe, 0x9b, 0x02, 0x99 };
static const uint8_t chiptest_0fd26a42aa63_out_point_47[] = { 0x04, 0xa4, 0x46, 0x6f, 0x1e, 0xf7, 0x91, 0xe4, 0x6e, 0xb4, 0xd0,
                                                              0xd6, 0xb5, 0xec, 0x55, 0x11, 0x4b, 0xe3, 0xa0, 0x45, 0x18, 0x0d,
                                                              0x90, 0xe5, 0xc7, 0x6e, 0x81, 0xa2, 0xd7, 0xe1, 0x6a, 0x5c, 0xb9,
                                                              0x3a, 0xf7, 0xe2, 0xd4, 0xc1, 0x25, 0x59, 0x6f, 0xf5, 0xa3, 0xf5,
                                                              0x21, 0x2a, 0xac, 0x55, 0x4e, 0x84, 0x05, 0xb6, 0x2f, 0xb6, 0x10,
                                                              0xae, 0xad, 0x92, 0x3f, 0x45, 0x4a, 0x10, 0x54, 0x6d, 0x1e };
static const struct spake2p_point_mul_tv chiptest_0fd26a42aa63_test_vector_48 = {
    .point         = chiptest_0fd26a42aa63_point_45,
    .point_len     = 65,
    .scalar        = chiptest_0fd26a42aa63_scalar_46,
    .scalar_len    = 32,
    .out_point     = chiptest_0fd26a42aa63_out_point_47,
    .out_point_len = 65,
};
static const uint8_t chiptest_0fd26a42aa63_point_49[]     = { 0x04, 0x25, 0xe9, 0xbf, 0x00, 0x4b, 0x69, 0x97, 0x29, 0xf7, 0x08,
                                                          0x4d, 0x3a, 0x78, 0x31, 0x7c, 0x2c, 0x79, 0x0e, 0x67, 0x13, 0x55,
                                                          0x9c, 0xd3, 0x5f, 0xeb, 0xa0, 0xd1, 0x43, 0x8b, 0x81, 0x83, 0x76,
                                                          0xa5, 0x92, 0x38, 0x0c, 0xff, 0xd4, 0x05, 0x10, 0xd2, 0xe0, 0xfc,
                                                          0x9f, 0xc4, 0xc6, 0x4a, 0x41, 0x03, 0x83, 0xeb, 0x41, 0x25, 0x63,
                                                          0xa1, 0x7d, 0x47, 0xf4, 0x13, 0xab, 0x1c, 0x7b, 0x46, 0x78 };
static const uint8_t chiptest_0fd26a42aa63_scalar_50[]    = { 0xd1, 0x02, 0x6d, 0x19, 0x7f, 0x73, 0x70, 0x3b, 0x16, 0x74, 0x15,
                                                           0xe1, 0x99, 0x96, 0x31, 0xc0, 0x51, 0x87, 0x22, 0xd0, 0x7c, 0x35,
                                                           0x6b, 0xd5, 0x84, 0x4e, 0x5e, 0x56, 0xa9, 0xb9, 0xb4, 0xdb };
static const uint8_t chiptest_0fd26a42aa63_out_point_51[] = { 0x04, 0x4d, 0x12, 0x56, 0x20, 0x34, 0x90, 0xc2, 0x7e, 0xb7, 0x9f,
                                                              0xfe, 0xf7, 0x22, 0x8f, 0x77, 0xa4, 0xe9, 0xa2, 0x7f, 0x87, 0x9a,
                                                              0xe5, 0x45, 0x40, 0xe9, 0x24, 0x90, 0x72, 0xcf, 0x8a, 0x02, 0xae,
                                                              0x91, 0x1a, 0x0e, 0x28, 0x67, 0x3f, 0x69, 0xd6, 0xaf, 0xfe, 0x3c,
                                                              0x74, 0xa1, 0x35, 0x6f, 0xd4, 0xbb, 0xbc, 0x9f, 0x74, 0x76, 0xea,
                                                              0xd3, 0x6e, 0xb3, 0x6b, 0x6c, 0xb6, 0x08, 0x22, 0x81, 0x3b };
static const struct spake2p_point_mul_tv chiptest_0fd26a42aa63_test_vector_52 = {
    .point         = chiptest_0fd26a42aa63_point_49,
    .point_len     = 65,
    .scalar        = chiptest_0fd26a42aa63_scalar_50,
    .scalar_len    = 32,
    .out_point     = chiptest_0fd26a42aa63_out_point_51,
    .out_point_len = 65,
};
static const uint8_t chiptest_0fd26a42aa63_point_53[]     = { 0x04, 0xbf, 0xee, 0x0a, 0x98, 0x12, 0x05, 0xf4, 0xb2, 0x2d, 0xa6,
                                                          0x16, 0x66, 0x63, 0x3d, 0x32, 0x45, 0x2a, 0x5e, 0xa9, 0x0a, 0x1f,
                                                          0x0f, 0x77, 0x55, 0x4d, 0x15, 0xd7, 0x25, 0x34, 0x70, 0x34, 0x16,
                                                          0xae, 0x2f, 0x1f, 0xd3, 0x42, 0x3a, 0x0a, 0xb4, 0x25, 0xd4, 0x2b,
                                                          0xbb, 0xf4, 0x4f, 0xaa, 0xe6, 0x85, 0x72, 0x3d, 0x07, 0xa1, 0xd1,
                                                          0xfb, 0x95, 0x17, 0xd4, 0xd4, 0xd0, 0x52, 0xa3, 0xb4, 0x73 };
static const uint8_t chiptest_0fd26a42aa63_scalar_54[]    = { 0x14, 0x75, 0x36, 0xe4, 0x65, 0xa7, 0xcf, 0x82, 0x9d, 0x42, 0xe3,
                                                           0xba, 0x15, 0x1f, 0xce, 0x6e, 0x0e, 0xd1, 0xcc, 0xfb, 0x34, 0x84,
                                                           0x8c, 0x54, 0x72, 0x6f, 0xfd, 0x9b, 0xa7, 0xed, 0x64, 0x8e };
static const uint8_t chiptest_0fd26a42aa63_out_point_55[] = { 0x04, 0x33, 0x27, 0x79, 0xda, 0x78, 0xff, 0x41, 0x68, 0xc0, 0x36,
                                                              0x55, 0xcd, 0x8a, 0xff, 0x9d, 0x95, 0x43, 0xe6, 0xd4, 0xbe, 0x3a,
                                                              0x53, 0x28, 0x23, 0x4f, 0x3f, 0x47, 0x5d, 0x16, 0x44, 0x3e, 0xc8,
                                                              0xf7, 0x0e, 0xce, 0x9f, 0x9e, 0xaa, 0x02, 0xea, 0x2e, 0x49, 0x96,
                                                              0x3d, 0x86, 0xcb, 0x33, 0x50, 0x4e, 0x91, 0x4f, 0xb9, 0x09, 0xf8,
                                                              0xe5, 0xa8, 0x53, 0x31, 0x5c, 0xf3, 0xe8, 0xef, 0x11, 0x3e };
static const struct spake2p_point_mul_tv chiptest_0fd26a42aa63_test_vector_56 = {
    .point         = chiptest_0fd26a42aa63_point_53,
    .point_len     = 65,
    .scalar        = chiptest_0fd26a42aa63_scalar_54,
    .scalar_len    = 32,
    .out_point     = chiptest_0fd26a42aa63_out_point_55,
    .out_point_len = 65,
};
static const uint8_t chiptest_0fd26a42aa63_point_57[]     = { 0x04, 0xfc, 0x03, 0x05, 0xd4, 0x05, 0x52, 0xd0, 0x0f, 0x5d, 0xb9,
                                                          0xcd, 0x43, 0xa5, 0x1c, 0x54, 0xf7, 0x9a, 0x98, 0x98, 0x9a, 0xac,
                                                          0x15, 0x77, 0x02, 0x8c, 0xcd, 0x9e, 0x92, 0x99, 0xca, 0x76, 0x53,
                                                          0x53, 0x54, 0x46, 0x57, 0x27, 0x56, 0x0a, 0x31, 0xe0, 0xc3, 0x4f,
                                                          0xdd, 0x3c, 0xf2, 0xae, 0xe7, 0x5d, 0x9d, 0x74, 0x5d, 0xbc, 0x01,
                                                          0x47, 0x62, 0x1d, 0x99, 0x60, 0x53, 0x5c, 0x45, 0x97, 0x8f };
static const uint8_t chiptest_0fd26a42aa63_scalar_58[]    = { 0x1d, 0x10, 0xa9, 0xc9, 0x0f, 0x54, 0x6b, 0xab, 0x00, 0xea, 0xc5,
                                                           0xe1, 0xf0, 0xcd, 0xc5, 0xbc, 0x60, 0xe0, 0x9e, 0x1b, 0xfc, 0x15,
                                                           0x91, 0xd0, 0x8d, 0x80, 0x8c, 0xeb, 0x0a, 0x21, 0x21, 0xef };
static const uint8_t chiptest_0fd26a42aa63_out_point_59[] = { 0x04, 0xec, 0xf6, 0xb5, 0xb2, 0x36, 0x33, 0x86, 0x94, 0x43, 0x21,
                                                              0x86, 0xdd, 0x68, 0xa0, 0xe8, 0xf7, 0xbf, 0xeb, 0x19, 0x0c, 0x4a,
                                                              0x1d, 0x3b, 0xdd, 0xe9, 0x5f, 0x3d, 0x3a, 0xd0, 0xed, 0x45, 0x44,
                                                              0xfd, 0x43, 0x10, 0xee, 0x41, 0x86, 0xa2, 0xf0, 0xa5, 0x80, 0xa3,
                                                              0x0e, 0xe6, 0x43, 0x18, 0x13, 0x77, 0x88, 0xed, 0x71, 0x17, 0xb7,
                                                              0xbb, 0xcf, 0xb3, 0x06, 0x73, 0x3e, 0x1c, 0xd8, 0x68, 0xdc };
static const struct spake2p_point_mul_tv chiptest_0fd26a42aa63_test_vector_60 = {
    .point         = chiptest_0fd26a42aa63_point_57,
    .point_len     = 65,
    .scalar        = chiptest_0fd26a42aa63_scalar_58,
    .scalar_len    = 32,
    .out_point     = chiptest_0fd26a42aa63_out_point_59,
    .out_point_len = 65,
};
static const uint8_t chiptest_0fd26a42aa63_point_61[]     = { 0x04, 0x4a, 0x7d, 0xcd, 0xcf, 0xe2, 0x5b, 0x5f, 0x03, 0xca, 0x29,
                                                          0x01, 0xdc, 0x3c, 0xad, 0xb8, 0x2e, 0xe0, 0x5a, 0xda, 0x6e, 0x31,
                                                          0xa9, 0x58, 0x97, 0x78, 0x65, 0xc0, 0x22, 0x36, 0xda, 0x9a, 0xe5,
                                                          0xda, 0x56, 0x42, 0x00, 0xd0, 0x1b, 0x5d, 0x11, 0xd2, 0xc6, 0x7b,
                                                          0x59, 0x8a, 0xeb, 0xf3, 0x6c, 0xc4, 0x81, 0x12, 0x77, 0xe8, 0x62,
                                                          0x49, 0x95, 0x14, 0xce, 0xe3, 0x95, 0x21, 0xb2, 0x37, 0xf6 };
static const uint8_t chiptest_0fd26a42aa63_scalar_62[]    = { 0x0c, 0xb6, 0x5f, 0xfd, 0x13, 0xf6, 0x43, 0xf9, 0xe0, 0xc1, 0xd0,
                                                           0xf4, 0x1d, 0xbc, 0x61, 0x11, 0x12, 0x89, 0xac, 0xe0, 0xd1, 0x92,
                                                           0xfc, 0x7e, 0x0d, 0xa4, 0x0b, 0x25, 0xc1, 0xa8, 0xa8, 0xf0 };
static const uint8_t chiptest_0fd26a42aa63_out_point_63[] = { 0x04, 0xfe, 0x72, 0xdb, 0xe8, 0xac, 0x31, 0x9b, 0xfd, 0xfd, 0x08,
                                                              0xbc, 0x97, 0xd3, 0x8c, 0x7d, 0xd5, 0x67, 0xf7, 0x41, 0xe4, 0x49,
                                                              0x8f, 0x09, 0xc4, 0xc1, 0x55, 0x8a, 0xdc, 0xe1, 0x95, 0x91, 0x43,
                                                              0x37, 0xe8, 0xcd, 0x08, 0xca, 0x3a, 0x55, 0x01, 0x84, 0x34, 0x72,
                                                              0xa0, 0xc6, 0x1d, 0xb3, 0xb3, 0x6d, 0xa3, 0x75, 0x13, 0x44, 0x21,
                                                              0xb1, 0x72, 0x74, 0x42, 0x75, 0xb0, 0x36, 0x34, 0xb7, 0x1d };
static const struct spake2p_point_mul_tv chiptest_0fd26a42aa63_test_vector_64 = {
    .point         = chiptest_0fd26a42aa63_point_61,
    .point_len     = 65,
    .scalar        = chiptest_0fd26a42aa63_scalar_62,
    .scalar_len    = 32,
    .out_point     = chiptest_0fd26a42aa63_out_point_63,
    .out_point_len = 65,
};
static const uint8_t chiptest_0fd26a42aa63_point_65[]     = { 0x04, 0x6f, 0x8e, 0x8b, 0x9b, 0x50, 0x49, 0x2c, 0xcc, 0x34, 0xd8,
                                                          0x24, 0x19, 0x9e, 0x1d, 0x68, 0x54, 0x0e, 0xb2, 0x7e, 0x98, 0xc3,
                                                          0xfd, 0x9f, 0xf0, 0x9a, 0xbb, 0xa2, 0xc4, 0xe0, 0x8b, 0x3f, 0x06,
                                                          0xd7, 0x93, 0xc4, 0x4d, 0x3e, 0xf6, 0x22, 0xbb, 0x2a, 0x57, 0x17,
                                                          0x69, 0xeb, 0x20, 0x4c, 0xae, 0x46, 0xcd, 0x15, 0x12, 0xf1, 0x55,
                                                          0x01, 0x4a, 0x70, 0x7e, 0xf2, 0x30, 0x43, 0x7c, 0x5e, 0x18 };
static const uint8_t chiptest_0fd26a42aa63_scalar_66[]    = { 0x7c, 0xe6, 0xe2, 0x6a, 0x2a, 0x0e, 0x0c, 0x5b, 0x9e, 0x19, 0x59,
                                                           0x55, 0x1e, 0x73, 0x7f, 0x38, 0xf6, 0x10, 0xdd, 0x79, 0xd2, 0x75,
                                                           0xcd, 0xf5, 0x6d, 0x3d, 0x7e, 0xd4, 0x65, 0x4c, 0x74, 0xbb };
static const uint8_t chiptest_0fd26a42aa63_out_point_67[] = { 0x04, 0xcd, 0x05, 0x21, 0xb5, 0xc9, 0x8b, 0x52, 0x8a, 0xb6, 0x87,
                                                              0x35, 0x5a, 0x21, 0xa4, 0xe9, 0x5b, 0xa0, 0x17, 0xf9, 0x6b, 0x46,
                                                              0x77, 0x8d, 0x6f, 0x89, 0xaf, 0xb7, 0xa4, 0x8f, 0x9f, 0xf6, 0x37,
                                                              0x0a, 0x79, 0x15, 0x98, 0x3c, 0x94, 0x81, 0xb9, 0x10, 0x61, 0x30,
                                                              0x82, 0x24, 0xd7, 0x1f, 0x97, 0xf8, 0x72, 0x32, 0xd0, 0x53, 0xad,
                                                              0xa7, 0x6c, 0xa0, 0x56, 0x05, 0x65, 0x73, 0xe8, 0x53, 0x68 };
static const struct spake2p_point_mul_tv chiptest_0fd26a42aa63_test_vector_68 = {
    .point         = chiptest_0fd26a42aa63_point_65,
    .point_len     = 65,
    .scalar        = chiptest_0fd26a42aa63_scalar_66,
    .scalar_len    = 32,
    .out_point     = chiptest_0fd26a42aa63_out_point_67,
    .out_point_len = 65,
};
static const uint8_t chiptest_0fd26a42aa63_point_69[]     = { 0x04, 0xef, 0xaa, 0xe4, 0x27, 0xe3, 0x17, 0xc7, 0xc3, 0x2b, 0x8a,
                                                          0xdc, 0x31, 0x5e, 0xdc, 0x7c, 0xa1, 0xa5, 0x79, 0x8b, 0xef, 0x0d,
                                                          0xf9, 0x28, 0xdb, 0x3a, 0x02, 0xcf, 0x45, 0x75, 0x48, 0x4a, 0xd9,
                                                          0x4c, 0xf3, 0xee, 0x79, 0x38, 0x72, 0xfb, 0x6a, 0xfd, 0xe8, 0x80,
                                                          0x73, 0x24, 0xa5, 0x19, 0x92, 0x7b, 0x6e, 0xe5, 0x9c, 0xcc, 0x4c,
                                                          0xd3, 0x5a, 0x4b, 0x41, 0x80, 0x52, 0xcf, 0xb7, 0x64, 0x32 };
static const uint8_t chiptest_0fd26a42aa63_scalar_70[]    = { 0xae, 0x96, 0x62, 0x8a, 0xae, 0xe5, 0xb5, 0x46, 0x95, 0xae, 0xcf,
                                                           0xa0, 0x79, 0x1b, 0xf9, 0x36, 0x0d, 0xd5, 0xbc, 0x34, 0x01, 0x78,
                                                           0xe8, 0x14, 0x6a, 0xe0, 0xce, 0xde, 0xe6, 0x5b, 0x31, 0xa0 };
static const uint8_t chiptest_0fd26a42aa63_out_point_71[] = { 0x04, 0x2c, 0x31, 0x70, 0x0b, 0x5b, 0x71, 0x20, 0x4c, 0xcc, 0x73,
                                                              0xe6, 0x14, 0xb7, 0x6f, 0xef, 0x7c, 0xc7, 0x12, 0x88, 0xcc, 0x50,
                                                              0x36, 0x1f, 0xb6, 0x47, 0xca, 0x97, 0xc9, 0x37, 0x20, 0x5e, 0x11,
                                                              0x59, 0xc5, 0x10, 0x72, 0x22, 0xa7, 0xce, 0x7d, 0x93, 0x7a, 0x0e,
                                                              0x7c, 0xa3, 0x9b, 0x97, 0x14, 0x27, 0xf8, 0x0e, 0x3a, 0xbd, 0x51,
                                                              0x00, 0xda, 0x00, 0x17, 0x8d, 0xf4, 0xe5, 0xc1, 0x60, 0xf4 };
static const struct spake2p_point_mul_tv chiptest_0fd26a42aa63_test_vector_72 = {
    .point         = chiptest_0fd26a42aa63_point_69,
    .point_len     = 65,
    .scalar        = chiptest_0fd26a42aa63_scalar_70,
    .scalar_len    = 32,
    .out_point     = chiptest_0fd26a42aa63_out_point_71,
    .out_point_len = 65,
};
static const uint8_t chiptest_0fd26a42aa63_point_73[]     = { 0x04, 0x52, 0x90, 0x95, 0x7e, 0x15, 0x3d, 0x11, 0x56, 0x9b, 0x92,
                                                          0xb8, 0x8c, 0x96, 0x32, 0x37, 0x66, 0xc7, 0x2e, 0x63, 0x88, 0x88,
                                                          0x1e, 0x1f, 0x6e, 0x98, 0x2d, 0x9b, 0x1c, 0xa0, 0x1e, 0x6b, 0x4c,
                                                          0xdc, 0x4f, 0xe6, 0xf2, 0x6e, 0x9c, 0x26, 0xfd, 0x5f, 0x5e, 0x3f,
                                                          0x7f, 0xc1, 0xc1, 0xc0, 0x07, 0xb2, 0x24, 0xb1, 0xb3, 0xe3, 0x24,
                                                          0x4c, 0xfc, 0x8c, 0x5b, 0x65, 0xa6, 0x7a, 0xbd, 0x1e, 0x37 };
static const uint8_t chiptest_0fd26a42aa63_scalar_74[]    = { 0x2f, 0xc1, 0xb4, 0xd5, 0x5b, 0xc9, 0x81, 0x01, 0xda, 0x97, 0x10,
                                                           0x23, 0xf5, 0x9b, 0xb0, 0x3d, 0xb3, 0x63, 0xc1, 0x0b, 0xa8, 0x84,
                                                           0x5a, 0x17, 0x4d, 0xb5, 0xa3, 0xe8, 0x6c, 0x22, 0x47, 0x7f };
static const uint8_t chiptest_0fd26a42aa63_out_point_75[] = { 0x04, 0x94, 0xdf, 0x0b, 0x77, 0x2f, 0x24, 0x00, 0x63, 0x8f, 0x46,
                                                              0x7c, 0x39, 0xdd, 0xa5, 0x42, 0xe0, 0xc4, 0x0e, 0x7e, 0x7d, 0xc7,
                                                              0x63, 0x4a, 0x2d, 0x6b, 0x31, 0xfd, 0x08, 0x39, 0x6a, 0xe5, 0xd4,
                                                              0x75, 0x73, 0x29, 0x5d, 0xb9, 0x7f, 0x53, 0xd6, 0x78, 0xc8, 0x5c,
                                                              0x51, 0x71, 0x4f, 0xba, 0x91, 0xf3, 0xf6, 0xbf, 0xa6, 0xd3, 0x82,
                                                              0x51, 0xcd, 0x42, 0xd2, 0x62, 0xe6, 0x3c, 0xd4, 0xca, 0x3b };
static const struct spake2p_point_mul_tv chiptest_0fd26a42aa63_test_vector_76 = {
    .point         = chiptest_0fd26a42aa63_point_73,
    .point_len     = 65,
    .scalar        = chiptest_0fd26a42aa63_scalar_74,
    .scalar_len    = 32,
    .out_point     = chiptest_0fd26a42aa63_out_point_75,
    .out_point_len = 65,
};
static const uint8_t chiptest_0fd26a42aa63_point_77[]     = { 0x04, 0x88, 0x44, 0x05, 0x46, 0xf0, 0xc1, 0xf8, 0x10, 0xeb, 0x18,
                                                          0xfc, 0xcb, 0xce, 0x50, 0x8d, 0xc8, 0x63, 0xf3, 0x9f, 0x27, 0xe3,
                                                          0x4c, 0x15, 0x07, 0x17, 0xf6, 0x56, 0xba, 0x8d, 0x8a, 0xea, 0x78,
                                                          0x4e, 0x7b, 0xcb, 0x3d, 0xd3, 0xf9, 0xb5, 0xff, 0xbd, 0xba, 0x22,
                                                          0xb9, 0xaa, 0x3d, 0xa6, 0xbf, 0x3d, 0x30, 0xe1, 0x85, 0x4f, 0x8a,
                                                          0xdd, 0x35, 0x5e, 0x0d, 0x8a, 0x52, 0xfc, 0x58, 0x96, 0x33 };
static const uint8_t chiptest_0fd26a42aa63_scalar_78[]    = { 0x03, 0xac, 0xd3, 0xb0, 0x2e, 0x9a, 0x26, 0xa0, 0x14, 0x2b, 0x64,
                                                           0xd7, 0x1e, 0x56, 0x8f, 0x85, 0x39, 0xd6, 0x94, 0xee, 0xf1, 0x15,
                                                           0x37, 0xff, 0xd7, 0x8b, 0x23, 0x94, 0x5d, 0x33, 0x44, 0x8f };
static const uint8_t chiptest_0fd26a42aa63_out_point_79[] = { 0x04, 0xab, 0x9b, 0xb7, 0x08, 0x43, 0x9d, 0xec, 0xf1, 0xb1, 0x3b,
                                                              0x3f, 0xf4, 0x4a, 0xa9, 0x94, 0x40, 0x41, 0x3c, 0x57, 0x2c, 0x90,
                                                              0x5f, 0x33, 0x6a, 0x21, 0xe2, 0xc7, 0x20, 0xbc, 0x50, 0x74, 0x26,
                                                              0x48, 0x8f, 0xf3, 0x65, 0xbc, 0xdd, 0xd7, 0x2e, 0x73, 0x41, 0x0f,
                                                              0xfc, 0x07, 0xa8, 0x3d, 0xa2, 0x2b, 0x47, 0xdd, 0xfb, 0xc1, 0x53,
                                                              0xbf, 0xdf, 0x80, 0x86, 0x6f, 0xe0, 0x7a, 0x9e, 0x24, 0x09 };
static const struct spake2p_point_mul_tv chiptest_0fd26a42aa63_test_vector_80 = {
    .point         = chiptest_0fd26a42aa63_point_77,
    .point_len     = 65,
    .scalar        = chiptest_0fd26a42aa63_scalar_78,
    .scalar_len    = 32,
    .out_point     = chiptest_0fd26a42aa63_out_point_79,
    .out_point_len = 65,
};
static const struct spake2p_point_mul_tv * point_mul_tvs[] = {
    &chiptest_0fd26a42aa63_test_vector_4,  &chiptest_0fd26a42aa63_test_vector_8,  &chiptest_0fd26a42aa63_test_vector_12,
    &chiptest_0fd26a42aa63_test_vector_16, &chiptest_0fd26a42aa63_test_vector_20, &chiptest_0fd26a42aa63_test_vector_24,
    &chiptest_0fd26a42aa63_test_vector_28, &chiptest_0fd26a42aa63_test_vector_32, &chiptest_0fd26a42aa63_test_vector_36,
    &chiptest_0fd26a42aa63_test_vector_40, &chiptest_0fd26a42aa63_test_vector_44, &chiptest_0fd26a42aa63_test_vector_48,
    &chiptest_0fd26a42aa63_test_vector_52, &chiptest_0fd26a42aa63_test_vector_56, &chiptest_0fd26a42aa63_test_vector_60,
    &chiptest_0fd26a42aa63_test_vector_64, &chiptest_0fd26a42aa63_test_vector_68, &chiptest_0fd26a42aa63_test_vector_72,
    &chiptest_0fd26a42aa63_test_vector_76, &chiptest_0fd26a42aa63_test_vector_80
};

} // namespace Crypto
} // namespace chip
