/*
 *
 *    Copyright (c) 2020 Project CHIP Authors
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

/**
 * @file - This file contains Spake2+ RFC test vectors.
 */

#pragma once

namespace chip {
namespace Crypto {

struct spake2p_rfc_tv
{
    const uint8_t * context;
    size_t context_len;
    const uint8_t * prover_identity;
    size_t prover_identity_len;
    const uint8_t * verifier_identity;
    size_t verifier_identity_len;
    const uint8_t * w0;
    size_t w0_len;
    const uint8_t * w1;
    size_t w1_len;
    const uint8_t * L;
    size_t L_len;
    const uint8_t * x;
    size_t x_len;
    const uint8_t * X;
    size_t X_len;
    const uint8_t * y;
    size_t y_len;
    const uint8_t * Y;
    size_t Y_len;
    const uint8_t * Z;
    size_t Z_len;
    const uint8_t * V;
    size_t V_len;
    const uint8_t * Ka;
    size_t Ka_len;
    const uint8_t * Ke;
    size_t Ke_len;
    const uint8_t * KcA;
    size_t KcA_len;
    const uint8_t * KcB;
    size_t KcB_len;
    const uint8_t * MAC_KcA;
    size_t MAC_KcA_len;
    const uint8_t * MAC_KcB;
    size_t MAC_KcB_len;
};

static const uint8_t chiptest_e4836c3b50dd_w0_1[]  = { 0xe6, 0x88, 0x7c, 0xf9, 0xbd, 0xfb, 0x75, 0x79, 0xc6, 0x9b, 0xf4,
                                                      0x79, 0x28, 0xa8, 0x45, 0x14, 0xb5, 0xe3, 0x55, 0xac, 0x03, 0x48,
                                                      0x63, 0xf7, 0xff, 0xaf, 0x43, 0x90, 0xe6, 0x7d, 0x79, 0x8c };
static const uint8_t chiptest_e4836c3b50dd_w1_2[]  = { 0x24, 0xb5, 0xae, 0x4a, 0xbd, 0xa8, 0x68, 0xec, 0x93, 0x36, 0xff,
                                                      0xc3, 0xb7, 0x8e, 0xe3, 0x1c, 0x57, 0x55, 0xbe, 0xf1, 0x75, 0x92,
                                                      0x27, 0xef, 0x53, 0x72, 0xca, 0x13, 0x9b, 0x94, 0xe5, 0x12 };
static const uint8_t chiptest_e4836c3b50dd_L_3[]   = { 0x04, 0x95, 0x64, 0x5c, 0xfb, 0x74, 0xdf, 0x6e, 0x58, 0xf9, 0x74, 0x8b, 0xb8,
                                                     0x3a, 0x86, 0x62, 0x0b, 0xab, 0x7c, 0x82, 0xe1, 0x07, 0xf5, 0x7d, 0x68, 0x70,
                                                     0xda, 0x8c, 0xbc, 0xb2, 0xff, 0x9f, 0x70, 0x63, 0xa1, 0x4b, 0x64, 0x02, 0xc6,
                                                     0x2f, 0x99, 0xaf, 0xcb, 0x97, 0x06, 0xa4, 0xd1, 0xa1, 0x43, 0x27, 0x32, 0x59,
                                                     0xfe, 0x76, 0xf1, 0xc6, 0x05, 0xa3, 0x63, 0x97, 0x45, 0xa9, 0x21, 0x54, 0xb9 };
static const uint8_t chiptest_e4836c3b50dd_x_4[]   = { 0x8b, 0x0f, 0x3f, 0x38, 0x39, 0x05, 0xcf, 0x3a, 0x3b, 0xb9, 0x55,
                                                     0xef, 0x8f, 0xb6, 0x2e, 0x24, 0x84, 0x9d, 0xd3, 0x49, 0xa0, 0x5c,
                                                     0xa7, 0x9a, 0xaf, 0xb1, 0x80, 0x41, 0xd3, 0x0c, 0xbd, 0xb6 };
static const uint8_t chiptest_e4836c3b50dd_X_5[]   = { 0x04, 0xaf, 0x09, 0x98, 0x7a, 0x59, 0x3d, 0x3b, 0xac, 0x86, 0x94, 0xb1, 0x23,
                                                     0x83, 0x94, 0x22, 0xc3, 0xcc, 0x87, 0xe3, 0x7d, 0x6b, 0x41, 0xc1, 0xd6, 0x30,
                                                     0xf0, 0x00, 0xdd, 0x64, 0x98, 0x0e, 0x53, 0x7a, 0xe7, 0x04, 0xbc, 0xed, 0xe0,
                                                     0x4e, 0xa3, 0xbe, 0xc9, 0xb7, 0x47, 0x5b, 0x32, 0xfa, 0x2c, 0xa3, 0xb6, 0x84,
                                                     0xbe, 0x14, 0xd1, 0x16, 0x45, 0xe3, 0x8e, 0xa6, 0x60, 0x9e, 0xb3, 0x9e, 0x7e };
static const uint8_t chiptest_e4836c3b50dd_y_6[]   = { 0x2e, 0x08, 0x95, 0xb0, 0xe7, 0x63, 0xd6, 0xd5, 0xa9, 0x56, 0x44,
                                                     0x33, 0xe6, 0x4a, 0xc3, 0xca, 0xc7, 0x4f, 0xf8, 0x97, 0xf6, 0xc3,
                                                     0x44, 0x52, 0x47, 0xba, 0x1b, 0xab, 0x40, 0x08, 0x2a, 0x91 };
static const uint8_t chiptest_e4836c3b50dd_Y_7[]   = { 0x04, 0x41, 0x75, 0x92, 0x62, 0x0a, 0xeb, 0xf9, 0xfd, 0x20, 0x36, 0x16, 0xbb,
                                                     0xb9, 0xf1, 0x21, 0xb7, 0x30, 0xc2, 0x58, 0xb2, 0x86, 0xf8, 0x90, 0xc5, 0xf1,
                                                     0x9f, 0xea, 0x83, 0x3a, 0x9c, 0x90, 0x0c, 0xbe, 0x90, 0x57, 0xbc, 0x54, 0x9a,
                                                     0x3e, 0x19, 0x97, 0x5b, 0xe9, 0x92, 0x7f, 0x0e, 0x76, 0x14, 0xf0, 0x8d, 0x1f,
                                                     0x0a, 0x10, 0x8e, 0xed, 0xe5, 0xfd, 0x7e, 0xb5, 0x62, 0x45, 0x84, 0xa4, 0xf4 };
static const uint8_t chiptest_e4836c3b50dd_Z_8[]   = { 0x04, 0x71, 0xa3, 0x52, 0x82, 0xd2, 0x02, 0x6f, 0x36, 0xbf, 0x3c, 0xeb, 0x38,
                                                     0xfc, 0xf8, 0x7e, 0x31, 0x12, 0xa4, 0x45, 0x2f, 0x46, 0xe9, 0xf7, 0xb4, 0x7f,
                                                     0xd7, 0x69, 0xcf, 0xb5, 0x70, 0x14, 0x5b, 0x62, 0x58, 0x9c, 0x76, 0xb7, 0xaa,
                                                     0x1e, 0xb6, 0x08, 0x0a, 0x83, 0x2e, 0x53, 0x32, 0xc3, 0x68, 0x98, 0x42, 0x69,
                                                     0x12, 0xe2, 0x9c, 0x40, 0xef, 0x9e, 0x9c, 0x74, 0x2e, 0xee, 0x82, 0xbf, 0x30 };
static const uint8_t chiptest_e4836c3b50dd_V_9[]   = { 0x04, 0x67, 0x18, 0x98, 0x1b, 0xf1, 0x5b, 0xc4, 0xdb, 0x53, 0x8f, 0xc1, 0xf1,
                                                     0xc1, 0xd0, 0x58, 0xcb, 0x0e, 0xec, 0xec, 0xf1, 0xdb, 0xe1, 0xb1, 0xea, 0x08,
                                                     0xa4, 0xe2, 0x52, 0x75, 0xd3, 0x82, 0xe8, 0x2b, 0x34, 0x8c, 0x81, 0x31, 0xd8,
                                                     0xed, 0x66, 0x9d, 0x16, 0x9c, 0x2e, 0x03, 0xa8, 0x58, 0xdb, 0x7c, 0xf6, 0xca,
                                                     0x28, 0x53, 0xa4, 0x07, 0x12, 0x51, 0xa3, 0x9f, 0xbe, 0x8c, 0xfc, 0x39, 0xbc };
static const uint8_t chiptest_e4836c3b50dd_Ka_10[] = { 0xf9, 0xca, 0xb9, 0xad, 0xcc, 0x0e, 0xd8, 0xe5,
                                                       0xa4, 0xdb, 0x11, 0xa8, 0x50, 0x59, 0x14, 0xb2 };
static const uint8_t chiptest_e4836c3b50dd_Ke_11[] = { 0x80, 0x1d, 0xb2, 0x97, 0x65, 0x48, 0x16, 0xeb,
                                                       0x4f, 0x02, 0x86, 0x81, 0x29, 0xb9, 0xdc, 0x89 };
static const uint8_t chiptest_e4836c3b50dd_KcA_12[]     = { 0x0d, 0x24, 0x8d, 0x7d, 0x19, 0x23, 0x4f, 0x14,
                                                        0x86, 0xb2, 0xef, 0xba, 0x51, 0x79, 0xc5, 0x2d };
static const uint8_t chiptest_e4836c3b50dd_KcB_13[]     = { 0x55, 0x62, 0x91, 0xdf, 0x26, 0xd7, 0x05, 0xa2,
                                                        0xca, 0xed, 0xd6, 0x47, 0x4d, 0xd0, 0x07, 0x9b };
static const uint8_t chiptest_e4836c3b50dd_MAC_KcA_14[] = { 0xd4, 0x37, 0x6f, 0x2d, 0xa9, 0xc7, 0x22, 0x26, 0xdd, 0x15, 0x1b,
                                                            0x77, 0xc2, 0x91, 0x90, 0x71, 0x15, 0x5f, 0xc2, 0x2a, 0x20, 0x68,
                                                            0xd9, 0x0b, 0x5f, 0xaa, 0x6c, 0x78, 0xc1, 0x1e, 0x77, 0xdd };
static const uint8_t chiptest_e4836c3b50dd_MAC_KcB_15[] = { 0x06, 0x60, 0xa6, 0x80, 0x66, 0x3e, 0x8c, 0x56, 0x95, 0x95, 0x6f,
                                                            0xb2, 0x2d, 0xff, 0x29, 0x8b, 0x1d, 0x07, 0xa5, 0x26, 0xcf, 0x3c,
                                                            0xc5, 0x91, 0xad, 0xfe, 0xcd, 0x1f, 0x6e, 0xf6, 0xe0, 0x2e };
static const struct spake2p_rfc_tv chiptest_e4836c3b50dd_test_vector_16 = {
    .context               = reinterpret_cast<const uint8_t *>("SPAKE2+-P256-SHA256-HKDF draft-01"),
    .context_len           = 33,
    .prover_identity       = reinterpret_cast<const uint8_t *>("client"),
    .prover_identity_len   = 6,
    .verifier_identity     = reinterpret_cast<const uint8_t *>("server"),
    .verifier_identity_len = 6,
    .w0                    = chiptest_e4836c3b50dd_w0_1,
    .w0_len                = 32,
    .w1                    = chiptest_e4836c3b50dd_w1_2,
    .w1_len                = 32,
    .L                     = chiptest_e4836c3b50dd_L_3,
    .L_len                 = 65,
    .x                     = chiptest_e4836c3b50dd_x_4,
    .x_len                 = 32,
    .X                     = chiptest_e4836c3b50dd_X_5,
    .X_len                 = 65,
    .y                     = chiptest_e4836c3b50dd_y_6,
    .y_len                 = 32,
    .Y                     = chiptest_e4836c3b50dd_Y_7,
    .Y_len                 = 65,
    .Z                     = chiptest_e4836c3b50dd_Z_8,
    .Z_len                 = 65,
    .V                     = chiptest_e4836c3b50dd_V_9,
    .V_len                 = 65,
    .Ka                    = chiptest_e4836c3b50dd_Ka_10,
    .Ka_len                = 16,
    .Ke                    = chiptest_e4836c3b50dd_Ke_11,
    .Ke_len                = 16,
    .KcA                   = chiptest_e4836c3b50dd_KcA_12,
    .KcA_len               = 16,
    .KcB                   = chiptest_e4836c3b50dd_KcB_13,
    .KcB_len               = 16,
    .MAC_KcA               = chiptest_e4836c3b50dd_MAC_KcA_14,
    .MAC_KcA_len           = 32,
    .MAC_KcB               = chiptest_e4836c3b50dd_MAC_KcB_15,
    .MAC_KcB_len           = 32
};
static const uint8_t chiptest_e4836c3b50dd_w0_17[] = { 0xe6, 0x88, 0x7c, 0xf9, 0xbd, 0xfb, 0x75, 0x79, 0xc6, 0x9b, 0xf4,
                                                       0x79, 0x28, 0xa8, 0x45, 0x14, 0xb5, 0xe3, 0x55, 0xac, 0x03, 0x48,
                                                       0x63, 0xf7, 0xff, 0xaf, 0x43, 0x90, 0xe6, 0x7d, 0x79, 0x8c };
static const uint8_t chiptest_e4836c3b50dd_w1_18[] = { 0x24, 0xb5, 0xae, 0x4a, 0xbd, 0xa8, 0x68, 0xec, 0x93, 0x36, 0xff,
                                                       0xc3, 0xb7, 0x8e, 0xe3, 0x1c, 0x57, 0x55, 0xbe, 0xf1, 0x75, 0x92,
                                                       0x27, 0xef, 0x53, 0x72, 0xca, 0x13, 0x9b, 0x94, 0xe5, 0x12 };
static const uint8_t chiptest_e4836c3b50dd_L_19[]  = {
    0x04, 0x95, 0x64, 0x5c, 0xfb, 0x74, 0xdf, 0x6e, 0x58, 0xf9, 0x74, 0x8b, 0xb8, 0x3a, 0x86, 0x62, 0x0b,
    0xab, 0x7c, 0x82, 0xe1, 0x07, 0xf5, 0x7d, 0x68, 0x70, 0xda, 0x8c, 0xbc, 0xb2, 0xff, 0x9f, 0x70, 0x63,
    0xa1, 0x4b, 0x64, 0x02, 0xc6, 0x2f, 0x99, 0xaf, 0xcb, 0x97, 0x06, 0xa4, 0xd1, 0xa1, 0x43, 0x27, 0x32,
    0x59, 0xfe, 0x76, 0xf1, 0xc6, 0x05, 0xa3, 0x63, 0x97, 0x45, 0xa9, 0x21, 0x54, 0xb9
};
static const uint8_t chiptest_e4836c3b50dd_x_20[] = { 0xec, 0x82, 0xd9, 0x25, 0x83, 0x37, 0xf6, 0x12, 0x39, 0xc9, 0xcd,
                                                      0x68, 0xe8, 0xe5, 0x32, 0xa3, 0xa6, 0xb8, 0x3d, 0x12, 0xd2, 0xb1,
                                                      0xca, 0x5d, 0x54, 0x3f, 0x44, 0xde, 0xf1, 0x7d, 0xfb, 0x8d };
static const uint8_t chiptest_e4836c3b50dd_X_21[] = {
    0x04, 0x23, 0x07, 0x79, 0x96, 0x08, 0x24, 0x07, 0x6d, 0x36, 0x66, 0xa7, 0x41, 0x8e, 0x4d, 0x43, 0x3e,
    0x2f, 0xa1, 0x5b, 0x06, 0x17, 0x6e, 0xab, 0xdd, 0x57, 0x2f, 0x43, 0xa3, 0x2e, 0xcc, 0x79, 0xa1, 0x92,
    0xb2, 0x43, 0xd2, 0x62, 0x43, 0x10, 0xa7, 0x35, 0x62, 0x73, 0xb8, 0x6e, 0x5f, 0xd9, 0xbd, 0x62, 0x7d,
    0x3a, 0xde, 0x76, 0x2b, 0xae, 0xff, 0x1a, 0x32, 0x0d, 0x4a, 0xd7, 0xa4, 0xe4, 0x7f
};
static const uint8_t chiptest_e4836c3b50dd_y_22[] = { 0xea, 0xc3, 0xf7, 0xde, 0x4b, 0x19, 0x8d, 0x5f, 0xe2, 0x5c, 0x44,
                                                      0x3c, 0x0c, 0xd4, 0x96, 0x38, 0x07, 0xad, 0xd7, 0x67, 0x81, 0x5d,
                                                      0xd0, 0x2a, 0x6f, 0x01, 0x33, 0xb4, 0xbc, 0x2c, 0x9e, 0xb0 };
static const uint8_t chiptest_e4836c3b50dd_Y_23[] = {
    0x04, 0x45, 0x58, 0x64, 0x2e, 0x71, 0xb6, 0x16, 0xb2, 0x48, 0xc9, 0x58, 0x3b, 0xd6, 0xd7, 0xaa, 0x1b,
    0x39, 0x52, 0xc6, 0xdf, 0x6a, 0x9f, 0x74, 0x92, 0xa0, 0x60, 0x35, 0xca, 0x5d, 0x92, 0x52, 0x2d, 0x84,
    0x44, 0x3d, 0xe7, 0xaa, 0x20, 0xa5, 0x93, 0x80, 0xfa, 0x4d, 0xe6, 0xb7, 0x43, 0x8d, 0x92, 0x5d, 0xbf,
    0xb7, 0xf1, 0xcf, 0xe6, 0x0d, 0x79, 0xac, 0xf9, 0x61, 0xee, 0x33, 0x98, 0x8c, 0x7d
};
static const uint8_t chiptest_e4836c3b50dd_Z_24[] = {
    0x04, 0xb4, 0xe8, 0x77, 0x0f, 0x19, 0xf5, 0x8d, 0xdf, 0x83, 0xf9, 0x22, 0x0c, 0x3a, 0x93, 0x05, 0x79,
    0x26, 0x65, 0xe0, 0xc6, 0x09, 0x89, 0xe6, 0xee, 0x9d, 0x7f, 0xa4, 0x49, 0xc7, 0x75, 0xd6, 0x39, 0x5f,
    0x6f, 0x25, 0xf3, 0x07, 0xe3, 0x90, 0x3a, 0xc0, 0x45, 0xa0, 0x13, 0xfb, 0xb5, 0xa6, 0x76, 0xe8, 0x72,
    0xa6, 0xab, 0xfc, 0xf4, 0xd7, 0xbb, 0x5a, 0xac, 0x69, 0xef, 0xd6, 0x14, 0x0e, 0xed
};
static const uint8_t chiptest_e4836c3b50dd_V_25[] = {
    0x04, 0x14, 0x1d, 0xb8, 0x3b, 0xc7, 0xd9, 0x6f, 0x41, 0xb6, 0x36, 0x62, 0x2e, 0x7a, 0x5c, 0x55, 0x2a,
    0xd8, 0x32, 0x11, 0xff, 0x55, 0x31, 0x9a, 0xc2, 0x5e, 0xd0, 0xa0, 0x9f, 0x08, 0x18, 0xbd, 0x94, 0x2e,
    0x81, 0x50, 0x31, 0x9b, 0xfb, 0xfa, 0x68, 0x61, 0x83, 0x80, 0x6d, 0xc6, 0x19, 0x11, 0x18, 0x3f, 0x6a,
    0x0f, 0x59, 0x56, 0x15, 0x60, 0x23, 0xd9, 0x6e, 0x0f, 0x93, 0xd2, 0x75, 0xbf, 0x50
};
static const uint8_t chiptest_e4836c3b50dd_Ka_26[]      = { 0xe2, 0xcb, 0xee, 0x3a, 0xe1, 0x9a, 0x4d, 0xbe,
                                                       0x9f, 0x14, 0x6b, 0xe6, 0xbe, 0xe9, 0xbf, 0xa1 };
static const uint8_t chiptest_e4836c3b50dd_Ke_27[]      = { 0x69, 0x89, 0xd8, 0xf9, 0x17, 0x7e, 0xf7, 0xdf,
                                                       0x67, 0xda, 0x43, 0x79, 0x87, 0xf0, 0x72, 0x55 };
static const uint8_t chiptest_e4836c3b50dd_KcA_28[]     = { 0x2f, 0x9e, 0x0b, 0xb6, 0x69, 0xd2, 0xc2, 0x26,
                                                        0x45, 0xbc, 0xe3, 0x4d, 0xa0, 0x4a, 0xc1, 0x6a };
static const uint8_t chiptest_e4836c3b50dd_KcB_29[]     = { 0xeb, 0x7a, 0x35, 0x16, 0x87, 0x59, 0xdd, 0x8e,
                                                        0x9c, 0xe4, 0x4e, 0x4d, 0xc5, 0x12, 0x77, 0xce };
static const uint8_t chiptest_e4836c3b50dd_MAC_KcA_30[] = { 0xe1, 0xb9, 0x25, 0x88, 0x07, 0xba, 0x47, 0x50, 0xda, 0xe1, 0xd7,
                                                            0xf3, 0xc3, 0xc2, 0x94, 0xf1, 0x3d, 0xc4, 0xfa, 0x60, 0xcd, 0xe3,
                                                            0x46, 0xd5, 0xde, 0x7d, 0x20, 0x0e, 0x2f, 0x8f, 0xd3, 0xfc };
static const uint8_t chiptest_e4836c3b50dd_MAC_KcB_31[] = { 0xb9, 0xc3, 0x9d, 0xfa, 0x49, 0xc4, 0x77, 0x57, 0xde, 0x77, 0x8d,
                                                            0x9b, 0xed, 0xea, 0xca, 0x24, 0x48, 0xb9, 0x05, 0xbe, 0x19, 0xa4,
                                                            0x3b, 0x94, 0xee, 0x24, 0xb7, 0x70, 0x20, 0x81, 0x35, 0xe3 };
static const struct spake2p_rfc_tv chiptest_e4836c3b50dd_test_vector_32 = {
    .context               = reinterpret_cast<const uint8_t *>("SPAKE2+-P256-SHA256-HKDF draft-01"),
    .context_len           = 33,
    .prover_identity       = reinterpret_cast<const uint8_t *>("client"),
    .prover_identity_len   = 6,
    .verifier_identity     = reinterpret_cast<const uint8_t *>(""),
    .verifier_identity_len = 0,
    .w0                    = chiptest_e4836c3b50dd_w0_17,
    .w0_len                = 32,
    .w1                    = chiptest_e4836c3b50dd_w1_18,
    .w1_len                = 32,
    .L                     = chiptest_e4836c3b50dd_L_19,
    .L_len                 = 65,
    .x                     = chiptest_e4836c3b50dd_x_20,
    .x_len                 = 32,
    .X                     = chiptest_e4836c3b50dd_X_21,
    .X_len                 = 65,
    .y                     = chiptest_e4836c3b50dd_y_22,
    .y_len                 = 32,
    .Y                     = chiptest_e4836c3b50dd_Y_23,
    .Y_len                 = 65,
    .Z                     = chiptest_e4836c3b50dd_Z_24,
    .Z_len                 = 65,
    .V                     = chiptest_e4836c3b50dd_V_25,
    .V_len                 = 65,
    .Ka                    = chiptest_e4836c3b50dd_Ka_26,
    .Ka_len                = 16,
    .Ke                    = chiptest_e4836c3b50dd_Ke_27,
    .Ke_len                = 16,
    .KcA                   = chiptest_e4836c3b50dd_KcA_28,
    .KcA_len               = 16,
    .KcB                   = chiptest_e4836c3b50dd_KcB_29,
    .KcB_len               = 16,
    .MAC_KcA               = chiptest_e4836c3b50dd_MAC_KcA_30,
    .MAC_KcA_len           = 32,
    .MAC_KcB               = chiptest_e4836c3b50dd_MAC_KcB_31,
    .MAC_KcB_len           = 32
};
static const uint8_t chiptest_e4836c3b50dd_w0_33[] = { 0xe6, 0x88, 0x7c, 0xf9, 0xbd, 0xfb, 0x75, 0x79, 0xc6, 0x9b, 0xf4,
                                                       0x79, 0x28, 0xa8, 0x45, 0x14, 0xb5, 0xe3, 0x55, 0xac, 0x03, 0x48,
                                                       0x63, 0xf7, 0xff, 0xaf, 0x43, 0x90, 0xe6, 0x7d, 0x79, 0x8c };
static const uint8_t chiptest_e4836c3b50dd_w1_34[] = { 0x24, 0xb5, 0xae, 0x4a, 0xbd, 0xa8, 0x68, 0xec, 0x93, 0x36, 0xff,
                                                       0xc3, 0xb7, 0x8e, 0xe3, 0x1c, 0x57, 0x55, 0xbe, 0xf1, 0x75, 0x92,
                                                       0x27, 0xef, 0x53, 0x72, 0xca, 0x13, 0x9b, 0x94, 0xe5, 0x12 };
static const uint8_t chiptest_e4836c3b50dd_L_35[]  = {
    0x04, 0x95, 0x64, 0x5c, 0xfb, 0x74, 0xdf, 0x6e, 0x58, 0xf9, 0x74, 0x8b, 0xb8, 0x3a, 0x86, 0x62, 0x0b,
    0xab, 0x7c, 0x82, 0xe1, 0x07, 0xf5, 0x7d, 0x68, 0x70, 0xda, 0x8c, 0xbc, 0xb2, 0xff, 0x9f, 0x70, 0x63,
    0xa1, 0x4b, 0x64, 0x02, 0xc6, 0x2f, 0x99, 0xaf, 0xcb, 0x97, 0x06, 0xa4, 0xd1, 0xa1, 0x43, 0x27, 0x32,
    0x59, 0xfe, 0x76, 0xf1, 0xc6, 0x05, 0xa3, 0x63, 0x97, 0x45, 0xa9, 0x21, 0x54, 0xb9
};
static const uint8_t chiptest_e4836c3b50dd_x_36[] = { 0xba, 0x0f, 0x0f, 0x5b, 0x78, 0xef, 0x23, 0xfd, 0x07, 0x86, 0x8e,
                                                      0x46, 0xae, 0xca, 0x63, 0xb5, 0x1f, 0xda, 0x51, 0x9a, 0x34, 0x20,
                                                      0x50, 0x1a, 0xcb, 0xe2, 0x3d, 0x53, 0xc2, 0x91, 0x87, 0x48 };
static const uint8_t chiptest_e4836c3b50dd_X_37[] = {
    0x04, 0xc1, 0x4d, 0x28, 0xf4, 0x37, 0x0f, 0xea, 0x20, 0x74, 0x51, 0x06, 0xce, 0xa5, 0x8b, 0xcf, 0xb6,
    0x0f, 0x29, 0x49, 0xfa, 0x4e, 0x13, 0x1b, 0x9a, 0xff, 0x5e, 0xa1, 0x3f, 0xd5, 0xaa, 0x79, 0xd5, 0x07,
    0xae, 0x1d, 0x22, 0x9e, 0x44, 0x7e, 0x00, 0x0f, 0x15, 0xeb, 0x78, 0xa9, 0xa3, 0x2c, 0x2b, 0x88, 0x65,
    0x2e, 0x34, 0x11, 0x64, 0x20, 0x43, 0xc1, 0xb2, 0xb7, 0x99, 0x2c, 0xf2, 0xd4, 0xde
};
static const uint8_t chiptest_e4836c3b50dd_y_38[] = { 0x39, 0x39, 0x7f, 0xbe, 0x6d, 0xb4, 0x7e, 0x9f, 0xbd, 0x1a, 0x26,
                                                      0x3d, 0x79, 0xf5, 0xd0, 0xaa, 0xa4, 0x4d, 0xf2, 0x6c, 0xe7, 0x55,
                                                      0xf7, 0x8e, 0x09, 0x26, 0x44, 0xb4, 0x34, 0x53, 0x3a, 0x42 };
static const uint8_t chiptest_e4836c3b50dd_Y_39[] = {
    0x04, 0xd1, 0xbe, 0xe3, 0x12, 0x0f, 0xd8, 0x7e, 0x86, 0xfe, 0x18, 0x9c, 0xb9, 0x52, 0xdc, 0x68, 0x88,
    0x23, 0x08, 0x0e, 0x62, 0x52, 0x4d, 0xd2, 0xc0, 0x8d, 0xff, 0xe3, 0xd2, 0x2a, 0x0a, 0x89, 0x86, 0xaa,
    0x64, 0xc9, 0xfe, 0x01, 0x91, 0x03, 0x3c, 0xaf, 0xbc, 0x9b, 0xca, 0xef, 0xc8, 0xe2, 0xba, 0x8b, 0xa8,
    0x60, 0xcd, 0x12, 0x7a, 0xf9, 0xef, 0xdd, 0x7f, 0x1c, 0x3a, 0x41, 0x92, 0x0f, 0xe8
};
static const uint8_t chiptest_e4836c3b50dd_Z_40[] = {
    0x04, 0xaa, 0xc7, 0x1c, 0xf4, 0xc8, 0xdf, 0x81, 0x81, 0xb8, 0x67, 0xc9, 0xec, 0xbe, 0xe9, 0xd0, 0x96,
    0x3c, 0xaf, 0x51, 0xf1, 0x53, 0x4a, 0x82, 0x34, 0x29, 0xc2, 0x6f, 0xe5, 0x24, 0x83, 0x13, 0xff, 0xc5,
    0xc5, 0xe4, 0x4e, 0xa8, 0x16, 0x21, 0x61, 0xab, 0x6b, 0x3d, 0x73, 0xb8, 0x77, 0x04, 0xa4, 0x58, 0x89,
    0xbf, 0x63, 0x43, 0xd9, 0x6f, 0xa9, 0x6c, 0xd1, 0x64, 0x1e, 0xfa, 0x71, 0x60, 0x7c
};
static const uint8_t chiptest_e4836c3b50dd_V_41[] = {
    0x04, 0xc7, 0xc9, 0x50, 0x53, 0x65, 0xf7, 0xce, 0x57, 0x29, 0x3c, 0x92, 0xa3, 0x7f, 0x1b, 0xbd, 0xc6,
    0x8e, 0x03, 0x22, 0x90, 0x1e, 0x61, 0xed, 0xef, 0x59, 0xfe, 0xe7, 0x87, 0x6b, 0x17, 0xb0, 0x63, 0xe0,
    0xfa, 0x4a, 0x12, 0x6e, 0xae, 0x0a, 0x67, 0x1b, 0x37, 0xf1, 0x46, 0x4c, 0xf1, 0xcc, 0xad, 0x59, 0x1c,
    0x33, 0xae, 0x94, 0x4e, 0x3b, 0x1f, 0x31, 0x8d, 0x76, 0xe3, 0x6f, 0xea, 0x99, 0x66
};
static const uint8_t chiptest_e4836c3b50dd_Ka_42[]      = { 0xec, 0x8d, 0x19, 0xb8, 0x07, 0xff, 0xb1, 0xd1,
                                                       0xee, 0xa8, 0x1a, 0x93, 0xba, 0x35, 0xcd, 0xfe };
static const uint8_t chiptest_e4836c3b50dd_Ke_43[]      = { 0x2e, 0xa4, 0x0e, 0x4b, 0xad, 0xfa, 0x54, 0x52,
                                                       0xb5, 0x74, 0x4d, 0xc5, 0x98, 0x3e, 0x99, 0xba };
static const uint8_t chiptest_e4836c3b50dd_KcA_44[]     = { 0x66, 0xde, 0x53, 0x4d, 0x9b, 0xf1, 0xe4, 0x4e,
                                                        0x96, 0xa5, 0x3a, 0x4b, 0x48, 0xd6, 0xb3, 0x53 };
static const uint8_t chiptest_e4836c3b50dd_KcB_45[]     = { 0x49, 0x45, 0xc3, 0x8b, 0xb4, 0x76, 0xcb, 0x0f,
                                                        0x34, 0x7f, 0x32, 0x22, 0xbe, 0x9b, 0x64, 0xa2 };
static const uint8_t chiptest_e4836c3b50dd_MAC_KcA_46[] = { 0xe5, 0x64, 0xc9, 0x3b, 0x30, 0x15, 0xef, 0xb9, 0x46, 0xdc, 0x16,
                                                            0xd6, 0x42, 0xbb, 0xe7, 0xd1, 0xc8, 0xda, 0x5b, 0xe1, 0x64, 0xed,
                                                            0x9f, 0xc3, 0xba, 0xe4, 0xe0, 0xff, 0x86, 0xe1, 0xbd, 0x3c };
static const uint8_t chiptest_e4836c3b50dd_MAC_KcB_47[] = { 0x07, 0x2a, 0x94, 0xd9, 0xa5, 0x4e, 0xdc, 0x20, 0x1d, 0x88, 0x91,
                                                            0x53, 0x4c, 0x23, 0x17, 0xca, 0xdf, 0x3e, 0xa3, 0x79, 0x28, 0x27,
                                                            0xf4, 0x79, 0xe8, 0x73, 0xf9, 0x3e, 0x90, 0xf2, 0x15, 0x52 };
static const struct spake2p_rfc_tv chiptest_e4836c3b50dd_test_vector_48 = {
    .context               = reinterpret_cast<const uint8_t *>("SPAKE2+-P256-SHA256-HKDF draft-01"),
    .context_len           = 33,
    .prover_identity       = reinterpret_cast<const uint8_t *>(""),
    .prover_identity_len   = 0,
    .verifier_identity     = reinterpret_cast<const uint8_t *>("server"),
    .verifier_identity_len = 6,
    .w0                    = chiptest_e4836c3b50dd_w0_33,
    .w0_len                = 32,
    .w1                    = chiptest_e4836c3b50dd_w1_34,
    .w1_len                = 32,
    .L                     = chiptest_e4836c3b50dd_L_35,
    .L_len                 = 65,
    .x                     = chiptest_e4836c3b50dd_x_36,
    .x_len                 = 32,
    .X                     = chiptest_e4836c3b50dd_X_37,
    .X_len                 = 65,
    .y                     = chiptest_e4836c3b50dd_y_38,
    .y_len                 = 32,
    .Y                     = chiptest_e4836c3b50dd_Y_39,
    .Y_len                 = 65,
    .Z                     = chiptest_e4836c3b50dd_Z_40,
    .Z_len                 = 65,
    .V                     = chiptest_e4836c3b50dd_V_41,
    .V_len                 = 65,
    .Ka                    = chiptest_e4836c3b50dd_Ka_42,
    .Ka_len                = 16,
    .Ke                    = chiptest_e4836c3b50dd_Ke_43,
    .Ke_len                = 16,
    .KcA                   = chiptest_e4836c3b50dd_KcA_44,
    .KcA_len               = 16,
    .KcB                   = chiptest_e4836c3b50dd_KcB_45,
    .KcB_len               = 16,
    .MAC_KcA               = chiptest_e4836c3b50dd_MAC_KcA_46,
    .MAC_KcA_len           = 32,
    .MAC_KcB               = chiptest_e4836c3b50dd_MAC_KcB_47,
    .MAC_KcB_len           = 32
};
static const uint8_t chiptest_e4836c3b50dd_w0_49[] = { 0xe6, 0x88, 0x7c, 0xf9, 0xbd, 0xfb, 0x75, 0x79, 0xc6, 0x9b, 0xf4,
                                                       0x79, 0x28, 0xa8, 0x45, 0x14, 0xb5, 0xe3, 0x55, 0xac, 0x03, 0x48,
                                                       0x63, 0xf7, 0xff, 0xaf, 0x43, 0x90, 0xe6, 0x7d, 0x79, 0x8c };
static const uint8_t chiptest_e4836c3b50dd_w1_50[] = { 0x24, 0xb5, 0xae, 0x4a, 0xbd, 0xa8, 0x68, 0xec, 0x93, 0x36, 0xff,
                                                       0xc3, 0xb7, 0x8e, 0xe3, 0x1c, 0x57, 0x55, 0xbe, 0xf1, 0x75, 0x92,
                                                       0x27, 0xef, 0x53, 0x72, 0xca, 0x13, 0x9b, 0x94, 0xe5, 0x12 };
static const uint8_t chiptest_e4836c3b50dd_L_51[]  = {
    0x04, 0x95, 0x64, 0x5c, 0xfb, 0x74, 0xdf, 0x6e, 0x58, 0xf9, 0x74, 0x8b, 0xb8, 0x3a, 0x86, 0x62, 0x0b,
    0xab, 0x7c, 0x82, 0xe1, 0x07, 0xf5, 0x7d, 0x68, 0x70, 0xda, 0x8c, 0xbc, 0xb2, 0xff, 0x9f, 0x70, 0x63,
    0xa1, 0x4b, 0x64, 0x02, 0xc6, 0x2f, 0x99, 0xaf, 0xcb, 0x97, 0x06, 0xa4, 0xd1, 0xa1, 0x43, 0x27, 0x32,
    0x59, 0xfe, 0x76, 0xf1, 0xc6, 0x05, 0xa3, 0x63, 0x97, 0x45, 0xa9, 0x21, 0x54, 0xb9
};
static const uint8_t chiptest_e4836c3b50dd_x_52[] = { 0x5b, 0x47, 0x86, 0x19, 0x80, 0x4f, 0x49, 0x38, 0xd3, 0x61, 0xfb,
                                                      0xba, 0x3a, 0x20, 0x64, 0x87, 0x25, 0x22, 0x2f, 0x0a, 0x54, 0xcc,
                                                      0x4c, 0x87, 0x61, 0x39, 0xef, 0xe7, 0xd9, 0xa2, 0x17, 0x86 };
static const uint8_t chiptest_e4836c3b50dd_X_53[] = {
    0x04, 0xa6, 0xdb, 0x23, 0xd0, 0x01, 0x72, 0x3f, 0xb0, 0x1f, 0xcf, 0xc9, 0xd0, 0x87, 0x46, 0xc3, 0xc2,
    0xa0, 0xa3, 0xfe, 0xff, 0x86, 0x35, 0xd2, 0x9c, 0xad, 0x28, 0x53, 0xe7, 0x35, 0x86, 0x23, 0x42, 0x5c,
    0xf3, 0x97, 0x12, 0xe9, 0x28, 0x05, 0x45, 0x61, 0xba, 0x71, 0xe2, 0xdc, 0x11, 0xf3, 0x00, 0xf1, 0x76,
    0x0e, 0x71, 0xeb, 0x17, 0x70, 0x21, 0xa8, 0xf8, 0x5e, 0x78, 0x68, 0x90, 0x71, 0xcd
};
static const uint8_t chiptest_e4836c3b50dd_y_54[] = { 0x76, 0x67, 0x70, 0xda, 0xd8, 0xc8, 0xee, 0xcb, 0xa9, 0x36, 0x82,
                                                      0x3c, 0x0a, 0xed, 0x04, 0x4b, 0x8c, 0x3c, 0x4f, 0x76, 0x55, 0xe8,
                                                      0xbe, 0xec, 0x44, 0xa1, 0x5d, 0xcb, 0xca, 0xf7, 0x8e, 0x5e };
static const uint8_t chiptest_e4836c3b50dd_Y_55[] = {
    0x04, 0x39, 0x0d, 0x29, 0xbf, 0x18, 0x5c, 0x3a, 0xbf, 0x99, 0xf1, 0x50, 0xae, 0x7c, 0x13, 0x38, 0x8c,
    0x82, 0xb6, 0xbe, 0x0c, 0x07, 0xb1, 0xb8, 0xd9, 0x0d, 0x26, 0x85, 0x3e, 0x84, 0x37, 0x4b, 0xbd, 0xc8,
    0x2b, 0xec, 0xdb, 0x97, 0x8c, 0xa3, 0x79, 0x2f, 0x47, 0x24, 0x24, 0x10, 0x6a, 0x25, 0x78, 0x01, 0x27,
    0x52, 0xc1, 0x19, 0x38, 0xfc, 0xf6, 0x0a, 0x41, 0xdf, 0x75, 0xff, 0x7c, 0xf9, 0x47
};
static const uint8_t chiptest_e4836c3b50dd_Z_56[] = {
    0x04, 0x0a, 0x15, 0x0d, 0x9a, 0x62, 0xf5, 0x14, 0xc9, 0xa1, 0xfe, 0xdd, 0x78, 0x2a, 0x02, 0x40, 0xa3,
    0x42, 0x72, 0x10, 0x46, 0xce, 0xfb, 0x11, 0x11, 0xc3, 0xad, 0xb3, 0xbe, 0x89, 0x3c, 0xe9, 0xfc, 0xd2,
    0xff, 0xa1, 0x37, 0x92, 0x2f, 0xcf, 0x8a, 0x58, 0x8d, 0x0f, 0x76, 0xba, 0x9c, 0x55, 0xc8, 0x5d, 0xa2,
    0xaf, 0x3f, 0x1c, 0x78, 0x9c, 0xa1, 0x79, 0x76, 0x81, 0x03, 0x87, 0xfb, 0x1d, 0x7e
};
static const uint8_t chiptest_e4836c3b50dd_V_57[] = {
    0x04, 0xf8, 0xe2, 0x47, 0xcc, 0x26, 0x3a, 0x18, 0x46, 0x27, 0x2f, 0x5a, 0x3b, 0x61, 0xb6, 0x8a, 0xa6,
    0x0a, 0x5a, 0x26, 0x65, 0xd1, 0x0c, 0xd2, 0x2c, 0x89, 0xcd, 0x6b, 0xad, 0x05, 0xdc, 0x0e, 0x5e, 0x65,
    0x0f, 0x21, 0xff, 0x01, 0x71, 0x86, 0xcc, 0x92, 0x65, 0x1a, 0x4c, 0xd7, 0xe6, 0x6c, 0xe8, 0x8f, 0x52,
    0x92, 0x99, 0xf3, 0x40, 0xea, 0x80, 0xfb, 0x90, 0xa9, 0xba, 0xd0, 0x94, 0xe1, 0xa6
};
static const uint8_t chiptest_e4836c3b50dd_Ka_58[]      = { 0x59, 0x29, 0xa3, 0xce, 0x98, 0x22, 0xc8, 0x14,
                                                       0x01, 0xbf, 0x0f, 0x76, 0x4f, 0x69, 0xaf, 0x08 };
static const uint8_t chiptest_e4836c3b50dd_Ke_59[]      = { 0xea, 0x32, 0x76, 0xd6, 0x83, 0x34, 0x57, 0x60,
                                                       0x97, 0xe0, 0x4b, 0x19, 0xee, 0x5a, 0x3a, 0x8b };
static const uint8_t chiptest_e4836c3b50dd_KcA_60[]     = { 0x7f, 0x84, 0xb9, 0x39, 0xd6, 0x00, 0x11, 0x72,
                                                        0x56, 0xb0, 0xc8, 0xa6, 0xd4, 0x0c, 0xf1, 0x81 };
static const uint8_t chiptest_e4836c3b50dd_KcB_61[]     = { 0xf7, 0xd7, 0x54, 0x7c, 0xed, 0x93, 0xf6, 0x81,
                                                        0xe8, 0xdf, 0x4c, 0x25, 0x8c, 0x45, 0x16, 0xfd };
static const uint8_t chiptest_e4836c3b50dd_MAC_KcA_62[] = { 0x71, 0xd9, 0x41, 0x27, 0x79, 0xb6, 0xc4, 0x5a, 0x2c, 0x61, 0x5c,
                                                            0x9d, 0xf3, 0xf1, 0xfd, 0x93, 0xdc, 0x0a, 0xaf, 0x63, 0x10, 0x4d,
                                                            0xa8, 0xec, 0xe4, 0xaa, 0x1b, 0x5a, 0x3a, 0x41, 0x5f, 0xea };
static const uint8_t chiptest_e4836c3b50dd_MAC_KcB_63[] = { 0x09, 0x5d, 0xc0, 0x40, 0x03, 0x55, 0xcc, 0x23, 0x3f, 0xde, 0x74,
                                                            0x37, 0x81, 0x18, 0x15, 0xb3, 0xc1, 0x52, 0x4a, 0xae, 0x80, 0xfd,
                                                            0x4e, 0x68, 0x10, 0xcf, 0x53, 0x1c, 0xf1, 0x1d, 0x20, 0xe3 };
static const struct spake2p_rfc_tv chiptest_e4836c3b50dd_test_vector_64 = {
    .context               = reinterpret_cast<const uint8_t *>("SPAKE2+-P256-SHA256-HKDF draft-01"),
    .context_len           = 33,
    .prover_identity       = reinterpret_cast<const uint8_t *>(""),
    .prover_identity_len   = 0,
    .verifier_identity     = reinterpret_cast<const uint8_t *>(""),
    .verifier_identity_len = 0,
    .w0                    = chiptest_e4836c3b50dd_w0_49,
    .w0_len                = 32,
    .w1                    = chiptest_e4836c3b50dd_w1_50,
    .w1_len                = 32,
    .L                     = chiptest_e4836c3b50dd_L_51,
    .L_len                 = 65,
    .x                     = chiptest_e4836c3b50dd_x_52,
    .x_len                 = 32,
    .X                     = chiptest_e4836c3b50dd_X_53,
    .X_len                 = 65,
    .y                     = chiptest_e4836c3b50dd_y_54,
    .y_len                 = 32,
    .Y                     = chiptest_e4836c3b50dd_Y_55,
    .Y_len                 = 65,
    .Z                     = chiptest_e4836c3b50dd_Z_56,
    .Z_len                 = 65,
    .V                     = chiptest_e4836c3b50dd_V_57,
    .V_len                 = 65,
    .Ka                    = chiptest_e4836c3b50dd_Ka_58,
    .Ka_len                = 16,
    .Ke                    = chiptest_e4836c3b50dd_Ke_59,
    .Ke_len                = 16,
    .KcA                   = chiptest_e4836c3b50dd_KcA_60,
    .KcA_len               = 16,
    .KcB                   = chiptest_e4836c3b50dd_KcB_61,
    .KcB_len               = 16,
    .MAC_KcA               = chiptest_e4836c3b50dd_MAC_KcA_62,
    .MAC_KcA_len           = 32,
    .MAC_KcB               = chiptest_e4836c3b50dd_MAC_KcB_63,
    .MAC_KcB_len           = 32
};
static const struct spake2p_rfc_tv * rfc_tvs[] = { &chiptest_e4836c3b50dd_test_vector_16, &chiptest_e4836c3b50dd_test_vector_32,
                                                   &chiptest_e4836c3b50dd_test_vector_48, &chiptest_e4836c3b50dd_test_vector_64 };

} // namespace Crypto
} // namespace chip
