/*
 *
 *    Copyright (c) 2022 Project CHIP Authors
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

#import "MTRAttributeTLVValueDecoder_Internal.h"

#import "MTRStructsObjc.h"
#import "NSDataSpanConversion.h"
#import "NSStringSpanConversion.h"

#include <app-common/zap-generated/cluster-objects.h>
#include <app-common/zap-generated/ids/Attributes.h>
#include <app-common/zap-generated/ids/Clusters.h>
#include <app/data-model/DecodableList.h>
#include <app/data-model/Decode.h>
#include <lib/support/TypeTraits.h>

using namespace chip;
using namespace chip::app;

static id _Nullable DecodeAttributeValueForIdentifyCluster(AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::Identify;
    switch (aAttributeId) {
    case Attributes::IdentifyTime::Id: {
        using TypeInfo = Attributes::IdentifyTime::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::IdentifyType::Id: {
        using TypeInfo = Attributes::IdentifyType::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForGroupsCluster(AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::Groups;
    switch (aAttributeId) {
    case Attributes::NameSupport::Id: {
        using TypeInfo = Attributes::NameSupport::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue.Raw()];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForScenesCluster(AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::Scenes;
    switch (aAttributeId) {
    case Attributes::SceneCount::Id: {
        using TypeInfo = Attributes::SceneCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::CurrentScene::Id: {
        using TypeInfo = Attributes::CurrentScene::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::CurrentGroup::Id: {
        using TypeInfo = Attributes::CurrentGroup::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::SceneValid::Id: {
        using TypeInfo = Attributes::SceneValid::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::NameSupport::Id: {
        using TypeInfo = Attributes::NameSupport::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::LastConfiguredBy::Id: {
        using TypeInfo = Attributes::LastConfiguredBy::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedLongLong:cppValue.Value()];
        }
        return value;
    }
    case Attributes::SceneTableSize::Id: {
        using TypeInfo = Attributes::SceneTableSize::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::RemainingCapacity::Id: {
        using TypeInfo = Attributes::RemainingCapacity::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForOnOffCluster(AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::OnOff;
    switch (aAttributeId) {
    case Attributes::OnOff::Id: {
        using TypeInfo = Attributes::OnOff::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::GlobalSceneControl::Id: {
        using TypeInfo = Attributes::GlobalSceneControl::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::OnTime::Id: {
        using TypeInfo = Attributes::OnTime::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::OffWaitTime::Id: {
        using TypeInfo = Attributes::OffWaitTime::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::StartUpOnOff::Id: {
        using TypeInfo = Attributes::StartUpOnOff::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue.Value())];
        }
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForOnOffSwitchConfigurationCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::OnOffSwitchConfiguration;
    switch (aAttributeId) {
    case Attributes::SwitchType::Id: {
        using TypeInfo = Attributes::SwitchType::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::SwitchActions::Id: {
        using TypeInfo = Attributes::SwitchActions::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForLevelControlCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::LevelControl;
    switch (aAttributeId) {
    case Attributes::CurrentLevel::Id: {
        using TypeInfo = Attributes::CurrentLevel::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::RemainingTime::Id: {
        using TypeInfo = Attributes::RemainingTime::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::MinLevel::Id: {
        using TypeInfo = Attributes::MinLevel::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::MaxLevel::Id: {
        using TypeInfo = Attributes::MaxLevel::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::CurrentFrequency::Id: {
        using TypeInfo = Attributes::CurrentFrequency::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::MinFrequency::Id: {
        using TypeInfo = Attributes::MinFrequency::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::MaxFrequency::Id: {
        using TypeInfo = Attributes::MaxFrequency::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::Options::Id: {
        using TypeInfo = Attributes::Options::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue.Raw()];
        return value;
    }
    case Attributes::OnOffTransitionTime::Id: {
        using TypeInfo = Attributes::OnOffTransitionTime::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::OnLevel::Id: {
        using TypeInfo = Attributes::OnLevel::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::OnTransitionTime::Id: {
        using TypeInfo = Attributes::OnTransitionTime::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::OffTransitionTime::Id: {
        using TypeInfo = Attributes::OffTransitionTime::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::DefaultMoveRate::Id: {
        using TypeInfo = Attributes::DefaultMoveRate::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::StartUpCurrentLevel::Id: {
        using TypeInfo = Attributes::StartUpCurrentLevel::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForBinaryInputBasicCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::BinaryInputBasic;
    switch (aAttributeId) {
    case Attributes::ActiveText::Id: {
        using TypeInfo = Attributes::ActiveText::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::Description::Id: {
        using TypeInfo = Attributes::Description::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::InactiveText::Id: {
        using TypeInfo = Attributes::InactiveText::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::OutOfService::Id: {
        using TypeInfo = Attributes::OutOfService::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::Polarity::Id: {
        using TypeInfo = Attributes::Polarity::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::PresentValue::Id: {
        using TypeInfo = Attributes::PresentValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::Reliability::Id: {
        using TypeInfo = Attributes::Reliability::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::StatusFlags::Id: {
        using TypeInfo = Attributes::StatusFlags::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::ApplicationType::Id: {
        using TypeInfo = Attributes::ApplicationType::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForDescriptorCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::Descriptor;
    switch (aAttributeId) {
    case Attributes::DeviceTypeList::Id: {
        using TypeInfo = Attributes::DeviceTypeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                MTRDescriptorClusterDeviceTypeStruct * newElement_0;
                newElement_0 = [MTRDescriptorClusterDeviceTypeStruct new];
                newElement_0.deviceType = [NSNumber numberWithUnsignedInt:entry_0.deviceType];
                newElement_0.revision = [NSNumber numberWithUnsignedShort:entry_0.revision];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::ServerList::Id: {
        using TypeInfo = Attributes::ServerList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::ClientList::Id: {
        using TypeInfo = Attributes::ClientList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::PartsList::Id: {
        using TypeInfo = Attributes::PartsList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedShort:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForBindingCluster(AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::Binding;
    switch (aAttributeId) {
    case Attributes::Binding::Id: {
        using TypeInfo = Attributes::Binding::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                MTRBindingClusterTargetStruct * newElement_0;
                newElement_0 = [MTRBindingClusterTargetStruct new];
                if (entry_0.node.HasValue()) {
                    newElement_0.node = [NSNumber numberWithUnsignedLongLong:entry_0.node.Value()];
                } else {
                    newElement_0.node = nil;
                }
                if (entry_0.group.HasValue()) {
                    newElement_0.group = [NSNumber numberWithUnsignedShort:entry_0.group.Value()];
                } else {
                    newElement_0.group = nil;
                }
                if (entry_0.endpoint.HasValue()) {
                    newElement_0.endpoint = [NSNumber numberWithUnsignedShort:entry_0.endpoint.Value()];
                } else {
                    newElement_0.endpoint = nil;
                }
                if (entry_0.cluster.HasValue()) {
                    newElement_0.cluster = [NSNumber numberWithUnsignedInt:entry_0.cluster.Value()];
                } else {
                    newElement_0.cluster = nil;
                }
                newElement_0.fabricIndex = [NSNumber numberWithUnsignedChar:entry_0.fabricIndex];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForAccessControlCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::AccessControl;
    switch (aAttributeId) {
    case Attributes::Acl::Id: {
        using TypeInfo = Attributes::Acl::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                MTRAccessControlClusterAccessControlEntryStruct * newElement_0;
                newElement_0 = [MTRAccessControlClusterAccessControlEntryStruct new];
                newElement_0.privilege = [NSNumber numberWithUnsignedChar:chip::to_underlying(entry_0.privilege)];
                newElement_0.authMode = [NSNumber numberWithUnsignedChar:chip::to_underlying(entry_0.authMode)];
                if (entry_0.subjects.IsNull()) {
                    newElement_0.subjects = nil;
                } else {
                    { // Scope for our temporary variables
                        auto * array_3 = [NSMutableArray new];
                        auto iter_3 = entry_0.subjects.Value().begin();
                        while (iter_3.Next()) {
                            auto & entry_3 = iter_3.GetValue();
                            NSNumber * newElement_3;
                            newElement_3 = [NSNumber numberWithUnsignedLongLong:entry_3];
                            [array_3 addObject:newElement_3];
                        }
                        CHIP_ERROR err = iter_3.GetStatus();
                        if (err != CHIP_NO_ERROR) {
                            *aError = err;
                            return nil;
                        }
                        newElement_0.subjects = array_3;
                    }
                }
                if (entry_0.targets.IsNull()) {
                    newElement_0.targets = nil;
                } else {
                    { // Scope for our temporary variables
                        auto * array_3 = [NSMutableArray new];
                        auto iter_3 = entry_0.targets.Value().begin();
                        while (iter_3.Next()) {
                            auto & entry_3 = iter_3.GetValue();
                            MTRAccessControlClusterAccessControlTargetStruct * newElement_3;
                            newElement_3 = [MTRAccessControlClusterAccessControlTargetStruct new];
                            if (entry_3.cluster.IsNull()) {
                                newElement_3.cluster = nil;
                            } else {
                                newElement_3.cluster = [NSNumber numberWithUnsignedInt:entry_3.cluster.Value()];
                            }
                            if (entry_3.endpoint.IsNull()) {
                                newElement_3.endpoint = nil;
                            } else {
                                newElement_3.endpoint = [NSNumber numberWithUnsignedShort:entry_3.endpoint.Value()];
                            }
                            if (entry_3.deviceType.IsNull()) {
                                newElement_3.deviceType = nil;
                            } else {
                                newElement_3.deviceType = [NSNumber numberWithUnsignedInt:entry_3.deviceType.Value()];
                            }
                            [array_3 addObject:newElement_3];
                        }
                        CHIP_ERROR err = iter_3.GetStatus();
                        if (err != CHIP_NO_ERROR) {
                            *aError = err;
                            return nil;
                        }
                        newElement_0.targets = array_3;
                    }
                }
                newElement_0.fabricIndex = [NSNumber numberWithUnsignedChar:entry_0.fabricIndex];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::Extension::Id: {
        using TypeInfo = Attributes::Extension::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                MTRAccessControlClusterAccessControlExtensionStruct * newElement_0;
                newElement_0 = [MTRAccessControlClusterAccessControlExtensionStruct new];
                newElement_0.data = AsData(entry_0.data);
                newElement_0.fabricIndex = [NSNumber numberWithUnsignedChar:entry_0.fabricIndex];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::SubjectsPerAccessControlEntry::Id: {
        using TypeInfo = Attributes::SubjectsPerAccessControlEntry::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::TargetsPerAccessControlEntry::Id: {
        using TypeInfo = Attributes::TargetsPerAccessControlEntry::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::AccessControlEntriesPerFabric::Id: {
        using TypeInfo = Attributes::AccessControlEntriesPerFabric::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForActionsCluster(AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::Actions;
    switch (aAttributeId) {
    case Attributes::ActionList::Id: {
        using TypeInfo = Attributes::ActionList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                MTRActionsClusterActionStruct * newElement_0;
                newElement_0 = [MTRActionsClusterActionStruct new];
                newElement_0.actionID = [NSNumber numberWithUnsignedShort:entry_0.actionID];
                newElement_0.name = AsString(entry_0.name);
                if (newElement_0.name == nil) {
                    CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
                    *aError = err;
                    return nil;
                }
                newElement_0.type = [NSNumber numberWithUnsignedChar:chip::to_underlying(entry_0.type)];
                newElement_0.endpointListID = [NSNumber numberWithUnsignedShort:entry_0.endpointListID];
                newElement_0.supportedCommands = [NSNumber numberWithUnsignedShort:entry_0.supportedCommands.Raw()];
                newElement_0.state = [NSNumber numberWithUnsignedChar:chip::to_underlying(entry_0.state)];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EndpointLists::Id: {
        using TypeInfo = Attributes::EndpointLists::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                MTRActionsClusterEndpointListStruct * newElement_0;
                newElement_0 = [MTRActionsClusterEndpointListStruct new];
                newElement_0.endpointListID = [NSNumber numberWithUnsignedShort:entry_0.endpointListID];
                newElement_0.name = AsString(entry_0.name);
                if (newElement_0.name == nil) {
                    CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
                    *aError = err;
                    return nil;
                }
                newElement_0.type = [NSNumber numberWithUnsignedChar:chip::to_underlying(entry_0.type)];
                { // Scope for our temporary variables
                    auto * array_2 = [NSMutableArray new];
                    auto iter_2 = entry_0.endpoints.begin();
                    while (iter_2.Next()) {
                        auto & entry_2 = iter_2.GetValue();
                        NSNumber * newElement_2;
                        newElement_2 = [NSNumber numberWithUnsignedShort:entry_2];
                        [array_2 addObject:newElement_2];
                    }
                    CHIP_ERROR err = iter_2.GetStatus();
                    if (err != CHIP_NO_ERROR) {
                        *aError = err;
                        return nil;
                    }
                    newElement_0.endpoints = array_2;
                }
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::SetupURL::Id: {
        using TypeInfo = Attributes::SetupURL::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForBasicInformationCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::BasicInformation;
    switch (aAttributeId) {
    case Attributes::DataModelRevision::Id: {
        using TypeInfo = Attributes::DataModelRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::VendorName::Id: {
        using TypeInfo = Attributes::VendorName::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::VendorID::Id: {
        using TypeInfo = Attributes::VendorID::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::ProductName::Id: {
        using TypeInfo = Attributes::ProductName::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::ProductID::Id: {
        using TypeInfo = Attributes::ProductID::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::NodeLabel::Id: {
        using TypeInfo = Attributes::NodeLabel::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::Location::Id: {
        using TypeInfo = Attributes::Location::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::HardwareVersion::Id: {
        using TypeInfo = Attributes::HardwareVersion::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::HardwareVersionString::Id: {
        using TypeInfo = Attributes::HardwareVersionString::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::SoftwareVersion::Id: {
        using TypeInfo = Attributes::SoftwareVersion::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::SoftwareVersionString::Id: {
        using TypeInfo = Attributes::SoftwareVersionString::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::ManufacturingDate::Id: {
        using TypeInfo = Attributes::ManufacturingDate::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::PartNumber::Id: {
        using TypeInfo = Attributes::PartNumber::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::ProductURL::Id: {
        using TypeInfo = Attributes::ProductURL::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::ProductLabel::Id: {
        using TypeInfo = Attributes::ProductLabel::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::SerialNumber::Id: {
        using TypeInfo = Attributes::SerialNumber::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::LocalConfigDisabled::Id: {
        using TypeInfo = Attributes::LocalConfigDisabled::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::Reachable::Id: {
        using TypeInfo = Attributes::Reachable::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::UniqueID::Id: {
        using TypeInfo = Attributes::UniqueID::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::CapabilityMinima::Id: {
        using TypeInfo = Attributes::CapabilityMinima::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        MTRBasicInformationClusterCapabilityMinimaStruct * _Nonnull value;
        value = [MTRBasicInformationClusterCapabilityMinimaStruct new];
        value.caseSessionsPerFabric = [NSNumber numberWithUnsignedShort:cppValue.caseSessionsPerFabric];
        value.subscriptionsPerFabric = [NSNumber numberWithUnsignedShort:cppValue.subscriptionsPerFabric];
        return value;
    }
    case Attributes::ProductAppearance::Id: {
        using TypeInfo = Attributes::ProductAppearance::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        MTRBasicInformationClusterProductAppearanceStruct * _Nonnull value;
        value = [MTRBasicInformationClusterProductAppearanceStruct new];
        value.finish = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue.finish)];
        if (cppValue.primaryColor.IsNull()) {
            value.primaryColor = nil;
        } else {
            value.primaryColor = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue.primaryColor.Value())];
        }
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForOTASoftwareUpdateProviderCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::OtaSoftwareUpdateProvider;
    switch (aAttributeId) {
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForOTASoftwareUpdateRequestorCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::OtaSoftwareUpdateRequestor;
    switch (aAttributeId) {
    case Attributes::DefaultOTAProviders::Id: {
        using TypeInfo = Attributes::DefaultOTAProviders::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                MTROTASoftwareUpdateRequestorClusterProviderLocation * newElement_0;
                newElement_0 = [MTROTASoftwareUpdateRequestorClusterProviderLocation new];
                newElement_0.providerNodeID = [NSNumber numberWithUnsignedLongLong:entry_0.providerNodeID];
                newElement_0.endpoint = [NSNumber numberWithUnsignedShort:entry_0.endpoint];
                newElement_0.fabricIndex = [NSNumber numberWithUnsignedChar:entry_0.fabricIndex];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::UpdatePossible::Id: {
        using TypeInfo = Attributes::UpdatePossible::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::UpdateState::Id: {
        using TypeInfo = Attributes::UpdateState::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::UpdateStateProgress::Id: {
        using TypeInfo = Attributes::UpdateStateProgress::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForLocalizationConfigurationCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::LocalizationConfiguration;
    switch (aAttributeId) {
    case Attributes::ActiveLocale::Id: {
        using TypeInfo = Attributes::ActiveLocale::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::SupportedLocales::Id: {
        using TypeInfo = Attributes::SupportedLocales::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSString * newElement_0;
                newElement_0 = AsString(entry_0);
                if (newElement_0 == nil) {
                    CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
                    *aError = err;
                    return nil;
                }
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForTimeFormatLocalizationCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::TimeFormatLocalization;
    switch (aAttributeId) {
    case Attributes::HourFormat::Id: {
        using TypeInfo = Attributes::HourFormat::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::ActiveCalendarType::Id: {
        using TypeInfo = Attributes::ActiveCalendarType::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::SupportedCalendarTypes::Id: {
        using TypeInfo = Attributes::SupportedCalendarTypes::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedChar:chip::to_underlying(entry_0)];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForUnitLocalizationCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::UnitLocalization;
    switch (aAttributeId) {
    case Attributes::TemperatureUnit::Id: {
        using TypeInfo = Attributes::TemperatureUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForPowerSourceConfigurationCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::PowerSourceConfiguration;
    switch (aAttributeId) {
    case Attributes::Sources::Id: {
        using TypeInfo = Attributes::Sources::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedChar:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForPowerSourceCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::PowerSource;
    switch (aAttributeId) {
    case Attributes::Status::Id: {
        using TypeInfo = Attributes::Status::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::Order::Id: {
        using TypeInfo = Attributes::Order::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::Description::Id: {
        using TypeInfo = Attributes::Description::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::WiredAssessedInputVoltage::Id: {
        using TypeInfo = Attributes::WiredAssessedInputVoltage::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedInt:cppValue.Value()];
        }
        return value;
    }
    case Attributes::WiredAssessedInputFrequency::Id: {
        using TypeInfo = Attributes::WiredAssessedInputFrequency::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::WiredCurrentType::Id: {
        using TypeInfo = Attributes::WiredCurrentType::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::WiredAssessedCurrent::Id: {
        using TypeInfo = Attributes::WiredAssessedCurrent::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedInt:cppValue.Value()];
        }
        return value;
    }
    case Attributes::WiredNominalVoltage::Id: {
        using TypeInfo = Attributes::WiredNominalVoltage::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::WiredMaximumCurrent::Id: {
        using TypeInfo = Attributes::WiredMaximumCurrent::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::WiredPresent::Id: {
        using TypeInfo = Attributes::WiredPresent::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::ActiveWiredFaults::Id: {
        using TypeInfo = Attributes::ActiveWiredFaults::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedChar:chip::to_underlying(entry_0)];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::BatVoltage::Id: {
        using TypeInfo = Attributes::BatVoltage::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedInt:cppValue.Value()];
        }
        return value;
    }
    case Attributes::BatPercentRemaining::Id: {
        using TypeInfo = Attributes::BatPercentRemaining::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::BatTimeRemaining::Id: {
        using TypeInfo = Attributes::BatTimeRemaining::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedInt:cppValue.Value()];
        }
        return value;
    }
    case Attributes::BatChargeLevel::Id: {
        using TypeInfo = Attributes::BatChargeLevel::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::BatReplacementNeeded::Id: {
        using TypeInfo = Attributes::BatReplacementNeeded::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::BatReplaceability::Id: {
        using TypeInfo = Attributes::BatReplaceability::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::BatPresent::Id: {
        using TypeInfo = Attributes::BatPresent::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::ActiveBatFaults::Id: {
        using TypeInfo = Attributes::ActiveBatFaults::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedChar:chip::to_underlying(entry_0)];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::BatReplacementDescription::Id: {
        using TypeInfo = Attributes::BatReplacementDescription::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::BatCommonDesignation::Id: {
        using TypeInfo = Attributes::BatCommonDesignation::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::BatANSIDesignation::Id: {
        using TypeInfo = Attributes::BatANSIDesignation::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::BatIECDesignation::Id: {
        using TypeInfo = Attributes::BatIECDesignation::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::BatApprovedChemistry::Id: {
        using TypeInfo = Attributes::BatApprovedChemistry::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::BatCapacity::Id: {
        using TypeInfo = Attributes::BatCapacity::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::BatQuantity::Id: {
        using TypeInfo = Attributes::BatQuantity::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::BatChargeState::Id: {
        using TypeInfo = Attributes::BatChargeState::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::BatTimeToFullCharge::Id: {
        using TypeInfo = Attributes::BatTimeToFullCharge::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedInt:cppValue.Value()];
        }
        return value;
    }
    case Attributes::BatFunctionalWhileCharging::Id: {
        using TypeInfo = Attributes::BatFunctionalWhileCharging::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::BatChargingCurrent::Id: {
        using TypeInfo = Attributes::BatChargingCurrent::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedInt:cppValue.Value()];
        }
        return value;
    }
    case Attributes::ActiveBatChargeFaults::Id: {
        using TypeInfo = Attributes::ActiveBatChargeFaults::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedChar:chip::to_underlying(entry_0)];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForGeneralCommissioningCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::GeneralCommissioning;
    switch (aAttributeId) {
    case Attributes::Breadcrumb::Id: {
        using TypeInfo = Attributes::Breadcrumb::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedLongLong:cppValue];
        return value;
    }
    case Attributes::BasicCommissioningInfo::Id: {
        using TypeInfo = Attributes::BasicCommissioningInfo::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        MTRGeneralCommissioningClusterBasicCommissioningInfo * _Nonnull value;
        value = [MTRGeneralCommissioningClusterBasicCommissioningInfo new];
        value.failSafeExpiryLengthSeconds = [NSNumber numberWithUnsignedShort:cppValue.failSafeExpiryLengthSeconds];
        value.maxCumulativeFailsafeSeconds = [NSNumber numberWithUnsignedShort:cppValue.maxCumulativeFailsafeSeconds];
        return value;
    }
    case Attributes::RegulatoryConfig::Id: {
        using TypeInfo = Attributes::RegulatoryConfig::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LocationCapability::Id: {
        using TypeInfo = Attributes::LocationCapability::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::SupportsConcurrentConnection::Id: {
        using TypeInfo = Attributes::SupportsConcurrentConnection::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForNetworkCommissioningCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::NetworkCommissioning;
    switch (aAttributeId) {
    case Attributes::MaxNetworks::Id: {
        using TypeInfo = Attributes::MaxNetworks::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::Networks::Id: {
        using TypeInfo = Attributes::Networks::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                MTRNetworkCommissioningClusterNetworkInfo * newElement_0;
                newElement_0 = [MTRNetworkCommissioningClusterNetworkInfo new];
                newElement_0.networkID = AsData(entry_0.networkID);
                newElement_0.connected = [NSNumber numberWithBool:entry_0.connected];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::ScanMaxTimeSeconds::Id: {
        using TypeInfo = Attributes::ScanMaxTimeSeconds::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::ConnectMaxTimeSeconds::Id: {
        using TypeInfo = Attributes::ConnectMaxTimeSeconds::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::InterfaceEnabled::Id: {
        using TypeInfo = Attributes::InterfaceEnabled::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::LastNetworkingStatus::Id: {
        using TypeInfo = Attributes::LastNetworkingStatus::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue.Value())];
        }
        return value;
    }
    case Attributes::LastNetworkID::Id: {
        using TypeInfo = Attributes::LastNetworkID::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSData * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = AsData(cppValue.Value());
        }
        return value;
    }
    case Attributes::LastConnectErrorValue::Id: {
        using TypeInfo = Attributes::LastConnectErrorValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithInt:cppValue.Value()];
        }
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForDiagnosticLogsCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::DiagnosticLogs;
    switch (aAttributeId) {
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForGeneralDiagnosticsCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::GeneralDiagnostics;
    switch (aAttributeId) {
    case Attributes::NetworkInterfaces::Id: {
        using TypeInfo = Attributes::NetworkInterfaces::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                MTRGeneralDiagnosticsClusterNetworkInterface * newElement_0;
                newElement_0 = [MTRGeneralDiagnosticsClusterNetworkInterface new];
                newElement_0.name = AsString(entry_0.name);
                if (newElement_0.name == nil) {
                    CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
                    *aError = err;
                    return nil;
                }
                newElement_0.isOperational = [NSNumber numberWithBool:entry_0.isOperational];
                if (entry_0.offPremiseServicesReachableIPv4.IsNull()) {
                    newElement_0.offPremiseServicesReachableIPv4 = nil;
                } else {
                    newElement_0.offPremiseServicesReachableIPv4 =
                        [NSNumber numberWithBool:entry_0.offPremiseServicesReachableIPv4.Value()];
                }
                if (entry_0.offPremiseServicesReachableIPv6.IsNull()) {
                    newElement_0.offPremiseServicesReachableIPv6 = nil;
                } else {
                    newElement_0.offPremiseServicesReachableIPv6 =
                        [NSNumber numberWithBool:entry_0.offPremiseServicesReachableIPv6.Value()];
                }
                newElement_0.hardwareAddress = AsData(entry_0.hardwareAddress);
                { // Scope for our temporary variables
                    auto * array_2 = [NSMutableArray new];
                    auto iter_2 = entry_0.IPv4Addresses.begin();
                    while (iter_2.Next()) {
                        auto & entry_2 = iter_2.GetValue();
                        NSData * newElement_2;
                        newElement_2 = AsData(entry_2);
                        [array_2 addObject:newElement_2];
                    }
                    CHIP_ERROR err = iter_2.GetStatus();
                    if (err != CHIP_NO_ERROR) {
                        *aError = err;
                        return nil;
                    }
                    newElement_0.iPv4Addresses = array_2;
                }
                { // Scope for our temporary variables
                    auto * array_2 = [NSMutableArray new];
                    auto iter_2 = entry_0.IPv6Addresses.begin();
                    while (iter_2.Next()) {
                        auto & entry_2 = iter_2.GetValue();
                        NSData * newElement_2;
                        newElement_2 = AsData(entry_2);
                        [array_2 addObject:newElement_2];
                    }
                    CHIP_ERROR err = iter_2.GetStatus();
                    if (err != CHIP_NO_ERROR) {
                        *aError = err;
                        return nil;
                    }
                    newElement_0.iPv6Addresses = array_2;
                }
                newElement_0.type = [NSNumber numberWithUnsignedChar:chip::to_underlying(entry_0.type)];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::RebootCount::Id: {
        using TypeInfo = Attributes::RebootCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::UpTime::Id: {
        using TypeInfo = Attributes::UpTime::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedLongLong:cppValue];
        return value;
    }
    case Attributes::TotalOperationalHours::Id: {
        using TypeInfo = Attributes::TotalOperationalHours::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::BootReason::Id: {
        using TypeInfo = Attributes::BootReason::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::ActiveHardwareFaults::Id: {
        using TypeInfo = Attributes::ActiveHardwareFaults::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedChar:chip::to_underlying(entry_0)];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::ActiveRadioFaults::Id: {
        using TypeInfo = Attributes::ActiveRadioFaults::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedChar:chip::to_underlying(entry_0)];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::ActiveNetworkFaults::Id: {
        using TypeInfo = Attributes::ActiveNetworkFaults::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedChar:chip::to_underlying(entry_0)];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::TestEventTriggersEnabled::Id: {
        using TypeInfo = Attributes::TestEventTriggersEnabled::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForSoftwareDiagnosticsCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::SoftwareDiagnostics;
    switch (aAttributeId) {
    case Attributes::ThreadMetrics::Id: {
        using TypeInfo = Attributes::ThreadMetrics::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                MTRSoftwareDiagnosticsClusterThreadMetricsStruct * newElement_0;
                newElement_0 = [MTRSoftwareDiagnosticsClusterThreadMetricsStruct new];
                newElement_0.id = [NSNumber numberWithUnsignedLongLong:entry_0.id];
                if (entry_0.name.HasValue()) {
                    newElement_0.name = AsString(entry_0.name.Value());
                    if (newElement_0.name == nil) {
                        CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
                        *aError = err;
                        return nil;
                    }
                } else {
                    newElement_0.name = nil;
                }
                if (entry_0.stackFreeCurrent.HasValue()) {
                    newElement_0.stackFreeCurrent = [NSNumber numberWithUnsignedInt:entry_0.stackFreeCurrent.Value()];
                } else {
                    newElement_0.stackFreeCurrent = nil;
                }
                if (entry_0.stackFreeMinimum.HasValue()) {
                    newElement_0.stackFreeMinimum = [NSNumber numberWithUnsignedInt:entry_0.stackFreeMinimum.Value()];
                } else {
                    newElement_0.stackFreeMinimum = nil;
                }
                if (entry_0.stackSize.HasValue()) {
                    newElement_0.stackSize = [NSNumber numberWithUnsignedInt:entry_0.stackSize.Value()];
                } else {
                    newElement_0.stackSize = nil;
                }
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::CurrentHeapFree::Id: {
        using TypeInfo = Attributes::CurrentHeapFree::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedLongLong:cppValue];
        return value;
    }
    case Attributes::CurrentHeapUsed::Id: {
        using TypeInfo = Attributes::CurrentHeapUsed::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedLongLong:cppValue];
        return value;
    }
    case Attributes::CurrentHeapHighWatermark::Id: {
        using TypeInfo = Attributes::CurrentHeapHighWatermark::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedLongLong:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForThreadNetworkDiagnosticsCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::ThreadNetworkDiagnostics;
    switch (aAttributeId) {
    case Attributes::Channel::Id: {
        using TypeInfo = Attributes::Channel::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::RoutingRole::Id: {
        using TypeInfo = Attributes::RoutingRole::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue.Value())];
        }
        return value;
    }
    case Attributes::NetworkName::Id: {
        using TypeInfo = Attributes::NetworkName::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = AsString(cppValue.Value());
            if (value == nil) {
                CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
                *aError = err;
                return nil;
            }
        }
        return value;
    }
    case Attributes::PanId::Id: {
        using TypeInfo = Attributes::PanId::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::ExtendedPanId::Id: {
        using TypeInfo = Attributes::ExtendedPanId::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedLongLong:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MeshLocalPrefix::Id: {
        using TypeInfo = Attributes::MeshLocalPrefix::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSData * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = AsData(cppValue.Value());
        }
        return value;
    }
    case Attributes::OverrunCount::Id: {
        using TypeInfo = Attributes::OverrunCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedLongLong:cppValue];
        return value;
    }
    case Attributes::NeighborTable::Id: {
        using TypeInfo = Attributes::NeighborTable::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                MTRThreadNetworkDiagnosticsClusterNeighborTableStruct * newElement_0;
                newElement_0 = [MTRThreadNetworkDiagnosticsClusterNeighborTableStruct new];
                newElement_0.extAddress = [NSNumber numberWithUnsignedLongLong:entry_0.extAddress];
                newElement_0.age = [NSNumber numberWithUnsignedInt:entry_0.age];
                newElement_0.rloc16 = [NSNumber numberWithUnsignedShort:entry_0.rloc16];
                newElement_0.linkFrameCounter = [NSNumber numberWithUnsignedInt:entry_0.linkFrameCounter];
                newElement_0.mleFrameCounter = [NSNumber numberWithUnsignedInt:entry_0.mleFrameCounter];
                newElement_0.lqi = [NSNumber numberWithUnsignedChar:entry_0.lqi];
                if (entry_0.averageRssi.IsNull()) {
                    newElement_0.averageRssi = nil;
                } else {
                    newElement_0.averageRssi = [NSNumber numberWithChar:entry_0.averageRssi.Value()];
                }
                if (entry_0.lastRssi.IsNull()) {
                    newElement_0.lastRssi = nil;
                } else {
                    newElement_0.lastRssi = [NSNumber numberWithChar:entry_0.lastRssi.Value()];
                }
                newElement_0.frameErrorRate = [NSNumber numberWithUnsignedChar:entry_0.frameErrorRate];
                newElement_0.messageErrorRate = [NSNumber numberWithUnsignedChar:entry_0.messageErrorRate];
                newElement_0.rxOnWhenIdle = [NSNumber numberWithBool:entry_0.rxOnWhenIdle];
                newElement_0.fullThreadDevice = [NSNumber numberWithBool:entry_0.fullThreadDevice];
                newElement_0.fullNetworkData = [NSNumber numberWithBool:entry_0.fullNetworkData];
                newElement_0.isChild = [NSNumber numberWithBool:entry_0.isChild];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::RouteTable::Id: {
        using TypeInfo = Attributes::RouteTable::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                MTRThreadNetworkDiagnosticsClusterRouteTableStruct * newElement_0;
                newElement_0 = [MTRThreadNetworkDiagnosticsClusterRouteTableStruct new];
                newElement_0.extAddress = [NSNumber numberWithUnsignedLongLong:entry_0.extAddress];
                newElement_0.rloc16 = [NSNumber numberWithUnsignedShort:entry_0.rloc16];
                newElement_0.routerId = [NSNumber numberWithUnsignedChar:entry_0.routerId];
                newElement_0.nextHop = [NSNumber numberWithUnsignedChar:entry_0.nextHop];
                newElement_0.pathCost = [NSNumber numberWithUnsignedChar:entry_0.pathCost];
                newElement_0.lqiIn = [NSNumber numberWithUnsignedChar:entry_0.LQIIn];
                newElement_0.lqiOut = [NSNumber numberWithUnsignedChar:entry_0.LQIOut];
                newElement_0.age = [NSNumber numberWithUnsignedChar:entry_0.age];
                newElement_0.allocated = [NSNumber numberWithBool:entry_0.allocated];
                newElement_0.linkEstablished = [NSNumber numberWithBool:entry_0.linkEstablished];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::PartitionId::Id: {
        using TypeInfo = Attributes::PartitionId::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedInt:cppValue.Value()];
        }
        return value;
    }
    case Attributes::Weighting::Id: {
        using TypeInfo = Attributes::Weighting::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::DataVersion::Id: {
        using TypeInfo = Attributes::DataVersion::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::StableDataVersion::Id: {
        using TypeInfo = Attributes::StableDataVersion::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::LeaderRouterId::Id: {
        using TypeInfo = Attributes::LeaderRouterId::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::DetachedRoleCount::Id: {
        using TypeInfo = Attributes::DetachedRoleCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::ChildRoleCount::Id: {
        using TypeInfo = Attributes::ChildRoleCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::RouterRoleCount::Id: {
        using TypeInfo = Attributes::RouterRoleCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::LeaderRoleCount::Id: {
        using TypeInfo = Attributes::LeaderRoleCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::AttachAttemptCount::Id: {
        using TypeInfo = Attributes::AttachAttemptCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::PartitionIdChangeCount::Id: {
        using TypeInfo = Attributes::PartitionIdChangeCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::BetterPartitionAttachAttemptCount::Id: {
        using TypeInfo = Attributes::BetterPartitionAttachAttemptCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::ParentChangeCount::Id: {
        using TypeInfo = Attributes::ParentChangeCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::TxTotalCount::Id: {
        using TypeInfo = Attributes::TxTotalCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::TxUnicastCount::Id: {
        using TypeInfo = Attributes::TxUnicastCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::TxBroadcastCount::Id: {
        using TypeInfo = Attributes::TxBroadcastCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::TxAckRequestedCount::Id: {
        using TypeInfo = Attributes::TxAckRequestedCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::TxAckedCount::Id: {
        using TypeInfo = Attributes::TxAckedCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::TxNoAckRequestedCount::Id: {
        using TypeInfo = Attributes::TxNoAckRequestedCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::TxDataCount::Id: {
        using TypeInfo = Attributes::TxDataCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::TxDataPollCount::Id: {
        using TypeInfo = Attributes::TxDataPollCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::TxBeaconCount::Id: {
        using TypeInfo = Attributes::TxBeaconCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::TxBeaconRequestCount::Id: {
        using TypeInfo = Attributes::TxBeaconRequestCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::TxOtherCount::Id: {
        using TypeInfo = Attributes::TxOtherCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::TxRetryCount::Id: {
        using TypeInfo = Attributes::TxRetryCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::TxDirectMaxRetryExpiryCount::Id: {
        using TypeInfo = Attributes::TxDirectMaxRetryExpiryCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::TxIndirectMaxRetryExpiryCount::Id: {
        using TypeInfo = Attributes::TxIndirectMaxRetryExpiryCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::TxErrCcaCount::Id: {
        using TypeInfo = Attributes::TxErrCcaCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::TxErrAbortCount::Id: {
        using TypeInfo = Attributes::TxErrAbortCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::TxErrBusyChannelCount::Id: {
        using TypeInfo = Attributes::TxErrBusyChannelCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::RxTotalCount::Id: {
        using TypeInfo = Attributes::RxTotalCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::RxUnicastCount::Id: {
        using TypeInfo = Attributes::RxUnicastCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::RxBroadcastCount::Id: {
        using TypeInfo = Attributes::RxBroadcastCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::RxDataCount::Id: {
        using TypeInfo = Attributes::RxDataCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::RxDataPollCount::Id: {
        using TypeInfo = Attributes::RxDataPollCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::RxBeaconCount::Id: {
        using TypeInfo = Attributes::RxBeaconCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::RxBeaconRequestCount::Id: {
        using TypeInfo = Attributes::RxBeaconRequestCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::RxOtherCount::Id: {
        using TypeInfo = Attributes::RxOtherCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::RxAddressFilteredCount::Id: {
        using TypeInfo = Attributes::RxAddressFilteredCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::RxDestAddrFilteredCount::Id: {
        using TypeInfo = Attributes::RxDestAddrFilteredCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::RxDuplicatedCount::Id: {
        using TypeInfo = Attributes::RxDuplicatedCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::RxErrNoFrameCount::Id: {
        using TypeInfo = Attributes::RxErrNoFrameCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::RxErrUnknownNeighborCount::Id: {
        using TypeInfo = Attributes::RxErrUnknownNeighborCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::RxErrInvalidSrcAddrCount::Id: {
        using TypeInfo = Attributes::RxErrInvalidSrcAddrCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::RxErrSecCount::Id: {
        using TypeInfo = Attributes::RxErrSecCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::RxErrFcsCount::Id: {
        using TypeInfo = Attributes::RxErrFcsCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::RxErrOtherCount::Id: {
        using TypeInfo = Attributes::RxErrOtherCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ActiveTimestamp::Id: {
        using TypeInfo = Attributes::ActiveTimestamp::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedLongLong:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PendingTimestamp::Id: {
        using TypeInfo = Attributes::PendingTimestamp::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedLongLong:cppValue.Value()];
        }
        return value;
    }
    case Attributes::Delay::Id: {
        using TypeInfo = Attributes::Delay::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedInt:cppValue.Value()];
        }
        return value;
    }
    case Attributes::SecurityPolicy::Id: {
        using TypeInfo = Attributes::SecurityPolicy::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        MTRThreadNetworkDiagnosticsClusterSecurityPolicy * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [MTRThreadNetworkDiagnosticsClusterSecurityPolicy new];
            value.rotationTime = [NSNumber numberWithUnsignedShort:cppValue.Value().rotationTime];
            value.flags = [NSNumber numberWithUnsignedShort:cppValue.Value().flags];
        }
        return value;
    }
    case Attributes::ChannelPage0Mask::Id: {
        using TypeInfo = Attributes::ChannelPage0Mask::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSData * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = AsData(cppValue.Value());
        }
        return value;
    }
    case Attributes::OperationalDatasetComponents::Id: {
        using TypeInfo = Attributes::OperationalDatasetComponents::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        MTRThreadNetworkDiagnosticsClusterOperationalDatasetComponents * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [MTRThreadNetworkDiagnosticsClusterOperationalDatasetComponents new];
            value.activeTimestampPresent = [NSNumber numberWithBool:cppValue.Value().activeTimestampPresent];
            value.pendingTimestampPresent = [NSNumber numberWithBool:cppValue.Value().pendingTimestampPresent];
            value.masterKeyPresent = [NSNumber numberWithBool:cppValue.Value().masterKeyPresent];
            value.networkNamePresent = [NSNumber numberWithBool:cppValue.Value().networkNamePresent];
            value.extendedPanIdPresent = [NSNumber numberWithBool:cppValue.Value().extendedPanIdPresent];
            value.meshLocalPrefixPresent = [NSNumber numberWithBool:cppValue.Value().meshLocalPrefixPresent];
            value.delayPresent = [NSNumber numberWithBool:cppValue.Value().delayPresent];
            value.panIdPresent = [NSNumber numberWithBool:cppValue.Value().panIdPresent];
            value.channelPresent = [NSNumber numberWithBool:cppValue.Value().channelPresent];
            value.pskcPresent = [NSNumber numberWithBool:cppValue.Value().pskcPresent];
            value.securityPolicyPresent = [NSNumber numberWithBool:cppValue.Value().securityPolicyPresent];
            value.channelMaskPresent = [NSNumber numberWithBool:cppValue.Value().channelMaskPresent];
        }
        return value;
    }
    case Attributes::ActiveNetworkFaultsList::Id: {
        using TypeInfo = Attributes::ActiveNetworkFaultsList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedChar:chip::to_underlying(entry_0)];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForWiFiNetworkDiagnosticsCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::WiFiNetworkDiagnostics;
    switch (aAttributeId) {
    case Attributes::Bssid::Id: {
        using TypeInfo = Attributes::Bssid::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSData * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = AsData(cppValue.Value());
        }
        return value;
    }
    case Attributes::SecurityType::Id: {
        using TypeInfo = Attributes::SecurityType::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue.Value())];
        }
        return value;
    }
    case Attributes::WiFiVersion::Id: {
        using TypeInfo = Attributes::WiFiVersion::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue.Value())];
        }
        return value;
    }
    case Attributes::ChannelNumber::Id: {
        using TypeInfo = Attributes::ChannelNumber::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::Rssi::Id: {
        using TypeInfo = Attributes::Rssi::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::BeaconLostCount::Id: {
        using TypeInfo = Attributes::BeaconLostCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedInt:cppValue.Value()];
        }
        return value;
    }
    case Attributes::BeaconRxCount::Id: {
        using TypeInfo = Attributes::BeaconRxCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedInt:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PacketMulticastRxCount::Id: {
        using TypeInfo = Attributes::PacketMulticastRxCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedInt:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PacketMulticastTxCount::Id: {
        using TypeInfo = Attributes::PacketMulticastTxCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedInt:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PacketUnicastRxCount::Id: {
        using TypeInfo = Attributes::PacketUnicastRxCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedInt:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PacketUnicastTxCount::Id: {
        using TypeInfo = Attributes::PacketUnicastTxCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedInt:cppValue.Value()];
        }
        return value;
    }
    case Attributes::CurrentMaxRate::Id: {
        using TypeInfo = Attributes::CurrentMaxRate::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedLongLong:cppValue.Value()];
        }
        return value;
    }
    case Attributes::OverrunCount::Id: {
        using TypeInfo = Attributes::OverrunCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedLongLong:cppValue.Value()];
        }
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForEthernetNetworkDiagnosticsCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::EthernetNetworkDiagnostics;
    switch (aAttributeId) {
    case Attributes::PHYRate::Id: {
        using TypeInfo = Attributes::PHYRate::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue.Value())];
        }
        return value;
    }
    case Attributes::FullDuplex::Id: {
        using TypeInfo = Attributes::FullDuplex::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithBool:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PacketRxCount::Id: {
        using TypeInfo = Attributes::PacketRxCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedLongLong:cppValue];
        return value;
    }
    case Attributes::PacketTxCount::Id: {
        using TypeInfo = Attributes::PacketTxCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedLongLong:cppValue];
        return value;
    }
    case Attributes::TxErrCount::Id: {
        using TypeInfo = Attributes::TxErrCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedLongLong:cppValue];
        return value;
    }
    case Attributes::CollisionCount::Id: {
        using TypeInfo = Attributes::CollisionCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedLongLong:cppValue];
        return value;
    }
    case Attributes::OverrunCount::Id: {
        using TypeInfo = Attributes::OverrunCount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedLongLong:cppValue];
        return value;
    }
    case Attributes::CarrierDetect::Id: {
        using TypeInfo = Attributes::CarrierDetect::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithBool:cppValue.Value()];
        }
        return value;
    }
    case Attributes::TimeSinceReset::Id: {
        using TypeInfo = Attributes::TimeSinceReset::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedLongLong:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForBridgedDeviceBasicInformationCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::BridgedDeviceBasicInformation;
    switch (aAttributeId) {
    case Attributes::VendorName::Id: {
        using TypeInfo = Attributes::VendorName::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::VendorID::Id: {
        using TypeInfo = Attributes::VendorID::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::ProductName::Id: {
        using TypeInfo = Attributes::ProductName::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::NodeLabel::Id: {
        using TypeInfo = Attributes::NodeLabel::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::HardwareVersion::Id: {
        using TypeInfo = Attributes::HardwareVersion::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::HardwareVersionString::Id: {
        using TypeInfo = Attributes::HardwareVersionString::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::SoftwareVersion::Id: {
        using TypeInfo = Attributes::SoftwareVersion::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::SoftwareVersionString::Id: {
        using TypeInfo = Attributes::SoftwareVersionString::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::ManufacturingDate::Id: {
        using TypeInfo = Attributes::ManufacturingDate::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::PartNumber::Id: {
        using TypeInfo = Attributes::PartNumber::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::ProductURL::Id: {
        using TypeInfo = Attributes::ProductURL::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::ProductLabel::Id: {
        using TypeInfo = Attributes::ProductLabel::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::SerialNumber::Id: {
        using TypeInfo = Attributes::SerialNumber::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::Reachable::Id: {
        using TypeInfo = Attributes::Reachable::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::UniqueID::Id: {
        using TypeInfo = Attributes::UniqueID::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::ProductAppearance::Id: {
        using TypeInfo = Attributes::ProductAppearance::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        MTRBridgedDeviceBasicInformationClusterProductAppearanceStruct * _Nonnull value;
        value = [MTRBridgedDeviceBasicInformationClusterProductAppearanceStruct new];
        value.finish = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue.finish)];
        if (cppValue.primaryColor.IsNull()) {
            value.primaryColor = nil;
        } else {
            value.primaryColor = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue.primaryColor.Value())];
        }
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForSwitchCluster(AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::Switch;
    switch (aAttributeId) {
    case Attributes::NumberOfPositions::Id: {
        using TypeInfo = Attributes::NumberOfPositions::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::CurrentPosition::Id: {
        using TypeInfo = Attributes::CurrentPosition::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::MultiPressMax::Id: {
        using TypeInfo = Attributes::MultiPressMax::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForAdministratorCommissioningCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::AdministratorCommissioning;
    switch (aAttributeId) {
    case Attributes::WindowStatus::Id: {
        using TypeInfo = Attributes::WindowStatus::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::AdminFabricIndex::Id: {
        using TypeInfo = Attributes::AdminFabricIndex::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AdminVendorId::Id: {
        using TypeInfo = Attributes::AdminVendorId::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForOperationalCredentialsCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::OperationalCredentials;
    switch (aAttributeId) {
    case Attributes::NOCs::Id: {
        using TypeInfo = Attributes::NOCs::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                MTROperationalCredentialsClusterNOCStruct * newElement_0;
                newElement_0 = [MTROperationalCredentialsClusterNOCStruct new];
                newElement_0.noc = AsData(entry_0.noc);
                if (entry_0.icac.IsNull()) {
                    newElement_0.icac = nil;
                } else {
                    newElement_0.icac = AsData(entry_0.icac.Value());
                }
                newElement_0.fabricIndex = [NSNumber numberWithUnsignedChar:entry_0.fabricIndex];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::Fabrics::Id: {
        using TypeInfo = Attributes::Fabrics::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                MTROperationalCredentialsClusterFabricDescriptorStruct * newElement_0;
                newElement_0 = [MTROperationalCredentialsClusterFabricDescriptorStruct new];
                newElement_0.rootPublicKey = AsData(entry_0.rootPublicKey);
                newElement_0.vendorID = [NSNumber numberWithUnsignedShort:chip::to_underlying(entry_0.vendorID)];
                newElement_0.fabricID = [NSNumber numberWithUnsignedLongLong:entry_0.fabricID];
                newElement_0.nodeID = [NSNumber numberWithUnsignedLongLong:entry_0.nodeID];
                newElement_0.label = AsString(entry_0.label);
                if (newElement_0.label == nil) {
                    CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
                    *aError = err;
                    return nil;
                }
                newElement_0.fabricIndex = [NSNumber numberWithUnsignedChar:entry_0.fabricIndex];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::SupportedFabrics::Id: {
        using TypeInfo = Attributes::SupportedFabrics::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::CommissionedFabrics::Id: {
        using TypeInfo = Attributes::CommissionedFabrics::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::TrustedRootCertificates::Id: {
        using TypeInfo = Attributes::TrustedRootCertificates::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSData * newElement_0;
                newElement_0 = AsData(entry_0);
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::CurrentFabricIndex::Id: {
        using TypeInfo = Attributes::CurrentFabricIndex::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForGroupKeyManagementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::GroupKeyManagement;
    switch (aAttributeId) {
    case Attributes::GroupKeyMap::Id: {
        using TypeInfo = Attributes::GroupKeyMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                MTRGroupKeyManagementClusterGroupKeyMapStruct * newElement_0;
                newElement_0 = [MTRGroupKeyManagementClusterGroupKeyMapStruct new];
                newElement_0.groupId = [NSNumber numberWithUnsignedShort:entry_0.groupId];
                newElement_0.groupKeySetID = [NSNumber numberWithUnsignedShort:entry_0.groupKeySetID];
                newElement_0.fabricIndex = [NSNumber numberWithUnsignedChar:entry_0.fabricIndex];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::GroupTable::Id: {
        using TypeInfo = Attributes::GroupTable::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                MTRGroupKeyManagementClusterGroupInfoMapStruct * newElement_0;
                newElement_0 = [MTRGroupKeyManagementClusterGroupInfoMapStruct new];
                newElement_0.groupId = [NSNumber numberWithUnsignedShort:entry_0.groupId];
                { // Scope for our temporary variables
                    auto * array_2 = [NSMutableArray new];
                    auto iter_2 = entry_0.endpoints.begin();
                    while (iter_2.Next()) {
                        auto & entry_2 = iter_2.GetValue();
                        NSNumber * newElement_2;
                        newElement_2 = [NSNumber numberWithUnsignedShort:entry_2];
                        [array_2 addObject:newElement_2];
                    }
                    CHIP_ERROR err = iter_2.GetStatus();
                    if (err != CHIP_NO_ERROR) {
                        *aError = err;
                        return nil;
                    }
                    newElement_0.endpoints = array_2;
                }
                if (entry_0.groupName.HasValue()) {
                    newElement_0.groupName = AsString(entry_0.groupName.Value());
                    if (newElement_0.groupName == nil) {
                        CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
                        *aError = err;
                        return nil;
                    }
                } else {
                    newElement_0.groupName = nil;
                }
                newElement_0.fabricIndex = [NSNumber numberWithUnsignedChar:entry_0.fabricIndex];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::MaxGroupsPerFabric::Id: {
        using TypeInfo = Attributes::MaxGroupsPerFabric::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::MaxGroupKeysPerFabric::Id: {
        using TypeInfo = Attributes::MaxGroupKeysPerFabric::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForFixedLabelCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::FixedLabel;
    switch (aAttributeId) {
    case Attributes::LabelList::Id: {
        using TypeInfo = Attributes::LabelList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                MTRFixedLabelClusterLabelStruct * newElement_0;
                newElement_0 = [MTRFixedLabelClusterLabelStruct new];
                newElement_0.label = AsString(entry_0.label);
                if (newElement_0.label == nil) {
                    CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
                    *aError = err;
                    return nil;
                }
                newElement_0.value = AsString(entry_0.value);
                if (newElement_0.value == nil) {
                    CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
                    *aError = err;
                    return nil;
                }
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForUserLabelCluster(AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::UserLabel;
    switch (aAttributeId) {
    case Attributes::LabelList::Id: {
        using TypeInfo = Attributes::LabelList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                MTRUserLabelClusterLabelStruct * newElement_0;
                newElement_0 = [MTRUserLabelClusterLabelStruct new];
                newElement_0.label = AsString(entry_0.label);
                if (newElement_0.label == nil) {
                    CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
                    *aError = err;
                    return nil;
                }
                newElement_0.value = AsString(entry_0.value);
                if (newElement_0.value == nil) {
                    CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
                    *aError = err;
                    return nil;
                }
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForBooleanStateCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::BooleanState;
    switch (aAttributeId) {
    case Attributes::StateValue::Id: {
        using TypeInfo = Attributes::StateValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForModeSelectCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::ModeSelect;
    switch (aAttributeId) {
    case Attributes::Description::Id: {
        using TypeInfo = Attributes::Description::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::StandardNamespace::Id: {
        using TypeInfo = Attributes::StandardNamespace::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::SupportedModes::Id: {
        using TypeInfo = Attributes::SupportedModes::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                MTRModeSelectClusterModeOptionStruct * newElement_0;
                newElement_0 = [MTRModeSelectClusterModeOptionStruct new];
                newElement_0.label = AsString(entry_0.label);
                if (newElement_0.label == nil) {
                    CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
                    *aError = err;
                    return nil;
                }
                newElement_0.mode = [NSNumber numberWithUnsignedChar:entry_0.mode];
                { // Scope for our temporary variables
                    auto * array_2 = [NSMutableArray new];
                    auto iter_2 = entry_0.semanticTags.begin();
                    while (iter_2.Next()) {
                        auto & entry_2 = iter_2.GetValue();
                        MTRModeSelectClusterSemanticTagStruct * newElement_2;
                        newElement_2 = [MTRModeSelectClusterSemanticTagStruct new];
                        newElement_2.mfgCode = [NSNumber numberWithUnsignedShort:chip::to_underlying(entry_2.mfgCode)];
                        newElement_2.value = [NSNumber numberWithUnsignedShort:entry_2.value];
                        [array_2 addObject:newElement_2];
                    }
                    CHIP_ERROR err = iter_2.GetStatus();
                    if (err != CHIP_NO_ERROR) {
                        *aError = err;
                        return nil;
                    }
                    newElement_0.semanticTags = array_2;
                }
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::CurrentMode::Id: {
        using TypeInfo = Attributes::CurrentMode::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::StartUpMode::Id: {
        using TypeInfo = Attributes::StartUpMode::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::OnMode::Id: {
        using TypeInfo = Attributes::OnMode::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForTemperatureControlCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::TemperatureControl;
    switch (aAttributeId) {
    case Attributes::TemperatureSetpoint::Id: {
        using TypeInfo = Attributes::TemperatureSetpoint::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithShort:cppValue];
        return value;
    }
    case Attributes::MinTemperature::Id: {
        using TypeInfo = Attributes::MinTemperature::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithShort:cppValue];
        return value;
    }
    case Attributes::MaxTemperature::Id: {
        using TypeInfo = Attributes::MaxTemperature::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithShort:cppValue];
        return value;
    }
    case Attributes::Step::Id: {
        using TypeInfo = Attributes::Step::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithShort:cppValue];
        return value;
    }
    case Attributes::CurrentTemperatureLevelIndex::Id: {
        using TypeInfo = Attributes::CurrentTemperatureLevelIndex::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::SupportedTemperatureLevels::Id: {
        using TypeInfo = Attributes::SupportedTemperatureLevels::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                MTRTemperatureControlClusterTemperatureLevelStruct * newElement_0;
                newElement_0 = [MTRTemperatureControlClusterTemperatureLevelStruct new];
                newElement_0.label = AsString(entry_0.label);
                if (newElement_0.label == nil) {
                    CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
                    *aError = err;
                    return nil;
                }
                newElement_0.temperatureLevel = [NSNumber numberWithUnsignedChar:entry_0.temperatureLevel];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForRefrigeratorAlarmCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::RefrigeratorAlarm;
    switch (aAttributeId) {
    case Attributes::Mask::Id: {
        using TypeInfo = Attributes::Mask::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue.Raw()];
        return value;
    }
    case Attributes::Latch::Id: {
        using TypeInfo = Attributes::Latch::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue.Raw()];
        return value;
    }
    case Attributes::State::Id: {
        using TypeInfo = Attributes::State::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue.Raw()];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForAirQualityCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::AirQuality;
    switch (aAttributeId) {
    case Attributes::AirQuality::Id: {
        using TypeInfo = Attributes::AirQuality::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForSmokeCOAlarmCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::SmokeCoAlarm;
    switch (aAttributeId) {
    case Attributes::ExpressedState::Id: {
        using TypeInfo = Attributes::ExpressedState::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::SmokeState::Id: {
        using TypeInfo = Attributes::SmokeState::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::COState::Id: {
        using TypeInfo = Attributes::COState::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::BatteryAlert::Id: {
        using TypeInfo = Attributes::BatteryAlert::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::DeviceMuted::Id: {
        using TypeInfo = Attributes::DeviceMuted::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::TestInProgress::Id: {
        using TypeInfo = Attributes::TestInProgress::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::HardwareFaultAlert::Id: {
        using TypeInfo = Attributes::HardwareFaultAlert::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::EndOfServiceAlert::Id: {
        using TypeInfo = Attributes::EndOfServiceAlert::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::InterconnectSmokeAlarm::Id: {
        using TypeInfo = Attributes::InterconnectSmokeAlarm::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::InterconnectCOAlarm::Id: {
        using TypeInfo = Attributes::InterconnectCOAlarm::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::ContaminationState::Id: {
        using TypeInfo = Attributes::ContaminationState::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::SensitivityLevel::Id: {
        using TypeInfo = Attributes::SensitivityLevel::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForOperationalStateCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::OperationalState;
    switch (aAttributeId) {
    case Attributes::PhaseList::Id: {
        using TypeInfo = Attributes::PhaseList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            { // Scope for our temporary variables
                auto * array_1 = [NSMutableArray new];
                auto iter_1 = cppValue.Value().begin();
                while (iter_1.Next()) {
                    auto & entry_1 = iter_1.GetValue();
                    NSString * newElement_1;
                    newElement_1 = AsString(entry_1);
                    if (newElement_1 == nil) {
                        CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
                        *aError = err;
                        return nil;
                    }
                    [array_1 addObject:newElement_1];
                }
                CHIP_ERROR err = iter_1.GetStatus();
                if (err != CHIP_NO_ERROR) {
                    *aError = err;
                    return nil;
                }
                value = array_1;
            }
        }
        return value;
    }
    case Attributes::CurrentPhase::Id: {
        using TypeInfo = Attributes::CurrentPhase::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::CountdownTime::Id: {
        using TypeInfo = Attributes::CountdownTime::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedInt:cppValue.Value()];
        }
        return value;
    }
    case Attributes::OperationalStateList::Id: {
        using TypeInfo = Attributes::OperationalStateList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                MTROperationalStateClusterOperationalStateStruct * newElement_0;
                newElement_0 = [MTROperationalStateClusterOperationalStateStruct new];
                newElement_0.operationalStateID = [NSNumber numberWithUnsignedChar:chip::to_underlying(entry_0.operationalStateID)];
                newElement_0.operationalStateLabel = AsString(entry_0.operationalStateLabel);
                if (newElement_0.operationalStateLabel == nil) {
                    CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
                    *aError = err;
                    return nil;
                }
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::OperationalState::Id: {
        using TypeInfo = Attributes::OperationalState::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        MTROperationalStateClusterOperationalStateStruct * _Nonnull value;
        value = [MTROperationalStateClusterOperationalStateStruct new];
        value.operationalStateID = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue.operationalStateID)];
        value.operationalStateLabel = AsString(cppValue.operationalStateLabel);
        if (value.operationalStateLabel == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::OperationalError::Id: {
        using TypeInfo = Attributes::OperationalError::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        MTROperationalStateClusterErrorStateStruct * _Nonnull value;
        value = [MTROperationalStateClusterErrorStateStruct new];
        value.errorStateID = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue.errorStateID)];
        if (cppValue.errorStateLabel.IsNull()) {
            value.errorStateLabel = nil;
        } else {
            value.errorStateLabel = AsString(cppValue.errorStateLabel.Value());
            if (value.errorStateLabel == nil) {
                CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
                *aError = err;
                return nil;
            }
        }
        if (cppValue.errorStateDetails.HasValue()) {
            value.errorStateDetails = AsString(cppValue.errorStateDetails.Value());
            if (value.errorStateDetails == nil) {
                CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
                *aError = err;
                return nil;
            }
        } else {
            value.errorStateDetails = nil;
        }
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForHEPAFilterMonitoringCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::HepaFilterMonitoring;
    switch (aAttributeId) {
    case Attributes::Condition::Id: {
        using TypeInfo = Attributes::Condition::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::DegradationDirection::Id: {
        using TypeInfo = Attributes::DegradationDirection::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::ChangeIndication::Id: {
        using TypeInfo = Attributes::ChangeIndication::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::InPlaceIndicator::Id: {
        using TypeInfo = Attributes::InPlaceIndicator::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForActivatedCarbonFilterMonitoringCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::ActivatedCarbonFilterMonitoring;
    switch (aAttributeId) {
    case Attributes::Condition::Id: {
        using TypeInfo = Attributes::Condition::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::DegradationDirection::Id: {
        using TypeInfo = Attributes::DegradationDirection::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::ChangeIndication::Id: {
        using TypeInfo = Attributes::ChangeIndication::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::InPlaceIndicator::Id: {
        using TypeInfo = Attributes::InPlaceIndicator::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForCeramicFilterMonitoringCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::CeramicFilterMonitoring;
    switch (aAttributeId) {
    case Attributes::Condition::Id: {
        using TypeInfo = Attributes::Condition::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::DegradationDirection::Id: {
        using TypeInfo = Attributes::DegradationDirection::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::ChangeIndication::Id: {
        using TypeInfo = Attributes::ChangeIndication::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::InPlaceIndicator::Id: {
        using TypeInfo = Attributes::InPlaceIndicator::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForElectrostaticFilterMonitoringCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::ElectrostaticFilterMonitoring;
    switch (aAttributeId) {
    case Attributes::Condition::Id: {
        using TypeInfo = Attributes::Condition::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::DegradationDirection::Id: {
        using TypeInfo = Attributes::DegradationDirection::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::ChangeIndication::Id: {
        using TypeInfo = Attributes::ChangeIndication::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::InPlaceIndicator::Id: {
        using TypeInfo = Attributes::InPlaceIndicator::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForUVFilterMonitoringCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::UvFilterMonitoring;
    switch (aAttributeId) {
    case Attributes::Condition::Id: {
        using TypeInfo = Attributes::Condition::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::DegradationDirection::Id: {
        using TypeInfo = Attributes::DegradationDirection::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::ChangeIndication::Id: {
        using TypeInfo = Attributes::ChangeIndication::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::InPlaceIndicator::Id: {
        using TypeInfo = Attributes::InPlaceIndicator::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForIonizingFilterMonitoringCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::IonizingFilterMonitoring;
    switch (aAttributeId) {
    case Attributes::Condition::Id: {
        using TypeInfo = Attributes::Condition::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::DegradationDirection::Id: {
        using TypeInfo = Attributes::DegradationDirection::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::ChangeIndication::Id: {
        using TypeInfo = Attributes::ChangeIndication::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::InPlaceIndicator::Id: {
        using TypeInfo = Attributes::InPlaceIndicator::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForZeoliteFilterMonitoringCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::ZeoliteFilterMonitoring;
    switch (aAttributeId) {
    case Attributes::Condition::Id: {
        using TypeInfo = Attributes::Condition::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::DegradationDirection::Id: {
        using TypeInfo = Attributes::DegradationDirection::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::ChangeIndication::Id: {
        using TypeInfo = Attributes::ChangeIndication::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::InPlaceIndicator::Id: {
        using TypeInfo = Attributes::InPlaceIndicator::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForOzoneFilterMonitoringCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::OzoneFilterMonitoring;
    switch (aAttributeId) {
    case Attributes::Condition::Id: {
        using TypeInfo = Attributes::Condition::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::DegradationDirection::Id: {
        using TypeInfo = Attributes::DegradationDirection::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::ChangeIndication::Id: {
        using TypeInfo = Attributes::ChangeIndication::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::InPlaceIndicator::Id: {
        using TypeInfo = Attributes::InPlaceIndicator::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForWaterTankMonitoringCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::WaterTankMonitoring;
    switch (aAttributeId) {
    case Attributes::Condition::Id: {
        using TypeInfo = Attributes::Condition::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::DegradationDirection::Id: {
        using TypeInfo = Attributes::DegradationDirection::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::ChangeIndication::Id: {
        using TypeInfo = Attributes::ChangeIndication::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::InPlaceIndicator::Id: {
        using TypeInfo = Attributes::InPlaceIndicator::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForFuelTankMonitoringCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::FuelTankMonitoring;
    switch (aAttributeId) {
    case Attributes::Condition::Id: {
        using TypeInfo = Attributes::Condition::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::DegradationDirection::Id: {
        using TypeInfo = Attributes::DegradationDirection::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::ChangeIndication::Id: {
        using TypeInfo = Attributes::ChangeIndication::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::InPlaceIndicator::Id: {
        using TypeInfo = Attributes::InPlaceIndicator::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForInkCartridgeMonitoringCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::InkCartridgeMonitoring;
    switch (aAttributeId) {
    case Attributes::Condition::Id: {
        using TypeInfo = Attributes::Condition::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::DegradationDirection::Id: {
        using TypeInfo = Attributes::DegradationDirection::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::ChangeIndication::Id: {
        using TypeInfo = Attributes::ChangeIndication::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::InPlaceIndicator::Id: {
        using TypeInfo = Attributes::InPlaceIndicator::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForTonerCartridgeMonitoringCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::TonerCartridgeMonitoring;
    switch (aAttributeId) {
    case Attributes::Condition::Id: {
        using TypeInfo = Attributes::Condition::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::DegradationDirection::Id: {
        using TypeInfo = Attributes::DegradationDirection::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::ChangeIndication::Id: {
        using TypeInfo = Attributes::ChangeIndication::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::InPlaceIndicator::Id: {
        using TypeInfo = Attributes::InPlaceIndicator::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForDoorLockCluster(AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::DoorLock;
    switch (aAttributeId) {
    case Attributes::LockState::Id: {
        using TypeInfo = Attributes::LockState::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue.Value())];
        }
        return value;
    }
    case Attributes::LockType::Id: {
        using TypeInfo = Attributes::LockType::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::ActuatorEnabled::Id: {
        using TypeInfo = Attributes::ActuatorEnabled::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::DoorState::Id: {
        using TypeInfo = Attributes::DoorState::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue.Value())];
        }
        return value;
    }
    case Attributes::DoorOpenEvents::Id: {
        using TypeInfo = Attributes::DoorOpenEvents::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::DoorClosedEvents::Id: {
        using TypeInfo = Attributes::DoorClosedEvents::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::OpenPeriod::Id: {
        using TypeInfo = Attributes::OpenPeriod::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::NumberOfTotalUsersSupported::Id: {
        using TypeInfo = Attributes::NumberOfTotalUsersSupported::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::NumberOfPINUsersSupported::Id: {
        using TypeInfo = Attributes::NumberOfPINUsersSupported::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::NumberOfRFIDUsersSupported::Id: {
        using TypeInfo = Attributes::NumberOfRFIDUsersSupported::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::NumberOfWeekDaySchedulesSupportedPerUser::Id: {
        using TypeInfo = Attributes::NumberOfWeekDaySchedulesSupportedPerUser::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::NumberOfYearDaySchedulesSupportedPerUser::Id: {
        using TypeInfo = Attributes::NumberOfYearDaySchedulesSupportedPerUser::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::NumberOfHolidaySchedulesSupported::Id: {
        using TypeInfo = Attributes::NumberOfHolidaySchedulesSupported::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::MaxPINCodeLength::Id: {
        using TypeInfo = Attributes::MaxPINCodeLength::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::MinPINCodeLength::Id: {
        using TypeInfo = Attributes::MinPINCodeLength::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::MaxRFIDCodeLength::Id: {
        using TypeInfo = Attributes::MaxRFIDCodeLength::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::MinRFIDCodeLength::Id: {
        using TypeInfo = Attributes::MinRFIDCodeLength::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::CredentialRulesSupport::Id: {
        using TypeInfo = Attributes::CredentialRulesSupport::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue.Raw()];
        return value;
    }
    case Attributes::NumberOfCredentialsSupportedPerUser::Id: {
        using TypeInfo = Attributes::NumberOfCredentialsSupportedPerUser::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::Language::Id: {
        using TypeInfo = Attributes::Language::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::LEDSettings::Id: {
        using TypeInfo = Attributes::LEDSettings::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::AutoRelockTime::Id: {
        using TypeInfo = Attributes::AutoRelockTime::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::SoundVolume::Id: {
        using TypeInfo = Attributes::SoundVolume::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::OperatingMode::Id: {
        using TypeInfo = Attributes::OperatingMode::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::SupportedOperatingModes::Id: {
        using TypeInfo = Attributes::SupportedOperatingModes::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue.Raw()];
        return value;
    }
    case Attributes::DefaultConfigurationRegister::Id: {
        using TypeInfo = Attributes::DefaultConfigurationRegister::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue.Raw()];
        return value;
    }
    case Attributes::EnableLocalProgramming::Id: {
        using TypeInfo = Attributes::EnableLocalProgramming::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::EnableOneTouchLocking::Id: {
        using TypeInfo = Attributes::EnableOneTouchLocking::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::EnableInsideStatusLED::Id: {
        using TypeInfo = Attributes::EnableInsideStatusLED::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::EnablePrivacyModeButton::Id: {
        using TypeInfo = Attributes::EnablePrivacyModeButton::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::LocalProgrammingFeatures::Id: {
        using TypeInfo = Attributes::LocalProgrammingFeatures::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue.Raw()];
        return value;
    }
    case Attributes::WrongCodeEntryLimit::Id: {
        using TypeInfo = Attributes::WrongCodeEntryLimit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::UserCodeTemporaryDisableTime::Id: {
        using TypeInfo = Attributes::UserCodeTemporaryDisableTime::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::SendPINOverTheAir::Id: {
        using TypeInfo = Attributes::SendPINOverTheAir::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::RequirePINforRemoteOperation::Id: {
        using TypeInfo = Attributes::RequirePINforRemoteOperation::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithBool:cppValue];
        return value;
    }
    case Attributes::ExpiringUserTimeout::Id: {
        using TypeInfo = Attributes::ExpiringUserTimeout::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForWindowCoveringCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::WindowCovering;
    switch (aAttributeId) {
    case Attributes::Type::Id: {
        using TypeInfo = Attributes::Type::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::PhysicalClosedLimitLift::Id: {
        using TypeInfo = Attributes::PhysicalClosedLimitLift::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::PhysicalClosedLimitTilt::Id: {
        using TypeInfo = Attributes::PhysicalClosedLimitTilt::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::CurrentPositionLift::Id: {
        using TypeInfo = Attributes::CurrentPositionLift::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::CurrentPositionTilt::Id: {
        using TypeInfo = Attributes::CurrentPositionTilt::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::NumberOfActuationsLift::Id: {
        using TypeInfo = Attributes::NumberOfActuationsLift::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::NumberOfActuationsTilt::Id: {
        using TypeInfo = Attributes::NumberOfActuationsTilt::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::ConfigStatus::Id: {
        using TypeInfo = Attributes::ConfigStatus::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue.Raw()];
        return value;
    }
    case Attributes::CurrentPositionLiftPercentage::Id: {
        using TypeInfo = Attributes::CurrentPositionLiftPercentage::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::CurrentPositionTiltPercentage::Id: {
        using TypeInfo = Attributes::CurrentPositionTiltPercentage::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::OperationalStatus::Id: {
        using TypeInfo = Attributes::OperationalStatus::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue.Raw()];
        return value;
    }
    case Attributes::TargetPositionLiftPercent100ths::Id: {
        using TypeInfo = Attributes::TargetPositionLiftPercent100ths::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::TargetPositionTiltPercent100ths::Id: {
        using TypeInfo = Attributes::TargetPositionTiltPercent100ths::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::EndProductType::Id: {
        using TypeInfo = Attributes::EndProductType::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::CurrentPositionLiftPercent100ths::Id: {
        using TypeInfo = Attributes::CurrentPositionLiftPercent100ths::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::CurrentPositionTiltPercent100ths::Id: {
        using TypeInfo = Attributes::CurrentPositionTiltPercent100ths::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::InstalledOpenLimitLift::Id: {
        using TypeInfo = Attributes::InstalledOpenLimitLift::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::InstalledClosedLimitLift::Id: {
        using TypeInfo = Attributes::InstalledClosedLimitLift::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::InstalledOpenLimitTilt::Id: {
        using TypeInfo = Attributes::InstalledOpenLimitTilt::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::InstalledClosedLimitTilt::Id: {
        using TypeInfo = Attributes::InstalledClosedLimitTilt::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::Mode::Id: {
        using TypeInfo = Attributes::Mode::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue.Raw()];
        return value;
    }
    case Attributes::SafetyStatus::Id: {
        using TypeInfo = Attributes::SafetyStatus::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue.Raw()];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForBarrierControlCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::BarrierControl;
    switch (aAttributeId) {
    case Attributes::BarrierMovingState::Id: {
        using TypeInfo = Attributes::BarrierMovingState::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::BarrierSafetyStatus::Id: {
        using TypeInfo = Attributes::BarrierSafetyStatus::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::BarrierCapabilities::Id: {
        using TypeInfo = Attributes::BarrierCapabilities::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::BarrierOpenEvents::Id: {
        using TypeInfo = Attributes::BarrierOpenEvents::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::BarrierCloseEvents::Id: {
        using TypeInfo = Attributes::BarrierCloseEvents::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::BarrierCommandOpenEvents::Id: {
        using TypeInfo = Attributes::BarrierCommandOpenEvents::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::BarrierCommandCloseEvents::Id: {
        using TypeInfo = Attributes::BarrierCommandCloseEvents::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::BarrierOpenPeriod::Id: {
        using TypeInfo = Attributes::BarrierOpenPeriod::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::BarrierClosePeriod::Id: {
        using TypeInfo = Attributes::BarrierClosePeriod::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::BarrierPosition::Id: {
        using TypeInfo = Attributes::BarrierPosition::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForPumpConfigurationAndControlCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::PumpConfigurationAndControl;
    switch (aAttributeId) {
    case Attributes::MaxPressure::Id: {
        using TypeInfo = Attributes::MaxPressure::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxSpeed::Id: {
        using TypeInfo = Attributes::MaxSpeed::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxFlow::Id: {
        using TypeInfo = Attributes::MaxFlow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinConstPressure::Id: {
        using TypeInfo = Attributes::MinConstPressure::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxConstPressure::Id: {
        using TypeInfo = Attributes::MaxConstPressure::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinCompPressure::Id: {
        using TypeInfo = Attributes::MinCompPressure::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxCompPressure::Id: {
        using TypeInfo = Attributes::MaxCompPressure::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinConstSpeed::Id: {
        using TypeInfo = Attributes::MinConstSpeed::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxConstSpeed::Id: {
        using TypeInfo = Attributes::MaxConstSpeed::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinConstFlow::Id: {
        using TypeInfo = Attributes::MinConstFlow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxConstFlow::Id: {
        using TypeInfo = Attributes::MaxConstFlow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinConstTemp::Id: {
        using TypeInfo = Attributes::MinConstTemp::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxConstTemp::Id: {
        using TypeInfo = Attributes::MaxConstTemp::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PumpStatus::Id: {
        using TypeInfo = Attributes::PumpStatus::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue.Raw()];
        return value;
    }
    case Attributes::EffectiveOperationMode::Id: {
        using TypeInfo = Attributes::EffectiveOperationMode::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::EffectiveControlMode::Id: {
        using TypeInfo = Attributes::EffectiveControlMode::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::Capacity::Id: {
        using TypeInfo = Attributes::Capacity::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::Speed::Id: {
        using TypeInfo = Attributes::Speed::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::LifetimeRunningHours::Id: {
        using TypeInfo = Attributes::LifetimeRunningHours::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedInt:cppValue.Value()];
        }
        return value;
    }
    case Attributes::Power::Id: {
        using TypeInfo = Attributes::Power::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedInt:cppValue.Value()];
        }
        return value;
    }
    case Attributes::LifetimeEnergyConsumed::Id: {
        using TypeInfo = Attributes::LifetimeEnergyConsumed::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedInt:cppValue.Value()];
        }
        return value;
    }
    case Attributes::OperationMode::Id: {
        using TypeInfo = Attributes::OperationMode::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::ControlMode::Id: {
        using TypeInfo = Attributes::ControlMode::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForThermostatCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::Thermostat;
    switch (aAttributeId) {
    case Attributes::LocalTemperature::Id: {
        using TypeInfo = Attributes::LocalTemperature::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::OutdoorTemperature::Id: {
        using TypeInfo = Attributes::OutdoorTemperature::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::Occupancy::Id: {
        using TypeInfo = Attributes::Occupancy::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::AbsMinHeatSetpointLimit::Id: {
        using TypeInfo = Attributes::AbsMinHeatSetpointLimit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithShort:cppValue];
        return value;
    }
    case Attributes::AbsMaxHeatSetpointLimit::Id: {
        using TypeInfo = Attributes::AbsMaxHeatSetpointLimit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithShort:cppValue];
        return value;
    }
    case Attributes::AbsMinCoolSetpointLimit::Id: {
        using TypeInfo = Attributes::AbsMinCoolSetpointLimit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithShort:cppValue];
        return value;
    }
    case Attributes::AbsMaxCoolSetpointLimit::Id: {
        using TypeInfo = Attributes::AbsMaxCoolSetpointLimit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithShort:cppValue];
        return value;
    }
    case Attributes::PICoolingDemand::Id: {
        using TypeInfo = Attributes::PICoolingDemand::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::PIHeatingDemand::Id: {
        using TypeInfo = Attributes::PIHeatingDemand::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::HVACSystemTypeConfiguration::Id: {
        using TypeInfo = Attributes::HVACSystemTypeConfiguration::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::LocalTemperatureCalibration::Id: {
        using TypeInfo = Attributes::LocalTemperatureCalibration::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithChar:cppValue];
        return value;
    }
    case Attributes::OccupiedCoolingSetpoint::Id: {
        using TypeInfo = Attributes::OccupiedCoolingSetpoint::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithShort:cppValue];
        return value;
    }
    case Attributes::OccupiedHeatingSetpoint::Id: {
        using TypeInfo = Attributes::OccupiedHeatingSetpoint::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithShort:cppValue];
        return value;
    }
    case Attributes::UnoccupiedCoolingSetpoint::Id: {
        using TypeInfo = Attributes::UnoccupiedCoolingSetpoint::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithShort:cppValue];
        return value;
    }
    case Attributes::UnoccupiedHeatingSetpoint::Id: {
        using TypeInfo = Attributes::UnoccupiedHeatingSetpoint::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithShort:cppValue];
        return value;
    }
    case Attributes::MinHeatSetpointLimit::Id: {
        using TypeInfo = Attributes::MinHeatSetpointLimit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithShort:cppValue];
        return value;
    }
    case Attributes::MaxHeatSetpointLimit::Id: {
        using TypeInfo = Attributes::MaxHeatSetpointLimit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithShort:cppValue];
        return value;
    }
    case Attributes::MinCoolSetpointLimit::Id: {
        using TypeInfo = Attributes::MinCoolSetpointLimit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithShort:cppValue];
        return value;
    }
    case Attributes::MaxCoolSetpointLimit::Id: {
        using TypeInfo = Attributes::MaxCoolSetpointLimit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithShort:cppValue];
        return value;
    }
    case Attributes::MinSetpointDeadBand::Id: {
        using TypeInfo = Attributes::MinSetpointDeadBand::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithChar:cppValue];
        return value;
    }
    case Attributes::RemoteSensing::Id: {
        using TypeInfo = Attributes::RemoteSensing::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::ControlSequenceOfOperation::Id: {
        using TypeInfo = Attributes::ControlSequenceOfOperation::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::SystemMode::Id: {
        using TypeInfo = Attributes::SystemMode::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::ThermostatRunningMode::Id: {
        using TypeInfo = Attributes::ThermostatRunningMode::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::StartOfWeek::Id: {
        using TypeInfo = Attributes::StartOfWeek::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::NumberOfWeeklyTransitions::Id: {
        using TypeInfo = Attributes::NumberOfWeeklyTransitions::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::NumberOfDailyTransitions::Id: {
        using TypeInfo = Attributes::NumberOfDailyTransitions::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::TemperatureSetpointHold::Id: {
        using TypeInfo = Attributes::TemperatureSetpointHold::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::TemperatureSetpointHoldDuration::Id: {
        using TypeInfo = Attributes::TemperatureSetpointHoldDuration::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::ThermostatProgrammingOperationMode::Id: {
        using TypeInfo = Attributes::ThermostatProgrammingOperationMode::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::ThermostatRunningState::Id: {
        using TypeInfo = Attributes::ThermostatRunningState::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::SetpointChangeSource::Id: {
        using TypeInfo = Attributes::SetpointChangeSource::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::SetpointChangeAmount::Id: {
        using TypeInfo = Attributes::SetpointChangeAmount::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::SetpointChangeSourceTimestamp::Id: {
        using TypeInfo = Attributes::SetpointChangeSourceTimestamp::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::OccupiedSetback::Id: {
        using TypeInfo = Attributes::OccupiedSetback::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::OccupiedSetbackMin::Id: {
        using TypeInfo = Attributes::OccupiedSetbackMin::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::OccupiedSetbackMax::Id: {
        using TypeInfo = Attributes::OccupiedSetbackMax::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::UnoccupiedSetback::Id: {
        using TypeInfo = Attributes::UnoccupiedSetback::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::UnoccupiedSetbackMin::Id: {
        using TypeInfo = Attributes::UnoccupiedSetbackMin::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::UnoccupiedSetbackMax::Id: {
        using TypeInfo = Attributes::UnoccupiedSetbackMax::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::EmergencyHeatDelta::Id: {
        using TypeInfo = Attributes::EmergencyHeatDelta::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::ACType::Id: {
        using TypeInfo = Attributes::ACType::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::ACCapacity::Id: {
        using TypeInfo = Attributes::ACCapacity::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::ACRefrigerantType::Id: {
        using TypeInfo = Attributes::ACRefrigerantType::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::ACCompressorType::Id: {
        using TypeInfo = Attributes::ACCompressorType::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::ACErrorCode::Id: {
        using TypeInfo = Attributes::ACErrorCode::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ACLouverPosition::Id: {
        using TypeInfo = Attributes::ACLouverPosition::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::ACCoilTemperature::Id: {
        using TypeInfo = Attributes::ACCoilTemperature::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::ACCapacityformat::Id: {
        using TypeInfo = Attributes::ACCapacityformat::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForFanControlCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::FanControl;
    switch (aAttributeId) {
    case Attributes::FanMode::Id: {
        using TypeInfo = Attributes::FanMode::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::FanModeSequence::Id: {
        using TypeInfo = Attributes::FanModeSequence::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::PercentSetting::Id: {
        using TypeInfo = Attributes::PercentSetting::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PercentCurrent::Id: {
        using TypeInfo = Attributes::PercentCurrent::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::SpeedMax::Id: {
        using TypeInfo = Attributes::SpeedMax::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::SpeedSetting::Id: {
        using TypeInfo = Attributes::SpeedSetting::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::SpeedCurrent::Id: {
        using TypeInfo = Attributes::SpeedCurrent::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::RockSupport::Id: {
        using TypeInfo = Attributes::RockSupport::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::RockSetting::Id: {
        using TypeInfo = Attributes::RockSetting::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::WindSupport::Id: {
        using TypeInfo = Attributes::WindSupport::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::WindSetting::Id: {
        using TypeInfo = Attributes::WindSetting::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::AirflowDirection::Id: {
        using TypeInfo = Attributes::AirflowDirection::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForThermostatUserInterfaceConfigurationCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::ThermostatUserInterfaceConfiguration;
    switch (aAttributeId) {
    case Attributes::TemperatureDisplayMode::Id: {
        using TypeInfo = Attributes::TemperatureDisplayMode::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::KeypadLockout::Id: {
        using TypeInfo = Attributes::KeypadLockout::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::ScheduleProgrammingVisibility::Id: {
        using TypeInfo = Attributes::ScheduleProgrammingVisibility::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForColorControlCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::ColorControl;
    switch (aAttributeId) {
    case Attributes::CurrentHue::Id: {
        using TypeInfo = Attributes::CurrentHue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::CurrentSaturation::Id: {
        using TypeInfo = Attributes::CurrentSaturation::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::RemainingTime::Id: {
        using TypeInfo = Attributes::RemainingTime::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::CurrentX::Id: {
        using TypeInfo = Attributes::CurrentX::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::CurrentY::Id: {
        using TypeInfo = Attributes::CurrentY::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::DriftCompensation::Id: {
        using TypeInfo = Attributes::DriftCompensation::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::CompensationText::Id: {
        using TypeInfo = Attributes::CompensationText::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::ColorTemperatureMireds::Id: {
        using TypeInfo = Attributes::ColorTemperatureMireds::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::ColorMode::Id: {
        using TypeInfo = Attributes::ColorMode::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::Options::Id: {
        using TypeInfo = Attributes::Options::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::NumberOfPrimaries::Id: {
        using TypeInfo = Attributes::NumberOfPrimaries::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::Primary1X::Id: {
        using TypeInfo = Attributes::Primary1X::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::Primary1Y::Id: {
        using TypeInfo = Attributes::Primary1Y::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::Primary1Intensity::Id: {
        using TypeInfo = Attributes::Primary1Intensity::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::Primary2X::Id: {
        using TypeInfo = Attributes::Primary2X::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::Primary2Y::Id: {
        using TypeInfo = Attributes::Primary2Y::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::Primary2Intensity::Id: {
        using TypeInfo = Attributes::Primary2Intensity::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::Primary3X::Id: {
        using TypeInfo = Attributes::Primary3X::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::Primary3Y::Id: {
        using TypeInfo = Attributes::Primary3Y::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::Primary3Intensity::Id: {
        using TypeInfo = Attributes::Primary3Intensity::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::Primary4X::Id: {
        using TypeInfo = Attributes::Primary4X::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::Primary4Y::Id: {
        using TypeInfo = Attributes::Primary4Y::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::Primary4Intensity::Id: {
        using TypeInfo = Attributes::Primary4Intensity::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::Primary5X::Id: {
        using TypeInfo = Attributes::Primary5X::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::Primary5Y::Id: {
        using TypeInfo = Attributes::Primary5Y::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::Primary5Intensity::Id: {
        using TypeInfo = Attributes::Primary5Intensity::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::Primary6X::Id: {
        using TypeInfo = Attributes::Primary6X::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::Primary6Y::Id: {
        using TypeInfo = Attributes::Primary6Y::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::Primary6Intensity::Id: {
        using TypeInfo = Attributes::Primary6Intensity::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::WhitePointX::Id: {
        using TypeInfo = Attributes::WhitePointX::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::WhitePointY::Id: {
        using TypeInfo = Attributes::WhitePointY::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::ColorPointRX::Id: {
        using TypeInfo = Attributes::ColorPointRX::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::ColorPointRY::Id: {
        using TypeInfo = Attributes::ColorPointRY::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::ColorPointRIntensity::Id: {
        using TypeInfo = Attributes::ColorPointRIntensity::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::ColorPointGX::Id: {
        using TypeInfo = Attributes::ColorPointGX::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::ColorPointGY::Id: {
        using TypeInfo = Attributes::ColorPointGY::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::ColorPointGIntensity::Id: {
        using TypeInfo = Attributes::ColorPointGIntensity::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::ColorPointBX::Id: {
        using TypeInfo = Attributes::ColorPointBX::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::ColorPointBY::Id: {
        using TypeInfo = Attributes::ColorPointBY::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::ColorPointBIntensity::Id: {
        using TypeInfo = Attributes::ColorPointBIntensity::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::EnhancedCurrentHue::Id: {
        using TypeInfo = Attributes::EnhancedCurrentHue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::EnhancedColorMode::Id: {
        using TypeInfo = Attributes::EnhancedColorMode::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::ColorLoopActive::Id: {
        using TypeInfo = Attributes::ColorLoopActive::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::ColorLoopDirection::Id: {
        using TypeInfo = Attributes::ColorLoopDirection::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::ColorLoopTime::Id: {
        using TypeInfo = Attributes::ColorLoopTime::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::ColorLoopStartEnhancedHue::Id: {
        using TypeInfo = Attributes::ColorLoopStartEnhancedHue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::ColorLoopStoredEnhancedHue::Id: {
        using TypeInfo = Attributes::ColorLoopStoredEnhancedHue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::ColorCapabilities::Id: {
        using TypeInfo = Attributes::ColorCapabilities::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::ColorTempPhysicalMinMireds::Id: {
        using TypeInfo = Attributes::ColorTempPhysicalMinMireds::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::ColorTempPhysicalMaxMireds::Id: {
        using TypeInfo = Attributes::ColorTempPhysicalMaxMireds::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::CoupleColorTempToLevelMinMireds::Id: {
        using TypeInfo = Attributes::CoupleColorTempToLevelMinMireds::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::StartUpColorTemperatureMireds::Id: {
        using TypeInfo = Attributes::StartUpColorTemperatureMireds::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForBallastConfigurationCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::BallastConfiguration;
    switch (aAttributeId) {
    case Attributes::PhysicalMinLevel::Id: {
        using TypeInfo = Attributes::PhysicalMinLevel::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::PhysicalMaxLevel::Id: {
        using TypeInfo = Attributes::PhysicalMaxLevel::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::BallastStatus::Id: {
        using TypeInfo = Attributes::BallastStatus::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::MinLevel::Id: {
        using TypeInfo = Attributes::MinLevel::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::MaxLevel::Id: {
        using TypeInfo = Attributes::MaxLevel::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::IntrinsicBallastFactor::Id: {
        using TypeInfo = Attributes::IntrinsicBallastFactor::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::BallastFactorAdjustment::Id: {
        using TypeInfo = Attributes::BallastFactorAdjustment::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:cppValue.Value()];
        }
        return value;
    }
    case Attributes::LampQuantity::Id: {
        using TypeInfo = Attributes::LampQuantity::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::LampType::Id: {
        using TypeInfo = Attributes::LampType::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::LampManufacturer::Id: {
        using TypeInfo = Attributes::LampManufacturer::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSString * _Nonnull value;
        value = AsString(cppValue);
        if (value == nil) {
            CHIP_ERROR err = CHIP_ERROR_INVALID_ARGUMENT;
            *aError = err;
            return nil;
        }
        return value;
    }
    case Attributes::LampRatedHours::Id: {
        using TypeInfo = Attributes::LampRatedHours::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedInt:cppValue.Value()];
        }
        return value;
    }
    case Attributes::LampBurnHours::Id: {
        using TypeInfo = Attributes::LampBurnHours::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedInt:cppValue.Value()];
        }
        return value;
    }
    case Attributes::LampAlarmMode::Id: {
        using TypeInfo = Attributes::LampAlarmMode::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::LampBurnHoursTripPoint::Id: {
        using TypeInfo = Attributes::LampBurnHoursTripPoint::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedInt:cppValue.Value()];
        }
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForIlluminanceMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::IlluminanceMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::Tolerance::Id: {
        using TypeInfo = Attributes::Tolerance::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::LightSensorType::Id: {
        using TypeInfo = Attributes::LightSensorType::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue.Value())];
        }
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForTemperatureMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::TemperatureMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::Tolerance::Id: {
        using TypeInfo = Attributes::Tolerance::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForPressureMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::PressureMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::Tolerance::Id: {
        using TypeInfo = Attributes::Tolerance::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::ScaledValue::Id: {
        using TypeInfo = Attributes::ScaledValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinScaledValue::Id: {
        using TypeInfo = Attributes::MinScaledValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxScaledValue::Id: {
        using TypeInfo = Attributes::MaxScaledValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::ScaledTolerance::Id: {
        using TypeInfo = Attributes::ScaledTolerance::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::Scale::Id: {
        using TypeInfo = Attributes::Scale::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithChar:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForFlowMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::FlowMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::Tolerance::Id: {
        using TypeInfo = Attributes::Tolerance::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForRelativeHumidityMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::RelativeHumidityMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithUnsignedShort:cppValue.Value()];
        }
        return value;
    }
    case Attributes::Tolerance::Id: {
        using TypeInfo = Attributes::Tolerance::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForOccupancySensingCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::OccupancySensing;
    switch (aAttributeId) {
    case Attributes::Occupancy::Id: {
        using TypeInfo = Attributes::Occupancy::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue.Raw()];
        return value;
    }
    case Attributes::OccupancySensorType::Id: {
        using TypeInfo = Attributes::OccupancySensorType::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::OccupancySensorTypeBitmap::Id: {
        using TypeInfo = Attributes::OccupancySensorTypeBitmap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue.Raw()];
        return value;
    }
    case Attributes::PIROccupiedToUnoccupiedDelay::Id: {
        using TypeInfo = Attributes::PIROccupiedToUnoccupiedDelay::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::PIRUnoccupiedToOccupiedDelay::Id: {
        using TypeInfo = Attributes::PIRUnoccupiedToOccupiedDelay::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::PIRUnoccupiedToOccupiedThreshold::Id: {
        using TypeInfo = Attributes::PIRUnoccupiedToOccupiedThreshold::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::UltrasonicOccupiedToUnoccupiedDelay::Id: {
        using TypeInfo = Attributes::UltrasonicOccupiedToUnoccupiedDelay::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::UltrasonicUnoccupiedToOccupiedDelay::Id: {
        using TypeInfo = Attributes::UltrasonicUnoccupiedToOccupiedDelay::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::UltrasonicUnoccupiedToOccupiedThreshold::Id: {
        using TypeInfo = Attributes::UltrasonicUnoccupiedToOccupiedThreshold::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::PhysicalContactOccupiedToUnoccupiedDelay::Id: {
        using TypeInfo = Attributes::PhysicalContactOccupiedToUnoccupiedDelay::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::PhysicalContactUnoccupiedToOccupiedDelay::Id: {
        using TypeInfo = Attributes::PhysicalContactUnoccupiedToOccupiedDelay::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    case Attributes::PhysicalContactUnoccupiedToOccupiedThreshold::Id: {
        using TypeInfo = Attributes::PhysicalContactUnoccupiedToOccupiedThreshold::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:cppValue];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForCarbonMonoxideConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::CarbonMonoxideConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForCarbonDioxideConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::CarbonDioxideConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForEthyleneConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::EthyleneConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForEthyleneOxideConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::EthyleneOxideConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForHydrogenConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::HydrogenConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForHydrogenSulfideConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::HydrogenSulfideConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForNitricOxideConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::NitricOxideConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForNitrogenDioxideConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::NitrogenDioxideConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForOxygenConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::OxygenConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForOzoneConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::OzoneConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForSulfurDioxideConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::SulfurDioxideConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForDissolvedOxygenConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::DissolvedOxygenConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForBromateConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::BromateConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForChloraminesConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::ChloraminesConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForChlorineConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::ChlorineConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForFecalColiformEColiConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::FecalColiformEColiConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForFluorideConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::FluorideConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForHaloaceticAcidsConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::HaloaceticAcidsConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForTotalTrihalomethanesConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::TotalTrihalomethanesConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForTotalColiformBacteriaConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::TotalColiformBacteriaConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForTurbidityConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::TurbidityConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForCopperConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::CopperConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForLeadConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::LeadConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForManganeseConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::ManganeseConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForSulfateConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::SulfateConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForBromodichloromethaneConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::BromodichloromethaneConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForBromoformConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::BromoformConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForChlorodibromomethaneConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::ChlorodibromomethaneConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForChloroformConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::ChloroformConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForSodiumConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::SodiumConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForPM25ConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::Pm25ConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForFormaldehydeConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::FormaldehydeConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForPM1ConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::Pm1ConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForPM10ConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::Pm10ConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForTotalVolatileOrganicCompoundsConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::TotalVolatileOrganicCompoundsConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::GeneratedCommandList::Id: {
        using TypeInfo = Attributes::GeneratedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AcceptedCommandList::Id: {
        using TypeInfo = Attributes::AcceptedCommandList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::EventList::Id: {
        using TypeInfo = Attributes::EventList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::AttributeList::Id: {
        using TypeInfo = Attributes::AttributeList::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSArray * _Nonnull value;
        { // Scope for our temporary variables
            auto * array_0 = [NSMutableArray new];
            auto iter_0 = cppValue.begin();
            while (iter_0.Next()) {
                auto & entry_0 = iter_0.GetValue();
                NSNumber * newElement_0;
                newElement_0 = [NSNumber numberWithUnsignedInt:entry_0];
                [array_0 addObject:newElement_0];
            }
            CHIP_ERROR err = iter_0.GetStatus();
            if (err != CHIP_NO_ERROR) {
                *aError = err;
                return nil;
            }
            value = array_0;
        }
        return value;
    }
    case Attributes::FeatureMap::Id: {
        using TypeInfo = Attributes::FeatureMap::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::ClusterRevision::Id: {
        using TypeInfo = Attributes::ClusterRevision::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedShort:cppValue];
        return value;
    }
    default: {
        break;
    }
    }

    *aError = CHIP_ERROR_IM_MALFORMED_ATTRIBUTE_PATH_IB;
    return nil;
}
static id _Nullable DecodeAttributeValueForRadonConcentrationMeasurementCluster(
    AttributeId aAttributeId, TLV::TLVReader & aReader, CHIP_ERROR * aError)
{
    using namespace Clusters::RadonConcentrationMeasurement;
    switch (aAttributeId) {
    case Attributes::MeasuredValue::Id: {
        using TypeInfo = Attributes::MeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MinMeasuredValue::Id: {
        using TypeInfo = Attributes::MinMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::MaxMeasuredValue::Id: {
        using TypeInfo = Attributes::MaxMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValue::Id: {
        using TypeInfo = Attributes::PeakMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::PeakMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::PeakMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::AverageMeasuredValue::Id: {
        using TypeInfo = Attributes::AverageMeasuredValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nullable value;
        if (cppValue.IsNull()) {
            value = nil;
        } else {
            value = [NSNumber numberWithFloat:cppValue.Value()];
        }
        return value;
    }
    case Attributes::AverageMeasuredValueWindow::Id: {
        using TypeInfo = Attributes::AverageMeasuredValueWindow::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedInt:cppValue];
        return value;
    }
    case Attributes::Uncertainty::Id: {
        using TypeInfo = Attributes::Uncertainty::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithFloat:cppValue];
        return value;
    }
    case Attributes::MeasurementUnit::Id: {
        using TypeInfo = Attributes::MeasurementUnit::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::MeasurementMedium::Id: {
        using TypeInfo = Attributes::MeasurementMedium::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    case Attributes::LevelValue::Id: {
        using TypeInfo = Attributes::LevelValue::TypeInfo;
        TypeInfo::DecodableType cppValue;
        *aError = DataModel::Decode(aReader, cppValue);
        if (*aError != CHIP_NO_ERROR) {
            return nil;
        }
        NSNumber * _Nonnull value;
        value = [NSNumber numberWithUnsignedChar:chip::to_underlying(cppValue)];
        return value;
    }
    ca