/*
 *
 *    Copyright (c) 2022 Project CHIP Authors
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

#import <Matter/MTRCluster.h>
#import <Matter/MTRClusterStateCacheContainer.h>
#import <Matter/MTRCommandPayloadsObjc.h>
#import <Matter/MTRDefines.h>

NS_ASSUME_NONNULL_BEGIN

/**
 * Cluster Identify
 *
 * Attributes and commands for putting a device into Identification mode (e.g. flashing a light).
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterIdentify : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

/**
 * Command Identify
 *
 * Command description for Identify
 */
- (void)identifyWithParams:(MTRIdentifyClusterIdentifyParams *)params
                completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command TriggerEffect
 *
 * Command description for TriggerEffect
 */
- (void)triggerEffectWithParams:(MTRIdentifyClusterTriggerEffectParams *)params
                     completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeIdentifyTimeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeIdentifyTimeWithValue:(NSNumber * _Nonnull)value
                                 completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeIdentifyTimeWithValue:(NSNumber * _Nonnull)value
                                     params:(MTRWriteParams * _Nullable)params
                                 completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeIdentifyTimeWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeIdentifyTimeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeIdentifyTypeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeIdentifyTypeWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeIdentifyTypeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Groups
 *
 * Attributes and commands for group configuration and manipulation.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterGroups : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

/**
 * Command AddGroup
 *
 * Command description for AddGroup
 */
- (void)addGroupWithParams:(MTRGroupsClusterAddGroupParams *)params
                completion:(void (^)(MTRGroupsClusterAddGroupResponseParams * _Nullable data, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command ViewGroup
 *
 * Command description for ViewGroup
 */
- (void)viewGroupWithParams:(MTRGroupsClusterViewGroupParams *)params
                 completion:
                     (void (^)(MTRGroupsClusterViewGroupResponseParams * _Nullable data, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command GetGroupMembership
 *
 * Command description for GetGroupMembership
 */
- (void)getGroupMembershipWithParams:(MTRGroupsClusterGetGroupMembershipParams *)params
                          completion:(void (^)(MTRGroupsClusterGetGroupMembershipResponseParams * _Nullable data,
                                         NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command RemoveGroup
 *
 * Command description for RemoveGroup
 */
- (void)removeGroupWithParams:(MTRGroupsClusterRemoveGroupParams *)params
                   completion:
                       (void (^)(MTRGroupsClusterRemoveGroupResponseParams * _Nullable data, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command RemoveAllGroups
 *
 * Command description for RemoveAllGroups
 */
- (void)removeAllGroupsWithParams:(MTRGroupsClusterRemoveAllGroupsParams * _Nullable)params
                       completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)removeAllGroupsWithCompletion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command AddGroupIfIdentifying
 *
 * Command description for AddGroupIfIdentifying
 */
- (void)addGroupIfIdentifyingWithParams:(MTRGroupsClusterAddGroupIfIdentifyingParams *)params
                             completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeNameSupportWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeNameSupportWithParams:(MTRSubscribeParams *)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeNameSupportWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Scenes
 *
 * Attributes and commands for scene configuration and manipulation.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterScenes : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

/**
 * Command AddScene
 *
 * Add a scene to the scene table. Extension field sets are supported, and are inputed as '{"ClusterID": VALUE,
 * "AttributeValueList":[{"AttributeId": VALUE, "AttributeValue": VALUE}]}'
 */
- (void)addSceneWithParams:(MTRScenesClusterAddSceneParams *)params
                completion:(void (^)(MTRScenesClusterAddSceneResponseParams * _Nullable data, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command ViewScene
 *
 * Retrieves the requested scene entry from its Scene table.
 */
- (void)viewSceneWithParams:(MTRScenesClusterViewSceneParams *)params
                 completion:
                     (void (^)(MTRScenesClusterViewSceneResponseParams * _Nullable data, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command RemoveScene
 *
 * Removes the requested scene entry, corresponding to the value of the GroupID field, from its Scene Table
 */
- (void)removeSceneWithParams:(MTRScenesClusterRemoveSceneParams *)params
                   completion:
                       (void (^)(MTRScenesClusterRemoveSceneResponseParams * _Nullable data, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command RemoveAllScenes
 *
 * Remove all scenes, corresponding to the value of the GroupID field, from its Scene Table
 */
- (void)removeAllScenesWithParams:(MTRScenesClusterRemoveAllScenesParams *)params
                       completion:(void (^)(MTRScenesClusterRemoveAllScenesResponseParams * _Nullable data,
                                      NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command StoreScene
 *
 * Adds the scene entry into its Scene Table along with all extension field sets corresponding to the current state of other
 * clusters on the same endpoint
 */
- (void)storeSceneWithParams:(MTRScenesClusterStoreSceneParams *)params
                  completion:
                      (void (^)(MTRScenesClusterStoreSceneResponseParams * _Nullable data, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command RecallScene
 *
 * Set the attributes and corresponding state for each other cluster implemented on the endpoint accordingly to the resquested scene
 * entry in the Scene Table
 */
- (void)recallSceneWithParams:(MTRScenesClusterRecallSceneParams *)params
                   completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command GetSceneMembership
 *
 * Get an unused scene identifier when no commissioning tool is in the network, or for a commissioning tool to get the used scene
 * identifiers within a certain group
 */
- (void)getSceneMembershipWithParams:(MTRScenesClusterGetSceneMembershipParams *)params
                          completion:(void (^)(MTRScenesClusterGetSceneMembershipResponseParams * _Nullable data,
                                         NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command EnhancedAddScene
 *
 * Allows a scene to be added using a finer scene transition time than the AddScene command.
 */
- (void)enhancedAddSceneWithParams:(MTRScenesClusterEnhancedAddSceneParams *)params
                        completion:(void (^)(MTRScenesClusterEnhancedAddSceneResponseParams * _Nullable data,
                                       NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command EnhancedViewScene
 *
 * Allows a scene to be retrieved using a finer scene transition time than the ViewScene command
 */
- (void)enhancedViewSceneWithParams:(MTRScenesClusterEnhancedViewSceneParams *)params
                         completion:(void (^)(MTRScenesClusterEnhancedViewSceneResponseParams * _Nullable data,
                                        NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command CopyScene
 *
 * Allows a client to efficiently copy scenes from one group/scene identifier pair to another group/scene identifier pair.
 */
- (void)copySceneWithParams:(MTRScenesClusterCopySceneParams *)params
                 completion:
                     (void (^)(MTRScenesClusterCopySceneResponseParams * _Nullable data, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeSceneCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeSceneCountWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeSceneCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeCurrentSceneWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeCurrentSceneWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeCurrentSceneWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeCurrentGroupWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeCurrentGroupWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeCurrentGroupWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeSceneValidWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeSceneValidWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeSceneValidWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeNameSupportWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeNameSupportWithParams:(MTRSubscribeParams *)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeNameSupportWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeLastConfiguredByWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeLastConfiguredByWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeLastConfiguredByWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:
                                                    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeSceneTableSizeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeSceneTableSizeWithParams:(MTRSubscribeParams *)params
                           subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                     reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeSceneTableSizeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                endpoint:(NSNumber *)endpoint
                                                   queue:(dispatch_queue_t)queue
                                              completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeRemainingCapacityWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeRemainingCapacityWithParams:(MTRSubscribeParams *)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeRemainingCapacityWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                                 completion:(void (^)(NSNumber * _Nullable value,
                                                                NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster On/Off
 *
 * Attributes and commands for switching devices between 'On' and 'Off' states.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterOnOff : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

/**
 * Command Off
 *
 * On receipt of this command, a device SHALL enter its ‘Off’ state. This state is device dependent, but it is recommended that it
 * is used for power off or similar functions. On receipt of the Off command, the OnTime attribute SHALL be set to 0.
 */
- (void)offWithParams:(MTROnOffClusterOffParams * _Nullable)params
           completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)offWithCompletion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command On
 *
 * On receipt of this command, a device SHALL enter its ‘On’ state. This state is device dependent, but it is recommended that it is
 * used for power on or similar functions. On receipt of the On command, if the value of the OnTime attribute is equal to 0, the
 * device SHALL set the OffWaitTime attribute to 0.
 */
- (void)onWithParams:(MTROnOffClusterOnParams * _Nullable)params
          completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)onWithCompletion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command Toggle
 *
 * On receipt of this command, if a device is in its ‘Off’ state it SHALL enter its ‘On’ state. Otherwise, if it is in its ‘On’
 * state it SHALL enter its ‘Off’ state. On receipt of the Toggle command, if the value of the OnOff attribute is equal to FALSE and
 * if the value of the OnTime attribute is equal to 0, the device SHALL set the OffWaitTime attribute to 0. If the value of the
 * OnOff attribute is equal to TRUE, the OnTime attribute SHALL be set to 0.
 */
- (void)toggleWithParams:(MTROnOffClusterToggleParams * _Nullable)params
              completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)toggleWithCompletion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command OffWithEffect
 *
 * The OffWithEffect command allows devices to be turned off using enhanced ways of fading.
 */
- (void)offWithEffectWithParams:(MTROnOffClusterOffWithEffectParams *)params
                     completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command OnWithRecallGlobalScene
 *
 * The OnWithRecallGlobalScene command allows the recall of the settings when the device was turned off.
 */
- (void)onWithRecallGlobalSceneWithParams:(MTROnOffClusterOnWithRecallGlobalSceneParams * _Nullable)params
                               completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)onWithRecallGlobalSceneWithCompletion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command OnWithTimedOff
 *
 * The OnWithTimedOff command allows devices to be turned on for a specific duration with a guarded off duration so that SHOULD the
 * device be subsequently switched off, further OnWithTimedOff commands, received during this time, are prevented from turning the
 * devices back on.
 */
- (void)onWithTimedOffWithParams:(MTROnOffClusterOnWithTimedOffParams *)params
                      completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeOnOffWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeOnOffWithParams:(MTRSubscribeParams *)params
                  subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                            reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeOnOffWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                       endpoint:(NSNumber *)endpoint
                                          queue:(dispatch_queue_t)queue
                                     completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGlobalSceneControlWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGlobalSceneControlWithParams:(MTRSubscribeParams *)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                         reportHandler:
                                             (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGlobalSceneControlWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                    endpoint:(NSNumber *)endpoint
                                                       queue:(dispatch_queue_t)queue
                                                  completion:
                                                      (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeOnTimeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeOnTimeWithValue:(NSNumber * _Nonnull)value
                           completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeOnTimeWithValue:(NSNumber * _Nonnull)value
                               params:(MTRWriteParams * _Nullable)params
                           completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeOnTimeWithParams:(MTRSubscribeParams *)params
                   subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                             reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeOnTimeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                        endpoint:(NSNumber *)endpoint
                                           queue:(dispatch_queue_t)queue
                                      completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeOffWaitTimeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeOffWaitTimeWithValue:(NSNumber * _Nonnull)value
                                completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeOffWaitTimeWithValue:(NSNumber * _Nonnull)value
                                    params:(MTRWriteParams * _Nullable)params
                                completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeOffWaitTimeWithParams:(MTRSubscribeParams *)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeOffWaitTimeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeStartUpOnOffWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeStartUpOnOffWithValue:(NSNumber * _Nullable)value
                                 completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeStartUpOnOffWithValue:(NSNumber * _Nullable)value
                                     params:(MTRWriteParams * _Nullable)params
                                 completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeStartUpOnOffWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeStartUpOnOffWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster On/off Switch Configuration
 *
 * Attributes and commands for configuring On/Off switching devices.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterOnOffSwitchConfiguration : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeSwitchTypeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeSwitchTypeWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeSwitchTypeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeSwitchActionsWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeSwitchActionsWithValue:(NSNumber * _Nonnull)value
                                  completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeSwitchActionsWithValue:(NSNumber * _Nonnull)value
                                      params:(MTRWriteParams * _Nullable)params
                                  completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeSwitchActionsWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeSwitchActionsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Level Control
 *
 * Attributes and commands for controlling devices that can be set to a level between fully 'On' and fully 'Off.'
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterLevelControl : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

/**
 * Command MoveToLevel
 *
 * Command description for MoveToLevel
 */
- (void)moveToLevelWithParams:(MTRLevelControlClusterMoveToLevelParams *)params
                   completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command Move
 *
 * Command description for Move
 */
- (void)moveWithParams:(MTRLevelControlClusterMoveParams *)params
            completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command Step
 *
 * Command description for Step
 */
- (void)stepWithParams:(MTRLevelControlClusterStepParams *)params
            completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command Stop
 *
 * Command description for Stop
 */
- (void)stopWithParams:(MTRLevelControlClusterStopParams *)params
            completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command MoveToLevelWithOnOff
 *
 * Command description for MoveToLevelWithOnOff
 */
- (void)moveToLevelWithOnOffWithParams:(MTRLevelControlClusterMoveToLevelWithOnOffParams *)params
                            completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command MoveWithOnOff
 *
 * Command description for MoveWithOnOff
 */
- (void)moveWithOnOffWithParams:(MTRLevelControlClusterMoveWithOnOffParams *)params
                     completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command StepWithOnOff
 *
 * Command description for StepWithOnOff
 */
- (void)stepWithOnOffWithParams:(MTRLevelControlClusterStepWithOnOffParams *)params
                     completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command StopWithOnOff
 *
 * Command description for StopWithOnOff
 */
- (void)stopWithOnOffWithParams:(MTRLevelControlClusterStopWithOnOffParams *)params
                     completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command MoveToClosestFrequency
 *
 * Change the currrent frequency to the provided one, or a close
        approximation if the exact provided one is not possible.
 */
- (void)moveToClosestFrequencyWithParams:(MTRLevelControlClusterMoveToClosestFrequencyParams *)params
                              completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeCurrentLevelWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeCurrentLevelWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeCurrentLevelWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeRemainingTimeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeRemainingTimeWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeRemainingTimeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeMinLevelWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeMinLevelWithParams:(MTRSubscribeParams *)params
                     subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                               reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeMinLevelWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                          endpoint:(NSNumber *)endpoint
                                             queue:(dispatch_queue_t)queue
                                        completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeMaxLevelWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeMaxLevelWithParams:(MTRSubscribeParams *)params
                     subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                               reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeMaxLevelWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                          endpoint:(NSNumber *)endpoint
                                             queue:(dispatch_queue_t)queue
                                        completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeCurrentFrequencyWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeCurrentFrequencyWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeCurrentFrequencyWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:
                                                    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeMinFrequencyWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeMinFrequencyWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeMinFrequencyWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeMaxFrequencyWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeMaxFrequencyWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeMaxFrequencyWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeOptionsWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeOptionsWithValue:(NSNumber * _Nonnull)value
                            completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeOptionsWithValue:(NSNumber * _Nonnull)value
                                params:(MTRWriteParams * _Nullable)params
                            completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeOptionsWithParams:(MTRSubscribeParams *)params
                    subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                              reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeOptionsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                       completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeOnOffTransitionTimeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeOnOffTransitionTimeWithValue:(NSNumber * _Nonnull)value
                                        completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeOnOffTransitionTimeWithValue:(NSNumber * _Nonnull)value
                                            params:(MTRWriteParams * _Nullable)params
                                        completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeOnOffTransitionTimeWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeOnOffTransitionTimeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeOnLevelWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeOnLevelWithValue:(NSNumber * _Nullable)value
                            completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeOnLevelWithValue:(NSNumber * _Nullable)value
                                params:(MTRWriteParams * _Nullable)params
                            completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeOnLevelWithParams:(MTRSubscribeParams *)params
                    subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                              reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeOnLevelWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                       completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeOnTransitionTimeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeOnTransitionTimeWithValue:(NSNumber * _Nullable)value
                                     completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeOnTransitionTimeWithValue:(NSNumber * _Nullable)value
                                         params:(MTRWriteParams * _Nullable)params
                                     completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeOnTransitionTimeWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeOnTransitionTimeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:
                                                    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeOffTransitionTimeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeOffTransitionTimeWithValue:(NSNumber * _Nullable)value
                                      completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeOffTransitionTimeWithValue:(NSNumber * _Nullable)value
                                          params:(MTRWriteParams * _Nullable)params
                                      completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeOffTransitionTimeWithParams:(MTRSubscribeParams *)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeOffTransitionTimeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                                 completion:
                                                     (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeDefaultMoveRateWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeDefaultMoveRateWithValue:(NSNumber * _Nullable)value
                                    completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeDefaultMoveRateWithValue:(NSNumber * _Nullable)value
                                        params:(MTRWriteParams * _Nullable)params
                                    completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeDefaultMoveRateWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeDefaultMoveRateWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeStartUpCurrentLevelWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeStartUpCurrentLevelWithValue:(NSNumber * _Nullable)value
                                        completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeStartUpCurrentLevelWithValue:(NSNumber * _Nullable)value
                                            params:(MTRWriteParams * _Nullable)params
                                        completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeStartUpCurrentLevelWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeStartUpCurrentLevelWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Binary Input (Basic)
 *
 * An interface for reading the value of a binary measurement and accessing various characteristics of that measurement.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterBinaryInputBasic : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeActiveTextWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeActiveTextWithValue:(NSString * _Nonnull)value
                               completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeActiveTextWithValue:(NSString * _Nonnull)value
                                   params:(MTRWriteParams * _Nullable)params
                               completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeActiveTextWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeActiveTextWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeDescriptionWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeDescriptionWithValue:(NSString * _Nonnull)value
                                completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeDescriptionWithValue:(NSString * _Nonnull)value
                                    params:(MTRWriteParams * _Nullable)params
                                completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeDescriptionWithParams:(MTRSubscribeParams *)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeDescriptionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeInactiveTextWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeInactiveTextWithValue:(NSString * _Nonnull)value
                                 completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeInactiveTextWithValue:(NSString * _Nonnull)value
                                     params:(MTRWriteParams * _Nullable)params
                                 completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeInactiveTextWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeInactiveTextWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeOutOfServiceWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeOutOfServiceWithValue:(NSNumber * _Nonnull)value
                                 completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeOutOfServiceWithValue:(NSNumber * _Nonnull)value
                                     params:(MTRWriteParams * _Nullable)params
                                 completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeOutOfServiceWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeOutOfServiceWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributePolarityWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributePolarityWithParams:(MTRSubscribeParams *)params
                     subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                               reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributePolarityWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                          endpoint:(NSNumber *)endpoint
                                             queue:(dispatch_queue_t)queue
                                        completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributePresentValueWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributePresentValueWithValue:(NSNumber * _Nonnull)value
                                 completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributePresentValueWithValue:(NSNumber * _Nonnull)value
                                     params:(MTRWriteParams * _Nullable)params
                                 completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributePresentValueWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributePresentValueWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeReliabilityWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeReliabilityWithValue:(NSNumber * _Nonnull)value
                                completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeReliabilityWithValue:(NSNumber * _Nonnull)value
                                    params:(MTRWriteParams * _Nullable)params
                                completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeReliabilityWithParams:(MTRSubscribeParams *)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeReliabilityWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeStatusFlagsWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeStatusFlagsWithParams:(MTRSubscribeParams *)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeStatusFlagsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeApplicationTypeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeApplicationTypeWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeApplicationTypeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Descriptor
 *
 * The Descriptor Cluster is meant to replace the support from the Zigbee Device Object (ZDO) for describing a node, its endpoints
 * and clusters.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterDescriptor : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeDeviceTypeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeDeviceTypeListWithParams:(MTRSubscribeParams *)params
                           subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                     reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeDeviceTypeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                endpoint:(NSNumber *)endpoint
                                                   queue:(dispatch_queue_t)queue
                                              completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeServerListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeServerListWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeServerListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClientListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClientListWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClientListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributePartsListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributePartsListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributePartsListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Binding
 *
 * The Binding Cluster is meant to replace the support from the Zigbee Device Object (ZDO) for supporting the binding table.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterBinding : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBindingWithParams:(MTRReadParams * _Nullable)params
                            completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeBindingWithValue:(NSArray * _Nonnull)value
                            completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeBindingWithValue:(NSArray * _Nonnull)value
                                params:(MTRWriteParams * _Nullable)params
                            completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBindingWithParams:(MTRSubscribeParams *)params
                    subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                              reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBindingWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                       completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Access Control
 *
 * The Access Control Cluster exposes a data model view of a
      Node's Access Control List (ACL), which codifies the rules used to manage
      and enforce Access Control for the Node's endpoints and their associated
      cluster instances.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterAccessControl : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeACLWithParams:(MTRReadParams * _Nullable)params
                        completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeACLWithValue:(NSArray * _Nonnull)value
                        completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeACLWithValue:(NSArray * _Nonnull)value
                            params:(MTRWriteParams * _Nullable)params
                        completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeACLWithParams:(MTRSubscribeParams *)params
                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                          reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeACLWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                     endpoint:(NSNumber *)endpoint
                                        queue:(dispatch_queue_t)queue
                                   completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeExtensionWithParams:(MTRReadParams * _Nullable)params
                              completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeExtensionWithValue:(NSArray * _Nonnull)value
                              completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeExtensionWithValue:(NSArray * _Nonnull)value
                                  params:(MTRWriteParams * _Nullable)params
                              completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeExtensionWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeExtensionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeSubjectsPerAccessControlEntryWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeSubjectsPerAccessControlEntryWithParams:(MTRSubscribeParams *)params
                                          subscriptionEstablished:
                                              (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                    reportHandler:(void (^)(NSNumber * _Nullable value,
                                                                      NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeSubjectsPerAccessControlEntryWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                               endpoint:(NSNumber *)endpoint
                                                                  queue:(dispatch_queue_t)queue
                                                             completion:(void (^)(NSNumber * _Nullable value,
                                                                            NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeTargetsPerAccessControlEntryWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeTargetsPerAccessControlEntryWithParams:(MTRSubscribeParams *)params
                                         subscriptionEstablished:
                                             (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                   reportHandler:(void (^)(NSNumber * _Nullable value,
                                                                     NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeTargetsPerAccessControlEntryWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                              endpoint:(NSNumber *)endpoint
                                                                 queue:(dispatch_queue_t)queue
                                                            completion:(void (^)(NSNumber * _Nullable value,
                                                                           NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAccessControlEntriesPerFabricWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAccessControlEntriesPerFabricWithParams:(MTRSubscribeParams *)params
                                          subscriptionEstablished:
                                              (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                    reportHandler:(void (^)(NSNumber * _Nullable value,
                                                                      NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAccessControlEntriesPerFabricWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                               endpoint:(NSNumber *)endpoint
                                                                  queue:(dispatch_queue_t)queue
                                                             completion:(void (^)(NSNumber * _Nullable value,
                                                                            NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Actions
 *
 * This cluster provides a standardized way for a Node (typically a Bridge, but could be any Node) to expose action information.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterActions : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

/**
 * Command InstantAction
 *
 * This command triggers an action (state change) on the involved endpoints.
 */
- (void)instantActionWithParams:(MTRActionsClusterInstantActionParams *)params
                     completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command InstantActionWithTransition
 *
 * This command triggers an action (state change) on the involved endpoints, with a specified time to transition from the current
 * state to the new state.
 */
- (void)instantActionWithTransitionWithParams:(MTRActionsClusterInstantActionWithTransitionParams *)params
                                   completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command StartAction
 *
 * This command triggers the commencement of an action on the involved endpoints.
 */
- (void)startActionWithParams:(MTRActionsClusterStartActionParams *)params
                   completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command StartActionWithDuration
 *
 * This command triggers the commencement of an action (with a duration) on the involved endpoints.
 */
- (void)startActionWithDurationWithParams:(MTRActionsClusterStartActionWithDurationParams *)params
                               completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command StopAction
 *
 * This command stops the ongoing action on the involved endpoints.
 */
- (void)stopActionWithParams:(MTRActionsClusterStopActionParams *)params
                  completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command PauseAction
 *
 * This command pauses an ongoing action.
 */
- (void)pauseActionWithParams:(MTRActionsClusterPauseActionParams *)params
                   completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command PauseActionWithDuration
 *
 * This command pauses an ongoing action with a duration.
 */
- (void)pauseActionWithDurationWithParams:(MTRActionsClusterPauseActionWithDurationParams *)params
                               completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command ResumeAction
 *
 * This command resumes a previously paused action.
 */
- (void)resumeActionWithParams:(MTRActionsClusterResumeActionParams *)params
                    completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command EnableAction
 *
 * This command enables a certain action or automation.
 */
- (void)enableActionWithParams:(MTRActionsClusterEnableActionParams *)params
                    completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command EnableActionWithDuration
 *
 * This command enables a certain action or automation with a duration.
 */
- (void)enableActionWithDurationWithParams:(MTRActionsClusterEnableActionWithDurationParams *)params
                                completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command DisableAction
 *
 * This command disables a certain action or automation.
 */
- (void)disableActionWithParams:(MTRActionsClusterDisableActionParams *)params
                     completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command DisableActionWithDuration
 *
 * This command disables a certain action or automation with a duration.
 */
- (void)disableActionWithDurationWithParams:(MTRActionsClusterDisableActionWithDurationParams *)params
                                 completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeActionListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeActionListWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeActionListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEndpointListsWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeEndpointListsWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeEndpointListsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeSetupURLWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeSetupURLWithParams:(MTRSubscribeParams *)params
                     subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                               reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeSetupURLWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                          endpoint:(NSNumber *)endpoint
                                             queue:(dispatch_queue_t)queue
                                        completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Basic Information
 *
 * This cluster provides attributes and events for determining basic information about Nodes, which supports both
      Commissioning and operational determination of Node characteristics, such as Vendor ID, Product ID and serial number,
      which apply to the whole Node. Also allows setting user device information such as location.
 */
API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4))
@interface MTRBaseClusterBasicInformation : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeDataModelRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeDataModelRevisionWithParams:(MTRSubscribeParams *)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeDataModelRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                                 completion:
                                                     (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeVendorNameWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeVendorNameWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeVendorNameWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeVendorIDWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeVendorIDWithParams:(MTRSubscribeParams *)params
                     subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                               reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeVendorIDWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                          endpoint:(NSNumber *)endpoint
                                             queue:(dispatch_queue_t)queue
                                        completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeProductNameWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeProductNameWithParams:(MTRSubscribeParams *)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeProductNameWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeProductIDWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeProductIDWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeProductIDWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeNodeLabelWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeNodeLabelWithValue:(NSString * _Nonnull)value
                              completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeNodeLabelWithValue:(NSString * _Nonnull)value
                                  params:(MTRWriteParams * _Nullable)params
                              completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeNodeLabelWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeNodeLabelWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeLocationWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeLocationWithValue:(NSString * _Nonnull)value
                             completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeLocationWithValue:(NSString * _Nonnull)value
                                 params:(MTRWriteParams * _Nullable)params
                             completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeLocationWithParams:(MTRSubscribeParams *)params
                     subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                               reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeLocationWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                          endpoint:(NSNumber *)endpoint
                                             queue:(dispatch_queue_t)queue
                                        completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeHardwareVersionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeHardwareVersionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeHardwareVersionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeHardwareVersionStringWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeHardwareVersionStringWithParams:(MTRSubscribeParams *)params
                                  subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                            reportHandler:
                                                (void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeHardwareVersionStringWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                       endpoint:(NSNumber *)endpoint
                                                          queue:(dispatch_queue_t)queue
                                                     completion:
                                                         (void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeSoftwareVersionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeSoftwareVersionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeSoftwareVersionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeSoftwareVersionStringWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeSoftwareVersionStringWithParams:(MTRSubscribeParams *)params
                                  subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                            reportHandler:
                                                (void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeSoftwareVersionStringWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                       endpoint:(NSNumber *)endpoint
                                                          queue:(dispatch_queue_t)queue
                                                     completion:
                                                         (void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeManufacturingDateWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeManufacturingDateWithParams:(MTRSubscribeParams *)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeManufacturingDateWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                                 completion:
                                                     (void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributePartNumberWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributePartNumberWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributePartNumberWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeProductURLWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeProductURLWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeProductURLWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeProductLabelWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeProductLabelWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeProductLabelWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeSerialNumberWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeSerialNumberWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeSerialNumberWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeLocalConfigDisabledWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeLocalConfigDisabledWithValue:(NSNumber * _Nonnull)value
                                        completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeLocalConfigDisabledWithValue:(NSNumber * _Nonnull)value
                                            params:(MTRWriteParams * _Nullable)params
                                        completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeLocalConfigDisabledWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeLocalConfigDisabledWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeReachableWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeReachableWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeReachableWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeUniqueIDWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeUniqueIDWithParams:(MTRSubscribeParams *)params
                     subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                               reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeUniqueIDWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                          endpoint:(NSNumber *)endpoint
                                             queue:(dispatch_queue_t)queue
                                        completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeCapabilityMinimaWithCompletion:
    (void (^)(MTRBasicInformationClusterCapabilityMinimaStruct * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeCapabilityMinimaWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(MTRBasicInformationClusterCapabilityMinimaStruct * _Nullable value,
                                                         NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeCapabilityMinimaWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:
                                                    (void (^)(MTRBasicInformationClusterCapabilityMinimaStruct * _Nullable value,
                                                        NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeProductAppearanceWithCompletion:(void (^)(MTRBasicInformationClusterProductAppearanceStruct * _Nullable value,
                                                         NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeProductAppearanceWithParams:(MTRSubscribeParams *)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:(void (^)(MTRBasicInformationClusterProductAppearanceStruct * _Nullable value,
                                                          NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeProductAppearanceWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                                 completion:
                                                     (void (^)(MTRBasicInformationClusterProductAppearanceStruct * _Nullable value,
                                                         NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster OTA Software Update Provider
 *
 * Provides an interface for providing OTA software updates
 */
API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4))
@interface MTRBaseClusterOTASoftwareUpdateProvider : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

/**
 * Command QueryImage
 *
 * Determine availability of a new Software Image
 */
- (void)queryImageWithParams:(MTROTASoftwareUpdateProviderClusterQueryImageParams *)params
                  completion:(void (^)(MTROTASoftwareUpdateProviderClusterQueryImageResponseParams * _Nullable data,
                                 NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command ApplyUpdateRequest
 *
 * Determine next action to take for a downloaded Software Image
 */
- (void)applyUpdateRequestWithParams:(MTROTASoftwareUpdateProviderClusterApplyUpdateRequestParams *)params
                          completion:(void (^)(MTROTASoftwareUpdateProviderClusterApplyUpdateResponseParams * _Nullable data,
                                         NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command NotifyUpdateApplied
 *
 * Notify OTA Provider that an update was applied
 */
- (void)notifyUpdateAppliedWithParams:(MTROTASoftwareUpdateProviderClusterNotifyUpdateAppliedParams *)params
                           completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster OTA Software Update Requestor
 *
 * Provides an interface for downloading and applying OTA software updates
 */
API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4))
@interface MTRBaseClusterOTASoftwareUpdateRequestor : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

/**
 * Command AnnounceOTAProvider
 *
 * Announce the presence of an OTA Provider
 */
- (void)announceOTAProviderWithParams:(MTROTASoftwareUpdateRequestorClusterAnnounceOTAProviderParams *)params
                           completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeDefaultOTAProvidersWithParams:(MTRReadParams * _Nullable)params
                                        completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeDefaultOTAProvidersWithValue:(NSArray * _Nonnull)value
                                        completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeDefaultOTAProvidersWithValue:(NSArray * _Nonnull)value
                                            params:(MTRWriteParams * _Nullable)params
                                        completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeDefaultOTAProvidersWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeDefaultOTAProvidersWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeUpdatePossibleWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeUpdatePossibleWithParams:(MTRSubscribeParams *)params
                           subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                     reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeUpdatePossibleWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                endpoint:(NSNumber *)endpoint
                                                   queue:(dispatch_queue_t)queue
                                              completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeUpdateStateWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeUpdateStateWithParams:(MTRSubscribeParams *)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeUpdateStateWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeUpdateStateProgressWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeUpdateStateProgressWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeUpdateStateProgressWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Localization Configuration
 *
 * Nodes should be expected to be deployed to any and all regions of the world. These global regions
      may have differing common languages, units of measurements, and numerical formatting
      standards. As such, Nodes that visually or audibly convey information need a mechanism by which
      they can be configured to use a user’s preferred language, units, etc
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterLocalizationConfiguration : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeActiveLocaleWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeActiveLocaleWithValue:(NSString * _Nonnull)value
                                 completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeActiveLocaleWithValue:(NSString * _Nonnull)value
                                     params:(MTRWriteParams * _Nullable)params
                                 completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeActiveLocaleWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeActiveLocaleWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeSupportedLocalesWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeSupportedLocalesWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeSupportedLocalesWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:
                                                    (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Time Format Localization
 *
 * Nodes should be expected to be deployed to any and all regions of the world. These global regions
      may have differing preferences for how dates and times are conveyed. As such, Nodes that visually
      or audibly convey time information need a mechanism by which they can be configured to use a
      user’s preferred format.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterTimeFormatLocalization : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeHourFormatWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeHourFormatWithValue:(NSNumber * _Nonnull)value
                               completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeHourFormatWithValue:(NSNumber * _Nonnull)value
                                   params:(MTRWriteParams * _Nullable)params
                               completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeHourFormatWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeHourFormatWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeActiveCalendarTypeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeActiveCalendarTypeWithValue:(NSNumber * _Nonnull)value
                                       completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeActiveCalendarTypeWithValue:(NSNumber * _Nonnull)value
                                           params:(MTRWriteParams * _Nullable)params
                                       completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeActiveCalendarTypeWithParams:(MTRSubscribeParams *)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                         reportHandler:
                                             (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeActiveCalendarTypeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                    endpoint:(NSNumber *)endpoint
                                                       queue:(dispatch_queue_t)queue
                                                  completion:
                                                      (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeSupportedCalendarTypesWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeSupportedCalendarTypesWithParams:(MTRSubscribeParams *)params
                                   subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                             reportHandler:
                                                 (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeSupportedCalendarTypesWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                        endpoint:(NSNumber *)endpoint
                                                           queue:(dispatch_queue_t)queue
                                                      completion:
                                                          (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Unit Localization
 *
 * Nodes should be expected to be deployed to any and all regions of the world. These global regions
      may have differing preferences for the units in which values are conveyed in communication to a
      user. As such, Nodes that visually or audibly convey measurable values to the user need a
      mechanism by which they can be configured to use a user’s preferred unit.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterUnitLocalization : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeTemperatureUnitWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeTemperatureUnitWithValue:(NSNumber * _Nonnull)value
                                    completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeTemperatureUnitWithValue:(NSNumber * _Nonnull)value
                                        params:(MTRWriteParams * _Nullable)params
                                    completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeTemperatureUnitWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeTemperatureUnitWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Power Source Configuration
 *
 * This cluster is used to describe the configuration and capabilities of a Device's power system.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterPowerSourceConfiguration : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeSourcesWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeSourcesWithParams:(MTRSubscribeParams *)params
                    subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                              reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeSourcesWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                       completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Power Source
 *
 * This cluster is used to describe the configuration and capabilities of a physical power source that provides power to the Node.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterPowerSource : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeStatusWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeStatusWithParams:(MTRSubscribeParams *)params
                   subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                             reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeStatusWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                        endpoint:(NSNumber *)endpoint
                                           queue:(dispatch_queue_t)queue
                                      completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeOrderWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeOrderWithParams:(MTRSubscribeParams *)params
                  subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                            reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeOrderWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                       endpoint:(NSNumber *)endpoint
                                          queue:(dispatch_queue_t)queue
                                     completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeDescriptionWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeDescriptionWithParams:(MTRSubscribeParams *)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeDescriptionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeWiredAssessedInputVoltageWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeWiredAssessedInputVoltageWithParams:(MTRSubscribeParams *)params
                                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                reportHandler:
                                                    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeWiredAssessedInputVoltageWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                           endpoint:(NSNumber *)endpoint
                                                              queue:(dispatch_queue_t)queue
                                                         completion:(void (^)(NSNumber * _Nullable value,
                                                                        NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeWiredAssessedInputFrequencyWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeWiredAssessedInputFrequencyWithParams:(MTRSubscribeParams *)params
                                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                  reportHandler:
                                                      (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeWiredAssessedInputFrequencyWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                             endpoint:(NSNumber *)endpoint
                                                                queue:(dispatch_queue_t)queue
                                                           completion:(void (^)(NSNumber * _Nullable value,
                                                                          NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeWiredCurrentTypeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeWiredCurrentTypeWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeWiredCurrentTypeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:
                                                    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeWiredAssessedCurrentWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeWiredAssessedCurrentWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeWiredAssessedCurrentWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeWiredNominalVoltageWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeWiredNominalVoltageWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeWiredNominalVoltageWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeWiredMaximumCurrentWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeWiredMaximumCurrentWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeWiredMaximumCurrentWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeWiredPresentWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeWiredPresentWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeWiredPresentWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeActiveWiredFaultsWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeActiveWiredFaultsWithParams:(MTRSubscribeParams *)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeActiveWiredFaultsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                                 completion:
                                                     (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBatVoltageWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBatVoltageWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBatVoltageWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBatPercentRemainingWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBatPercentRemainingWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBatPercentRemainingWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBatTimeRemainingWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBatTimeRemainingWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBatTimeRemainingWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:
                                                    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBatChargeLevelWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBatChargeLevelWithParams:(MTRSubscribeParams *)params
                           subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                     reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBatChargeLevelWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                endpoint:(NSNumber *)endpoint
                                                   queue:(dispatch_queue_t)queue
                                              completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBatReplacementNeededWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBatReplacementNeededWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBatReplacementNeededWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBatReplaceabilityWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBatReplaceabilityWithParams:(MTRSubscribeParams *)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBatReplaceabilityWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                                 completion:
                                                     (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBatPresentWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBatPresentWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBatPresentWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeActiveBatFaultsWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeActiveBatFaultsWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeActiveBatFaultsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBatReplacementDescriptionWithCompletion:
    (void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBatReplacementDescriptionWithParams:(MTRSubscribeParams *)params
                                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                reportHandler:
                                                    (void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBatReplacementDescriptionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                           endpoint:(NSNumber *)endpoint
                                                              queue:(dispatch_queue_t)queue
                                                         completion:(void (^)(NSString * _Nullable value,
                                                                        NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBatCommonDesignationWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBatCommonDesignationWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBatCommonDesignationWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBatANSIDesignationWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBatANSIDesignationWithParams:(MTRSubscribeParams *)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                         reportHandler:
                                             (void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBatANSIDesignationWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                    endpoint:(NSNumber *)endpoint
                                                       queue:(dispatch_queue_t)queue
                                                  completion:
                                                      (void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBatIECDesignationWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBatIECDesignationWithParams:(MTRSubscribeParams *)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBatIECDesignationWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                                 completion:
                                                     (void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBatApprovedChemistryWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBatApprovedChemistryWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBatApprovedChemistryWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBatCapacityWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBatCapacityWithParams:(MTRSubscribeParams *)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBatCapacityWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBatQuantityWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBatQuantityWithParams:(MTRSubscribeParams *)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBatQuantityWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBatChargeStateWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBatChargeStateWithParams:(MTRSubscribeParams *)params
                           subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                     reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBatChargeStateWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                endpoint:(NSNumber *)endpoint
                                                   queue:(dispatch_queue_t)queue
                                              completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBatTimeToFullChargeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBatTimeToFullChargeWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBatTimeToFullChargeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBatFunctionalWhileChargingWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBatFunctionalWhileChargingWithParams:(MTRSubscribeParams *)params
                                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                 reportHandler:
                                                     (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBatFunctionalWhileChargingWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                            endpoint:(NSNumber *)endpoint
                                                               queue:(dispatch_queue_t)queue
                                                          completion:(void (^)(NSNumber * _Nullable value,
                                                                         NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBatChargingCurrentWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBatChargingCurrentWithParams:(MTRSubscribeParams *)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                         reportHandler:
                                             (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBatChargingCurrentWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                    endpoint:(NSNumber *)endpoint
                                                       queue:(dispatch_queue_t)queue
                                                  completion:
                                                      (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeActiveBatChargeFaultsWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeActiveBatChargeFaultsWithParams:(MTRSubscribeParams *)params
                                  subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                            reportHandler:
                                                (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeActiveBatChargeFaultsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                       endpoint:(NSNumber *)endpoint
                                                          queue:(dispatch_queue_t)queue
                                                     completion:
                                                         (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster General Commissioning
 *
 * This cluster is used to manage global aspects of the Commissioning flow.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterGeneralCommissioning : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

/**
 * Command ArmFailSafe
 *
 * Arm the persistent fail-safe timer with an expiry time of now + ExpiryLengthSeconds using device clock
 */
- (void)armFailSafeWithParams:(MTRGeneralCommissioningClusterArmFailSafeParams *)params
                   completion:(void (^)(MTRGeneralCommissioningClusterArmFailSafeResponseParams * _Nullable data,
                                  NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command SetRegulatoryConfig
 *
 * Set the regulatory configuration to be used during commissioning
 */
- (void)setRegulatoryConfigWithParams:(MTRGeneralCommissioningClusterSetRegulatoryConfigParams *)params
                           completion:(void (^)(MTRGeneralCommissioningClusterSetRegulatoryConfigResponseParams * _Nullable data,
                                          NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command CommissioningComplete
 *
 * Signals the Server that the Client has successfully completed all steps of Commissioning/Recofiguration needed during fail-safe
 * period.
 */
- (void)commissioningCompleteWithParams:(MTRGeneralCommissioningClusterCommissioningCompleteParams * _Nullable)params
                             completion:
                                 (void (^)(MTRGeneralCommissioningClusterCommissioningCompleteResponseParams * _Nullable data,
                                     NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)commissioningCompleteWithCompletion:
    (void (^)(MTRGeneralCommissioningClusterCommissioningCompleteResponseParams * _Nullable data,
        NSError * _Nullable error))completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBreadcrumbWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeBreadcrumbWithValue:(NSNumber * _Nonnull)value
                               completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeBreadcrumbWithValue:(NSNumber * _Nonnull)value
                                   params:(MTRWriteParams * _Nullable)params
                               completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBreadcrumbWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBreadcrumbWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBasicCommissioningInfoWithCompletion:
    (void (^)(MTRGeneralCommissioningClusterBasicCommissioningInfo * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBasicCommissioningInfoWithParams:(MTRSubscribeParams *)params
                                   subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                             reportHandler:
                                                 (void (^)(MTRGeneralCommissioningClusterBasicCommissioningInfo * _Nullable value,
                                                     NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)
    readAttributeBasicCommissioningInfoWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                    endpoint:(NSNumber *)endpoint
                                                       queue:(dispatch_queue_t)queue
                                                  completion:
                                                      (void (^)(
                                                          MTRGeneralCommissioningClusterBasicCommissioningInfo * _Nullable value,
                                                          NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeRegulatoryConfigWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeRegulatoryConfigWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeRegulatoryConfigWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:
                                                    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeLocationCapabilityWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeLocationCapabilityWithParams:(MTRSubscribeParams *)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                         reportHandler:
                                             (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeLocationCapabilityWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                    endpoint:(NSNumber *)endpoint
                                                       queue:(dispatch_queue_t)queue
                                                  completion:
                                                      (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeSupportsConcurrentConnectionWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeSupportsConcurrentConnectionWithParams:(MTRSubscribeParams *)params
                                         subscriptionEstablished:
                                             (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                   reportHandler:(void (^)(NSNumber * _Nullable value,
                                                                     NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeSupportsConcurrentConnectionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                              endpoint:(NSNumber *)endpoint
                                                                 queue:(dispatch_queue_t)queue
                                                            completion:(void (^)(NSNumber * _Nullable value,
                                                                           NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Network Commissioning
 *
 * Functionality to configure, enable, disable network credentials and access on a Matter device.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterNetworkCommissioning : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

/**
 * Command ScanNetworks
 *
 * Detemine the set of networks the device sees as available.
 */
- (void)scanNetworksWithParams:(MTRNetworkCommissioningClusterScanNetworksParams * _Nullable)params
                    completion:(void (^)(MTRNetworkCommissioningClusterScanNetworksResponseParams * _Nullable data,
                                   NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command AddOrUpdateWiFiNetwork
 *
 * Add or update the credentials for a given Wi-Fi network.
 */
- (void)addOrUpdateWiFiNetworkWithParams:(MTRNetworkCommissioningClusterAddOrUpdateWiFiNetworkParams *)params
                              completion:(void (^)(MTRNetworkCommissioningClusterNetworkConfigResponseParams * _Nullable data,
                                             NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command AddOrUpdateThreadNetwork
 *
 * Add or update the credentials for a given Thread network.
 */
- (void)addOrUpdateThreadNetworkWithParams:(MTRNetworkCommissioningClusterAddOrUpdateThreadNetworkParams *)params
                                completion:(void (^)(MTRNetworkCommissioningClusterNetworkConfigResponseParams * _Nullable data,
                                               NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command RemoveNetwork
 *
 * Remove the definition of a given network (including its credentials).
 */
- (void)removeNetworkWithParams:(MTRNetworkCommissioningClusterRemoveNetworkParams *)params
                     completion:(void (^)(MTRNetworkCommissioningClusterNetworkConfigResponseParams * _Nullable data,
                                    NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command ConnectNetwork
 *
 * Connect to the specified network, using previously-defined credentials.
 */
- (void)connectNetworkWithParams:(MTRNetworkCommissioningClusterConnectNetworkParams *)params
                      completion:(void (^)(MTRNetworkCommissioningClusterConnectNetworkResponseParams * _Nullable data,
                                     NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command ReorderNetwork
 *
 * Modify the order in which networks will be presented in the Networks attribute.
 */
- (void)reorderNetworkWithParams:(MTRNetworkCommissioningClusterReorderNetworkParams *)params
                      completion:(void (^)(MTRNetworkCommissioningClusterNetworkConfigResponseParams * _Nullable data,
                                     NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeMaxNetworksWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeMaxNetworksWithParams:(MTRSubscribeParams *)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeMaxNetworksWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeNetworksWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeNetworksWithParams:(MTRSubscribeParams *)params
                     subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                               reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeNetworksWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                          endpoint:(NSNumber *)endpoint
                                             queue:(dispatch_queue_t)queue
                                        completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeScanMaxTimeSecondsWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeScanMaxTimeSecondsWithParams:(MTRSubscribeParams *)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                         reportHandler:
                                             (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeScanMaxTimeSecondsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                    endpoint:(NSNumber *)endpoint
                                                       queue:(dispatch_queue_t)queue
                                                  completion:
                                                      (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeConnectMaxTimeSecondsWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeConnectMaxTimeSecondsWithParams:(MTRSubscribeParams *)params
                                  subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                            reportHandler:
                                                (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeConnectMaxTimeSecondsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                       endpoint:(NSNumber *)endpoint
                                                          queue:(dispatch_queue_t)queue
                                                     completion:
                                                         (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeInterfaceEnabledWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeInterfaceEnabledWithValue:(NSNumber * _Nonnull)value
                                     completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeInterfaceEnabledWithValue:(NSNumber * _Nonnull)value
                                         params:(MTRWriteParams * _Nullable)params
                                     completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeInterfaceEnabledWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeInterfaceEnabledWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:
                                                    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeLastNetworkingStatusWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeLastNetworkingStatusWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeLastNetworkingStatusWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeLastNetworkIDWithCompletion:(void (^)(NSData * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeLastNetworkIDWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSData * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeLastNetworkIDWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSData * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeLastConnectErrorValueWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeLastConnectErrorValueWithParams:(MTRSubscribeParams *)params
                                  subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                            reportHandler:
                                                (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeLastConnectErrorValueWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                       endpoint:(NSNumber *)endpoint
                                                          queue:(dispatch_queue_t)queue
                                                     completion:
                                                         (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Diagnostic Logs
 *
 * The cluster provides commands for retrieving unstructured diagnostic logs from a Node that may be used to aid in diagnostics.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterDiagnosticLogs : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

/**
 * Command RetrieveLogsRequest
 *
 * Retrieving diagnostic logs from a Node
 */
- (void)retrieveLogsRequestWithParams:(MTRDiagnosticLogsClusterRetrieveLogsRequestParams *)params
                           completion:(void (^)(MTRDiagnosticLogsClusterRetrieveLogsResponseParams * _Nullable data,
                                          NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster General Diagnostics
 *
 * The General Diagnostics Cluster, along with other diagnostics clusters, provide a means to acquire standardized diagnostics
 * metrics that MAY be used by a Node to assist a user or Administrative Node in diagnosing potential problems.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterGeneralDiagnostics : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

/**
 * Command TestEventTrigger
 *
 * Provide a means for certification tests to trigger some test-plan-specific events
 */
- (void)testEventTriggerWithParams:(MTRGeneralDiagnosticsClusterTestEventTriggerParams *)params
                        completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeNetworkInterfacesWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeNetworkInterfacesWithParams:(MTRSubscribeParams *)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeNetworkInterfacesWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                                 completion:
                                                     (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeRebootCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeRebootCountWithParams:(MTRSubscribeParams *)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeRebootCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeUpTimeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeUpTimeWithParams:(MTRSubscribeParams *)params
                   subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                             reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeUpTimeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                        endpoint:(NSNumber *)endpoint
                                           queue:(dispatch_queue_t)queue
                                      completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeTotalOperationalHoursWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeTotalOperationalHoursWithParams:(MTRSubscribeParams *)params
                                  subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                            reportHandler:
                                                (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeTotalOperationalHoursWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                       endpoint:(NSNumber *)endpoint
                                                          queue:(dispatch_queue_t)queue
                                                     completion:
                                                         (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBootReasonWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBootReasonWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBootReasonWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeActiveHardwareFaultsWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeActiveHardwareFaultsWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeActiveHardwareFaultsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeActiveRadioFaultsWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeActiveRadioFaultsWithParams:(MTRSubscribeParams *)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeActiveRadioFaultsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                                 completion:
                                                     (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeActiveNetworkFaultsWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeActiveNetworkFaultsWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeActiveNetworkFaultsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeTestEventTriggersEnabledWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeTestEventTriggersEnabledWithParams:(MTRSubscribeParams *)params
                                     subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                               reportHandler:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeTestEventTriggersEnabledWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                          endpoint:(NSNumber *)endpoint
                                                             queue:(dispatch_queue_t)queue
                                                        completion:(void (^)(NSNumber * _Nullable value,
                                                                       NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Software Diagnostics
 *
 * The Software Diagnostics Cluster provides a means to acquire standardized diagnostics metrics that MAY be used by a Node to
 * assist a user or Administrative Node in diagnosing potential problems.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterSoftwareDiagnostics : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

/**
 * Command ResetWatermarks
 *
 * Reception of this command SHALL reset the values: The StackFreeMinimum field of the ThreadMetrics attribute,
 * CurrentHeapHighWaterMark attribute.
 */
- (void)resetWatermarksWithParams:(MTRSoftwareDiagnosticsClusterResetWatermarksParams * _Nullable)params
                       completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)resetWatermarksWithCompletion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeThreadMetricsWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeThreadMetricsWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeThreadMetricsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeCurrentHeapFreeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeCurrentHeapFreeWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeCurrentHeapFreeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeCurrentHeapUsedWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeCurrentHeapUsedWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeCurrentHeapUsedWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeCurrentHeapHighWatermarkWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeCurrentHeapHighWatermarkWithParams:(MTRSubscribeParams *)params
                                     subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                               reportHandler:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeCurrentHeapHighWatermarkWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                          endpoint:(NSNumber *)endpoint
                                                             queue:(dispatch_queue_t)queue
                                                        completion:(void (^)(NSNumber * _Nullable value,
                                                                       NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Thread Network Diagnostics
 *
 * The Thread Network Diagnostics Cluster provides a means to acquire standardized diagnostics metrics that MAY be used by a Node to
 * assist a user or Administrative Node in diagnosing potential problems
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterThreadNetworkDiagnostics : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

/**
 * Command ResetCounts
 *
 * Reception of this command SHALL reset the OverrunCount attributes to 0
 */
- (void)resetCountsWithParams:(MTRThreadNetworkDiagnosticsClusterResetCountsParams * _Nullable)params
                   completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)resetCountsWithCompletion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeChannelWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeChannelWithParams:(MTRSubscribeParams *)params
                    subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                              reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeChannelWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                       completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeRoutingRoleWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeRoutingRoleWithParams:(MTRSubscribeParams *)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeRoutingRoleWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeNetworkNameWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeNetworkNameWithParams:(MTRSubscribeParams *)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeNetworkNameWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributePanIdWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributePanIdWithParams:(MTRSubscribeParams *)params
                  subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                            reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributePanIdWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                       endpoint:(NSNumber *)endpoint
                                          queue:(dispatch_queue_t)queue
                                     completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeExtendedPanIdWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeExtendedPanIdWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeExtendedPanIdWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeMeshLocalPrefixWithCompletion:(void (^)(NSData * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeMeshLocalPrefixWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSData * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeMeshLocalPrefixWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:(void (^)(NSData * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeOverrunCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeOverrunCountWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeOverrunCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeNeighborTableWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeNeighborTableWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeNeighborTableWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeRouteTableWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeRouteTableWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeRouteTableWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributePartitionIdWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributePartitionIdWithParams:(MTRSubscribeParams *)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributePartitionIdWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeWeightingWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeWeightingWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeWeightingWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeDataVersionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeDataVersionWithParams:(MTRSubscribeParams *)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeDataVersionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeStableDataVersionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeStableDataVersionWithParams:(MTRSubscribeParams *)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeStableDataVersionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                                 completion:
                                                     (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeLeaderRouterIdWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeLeaderRouterIdWithParams:(MTRSubscribeParams *)params
                           subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                     reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeLeaderRouterIdWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                endpoint:(NSNumber *)endpoint
                                                   queue:(dispatch_queue_t)queue
                                              completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeDetachedRoleCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeDetachedRoleCountWithParams:(MTRSubscribeParams *)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeDetachedRoleCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                                 completion:
                                                     (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeChildRoleCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeChildRoleCountWithParams:(MTRSubscribeParams *)params
                           subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                     reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeChildRoleCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                endpoint:(NSNumber *)endpoint
                                                   queue:(dispatch_queue_t)queue
                                              completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeRouterRoleCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeRouterRoleCountWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeRouterRoleCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeLeaderRoleCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeLeaderRoleCountWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeLeaderRoleCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAttachAttemptCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttachAttemptCountWithParams:(MTRSubscribeParams *)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                         reportHandler:
                                             (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttachAttemptCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                    endpoint:(NSNumber *)endpoint
                                                       queue:(dispatch_queue_t)queue
                                                  completion:
                                                      (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributePartitionIdChangeCountWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributePartitionIdChangeCountWithParams:(MTRSubscribeParams *)params
                                   subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                             reportHandler:
                                                 (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributePartitionIdChangeCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                        endpoint:(NSNumber *)endpoint
                                                           queue:(dispatch_queue_t)queue
                                                      completion:(void (^)(NSNumber * _Nullable value,
                                                                     NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBetterPartitionAttachAttemptCountWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBetterPartitionAttachAttemptCountWithParams:(MTRSubscribeParams *)params
                                              subscriptionEstablished:
                                                  (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                        reportHandler:(void (^)(NSNumber * _Nullable value,
                                                                          NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBetterPartitionAttachAttemptCountWithClusterStateCache:
            (MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                                   endpoint:(NSNumber *)endpoint
                                                                      queue:(dispatch_queue_t)queue
                                                                 completion:(void (^)(NSNumber * _Nullable value,
                                                                                NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeParentChangeCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeParentChangeCountWithParams:(MTRSubscribeParams *)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeParentChangeCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                                 completion:
                                                     (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeTxTotalCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeTxTotalCountWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeTxTotalCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeTxUnicastCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeTxUnicastCountWithParams:(MTRSubscribeParams *)params
                           subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                     reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeTxUnicastCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                endpoint:(NSNumber *)endpoint
                                                   queue:(dispatch_queue_t)queue
                                              completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeTxBroadcastCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeTxBroadcastCountWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeTxBroadcastCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:
                                                    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeTxAckRequestedCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeTxAckRequestedCountWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeTxAckRequestedCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeTxAckedCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeTxAckedCountWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeTxAckedCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeTxNoAckRequestedCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeTxNoAckRequestedCountWithParams:(MTRSubscribeParams *)params
                                  subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                            reportHandler:
                                                (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeTxNoAckRequestedCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                       endpoint:(NSNumber *)endpoint
                                                          queue:(dispatch_queue_t)queue
                                                     completion:
                                                         (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeTxDataCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeTxDataCountWithParams:(MTRSubscribeParams *)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeTxDataCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeTxDataPollCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeTxDataPollCountWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeTxDataPollCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeTxBeaconCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeTxBeaconCountWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeTxBeaconCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeTxBeaconRequestCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeTxBeaconRequestCountWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeTxBeaconRequestCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeTxOtherCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeTxOtherCountWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeTxOtherCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeTxRetryCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeTxRetryCountWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeTxRetryCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeTxDirectMaxRetryExpiryCountWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeTxDirectMaxRetryExpiryCountWithParams:(MTRSubscribeParams *)params
                                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                  reportHandler:
                                                      (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeTxDirectMaxRetryExpiryCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                             endpoint:(NSNumber *)endpoint
                                                                queue:(dispatch_queue_t)queue
                                                           completion:(void (^)(NSNumber * _Nullable value,
                                                                          NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeTxIndirectMaxRetryExpiryCountWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeTxIndirectMaxRetryExpiryCountWithParams:(MTRSubscribeParams *)params
                                          subscriptionEstablished:
                                              (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                    reportHandler:(void (^)(NSNumber * _Nullable value,
                                                                      NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeTxIndirectMaxRetryExpiryCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                               endpoint:(NSNumber *)endpoint
                                                                  queue:(dispatch_queue_t)queue
                                                             completion:(void (^)(NSNumber * _Nullable value,
                                                                            NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeTxErrCcaCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeTxErrCcaCountWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeTxErrCcaCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeTxErrAbortCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeTxErrAbortCountWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeTxErrAbortCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeTxErrBusyChannelCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeTxErrBusyChannelCountWithParams:(MTRSubscribeParams *)params
                                  subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                            reportHandler:
                                                (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeTxErrBusyChannelCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                       endpoint:(NSNumber *)endpoint
                                                          queue:(dispatch_queue_t)queue
                                                     completion:
                                                         (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeRxTotalCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeRxTotalCountWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeRxTotalCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeRxUnicastCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeRxUnicastCountWithParams:(MTRSubscribeParams *)params
                           subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                     reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeRxUnicastCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                endpoint:(NSNumber *)endpoint
                                                   queue:(dispatch_queue_t)queue
                                              completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeRxBroadcastCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeRxBroadcastCountWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeRxBroadcastCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:
                                                    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeRxDataCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeRxDataCountWithParams:(MTRSubscribeParams *)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeRxDataCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeRxDataPollCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeRxDataPollCountWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeRxDataPollCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeRxBeaconCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeRxBeaconCountWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeRxBeaconCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeRxBeaconRequestCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeRxBeaconRequestCountWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeRxBeaconRequestCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeRxOtherCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeRxOtherCountWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeRxOtherCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeRxAddressFilteredCountWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeRxAddressFilteredCountWithParams:(MTRSubscribeParams *)params
                                   subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                             reportHandler:
                                                 (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeRxAddressFilteredCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                        endpoint:(NSNumber *)endpoint
                                                           queue:(dispatch_queue_t)queue
                                                      completion:(void (^)(NSNumber * _Nullable value,
                                                                     NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeRxDestAddrFilteredCountWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeRxDestAddrFilteredCountWithParams:(MTRSubscribeParams *)params
                                    subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                              reportHandler:
                                                  (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeRxDestAddrFilteredCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                         endpoint:(NSNumber *)endpoint
                                                            queue:(dispatch_queue_t)queue
                                                       completion:(void (^)(NSNumber * _Nullable value,
                                                                      NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeRxDuplicatedCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeRxDuplicatedCountWithParams:(MTRSubscribeParams *)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeRxDuplicatedCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                                 completion:
                                                     (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeRxErrNoFrameCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeRxErrNoFrameCountWithParams:(MTRSubscribeParams *)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeRxErrNoFrameCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                                 completion:
                                                     (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeRxErrUnknownNeighborCountWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeRxErrUnknownNeighborCountWithParams:(MTRSubscribeParams *)params
                                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                reportHandler:
                                                    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeRxErrUnknownNeighborCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                           endpoint:(NSNumber *)endpoint
                                                              queue:(dispatch_queue_t)queue
                                                         completion:(void (^)(NSNumber * _Nullable value,
                                                                        NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeRxErrInvalidSrcAddrCountWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeRxErrInvalidSrcAddrCountWithParams:(MTRSubscribeParams *)params
                                     subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                               reportHandler:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeRxErrInvalidSrcAddrCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                          endpoint:(NSNumber *)endpoint
                                                             queue:(dispatch_queue_t)queue
                                                        completion:(void (^)(NSNumber * _Nullable value,
                                                                       NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeRxErrSecCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeRxErrSecCountWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeRxErrSecCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeRxErrFcsCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeRxErrFcsCountWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeRxErrFcsCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeRxErrOtherCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeRxErrOtherCountWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeRxErrOtherCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeActiveTimestampWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeActiveTimestampWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeActiveTimestampWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributePendingTimestampWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributePendingTimestampWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributePendingTimestampWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:
                                                    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeDelayWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeDelayWithParams:(MTRSubscribeParams *)params
                  subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                            reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeDelayWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                       endpoint:(NSNumber *)endpoint
                                          queue:(dispatch_queue_t)queue
                                     completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeSecurityPolicyWithCompletion:
    (void (^)(MTRThreadNetworkDiagnosticsClusterSecurityPolicy * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeSecurityPolicyWithParams:(MTRSubscribeParams *)params
                           subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                     reportHandler:(void (^)(MTRThreadNetworkDiagnosticsClusterSecurityPolicy * _Nullable value,
                                                       NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeSecurityPolicyWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                endpoint:(NSNumber *)endpoint
                                                   queue:(dispatch_queue_t)queue
                                              completion:
                                                  (void (^)(MTRThreadNetworkDiagnosticsClusterSecurityPolicy * _Nullable value,
                                                      NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeChannelPage0MaskWithCompletion:(void (^)(NSData * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeChannelPage0MaskWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSData * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeChannelPage0MaskWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:(void (^)(NSData * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeOperationalDatasetComponentsWithCompletion:
    (void (^)(MTRThreadNetworkDiagnosticsClusterOperationalDatasetComponents * _Nullable value,
        NSError * _Nullable error))completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)
    subscribeAttributeOperationalDatasetComponentsWithParams:(MTRSubscribeParams *)params
                                     subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                               reportHandler:
                                                   (void (^)(
                                                       MTRThreadNetworkDiagnosticsClusterOperationalDatasetComponents * _Nullable value,
                                                       NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)
    readAttributeOperationalDatasetComponentsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                          endpoint:(NSNumber *)endpoint
                                                             queue:(dispatch_queue_t)queue
                                                        completion:
                                                            (void (^)(
                                                                MTRThreadNetworkDiagnosticsClusterOperationalDatasetComponents * _Nullable value,
                                                                NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeActiveNetworkFaultsListWithCompletion:
    (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeActiveNetworkFaultsListWithParams:(MTRSubscribeParams *)params
                                    subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                              reportHandler:
                                                  (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeActiveNetworkFaultsListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                         endpoint:(NSNumber *)endpoint
                                                            queue:(dispatch_queue_t)queue
                                                       completion:(void (^)(NSArray * _Nullable value,
                                                                      NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster WiFi Network Diagnostics
 *
 * The Wi-Fi Network Diagnostics Cluster provides a means to acquire standardized diagnostics metrics that MAY be used by a Node to
 * assist a user or Administrative Node in diagnosing potential problems.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterWiFiNetworkDiagnostics : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

/**
 * Command ResetCounts
 *
 * Reception of this command SHALL reset the Breacon and Packet related count attributes to 0
 */
- (void)resetCountsWithParams:(MTRWiFiNetworkDiagnosticsClusterResetCountsParams * _Nullable)params
                   completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)resetCountsWithCompletion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBSSIDWithCompletion:(void (^)(NSData * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBSSIDWithParams:(MTRSubscribeParams *)params
                  subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                            reportHandler:(void (^)(NSData * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBSSIDWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                       endpoint:(NSNumber *)endpoint
                                          queue:(dispatch_queue_t)queue
                                     completion:(void (^)(NSData * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeSecurityTypeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeSecurityTypeWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeSecurityTypeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeWiFiVersionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeWiFiVersionWithParams:(MTRSubscribeParams *)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeWiFiVersionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeChannelNumberWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeChannelNumberWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeChannelNumberWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeRSSIWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeRSSIWithParams:(MTRSubscribeParams *)params
                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                           reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeRSSIWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                      endpoint:(NSNumber *)endpoint
                                         queue:(dispatch_queue_t)queue
                                    completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBeaconLostCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBeaconLostCountWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBeaconLostCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBeaconRxCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBeaconRxCountWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBeaconRxCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributePacketMulticastRxCountWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributePacketMulticastRxCountWithParams:(MTRSubscribeParams *)params
                                   subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                             reportHandler:
                                                 (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributePacketMulticastRxCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                        endpoint:(NSNumber *)endpoint
                                                           queue:(dispatch_queue_t)queue
                                                      completion:(void (^)(NSNumber * _Nullable value,
                                                                     NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributePacketMulticastTxCountWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributePacketMulticastTxCountWithParams:(MTRSubscribeParams *)params
                                   subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                             reportHandler:
                                                 (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributePacketMulticastTxCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                        endpoint:(NSNumber *)endpoint
                                                           queue:(dispatch_queue_t)queue
                                                      completion:(void (^)(NSNumber * _Nullable value,
                                                                     NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributePacketUnicastRxCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributePacketUnicastRxCountWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributePacketUnicastRxCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributePacketUnicastTxCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributePacketUnicastTxCountWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributePacketUnicastTxCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeCurrentMaxRateWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeCurrentMaxRateWithParams:(MTRSubscribeParams *)params
                           subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                     reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeCurrentMaxRateWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                endpoint:(NSNumber *)endpoint
                                                   queue:(dispatch_queue_t)queue
                                              completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeOverrunCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeOverrunCountWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeOverrunCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Ethernet Network Diagnostics
 *
 * The Ethernet Network Diagnostics Cluster provides a means to acquire standardized diagnostics metrics that MAY be used by a Node
 * to assist a user or Administrative Node in diagnosing potential problems.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterEthernetNetworkDiagnostics : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

/**
 * Command ResetCounts
 *
 * Reception of this command SHALL reset the attributes: PacketRxCount, PacketTxCount, TxErrCount, CollisionCount, OverrunCount to 0
 */
- (void)resetCountsWithParams:(MTREthernetNetworkDiagnosticsClusterResetCountsParams * _Nullable)params
                   completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)resetCountsWithCompletion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributePHYRateWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributePHYRateWithParams:(MTRSubscribeParams *)params
                    subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                              reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributePHYRateWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                       completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFullDuplexWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFullDuplexWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFullDuplexWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributePacketRxCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributePacketRxCountWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributePacketRxCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributePacketTxCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributePacketTxCountWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributePacketTxCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeTxErrCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeTxErrCountWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeTxErrCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeCollisionCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeCollisionCountWithParams:(MTRSubscribeParams *)params
                           subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                     reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeCollisionCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                endpoint:(NSNumber *)endpoint
                                                   queue:(dispatch_queue_t)queue
                                              completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeOverrunCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeOverrunCountWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeOverrunCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeCarrierDetectWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeCarrierDetectWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeCarrierDetectWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeTimeSinceResetWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeTimeSinceResetWithParams:(MTRSubscribeParams *)params
                           subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                     reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeTimeSinceResetWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                endpoint:(NSNumber *)endpoint
                                                   queue:(dispatch_queue_t)queue
                                              completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Bridged Device Basic Information
 *
 * This Cluster serves two purposes towards a Node communicating with a Bridge: indicate that the functionality on
          the Endpoint where it is placed (and its Parts) is bridged from a non-CHIP technology; and provide a centralized
          collection of attributes that the Node MAY collect to aid in conveying information regarding the Bridged Device to a user,
          such as the vendor name, the model name, or user-assigned name.
 */
API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4))
@interface MTRBaseClusterBridgedDeviceBasicInformation : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeVendorNameWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeVendorNameWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeVendorNameWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeVendorIDWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeVendorIDWithParams:(MTRSubscribeParams *)params
                     subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                               reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeVendorIDWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                          endpoint:(NSNumber *)endpoint
                                             queue:(dispatch_queue_t)queue
                                        completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeProductNameWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeProductNameWithParams:(MTRSubscribeParams *)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeProductNameWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeNodeLabelWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeNodeLabelWithValue:(NSString * _Nonnull)value
                              completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeNodeLabelWithValue:(NSString * _Nonnull)value
                                  params:(MTRWriteParams * _Nullable)params
                              completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeNodeLabelWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeNodeLabelWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeHardwareVersionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeHardwareVersionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeHardwareVersionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeHardwareVersionStringWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeHardwareVersionStringWithParams:(MTRSubscribeParams *)params
                                  subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                            reportHandler:
                                                (void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeHardwareVersionStringWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                       endpoint:(NSNumber *)endpoint
                                                          queue:(dispatch_queue_t)queue
                                                     completion:
                                                         (void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeSoftwareVersionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeSoftwareVersionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeSoftwareVersionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeSoftwareVersionStringWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeSoftwareVersionStringWithParams:(MTRSubscribeParams *)params
                                  subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                            reportHandler:
                                                (void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeSoftwareVersionStringWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                       endpoint:(NSNumber *)endpoint
                                                          queue:(dispatch_queue_t)queue
                                                     completion:
                                                         (void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeManufacturingDateWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeManufacturingDateWithParams:(MTRSubscribeParams *)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeManufacturingDateWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                                 completion:
                                                     (void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributePartNumberWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributePartNumberWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributePartNumberWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeProductURLWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeProductURLWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeProductURLWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeProductLabelWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeProductLabelWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeProductLabelWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeSerialNumberWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeSerialNumberWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeSerialNumberWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeReachableWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeReachableWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeReachableWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeUniqueIDWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeUniqueIDWithParams:(MTRSubscribeParams *)params
                     subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                               reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeUniqueIDWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                          endpoint:(NSNumber *)endpoint
                                             queue:(dispatch_queue_t)queue
                                        completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeProductAppearanceWithCompletion:
    (void (^)(MTRBridgedDeviceBasicInformationClusterProductAppearanceStruct * _Nullable value,
        NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeProductAppearanceWithParams:(MTRSubscribeParams *)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:
                                            (void (^)(
                                                MTRBridgedDeviceBasicInformationClusterProductAppearanceStruct * _Nullable value,
                                                NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)
    readAttributeProductAppearanceWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:
                                                 (void (^)(
                                                     MTRBridgedDeviceBasicInformationClusterProductAppearanceStruct * _Nullable value,
                                                     NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Switch
 *
 * This cluster exposes interactions with a switch device, for the purpose of using those interactions by other devices.
Two types of switch devices are supported: latching switch (e.g. rocker switch) and momentary switch (e.g. push button),
distinguished with their feature flags. Interactions with the switch device are exposed as attributes (for the latching switch) and
as events (for both types of switches). An interested party MAY subscribe to these attributes/events and thus be informed of the
interactions, and can perform actions based on this, for example by sending commands to perform an action such as controlling a
light or a window shade.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterSwitch : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeNumberOfPositionsWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeNumberOfPositionsWithParams:(MTRSubscribeParams *)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeNumberOfPositionsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                                 completion:
                                                     (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeCurrentPositionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeCurrentPositionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeCurrentPositionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeMultiPressMaxWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeMultiPressMaxWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeMultiPressMaxWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Administrator Commissioning
 *
 * Commands to trigger a Node to allow a new Administrator to commission it.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterAdministratorCommissioning : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

/**
 * Command OpenCommissioningWindow
 *
 * This command is used by a current Administrator to instruct a Node to go into commissioning mode using enhanced commissioning
 * method.
 */
- (void)openCommissioningWindowWithParams:(MTRAdministratorCommissioningClusterOpenCommissioningWindowParams *)params
                               completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command OpenBasicCommissioningWindow
 *
 * This command is used by a current Administrator to instruct a Node to go into commissioning mode using basic commissioning
 * method, if the node supports it.
 */
- (void)openBasicCommissioningWindowWithParams:(MTRAdministratorCommissioningClusterOpenBasicCommissioningWindowParams *)params
                                    completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command RevokeCommissioning
 *
 * This command is used by a current Administrator to instruct a Node to revoke any active Open Commissioning Window or Open Basic
 * Commissioning Window command.
 */
- (void)revokeCommissioningWithParams:(MTRAdministratorCommissioningClusterRevokeCommissioningParams * _Nullable)params
                           completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)revokeCommissioningWithCompletion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeWindowStatusWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeWindowStatusWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeWindowStatusWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAdminFabricIndexWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAdminFabricIndexWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAdminFabricIndexWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:
                                                    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAdminVendorIdWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAdminVendorIdWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAdminVendorIdWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Operational Credentials
 *
 * This cluster is used to add or remove Operational Credentials on a Commissionee or Node, as well as manage the associated
 * Fabrics.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterOperationalCredentials : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

/**
 * Command AttestationRequest
 *
 * Sender is requesting attestation information from the receiver.
 */
- (void)attestationRequestWithParams:(MTROperationalCredentialsClusterAttestationRequestParams *)params
                          completion:(void (^)(MTROperationalCredentialsClusterAttestationResponseParams * _Nullable data,
                                         NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command CertificateChainRequest
 *
 * Sender is requesting a device attestation certificate from the receiver.
 */
- (void)certificateChainRequestWithParams:(MTROperationalCredentialsClusterCertificateChainRequestParams *)params
                               completion:(void (^)(MTROperationalCredentialsClusterCertificateChainResponseParams * _Nullable data,
                                              NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command CSRRequest
 *
 * Sender is requesting a certificate signing request (CSR) from the receiver.
 */
- (void)CSRRequestWithParams:(MTROperationalCredentialsClusterCSRRequestParams *)params
                  completion:(void (^)(MTROperationalCredentialsClusterCSRResponseParams * _Nullable data,
                                 NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command AddNOC
 *
 * Sender is requesting to add the new node operational certificates.
 */
- (void)addNOCWithParams:(MTROperationalCredentialsClusterAddNOCParams *)params
              completion:(void (^)(MTROperationalCredentialsClusterNOCResponseParams * _Nullable data,
                             NSError * _Nullable error))completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command UpdateNOC
 *
 * Sender is requesting to update the node operational certificates.
 */
- (void)updateNOCWithParams:(MTROperationalCredentialsClusterUpdateNOCParams *)params
                 completion:(void (^)(MTROperationalCredentialsClusterNOCResponseParams * _Nullable data,
                                NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command UpdateFabricLabel
 *
 * This command SHALL be used by an Administrative Node to set the user-visible Label field for a given Fabric, as reflected by
 * entries in the Fabrics attribute.
 */
- (void)updateFabricLabelWithParams:(MTROperationalCredentialsClusterUpdateFabricLabelParams *)params
                         completion:(void (^)(MTROperationalCredentialsClusterNOCResponseParams * _Nullable data,
                                        NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command RemoveFabric
 *
 * This command is used by Administrative Nodes to remove a given fabric index and delete all associated fabric-scoped data.
 */
- (void)removeFabricWithParams:(MTROperationalCredentialsClusterRemoveFabricParams *)params
                    completion:(void (^)(MTROperationalCredentialsClusterNOCResponseParams * _Nullable data,
                                   NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command AddTrustedRootCertificate
 *
 * This command SHALL add a Trusted Root CA Certificate, provided as its CHIP Certificate representation.
 */
- (void)addTrustedRootCertificateWithParams:(MTROperationalCredentialsClusterAddTrustedRootCertificateParams *)params
                                 completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeNOCsWithParams:(MTRReadParams * _Nullable)params
                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeNOCsWithParams:(MTRSubscribeParams *)params
                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                           reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeNOCsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                      endpoint:(NSNumber *)endpoint
                                         queue:(dispatch_queue_t)queue
                                    completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFabricsWithParams:(MTRReadParams * _Nullable)params
                            completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFabricsWithParams:(MTRSubscribeParams *)params
                    subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                              reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFabricsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                       completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeSupportedFabricsWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeSupportedFabricsWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeSupportedFabricsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:
                                                    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeCommissionedFabricsWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeCommissionedFabricsWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeCommissionedFabricsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeTrustedRootCertificatesWithCompletion:
    (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeTrustedRootCertificatesWithParams:(MTRSubscribeParams *)params
                                    subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                              reportHandler:
                                                  (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeTrustedRootCertificatesWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                         endpoint:(NSNumber *)endpoint
                                                            queue:(dispatch_queue_t)queue
                                                       completion:(void (^)(NSArray * _Nullable value,
                                                                      NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeCurrentFabricIndexWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeCurrentFabricIndexWithParams:(MTRSubscribeParams *)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                         reportHandler:
                                             (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeCurrentFabricIndexWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                    endpoint:(NSNumber *)endpoint
                                                       queue:(dispatch_queue_t)queue
                                                  completion:
                                                      (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Group Key Management
 *
 * The Group Key Management Cluster is the mechanism by which group keys are managed.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterGroupKeyManagement : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

/**
 * Command KeySetWrite
 *
 * Write a new set of keys for the given key set id.
 */
- (void)keySetWriteWithParams:(MTRGroupKeyManagementClusterKeySetWriteParams *)params
                   completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command KeySetRead
 *
 * Read the keys for a given key set id.
 */
- (void)keySetReadWithParams:(MTRGroupKeyManagementClusterKeySetReadParams *)params
                  completion:(void (^)(MTRGroupKeyManagementClusterKeySetReadResponseParams * _Nullable data,
                                 NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command KeySetRemove
 *
 * Revoke a Root Key from a Group
 */
- (void)keySetRemoveWithParams:(MTRGroupKeyManagementClusterKeySetRemoveParams *)params
                    completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command KeySetReadAllIndices
 *
 * Return the list of Group Key Sets associated with the accessing fabric
 */
- (void)keySetReadAllIndicesWithParams:(MTRGroupKeyManagementClusterKeySetReadAllIndicesParams *)params
                            completion:(void (^)(MTRGroupKeyManagementClusterKeySetReadAllIndicesResponseParams * _Nullable data,
                                           NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGroupKeyMapWithParams:(MTRReadParams * _Nullable)params
                                completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeGroupKeyMapWithValue:(NSArray * _Nonnull)value
                                completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeGroupKeyMapWithValue:(NSArray * _Nonnull)value
                                    params:(MTRWriteParams * _Nullable)params
                                completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGroupKeyMapWithParams:(MTRSubscribeParams *)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGroupKeyMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGroupTableWithParams:(MTRReadParams * _Nullable)params
                               completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGroupTableWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGroupTableWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeMaxGroupsPerFabricWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeMaxGroupsPerFabricWithParams:(MTRSubscribeParams *)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                         reportHandler:
                                             (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeMaxGroupsPerFabricWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                    endpoint:(NSNumber *)endpoint
                                                       queue:(dispatch_queue_t)queue
                                                  completion:
                                                      (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeMaxGroupKeysPerFabricWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeMaxGroupKeysPerFabricWithParams:(MTRSubscribeParams *)params
                                  subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                            reportHandler:
                                                (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeMaxGroupKeysPerFabricWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                       endpoint:(NSNumber *)endpoint
                                                          queue:(dispatch_queue_t)queue
                                                     completion:
                                                         (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Fixed Label
 *
 * The Fixed Label Cluster provides a feature for the device to tag an endpoint with zero or more read only
labels.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterFixedLabel : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeLabelListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeLabelListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeLabelListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster User Label
 *
 * The User Label Cluster provides a feature to tag an endpoint with zero or more labels.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterUserLabel : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeLabelListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeLabelListWithValue:(NSArray * _Nonnull)value
                              completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeLabelListWithValue:(NSArray * _Nonnull)value
                                  params:(MTRWriteParams * _Nullable)params
                              completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeLabelListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeLabelListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Boolean State
 *
 * This cluster provides an interface to a boolean state called StateValue.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterBooleanState : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeStateValueWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeStateValueWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeStateValueWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Mode Select
 *
 * Attributes and commands for selecting a mode from a list of supported options.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterModeSelect : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

/**
 * Command ChangeToMode
 *
 * On receipt of this command, if the NewMode field matches the Mode field in an entry of the SupportedModes list, the server SHALL
 * set the CurrentMode attribute to the NewMode value, otherwise, the server SHALL respond with an INVALID_COMMAND status response.
 */
- (void)changeToModeWithParams:(MTRModeSelectClusterChangeToModeParams *)params
                    completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeDescriptionWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeDescriptionWithParams:(MTRSubscribeParams *)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeDescriptionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeStandardNamespaceWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeStandardNamespaceWithParams:(MTRSubscribeParams *)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeStandardNamespaceWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                                 completion:
                                                     (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeSupportedModesWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeSupportedModesWithParams:(MTRSubscribeParams *)params
                           subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                     reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeSupportedModesWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                endpoint:(NSNumber *)endpoint
                                                   queue:(dispatch_queue_t)queue
                                              completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeCurrentModeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeCurrentModeWithParams:(MTRSubscribeParams *)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeCurrentModeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeStartUpModeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeStartUpModeWithValue:(NSNumber * _Nullable)value
                                completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeStartUpModeWithValue:(NSNumber * _Nullable)value
                                    params:(MTRWriteParams * _Nullable)params
                                completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeStartUpModeWithParams:(MTRSubscribeParams *)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeStartUpModeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeOnModeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeOnModeWithValue:(NSNumber * _Nullable)value
                           completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeOnModeWithValue:(NSNumber * _Nullable)value
                               params:(MTRWriteParams * _Nullable)params
                           completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeOnModeWithParams:(MTRSubscribeParams *)params
                   subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                             reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeOnModeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                        endpoint:(NSNumber *)endpoint
                                           queue:(dispatch_queue_t)queue
                                      completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Temperature Control
 *
 * Attributes and commands for configuring the temperature control, and reporting temperature.
 */
MTR_NEWLY_AVAILABLE
@interface MTRBaseClusterTemperatureControl : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER MTR_NEWLY_AVAILABLE;

/**
 * Command SetTemperature
 *
 * Set Temperature
 */
- (void)setTemperatureWithParams:(MTRTemperatureControlClusterSetTemperatureParams * _Nullable)params
                      completion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeTemperatureSetpointWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeTemperatureSetpointWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:(void (^)(NSNumber * _Nullable value,
                                                            NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeTemperatureSetpointWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:(void (^)(NSNumber * _Nullable value,
                                                                  NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeMinTemperatureWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeMinTemperatureWithParams:(MTRSubscribeParams *)params
                           subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                     reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeMinTemperatureWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                endpoint:(NSNumber *)endpoint
                                                   queue:(dispatch_queue_t)queue
                                              completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeMaxTemperatureWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeMaxTemperatureWithParams:(MTRSubscribeParams *)params
                           subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                     reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeMaxTemperatureWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                endpoint:(NSNumber *)endpoint
                                                   queue:(dispatch_queue_t)queue
                                              completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeStepWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeStepWithParams:(MTRSubscribeParams *)params
                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                           reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeStepWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                      endpoint:(NSNumber *)endpoint
                                         queue:(dispatch_queue_t)queue
                                    completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeCurrentTemperatureLevelIndexWithCompletion:(void (^)(NSNumber * _Nullable value,
                                                                    NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeCurrentTemperatureLevelIndexWithParams:(MTRSubscribeParams *)params
                                         subscriptionEstablished:
                                             (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                   reportHandler:(void (^)(NSNumber * _Nullable value,
                                                                     NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeCurrentTemperatureLevelIndexWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                              endpoint:(NSNumber *)endpoint
                                                                 queue:(dispatch_queue_t)queue
                                                            completion:(void (^)(NSNumber * _Nullable value,
                                                                           NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeSupportedTemperatureLevelsWithCompletion:(void (^)(NSArray * _Nullable value,
                                                                  NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeSupportedTemperatureLevelsWithParams:(MTRSubscribeParams *)params
                                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                 reportHandler:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeSupportedTemperatureLevelsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                            endpoint:(NSNumber *)endpoint
                                                               queue:(dispatch_queue_t)queue
                                                          completion:(void (^)(NSArray * _Nullable value,
                                                                         NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:(void (^)(NSArray * _Nullable value,
                                                             NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:(void (^)(NSArray * _Nullable value,
                                                            NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:(void (^)(NSArray * _Nullable value,
                                                                  NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Refrigerator Alarm
 *
 * Attributes and commands for configuring the Refrigerator alarm.
 */
MTR_NEWLY_AVAILABLE
@interface MTRBaseClusterRefrigeratorAlarm : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER MTR_NEWLY_AVAILABLE;

/**
 * Command Reset
 *
 * Reset alarm
 */
- (void)resetWithParams:(MTRRefrigeratorAlarmClusterResetParams *)params
             completion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeMaskWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)writeAttributeMaskWithValue:(NSNumber * _Nonnull)value completion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;
- (void)writeAttributeMaskWithValue:(NSNumber * _Nonnull)value
                             params:(MTRWriteParams * _Nullable)params
                         completion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeMaskWithParams:(MTRSubscribeParams *)params
                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                           reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeMaskWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                      endpoint:(NSNumber *)endpoint
                                         queue:(dispatch_queue_t)queue
                                    completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeLatchWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeLatchWithParams:(MTRSubscribeParams *)params
                  subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                            reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeLatchWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                       endpoint:(NSNumber *)endpoint
                                          queue:(dispatch_queue_t)queue
                                     completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeStateWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeStateWithParams:(MTRSubscribeParams *)params
                  subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                            reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeStateWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                       endpoint:(NSNumber *)endpoint
                                          queue:(dispatch_queue_t)queue
                                     completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:(void (^)(NSArray * _Nullable value,
                                                             NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:(void (^)(NSArray * _Nullable value,
                                                            NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:(void (^)(NSArray * _Nullable value,
                                                                  NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Air Quality
 *
 * Attributes for reporting air quality classification
 */
MTR_NEWLY_AVAILABLE
@interface MTRBaseClusterAirQuality : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER MTR_NEWLY_AVAILABLE;

- (void)readAttributeAirQualityWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAirQualityWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAirQualityWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:(void (^)(NSArray * _Nullable value,
                                                             NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:(void (^)(NSArray * _Nullable value,
                                                            NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:(void (^)(NSArray * _Nullable value,
                                                                  NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Smoke CO Alarm
 *
 * This cluster provides an interface for observing and managing the state of smoke and CO alarms.
 */
MTR_NEWLY_AVAILABLE
@interface MTRBaseClusterSmokeCOAlarm : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER MTR_NEWLY_AVAILABLE;

/**
 * Command SelfTestRequest
 *
 * This command SHALL initiate a device self-test.
 */
- (void)selfTestRequestWithParams:(MTRSmokeCOAlarmClusterSelfTestRequestParams * _Nullable)params
                       completion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;
- (void)selfTestRequestWithCompletion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeExpressedStateWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeExpressedStateWithParams:(MTRSubscribeParams *)params
                           subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                     reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeExpressedStateWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                endpoint:(NSNumber *)endpoint
                                                   queue:(dispatch_queue_t)queue
                                              completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeSmokeStateWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeSmokeStateWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeSmokeStateWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeCOStateWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeCOStateWithParams:(MTRSubscribeParams *)params
                    subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                              reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeCOStateWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                       completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeBatteryAlertWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeBatteryAlertWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeBatteryAlertWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeDeviceMutedWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeDeviceMutedWithParams:(MTRSubscribeParams *)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeDeviceMutedWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeTestInProgressWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeTestInProgressWithParams:(MTRSubscribeParams *)params
                           subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                     reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeTestInProgressWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                endpoint:(NSNumber *)endpoint
                                                   queue:(dispatch_queue_t)queue
                                              completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeHardwareFaultAlertWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeHardwareFaultAlertWithParams:(MTRSubscribeParams *)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                         reportHandler:(void (^)(NSNumber * _Nullable value,
                                                           NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeHardwareFaultAlertWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                    endpoint:(NSNumber *)endpoint
                                                       queue:(dispatch_queue_t)queue
                                                  completion:(void (^)(NSNumber * _Nullable value,
                                                                 NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeEndOfServiceAlertWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEndOfServiceAlertWithParams:(MTRSubscribeParams *)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEndOfServiceAlertWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                                 completion:(void (^)(NSNumber * _Nullable value,
                                                                NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeInterconnectSmokeAlarmWithCompletion:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeInterconnectSmokeAlarmWithParams:(MTRSubscribeParams *)params
                                   subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                             reportHandler:(void (^)(NSNumber * _Nullable value,
                                                               NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeInterconnectSmokeAlarmWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                        endpoint:(NSNumber *)endpoint
                                                           queue:(dispatch_queue_t)queue
                                                      completion:(void (^)(NSNumber * _Nullable value,
                                                                     NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeInterconnectCOAlarmWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeInterconnectCOAlarmWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:(void (^)(NSNumber * _Nullable value,
                                                            NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeInterconnectCOAlarmWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:(void (^)(NSNumber * _Nullable value,
                                                                  NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeContaminationStateWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeContaminationStateWithParams:(MTRSubscribeParams *)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                         reportHandler:(void (^)(NSNumber * _Nullable value,
                                                           NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeContaminationStateWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                    endpoint:(NSNumber *)endpoint
                                                       queue:(dispatch_queue_t)queue
                                                  completion:(void (^)(NSNumber * _Nullable value,
                                                                 NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeSensitivityLevelWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)writeAttributeSensitivityLevelWithValue:(NSNumber * _Nonnull)value
                                     completion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;
- (void)writeAttributeSensitivityLevelWithValue:(NSNumber * _Nonnull)value
                                         params:(MTRWriteParams * _Nullable)params
                                     completion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeSensitivityLevelWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeSensitivityLevelWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:(void (^)(NSNumber * _Nullable value,
                                                               NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:(void (^)(NSArray * _Nullable value,
                                                             NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:(void (^)(NSArray * _Nullable value,
                                                            NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:(void (^)(NSArray * _Nullable value,
                                                                  NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Operational State
 *
 * This cluster supports remotely monitoring and, where supported, changing the operational state of any device where a state
 * machine is a part of the operation.
 */
MTR_NEWLY_AVAILABLE
@interface MTRBaseClusterOperationalState : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER MTR_NEWLY_AVAILABLE;

/**
 * Command Pause
 *
 * Upon receipt, the device SHALL pause its operation if it is possible based on the current function of the server.
 */
- (void)pauseWithParams:(MTROperationalStateClusterPauseParams * _Nullable)params
             completion:(void (^)(MTROperationalStateClusterOperationalCommandResponseParams * _Nullable data,
                            NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;
- (void)pauseWithCompletion:(void (^)(MTROperationalStateClusterOperationalCommandResponseParams * _Nullable data,
                                NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;
/**
 * Command Stop
 *
 * Upon receipt, the device SHALL stop its operation if it is at a position where it is safe to do so and/or permitted.
 */
- (void)stopWithParams:(MTROperationalStateClusterStopParams * _Nullable)params
            completion:(void (^)(MTROperationalStateClusterOperationalCommandResponseParams * _Nullable data,
                           NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;
- (void)stopWithCompletion:(void (^)(MTROperationalStateClusterOperationalCommandResponseParams * _Nullable data,
                               NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;
/**
 * Command Start
 *
 * Upon receipt, the device SHALL start its operation if it is safe to do so and the device is in an operational state from which it
 * can be started.
 */
- (void)startWithParams:(MTROperationalStateClusterStartParams * _Nullable)params
             completion:(void (^)(MTROperationalStateClusterOperationalCommandResponseParams * _Nullable data,
                            NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;
- (void)startWithCompletion:(void (^)(MTROperationalStateClusterOperationalCommandResponseParams * _Nullable data,
                                NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;
/**
 * Command Resume
 *
 * Upon receipt, the device SHALL resume its operation from the point it was at when it received the Pause command, or from the
 * point when it was paused by means outside of this cluster (for example by manual button press).
 */
- (void)resumeWithParams:(MTROperationalStateClusterResumeParams * _Nullable)params
              completion:(void (^)(MTROperationalStateClusterOperationalCommandResponseParams * _Nullable data,
                             NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;
- (void)resumeWithCompletion:(void (^)(MTROperationalStateClusterOperationalCommandResponseParams * _Nullable data,
                                 NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributePhaseListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributePhaseListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributePhaseListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeCurrentPhaseWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeCurrentPhaseWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeCurrentPhaseWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeCountdownTimeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeCountdownTimeWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeCountdownTimeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeOperationalStateListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeOperationalStateListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:(void (^)(NSArray * _Nullable value,
                                                             NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeOperationalStateListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeOperationalStateWithCompletion:(void (^)(MTROperationalStateClusterOperationalStateStruct * _Nullable value,
                                                        NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeOperationalStateWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(MTROperationalStateClusterOperationalStateStruct * _Nullable value,
                                                         NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeOperationalStateWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:
                                                    (void (^)(MTROperationalStateClusterOperationalStateStruct * _Nullable value,
                                                        NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeOperationalErrorWithCompletion:(void (^)(MTROperationalStateClusterErrorStateStruct * _Nullable value,
                                                        NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeOperationalErrorWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(MTROperationalStateClusterErrorStateStruct * _Nullable value,
                                                         NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeOperationalErrorWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:(void (^)(MTROperationalStateClusterErrorStateStruct * _Nullable value,
                                                               NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:(void (^)(NSArray * _Nullable value,
                                                             NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:(void (^)(NSArray * _Nullable value,
                                                            NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:(void (^)(NSArray * _Nullable value,
                                                                  NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster HEPA Filter Monitoring
 *
 * Attributes and commands for monitoring HEPA filters in a device
 */
MTR_NEWLY_AVAILABLE
@interface MTRBaseClusterHEPAFilterMonitoring : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER MTR_NEWLY_AVAILABLE;

/**
 * Command ResetCondition
 *
 * Reset the condition of the replaceable to the non degraded state
 */
- (void)resetConditionWithParams:(MTRHEPAFilterMonitoringClusterResetConditionParams * _Nullable)params
                      completion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;
- (void)resetConditionWithCompletion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeConditionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeConditionWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeConditionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeDegradationDirectionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeDegradationDirectionWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:(void (^)(NSNumber * _Nullable value,
                                                             NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeDegradationDirectionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:(void (^)(NSNumber * _Nullable value,
                                                                   NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeChangeIndicationWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeChangeIndicationWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeChangeIndicationWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:(void (^)(NSNumber * _Nullable value,
                                                               NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeInPlaceIndicatorWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeInPlaceIndicatorWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeInPlaceIndicatorWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:(void (^)(NSNumber * _Nullable value,
                                                               NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:(void (^)(NSArray * _Nullable value,
                                                             NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:(void (^)(NSArray * _Nullable value,
                                                            NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:(void (^)(NSArray * _Nullable value,
                                                                  NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Activated Carbon Filter Monitoring
 *
 * Attributes and commands for monitoring activated carbon filters in a device
 */
MTR_NEWLY_AVAILABLE
@interface MTRBaseClusterActivatedCarbonFilterMonitoring : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER MTR_NEWLY_AVAILABLE;

/**
 * Command ResetCondition
 *
 * Reset the condition of the replaceable to the non degraded state
 */
- (void)resetConditionWithParams:(MTRActivatedCarbonFilterMonitoringClusterResetConditionParams * _Nullable)params
                      completion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;
- (void)resetConditionWithCompletion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeConditionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeConditionWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeConditionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeDegradationDirectionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeDegradationDirectionWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:(void (^)(NSNumber * _Nullable value,
                                                             NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeDegradationDirectionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:(void (^)(NSNumber * _Nullable value,
                                                                   NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeChangeIndicationWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeChangeIndicationWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeChangeIndicationWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:(void (^)(NSNumber * _Nullable value,
                                                               NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeInPlaceIndicatorWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeInPlaceIndicatorWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeInPlaceIndicatorWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:(void (^)(NSNumber * _Nullable value,
                                                               NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:(void (^)(NSArray * _Nullable value,
                                                             NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:(void (^)(NSArray * _Nullable value,
                                                            NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:(void (^)(NSArray * _Nullable value,
                                                                  NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Ceramic Filter Monitoring
 *
 * Attributes and commands for monitoring ceramic filters in a device
 */
MTR_NEWLY_AVAILABLE
@interface MTRBaseClusterCeramicFilterMonitoring : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER MTR_NEWLY_AVAILABLE;

/**
 * Command ResetCondition
 *
 * Reset the condition of the replaceable to the non degraded state
 */
- (void)resetConditionWithParams:(MTRCeramicFilterMonitoringClusterResetConditionParams * _Nullable)params
                      completion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;
- (void)resetConditionWithCompletion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeConditionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeConditionWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeConditionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeDegradationDirectionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeDegradationDirectionWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:(void (^)(NSNumber * _Nullable value,
                                                             NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeDegradationDirectionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:(void (^)(NSNumber * _Nullable value,
                                                                   NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeChangeIndicationWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeChangeIndicationWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeChangeIndicationWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:(void (^)(NSNumber * _Nullable value,
                                                               NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeInPlaceIndicatorWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeInPlaceIndicatorWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeInPlaceIndicatorWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:(void (^)(NSNumber * _Nullable value,
                                                               NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:(void (^)(NSArray * _Nullable value,
                                                             NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:(void (^)(NSArray * _Nullable value,
                                                            NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:(void (^)(NSArray * _Nullable value,
                                                                  NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Electrostatic Filter Monitoring
 *
 * Attributes and commands for monitoring electrostatic filters in a device
 */
MTR_NEWLY_AVAILABLE
@interface MTRBaseClusterElectrostaticFilterMonitoring : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER MTR_NEWLY_AVAILABLE;

/**
 * Command ResetCondition
 *
 * Reset the condition of the replaceable to the non degraded state
 */
- (void)resetConditionWithParams:(MTRElectrostaticFilterMonitoringClusterResetConditionParams * _Nullable)params
                      completion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;
- (void)resetConditionWithCompletion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeConditionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeConditionWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeConditionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeDegradationDirectionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeDegradationDirectionWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:(void (^)(NSNumber * _Nullable value,
                                                             NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeDegradationDirectionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:(void (^)(NSNumber * _Nullable value,
                                                                   NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeChangeIndicationWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeChangeIndicationWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeChangeIndicationWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:(void (^)(NSNumber * _Nullable value,
                                                               NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeInPlaceIndicatorWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeInPlaceIndicatorWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeInPlaceIndicatorWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:(void (^)(NSNumber * _Nullable value,
                                                               NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:(void (^)(NSArray * _Nullable value,
                                                             NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:(void (^)(NSArray * _Nullable value,
                                                            NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:(void (^)(NSArray * _Nullable value,
                                                                  NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster UV Filter Monitoring
 *
 * Attributes and commands for monitoring UV filters in a device
 */
MTR_NEWLY_AVAILABLE
@interface MTRBaseClusterUVFilterMonitoring : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER MTR_NEWLY_AVAILABLE;

/**
 * Command ResetCondition
 *
 * Reset the condition of the replaceable to the non degraded state
 */
- (void)resetConditionWithParams:(MTRUVFilterMonitoringClusterResetConditionParams * _Nullable)params
                      completion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;
- (void)resetConditionWithCompletion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeConditionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeConditionWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeConditionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeDegradationDirectionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeDegradationDirectionWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:(void (^)(NSNumber * _Nullable value,
                                                             NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeDegradationDirectionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:(void (^)(NSNumber * _Nullable value,
                                                                   NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeChangeIndicationWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeChangeIndicationWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeChangeIndicationWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:(void (^)(NSNumber * _Nullable value,
                                                               NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeInPlaceIndicatorWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeInPlaceIndicatorWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeInPlaceIndicatorWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:(void (^)(NSNumber * _Nullable value,
                                                               NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:(void (^)(NSArray * _Nullable value,
                                                             NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:(void (^)(NSArray * _Nullable value,
                                                            NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:(void (^)(NSArray * _Nullable value,
                                                                  NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Ionizing Filter Monitoring
 *
 * Attributes and commands for monitoring ionizing filters in a device
 */
MTR_NEWLY_AVAILABLE
@interface MTRBaseClusterIonizingFilterMonitoring : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER MTR_NEWLY_AVAILABLE;

/**
 * Command ResetCondition
 *
 * Reset the condition of the replaceable to the non degraded state
 */
- (void)resetConditionWithParams:(MTRIonizingFilterMonitoringClusterResetConditionParams * _Nullable)params
                      completion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;
- (void)resetConditionWithCompletion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeConditionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeConditionWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeConditionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeDegradationDirectionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeDegradationDirectionWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:(void (^)(NSNumber * _Nullable value,
                                                             NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeDegradationDirectionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:(void (^)(NSNumber * _Nullable value,
                                                                   NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeChangeIndicationWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeChangeIndicationWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeChangeIndicationWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:(void (^)(NSNumber * _Nullable value,
                                                               NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeInPlaceIndicatorWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeInPlaceIndicatorWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeInPlaceIndicatorWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:(void (^)(NSNumber * _Nullable value,
                                                               NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:(void (^)(NSArray * _Nullable value,
                                                             NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:(void (^)(NSArray * _Nullable value,
                                                            NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:(void (^)(NSArray * _Nullable value,
                                                                  NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Zeolite Filter Monitoring
 *
 * Attributes and commands for monitoring zeolite filters in a device
 */
MTR_NEWLY_AVAILABLE
@interface MTRBaseClusterZeoliteFilterMonitoring : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER MTR_NEWLY_AVAILABLE;

/**
 * Command ResetCondition
 *
 * Reset the condition of the replaceable to the non degraded state
 */
- (void)resetConditionWithParams:(MTRZeoliteFilterMonitoringClusterResetConditionParams * _Nullable)params
                      completion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;
- (void)resetConditionWithCompletion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeConditionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeConditionWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeConditionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeDegradationDirectionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeDegradationDirectionWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:(void (^)(NSNumber * _Nullable value,
                                                             NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeDegradationDirectionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:(void (^)(NSNumber * _Nullable value,
                                                                   NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeChangeIndicationWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeChangeIndicationWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeChangeIndicationWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:(void (^)(NSNumber * _Nullable value,
                                                               NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeInPlaceIndicatorWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeInPlaceIndicatorWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeInPlaceIndicatorWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:(void (^)(NSNumber * _Nullable value,
                                                               NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:(void (^)(NSArray * _Nullable value,
                                                             NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:(void (^)(NSArray * _Nullable value,
                                                            NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:(void (^)(NSArray * _Nullable value,
                                                                  NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Ozone Filter Monitoring
 *
 * Attributes and commands for monitoring ozone filters in a device
 */
MTR_NEWLY_AVAILABLE
@interface MTRBaseClusterOzoneFilterMonitoring : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER MTR_NEWLY_AVAILABLE;

/**
 * Command ResetCondition
 *
 * Reset the condition of the replaceable to the non degraded state
 */
- (void)resetConditionWithParams:(MTROzoneFilterMonitoringClusterResetConditionParams * _Nullable)params
                      completion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;
- (void)resetConditionWithCompletion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeConditionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeConditionWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeConditionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeDegradationDirectionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeDegradationDirectionWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:(void (^)(NSNumber * _Nullable value,
                                                             NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeDegradationDirectionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:(void (^)(NSNumber * _Nullable value,
                                                                   NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeChangeIndicationWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeChangeIndicationWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeChangeIndicationWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:(void (^)(NSNumber * _Nullable value,
                                                               NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeInPlaceIndicatorWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeInPlaceIndicatorWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeInPlaceIndicatorWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:(void (^)(NSNumber * _Nullable value,
                                                               NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:(void (^)(NSArray * _Nullable value,
                                                             NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:(void (^)(NSArray * _Nullable value,
                                                            NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:(void (^)(NSArray * _Nullable value,
                                                                  NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Water Tank Monitoring
 *
 * Attributes and commands for monitoring water tanks in a device
 */
MTR_NEWLY_AVAILABLE
@interface MTRBaseClusterWaterTankMonitoring : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER MTR_NEWLY_AVAILABLE;

/**
 * Command ResetCondition
 *
 * Reset the condition of the replaceable to the non degraded state
 */
- (void)resetConditionWithParams:(MTRWaterTankMonitoringClusterResetConditionParams * _Nullable)params
                      completion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;
- (void)resetConditionWithCompletion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeConditionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeConditionWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeConditionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeDegradationDirectionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeDegradationDirectionWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:(void (^)(NSNumber * _Nullable value,
                                                             NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeDegradationDirectionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:(void (^)(NSNumber * _Nullable value,
                                                                   NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeChangeIndicationWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeChangeIndicationWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeChangeIndicationWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:(void (^)(NSNumber * _Nullable value,
                                                               NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeInPlaceIndicatorWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeInPlaceIndicatorWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeInPlaceIndicatorWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:(void (^)(NSNumber * _Nullable value,
                                                               NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:(void (^)(NSArray * _Nullable value,
                                                             NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:(void (^)(NSArray * _Nullable value,
                                                            NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:(void (^)(NSArray * _Nullable value,
                                                                  NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Fuel Tank Monitoring
 *
 * Attributes and commands for monitoring fuel tanks in a device
 */
MTR_NEWLY_AVAILABLE
@interface MTRBaseClusterFuelTankMonitoring : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER MTR_NEWLY_AVAILABLE;

/**
 * Command ResetCondition
 *
 * Reset the condition of the replaceable to the non degraded state
 */
- (void)resetConditionWithParams:(MTRFuelTankMonitoringClusterResetConditionParams * _Nullable)params
                      completion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;
- (void)resetConditionWithCompletion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeConditionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeConditionWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeConditionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeDegradationDirectionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeDegradationDirectionWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:(void (^)(NSNumber * _Nullable value,
                                                             NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeDegradationDirectionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:(void (^)(NSNumber * _Nullable value,
                                                                   NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeChangeIndicationWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeChangeIndicationWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeChangeIndicationWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:(void (^)(NSNumber * _Nullable value,
                                                               NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeInPlaceIndicatorWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeInPlaceIndicatorWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeInPlaceIndicatorWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:(void (^)(NSNumber * _Nullable value,
                                                               NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:(void (^)(NSArray * _Nullable value,
                                                             NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:(void (^)(NSArray * _Nullable value,
                                                            NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:(void (^)(NSArray * _Nullable value,
                                                                  NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Ink Cartridge Monitoring
 *
 * Attributes and commands for monitoring ink cartridges in a device
 */
MTR_NEWLY_AVAILABLE
@interface MTRBaseClusterInkCartridgeMonitoring : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER MTR_NEWLY_AVAILABLE;

/**
 * Command ResetCondition
 *
 * Reset the condition of the replaceable to the non degraded state
 */
- (void)resetConditionWithParams:(MTRInkCartridgeMonitoringClusterResetConditionParams * _Nullable)params
                      completion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;
- (void)resetConditionWithCompletion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeConditionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeConditionWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeConditionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeDegradationDirectionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeDegradationDirectionWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:(void (^)(NSNumber * _Nullable value,
                                                             NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeDegradationDirectionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:(void (^)(NSNumber * _Nullable value,
                                                                   NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeChangeIndicationWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeChangeIndicationWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeChangeIndicationWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:(void (^)(NSNumber * _Nullable value,
                                                               NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeInPlaceIndicatorWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeInPlaceIndicatorWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeInPlaceIndicatorWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:(void (^)(NSNumber * _Nullable value,
                                                               NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:(void (^)(NSArray * _Nullable value,
                                                             NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:(void (^)(NSArray * _Nullable value,
                                                            NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:(void (^)(NSArray * _Nullable value,
                                                                  NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Toner Cartridge Monitoring
 *
 * Attributes and commands for monitoring toner cartridges in a device
 */
MTR_NEWLY_AVAILABLE
@interface MTRBaseClusterTonerCartridgeMonitoring : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER MTR_NEWLY_AVAILABLE;

/**
 * Command ResetCondition
 *
 * Reset the condition of the replaceable to the non degraded state
 */
- (void)resetConditionWithParams:(MTRTonerCartridgeMonitoringClusterResetConditionParams * _Nullable)params
                      completion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;
- (void)resetConditionWithCompletion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeConditionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeConditionWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeConditionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeDegradationDirectionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeDegradationDirectionWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:(void (^)(NSNumber * _Nullable value,
                                                             NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeDegradationDirectionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:(void (^)(NSNumber * _Nullable value,
                                                                   NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeChangeIndicationWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeChangeIndicationWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeChangeIndicationWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:(void (^)(NSNumber * _Nullable value,
                                                               NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeInPlaceIndicatorWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeInPlaceIndicatorWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeInPlaceIndicatorWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:(void (^)(NSNumber * _Nullable value,
                                                               NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:(void (^)(NSArray * _Nullable value,
                                                             NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:(void (^)(NSArray * _Nullable value,
                                                            NSError * _Nullable error))reportHandler MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:(void (^)(NSArray * _Nullable value,
                                                                  NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completion MTR_NEWLY_AVAILABLE;

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Door Lock
 *
 * An interface to a generic way to secure a door
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterDoorLock : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

/**
 * Command LockDoor
 *
 * This command causes the lock device to lock the door.
 */
- (void)lockDoorWithParams:(MTRDoorLockClusterLockDoorParams * _Nullable)params
                completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command UnlockDoor
 *
 * This command causes the lock device to unlock the door.
 */
- (void)unlockDoorWithParams:(MTRDoorLockClusterUnlockDoorParams * _Nullable)params
                  completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command UnlockWithTimeout
 *
 * This command causes the lock device to unlock the door with a timeout parameter.
 */
- (void)unlockWithTimeoutWithParams:(MTRDoorLockClusterUnlockWithTimeoutParams *)params
                         completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command SetWeekDaySchedule
 *
 * Set a weekly repeating schedule for a specified user.
 */
- (void)setWeekDayScheduleWithParams:(MTRDoorLockClusterSetWeekDayScheduleParams *)params
                          completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command GetWeekDaySchedule
 *
 * Retrieve the specific weekly schedule for the specific user.
 */
- (void)getWeekDayScheduleWithParams:(MTRDoorLockClusterGetWeekDayScheduleParams *)params
                          completion:(void (^)(MTRDoorLockClusterGetWeekDayScheduleResponseParams * _Nullable data,
                                         NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command ClearWeekDaySchedule
 *
 * Clear the specific weekly schedule or all weekly schedules for the specific user.
 */
- (void)clearWeekDayScheduleWithParams:(MTRDoorLockClusterClearWeekDayScheduleParams *)params
                            completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command SetYearDaySchedule
 *
 * Set a time-specific schedule ID for a specified user.
 */
- (void)setYearDayScheduleWithParams:(MTRDoorLockClusterSetYearDayScheduleParams *)params
                          completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command GetYearDaySchedule
 *
 * Returns the year day schedule data for the specified schedule and user indexes.
 */
- (void)getYearDayScheduleWithParams:(MTRDoorLockClusterGetYearDayScheduleParams *)params
                          completion:(void (^)(MTRDoorLockClusterGetYearDayScheduleResponseParams * _Nullable data,
                                         NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command ClearYearDaySchedule
 *
 * Clears the specific year day schedule or all year day schedules for the specific user.
 */
- (void)clearYearDayScheduleWithParams:(MTRDoorLockClusterClearYearDayScheduleParams *)params
                            completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command SetHolidaySchedule
 *
 * Set the holiday Schedule by specifying local start time and local end time with respect to any Lock Operating Mode.
 */
- (void)setHolidayScheduleWithParams:(MTRDoorLockClusterSetHolidayScheduleParams *)params
                          completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command GetHolidaySchedule
 *
 * Get the holiday schedule for the specified index.
 */
- (void)getHolidayScheduleWithParams:(MTRDoorLockClusterGetHolidayScheduleParams *)params
                          completion:(void (^)(MTRDoorLockClusterGetHolidayScheduleResponseParams * _Nullable data,
                                         NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command ClearHolidaySchedule
 *
 * Clears the holiday schedule or all holiday schedules.
 */
- (void)clearHolidayScheduleWithParams:(MTRDoorLockClusterClearHolidayScheduleParams *)params
                            completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command SetUser
 *
 * Set User into the lock.
 */
- (void)setUserWithParams:(MTRDoorLockClusterSetUserParams *)params
               completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command GetUser
 *
 * Retrieve User.
 */
- (void)getUserWithParams:(MTRDoorLockClusterGetUserParams *)params
               completion:(void (^)(MTRDoorLockClusterGetUserResponseParams * _Nullable data, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command ClearUser
 *
 * Clears a User or all Users.
 */
- (void)clearUserWithParams:(MTRDoorLockClusterClearUserParams *)params
                 completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command SetCredential
 *
 * Set a credential (e.g. PIN, RFID, Fingerprint, etc.) into the lock for a new user, existing user, or ProgrammingUser.
 */
- (void)setCredentialWithParams:(MTRDoorLockClusterSetCredentialParams *)params
                     completion:(void (^)(MTRDoorLockClusterSetCredentialResponseParams * _Nullable data,
                                    NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command GetCredentialStatus
 *
 * Retrieve the status of a particular credential (e.g. PIN, RFID, Fingerprint, etc.) by index.
 */
- (void)getCredentialStatusWithParams:(MTRDoorLockClusterGetCredentialStatusParams *)params
                           completion:(void (^)(MTRDoorLockClusterGetCredentialStatusResponseParams * _Nullable data,
                                          NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command ClearCredential
 *
 * Clear one, one type, or all credentials except ProgrammingPIN credential.
 */
- (void)clearCredentialWithParams:(MTRDoorLockClusterClearCredentialParams *)params
                       completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command UnboltDoor
 *
 * This command causes the lock device to unlock the door without pulling the latch.
 */
- (void)unboltDoorWithParams:(MTRDoorLockClusterUnboltDoorParams * _Nullable)params
                  completion:(MTRStatusCompletion)completion MTR_NEWLY_AVAILABLE;

- (void)readAttributeLockStateWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeLockStateWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeLockStateWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeLockTypeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeLockTypeWithParams:(MTRSubscribeParams *)params
                     subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                               reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeLockTypeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                          endpoint:(NSNumber *)endpoint
                                             queue:(dispatch_queue_t)queue
                                        completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeActuatorEnabledWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeActuatorEnabledWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeActuatorEnabledWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeDoorStateWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeDoorStateWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeDoorStateWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeDoorOpenEventsWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeDoorOpenEventsWithValue:(NSNumber * _Nonnull)value
                                   completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeDoorOpenEventsWithValue:(NSNumber * _Nonnull)value
                                       params:(MTRWriteParams * _Nullable)params
                                   completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeDoorOpenEventsWithParams:(MTRSubscribeParams *)params
                           subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                     reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeDoorOpenEventsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                endpoint:(NSNumber *)endpoint
                                                   queue:(dispatch_queue_t)queue
                                              completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeDoorClosedEventsWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeDoorClosedEventsWithValue:(NSNumber * _Nonnull)value
                                     completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeDoorClosedEventsWithValue:(NSNumber * _Nonnull)value
                                         params:(MTRWriteParams * _Nullable)params
                                     completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeDoorClosedEventsWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeDoorClosedEventsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:
                                                    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeOpenPeriodWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeOpenPeriodWithValue:(NSNumber * _Nonnull)value
                               completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeOpenPeriodWithValue:(NSNumber * _Nonnull)value
                                   params:(MTRWriteParams * _Nullable)params
                               completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeOpenPeriodWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeOpenPeriodWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeNumberOfTotalUsersSupportedWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeNumberOfTotalUsersSupportedWithParams:(MTRSubscribeParams *)params
                                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                  reportHandler:
                                                      (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeNumberOfTotalUsersSupportedWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                             endpoint:(NSNumber *)endpoint
                                                                queue:(dispatch_queue_t)queue
                                                           completion:(void (^)(NSNumber * _Nullable value,
                                                                          NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeNumberOfPINUsersSupportedWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeNumberOfPINUsersSupportedWithParams:(MTRSubscribeParams *)params
                                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                reportHandler:
                                                    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeNumberOfPINUsersSupportedWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                           endpoint:(NSNumber *)endpoint
                                                              queue:(dispatch_queue_t)queue
                                                         completion:(void (^)(NSNumber * _Nullable value,
                                                                        NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeNumberOfRFIDUsersSupportedWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeNumberOfRFIDUsersSupportedWithParams:(MTRSubscribeParams *)params
                                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                 reportHandler:
                                                     (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeNumberOfRFIDUsersSupportedWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                            endpoint:(NSNumber *)endpoint
                                                               queue:(dispatch_queue_t)queue
                                                          completion:(void (^)(NSNumber * _Nullable value,
                                                                         NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeNumberOfWeekDaySchedulesSupportedPerUserWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeNumberOfWeekDaySchedulesSupportedPerUserWithParams:(MTRSubscribeParams *)params
                                                     subscriptionEstablished:
                                                         (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                               reportHandler:(void (^)(NSNumber * _Nullable value,
                                                                                 NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeNumberOfWeekDaySchedulesSupportedPerUserWithClusterStateCache:
            (MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                                          endpoint:(NSNumber *)endpoint
                                                                             queue:(dispatch_queue_t)queue
                                                                        completion:(void (^)(NSNumber * _Nullable value,
                                                                                       NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeNumberOfYearDaySchedulesSupportedPerUserWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeNumberOfYearDaySchedulesSupportedPerUserWithParams:(MTRSubscribeParams *)params
                                                     subscriptionEstablished:
                                                         (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                               reportHandler:(void (^)(NSNumber * _Nullable value,
                                                                                 NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeNumberOfYearDaySchedulesSupportedPerUserWithClusterStateCache:
            (MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                                          endpoint:(NSNumber *)endpoint
                                                                             queue:(dispatch_queue_t)queue
                                                                        completion:(void (^)(NSNumber * _Nullable value,
                                                                                       NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeNumberOfHolidaySchedulesSupportedWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeNumberOfHolidaySchedulesSupportedWithParams:(MTRSubscribeParams *)params
                                              subscriptionEstablished:
                                                  (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                        reportHandler:(void (^)(NSNumber * _Nullable value,
                                                                          NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeNumberOfHolidaySchedulesSupportedWithClusterStateCache:
            (MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                                   endpoint:(NSNumber *)endpoint
                                                                      queue:(dispatch_queue_t)queue
                                                                 completion:(void (^)(NSNumber * _Nullable value,
                                                                                NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeMaxPINCodeLengthWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeMaxPINCodeLengthWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeMaxPINCodeLengthWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:
                                                    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeMinPINCodeLengthWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeMinPINCodeLengthWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeMinPINCodeLengthWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:
                                                    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeMaxRFIDCodeLengthWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeMaxRFIDCodeLengthWithParams:(MTRSubscribeParams *)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeMaxRFIDCodeLengthWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                                 completion:
                                                     (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeMinRFIDCodeLengthWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeMinRFIDCodeLengthWithParams:(MTRSubscribeParams *)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeMinRFIDCodeLengthWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                                 completion:
                                                     (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeCredentialRulesSupportWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeCredentialRulesSupportWithParams:(MTRSubscribeParams *)params
                                   subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                             reportHandler:
                                                 (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeCredentialRulesSupportWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                        endpoint:(NSNumber *)endpoint
                                                           queue:(dispatch_queue_t)queue
                                                      completion:(void (^)(NSNumber * _Nullable value,
                                                                     NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeNumberOfCredentialsSupportedPerUserWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeNumberOfCredentialsSupportedPerUserWithParams:(MTRSubscribeParams *)params
                                                subscriptionEstablished:
                                                    (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                          reportHandler:(void (^)(NSNumber * _Nullable value,
                                                                            NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeNumberOfCredentialsSupportedPerUserWithClusterStateCache:
            (MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                                     endpoint:(NSNumber *)endpoint
                                                                        queue:(dispatch_queue_t)queue
                                                                   completion:(void (^)(NSNumber * _Nullable value,
                                                                                  NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeLanguageWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeLanguageWithValue:(NSString * _Nonnull)value
                             completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeLanguageWithValue:(NSString * _Nonnull)value
                                 params:(MTRWriteParams * _Nullable)params
                             completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeLanguageWithParams:(MTRSubscribeParams *)params
                     subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                               reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeLanguageWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                          endpoint:(NSNumber *)endpoint
                                             queue:(dispatch_queue_t)queue
                                        completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeLEDSettingsWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeLEDSettingsWithValue:(NSNumber * _Nonnull)value
                                completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeLEDSettingsWithValue:(NSNumber * _Nonnull)value
                                    params:(MTRWriteParams * _Nullable)params
                                completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeLEDSettingsWithParams:(MTRSubscribeParams *)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeLEDSettingsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAutoRelockTimeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeAutoRelockTimeWithValue:(NSNumber * _Nonnull)value
                                   completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeAutoRelockTimeWithValue:(NSNumber * _Nonnull)value
                                       params:(MTRWriteParams * _Nullable)params
                                   completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAutoRelockTimeWithParams:(MTRSubscribeParams *)params
                           subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                     reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAutoRelockTimeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                endpoint:(NSNumber *)endpoint
                                                   queue:(dispatch_queue_t)queue
                                              completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeSoundVolumeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeSoundVolumeWithValue:(NSNumber * _Nonnull)value
                                completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeSoundVolumeWithValue:(NSNumber * _Nonnull)value
                                    params:(MTRWriteParams * _Nullable)params
                                completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeSoundVolumeWithParams:(MTRSubscribeParams *)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeSoundVolumeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeOperatingModeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeOperatingModeWithValue:(NSNumber * _Nonnull)value
                                  completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeOperatingModeWithValue:(NSNumber * _Nonnull)value
                                      params:(MTRWriteParams * _Nullable)params
                                  completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeOperatingModeWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeOperatingModeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeSupportedOperatingModesWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeSupportedOperatingModesWithParams:(MTRSubscribeParams *)params
                                    subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                              reportHandler:
                                                  (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeSupportedOperatingModesWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                         endpoint:(NSNumber *)endpoint
                                                            queue:(dispatch_queue_t)queue
                                                       completion:(void (^)(NSNumber * _Nullable value,
                                                                      NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeDefaultConfigurationRegisterWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeDefaultConfigurationRegisterWithParams:(MTRSubscribeParams *)params
                                         subscriptionEstablished:
                                             (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                   reportHandler:(void (^)(NSNumber * _Nullable value,
                                                                     NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeDefaultConfigurationRegisterWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                              endpoint:(NSNumber *)endpoint
                                                                 queue:(dispatch_queue_t)queue
                                                            completion:(void (^)(NSNumber * _Nullable value,
                                                                           NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEnableLocalProgrammingWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeEnableLocalProgrammingWithValue:(NSNumber * _Nonnull)value
                                           completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeEnableLocalProgrammingWithValue:(NSNumber * _Nonnull)value
                                               params:(MTRWriteParams * _Nullable)params
                                           completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeEnableLocalProgrammingWithParams:(MTRSubscribeParams *)params
                                   subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                             reportHandler:
                                                 (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeEnableLocalProgrammingWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                        endpoint:(NSNumber *)endpoint
                                                           queue:(dispatch_queue_t)queue
                                                      completion:(void (^)(NSNumber * _Nullable value,
                                                                     NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEnableOneTouchLockingWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeEnableOneTouchLockingWithValue:(NSNumber * _Nonnull)value
                                          completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeEnableOneTouchLockingWithValue:(NSNumber * _Nonnull)value
                                              params:(MTRWriteParams * _Nullable)params
                                          completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeEnableOneTouchLockingWithParams:(MTRSubscribeParams *)params
                                  subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                            reportHandler:
                                                (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeEnableOneTouchLockingWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                       endpoint:(NSNumber *)endpoint
                                                          queue:(dispatch_queue_t)queue
                                                     completion:
                                                         (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEnableInsideStatusLEDWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeEnableInsideStatusLEDWithValue:(NSNumber * _Nonnull)value
                                          completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeEnableInsideStatusLEDWithValue:(NSNumber * _Nonnull)value
                                              params:(MTRWriteParams * _Nullable)params
                                          completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeEnableInsideStatusLEDWithParams:(MTRSubscribeParams *)params
                                  subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                            reportHandler:
                                                (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeEnableInsideStatusLEDWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                       endpoint:(NSNumber *)endpoint
                                                          queue:(dispatch_queue_t)queue
                                                     completion:
                                                         (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEnablePrivacyModeButtonWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeEnablePrivacyModeButtonWithValue:(NSNumber * _Nonnull)value
                                            completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeEnablePrivacyModeButtonWithValue:(NSNumber * _Nonnull)value
                                                params:(MTRWriteParams * _Nullable)params
                                            completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeEnablePrivacyModeButtonWithParams:(MTRSubscribeParams *)params
                                    subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                              reportHandler:
                                                  (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeEnablePrivacyModeButtonWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                         endpoint:(NSNumber *)endpoint
                                                            queue:(dispatch_queue_t)queue
                                                       completion:(void (^)(NSNumber * _Nullable value,
                                                                      NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeLocalProgrammingFeaturesWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeLocalProgrammingFeaturesWithValue:(NSNumber * _Nonnull)value
                                             completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeLocalProgrammingFeaturesWithValue:(NSNumber * _Nonnull)value
                                                 params:(MTRWriteParams * _Nullable)params
                                             completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeLocalProgrammingFeaturesWithParams:(MTRSubscribeParams *)params
                                     subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                               reportHandler:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeLocalProgrammingFeaturesWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                          endpoint:(NSNumber *)endpoint
                                                             queue:(dispatch_queue_t)queue
                                                        completion:(void (^)(NSNumber * _Nullable value,
                                                                       NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeWrongCodeEntryLimitWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeWrongCodeEntryLimitWithValue:(NSNumber * _Nonnull)value
                                        completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeWrongCodeEntryLimitWithValue:(NSNumber * _Nonnull)value
                                            params:(MTRWriteParams * _Nullable)params
                                        completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeWrongCodeEntryLimitWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeWrongCodeEntryLimitWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeUserCodeTemporaryDisableTimeWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeUserCodeTemporaryDisableTimeWithValue:(NSNumber * _Nonnull)value
                                                 completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeUserCodeTemporaryDisableTimeWithValue:(NSNumber * _Nonnull)value
                                                     params:(MTRWriteParams * _Nullable)params
                                                 completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeUserCodeTemporaryDisableTimeWithParams:(MTRSubscribeParams *)params
                                         subscriptionEstablished:
                                             (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                   reportHandler:(void (^)(NSNumber * _Nullable value,
                                                                     NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeUserCodeTemporaryDisableTimeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                              endpoint:(NSNumber *)endpoint
                                                                 queue:(dispatch_queue_t)queue
                                                            completion:(void (^)(NSNumber * _Nullable value,
                                                                           NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeSendPINOverTheAirWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeSendPINOverTheAirWithValue:(NSNumber * _Nonnull)value
                                      completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeSendPINOverTheAirWithValue:(NSNumber * _Nonnull)value
                                          params:(MTRWriteParams * _Nullable)params
                                      completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeSendPINOverTheAirWithParams:(MTRSubscribeParams *)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeSendPINOverTheAirWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                                 completion:
                                                     (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeRequirePINforRemoteOperationWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeRequirePINforRemoteOperationWithValue:(NSNumber * _Nonnull)value
                                                 completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeRequirePINforRemoteOperationWithValue:(NSNumber * _Nonnull)value
                                                     params:(MTRWriteParams * _Nullable)params
                                                 completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeRequirePINforRemoteOperationWithParams:(MTRSubscribeParams *)params
                                         subscriptionEstablished:
                                             (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                   reportHandler:(void (^)(NSNumber * _Nullable value,
                                                                     NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeRequirePINforRemoteOperationWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                              endpoint:(NSNumber *)endpoint
                                                                 queue:(dispatch_queue_t)queue
                                                            completion:(void (^)(NSNumber * _Nullable value,
                                                                           NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeExpiringUserTimeoutWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeExpiringUserTimeoutWithValue:(NSNumber * _Nonnull)value
                                        completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeExpiringUserTimeoutWithValue:(NSNumber * _Nonnull)value
                                            params:(MTRWriteParams * _Nullable)params
                                        completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeExpiringUserTimeoutWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeExpiringUserTimeoutWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Window Covering
 *
 * Provides an interface for controlling and adjusting automatic window coverings.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterWindowCovering : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

/**
 * Command UpOrOpen
 *
 * Moves window covering to InstalledOpenLimitLift and InstalledOpenLimitTilt
 */
- (void)upOrOpenWithParams:(MTRWindowCoveringClusterUpOrOpenParams * _Nullable)params
                completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)upOrOpenWithCompletion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command DownOrClose
 *
 * Moves window covering to InstalledClosedLimitLift and InstalledCloseLimitTilt
 */
- (void)downOrCloseWithParams:(MTRWindowCoveringClusterDownOrCloseParams * _Nullable)params
                   completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)downOrCloseWithCompletion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command StopMotion
 *
 * Stop any adjusting of window covering
 */
- (void)stopMotionWithParams:(MTRWindowCoveringClusterStopMotionParams * _Nullable)params
                  completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)stopMotionWithCompletion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command GoToLiftValue
 *
 * Go to lift value specified
 */
- (void)goToLiftValueWithParams:(MTRWindowCoveringClusterGoToLiftValueParams *)params
                     completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command GoToLiftPercentage
 *
 * Go to lift percentage specified
 */
- (void)goToLiftPercentageWithParams:(MTRWindowCoveringClusterGoToLiftPercentageParams *)params
                          completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command GoToTiltValue
 *
 * Go to tilt value specified
 */
- (void)goToTiltValueWithParams:(MTRWindowCoveringClusterGoToTiltValueParams *)params
                     completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command GoToTiltPercentage
 *
 * Go to tilt percentage specified
 */
- (void)goToTiltPercentageWithParams:(MTRWindowCoveringClusterGoToTiltPercentageParams *)params
                          completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeTypeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeTypeWithParams:(MTRSubscribeParams *)params
                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                           reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeTypeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                      endpoint:(NSNumber *)endpoint
                                         queue:(dispatch_queue_t)queue
                                    completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributePhysicalClosedLimitLiftWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributePhysicalClosedLimitLiftWithParams:(MTRSubscribeParams *)params
                                    subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                              reportHandler:
                                                  (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributePhysicalClosedLimitLiftWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                         endpoint:(NSNumber *)endpoint
                                                            queue:(dispatch_queue_t)queue
                                                       completion:(void (^)(NSNumber * _Nullable value,
                                                                      NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributePhysicalClosedLimitTiltWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributePhysicalClosedLimitTiltWithParams:(MTRSubscribeParams *)params
                                    subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                              reportHandler:
                                                  (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributePhysicalClosedLimitTiltWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                         endpoint:(NSNumber *)endpoint
                                                            queue:(dispatch_queue_t)queue
                                                       completion:(void (^)(NSNumber * _Nullable value,
                                                                      NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeCurrentPositionLiftWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeCurrentPositionLiftWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeCurrentPositionLiftWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeCurrentPositionTiltWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeCurrentPositionTiltWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeCurrentPositionTiltWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeNumberOfActuationsLiftWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeNumberOfActuationsLiftWithParams:(MTRSubscribeParams *)params
                                   subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                             reportHandler:
                                                 (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeNumberOfActuationsLiftWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                        endpoint:(NSNumber *)endpoint
                                                           queue:(dispatch_queue_t)queue
                                                      completion:(void (^)(NSNumber * _Nullable value,
                                                                     NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeNumberOfActuationsTiltWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeNumberOfActuationsTiltWithParams:(MTRSubscribeParams *)params
                                   subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                             reportHandler:
                                                 (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeNumberOfActuationsTiltWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                        endpoint:(NSNumber *)endpoint
                                                           queue:(dispatch_queue_t)queue
                                                      completion:(void (^)(NSNumber * _Nullable value,
                                                                     NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeConfigStatusWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeConfigStatusWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeConfigStatusWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeCurrentPositionLiftPercentageWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeCurrentPositionLiftPercentageWithParams:(MTRSubscribeParams *)params
                                          subscriptionEstablished:
                                              (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                    reportHandler:(void (^)(NSNumber * _Nullable value,
                                                                      NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeCurrentPositionLiftPercentageWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                               endpoint:(NSNumber *)endpoint
                                                                  queue:(dispatch_queue_t)queue
                                                             completion:(void (^)(NSNumber * _Nullable value,
                                                                            NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeCurrentPositionTiltPercentageWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeCurrentPositionTiltPercentageWithParams:(MTRSubscribeParams *)params
                                          subscriptionEstablished:
                                              (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                    reportHandler:(void (^)(NSNumber * _Nullable value,
                                                                      NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeCurrentPositionTiltPercentageWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                               endpoint:(NSNumber *)endpoint
                                                                  queue:(dispatch_queue_t)queue
                                                             completion:(void (^)(NSNumber * _Nullable value,
                                                                            NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeOperationalStatusWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeOperationalStatusWithParams:(MTRSubscribeParams *)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeOperationalStatusWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                                 completion:
                                                     (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeTargetPositionLiftPercent100thsWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeTargetPositionLiftPercent100thsWithParams:(MTRSubscribeParams *)params
                                            subscriptionEstablished:
                                                (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                      reportHandler:(void (^)(NSNumber * _Nullable value,
                                                                        NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeTargetPositionLiftPercent100thsWithClusterStateCache:
            (MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                                 endpoint:(NSNumber *)endpoint
                                                                    queue:(dispatch_queue_t)queue
                                                               completion:(void (^)(NSNumber * _Nullable value,
                                                                              NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeTargetPositionTiltPercent100thsWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeTargetPositionTiltPercent100thsWithParams:(MTRSubscribeParams *)params
                                            subscriptionEstablished:
                                                (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                      reportHandler:(void (^)(NSNumber * _Nullable value,
                                                                        NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeTargetPositionTiltPercent100thsWithClusterStateCache:
            (MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                                 endpoint:(NSNumber *)endpoint
                                                                    queue:(dispatch_queue_t)queue
                                                               completion:(void (^)(NSNumber * _Nullable value,
                                                                              NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEndProductTypeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeEndProductTypeWithParams:(MTRSubscribeParams *)params
                           subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                     reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeEndProductTypeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                endpoint:(NSNumber *)endpoint
                                                   queue:(dispatch_queue_t)queue
                                              completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeCurrentPositionLiftPercent100thsWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeCurrentPositionLiftPercent100thsWithParams:(MTRSubscribeParams *)params
                                             subscriptionEstablished:
                                                 (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                       reportHandler:(void (^)(NSNumber * _Nullable value,
                                                                         NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeCurrentPositionLiftPercent100thsWithClusterStateCache:
            (MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                                  endpoint:(NSNumber *)endpoint
                                                                     queue:(dispatch_queue_t)queue
                                                                completion:(void (^)(NSNumber * _Nullable value,
                                                                               NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeCurrentPositionTiltPercent100thsWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeCurrentPositionTiltPercent100thsWithParams:(MTRSubscribeParams *)params
                                             subscriptionEstablished:
                                                 (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                       reportHandler:(void (^)(NSNumber * _Nullable value,
                                                                         NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeCurrentPositionTiltPercent100thsWithClusterStateCache:
            (MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                                  endpoint:(NSNumber *)endpoint
                                                                     queue:(dispatch_queue_t)queue
                                                                completion:(void (^)(NSNumber * _Nullable value,
                                                                               NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeInstalledOpenLimitLiftWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeInstalledOpenLimitLiftWithParams:(MTRSubscribeParams *)params
                                   subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                             reportHandler:
                                                 (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeInstalledOpenLimitLiftWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                        endpoint:(NSNumber *)endpoint
                                                           queue:(dispatch_queue_t)queue
                                                      completion:(void (^)(NSNumber * _Nullable value,
                                                                     NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeInstalledClosedLimitLiftWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeInstalledClosedLimitLiftWithParams:(MTRSubscribeParams *)params
                                     subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                               reportHandler:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeInstalledClosedLimitLiftWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                          endpoint:(NSNumber *)endpoint
                                                             queue:(dispatch_queue_t)queue
                                                        completion:(void (^)(NSNumber * _Nullable value,
                                                                       NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeInstalledOpenLimitTiltWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeInstalledOpenLimitTiltWithParams:(MTRSubscribeParams *)params
                                   subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                             reportHandler:
                                                 (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeInstalledOpenLimitTiltWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                        endpoint:(NSNumber *)endpoint
                                                           queue:(dispatch_queue_t)queue
                                                      completion:(void (^)(NSNumber * _Nullable value,
                                                                     NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeInstalledClosedLimitTiltWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeInstalledClosedLimitTiltWithParams:(MTRSubscribeParams *)params
                                     subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                               reportHandler:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeInstalledClosedLimitTiltWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                          endpoint:(NSNumber *)endpoint
                                                             queue:(dispatch_queue_t)queue
                                                        completion:(void (^)(NSNumber * _Nullable value,
                                                                       NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeModeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeModeWithValue:(NSNumber * _Nonnull)value
                         completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeModeWithValue:(NSNumber * _Nonnull)value
                             params:(MTRWriteParams * _Nullable)params
                         completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeModeWithParams:(MTRSubscribeParams *)params
                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                           reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeModeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                      endpoint:(NSNumber *)endpoint
                                         queue:(dispatch_queue_t)queue
                                    completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeSafetyStatusWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeSafetyStatusWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeSafetyStatusWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Barrier Control
 *
 * This cluster provides control of a barrier (garage door).
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterBarrierControl : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

/**
 * Command BarrierControlGoToPercent
 *
 * Command to instruct a barrier to go to a percent open state.
 */
- (void)barrierControlGoToPercentWithParams:(MTRBarrierControlClusterBarrierControlGoToPercentParams *)params
                                 completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command BarrierControlStop
 *
 * Command that instructs the barrier to stop moving.
 */
- (void)barrierControlStopWithParams:(MTRBarrierControlClusterBarrierControlStopParams * _Nullable)params
                          completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)barrierControlStopWithCompletion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBarrierMovingStateWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBarrierMovingStateWithParams:(MTRSubscribeParams *)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                         reportHandler:
                                             (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBarrierMovingStateWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                    endpoint:(NSNumber *)endpoint
                                                       queue:(dispatch_queue_t)queue
                                                  completion:
                                                      (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBarrierSafetyStatusWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBarrierSafetyStatusWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBarrierSafetyStatusWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBarrierCapabilitiesWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBarrierCapabilitiesWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBarrierCapabilitiesWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBarrierOpenEventsWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeBarrierOpenEventsWithValue:(NSNumber * _Nonnull)value
                                      completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeBarrierOpenEventsWithValue:(NSNumber * _Nonnull)value
                                          params:(MTRWriteParams * _Nullable)params
                                      completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBarrierOpenEventsWithParams:(MTRSubscribeParams *)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBarrierOpenEventsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                                 completion:
                                                     (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBarrierCloseEventsWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeBarrierCloseEventsWithValue:(NSNumber * _Nonnull)value
                                       completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeBarrierCloseEventsWithValue:(NSNumber * _Nonnull)value
                                           params:(MTRWriteParams * _Nullable)params
                                       completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBarrierCloseEventsWithParams:(MTRSubscribeParams *)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                         reportHandler:
                                             (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBarrierCloseEventsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                    endpoint:(NSNumber *)endpoint
                                                       queue:(dispatch_queue_t)queue
                                                  completion:
                                                      (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBarrierCommandOpenEventsWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeBarrierCommandOpenEventsWithValue:(NSNumber * _Nonnull)value
                                             completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeBarrierCommandOpenEventsWithValue:(NSNumber * _Nonnull)value
                                                 params:(MTRWriteParams * _Nullable)params
                                             completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBarrierCommandOpenEventsWithParams:(MTRSubscribeParams *)params
                                     subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                               reportHandler:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBarrierCommandOpenEventsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                          endpoint:(NSNumber *)endpoint
                                                             queue:(dispatch_queue_t)queue
                                                        completion:(void (^)(NSNumber * _Nullable value,
                                                                       NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBarrierCommandCloseEventsWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeBarrierCommandCloseEventsWithValue:(NSNumber * _Nonnull)value
                                              completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeBarrierCommandCloseEventsWithValue:(NSNumber * _Nonnull)value
                                                  params:(MTRWriteParams * _Nullable)params
                                              completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBarrierCommandCloseEventsWithParams:(MTRSubscribeParams *)params
                                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                reportHandler:
                                                    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBarrierCommandCloseEventsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                           endpoint:(NSNumber *)endpoint
                                                              queue:(dispatch_queue_t)queue
                                                         completion:(void (^)(NSNumber * _Nullable value,
                                                                        NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBarrierOpenPeriodWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeBarrierOpenPeriodWithValue:(NSNumber * _Nonnull)value
                                      completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeBarrierOpenPeriodWithValue:(NSNumber * _Nonnull)value
                                          params:(MTRWriteParams * _Nullable)params
                                      completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBarrierOpenPeriodWithParams:(MTRSubscribeParams *)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBarrierOpenPeriodWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                                 completion:
                                                     (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBarrierClosePeriodWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeBarrierClosePeriodWithValue:(NSNumber * _Nonnull)value
                                       completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeBarrierClosePeriodWithValue:(NSNumber * _Nonnull)value
                                           params:(MTRWriteParams * _Nullable)params
                                       completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBarrierClosePeriodWithParams:(MTRSubscribeParams *)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                         reportHandler:
                                             (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBarrierClosePeriodWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                    endpoint:(NSNumber *)endpoint
                                                       queue:(dispatch_queue_t)queue
                                                  completion:
                                                      (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeBarrierPositionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeBarrierPositionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeBarrierPositionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Pump Configuration and Control
 *
 * An interface for configuring and controlling pumps.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterPumpConfigurationAndControl : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeMaxPressureWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeMaxPressureWithParams:(MTRSubscribeParams *)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeMaxPressureWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeMaxSpeedWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeMaxSpeedWithParams:(MTRSubscribeParams *)params
                     subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                               reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeMaxSpeedWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                          endpoint:(NSNumber *)endpoint
                                             queue:(dispatch_queue_t)queue
                                        completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeMaxFlowWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeMaxFlowWithParams:(MTRSubscribeParams *)params
                    subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                              reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeMaxFlowWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                       completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeMinConstPressureWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeMinConstPressureWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeMinConstPressureWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:
                                                    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeMaxConstPressureWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeMaxConstPressureWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeMaxConstPressureWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:
                                                    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeMinCompPressureWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeMinCompPressureWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeMinCompPressureWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeMaxCompPressureWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeMaxCompPressureWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeMaxCompPressureWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeMinConstSpeedWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeMinConstSpeedWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeMinConstSpeedWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeMaxConstSpeedWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeMaxConstSpeedWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeMaxConstSpeedWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeMinConstFlowWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeMinConstFlowWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeMinConstFlowWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeMaxConstFlowWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeMaxConstFlowWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeMaxConstFlowWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeMinConstTempWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeMinConstTempWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeMinConstTempWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeMaxConstTempWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeMaxConstTempWithParams:(MTRSubscribeParams *)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeMaxConstTempWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributePumpStatusWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributePumpStatusWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributePumpStatusWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEffectiveOperationModeWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeEffectiveOperationModeWithParams:(MTRSubscribeParams *)params
                                   subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                             reportHandler:
                                                 (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeEffectiveOperationModeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                        endpoint:(NSNumber *)endpoint
                                                           queue:(dispatch_queue_t)queue
                                                      completion:(void (^)(NSNumber * _Nullable value,
                                                                     NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEffectiveControlModeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeEffectiveControlModeWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeEffectiveControlModeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeCapacityWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeCapacityWithParams:(MTRSubscribeParams *)params
                     subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                               reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeCapacityWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                          endpoint:(NSNumber *)endpoint
                                             queue:(dispatch_queue_t)queue
                                        completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeSpeedWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeSpeedWithParams:(MTRSubscribeParams *)params
                  subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                            reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeSpeedWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                       endpoint:(NSNumber *)endpoint
                                          queue:(dispatch_queue_t)queue
                                     completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeLifetimeRunningHoursWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeLifetimeRunningHoursWithValue:(NSNumber * _Nullable)value
                                         completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeLifetimeRunningHoursWithValue:(NSNumber * _Nullable)value
                                             params:(MTRWriteParams * _Nullable)params
                                         completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeLifetimeRunningHoursWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeLifetimeRunningHoursWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributePowerWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributePowerWithParams:(MTRSubscribeParams *)params
                  subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                            reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributePowerWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                       endpoint:(NSNumber *)endpoint
                                          queue:(dispatch_queue_t)queue
                                     completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeLifetimeEnergyConsumedWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeLifetimeEnergyConsumedWithValue:(NSNumber * _Nullable)value
                                           completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeLifetimeEnergyConsumedWithValue:(NSNumber * _Nullable)value
                                               params:(MTRWriteParams * _Nullable)params
                                           completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeLifetimeEnergyConsumedWithParams:(MTRSubscribeParams *)params
                                   subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                             reportHandler:
                                                 (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeLifetimeEnergyConsumedWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                        endpoint:(NSNumber *)endpoint
                                                           queue:(dispatch_queue_t)queue
                                                      completion:(void (^)(NSNumber * _Nullable value,
                                                                     NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeOperationModeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeOperationModeWithValue:(NSNumber * _Nonnull)value
                                  completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeOperationModeWithValue:(NSNumber * _Nonnull)value
                                      params:(MTRWriteParams * _Nullable)params
                                  completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeOperationModeWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeOperationModeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeControlModeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeControlModeWithValue:(NSNumber * _Nonnull)value
                                completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeControlModeWithValue:(NSNumber * _Nonnull)value
                                    params:(MTRWriteParams * _Nullable)params
                                completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeControlModeWithParams:(MTRSubscribeParams *)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeControlModeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams *)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams *)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;
- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    MTR_NEWLY_AVAILABLE;
+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    MTR_NEWLY_AVAILABLE;

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams *)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams *)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (instancetype)init NS_UNAVAILABLE;
+ (instancetype)new NS_UNAVAILABLE;

@end

/**
 * Cluster Thermostat
 *
 * An interface for configuring and controlling the functionality of a thermostat.
 */
API_AVAILABLE(ios(16.1), macos(13.0), watchos(9.1), tvos(16.1))
@interface MTRBaseClusterThermostat : MTRCluster

- (instancetype _Nullable)initWithDevice:(MTRBaseDevice *)device
                              endpointID:(NSNumber *)endpointID
                                   queue:(dispatch_queue_t)queue NS_DESIGNATED_INITIALIZER
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

/**
 * Command SetpointRaiseLower
 *
 * Command description for SetpointRaiseLower
 */
- (void)setpointRaiseLowerWithParams:(MTRThermostatClusterSetpointRaiseLowerParams *)params
                          completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command SetWeeklySchedule
 *
 * Command description for SetWeeklySchedule
 */
- (void)setWeeklyScheduleWithParams:(MTRThermostatClusterSetWeeklyScheduleParams *)params
                         completion:(MTRStatusCompletion)completion API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command GetWeeklySchedule
 *
 * Command description for GetWeeklySchedule
 */
- (void)getWeeklyScheduleWithParams:(MTRThermostatClusterGetWeeklyScheduleParams *)params
                         completion:(void (^)(MTRThermostatClusterGetWeeklyScheduleResponseParams * _Nullable data,
                                        NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
/**
 * Command ClearWeeklySchedule
 *
 * The Clear Weekly Schedule command is used to clear the weekly schedule.
 */
- (void)clearWeeklyScheduleWithParams:(MTRThermostatClusterClearWeeklyScheduleParams * _Nullable)params
                           completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)clearWeeklyScheduleWithCompletion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeLocalTemperatureWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeLocalTemperatureWithParams:(MTRSubscribeParams *)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeLocalTemperatureWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:
                                                    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeOutdoorTemperatureWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeOutdoorTemperatureWithParams:(MTRSubscribeParams *)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                         reportHandler:
                                             (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeOutdoorTemperatureWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                    endpoint:(NSNumber *)endpoint
                                                       queue:(dispatch_queue_t)queue
                                                  completion:
                                                      (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeOccupancyWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeOccupancyWithParams:(MTRSubscribeParams *)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeOccupancyWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAbsMinHeatSetpointLimitWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAbsMinHeatSetpointLimitWithParams:(MTRSubscribeParams *)params
                                    subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                              reportHandler:
                                                  (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAbsMinHeatSetpointLimitWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                         endpoint:(NSNumber *)endpoint
                                                            queue:(dispatch_queue_t)queue
                                                       completion:(void (^)(NSNumber * _Nullable value,
                                                                      NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAbsMaxHeatSetpointLimitWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAbsMaxHeatSetpointLimitWithParams:(MTRSubscribeParams *)params
                                    subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                              reportHandler:
                                                  (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAbsMaxHeatSetpointLimitWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                         endpoint:(NSNumber *)endpoint
                                                            queue:(dispatch_queue_t)queue
                                                       completion:(void (^)(NSNumber * _Nullable value,
                                                                      NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAbsMinCoolSetpointLimitWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAbsMinCoolSetpointLimitWithParams:(MTRSubscribeParams *)params
                                    subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                              reportHandler:
                                                  (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAbsMinCoolSetpointLimitWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                         endpoint:(NSNumber *)endpoint
                                                            queue:(dispatch_queue_t)queue
                                                       completion:(void (^)(NSNumber * _Nullable value,
                                                                      NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeAbsMaxCoolSetpointLimitWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeAbsMaxCoolSetpointLimitWithParams:(MTRSubscribeParams *)params
                                    subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                              reportHandler:
                                                  (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeAbsMaxCoolSetpointLimitWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                         endpoint:(NSNumber *)endpoint
                                                            queue:(dispatch_queue_t)queue
                                                       completion:(void (^)(NSNumber * _Nullable value,
                                                                      NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributePICoolingDemandWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributePICoolingDemandWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributePICoolingDemandWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributePIHeatingDemandWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributePIHeatingDemandWithParams:(MTRSubscribeParams *)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributePIHeatingDemandWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeHVACSystemTypeConfigurationWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeHVACSystemTypeConfigurationWithValue:(NSNumber * _Nonnull)value
                                                completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeHVACSystemTypeConfigurationWithValue:(NSNumber * _Nonnull)value
                                                    params:(MTRWriteParams * _Nullable)params
                                                completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeHVACSystemTypeConfigurationWithParams:(MTRSubscribeParams *)params
                                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                  reportHandler:
                                                      (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeHVACSystemTypeConfigurationWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                             endpoint:(NSNumber *)endpoint
                                                                queue:(dispatch_queue_t)queue
                                                           completion:(void (^)(NSNumber * _Nullable value,
                                                                          NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeLocalTemperatureCalibrationWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeLocalTemperatureCalibrationWithValue:(NSNumber * _Nonnull)value
                                                completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeLocalTemperatureCalibrationWithValue:(NSNumber * _Nonnull)value
                                                    params:(MTRWriteParams * _Nullable)params
                                                completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeLocalTemperatureCalibrationWithParams:(MTRSubscribeParams *)params
                                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                  reportHandler:
                                                      (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeLocalTemperatureCalibrationWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                             endpoint:(NSNumber *)endpoint
                                                                queue:(dispatch_queue_t)queue
                                                           completion:(void (^)(NSNumber * _Nullable value,
                                                                          NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeOccupiedCoolingSetpointWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeOccupiedCoolingSetpointWithValue:(NSNumber * _Nonnull)value
                                            completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeOccupiedCoolingSetpointWithValue:(NSNumber * _Nonnull)value
                                                params:(MTRWriteParams * _Nullable)params
                                            completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeOccupiedCoolingSetpointWithParams:(MTRSubscribeParams *)params
                                    subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                              reportHandler:
                                                  (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeOccupiedCoolingSetpointWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                         endpoint:(NSNumber *)endpoint
                                                            queue:(dispatch_queue_t)queue
                                                       completion:(void (^)(NSNumber * _Nullable value,
                                                                      NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeOccupiedHeatingSetpointWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeOccupiedHeatingSetpointWithValue:(NSNumber * _Nonnull)value
                                            completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeOccupiedHeatingSetpointWithValue:(NSNumber * _Nonnull)value
                                                params:(MTRWriteParams * _Nullable)params
                                            completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeOccupiedHeatingSetpointWithParams:(MTRSubscribeParams *)params
                                    subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                              reportHandler:
                                                  (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeOccupiedHeatingSetpointWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                         endpoint:(NSNumber *)endpoint
                                                            queue:(dispatch_queue_t)queue
                                                       completion:(void (^)(NSNumber * _Nullable value,
                                                                      NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeUnoccupiedCoolingSetpointWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeUnoccupiedCoolingSetpointWithValue:(NSNumber * _Nonnull)value
                                              completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeUnoccupiedCoolingSetpointWithValue:(NSNumber * _Nonnull)value
                                                  params:(MTRWriteParams * _Nullable)params
                                              completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeUnoccupiedCoolingSetpointWithParams:(MTRSubscribeParams *)params
                                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                reportHandler:
                                                    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeUnoccupiedCoolingSetpointWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                           endpoint:(NSNumber *)endpoint
                                                              queue:(dispatch_queue_t)queue
                                                         completion:(void (^)(NSNumber * _Nullable value,
                                                                        NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeUnoccupiedHeatingSetpointWithCompletion:
    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeUnoccupiedHeatingSetpointWithValue:(NSNumber * _Nonnull)value
                                              completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeUnoccupiedHeatingSetpointWithValue:(NSNumber * _Nonnull)value
                                                  params:(MTRWriteParams * _Nullable)params
                                              completion:(MTRStatusCompletion)completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)subscribeAttributeUnoccupiedHeatingSetpointWithParams:(MTRSubscribeParams *)params
                                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                reportHandler:
                                                    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
+ (void)readAttributeUnoccupiedHeatingSetpointWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                           endpoint:(NSNumber *)endpoint
                                                              queue:(dispatch_queue_t)queue
                                                         completion:(void (^)(NSNumber * _Nullable value,
                                                                        NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));

- (void)readAttributeMinHeatSetpointLimitWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
    API_AVAILABLE(ios(16.4), macos(13.3), watchos(9.4), tvos(16.4));
- (void)writeAttributeMinHeatSetpointLimitWithValue:(NSNumber * _Nonnull)value
                                         completion: