/*
 *
 *    Copyright (c) 2022 Project CHIP Authors
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

#import <Foundation/Foundation.h>

#import "MTRBaseClusterUtils.h"
#import "MTRBaseClusters_Internal.h"
#import "MTRBaseDevice_Internal.h"
#import "MTRCallbackBridge.h"
#import "MTRClusterStateCacheContainer_Internal.h"
#import "MTRCluster_Internal.h"
#import "MTRCommandPayloadsObjc.h"
#import "MTRDevice_Internal.h"
#import "MTRStructsObjc.h"

#include <controller/CHIPCluster.h>
#include <lib/support/CHIPListUtils.h>
#include <platform/CHIPDeviceLayer.h>

#include <type_traits>

using chip::Callback::Callback;
using chip::Callback::Cancelable;
using namespace chip::app::Clusters;
using chip::Optional;
using chip::SessionHandle;
using chip::Messaging::ExchangeManager;
using chip::System::Clock::Seconds16;
using chip::System::Clock::Timeout;

// NOLINTBEGIN(clang-analyzer-cplusplus.NewDeleteLeaks): Linter is unable to locate the delete on these objects.
@implementation MTRBaseClusterIdentify

- (instancetype)initWithDevice:(MTRBaseDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _device = device;
        _endpoint = [endpointID unsignedShortValue];
    }
    return self;
}

- (void)identifyWithParams:(MTRIdentifyClusterIdentifyParams *)params completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            Identify::Commands::Identify::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.identifyTime = params.identifyTime.unsignedShortValue;

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)triggerEffectWithParams:(MTRIdentifyClusterTriggerEffectParams *)params completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            Identify::Commands::TriggerEffect::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.effectIdentifier = static_cast<std::remove_reference_t<decltype(request.effectIdentifier)>>(
                params.effectIdentifier.unsignedCharValue);
            request.effectVariant
                = static_cast<std::remove_reference_t<decltype(request.effectVariant)>>(params.effectVariant.unsignedCharValue);

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)readAttributeIdentifyTimeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Identify::Attributes::IdentifyTime::TypeInfo;
    return MTRReadAttribute<MTRInt16uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)writeAttributeIdentifyTimeWithValue:(NSNumber * _Nonnull)value completion:(MTRStatusCompletion)completion
{
    [self writeAttributeIdentifyTimeWithValue:(NSNumber * _Nonnull) value params:nil completion:completion];
}
- (void)writeAttributeIdentifyTimeWithValue:(NSNumber * _Nonnull)value
                                     params:(MTRWriteParams * _Nullable)params
                                 completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    value = [value copy];

    auto * bridge = new MTRDefaultSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable ignored, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, DefaultSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            chip::Optional<uint16_t> timedWriteTimeout;
            if (params != nil) {
                if (params.timedWriteTimeout != nil) {
                    timedWriteTimeout.SetValue(params.timedWriteTimeout.unsignedShortValue);
                }
            }

            ListFreer listFreer;
            using TypeInfo = Identify::Attributes::IdentifyTime::TypeInfo;
            TypeInfo::Type cppValue;
            cppValue = value.unsignedShortValue;

            chip::Controller::ClusterBase cppCluster(exchangeManager, session, self->_endpoint);
            return cppCluster.WriteAttribute<TypeInfo>(cppValue, bridge, successCb, failureCb, timedWriteTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)subscribeAttributeIdentifyTimeWithParams:(MTRSubscribeParams * _Nonnull)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Identify::Attributes::IdentifyTime::TypeInfo;
    MTRSubscribeAttribute<MTRInt16uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeIdentifyTimeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt16uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int16uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Identify::Attributes::IdentifyTime::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeIdentifyTypeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Identify::Attributes::IdentifyType::TypeInfo;
    return MTRReadAttribute<MTRIdentifyClusterIdentifyTypeEnumAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeIdentifyTypeWithParams:(MTRSubscribeParams * _Nonnull)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Identify::Attributes::IdentifyType::TypeInfo;
    MTRSubscribeAttribute<MTRIdentifyClusterIdentifyTypeEnumAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(
        params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeIdentifyTypeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRIdentifyClusterIdentifyTypeEnumAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(IdentifyClusterIdentifyTypeEnumAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Identify::Attributes::IdentifyType::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Identify::Attributes::GeneratedCommandList::TypeInfo;
    return MTRReadAttribute<MTRIdentifyGeneratedCommandListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams * _Nonnull)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Identify::Attributes::GeneratedCommandList::TypeInfo;
    MTRSubscribeAttribute<MTRIdentifyGeneratedCommandListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(
        params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRIdentifyGeneratedCommandListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(IdentifyGeneratedCommandListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Identify::Attributes::GeneratedCommandList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Identify::Attributes::AcceptedCommandList::TypeInfo;
    return MTRReadAttribute<MTRIdentifyAcceptedCommandListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams * _Nonnull)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Identify::Attributes::AcceptedCommandList::TypeInfo;
    MTRSubscribeAttribute<MTRIdentifyAcceptedCommandListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(
        params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRIdentifyAcceptedCommandListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(IdentifyAcceptedCommandListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Identify::Attributes::AcceptedCommandList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Identify::Attributes::EventList::TypeInfo;
    return MTRReadAttribute<MTRIdentifyEventListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams * _Nonnull)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Identify::Attributes::EventList::TypeInfo;
    MTRSubscribeAttribute<MTRIdentifyEventListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRIdentifyEventListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(IdentifyEventListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Identify::Attributes::EventList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Identify::Attributes::AttributeList::TypeInfo;
    return MTRReadAttribute<MTRIdentifyAttributeListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams * _Nonnull)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Identify::Attributes::AttributeList::TypeInfo;
    MTRSubscribeAttribute<MTRIdentifyAttributeListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRIdentifyAttributeListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(IdentifyAttributeListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Identify::Attributes::AttributeList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Identify::Attributes::FeatureMap::TypeInfo;
    return MTRReadAttribute<MTRInt32uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams * _Nonnull)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Identify::Attributes::FeatureMap::TypeInfo;
    MTRSubscribeAttribute<MTRInt32uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt32uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int32uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Identify::Attributes::FeatureMap::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Identify::Attributes::ClusterRevision::TypeInfo;
    return MTRReadAttribute<MTRInt16uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams * _Nonnull)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Identify::Attributes::ClusterRevision::TypeInfo;
    MTRSubscribeAttribute<MTRInt16uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt16uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int16uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Identify::Attributes::ClusterRevision::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

@end

@implementation MTRBaseClusterIdentify (Deprecated)

- (void)identifyWithParams:(MTRIdentifyClusterIdentifyParams *)params completionHandler:(MTRStatusCompletion)completionHandler
{
    [self identifyWithParams:params completion:completionHandler];
}
- (void)triggerEffectWithParams:(MTRIdentifyClusterTriggerEffectParams *)params
              completionHandler:(MTRStatusCompletion)completionHandler
{
    [self triggerEffectWithParams:params completion:completionHandler];
}

- (void)readAttributeIdentifyTimeWithCompletionHandler:(void (^)(
                                                           NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeIdentifyTimeWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)writeAttributeIdentifyTimeWithValue:(NSNumber * _Nonnull)value completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeIdentifyTimeWithValue:value params:nil completion:completionHandler];
}
- (void)writeAttributeIdentifyTimeWithValue:(NSNumber * _Nonnull)value
                                     params:(MTRWriteParams * _Nullable)params
                          completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeIdentifyTimeWithValue:value params:params completion:completionHandler];
}
- (void)subscribeAttributeIdentifyTimeWithMinInterval:(NSNumber * _Nonnull)minInterval
                                          maxInterval:(NSNumber * _Nonnull)maxInterval
                                               params:(MTRSubscribeParams * _Nullable)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeIdentifyTimeWithParams:subscribeParams
                           subscriptionEstablished:subscriptionEstablishedHandler
                                     reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                         // Cast is safe because subclass does not add any selectors.
                                         reportHandler(static_cast<NSNumber *>(value), error);
                                     }];
}
+ (void)readAttributeIdentifyTimeWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                  completionHandler:
                                      (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeIdentifyTimeWithClusterStateCache:attributeCacheContainer.realContainer
                                                endpoint:endpoint
                                                   queue:queue
                                              completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                  // Cast is safe because subclass does not add any selectors.
                                                  completionHandler(static_cast<NSNumber *>(value), error);
                                              }];
}

- (void)readAttributeIdentifyTypeWithCompletionHandler:(void (^)(
                                                           NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeIdentifyTypeWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeIdentifyTypeWithMinInterval:(NSNumber * _Nonnull)minInterval
                                          maxInterval:(NSNumber * _Nonnull)maxInterval
                                               params:(MTRSubscribeParams * _Nullable)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeIdentifyTypeWithParams:subscribeParams
                           subscriptionEstablished:subscriptionEstablishedHandler
                                     reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                         // Cast is safe because subclass does not add any selectors.
                                         reportHandler(static_cast<NSNumber *>(value), error);
                                     }];
}
+ (void)readAttributeIdentifyTypeWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                  completionHandler:
                                      (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeIdentifyTypeWithClusterStateCache:attributeCacheContainer.realContainer
                                                endpoint:endpoint
                                                   queue:queue
                                              completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                  // Cast is safe because subclass does not add any selectors.
                                                  completionHandler(static_cast<NSNumber *>(value), error);
                                              }];
}

- (void)readAttributeGeneratedCommandListWithCompletionHandler:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))completionHandler
{
    [self readAttributeGeneratedCommandListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeGeneratedCommandListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                  maxInterval:(NSNumber * _Nonnull)maxInterval
                                                       params:(MTRSubscribeParams * _Nullable)params
                                      subscriptionEstablished:
                                          (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                                reportHandler:
                                                    (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeGeneratedCommandListWithParams:subscribeParams
                                   subscriptionEstablished:subscriptionEstablishedHandler
                                             reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                 // Cast is safe because subclass does not add any selectors.
                                                 reportHandler(static_cast<NSArray *>(value), error);
                                             }];
}
+ (void)readAttributeGeneratedCommandListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                          completionHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeGeneratedCommandListWithClusterStateCache:attributeCacheContainer.realContainer
                                                        endpoint:endpoint
                                                           queue:queue
                                                      completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                          // Cast is safe because subclass does not add any selectors.
                                                          completionHandler(static_cast<NSArray *>(value), error);
                                                      }];
}

- (void)readAttributeAcceptedCommandListWithCompletionHandler:(void (^)(NSArray * _Nullable value,
                                                                  NSError * _Nullable error))completionHandler
{
    [self readAttributeAcceptedCommandListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeAcceptedCommandListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                 maxInterval:(NSNumber * _Nonnull)maxInterval
                                                      params:(MTRSubscribeParams * _Nullable)params
                                     subscriptionEstablished:
                                         (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                               reportHandler:
                                                   (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeAcceptedCommandListWithParams:subscribeParams
                                  subscriptionEstablished:subscriptionEstablishedHandler
                                            reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                reportHandler(static_cast<NSArray *>(value), error);
                                            }];
}
+ (void)readAttributeAcceptedCommandListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                         completionHandler:
                                             (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAcceptedCommandListWithClusterStateCache:attributeCacheContainer.realContainer
                                                       endpoint:endpoint
                                                          queue:queue
                                                     completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                         // Cast is safe because subclass does not add any selectors.
                                                         completionHandler(static_cast<NSArray *>(value), error);
                                                     }];
}

- (void)readAttributeAttributeListWithCompletionHandler:(void (^)(
                                                            NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAttributeListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeAttributeListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                           maxInterval:(NSNumber * _Nonnull)maxInterval
                                                params:(MTRSubscribeParams * _Nullable)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                         reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeAttributeListWithParams:subscribeParams
                            subscriptionEstablished:subscriptionEstablishedHandler
                                      reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                          // Cast is safe because subclass does not add any selectors.
                                          reportHandler(static_cast<NSArray *>(value), error);
                                      }];
}
+ (void)readAttributeAttributeListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                   completionHandler:
                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAttributeListWithClusterStateCache:attributeCacheContainer.realContainer
                                                 endpoint:endpoint
                                                    queue:queue
                                               completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                   // Cast is safe because subclass does not add any selectors.
                                                   completionHandler(static_cast<NSArray *>(value), error);
                                               }];
}

- (void)readAttributeFeatureMapWithCompletionHandler:(void (^)(
                                                         NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeFeatureMapWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeFeatureMapWithMinInterval:(NSNumber * _Nonnull)minInterval
                                        maxInterval:(NSNumber * _Nonnull)maxInterval
                                             params:(MTRSubscribeParams * _Nullable)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeFeatureMapWithParams:subscribeParams
                         subscriptionEstablished:subscriptionEstablishedHandler
                                   reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                       // Cast is safe because subclass does not add any selectors.
                                       reportHandler(static_cast<NSNumber *>(value), error);
                                   }];
}
+ (void)readAttributeFeatureMapWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                completionHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeFeatureMapWithClusterStateCache:attributeCacheContainer.realContainer
                                              endpoint:endpoint
                                                 queue:queue
                                            completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                completionHandler(static_cast<NSNumber *>(value), error);
                                            }];
}

- (void)readAttributeClusterRevisionWithCompletionHandler:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completionHandler
{
    [self readAttributeClusterRevisionWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeClusterRevisionWithMinInterval:(NSNumber * _Nonnull)minInterval
                                             maxInterval:(NSNumber * _Nonnull)maxInterval
                                                  params:(MTRSubscribeParams * _Nullable)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                           reportHandler:
                                               (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeClusterRevisionWithParams:subscribeParams
                              subscriptionEstablished:subscriptionEstablishedHandler
                                        reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                            // Cast is safe because subclass does not add any selectors.
                                            reportHandler(static_cast<NSNumber *>(value), error);
                                        }];
}
+ (void)readAttributeClusterRevisionWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                     completionHandler:
                                         (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeClusterRevisionWithClusterStateCache:attributeCacheContainer.realContainer
                                                   endpoint:endpoint
                                                      queue:queue
                                                 completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                     // Cast is safe because subclass does not add any selectors.
                                                     completionHandler(static_cast<NSNumber *>(value), error);
                                                 }];
}

- (nullable instancetype)initWithDevice:(MTRBaseDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

@end

@implementation MTRBaseClusterGroups

- (instancetype)initWithDevice:(MTRBaseDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _device = device;
        _endpoint = [endpointID unsignedShortValue];
    }
    return self;
}

- (void)addGroupWithParams:(MTRGroupsClusterAddGroupParams *)params
                completion:(void (^)(MTRGroupsClusterAddGroupResponseParams * _Nullable data, NSError * _Nullable error))completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRGroupsClusterAddGroupResponseCallbackBridge(self.callbackQueue, completion,
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, GroupsClusterAddGroupResponseCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRGroupsClusterAddGroupResponseCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            Groups::Commands::AddGroup::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.groupID = params.groupID.unsignedShortValue;
            request.groupName = [self asCharSpan:params.groupName];

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)viewGroupWithParams:(MTRGroupsClusterViewGroupParams *)params
                 completion:
                     (void (^)(MTRGroupsClusterViewGroupResponseParams * _Nullable data, NSError * _Nullable error))completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRGroupsClusterViewGroupResponseCallbackBridge(self.callbackQueue, completion,
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, GroupsClusterViewGroupResponseCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRGroupsClusterViewGroupResponseCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            Groups::Commands::ViewGroup::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.groupID = params.groupID.unsignedShortValue;

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)getGroupMembershipWithParams:(MTRGroupsClusterGetGroupMembershipParams *)params
                          completion:(void (^)(MTRGroupsClusterGetGroupMembershipResponseParams * _Nullable data,
                                         NSError * _Nullable error))completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRGroupsClusterGetGroupMembershipResponseCallbackBridge(self.callbackQueue, completion,
        ^(ExchangeManager & exchangeManager, const SessionHandle & session,
            GroupsClusterGetGroupMembershipResponseCallbackType successCb, MTRErrorCallback failureCb,
            MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRGroupsClusterGetGroupMembershipResponseCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            Groups::Commands::GetGroupMembership::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            {
                using ListType_0 = std::remove_reference_t<decltype(request.groupList)>;
                using ListMemberType_0 = ListMemberTypeGetter<ListType_0>::Type;
                if (params.groupList.count != 0) {
                    auto * listHolder_0 = new ListHolder<ListMemberType_0>(params.groupList.count);
                    if (listHolder_0 == nullptr || listHolder_0->mList == nullptr) {
                        return CHIP_ERROR_INVALID_ARGUMENT;
                    }
                    listFreer.add(listHolder_0);
                    for (size_t i_0 = 0; i_0 < params.groupList.count; ++i_0) {
                        if (![params.groupList[i_0] isKindOfClass:[NSNumber class]]) {
                            // Wrong kind of value.
                            return CHIP_ERROR_INVALID_ARGUMENT;
                        }
                        auto element_0 = (NSNumber *) params.groupList[i_0];
                        listHolder_0->mList[i_0] = element_0.unsignedShortValue;
                    }
                    request.groupList = ListType_0(listHolder_0->mList, params.groupList.count);
                } else {
                    request.groupList = ListType_0();
                }
            }

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)removeGroupWithParams:(MTRGroupsClusterRemoveGroupParams *)params
                   completion:
                       (void (^)(MTRGroupsClusterRemoveGroupResponseParams * _Nullable data, NSError * _Nullable error))completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRGroupsClusterRemoveGroupResponseCallbackBridge(self.callbackQueue, completion,
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, GroupsClusterRemoveGroupResponseCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRGroupsClusterRemoveGroupResponseCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            Groups::Commands::RemoveGroup::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.groupID = params.groupID.unsignedShortValue;

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)removeAllGroupsWithCompletion:(MTRStatusCompletion)completion
{
    [self removeAllGroupsWithParams:nil completion:completion];
}
- (void)removeAllGroupsWithParams:(MTRGroupsClusterRemoveAllGroupsParams * _Nullable)params
                       completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            Groups::Commands::RemoveAllGroups::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)addGroupIfIdentifyingWithParams:(MTRGroupsClusterAddGroupIfIdentifyingParams *)params
                             completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            Groups::Commands::AddGroupIfIdentifying::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.groupID = params.groupID.unsignedShortValue;
            request.groupName = [self asCharSpan:params.groupName];

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)readAttributeNameSupportWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Groups::Attributes::NameSupport::TypeInfo;
    return MTRReadAttribute<MTRGroupsNameSupportAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeNameSupportWithParams:(MTRSubscribeParams * _Nonnull)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Groups::Attributes::NameSupport::TypeInfo;
    MTRSubscribeAttribute<MTRGroupsNameSupportAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeNameSupportWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRGroupsNameSupportAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(GroupsNameSupportAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Groups::Attributes::NameSupport::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Groups::Attributes::GeneratedCommandList::TypeInfo;
    return MTRReadAttribute<MTRGroupsGeneratedCommandListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams * _Nonnull)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Groups::Attributes::GeneratedCommandList::TypeInfo;
    MTRSubscribeAttribute<MTRGroupsGeneratedCommandListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(
        params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRGroupsGeneratedCommandListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(GroupsGeneratedCommandListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Groups::Attributes::GeneratedCommandList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Groups::Attributes::AcceptedCommandList::TypeInfo;
    return MTRReadAttribute<MTRGroupsAcceptedCommandListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams * _Nonnull)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Groups::Attributes::AcceptedCommandList::TypeInfo;
    MTRSubscribeAttribute<MTRGroupsAcceptedCommandListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(
        params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRGroupsAcceptedCommandListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(GroupsAcceptedCommandListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Groups::Attributes::AcceptedCommandList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Groups::Attributes::EventList::TypeInfo;
    return MTRReadAttribute<MTRGroupsEventListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams * _Nonnull)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Groups::Attributes::EventList::TypeInfo;
    MTRSubscribeAttribute<MTRGroupsEventListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRGroupsEventListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(GroupsEventListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Groups::Attributes::EventList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Groups::Attributes::AttributeList::TypeInfo;
    return MTRReadAttribute<MTRGroupsAttributeListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams * _Nonnull)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Groups::Attributes::AttributeList::TypeInfo;
    MTRSubscribeAttribute<MTRGroupsAttributeListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRGroupsAttributeListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(GroupsAttributeListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Groups::Attributes::AttributeList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Groups::Attributes::FeatureMap::TypeInfo;
    return MTRReadAttribute<MTRInt32uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams * _Nonnull)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Groups::Attributes::FeatureMap::TypeInfo;
    MTRSubscribeAttribute<MTRInt32uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt32uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int32uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Groups::Attributes::FeatureMap::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Groups::Attributes::ClusterRevision::TypeInfo;
    return MTRReadAttribute<MTRInt16uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams * _Nonnull)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Groups::Attributes::ClusterRevision::TypeInfo;
    MTRSubscribeAttribute<MTRInt16uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt16uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int16uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Groups::Attributes::ClusterRevision::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

@end

@implementation MTRBaseClusterGroups (Deprecated)

- (void)addGroupWithParams:(MTRGroupsClusterAddGroupParams *)params
         completionHandler:
             (void (^)(MTRGroupsClusterAddGroupResponseParams * _Nullable data, NSError * _Nullable error))completionHandler
{
    [self addGroupWithParams:params
                  completion:^(MTRGroupsClusterAddGroupResponseParams * _Nullable data, NSError * _Nullable error) {
                      // Cast is safe because subclass does not add any selectors.
                      completionHandler(static_cast<MTRGroupsClusterAddGroupResponseParams *>(data), error);
                  }];
}
- (void)viewGroupWithParams:(MTRGroupsClusterViewGroupParams *)params
          completionHandler:
              (void (^)(MTRGroupsClusterViewGroupResponseParams * _Nullable data, NSError * _Nullable error))completionHandler
{
    [self viewGroupWithParams:params
                   completion:^(MTRGroupsClusterViewGroupResponseParams * _Nullable data, NSError * _Nullable error) {
                       // Cast is safe because subclass does not add any selectors.
                       completionHandler(static_cast<MTRGroupsClusterViewGroupResponseParams *>(data), error);
                   }];
}
- (void)getGroupMembershipWithParams:(MTRGroupsClusterGetGroupMembershipParams *)params
                   completionHandler:(void (^)(MTRGroupsClusterGetGroupMembershipResponseParams * _Nullable data,
                                         NSError * _Nullable error))completionHandler
{
    [self getGroupMembershipWithParams:params
                            completion:^(
                                MTRGroupsClusterGetGroupMembershipResponseParams * _Nullable data, NSError * _Nullable error) {
                                // Cast is safe because subclass does not add any selectors.
                                completionHandler(static_cast<MTRGroupsClusterGetGroupMembershipResponseParams *>(data), error);
                            }];
}
- (void)removeGroupWithParams:(MTRGroupsClusterRemoveGroupParams *)params
            completionHandler:
                (void (^)(MTRGroupsClusterRemoveGroupResponseParams * _Nullable data, NSError * _Nullable error))completionHandler
{
    [self removeGroupWithParams:params
                     completion:^(MTRGroupsClusterRemoveGroupResponseParams * _Nullable data, NSError * _Nullable error) {
                         // Cast is safe because subclass does not add any selectors.
                         completionHandler(static_cast<MTRGroupsClusterRemoveGroupResponseParams *>(data), error);
                     }];
}
- (void)removeAllGroupsWithParams:(MTRGroupsClusterRemoveAllGroupsParams * _Nullable)params
                completionHandler:(MTRStatusCompletion)completionHandler
{
    [self removeAllGroupsWithParams:params completion:completionHandler];
}
- (void)removeAllGroupsWithCompletionHandler:(MTRStatusCompletion)completionHandler
{
    [self removeAllGroupsWithParams:nil completionHandler:completionHandler];
}
- (void)addGroupIfIdentifyingWithParams:(MTRGroupsClusterAddGroupIfIdentifyingParams *)params
                      completionHandler:(MTRStatusCompletion)completionHandler
{
    [self addGroupIfIdentifyingWithParams:params completion:completionHandler];
}

- (void)readAttributeNameSupportWithCompletionHandler:(void (^)(
                                                          NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeNameSupportWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeNameSupportWithMinInterval:(NSNumber * _Nonnull)minInterval
                                         maxInterval:(NSNumber * _Nonnull)maxInterval
                                              params:(MTRSubscribeParams * _Nullable)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeNameSupportWithParams:subscribeParams
                          subscriptionEstablished:subscriptionEstablishedHandler
                                    reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                        // Cast is safe because subclass does not add any selectors.
                                        reportHandler(static_cast<NSNumber *>(value), error);
                                    }];
}
+ (void)readAttributeNameSupportWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                          endpoint:(NSNumber *)endpoint
                                             queue:(dispatch_queue_t)queue
                                 completionHandler:
                                     (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeNameSupportWithClusterStateCache:attributeCacheContainer.realContainer
                                               endpoint:endpoint
                                                  queue:queue
                                             completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                 // Cast is safe because subclass does not add any selectors.
                                                 completionHandler(static_cast<NSNumber *>(value), error);
                                             }];
}

- (void)readAttributeGeneratedCommandListWithCompletionHandler:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))completionHandler
{
    [self readAttributeGeneratedCommandListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeGeneratedCommandListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                  maxInterval:(NSNumber * _Nonnull)maxInterval
                                                       params:(MTRSubscribeParams * _Nullable)params
                                      subscriptionEstablished:
                                          (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                                reportHandler:
                                                    (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeGeneratedCommandListWithParams:subscribeParams
                                   subscriptionEstablished:subscriptionEstablishedHandler
                                             reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                 // Cast is safe because subclass does not add any selectors.
                                                 reportHandler(static_cast<NSArray *>(value), error);
                                             }];
}
+ (void)readAttributeGeneratedCommandListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                          completionHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeGeneratedCommandListWithClusterStateCache:attributeCacheContainer.realContainer
                                                        endpoint:endpoint
                                                           queue:queue
                                                      completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                          // Cast is safe because subclass does not add any selectors.
                                                          completionHandler(static_cast<NSArray *>(value), error);
                                                      }];
}

- (void)readAttributeAcceptedCommandListWithCompletionHandler:(void (^)(NSArray * _Nullable value,
                                                                  NSError * _Nullable error))completionHandler
{
    [self readAttributeAcceptedCommandListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeAcceptedCommandListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                 maxInterval:(NSNumber * _Nonnull)maxInterval
                                                      params:(MTRSubscribeParams * _Nullable)params
                                     subscriptionEstablished:
                                         (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                               reportHandler:
                                                   (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeAcceptedCommandListWithParams:subscribeParams
                                  subscriptionEstablished:subscriptionEstablishedHandler
                                            reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                reportHandler(static_cast<NSArray *>(value), error);
                                            }];
}
+ (void)readAttributeAcceptedCommandListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                         completionHandler:
                                             (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAcceptedCommandListWithClusterStateCache:attributeCacheContainer.realContainer
                                                       endpoint:endpoint
                                                          queue:queue
                                                     completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                         // Cast is safe because subclass does not add any selectors.
                                                         completionHandler(static_cast<NSArray *>(value), error);
                                                     }];
}

- (void)readAttributeAttributeListWithCompletionHandler:(void (^)(
                                                            NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAttributeListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeAttributeListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                           maxInterval:(NSNumber * _Nonnull)maxInterval
                                                params:(MTRSubscribeParams * _Nullable)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                         reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeAttributeListWithParams:subscribeParams
                            subscriptionEstablished:subscriptionEstablishedHandler
                                      reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                          // Cast is safe because subclass does not add any selectors.
                                          reportHandler(static_cast<NSArray *>(value), error);
                                      }];
}
+ (void)readAttributeAttributeListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                   completionHandler:
                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAttributeListWithClusterStateCache:attributeCacheContainer.realContainer
                                                 endpoint:endpoint
                                                    queue:queue
                                               completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                   // Cast is safe because subclass does not add any selectors.
                                                   completionHandler(static_cast<NSArray *>(value), error);
                                               }];
}

- (void)readAttributeFeatureMapWithCompletionHandler:(void (^)(
                                                         NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeFeatureMapWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeFeatureMapWithMinInterval:(NSNumber * _Nonnull)minInterval
                                        maxInterval:(NSNumber * _Nonnull)maxInterval
                                             params:(MTRSubscribeParams * _Nullable)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeFeatureMapWithParams:subscribeParams
                         subscriptionEstablished:subscriptionEstablishedHandler
                                   reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                       // Cast is safe because subclass does not add any selectors.
                                       reportHandler(static_cast<NSNumber *>(value), error);
                                   }];
}
+ (void)readAttributeFeatureMapWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                completionHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeFeatureMapWithClusterStateCache:attributeCacheContainer.realContainer
                                              endpoint:endpoint
                                                 queue:queue
                                            completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                completionHandler(static_cast<NSNumber *>(value), error);
                                            }];
}

- (void)readAttributeClusterRevisionWithCompletionHandler:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completionHandler
{
    [self readAttributeClusterRevisionWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeClusterRevisionWithMinInterval:(NSNumber * _Nonnull)minInterval
                                             maxInterval:(NSNumber * _Nonnull)maxInterval
                                                  params:(MTRSubscribeParams * _Nullable)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                           reportHandler:
                                               (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeClusterRevisionWithParams:subscribeParams
                              subscriptionEstablished:subscriptionEstablishedHandler
                                        reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                            // Cast is safe because subclass does not add any selectors.
                                            reportHandler(static_cast<NSNumber *>(value), error);
                                        }];
}
+ (void)readAttributeClusterRevisionWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                     completionHandler:
                                         (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeClusterRevisionWithClusterStateCache:attributeCacheContainer.realContainer
                                                   endpoint:endpoint
                                                      queue:queue
                                                 completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                     // Cast is safe because subclass does not add any selectors.
                                                     completionHandler(static_cast<NSNumber *>(value), error);
                                                 }];
}

- (nullable instancetype)initWithDevice:(MTRBaseDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

@end

@implementation MTRBaseClusterScenes

- (instancetype)initWithDevice:(MTRBaseDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _device = device;
        _endpoint = [endpointID unsignedShortValue];
    }
    return self;
}

- (void)addSceneWithParams:(MTRScenesClusterAddSceneParams *)params
                completion:(void (^)(MTRScenesClusterAddSceneResponseParams * _Nullable data, NSError * _Nullable error))completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRScenesClusterAddSceneResponseCallbackBridge(self.callbackQueue, completion,
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, ScenesClusterAddSceneResponseCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRScenesClusterAddSceneResponseCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            Scenes::Commands::AddScene::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.groupID = params.groupID.unsignedShortValue;
            request.sceneID = params.sceneID.unsignedCharValue;
            request.transitionTime = params.transitionTime.unsignedShortValue;
            request.sceneName = [self asCharSpan:params.sceneName];
            {
                using ListType_0 = std::remove_reference_t<decltype(request.extensionFieldSets)>;
                using ListMemberType_0 = ListMemberTypeGetter<ListType_0>::Type;
                if (params.extensionFieldSets.count != 0) {
                    auto * listHolder_0 = new ListHolder<ListMemberType_0>(params.extensionFieldSets.count);
                    if (listHolder_0 == nullptr || listHolder_0->mList == nullptr) {
                        return CHIP_ERROR_INVALID_ARGUMENT;
                    }
                    listFreer.add(listHolder_0);
                    for (size_t i_0 = 0; i_0 < params.extensionFieldSets.count; ++i_0) {
                        if (![params.extensionFieldSets[i_0] isKindOfClass:[MTRScenesClusterExtensionFieldSet class]]) {
                            // Wrong kind of value.
                            return CHIP_ERROR_INVALID_ARGUMENT;
                        }
                        auto element_0 = (MTRScenesClusterExtensionFieldSet *) params.extensionFieldSets[i_0];
                        listHolder_0->mList[i_0].clusterID = element_0.clusterID.unsignedIntValue;
                        {
                            using ListType_2 = std::remove_reference_t<decltype(listHolder_0->mList[i_0].attributeValueList)>;
                            using ListMemberType_2 = ListMemberTypeGetter<ListType_2>::Type;
                            if (element_0.attributeValueList.count != 0) {
                                auto * listHolder_2 = new ListHolder<ListMemberType_2>(element_0.attributeValueList.count);
                                if (listHolder_2 == nullptr || listHolder_2->mList == nullptr) {
                                    return CHIP_ERROR_INVALID_ARGUMENT;
                                }
                                listFreer.add(listHolder_2);
                                for (size_t i_2 = 0; i_2 < element_0.attributeValueList.count; ++i_2) {
                                    if (![element_0.attributeValueList[i_2]
                                            isKindOfClass:[MTRScenesClusterAttributeValuePair class]]) {
                                        // Wrong kind of value.
                                        return CHIP_ERROR_INVALID_ARGUMENT;
                                    }
                                    auto element_2 = (MTRScenesClusterAttributeValuePair *) element_0.attributeValueList[i_2];
                                    if (element_2.attributeID != nil) {
                                        auto & definedValue_4 = listHolder_2->mList[i_2].attributeID.Emplace();
                                        definedValue_4 = element_2.attributeID.unsignedIntValue;
                                    }
                                    listHolder_2->mList[i_2].attributeValue = element_2.attributeValue.unsignedIntValue;
                                }
                                listHolder_0->mList[i_0].attributeValueList
                                    = ListType_2(listHolder_2->mList, element_0.attributeValueList.count);
                            } else {
                                listHolder_0->mList[i_0].attributeValueList = ListType_2();
                            }
                        }
                    }
                    request.extensionFieldSets = ListType_0(listHolder_0->mList, params.extensionFieldSets.count);
                } else {
                    request.extensionFieldSets = ListType_0();
                }
            }

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)viewSceneWithParams:(MTRScenesClusterViewSceneParams *)params
                 completion:
                     (void (^)(MTRScenesClusterViewSceneResponseParams * _Nullable data, NSError * _Nullable error))completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRScenesClusterViewSceneResponseCallbackBridge(self.callbackQueue, completion,
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, ScenesClusterViewSceneResponseCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRScenesClusterViewSceneResponseCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            Scenes::Commands::ViewScene::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.groupID = params.groupID.unsignedShortValue;
            request.sceneID = params.sceneID.unsignedCharValue;

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)removeSceneWithParams:(MTRScenesClusterRemoveSceneParams *)params
                   completion:
                       (void (^)(MTRScenesClusterRemoveSceneResponseParams * _Nullable data, NSError * _Nullable error))completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRScenesClusterRemoveSceneResponseCallbackBridge(self.callbackQueue, completion,
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, ScenesClusterRemoveSceneResponseCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRScenesClusterRemoveSceneResponseCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            Scenes::Commands::RemoveScene::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.groupID = params.groupID.unsignedShortValue;
            request.sceneID = params.sceneID.unsignedCharValue;

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)removeAllScenesWithParams:(MTRScenesClusterRemoveAllScenesParams *)params
                       completion:(void (^)(MTRScenesClusterRemoveAllScenesResponseParams * _Nullable data,
                                      NSError * _Nullable error))completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRScenesClusterRemoveAllScenesResponseCallbackBridge(self.callbackQueue, completion,
        ^(ExchangeManager & exchangeManager, const SessionHandle & session,
            ScenesClusterRemoveAllScenesResponseCallbackType successCb, MTRErrorCallback failureCb,
            MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRScenesClusterRemoveAllScenesResponseCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            Scenes::Commands::RemoveAllScenes::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.groupID = params.groupID.unsignedShortValue;

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)storeSceneWithParams:(MTRScenesClusterStoreSceneParams *)params
                  completion:
                      (void (^)(MTRScenesClusterStoreSceneResponseParams * _Nullable data, NSError * _Nullable error))completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRScenesClusterStoreSceneResponseCallbackBridge(self.callbackQueue, completion,
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, ScenesClusterStoreSceneResponseCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRScenesClusterStoreSceneResponseCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            Scenes::Commands::StoreScene::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.groupID = params.groupID.unsignedShortValue;
            request.sceneID = params.sceneID.unsignedCharValue;

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)recallSceneWithParams:(MTRScenesClusterRecallSceneParams *)params completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            Scenes::Commands::RecallScene::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.groupID = params.groupID.unsignedShortValue;
            request.sceneID = params.sceneID.unsignedCharValue;
            if (params.transitionTime != nil) {
                auto & definedValue_0 = request.transitionTime.Emplace();
                if (params.transitionTime == nil) {
                    definedValue_0.SetNull();
                } else {
                    auto & nonNullValue_1 = definedValue_0.SetNonNull();
                    nonNullValue_1 = params.transitionTime.unsignedShortValue;
                }
            }

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)getSceneMembershipWithParams:(MTRScenesClusterGetSceneMembershipParams *)params
                          completion:(void (^)(MTRScenesClusterGetSceneMembershipResponseParams * _Nullable data,
                                         NSError * _Nullable error))completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRScenesClusterGetSceneMembershipResponseCallbackBridge(self.callbackQueue, completion,
        ^(ExchangeManager & exchangeManager, const SessionHandle & session,
            ScenesClusterGetSceneMembershipResponseCallbackType successCb, MTRErrorCallback failureCb,
            MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRScenesClusterGetSceneMembershipResponseCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            Scenes::Commands::GetSceneMembership::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.groupID = params.groupID.unsignedShortValue;

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)enhancedAddSceneWithParams:(MTRScenesClusterEnhancedAddSceneParams *)params
                        completion:(void (^)(MTRScenesClusterEnhancedAddSceneResponseParams * _Nullable data,
                                       NSError * _Nullable error))completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRScenesClusterEnhancedAddSceneResponseCallbackBridge(self.callbackQueue, completion,
        ^(ExchangeManager & exchangeManager, const SessionHandle & session,
            ScenesClusterEnhancedAddSceneResponseCallbackType successCb, MTRErrorCallback failureCb,
            MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRScenesClusterEnhancedAddSceneResponseCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            Scenes::Commands::EnhancedAddScene::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.groupID = params.groupID.unsignedShortValue;
            request.sceneID = params.sceneID.unsignedCharValue;
            request.transitionTime = params.transitionTime.unsignedShortValue;
            request.sceneName = [self asCharSpan:params.sceneName];
            {
                using ListType_0 = std::remove_reference_t<decltype(request.extensionFieldSets)>;
                using ListMemberType_0 = ListMemberTypeGetter<ListType_0>::Type;
                if (params.extensionFieldSets.count != 0) {
                    auto * listHolder_0 = new ListHolder<ListMemberType_0>(params.extensionFieldSets.count);
                    if (listHolder_0 == nullptr || listHolder_0->mList == nullptr) {
                        return CHIP_ERROR_INVALID_ARGUMENT;
                    }
                    listFreer.add(listHolder_0);
                    for (size_t i_0 = 0; i_0 < params.extensionFieldSets.count; ++i_0) {
                        if (![params.extensionFieldSets[i_0] isKindOfClass:[MTRScenesClusterExtensionFieldSet class]]) {
                            // Wrong kind of value.
                            return CHIP_ERROR_INVALID_ARGUMENT;
                        }
                        auto element_0 = (MTRScenesClusterExtensionFieldSet *) params.extensionFieldSets[i_0];
                        listHolder_0->mList[i_0].clusterID = element_0.clusterID.unsignedIntValue;
                        {
                            using ListType_2 = std::remove_reference_t<decltype(listHolder_0->mList[i_0].attributeValueList)>;
                            using ListMemberType_2 = ListMemberTypeGetter<ListType_2>::Type;
                            if (element_0.attributeValueList.count != 0) {
                                auto * listHolder_2 = new ListHolder<ListMemberType_2>(element_0.attributeValueList.count);
                                if (listHolder_2 == nullptr || listHolder_2->mList == nullptr) {
                                    return CHIP_ERROR_INVALID_ARGUMENT;
                                }
                                listFreer.add(listHolder_2);
                                for (size_t i_2 = 0; i_2 < element_0.attributeValueList.count; ++i_2) {
                                    if (![element_0.attributeValueList[i_2]
                                            isKindOfClass:[MTRScenesClusterAttributeValuePair class]]) {
                                        // Wrong kind of value.
                                        return CHIP_ERROR_INVALID_ARGUMENT;
                                    }
                                    auto element_2 = (MTRScenesClusterAttributeValuePair *) element_0.attributeValueList[i_2];
                                    if (element_2.attributeID != nil) {
                                        auto & definedValue_4 = listHolder_2->mList[i_2].attributeID.Emplace();
                                        definedValue_4 = element_2.attributeID.unsignedIntValue;
                                    }
                                    listHolder_2->mList[i_2].attributeValue = element_2.attributeValue.unsignedIntValue;
                                }
                                listHolder_0->mList[i_0].attributeValueList
                                    = ListType_2(listHolder_2->mList, element_0.attributeValueList.count);
                            } else {
                                listHolder_0->mList[i_0].attributeValueList = ListType_2();
                            }
                        }
                    }
                    request.extensionFieldSets = ListType_0(listHolder_0->mList, params.extensionFieldSets.count);
                } else {
                    request.extensionFieldSets = ListType_0();
                }
            }

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)enhancedViewSceneWithParams:(MTRScenesClusterEnhancedViewSceneParams *)params
                         completion:(void (^)(MTRScenesClusterEnhancedViewSceneResponseParams * _Nullable data,
                                        NSError * _Nullable error))completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRScenesClusterEnhancedViewSceneResponseCallbackBridge(self.callbackQueue, completion,
        ^(ExchangeManager & exchangeManager, const SessionHandle & session,
            ScenesClusterEnhancedViewSceneResponseCallbackType successCb, MTRErrorCallback failureCb,
            MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRScenesClusterEnhancedViewSceneResponseCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            Scenes::Commands::EnhancedViewScene::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.groupID = params.groupID.unsignedShortValue;
            request.sceneID = params.sceneID.unsignedCharValue;

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)copySceneWithParams:(MTRScenesClusterCopySceneParams *)params
                 completion:
                     (void (^)(MTRScenesClusterCopySceneResponseParams * _Nullable data, NSError * _Nullable error))completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRScenesClusterCopySceneResponseCallbackBridge(self.callbackQueue, completion,
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, ScenesClusterCopySceneResponseCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRScenesClusterCopySceneResponseCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            Scenes::Commands::CopyScene::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.mode = static_cast<std::remove_reference_t<decltype(request.mode)>>(params.mode.unsignedCharValue);
            request.groupIdentifierFrom = params.groupIdentifierFrom.unsignedShortValue;
            request.sceneIdentifierFrom = params.sceneIdentifierFrom.unsignedCharValue;
            request.groupIdentifierTo = params.groupIdentifierTo.unsignedShortValue;
            request.sceneIdentifierTo = params.sceneIdentifierTo.unsignedCharValue;

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)readAttributeSceneCountWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Scenes::Attributes::SceneCount::TypeInfo;
    return MTRReadAttribute<MTRInt8uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeSceneCountWithParams:(MTRSubscribeParams * _Nonnull)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Scenes::Attributes::SceneCount::TypeInfo;
    MTRSubscribeAttribute<MTRInt8uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeSceneCountWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt8uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int8uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Scenes::Attributes::SceneCount::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeCurrentSceneWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Scenes::Attributes::CurrentScene::TypeInfo;
    return MTRReadAttribute<MTRInt8uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeCurrentSceneWithParams:(MTRSubscribeParams * _Nonnull)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Scenes::Attributes::CurrentScene::TypeInfo;
    MTRSubscribeAttribute<MTRInt8uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeCurrentSceneWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt8uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int8uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Scenes::Attributes::CurrentScene::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeCurrentGroupWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Scenes::Attributes::CurrentGroup::TypeInfo;
    return MTRReadAttribute<MTRInt16uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeCurrentGroupWithParams:(MTRSubscribeParams * _Nonnull)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Scenes::Attributes::CurrentGroup::TypeInfo;
    MTRSubscribeAttribute<MTRInt16uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeCurrentGroupWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt16uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int16uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Scenes::Attributes::CurrentGroup::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeSceneValidWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Scenes::Attributes::SceneValid::TypeInfo;
    return MTRReadAttribute<MTRBooleanAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeSceneValidWithParams:(MTRSubscribeParams * _Nonnull)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Scenes::Attributes::SceneValid::TypeInfo;
    MTRSubscribeAttribute<MTRBooleanAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeSceneValidWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRBooleanAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(BooleanAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Scenes::Attributes::SceneValid::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeNameSupportWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Scenes::Attributes::NameSupport::TypeInfo;
    return MTRReadAttribute<MTRInt8uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeNameSupportWithParams:(MTRSubscribeParams * _Nonnull)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Scenes::Attributes::NameSupport::TypeInfo;
    MTRSubscribeAttribute<MTRInt8uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeNameSupportWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt8uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int8uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Scenes::Attributes::NameSupport::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeLastConfiguredByWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Scenes::Attributes::LastConfiguredBy::TypeInfo;
    return MTRReadAttribute<MTRNullableInt64uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeLastConfiguredByWithParams:(MTRSubscribeParams * _Nonnull)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Scenes::Attributes::LastConfiguredBy::TypeInfo;
    MTRSubscribeAttribute<MTRNullableInt64uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeLastConfiguredByWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:
                                                    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRNullableInt64uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(NullableInt64uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Scenes::Attributes::LastConfiguredBy::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeSceneTableSizeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Scenes::Attributes::SceneTableSize::TypeInfo;
    return MTRReadAttribute<MTRInt16uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeSceneTableSizeWithParams:(MTRSubscribeParams * _Nonnull)params
                           subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                     reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Scenes::Attributes::SceneTableSize::TypeInfo;
    MTRSubscribeAttribute<MTRInt16uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeSceneTableSizeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                endpoint:(NSNumber *)endpoint
                                                   queue:(dispatch_queue_t)queue
                                              completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt16uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int16uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Scenes::Attributes::SceneTableSize::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeRemainingCapacityWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Scenes::Attributes::RemainingCapacity::TypeInfo;
    return MTRReadAttribute<MTRInt8uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeRemainingCapacityWithParams:(MTRSubscribeParams * _Nonnull)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Scenes::Attributes::RemainingCapacity::TypeInfo;
    MTRSubscribeAttribute<MTRInt8uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeRemainingCapacityWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                                 completion:
                                                     (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt8uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int8uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Scenes::Attributes::RemainingCapacity::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Scenes::Attributes::GeneratedCommandList::TypeInfo;
    return MTRReadAttribute<MTRScenesGeneratedCommandListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams * _Nonnull)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Scenes::Attributes::GeneratedCommandList::TypeInfo;
    MTRSubscribeAttribute<MTRScenesGeneratedCommandListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(
        params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRScenesGeneratedCommandListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(ScenesGeneratedCommandListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Scenes::Attributes::GeneratedCommandList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Scenes::Attributes::AcceptedCommandList::TypeInfo;
    return MTRReadAttribute<MTRScenesAcceptedCommandListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams * _Nonnull)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Scenes::Attributes::AcceptedCommandList::TypeInfo;
    MTRSubscribeAttribute<MTRScenesAcceptedCommandListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(
        params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRScenesAcceptedCommandListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(ScenesAcceptedCommandListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Scenes::Attributes::AcceptedCommandList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Scenes::Attributes::EventList::TypeInfo;
    return MTRReadAttribute<MTRScenesEventListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams * _Nonnull)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Scenes::Attributes::EventList::TypeInfo;
    MTRSubscribeAttribute<MTRScenesEventListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRScenesEventListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(ScenesEventListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Scenes::Attributes::EventList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Scenes::Attributes::AttributeList::TypeInfo;
    return MTRReadAttribute<MTRScenesAttributeListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams * _Nonnull)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Scenes::Attributes::AttributeList::TypeInfo;
    MTRSubscribeAttribute<MTRScenesAttributeListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRScenesAttributeListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(ScenesAttributeListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Scenes::Attributes::AttributeList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Scenes::Attributes::FeatureMap::TypeInfo;
    return MTRReadAttribute<MTRInt32uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams * _Nonnull)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Scenes::Attributes::FeatureMap::TypeInfo;
    MTRSubscribeAttribute<MTRInt32uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt32uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int32uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Scenes::Attributes::FeatureMap::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Scenes::Attributes::ClusterRevision::TypeInfo;
    return MTRReadAttribute<MTRInt16uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams * _Nonnull)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Scenes::Attributes::ClusterRevision::TypeInfo;
    MTRSubscribeAttribute<MTRInt16uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt16uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int16uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Scenes::Attributes::ClusterRevision::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

@end

@implementation MTRBaseClusterScenes (Deprecated)

- (void)addSceneWithParams:(MTRScenesClusterAddSceneParams *)params
         completionHandler:
             (void (^)(MTRScenesClusterAddSceneResponseParams * _Nullable data, NSError * _Nullable error))completionHandler
{
    [self addSceneWithParams:params
                  completion:^(MTRScenesClusterAddSceneResponseParams * _Nullable data, NSError * _Nullable error) {
                      // Cast is safe because subclass does not add any selectors.
                      completionHandler(static_cast<MTRScenesClusterAddSceneResponseParams *>(data), error);
                  }];
}
- (void)viewSceneWithParams:(MTRScenesClusterViewSceneParams *)params
          completionHandler:
              (void (^)(MTRScenesClusterViewSceneResponseParams * _Nullable data, NSError * _Nullable error))completionHandler
{
    [self viewSceneWithParams:params
                   completion:^(MTRScenesClusterViewSceneResponseParams * _Nullable data, NSError * _Nullable error) {
                       // Cast is safe because subclass does not add any selectors.
                       completionHandler(static_cast<MTRScenesClusterViewSceneResponseParams *>(data), error);
                   }];
}
- (void)removeSceneWithParams:(MTRScenesClusterRemoveSceneParams *)params
            completionHandler:
                (void (^)(MTRScenesClusterRemoveSceneResponseParams * _Nullable data, NSError * _Nullable error))completionHandler
{
    [self removeSceneWithParams:params
                     completion:^(MTRScenesClusterRemoveSceneResponseParams * _Nullable data, NSError * _Nullable error) {
                         // Cast is safe because subclass does not add any selectors.
                         completionHandler(static_cast<MTRScenesClusterRemoveSceneResponseParams *>(data), error);
                     }];
}
- (void)removeAllScenesWithParams:(MTRScenesClusterRemoveAllScenesParams *)params
                completionHandler:(void (^)(MTRScenesClusterRemoveAllScenesResponseParams * _Nullable data,
                                      NSError * _Nullable error))completionHandler
{
    [self removeAllScenesWithParams:params
                         completion:^(MTRScenesClusterRemoveAllScenesResponseParams * _Nullable data, NSError * _Nullable error) {
                             // Cast is safe because subclass does not add any selectors.
                             completionHandler(static_cast<MTRScenesClusterRemoveAllScenesResponseParams *>(data), error);
                         }];
}
- (void)storeSceneWithParams:(MTRScenesClusterStoreSceneParams *)params
           completionHandler:
               (void (^)(MTRScenesClusterStoreSceneResponseParams * _Nullable data, NSError * _Nullable error))completionHandler
{
    [self storeSceneWithParams:params
                    completion:^(MTRScenesClusterStoreSceneResponseParams * _Nullable data, NSError * _Nullable error) {
                        // Cast is safe because subclass does not add any selectors.
                        completionHandler(static_cast<MTRScenesClusterStoreSceneResponseParams *>(data), error);
                    }];
}
- (void)recallSceneWithParams:(MTRScenesClusterRecallSceneParams *)params completionHandler:(MTRStatusCompletion)completionHandler
{
    [self recallSceneWithParams:params completion:completionHandler];
}
- (void)getSceneMembershipWithParams:(MTRScenesClusterGetSceneMembershipParams *)params
                   completionHandler:(void (^)(MTRScenesClusterGetSceneMembershipResponseParams * _Nullable data,
                                         NSError * _Nullable error))completionHandler
{
    [self getSceneMembershipWithParams:params
                            completion:^(
                                MTRScenesClusterGetSceneMembershipResponseParams * _Nullable data, NSError * _Nullable error) {
                                // Cast is safe because subclass does not add any selectors.
                                completionHandler(static_cast<MTRScenesClusterGetSceneMembershipResponseParams *>(data), error);
                            }];
}
- (void)enhancedAddSceneWithParams:(MTRScenesClusterEnhancedAddSceneParams *)params
                 completionHandler:(void (^)(MTRScenesClusterEnhancedAddSceneResponseParams * _Nullable data,
                                       NSError * _Nullable error))completionHandler
{
    [self enhancedAddSceneWithParams:params
                          completion:^(MTRScenesClusterEnhancedAddSceneResponseParams * _Nullable data, NSError * _Nullable error) {
                              // Cast is safe because subclass does not add any selectors.
                              completionHandler(static_cast<MTRScenesClusterEnhancedAddSceneResponseParams *>(data), error);
                          }];
}
- (void)enhancedViewSceneWithParams:(MTRScenesClusterEnhancedViewSceneParams *)params
                  completionHandler:(void (^)(MTRScenesClusterEnhancedViewSceneResponseParams * _Nullable data,
                                        NSError * _Nullable error))completionHandler
{
    [self
        enhancedViewSceneWithParams:params
                         completion:^(MTRScenesClusterEnhancedViewSceneResponseParams * _Nullable data, NSError * _Nullable error) {
                             // Cast is safe because subclass does not add any selectors.
                             completionHandler(static_cast<MTRScenesClusterEnhancedViewSceneResponseParams *>(data), error);
                         }];
}
- (void)copySceneWithParams:(MTRScenesClusterCopySceneParams *)params
          completionHandler:
              (void (^)(MTRScenesClusterCopySceneResponseParams * _Nullable data, NSError * _Nullable error))completionHandler
{
    [self copySceneWithParams:params
                   completion:^(MTRScenesClusterCopySceneResponseParams * _Nullable data, NSError * _Nullable error) {
                       // Cast is safe because subclass does not add any selectors.
                       completionHandler(static_cast<MTRScenesClusterCopySceneResponseParams *>(data), error);
                   }];
}

- (void)readAttributeSceneCountWithCompletionHandler:(void (^)(
                                                         NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeSceneCountWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeSceneCountWithMinInterval:(NSNumber * _Nonnull)minInterval
                                        maxInterval:(NSNumber * _Nonnull)maxInterval
                                             params:(MTRSubscribeParams * _Nullable)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeSceneCountWithParams:subscribeParams
                         subscriptionEstablished:subscriptionEstablishedHandler
                                   reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                       // Cast is safe because subclass does not add any selectors.
                                       reportHandler(static_cast<NSNumber *>(value), error);
                                   }];
}
+ (void)readAttributeSceneCountWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                completionHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeSceneCountWithClusterStateCache:attributeCacheContainer.realContainer
                                              endpoint:endpoint
                                                 queue:queue
                                            completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                completionHandler(static_cast<NSNumber *>(value), error);
                                            }];
}

- (void)readAttributeCurrentSceneWithCompletionHandler:(void (^)(
                                                           NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeCurrentSceneWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeCurrentSceneWithMinInterval:(NSNumber * _Nonnull)minInterval
                                          maxInterval:(NSNumber * _Nonnull)maxInterval
                                               params:(MTRSubscribeParams * _Nullable)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeCurrentSceneWithParams:subscribeParams
                           subscriptionEstablished:subscriptionEstablishedHandler
                                     reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                         // Cast is safe because subclass does not add any selectors.
                                         reportHandler(static_cast<NSNumber *>(value), error);
                                     }];
}
+ (void)readAttributeCurrentSceneWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                  completionHandler:
                                      (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeCurrentSceneWithClusterStateCache:attributeCacheContainer.realContainer
                                                endpoint:endpoint
                                                   queue:queue
                                              completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                  // Cast is safe because subclass does not add any selectors.
                                                  completionHandler(static_cast<NSNumber *>(value), error);
                                              }];
}

- (void)readAttributeCurrentGroupWithCompletionHandler:(void (^)(
                                                           NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeCurrentGroupWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeCurrentGroupWithMinInterval:(NSNumber * _Nonnull)minInterval
                                          maxInterval:(NSNumber * _Nonnull)maxInterval
                                               params:(MTRSubscribeParams * _Nullable)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeCurrentGroupWithParams:subscribeParams
                           subscriptionEstablished:subscriptionEstablishedHandler
                                     reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                         // Cast is safe because subclass does not add any selectors.
                                         reportHandler(static_cast<NSNumber *>(value), error);
                                     }];
}
+ (void)readAttributeCurrentGroupWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                  completionHandler:
                                      (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeCurrentGroupWithClusterStateCache:attributeCacheContainer.realContainer
                                                endpoint:endpoint
                                                   queue:queue
                                              completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                  // Cast is safe because subclass does not add any selectors.
                                                  completionHandler(static_cast<NSNumber *>(value), error);
                                              }];
}

- (void)readAttributeSceneValidWithCompletionHandler:(void (^)(
                                                         NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeSceneValidWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeSceneValidWithMinInterval:(NSNumber * _Nonnull)minInterval
                                        maxInterval:(NSNumber * _Nonnull)maxInterval
                                             params:(MTRSubscribeParams * _Nullable)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeSceneValidWithParams:subscribeParams
                         subscriptionEstablished:subscriptionEstablishedHandler
                                   reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                       // Cast is safe because subclass does not add any selectors.
                                       reportHandler(static_cast<NSNumber *>(value), error);
                                   }];
}
+ (void)readAttributeSceneValidWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                completionHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeSceneValidWithClusterStateCache:attributeCacheContainer.realContainer
                                              endpoint:endpoint
                                                 queue:queue
                                            completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                completionHandler(static_cast<NSNumber *>(value), error);
                                            }];
}

- (void)readAttributeNameSupportWithCompletionHandler:(void (^)(
                                                          NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeNameSupportWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeNameSupportWithMinInterval:(NSNumber * _Nonnull)minInterval
                                         maxInterval:(NSNumber * _Nonnull)maxInterval
                                              params:(MTRSubscribeParams * _Nullable)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeNameSupportWithParams:subscribeParams
                          subscriptionEstablished:subscriptionEstablishedHandler
                                    reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                        // Cast is safe because subclass does not add any selectors.
                                        reportHandler(static_cast<NSNumber *>(value), error);
                                    }];
}
+ (void)readAttributeNameSupportWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                          endpoint:(NSNumber *)endpoint
                                             queue:(dispatch_queue_t)queue
                                 completionHandler:
                                     (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeNameSupportWithClusterStateCache:attributeCacheContainer.realContainer
                                               endpoint:endpoint
                                                  queue:queue
                                             completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                 // Cast is safe because subclass does not add any selectors.
                                                 completionHandler(static_cast<NSNumber *>(value), error);
                                             }];
}

- (void)readAttributeLastConfiguredByWithCompletionHandler:(void (^)(NSNumber * _Nullable value,
                                                               NSError * _Nullable error))completionHandler
{
    [self readAttributeLastConfiguredByWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)
    subscribeAttributeLastConfiguredByWithMinInterval:(NSNumber * _Nonnull)minInterval
                                          maxInterval:(NSNumber * _Nonnull)maxInterval
                                               params:(MTRSubscribeParams * _Nullable)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeLastConfiguredByWithParams:subscribeParams
                               subscriptionEstablished:subscriptionEstablishedHandler
                                         reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                             // Cast is safe because subclass does not add any selectors.
                                             reportHandler(static_cast<NSNumber *>(value), error);
                                         }];
}
+ (void)readAttributeLastConfiguredByWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                      completionHandler:
                                          (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeLastConfiguredByWithClusterStateCache:attributeCacheContainer.realContainer
                                                    endpoint:endpoint
                                                       queue:queue
                                                  completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                      // Cast is safe because subclass does not add any selectors.
                                                      completionHandler(static_cast<NSNumber *>(value), error);
                                                  }];
}

- (void)readAttributeGeneratedCommandListWithCompletionHandler:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))completionHandler
{
    [self readAttributeGeneratedCommandListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeGeneratedCommandListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                  maxInterval:(NSNumber * _Nonnull)maxInterval
                                                       params:(MTRSubscribeParams * _Nullable)params
                                      subscriptionEstablished:
                                          (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                                reportHandler:
                                                    (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeGeneratedCommandListWithParams:subscribeParams
                                   subscriptionEstablished:subscriptionEstablishedHandler
                                             reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                 // Cast is safe because subclass does not add any selectors.
                                                 reportHandler(static_cast<NSArray *>(value), error);
                                             }];
}
+ (void)readAttributeGeneratedCommandListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                          completionHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeGeneratedCommandListWithClusterStateCache:attributeCacheContainer.realContainer
                                                        endpoint:endpoint
                                                           queue:queue
                                                      completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                          // Cast is safe because subclass does not add any selectors.
                                                          completionHandler(static_cast<NSArray *>(value), error);
                                                      }];
}

- (void)readAttributeAcceptedCommandListWithCompletionHandler:(void (^)(NSArray * _Nullable value,
                                                                  NSError * _Nullable error))completionHandler
{
    [self readAttributeAcceptedCommandListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeAcceptedCommandListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                 maxInterval:(NSNumber * _Nonnull)maxInterval
                                                      params:(MTRSubscribeParams * _Nullable)params
                                     subscriptionEstablished:
                                         (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                               reportHandler:
                                                   (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeAcceptedCommandListWithParams:subscribeParams
                                  subscriptionEstablished:subscriptionEstablishedHandler
                                            reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                reportHandler(static_cast<NSArray *>(value), error);
                                            }];
}
+ (void)readAttributeAcceptedCommandListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                         completionHandler:
                                             (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAcceptedCommandListWithClusterStateCache:attributeCacheContainer.realContainer
                                                       endpoint:endpoint
                                                          queue:queue
                                                     completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                         // Cast is safe because subclass does not add any selectors.
                                                         completionHandler(static_cast<NSArray *>(value), error);
                                                     }];
}

- (void)readAttributeAttributeListWithCompletionHandler:(void (^)(
                                                            NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAttributeListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeAttributeListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                           maxInterval:(NSNumber * _Nonnull)maxInterval
                                                params:(MTRSubscribeParams * _Nullable)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                         reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeAttributeListWithParams:subscribeParams
                            subscriptionEstablished:subscriptionEstablishedHandler
                                      reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                          // Cast is safe because subclass does not add any selectors.
                                          reportHandler(static_cast<NSArray *>(value), error);
                                      }];
}
+ (void)readAttributeAttributeListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                   completionHandler:
                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAttributeListWithClusterStateCache:attributeCacheContainer.realContainer
                                                 endpoint:endpoint
                                                    queue:queue
                                               completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                   // Cast is safe because subclass does not add any selectors.
                                                   completionHandler(static_cast<NSArray *>(value), error);
                                               }];
}

- (void)readAttributeFeatureMapWithCompletionHandler:(void (^)(
                                                         NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeFeatureMapWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeFeatureMapWithMinInterval:(NSNumber * _Nonnull)minInterval
                                        maxInterval:(NSNumber * _Nonnull)maxInterval
                                             params:(MTRSubscribeParams * _Nullable)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeFeatureMapWithParams:subscribeParams
                         subscriptionEstablished:subscriptionEstablishedHandler
                                   reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                       // Cast is safe because subclass does not add any selectors.
                                       reportHandler(static_cast<NSNumber *>(value), error);
                                   }];
}
+ (void)readAttributeFeatureMapWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                completionHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeFeatureMapWithClusterStateCache:attributeCacheContainer.realContainer
                                              endpoint:endpoint
                                                 queue:queue
                                            completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                completionHandler(static_cast<NSNumber *>(value), error);
                                            }];
}

- (void)readAttributeClusterRevisionWithCompletionHandler:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completionHandler
{
    [self readAttributeClusterRevisionWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeClusterRevisionWithMinInterval:(NSNumber * _Nonnull)minInterval
                                             maxInterval:(NSNumber * _Nonnull)maxInterval
                                                  params:(MTRSubscribeParams * _Nullable)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                           reportHandler:
                                               (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeClusterRevisionWithParams:subscribeParams
                              subscriptionEstablished:subscriptionEstablishedHandler
                                        reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                            // Cast is safe because subclass does not add any selectors.
                                            reportHandler(static_cast<NSNumber *>(value), error);
                                        }];
}
+ (void)readAttributeClusterRevisionWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                     completionHandler:
                                         (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeClusterRevisionWithClusterStateCache:attributeCacheContainer.realContainer
                                                   endpoint:endpoint
                                                      queue:queue
                                                 completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                     // Cast is safe because subclass does not add any selectors.
                                                     completionHandler(static_cast<NSNumber *>(value), error);
                                                 }];
}

- (nullable instancetype)initWithDevice:(MTRBaseDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

@end

@implementation MTRBaseClusterOnOff

- (instancetype)initWithDevice:(MTRBaseDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _device = device;
        _endpoint = [endpointID unsignedShortValue];
    }
    return self;
}

- (void)offWithCompletion:(MTRStatusCompletion)completion
{
    [self offWithParams:nil completion:completion];
}
- (void)offWithParams:(MTROnOffClusterOffParams * _Nullable)params completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            OnOff::Commands::Off::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)onWithCompletion:(MTRStatusCompletion)completion
{
    [self onWithParams:nil completion:completion];
}
- (void)onWithParams:(MTROnOffClusterOnParams * _Nullable)params completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            OnOff::Commands::On::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)toggleWithCompletion:(MTRStatusCompletion)completion
{
    [self toggleWithParams:nil completion:completion];
}
- (void)toggleWithParams:(MTROnOffClusterToggleParams * _Nullable)params completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            OnOff::Commands::Toggle::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)offWithEffectWithParams:(MTROnOffClusterOffWithEffectParams *)params completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            OnOff::Commands::OffWithEffect::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.effectIdentifier = static_cast<std::remove_reference_t<decltype(request.effectIdentifier)>>(
                params.effectIdentifier.unsignedCharValue);
            request.effectVariant = params.effectVariant.unsignedCharValue;

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)onWithRecallGlobalSceneWithCompletion:(MTRStatusCompletion)completion
{
    [self onWithRecallGlobalSceneWithParams:nil completion:completion];
}
- (void)onWithRecallGlobalSceneWithParams:(MTROnOffClusterOnWithRecallGlobalSceneParams * _Nullable)params
                               completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            OnOff::Commands::OnWithRecallGlobalScene::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)onWithTimedOffWithParams:(MTROnOffClusterOnWithTimedOffParams *)params completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            OnOff::Commands::OnWithTimedOff::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.onOffControl
                = static_cast<std::remove_reference_t<decltype(request.onOffControl)>>(params.onOffControl.unsignedCharValue);
            request.onTime = params.onTime.unsignedShortValue;
            request.offWaitTime = params.offWaitTime.unsignedShortValue;

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)readAttributeOnOffWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = OnOff::Attributes::OnOff::TypeInfo;
    return MTRReadAttribute<MTRBooleanAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeOnOffWithParams:(MTRSubscribeParams * _Nonnull)params
                  subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                            reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = OnOff::Attributes::OnOff::TypeInfo;
    MTRSubscribeAttribute<MTRBooleanAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeOnOffWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                       endpoint:(NSNumber *)endpoint
                                          queue:(dispatch_queue_t)queue
                                     completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRBooleanAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(BooleanAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = OnOff::Attributes::OnOff::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeGlobalSceneControlWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = OnOff::Attributes::GlobalSceneControl::TypeInfo;
    return MTRReadAttribute<MTRBooleanAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeGlobalSceneControlWithParams:(MTRSubscribeParams * _Nonnull)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                         reportHandler:
                                             (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = OnOff::Attributes::GlobalSceneControl::TypeInfo;
    MTRSubscribeAttribute<MTRBooleanAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeGlobalSceneControlWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                    endpoint:(NSNumber *)endpoint
                                                       queue:(dispatch_queue_t)queue
                                                  completion:
                                                      (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRBooleanAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(BooleanAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = OnOff::Attributes::GlobalSceneControl::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeOnTimeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = OnOff::Attributes::OnTime::TypeInfo;
    return MTRReadAttribute<MTRInt16uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)writeAttributeOnTimeWithValue:(NSNumber * _Nonnull)value completion:(MTRStatusCompletion)completion
{
    [self writeAttributeOnTimeWithValue:(NSNumber * _Nonnull) value params:nil completion:completion];
}
- (void)writeAttributeOnTimeWithValue:(NSNumber * _Nonnull)value
                               params:(MTRWriteParams * _Nullable)params
                           completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    value = [value copy];

    auto * bridge = new MTRDefaultSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable ignored, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, DefaultSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            chip::Optional<uint16_t> timedWriteTimeout;
            if (params != nil) {
                if (params.timedWriteTimeout != nil) {
                    timedWriteTimeout.SetValue(params.timedWriteTimeout.unsignedShortValue);
                }
            }

            ListFreer listFreer;
            using TypeInfo = OnOff::Attributes::OnTime::TypeInfo;
            TypeInfo::Type cppValue;
            cppValue = value.unsignedShortValue;

            chip::Controller::ClusterBase cppCluster(exchangeManager, session, self->_endpoint);
            return cppCluster.WriteAttribute<TypeInfo>(cppValue, bridge, successCb, failureCb, timedWriteTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)subscribeAttributeOnTimeWithParams:(MTRSubscribeParams * _Nonnull)params
                   subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                             reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = OnOff::Attributes::OnTime::TypeInfo;
    MTRSubscribeAttribute<MTRInt16uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeOnTimeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                        endpoint:(NSNumber *)endpoint
                                           queue:(dispatch_queue_t)queue
                                      completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt16uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int16uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = OnOff::Attributes::OnTime::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeOffWaitTimeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = OnOff::Attributes::OffWaitTime::TypeInfo;
    return MTRReadAttribute<MTRInt16uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)writeAttributeOffWaitTimeWithValue:(NSNumber * _Nonnull)value completion:(MTRStatusCompletion)completion
{
    [self writeAttributeOffWaitTimeWithValue:(NSNumber * _Nonnull) value params:nil completion:completion];
}
- (void)writeAttributeOffWaitTimeWithValue:(NSNumber * _Nonnull)value
                                    params:(MTRWriteParams * _Nullable)params
                                completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    value = [value copy];

    auto * bridge = new MTRDefaultSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable ignored, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, DefaultSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            chip::Optional<uint16_t> timedWriteTimeout;
            if (params != nil) {
                if (params.timedWriteTimeout != nil) {
                    timedWriteTimeout.SetValue(params.timedWriteTimeout.unsignedShortValue);
                }
            }

            ListFreer listFreer;
            using TypeInfo = OnOff::Attributes::OffWaitTime::TypeInfo;
            TypeInfo::Type cppValue;
            cppValue = value.unsignedShortValue;

            chip::Controller::ClusterBase cppCluster(exchangeManager, session, self->_endpoint);
            return cppCluster.WriteAttribute<TypeInfo>(cppValue, bridge, successCb, failureCb, timedWriteTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)subscribeAttributeOffWaitTimeWithParams:(MTRSubscribeParams * _Nonnull)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = OnOff::Attributes::OffWaitTime::TypeInfo;
    MTRSubscribeAttribute<MTRInt16uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeOffWaitTimeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt16uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int16uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = OnOff::Attributes::OffWaitTime::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeStartUpOnOffWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = OnOff::Attributes::StartUpOnOff::TypeInfo;
    return MTRReadAttribute<MTRNullableOnOffClusterOnOffStartUpOnOffAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)writeAttributeStartUpOnOffWithValue:(NSNumber * _Nullable)value completion:(MTRStatusCompletion)completion
{
    [self writeAttributeStartUpOnOffWithValue:(NSNumber * _Nullable) value params:nil completion:completion];
}
- (void)writeAttributeStartUpOnOffWithValue:(NSNumber * _Nullable)value
                                     params:(MTRWriteParams * _Nullable)params
                                 completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    value = [value copy];

    auto * bridge = new MTRDefaultSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable ignored, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, DefaultSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            chip::Optional<uint16_t> timedWriteTimeout;
            if (params != nil) {
                if (params.timedWriteTimeout != nil) {
                    timedWriteTimeout.SetValue(params.timedWriteTimeout.unsignedShortValue);
                }
            }

            ListFreer listFreer;
            using TypeInfo = OnOff::Attributes::StartUpOnOff::TypeInfo;
            TypeInfo::Type cppValue;
            if (value == nil) {
                cppValue.SetNull();
            } else {
                auto & nonNullValue_0 = cppValue.SetNonNull();
                nonNullValue_0 = static_cast<std::remove_reference_t<decltype(nonNullValue_0)>>(value.unsignedCharValue);
            }

            chip::Controller::ClusterBase cppCluster(exchangeManager, session, self->_endpoint);
            return cppCluster.WriteAttribute<TypeInfo>(cppValue, bridge, successCb, failureCb, timedWriteTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)subscribeAttributeStartUpOnOffWithParams:(MTRSubscribeParams * _Nonnull)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = OnOff::Attributes::StartUpOnOff::TypeInfo;
    MTRSubscribeAttribute<MTRNullableOnOffClusterOnOffStartUpOnOffAttributeCallbackSubscriptionBridge, NSNumber,
        TypeInfo::DecodableType>(params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint,
        TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

+ (void)readAttributeStartUpOnOffWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRNullableOnOffClusterOnOffStartUpOnOffAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(NullableOnOffClusterOnOffStartUpOnOffAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = OnOff::Attributes::StartUpOnOff::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = OnOff::Attributes::GeneratedCommandList::TypeInfo;
    return MTRReadAttribute<MTROnOffGeneratedCommandListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams * _Nonnull)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = OnOff::Attributes::GeneratedCommandList::TypeInfo;
    MTRSubscribeAttribute<MTROnOffGeneratedCommandListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(
        params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTROnOffGeneratedCommandListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(OnOffGeneratedCommandListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = OnOff::Attributes::GeneratedCommandList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = OnOff::Attributes::AcceptedCommandList::TypeInfo;
    return MTRReadAttribute<MTROnOffAcceptedCommandListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams * _Nonnull)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = OnOff::Attributes::AcceptedCommandList::TypeInfo;
    MTRSubscribeAttribute<MTROnOffAcceptedCommandListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(
        params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTROnOffAcceptedCommandListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(OnOffAcceptedCommandListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = OnOff::Attributes::AcceptedCommandList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = OnOff::Attributes::EventList::TypeInfo;
    return MTRReadAttribute<MTROnOffEventListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams * _Nonnull)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = OnOff::Attributes::EventList::TypeInfo;
    MTRSubscribeAttribute<MTROnOffEventListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTROnOffEventListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(OnOffEventListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = OnOff::Attributes::EventList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = OnOff::Attributes::AttributeList::TypeInfo;
    return MTRReadAttribute<MTROnOffAttributeListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams * _Nonnull)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = OnOff::Attributes::AttributeList::TypeInfo;
    MTRSubscribeAttribute<MTROnOffAttributeListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTROnOffAttributeListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(OnOffAttributeListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = OnOff::Attributes::AttributeList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = OnOff::Attributes::FeatureMap::TypeInfo;
    return MTRReadAttribute<MTRInt32uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams * _Nonnull)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = OnOff::Attributes::FeatureMap::TypeInfo;
    MTRSubscribeAttribute<MTRInt32uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt32uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int32uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = OnOff::Attributes::FeatureMap::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = OnOff::Attributes::ClusterRevision::TypeInfo;
    return MTRReadAttribute<MTRInt16uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams * _Nonnull)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = OnOff::Attributes::ClusterRevision::TypeInfo;
    MTRSubscribeAttribute<MTRInt16uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt16uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int16uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = OnOff::Attributes::ClusterRevision::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

@end

@implementation MTRBaseClusterOnOff (Deprecated)

- (void)offWithParams:(MTROnOffClusterOffParams * _Nullable)params completionHandler:(MTRStatusCompletion)completionHandler
{
    [self offWithParams:params completion:completionHandler];
}
- (void)offWithCompletionHandler:(MTRStatusCompletion)completionHandler
{
    [self offWithParams:nil completionHandler:completionHandler];
}
- (void)onWithParams:(MTROnOffClusterOnParams * _Nullable)params completionHandler:(MTRStatusCompletion)completionHandler
{
    [self onWithParams:params completion:completionHandler];
}
- (void)onWithCompletionHandler:(MTRStatusCompletion)completionHandler
{
    [self onWithParams:nil completionHandler:completionHandler];
}
- (void)toggleWithParams:(MTROnOffClusterToggleParams * _Nullable)params completionHandler:(MTRStatusCompletion)completionHandler
{
    [self toggleWithParams:params completion:completionHandler];
}
- (void)toggleWithCompletionHandler:(MTRStatusCompletion)completionHandler
{
    [self toggleWithParams:nil completionHandler:completionHandler];
}
- (void)offWithEffectWithParams:(MTROnOffClusterOffWithEffectParams *)params
              completionHandler:(MTRStatusCompletion)completionHandler
{
    [self offWithEffectWithParams:params completion:completionHandler];
}
- (void)onWithRecallGlobalSceneWithParams:(MTROnOffClusterOnWithRecallGlobalSceneParams * _Nullable)params
                        completionHandler:(MTRStatusCompletion)completionHandler
{
    [self onWithRecallGlobalSceneWithParams:params completion:completionHandler];
}
- (void)onWithRecallGlobalSceneWithCompletionHandler:(MTRStatusCompletion)completionHandler
{
    [self onWithRecallGlobalSceneWithParams:nil completionHandler:completionHandler];
}
- (void)onWithTimedOffWithParams:(MTROnOffClusterOnWithTimedOffParams *)params
               completionHandler:(MTRStatusCompletion)completionHandler
{
    [self onWithTimedOffWithParams:params completion:completionHandler];
}

- (void)readAttributeOnOffWithCompletionHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeOnOffWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeOnOffWithMinInterval:(NSNumber * _Nonnull)minInterval
                                   maxInterval:(NSNumber * _Nonnull)maxInterval
                                        params:(MTRSubscribeParams * _Nullable)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeOnOffWithParams:subscribeParams
                    subscriptionEstablished:subscriptionEstablishedHandler
                              reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                  // Cast is safe because subclass does not add any selectors.
                                  reportHandler(static_cast<NSNumber *>(value), error);
                              }];
}
+ (void)readAttributeOnOffWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                    endpoint:(NSNumber *)endpoint
                                       queue:(dispatch_queue_t)queue
                           completionHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeOnOffWithClusterStateCache:attributeCacheContainer.realContainer
                                         endpoint:endpoint
                                            queue:queue
                                       completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                           // Cast is safe because subclass does not add any selectors.
                                           completionHandler(static_cast<NSNumber *>(value), error);
                                       }];
}

- (void)readAttributeGlobalSceneControlWithCompletionHandler:(void (^)(NSNumber * _Nullable value,
                                                                 NSError * _Nullable error))completionHandler
{
    [self readAttributeGlobalSceneControlWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeGlobalSceneControlWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                maxInterval:(NSNumber * _Nonnull)maxInterval
                                                     params:(MTRSubscribeParams * _Nullable)params
                                    subscriptionEstablished:
                                        (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                              reportHandler:
                                                  (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeGlobalSceneControlWithParams:subscribeParams
                                 subscriptionEstablished:subscriptionEstablishedHandler
                                           reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                               // Cast is safe because subclass does not add any selectors.
                                               reportHandler(static_cast<NSNumber *>(value), error);
                                           }];
}
+ (void)readAttributeGlobalSceneControlWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                        completionHandler:
                                            (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeGlobalSceneControlWithClusterStateCache:attributeCacheContainer.realContainer
                                                      endpoint:endpoint
                                                         queue:queue
                                                    completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                        // Cast is safe because subclass does not add any selectors.
                                                        completionHandler(static_cast<NSNumber *>(value), error);
                                                    }];
}

- (void)readAttributeOnTimeWithCompletionHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeOnTimeWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)writeAttributeOnTimeWithValue:(NSNumber * _Nonnull)value completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeOnTimeWithValue:value params:nil completion:completionHandler];
}
- (void)writeAttributeOnTimeWithValue:(NSNumber * _Nonnull)value
                               params:(MTRWriteParams * _Nullable)params
                    completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeOnTimeWithValue:value params:params completion:completionHandler];
}
- (void)subscribeAttributeOnTimeWithMinInterval:(NSNumber * _Nonnull)minInterval
                                    maxInterval:(NSNumber * _Nonnull)maxInterval
                                         params:(MTRSubscribeParams * _Nullable)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                  reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeOnTimeWithParams:subscribeParams
                     subscriptionEstablished:subscriptionEstablishedHandler
                               reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                   // Cast is safe because subclass does not add any selectors.
                                   reportHandler(static_cast<NSNumber *>(value), error);
                               }];
}
+ (void)readAttributeOnTimeWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                     endpoint:(NSNumber *)endpoint
                                        queue:(dispatch_queue_t)queue
                            completionHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeOnTimeWithClusterStateCache:attributeCacheContainer.realContainer
                                          endpoint:endpoint
                                             queue:queue
                                        completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                            // Cast is safe because subclass does not add any selectors.
                                            completionHandler(static_cast<NSNumber *>(value), error);
                                        }];
}

- (void)readAttributeOffWaitTimeWithCompletionHandler:(void (^)(
                                                          NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeOffWaitTimeWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)writeAttributeOffWaitTimeWithValue:(NSNumber * _Nonnull)value completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeOffWaitTimeWithValue:value params:nil completion:completionHandler];
}
- (void)writeAttributeOffWaitTimeWithValue:(NSNumber * _Nonnull)value
                                    params:(MTRWriteParams * _Nullable)params
                         completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeOffWaitTimeWithValue:value params:params completion:completionHandler];
}
- (void)subscribeAttributeOffWaitTimeWithMinInterval:(NSNumber * _Nonnull)minInterval
                                         maxInterval:(NSNumber * _Nonnull)maxInterval
                                              params:(MTRSubscribeParams * _Nullable)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeOffWaitTimeWithParams:subscribeParams
                          subscriptionEstablished:subscriptionEstablishedHandler
                                    reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                        // Cast is safe because subclass does not add any selectors.
                                        reportHandler(static_cast<NSNumber *>(value), error);
                                    }];
}
+ (void)readAttributeOffWaitTimeWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                          endpoint:(NSNumber *)endpoint
                                             queue:(dispatch_queue_t)queue
                                 completionHandler:
                                     (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeOffWaitTimeWithClusterStateCache:attributeCacheContainer.realContainer
                                               endpoint:endpoint
                                                  queue:queue
                                             completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                 // Cast is safe because subclass does not add any selectors.
                                                 completionHandler(static_cast<NSNumber *>(value), error);
                                             }];
}

- (void)readAttributeStartUpOnOffWithCompletionHandler:(void (^)(
                                                           NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeStartUpOnOffWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)writeAttributeStartUpOnOffWithValue:(NSNumber * _Nullable)value completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeStartUpOnOffWithValue:value params:nil completion:completionHandler];
}
- (void)writeAttributeStartUpOnOffWithValue:(NSNumber * _Nullable)value
                                     params:(MTRWriteParams * _Nullable)params
                          completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeStartUpOnOffWithValue:value params:params completion:completionHandler];
}
- (void)subscribeAttributeStartUpOnOffWithMinInterval:(NSNumber * _Nonnull)minInterval
                                          maxInterval:(NSNumber * _Nonnull)maxInterval
                                               params:(MTRSubscribeParams * _Nullable)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeStartUpOnOffWithParams:subscribeParams
                           subscriptionEstablished:subscriptionEstablishedHandler
                                     reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                         // Cast is safe because subclass does not add any selectors.
                                         reportHandler(static_cast<NSNumber *>(value), error);
                                     }];
}
+ (void)readAttributeStartUpOnOffWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                  completionHandler:
                                      (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeStartUpOnOffWithClusterStateCache:attributeCacheContainer.realContainer
                                                endpoint:endpoint
                                                   queue:queue
                                              completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                  // Cast is safe because subclass does not add any selectors.
                                                  completionHandler(static_cast<NSNumber *>(value), error);
                                              }];
}

- (void)readAttributeGeneratedCommandListWithCompletionHandler:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))completionHandler
{
    [self readAttributeGeneratedCommandListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeGeneratedCommandListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                  maxInterval:(NSNumber * _Nonnull)maxInterval
                                                       params:(MTRSubscribeParams * _Nullable)params
                                      subscriptionEstablished:
                                          (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                                reportHandler:
                                                    (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeGeneratedCommandListWithParams:subscribeParams
                                   subscriptionEstablished:subscriptionEstablishedHandler
                                             reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                 // Cast is safe because subclass does not add any selectors.
                                                 reportHandler(static_cast<NSArray *>(value), error);
                                             }];
}
+ (void)readAttributeGeneratedCommandListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                          completionHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeGeneratedCommandListWithClusterStateCache:attributeCacheContainer.realContainer
                                                        endpoint:endpoint
                                                           queue:queue
                                                      completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                          // Cast is safe because subclass does not add any selectors.
                                                          completionHandler(static_cast<NSArray *>(value), error);
                                                      }];
}

- (void)readAttributeAcceptedCommandListWithCompletionHandler:(void (^)(NSArray * _Nullable value,
                                                                  NSError * _Nullable error))completionHandler
{
    [self readAttributeAcceptedCommandListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeAcceptedCommandListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                 maxInterval:(NSNumber * _Nonnull)maxInterval
                                                      params:(MTRSubscribeParams * _Nullable)params
                                     subscriptionEstablished:
                                         (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                               reportHandler:
                                                   (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeAcceptedCommandListWithParams:subscribeParams
                                  subscriptionEstablished:subscriptionEstablishedHandler
                                            reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                reportHandler(static_cast<NSArray *>(value), error);
                                            }];
}
+ (void)readAttributeAcceptedCommandListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                         completionHandler:
                                             (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAcceptedCommandListWithClusterStateCache:attributeCacheContainer.realContainer
                                                       endpoint:endpoint
                                                          queue:queue
                                                     completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                         // Cast is safe because subclass does not add any selectors.
                                                         completionHandler(static_cast<NSArray *>(value), error);
                                                     }];
}

- (void)readAttributeAttributeListWithCompletionHandler:(void (^)(
                                                            NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAttributeListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeAttributeListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                           maxInterval:(NSNumber * _Nonnull)maxInterval
                                                params:(MTRSubscribeParams * _Nullable)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                         reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeAttributeListWithParams:subscribeParams
                            subscriptionEstablished:subscriptionEstablishedHandler
                                      reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                          // Cast is safe because subclass does not add any selectors.
                                          reportHandler(static_cast<NSArray *>(value), error);
                                      }];
}
+ (void)readAttributeAttributeListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                   completionHandler:
                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAttributeListWithClusterStateCache:attributeCacheContainer.realContainer
                                                 endpoint:endpoint
                                                    queue:queue
                                               completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                   // Cast is safe because subclass does not add any selectors.
                                                   completionHandler(static_cast<NSArray *>(value), error);
                                               }];
}

- (void)readAttributeFeatureMapWithCompletionHandler:(void (^)(
                                                         NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeFeatureMapWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeFeatureMapWithMinInterval:(NSNumber * _Nonnull)minInterval
                                        maxInterval:(NSNumber * _Nonnull)maxInterval
                                             params:(MTRSubscribeParams * _Nullable)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeFeatureMapWithParams:subscribeParams
                         subscriptionEstablished:subscriptionEstablishedHandler
                                   reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                       // Cast is safe because subclass does not add any selectors.
                                       reportHandler(static_cast<NSNumber *>(value), error);
                                   }];
}
+ (void)readAttributeFeatureMapWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                completionHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeFeatureMapWithClusterStateCache:attributeCacheContainer.realContainer
                                              endpoint:endpoint
                                                 queue:queue
                                            completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                completionHandler(static_cast<NSNumber *>(value), error);
                                            }];
}

- (void)readAttributeClusterRevisionWithCompletionHandler:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completionHandler
{
    [self readAttributeClusterRevisionWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeClusterRevisionWithMinInterval:(NSNumber * _Nonnull)minInterval
                                             maxInterval:(NSNumber * _Nonnull)maxInterval
                                                  params:(MTRSubscribeParams * _Nullable)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                           reportHandler:
                                               (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeClusterRevisionWithParams:subscribeParams
                              subscriptionEstablished:subscriptionEstablishedHandler
                                        reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                            // Cast is safe because subclass does not add any selectors.
                                            reportHandler(static_cast<NSNumber *>(value), error);
                                        }];
}
+ (void)readAttributeClusterRevisionWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                     completionHandler:
                                         (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeClusterRevisionWithClusterStateCache:attributeCacheContainer.realContainer
                                                   endpoint:endpoint
                                                      queue:queue
                                                 completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                     // Cast is safe because subclass does not add any selectors.
                                                     completionHandler(static_cast<NSNumber *>(value), error);
                                                 }];
}

- (nullable instancetype)initWithDevice:(MTRBaseDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

@end

@implementation MTRBaseClusterOnOffSwitchConfiguration

- (instancetype)initWithDevice:(MTRBaseDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _device = device;
        _endpoint = [endpointID unsignedShortValue];
    }
    return self;
}

- (void)readAttributeSwitchTypeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = OnOffSwitchConfiguration::Attributes::SwitchType::TypeInfo;
    return MTRReadAttribute<MTRInt8uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeSwitchTypeWithParams:(MTRSubscribeParams * _Nonnull)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = OnOffSwitchConfiguration::Attributes::SwitchType::TypeInfo;
    MTRSubscribeAttribute<MTRInt8uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeSwitchTypeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt8uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int8uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = OnOffSwitchConfiguration::Attributes::SwitchType::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeSwitchActionsWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = OnOffSwitchConfiguration::Attributes::SwitchActions::TypeInfo;
    return MTRReadAttribute<MTRInt8uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)writeAttributeSwitchActionsWithValue:(NSNumber * _Nonnull)value completion:(MTRStatusCompletion)completion
{
    [self writeAttributeSwitchActionsWithValue:(NSNumber * _Nonnull) value params:nil completion:completion];
}
- (void)writeAttributeSwitchActionsWithValue:(NSNumber * _Nonnull)value
                                      params:(MTRWriteParams * _Nullable)params
                                  completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    value = [value copy];

    auto * bridge = new MTRDefaultSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable ignored, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, DefaultSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            chip::Optional<uint16_t> timedWriteTimeout;
            if (params != nil) {
                if (params.timedWriteTimeout != nil) {
                    timedWriteTimeout.SetValue(params.timedWriteTimeout.unsignedShortValue);
                }
            }

            ListFreer listFreer;
            using TypeInfo = OnOffSwitchConfiguration::Attributes::SwitchActions::TypeInfo;
            TypeInfo::Type cppValue;
            cppValue = value.unsignedCharValue;

            chip::Controller::ClusterBase cppCluster(exchangeManager, session, self->_endpoint);
            return cppCluster.WriteAttribute<TypeInfo>(cppValue, bridge, successCb, failureCb, timedWriteTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)subscribeAttributeSwitchActionsWithParams:(MTRSubscribeParams * _Nonnull)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = OnOffSwitchConfiguration::Attributes::SwitchActions::TypeInfo;
    MTRSubscribeAttribute<MTRInt8uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeSwitchActionsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt8uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int8uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = OnOffSwitchConfiguration::Attributes::SwitchActions::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = OnOffSwitchConfiguration::Attributes::GeneratedCommandList::TypeInfo;
    return MTRReadAttribute<MTROnOffSwitchConfigurationGeneratedCommandListListAttributeCallbackBridge, NSArray,
        TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams * _Nonnull)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = OnOffSwitchConfiguration::Attributes::GeneratedCommandList::TypeInfo;
    MTRSubscribeAttribute<MTROnOffSwitchConfigurationGeneratedCommandListListAttributeCallbackSubscriptionBridge, NSArray,
        TypeInfo::DecodableType>(params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint,
        TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTROnOffSwitchConfigurationGeneratedCommandListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(OnOffSwitchConfigurationGeneratedCommandListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = OnOffSwitchConfiguration::Attributes::GeneratedCommandList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = OnOffSwitchConfiguration::Attributes::AcceptedCommandList::TypeInfo;
    return MTRReadAttribute<MTROnOffSwitchConfigurationAcceptedCommandListListAttributeCallbackBridge, NSArray,
        TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams * _Nonnull)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = OnOffSwitchConfiguration::Attributes::AcceptedCommandList::TypeInfo;
    MTRSubscribeAttribute<MTROnOffSwitchConfigurationAcceptedCommandListListAttributeCallbackSubscriptionBridge, NSArray,
        TypeInfo::DecodableType>(params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint,
        TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTROnOffSwitchConfigurationAcceptedCommandListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(OnOffSwitchConfigurationAcceptedCommandListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = OnOffSwitchConfiguration::Attributes::AcceptedCommandList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = OnOffSwitchConfiguration::Attributes::EventList::TypeInfo;
    return MTRReadAttribute<MTROnOffSwitchConfigurationEventListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams * _Nonnull)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = OnOffSwitchConfiguration::Attributes::EventList::TypeInfo;
    MTRSubscribeAttribute<MTROnOffSwitchConfigurationEventListListAttributeCallbackSubscriptionBridge, NSArray,
        TypeInfo::DecodableType>(params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint,
        TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTROnOffSwitchConfigurationEventListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(OnOffSwitchConfigurationEventListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = OnOffSwitchConfiguration::Attributes::EventList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = OnOffSwitchConfiguration::Attributes::AttributeList::TypeInfo;
    return MTRReadAttribute<MTROnOffSwitchConfigurationAttributeListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams * _Nonnull)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = OnOffSwitchConfiguration::Attributes::AttributeList::TypeInfo;
    MTRSubscribeAttribute<MTROnOffSwitchConfigurationAttributeListListAttributeCallbackSubscriptionBridge, NSArray,
        TypeInfo::DecodableType>(params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint,
        TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTROnOffSwitchConfigurationAttributeListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(OnOffSwitchConfigurationAttributeListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = OnOffSwitchConfiguration::Attributes::AttributeList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = OnOffSwitchConfiguration::Attributes::FeatureMap::TypeInfo;
    return MTRReadAttribute<MTRInt32uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams * _Nonnull)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = OnOffSwitchConfiguration::Attributes::FeatureMap::TypeInfo;
    MTRSubscribeAttribute<MTRInt32uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt32uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int32uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = OnOffSwitchConfiguration::Attributes::FeatureMap::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = OnOffSwitchConfiguration::Attributes::ClusterRevision::TypeInfo;
    return MTRReadAttribute<MTRInt16uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams * _Nonnull)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = OnOffSwitchConfiguration::Attributes::ClusterRevision::TypeInfo;
    MTRSubscribeAttribute<MTRInt16uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt16uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int16uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = OnOffSwitchConfiguration::Attributes::ClusterRevision::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

@end

@implementation MTRBaseClusterOnOffSwitchConfiguration (Deprecated)

- (void)readAttributeSwitchTypeWithCompletionHandler:(void (^)(
                                                         NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeSwitchTypeWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeSwitchTypeWithMinInterval:(NSNumber * _Nonnull)minInterval
                                        maxInterval:(NSNumber * _Nonnull)maxInterval
                                             params:(MTRSubscribeParams * _Nullable)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeSwitchTypeWithParams:subscribeParams
                         subscriptionEstablished:subscriptionEstablishedHandler
                                   reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                       // Cast is safe because subclass does not add any selectors.
                                       reportHandler(static_cast<NSNumber *>(value), error);
                                   }];
}
+ (void)readAttributeSwitchTypeWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                completionHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeSwitchTypeWithClusterStateCache:attributeCacheContainer.realContainer
                                              endpoint:endpoint
                                                 queue:queue
                                            completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                completionHandler(static_cast<NSNumber *>(value), error);
                                            }];
}

- (void)readAttributeSwitchActionsWithCompletionHandler:(void (^)(
                                                            NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeSwitchActionsWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)writeAttributeSwitchActionsWithValue:(NSNumber * _Nonnull)value completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeSwitchActionsWithValue:value params:nil completion:completionHandler];
}
- (void)writeAttributeSwitchActionsWithValue:(NSNumber * _Nonnull)value
                                      params:(MTRWriteParams * _Nullable)params
                           completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeSwitchActionsWithValue:value params:params completion:completionHandler];
}
- (void)subscribeAttributeSwitchActionsWithMinInterval:(NSNumber * _Nonnull)minInterval
                                           maxInterval:(NSNumber * _Nonnull)maxInterval
                                                params:(MTRSubscribeParams * _Nullable)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                         reportHandler:
                                             (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeSwitchActionsWithParams:subscribeParams
                            subscriptionEstablished:subscriptionEstablishedHandler
                                      reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                          // Cast is safe because subclass does not add any selectors.
                                          reportHandler(static_cast<NSNumber *>(value), error);
                                      }];
}
+ (void)readAttributeSwitchActionsWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                   completionHandler:
                                       (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeSwitchActionsWithClusterStateCache:attributeCacheContainer.realContainer
                                                 endpoint:endpoint
                                                    queue:queue
                                               completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                   // Cast is safe because subclass does not add any selectors.
                                                   completionHandler(static_cast<NSNumber *>(value), error);
                                               }];
}

- (void)readAttributeGeneratedCommandListWithCompletionHandler:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))completionHandler
{
    [self readAttributeGeneratedCommandListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeGeneratedCommandListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                  maxInterval:(NSNumber * _Nonnull)maxInterval
                                                       params:(MTRSubscribeParams * _Nullable)params
                                      subscriptionEstablished:
                                          (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                                reportHandler:
                                                    (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeGeneratedCommandListWithParams:subscribeParams
                                   subscriptionEstablished:subscriptionEstablishedHandler
                                             reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                 // Cast is safe because subclass does not add any selectors.
                                                 reportHandler(static_cast<NSArray *>(value), error);
                                             }];
}
+ (void)readAttributeGeneratedCommandListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                          completionHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeGeneratedCommandListWithClusterStateCache:attributeCacheContainer.realContainer
                                                        endpoint:endpoint
                                                           queue:queue
                                                      completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                          // Cast is safe because subclass does not add any selectors.
                                                          completionHandler(static_cast<NSArray *>(value), error);
                                                      }];
}

- (void)readAttributeAcceptedCommandListWithCompletionHandler:(void (^)(NSArray * _Nullable value,
                                                                  NSError * _Nullable error))completionHandler
{
    [self readAttributeAcceptedCommandListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeAcceptedCommandListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                 maxInterval:(NSNumber * _Nonnull)maxInterval
                                                      params:(MTRSubscribeParams * _Nullable)params
                                     subscriptionEstablished:
                                         (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                               reportHandler:
                                                   (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeAcceptedCommandListWithParams:subscribeParams
                                  subscriptionEstablished:subscriptionEstablishedHandler
                                            reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                reportHandler(static_cast<NSArray *>(value), error);
                                            }];
}
+ (void)readAttributeAcceptedCommandListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                         completionHandler:
                                             (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAcceptedCommandListWithClusterStateCache:attributeCacheContainer.realContainer
                                                       endpoint:endpoint
                                                          queue:queue
                                                     completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                         // Cast is safe because subclass does not add any selectors.
                                                         completionHandler(static_cast<NSArray *>(value), error);
                                                     }];
}

- (void)readAttributeAttributeListWithCompletionHandler:(void (^)(
                                                            NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAttributeListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeAttributeListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                           maxInterval:(NSNumber * _Nonnull)maxInterval
                                                params:(MTRSubscribeParams * _Nullable)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                         reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeAttributeListWithParams:subscribeParams
                            subscriptionEstablished:subscriptionEstablishedHandler
                                      reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                          // Cast is safe because subclass does not add any selectors.
                                          reportHandler(static_cast<NSArray *>(value), error);
                                      }];
}
+ (void)readAttributeAttributeListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                   completionHandler:
                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAttributeListWithClusterStateCache:attributeCacheContainer.realContainer
                                                 endpoint:endpoint
                                                    queue:queue
                                               completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                   // Cast is safe because subclass does not add any selectors.
                                                   completionHandler(static_cast<NSArray *>(value), error);
                                               }];
}

- (void)readAttributeFeatureMapWithCompletionHandler:(void (^)(
                                                         NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeFeatureMapWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeFeatureMapWithMinInterval:(NSNumber * _Nonnull)minInterval
                                        maxInterval:(NSNumber * _Nonnull)maxInterval
                                             params:(MTRSubscribeParams * _Nullable)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeFeatureMapWithParams:subscribeParams
                         subscriptionEstablished:subscriptionEstablishedHandler
                                   reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                       // Cast is safe because subclass does not add any selectors.
                                       reportHandler(static_cast<NSNumber *>(value), error);
                                   }];
}
+ (void)readAttributeFeatureMapWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                completionHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeFeatureMapWithClusterStateCache:attributeCacheContainer.realContainer
                                              endpoint:endpoint
                                                 queue:queue
                                            completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                completionHandler(static_cast<NSNumber *>(value), error);
                                            }];
}

- (void)readAttributeClusterRevisionWithCompletionHandler:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completionHandler
{
    [self readAttributeClusterRevisionWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeClusterRevisionWithMinInterval:(NSNumber * _Nonnull)minInterval
                                             maxInterval:(NSNumber * _Nonnull)maxInterval
                                                  params:(MTRSubscribeParams * _Nullable)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                           reportHandler:
                                               (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeClusterRevisionWithParams:subscribeParams
                              subscriptionEstablished:subscriptionEstablishedHandler
                                        reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                            // Cast is safe because subclass does not add any selectors.
                                            reportHandler(static_cast<NSNumber *>(value), error);
                                        }];
}
+ (void)readAttributeClusterRevisionWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                     completionHandler:
                                         (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeClusterRevisionWithClusterStateCache:attributeCacheContainer.realContainer
                                                   endpoint:endpoint
                                                      queue:queue
                                                 completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                     // Cast is safe because subclass does not add any selectors.
                                                     completionHandler(static_cast<NSNumber *>(value), error);
                                                 }];
}

- (nullable instancetype)initWithDevice:(MTRBaseDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

@end

@implementation MTRBaseClusterLevelControl

- (instancetype)initWithDevice:(MTRBaseDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _device = device;
        _endpoint = [endpointID unsignedShortValue];
    }
    return self;
}

- (void)moveToLevelWithParams:(MTRLevelControlClusterMoveToLevelParams *)params completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            LevelControl::Commands::MoveToLevel::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.level = params.level.unsignedCharValue;
            if (params.transitionTime == nil) {
                request.transitionTime.SetNull();
            } else {
                auto & nonNullValue_0 = request.transitionTime.SetNonNull();
                nonNullValue_0 = params.transitionTime.unsignedShortValue;
            }
            request.optionsMask
                = static_cast<std::remove_reference_t<decltype(request.optionsMask)>>(params.optionsMask.unsignedCharValue);
            request.optionsOverride
                = static_cast<std::remove_reference_t<decltype(request.optionsOverride)>>(params.optionsOverride.unsignedCharValue);

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)moveWithParams:(MTRLevelControlClusterMoveParams *)params completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            LevelControl::Commands::Move::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.moveMode = static_cast<std::remove_reference_t<decltype(request.moveMode)>>(params.moveMode.unsignedCharValue);
            if (params.rate == nil) {
                request.rate.SetNull();
            } else {
                auto & nonNullValue_0 = request.rate.SetNonNull();
                nonNullValue_0 = params.rate.unsignedCharValue;
            }
            request.optionsMask
                = static_cast<std::remove_reference_t<decltype(request.optionsMask)>>(params.optionsMask.unsignedCharValue);
            request.optionsOverride
                = static_cast<std::remove_reference_t<decltype(request.optionsOverride)>>(params.optionsOverride.unsignedCharValue);

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)stepWithParams:(MTRLevelControlClusterStepParams *)params completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            LevelControl::Commands::Step::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.stepMode = static_cast<std::remove_reference_t<decltype(request.stepMode)>>(params.stepMode.unsignedCharValue);
            request.stepSize = params.stepSize.unsignedCharValue;
            if (params.transitionTime == nil) {
                request.transitionTime.SetNull();
            } else {
                auto & nonNullValue_0 = request.transitionTime.SetNonNull();
                nonNullValue_0 = params.transitionTime.unsignedShortValue;
            }
            request.optionsMask
                = static_cast<std::remove_reference_t<decltype(request.optionsMask)>>(params.optionsMask.unsignedCharValue);
            request.optionsOverride
                = static_cast<std::remove_reference_t<decltype(request.optionsOverride)>>(params.optionsOverride.unsignedCharValue);

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)stopWithParams:(MTRLevelControlClusterStopParams *)params completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            LevelControl::Commands::Stop::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.optionsMask
                = static_cast<std::remove_reference_t<decltype(request.optionsMask)>>(params.optionsMask.unsignedCharValue);
            request.optionsOverride
                = static_cast<std::remove_reference_t<decltype(request.optionsOverride)>>(params.optionsOverride.unsignedCharValue);

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)moveToLevelWithOnOffWithParams:(MTRLevelControlClusterMoveToLevelWithOnOffParams *)params
                            completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            LevelControl::Commands::MoveToLevelWithOnOff::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.level = params.level.unsignedCharValue;
            if (params.transitionTime == nil) {
                request.transitionTime.SetNull();
            } else {
                auto & nonNullValue_0 = request.transitionTime.SetNonNull();
                nonNullValue_0 = params.transitionTime.unsignedShortValue;
            }
            request.optionsMask
                = static_cast<std::remove_reference_t<decltype(request.optionsMask)>>(params.optionsMask.unsignedCharValue);
            request.optionsOverride
                = static_cast<std::remove_reference_t<decltype(request.optionsOverride)>>(params.optionsOverride.unsignedCharValue);

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)moveWithOnOffWithParams:(MTRLevelControlClusterMoveWithOnOffParams *)params completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            LevelControl::Commands::MoveWithOnOff::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.moveMode = static_cast<std::remove_reference_t<decltype(request.moveMode)>>(params.moveMode.unsignedCharValue);
            if (params.rate == nil) {
                request.rate.SetNull();
            } else {
                auto & nonNullValue_0 = request.rate.SetNonNull();
                nonNullValue_0 = params.rate.unsignedCharValue;
            }
            request.optionsMask
                = static_cast<std::remove_reference_t<decltype(request.optionsMask)>>(params.optionsMask.unsignedCharValue);
            request.optionsOverride
                = static_cast<std::remove_reference_t<decltype(request.optionsOverride)>>(params.optionsOverride.unsignedCharValue);

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)stepWithOnOffWithParams:(MTRLevelControlClusterStepWithOnOffParams *)params completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            LevelControl::Commands::StepWithOnOff::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.stepMode = static_cast<std::remove_reference_t<decltype(request.stepMode)>>(params.stepMode.unsignedCharValue);
            request.stepSize = params.stepSize.unsignedCharValue;
            if (params.transitionTime == nil) {
                request.transitionTime.SetNull();
            } else {
                auto & nonNullValue_0 = request.transitionTime.SetNonNull();
                nonNullValue_0 = params.transitionTime.unsignedShortValue;
            }
            request.optionsMask
                = static_cast<std::remove_reference_t<decltype(request.optionsMask)>>(params.optionsMask.unsignedCharValue);
            request.optionsOverride
                = static_cast<std::remove_reference_t<decltype(request.optionsOverride)>>(params.optionsOverride.unsignedCharValue);

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)stopWithOnOffWithParams:(MTRLevelControlClusterStopWithOnOffParams *)params completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            LevelControl::Commands::StopWithOnOff::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.optionsMask
                = static_cast<std::remove_reference_t<decltype(request.optionsMask)>>(params.optionsMask.unsignedCharValue);
            request.optionsOverride
                = static_cast<std::remove_reference_t<decltype(request.optionsOverride)>>(params.optionsOverride.unsignedCharValue);

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)moveToClosestFrequencyWithParams:(MTRLevelControlClusterMoveToClosestFrequencyParams *)params
                              completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            LevelControl::Commands::MoveToClosestFrequency::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.frequency = params.frequency.unsignedShortValue;

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)readAttributeCurrentLevelWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = LevelControl::Attributes::CurrentLevel::TypeInfo;
    return MTRReadAttribute<MTRNullableInt8uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeCurrentLevelWithParams:(MTRSubscribeParams * _Nonnull)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = LevelControl::Attributes::CurrentLevel::TypeInfo;
    MTRSubscribeAttribute<MTRNullableInt8uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeCurrentLevelWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRNullableInt8uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(NullableInt8uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = LevelControl::Attributes::CurrentLevel::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeRemainingTimeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = LevelControl::Attributes::RemainingTime::TypeInfo;
    return MTRReadAttribute<MTRInt16uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeRemainingTimeWithParams:(MTRSubscribeParams * _Nonnull)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = LevelControl::Attributes::RemainingTime::TypeInfo;
    MTRSubscribeAttribute<MTRInt16uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeRemainingTimeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt16uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int16uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = LevelControl::Attributes::RemainingTime::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeMinLevelWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = LevelControl::Attributes::MinLevel::TypeInfo;
    return MTRReadAttribute<MTRInt8uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeMinLevelWithParams:(MTRSubscribeParams * _Nonnull)params
                     subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                               reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = LevelControl::Attributes::MinLevel::TypeInfo;
    MTRSubscribeAttribute<MTRInt8uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeMinLevelWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                          endpoint:(NSNumber *)endpoint
                                             queue:(dispatch_queue_t)queue
                                        completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt8uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int8uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = LevelControl::Attributes::MinLevel::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeMaxLevelWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = LevelControl::Attributes::MaxLevel::TypeInfo;
    return MTRReadAttribute<MTRInt8uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeMaxLevelWithParams:(MTRSubscribeParams * _Nonnull)params
                     subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                               reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = LevelControl::Attributes::MaxLevel::TypeInfo;
    MTRSubscribeAttribute<MTRInt8uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeMaxLevelWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                          endpoint:(NSNumber *)endpoint
                                             queue:(dispatch_queue_t)queue
                                        completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt8uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int8uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = LevelControl::Attributes::MaxLevel::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeCurrentFrequencyWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = LevelControl::Attributes::CurrentFrequency::TypeInfo;
    return MTRReadAttribute<MTRInt16uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeCurrentFrequencyWithParams:(MTRSubscribeParams * _Nonnull)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = LevelControl::Attributes::CurrentFrequency::TypeInfo;
    MTRSubscribeAttribute<MTRInt16uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeCurrentFrequencyWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:
                                                    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt16uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int16uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = LevelControl::Attributes::CurrentFrequency::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeMinFrequencyWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = LevelControl::Attributes::MinFrequency::TypeInfo;
    return MTRReadAttribute<MTRInt16uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeMinFrequencyWithParams:(MTRSubscribeParams * _Nonnull)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = LevelControl::Attributes::MinFrequency::TypeInfo;
    MTRSubscribeAttribute<MTRInt16uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeMinFrequencyWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt16uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int16uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = LevelControl::Attributes::MinFrequency::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeMaxFrequencyWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = LevelControl::Attributes::MaxFrequency::TypeInfo;
    return MTRReadAttribute<MTRInt16uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeMaxFrequencyWithParams:(MTRSubscribeParams * _Nonnull)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = LevelControl::Attributes::MaxFrequency::TypeInfo;
    MTRSubscribeAttribute<MTRInt16uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeMaxFrequencyWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt16uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int16uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = LevelControl::Attributes::MaxFrequency::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeOptionsWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = LevelControl::Attributes::Options::TypeInfo;
    return MTRReadAttribute<MTRLevelControlOptionsAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)writeAttributeOptionsWithValue:(NSNumber * _Nonnull)value completion:(MTRStatusCompletion)completion
{
    [self writeAttributeOptionsWithValue:(NSNumber * _Nonnull) value params:nil completion:completion];
}
- (void)writeAttributeOptionsWithValue:(NSNumber * _Nonnull)value
                                params:(MTRWriteParams * _Nullable)params
                            completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    value = [value copy];

    auto * bridge = new MTRDefaultSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable ignored, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, DefaultSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            chip::Optional<uint16_t> timedWriteTimeout;
            if (params != nil) {
                if (params.timedWriteTimeout != nil) {
                    timedWriteTimeout.SetValue(params.timedWriteTimeout.unsignedShortValue);
                }
            }

            ListFreer listFreer;
            using TypeInfo = LevelControl::Attributes::Options::TypeInfo;
            TypeInfo::Type cppValue;
            cppValue = static_cast<std::remove_reference_t<decltype(cppValue)>>(value.unsignedCharValue);

            chip::Controller::ClusterBase cppCluster(exchangeManager, session, self->_endpoint);
            return cppCluster.WriteAttribute<TypeInfo>(cppValue, bridge, successCb, failureCb, timedWriteTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)subscribeAttributeOptionsWithParams:(MTRSubscribeParams * _Nonnull)params
                    subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                              reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = LevelControl::Attributes::Options::TypeInfo;
    MTRSubscribeAttribute<MTRLevelControlOptionsAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeOptionsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                       completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRLevelControlOptionsAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(LevelControlOptionsAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = LevelControl::Attributes::Options::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeOnOffTransitionTimeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = LevelControl::Attributes::OnOffTransitionTime::TypeInfo;
    return MTRReadAttribute<MTRInt16uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)writeAttributeOnOffTransitionTimeWithValue:(NSNumber * _Nonnull)value completion:(MTRStatusCompletion)completion
{
    [self writeAttributeOnOffTransitionTimeWithValue:(NSNumber * _Nonnull) value params:nil completion:completion];
}
- (void)writeAttributeOnOffTransitionTimeWithValue:(NSNumber * _Nonnull)value
                                            params:(MTRWriteParams * _Nullable)params
                                        completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    value = [value copy];

    auto * bridge = new MTRDefaultSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable ignored, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, DefaultSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            chip::Optional<uint16_t> timedWriteTimeout;
            if (params != nil) {
                if (params.timedWriteTimeout != nil) {
                    timedWriteTimeout.SetValue(params.timedWriteTimeout.unsignedShortValue);
                }
            }

            ListFreer listFreer;
            using TypeInfo = LevelControl::Attributes::OnOffTransitionTime::TypeInfo;
            TypeInfo::Type cppValue;
            cppValue = value.unsignedShortValue;

            chip::Controller::ClusterBase cppCluster(exchangeManager, session, self->_endpoint);
            return cppCluster.WriteAttribute<TypeInfo>(cppValue, bridge, successCb, failureCb, timedWriteTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)subscribeAttributeOnOffTransitionTimeWithParams:(MTRSubscribeParams * _Nonnull)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = LevelControl::Attributes::OnOffTransitionTime::TypeInfo;
    MTRSubscribeAttribute<MTRInt16uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeOnOffTransitionTimeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt16uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int16uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = LevelControl::Attributes::OnOffTransitionTime::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeOnLevelWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = LevelControl::Attributes::OnLevel::TypeInfo;
    return MTRReadAttribute<MTRNullableInt8uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)writeAttributeOnLevelWithValue:(NSNumber * _Nullable)value completion:(MTRStatusCompletion)completion
{
    [self writeAttributeOnLevelWithValue:(NSNumber * _Nullable) value params:nil completion:completion];
}
- (void)writeAttributeOnLevelWithValue:(NSNumber * _Nullable)value
                                params:(MTRWriteParams * _Nullable)params
                            completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    value = [value copy];

    auto * bridge = new MTRDefaultSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable ignored, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, DefaultSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            chip::Optional<uint16_t> timedWriteTimeout;
            if (params != nil) {
                if (params.timedWriteTimeout != nil) {
                    timedWriteTimeout.SetValue(params.timedWriteTimeout.unsignedShortValue);
                }
            }

            ListFreer listFreer;
            using TypeInfo = LevelControl::Attributes::OnLevel::TypeInfo;
            TypeInfo::Type cppValue;
            if (value == nil) {
                cppValue.SetNull();
            } else {
                auto & nonNullValue_0 = cppValue.SetNonNull();
                nonNullValue_0 = value.unsignedCharValue;
            }

            chip::Controller::ClusterBase cppCluster(exchangeManager, session, self->_endpoint);
            return cppCluster.WriteAttribute<TypeInfo>(cppValue, bridge, successCb, failureCb, timedWriteTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)subscribeAttributeOnLevelWithParams:(MTRSubscribeParams * _Nonnull)params
                    subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                              reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = LevelControl::Attributes::OnLevel::TypeInfo;
    MTRSubscribeAttribute<MTRNullableInt8uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeOnLevelWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                       completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRNullableInt8uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(NullableInt8uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = LevelControl::Attributes::OnLevel::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeOnTransitionTimeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = LevelControl::Attributes::OnTransitionTime::TypeInfo;
    return MTRReadAttribute<MTRNullableInt16uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)writeAttributeOnTransitionTimeWithValue:(NSNumber * _Nullable)value completion:(MTRStatusCompletion)completion
{
    [self writeAttributeOnTransitionTimeWithValue:(NSNumber * _Nullable) value params:nil completion:completion];
}
- (void)writeAttributeOnTransitionTimeWithValue:(NSNumber * _Nullable)value
                                         params:(MTRWriteParams * _Nullable)params
                                     completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    value = [value copy];

    auto * bridge = new MTRDefaultSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable ignored, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, DefaultSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            chip::Optional<uint16_t> timedWriteTimeout;
            if (params != nil) {
                if (params.timedWriteTimeout != nil) {
                    timedWriteTimeout.SetValue(params.timedWriteTimeout.unsignedShortValue);
                }
            }

            ListFreer listFreer;
            using TypeInfo = LevelControl::Attributes::OnTransitionTime::TypeInfo;
            TypeInfo::Type cppValue;
            if (value == nil) {
                cppValue.SetNull();
            } else {
                auto & nonNullValue_0 = cppValue.SetNonNull();
                nonNullValue_0 = value.unsignedShortValue;
            }

            chip::Controller::ClusterBase cppCluster(exchangeManager, session, self->_endpoint);
            return cppCluster.WriteAttribute<TypeInfo>(cppValue, bridge, successCb, failureCb, timedWriteTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)subscribeAttributeOnTransitionTimeWithParams:(MTRSubscribeParams * _Nonnull)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = LevelControl::Attributes::OnTransitionTime::TypeInfo;
    MTRSubscribeAttribute<MTRNullableInt16uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeOnTransitionTimeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:
                                                    (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRNullableInt16uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(NullableInt16uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = LevelControl::Attributes::OnTransitionTime::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeOffTransitionTimeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = LevelControl::Attributes::OffTransitionTime::TypeInfo;
    return MTRReadAttribute<MTRNullableInt16uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)writeAttributeOffTransitionTimeWithValue:(NSNumber * _Nullable)value completion:(MTRStatusCompletion)completion
{
    [self writeAttributeOffTransitionTimeWithValue:(NSNumber * _Nullable) value params:nil completion:completion];
}
- (void)writeAttributeOffTransitionTimeWithValue:(NSNumber * _Nullable)value
                                          params:(MTRWriteParams * _Nullable)params
                                      completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    value = [value copy];

    auto * bridge = new MTRDefaultSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable ignored, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, DefaultSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            chip::Optional<uint16_t> timedWriteTimeout;
            if (params != nil) {
                if (params.timedWriteTimeout != nil) {
                    timedWriteTimeout.SetValue(params.timedWriteTimeout.unsignedShortValue);
                }
            }

            ListFreer listFreer;
            using TypeInfo = LevelControl::Attributes::OffTransitionTime::TypeInfo;
            TypeInfo::Type cppValue;
            if (value == nil) {
                cppValue.SetNull();
            } else {
                auto & nonNullValue_0 = cppValue.SetNonNull();
                nonNullValue_0 = value.unsignedShortValue;
            }

            chip::Controller::ClusterBase cppCluster(exchangeManager, session, self->_endpoint);
            return cppCluster.WriteAttribute<TypeInfo>(cppValue, bridge, successCb, failureCb, timedWriteTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)subscribeAttributeOffTransitionTimeWithParams:(MTRSubscribeParams * _Nonnull)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = LevelControl::Attributes::OffTransitionTime::TypeInfo;
    MTRSubscribeAttribute<MTRNullableInt16uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeOffTransitionTimeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                                 completion:
                                                     (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRNullableInt16uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(NullableInt16uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = LevelControl::Attributes::OffTransitionTime::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeDefaultMoveRateWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = LevelControl::Attributes::DefaultMoveRate::TypeInfo;
    return MTRReadAttribute<MTRNullableInt8uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)writeAttributeDefaultMoveRateWithValue:(NSNumber * _Nullable)value completion:(MTRStatusCompletion)completion
{
    [self writeAttributeDefaultMoveRateWithValue:(NSNumber * _Nullable) value params:nil completion:completion];
}
- (void)writeAttributeDefaultMoveRateWithValue:(NSNumber * _Nullable)value
                                        params:(MTRWriteParams * _Nullable)params
                                    completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    value = [value copy];

    auto * bridge = new MTRDefaultSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable ignored, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, DefaultSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            chip::Optional<uint16_t> timedWriteTimeout;
            if (params != nil) {
                if (params.timedWriteTimeout != nil) {
                    timedWriteTimeout.SetValue(params.timedWriteTimeout.unsignedShortValue);
                }
            }

            ListFreer listFreer;
            using TypeInfo = LevelControl::Attributes::DefaultMoveRate::TypeInfo;
            TypeInfo::Type cppValue;
            if (value == nil) {
                cppValue.SetNull();
            } else {
                auto & nonNullValue_0 = cppValue.SetNonNull();
                nonNullValue_0 = value.unsignedCharValue;
            }

            chip::Controller::ClusterBase cppCluster(exchangeManager, session, self->_endpoint);
            return cppCluster.WriteAttribute<TypeInfo>(cppValue, bridge, successCb, failureCb, timedWriteTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)subscribeAttributeDefaultMoveRateWithParams:(MTRSubscribeParams * _Nonnull)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = LevelControl::Attributes::DefaultMoveRate::TypeInfo;
    MTRSubscribeAttribute<MTRNullableInt8uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeDefaultMoveRateWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRNullableInt8uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(NullableInt8uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = LevelControl::Attributes::DefaultMoveRate::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeStartUpCurrentLevelWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = LevelControl::Attributes::StartUpCurrentLevel::TypeInfo;
    return MTRReadAttribute<MTRNullableInt8uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)writeAttributeStartUpCurrentLevelWithValue:(NSNumber * _Nullable)value completion:(MTRStatusCompletion)completion
{
    [self writeAttributeStartUpCurrentLevelWithValue:(NSNumber * _Nullable) value params:nil completion:completion];
}
- (void)writeAttributeStartUpCurrentLevelWithValue:(NSNumber * _Nullable)value
                                            params:(MTRWriteParams * _Nullable)params
                                        completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    value = [value copy];

    auto * bridge = new MTRDefaultSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable ignored, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, DefaultSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            chip::Optional<uint16_t> timedWriteTimeout;
            if (params != nil) {
                if (params.timedWriteTimeout != nil) {
                    timedWriteTimeout.SetValue(params.timedWriteTimeout.unsignedShortValue);
                }
            }

            ListFreer listFreer;
            using TypeInfo = LevelControl::Attributes::StartUpCurrentLevel::TypeInfo;
            TypeInfo::Type cppValue;
            if (value == nil) {
                cppValue.SetNull();
            } else {
                auto & nonNullValue_0 = cppValue.SetNonNull();
                nonNullValue_0 = value.unsignedCharValue;
            }

            chip::Controller::ClusterBase cppCluster(exchangeManager, session, self->_endpoint);
            return cppCluster.WriteAttribute<TypeInfo>(cppValue, bridge, successCb, failureCb, timedWriteTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)subscribeAttributeStartUpCurrentLevelWithParams:(MTRSubscribeParams * _Nonnull)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = LevelControl::Attributes::StartUpCurrentLevel::TypeInfo;
    MTRSubscribeAttribute<MTRNullableInt8uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeStartUpCurrentLevelWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRNullableInt8uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(NullableInt8uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = LevelControl::Attributes::StartUpCurrentLevel::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = LevelControl::Attributes::GeneratedCommandList::TypeInfo;
    return MTRReadAttribute<MTRLevelControlGeneratedCommandListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams * _Nonnull)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = LevelControl::Attributes::GeneratedCommandList::TypeInfo;
    MTRSubscribeAttribute<MTRLevelControlGeneratedCommandListListAttributeCallbackSubscriptionBridge, NSArray,
        TypeInfo::DecodableType>(params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint,
        TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRLevelControlGeneratedCommandListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(LevelControlGeneratedCommandListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = LevelControl::Attributes::GeneratedCommandList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = LevelControl::Attributes::AcceptedCommandList::TypeInfo;
    return MTRReadAttribute<MTRLevelControlAcceptedCommandListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams * _Nonnull)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = LevelControl::Attributes::AcceptedCommandList::TypeInfo;
    MTRSubscribeAttribute<MTRLevelControlAcceptedCommandListListAttributeCallbackSubscriptionBridge, NSArray,
        TypeInfo::DecodableType>(params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint,
        TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRLevelControlAcceptedCommandListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(LevelControlAcceptedCommandListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = LevelControl::Attributes::AcceptedCommandList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = LevelControl::Attributes::EventList::TypeInfo;
    return MTRReadAttribute<MTRLevelControlEventListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams * _Nonnull)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = LevelControl::Attributes::EventList::TypeInfo;
    MTRSubscribeAttribute<MTRLevelControlEventListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRLevelControlEventListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(LevelControlEventListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = LevelControl::Attributes::EventList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = LevelControl::Attributes::AttributeList::TypeInfo;
    return MTRReadAttribute<MTRLevelControlAttributeListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams * _Nonnull)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = LevelControl::Attributes::AttributeList::TypeInfo;
    MTRSubscribeAttribute<MTRLevelControlAttributeListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(
        params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRLevelControlAttributeListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(LevelControlAttributeListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = LevelControl::Attributes::AttributeList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = LevelControl::Attributes::FeatureMap::TypeInfo;
    return MTRReadAttribute<MTRInt32uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams * _Nonnull)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = LevelControl::Attributes::FeatureMap::TypeInfo;
    MTRSubscribeAttribute<MTRInt32uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt32uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int32uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = LevelControl::Attributes::FeatureMap::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = LevelControl::Attributes::ClusterRevision::TypeInfo;
    return MTRReadAttribute<MTRInt16uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams * _Nonnull)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = LevelControl::Attributes::ClusterRevision::TypeInfo;
    MTRSubscribeAttribute<MTRInt16uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt16uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int16uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = LevelControl::Attributes::ClusterRevision::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

@end

@implementation MTRBaseClusterLevelControl (Deprecated)

- (void)moveToLevelWithParams:(MTRLevelControlClusterMoveToLevelParams *)params
            completionHandler:(MTRStatusCompletion)completionHandler
{
    [self moveToLevelWithParams:params completion:completionHandler];
}
- (void)moveWithParams:(MTRLevelControlClusterMoveParams *)params completionHandler:(MTRStatusCompletion)completionHandler
{
    [self moveWithParams:params completion:completionHandler];
}
- (void)stepWithParams:(MTRLevelControlClusterStepParams *)params completionHandler:(MTRStatusCompletion)completionHandler
{
    [self stepWithParams:params completion:completionHandler];
}
- (void)stopWithParams:(MTRLevelControlClusterStopParams *)params completionHandler:(MTRStatusCompletion)completionHandler
{
    [self stopWithParams:params completion:completionHandler];
}
- (void)moveToLevelWithOnOffWithParams:(MTRLevelControlClusterMoveToLevelWithOnOffParams *)params
                     completionHandler:(MTRStatusCompletion)completionHandler
{
    [self moveToLevelWithOnOffWithParams:params completion:completionHandler];
}
- (void)moveWithOnOffWithParams:(MTRLevelControlClusterMoveWithOnOffParams *)params
              completionHandler:(MTRStatusCompletion)completionHandler
{
    [self moveWithOnOffWithParams:params completion:completionHandler];
}
- (void)stepWithOnOffWithParams:(MTRLevelControlClusterStepWithOnOffParams *)params
              completionHandler:(MTRStatusCompletion)completionHandler
{
    [self stepWithOnOffWithParams:params completion:completionHandler];
}
- (void)stopWithOnOffWithParams:(MTRLevelControlClusterStopWithOnOffParams *)params
              completionHandler:(MTRStatusCompletion)completionHandler
{
    [self stopWithOnOffWithParams:params completion:completionHandler];
}
- (void)moveToClosestFrequencyWithParams:(MTRLevelControlClusterMoveToClosestFrequencyParams *)params
                       completionHandler:(MTRStatusCompletion)completionHandler
{
    [self moveToClosestFrequencyWithParams:params completion:completionHandler];
}

- (void)readAttributeCurrentLevelWithCompletionHandler:(void (^)(
                                                           NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeCurrentLevelWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeCurrentLevelWithMinInterval:(NSNumber * _Nonnull)minInterval
                                          maxInterval:(NSNumber * _Nonnull)maxInterval
                                               params:(MTRSubscribeParams * _Nullable)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeCurrentLevelWithParams:subscribeParams
                           subscriptionEstablished:subscriptionEstablishedHandler
                                     reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                         // Cast is safe because subclass does not add any selectors.
                                         reportHandler(static_cast<NSNumber *>(value), error);
                                     }];
}
+ (void)readAttributeCurrentLevelWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                  completionHandler:
                                      (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeCurrentLevelWithClusterStateCache:attributeCacheContainer.realContainer
                                                endpoint:endpoint
                                                   queue:queue
                                              completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                  // Cast is safe because subclass does not add any selectors.
                                                  completionHandler(static_cast<NSNumber *>(value), error);
                                              }];
}

- (void)readAttributeRemainingTimeWithCompletionHandler:(void (^)(
                                                            NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeRemainingTimeWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeRemainingTimeWithMinInterval:(NSNumber * _Nonnull)minInterval
                                           maxInterval:(NSNumber * _Nonnull)maxInterval
                                                params:(MTRSubscribeParams * _Nullable)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                         reportHandler:
                                             (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeRemainingTimeWithParams:subscribeParams
                            subscriptionEstablished:subscriptionEstablishedHandler
                                      reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                          // Cast is safe because subclass does not add any selectors.
                                          reportHandler(static_cast<NSNumber *>(value), error);
                                      }];
}
+ (void)readAttributeRemainingTimeWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                   completionHandler:
                                       (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeRemainingTimeWithClusterStateCache:attributeCacheContainer.realContainer
                                                 endpoint:endpoint
                                                    queue:queue
                                               completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                   // Cast is safe because subclass does not add any selectors.
                                                   completionHandler(static_cast<NSNumber *>(value), error);
                                               }];
}

- (void)readAttributeMinLevelWithCompletionHandler:(void (^)(
                                                       NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeMinLevelWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeMinLevelWithMinInterval:(NSNumber * _Nonnull)minInterval
                                      maxInterval:(NSNumber * _Nonnull)maxInterval
                                           params:(MTRSubscribeParams * _Nullable)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                    reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeMinLevelWithParams:subscribeParams
                       subscriptionEstablished:subscriptionEstablishedHandler
                                 reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                     // Cast is safe because subclass does not add any selectors.
                                     reportHandler(static_cast<NSNumber *>(value), error);
                                 }];
}
+ (void)readAttributeMinLevelWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                       endpoint:(NSNumber *)endpoint
                                          queue:(dispatch_queue_t)queue
                              completionHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeMinLevelWithClusterStateCache:attributeCacheContainer.realContainer
                                            endpoint:endpoint
                                               queue:queue
                                          completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                              // Cast is safe because subclass does not add any selectors.
                                              completionHandler(static_cast<NSNumber *>(value), error);
                                          }];
}

- (void)readAttributeMaxLevelWithCompletionHandler:(void (^)(
                                                       NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeMaxLevelWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeMaxLevelWithMinInterval:(NSNumber * _Nonnull)minInterval
                                      maxInterval:(NSNumber * _Nonnull)maxInterval
                                           params:(MTRSubscribeParams * _Nullable)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                    reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeMaxLevelWithParams:subscribeParams
                       subscriptionEstablished:subscriptionEstablishedHandler
                                 reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                     // Cast is safe because subclass does not add any selectors.
                                     reportHandler(static_cast<NSNumber *>(value), error);
                                 }];
}
+ (void)readAttributeMaxLevelWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                       endpoint:(NSNumber *)endpoint
                                          queue:(dispatch_queue_t)queue
                              completionHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeMaxLevelWithClusterStateCache:attributeCacheContainer.realContainer
                                            endpoint:endpoint
                                               queue:queue
                                          completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                              // Cast is safe because subclass does not add any selectors.
                                              completionHandler(static_cast<NSNumber *>(value), error);
                                          }];
}

- (void)readAttributeCurrentFrequencyWithCompletionHandler:(void (^)(NSNumber * _Nullable value,
                                                               NSError * _Nullable error))completionHandler
{
    [self readAttributeCurrentFrequencyWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)
    subscribeAttributeCurrentFrequencyWithMinInterval:(NSNumber * _Nonnull)minInterval
                                          maxInterval:(NSNumber * _Nonnull)maxInterval
                                               params:(MTRSubscribeParams * _Nullable)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeCurrentFrequencyWithParams:subscribeParams
                               subscriptionEstablished:subscriptionEstablishedHandler
                                         reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                             // Cast is safe because subclass does not add any selectors.
                                             reportHandler(static_cast<NSNumber *>(value), error);
                                         }];
}
+ (void)readAttributeCurrentFrequencyWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                      completionHandler:
                                          (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeCurrentFrequencyWithClusterStateCache:attributeCacheContainer.realContainer
                                                    endpoint:endpoint
                                                       queue:queue
                                                  completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                      // Cast is safe because subclass does not add any selectors.
                                                      completionHandler(static_cast<NSNumber *>(value), error);
                                                  }];
}

- (void)readAttributeMinFrequencyWithCompletionHandler:(void (^)(
                                                           NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeMinFrequencyWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeMinFrequencyWithMinInterval:(NSNumber * _Nonnull)minInterval
                                          maxInterval:(NSNumber * _Nonnull)maxInterval
                                               params:(MTRSubscribeParams * _Nullable)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeMinFrequencyWithParams:subscribeParams
                           subscriptionEstablished:subscriptionEstablishedHandler
                                     reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                         // Cast is safe because subclass does not add any selectors.
                                         reportHandler(static_cast<NSNumber *>(value), error);
                                     }];
}
+ (void)readAttributeMinFrequencyWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                  completionHandler:
                                      (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeMinFrequencyWithClusterStateCache:attributeCacheContainer.realContainer
                                                endpoint:endpoint
                                                   queue:queue
                                              completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                  // Cast is safe because subclass does not add any selectors.
                                                  completionHandler(static_cast<NSNumber *>(value), error);
                                              }];
}

- (void)readAttributeMaxFrequencyWithCompletionHandler:(void (^)(
                                                           NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeMaxFrequencyWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeMaxFrequencyWithMinInterval:(NSNumber * _Nonnull)minInterval
                                          maxInterval:(NSNumber * _Nonnull)maxInterval
                                               params:(MTRSubscribeParams * _Nullable)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeMaxFrequencyWithParams:subscribeParams
                           subscriptionEstablished:subscriptionEstablishedHandler
                                     reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                         // Cast is safe because subclass does not add any selectors.
                                         reportHandler(static_cast<NSNumber *>(value), error);
                                     }];
}
+ (void)readAttributeMaxFrequencyWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                  completionHandler:
                                      (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeMaxFrequencyWithClusterStateCache:attributeCacheContainer.realContainer
                                                endpoint:endpoint
                                                   queue:queue
                                              completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                  // Cast is safe because subclass does not add any selectors.
                                                  completionHandler(static_cast<NSNumber *>(value), error);
                                              }];
}

- (void)readAttributeOptionsWithCompletionHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeOptionsWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)writeAttributeOptionsWithValue:(NSNumber * _Nonnull)value completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeOptionsWithValue:value params:nil completion:completionHandler];
}
- (void)writeAttributeOptionsWithValue:(NSNumber * _Nonnull)value
                                params:(MTRWriteParams * _Nullable)params
                     completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeOptionsWithValue:value params:params completion:completionHandler];
}
- (void)subscribeAttributeOptionsWithMinInterval:(NSNumber * _Nonnull)minInterval
                                     maxInterval:(NSNumber * _Nonnull)maxInterval
                                          params:(MTRSubscribeParams * _Nullable)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeOptionsWithParams:subscribeParams
                      subscriptionEstablished:subscriptionEstablishedHandler
                                reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                    // Cast is safe because subclass does not add any selectors.
                                    reportHandler(static_cast<NSNumber *>(value), error);
                                }];
}
+ (void)readAttributeOptionsWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                      endpoint:(NSNumber *)endpoint
                                         queue:(dispatch_queue_t)queue
                             completionHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeOptionsWithClusterStateCache:attributeCacheContainer.realContainer
                                           endpoint:endpoint
                                              queue:queue
                                         completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                             // Cast is safe because subclass does not add any selectors.
                                             completionHandler(static_cast<NSNumber *>(value), error);
                                         }];
}

- (void)readAttributeOnOffTransitionTimeWithCompletionHandler:(void (^)(NSNumber * _Nullable value,
                                                                  NSError * _Nullable error))completionHandler
{
    [self readAttributeOnOffTransitionTimeWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)writeAttributeOnOffTransitionTimeWithValue:(NSNumber * _Nonnull)value
                                 completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeOnOffTransitionTimeWithValue:value params:nil completion:completionHandler];
}
- (void)writeAttributeOnOffTransitionTimeWithValue:(NSNumber * _Nonnull)value
                                            params:(MTRWriteParams * _Nullable)params
                                 completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeOnOffTransitionTimeWithValue:value params:params completion:completionHandler];
}
- (void)subscribeAttributeOnOffTransitionTimeWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                 maxInterval:(NSNumber * _Nonnull)maxInterval
                                                      params:(MTRSubscribeParams * _Nullable)params
                                     subscriptionEstablished:
                                         (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                               reportHandler:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeOnOffTransitionTimeWithParams:subscribeParams
                                  subscriptionEstablished:subscriptionEstablishedHandler
                                            reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                reportHandler(static_cast<NSNumber *>(value), error);
                                            }];
}
+ (void)readAttributeOnOffTransitionTimeWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                         completionHandler:
                                             (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeOnOffTransitionTimeWithClusterStateCache:attributeCacheContainer.realContainer
                                                       endpoint:endpoint
                                                          queue:queue
                                                     completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                         // Cast is safe because subclass does not add any selectors.
                                                         completionHandler(static_cast<NSNumber *>(value), error);
                                                     }];
}

- (void)readAttributeOnLevelWithCompletionHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeOnLevelWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)writeAttributeOnLevelWithValue:(NSNumber * _Nullable)value completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeOnLevelWithValue:value params:nil completion:completionHandler];
}
- (void)writeAttributeOnLevelWithValue:(NSNumber * _Nullable)value
                                params:(MTRWriteParams * _Nullable)params
                     completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeOnLevelWithValue:value params:params completion:completionHandler];
}
- (void)subscribeAttributeOnLevelWithMinInterval:(NSNumber * _Nonnull)minInterval
                                     maxInterval:(NSNumber * _Nonnull)maxInterval
                                          params:(MTRSubscribeParams * _Nullable)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeOnLevelWithParams:subscribeParams
                      subscriptionEstablished:subscriptionEstablishedHandler
                                reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                    // Cast is safe because subclass does not add any selectors.
                                    reportHandler(static_cast<NSNumber *>(value), error);
                                }];
}
+ (void)readAttributeOnLevelWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                      endpoint:(NSNumber *)endpoint
                                         queue:(dispatch_queue_t)queue
                             completionHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeOnLevelWithClusterStateCache:attributeCacheContainer.realContainer
                                           endpoint:endpoint
                                              queue:queue
                                         completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                             // Cast is safe because subclass does not add any selectors.
                                             completionHandler(static_cast<NSNumber *>(value), error);
                                         }];
}

- (void)readAttributeOnTransitionTimeWithCompletionHandler:(void (^)(NSNumber * _Nullable value,
                                                               NSError * _Nullable error))completionHandler
{
    [self readAttributeOnTransitionTimeWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)writeAttributeOnTransitionTimeWithValue:(NSNumber * _Nullable)value completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeOnTransitionTimeWithValue:value params:nil completion:completionHandler];
}
- (void)writeAttributeOnTransitionTimeWithValue:(NSNumber * _Nullable)value
                                         params:(MTRWriteParams * _Nullable)params
                              completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeOnTransitionTimeWithValue:value params:params completion:completionHandler];
}
- (void)
    subscribeAttributeOnTransitionTimeWithMinInterval:(NSNumber * _Nonnull)minInterval
                                          maxInterval:(NSNumber * _Nonnull)maxInterval
                                               params:(MTRSubscribeParams * _Nullable)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeOnTransitionTimeWithParams:subscribeParams
                               subscriptionEstablished:subscriptionEstablishedHandler
                                         reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                             // Cast is safe because subclass does not add any selectors.
                                             reportHandler(static_cast<NSNumber *>(value), error);
                                         }];
}
+ (void)readAttributeOnTransitionTimeWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                      completionHandler:
                                          (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeOnTransitionTimeWithClusterStateCache:attributeCacheContainer.realContainer
                                                    endpoint:endpoint
                                                       queue:queue
                                                  completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                      // Cast is safe because subclass does not add any selectors.
                                                      completionHandler(static_cast<NSNumber *>(value), error);
                                                  }];
}

- (void)readAttributeOffTransitionTimeWithCompletionHandler:(void (^)(NSNumber * _Nullable value,
                                                                NSError * _Nullable error))completionHandler
{
    [self readAttributeOffTransitionTimeWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)writeAttributeOffTransitionTimeWithValue:(NSNumber * _Nullable)value
                               completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeOffTransitionTimeWithValue:value params:nil completion:completionHandler];
}
- (void)writeAttributeOffTransitionTimeWithValue:(NSNumber * _Nullable)value
                                          params:(MTRWriteParams * _Nullable)params
                               completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeOffTransitionTimeWithValue:value params:params completion:completionHandler];
}
- (void)subscribeAttributeOffTransitionTimeWithMinInterval:(NSNumber * _Nonnull)minInterval
                                               maxInterval:(NSNumber * _Nonnull)maxInterval
                                                    params:(MTRSubscribeParams * _Nullable)params
                                   subscriptionEstablished:
                                       (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                             reportHandler:
                                                 (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeOffTransitionTimeWithParams:subscribeParams
                                subscriptionEstablished:subscriptionEstablishedHandler
                                          reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                              // Cast is safe because subclass does not add any selectors.
                                              reportHandler(static_cast<NSNumber *>(value), error);
                                          }];
}
+ (void)readAttributeOffTransitionTimeWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                endpoint:(NSNumber *)endpoint
                                                   queue:(dispatch_queue_t)queue
                                       completionHandler:
                                           (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeOffTransitionTimeWithClusterStateCache:attributeCacheContainer.realContainer
                                                     endpoint:endpoint
                                                        queue:queue
                                                   completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                       // Cast is safe because subclass does not add any selectors.
                                                       completionHandler(static_cast<NSNumber *>(value), error);
                                                   }];
}

- (void)readAttributeDefaultMoveRateWithCompletionHandler:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completionHandler
{
    [self readAttributeDefaultMoveRateWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)writeAttributeDefaultMoveRateWithValue:(NSNumber * _Nullable)value completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeDefaultMoveRateWithValue:value params:nil completion:completionHandler];
}
- (void)writeAttributeDefaultMoveRateWithValue:(NSNumber * _Nullable)value
                                        params:(MTRWriteParams * _Nullable)params
                             completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeDefaultMoveRateWithValue:value params:params completion:completionHandler];
}
- (void)subscribeAttributeDefaultMoveRateWithMinInterval:(NSNumber * _Nonnull)minInterval
                                             maxInterval:(NSNumber * _Nonnull)maxInterval
                                                  params:(MTRSubscribeParams * _Nullable)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                           reportHandler:
                                               (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeDefaultMoveRateWithParams:subscribeParams
                              subscriptionEstablished:subscriptionEstablishedHandler
                                        reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                            // Cast is safe because subclass does not add any selectors.
                                            reportHandler(static_cast<NSNumber *>(value), error);
                                        }];
}
+ (void)readAttributeDefaultMoveRateWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                     completionHandler:
                                         (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeDefaultMoveRateWithClusterStateCache:attributeCacheContainer.realContainer
                                                   endpoint:endpoint
                                                      queue:queue
                                                 completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                     // Cast is safe because subclass does not add any selectors.
                                                     completionHandler(static_cast<NSNumber *>(value), error);
                                                 }];
}

- (void)readAttributeStartUpCurrentLevelWithCompletionHandler:(void (^)(NSNumber * _Nullable value,
                                                                  NSError * _Nullable error))completionHandler
{
    [self readAttributeStartUpCurrentLevelWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)writeAttributeStartUpCurrentLevelWithValue:(NSNumber * _Nullable)value
                                 completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeStartUpCurrentLevelWithValue:value params:nil completion:completionHandler];
}
- (void)writeAttributeStartUpCurrentLevelWithValue:(NSNumber * _Nullable)value
                                            params:(MTRWriteParams * _Nullable)params
                                 completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeStartUpCurrentLevelWithValue:value params:params completion:completionHandler];
}
- (void)subscribeAttributeStartUpCurrentLevelWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                 maxInterval:(NSNumber * _Nonnull)maxInterval
                                                      params:(MTRSubscribeParams * _Nullable)params
                                     subscriptionEstablished:
                                         (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                               reportHandler:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeStartUpCurrentLevelWithParams:subscribeParams
                                  subscriptionEstablished:subscriptionEstablishedHandler
                                            reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                reportHandler(static_cast<NSNumber *>(value), error);
                                            }];
}
+ (void)readAttributeStartUpCurrentLevelWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                         completionHandler:
                                             (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeStartUpCurrentLevelWithClusterStateCache:attributeCacheContainer.realContainer
                                                       endpoint:endpoint
                                                          queue:queue
                                                     completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                         // Cast is safe because subclass does not add any selectors.
                                                         completionHandler(static_cast<NSNumber *>(value), error);
                                                     }];
}

- (void)readAttributeGeneratedCommandListWithCompletionHandler:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))completionHandler
{
    [self readAttributeGeneratedCommandListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeGeneratedCommandListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                  maxInterval:(NSNumber * _Nonnull)maxInterval
                                                       params:(MTRSubscribeParams * _Nullable)params
                                      subscriptionEstablished:
                                          (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                                reportHandler:
                                                    (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeGeneratedCommandListWithParams:subscribeParams
                                   subscriptionEstablished:subscriptionEstablishedHandler
                                             reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                 // Cast is safe because subclass does not add any selectors.
                                                 reportHandler(static_cast<NSArray *>(value), error);
                                             }];
}
+ (void)readAttributeGeneratedCommandListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                          completionHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeGeneratedCommandListWithClusterStateCache:attributeCacheContainer.realContainer
                                                        endpoint:endpoint
                                                           queue:queue
                                                      completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                          // Cast is safe because subclass does not add any selectors.
                                                          completionHandler(static_cast<NSArray *>(value), error);
                                                      }];
}

- (void)readAttributeAcceptedCommandListWithCompletionHandler:(void (^)(NSArray * _Nullable value,
                                                                  NSError * _Nullable error))completionHandler
{
    [self readAttributeAcceptedCommandListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeAcceptedCommandListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                 maxInterval:(NSNumber * _Nonnull)maxInterval
                                                      params:(MTRSubscribeParams * _Nullable)params
                                     subscriptionEstablished:
                                         (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                               reportHandler:
                                                   (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeAcceptedCommandListWithParams:subscribeParams
                                  subscriptionEstablished:subscriptionEstablishedHandler
                                            reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                reportHandler(static_cast<NSArray *>(value), error);
                                            }];
}
+ (void)readAttributeAcceptedCommandListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                         completionHandler:
                                             (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAcceptedCommandListWithClusterStateCache:attributeCacheContainer.realContainer
                                                       endpoint:endpoint
                                                          queue:queue
                                                     completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                         // Cast is safe because subclass does not add any selectors.
                                                         completionHandler(static_cast<NSArray *>(value), error);
                                                     }];
}

- (void)readAttributeAttributeListWithCompletionHandler:(void (^)(
                                                            NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAttributeListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeAttributeListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                           maxInterval:(NSNumber * _Nonnull)maxInterval
                                                params:(MTRSubscribeParams * _Nullable)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                         reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeAttributeListWithParams:subscribeParams
                            subscriptionEstablished:subscriptionEstablishedHandler
                                      reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                          // Cast is safe because subclass does not add any selectors.
                                          reportHandler(static_cast<NSArray *>(value), error);
                                      }];
}
+ (void)readAttributeAttributeListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                   completionHandler:
                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAttributeListWithClusterStateCache:attributeCacheContainer.realContainer
                                                 endpoint:endpoint
                                                    queue:queue
                                               completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                   // Cast is safe because subclass does not add any selectors.
                                                   completionHandler(static_cast<NSArray *>(value), error);
                                               }];
}

- (void)readAttributeFeatureMapWithCompletionHandler:(void (^)(
                                                         NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeFeatureMapWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeFeatureMapWithMinInterval:(NSNumber * _Nonnull)minInterval
                                        maxInterval:(NSNumber * _Nonnull)maxInterval
                                             params:(MTRSubscribeParams * _Nullable)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeFeatureMapWithParams:subscribeParams
                         subscriptionEstablished:subscriptionEstablishedHandler
                                   reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                       // Cast is safe because subclass does not add any selectors.
                                       reportHandler(static_cast<NSNumber *>(value), error);
                                   }];
}
+ (void)readAttributeFeatureMapWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                completionHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeFeatureMapWithClusterStateCache:attributeCacheContainer.realContainer
                                              endpoint:endpoint
                                                 queue:queue
                                            completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                completionHandler(static_cast<NSNumber *>(value), error);
                                            }];
}

- (void)readAttributeClusterRevisionWithCompletionHandler:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completionHandler
{
    [self readAttributeClusterRevisionWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeClusterRevisionWithMinInterval:(NSNumber * _Nonnull)minInterval
                                             maxInterval:(NSNumber * _Nonnull)maxInterval
                                                  params:(MTRSubscribeParams * _Nullable)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                           reportHandler:
                                               (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeClusterRevisionWithParams:subscribeParams
                              subscriptionEstablished:subscriptionEstablishedHandler
                                        reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                            // Cast is safe because subclass does not add any selectors.
                                            reportHandler(static_cast<NSNumber *>(value), error);
                                        }];
}
+ (void)readAttributeClusterRevisionWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                     completionHandler:
                                         (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeClusterRevisionWithClusterStateCache:attributeCacheContainer.realContainer
                                                   endpoint:endpoint
                                                      queue:queue
                                                 completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                     // Cast is safe because subclass does not add any selectors.
                                                     completionHandler(static_cast<NSNumber *>(value), error);
                                                 }];
}

- (nullable instancetype)initWithDevice:(MTRBaseDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

@end

@implementation MTRBaseClusterBinaryInputBasic

- (instancetype)initWithDevice:(MTRBaseDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _device = device;
        _endpoint = [endpointID unsignedShortValue];
    }
    return self;
}

- (void)readAttributeActiveTextWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BinaryInputBasic::Attributes::ActiveText::TypeInfo;
    return MTRReadAttribute<MTRCharStringAttributeCallbackBridge, NSString, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)writeAttributeActiveTextWithValue:(NSString * _Nonnull)value completion:(MTRStatusCompletion)completion
{
    [self writeAttributeActiveTextWithValue:(NSString * _Nonnull) value params:nil completion:completion];
}
- (void)writeAttributeActiveTextWithValue:(NSString * _Nonnull)value
                                   params:(MTRWriteParams * _Nullable)params
                               completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    value = [value copy];

    auto * bridge = new MTRDefaultSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable ignored, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, DefaultSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            chip::Optional<uint16_t> timedWriteTimeout;
            if (params != nil) {
                if (params.timedWriteTimeout != nil) {
                    timedWriteTimeout.SetValue(params.timedWriteTimeout.unsignedShortValue);
                }
            }

            ListFreer listFreer;
            using TypeInfo = BinaryInputBasic::Attributes::ActiveText::TypeInfo;
            TypeInfo::Type cppValue;
            cppValue = [self asCharSpan:value];

            chip::Controller::ClusterBase cppCluster(exchangeManager, session, self->_endpoint);
            return cppCluster.WriteAttribute<TypeInfo>(cppValue, bridge, successCb, failureCb, timedWriteTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)subscribeAttributeActiveTextWithParams:(MTRSubscribeParams * _Nonnull)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BinaryInputBasic::Attributes::ActiveText::TypeInfo;
    MTRSubscribeAttribute<MTRCharStringAttributeCallbackSubscriptionBridge, NSString, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeActiveTextWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRCharStringAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(CharStringAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BinaryInputBasic::Attributes::ActiveText::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeDescriptionWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BinaryInputBasic::Attributes::Description::TypeInfo;
    return MTRReadAttribute<MTRCharStringAttributeCallbackBridge, NSString, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)writeAttributeDescriptionWithValue:(NSString * _Nonnull)value completion:(MTRStatusCompletion)completion
{
    [self writeAttributeDescriptionWithValue:(NSString * _Nonnull) value params:nil completion:completion];
}
- (void)writeAttributeDescriptionWithValue:(NSString * _Nonnull)value
                                    params:(MTRWriteParams * _Nullable)params
                                completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    value = [value copy];

    auto * bridge = new MTRDefaultSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable ignored, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, DefaultSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            chip::Optional<uint16_t> timedWriteTimeout;
            if (params != nil) {
                if (params.timedWriteTimeout != nil) {
                    timedWriteTimeout.SetValue(params.timedWriteTimeout.unsignedShortValue);
                }
            }

            ListFreer listFreer;
            using TypeInfo = BinaryInputBasic::Attributes::Description::TypeInfo;
            TypeInfo::Type cppValue;
            cppValue = [self asCharSpan:value];

            chip::Controller::ClusterBase cppCluster(exchangeManager, session, self->_endpoint);
            return cppCluster.WriteAttribute<TypeInfo>(cppValue, bridge, successCb, failureCb, timedWriteTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)subscribeAttributeDescriptionWithParams:(MTRSubscribeParams * _Nonnull)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BinaryInputBasic::Attributes::Description::TypeInfo;
    MTRSubscribeAttribute<MTRCharStringAttributeCallbackSubscriptionBridge, NSString, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeDescriptionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRCharStringAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(CharStringAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BinaryInputBasic::Attributes::Description::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeInactiveTextWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BinaryInputBasic::Attributes::InactiveText::TypeInfo;
    return MTRReadAttribute<MTRCharStringAttributeCallbackBridge, NSString, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)writeAttributeInactiveTextWithValue:(NSString * _Nonnull)value completion:(MTRStatusCompletion)completion
{
    [self writeAttributeInactiveTextWithValue:(NSString * _Nonnull) value params:nil completion:completion];
}
- (void)writeAttributeInactiveTextWithValue:(NSString * _Nonnull)value
                                     params:(MTRWriteParams * _Nullable)params
                                 completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    value = [value copy];

    auto * bridge = new MTRDefaultSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable ignored, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, DefaultSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            chip::Optional<uint16_t> timedWriteTimeout;
            if (params != nil) {
                if (params.timedWriteTimeout != nil) {
                    timedWriteTimeout.SetValue(params.timedWriteTimeout.unsignedShortValue);
                }
            }

            ListFreer listFreer;
            using TypeInfo = BinaryInputBasic::Attributes::InactiveText::TypeInfo;
            TypeInfo::Type cppValue;
            cppValue = [self asCharSpan:value];

            chip::Controller::ClusterBase cppCluster(exchangeManager, session, self->_endpoint);
            return cppCluster.WriteAttribute<TypeInfo>(cppValue, bridge, successCb, failureCb, timedWriteTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)subscribeAttributeInactiveTextWithParams:(MTRSubscribeParams * _Nonnull)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BinaryInputBasic::Attributes::InactiveText::TypeInfo;
    MTRSubscribeAttribute<MTRCharStringAttributeCallbackSubscriptionBridge, NSString, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeInactiveTextWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRCharStringAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(CharStringAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BinaryInputBasic::Attributes::InactiveText::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeOutOfServiceWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BinaryInputBasic::Attributes::OutOfService::TypeInfo;
    return MTRReadAttribute<MTRBooleanAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)writeAttributeOutOfServiceWithValue:(NSNumber * _Nonnull)value completion:(MTRStatusCompletion)completion
{
    [self writeAttributeOutOfServiceWithValue:(NSNumber * _Nonnull) value params:nil completion:completion];
}
- (void)writeAttributeOutOfServiceWithValue:(NSNumber * _Nonnull)value
                                     params:(MTRWriteParams * _Nullable)params
                                 completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    value = [value copy];

    auto * bridge = new MTRDefaultSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable ignored, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, DefaultSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            chip::Optional<uint16_t> timedWriteTimeout;
            if (params != nil) {
                if (params.timedWriteTimeout != nil) {
                    timedWriteTimeout.SetValue(params.timedWriteTimeout.unsignedShortValue);
                }
            }

            ListFreer listFreer;
            using TypeInfo = BinaryInputBasic::Attributes::OutOfService::TypeInfo;
            TypeInfo::Type cppValue;
            cppValue = value.boolValue;

            chip::Controller::ClusterBase cppCluster(exchangeManager, session, self->_endpoint);
            return cppCluster.WriteAttribute<TypeInfo>(cppValue, bridge, successCb, failureCb, timedWriteTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)subscribeAttributeOutOfServiceWithParams:(MTRSubscribeParams * _Nonnull)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BinaryInputBasic::Attributes::OutOfService::TypeInfo;
    MTRSubscribeAttribute<MTRBooleanAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeOutOfServiceWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRBooleanAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(BooleanAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BinaryInputBasic::Attributes::OutOfService::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributePolarityWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BinaryInputBasic::Attributes::Polarity::TypeInfo;
    return MTRReadAttribute<MTRInt8uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributePolarityWithParams:(MTRSubscribeParams * _Nonnull)params
                     subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                               reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BinaryInputBasic::Attributes::Polarity::TypeInfo;
    MTRSubscribeAttribute<MTRInt8uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributePolarityWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                          endpoint:(NSNumber *)endpoint
                                             queue:(dispatch_queue_t)queue
                                        completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt8uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int8uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BinaryInputBasic::Attributes::Polarity::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributePresentValueWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BinaryInputBasic::Attributes::PresentValue::TypeInfo;
    return MTRReadAttribute<MTRBooleanAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)writeAttributePresentValueWithValue:(NSNumber * _Nonnull)value completion:(MTRStatusCompletion)completion
{
    [self writeAttributePresentValueWithValue:(NSNumber * _Nonnull) value params:nil completion:completion];
}
- (void)writeAttributePresentValueWithValue:(NSNumber * _Nonnull)value
                                     params:(MTRWriteParams * _Nullable)params
                                 completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    value = [value copy];

    auto * bridge = new MTRDefaultSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable ignored, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, DefaultSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            chip::Optional<uint16_t> timedWriteTimeout;
            if (params != nil) {
                if (params.timedWriteTimeout != nil) {
                    timedWriteTimeout.SetValue(params.timedWriteTimeout.unsignedShortValue);
                }
            }

            ListFreer listFreer;
            using TypeInfo = BinaryInputBasic::Attributes::PresentValue::TypeInfo;
            TypeInfo::Type cppValue;
            cppValue = value.boolValue;

            chip::Controller::ClusterBase cppCluster(exchangeManager, session, self->_endpoint);
            return cppCluster.WriteAttribute<TypeInfo>(cppValue, bridge, successCb, failureCb, timedWriteTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)subscribeAttributePresentValueWithParams:(MTRSubscribeParams * _Nonnull)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BinaryInputBasic::Attributes::PresentValue::TypeInfo;
    MTRSubscribeAttribute<MTRBooleanAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributePresentValueWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRBooleanAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(BooleanAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BinaryInputBasic::Attributes::PresentValue::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeReliabilityWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BinaryInputBasic::Attributes::Reliability::TypeInfo;
    return MTRReadAttribute<MTRInt8uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)writeAttributeReliabilityWithValue:(NSNumber * _Nonnull)value completion:(MTRStatusCompletion)completion
{
    [self writeAttributeReliabilityWithValue:(NSNumber * _Nonnull) value params:nil completion:completion];
}
- (void)writeAttributeReliabilityWithValue:(NSNumber * _Nonnull)value
                                    params:(MTRWriteParams * _Nullable)params
                                completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    value = [value copy];

    auto * bridge = new MTRDefaultSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable ignored, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, DefaultSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            chip::Optional<uint16_t> timedWriteTimeout;
            if (params != nil) {
                if (params.timedWriteTimeout != nil) {
                    timedWriteTimeout.SetValue(params.timedWriteTimeout.unsignedShortValue);
                }
            }

            ListFreer listFreer;
            using TypeInfo = BinaryInputBasic::Attributes::Reliability::TypeInfo;
            TypeInfo::Type cppValue;
            cppValue = value.unsignedCharValue;

            chip::Controller::ClusterBase cppCluster(exchangeManager, session, self->_endpoint);
            return cppCluster.WriteAttribute<TypeInfo>(cppValue, bridge, successCb, failureCb, timedWriteTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)subscribeAttributeReliabilityWithParams:(MTRSubscribeParams * _Nonnull)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BinaryInputBasic::Attributes::Reliability::TypeInfo;
    MTRSubscribeAttribute<MTRInt8uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeReliabilityWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt8uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int8uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BinaryInputBasic::Attributes::Reliability::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeStatusFlagsWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BinaryInputBasic::Attributes::StatusFlags::TypeInfo;
    return MTRReadAttribute<MTRInt8uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeStatusFlagsWithParams:(MTRSubscribeParams * _Nonnull)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BinaryInputBasic::Attributes::StatusFlags::TypeInfo;
    MTRSubscribeAttribute<MTRInt8uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeStatusFlagsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt8uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int8uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BinaryInputBasic::Attributes::StatusFlags::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeApplicationTypeWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BinaryInputBasic::Attributes::ApplicationType::TypeInfo;
    return MTRReadAttribute<MTRInt32uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeApplicationTypeWithParams:(MTRSubscribeParams * _Nonnull)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BinaryInputBasic::Attributes::ApplicationType::TypeInfo;
    MTRSubscribeAttribute<MTRInt32uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeApplicationTypeWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt32uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int32uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BinaryInputBasic::Attributes::ApplicationType::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BinaryInputBasic::Attributes::GeneratedCommandList::TypeInfo;
    return MTRReadAttribute<MTRBinaryInputBasicGeneratedCommandListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams * _Nonnull)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BinaryInputBasic::Attributes::GeneratedCommandList::TypeInfo;
    MTRSubscribeAttribute<MTRBinaryInputBasicGeneratedCommandListListAttributeCallbackSubscriptionBridge, NSArray,
        TypeInfo::DecodableType>(params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint,
        TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRBinaryInputBasicGeneratedCommandListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(BinaryInputBasicGeneratedCommandListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BinaryInputBasic::Attributes::GeneratedCommandList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BinaryInputBasic::Attributes::AcceptedCommandList::TypeInfo;
    return MTRReadAttribute<MTRBinaryInputBasicAcceptedCommandListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams * _Nonnull)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BinaryInputBasic::Attributes::AcceptedCommandList::TypeInfo;
    MTRSubscribeAttribute<MTRBinaryInputBasicAcceptedCommandListListAttributeCallbackSubscriptionBridge, NSArray,
        TypeInfo::DecodableType>(params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint,
        TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRBinaryInputBasicAcceptedCommandListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(BinaryInputBasicAcceptedCommandListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BinaryInputBasic::Attributes::AcceptedCommandList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BinaryInputBasic::Attributes::EventList::TypeInfo;
    return MTRReadAttribute<MTRBinaryInputBasicEventListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams * _Nonnull)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BinaryInputBasic::Attributes::EventList::TypeInfo;
    MTRSubscribeAttribute<MTRBinaryInputBasicEventListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(
        params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRBinaryInputBasicEventListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(BinaryInputBasicEventListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BinaryInputBasic::Attributes::EventList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BinaryInputBasic::Attributes::AttributeList::TypeInfo;
    return MTRReadAttribute<MTRBinaryInputBasicAttributeListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams * _Nonnull)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BinaryInputBasic::Attributes::AttributeList::TypeInfo;
    MTRSubscribeAttribute<MTRBinaryInputBasicAttributeListListAttributeCallbackSubscriptionBridge, NSArray,
        TypeInfo::DecodableType>(params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint,
        TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRBinaryInputBasicAttributeListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(BinaryInputBasicAttributeListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BinaryInputBasic::Attributes::AttributeList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BinaryInputBasic::Attributes::FeatureMap::TypeInfo;
    return MTRReadAttribute<MTRInt32uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams * _Nonnull)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BinaryInputBasic::Attributes::FeatureMap::TypeInfo;
    MTRSubscribeAttribute<MTRInt32uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt32uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int32uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BinaryInputBasic::Attributes::FeatureMap::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BinaryInputBasic::Attributes::ClusterRevision::TypeInfo;
    return MTRReadAttribute<MTRInt16uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams * _Nonnull)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BinaryInputBasic::Attributes::ClusterRevision::TypeInfo;
    MTRSubscribeAttribute<MTRInt16uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt16uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int16uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BinaryInputBasic::Attributes::ClusterRevision::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

@end

@implementation MTRBaseClusterBinaryInputBasic (Deprecated)

- (void)readAttributeActiveTextWithCompletionHandler:(void (^)(
                                                         NSString * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeActiveTextWithCompletion:^(NSString * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSString *>(value), error);
    }];
}
- (void)writeAttributeActiveTextWithValue:(NSString * _Nonnull)value completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeActiveTextWithValue:value params:nil completion:completionHandler];
}
- (void)writeAttributeActiveTextWithValue:(NSString * _Nonnull)value
                                   params:(MTRWriteParams * _Nullable)params
                        completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeActiveTextWithValue:value params:params completion:completionHandler];
}
- (void)subscribeAttributeActiveTextWithMinInterval:(NSNumber * _Nonnull)minInterval
                                        maxInterval:(NSNumber * _Nonnull)maxInterval
                                             params:(MTRSubscribeParams * _Nullable)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                      reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeActiveTextWithParams:subscribeParams
                         subscriptionEstablished:subscriptionEstablishedHandler
                                   reportHandler:^(NSString * _Nullable value, NSError * _Nullable error) {
                                       // Cast is safe because subclass does not add any selectors.
                                       reportHandler(static_cast<NSString *>(value), error);
                                   }];
}
+ (void)readAttributeActiveTextWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                completionHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeActiveTextWithClusterStateCache:attributeCacheContainer.realContainer
                                              endpoint:endpoint
                                                 queue:queue
                                            completion:^(NSString * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                completionHandler(static_cast<NSString *>(value), error);
                                            }];
}

- (void)readAttributeDescriptionWithCompletionHandler:(void (^)(
                                                          NSString * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeDescriptionWithCompletion:^(NSString * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSString *>(value), error);
    }];
}
- (void)writeAttributeDescriptionWithValue:(NSString * _Nonnull)value completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeDescriptionWithValue:value params:nil completion:completionHandler];
}
- (void)writeAttributeDescriptionWithValue:(NSString * _Nonnull)value
                                    params:(MTRWriteParams * _Nullable)params
                         completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeDescriptionWithValue:value params:params completion:completionHandler];
}
- (void)subscribeAttributeDescriptionWithMinInterval:(NSNumber * _Nonnull)minInterval
                                         maxInterval:(NSNumber * _Nonnull)maxInterval
                                              params:(MTRSubscribeParams * _Nullable)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                       reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeDescriptionWithParams:subscribeParams
                          subscriptionEstablished:subscriptionEstablishedHandler
                                    reportHandler:^(NSString * _Nullable value, NSError * _Nullable error) {
                                        // Cast is safe because subclass does not add any selectors.
                                        reportHandler(static_cast<NSString *>(value), error);
                                    }];
}
+ (void)readAttributeDescriptionWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                          endpoint:(NSNumber *)endpoint
                                             queue:(dispatch_queue_t)queue
                                 completionHandler:
                                     (void (^)(NSString * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeDescriptionWithClusterStateCache:attributeCacheContainer.realContainer
                                               endpoint:endpoint
                                                  queue:queue
                                             completion:^(NSString * _Nullable value, NSError * _Nullable error) {
                                                 // Cast is safe because subclass does not add any selectors.
                                                 completionHandler(static_cast<NSString *>(value), error);
                                             }];
}

- (void)readAttributeInactiveTextWithCompletionHandler:(void (^)(
                                                           NSString * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeInactiveTextWithCompletion:^(NSString * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSString *>(value), error);
    }];
}
- (void)writeAttributeInactiveTextWithValue:(NSString * _Nonnull)value completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeInactiveTextWithValue:value params:nil completion:completionHandler];
}
- (void)writeAttributeInactiveTextWithValue:(NSString * _Nonnull)value
                                     params:(MTRWriteParams * _Nullable)params
                          completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeInactiveTextWithValue:value params:params completion:completionHandler];
}
- (void)subscribeAttributeInactiveTextWithMinInterval:(NSNumber * _Nonnull)minInterval
                                          maxInterval:(NSNumber * _Nonnull)maxInterval
                                               params:(MTRSubscribeParams * _Nullable)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                        reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeInactiveTextWithParams:subscribeParams
                           subscriptionEstablished:subscriptionEstablishedHandler
                                     reportHandler:^(NSString * _Nullable value, NSError * _Nullable error) {
                                         // Cast is safe because subclass does not add any selectors.
                                         reportHandler(static_cast<NSString *>(value), error);
                                     }];
}
+ (void)readAttributeInactiveTextWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                  completionHandler:
                                      (void (^)(NSString * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeInactiveTextWithClusterStateCache:attributeCacheContainer.realContainer
                                                endpoint:endpoint
                                                   queue:queue
                                              completion:^(NSString * _Nullable value, NSError * _Nullable error) {
                                                  // Cast is safe because subclass does not add any selectors.
                                                  completionHandler(static_cast<NSString *>(value), error);
                                              }];
}

- (void)readAttributeOutOfServiceWithCompletionHandler:(void (^)(
                                                           NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeOutOfServiceWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)writeAttributeOutOfServiceWithValue:(NSNumber * _Nonnull)value completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeOutOfServiceWithValue:value params:nil completion:completionHandler];
}
- (void)writeAttributeOutOfServiceWithValue:(NSNumber * _Nonnull)value
                                     params:(MTRWriteParams * _Nullable)params
                          completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeOutOfServiceWithValue:value params:params completion:completionHandler];
}
- (void)subscribeAttributeOutOfServiceWithMinInterval:(NSNumber * _Nonnull)minInterval
                                          maxInterval:(NSNumber * _Nonnull)maxInterval
                                               params:(MTRSubscribeParams * _Nullable)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeOutOfServiceWithParams:subscribeParams
                           subscriptionEstablished:subscriptionEstablishedHandler
                                     reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                         // Cast is safe because subclass does not add any selectors.
                                         reportHandler(static_cast<NSNumber *>(value), error);
                                     }];
}
+ (void)readAttributeOutOfServiceWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                  completionHandler:
                                      (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeOutOfServiceWithClusterStateCache:attributeCacheContainer.realContainer
                                                endpoint:endpoint
                                                   queue:queue
                                              completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                  // Cast is safe because subclass does not add any selectors.
                                                  completionHandler(static_cast<NSNumber *>(value), error);
                                              }];
}

- (void)readAttributePolarityWithCompletionHandler:(void (^)(
                                                       NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributePolarityWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributePolarityWithMinInterval:(NSNumber * _Nonnull)minInterval
                                      maxInterval:(NSNumber * _Nonnull)maxInterval
                                           params:(MTRSubscribeParams * _Nullable)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                    reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributePolarityWithParams:subscribeParams
                       subscriptionEstablished:subscriptionEstablishedHandler
                                 reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                     // Cast is safe because subclass does not add any selectors.
                                     reportHandler(static_cast<NSNumber *>(value), error);
                                 }];
}
+ (void)readAttributePolarityWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                       endpoint:(NSNumber *)endpoint
                                          queue:(dispatch_queue_t)queue
                              completionHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributePolarityWithClusterStateCache:attributeCacheContainer.realContainer
                                            endpoint:endpoint
                                               queue:queue
                                          completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                              // Cast is safe because subclass does not add any selectors.
                                              completionHandler(static_cast<NSNumber *>(value), error);
                                          }];
}

- (void)readAttributePresentValueWithCompletionHandler:(void (^)(
                                                           NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributePresentValueWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)writeAttributePresentValueWithValue:(NSNumber * _Nonnull)value completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributePresentValueWithValue:value params:nil completion:completionHandler];
}
- (void)writeAttributePresentValueWithValue:(NSNumber * _Nonnull)value
                                     params:(MTRWriteParams * _Nullable)params
                          completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributePresentValueWithValue:value params:params completion:completionHandler];
}
- (void)subscribeAttributePresentValueWithMinInterval:(NSNumber * _Nonnull)minInterval
                                          maxInterval:(NSNumber * _Nonnull)maxInterval
                                               params:(MTRSubscribeParams * _Nullable)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributePresentValueWithParams:subscribeParams
                           subscriptionEstablished:subscriptionEstablishedHandler
                                     reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                         // Cast is safe because subclass does not add any selectors.
                                         reportHandler(static_cast<NSNumber *>(value), error);
                                     }];
}
+ (void)readAttributePresentValueWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                  completionHandler:
                                      (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributePresentValueWithClusterStateCache:attributeCacheContainer.realContainer
                                                endpoint:endpoint
                                                   queue:queue
                                              completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                  // Cast is safe because subclass does not add any selectors.
                                                  completionHandler(static_cast<NSNumber *>(value), error);
                                              }];
}

- (void)readAttributeReliabilityWithCompletionHandler:(void (^)(
                                                          NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeReliabilityWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)writeAttributeReliabilityWithValue:(NSNumber * _Nonnull)value completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeReliabilityWithValue:value params:nil completion:completionHandler];
}
- (void)writeAttributeReliabilityWithValue:(NSNumber * _Nonnull)value
                                    params:(MTRWriteParams * _Nullable)params
                         completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeReliabilityWithValue:value params:params completion:completionHandler];
}
- (void)subscribeAttributeReliabilityWithMinInterval:(NSNumber * _Nonnull)minInterval
                                         maxInterval:(NSNumber * _Nonnull)maxInterval
                                              params:(MTRSubscribeParams * _Nullable)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeReliabilityWithParams:subscribeParams
                          subscriptionEstablished:subscriptionEstablishedHandler
                                    reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                        // Cast is safe because subclass does not add any selectors.
                                        reportHandler(static_cast<NSNumber *>(value), error);
                                    }];
}
+ (void)readAttributeReliabilityWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                          endpoint:(NSNumber *)endpoint
                                             queue:(dispatch_queue_t)queue
                                 completionHandler:
                                     (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeReliabilityWithClusterStateCache:attributeCacheContainer.realContainer
                                               endpoint:endpoint
                                                  queue:queue
                                             completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                 // Cast is safe because subclass does not add any selectors.
                                                 completionHandler(static_cast<NSNumber *>(value), error);
                                             }];
}

- (void)readAttributeStatusFlagsWithCompletionHandler:(void (^)(
                                                          NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeStatusFlagsWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeStatusFlagsWithMinInterval:(NSNumber * _Nonnull)minInterval
                                         maxInterval:(NSNumber * _Nonnull)maxInterval
                                              params:(MTRSubscribeParams * _Nullable)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                       reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeStatusFlagsWithParams:subscribeParams
                          subscriptionEstablished:subscriptionEstablishedHandler
                                    reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                        // Cast is safe because subclass does not add any selectors.
                                        reportHandler(static_cast<NSNumber *>(value), error);
                                    }];
}
+ (void)readAttributeStatusFlagsWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                          endpoint:(NSNumber *)endpoint
                                             queue:(dispatch_queue_t)queue
                                 completionHandler:
                                     (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeStatusFlagsWithClusterStateCache:attributeCacheContainer.realContainer
                                               endpoint:endpoint
                                                  queue:queue
                                             completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                 // Cast is safe because subclass does not add any selectors.
                                                 completionHandler(static_cast<NSNumber *>(value), error);
                                             }];
}

- (void)readAttributeApplicationTypeWithCompletionHandler:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completionHandler
{
    [self readAttributeApplicationTypeWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeApplicationTypeWithMinInterval:(NSNumber * _Nonnull)minInterval
                                             maxInterval:(NSNumber * _Nonnull)maxInterval
                                                  params:(MTRSubscribeParams * _Nullable)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                           reportHandler:
                                               (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeApplicationTypeWithParams:subscribeParams
                              subscriptionEstablished:subscriptionEstablishedHandler
                                        reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                            // Cast is safe because subclass does not add any selectors.
                                            reportHandler(static_cast<NSNumber *>(value), error);
                                        }];
}
+ (void)readAttributeApplicationTypeWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                     completionHandler:
                                         (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeApplicationTypeWithClusterStateCache:attributeCacheContainer.realContainer
                                                   endpoint:endpoint
                                                      queue:queue
                                                 completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                     // Cast is safe because subclass does not add any selectors.
                                                     completionHandler(static_cast<NSNumber *>(value), error);
                                                 }];
}

- (void)readAttributeGeneratedCommandListWithCompletionHandler:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))completionHandler
{
    [self readAttributeGeneratedCommandListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeGeneratedCommandListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                  maxInterval:(NSNumber * _Nonnull)maxInterval
                                                       params:(MTRSubscribeParams * _Nullable)params
                                      subscriptionEstablished:
                                          (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                                reportHandler:
                                                    (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeGeneratedCommandListWithParams:subscribeParams
                                   subscriptionEstablished:subscriptionEstablishedHandler
                                             reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                 // Cast is safe because subclass does not add any selectors.
                                                 reportHandler(static_cast<NSArray *>(value), error);
                                             }];
}
+ (void)readAttributeGeneratedCommandListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                          completionHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeGeneratedCommandListWithClusterStateCache:attributeCacheContainer.realContainer
                                                        endpoint:endpoint
                                                           queue:queue
                                                      completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                          // Cast is safe because subclass does not add any selectors.
                                                          completionHandler(static_cast<NSArray *>(value), error);
                                                      }];
}

- (void)readAttributeAcceptedCommandListWithCompletionHandler:(void (^)(NSArray * _Nullable value,
                                                                  NSError * _Nullable error))completionHandler
{
    [self readAttributeAcceptedCommandListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeAcceptedCommandListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                 maxInterval:(NSNumber * _Nonnull)maxInterval
                                                      params:(MTRSubscribeParams * _Nullable)params
                                     subscriptionEstablished:
                                         (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                               reportHandler:
                                                   (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeAcceptedCommandListWithParams:subscribeParams
                                  subscriptionEstablished:subscriptionEstablishedHandler
                                            reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                reportHandler(static_cast<NSArray *>(value), error);
                                            }];
}
+ (void)readAttributeAcceptedCommandListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                         completionHandler:
                                             (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAcceptedCommandListWithClusterStateCache:attributeCacheContainer.realContainer
                                                       endpoint:endpoint
                                                          queue:queue
                                                     completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                         // Cast is safe because subclass does not add any selectors.
                                                         completionHandler(static_cast<NSArray *>(value), error);
                                                     }];
}

- (void)readAttributeAttributeListWithCompletionHandler:(void (^)(
                                                            NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAttributeListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeAttributeListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                           maxInterval:(NSNumber * _Nonnull)maxInterval
                                                params:(MTRSubscribeParams * _Nullable)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                         reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeAttributeListWithParams:subscribeParams
                            subscriptionEstablished:subscriptionEstablishedHandler
                                      reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                          // Cast is safe because subclass does not add any selectors.
                                          reportHandler(static_cast<NSArray *>(value), error);
                                      }];
}
+ (void)readAttributeAttributeListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                   completionHandler:
                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAttributeListWithClusterStateCache:attributeCacheContainer.realContainer
                                                 endpoint:endpoint
                                                    queue:queue
                                               completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                   // Cast is safe because subclass does not add any selectors.
                                                   completionHandler(static_cast<NSArray *>(value), error);
                                               }];
}

- (void)readAttributeFeatureMapWithCompletionHandler:(void (^)(
                                                         NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeFeatureMapWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeFeatureMapWithMinInterval:(NSNumber * _Nonnull)minInterval
                                        maxInterval:(NSNumber * _Nonnull)maxInterval
                                             params:(MTRSubscribeParams * _Nullable)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeFeatureMapWithParams:subscribeParams
                         subscriptionEstablished:subscriptionEstablishedHandler
                                   reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                       // Cast is safe because subclass does not add any selectors.
                                       reportHandler(static_cast<NSNumber *>(value), error);
                                   }];
}
+ (void)readAttributeFeatureMapWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                completionHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeFeatureMapWithClusterStateCache:attributeCacheContainer.realContainer
                                              endpoint:endpoint
                                                 queue:queue
                                            completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                completionHandler(static_cast<NSNumber *>(value), error);
                                            }];
}

- (void)readAttributeClusterRevisionWithCompletionHandler:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completionHandler
{
    [self readAttributeClusterRevisionWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeClusterRevisionWithMinInterval:(NSNumber * _Nonnull)minInterval
                                             maxInterval:(NSNumber * _Nonnull)maxInterval
                                                  params:(MTRSubscribeParams * _Nullable)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                           reportHandler:
                                               (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeClusterRevisionWithParams:subscribeParams
                              subscriptionEstablished:subscriptionEstablishedHandler
                                        reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                            // Cast is safe because subclass does not add any selectors.
                                            reportHandler(static_cast<NSNumber *>(value), error);
                                        }];
}
+ (void)readAttributeClusterRevisionWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                     completionHandler:
                                         (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeClusterRevisionWithClusterStateCache:attributeCacheContainer.realContainer
                                                   endpoint:endpoint
                                                      queue:queue
                                                 completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                     // Cast is safe because subclass does not add any selectors.
                                                     completionHandler(static_cast<NSNumber *>(value), error);
                                                 }];
}

- (nullable instancetype)initWithDevice:(MTRBaseDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

@end

@implementation MTRBaseClusterDescriptor

- (instancetype)initWithDevice:(MTRBaseDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _device = device;
        _endpoint = [endpointID unsignedShortValue];
    }
    return self;
}

- (void)readAttributeDeviceTypeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Descriptor::Attributes::DeviceTypeList::TypeInfo;
    return MTRReadAttribute<MTRDescriptorDeviceTypeListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeDeviceTypeListWithParams:(MTRSubscribeParams * _Nonnull)params
                           subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                     reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Descriptor::Attributes::DeviceTypeList::TypeInfo;
    MTRSubscribeAttribute<MTRDescriptorDeviceTypeListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(
        params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeDeviceTypeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                endpoint:(NSNumber *)endpoint
                                                   queue:(dispatch_queue_t)queue
                                              completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRDescriptorDeviceTypeListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(DescriptorDeviceTypeListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Descriptor::Attributes::DeviceTypeList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeServerListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Descriptor::Attributes::ServerList::TypeInfo;
    return MTRReadAttribute<MTRDescriptorServerListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeServerListWithParams:(MTRSubscribeParams * _Nonnull)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Descriptor::Attributes::ServerList::TypeInfo;
    MTRSubscribeAttribute<MTRDescriptorServerListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeServerListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRDescriptorServerListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(DescriptorServerListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Descriptor::Attributes::ServerList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeClientListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Descriptor::Attributes::ClientList::TypeInfo;
    return MTRReadAttribute<MTRDescriptorClientListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeClientListWithParams:(MTRSubscribeParams * _Nonnull)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Descriptor::Attributes::ClientList::TypeInfo;
    MTRSubscribeAttribute<MTRDescriptorClientListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeClientListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRDescriptorClientListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(DescriptorClientListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Descriptor::Attributes::ClientList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributePartsListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Descriptor::Attributes::PartsList::TypeInfo;
    return MTRReadAttribute<MTRDescriptorPartsListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributePartsListWithParams:(MTRSubscribeParams * _Nonnull)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Descriptor::Attributes::PartsList::TypeInfo;
    MTRSubscribeAttribute<MTRDescriptorPartsListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributePartsListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRDescriptorPartsListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(DescriptorPartsListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Descriptor::Attributes::PartsList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Descriptor::Attributes::GeneratedCommandList::TypeInfo;
    return MTRReadAttribute<MTRDescriptorGeneratedCommandListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams * _Nonnull)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Descriptor::Attributes::GeneratedCommandList::TypeInfo;
    MTRSubscribeAttribute<MTRDescriptorGeneratedCommandListListAttributeCallbackSubscriptionBridge, NSArray,
        TypeInfo::DecodableType>(params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint,
        TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRDescriptorGeneratedCommandListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(DescriptorGeneratedCommandListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Descriptor::Attributes::GeneratedCommandList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Descriptor::Attributes::AcceptedCommandList::TypeInfo;
    return MTRReadAttribute<MTRDescriptorAcceptedCommandListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams * _Nonnull)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Descriptor::Attributes::AcceptedCommandList::TypeInfo;
    MTRSubscribeAttribute<MTRDescriptorAcceptedCommandListListAttributeCallbackSubscriptionBridge, NSArray,
        TypeInfo::DecodableType>(params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint,
        TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRDescriptorAcceptedCommandListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(DescriptorAcceptedCommandListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Descriptor::Attributes::AcceptedCommandList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Descriptor::Attributes::EventList::TypeInfo;
    return MTRReadAttribute<MTRDescriptorEventListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams * _Nonnull)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Descriptor::Attributes::EventList::TypeInfo;
    MTRSubscribeAttribute<MTRDescriptorEventListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRDescriptorEventListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(DescriptorEventListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Descriptor::Attributes::EventList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Descriptor::Attributes::AttributeList::TypeInfo;
    return MTRReadAttribute<MTRDescriptorAttributeListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams * _Nonnull)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Descriptor::Attributes::AttributeList::TypeInfo;
    MTRSubscribeAttribute<MTRDescriptorAttributeListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(
        params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRDescriptorAttributeListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(DescriptorAttributeListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Descriptor::Attributes::AttributeList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Descriptor::Attributes::FeatureMap::TypeInfo;
    return MTRReadAttribute<MTRInt32uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams * _Nonnull)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Descriptor::Attributes::FeatureMap::TypeInfo;
    MTRSubscribeAttribute<MTRInt32uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt32uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int32uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Descriptor::Attributes::FeatureMap::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Descriptor::Attributes::ClusterRevision::TypeInfo;
    return MTRReadAttribute<MTRInt16uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams * _Nonnull)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Descriptor::Attributes::ClusterRevision::TypeInfo;
    MTRSubscribeAttribute<MTRInt16uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt16uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int16uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Descriptor::Attributes::ClusterRevision::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

@end

@implementation MTRBaseClusterDescriptor (Deprecated)

- (void)readAttributeDeviceListWithCompletionHandler:(void (^)(
                                                         NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeDeviceTypeListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeDeviceListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                        maxInterval:(NSNumber * _Nonnull)maxInterval
                                             params:(MTRSubscribeParams * _Nullable)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                      reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeDeviceTypeListWithParams:subscribeParams
                             subscriptionEstablished:subscriptionEstablishedHandler
                                       reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                           // Cast is safe because subclass does not add any selectors.
                                           reportHandler(static_cast<NSArray *>(value), error);
                                       }];
}
+ (void)readAttributeDeviceListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                completionHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeDeviceTypeListWithClusterStateCache:attributeCacheContainer.realContainer
                                                  endpoint:endpoint
                                                     queue:queue
                                                completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                    // Cast is safe because subclass does not add any selectors.
                                                    completionHandler(static_cast<NSArray *>(value), error);
                                                }];
}

- (void)readAttributeServerListWithCompletionHandler:(void (^)(
                                                         NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeServerListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeServerListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                        maxInterval:(NSNumber * _Nonnull)maxInterval
                                             params:(MTRSubscribeParams * _Nullable)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                      reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeServerListWithParams:subscribeParams
                         subscriptionEstablished:subscriptionEstablishedHandler
                                   reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                       // Cast is safe because subclass does not add any selectors.
                                       reportHandler(static_cast<NSArray *>(value), error);
                                   }];
}
+ (void)readAttributeServerListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                completionHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeServerListWithClusterStateCache:attributeCacheContainer.realContainer
                                              endpoint:endpoint
                                                 queue:queue
                                            completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                completionHandler(static_cast<NSArray *>(value), error);
                                            }];
}

- (void)readAttributeClientListWithCompletionHandler:(void (^)(
                                                         NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeClientListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeClientListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                        maxInterval:(NSNumber * _Nonnull)maxInterval
                                             params:(MTRSubscribeParams * _Nullable)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                      reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeClientListWithParams:subscribeParams
                         subscriptionEstablished:subscriptionEstablishedHandler
                                   reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                       // Cast is safe because subclass does not add any selectors.
                                       reportHandler(static_cast<NSArray *>(value), error);
                                   }];
}
+ (void)readAttributeClientListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                completionHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeClientListWithClusterStateCache:attributeCacheContainer.realContainer
                                              endpoint:endpoint
                                                 queue:queue
                                            completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                completionHandler(static_cast<NSArray *>(value), error);
                                            }];
}

- (void)readAttributePartsListWithCompletionHandler:(void (^)(
                                                        NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributePartsListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributePartsListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                       maxInterval:(NSNumber * _Nonnull)maxInterval
                                            params:(MTRSubscribeParams * _Nullable)params
                           subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                     reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributePartsListWithParams:subscribeParams
                        subscriptionEstablished:subscriptionEstablishedHandler
                                  reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                      // Cast is safe because subclass does not add any selectors.
                                      reportHandler(static_cast<NSArray *>(value), error);
                                  }];
}
+ (void)readAttributePartsListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                        endpoint:(NSNumber *)endpoint
                                           queue:(dispatch_queue_t)queue
                               completionHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributePartsListWithClusterStateCache:attributeCacheContainer.realContainer
                                             endpoint:endpoint
                                                queue:queue
                                           completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                               // Cast is safe because subclass does not add any selectors.
                                               completionHandler(static_cast<NSArray *>(value), error);
                                           }];
}

- (void)readAttributeGeneratedCommandListWithCompletionHandler:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))completionHandler
{
    [self readAttributeGeneratedCommandListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeGeneratedCommandListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                  maxInterval:(NSNumber * _Nonnull)maxInterval
                                                       params:(MTRSubscribeParams * _Nullable)params
                                      subscriptionEstablished:
                                          (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                                reportHandler:
                                                    (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeGeneratedCommandListWithParams:subscribeParams
                                   subscriptionEstablished:subscriptionEstablishedHandler
                                             reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                 // Cast is safe because subclass does not add any selectors.
                                                 reportHandler(static_cast<NSArray *>(value), error);
                                             }];
}
+ (void)readAttributeGeneratedCommandListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                          completionHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeGeneratedCommandListWithClusterStateCache:attributeCacheContainer.realContainer
                                                        endpoint:endpoint
                                                           queue:queue
                                                      completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                          // Cast is safe because subclass does not add any selectors.
                                                          completionHandler(static_cast<NSArray *>(value), error);
                                                      }];
}

- (void)readAttributeAcceptedCommandListWithCompletionHandler:(void (^)(NSArray * _Nullable value,
                                                                  NSError * _Nullable error))completionHandler
{
    [self readAttributeAcceptedCommandListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeAcceptedCommandListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                 maxInterval:(NSNumber * _Nonnull)maxInterval
                                                      params:(MTRSubscribeParams * _Nullable)params
                                     subscriptionEstablished:
                                         (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                               reportHandler:
                                                   (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeAcceptedCommandListWithParams:subscribeParams
                                  subscriptionEstablished:subscriptionEstablishedHandler
                                            reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                reportHandler(static_cast<NSArray *>(value), error);
                                            }];
}
+ (void)readAttributeAcceptedCommandListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                         completionHandler:
                                             (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAcceptedCommandListWithClusterStateCache:attributeCacheContainer.realContainer
                                                       endpoint:endpoint
                                                          queue:queue
                                                     completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                         // Cast is safe because subclass does not add any selectors.
                                                         completionHandler(static_cast<NSArray *>(value), error);
                                                     }];
}

- (void)readAttributeAttributeListWithCompletionHandler:(void (^)(
                                                            NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAttributeListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeAttributeListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                           maxInterval:(NSNumber * _Nonnull)maxInterval
                                                params:(MTRSubscribeParams * _Nullable)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                         reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeAttributeListWithParams:subscribeParams
                            subscriptionEstablished:subscriptionEstablishedHandler
                                      reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                          // Cast is safe because subclass does not add any selectors.
                                          reportHandler(static_cast<NSArray *>(value), error);
                                      }];
}
+ (void)readAttributeAttributeListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                   completionHandler:
                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAttributeListWithClusterStateCache:attributeCacheContainer.realContainer
                                                 endpoint:endpoint
                                                    queue:queue
                                               completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                   // Cast is safe because subclass does not add any selectors.
                                                   completionHandler(static_cast<NSArray *>(value), error);
                                               }];
}

- (void)readAttributeFeatureMapWithCompletionHandler:(void (^)(
                                                         NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeFeatureMapWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeFeatureMapWithMinInterval:(NSNumber * _Nonnull)minInterval
                                        maxInterval:(NSNumber * _Nonnull)maxInterval
                                             params:(MTRSubscribeParams * _Nullable)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeFeatureMapWithParams:subscribeParams
                         subscriptionEstablished:subscriptionEstablishedHandler
                                   reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                       // Cast is safe because subclass does not add any selectors.
                                       reportHandler(static_cast<NSNumber *>(value), error);
                                   }];
}
+ (void)readAttributeFeatureMapWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                completionHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeFeatureMapWithClusterStateCache:attributeCacheContainer.realContainer
                                              endpoint:endpoint
                                                 queue:queue
                                            completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                completionHandler(static_cast<NSNumber *>(value), error);
                                            }];
}

- (void)readAttributeClusterRevisionWithCompletionHandler:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completionHandler
{
    [self readAttributeClusterRevisionWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeClusterRevisionWithMinInterval:(NSNumber * _Nonnull)minInterval
                                             maxInterval:(NSNumber * _Nonnull)maxInterval
                                                  params:(MTRSubscribeParams * _Nullable)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                           reportHandler:
                                               (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeClusterRevisionWithParams:subscribeParams
                              subscriptionEstablished:subscriptionEstablishedHandler
                                        reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                            // Cast is safe because subclass does not add any selectors.
                                            reportHandler(static_cast<NSNumber *>(value), error);
                                        }];
}
+ (void)readAttributeClusterRevisionWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                     completionHandler:
                                         (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeClusterRevisionWithClusterStateCache:attributeCacheContainer.realContainer
                                                   endpoint:endpoint
                                                      queue:queue
                                                 completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                     // Cast is safe because subclass does not add any selectors.
                                                     completionHandler(static_cast<NSNumber *>(value), error);
                                                 }];
}

- (nullable instancetype)initWithDevice:(MTRBaseDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

@end

@implementation MTRBaseClusterBinding

- (instancetype)initWithDevice:(MTRBaseDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _device = device;
        _endpoint = [endpointID unsignedShortValue];
    }
    return self;
}

- (void)readAttributeBindingWithParams:(MTRReadParams * _Nullable)params
                            completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{ // Make a copy of params before we go async.
    params = [params copy];
    using TypeInfo = Binding::Attributes::Binding::TypeInfo;
    return MTRReadAttribute<MTRBindingBindingListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)writeAttributeBindingWithValue:(NSArray * _Nonnull)value completion:(MTRStatusCompletion)completion
{
    [self writeAttributeBindingWithValue:(NSArray * _Nonnull) value params:nil completion:completion];
}
- (void)writeAttributeBindingWithValue:(NSArray * _Nonnull)value
                                params:(MTRWriteParams * _Nullable)params
                            completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    value = [value copy];

    auto * bridge = new MTRDefaultSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable ignored, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, DefaultSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            chip::Optional<uint16_t> timedWriteTimeout;
            if (params != nil) {
                if (params.timedWriteTimeout != nil) {
                    timedWriteTimeout.SetValue(params.timedWriteTimeout.unsignedShortValue);
                }
            }

            ListFreer listFreer;
            using TypeInfo = Binding::Attributes::Binding::TypeInfo;
            TypeInfo::Type cppValue;
            {
                using ListType_0 = std::remove_reference_t<decltype(cppValue)>;
                using ListMemberType_0 = ListMemberTypeGetter<ListType_0>::Type;
                if (value.count != 0) {
                    auto * listHolder_0 = new ListHolder<ListMemberType_0>(value.count);
                    if (listHolder_0 == nullptr || listHolder_0->mList == nullptr) {
                        return CHIP_ERROR_INVALID_ARGUMENT;
                    }
                    listFreer.add(listHolder_0);
                    for (size_t i_0 = 0; i_0 < value.count; ++i_0) {
                        if (![value[i_0] isKindOfClass:[MTRBindingClusterTargetStruct class]]) {
                            // Wrong kind of value.
                            return CHIP_ERROR_INVALID_ARGUMENT;
                        }
                        auto element_0 = (MTRBindingClusterTargetStruct *) value[i_0];
                        if (element_0.node != nil) {
                            auto & definedValue_2 = listHolder_0->mList[i_0].node.Emplace();
                            definedValue_2 = element_0.node.unsignedLongLongValue;
                        }
                        if (element_0.group != nil) {
                            auto & definedValue_2 = listHolder_0->mList[i_0].group.Emplace();
                            definedValue_2 = element_0.group.unsignedShortValue;
                        }
                        if (element_0.endpoint != nil) {
                            auto & definedValue_2 = listHolder_0->mList[i_0].endpoint.Emplace();
                            definedValue_2 = element_0.endpoint.unsignedShortValue;
                        }
                        if (element_0.cluster != nil) {
                            auto & definedValue_2 = listHolder_0->mList[i_0].cluster.Emplace();
                            definedValue_2 = element_0.cluster.unsignedIntValue;
                        }
                        listHolder_0->mList[i_0].fabricIndex = element_0.fabricIndex.unsignedCharValue;
                    }
                    cppValue = ListType_0(listHolder_0->mList, value.count);
                } else {
                    cppValue = ListType_0();
                }
            }

            chip::Controller::ClusterBase cppCluster(exchangeManager, session, self->_endpoint);
            return cppCluster.WriteAttribute<TypeInfo>(cppValue, bridge, successCb, failureCb, timedWriteTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)subscribeAttributeBindingWithParams:(MTRSubscribeParams * _Nonnull)params
                    subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                              reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Binding::Attributes::Binding::TypeInfo;
    MTRSubscribeAttribute<MTRBindingBindingListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeBindingWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                       completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRBindingBindingListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(BindingBindingListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Binding::Attributes::Binding::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Binding::Attributes::GeneratedCommandList::TypeInfo;
    return MTRReadAttribute<MTRBindingGeneratedCommandListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams * _Nonnull)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Binding::Attributes::GeneratedCommandList::TypeInfo;
    MTRSubscribeAttribute<MTRBindingGeneratedCommandListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(
        params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRBindingGeneratedCommandListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(BindingGeneratedCommandListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Binding::Attributes::GeneratedCommandList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Binding::Attributes::AcceptedCommandList::TypeInfo;
    return MTRReadAttribute<MTRBindingAcceptedCommandListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams * _Nonnull)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Binding::Attributes::AcceptedCommandList::TypeInfo;
    MTRSubscribeAttribute<MTRBindingAcceptedCommandListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(
        params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRBindingAcceptedCommandListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(BindingAcceptedCommandListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Binding::Attributes::AcceptedCommandList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Binding::Attributes::EventList::TypeInfo;
    return MTRReadAttribute<MTRBindingEventListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams * _Nonnull)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Binding::Attributes::EventList::TypeInfo;
    MTRSubscribeAttribute<MTRBindingEventListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRBindingEventListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(BindingEventListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Binding::Attributes::EventList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Binding::Attributes::AttributeList::TypeInfo;
    return MTRReadAttribute<MTRBindingAttributeListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams * _Nonnull)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Binding::Attributes::AttributeList::TypeInfo;
    MTRSubscribeAttribute<MTRBindingAttributeListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRBindingAttributeListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(BindingAttributeListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Binding::Attributes::AttributeList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Binding::Attributes::FeatureMap::TypeInfo;
    return MTRReadAttribute<MTRInt32uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams * _Nonnull)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Binding::Attributes::FeatureMap::TypeInfo;
    MTRSubscribeAttribute<MTRInt32uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt32uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int32uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Binding::Attributes::FeatureMap::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Binding::Attributes::ClusterRevision::TypeInfo;
    return MTRReadAttribute<MTRInt16uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams * _Nonnull)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Binding::Attributes::ClusterRevision::TypeInfo;
    MTRSubscribeAttribute<MTRInt16uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt16uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int16uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Binding::Attributes::ClusterRevision::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

@end

@implementation MTRBaseClusterBinding (Deprecated)

- (void)readAttributeBindingWithParams:(MTRReadParams * _Nullable)params
                     completionHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeBindingWithParams:params
                              completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                  // Cast is safe because subclass does not add any selectors.
                                  completionHandler(static_cast<NSArray *>(value), error);
                              }];
}
- (void)writeAttributeBindingWithValue:(NSArray * _Nonnull)value completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeBindingWithValue:value params:nil completion:completionHandler];
}
- (void)writeAttributeBindingWithValue:(NSArray * _Nonnull)value
                                params:(MTRWriteParams * _Nullable)params
                     completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeBindingWithValue:value params:params completion:completionHandler];
}
- (void)subscribeAttributeBindingWithMinInterval:(NSNumber * _Nonnull)minInterval
                                     maxInterval:(NSNumber * _Nonnull)maxInterval
                                          params:(MTRSubscribeParams * _Nullable)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                   reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeBindingWithParams:subscribeParams
                      subscriptionEstablished:subscriptionEstablishedHandler
                                reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                    // Cast is safe because subclass does not add any selectors.
                                    reportHandler(static_cast<NSArray *>(value), error);
                                }];
}
+ (void)readAttributeBindingWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                      endpoint:(NSNumber *)endpoint
                                         queue:(dispatch_queue_t)queue
                             completionHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeBindingWithClusterStateCache:attributeCacheContainer.realContainer
                                           endpoint:endpoint
                                              queue:queue
                                         completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                             // Cast is safe because subclass does not add any selectors.
                                             completionHandler(static_cast<NSArray *>(value), error);
                                         }];
}

- (void)readAttributeGeneratedCommandListWithCompletionHandler:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))completionHandler
{
    [self readAttributeGeneratedCommandListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeGeneratedCommandListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                  maxInterval:(NSNumber * _Nonnull)maxInterval
                                                       params:(MTRSubscribeParams * _Nullable)params
                                      subscriptionEstablished:
                                          (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                                reportHandler:
                                                    (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeGeneratedCommandListWithParams:subscribeParams
                                   subscriptionEstablished:subscriptionEstablishedHandler
                                             reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                 // Cast is safe because subclass does not add any selectors.
                                                 reportHandler(static_cast<NSArray *>(value), error);
                                             }];
}
+ (void)readAttributeGeneratedCommandListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                          completionHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeGeneratedCommandListWithClusterStateCache:attributeCacheContainer.realContainer
                                                        endpoint:endpoint
                                                           queue:queue
                                                      completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                          // Cast is safe because subclass does not add any selectors.
                                                          completionHandler(static_cast<NSArray *>(value), error);
                                                      }];
}

- (void)readAttributeAcceptedCommandListWithCompletionHandler:(void (^)(NSArray * _Nullable value,
                                                                  NSError * _Nullable error))completionHandler
{
    [self readAttributeAcceptedCommandListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeAcceptedCommandListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                 maxInterval:(NSNumber * _Nonnull)maxInterval
                                                      params:(MTRSubscribeParams * _Nullable)params
                                     subscriptionEstablished:
                                         (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                               reportHandler:
                                                   (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeAcceptedCommandListWithParams:subscribeParams
                                  subscriptionEstablished:subscriptionEstablishedHandler
                                            reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                reportHandler(static_cast<NSArray *>(value), error);
                                            }];
}
+ (void)readAttributeAcceptedCommandListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                         completionHandler:
                                             (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAcceptedCommandListWithClusterStateCache:attributeCacheContainer.realContainer
                                                       endpoint:endpoint
                                                          queue:queue
                                                     completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                         // Cast is safe because subclass does not add any selectors.
                                                         completionHandler(static_cast<NSArray *>(value), error);
                                                     }];
}

- (void)readAttributeAttributeListWithCompletionHandler:(void (^)(
                                                            NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAttributeListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeAttributeListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                           maxInterval:(NSNumber * _Nonnull)maxInterval
                                                params:(MTRSubscribeParams * _Nullable)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                         reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeAttributeListWithParams:subscribeParams
                            subscriptionEstablished:subscriptionEstablishedHandler
                                      reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                          // Cast is safe because subclass does not add any selectors.
                                          reportHandler(static_cast<NSArray *>(value), error);
                                      }];
}
+ (void)readAttributeAttributeListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                   completionHandler:
                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAttributeListWithClusterStateCache:attributeCacheContainer.realContainer
                                                 endpoint:endpoint
                                                    queue:queue
                                               completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                   // Cast is safe because subclass does not add any selectors.
                                                   completionHandler(static_cast<NSArray *>(value), error);
                                               }];
}

- (void)readAttributeFeatureMapWithCompletionHandler:(void (^)(
                                                         NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeFeatureMapWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeFeatureMapWithMinInterval:(NSNumber * _Nonnull)minInterval
                                        maxInterval:(NSNumber * _Nonnull)maxInterval
                                             params:(MTRSubscribeParams * _Nullable)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeFeatureMapWithParams:subscribeParams
                         subscriptionEstablished:subscriptionEstablishedHandler
                                   reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                       // Cast is safe because subclass does not add any selectors.
                                       reportHandler(static_cast<NSNumber *>(value), error);
                                   }];
}
+ (void)readAttributeFeatureMapWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                completionHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeFeatureMapWithClusterStateCache:attributeCacheContainer.realContainer
                                              endpoint:endpoint
                                                 queue:queue
                                            completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                completionHandler(static_cast<NSNumber *>(value), error);
                                            }];
}

- (void)readAttributeClusterRevisionWithCompletionHandler:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completionHandler
{
    [self readAttributeClusterRevisionWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeClusterRevisionWithMinInterval:(NSNumber * _Nonnull)minInterval
                                             maxInterval:(NSNumber * _Nonnull)maxInterval
                                                  params:(MTRSubscribeParams * _Nullable)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                           reportHandler:
                                               (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeClusterRevisionWithParams:subscribeParams
                              subscriptionEstablished:subscriptionEstablishedHandler
                                        reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                            // Cast is safe because subclass does not add any selectors.
                                            reportHandler(static_cast<NSNumber *>(value), error);
                                        }];
}
+ (void)readAttributeClusterRevisionWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                     completionHandler:
                                         (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeClusterRevisionWithClusterStateCache:attributeCacheContainer.realContainer
                                                   endpoint:endpoint
                                                      queue:queue
                                                 completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                     // Cast is safe because subclass does not add any selectors.
                                                     completionHandler(static_cast<NSNumber *>(value), error);
                                                 }];
}

- (nullable instancetype)initWithDevice:(MTRBaseDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

@end

@implementation MTRBaseClusterAccessControl

- (instancetype)initWithDevice:(MTRBaseDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _device = device;
        _endpoint = [endpointID unsignedShortValue];
    }
    return self;
}

- (void)readAttributeACLWithParams:(MTRReadParams * _Nullable)params
                        completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{ // Make a copy of params before we go async.
    params = [params copy];
    using TypeInfo = AccessControl::Attributes::Acl::TypeInfo;
    return MTRReadAttribute<MTRAccessControlACLListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)writeAttributeACLWithValue:(NSArray * _Nonnull)value completion:(MTRStatusCompletion)completion
{
    [self writeAttributeACLWithValue:(NSArray * _Nonnull) value params:nil completion:completion];
}
- (void)writeAttributeACLWithValue:(NSArray * _Nonnull)value
                            params:(MTRWriteParams * _Nullable)params
                        completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    value = [value copy];

    auto * bridge = new MTRDefaultSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable ignored, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, DefaultSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            chip::Optional<uint16_t> timedWriteTimeout;
            if (params != nil) {
                if (params.timedWriteTimeout != nil) {
                    timedWriteTimeout.SetValue(params.timedWriteTimeout.unsignedShortValue);
                }
            }

            ListFreer listFreer;
            using TypeInfo = AccessControl::Attributes::Acl::TypeInfo;
            TypeInfo::Type cppValue;
            {
                using ListType_0 = std::remove_reference_t<decltype(cppValue)>;
                using ListMemberType_0 = ListMemberTypeGetter<ListType_0>::Type;
                if (value.count != 0) {
                    auto * listHolder_0 = new ListHolder<ListMemberType_0>(value.count);
                    if (listHolder_0 == nullptr || listHolder_0->mList == nullptr) {
                        return CHIP_ERROR_INVALID_ARGUMENT;
                    }
                    listFreer.add(listHolder_0);
                    for (size_t i_0 = 0; i_0 < value.count; ++i_0) {
                        if (![value[i_0] isKindOfClass:[MTRAccessControlClusterAccessControlEntryStruct class]]) {
                            // Wrong kind of value.
                            return CHIP_ERROR_INVALID_ARGUMENT;
                        }
                        auto element_0 = (MTRAccessControlClusterAccessControlEntryStruct *) value[i_0];
                        listHolder_0->mList[i_0].privilege
                            = static_cast<std::remove_reference_t<decltype(listHolder_0->mList[i_0].privilege)>>(
                                element_0.privilege.unsignedCharValue);
                        listHolder_0->mList[i_0].authMode
                            = static_cast<std::remove_reference_t<decltype(listHolder_0->mList[i_0].authMode)>>(
                                element_0.authMode.unsignedCharValue);
                        if (element_0.subjects == nil) {
                            listHolder_0->mList[i_0].subjects.SetNull();
                        } else {
                            auto & nonNullValue_2 = listHolder_0->mList[i_0].subjects.SetNonNull();
                            {
                                using ListType_3 = std::remove_reference_t<decltype(nonNullValue_2)>;
                                using ListMemberType_3 = ListMemberTypeGetter<ListType_3>::Type;
                                if (element_0.subjects.count != 0) {
                                    auto * listHolder_3 = new ListHolder<ListMemberType_3>(element_0.subjects.count);
                                    if (listHolder_3 == nullptr || listHolder_3->mList == nullptr) {
                                        return CHIP_ERROR_INVALID_ARGUMENT;
                                    }
                                    listFreer.add(listHolder_3);
                                    for (size_t i_3 = 0; i_3 < element_0.subjects.count; ++i_3) {
                                        if (![element_0.subjects[i_3] isKindOfClass:[NSNumber class]]) {
                                            // Wrong kind of value.
                                            return CHIP_ERROR_INVALID_ARGUMENT;
                                        }
                                        auto element_3 = (NSNumber *) element_0.subjects[i_3];
                                        listHolder_3->mList[i_3] = element_3.unsignedLongLongValue;
                                    }
                                    nonNullValue_2 = ListType_3(listHolder_3->mList, element_0.subjects.count);
                                } else {
                                    nonNullValue_2 = ListType_3();
                                }
                            }
                        }
                        if (element_0.targets == nil) {
                            listHolder_0->mList[i_0].targets.SetNull();
                        } else {
                            auto & nonNullValue_2 = listHolder_0->mList[i_0].targets.SetNonNull();
                            {
                                using ListType_3 = std::remove_reference_t<decltype(nonNullValue_2)>;
                                using ListMemberType_3 = ListMemberTypeGetter<ListType_3>::Type;
                                if (element_0.targets.count != 0) {
                                    auto * listHolder_3 = new ListHolder<ListMemberType_3>(element_0.targets.count);
                                    if (listHolder_3 == nullptr || listHolder_3->mList == nullptr) {
                                        return CHIP_ERROR_INVALID_ARGUMENT;
                                    }
                                    listFreer.add(listHolder_3);
                                    for (size_t i_3 = 0; i_3 < element_0.targets.count; ++i_3) {
                                        if (![element_0.targets[i_3]
                                                isKindOfClass:[MTRAccessControlClusterAccessControlTargetStruct class]]) {
                                            // Wrong kind of value.
                                            return CHIP_ERROR_INVALID_ARGUMENT;
                                        }
                                        auto element_3
                                            = (MTRAccessControlClusterAccessControlTargetStruct *) element_0.targets[i_3];
                                        if (element_3.cluster == nil) {
                                            listHolder_3->mList[i_3].cluster.SetNull();
                                        } else {
                                            auto & nonNullValue_5 = listHolder_3->mList[i_3].cluster.SetNonNull();
                                            nonNullValue_5 = element_3.cluster.unsignedIntValue;
                                        }
                                        if (element_3.endpoint == nil) {
                                            listHolder_3->mList[i_3].endpoint.SetNull();
                                        } else {
                                            auto & nonNullValue_5 = listHolder_3->mList[i_3].endpoint.SetNonNull();
                                            nonNullValue_5 = element_3.endpoint.unsignedShortValue;
                                        }
                                        if (element_3.deviceType == nil) {
                                            listHolder_3->mList[i_3].deviceType.SetNull();
                                        } else {
                                            auto & nonNullValue_5 = listHolder_3->mList[i_3].deviceType.SetNonNull();
                                            nonNullValue_5 = element_3.deviceType.unsignedIntValue;
                                        }
                                    }
                                    nonNullValue_2 = ListType_3(listHolder_3->mList, element_0.targets.count);
                                } else {
                                    nonNullValue_2 = ListType_3();
                                }
                            }
                        }
                        listHolder_0->mList[i_0].fabricIndex = element_0.fabricIndex.unsignedCharValue;
                    }
                    cppValue = ListType_0(listHolder_0->mList, value.count);
                } else {
                    cppValue = ListType_0();
                }
            }

            chip::Controller::ClusterBase cppCluster(exchangeManager, session, self->_endpoint);
            return cppCluster.WriteAttribute<TypeInfo>(cppValue, bridge, successCb, failureCb, timedWriteTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)subscribeAttributeACLWithParams:(MTRSubscribeParams * _Nonnull)params
                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                          reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = AccessControl::Attributes::Acl::TypeInfo;
    MTRSubscribeAttribute<MTRAccessControlACLListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeACLWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                     endpoint:(NSNumber *)endpoint
                                        queue:(dispatch_queue_t)queue
                                   completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRAccessControlACLListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(AccessControlACLListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = AccessControl::Attributes::Acl::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeExtensionWithParams:(MTRReadParams * _Nullable)params
                              completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{ // Make a copy of params before we go async.
    params = [params copy];
    using TypeInfo = AccessControl::Attributes::Extension::TypeInfo;
    return MTRReadAttribute<MTRAccessControlExtensionListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)writeAttributeExtensionWithValue:(NSArray * _Nonnull)value completion:(MTRStatusCompletion)completion
{
    [self writeAttributeExtensionWithValue:(NSArray * _Nonnull) value params:nil completion:completion];
}
- (void)writeAttributeExtensionWithValue:(NSArray * _Nonnull)value
                                  params:(MTRWriteParams * _Nullable)params
                              completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    value = [value copy];

    auto * bridge = new MTRDefaultSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable ignored, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, DefaultSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            chip::Optional<uint16_t> timedWriteTimeout;
            if (params != nil) {
                if (params.timedWriteTimeout != nil) {
                    timedWriteTimeout.SetValue(params.timedWriteTimeout.unsignedShortValue);
                }
            }

            ListFreer listFreer;
            using TypeInfo = AccessControl::Attributes::Extension::TypeInfo;
            TypeInfo::Type cppValue;
            {
                using ListType_0 = std::remove_reference_t<decltype(cppValue)>;
                using ListMemberType_0 = ListMemberTypeGetter<ListType_0>::Type;
                if (value.count != 0) {
                    auto * listHolder_0 = new ListHolder<ListMemberType_0>(value.count);
                    if (listHolder_0 == nullptr || listHolder_0->mList == nullptr) {
                        return CHIP_ERROR_INVALID_ARGUMENT;
                    }
                    listFreer.add(listHolder_0);
                    for (size_t i_0 = 0; i_0 < value.count; ++i_0) {
                        if (![value[i_0] isKindOfClass:[MTRAccessControlClusterAccessControlExtensionStruct class]]) {
                            // Wrong kind of value.
                            return CHIP_ERROR_INVALID_ARGUMENT;
                        }
                        auto element_0 = (MTRAccessControlClusterAccessControlExtensionStruct *) value[i_0];
                        listHolder_0->mList[i_0].data = [self asByteSpan:element_0.data];
                        listHolder_0->mList[i_0].fabricIndex = element_0.fabricIndex.unsignedCharValue;
                    }
                    cppValue = ListType_0(listHolder_0->mList, value.count);
                } else {
                    cppValue = ListType_0();
                }
            }

            chip::Controller::ClusterBase cppCluster(exchangeManager, session, self->_endpoint);
            return cppCluster.WriteAttribute<TypeInfo>(cppValue, bridge, successCb, failureCb, timedWriteTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)subscribeAttributeExtensionWithParams:(MTRSubscribeParams * _Nonnull)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = AccessControl::Attributes::Extension::TypeInfo;
    MTRSubscribeAttribute<MTRAccessControlExtensionListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(
        params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeExtensionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRAccessControlExtensionListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(AccessControlExtensionListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = AccessControl::Attributes::Extension::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeSubjectsPerAccessControlEntryWithCompletion:(void (^)(NSNumber * _Nullable value,
                                                                     NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = AccessControl::Attributes::SubjectsPerAccessControlEntry::TypeInfo;
    return MTRReadAttribute<MTRInt16uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeSubjectsPerAccessControlEntryWithParams:(MTRSubscribeParams * _Nonnull)params
                                          subscriptionEstablished:
                                              (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                    reportHandler:(void (^)(NSNumber * _Nullable value,
                                                                      NSError * _Nullable error))reportHandler
{
    using TypeInfo = AccessControl::Attributes::SubjectsPerAccessControlEntry::TypeInfo;
    MTRSubscribeAttribute<MTRInt16uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeSubjectsPerAccessControlEntryWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                               endpoint:(NSNumber *)endpoint
                                                                  queue:(dispatch_queue_t)queue
                                                             completion:(void (^)(NSNumber * _Nullable value,
                                                                            NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt16uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int16uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = AccessControl::Attributes::SubjectsPerAccessControlEntry::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeTargetsPerAccessControlEntryWithCompletion:(void (^)(NSNumber * _Nullable value,
                                                                    NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = AccessControl::Attributes::TargetsPerAccessControlEntry::TypeInfo;
    return MTRReadAttribute<MTRInt16uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeTargetsPerAccessControlEntryWithParams:(MTRSubscribeParams * _Nonnull)params
                                         subscriptionEstablished:
                                             (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                   reportHandler:(void (^)(NSNumber * _Nullable value,
                                                                     NSError * _Nullable error))reportHandler
{
    using TypeInfo = AccessControl::Attributes::TargetsPerAccessControlEntry::TypeInfo;
    MTRSubscribeAttribute<MTRInt16uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeTargetsPerAccessControlEntryWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                              endpoint:(NSNumber *)endpoint
                                                                 queue:(dispatch_queue_t)queue
                                                            completion:(void (^)(NSNumber * _Nullable value,
                                                                           NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt16uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int16uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = AccessControl::Attributes::TargetsPerAccessControlEntry::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeAccessControlEntriesPerFabricWithCompletion:(void (^)(NSNumber * _Nullable value,
                                                                     NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = AccessControl::Attributes::AccessControlEntriesPerFabric::TypeInfo;
    return MTRReadAttribute<MTRInt16uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeAccessControlEntriesPerFabricWithParams:(MTRSubscribeParams * _Nonnull)params
                                          subscriptionEstablished:
                                              (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                                    reportHandler:(void (^)(NSNumber * _Nullable value,
                                                                      NSError * _Nullable error))reportHandler
{
    using TypeInfo = AccessControl::Attributes::AccessControlEntriesPerFabric::TypeInfo;
    MTRSubscribeAttribute<MTRInt16uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeAccessControlEntriesPerFabricWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                               endpoint:(NSNumber *)endpoint
                                                                  queue:(dispatch_queue_t)queue
                                                             completion:(void (^)(NSNumber * _Nullable value,
                                                                            NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt16uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int16uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = AccessControl::Attributes::AccessControlEntriesPerFabric::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = AccessControl::Attributes::GeneratedCommandList::TypeInfo;
    return MTRReadAttribute<MTRAccessControlGeneratedCommandListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams * _Nonnull)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = AccessControl::Attributes::GeneratedCommandList::TypeInfo;
    MTRSubscribeAttribute<MTRAccessControlGeneratedCommandListListAttributeCallbackSubscriptionBridge, NSArray,
        TypeInfo::DecodableType>(params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint,
        TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRAccessControlGeneratedCommandListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(AccessControlGeneratedCommandListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = AccessControl::Attributes::GeneratedCommandList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = AccessControl::Attributes::AcceptedCommandList::TypeInfo;
    return MTRReadAttribute<MTRAccessControlAcceptedCommandListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams * _Nonnull)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = AccessControl::Attributes::AcceptedCommandList::TypeInfo;
    MTRSubscribeAttribute<MTRAccessControlAcceptedCommandListListAttributeCallbackSubscriptionBridge, NSArray,
        TypeInfo::DecodableType>(params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint,
        TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRAccessControlAcceptedCommandListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(AccessControlAcceptedCommandListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = AccessControl::Attributes::AcceptedCommandList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = AccessControl::Attributes::EventList::TypeInfo;
    return MTRReadAttribute<MTRAccessControlEventListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams * _Nonnull)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = AccessControl::Attributes::EventList::TypeInfo;
    MTRSubscribeAttribute<MTRAccessControlEventListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(
        params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRAccessControlEventListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(AccessControlEventListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = AccessControl::Attributes::EventList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = AccessControl::Attributes::AttributeList::TypeInfo;
    return MTRReadAttribute<MTRAccessControlAttributeListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams * _Nonnull)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = AccessControl::Attributes::AttributeList::TypeInfo;
    MTRSubscribeAttribute<MTRAccessControlAttributeListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(
        params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRAccessControlAttributeListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(AccessControlAttributeListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = AccessControl::Attributes::AttributeList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = AccessControl::Attributes::FeatureMap::TypeInfo;
    return MTRReadAttribute<MTRInt32uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams * _Nonnull)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = AccessControl::Attributes::FeatureMap::TypeInfo;
    MTRSubscribeAttribute<MTRInt32uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt32uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int32uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = AccessControl::Attributes::FeatureMap::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = AccessControl::Attributes::ClusterRevision::TypeInfo;
    return MTRReadAttribute<MTRInt16uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams * _Nonnull)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = AccessControl::Attributes::ClusterRevision::TypeInfo;
    MTRSubscribeAttribute<MTRInt16uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt16uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int16uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = AccessControl::Attributes::ClusterRevision::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

@end

@implementation MTRBaseClusterAccessControl (Deprecated)

- (void)readAttributeAclWithParams:(MTRReadParams * _Nullable)params
                 completionHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeACLWithParams:params
                          completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                              // Cast is safe because subclass does not add any selectors.
                              completionHandler(static_cast<NSArray *>(value), error);
                          }];
}
- (void)writeAttributeAclWithValue:(NSArray * _Nonnull)value completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeACLWithValue:value params:nil completion:completionHandler];
}
- (void)writeAttributeAclWithValue:(NSArray * _Nonnull)value
                            params:(MTRWriteParams * _Nullable)params
                 completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeACLWithValue:value params:params completion:completionHandler];
}
- (void)subscribeAttributeAclWithMinInterval:(NSNumber * _Nonnull)minInterval
                                 maxInterval:(NSNumber * _Nonnull)maxInterval
                                      params:(MTRSubscribeParams * _Nullable)params
                     subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                               reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeACLWithParams:subscribeParams
                  subscriptionEstablished:subscriptionEstablishedHandler
                            reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                // Cast is safe because subclass does not add any selectors.
                                reportHandler(static_cast<NSArray *>(value), error);
                            }];
}
+ (void)readAttributeAclWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                  endpoint:(NSNumber *)endpoint
                                     queue:(dispatch_queue_t)queue
                         completionHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeACLWithClusterStateCache:attributeCacheContainer.realContainer
                                       endpoint:endpoint
                                          queue:queue
                                     completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                         // Cast is safe because subclass does not add any selectors.
                                         completionHandler(static_cast<NSArray *>(value), error);
                                     }];
}

- (void)readAttributeExtensionWithParams:(MTRReadParams * _Nullable)params
                       completionHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeExtensionWithParams:params
                                completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                    // Cast is safe because subclass does not add any selectors.
                                    completionHandler(static_cast<NSArray *>(value), error);
                                }];
}
- (void)writeAttributeExtensionWithValue:(NSArray * _Nonnull)value completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeExtensionWithValue:value params:nil completion:completionHandler];
}
- (void)writeAttributeExtensionWithValue:(NSArray * _Nonnull)value
                                  params:(MTRWriteParams * _Nullable)params
                       completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeExtensionWithValue:value params:params completion:completionHandler];
}
- (void)subscribeAttributeExtensionWithMinInterval:(NSNumber * _Nonnull)minInterval
                                       maxInterval:(NSNumber * _Nonnull)maxInterval
                                            params:(MTRSubscribeParams * _Nullable)params
                           subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                     reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeExtensionWithParams:subscribeParams
                        subscriptionEstablished:subscriptionEstablishedHandler
                                  reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                      // Cast is safe because subclass does not add any selectors.
                                      reportHandler(static_cast<NSArray *>(value), error);
                                  }];
}
+ (void)readAttributeExtensionWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                        endpoint:(NSNumber *)endpoint
                                           queue:(dispatch_queue_t)queue
                               completionHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeExtensionWithClusterStateCache:attributeCacheContainer.realContainer
                                             endpoint:endpoint
                                                queue:queue
                                           completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                               // Cast is safe because subclass does not add any selectors.
                                               completionHandler(static_cast<NSArray *>(value), error);
                                           }];
}

- (void)readAttributeSubjectsPerAccessControlEntryWithCompletionHandler:(void (^)(NSNumber * _Nullable value,
                                                                            NSError * _Nullable error))completionHandler
{
    [self readAttributeSubjectsPerAccessControlEntryWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeSubjectsPerAccessControlEntryWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                           maxInterval:(NSNumber * _Nonnull)maxInterval
                                                                params:(MTRSubscribeParams * _Nullable)params
                                               subscriptionEstablished:
                                                   (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                                         reportHandler:(void (^)(NSNumber * _Nullable value,
                                                                           NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeSubjectsPerAccessControlEntryWithParams:subscribeParams
                                            subscriptionEstablished:subscriptionEstablishedHandler
                                                      reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                          // Cast is safe because subclass does not add any selectors.
                                                          reportHandler(static_cast<NSNumber *>(value), error);
                                                      }];
}
+ (void)readAttributeSubjectsPerAccessControlEntryWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                            endpoint:(NSNumber *)endpoint
                                                               queue:(dispatch_queue_t)queue
                                                   completionHandler:(void (^)(NSNumber * _Nullable value,
                                                                         NSError * _Nullable error))completionHandler
{
    [self readAttributeSubjectsPerAccessControlEntryWithClusterStateCache:attributeCacheContainer.realContainer
                                                                 endpoint:endpoint
                                                                    queue:queue
                                                               completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                                   // Cast is safe because subclass does not add any selectors.
                                                                   completionHandler(static_cast<NSNumber *>(value), error);
                                                               }];
}

- (void)readAttributeTargetsPerAccessControlEntryWithCompletionHandler:(void (^)(NSNumber * _Nullable value,
                                                                           NSError * _Nullable error))completionHandler
{
    [self readAttributeTargetsPerAccessControlEntryWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeTargetsPerAccessControlEntryWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                          maxInterval:(NSNumber * _Nonnull)maxInterval
                                                               params:(MTRSubscribeParams * _Nullable)params
                                              subscriptionEstablished:
                                                  (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                                        reportHandler:(void (^)(NSNumber * _Nullable value,
                                                                          NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeTargetsPerAccessControlEntryWithParams:subscribeParams
                                           subscriptionEstablished:subscriptionEstablishedHandler
                                                     reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                         // Cast is safe because subclass does not add any selectors.
                                                         reportHandler(static_cast<NSNumber *>(value), error);
                                                     }];
}
+ (void)readAttributeTargetsPerAccessControlEntryWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                           endpoint:(NSNumber *)endpoint
                                                              queue:(dispatch_queue_t)queue
                                                  completionHandler:(void (^)(NSNumber * _Nullable value,
                                                                        NSError * _Nullable error))completionHandler
{
    [self readAttributeTargetsPerAccessControlEntryWithClusterStateCache:attributeCacheContainer.realContainer
                                                                endpoint:endpoint
                                                                   queue:queue
                                                              completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                                  // Cast is safe because subclass does not add any selectors.
                                                                  completionHandler(static_cast<NSNumber *>(value), error);
                                                              }];
}

- (void)readAttributeAccessControlEntriesPerFabricWithCompletionHandler:(void (^)(NSNumber * _Nullable value,
                                                                            NSError * _Nullable error))completionHandler
{
    [self readAttributeAccessControlEntriesPerFabricWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeAccessControlEntriesPerFabricWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                           maxInterval:(NSNumber * _Nonnull)maxInterval
                                                                params:(MTRSubscribeParams * _Nullable)params
                                               subscriptionEstablished:
                                                   (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                                         reportHandler:(void (^)(NSNumber * _Nullable value,
                                                                           NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeAccessControlEntriesPerFabricWithParams:subscribeParams
                                            subscriptionEstablished:subscriptionEstablishedHandler
                                                      reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                          // Cast is safe because subclass does not add any selectors.
                                                          reportHandler(static_cast<NSNumber *>(value), error);
                                                      }];
}
+ (void)readAttributeAccessControlEntriesPerFabricWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                            endpoint:(NSNumber *)endpoint
                                                               queue:(dispatch_queue_t)queue
                                                   completionHandler:(void (^)(NSNumber * _Nullable value,
                                                                         NSError * _Nullable error))completionHandler
{
    [self readAttributeAccessControlEntriesPerFabricWithClusterStateCache:attributeCacheContainer.realContainer
                                                                 endpoint:endpoint
                                                                    queue:queue
                                                               completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                                   // Cast is safe because subclass does not add any selectors.
                                                                   completionHandler(static_cast<NSNumber *>(value), error);
                                                               }];
}

- (void)readAttributeGeneratedCommandListWithCompletionHandler:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))completionHandler
{
    [self readAttributeGeneratedCommandListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeGeneratedCommandListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                  maxInterval:(NSNumber * _Nonnull)maxInterval
                                                       params:(MTRSubscribeParams * _Nullable)params
                                      subscriptionEstablished:
                                          (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                                reportHandler:
                                                    (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeGeneratedCommandListWithParams:subscribeParams
                                   subscriptionEstablished:subscriptionEstablishedHandler
                                             reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                 // Cast is safe because subclass does not add any selectors.
                                                 reportHandler(static_cast<NSArray *>(value), error);
                                             }];
}
+ (void)readAttributeGeneratedCommandListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                          completionHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeGeneratedCommandListWithClusterStateCache:attributeCacheContainer.realContainer
                                                        endpoint:endpoint
                                                           queue:queue
                                                      completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                          // Cast is safe because subclass does not add any selectors.
                                                          completionHandler(static_cast<NSArray *>(value), error);
                                                      }];
}

- (void)readAttributeAcceptedCommandListWithCompletionHandler:(void (^)(NSArray * _Nullable value,
                                                                  NSError * _Nullable error))completionHandler
{
    [self readAttributeAcceptedCommandListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeAcceptedCommandListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                 maxInterval:(NSNumber * _Nonnull)maxInterval
                                                      params:(MTRSubscribeParams * _Nullable)params
                                     subscriptionEstablished:
                                         (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                               reportHandler:
                                                   (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeAcceptedCommandListWithParams:subscribeParams
                                  subscriptionEstablished:subscriptionEstablishedHandler
                                            reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                reportHandler(static_cast<NSArray *>(value), error);
                                            }];
}
+ (void)readAttributeAcceptedCommandListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                         completionHandler:
                                             (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAcceptedCommandListWithClusterStateCache:attributeCacheContainer.realContainer
                                                       endpoint:endpoint
                                                          queue:queue
                                                     completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                         // Cast is safe because subclass does not add any selectors.
                                                         completionHandler(static_cast<NSArray *>(value), error);
                                                     }];
}

- (void)readAttributeAttributeListWithCompletionHandler:(void (^)(
                                                            NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAttributeListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeAttributeListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                           maxInterval:(NSNumber * _Nonnull)maxInterval
                                                params:(MTRSubscribeParams * _Nullable)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                         reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeAttributeListWithParams:subscribeParams
                            subscriptionEstablished:subscriptionEstablishedHandler
                                      reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                          // Cast is safe because subclass does not add any selectors.
                                          reportHandler(static_cast<NSArray *>(value), error);
                                      }];
}
+ (void)readAttributeAttributeListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                   completionHandler:
                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAttributeListWithClusterStateCache:attributeCacheContainer.realContainer
                                                 endpoint:endpoint
                                                    queue:queue
                                               completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                   // Cast is safe because subclass does not add any selectors.
                                                   completionHandler(static_cast<NSArray *>(value), error);
                                               }];
}

- (void)readAttributeFeatureMapWithCompletionHandler:(void (^)(
                                                         NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeFeatureMapWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeFeatureMapWithMinInterval:(NSNumber * _Nonnull)minInterval
                                        maxInterval:(NSNumber * _Nonnull)maxInterval
                                             params:(MTRSubscribeParams * _Nullable)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeFeatureMapWithParams:subscribeParams
                         subscriptionEstablished:subscriptionEstablishedHandler
                                   reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                       // Cast is safe because subclass does not add any selectors.
                                       reportHandler(static_cast<NSNumber *>(value), error);
                                   }];
}
+ (void)readAttributeFeatureMapWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                completionHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeFeatureMapWithClusterStateCache:attributeCacheContainer.realContainer
                                              endpoint:endpoint
                                                 queue:queue
                                            completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                completionHandler(static_cast<NSNumber *>(value), error);
                                            }];
}

- (void)readAttributeClusterRevisionWithCompletionHandler:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completionHandler
{
    [self readAttributeClusterRevisionWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeClusterRevisionWithMinInterval:(NSNumber * _Nonnull)minInterval
                                             maxInterval:(NSNumber * _Nonnull)maxInterval
                                                  params:(MTRSubscribeParams * _Nullable)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                           reportHandler:
                                               (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeClusterRevisionWithParams:subscribeParams
                              subscriptionEstablished:subscriptionEstablishedHandler
                                        reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                            // Cast is safe because subclass does not add any selectors.
                                            reportHandler(static_cast<NSNumber *>(value), error);
                                        }];
}
+ (void)readAttributeClusterRevisionWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                     completionHandler:
                                         (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeClusterRevisionWithClusterStateCache:attributeCacheContainer.realContainer
                                                   endpoint:endpoint
                                                      queue:queue
                                                 completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                     // Cast is safe because subclass does not add any selectors.
                                                     completionHandler(static_cast<NSNumber *>(value), error);
                                                 }];
}

- (nullable instancetype)initWithDevice:(MTRBaseDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

@end

@implementation MTRBaseClusterActions

- (instancetype)initWithDevice:(MTRBaseDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _device = device;
        _endpoint = [endpointID unsignedShortValue];
    }
    return self;
}

- (void)instantActionWithParams:(MTRActionsClusterInstantActionParams *)params completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            Actions::Commands::InstantAction::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.actionID = params.actionID.unsignedShortValue;
            if (params.invokeID != nil) {
                auto & definedValue_0 = request.invokeID.Emplace();
                definedValue_0 = params.invokeID.unsignedIntValue;
            }

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)instantActionWithTransitionWithParams:(MTRActionsClusterInstantActionWithTransitionParams *)params
                                   completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            Actions::Commands::InstantActionWithTransition::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.actionID = params.actionID.unsignedShortValue;
            if (params.invokeID != nil) {
                auto & definedValue_0 = request.invokeID.Emplace();
                definedValue_0 = params.invokeID.unsignedIntValue;
            }
            request.transitionTime = params.transitionTime.unsignedShortValue;

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)startActionWithParams:(MTRActionsClusterStartActionParams *)params completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            Actions::Commands::StartAction::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.actionID = params.actionID.unsignedShortValue;
            if (params.invokeID != nil) {
                auto & definedValue_0 = request.invokeID.Emplace();
                definedValue_0 = params.invokeID.unsignedIntValue;
            }

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)startActionWithDurationWithParams:(MTRActionsClusterStartActionWithDurationParams *)params
                               completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            Actions::Commands::StartActionWithDuration::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.actionID = params.actionID.unsignedShortValue;
            if (params.invokeID != nil) {
                auto & definedValue_0 = request.invokeID.Emplace();
                definedValue_0 = params.invokeID.unsignedIntValue;
            }
            request.duration = params.duration.unsignedIntValue;

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)stopActionWithParams:(MTRActionsClusterStopActionParams *)params completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            Actions::Commands::StopAction::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.actionID = params.actionID.unsignedShortValue;
            if (params.invokeID != nil) {
                auto & definedValue_0 = request.invokeID.Emplace();
                definedValue_0 = params.invokeID.unsignedIntValue;
            }

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)pauseActionWithParams:(MTRActionsClusterPauseActionParams *)params completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            Actions::Commands::PauseAction::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.actionID = params.actionID.unsignedShortValue;
            if (params.invokeID != nil) {
                auto & definedValue_0 = request.invokeID.Emplace();
                definedValue_0 = params.invokeID.unsignedIntValue;
            }

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)pauseActionWithDurationWithParams:(MTRActionsClusterPauseActionWithDurationParams *)params
                               completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            Actions::Commands::PauseActionWithDuration::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.actionID = params.actionID.unsignedShortValue;
            if (params.invokeID != nil) {
                auto & definedValue_0 = request.invokeID.Emplace();
                definedValue_0 = params.invokeID.unsignedIntValue;
            }
            request.duration = params.duration.unsignedIntValue;

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)resumeActionWithParams:(MTRActionsClusterResumeActionParams *)params completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            Actions::Commands::ResumeAction::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.actionID = params.actionID.unsignedShortValue;
            if (params.invokeID != nil) {
                auto & definedValue_0 = request.invokeID.Emplace();
                definedValue_0 = params.invokeID.unsignedIntValue;
            }

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)enableActionWithParams:(MTRActionsClusterEnableActionParams *)params completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            Actions::Commands::EnableAction::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.actionID = params.actionID.unsignedShortValue;
            if (params.invokeID != nil) {
                auto & definedValue_0 = request.invokeID.Emplace();
                definedValue_0 = params.invokeID.unsignedIntValue;
            }

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)enableActionWithDurationWithParams:(MTRActionsClusterEnableActionWithDurationParams *)params
                                completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            Actions::Commands::EnableActionWithDuration::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.actionID = params.actionID.unsignedShortValue;
            if (params.invokeID != nil) {
                auto & definedValue_0 = request.invokeID.Emplace();
                definedValue_0 = params.invokeID.unsignedIntValue;
            }
            request.duration = params.duration.unsignedIntValue;

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)disableActionWithParams:(MTRActionsClusterDisableActionParams *)params completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            Actions::Commands::DisableAction::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.actionID = params.actionID.unsignedShortValue;
            if (params.invokeID != nil) {
                auto & definedValue_0 = request.invokeID.Emplace();
                definedValue_0 = params.invokeID.unsignedIntValue;
            }

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)disableActionWithDurationWithParams:(MTRActionsClusterDisableActionWithDurationParams *)params
                                 completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            Actions::Commands::DisableActionWithDuration::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.actionID = params.actionID.unsignedShortValue;
            if (params.invokeID != nil) {
                auto & definedValue_0 = request.invokeID.Emplace();
                definedValue_0 = params.invokeID.unsignedIntValue;
            }
            request.duration = params.duration.unsignedIntValue;

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)readAttributeActionListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Actions::Attributes::ActionList::TypeInfo;
    return MTRReadAttribute<MTRActionsActionListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeActionListWithParams:(MTRSubscribeParams * _Nonnull)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Actions::Attributes::ActionList::TypeInfo;
    MTRSubscribeAttribute<MTRActionsActionListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeActionListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRActionsActionListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(ActionsActionListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Actions::Attributes::ActionList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeEndpointListsWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Actions::Attributes::EndpointLists::TypeInfo;
    return MTRReadAttribute<MTRActionsEndpointListsListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeEndpointListsWithParams:(MTRSubscribeParams * _Nonnull)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Actions::Attributes::EndpointLists::TypeInfo;
    MTRSubscribeAttribute<MTRActionsEndpointListsListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeEndpointListsWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRActionsEndpointListsListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(ActionsEndpointListsListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Actions::Attributes::EndpointLists::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeSetupURLWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Actions::Attributes::SetupURL::TypeInfo;
    return MTRReadAttribute<MTRCharStringAttributeCallbackBridge, NSString, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeSetupURLWithParams:(MTRSubscribeParams * _Nonnull)params
                     subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                               reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Actions::Attributes::SetupURL::TypeInfo;
    MTRSubscribeAttribute<MTRCharStringAttributeCallbackSubscriptionBridge, NSString, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeSetupURLWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                          endpoint:(NSNumber *)endpoint
                                             queue:(dispatch_queue_t)queue
                                        completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRCharStringAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(CharStringAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Actions::Attributes::SetupURL::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Actions::Attributes::GeneratedCommandList::TypeInfo;
    return MTRReadAttribute<MTRActionsGeneratedCommandListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams * _Nonnull)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Actions::Attributes::GeneratedCommandList::TypeInfo;
    MTRSubscribeAttribute<MTRActionsGeneratedCommandListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(
        params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRActionsGeneratedCommandListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(ActionsGeneratedCommandListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Actions::Attributes::GeneratedCommandList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Actions::Attributes::AcceptedCommandList::TypeInfo;
    return MTRReadAttribute<MTRActionsAcceptedCommandListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams * _Nonnull)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Actions::Attributes::AcceptedCommandList::TypeInfo;
    MTRSubscribeAttribute<MTRActionsAcceptedCommandListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(
        params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRActionsAcceptedCommandListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(ActionsAcceptedCommandListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Actions::Attributes::AcceptedCommandList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Actions::Attributes::EventList::TypeInfo;
    return MTRReadAttribute<MTRActionsEventListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams * _Nonnull)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Actions::Attributes::EventList::TypeInfo;
    MTRSubscribeAttribute<MTRActionsEventListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRActionsEventListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(ActionsEventListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Actions::Attributes::EventList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Actions::Attributes::AttributeList::TypeInfo;
    return MTRReadAttribute<MTRActionsAttributeListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams * _Nonnull)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Actions::Attributes::AttributeList::TypeInfo;
    MTRSubscribeAttribute<MTRActionsAttributeListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRActionsAttributeListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(ActionsAttributeListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Actions::Attributes::AttributeList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Actions::Attributes::FeatureMap::TypeInfo;
    return MTRReadAttribute<MTRInt32uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams * _Nonnull)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Actions::Attributes::FeatureMap::TypeInfo;
    MTRSubscribeAttribute<MTRInt32uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt32uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int32uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Actions::Attributes::FeatureMap::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = Actions::Attributes::ClusterRevision::TypeInfo;
    return MTRReadAttribute<MTRInt16uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams * _Nonnull)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = Actions::Attributes::ClusterRevision::TypeInfo;
    MTRSubscribeAttribute<MTRInt16uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt16uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int16uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = Actions::Attributes::ClusterRevision::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

@end

@implementation MTRBaseClusterActions (Deprecated)

- (void)instantActionWithParams:(MTRActionsClusterInstantActionParams *)params
              completionHandler:(MTRStatusCompletion)completionHandler
{
    [self instantActionWithParams:params completion:completionHandler];
}
- (void)instantActionWithTransitionWithParams:(MTRActionsClusterInstantActionWithTransitionParams *)params
                            completionHandler:(MTRStatusCompletion)completionHandler
{
    [self instantActionWithTransitionWithParams:params completion:completionHandler];
}
- (void)startActionWithParams:(MTRActionsClusterStartActionParams *)params completionHandler:(MTRStatusCompletion)completionHandler
{
    [self startActionWithParams:params completion:completionHandler];
}
- (void)startActionWithDurationWithParams:(MTRActionsClusterStartActionWithDurationParams *)params
                        completionHandler:(MTRStatusCompletion)completionHandler
{
    [self startActionWithDurationWithParams:params completion:completionHandler];
}
- (void)stopActionWithParams:(MTRActionsClusterStopActionParams *)params completionHandler:(MTRStatusCompletion)completionHandler
{
    [self stopActionWithParams:params completion:completionHandler];
}
- (void)pauseActionWithParams:(MTRActionsClusterPauseActionParams *)params completionHandler:(MTRStatusCompletion)completionHandler
{
    [self pauseActionWithParams:params completion:completionHandler];
}
- (void)pauseActionWithDurationWithParams:(MTRActionsClusterPauseActionWithDurationParams *)params
                        completionHandler:(MTRStatusCompletion)completionHandler
{
    [self pauseActionWithDurationWithParams:params completion:completionHandler];
}
- (void)resumeActionWithParams:(MTRActionsClusterResumeActionParams *)params
             completionHandler:(MTRStatusCompletion)completionHandler
{
    [self resumeActionWithParams:params completion:completionHandler];
}
- (void)enableActionWithParams:(MTRActionsClusterEnableActionParams *)params
             completionHandler:(MTRStatusCompletion)completionHandler
{
    [self enableActionWithParams:params completion:completionHandler];
}
- (void)enableActionWithDurationWithParams:(MTRActionsClusterEnableActionWithDurationParams *)params
                         completionHandler:(MTRStatusCompletion)completionHandler
{
    [self enableActionWithDurationWithParams:params completion:completionHandler];
}
- (void)disableActionWithParams:(MTRActionsClusterDisableActionParams *)params
              completionHandler:(MTRStatusCompletion)completionHandler
{
    [self disableActionWithParams:params completion:completionHandler];
}
- (void)disableActionWithDurationWithParams:(MTRActionsClusterDisableActionWithDurationParams *)params
                          completionHandler:(MTRStatusCompletion)completionHandler
{
    [self disableActionWithDurationWithParams:params completion:completionHandler];
}

- (void)readAttributeActionListWithCompletionHandler:(void (^)(
                                                         NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeActionListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeActionListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                        maxInterval:(NSNumber * _Nonnull)maxInterval
                                             params:(MTRSubscribeParams * _Nullable)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                      reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeActionListWithParams:subscribeParams
                         subscriptionEstablished:subscriptionEstablishedHandler
                                   reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                       // Cast is safe because subclass does not add any selectors.
                                       reportHandler(static_cast<NSArray *>(value), error);
                                   }];
}
+ (void)readAttributeActionListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                completionHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeActionListWithClusterStateCache:attributeCacheContainer.realContainer
                                              endpoint:endpoint
                                                 queue:queue
                                            completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                completionHandler(static_cast<NSArray *>(value), error);
                                            }];
}

- (void)readAttributeEndpointListsWithCompletionHandler:(void (^)(
                                                            NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeEndpointListsWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeEndpointListsWithMinInterval:(NSNumber * _Nonnull)minInterval
                                           maxInterval:(NSNumber * _Nonnull)maxInterval
                                                params:(MTRSubscribeParams * _Nullable)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                         reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeEndpointListsWithParams:subscribeParams
                            subscriptionEstablished:subscriptionEstablishedHandler
                                      reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                          // Cast is safe because subclass does not add any selectors.
                                          reportHandler(static_cast<NSArray *>(value), error);
                                      }];
}
+ (void)readAttributeEndpointListsWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                   completionHandler:
                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeEndpointListsWithClusterStateCache:attributeCacheContainer.realContainer
                                                 endpoint:endpoint
                                                    queue:queue
                                               completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                   // Cast is safe because subclass does not add any selectors.
                                                   completionHandler(static_cast<NSArray *>(value), error);
                                               }];
}

- (void)readAttributeSetupURLWithCompletionHandler:(void (^)(
                                                       NSString * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeSetupURLWithCompletion:^(NSString * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSString *>(value), error);
    }];
}
- (void)subscribeAttributeSetupURLWithMinInterval:(NSNumber * _Nonnull)minInterval
                                      maxInterval:(NSNumber * _Nonnull)maxInterval
                                           params:(MTRSubscribeParams * _Nullable)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                    reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeSetupURLWithParams:subscribeParams
                       subscriptionEstablished:subscriptionEstablishedHandler
                                 reportHandler:^(NSString * _Nullable value, NSError * _Nullable error) {
                                     // Cast is safe because subclass does not add any selectors.
                                     reportHandler(static_cast<NSString *>(value), error);
                                 }];
}
+ (void)readAttributeSetupURLWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                       endpoint:(NSNumber *)endpoint
                                          queue:(dispatch_queue_t)queue
                              completionHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeSetupURLWithClusterStateCache:attributeCacheContainer.realContainer
                                            endpoint:endpoint
                                               queue:queue
                                          completion:^(NSString * _Nullable value, NSError * _Nullable error) {
                                              // Cast is safe because subclass does not add any selectors.
                                              completionHandler(static_cast<NSString *>(value), error);
                                          }];
}

- (void)readAttributeGeneratedCommandListWithCompletionHandler:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))completionHandler
{
    [self readAttributeGeneratedCommandListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeGeneratedCommandListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                  maxInterval:(NSNumber * _Nonnull)maxInterval
                                                       params:(MTRSubscribeParams * _Nullable)params
                                      subscriptionEstablished:
                                          (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                                reportHandler:
                                                    (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeGeneratedCommandListWithParams:subscribeParams
                                   subscriptionEstablished:subscriptionEstablishedHandler
                                             reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                 // Cast is safe because subclass does not add any selectors.
                                                 reportHandler(static_cast<NSArray *>(value), error);
                                             }];
}
+ (void)readAttributeGeneratedCommandListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                          completionHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeGeneratedCommandListWithClusterStateCache:attributeCacheContainer.realContainer
                                                        endpoint:endpoint
                                                           queue:queue
                                                      completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                          // Cast is safe because subclass does not add any selectors.
                                                          completionHandler(static_cast<NSArray *>(value), error);
                                                      }];
}

- (void)readAttributeAcceptedCommandListWithCompletionHandler:(void (^)(NSArray * _Nullable value,
                                                                  NSError * _Nullable error))completionHandler
{
    [self readAttributeAcceptedCommandListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeAcceptedCommandListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                 maxInterval:(NSNumber * _Nonnull)maxInterval
                                                      params:(MTRSubscribeParams * _Nullable)params
                                     subscriptionEstablished:
                                         (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                               reportHandler:
                                                   (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeAcceptedCommandListWithParams:subscribeParams
                                  subscriptionEstablished:subscriptionEstablishedHandler
                                            reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                reportHandler(static_cast<NSArray *>(value), error);
                                            }];
}
+ (void)readAttributeAcceptedCommandListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                         completionHandler:
                                             (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAcceptedCommandListWithClusterStateCache:attributeCacheContainer.realContainer
                                                       endpoint:endpoint
                                                          queue:queue
                                                     completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                         // Cast is safe because subclass does not add any selectors.
                                                         completionHandler(static_cast<NSArray *>(value), error);
                                                     }];
}

- (void)readAttributeAttributeListWithCompletionHandler:(void (^)(
                                                            NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAttributeListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeAttributeListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                           maxInterval:(NSNumber * _Nonnull)maxInterval
                                                params:(MTRSubscribeParams * _Nullable)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                         reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeAttributeListWithParams:subscribeParams
                            subscriptionEstablished:subscriptionEstablishedHandler
                                      reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                          // Cast is safe because subclass does not add any selectors.
                                          reportHandler(static_cast<NSArray *>(value), error);
                                      }];
}
+ (void)readAttributeAttributeListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                   completionHandler:
                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAttributeListWithClusterStateCache:attributeCacheContainer.realContainer
                                                 endpoint:endpoint
                                                    queue:queue
                                               completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                   // Cast is safe because subclass does not add any selectors.
                                                   completionHandler(static_cast<NSArray *>(value), error);
                                               }];
}

- (void)readAttributeFeatureMapWithCompletionHandler:(void (^)(
                                                         NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeFeatureMapWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeFeatureMapWithMinInterval:(NSNumber * _Nonnull)minInterval
                                        maxInterval:(NSNumber * _Nonnull)maxInterval
                                             params:(MTRSubscribeParams * _Nullable)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeFeatureMapWithParams:subscribeParams
                         subscriptionEstablished:subscriptionEstablishedHandler
                                   reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                       // Cast is safe because subclass does not add any selectors.
                                       reportHandler(static_cast<NSNumber *>(value), error);
                                   }];
}
+ (void)readAttributeFeatureMapWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                completionHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeFeatureMapWithClusterStateCache:attributeCacheContainer.realContainer
                                              endpoint:endpoint
                                                 queue:queue
                                            completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                completionHandler(static_cast<NSNumber *>(value), error);
                                            }];
}

- (void)readAttributeClusterRevisionWithCompletionHandler:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completionHandler
{
    [self readAttributeClusterRevisionWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeClusterRevisionWithMinInterval:(NSNumber * _Nonnull)minInterval
                                             maxInterval:(NSNumber * _Nonnull)maxInterval
                                                  params:(MTRSubscribeParams * _Nullable)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                           reportHandler:
                                               (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeClusterRevisionWithParams:subscribeParams
                              subscriptionEstablished:subscriptionEstablishedHandler
                                        reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                            // Cast is safe because subclass does not add any selectors.
                                            reportHandler(static_cast<NSNumber *>(value), error);
                                        }];
}
+ (void)readAttributeClusterRevisionWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                     completionHandler:
                                         (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeClusterRevisionWithClusterStateCache:attributeCacheContainer.realContainer
                                                   endpoint:endpoint
                                                      queue:queue
                                                 completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                     // Cast is safe because subclass does not add any selectors.
                                                     completionHandler(static_cast<NSNumber *>(value), error);
                                                 }];
}

- (nullable instancetype)initWithDevice:(MTRBaseDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

@end

@implementation MTRBaseClusterBasicInformation

- (instancetype)initWithDevice:(MTRBaseDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _device = device;
        _endpoint = [endpointID unsignedShortValue];
    }
    return self;
}

- (void)mfgSpecificPingWithCompletion:(MTRStatusCompletion)completion
{
    [self mfgSpecificPingWithParams:nil completion:completion];
}
- (void)mfgSpecificPingWithParams:(MTRBasicClusterMfgSpecificPingParams * _Nullable)params
                       completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            BasicInformation::Commands::MfgSpecificPing::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)readAttributeDataModelRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BasicInformation::Attributes::DataModelRevision::TypeInfo;
    return MTRReadAttribute<MTRInt16uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeDataModelRevisionWithParams:(MTRSubscribeParams * _Nonnull)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BasicInformation::Attributes::DataModelRevision::TypeInfo;
    MTRSubscribeAttribute<MTRInt16uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeDataModelRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                                 completion:
                                                     (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt16uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int16uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BasicInformation::Attributes::DataModelRevision::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeVendorNameWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BasicInformation::Attributes::VendorName::TypeInfo;
    return MTRReadAttribute<MTRCharStringAttributeCallbackBridge, NSString, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeVendorNameWithParams:(MTRSubscribeParams * _Nonnull)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BasicInformation::Attributes::VendorName::TypeInfo;
    MTRSubscribeAttribute<MTRCharStringAttributeCallbackSubscriptionBridge, NSString, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeVendorNameWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRCharStringAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(CharStringAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BasicInformation::Attributes::VendorName::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeVendorIDWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BasicInformation::Attributes::VendorID::TypeInfo;
    return MTRReadAttribute<MTRVendorIdAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeVendorIDWithParams:(MTRSubscribeParams * _Nonnull)params
                     subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                               reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BasicInformation::Attributes::VendorID::TypeInfo;
    MTRSubscribeAttribute<MTRVendorIdAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeVendorIDWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                          endpoint:(NSNumber *)endpoint
                                             queue:(dispatch_queue_t)queue
                                        completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRVendorIdAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(VendorIdAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BasicInformation::Attributes::VendorID::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeProductNameWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BasicInformation::Attributes::ProductName::TypeInfo;
    return MTRReadAttribute<MTRCharStringAttributeCallbackBridge, NSString, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeProductNameWithParams:(MTRSubscribeParams * _Nonnull)params
                        subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                  reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BasicInformation::Attributes::ProductName::TypeInfo;
    MTRSubscribeAttribute<MTRCharStringAttributeCallbackSubscriptionBridge, NSString, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeProductNameWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                             endpoint:(NSNumber *)endpoint
                                                queue:(dispatch_queue_t)queue
                                           completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRCharStringAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(CharStringAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BasicInformation::Attributes::ProductName::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeProductIDWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BasicInformation::Attributes::ProductID::TypeInfo;
    return MTRReadAttribute<MTRInt16uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeProductIDWithParams:(MTRSubscribeParams * _Nonnull)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BasicInformation::Attributes::ProductID::TypeInfo;
    MTRSubscribeAttribute<MTRInt16uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeProductIDWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt16uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int16uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BasicInformation::Attributes::ProductID::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeNodeLabelWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BasicInformation::Attributes::NodeLabel::TypeInfo;
    return MTRReadAttribute<MTRCharStringAttributeCallbackBridge, NSString, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)writeAttributeNodeLabelWithValue:(NSString * _Nonnull)value completion:(MTRStatusCompletion)completion
{
    [self writeAttributeNodeLabelWithValue:(NSString * _Nonnull) value params:nil completion:completion];
}
- (void)writeAttributeNodeLabelWithValue:(NSString * _Nonnull)value
                                  params:(MTRWriteParams * _Nullable)params
                              completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    value = [value copy];

    auto * bridge = new MTRDefaultSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable ignored, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, DefaultSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            chip::Optional<uint16_t> timedWriteTimeout;
            if (params != nil) {
                if (params.timedWriteTimeout != nil) {
                    timedWriteTimeout.SetValue(params.timedWriteTimeout.unsignedShortValue);
                }
            }

            ListFreer listFreer;
            using TypeInfo = BasicInformation::Attributes::NodeLabel::TypeInfo;
            TypeInfo::Type cppValue;
            cppValue = [self asCharSpan:value];

            chip::Controller::ClusterBase cppCluster(exchangeManager, session, self->_endpoint);
            return cppCluster.WriteAttribute<TypeInfo>(cppValue, bridge, successCb, failureCb, timedWriteTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)subscribeAttributeNodeLabelWithParams:(MTRSubscribeParams * _Nonnull)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BasicInformation::Attributes::NodeLabel::TypeInfo;
    MTRSubscribeAttribute<MTRCharStringAttributeCallbackSubscriptionBridge, NSString, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeNodeLabelWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRCharStringAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(CharStringAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BasicInformation::Attributes::NodeLabel::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeLocationWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BasicInformation::Attributes::Location::TypeInfo;
    return MTRReadAttribute<MTRCharStringAttributeCallbackBridge, NSString, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)writeAttributeLocationWithValue:(NSString * _Nonnull)value completion:(MTRStatusCompletion)completion
{
    [self writeAttributeLocationWithValue:(NSString * _Nonnull) value params:nil completion:completion];
}
- (void)writeAttributeLocationWithValue:(NSString * _Nonnull)value
                                 params:(MTRWriteParams * _Nullable)params
                             completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    value = [value copy];

    auto * bridge = new MTRDefaultSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable ignored, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, DefaultSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            chip::Optional<uint16_t> timedWriteTimeout;
            if (params != nil) {
                if (params.timedWriteTimeout != nil) {
                    timedWriteTimeout.SetValue(params.timedWriteTimeout.unsignedShortValue);
                }
            }

            ListFreer listFreer;
            using TypeInfo = BasicInformation::Attributes::Location::TypeInfo;
            TypeInfo::Type cppValue;
            cppValue = [self asCharSpan:value];

            chip::Controller::ClusterBase cppCluster(exchangeManager, session, self->_endpoint);
            return cppCluster.WriteAttribute<TypeInfo>(cppValue, bridge, successCb, failureCb, timedWriteTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)subscribeAttributeLocationWithParams:(MTRSubscribeParams * _Nonnull)params
                     subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                               reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BasicInformation::Attributes::Location::TypeInfo;
    MTRSubscribeAttribute<MTRCharStringAttributeCallbackSubscriptionBridge, NSString, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeLocationWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                          endpoint:(NSNumber *)endpoint
                                             queue:(dispatch_queue_t)queue
                                        completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRCharStringAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(CharStringAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BasicInformation::Attributes::Location::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeHardwareVersionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BasicInformation::Attributes::HardwareVersion::TypeInfo;
    return MTRReadAttribute<MTRInt16uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeHardwareVersionWithParams:(MTRSubscribeParams * _Nonnull)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BasicInformation::Attributes::HardwareVersion::TypeInfo;
    MTRSubscribeAttribute<MTRInt16uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeHardwareVersionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt16uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int16uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BasicInformation::Attributes::HardwareVersion::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeHardwareVersionStringWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BasicInformation::Attributes::HardwareVersionString::TypeInfo;
    return MTRReadAttribute<MTRCharStringAttributeCallbackBridge, NSString, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeHardwareVersionStringWithParams:(MTRSubscribeParams * _Nonnull)params
                                  subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                            reportHandler:
                                                (void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BasicInformation::Attributes::HardwareVersionString::TypeInfo;
    MTRSubscribeAttribute<MTRCharStringAttributeCallbackSubscriptionBridge, NSString, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeHardwareVersionStringWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                       endpoint:(NSNumber *)endpoint
                                                          queue:(dispatch_queue_t)queue
                                                     completion:
                                                         (void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRCharStringAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(CharStringAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BasicInformation::Attributes::HardwareVersionString::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeSoftwareVersionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BasicInformation::Attributes::SoftwareVersion::TypeInfo;
    return MTRReadAttribute<MTRInt32uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeSoftwareVersionWithParams:(MTRSubscribeParams * _Nonnull)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BasicInformation::Attributes::SoftwareVersion::TypeInfo;
    MTRSubscribeAttribute<MTRInt32uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeSoftwareVersionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt32uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int32uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BasicInformation::Attributes::SoftwareVersion::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeSoftwareVersionStringWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BasicInformation::Attributes::SoftwareVersionString::TypeInfo;
    return MTRReadAttribute<MTRCharStringAttributeCallbackBridge, NSString, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeSoftwareVersionStringWithParams:(MTRSubscribeParams * _Nonnull)params
                                  subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                            reportHandler:
                                                (void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BasicInformation::Attributes::SoftwareVersionString::TypeInfo;
    MTRSubscribeAttribute<MTRCharStringAttributeCallbackSubscriptionBridge, NSString, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeSoftwareVersionStringWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                       endpoint:(NSNumber *)endpoint
                                                          queue:(dispatch_queue_t)queue
                                                     completion:
                                                         (void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRCharStringAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(CharStringAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BasicInformation::Attributes::SoftwareVersionString::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeManufacturingDateWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BasicInformation::Attributes::ManufacturingDate::TypeInfo;
    return MTRReadAttribute<MTRCharStringAttributeCallbackBridge, NSString, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeManufacturingDateWithParams:(MTRSubscribeParams * _Nonnull)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BasicInformation::Attributes::ManufacturingDate::TypeInfo;
    MTRSubscribeAttribute<MTRCharStringAttributeCallbackSubscriptionBridge, NSString, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeManufacturingDateWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                                 completion:
                                                     (void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRCharStringAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(CharStringAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BasicInformation::Attributes::ManufacturingDate::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributePartNumberWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BasicInformation::Attributes::PartNumber::TypeInfo;
    return MTRReadAttribute<MTRCharStringAttributeCallbackBridge, NSString, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributePartNumberWithParams:(MTRSubscribeParams * _Nonnull)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BasicInformation::Attributes::PartNumber::TypeInfo;
    MTRSubscribeAttribute<MTRCharStringAttributeCallbackSubscriptionBridge, NSString, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributePartNumberWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRCharStringAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(CharStringAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BasicInformation::Attributes::PartNumber::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeProductURLWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BasicInformation::Attributes::ProductURL::TypeInfo;
    return MTRReadAttribute<MTRCharStringAttributeCallbackBridge, NSString, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeProductURLWithParams:(MTRSubscribeParams * _Nonnull)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BasicInformation::Attributes::ProductURL::TypeInfo;
    MTRSubscribeAttribute<MTRCharStringAttributeCallbackSubscriptionBridge, NSString, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeProductURLWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRCharStringAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(CharStringAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BasicInformation::Attributes::ProductURL::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeProductLabelWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BasicInformation::Attributes::ProductLabel::TypeInfo;
    return MTRReadAttribute<MTRCharStringAttributeCallbackBridge, NSString, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeProductLabelWithParams:(MTRSubscribeParams * _Nonnull)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BasicInformation::Attributes::ProductLabel::TypeInfo;
    MTRSubscribeAttribute<MTRCharStringAttributeCallbackSubscriptionBridge, NSString, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeProductLabelWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRCharStringAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(CharStringAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BasicInformation::Attributes::ProductLabel::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeSerialNumberWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BasicInformation::Attributes::SerialNumber::TypeInfo;
    return MTRReadAttribute<MTRCharStringAttributeCallbackBridge, NSString, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeSerialNumberWithParams:(MTRSubscribeParams * _Nonnull)params
                         subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                   reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BasicInformation::Attributes::SerialNumber::TypeInfo;
    MTRSubscribeAttribute<MTRCharStringAttributeCallbackSubscriptionBridge, NSString, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeSerialNumberWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                            completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRCharStringAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(CharStringAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BasicInformation::Attributes::SerialNumber::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeLocalConfigDisabledWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BasicInformation::Attributes::LocalConfigDisabled::TypeInfo;
    return MTRReadAttribute<MTRBooleanAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)writeAttributeLocalConfigDisabledWithValue:(NSNumber * _Nonnull)value completion:(MTRStatusCompletion)completion
{
    [self writeAttributeLocalConfigDisabledWithValue:(NSNumber * _Nonnull) value params:nil completion:completion];
}
- (void)writeAttributeLocalConfigDisabledWithValue:(NSNumber * _Nonnull)value
                                            params:(MTRWriteParams * _Nullable)params
                                        completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    value = [value copy];

    auto * bridge = new MTRDefaultSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable ignored, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, DefaultSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            chip::Optional<uint16_t> timedWriteTimeout;
            if (params != nil) {
                if (params.timedWriteTimeout != nil) {
                    timedWriteTimeout.SetValue(params.timedWriteTimeout.unsignedShortValue);
                }
            }

            ListFreer listFreer;
            using TypeInfo = BasicInformation::Attributes::LocalConfigDisabled::TypeInfo;
            TypeInfo::Type cppValue;
            cppValue = value.boolValue;

            chip::Controller::ClusterBase cppCluster(exchangeManager, session, self->_endpoint);
            return cppCluster.WriteAttribute<TypeInfo>(cppValue, bridge, successCb, failureCb, timedWriteTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)subscribeAttributeLocalConfigDisabledWithParams:(MTRSubscribeParams * _Nonnull)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BasicInformation::Attributes::LocalConfigDisabled::TypeInfo;
    MTRSubscribeAttribute<MTRBooleanAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeLocalConfigDisabledWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRBooleanAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(BooleanAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BasicInformation::Attributes::LocalConfigDisabled::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeReachableWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BasicInformation::Attributes::Reachable::TypeInfo;
    return MTRReadAttribute<MTRBooleanAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeReachableWithParams:(MTRSubscribeParams * _Nonnull)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BasicInformation::Attributes::Reachable::TypeInfo;
    MTRSubscribeAttribute<MTRBooleanAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeReachableWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRBooleanAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(BooleanAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BasicInformation::Attributes::Reachable::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeUniqueIDWithCompletion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BasicInformation::Attributes::UniqueID::TypeInfo;
    return MTRReadAttribute<MTRCharStringAttributeCallbackBridge, NSString, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeUniqueIDWithParams:(MTRSubscribeParams * _Nonnull)params
                     subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                               reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BasicInformation::Attributes::UniqueID::TypeInfo;
    MTRSubscribeAttribute<MTRCharStringAttributeCallbackSubscriptionBridge, NSString, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeUniqueIDWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                          endpoint:(NSNumber *)endpoint
                                             queue:(dispatch_queue_t)queue
                                        completion:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRCharStringAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(CharStringAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BasicInformation::Attributes::UniqueID::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeCapabilityMinimaWithCompletion:(void (^)(MTRBasicInformationClusterCapabilityMinimaStruct * _Nullable value,
                                                        NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BasicInformation::Attributes::CapabilityMinima::TypeInfo;
    return MTRReadAttribute<MTRBasicInformationCapabilityMinimaStructAttributeCallbackBridge,
        MTRBasicInformationClusterCapabilityMinimaStruct, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeCapabilityMinimaWithParams:(MTRSubscribeParams * _Nonnull)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                       reportHandler:(void (^)(MTRBasicInformationClusterCapabilityMinimaStruct * _Nullable value,
                                                         NSError * _Nullable error))reportHandler
{
    using TypeInfo = BasicInformation::Attributes::CapabilityMinima::TypeInfo;
    MTRSubscribeAttribute<MTRBasicInformationCapabilityMinimaStructAttributeCallbackSubscriptionBridge,
        MTRBasicInformationClusterCapabilityMinimaStruct, TypeInfo::DecodableType>(params, subscriptionEstablished, reportHandler,
        self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

+ (void)readAttributeCapabilityMinimaWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                                completion:
                                                    (void (^)(MTRBasicInformationClusterCapabilityMinimaStruct * _Nullable value,
                                                        NSError * _Nullable error))completion
{
    auto * bridge = new MTRBasicInformationCapabilityMinimaStructAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(BasicInformationCapabilityMinimaStructAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BasicInformation::Attributes::CapabilityMinima::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeProductAppearanceWithCompletion:(void (^)(MTRBasicInformationClusterProductAppearanceStruct * _Nullable value,
                                                         NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BasicInformation::Attributes::ProductAppearance::TypeInfo;
    return MTRReadAttribute<MTRBasicInformationProductAppearanceStructAttributeCallbackBridge,
        MTRBasicInformationClusterProductAppearanceStruct, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeProductAppearanceWithParams:(MTRSubscribeParams * _Nonnull)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                        reportHandler:(void (^)(MTRBasicInformationClusterProductAppearanceStruct * _Nullable value,
                                                          NSError * _Nullable error))reportHandler
{
    using TypeInfo = BasicInformation::Attributes::ProductAppearance::TypeInfo;
    MTRSubscribeAttribute<MTRBasicInformationProductAppearanceStructAttributeCallbackSubscriptionBridge,
        MTRBasicInformationClusterProductAppearanceStruct, TypeInfo::DecodableType>(params, subscriptionEstablished, reportHandler,
        self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

+ (void)readAttributeProductAppearanceWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                                 completion:
                                                     (void (^)(MTRBasicInformationClusterProductAppearanceStruct * _Nullable value,
                                                         NSError * _Nullable error))completion
{
    auto * bridge = new MTRBasicInformationProductAppearanceStructAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(BasicInformationProductAppearanceStructAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BasicInformation::Attributes::ProductAppearance::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BasicInformation::Attributes::GeneratedCommandList::TypeInfo;
    return MTRReadAttribute<MTRBasicInformationGeneratedCommandListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams * _Nonnull)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BasicInformation::Attributes::GeneratedCommandList::TypeInfo;
    MTRSubscribeAttribute<MTRBasicInformationGeneratedCommandListListAttributeCallbackSubscriptionBridge, NSArray,
        TypeInfo::DecodableType>(params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint,
        TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRBasicInformationGeneratedCommandListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(BasicInformationGeneratedCommandListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BasicInformation::Attributes::GeneratedCommandList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BasicInformation::Attributes::AcceptedCommandList::TypeInfo;
    return MTRReadAttribute<MTRBasicInformationAcceptedCommandListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams * _Nonnull)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BasicInformation::Attributes::AcceptedCommandList::TypeInfo;
    MTRSubscribeAttribute<MTRBasicInformationAcceptedCommandListListAttributeCallbackSubscriptionBridge, NSArray,
        TypeInfo::DecodableType>(params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint,
        TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

+ (void)readAttributeAcceptedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                     endpoint:(NSNumber *)endpoint
                                                        queue:(dispatch_queue_t)queue
                                                   completion:
                                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRBasicInformationAcceptedCommandListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(BasicInformationAcceptedCommandListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BasicInformation::Attributes::AcceptedCommandList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeEventListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BasicInformation::Attributes::EventList::TypeInfo;
    return MTRReadAttribute<MTRBasicInformationEventListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeEventListWithParams:(MTRSubscribeParams * _Nonnull)params
                      subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BasicInformation::Attributes::EventList::TypeInfo;
    MTRSubscribeAttribute<MTRBasicInformationEventListListAttributeCallbackSubscriptionBridge, NSArray, TypeInfo::DecodableType>(
        params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeEventListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                         completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRBasicInformationEventListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(BasicInformationEventListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BasicInformation::Attributes::EventList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeAttributeListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BasicInformation::Attributes::AttributeList::TypeInfo;
    return MTRReadAttribute<MTRBasicInformationAttributeListListAttributeCallbackBridge, NSArray, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeAttributeListWithParams:(MTRSubscribeParams * _Nonnull)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                    reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BasicInformation::Attributes::AttributeList::TypeInfo;
    MTRSubscribeAttribute<MTRBasicInformationAttributeListListAttributeCallbackSubscriptionBridge, NSArray,
        TypeInfo::DecodableType>(params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint,
        TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

+ (void)readAttributeAttributeListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                             completion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRBasicInformationAttributeListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(BasicInformationAttributeListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BasicInformation::Attributes::AttributeList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeFeatureMapWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BasicInformation::Attributes::FeatureMap::TypeInfo;
    return MTRReadAttribute<MTRInt32uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeFeatureMapWithParams:(MTRSubscribeParams * _Nonnull)params
                       subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                 reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BasicInformation::Attributes::FeatureMap::TypeInfo;
    MTRSubscribeAttribute<MTRInt32uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeFeatureMapWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                          completion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt32uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int32uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BasicInformation::Attributes::FeatureMap::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeClusterRevisionWithCompletion:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = BasicInformation::Attributes::ClusterRevision::TypeInfo;
    return MTRReadAttribute<MTRInt16uAttributeCallbackBridge, NSNumber, TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeClusterRevisionWithParams:(MTRSubscribeParams * _Nonnull)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = BasicInformation::Attributes::ClusterRevision::TypeInfo;
    MTRSubscribeAttribute<MTRInt16uAttributeCallbackSubscriptionBridge, NSNumber, TypeInfo::DecodableType>(params,
        subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(),
        TypeInfo::GetAttributeId());
}

+ (void)readAttributeClusterRevisionWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                 endpoint:(NSNumber *)endpoint
                                                    queue:(dispatch_queue_t)queue
                                               completion:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTRInt16uAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(
        clusterStateCacheContainer.baseDevice, ^(Int16uAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = BasicInformation::Attributes::ClusterRevision::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

@end

@implementation MTRBaseClusterBasic
@end

@implementation MTRBaseClusterBasic (Deprecated)

- (void)mfgSpecificPingWithParams:(MTRBasicClusterMfgSpecificPingParams * _Nullable)params
                completionHandler:(MTRStatusCompletion)completionHandler
{
    [self mfgSpecificPingWithParams:params completion:completionHandler];
}
- (void)mfgSpecificPingWithCompletionHandler:(MTRStatusCompletion)completionHandler
{
    [self mfgSpecificPingWithParams:nil completionHandler:completionHandler];
}

- (void)readAttributeDataModelRevisionWithCompletionHandler:(void (^)(NSNumber * _Nullable value,
                                                                NSError * _Nullable error))completionHandler
{
    [self readAttributeDataModelRevisionWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeDataModelRevisionWithMinInterval:(NSNumber * _Nonnull)minInterval
                                               maxInterval:(NSNumber * _Nonnull)maxInterval
                                                    params:(MTRSubscribeParams * _Nullable)params
                                   subscriptionEstablished:
                                       (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                             reportHandler:
                                                 (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeDataModelRevisionWithParams:subscribeParams
                                subscriptionEstablished:subscriptionEstablishedHandler
                                          reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                              // Cast is safe because subclass does not add any selectors.
                                              reportHandler(static_cast<NSNumber *>(value), error);
                                          }];
}
+ (void)readAttributeDataModelRevisionWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                endpoint:(NSNumber *)endpoint
                                                   queue:(dispatch_queue_t)queue
                                       completionHandler:
                                           (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeDataModelRevisionWithClusterStateCache:attributeCacheContainer.realContainer
                                                     endpoint:endpoint
                                                        queue:queue
                                                   completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                       // Cast is safe because subclass does not add any selectors.
                                                       completionHandler(static_cast<NSNumber *>(value), error);
                                                   }];
}

- (void)readAttributeVendorNameWithCompletionHandler:(void (^)(
                                                         NSString * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeVendorNameWithCompletion:^(NSString * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSString *>(value), error);
    }];
}
- (void)subscribeAttributeVendorNameWithMinInterval:(NSNumber * _Nonnull)minInterval
                                        maxInterval:(NSNumber * _Nonnull)maxInterval
                                             params:(MTRSubscribeParams * _Nullable)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                      reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeVendorNameWithParams:subscribeParams
                         subscriptionEstablished:subscriptionEstablishedHandler
                                   reportHandler:^(NSString * _Nullable value, NSError * _Nullable error) {
                                       // Cast is safe because subclass does not add any selectors.
                                       reportHandler(static_cast<NSString *>(value), error);
                                   }];
}
+ (void)readAttributeVendorNameWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                completionHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeVendorNameWithClusterStateCache:attributeCacheContainer.realContainer
                                              endpoint:endpoint
                                                 queue:queue
                                            completion:^(NSString * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                completionHandler(static_cast<NSString *>(value), error);
                                            }];
}

- (void)readAttributeVendorIDWithCompletionHandler:(void (^)(
                                                       NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeVendorIDWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeVendorIDWithMinInterval:(NSNumber * _Nonnull)minInterval
                                      maxInterval:(NSNumber * _Nonnull)maxInterval
                                           params:(MTRSubscribeParams * _Nullable)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                    reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeVendorIDWithParams:subscribeParams
                       subscriptionEstablished:subscriptionEstablishedHandler
                                 reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                     // Cast is safe because subclass does not add any selectors.
                                     reportHandler(static_cast<NSNumber *>(value), error);
                                 }];
}
+ (void)readAttributeVendorIDWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                       endpoint:(NSNumber *)endpoint
                                          queue:(dispatch_queue_t)queue
                              completionHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeVendorIDWithClusterStateCache:attributeCacheContainer.realContainer
                                            endpoint:endpoint
                                               queue:queue
                                          completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                              // Cast is safe because subclass does not add any selectors.
                                              completionHandler(static_cast<NSNumber *>(value), error);
                                          }];
}

- (void)readAttributeProductNameWithCompletionHandler:(void (^)(
                                                          NSString * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeProductNameWithCompletion:^(NSString * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSString *>(value), error);
    }];
}
- (void)subscribeAttributeProductNameWithMinInterval:(NSNumber * _Nonnull)minInterval
                                         maxInterval:(NSNumber * _Nonnull)maxInterval
                                              params:(MTRSubscribeParams * _Nullable)params
                             subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                       reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeProductNameWithParams:subscribeParams
                          subscriptionEstablished:subscriptionEstablishedHandler
                                    reportHandler:^(NSString * _Nullable value, NSError * _Nullable error) {
                                        // Cast is safe because subclass does not add any selectors.
                                        reportHandler(static_cast<NSString *>(value), error);
                                    }];
}
+ (void)readAttributeProductNameWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                          endpoint:(NSNumber *)endpoint
                                             queue:(dispatch_queue_t)queue
                                 completionHandler:
                                     (void (^)(NSString * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeProductNameWithClusterStateCache:attributeCacheContainer.realContainer
                                               endpoint:endpoint
                                                  queue:queue
                                             completion:^(NSString * _Nullable value, NSError * _Nullable error) {
                                                 // Cast is safe because subclass does not add any selectors.
                                                 completionHandler(static_cast<NSString *>(value), error);
                                             }];
}

- (void)readAttributeProductIDWithCompletionHandler:(void (^)(
                                                        NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeProductIDWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeProductIDWithMinInterval:(NSNumber * _Nonnull)minInterval
                                       maxInterval:(NSNumber * _Nonnull)maxInterval
                                            params:(MTRSubscribeParams * _Nullable)params
                           subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                     reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeProductIDWithParams:subscribeParams
                        subscriptionEstablished:subscriptionEstablishedHandler
                                  reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                      // Cast is safe because subclass does not add any selectors.
                                      reportHandler(static_cast<NSNumber *>(value), error);
                                  }];
}
+ (void)readAttributeProductIDWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                        endpoint:(NSNumber *)endpoint
                                           queue:(dispatch_queue_t)queue
                               completionHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeProductIDWithClusterStateCache:attributeCacheContainer.realContainer
                                             endpoint:endpoint
                                                queue:queue
                                           completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                               // Cast is safe because subclass does not add any selectors.
                                               completionHandler(static_cast<NSNumber *>(value), error);
                                           }];
}

- (void)readAttributeNodeLabelWithCompletionHandler:(void (^)(
                                                        NSString * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeNodeLabelWithCompletion:^(NSString * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSString *>(value), error);
    }];
}
- (void)writeAttributeNodeLabelWithValue:(NSString * _Nonnull)value completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeNodeLabelWithValue:value params:nil completion:completionHandler];
}
- (void)writeAttributeNodeLabelWithValue:(NSString * _Nonnull)value
                                  params:(MTRWriteParams * _Nullable)params
                       completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeNodeLabelWithValue:value params:params completion:completionHandler];
}
- (void)subscribeAttributeNodeLabelWithMinInterval:(NSNumber * _Nonnull)minInterval
                                       maxInterval:(NSNumber * _Nonnull)maxInterval
                                            params:(MTRSubscribeParams * _Nullable)params
                           subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                     reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeNodeLabelWithParams:subscribeParams
                        subscriptionEstablished:subscriptionEstablishedHandler
                                  reportHandler:^(NSString * _Nullable value, NSError * _Nullable error) {
                                      // Cast is safe because subclass does not add any selectors.
                                      reportHandler(static_cast<NSString *>(value), error);
                                  }];
}
+ (void)readAttributeNodeLabelWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                        endpoint:(NSNumber *)endpoint
                                           queue:(dispatch_queue_t)queue
                               completionHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeNodeLabelWithClusterStateCache:attributeCacheContainer.realContainer
                                             endpoint:endpoint
                                                queue:queue
                                           completion:^(NSString * _Nullable value, NSError * _Nullable error) {
                                               // Cast is safe because subclass does not add any selectors.
                                               completionHandler(static_cast<NSString *>(value), error);
                                           }];
}

- (void)readAttributeLocationWithCompletionHandler:(void (^)(
                                                       NSString * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeLocationWithCompletion:^(NSString * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSString *>(value), error);
    }];
}
- (void)writeAttributeLocationWithValue:(NSString * _Nonnull)value completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeLocationWithValue:value params:nil completion:completionHandler];
}
- (void)writeAttributeLocationWithValue:(NSString * _Nonnull)value
                                 params:(MTRWriteParams * _Nullable)params
                      completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeLocationWithValue:value params:params completion:completionHandler];
}
- (void)subscribeAttributeLocationWithMinInterval:(NSNumber * _Nonnull)minInterval
                                      maxInterval:(NSNumber * _Nonnull)maxInterval
                                           params:(MTRSubscribeParams * _Nullable)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                    reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeLocationWithParams:subscribeParams
                       subscriptionEstablished:subscriptionEstablishedHandler
                                 reportHandler:^(NSString * _Nullable value, NSError * _Nullable error) {
                                     // Cast is safe because subclass does not add any selectors.
                                     reportHandler(static_cast<NSString *>(value), error);
                                 }];
}
+ (void)readAttributeLocationWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                       endpoint:(NSNumber *)endpoint
                                          queue:(dispatch_queue_t)queue
                              completionHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeLocationWithClusterStateCache:attributeCacheContainer.realContainer
                                            endpoint:endpoint
                                               queue:queue
                                          completion:^(NSString * _Nullable value, NSError * _Nullable error) {
                                              // Cast is safe because subclass does not add any selectors.
                                              completionHandler(static_cast<NSString *>(value), error);
                                          }];
}

- (void)readAttributeHardwareVersionWithCompletionHandler:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completionHandler
{
    [self readAttributeHardwareVersionWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeHardwareVersionWithMinInterval:(NSNumber * _Nonnull)minInterval
                                             maxInterval:(NSNumber * _Nonnull)maxInterval
                                                  params:(MTRSubscribeParams * _Nullable)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                           reportHandler:
                                               (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeHardwareVersionWithParams:subscribeParams
                              subscriptionEstablished:subscriptionEstablishedHandler
                                        reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                            // Cast is safe because subclass does not add any selectors.
                                            reportHandler(static_cast<NSNumber *>(value), error);
                                        }];
}
+ (void)readAttributeHardwareVersionWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                     completionHandler:
                                         (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeHardwareVersionWithClusterStateCache:attributeCacheContainer.realContainer
                                                   endpoint:endpoint
                                                      queue:queue
                                                 completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                     // Cast is safe because subclass does not add any selectors.
                                                     completionHandler(static_cast<NSNumber *>(value), error);
                                                 }];
}

- (void)readAttributeHardwareVersionStringWithCompletionHandler:(void (^)(NSString * _Nullable value,
                                                                    NSError * _Nullable error))completionHandler
{
    [self readAttributeHardwareVersionStringWithCompletion:^(NSString * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSString *>(value), error);
    }];
}
- (void)subscribeAttributeHardwareVersionStringWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                   maxInterval:(NSNumber * _Nonnull)maxInterval
                                                        params:(MTRSubscribeParams * _Nullable)params
                                       subscriptionEstablished:
                                           (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                                 reportHandler:
                                                     (void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeHardwareVersionStringWithParams:subscribeParams
                                    subscriptionEstablished:subscriptionEstablishedHandler
                                              reportHandler:^(NSString * _Nullable value, NSError * _Nullable error) {
                                                  // Cast is safe because subclass does not add any selectors.
                                                  reportHandler(static_cast<NSString *>(value), error);
                                              }];
}
+ (void)readAttributeHardwareVersionStringWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                    endpoint:(NSNumber *)endpoint
                                                       queue:(dispatch_queue_t)queue
                                           completionHandler:
                                               (void (^)(NSString * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeHardwareVersionStringWithClusterStateCache:attributeCacheContainer.realContainer
                                                         endpoint:endpoint
                                                            queue:queue
                                                       completion:^(NSString * _Nullable value, NSError * _Nullable error) {
                                                           // Cast is safe because subclass does not add any selectors.
                                                           completionHandler(static_cast<NSString *>(value), error);
                                                       }];
}

- (void)readAttributeSoftwareVersionWithCompletionHandler:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completionHandler
{
    [self readAttributeSoftwareVersionWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeSoftwareVersionWithMinInterval:(NSNumber * _Nonnull)minInterval
                                             maxInterval:(NSNumber * _Nonnull)maxInterval
                                                  params:(MTRSubscribeParams * _Nullable)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                           reportHandler:
                                               (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeSoftwareVersionWithParams:subscribeParams
                              subscriptionEstablished:subscriptionEstablishedHandler
                                        reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                            // Cast is safe because subclass does not add any selectors.
                                            reportHandler(static_cast<NSNumber *>(value), error);
                                        }];
}
+ (void)readAttributeSoftwareVersionWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                     completionHandler:
                                         (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeSoftwareVersionWithClusterStateCache:attributeCacheContainer.realContainer
                                                   endpoint:endpoint
                                                      queue:queue
                                                 completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                     // Cast is safe because subclass does not add any selectors.
                                                     completionHandler(static_cast<NSNumber *>(value), error);
                                                 }];
}

- (void)readAttributeSoftwareVersionStringWithCompletionHandler:(void (^)(NSString * _Nullable value,
                                                                    NSError * _Nullable error))completionHandler
{
    [self readAttributeSoftwareVersionStringWithCompletion:^(NSString * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSString *>(value), error);
    }];
}
- (void)subscribeAttributeSoftwareVersionStringWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                   maxInterval:(NSNumber * _Nonnull)maxInterval
                                                        params:(MTRSubscribeParams * _Nullable)params
                                       subscriptionEstablished:
                                           (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                                 reportHandler:
                                                     (void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeSoftwareVersionStringWithParams:subscribeParams
                                    subscriptionEstablished:subscriptionEstablishedHandler
                                              reportHandler:^(NSString * _Nullable value, NSError * _Nullable error) {
                                                  // Cast is safe because subclass does not add any selectors.
                                                  reportHandler(static_cast<NSString *>(value), error);
                                              }];
}
+ (void)readAttributeSoftwareVersionStringWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                    endpoint:(NSNumber *)endpoint
                                                       queue:(dispatch_queue_t)queue
                                           completionHandler:
                                               (void (^)(NSString * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeSoftwareVersionStringWithClusterStateCache:attributeCacheContainer.realContainer
                                                         endpoint:endpoint
                                                            queue:queue
                                                       completion:^(NSString * _Nullable value, NSError * _Nullable error) {
                                                           // Cast is safe because subclass does not add any selectors.
                                                           completionHandler(static_cast<NSString *>(value), error);
                                                       }];
}

- (void)readAttributeManufacturingDateWithCompletionHandler:(void (^)(NSString * _Nullable value,
                                                                NSError * _Nullable error))completionHandler
{
    [self readAttributeManufacturingDateWithCompletion:^(NSString * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSString *>(value), error);
    }];
}
- (void)subscribeAttributeManufacturingDateWithMinInterval:(NSNumber * _Nonnull)minInterval
                                               maxInterval:(NSNumber * _Nonnull)maxInterval
                                                    params:(MTRSubscribeParams * _Nullable)params
                                   subscriptionEstablished:
                                       (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                             reportHandler:
                                                 (void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeManufacturingDateWithParams:subscribeParams
                                subscriptionEstablished:subscriptionEstablishedHandler
                                          reportHandler:^(NSString * _Nullable value, NSError * _Nullable error) {
                                              // Cast is safe because subclass does not add any selectors.
                                              reportHandler(static_cast<NSString *>(value), error);
                                          }];
}
+ (void)readAttributeManufacturingDateWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                endpoint:(NSNumber *)endpoint
                                                   queue:(dispatch_queue_t)queue
                                       completionHandler:
                                           (void (^)(NSString * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeManufacturingDateWithClusterStateCache:attributeCacheContainer.realContainer
                                                     endpoint:endpoint
                                                        queue:queue
                                                   completion:^(NSString * _Nullable value, NSError * _Nullable error) {
                                                       // Cast is safe because subclass does not add any selectors.
                                                       completionHandler(static_cast<NSString *>(value), error);
                                                   }];
}

- (void)readAttributePartNumberWithCompletionHandler:(void (^)(
                                                         NSString * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributePartNumberWithCompletion:^(NSString * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSString *>(value), error);
    }];
}
- (void)subscribeAttributePartNumberWithMinInterval:(NSNumber * _Nonnull)minInterval
                                        maxInterval:(NSNumber * _Nonnull)maxInterval
                                             params:(MTRSubscribeParams * _Nullable)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                      reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributePartNumberWithParams:subscribeParams
                         subscriptionEstablished:subscriptionEstablishedHandler
                                   reportHandler:^(NSString * _Nullable value, NSError * _Nullable error) {
                                       // Cast is safe because subclass does not add any selectors.
                                       reportHandler(static_cast<NSString *>(value), error);
                                   }];
}
+ (void)readAttributePartNumberWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                completionHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributePartNumberWithClusterStateCache:attributeCacheContainer.realContainer
                                              endpoint:endpoint
                                                 queue:queue
                                            completion:^(NSString * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                completionHandler(static_cast<NSString *>(value), error);
                                            }];
}

- (void)readAttributeProductURLWithCompletionHandler:(void (^)(
                                                         NSString * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeProductURLWithCompletion:^(NSString * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSString *>(value), error);
    }];
}
- (void)subscribeAttributeProductURLWithMinInterval:(NSNumber * _Nonnull)minInterval
                                        maxInterval:(NSNumber * _Nonnull)maxInterval
                                             params:(MTRSubscribeParams * _Nullable)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                      reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeProductURLWithParams:subscribeParams
                         subscriptionEstablished:subscriptionEstablishedHandler
                                   reportHandler:^(NSString * _Nullable value, NSError * _Nullable error) {
                                       // Cast is safe because subclass does not add any selectors.
                                       reportHandler(static_cast<NSString *>(value), error);
                                   }];
}
+ (void)readAttributeProductURLWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                completionHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeProductURLWithClusterStateCache:attributeCacheContainer.realContainer
                                              endpoint:endpoint
                                                 queue:queue
                                            completion:^(NSString * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                completionHandler(static_cast<NSString *>(value), error);
                                            }];
}

- (void)readAttributeProductLabelWithCompletionHandler:(void (^)(
                                                           NSString * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeProductLabelWithCompletion:^(NSString * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSString *>(value), error);
    }];
}
- (void)subscribeAttributeProductLabelWithMinInterval:(NSNumber * _Nonnull)minInterval
                                          maxInterval:(NSNumber * _Nonnull)maxInterval
                                               params:(MTRSubscribeParams * _Nullable)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                        reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeProductLabelWithParams:subscribeParams
                           subscriptionEstablished:subscriptionEstablishedHandler
                                     reportHandler:^(NSString * _Nullable value, NSError * _Nullable error) {
                                         // Cast is safe because subclass does not add any selectors.
                                         reportHandler(static_cast<NSString *>(value), error);
                                     }];
}
+ (void)readAttributeProductLabelWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                  completionHandler:
                                      (void (^)(NSString * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeProductLabelWithClusterStateCache:attributeCacheContainer.realContainer
                                                endpoint:endpoint
                                                   queue:queue
                                              completion:^(NSString * _Nullable value, NSError * _Nullable error) {
                                                  // Cast is safe because subclass does not add any selectors.
                                                  completionHandler(static_cast<NSString *>(value), error);
                                              }];
}

- (void)readAttributeSerialNumberWithCompletionHandler:(void (^)(
                                                           NSString * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeSerialNumberWithCompletion:^(NSString * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSString *>(value), error);
    }];
}
- (void)subscribeAttributeSerialNumberWithMinInterval:(NSNumber * _Nonnull)minInterval
                                          maxInterval:(NSNumber * _Nonnull)maxInterval
                                               params:(MTRSubscribeParams * _Nullable)params
                              subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                        reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeSerialNumberWithParams:subscribeParams
                           subscriptionEstablished:subscriptionEstablishedHandler
                                     reportHandler:^(NSString * _Nullable value, NSError * _Nullable error) {
                                         // Cast is safe because subclass does not add any selectors.
                                         reportHandler(static_cast<NSString *>(value), error);
                                     }];
}
+ (void)readAttributeSerialNumberWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                           endpoint:(NSNumber *)endpoint
                                              queue:(dispatch_queue_t)queue
                                  completionHandler:
                                      (void (^)(NSString * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeSerialNumberWithClusterStateCache:attributeCacheContainer.realContainer
                                                endpoint:endpoint
                                                   queue:queue
                                              completion:^(NSString * _Nullable value, NSError * _Nullable error) {
                                                  // Cast is safe because subclass does not add any selectors.
                                                  completionHandler(static_cast<NSString *>(value), error);
                                              }];
}

- (void)readAttributeLocalConfigDisabledWithCompletionHandler:(void (^)(NSNumber * _Nullable value,
                                                                  NSError * _Nullable error))completionHandler
{
    [self readAttributeLocalConfigDisabledWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)writeAttributeLocalConfigDisabledWithValue:(NSNumber * _Nonnull)value
                                 completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeLocalConfigDisabledWithValue:value params:nil completion:completionHandler];
}
- (void)writeAttributeLocalConfigDisabledWithValue:(NSNumber * _Nonnull)value
                                            params:(MTRWriteParams * _Nullable)params
                                 completionHandler:(MTRStatusCompletion)completionHandler
{
    [self writeAttributeLocalConfigDisabledWithValue:value params:params completion:completionHandler];
}
- (void)subscribeAttributeLocalConfigDisabledWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                 maxInterval:(NSNumber * _Nonnull)maxInterval
                                                      params:(MTRSubscribeParams * _Nullable)params
                                     subscriptionEstablished:
                                         (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                               reportHandler:
                                                   (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeLocalConfigDisabledWithParams:subscribeParams
                                  subscriptionEstablished:subscriptionEstablishedHandler
                                            reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                reportHandler(static_cast<NSNumber *>(value), error);
                                            }];
}
+ (void)readAttributeLocalConfigDisabledWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                         completionHandler:
                                             (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeLocalConfigDisabledWithClusterStateCache:attributeCacheContainer.realContainer
                                                       endpoint:endpoint
                                                          queue:queue
                                                     completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                         // Cast is safe because subclass does not add any selectors.
                                                         completionHandler(static_cast<NSNumber *>(value), error);
                                                     }];
}

- (void)readAttributeReachableWithCompletionHandler:(void (^)(
                                                        NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeReachableWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeReachableWithMinInterval:(NSNumber * _Nonnull)minInterval
                                       maxInterval:(NSNumber * _Nonnull)maxInterval
                                            params:(MTRSubscribeParams * _Nullable)params
                           subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                     reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeReachableWithParams:subscribeParams
                        subscriptionEstablished:subscriptionEstablishedHandler
                                  reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                      // Cast is safe because subclass does not add any selectors.
                                      reportHandler(static_cast<NSNumber *>(value), error);
                                  }];
}
+ (void)readAttributeReachableWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                        endpoint:(NSNumber *)endpoint
                                           queue:(dispatch_queue_t)queue
                               completionHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeReachableWithClusterStateCache:attributeCacheContainer.realContainer
                                             endpoint:endpoint
                                                queue:queue
                                           completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                               // Cast is safe because subclass does not add any selectors.
                                               completionHandler(static_cast<NSNumber *>(value), error);
                                           }];
}

- (void)readAttributeUniqueIDWithCompletionHandler:(void (^)(
                                                       NSString * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeUniqueIDWithCompletion:^(NSString * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSString *>(value), error);
    }];
}
- (void)subscribeAttributeUniqueIDWithMinInterval:(NSNumber * _Nonnull)minInterval
                                      maxInterval:(NSNumber * _Nonnull)maxInterval
                                           params:(MTRSubscribeParams * _Nullable)params
                          subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                    reportHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeUniqueIDWithParams:subscribeParams
                       subscriptionEstablished:subscriptionEstablishedHandler
                                 reportHandler:^(NSString * _Nullable value, NSError * _Nullable error) {
                                     // Cast is safe because subclass does not add any selectors.
                                     reportHandler(static_cast<NSString *>(value), error);
                                 }];
}
+ (void)readAttributeUniqueIDWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                       endpoint:(NSNumber *)endpoint
                                          queue:(dispatch_queue_t)queue
                              completionHandler:(void (^)(NSString * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeUniqueIDWithClusterStateCache:attributeCacheContainer.realContainer
                                            endpoint:endpoint
                                               queue:queue
                                          completion:^(NSString * _Nullable value, NSError * _Nullable error) {
                                              // Cast is safe because subclass does not add any selectors.
                                              completionHandler(static_cast<NSString *>(value), error);
                                          }];
}

- (void)readAttributeCapabilityMinimaWithCompletionHandler:(void (^)(MTRBasicClusterCapabilityMinimaStruct * _Nullable value,
                                                               NSError * _Nullable error))completionHandler
{
    [self readAttributeCapabilityMinimaWithCompletion:^(
        MTRBasicInformationClusterCapabilityMinimaStruct * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<MTRBasicClusterCapabilityMinimaStruct *>(value), error);
    }];
}
- (void)subscribeAttributeCapabilityMinimaWithMinInterval:(NSNumber * _Nonnull)minInterval
                                              maxInterval:(NSNumber * _Nonnull)maxInterval
                                                   params:(MTRSubscribeParams * _Nullable)params
                                  subscriptionEstablished:
                                      (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                            reportHandler:(void (^)(MTRBasicClusterCapabilityMinimaStruct * _Nullable value,
                                                              NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeCapabilityMinimaWithParams:subscribeParams
                               subscriptionEstablished:subscriptionEstablishedHandler
                                         reportHandler:^(MTRBasicInformationClusterCapabilityMinimaStruct * _Nullable value,
                                             NSError * _Nullable error) {
                                             // Cast is safe because subclass does not add any selectors.
                                             reportHandler(static_cast<MTRBasicClusterCapabilityMinimaStruct *>(value), error);
                                         }];
}
+ (void)readAttributeCapabilityMinimaWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                               endpoint:(NSNumber *)endpoint
                                                  queue:(dispatch_queue_t)queue
                                      completionHandler:(void (^)(MTRBasicClusterCapabilityMinimaStruct * _Nullable value,
                                                            NSError * _Nullable error))completionHandler
{
    [self readAttributeCapabilityMinimaWithClusterStateCache:attributeCacheContainer.realContainer
                                                    endpoint:endpoint
                                                       queue:queue
                                                  completion:^(MTRBasicInformationClusterCapabilityMinimaStruct * _Nullable value,
                                                      NSError * _Nullable error) {
                                                      // Cast is safe because subclass does not add any selectors.
                                                      completionHandler(
                                                          static_cast<MTRBasicClusterCapabilityMinimaStruct *>(value), error);
                                                  }];
}

- (void)readAttributeGeneratedCommandListWithCompletionHandler:(void (^)(NSArray * _Nullable value,
                                                                   NSError * _Nullable error))completionHandler
{
    [self readAttributeGeneratedCommandListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeGeneratedCommandListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                  maxInterval:(NSNumber * _Nonnull)maxInterval
                                                       params:(MTRSubscribeParams * _Nullable)params
                                      subscriptionEstablished:
                                          (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                                reportHandler:
                                                    (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeGeneratedCommandListWithParams:subscribeParams
                                   subscriptionEstablished:subscriptionEstablishedHandler
                                             reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                 // Cast is safe because subclass does not add any selectors.
                                                 reportHandler(static_cast<NSArray *>(value), error);
                                             }];
}
+ (void)readAttributeGeneratedCommandListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                   endpoint:(NSNumber *)endpoint
                                                      queue:(dispatch_queue_t)queue
                                          completionHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeGeneratedCommandListWithClusterStateCache:attributeCacheContainer.realContainer
                                                        endpoint:endpoint
                                                           queue:queue
                                                      completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                          // Cast is safe because subclass does not add any selectors.
                                                          completionHandler(static_cast<NSArray *>(value), error);
                                                      }];
}

- (void)readAttributeAcceptedCommandListWithCompletionHandler:(void (^)(NSArray * _Nullable value,
                                                                  NSError * _Nullable error))completionHandler
{
    [self readAttributeAcceptedCommandListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeAcceptedCommandListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                                 maxInterval:(NSNumber * _Nonnull)maxInterval
                                                      params:(MTRSubscribeParams * _Nullable)params
                                     subscriptionEstablished:
                                         (MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                               reportHandler:
                                                   (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeAcceptedCommandListWithParams:subscribeParams
                                  subscriptionEstablished:subscriptionEstablishedHandler
                                            reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                reportHandler(static_cast<NSArray *>(value), error);
                                            }];
}
+ (void)readAttributeAcceptedCommandListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                                  endpoint:(NSNumber *)endpoint
                                                     queue:(dispatch_queue_t)queue
                                         completionHandler:
                                             (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAcceptedCommandListWithClusterStateCache:attributeCacheContainer.realContainer
                                                       endpoint:endpoint
                                                          queue:queue
                                                     completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                         // Cast is safe because subclass does not add any selectors.
                                                         completionHandler(static_cast<NSArray *>(value), error);
                                                     }];
}

- (void)readAttributeAttributeListWithCompletionHandler:(void (^)(
                                                            NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAttributeListWithCompletion:^(NSArray * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSArray *>(value), error);
    }];
}
- (void)subscribeAttributeAttributeListWithMinInterval:(NSNumber * _Nonnull)minInterval
                                           maxInterval:(NSNumber * _Nonnull)maxInterval
                                                params:(MTRSubscribeParams * _Nullable)params
                               subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                         reportHandler:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeAttributeListWithParams:subscribeParams
                            subscriptionEstablished:subscriptionEstablishedHandler
                                      reportHandler:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                          // Cast is safe because subclass does not add any selectors.
                                          reportHandler(static_cast<NSArray *>(value), error);
                                      }];
}
+ (void)readAttributeAttributeListWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                            endpoint:(NSNumber *)endpoint
                                               queue:(dispatch_queue_t)queue
                                   completionHandler:
                                       (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeAttributeListWithClusterStateCache:attributeCacheContainer.realContainer
                                                 endpoint:endpoint
                                                    queue:queue
                                               completion:^(NSArray * _Nullable value, NSError * _Nullable error) {
                                                   // Cast is safe because subclass does not add any selectors.
                                                   completionHandler(static_cast<NSArray *>(value), error);
                                               }];
}

- (void)readAttributeFeatureMapWithCompletionHandler:(void (^)(
                                                         NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeFeatureMapWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeFeatureMapWithMinInterval:(NSNumber * _Nonnull)minInterval
                                        maxInterval:(NSNumber * _Nonnull)maxInterval
                                             params:(MTRSubscribeParams * _Nullable)params
                            subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                      reportHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeFeatureMapWithParams:subscribeParams
                         subscriptionEstablished:subscriptionEstablishedHandler
                                   reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                       // Cast is safe because subclass does not add any selectors.
                                       reportHandler(static_cast<NSNumber *>(value), error);
                                   }];
}
+ (void)readAttributeFeatureMapWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                         endpoint:(NSNumber *)endpoint
                                            queue:(dispatch_queue_t)queue
                                completionHandler:(void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeFeatureMapWithClusterStateCache:attributeCacheContainer.realContainer
                                              endpoint:endpoint
                                                 queue:queue
                                            completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                // Cast is safe because subclass does not add any selectors.
                                                completionHandler(static_cast<NSNumber *>(value), error);
                                            }];
}

- (void)readAttributeClusterRevisionWithCompletionHandler:(void (^)(NSNumber * _Nullable value,
                                                              NSError * _Nullable error))completionHandler
{
    [self readAttributeClusterRevisionWithCompletion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
        // Cast is safe because subclass does not add any selectors.
        completionHandler(static_cast<NSNumber *>(value), error);
    }];
}
- (void)subscribeAttributeClusterRevisionWithMinInterval:(NSNumber * _Nonnull)minInterval
                                             maxInterval:(NSNumber * _Nonnull)maxInterval
                                                  params:(MTRSubscribeParams * _Nullable)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablishedHandler
                                           reportHandler:
                                               (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))reportHandler
{
    MTRSubscribeParams * _Nullable subscribeParams = [params copy];
    if (subscribeParams == nil) {
        subscribeParams = [[MTRSubscribeParams alloc] initWithMinInterval:minInterval maxInterval:maxInterval];
    } else {
        subscribeParams.minInterval = minInterval;
        subscribeParams.maxInterval = maxInterval;
    }
    [self subscribeAttributeClusterRevisionWithParams:subscribeParams
                              subscriptionEstablished:subscriptionEstablishedHandler
                                        reportHandler:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                            // Cast is safe because subclass does not add any selectors.
                                            reportHandler(static_cast<NSNumber *>(value), error);
                                        }];
}
+ (void)readAttributeClusterRevisionWithAttributeCache:(MTRAttributeCacheContainer *)attributeCacheContainer
                                              endpoint:(NSNumber *)endpoint
                                                 queue:(dispatch_queue_t)queue
                                     completionHandler:
                                         (void (^)(NSNumber * _Nullable value, NSError * _Nullable error))completionHandler
{
    [self readAttributeClusterRevisionWithClusterStateCache:attributeCacheContainer.realContainer
                                                   endpoint:endpoint
                                                      queue:queue
                                                 completion:^(NSNumber * _Nullable value, NSError * _Nullable error) {
                                                     // Cast is safe because subclass does not add any selectors.
                                                     completionHandler(static_cast<NSNumber *>(value), error);
                                                 }];
}

- (nullable instancetype)initWithDevice:(MTRBaseDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

@end

@implementation MTRBaseClusterOTASoftwareUpdateProvider

- (instancetype)initWithDevice:(MTRBaseDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _device = device;
        _endpoint = [endpointID unsignedShortValue];
    }
    return self;
}

- (void)queryImageWithParams:(MTROTASoftwareUpdateProviderClusterQueryImageParams *)params
                  completion:(void (^)(MTROTASoftwareUpdateProviderClusterQueryImageResponseParams * _Nullable data,
                                 NSError * _Nullable error))completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTROTASoftwareUpdateProviderClusterQueryImageResponseCallbackBridge(self.callbackQueue, completion,
        ^(ExchangeManager & exchangeManager, const SessionHandle & session,
            OTASoftwareUpdateProviderClusterQueryImageResponseCallbackType successCb, MTRErrorCallback failureCb,
            MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTROTASoftwareUpdateProviderClusterQueryImageResponseCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            OtaSoftwareUpdateProvider::Commands::QueryImage::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.vendorID = static_cast<std::remove_reference_t<decltype(request.vendorID)>>(params.vendorID.unsignedShortValue);
            request.productID = params.productID.unsignedShortValue;
            request.softwareVersion = params.softwareVersion.unsignedIntValue;
            {
                using ListType_0 = std::remove_reference_t<decltype(request.protocolsSupported)>;
                using ListMemberType_0 = ListMemberTypeGetter<ListType_0>::Type;
                if (params.protocolsSupported.count != 0) {
                    auto * listHolder_0 = new ListHolder<ListMemberType_0>(params.protocolsSupported.count);
                    if (listHolder_0 == nullptr || listHolder_0->mList == nullptr) {
                        return CHIP_ERROR_INVALID_ARGUMENT;
                    }
                    listFreer.add(listHolder_0);
                    for (size_t i_0 = 0; i_0 < params.protocolsSupported.count; ++i_0) {
                        if (![params.protocolsSupported[i_0] isKindOfClass:[NSNumber class]]) {
                            // Wrong kind of value.
                            return CHIP_ERROR_INVALID_ARGUMENT;
                        }
                        auto element_0 = (NSNumber *) params.protocolsSupported[i_0];
                        listHolder_0->mList[i_0]
                            = static_cast<std::remove_reference_t<decltype(listHolder_0->mList[i_0])>>(element_0.unsignedCharValue);
                    }
                    request.protocolsSupported = ListType_0(listHolder_0->mList, params.protocolsSupported.count);
                } else {
                    request.protocolsSupported = ListType_0();
                }
            }
            if (params.hardwareVersion != nil) {
                auto & definedValue_0 = request.hardwareVersion.Emplace();
                definedValue_0 = params.hardwareVersion.unsignedShortValue;
            }
            if (params.location != nil) {
                auto & definedValue_0 = request.location.Emplace();
                definedValue_0 = [self asCharSpan:params.location];
            }
            if (params.requestorCanConsent != nil) {
                auto & definedValue_0 = request.requestorCanConsent.Emplace();
                definedValue_0 = params.requestorCanConsent.boolValue;
            }
            if (params.metadataForProvider != nil) {
                auto & definedValue_0 = request.metadataForProvider.Emplace();
                definedValue_0 = [self asByteSpan:params.metadataForProvider];
            }

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)applyUpdateRequestWithParams:(MTROTASoftwareUpdateProviderClusterApplyUpdateRequestParams *)params
                          completion:(void (^)(MTROTASoftwareUpdateProviderClusterApplyUpdateResponseParams * _Nullable data,
                                         NSError * _Nullable error))completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTROTASoftwareUpdateProviderClusterApplyUpdateResponseCallbackBridge(self.callbackQueue, completion,
        ^(ExchangeManager & exchangeManager, const SessionHandle & session,
            OTASoftwareUpdateProviderClusterApplyUpdateResponseCallbackType successCb, MTRErrorCallback failureCb,
            MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTROTASoftwareUpdateProviderClusterApplyUpdateResponseCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            OtaSoftwareUpdateProvider::Commands::ApplyUpdateRequest::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.updateToken = [self asByteSpan:params.updateToken];
            request.newVersion = params.newVersion.unsignedIntValue;

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)notifyUpdateAppliedWithParams:(MTROTASoftwareUpdateProviderClusterNotifyUpdateAppliedParams *)params
                           completion:(MTRStatusCompletion)completion
{
    // Make a copy of params before we go async.
    params = [params copy];
    auto * bridge = new MTRCommandSuccessCallbackBridge(
        self.callbackQueue,
        ^(id _Nullable value, NSError * _Nullable error) {
            completion(error);
        },
        ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
            MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
            auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
            Optional<uint16_t> timedInvokeTimeoutMs;
            Optional<Timeout> invokeTimeout;
            ListFreer listFreer;
            OtaSoftwareUpdateProvider::Commands::NotifyUpdateApplied::Type request;
            if (params != nil) {
                if (params.timedInvokeTimeoutMs != nil) {
                    params.timedInvokeTimeoutMs = MTRClampedNumber(params.timedInvokeTimeoutMs, @(1), @(UINT16_MAX));
                    timedInvokeTimeoutMs.SetValue(params.timedInvokeTimeoutMs.unsignedShortValue);
                }
                if (params.serverSideProcessingTimeout != nil) {
                    // Clamp to a number of seconds that will not overflow 32-bit
                    // int when converted to ms.
                    auto * serverSideProcessingTimeout = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                    invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                }
            }
            request.updateToken = [self asByteSpan:params.updateToken];
            request.softwareVersion = params.softwareVersion.unsignedIntValue;

            return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb, self->_endpoint,
                timedInvokeTimeoutMs, invokeTimeout);
        });
    std::move(*bridge).DispatchAction(self.device);
}

- (void)readAttributeGeneratedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = OtaSoftwareUpdateProvider::Attributes::GeneratedCommandList::TypeInfo;
    return MTRReadAttribute<MTROTASoftwareUpdateProviderGeneratedCommandListListAttributeCallbackBridge, NSArray,
        TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeGeneratedCommandListWithParams:(MTRSubscribeParams * _Nonnull)params
                                 subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                           reportHandler:
                                               (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = OtaSoftwareUpdateProvider::Attributes::GeneratedCommandList::TypeInfo;
    MTRSubscribeAttribute<MTROTASoftwareUpdateProviderGeneratedCommandListListAttributeCallbackSubscriptionBridge, NSArray,
        TypeInfo::DecodableType>(params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint,
        TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

+ (void)readAttributeGeneratedCommandListWithClusterStateCache:(MTRClusterStateCacheContainer *)clusterStateCacheContainer
                                                      endpoint:(NSNumber *)endpoint
                                                         queue:(dispatch_queue_t)queue
                                                    completion:
                                                        (void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    auto * bridge = new MTROTASoftwareUpdateProviderGeneratedCommandListListAttributeCallbackBridge(queue, completion);
    std::move(*bridge).DispatchLocalAction(clusterStateCacheContainer.baseDevice,
        ^(OTASoftwareUpdateProviderGeneratedCommandListListAttributeCallback successCb, MTRErrorCallback failureCb) {
            if (clusterStateCacheContainer.cppClusterStateCache) {
                chip::app::ConcreteAttributePath path;
                using TypeInfo = OtaSoftwareUpdateProvider::Attributes::GeneratedCommandList::TypeInfo;
                path.mEndpointId = static_cast<chip::EndpointId>([endpoint unsignedShortValue]);
                path.mClusterId = TypeInfo::GetClusterId();
                path.mAttributeId = TypeInfo::GetAttributeId();
                TypeInfo::DecodableType value;
                CHIP_ERROR err = clusterStateCacheContainer.cppClusterStateCache->Get<TypeInfo>(path, value);
                if (err == CHIP_NO_ERROR) {
                    successCb(bridge, value);
                }
                return err;
            }
            return CHIP_ERROR_NOT_FOUND;
        });
}

- (void)readAttributeAcceptedCommandListWithCompletion:(void (^)(NSArray * _Nullable value, NSError * _Nullable error))completion
{
    MTRReadParams * params = [[MTRReadParams alloc] init];
    using TypeInfo = OtaSoftwareUpdateProvider::Attributes::AcceptedCommandList::TypeInfo;
    return MTRReadAttribute<MTROTASoftwareUpdateProviderAcceptedCommandListListAttributeCallbackBridge, NSArray,
        TypeInfo::DecodableType>(
        params, completion, self.callbackQueue, self.device, self->_endpoint, TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

- (void)subscribeAttributeAcceptedCommandListWithParams:(MTRSubscribeParams * _Nonnull)params
                                subscriptionEstablished:(MTRSubscriptionEstablishedHandler _Nullable)subscriptionEstablished
                                          reportHandler:
                                              (void (^)(NSArray * _Nullable value, NSError * _Nullable error))reportHandler
{
    using TypeInfo = OtaSoftwareUpdateProvider::Attributes::AcceptedCommandList::TypeInfo;
    MTRSubscribeAttribute<MTROTASoftwareUpdateProviderAcceptedCommandListListAttributeCallbackSubscriptionBridge, NSArray,
        TypeInfo::DecodableType>(params, subscriptionEstablished, reportHandler, self.callbackQueue, self.device, self->_endpoint,
        TypeInfo::GetClusterId(), TypeInfo::GetAttributeId());
}

+ (void)readAttributeAcceptedCommandListWith