/*
 *
 *    Copyright (c) 2022 Project CHIP Authors
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

#import <Foundation/Foundation.h>

#import "MTRAsyncCallbackWorkQueue.h"
#import "MTRBaseClusterUtils.h"
#import "MTRBaseDevice_Internal.h"
#import "MTRCallbackBridge.h"
#import "MTRClusterConstants.h"
#import "MTRCluster_Internal.h"
#import "MTRClusters_Internal.h"
#import "MTRCommandPayloadsObjc.h"
#import "MTRDevice_Internal.h"
#import "MTRLogging_Internal.h"
#import "MTRStructsObjc.h"

#include <lib/support/CHIPListUtils.h>
#include <platform/CHIPDeviceLayer.h>
#include <type_traits>

using chip::Callback::Callback;
using chip::Callback::Cancelable;
using namespace chip::app::Clusters;
using chip::Optional;
using chip::SessionHandle;
using chip::Messaging::ExchangeManager;
using chip::System::Clock::Seconds16;
using chip::System::Clock::Timeout;

static void MTRClustersLogEnqueue(NSString * logPrefix, MTRAsyncCallbackWorkQueue * workQueue)
{
    MTR_LOG_DEFAULT("%@ enqueueWorkItem %@", logPrefix, workQueue);
}

static void MTRClustersLogDequeue(NSString * logPrefix, MTRAsyncCallbackWorkQueue * workQueue)
{
    MTR_LOG_DEFAULT("%@ dequeueWorkItem %@", logPrefix, workQueue);
}

static void MTRClustersLogCompletion(NSString * logPrefix, id value, NSError * error)
{
    // Log the data at the INFO level (not usually persisted permanently),
    // but make sure we log the work completion at the DEFAULT level.
    MTR_LOG_INFO("%@ received response: %@ error: %@", logPrefix, value, error);
    MTR_LOG_DEFAULT("%@ endWork", logPrefix);
}

// NOLINTBEGIN(clang-analyzer-cplusplus.NewDeleteLeaks): Linter is unable to locate the delete on these objects.
@implementation MTRClusterIdentify

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)identifyWithParams:(MTRIdentifyClusterIdentifyParams *)params
            expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
     expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeIdentifyID, (unsigned int) MTRCommandIDTypeClusterIdentifyCommandIdentifyID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                Identify::Commands::Identify::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.identifyTime = params.identifyTime.unsignedShortValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)triggerEffectWithParams:(MTRIdentifyClusterTriggerEffectParams *)params
                 expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
          expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                     completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeIdentifyID, (unsigned int) MTRCommandIDTypeClusterIdentifyCommandTriggerEffectID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                Identify::Commands::TriggerEffect::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.effectIdentifier = static_cast<std::remove_reference_t<decltype(request.effectIdentifier)>>(
                    params.effectIdentifier.unsignedCharValue);
                request.effectVariant
                    = static_cast<std::remove_reference_t<decltype(request.effectVariant)>>(params.effectVariant.unsignedCharValue);

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeIdentifyTimeWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeIdentifyID)
                                        attributeID:@(MTRAttributeIDTypeClusterIdentifyAttributeIdentifyTimeID)
                                             params:params];
}

- (void)writeAttributeIdentifyTimeWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                      expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeIdentifyTimeWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeIdentifyTimeWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                      expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                     params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeIdentifyID)
                                  attributeID:@(MTRAttributeIDTypeClusterIdentifyAttributeIdentifyTimeID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeIdentifyTypeWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeIdentifyID)
                                        attributeID:@(MTRAttributeIDTypeClusterIdentifyAttributeIdentifyTypeID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeIdentifyID)
                                        attributeID:@(MTRAttributeIDTypeClusterIdentifyAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeIdentifyID)
                                        attributeID:@(MTRAttributeIDTypeClusterIdentifyAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeIdentifyID)
                                        attributeID:@(MTRAttributeIDTypeClusterIdentifyAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeIdentifyID)
                                        attributeID:@(MTRAttributeIDTypeClusterIdentifyAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeIdentifyID)
                                        attributeID:@(MTRAttributeIDTypeClusterIdentifyAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeIdentifyID)
                                        attributeID:@(MTRAttributeIDTypeClusterIdentifyAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterIdentify (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

- (void)identifyWithParams:(MTRIdentifyClusterIdentifyParams *)params
            expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
     expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
         completionHandler:(MTRStatusCompletion)completionHandler
{
    [self identifyWithParams:params
               expectedValues:expectedDataValueDictionaries
        expectedValueInterval:expectedValueIntervalMs
                   completion:completionHandler];
}
- (void)triggerEffectWithParams:(MTRIdentifyClusterTriggerEffectParams *)params
                 expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
          expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
              completionHandler:(MTRStatusCompletion)completionHandler
{
    [self triggerEffectWithParams:params
                   expectedValues:expectedDataValueDictionaries
            expectedValueInterval:expectedValueIntervalMs
                       completion:completionHandler];
}
@end

@implementation MTRClusterGroups

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)addGroupWithParams:(MTRGroupsClusterAddGroupParams *)params
            expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
     expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                completion:(void (^)(MTRGroupsClusterAddGroupResponseParams * _Nullable data, NSError * _Nullable error))completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeGroupsID, (unsigned int) MTRCommandIDTypeClusterGroupsCommandAddGroupID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRGroupsClusterAddGroupResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, GroupsClusterAddGroupResponseCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRGroupsClusterAddGroupResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                Groups::Commands::AddGroup::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.groupID = params.groupID.unsignedShortValue;
                request.groupName = [self asCharSpan:params.groupName];

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)viewGroupWithParams:(MTRGroupsClusterViewGroupParams *)params
             expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
      expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                 completion:
                     (void (^)(MTRGroupsClusterViewGroupResponseParams * _Nullable data, NSError * _Nullable error))completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeGroupsID, (unsigned int) MTRCommandIDTypeClusterGroupsCommandViewGroupID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRGroupsClusterViewGroupResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                GroupsClusterViewGroupResponseCallbackType successCb, MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRGroupsClusterViewGroupResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                Groups::Commands::ViewGroup::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.groupID = params.groupID.unsignedShortValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)getGroupMembershipWithParams:(MTRGroupsClusterGetGroupMembershipParams *)params
                      expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
               expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                          completion:(void (^)(MTRGroupsClusterGetGroupMembershipResponseParams * _Nullable data,
                                         NSError * _Nullable error))completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeGroupsID, (unsigned int) MTRCommandIDTypeClusterGroupsCommandGetGroupMembershipID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRGroupsClusterGetGroupMembershipResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                GroupsClusterGetGroupMembershipResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRGroupsClusterGetGroupMembershipResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                Groups::Commands::GetGroupMembership::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                {
                    using ListType_0 = std::remove_reference_t<decltype(request.groupList)>;
                    using ListMemberType_0 = ListMemberTypeGetter<ListType_0>::Type;
                    if (params.groupList.count != 0) {
                        auto * listHolder_0 = new ListHolder<ListMemberType_0>(params.groupList.count);
                        if (listHolder_0 == nullptr || listHolder_0->mList == nullptr) {
                            return CHIP_ERROR_INVALID_ARGUMENT;
                        }
                        listFreer.add(listHolder_0);
                        for (size_t i_0 = 0; i_0 < params.groupList.count; ++i_0) {
                            if (![params.groupList[i_0] isKindOfClass:[NSNumber class]]) {
                                // Wrong kind of value.
                                return CHIP_ERROR_INVALID_ARGUMENT;
                            }
                            auto element_0 = (NSNumber *) params.groupList[i_0];
                            listHolder_0->mList[i_0] = element_0.unsignedShortValue;
                        }
                        request.groupList = ListType_0(listHolder_0->mList, params.groupList.count);
                    } else {
                        request.groupList = ListType_0();
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)removeGroupWithParams:(MTRGroupsClusterRemoveGroupParams *)params
               expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
        expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                   completion:
                       (void (^)(MTRGroupsClusterRemoveGroupResponseParams * _Nullable data, NSError * _Nullable error))completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeGroupsID, (unsigned int) MTRCommandIDTypeClusterGroupsCommandRemoveGroupID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRGroupsClusterRemoveGroupResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                GroupsClusterRemoveGroupResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRGroupsClusterRemoveGroupResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                Groups::Commands::RemoveGroup::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.groupID = params.groupID.unsignedShortValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)removeAllGroupsWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                               completion:(MTRStatusCompletion)completion
{
    [self removeAllGroupsWithParams:nil
                     expectedValues:expectedValues
              expectedValueInterval:expectedValueIntervalMs
                         completion:completion];
}
- (void)removeAllGroupsWithParams:(MTRGroupsClusterRemoveAllGroupsParams * _Nullable)params
                   expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
            expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                       completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeGroupsID, (unsigned int) MTRCommandIDTypeClusterGroupsCommandRemoveAllGroupsID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                Groups::Commands::RemoveAllGroups::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)addGroupIfIdentifyingWithParams:(MTRGroupsClusterAddGroupIfIdentifyingParams *)params
                         expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                  expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                             completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeGroupsID,
                                     (unsigned int) MTRCommandIDTypeClusterGroupsCommandAddGroupIfIdentifyingID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                Groups::Commands::AddGroupIfIdentifying::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.groupID = params.groupID.unsignedShortValue;
                request.groupName = [self asCharSpan:params.groupName];

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeNameSupportWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGroupsID)
                                        attributeID:@(MTRAttributeIDTypeClusterGroupsAttributeNameSupportID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGroupsID)
                                        attributeID:@(MTRAttributeIDTypeClusterGroupsAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGroupsID)
                                        attributeID:@(MTRAttributeIDTypeClusterGroupsAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGroupsID)
                                        attributeID:@(MTRAttributeIDTypeClusterGroupsAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGroupsID)
                                        attributeID:@(MTRAttributeIDTypeClusterGroupsAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGroupsID)
                                        attributeID:@(MTRAttributeIDTypeClusterGroupsAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGroupsID)
                                        attributeID:@(MTRAttributeIDTypeClusterGroupsAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterGroups (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

- (void)addGroupWithParams:(MTRGroupsClusterAddGroupParams *)params
            expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
     expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
         completionHandler:
             (void (^)(MTRGroupsClusterAddGroupResponseParams * _Nullable data, NSError * _Nullable error))completionHandler
{
    [self addGroupWithParams:params
               expectedValues:expectedDataValueDictionaries
        expectedValueInterval:expectedValueIntervalMs
                   completion:^(MTRGroupsClusterAddGroupResponseParams * _Nullable data, NSError * _Nullable error) {
                       // Cast is safe because subclass does not add any selectors.
                       completionHandler(static_cast<MTRGroupsClusterAddGroupResponseParams *>(data), error);
                   }];
}
- (void)viewGroupWithParams:(MTRGroupsClusterViewGroupParams *)params
             expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
      expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
          completionHandler:
              (void (^)(MTRGroupsClusterViewGroupResponseParams * _Nullable data, NSError * _Nullable error))completionHandler
{
    [self viewGroupWithParams:params
               expectedValues:expectedDataValueDictionaries
        expectedValueInterval:expectedValueIntervalMs
                   completion:^(MTRGroupsClusterViewGroupResponseParams * _Nullable data, NSError * _Nullable error) {
                       // Cast is safe because subclass does not add any selectors.
                       completionHandler(static_cast<MTRGroupsClusterViewGroupResponseParams *>(data), error);
                   }];
}
- (void)getGroupMembershipWithParams:(MTRGroupsClusterGetGroupMembershipParams *)params
                      expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
               expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                   completionHandler:(void (^)(MTRGroupsClusterGetGroupMembershipResponseParams * _Nullable data,
                                         NSError * _Nullable error))completionHandler
{
    [self getGroupMembershipWithParams:params
                        expectedValues:expectedDataValueDictionaries
                 expectedValueInterval:expectedValueIntervalMs
                            completion:^(
                                MTRGroupsClusterGetGroupMembershipResponseParams * _Nullable data, NSError * _Nullable error) {
                                // Cast is safe because subclass does not add any selectors.
                                completionHandler(static_cast<MTRGroupsClusterGetGroupMembershipResponseParams *>(data), error);
                            }];
}
- (void)removeGroupWithParams:(MTRGroupsClusterRemoveGroupParams *)params
               expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
        expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
            completionHandler:
                (void (^)(MTRGroupsClusterRemoveGroupResponseParams * _Nullable data, NSError * _Nullable error))completionHandler
{
    [self removeGroupWithParams:params
                 expectedValues:expectedDataValueDictionaries
          expectedValueInterval:expectedValueIntervalMs
                     completion:^(MTRGroupsClusterRemoveGroupResponseParams * _Nullable data, NSError * _Nullable error) {
                         // Cast is safe because subclass does not add any selectors.
                         completionHandler(static_cast<MTRGroupsClusterRemoveGroupResponseParams *>(data), error);
                     }];
}
- (void)removeAllGroupsWithParams:(MTRGroupsClusterRemoveAllGroupsParams * _Nullable)params
                   expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
            expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                completionHandler:(MTRStatusCompletion)completionHandler
{
    [self removeAllGroupsWithParams:params
                     expectedValues:expectedDataValueDictionaries
              expectedValueInterval:expectedValueIntervalMs
                         completion:completionHandler];
}
- (void)removeAllGroupsWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                        completionHandler:(MTRStatusCompletion)completionHandler
{
    [self removeAllGroupsWithParams:nil
                     expectedValues:expectedValues
              expectedValueInterval:expectedValueIntervalMs
                  completionHandler:completionHandler];
}
- (void)addGroupIfIdentifyingWithParams:(MTRGroupsClusterAddGroupIfIdentifyingParams *)params
                         expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
                  expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                      completionHandler:(MTRStatusCompletion)completionHandler
{
    [self addGroupIfIdentifyingWithParams:params
                           expectedValues:expectedDataValueDictionaries
                    expectedValueInterval:expectedValueIntervalMs
                               completion:completionHandler];
}
@end

@implementation MTRClusterScenes

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)addSceneWithParams:(MTRScenesClusterAddSceneParams *)params
            expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
     expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                completion:(void (^)(MTRScenesClusterAddSceneResponseParams * _Nullable data, NSError * _Nullable error))completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeScenesID, (unsigned int) MTRCommandIDTypeClusterScenesCommandAddSceneID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRScenesClusterAddSceneResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, ScenesClusterAddSceneResponseCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRScenesClusterAddSceneResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                Scenes::Commands::AddScene::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.groupID = params.groupID.unsignedShortValue;
                request.sceneID = params.sceneID.unsignedCharValue;
                request.transitionTime = params.transitionTime.unsignedShortValue;
                request.sceneName = [self asCharSpan:params.sceneName];
                {
                    using ListType_0 = std::remove_reference_t<decltype(request.extensionFieldSets)>;
                    using ListMemberType_0 = ListMemberTypeGetter<ListType_0>::Type;
                    if (params.extensionFieldSets.count != 0) {
                        auto * listHolder_0 = new ListHolder<ListMemberType_0>(params.extensionFieldSets.count);
                        if (listHolder_0 == nullptr || listHolder_0->mList == nullptr) {
                            return CHIP_ERROR_INVALID_ARGUMENT;
                        }
                        listFreer.add(listHolder_0);
                        for (size_t i_0 = 0; i_0 < params.extensionFieldSets.count; ++i_0) {
                            if (![params.extensionFieldSets[i_0] isKindOfClass:[MTRScenesClusterExtensionFieldSet class]]) {
                                // Wrong kind of value.
                                return CHIP_ERROR_INVALID_ARGUMENT;
                            }
                            auto element_0 = (MTRScenesClusterExtensionFieldSet *) params.extensionFieldSets[i_0];
                            listHolder_0->mList[i_0].clusterID = element_0.clusterID.unsignedIntValue;
                            {
                                using ListType_2 = std::remove_reference_t<decltype(listHolder_0->mList[i_0].attributeValueList)>;
                                using ListMemberType_2 = ListMemberTypeGetter<ListType_2>::Type;
                                if (element_0.attributeValueList.count != 0) {
                                    auto * listHolder_2 = new ListHolder<ListMemberType_2>(element_0.attributeValueList.count);
                                    if (listHolder_2 == nullptr || listHolder_2->mList == nullptr) {
                                        return CHIP_ERROR_INVALID_ARGUMENT;
                                    }
                                    listFreer.add(listHolder_2);
                                    for (size_t i_2 = 0; i_2 < element_0.attributeValueList.count; ++i_2) {
                                        if (![element_0.attributeValueList[i_2]
                                                isKindOfClass:[MTRScenesClusterAttributeValuePair class]]) {
                                            // Wrong kind of value.
                                            return CHIP_ERROR_INVALID_ARGUMENT;
                                        }
                                        auto element_2 = (MTRScenesClusterAttributeValuePair *) element_0.attributeValueList[i_2];
                                        if (element_2.attributeID != nil) {
                                            auto & definedValue_4 = listHolder_2->mList[i_2].attributeID.Emplace();
                                            definedValue_4 = element_2.attributeID.unsignedIntValue;
                                        }
                                        listHolder_2->mList[i_2].attributeValue = element_2.attributeValue.unsignedIntValue;
                                    }
                                    listHolder_0->mList[i_0].attributeValueList
                                        = ListType_2(listHolder_2->mList, element_0.attributeValueList.count);
                                } else {
                                    listHolder_0->mList[i_0].attributeValueList = ListType_2();
                                }
                            }
                        }
                        request.extensionFieldSets = ListType_0(listHolder_0->mList, params.extensionFieldSets.count);
                    } else {
                        request.extensionFieldSets = ListType_0();
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)viewSceneWithParams:(MTRScenesClusterViewSceneParams *)params
             expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
      expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                 completion:
                     (void (^)(MTRScenesClusterViewSceneResponseParams * _Nullable data, NSError * _Nullable error))completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeScenesID, (unsigned int) MTRCommandIDTypeClusterScenesCommandViewSceneID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRScenesClusterViewSceneResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                ScenesClusterViewSceneResponseCallbackType successCb, MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRScenesClusterViewSceneResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                Scenes::Commands::ViewScene::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.groupID = params.groupID.unsignedShortValue;
                request.sceneID = params.sceneID.unsignedCharValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)removeSceneWithParams:(MTRScenesClusterRemoveSceneParams *)params
               expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
        expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                   completion:
                       (void (^)(MTRScenesClusterRemoveSceneResponseParams * _Nullable data, NSError * _Nullable error))completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeScenesID, (unsigned int) MTRCommandIDTypeClusterScenesCommandRemoveSceneID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRScenesClusterRemoveSceneResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                ScenesClusterRemoveSceneResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRScenesClusterRemoveSceneResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                Scenes::Commands::RemoveScene::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.groupID = params.groupID.unsignedShortValue;
                request.sceneID = params.sceneID.unsignedCharValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)removeAllScenesWithParams:(MTRScenesClusterRemoveAllScenesParams *)params
                   expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
            expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                       completion:(void (^)(MTRScenesClusterRemoveAllScenesResponseParams * _Nullable data,
                                      NSError * _Nullable error))completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeScenesID, (unsigned int) MTRCommandIDTypeClusterScenesCommandRemoveAllScenesID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRScenesClusterRemoveAllScenesResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                ScenesClusterRemoveAllScenesResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRScenesClusterRemoveAllScenesResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                Scenes::Commands::RemoveAllScenes::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.groupID = params.groupID.unsignedShortValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)storeSceneWithParams:(MTRScenesClusterStoreSceneParams *)params
              expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
       expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                  completion:
                      (void (^)(MTRScenesClusterStoreSceneResponseParams * _Nullable data, NSError * _Nullable error))completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeScenesID, (unsigned int) MTRCommandIDTypeClusterScenesCommandStoreSceneID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRScenesClusterStoreSceneResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                ScenesClusterStoreSceneResponseCallbackType successCb, MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRScenesClusterStoreSceneResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                Scenes::Commands::StoreScene::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.groupID = params.groupID.unsignedShortValue;
                request.sceneID = params.sceneID.unsignedCharValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)recallSceneWithParams:(MTRScenesClusterRecallSceneParams *)params
               expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
        expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                   completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeScenesID, (unsigned int) MTRCommandIDTypeClusterScenesCommandRecallSceneID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                Scenes::Commands::RecallScene::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.groupID = params.groupID.unsignedShortValue;
                request.sceneID = params.sceneID.unsignedCharValue;
                if (params.transitionTime != nil) {
                    auto & definedValue_0 = request.transitionTime.Emplace();
                    if (params.transitionTime == nil) {
                        definedValue_0.SetNull();
                    } else {
                        auto & nonNullValue_1 = definedValue_0.SetNonNull();
                        nonNullValue_1 = params.transitionTime.unsignedShortValue;
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)getSceneMembershipWithParams:(MTRScenesClusterGetSceneMembershipParams *)params
                      expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
               expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                          completion:(void (^)(MTRScenesClusterGetSceneMembershipResponseParams * _Nullable data,
                                         NSError * _Nullable error))completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeScenesID, (unsigned int) MTRCommandIDTypeClusterScenesCommandGetSceneMembershipID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRScenesClusterGetSceneMembershipResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                ScenesClusterGetSceneMembershipResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRScenesClusterGetSceneMembershipResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                Scenes::Commands::GetSceneMembership::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.groupID = params.groupID.unsignedShortValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)enhancedAddSceneWithParams:(MTRScenesClusterEnhancedAddSceneParams *)params
                    expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
             expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                        completion:(void (^)(MTRScenesClusterEnhancedAddSceneResponseParams * _Nullable data,
                                       NSError * _Nullable error))completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeScenesID, (unsigned int) MTRCommandIDTypeClusterScenesCommandEnhancedAddSceneID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRScenesClusterEnhancedAddSceneResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                ScenesClusterEnhancedAddSceneResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRScenesClusterEnhancedAddSceneResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                Scenes::Commands::EnhancedAddScene::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.groupID = params.groupID.unsignedShortValue;
                request.sceneID = params.sceneID.unsignedCharValue;
                request.transitionTime = params.transitionTime.unsignedShortValue;
                request.sceneName = [self asCharSpan:params.sceneName];
                {
                    using ListType_0 = std::remove_reference_t<decltype(request.extensionFieldSets)>;
                    using ListMemberType_0 = ListMemberTypeGetter<ListType_0>::Type;
                    if (params.extensionFieldSets.count != 0) {
                        auto * listHolder_0 = new ListHolder<ListMemberType_0>(params.extensionFieldSets.count);
                        if (listHolder_0 == nullptr || listHolder_0->mList == nullptr) {
                            return CHIP_ERROR_INVALID_ARGUMENT;
                        }
                        listFreer.add(listHolder_0);
                        for (size_t i_0 = 0; i_0 < params.extensionFieldSets.count; ++i_0) {
                            if (![params.extensionFieldSets[i_0] isKindOfClass:[MTRScenesClusterExtensionFieldSet class]]) {
                                // Wrong kind of value.
                                return CHIP_ERROR_INVALID_ARGUMENT;
                            }
                            auto element_0 = (MTRScenesClusterExtensionFieldSet *) params.extensionFieldSets[i_0];
                            listHolder_0->mList[i_0].clusterID = element_0.clusterID.unsignedIntValue;
                            {
                                using ListType_2 = std::remove_reference_t<decltype(listHolder_0->mList[i_0].attributeValueList)>;
                                using ListMemberType_2 = ListMemberTypeGetter<ListType_2>::Type;
                                if (element_0.attributeValueList.count != 0) {
                                    auto * listHolder_2 = new ListHolder<ListMemberType_2>(element_0.attributeValueList.count);
                                    if (listHolder_2 == nullptr || listHolder_2->mList == nullptr) {
                                        return CHIP_ERROR_INVALID_ARGUMENT;
                                    }
                                    listFreer.add(listHolder_2);
                                    for (size_t i_2 = 0; i_2 < element_0.attributeValueList.count; ++i_2) {
                                        if (![element_0.attributeValueList[i_2]
                                                isKindOfClass:[MTRScenesClusterAttributeValuePair class]]) {
                                            // Wrong kind of value.
                                            return CHIP_ERROR_INVALID_ARGUMENT;
                                        }
                                        auto element_2 = (MTRScenesClusterAttributeValuePair *) element_0.attributeValueList[i_2];
                                        if (element_2.attributeID != nil) {
                                            auto & definedValue_4 = listHolder_2->mList[i_2].attributeID.Emplace();
                                            definedValue_4 = element_2.attributeID.unsignedIntValue;
                                        }
                                        listHolder_2->mList[i_2].attributeValue = element_2.attributeValue.unsignedIntValue;
                                    }
                                    listHolder_0->mList[i_0].attributeValueList
                                        = ListType_2(listHolder_2->mList, element_0.attributeValueList.count);
                                } else {
                                    listHolder_0->mList[i_0].attributeValueList = ListType_2();
                                }
                            }
                        }
                        request.extensionFieldSets = ListType_0(listHolder_0->mList, params.extensionFieldSets.count);
                    } else {
                        request.extensionFieldSets = ListType_0();
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)enhancedViewSceneWithParams:(MTRScenesClusterEnhancedViewSceneParams *)params
                     expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
              expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                         completion:(void (^)(MTRScenesClusterEnhancedViewSceneResponseParams * _Nullable data,
                                        NSError * _Nullable error))completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeScenesID, (unsigned int) MTRCommandIDTypeClusterScenesCommandEnhancedViewSceneID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRScenesClusterEnhancedViewSceneResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                ScenesClusterEnhancedViewSceneResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRScenesClusterEnhancedViewSceneResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                Scenes::Commands::EnhancedViewScene::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.groupID = params.groupID.unsignedShortValue;
                request.sceneID = params.sceneID.unsignedCharValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)copySceneWithParams:(MTRScenesClusterCopySceneParams *)params
             expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
      expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                 completion:
                     (void (^)(MTRScenesClusterCopySceneResponseParams * _Nullable data, NSError * _Nullable error))completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeScenesID, (unsigned int) MTRCommandIDTypeClusterScenesCommandCopySceneID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRScenesClusterCopySceneResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                ScenesClusterCopySceneResponseCallbackType successCb, MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRScenesClusterCopySceneResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                Scenes::Commands::CopyScene::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.mode = static_cast<std::remove_reference_t<decltype(request.mode)>>(params.mode.unsignedCharValue);
                request.groupIdentifierFrom = params.groupIdentifierFrom.unsignedShortValue;
                request.sceneIdentifierFrom = params.sceneIdentifierFrom.unsignedCharValue;
                request.groupIdentifierTo = params.groupIdentifierTo.unsignedShortValue;
                request.sceneIdentifierTo = params.sceneIdentifierTo.unsignedCharValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeSceneCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeScenesID)
                                        attributeID:@(MTRAttributeIDTypeClusterScenesAttributeSceneCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeCurrentSceneWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeScenesID)
                                        attributeID:@(MTRAttributeIDTypeClusterScenesAttributeCurrentSceneID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeCurrentGroupWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeScenesID)
                                        attributeID:@(MTRAttributeIDTypeClusterScenesAttributeCurrentGroupID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeSceneValidWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeScenesID)
                                        attributeID:@(MTRAttributeIDTypeClusterScenesAttributeSceneValidID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeNameSupportWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeScenesID)
                                        attributeID:@(MTRAttributeIDTypeClusterScenesAttributeNameSupportID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeLastConfiguredByWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeScenesID)
                                        attributeID:@(MTRAttributeIDTypeClusterScenesAttributeLastConfiguredByID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeSceneTableSizeWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeScenesID)
                                        attributeID:@(MTRAttributeIDTypeClusterScenesAttributeSceneTableSizeID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeRemainingCapacityWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeScenesID)
                                        attributeID:@(MTRAttributeIDTypeClusterScenesAttributeRemainingCapacityID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeScenesID)
                                        attributeID:@(MTRAttributeIDTypeClusterScenesAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeScenesID)
                                        attributeID:@(MTRAttributeIDTypeClusterScenesAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeScenesID)
                                        attributeID:@(MTRAttributeIDTypeClusterScenesAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeScenesID)
                                        attributeID:@(MTRAttributeIDTypeClusterScenesAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeScenesID)
                                        attributeID:@(MTRAttributeIDTypeClusterScenesAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeScenesID)
                                        attributeID:@(MTRAttributeIDTypeClusterScenesAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterScenes (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

- (void)addSceneWithParams:(MTRScenesClusterAddSceneParams *)params
            expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
     expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
         completionHandler:
             (void (^)(MTRScenesClusterAddSceneResponseParams * _Nullable data, NSError * _Nullable error))completionHandler
{
    [self addSceneWithParams:params
               expectedValues:expectedDataValueDictionaries
        expectedValueInterval:expectedValueIntervalMs
                   completion:^(MTRScenesClusterAddSceneResponseParams * _Nullable data, NSError * _Nullable error) {
                       // Cast is safe because subclass does not add any selectors.
                       completionHandler(static_cast<MTRScenesClusterAddSceneResponseParams *>(data), error);
                   }];
}
- (void)viewSceneWithParams:(MTRScenesClusterViewSceneParams *)params
             expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
      expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
          completionHandler:
              (void (^)(MTRScenesClusterViewSceneResponseParams * _Nullable data, NSError * _Nullable error))completionHandler
{
    [self viewSceneWithParams:params
               expectedValues:expectedDataValueDictionaries
        expectedValueInterval:expectedValueIntervalMs
                   completion:^(MTRScenesClusterViewSceneResponseParams * _Nullable data, NSError * _Nullable error) {
                       // Cast is safe because subclass does not add any selectors.
                       completionHandler(static_cast<MTRScenesClusterViewSceneResponseParams *>(data), error);
                   }];
}
- (void)removeSceneWithParams:(MTRScenesClusterRemoveSceneParams *)params
               expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
        expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
            completionHandler:
                (void (^)(MTRScenesClusterRemoveSceneResponseParams * _Nullable data, NSError * _Nullable error))completionHandler
{
    [self removeSceneWithParams:params
                 expectedValues:expectedDataValueDictionaries
          expectedValueInterval:expectedValueIntervalMs
                     completion:^(MTRScenesClusterRemoveSceneResponseParams * _Nullable data, NSError * _Nullable error) {
                         // Cast is safe because subclass does not add any selectors.
                         completionHandler(static_cast<MTRScenesClusterRemoveSceneResponseParams *>(data), error);
                     }];
}
- (void)removeAllScenesWithParams:(MTRScenesClusterRemoveAllScenesParams *)params
                   expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
            expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                completionHandler:(void (^)(MTRScenesClusterRemoveAllScenesResponseParams * _Nullable data,
                                      NSError * _Nullable error))completionHandler
{
    [self removeAllScenesWithParams:params
                     expectedValues:expectedDataValueDictionaries
              expectedValueInterval:expectedValueIntervalMs
                         completion:^(MTRScenesClusterRemoveAllScenesResponseParams * _Nullable data, NSError * _Nullable error) {
                             // Cast is safe because subclass does not add any selectors.
                             completionHandler(static_cast<MTRScenesClusterRemoveAllScenesResponseParams *>(data), error);
                         }];
}
- (void)storeSceneWithParams:(MTRScenesClusterStoreSceneParams *)params
              expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
       expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
           completionHandler:
               (void (^)(MTRScenesClusterStoreSceneResponseParams * _Nullable data, NSError * _Nullable error))completionHandler
{
    [self storeSceneWithParams:params
                expectedValues:expectedDataValueDictionaries
         expectedValueInterval:expectedValueIntervalMs
                    completion:^(MTRScenesClusterStoreSceneResponseParams * _Nullable data, NSError * _Nullable error) {
                        // Cast is safe because subclass does not add any selectors.
                        completionHandler(static_cast<MTRScenesClusterStoreSceneResponseParams *>(data), error);
                    }];
}
- (void)recallSceneWithParams:(MTRScenesClusterRecallSceneParams *)params
               expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
        expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
            completionHandler:(MTRStatusCompletion)completionHandler
{
    [self recallSceneWithParams:params
                 expectedValues:expectedDataValueDictionaries
          expectedValueInterval:expectedValueIntervalMs
                     completion:completionHandler];
}
- (void)getSceneMembershipWithParams:(MTRScenesClusterGetSceneMembershipParams *)params
                      expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
               expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                   completionHandler:(void (^)(MTRScenesClusterGetSceneMembershipResponseParams * _Nullable data,
                                         NSError * _Nullable error))completionHandler
{
    [self getSceneMembershipWithParams:params
                        expectedValues:expectedDataValueDictionaries
                 expectedValueInterval:expectedValueIntervalMs
                            completion:^(
                                MTRScenesClusterGetSceneMembershipResponseParams * _Nullable data, NSError * _Nullable error) {
                                // Cast is safe because subclass does not add any selectors.
                                completionHandler(static_cast<MTRScenesClusterGetSceneMembershipResponseParams *>(data), error);
                            }];
}
- (void)enhancedAddSceneWithParams:(MTRScenesClusterEnhancedAddSceneParams *)params
                    expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
             expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                 completionHandler:(void (^)(MTRScenesClusterEnhancedAddSceneResponseParams * _Nullable data,
                                       NSError * _Nullable error))completionHandler
{
    [self enhancedAddSceneWithParams:params
                      expectedValues:expectedDataValueDictionaries
               expectedValueInterval:expectedValueIntervalMs
                          completion:^(MTRScenesClusterEnhancedAddSceneResponseParams * _Nullable data, NSError * _Nullable error) {
                              // Cast is safe because subclass does not add any selectors.
                              completionHandler(static_cast<MTRScenesClusterEnhancedAddSceneResponseParams *>(data), error);
                          }];
}
- (void)enhancedViewSceneWithParams:(MTRScenesClusterEnhancedViewSceneParams *)params
                     expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
              expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                  completionHandler:(void (^)(MTRScenesClusterEnhancedViewSceneResponseParams * _Nullable data,
                                        NSError * _Nullable error))completionHandler
{
    [self
        enhancedViewSceneWithParams:params
                     expectedValues:expectedDataValueDictionaries
              expectedValueInterval:expectedValueIntervalMs
                         completion:^(MTRScenesClusterEnhancedViewSceneResponseParams * _Nullable data, NSError * _Nullable error) {
                             // Cast is safe because subclass does not add any selectors.
                             completionHandler(static_cast<MTRScenesClusterEnhancedViewSceneResponseParams *>(data), error);
                         }];
}
- (void)copySceneWithParams:(MTRScenesClusterCopySceneParams *)params
             expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
      expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
          completionHandler:
              (void (^)(MTRScenesClusterCopySceneResponseParams * _Nullable data, NSError * _Nullable error))completionHandler
{
    [self copySceneWithParams:params
               expectedValues:expectedDataValueDictionaries
        expectedValueInterval:expectedValueIntervalMs
                   completion:^(MTRScenesClusterCopySceneResponseParams * _Nullable data, NSError * _Nullable error) {
                       // Cast is safe because subclass does not add any selectors.
                       completionHandler(static_cast<MTRScenesClusterCopySceneResponseParams *>(data), error);
                   }];
}
@end

@implementation MTRClusterOnOff

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)offWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
        expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                   completion:(MTRStatusCompletion)completion
{
    [self offWithParams:nil expectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs completion:completion];
}
- (void)offWithParams:(MTROnOffClusterOffParams * _Nullable)params
           expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
               completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeOnOffID, (unsigned int) MTRCommandIDTypeClusterOnOffCommandOffID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                OnOff::Commands::Off::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)onWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
       expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                  completion:(MTRStatusCompletion)completion
{
    [self onWithParams:nil expectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs completion:completion];
}
- (void)onWithParams:(MTROnOffClusterOnParams * _Nullable)params
           expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
               completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeOnOffID, (unsigned int) MTRCommandIDTypeClusterOnOffCommandOnID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                OnOff::Commands::On::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)toggleWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
           expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                      completion:(MTRStatusCompletion)completion
{
    [self toggleWithParams:nil expectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs completion:completion];
}
- (void)toggleWithParams:(MTROnOffClusterToggleParams * _Nullable)params
           expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
               completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeOnOffID, (unsigned int) MTRCommandIDTypeClusterOnOffCommandToggleID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                OnOff::Commands::Toggle::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)offWithEffectWithParams:(MTROnOffClusterOffWithEffectParams *)params
                 expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
          expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                     completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeOnOffID, (unsigned int) MTRCommandIDTypeClusterOnOffCommandOffWithEffectID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                OnOff::Commands::OffWithEffect::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.effectIdentifier = static_cast<std::remove_reference_t<decltype(request.effectIdentifier)>>(
                    params.effectIdentifier.unsignedCharValue);
                request.effectVariant = params.effectVariant.unsignedCharValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)onWithRecallGlobalSceneWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                            expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                       completion:(MTRStatusCompletion)completion
{
    [self onWithRecallGlobalSceneWithParams:nil
                             expectedValues:expectedValues
                      expectedValueInterval:expectedValueIntervalMs
                                 completion:completion];
}
- (void)onWithRecallGlobalSceneWithParams:(MTROnOffClusterOnWithRecallGlobalSceneParams * _Nullable)params
                           expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                               completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeOnOffID,
                                     (unsigned int) MTRCommandIDTypeClusterOnOffCommandOnWithRecallGlobalSceneID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                OnOff::Commands::OnWithRecallGlobalScene::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)onWithTimedOffWithParams:(MTROnOffClusterOnWithTimedOffParams *)params
                  expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
           expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                      completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeOnOffID, (unsigned int) MTRCommandIDTypeClusterOnOffCommandOnWithTimedOffID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                OnOff::Commands::OnWithTimedOff::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.onOffControl
                    = static_cast<std::remove_reference_t<decltype(request.onOffControl)>>(params.onOffControl.unsignedCharValue);
                request.onTime = params.onTime.unsignedShortValue;
                request.offWaitTime = params.offWaitTime.unsignedShortValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeOnOffWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOnOffID)
                                        attributeID:@(MTRAttributeIDTypeClusterOnOffAttributeOnOffID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGlobalSceneControlWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOnOffID)
                                        attributeID:@(MTRAttributeIDTypeClusterOnOffAttributeGlobalSceneControlID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeOnTimeWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOnOffID)
                                        attributeID:@(MTRAttributeIDTypeClusterOnOffAttributeOnTimeID)
                                             params:params];
}

- (void)writeAttributeOnTimeWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeOnTimeWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeOnTimeWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                               params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeOnOffID)
                                  attributeID:@(MTRAttributeIDTypeClusterOnOffAttributeOnTimeID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeOffWaitTimeWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOnOffID)
                                        attributeID:@(MTRAttributeIDTypeClusterOnOffAttributeOffWaitTimeID)
                                             params:params];
}

- (void)writeAttributeOffWaitTimeWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                     expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeOffWaitTimeWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeOffWaitTimeWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                     expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                    params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeOnOffID)
                                  attributeID:@(MTRAttributeIDTypeClusterOnOffAttributeOffWaitTimeID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeStartUpOnOffWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOnOffID)
                                        attributeID:@(MTRAttributeIDTypeClusterOnOffAttributeStartUpOnOffID)
                                             params:params];
}

- (void)writeAttributeStartUpOnOffWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                      expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeStartUpOnOffWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeStartUpOnOffWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                      expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                     params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeOnOffID)
                                  attributeID:@(MTRAttributeIDTypeClusterOnOffAttributeStartUpOnOffID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOnOffID)
                                        attributeID:@(MTRAttributeIDTypeClusterOnOffAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOnOffID)
                                        attributeID:@(MTRAttributeIDTypeClusterOnOffAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOnOffID)
                                        attributeID:@(MTRAttributeIDTypeClusterOnOffAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOnOffID)
                                        attributeID:@(MTRAttributeIDTypeClusterOnOffAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOnOffID)
                                        attributeID:@(MTRAttributeIDTypeClusterOnOffAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOnOffID)
                                        attributeID:@(MTRAttributeIDTypeClusterOnOffAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterOnOff (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

- (void)offWithParams:(MTROnOffClusterOffParams * _Nullable)params
           expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
    expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
        completionHandler:(MTRStatusCompletion)completionHandler
{
    [self offWithParams:params
               expectedValues:expectedDataValueDictionaries
        expectedValueInterval:expectedValueIntervalMs
                   completion:completionHandler];
}
- (void)offWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
        expectedValueInterval:(NSNumber *)expectedValueIntervalMs
            completionHandler:(MTRStatusCompletion)completionHandler
{
    [self offWithParams:nil
               expectedValues:expectedValues
        expectedValueInterval:expectedValueIntervalMs
            completionHandler:completionHandler];
}
- (void)onWithParams:(MTROnOffClusterOnParams * _Nullable)params
           expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
    expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
        completionHandler:(MTRStatusCompletion)completionHandler
{
    [self onWithParams:params
               expectedValues:expectedDataValueDictionaries
        expectedValueInterval:expectedValueIntervalMs
                   completion:completionHandler];
}
- (void)onWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
       expectedValueInterval:(NSNumber *)expectedValueIntervalMs
           completionHandler:(MTRStatusCompletion)completionHandler
{
    [self onWithParams:nil
               expectedValues:expectedValues
        expectedValueInterval:expectedValueIntervalMs
            completionHandler:completionHandler];
}
- (void)toggleWithParams:(MTROnOffClusterToggleParams * _Nullable)params
           expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
    expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
        completionHandler:(MTRStatusCompletion)completionHandler
{
    [self toggleWithParams:params
               expectedValues:expectedDataValueDictionaries
        expectedValueInterval:expectedValueIntervalMs
                   completion:completionHandler];
}
- (void)toggleWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
           expectedValueInterval:(NSNumber *)expectedValueIntervalMs
               completionHandler:(MTRStatusCompletion)completionHandler
{
    [self toggleWithParams:nil
               expectedValues:expectedValues
        expectedValueInterval:expectedValueIntervalMs
            completionHandler:completionHandler];
}
- (void)offWithEffectWithParams:(MTROnOffClusterOffWithEffectParams *)params
                 expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
          expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
              completionHandler:(MTRStatusCompletion)completionHandler
{
    [self offWithEffectWithParams:params
                   expectedValues:expectedDataValueDictionaries
            expectedValueInterval:expectedValueIntervalMs
                       completion:completionHandler];
}
- (void)onWithRecallGlobalSceneWithParams:(MTROnOffClusterOnWithRecallGlobalSceneParams * _Nullable)params
                           expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
                    expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                        completionHandler:(MTRStatusCompletion)completionHandler
{
    [self onWithRecallGlobalSceneWithParams:params
                             expectedValues:expectedDataValueDictionaries
                      expectedValueInterval:expectedValueIntervalMs
                                 completion:completionHandler];
}
- (void)onWithRecallGlobalSceneWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                            expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                completionHandler:(MTRStatusCompletion)completionHandler
{
    [self onWithRecallGlobalSceneWithParams:nil
                             expectedValues:expectedValues
                      expectedValueInterval:expectedValueIntervalMs
                          completionHandler:completionHandler];
}
- (void)onWithTimedOffWithParams:(MTROnOffClusterOnWithTimedOffParams *)params
                  expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
           expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
               completionHandler:(MTRStatusCompletion)completionHandler
{
    [self onWithTimedOffWithParams:params
                    expectedValues:expectedDataValueDictionaries
             expectedValueInterval:expectedValueIntervalMs
                        completion:completionHandler];
}
@end

@implementation MTRClusterOnOffSwitchConfiguration

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (NSDictionary<NSString *, id> *)readAttributeSwitchTypeWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOnOffSwitchConfigurationID)
                                        attributeID:@(MTRAttributeIDTypeClusterOnOffSwitchConfigurationAttributeSwitchTypeID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeSwitchActionsWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOnOffSwitchConfigurationID)
                                        attributeID:@(MTRAttributeIDTypeClusterOnOffSwitchConfigurationAttributeSwitchActionsID)
                                             params:params];
}

- (void)writeAttributeSwitchActionsWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                       expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeSwitchActionsWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeSwitchActionsWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                       expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                      params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeOnOffSwitchConfigurationID)
                                  attributeID:@(MTRAttributeIDTypeClusterOnOffSwitchConfigurationAttributeSwitchActionsID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeOnOffSwitchConfigurationID)
                                     attributeID:@(MTRAttributeIDTypeClusterOnOffSwitchConfigurationAttributeGeneratedCommandListID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeOnOffSwitchConfigurationID)
                                     attributeID:@(MTRAttributeIDTypeClusterOnOffSwitchConfigurationAttributeAcceptedCommandListID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOnOffSwitchConfigurationID)
                                        attributeID:@(MTRAttributeIDTypeClusterOnOffSwitchConfigurationAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOnOffSwitchConfigurationID)
                                        attributeID:@(MTRAttributeIDTypeClusterOnOffSwitchConfigurationAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOnOffSwitchConfigurationID)
                                        attributeID:@(MTRAttributeIDTypeClusterOnOffSwitchConfigurationAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOnOffSwitchConfigurationID)
                                        attributeID:@(MTRAttributeIDTypeClusterOnOffSwitchConfigurationAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterOnOffSwitchConfiguration (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

@end

@implementation MTRClusterLevelControl

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)moveToLevelWithParams:(MTRLevelControlClusterMoveToLevelParams *)params
               expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
        expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                   completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeLevelControlID,
                                     (unsigned int) MTRCommandIDTypeClusterLevelControlCommandMoveToLevelID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                LevelControl::Commands::MoveToLevel::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.level = params.level.unsignedCharValue;
                if (params.transitionTime == nil) {
                    request.transitionTime.SetNull();
                } else {
                    auto & nonNullValue_0 = request.transitionTime.SetNonNull();
                    nonNullValue_0 = params.transitionTime.unsignedShortValue;
                }
                request.optionsMask
                    = static_cast<std::remove_reference_t<decltype(request.optionsMask)>>(params.optionsMask.unsignedCharValue);
                request.optionsOverride = static_cast<std::remove_reference_t<decltype(request.optionsOverride)>>(
                    params.optionsOverride.unsignedCharValue);

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)moveWithParams:(MTRLevelControlClusterMoveParams *)params
           expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
               completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeLevelControlID, (unsigned int) MTRCommandIDTypeClusterLevelControlCommandMoveID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                LevelControl::Commands::Move::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.moveMode
                    = static_cast<std::remove_reference_t<decltype(request.moveMode)>>(params.moveMode.unsignedCharValue);
                if (params.rate == nil) {
                    request.rate.SetNull();
                } else {
                    auto & nonNullValue_0 = request.rate.SetNonNull();
                    nonNullValue_0 = params.rate.unsignedCharValue;
                }
                request.optionsMask
                    = static_cast<std::remove_reference_t<decltype(request.optionsMask)>>(params.optionsMask.unsignedCharValue);
                request.optionsOverride = static_cast<std::remove_reference_t<decltype(request.optionsOverride)>>(
                    params.optionsOverride.unsignedCharValue);

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)stepWithParams:(MTRLevelControlClusterStepParams *)params
           expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
               completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeLevelControlID, (unsigned int) MTRCommandIDTypeClusterLevelControlCommandStepID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                LevelControl::Commands::Step::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.stepMode
                    = static_cast<std::remove_reference_t<decltype(request.stepMode)>>(params.stepMode.unsignedCharValue);
                request.stepSize = params.stepSize.unsignedCharValue;
                if (params.transitionTime == nil) {
                    request.transitionTime.SetNull();
                } else {
                    auto & nonNullValue_0 = request.transitionTime.SetNonNull();
                    nonNullValue_0 = params.transitionTime.unsignedShortValue;
                }
                request.optionsMask
                    = static_cast<std::remove_reference_t<decltype(request.optionsMask)>>(params.optionsMask.unsignedCharValue);
                request.optionsOverride = static_cast<std::remove_reference_t<decltype(request.optionsOverride)>>(
                    params.optionsOverride.unsignedCharValue);

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)stopWithParams:(MTRLevelControlClusterStopParams *)params
           expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
               completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeLevelControlID, (unsigned int) MTRCommandIDTypeClusterLevelControlCommandStopID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                LevelControl::Commands::Stop::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.optionsMask
                    = static_cast<std::remove_reference_t<decltype(request.optionsMask)>>(params.optionsMask.unsignedCharValue);
                request.optionsOverride = static_cast<std::remove_reference_t<decltype(request.optionsOverride)>>(
                    params.optionsOverride.unsignedCharValue);

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)moveToLevelWithOnOffWithParams:(MTRLevelControlClusterMoveToLevelWithOnOffParams *)params
                        expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                 expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                            completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeLevelControlID,
                                     (unsigned int) MTRCommandIDTypeClusterLevelControlCommandMoveToLevelWithOnOffID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                LevelControl::Commands::MoveToLevelWithOnOff::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.level = params.level.unsignedCharValue;
                if (params.transitionTime == nil) {
                    request.transitionTime.SetNull();
                } else {
                    auto & nonNullValue_0 = request.transitionTime.SetNonNull();
                    nonNullValue_0 = params.transitionTime.unsignedShortValue;
                }
                request.optionsMask
                    = static_cast<std::remove_reference_t<decltype(request.optionsMask)>>(params.optionsMask.unsignedCharValue);
                request.optionsOverride = static_cast<std::remove_reference_t<decltype(request.optionsOverride)>>(
                    params.optionsOverride.unsignedCharValue);

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)moveWithOnOffWithParams:(MTRLevelControlClusterMoveWithOnOffParams *)params
                 expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
          expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                     completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeLevelControlID,
                                     (unsigned int) MTRCommandIDTypeClusterLevelControlCommandMoveWithOnOffID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                LevelControl::Commands::MoveWithOnOff::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.moveMode
                    = static_cast<std::remove_reference_t<decltype(request.moveMode)>>(params.moveMode.unsignedCharValue);
                if (params.rate == nil) {
                    request.rate.SetNull();
                } else {
                    auto & nonNullValue_0 = request.rate.SetNonNull();
                    nonNullValue_0 = params.rate.unsignedCharValue;
                }
                request.optionsMask
                    = static_cast<std::remove_reference_t<decltype(request.optionsMask)>>(params.optionsMask.unsignedCharValue);
                request.optionsOverride = static_cast<std::remove_reference_t<decltype(request.optionsOverride)>>(
                    params.optionsOverride.unsignedCharValue);

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)stepWithOnOffWithParams:(MTRLevelControlClusterStepWithOnOffParams *)params
                 expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
          expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                     completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeLevelControlID,
                                     (unsigned int) MTRCommandIDTypeClusterLevelControlCommandStepWithOnOffID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                LevelControl::Commands::StepWithOnOff::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.stepMode
                    = static_cast<std::remove_reference_t<decltype(request.stepMode)>>(params.stepMode.unsignedCharValue);
                request.stepSize = params.stepSize.unsignedCharValue;
                if (params.transitionTime == nil) {
                    request.transitionTime.SetNull();
                } else {
                    auto & nonNullValue_0 = request.transitionTime.SetNonNull();
                    nonNullValue_0 = params.transitionTime.unsignedShortValue;
                }
                request.optionsMask
                    = static_cast<std::remove_reference_t<decltype(request.optionsMask)>>(params.optionsMask.unsignedCharValue);
                request.optionsOverride = static_cast<std::remove_reference_t<decltype(request.optionsOverride)>>(
                    params.optionsOverride.unsignedCharValue);

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)stopWithOnOffWithParams:(MTRLevelControlClusterStopWithOnOffParams *)params
                 expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
          expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                     completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeLevelControlID,
                                     (unsigned int) MTRCommandIDTypeClusterLevelControlCommandStopWithOnOffID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                LevelControl::Commands::StopWithOnOff::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.optionsMask
                    = static_cast<std::remove_reference_t<decltype(request.optionsMask)>>(params.optionsMask.unsignedCharValue);
                request.optionsOverride = static_cast<std::remove_reference_t<decltype(request.optionsOverride)>>(
                    params.optionsOverride.unsignedCharValue);

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)moveToClosestFrequencyWithParams:(MTRLevelControlClusterMoveToClosestFrequencyParams *)params
                          expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                   expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                              completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeLevelControlID,
                                     (unsigned int) MTRCommandIDTypeClusterLevelControlCommandMoveToClosestFrequencyID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                LevelControl::Commands::MoveToClosestFrequency::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.frequency = params.frequency.unsignedShortValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeCurrentLevelWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeLevelControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterLevelControlAttributeCurrentLevelID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeRemainingTimeWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeLevelControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterLevelControlAttributeRemainingTimeID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeMinLevelWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeLevelControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterLevelControlAttributeMinLevelID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeMaxLevelWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeLevelControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterLevelControlAttributeMaxLevelID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeCurrentFrequencyWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeLevelControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterLevelControlAttributeCurrentFrequencyID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeMinFrequencyWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeLevelControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterLevelControlAttributeMinFrequencyID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeMaxFrequencyWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeLevelControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterLevelControlAttributeMaxFrequencyID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeOptionsWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeLevelControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterLevelControlAttributeOptionsID)
                                             params:params];
}

- (void)writeAttributeOptionsWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                 expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeOptionsWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeOptionsWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                 expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeLevelControlID)
                                  attributeID:@(MTRAttributeIDTypeClusterLevelControlAttributeOptionsID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeOnOffTransitionTimeWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeLevelControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterLevelControlAttributeOnOffTransitionTimeID)
                                             params:params];
}

- (void)writeAttributeOnOffTransitionTimeWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                             expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeOnOffTransitionTimeWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeOnOffTransitionTimeWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                             expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                            params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeLevelControlID)
                                  attributeID:@(MTRAttributeIDTypeClusterLevelControlAttributeOnOffTransitionTimeID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeOnLevelWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeLevelControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterLevelControlAttributeOnLevelID)
                                             params:params];
}

- (void)writeAttributeOnLevelWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                 expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeOnLevelWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeOnLevelWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                 expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeLevelControlID)
                                  attributeID:@(MTRAttributeIDTypeClusterLevelControlAttributeOnLevelID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeOnTransitionTimeWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeLevelControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterLevelControlAttributeOnTransitionTimeID)
                                             params:params];
}

- (void)writeAttributeOnTransitionTimeWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                          expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeOnTransitionTimeWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeOnTransitionTimeWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                          expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                         params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeLevelControlID)
                                  attributeID:@(MTRAttributeIDTypeClusterLevelControlAttributeOnTransitionTimeID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeOffTransitionTimeWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeLevelControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterLevelControlAttributeOffTransitionTimeID)
                                             params:params];
}

- (void)writeAttributeOffTransitionTimeWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                           expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeOffTransitionTimeWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeOffTransitionTimeWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                           expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                          params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeLevelControlID)
                                  attributeID:@(MTRAttributeIDTypeClusterLevelControlAttributeOffTransitionTimeID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeDefaultMoveRateWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeLevelControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterLevelControlAttributeDefaultMoveRateID)
                                             params:params];
}

- (void)writeAttributeDefaultMoveRateWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                         expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeDefaultMoveRateWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeDefaultMoveRateWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                         expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                        params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeLevelControlID)
                                  attributeID:@(MTRAttributeIDTypeClusterLevelControlAttributeDefaultMoveRateID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeStartUpCurrentLevelWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeLevelControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterLevelControlAttributeStartUpCurrentLevelID)
                                             params:params];
}

- (void)writeAttributeStartUpCurrentLevelWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                             expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeStartUpCurrentLevelWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeStartUpCurrentLevelWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                             expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                            params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeLevelControlID)
                                  attributeID:@(MTRAttributeIDTypeClusterLevelControlAttributeStartUpCurrentLevelID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeLevelControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterLevelControlAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeLevelControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterLevelControlAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeLevelControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterLevelControlAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeLevelControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterLevelControlAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeLevelControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterLevelControlAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeLevelControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterLevelControlAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterLevelControl (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

- (void)moveToLevelWithParams:(MTRLevelControlClusterMoveToLevelParams *)params
               expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
        expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
            completionHandler:(MTRStatusCompletion)completionHandler
{
    [self moveToLevelWithParams:params
                 expectedValues:expectedDataValueDictionaries
          expectedValueInterval:expectedValueIntervalMs
                     completion:completionHandler];
}
- (void)moveWithParams:(MTRLevelControlClusterMoveParams *)params
           expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
    expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
        completionHandler:(MTRStatusCompletion)completionHandler
{
    [self moveWithParams:params
               expectedValues:expectedDataValueDictionaries
        expectedValueInterval:expectedValueIntervalMs
                   completion:completionHandler];
}
- (void)stepWithParams:(MTRLevelControlClusterStepParams *)params
           expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
    expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
        completionHandler:(MTRStatusCompletion)completionHandler
{
    [self stepWithParams:params
               expectedValues:expectedDataValueDictionaries
        expectedValueInterval:expectedValueIntervalMs
                   completion:completionHandler];
}
- (void)stopWithParams:(MTRLevelControlClusterStopParams *)params
           expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
    expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
        completionHandler:(MTRStatusCompletion)completionHandler
{
    [self stopWithParams:params
               expectedValues:expectedDataValueDictionaries
        expectedValueInterval:expectedValueIntervalMs
                   completion:completionHandler];
}
- (void)moveToLevelWithOnOffWithParams:(MTRLevelControlClusterMoveToLevelWithOnOffParams *)params
                        expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
                 expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                     completionHandler:(MTRStatusCompletion)completionHandler
{
    [self moveToLevelWithOnOffWithParams:params
                          expectedValues:expectedDataValueDictionaries
                   expectedValueInterval:expectedValueIntervalMs
                              completion:completionHandler];
}
- (void)moveWithOnOffWithParams:(MTRLevelControlClusterMoveWithOnOffParams *)params
                 expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
          expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
              completionHandler:(MTRStatusCompletion)completionHandler
{
    [self moveWithOnOffWithParams:params
                   expectedValues:expectedDataValueDictionaries
            expectedValueInterval:expectedValueIntervalMs
                       completion:completionHandler];
}
- (void)stepWithOnOffWithParams:(MTRLevelControlClusterStepWithOnOffParams *)params
                 expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
          expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
              completionHandler:(MTRStatusCompletion)completionHandler
{
    [self stepWithOnOffWithParams:params
                   expectedValues:expectedDataValueDictionaries
            expectedValueInterval:expectedValueIntervalMs
                       completion:completionHandler];
}
- (void)stopWithOnOffWithParams:(MTRLevelControlClusterStopWithOnOffParams *)params
                 expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
          expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
              completionHandler:(MTRStatusCompletion)completionHandler
{
    [self stopWithOnOffWithParams:params
                   expectedValues:expectedDataValueDictionaries
            expectedValueInterval:expectedValueIntervalMs
                       completion:completionHandler];
}
- (void)moveToClosestFrequencyWithParams:(MTRLevelControlClusterMoveToClosestFrequencyParams *)params
                          expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
                   expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                       completionHandler:(MTRStatusCompletion)completionHandler
{
    [self moveToClosestFrequencyWithParams:params
                            expectedValues:expectedDataValueDictionaries
                     expectedValueInterval:expectedValueIntervalMs
                                completion:completionHandler];
}
@end

@implementation MTRClusterBinaryInputBasic

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (NSDictionary<NSString *, id> *)readAttributeActiveTextWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBinaryInputBasicID)
                                        attributeID:@(MTRAttributeIDTypeClusterBinaryInputBasicAttributeActiveTextID)
                                             params:params];
}

- (void)writeAttributeActiveTextWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeActiveTextWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeActiveTextWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                   params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeBinaryInputBasicID)
                                  attributeID:@(MTRAttributeIDTypeClusterBinaryInputBasicAttributeActiveTextID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeDescriptionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBinaryInputBasicID)
                                        attributeID:@(MTRAttributeIDTypeClusterBinaryInputBasicAttributeDescriptionID)
                                             params:params];
}

- (void)writeAttributeDescriptionWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                     expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeDescriptionWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeDescriptionWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                     expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                    params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeBinaryInputBasicID)
                                  attributeID:@(MTRAttributeIDTypeClusterBinaryInputBasicAttributeDescriptionID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeInactiveTextWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBinaryInputBasicID)
                                        attributeID:@(MTRAttributeIDTypeClusterBinaryInputBasicAttributeInactiveTextID)
                                             params:params];
}

- (void)writeAttributeInactiveTextWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                      expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeInactiveTextWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeInactiveTextWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                      expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                     params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeBinaryInputBasicID)
                                  attributeID:@(MTRAttributeIDTypeClusterBinaryInputBasicAttributeInactiveTextID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeOutOfServiceWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBinaryInputBasicID)
                                        attributeID:@(MTRAttributeIDTypeClusterBinaryInputBasicAttributeOutOfServiceID)
                                             params:params];
}

- (void)writeAttributeOutOfServiceWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                      expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeOutOfServiceWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeOutOfServiceWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                      expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                     params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeBinaryInputBasicID)
                                  attributeID:@(MTRAttributeIDTypeClusterBinaryInputBasicAttributeOutOfServiceID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributePolarityWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBinaryInputBasicID)
                                        attributeID:@(MTRAttributeIDTypeClusterBinaryInputBasicAttributePolarityID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributePresentValueWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBinaryInputBasicID)
                                        attributeID:@(MTRAttributeIDTypeClusterBinaryInputBasicAttributePresentValueID)
                                             params:params];
}

- (void)writeAttributePresentValueWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                      expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributePresentValueWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributePresentValueWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                      expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                     params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeBinaryInputBasicID)
                                  attributeID:@(MTRAttributeIDTypeClusterBinaryInputBasicAttributePresentValueID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeReliabilityWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBinaryInputBasicID)
                                        attributeID:@(MTRAttributeIDTypeClusterBinaryInputBasicAttributeReliabilityID)
                                             params:params];
}

- (void)writeAttributeReliabilityWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                     expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeReliabilityWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeReliabilityWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                     expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                    params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeBinaryInputBasicID)
                                  attributeID:@(MTRAttributeIDTypeClusterBinaryInputBasicAttributeReliabilityID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeStatusFlagsWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBinaryInputBasicID)
                                        attributeID:@(MTRAttributeIDTypeClusterBinaryInputBasicAttributeStatusFlagsID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeApplicationTypeWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBinaryInputBasicID)
                                        attributeID:@(MTRAttributeIDTypeClusterBinaryInputBasicAttributeApplicationTypeID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBinaryInputBasicID)
                                        attributeID:@(MTRAttributeIDTypeClusterBinaryInputBasicAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBinaryInputBasicID)
                                        attributeID:@(MTRAttributeIDTypeClusterBinaryInputBasicAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBinaryInputBasicID)
                                        attributeID:@(MTRAttributeIDTypeClusterBinaryInputBasicAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBinaryInputBasicID)
                                        attributeID:@(MTRAttributeIDTypeClusterBinaryInputBasicAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBinaryInputBasicID)
                                        attributeID:@(MTRAttributeIDTypeClusterBinaryInputBasicAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBinaryInputBasicID)
                                        attributeID:@(MTRAttributeIDTypeClusterBinaryInputBasicAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterBinaryInputBasic (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

@end

@implementation MTRClusterDescriptor

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (NSDictionary<NSString *, id> *)readAttributeDeviceTypeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDescriptorID)
                                        attributeID:@(MTRAttributeIDTypeClusterDescriptorAttributeDeviceTypeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeServerListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDescriptorID)
                                        attributeID:@(MTRAttributeIDTypeClusterDescriptorAttributeServerListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClientListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDescriptorID)
                                        attributeID:@(MTRAttributeIDTypeClusterDescriptorAttributeClientListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributePartsListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDescriptorID)
                                        attributeID:@(MTRAttributeIDTypeClusterDescriptorAttributePartsListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDescriptorID)
                                        attributeID:@(MTRAttributeIDTypeClusterDescriptorAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDescriptorID)
                                        attributeID:@(MTRAttributeIDTypeClusterDescriptorAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDescriptorID)
                                        attributeID:@(MTRAttributeIDTypeClusterDescriptorAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDescriptorID)
                                        attributeID:@(MTRAttributeIDTypeClusterDescriptorAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDescriptorID)
                                        attributeID:@(MTRAttributeIDTypeClusterDescriptorAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDescriptorID)
                                        attributeID:@(MTRAttributeIDTypeClusterDescriptorAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterDescriptor (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

- (NSDictionary<NSString *, id> *)readAttributeDeviceListWithParams:(MTRReadParams * _Nullable)params
{
    return [self readAttributeDeviceTypeListWithParams:params];
}
@end

@implementation MTRClusterBinding

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (NSDictionary<NSString *, id> *)readAttributeBindingWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBindingID)
                                        attributeID:@(MTRAttributeIDTypeClusterBindingAttributeBindingID)
                                             params:params];
}

- (void)writeAttributeBindingWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                 expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeBindingWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeBindingWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                 expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeBindingID)
                                  attributeID:@(MTRAttributeIDTypeClusterBindingAttributeBindingID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBindingID)
                                        attributeID:@(MTRAttributeIDTypeClusterBindingAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBindingID)
                                        attributeID:@(MTRAttributeIDTypeClusterBindingAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBindingID)
                                        attributeID:@(MTRAttributeIDTypeClusterBindingAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBindingID)
                                        attributeID:@(MTRAttributeIDTypeClusterBindingAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBindingID)
                                        attributeID:@(MTRAttributeIDTypeClusterBindingAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBindingID)
                                        attributeID:@(MTRAttributeIDTypeClusterBindingAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterBinding (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

@end

@implementation MTRClusterAccessControl

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (NSDictionary<NSString *, id> *)readAttributeACLWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeAccessControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterAccessControlAttributeACLID)
                                             params:params];
}

- (void)writeAttributeACLWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
             expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeACLWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeACLWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
             expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                            params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeAccessControlID)
                                  attributeID:@(MTRAttributeIDTypeClusterAccessControlAttributeACLID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeExtensionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeAccessControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterAccessControlAttributeExtensionID)
                                             params:params];
}

- (void)writeAttributeExtensionWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                   expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeExtensionWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeExtensionWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                   expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                  params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeAccessControlID)
                                  attributeID:@(MTRAttributeIDTypeClusterAccessControlAttributeExtensionID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeSubjectsPerAccessControlEntryWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeAccessControlID)
                                     attributeID:@(MTRAttributeIDTypeClusterAccessControlAttributeSubjectsPerAccessControlEntryID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeTargetsPerAccessControlEntryWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeAccessControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterAccessControlAttributeTargetsPerAccessControlEntryID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAccessControlEntriesPerFabricWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeAccessControlID)
                                     attributeID:@(MTRAttributeIDTypeClusterAccessControlAttributeAccessControlEntriesPerFabricID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeAccessControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterAccessControlAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeAccessControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterAccessControlAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeAccessControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterAccessControlAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeAccessControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterAccessControlAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeAccessControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterAccessControlAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeAccessControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterAccessControlAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterAccessControl (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

- (NSDictionary<NSString *, id> *)readAttributeAclWithParams:(MTRReadParams * _Nullable)params
{
    return [self readAttributeACLWithParams:params];
}
- (void)writeAttributeAclWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
             expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeACLWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs];
}
- (void)writeAttributeAclWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
             expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                            params:(MTRWriteParams * _Nullable)params
{
    [self writeAttributeACLWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:params];
}
@end

@implementation MTRClusterActions

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)instantActionWithParams:(MTRActionsClusterInstantActionParams *)params
                 expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
          expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                     completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeActionsID, (unsigned int) MTRCommandIDTypeClusterActionsCommandInstantActionID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                Actions::Commands::InstantAction::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.actionID = params.actionID.unsignedShortValue;
                if (params.invokeID != nil) {
                    auto & definedValue_0 = request.invokeID.Emplace();
                    definedValue_0 = params.invokeID.unsignedIntValue;
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)instantActionWithTransitionWithParams:(MTRActionsClusterInstantActionWithTransitionParams *)params
                               expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                        expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                   completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeActionsID,
                                     (unsigned int) MTRCommandIDTypeClusterActionsCommandInstantActionWithTransitionID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                Actions::Commands::InstantActionWithTransition::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.actionID = params.actionID.unsignedShortValue;
                if (params.invokeID != nil) {
                    auto & definedValue_0 = request.invokeID.Emplace();
                    definedValue_0 = params.invokeID.unsignedIntValue;
                }
                request.transitionTime = params.transitionTime.unsignedShortValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)startActionWithParams:(MTRActionsClusterStartActionParams *)params
               expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
        expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                   completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeActionsID, (unsigned int) MTRCommandIDTypeClusterActionsCommandStartActionID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                Actions::Commands::StartAction::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.actionID = params.actionID.unsignedShortValue;
                if (params.invokeID != nil) {
                    auto & definedValue_0 = request.invokeID.Emplace();
                    definedValue_0 = params.invokeID.unsignedIntValue;
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)startActionWithDurationWithParams:(MTRActionsClusterStartActionWithDurationParams *)params
                           expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                               completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeActionsID,
                                     (unsigned int) MTRCommandIDTypeClusterActionsCommandStartActionWithDurationID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                Actions::Commands::StartActionWithDuration::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.actionID = params.actionID.unsignedShortValue;
                if (params.invokeID != nil) {
                    auto & definedValue_0 = request.invokeID.Emplace();
                    definedValue_0 = params.invokeID.unsignedIntValue;
                }
                request.duration = params.duration.unsignedIntValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)stopActionWithParams:(MTRActionsClusterStopActionParams *)params
              expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
       expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                  completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeActionsID, (unsigned int) MTRCommandIDTypeClusterActionsCommandStopActionID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                Actions::Commands::StopAction::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.actionID = params.actionID.unsignedShortValue;
                if (params.invokeID != nil) {
                    auto & definedValue_0 = request.invokeID.Emplace();
                    definedValue_0 = params.invokeID.unsignedIntValue;
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)pauseActionWithParams:(MTRActionsClusterPauseActionParams *)params
               expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
        expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                   completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeActionsID, (unsigned int) MTRCommandIDTypeClusterActionsCommandPauseActionID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                Actions::Commands::PauseAction::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.actionID = params.actionID.unsignedShortValue;
                if (params.invokeID != nil) {
                    auto & definedValue_0 = request.invokeID.Emplace();
                    definedValue_0 = params.invokeID.unsignedIntValue;
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)pauseActionWithDurationWithParams:(MTRActionsClusterPauseActionWithDurationParams *)params
                           expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                               completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeActionsID,
                                     (unsigned int) MTRCommandIDTypeClusterActionsCommandPauseActionWithDurationID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                Actions::Commands::PauseActionWithDuration::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.actionID = params.actionID.unsignedShortValue;
                if (params.invokeID != nil) {
                    auto & definedValue_0 = request.invokeID.Emplace();
                    definedValue_0 = params.invokeID.unsignedIntValue;
                }
                request.duration = params.duration.unsignedIntValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)resumeActionWithParams:(MTRActionsClusterResumeActionParams *)params
                expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
         expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                    completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeActionsID, (unsigned int) MTRCommandIDTypeClusterActionsCommandResumeActionID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                Actions::Commands::ResumeAction::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.actionID = params.actionID.unsignedShortValue;
                if (params.invokeID != nil) {
                    auto & definedValue_0 = request.invokeID.Emplace();
                    definedValue_0 = params.invokeID.unsignedIntValue;
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)enableActionWithParams:(MTRActionsClusterEnableActionParams *)params
                expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
         expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                    completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeActionsID, (unsigned int) MTRCommandIDTypeClusterActionsCommandEnableActionID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                Actions::Commands::EnableAction::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.actionID = params.actionID.unsignedShortValue;
                if (params.invokeID != nil) {
                    auto & definedValue_0 = request.invokeID.Emplace();
                    definedValue_0 = params.invokeID.unsignedIntValue;
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)enableActionWithDurationWithParams:(MTRActionsClusterEnableActionWithDurationParams *)params
                            expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                     expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeActionsID,
                                     (unsigned int) MTRCommandIDTypeClusterActionsCommandEnableActionWithDurationID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                Actions::Commands::EnableActionWithDuration::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.actionID = params.actionID.unsignedShortValue;
                if (params.invokeID != nil) {
                    auto & definedValue_0 = request.invokeID.Emplace();
                    definedValue_0 = params.invokeID.unsignedIntValue;
                }
                request.duration = params.duration.unsignedIntValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)disableActionWithParams:(MTRActionsClusterDisableActionParams *)params
                 expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
          expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                     completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeActionsID, (unsigned int) MTRCommandIDTypeClusterActionsCommandDisableActionID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                Actions::Commands::DisableAction::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.actionID = params.actionID.unsignedShortValue;
                if (params.invokeID != nil) {
                    auto & definedValue_0 = request.invokeID.Emplace();
                    definedValue_0 = params.invokeID.unsignedIntValue;
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)disableActionWithDurationWithParams:(MTRActionsClusterDisableActionWithDurationParams *)params
                             expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                      expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                 completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeActionsID,
                                     (unsigned int) MTRCommandIDTypeClusterActionsCommandDisableActionWithDurationID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                Actions::Commands::DisableActionWithDuration::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.actionID = params.actionID.unsignedShortValue;
                if (params.invokeID != nil) {
                    auto & definedValue_0 = request.invokeID.Emplace();
                    definedValue_0 = params.invokeID.unsignedIntValue;
                }
                request.duration = params.duration.unsignedIntValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeActionListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeActionsID)
                                        attributeID:@(MTRAttributeIDTypeClusterActionsAttributeActionListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEndpointListsWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeActionsID)
                                        attributeID:@(MTRAttributeIDTypeClusterActionsAttributeEndpointListsID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeSetupURLWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeActionsID)
                                        attributeID:@(MTRAttributeIDTypeClusterActionsAttributeSetupURLID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeActionsID)
                                        attributeID:@(MTRAttributeIDTypeClusterActionsAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeActionsID)
                                        attributeID:@(MTRAttributeIDTypeClusterActionsAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeActionsID)
                                        attributeID:@(MTRAttributeIDTypeClusterActionsAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeActionsID)
                                        attributeID:@(MTRAttributeIDTypeClusterActionsAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeActionsID)
                                        attributeID:@(MTRAttributeIDTypeClusterActionsAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeActionsID)
                                        attributeID:@(MTRAttributeIDTypeClusterActionsAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterActions (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

- (void)instantActionWithParams:(MTRActionsClusterInstantActionParams *)params
                 expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
          expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
              completionHandler:(MTRStatusCompletion)completionHandler
{
    [self instantActionWithParams:params
                   expectedValues:expectedDataValueDictionaries
            expectedValueInterval:expectedValueIntervalMs
                       completion:completionHandler];
}
- (void)instantActionWithTransitionWithParams:(MTRActionsClusterInstantActionWithTransitionParams *)params
                               expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
                        expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                            completionHandler:(MTRStatusCompletion)completionHandler
{
    [self instantActionWithTransitionWithParams:params
                                 expectedValues:expectedDataValueDictionaries
                          expectedValueInterval:expectedValueIntervalMs
                                     completion:completionHandler];
}
- (void)startActionWithParams:(MTRActionsClusterStartActionParams *)params
               expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
        expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
            completionHandler:(MTRStatusCompletion)completionHandler
{
    [self startActionWithParams:params
                 expectedValues:expectedDataValueDictionaries
          expectedValueInterval:expectedValueIntervalMs
                     completion:completionHandler];
}
- (void)startActionWithDurationWithParams:(MTRActionsClusterStartActionWithDurationParams *)params
                           expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
                    expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                        completionHandler:(MTRStatusCompletion)completionHandler
{
    [self startActionWithDurationWithParams:params
                             expectedValues:expectedDataValueDictionaries
                      expectedValueInterval:expectedValueIntervalMs
                                 completion:completionHandler];
}
- (void)stopActionWithParams:(MTRActionsClusterStopActionParams *)params
              expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
       expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
           completionHandler:(MTRStatusCompletion)completionHandler
{
    [self stopActionWithParams:params
                expectedValues:expectedDataValueDictionaries
         expectedValueInterval:expectedValueIntervalMs
                    completion:completionHandler];
}
- (void)pauseActionWithParams:(MTRActionsClusterPauseActionParams *)params
               expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
        expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
            completionHandler:(MTRStatusCompletion)completionHandler
{
    [self pauseActionWithParams:params
                 expectedValues:expectedDataValueDictionaries
          expectedValueInterval:expectedValueIntervalMs
                     completion:completionHandler];
}
- (void)pauseActionWithDurationWithParams:(MTRActionsClusterPauseActionWithDurationParams *)params
                           expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
                    expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                        completionHandler:(MTRStatusCompletion)completionHandler
{
    [self pauseActionWithDurationWithParams:params
                             expectedValues:expectedDataValueDictionaries
                      expectedValueInterval:expectedValueIntervalMs
                                 completion:completionHandler];
}
- (void)resumeActionWithParams:(MTRActionsClusterResumeActionParams *)params
                expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
         expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
             completionHandler:(MTRStatusCompletion)completionHandler
{
    [self resumeActionWithParams:params
                  expectedValues:expectedDataValueDictionaries
           expectedValueInterval:expectedValueIntervalMs
                      completion:completionHandler];
}
- (void)enableActionWithParams:(MTRActionsClusterEnableActionParams *)params
                expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
         expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
             completionHandler:(MTRStatusCompletion)completionHandler
{
    [self enableActionWithParams:params
                  expectedValues:expectedDataValueDictionaries
           expectedValueInterval:expectedValueIntervalMs
                      completion:completionHandler];
}
- (void)enableActionWithDurationWithParams:(MTRActionsClusterEnableActionWithDurationParams *)params
                            expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
                     expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                         completionHandler:(MTRStatusCompletion)completionHandler
{
    [self enableActionWithDurationWithParams:params
                              expectedValues:expectedDataValueDictionaries
                       expectedValueInterval:expectedValueIntervalMs
                                  completion:completionHandler];
}
- (void)disableActionWithParams:(MTRActionsClusterDisableActionParams *)params
                 expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
          expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
              completionHandler:(MTRStatusCompletion)completionHandler
{
    [self disableActionWithParams:params
                   expectedValues:expectedDataValueDictionaries
            expectedValueInterval:expectedValueIntervalMs
                       completion:completionHandler];
}
- (void)disableActionWithDurationWithParams:(MTRActionsClusterDisableActionWithDurationParams *)params
                             expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
                      expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                          completionHandler:(MTRStatusCompletion)completionHandler
{
    [self disableActionWithDurationWithParams:params
                               expectedValues:expectedDataValueDictionaries
                        expectedValueInterval:expectedValueIntervalMs
                                   completion:completionHandler];
}
@end

@implementation MTRClusterBasicInformation

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)mfgSpecificPingWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                               completion:(MTRStatusCompletion)completion
{
    [self mfgSpecificPingWithParams:nil
                     expectedValues:expectedValues
              expectedValueInterval:expectedValueIntervalMs
                         completion:completion];
}
- (void)mfgSpecificPingWithParams:(MTRBasicClusterMfgSpecificPingParams * _Nullable)params
                   expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
            expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                       completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) 0x00000028, (unsigned int) 0x10020000];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                BasicInformation::Commands::MfgSpecificPing::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeDataModelRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBasicInformationAttributeDataModelRevisionID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeVendorNameWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBasicInformationAttributeVendorNameID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeVendorIDWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBasicInformationAttributeVendorIDID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeProductNameWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBasicInformationAttributeProductNameID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeProductIDWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBasicInformationAttributeProductIDID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeNodeLabelWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBasicInformationAttributeNodeLabelID)
                                             params:params];
}

- (void)writeAttributeNodeLabelWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                   expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeNodeLabelWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeNodeLabelWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                   expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                  params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeBasicInformationID)
                                  attributeID:@(MTRAttributeIDTypeClusterBasicInformationAttributeNodeLabelID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeLocationWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBasicInformationAttributeLocationID)
                                             params:params];
}

- (void)writeAttributeLocationWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                  expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeLocationWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeLocationWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                  expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                 params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeBasicInformationID)
                                  attributeID:@(MTRAttributeIDTypeClusterBasicInformationAttributeLocationID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeHardwareVersionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBasicInformationAttributeHardwareVersionID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeHardwareVersionStringWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBasicInformationAttributeHardwareVersionStringID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeSoftwareVersionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBasicInformationAttributeSoftwareVersionID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeSoftwareVersionStringWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBasicInformationAttributeSoftwareVersionStringID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeManufacturingDateWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBasicInformationAttributeManufacturingDateID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributePartNumberWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBasicInformationAttributePartNumberID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeProductURLWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBasicInformationAttributeProductURLID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeProductLabelWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBasicInformationAttributeProductLabelID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeSerialNumberWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBasicInformationAttributeSerialNumberID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeLocalConfigDisabledWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBasicInformationAttributeLocalConfigDisabledID)
                                             params:params];
}

- (void)writeAttributeLocalConfigDisabledWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                             expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeLocalConfigDisabledWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeLocalConfigDisabledWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                             expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                            params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeBasicInformationID)
                                  attributeID:@(MTRAttributeIDTypeClusterBasicInformationAttributeLocalConfigDisabledID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeReachableWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBasicInformationAttributeReachableID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeUniqueIDWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBasicInformationAttributeUniqueIDID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeCapabilityMinimaWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBasicInformationAttributeCapabilityMinimaID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeProductAppearanceWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBasicInformationAttributeProductAppearanceID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBasicInformationAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBasicInformationAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBasicInformationAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBasicInformationAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBasicInformationAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBasicInformationAttributeClusterRevisionID)
                                             params:params];
}

@end
@implementation MTRClusterBasic
@end

@implementation MTRClusterBasic (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

- (void)mfgSpecificPingWithParams:(MTRBasicClusterMfgSpecificPingParams * _Nullable)params
                   expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
            expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                completionHandler:(MTRStatusCompletion)completionHandler
{
    [self mfgSpecificPingWithParams:params
                     expectedValues:expectedDataValueDictionaries
              expectedValueInterval:expectedValueIntervalMs
                         completion:completionHandler];
}
- (void)mfgSpecificPingWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                        completionHandler:(MTRStatusCompletion)completionHandler
{
    [self mfgSpecificPingWithParams:nil
                     expectedValues:expectedValues
              expectedValueInterval:expectedValueIntervalMs
                  completionHandler:completionHandler];
}
@end

@implementation MTRClusterOTASoftwareUpdateProvider

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)queryImageWithParams:(MTROTASoftwareUpdateProviderClusterQueryImageParams *)params
              expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
       expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                  completion:(void (^)(MTROTASoftwareUpdateProviderClusterQueryImageResponseParams * _Nullable data,
                                 NSError * _Nullable error))completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeOTASoftwareUpdateProviderID,
                                     (unsigned int) MTRCommandIDTypeClusterOTASoftwareUpdateProviderCommandQueryImageID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTROTASoftwareUpdateProviderClusterQueryImageResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                OTASoftwareUpdateProviderClusterQueryImageResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTROTASoftwareUpdateProviderClusterQueryImageResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                OtaSoftwareUpdateProvider::Commands::QueryImage::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.vendorID
                    = static_cast<std::remove_reference_t<decltype(request.vendorID)>>(params.vendorID.unsignedShortValue);
                request.productID = params.productID.unsignedShortValue;
                request.softwareVersion = params.softwareVersion.unsignedIntValue;
                {
                    using ListType_0 = std::remove_reference_t<decltype(request.protocolsSupported)>;
                    using ListMemberType_0 = ListMemberTypeGetter<ListType_0>::Type;
                    if (params.protocolsSupported.count != 0) {
                        auto * listHolder_0 = new ListHolder<ListMemberType_0>(params.protocolsSupported.count);
                        if (listHolder_0 == nullptr || listHolder_0->mList == nullptr) {
                            return CHIP_ERROR_INVALID_ARGUMENT;
                        }
                        listFreer.add(listHolder_0);
                        for (size_t i_0 = 0; i_0 < params.protocolsSupported.count; ++i_0) {
                            if (![params.protocolsSupported[i_0] isKindOfClass:[NSNumber class]]) {
                                // Wrong kind of value.
                                return CHIP_ERROR_INVALID_ARGUMENT;
                            }
                            auto element_0 = (NSNumber *) params.protocolsSupported[i_0];
                            listHolder_0->mList[i_0] = static_cast<std::remove_reference_t<decltype(listHolder_0->mList[i_0])>>(
                                element_0.unsignedCharValue);
                        }
                        request.protocolsSupported = ListType_0(listHolder_0->mList, params.protocolsSupported.count);
                    } else {
                        request.protocolsSupported = ListType_0();
                    }
                }
                if (params.hardwareVersion != nil) {
                    auto & definedValue_0 = request.hardwareVersion.Emplace();
                    definedValue_0 = params.hardwareVersion.unsignedShortValue;
                }
                if (params.location != nil) {
                    auto & definedValue_0 = request.location.Emplace();
                    definedValue_0 = [self asCharSpan:params.location];
                }
                if (params.requestorCanConsent != nil) {
                    auto & definedValue_0 = request.requestorCanConsent.Emplace();
                    definedValue_0 = params.requestorCanConsent.boolValue;
                }
                if (params.metadataForProvider != nil) {
                    auto & definedValue_0 = request.metadataForProvider.Emplace();
                    definedValue_0 = [self asByteSpan:params.metadataForProvider];
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)applyUpdateRequestWithParams:(MTROTASoftwareUpdateProviderClusterApplyUpdateRequestParams *)params
                      expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
               expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                          completion:(void (^)(MTROTASoftwareUpdateProviderClusterApplyUpdateResponseParams * _Nullable data,
                                         NSError * _Nullable error))completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeOTASoftwareUpdateProviderID,
                                     (unsigned int) MTRCommandIDTypeClusterOTASoftwareUpdateProviderCommandApplyUpdateRequestID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTROTASoftwareUpdateProviderClusterApplyUpdateResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                OTASoftwareUpdateProviderClusterApplyUpdateResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTROTASoftwareUpdateProviderClusterApplyUpdateResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                OtaSoftwareUpdateProvider::Commands::ApplyUpdateRequest::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.updateToken = [self asByteSpan:params.updateToken];
                request.newVersion = params.newVersion.unsignedIntValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)notifyUpdateAppliedWithParams:(MTROTASoftwareUpdateProviderClusterNotifyUpdateAppliedParams *)params
                       expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                           completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeOTASoftwareUpdateProviderID,
                                     (unsigned int) MTRCommandIDTypeClusterOTASoftwareUpdateProviderCommandNotifyUpdateAppliedID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                OtaSoftwareUpdateProvider::Commands::NotifyUpdateApplied::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.updateToken = [self asByteSpan:params.updateToken];
                request.softwareVersion = params.softwareVersion.unsignedIntValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeOTASoftwareUpdateProviderID)
                        attributeID:@(MTRAttributeIDTypeClusterOTASoftwareUpdateProviderAttributeGeneratedCommandListID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeOTASoftwareUpdateProviderID)
                                     attributeID:@(MTRAttributeIDTypeClusterOTASoftwareUpdateProviderAttributeAcceptedCommandListID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOTASoftwareUpdateProviderID)
                                        attributeID:@(MTRAttributeIDTypeClusterOTASoftwareUpdateProviderAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOTASoftwareUpdateProviderID)
                                        attributeID:@(MTRAttributeIDTypeClusterOTASoftwareUpdateProviderAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOTASoftwareUpdateProviderID)
                                        attributeID:@(MTRAttributeIDTypeClusterOTASoftwareUpdateProviderAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOTASoftwareUpdateProviderID)
                                        attributeID:@(MTRAttributeIDTypeClusterOTASoftwareUpdateProviderAttributeClusterRevisionID)
                                             params:params];
}

@end
@implementation MTRClusterOtaSoftwareUpdateProvider
@end

@implementation MTRClusterOtaSoftwareUpdateProvider (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

- (void)queryImageWithParams:(MTROtaSoftwareUpdateProviderClusterQueryImageParams *)params
              expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
       expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
           completionHandler:(void (^)(MTROtaSoftwareUpdateProviderClusterQueryImageResponseParams * _Nullable data,
                                 NSError * _Nullable error))completionHandler
{
    [self queryImageWithParams:params
                expectedValues:expectedDataValueDictionaries
         expectedValueInterval:expectedValueIntervalMs
                    completion:^(
                        MTROTASoftwareUpdateProviderClusterQueryImageResponseParams * _Nullable data, NSError * _Nullable error) {
                        // Cast is safe because subclass does not add any selectors.
                        completionHandler(static_cast<MTROtaSoftwareUpdateProviderClusterQueryImageResponseParams *>(data), error);
                    }];
}
- (void)applyUpdateRequestWithParams:(MTROtaSoftwareUpdateProviderClusterApplyUpdateRequestParams *)params
                      expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
               expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                   completionHandler:(void (^)(MTROtaSoftwareUpdateProviderClusterApplyUpdateResponseParams * _Nullable data,
                                         NSError * _Nullable error))completionHandler
{
    [self applyUpdateRequestWithParams:params
                        expectedValues:expectedDataValueDictionaries
                 expectedValueInterval:expectedValueIntervalMs
                            completion:^(MTROTASoftwareUpdateProviderClusterApplyUpdateResponseParams * _Nullable data,
                                NSError * _Nullable error) {
                                // Cast is safe because subclass does not add any selectors.
                                completionHandler(
                                    static_cast<MTROtaSoftwareUpdateProviderClusterApplyUpdateResponseParams *>(data), error);
                            }];
}
- (void)notifyUpdateAppliedWithParams:(MTROtaSoftwareUpdateProviderClusterNotifyUpdateAppliedParams *)params
                       expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
                expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                    completionHandler:(MTRStatusCompletion)completionHandler
{
    [self notifyUpdateAppliedWithParams:params
                         expectedValues:expectedDataValueDictionaries
                  expectedValueInterval:expectedValueIntervalMs
                             completion:completionHandler];
}
@end

@implementation MTRClusterOTASoftwareUpdateRequestor

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)announceOTAProviderWithParams:(MTROTASoftwareUpdateRequestorClusterAnnounceOTAProviderParams *)params
                       expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                           completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeOTASoftwareUpdateRequestorID,
                                     (unsigned int) MTRCommandIDTypeClusterOTASoftwareUpdateRequestorCommandAnnounceOTAProviderID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                OtaSoftwareUpdateRequestor::Commands::AnnounceOTAProvider::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.providerNodeID = params.providerNodeID.unsignedLongLongValue;
                request.vendorID
                    = static_cast<std::remove_reference_t<decltype(request.vendorID)>>(params.vendorID.unsignedShortValue);
                request.announcementReason = static_cast<std::remove_reference_t<decltype(request.announcementReason)>>(
                    params.announcementReason.unsignedCharValue);
                if (params.metadataForNode != nil) {
                    auto & definedValue_0 = request.metadataForNode.Emplace();
                    definedValue_0 = [self asByteSpan:params.metadataForNode];
                }
                request.endpoint = params.endpoint.unsignedShortValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeDefaultOTAProvidersWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeOTASoftwareUpdateRequestorID)
                        attributeID:@(MTRAttributeIDTypeClusterOTASoftwareUpdateRequestorAttributeDefaultOTAProvidersID)
                             params:params];
}

- (void)writeAttributeDefaultOTAProvidersWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                             expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeDefaultOTAProvidersWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeDefaultOTAProvidersWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                             expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                            params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeOTASoftwareUpdateRequestorID)
                                  attributeID:@(MTRAttributeIDTypeClusterOTASoftwareUpdateRequestorAttributeDefaultOTAProvidersID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeUpdatePossibleWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOTASoftwareUpdateRequestorID)
                                        attributeID:@(MTRAttributeIDTypeClusterOTASoftwareUpdateRequestorAttributeUpdatePossibleID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeUpdateStateWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOTASoftwareUpdateRequestorID)
                                        attributeID:@(MTRAttributeIDTypeClusterOTASoftwareUpdateRequestorAttributeUpdateStateID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeUpdateStateProgressWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeOTASoftwareUpdateRequestorID)
                        attributeID:@(MTRAttributeIDTypeClusterOTASoftwareUpdateRequestorAttributeUpdateStateProgressID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeOTASoftwareUpdateRequestorID)
                        attributeID:@(MTRAttributeIDTypeClusterOTASoftwareUpdateRequestorAttributeGeneratedCommandListID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeOTASoftwareUpdateRequestorID)
                        attributeID:@(MTRAttributeIDTypeClusterOTASoftwareUpdateRequestorAttributeAcceptedCommandListID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOTASoftwareUpdateRequestorID)
                                        attributeID:@(MTRAttributeIDTypeClusterOTASoftwareUpdateRequestorAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOTASoftwareUpdateRequestorID)
                                        attributeID:@(MTRAttributeIDTypeClusterOTASoftwareUpdateRequestorAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOTASoftwareUpdateRequestorID)
                                        attributeID:@(MTRAttributeIDTypeClusterOTASoftwareUpdateRequestorAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOTASoftwareUpdateRequestorID)
                                        attributeID:@(MTRAttributeIDTypeClusterOTASoftwareUpdateRequestorAttributeClusterRevisionID)
                                             params:params];
}

@end
@implementation MTRClusterOtaSoftwareUpdateRequestor
@end

@implementation MTRClusterOtaSoftwareUpdateRequestor (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

- (void)announceOtaProviderWithParams:(MTROtaSoftwareUpdateRequestorClusterAnnounceOtaProviderParams *)params
                       expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
                expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                    completionHandler:(MTRStatusCompletion)completionHandler
{
    [self announceOTAProviderWithParams:params
                         expectedValues:expectedDataValueDictionaries
                  expectedValueInterval:expectedValueIntervalMs
                             completion:completionHandler];
}
- (NSDictionary<NSString *, id> *)readAttributeDefaultOtaProvidersWithParams:(MTRReadParams * _Nullable)params
{
    return [self readAttributeDefaultOTAProvidersWithParams:params];
}
- (void)writeAttributeDefaultOtaProvidersWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                             expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeDefaultOTAProvidersWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs];
}
- (void)writeAttributeDefaultOtaProvidersWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                             expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                            params:(MTRWriteParams * _Nullable)params
{
    [self writeAttributeDefaultOTAProvidersWithValue:dataValueDictionary
                               expectedValueInterval:expectedValueIntervalMs
                                              params:params];
}
@end

@implementation MTRClusterLocalizationConfiguration

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (NSDictionary<NSString *, id> *)readAttributeActiveLocaleWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeLocalizationConfigurationID)
                                        attributeID:@(MTRAttributeIDTypeClusterLocalizationConfigurationAttributeActiveLocaleID)
                                             params:params];
}

- (void)writeAttributeActiveLocaleWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                      expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeActiveLocaleWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeActiveLocaleWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                      expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                     params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeLocalizationConfigurationID)
                                  attributeID:@(MTRAttributeIDTypeClusterLocalizationConfigurationAttributeActiveLocaleID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeSupportedLocalesWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeLocalizationConfigurationID)
                                        attributeID:@(MTRAttributeIDTypeClusterLocalizationConfigurationAttributeSupportedLocalesID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeLocalizationConfigurationID)
                        attributeID:@(MTRAttributeIDTypeClusterLocalizationConfigurationAttributeGeneratedCommandListID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeLocalizationConfigurationID)
                                     attributeID:@(MTRAttributeIDTypeClusterLocalizationConfigurationAttributeAcceptedCommandListID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeLocalizationConfigurationID)
                                        attributeID:@(MTRAttributeIDTypeClusterLocalizationConfigurationAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeLocalizationConfigurationID)
                                        attributeID:@(MTRAttributeIDTypeClusterLocalizationConfigurationAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeLocalizationConfigurationID)
                                        attributeID:@(MTRAttributeIDTypeClusterLocalizationConfigurationAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeLocalizationConfigurationID)
                                        attributeID:@(MTRAttributeIDTypeClusterLocalizationConfigurationAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterLocalizationConfiguration (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

@end

@implementation MTRClusterTimeFormatLocalization

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (NSDictionary<NSString *, id> *)readAttributeHourFormatWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeTimeFormatLocalizationID)
                                        attributeID:@(MTRAttributeIDTypeClusterTimeFormatLocalizationAttributeHourFormatID)
                                             params:params];
}

- (void)writeAttributeHourFormatWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeHourFormatWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeHourFormatWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                   params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeTimeFormatLocalizationID)
                                  attributeID:@(MTRAttributeIDTypeClusterTimeFormatLocalizationAttributeHourFormatID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeActiveCalendarTypeWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeTimeFormatLocalizationID)
                                        attributeID:@(MTRAttributeIDTypeClusterTimeFormatLocalizationAttributeActiveCalendarTypeID)
                                             params:params];
}

- (void)writeAttributeActiveCalendarTypeWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                            expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeActiveCalendarTypeWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeActiveCalendarTypeWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                            expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                           params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeTimeFormatLocalizationID)
                                  attributeID:@(MTRAttributeIDTypeClusterTimeFormatLocalizationAttributeActiveCalendarTypeID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeSupportedCalendarTypesWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeTimeFormatLocalizationID)
                                     attributeID:@(MTRAttributeIDTypeClusterTimeFormatLocalizationAttributeSupportedCalendarTypesID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeTimeFormatLocalizationID)
                                     attributeID:@(MTRAttributeIDTypeClusterTimeFormatLocalizationAttributeGeneratedCommandListID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeTimeFormatLocalizationID)
                                        attributeID:@(MTRAttributeIDTypeClusterTimeFormatLocalizationAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeTimeFormatLocalizationID)
                                        attributeID:@(MTRAttributeIDTypeClusterTimeFormatLocalizationAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeTimeFormatLocalizationID)
                                        attributeID:@(MTRAttributeIDTypeClusterTimeFormatLocalizationAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeTimeFormatLocalizationID)
                                        attributeID:@(MTRAttributeIDTypeClusterTimeFormatLocalizationAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeTimeFormatLocalizationID)
                                        attributeID:@(MTRAttributeIDTypeClusterTimeFormatLocalizationAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterTimeFormatLocalization (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

@end

@implementation MTRClusterUnitLocalization

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (NSDictionary<NSString *, id> *)readAttributeTemperatureUnitWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeUnitLocalizationID)
                                        attributeID:@(MTRAttributeIDTypeClusterUnitLocalizationAttributeTemperatureUnitID)
                                             params:params];
}

- (void)writeAttributeTemperatureUnitWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                         expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeTemperatureUnitWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeTemperatureUnitWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                         expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                        params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeUnitLocalizationID)
                                  attributeID:@(MTRAttributeIDTypeClusterUnitLocalizationAttributeTemperatureUnitID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeUnitLocalizationID)
                                        attributeID:@(MTRAttributeIDTypeClusterUnitLocalizationAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeUnitLocalizationID)
                                        attributeID:@(MTRAttributeIDTypeClusterUnitLocalizationAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeUnitLocalizationID)
                                        attributeID:@(MTRAttributeIDTypeClusterUnitLocalizationAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeUnitLocalizationID)
                                        attributeID:@(MTRAttributeIDTypeClusterUnitLocalizationAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeUnitLocalizationID)
                                        attributeID:@(MTRAttributeIDTypeClusterUnitLocalizationAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeUnitLocalizationID)
                                        attributeID:@(MTRAttributeIDTypeClusterUnitLocalizationAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterUnitLocalization (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

@end

@implementation MTRClusterPowerSourceConfiguration

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (NSDictionary<NSString *, id> *)readAttributeSourcesWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceConfigurationID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceConfigurationAttributeSourcesID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypePowerSourceConfigurationID)
                                     attributeID:@(MTRAttributeIDTypeClusterPowerSourceConfigurationAttributeGeneratedCommandListID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypePowerSourceConfigurationID)
                                     attributeID:@(MTRAttributeIDTypeClusterPowerSourceConfigurationAttributeAcceptedCommandListID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceConfigurationID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceConfigurationAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceConfigurationID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceConfigurationAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceConfigurationID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceConfigurationAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceConfigurationID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceConfigurationAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterPowerSourceConfiguration (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

@end

@implementation MTRClusterPowerSource

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (NSDictionary<NSString *, id> *)readAttributeStatusWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeStatusID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeOrderWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeOrderID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeDescriptionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeDescriptionID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeWiredAssessedInputVoltageWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeWiredAssessedInputVoltageID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeWiredAssessedInputFrequencyWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeWiredAssessedInputFrequencyID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeWiredCurrentTypeWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeWiredCurrentTypeID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeWiredAssessedCurrentWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeWiredAssessedCurrentID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeWiredNominalVoltageWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeWiredNominalVoltageID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeWiredMaximumCurrentWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeWiredMaximumCurrentID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeWiredPresentWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeWiredPresentID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeActiveWiredFaultsWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeActiveWiredFaultsID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeBatVoltageWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeBatVoltageID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeBatPercentRemainingWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeBatPercentRemainingID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeBatTimeRemainingWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeBatTimeRemainingID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeBatChargeLevelWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeBatChargeLevelID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeBatReplacementNeededWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeBatReplacementNeededID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeBatReplaceabilityWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeBatReplaceabilityID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeBatPresentWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeBatPresentID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeActiveBatFaultsWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeActiveBatFaultsID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeBatReplacementDescriptionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeBatReplacementDescriptionID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeBatCommonDesignationWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeBatCommonDesignationID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeBatANSIDesignationWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeBatANSIDesignationID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeBatIECDesignationWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeBatIECDesignationID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeBatApprovedChemistryWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeBatApprovedChemistryID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeBatCapacityWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeBatCapacityID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeBatQuantityWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeBatQuantityID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeBatChargeStateWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeBatChargeStateID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeBatTimeToFullChargeWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeBatTimeToFullChargeID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeBatFunctionalWhileChargingWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeBatFunctionalWhileChargingID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeBatChargingCurrentWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeBatChargingCurrentID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeActiveBatChargeFaultsWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeActiveBatChargeFaultsID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypePowerSourceID)
                                        attributeID:@(MTRAttributeIDTypeClusterPowerSourceAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterPowerSource (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

@end

@implementation MTRClusterGeneralCommissioning

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)armFailSafeWithParams:(MTRGeneralCommissioningClusterArmFailSafeParams *)params
               expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
        expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                   completion:(void (^)(MTRGeneralCommissioningClusterArmFailSafeResponseParams * _Nullable data,
                                  NSError * _Nullable error))completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeGeneralCommissioningID,
                                     (unsigned int) MTRCommandIDTypeClusterGeneralCommissioningCommandArmFailSafeID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRGeneralCommissioningClusterArmFailSafeResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                GeneralCommissioningClusterArmFailSafeResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRGeneralCommissioningClusterArmFailSafeResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                GeneralCommissioning::Commands::ArmFailSafe::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.expiryLengthSeconds = params.expiryLengthSeconds.unsignedShortValue;
                request.breadcrumb = params.breadcrumb.unsignedLongLongValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)setRegulatoryConfigWithParams:(MTRGeneralCommissioningClusterSetRegulatoryConfigParams *)params
                       expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                           completion:(void (^)(MTRGeneralCommissioningClusterSetRegulatoryConfigResponseParams * _Nullable data,
                                          NSError * _Nullable error))completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeGeneralCommissioningID,
                                     (unsigned int) MTRCommandIDTypeClusterGeneralCommissioningCommandSetRegulatoryConfigID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRGeneralCommissioningClusterSetRegulatoryConfigResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                GeneralCommissioningClusterSetRegulatoryConfigResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRGeneralCommissioningClusterSetRegulatoryConfigResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                GeneralCommissioning::Commands::SetRegulatoryConfig::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.newRegulatoryConfig = static_cast<std::remove_reference_t<decltype(request.newRegulatoryConfig)>>(
                    params.newRegulatoryConfig.unsignedCharValue);
                request.countryCode = [self asCharSpan:params.countryCode];
                request.breadcrumb = params.breadcrumb.unsignedLongLongValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)commissioningCompleteWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                          expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                     completion:
                                         (void (^)(
                                             MTRGeneralCommissioningClusterCommissioningCompleteResponseParams * _Nullable data,
                                             NSError * _Nullable error))completion
{
    [self commissioningCompleteWithParams:nil
                           expectedValues:expectedValues
                    expectedValueInterval:expectedValueIntervalMs
                               completion:completion];
}
- (void)commissioningCompleteWithParams:(MTRGeneralCommissioningClusterCommissioningCompleteParams * _Nullable)params
                         expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                  expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                             completion:
                                 (void (^)(MTRGeneralCommissioningClusterCommissioningCompleteResponseParams * _Nullable data,
                                     NSError * _Nullable error))completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeGeneralCommissioningID,
                                     (unsigned int) MTRCommandIDTypeClusterGeneralCommissioningCommandCommissioningCompleteID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRGeneralCommissioningClusterCommissioningCompleteResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                GeneralCommissioningClusterCommissioningCompleteResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge
                    = static_cast<MTRGeneralCommissioningClusterCommissioningCompleteResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                GeneralCommissioning::Commands::CommissioningComplete::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeBreadcrumbWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGeneralCommissioningID)
                                        attributeID:@(MTRAttributeIDTypeClusterGeneralCommissioningAttributeBreadcrumbID)
                                             params:params];
}

- (void)writeAttributeBreadcrumbWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeBreadcrumbWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeBreadcrumbWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                   params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeGeneralCommissioningID)
                                  attributeID:@(MTRAttributeIDTypeClusterGeneralCommissioningAttributeBreadcrumbID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeBasicCommissioningInfoWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeGeneralCommissioningID)
                                     attributeID:@(MTRAttributeIDTypeClusterGeneralCommissioningAttributeBasicCommissioningInfoID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeRegulatoryConfigWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGeneralCommissioningID)
                                        attributeID:@(MTRAttributeIDTypeClusterGeneralCommissioningAttributeRegulatoryConfigID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeLocationCapabilityWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGeneralCommissioningID)
                                        attributeID:@(MTRAttributeIDTypeClusterGeneralCommissioningAttributeLocationCapabilityID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeSupportsConcurrentConnectionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeGeneralCommissioningID)
                        attributeID:@(MTRAttributeIDTypeClusterGeneralCommissioningAttributeSupportsConcurrentConnectionID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGeneralCommissioningID)
                                        attributeID:@(MTRAttributeIDTypeClusterGeneralCommissioningAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGeneralCommissioningID)
                                        attributeID:@(MTRAttributeIDTypeClusterGeneralCommissioningAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGeneralCommissioningID)
                                        attributeID:@(MTRAttributeIDTypeClusterGeneralCommissioningAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGeneralCommissioningID)
                                        attributeID:@(MTRAttributeIDTypeClusterGeneralCommissioningAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGeneralCommissioningID)
                                        attributeID:@(MTRAttributeIDTypeClusterGeneralCommissioningAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGeneralCommissioningID)
                                        attributeID:@(MTRAttributeIDTypeClusterGeneralCommissioningAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterGeneralCommissioning (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

- (void)armFailSafeWithParams:(MTRGeneralCommissioningClusterArmFailSafeParams *)params
               expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
        expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
            completionHandler:(void (^)(MTRGeneralCommissioningClusterArmFailSafeResponseParams * _Nullable data,
                                  NSError * _Nullable error))completionHandler
{
    [self armFailSafeWithParams:params
                 expectedValues:expectedDataValueDictionaries
          expectedValueInterval:expectedValueIntervalMs
                     completion:^(
                         MTRGeneralCommissioningClusterArmFailSafeResponseParams * _Nullable data, NSError * _Nullable error) {
                         // Cast is safe because subclass does not add any selectors.
                         completionHandler(static_cast<MTRGeneralCommissioningClusterArmFailSafeResponseParams *>(data), error);
                     }];
}
- (void)setRegulatoryConfigWithParams:(MTRGeneralCommissioningClusterSetRegulatoryConfigParams *)params
                       expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
                expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                    completionHandler:(void (^)(MTRGeneralCommissioningClusterSetRegulatoryConfigResponseParams * _Nullable data,
                                          NSError * _Nullable error))completionHandler
{
    [self setRegulatoryConfigWithParams:params
                         expectedValues:expectedDataValueDictionaries
                  expectedValueInterval:expectedValueIntervalMs
                             completion:^(MTRGeneralCommissioningClusterSetRegulatoryConfigResponseParams * _Nullable data,
                                 NSError * _Nullable error) {
                                 // Cast is safe because subclass does not add any selectors.
                                 completionHandler(
                                     static_cast<MTRGeneralCommissioningClusterSetRegulatoryConfigResponseParams *>(data), error);
                             }];
}
- (void)commissioningCompleteWithParams:(MTRGeneralCommissioningClusterCommissioningCompleteParams * _Nullable)params
                         expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
                  expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                      completionHandler:
                          (void (^)(MTRGeneralCommissioningClusterCommissioningCompleteResponseParams * _Nullable data,
                              NSError * _Nullable error))completionHandler
{
    [self
        commissioningCompleteWithParams:params
                         expectedValues:expectedDataValueDictionaries
                  expectedValueInterval:expectedValueIntervalMs
                             completion:^(MTRGeneralCommissioningClusterCommissioningCompleteResponseParams * _Nullable data,
                                 NSError * _Nullable error) {
                                 // Cast is safe because subclass does not add any selectors.
                                 completionHandler(
                                     static_cast<MTRGeneralCommissioningClusterCommissioningCompleteResponseParams *>(data), error);
                             }];
}
- (void)commissioningCompleteWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                          expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                              completionHandler:
                                  (void (^)(MTRGeneralCommissioningClusterCommissioningCompleteResponseParams * _Nullable data,
                                      NSError * _Nullable error))completionHandler
{
    [self commissioningCompleteWithParams:nil
                           expectedValues:expectedValues
                    expectedValueInterval:expectedValueIntervalMs
                        completionHandler:completionHandler];
}
@end

@implementation MTRClusterNetworkCommissioning

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)scanNetworksWithParams:(MTRNetworkCommissioningClusterScanNetworksParams * _Nullable)params
                expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
         expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                    completion:(void (^)(MTRNetworkCommissioningClusterScanNetworksResponseParams * _Nullable data,
                                   NSError * _Nullable error))completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeNetworkCommissioningID,
                                     (unsigned int) MTRCommandIDTypeClusterNetworkCommissioningCommandScanNetworksID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRNetworkCommissioningClusterScanNetworksResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                NetworkCommissioningClusterScanNetworksResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRNetworkCommissioningClusterScanNetworksResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                NetworkCommissioning::Commands::ScanNetworks::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                if (params != nil) {
                    if (params.ssid != nil) {
                        auto & definedValue_0 = request.ssid.Emplace();
                        if (params.ssid == nil) {
                            definedValue_0.SetNull();
                        } else {
                            auto & nonNullValue_1 = definedValue_0.SetNonNull();
                            nonNullValue_1 = [self asByteSpan:params.ssid];
                        }
                    }
                    if (params.breadcrumb != nil) {
                        auto & definedValue_0 = request.breadcrumb.Emplace();
                        definedValue_0 = params.breadcrumb.unsignedLongLongValue;
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)addOrUpdateWiFiNetworkWithParams:(MTRNetworkCommissioningClusterAddOrUpdateWiFiNetworkParams *)params
                          expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                   expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                              completion:(void (^)(MTRNetworkCommissioningClusterNetworkConfigResponseParams * _Nullable data,
                                             NSError * _Nullable error))completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeNetworkCommissioningID,
                                     (unsigned int) MTRCommandIDTypeClusterNetworkCommissioningCommandAddOrUpdateWiFiNetworkID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRNetworkCommissioningClusterNetworkConfigResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                NetworkCommissioningClusterNetworkConfigResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRNetworkCommissioningClusterNetworkConfigResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                NetworkCommissioning::Commands::AddOrUpdateWiFiNetwork::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.ssid = [self asByteSpan:params.ssid];
                request.credentials = [self asByteSpan:params.credentials];
                if (params.breadcrumb != nil) {
                    auto & definedValue_0 = request.breadcrumb.Emplace();
                    definedValue_0 = params.breadcrumb.unsignedLongLongValue;
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)addOrUpdateThreadNetworkWithParams:(MTRNetworkCommissioningClusterAddOrUpdateThreadNetworkParams *)params
                            expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                     expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                completion:(void (^)(MTRNetworkCommissioningClusterNetworkConfigResponseParams * _Nullable data,
                                               NSError * _Nullable error))completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeNetworkCommissioningID,
                                     (unsigned int) MTRCommandIDTypeClusterNetworkCommissioningCommandAddOrUpdateThreadNetworkID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRNetworkCommissioningClusterNetworkConfigResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                NetworkCommissioningClusterNetworkConfigResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRNetworkCommissioningClusterNetworkConfigResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                NetworkCommissioning::Commands::AddOrUpdateThreadNetwork::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.operationalDataset = [self asByteSpan:params.operationalDataset];
                if (params.breadcrumb != nil) {
                    auto & definedValue_0 = request.breadcrumb.Emplace();
                    definedValue_0 = params.breadcrumb.unsignedLongLongValue;
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)removeNetworkWithParams:(MTRNetworkCommissioningClusterRemoveNetworkParams *)params
                 expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
          expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                     completion:(void (^)(MTRNetworkCommissioningClusterNetworkConfigResponseParams * _Nullable data,
                                    NSError * _Nullable error))completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeNetworkCommissioningID,
                                     (unsigned int) MTRCommandIDTypeClusterNetworkCommissioningCommandRemoveNetworkID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRNetworkCommissioningClusterNetworkConfigResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                NetworkCommissioningClusterNetworkConfigResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRNetworkCommissioningClusterNetworkConfigResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                NetworkCommissioning::Commands::RemoveNetwork::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.networkID = [self asByteSpan:params.networkID];
                if (params.breadcrumb != nil) {
                    auto & definedValue_0 = request.breadcrumb.Emplace();
                    definedValue_0 = params.breadcrumb.unsignedLongLongValue;
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)connectNetworkWithParams:(MTRNetworkCommissioningClusterConnectNetworkParams *)params
                  expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
           expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                      completion:(void (^)(MTRNetworkCommissioningClusterConnectNetworkResponseParams * _Nullable data,
                                     NSError * _Nullable error))completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeNetworkCommissioningID,
                                     (unsigned int) MTRCommandIDTypeClusterNetworkCommissioningCommandConnectNetworkID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRNetworkCommissioningClusterConnectNetworkResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                NetworkCommissioningClusterConnectNetworkResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRNetworkCommissioningClusterConnectNetworkResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                NetworkCommissioning::Commands::ConnectNetwork::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.networkID = [self asByteSpan:params.networkID];
                if (params.breadcrumb != nil) {
                    auto & definedValue_0 = request.breadcrumb.Emplace();
                    definedValue_0 = params.breadcrumb.unsignedLongLongValue;
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)reorderNetworkWithParams:(MTRNetworkCommissioningClusterReorderNetworkParams *)params
                  expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
           expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                      completion:(void (^)(MTRNetworkCommissioningClusterNetworkConfigResponseParams * _Nullable data,
                                     NSError * _Nullable error))completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeNetworkCommissioningID,
                                     (unsigned int) MTRCommandIDTypeClusterNetworkCommissioningCommandReorderNetworkID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRNetworkCommissioningClusterNetworkConfigResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                NetworkCommissioningClusterNetworkConfigResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRNetworkCommissioningClusterNetworkConfigResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                NetworkCommissioning::Commands::ReorderNetwork::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.networkID = [self asByteSpan:params.networkID];
                request.networkIndex = params.networkIndex.unsignedCharValue;
                if (params.breadcrumb != nil) {
                    auto & definedValue_0 = request.breadcrumb.Emplace();
                    definedValue_0 = params.breadcrumb.unsignedLongLongValue;
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeMaxNetworksWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeNetworkCommissioningID)
                                        attributeID:@(MTRAttributeIDTypeClusterNetworkCommissioningAttributeMaxNetworksID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeNetworksWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeNetworkCommissioningID)
                                        attributeID:@(MTRAttributeIDTypeClusterNetworkCommissioningAttributeNetworksID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeScanMaxTimeSecondsWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeNetworkCommissioningID)
                                        attributeID:@(MTRAttributeIDTypeClusterNetworkCommissioningAttributeScanMaxTimeSecondsID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeConnectMaxTimeSecondsWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeNetworkCommissioningID)
                                        attributeID:@(MTRAttributeIDTypeClusterNetworkCommissioningAttributeConnectMaxTimeSecondsID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeInterfaceEnabledWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeNetworkCommissioningID)
                                        attributeID:@(MTRAttributeIDTypeClusterNetworkCommissioningAttributeInterfaceEnabledID)
                                             params:params];
}

- (void)writeAttributeInterfaceEnabledWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                          expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeInterfaceEnabledWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeInterfaceEnabledWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                          expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                         params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeNetworkCommissioningID)
                                  attributeID:@(MTRAttributeIDTypeClusterNetworkCommissioningAttributeInterfaceEnabledID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeLastNetworkingStatusWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeNetworkCommissioningID)
                                        attributeID:@(MTRAttributeIDTypeClusterNetworkCommissioningAttributeLastNetworkingStatusID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeLastNetworkIDWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeNetworkCommissioningID)
                                        attributeID:@(MTRAttributeIDTypeClusterNetworkCommissioningAttributeLastNetworkIDID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeLastConnectErrorValueWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeNetworkCommissioningID)
                                        attributeID:@(MTRAttributeIDTypeClusterNetworkCommissioningAttributeLastConnectErrorValueID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeNetworkCommissioningID)
                                        attributeID:@(MTRAttributeIDTypeClusterNetworkCommissioningAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeNetworkCommissioningID)
                                        attributeID:@(MTRAttributeIDTypeClusterNetworkCommissioningAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeNetworkCommissioningID)
                                        attributeID:@(MTRAttributeIDTypeClusterNetworkCommissioningAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeNetworkCommissioningID)
                                        attributeID:@(MTRAttributeIDTypeClusterNetworkCommissioningAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeNetworkCommissioningID)
                                        attributeID:@(MTRAttributeIDTypeClusterNetworkCommissioningAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeNetworkCommissioningID)
                                        attributeID:@(MTRAttributeIDTypeClusterNetworkCommissioningAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterNetworkCommissioning (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

- (void)scanNetworksWithParams:(MTRNetworkCommissioningClusterScanNetworksParams * _Nullable)params
                expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
         expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
             completionHandler:(void (^)(MTRNetworkCommissioningClusterScanNetworksResponseParams * _Nullable data,
                                   NSError * _Nullable error))completionHandler
{
    [self scanNetworksWithParams:params
                  expectedValues:expectedDataValueDictionaries
           expectedValueInterval:expectedValueIntervalMs
                      completion:^(
                          MTRNetworkCommissioningClusterScanNetworksResponseParams * _Nullable data, NSError * _Nullable error) {
                          // Cast is safe because subclass does not add any selectors.
                          completionHandler(static_cast<MTRNetworkCommissioningClusterScanNetworksResponseParams *>(data), error);
                      }];
}
- (void)addOrUpdateWiFiNetworkWithParams:(MTRNetworkCommissioningClusterAddOrUpdateWiFiNetworkParams *)params
                          expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
                   expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                       completionHandler:(void (^)(MTRNetworkCommissioningClusterNetworkConfigResponseParams * _Nullable data,
                                             NSError * _Nullable error))completionHandler
{
    [self addOrUpdateWiFiNetworkWithParams:params
                            expectedValues:expectedDataValueDictionaries
                     expectedValueInterval:expectedValueIntervalMs
                                completion:^(MTRNetworkCommissioningClusterNetworkConfigResponseParams * _Nullable data,
                                    NSError * _Nullable error) {
                                    // Cast is safe because subclass does not add any selectors.
                                    completionHandler(
                                        static_cast<MTRNetworkCommissioningClusterNetworkConfigResponseParams *>(data), error);
                                }];
}
- (void)addOrUpdateThreadNetworkWithParams:(MTRNetworkCommissioningClusterAddOrUpdateThreadNetworkParams *)params
                            expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
                     expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                         completionHandler:(void (^)(MTRNetworkCommissioningClusterNetworkConfigResponseParams * _Nullable data,
                                               NSError * _Nullable error))completionHandler
{
    [self addOrUpdateThreadNetworkWithParams:params
                              expectedValues:expectedDataValueDictionaries
                       expectedValueInterval:expectedValueIntervalMs
                                  completion:^(MTRNetworkCommissioningClusterNetworkConfigResponseParams * _Nullable data,
                                      NSError * _Nullable error) {
                                      // Cast is safe because subclass does not add any selectors.
                                      completionHandler(
                                          static_cast<MTRNetworkCommissioningClusterNetworkConfigResponseParams *>(data), error);
                                  }];
}
- (void)removeNetworkWithParams:(MTRNetworkCommissioningClusterRemoveNetworkParams *)params
                 expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
          expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
              completionHandler:(void (^)(MTRNetworkCommissioningClusterNetworkConfigResponseParams * _Nullable data,
                                    NSError * _Nullable error))completionHandler
{
    [self removeNetworkWithParams:params
                   expectedValues:expectedDataValueDictionaries
            expectedValueInterval:expectedValueIntervalMs
                       completion:^(
                           MTRNetworkCommissioningClusterNetworkConfigResponseParams * _Nullable data, NSError * _Nullable error) {
                           // Cast is safe because subclass does not add any selectors.
                           completionHandler(static_cast<MTRNetworkCommissioningClusterNetworkConfigResponseParams *>(data), error);
                       }];
}
- (void)connectNetworkWithParams:(MTRNetworkCommissioningClusterConnectNetworkParams *)params
                  expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
           expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
               completionHandler:(void (^)(MTRNetworkCommissioningClusterConnectNetworkResponseParams * _Nullable data,
                                     NSError * _Nullable error))completionHandler
{
    [self
        connectNetworkWithParams:params
                  expectedValues:expectedDataValueDictionaries
           expectedValueInterval:expectedValueIntervalMs
                      completion:^(
                          MTRNetworkCommissioningClusterConnectNetworkResponseParams * _Nullable data, NSError * _Nullable error) {
                          // Cast is safe because subclass does not add any selectors.
                          completionHandler(static_cast<MTRNetworkCommissioningClusterConnectNetworkResponseParams *>(data), error);
                      }];
}
- (void)reorderNetworkWithParams:(MTRNetworkCommissioningClusterReorderNetworkParams *)params
                  expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
           expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
               completionHandler:(void (^)(MTRNetworkCommissioningClusterNetworkConfigResponseParams * _Nullable data,
                                     NSError * _Nullable error))completionHandler
{
    [self
        reorderNetworkWithParams:params
                  expectedValues:expectedDataValueDictionaries
           expectedValueInterval:expectedValueIntervalMs
                      completion:^(
                          MTRNetworkCommissioningClusterNetworkConfigResponseParams * _Nullable data, NSError * _Nullable error) {
                          // Cast is safe because subclass does not add any selectors.
                          completionHandler(static_cast<MTRNetworkCommissioningClusterNetworkConfigResponseParams *>(data), error);
                      }];
}
@end

@implementation MTRClusterDiagnosticLogs

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)retrieveLogsRequestWithParams:(MTRDiagnosticLogsClusterRetrieveLogsRequestParams *)params
                       expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                           completion:(void (^)(MTRDiagnosticLogsClusterRetrieveLogsResponseParams * _Nullable data,
                                          NSError * _Nullable error))completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeDiagnosticLogsID,
                                     (unsigned int) MTRCommandIDTypeClusterDiagnosticLogsCommandRetrieveLogsRequestID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRDiagnosticLogsClusterRetrieveLogsResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                DiagnosticLogsClusterRetrieveLogsResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRDiagnosticLogsClusterRetrieveLogsResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                DiagnosticLogs::Commands::RetrieveLogsRequest::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.intent = static_cast<std::remove_reference_t<decltype(request.intent)>>(params.intent.unsignedCharValue);
                request.requestedProtocol = static_cast<std::remove_reference_t<decltype(request.requestedProtocol)>>(
                    params.requestedProtocol.unsignedCharValue);
                if (params.transferFileDesignator != nil) {
                    auto & definedValue_0 = request.transferFileDesignator.Emplace();
                    definedValue_0 = [self asCharSpan:params.transferFileDesignator];
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDiagnosticLogsID)
                                        attributeID:@(MTRAttributeIDTypeClusterDiagnosticLogsAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDiagnosticLogsID)
                                        attributeID:@(MTRAttributeIDTypeClusterDiagnosticLogsAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDiagnosticLogsID)
                                        attributeID:@(MTRAttributeIDTypeClusterDiagnosticLogsAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDiagnosticLogsID)
                                        attributeID:@(MTRAttributeIDTypeClusterDiagnosticLogsAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDiagnosticLogsID)
                                        attributeID:@(MTRAttributeIDTypeClusterDiagnosticLogsAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDiagnosticLogsID)
                                        attributeID:@(MTRAttributeIDTypeClusterDiagnosticLogsAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterDiagnosticLogs (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

- (void)retrieveLogsRequestWithParams:(MTRDiagnosticLogsClusterRetrieveLogsRequestParams *)params
                       expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
                expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                    completionHandler:(void (^)(MTRDiagnosticLogsClusterRetrieveLogsResponseParams * _Nullable data,
                                          NSError * _Nullable error))completionHandler
{
    [self retrieveLogsRequestWithParams:params
                         expectedValues:expectedDataValueDictionaries
                  expectedValueInterval:expectedValueIntervalMs
                             completion:^(
                                 MTRDiagnosticLogsClusterRetrieveLogsResponseParams * _Nullable data, NSError * _Nullable error) {
                                 // Cast is safe because subclass does not add any selectors.
                                 completionHandler(static_cast<MTRDiagnosticLogsClusterRetrieveLogsResponseParams *>(data), error);
                             }];
}
@end

@implementation MTRClusterGeneralDiagnostics

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)testEventTriggerWithParams:(MTRGeneralDiagnosticsClusterTestEventTriggerParams *)params
                    expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
             expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                        completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeGeneralDiagnosticsID,
                                     (unsigned int) MTRCommandIDTypeClusterGeneralDiagnosticsCommandTestEventTriggerID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                GeneralDiagnostics::Commands::TestEventTrigger::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.enableKey = [self asByteSpan:params.enableKey];
                request.eventTrigger = params.eventTrigger.unsignedLongLongValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeNetworkInterfacesWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGeneralDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterGeneralDiagnosticsAttributeNetworkInterfacesID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeRebootCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGeneralDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterGeneralDiagnosticsAttributeRebootCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeUpTimeWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGeneralDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterGeneralDiagnosticsAttributeUpTimeID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeTotalOperationalHoursWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGeneralDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterGeneralDiagnosticsAttributeTotalOperationalHoursID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeBootReasonWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGeneralDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterGeneralDiagnosticsAttributeBootReasonID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeActiveHardwareFaultsWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGeneralDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterGeneralDiagnosticsAttributeActiveHardwareFaultsID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeActiveRadioFaultsWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGeneralDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterGeneralDiagnosticsAttributeActiveRadioFaultsID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeActiveNetworkFaultsWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGeneralDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterGeneralDiagnosticsAttributeActiveNetworkFaultsID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeTestEventTriggersEnabledWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeGeneralDiagnosticsID)
                                     attributeID:@(MTRAttributeIDTypeClusterGeneralDiagnosticsAttributeTestEventTriggersEnabledID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGeneralDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterGeneralDiagnosticsAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGeneralDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterGeneralDiagnosticsAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGeneralDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterGeneralDiagnosticsAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGeneralDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterGeneralDiagnosticsAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGeneralDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterGeneralDiagnosticsAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGeneralDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterGeneralDiagnosticsAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterGeneralDiagnostics (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

- (void)testEventTriggerWithParams:(MTRGeneralDiagnosticsClusterTestEventTriggerParams *)params
                    expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
             expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                 completionHandler:(MTRStatusCompletion)completionHandler
{
    [self testEventTriggerWithParams:params
                      expectedValues:expectedDataValueDictionaries
               expectedValueInterval:expectedValueIntervalMs
                          completion:completionHandler];
}
- (NSDictionary<NSString *, id> *)readAttributeBootReasonsWithParams:(MTRReadParams * _Nullable)params
{
    return [self readAttributeBootReasonWithParams:params];
}
@end

@implementation MTRClusterSoftwareDiagnostics

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)resetWatermarksWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                               completion:(MTRStatusCompletion)completion
{
    [self resetWatermarksWithParams:nil
                     expectedValues:expectedValues
              expectedValueInterval:expectedValueIntervalMs
                         completion:completion];
}
- (void)resetWatermarksWithParams:(MTRSoftwareDiagnosticsClusterResetWatermarksParams * _Nullable)params
                   expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
            expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                       completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeSoftwareDiagnosticsID,
                                     (unsigned int) MTRCommandIDTypeClusterSoftwareDiagnosticsCommandResetWatermarksID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                SoftwareDiagnostics::Commands::ResetWatermarks::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeThreadMetricsWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSoftwareDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterSoftwareDiagnosticsAttributeThreadMetricsID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeCurrentHeapFreeWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSoftwareDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterSoftwareDiagnosticsAttributeCurrentHeapFreeID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeCurrentHeapUsedWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSoftwareDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterSoftwareDiagnosticsAttributeCurrentHeapUsedID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeCurrentHeapHighWatermarkWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeSoftwareDiagnosticsID)
                                     attributeID:@(MTRAttributeIDTypeClusterSoftwareDiagnosticsAttributeCurrentHeapHighWatermarkID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSoftwareDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterSoftwareDiagnosticsAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSoftwareDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterSoftwareDiagnosticsAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSoftwareDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterSoftwareDiagnosticsAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSoftwareDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterSoftwareDiagnosticsAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSoftwareDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterSoftwareDiagnosticsAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSoftwareDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterSoftwareDiagnosticsAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterSoftwareDiagnostics (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

- (void)resetWatermarksWithParams:(MTRSoftwareDiagnosticsClusterResetWatermarksParams * _Nullable)params
                   expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
            expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                completionHandler:(MTRStatusCompletion)completionHandler
{
    [self resetWatermarksWithParams:params
                     expectedValues:expectedDataValueDictionaries
              expectedValueInterval:expectedValueIntervalMs
                         completion:completionHandler];
}
- (void)resetWatermarksWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                        completionHandler:(MTRStatusCompletion)completionHandler
{
    [self resetWatermarksWithParams:nil
                     expectedValues:expectedValues
              expectedValueInterval:expectedValueIntervalMs
                  completionHandler:completionHandler];
}
@end

@implementation MTRClusterThreadNetworkDiagnostics

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)resetCountsWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                           completion:(MTRStatusCompletion)completion
{
    [self resetCountsWithParams:nil
                 expectedValues:expectedValues
          expectedValueInterval:expectedValueIntervalMs
                     completion:completion];
}
- (void)resetCountsWithParams:(MTRThreadNetworkDiagnosticsClusterResetCountsParams * _Nullable)params
               expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
        expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                   completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeThreadNetworkDiagnosticsID,
                                     (unsigned int) MTRCommandIDTypeClusterThreadNetworkDiagnosticsCommandResetCountsID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                ThreadNetworkDiagnostics::Commands::ResetCounts::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeChannelWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeChannelID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeRoutingRoleWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeRoutingRoleID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeNetworkNameWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeNetworkNameID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributePanIdWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributePanIdID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeExtendedPanIdWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeExtendedPanIdID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeMeshLocalPrefixWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeMeshLocalPrefixID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeOverrunCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeOverrunCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeNeighborTableWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeNeighborTableID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeRouteTableWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeRouteTableID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributePartitionIdWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributePartitionIdID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeWeightingWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeWeightingID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeDataVersionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeDataVersionID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeStableDataVersionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeStableDataVersionID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeLeaderRouterIdWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeLeaderRouterIdID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeDetachedRoleCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeDetachedRoleCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeChildRoleCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeChildRoleCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeRouterRoleCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeRouterRoleCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeLeaderRoleCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeLeaderRoleCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttachAttemptCountWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                     attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeAttachAttemptCountID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributePartitionIdChangeCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributePartitionIdChangeCountID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeBetterPartitionAttachAttemptCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeBetterPartitionAttachAttemptCountID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeParentChangeCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeParentChangeCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeTxTotalCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeTxTotalCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeTxUnicastCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeTxUnicastCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeTxBroadcastCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeTxBroadcastCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeTxAckRequestedCountWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                     attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeTxAckRequestedCountID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeTxAckedCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeTxAckedCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeTxNoAckRequestedCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeTxNoAckRequestedCountID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeTxDataCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeTxDataCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeTxDataPollCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeTxDataPollCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeTxBeaconCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeTxBeaconCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeTxBeaconRequestCountWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                     attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeTxBeaconRequestCountID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeTxOtherCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeTxOtherCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeTxRetryCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeTxRetryCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeTxDirectMaxRetryExpiryCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeTxDirectMaxRetryExpiryCountID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeTxIndirectMaxRetryExpiryCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeTxIndirectMaxRetryExpiryCountID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeTxErrCcaCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeTxErrCcaCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeTxErrAbortCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeTxErrAbortCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeTxErrBusyChannelCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeTxErrBusyChannelCountID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeRxTotalCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeRxTotalCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeRxUnicastCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeRxUnicastCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeRxBroadcastCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeRxBroadcastCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeRxDataCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeRxDataCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeRxDataPollCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeRxDataPollCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeRxBeaconCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeRxBeaconCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeRxBeaconRequestCountWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                     attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeRxBeaconRequestCountID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeRxOtherCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeRxOtherCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeRxAddressFilteredCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeRxAddressFilteredCountID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeRxDestAddrFilteredCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeRxDestAddrFilteredCountID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeRxDuplicatedCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeRxDuplicatedCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeRxErrNoFrameCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeRxErrNoFrameCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeRxErrUnknownNeighborCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeRxErrUnknownNeighborCountID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeRxErrInvalidSrcAddrCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeRxErrInvalidSrcAddrCountID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeRxErrSecCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeRxErrSecCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeRxErrFcsCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeRxErrFcsCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeRxErrOtherCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeRxErrOtherCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeActiveTimestampWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeActiveTimestampID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributePendingTimestampWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributePendingTimestampID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeDelayWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeDelayID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeSecurityPolicyWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeSecurityPolicyID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeChannelPage0MaskWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeChannelPage0MaskID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeOperationalDatasetComponentsWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeOperationalDatasetComponentsID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeActiveNetworkFaultsListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeActiveNetworkFaultsListID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                     attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeGeneratedCommandListID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                     attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeAcceptedCommandListID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeThreadNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterThreadNetworkDiagnosticsAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterThreadNetworkDiagnostics (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

- (void)resetCountsWithParams:(MTRThreadNetworkDiagnosticsClusterResetCountsParams * _Nullable)params
               expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
        expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
            completionHandler:(MTRStatusCompletion)completionHandler
{
    [self resetCountsWithParams:params
                 expectedValues:expectedDataValueDictionaries
          expectedValueInterval:expectedValueIntervalMs
                     completion:completionHandler];
}
- (void)resetCountsWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                    completionHandler:(MTRStatusCompletion)completionHandler
{
    [self resetCountsWithParams:nil
                 expectedValues:expectedValues
          expectedValueInterval:expectedValueIntervalMs
              completionHandler:completionHandler];
}
- (NSDictionary<NSString *, id> *)readAttributeNeighborTableListWithParams:(MTRReadParams * _Nullable)params
{
    return [self readAttributeNeighborTableWithParams:params];
}
- (NSDictionary<NSString *, id> *)readAttributeRouteTableListWithParams:(MTRReadParams * _Nullable)params
{
    return [self readAttributeRouteTableWithParams:params];
}
@end

@implementation MTRClusterWiFiNetworkDiagnostics

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)resetCountsWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                           completion:(MTRStatusCompletion)completion
{
    [self resetCountsWithParams:nil
                 expectedValues:expectedValues
          expectedValueInterval:expectedValueIntervalMs
                     completion:completion];
}
- (void)resetCountsWithParams:(MTRWiFiNetworkDiagnosticsClusterResetCountsParams * _Nullable)params
               expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
        expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                   completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeWiFiNetworkDiagnosticsID,
                                     (unsigned int) MTRCommandIDTypeClusterWiFiNetworkDiagnosticsCommandResetCountsID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                WiFiNetworkDiagnostics::Commands::ResetCounts::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeBSSIDWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeWiFiNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterWiFiNetworkDiagnosticsAttributeBSSIDID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeSecurityTypeWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeWiFiNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterWiFiNetworkDiagnosticsAttributeSecurityTypeID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeWiFiVersionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeWiFiNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterWiFiNetworkDiagnosticsAttributeWiFiVersionID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeChannelNumberWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeWiFiNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterWiFiNetworkDiagnosticsAttributeChannelNumberID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeRSSIWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeWiFiNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterWiFiNetworkDiagnosticsAttributeRSSIID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeBeaconLostCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeWiFiNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterWiFiNetworkDiagnosticsAttributeBeaconLostCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeBeaconRxCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeWiFiNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterWiFiNetworkDiagnosticsAttributeBeaconRxCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributePacketMulticastRxCountWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeWiFiNetworkDiagnosticsID)
                                     attributeID:@(MTRAttributeIDTypeClusterWiFiNetworkDiagnosticsAttributePacketMulticastRxCountID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributePacketMulticastTxCountWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeWiFiNetworkDiagnosticsID)
                                     attributeID:@(MTRAttributeIDTypeClusterWiFiNetworkDiagnosticsAttributePacketMulticastTxCountID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributePacketUnicastRxCountWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeWiFiNetworkDiagnosticsID)
                                     attributeID:@(MTRAttributeIDTypeClusterWiFiNetworkDiagnosticsAttributePacketUnicastRxCountID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributePacketUnicastTxCountWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeWiFiNetworkDiagnosticsID)
                                     attributeID:@(MTRAttributeIDTypeClusterWiFiNetworkDiagnosticsAttributePacketUnicastTxCountID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeCurrentMaxRateWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeWiFiNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterWiFiNetworkDiagnosticsAttributeCurrentMaxRateID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeOverrunCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeWiFiNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterWiFiNetworkDiagnosticsAttributeOverrunCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeWiFiNetworkDiagnosticsID)
                                     attributeID:@(MTRAttributeIDTypeClusterWiFiNetworkDiagnosticsAttributeGeneratedCommandListID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeWiFiNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterWiFiNetworkDiagnosticsAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeWiFiNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterWiFiNetworkDiagnosticsAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeWiFiNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterWiFiNetworkDiagnosticsAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeWiFiNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterWiFiNetworkDiagnosticsAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeWiFiNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterWiFiNetworkDiagnosticsAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterWiFiNetworkDiagnostics (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

- (void)resetCountsWithParams:(MTRWiFiNetworkDiagnosticsClusterResetCountsParams * _Nullable)params
               expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
        expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
            completionHandler:(MTRStatusCompletion)completionHandler
{
    [self resetCountsWithParams:params
                 expectedValues:expectedDataValueDictionaries
          expectedValueInterval:expectedValueIntervalMs
                     completion:completionHandler];
}
- (void)resetCountsWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                    completionHandler:(MTRStatusCompletion)completionHandler
{
    [self resetCountsWithParams:nil
                 expectedValues:expectedValues
          expectedValueInterval:expectedValueIntervalMs
              completionHandler:completionHandler];
}
- (NSDictionary<NSString *, id> *)readAttributeBssidWithParams:(MTRReadParams * _Nullable)params
{
    return [self readAttributeBSSIDWithParams:params];
}
- (NSDictionary<NSString *, id> *)readAttributeRssiWithParams:(MTRReadParams * _Nullable)params
{
    return [self readAttributeRSSIWithParams:params];
}
@end

@implementation MTRClusterEthernetNetworkDiagnostics

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)resetCountsWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                           completion:(MTRStatusCompletion)completion
{
    [self resetCountsWithParams:nil
                 expectedValues:expectedValues
          expectedValueInterval:expectedValueIntervalMs
                     completion:completion];
}
- (void)resetCountsWithParams:(MTREthernetNetworkDiagnosticsClusterResetCountsParams * _Nullable)params
               expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
        expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                   completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeEthernetNetworkDiagnosticsID,
                                     (unsigned int) MTRCommandIDTypeClusterEthernetNetworkDiagnosticsCommandResetCountsID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                EthernetNetworkDiagnostics::Commands::ResetCounts::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributePHYRateWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeEthernetNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterEthernetNetworkDiagnosticsAttributePHYRateID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFullDuplexWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeEthernetNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterEthernetNetworkDiagnosticsAttributeFullDuplexID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributePacketRxCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeEthernetNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterEthernetNetworkDiagnosticsAttributePacketRxCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributePacketTxCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeEthernetNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterEthernetNetworkDiagnosticsAttributePacketTxCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeTxErrCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeEthernetNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterEthernetNetworkDiagnosticsAttributeTxErrCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeCollisionCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeEthernetNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterEthernetNetworkDiagnosticsAttributeCollisionCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeOverrunCountWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeEthernetNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterEthernetNetworkDiagnosticsAttributeOverrunCountID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeCarrierDetectWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeEthernetNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterEthernetNetworkDiagnosticsAttributeCarrierDetectID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeTimeSinceResetWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeEthernetNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterEthernetNetworkDiagnosticsAttributeTimeSinceResetID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeEthernetNetworkDiagnosticsID)
                        attributeID:@(MTRAttributeIDTypeClusterEthernetNetworkDiagnosticsAttributeGeneratedCommandListID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeEthernetNetworkDiagnosticsID)
                        attributeID:@(MTRAttributeIDTypeClusterEthernetNetworkDiagnosticsAttributeAcceptedCommandListID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeEthernetNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterEthernetNetworkDiagnosticsAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeEthernetNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterEthernetNetworkDiagnosticsAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeEthernetNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterEthernetNetworkDiagnosticsAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeEthernetNetworkDiagnosticsID)
                                        attributeID:@(MTRAttributeIDTypeClusterEthernetNetworkDiagnosticsAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterEthernetNetworkDiagnostics (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

- (void)resetCountsWithParams:(MTREthernetNetworkDiagnosticsClusterResetCountsParams * _Nullable)params
               expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
        expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
            completionHandler:(MTRStatusCompletion)completionHandler
{
    [self resetCountsWithParams:params
                 expectedValues:expectedDataValueDictionaries
          expectedValueInterval:expectedValueIntervalMs
                     completion:completionHandler];
}
- (void)resetCountsWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                    completionHandler:(MTRStatusCompletion)completionHandler
{
    [self resetCountsWithParams:nil
                 expectedValues:expectedValues
          expectedValueInterval:expectedValueIntervalMs
              completionHandler:completionHandler];
}
@end

@implementation MTRClusterBridgedDeviceBasicInformation

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (NSDictionary<NSString *, id> *)readAttributeVendorNameWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBridgedDeviceBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBridgedDeviceBasicInformationAttributeVendorNameID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeVendorIDWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBridgedDeviceBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBridgedDeviceBasicInformationAttributeVendorIDID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeProductNameWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBridgedDeviceBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBridgedDeviceBasicInformationAttributeProductNameID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeNodeLabelWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBridgedDeviceBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBridgedDeviceBasicInformationAttributeNodeLabelID)
                                             params:params];
}

- (void)writeAttributeNodeLabelWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                   expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeNodeLabelWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeNodeLabelWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                   expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                  params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeBridgedDeviceBasicInformationID)
                                  attributeID:@(MTRAttributeIDTypeClusterBridgedDeviceBasicInformationAttributeNodeLabelID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeHardwareVersionWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeBridgedDeviceBasicInformationID)
                                     attributeID:@(MTRAttributeIDTypeClusterBridgedDeviceBasicInformationAttributeHardwareVersionID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeHardwareVersionStringWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeBridgedDeviceBasicInformationID)
                        attributeID:@(MTRAttributeIDTypeClusterBridgedDeviceBasicInformationAttributeHardwareVersionStringID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeSoftwareVersionWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeBridgedDeviceBasicInformationID)
                                     attributeID:@(MTRAttributeIDTypeClusterBridgedDeviceBasicInformationAttributeSoftwareVersionID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeSoftwareVersionStringWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeBridgedDeviceBasicInformationID)
                        attributeID:@(MTRAttributeIDTypeClusterBridgedDeviceBasicInformationAttributeSoftwareVersionStringID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeManufacturingDateWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeBridgedDeviceBasicInformationID)
                        attributeID:@(MTRAttributeIDTypeClusterBridgedDeviceBasicInformationAttributeManufacturingDateID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributePartNumberWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBridgedDeviceBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBridgedDeviceBasicInformationAttributePartNumberID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeProductURLWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBridgedDeviceBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBridgedDeviceBasicInformationAttributeProductURLID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeProductLabelWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBridgedDeviceBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBridgedDeviceBasicInformationAttributeProductLabelID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeSerialNumberWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBridgedDeviceBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBridgedDeviceBasicInformationAttributeSerialNumberID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeReachableWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBridgedDeviceBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBridgedDeviceBasicInformationAttributeReachableID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeUniqueIDWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBridgedDeviceBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBridgedDeviceBasicInformationAttributeUniqueIDID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeProductAppearanceWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeBridgedDeviceBasicInformationID)
                        attributeID:@(MTRAttributeIDTypeClusterBridgedDeviceBasicInformationAttributeProductAppearanceID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeBridgedDeviceBasicInformationID)
                        attributeID:@(MTRAttributeIDTypeClusterBridgedDeviceBasicInformationAttributeGeneratedCommandListID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeBridgedDeviceBasicInformationID)
                        attributeID:@(MTRAttributeIDTypeClusterBridgedDeviceBasicInformationAttributeAcceptedCommandListID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBridgedDeviceBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBridgedDeviceBasicInformationAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeBridgedDeviceBasicInformationID)
                                     attributeID:@(MTRAttributeIDTypeClusterBridgedDeviceBasicInformationAttributeAttributeListID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBridgedDeviceBasicInformationID)
                                        attributeID:@(MTRAttributeIDTypeClusterBridgedDeviceBasicInformationAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeBridgedDeviceBasicInformationID)
                                     attributeID:@(MTRAttributeIDTypeClusterBridgedDeviceBasicInformationAttributeClusterRevisionID)
                                          params:params];
}

@end
@implementation MTRClusterBridgedDeviceBasic
@end

@implementation MTRClusterBridgedDeviceBasic (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

@end

@implementation MTRClusterSwitch

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (NSDictionary<NSString *, id> *)readAttributeNumberOfPositionsWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSwitchID)
                                        attributeID:@(MTRAttributeIDTypeClusterSwitchAttributeNumberOfPositionsID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeCurrentPositionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSwitchID)
                                        attributeID:@(MTRAttributeIDTypeClusterSwitchAttributeCurrentPositionID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeMultiPressMaxWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSwitchID)
                                        attributeID:@(MTRAttributeIDTypeClusterSwitchAttributeMultiPressMaxID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSwitchID)
                                        attributeID:@(MTRAttributeIDTypeClusterSwitchAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSwitchID)
                                        attributeID:@(MTRAttributeIDTypeClusterSwitchAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSwitchID)
                                        attributeID:@(MTRAttributeIDTypeClusterSwitchAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSwitchID)
                                        attributeID:@(MTRAttributeIDTypeClusterSwitchAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSwitchID)
                                        attributeID:@(MTRAttributeIDTypeClusterSwitchAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSwitchID)
                                        attributeID:@(MTRAttributeIDTypeClusterSwitchAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterSwitch (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

@end

@implementation MTRClusterAdministratorCommissioning

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)openCommissioningWindowWithParams:(MTRAdministratorCommissioningClusterOpenCommissioningWindowParams *)params
                           expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                               completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeAdministratorCommissioningID,
                  (unsigned int) MTRCommandIDTypeClusterAdministratorCommissioningCommandOpenCommissioningWindowID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                AdministratorCommissioning::Commands::OpenCommissioningWindow::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                if (!timedInvokeTimeoutMs.HasValue()) {
                    timedInvokeTimeoutMs.SetValue(10000);
                }
                request.commissioningTimeout = params.commissioningTimeout.unsignedShortValue;
                request.PAKEPasscodeVerifier = [self asByteSpan:params.pakePasscodeVerifier];
                request.discriminator = params.discriminator.unsignedShortValue;
                request.iterations = params.iterations.unsignedIntValue;
                request.salt = [self asByteSpan:params.salt];

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)openBasicCommissioningWindowWithParams:(MTRAdministratorCommissioningClusterOpenBasicCommissioningWindowParams *)params
                                expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                         expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                    completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeAdministratorCommissioningID,
                  (unsigned int) MTRCommandIDTypeClusterAdministratorCommissioningCommandOpenBasicCommissioningWindowID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                AdministratorCommissioning::Commands::OpenBasicCommissioningWindow::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                if (!timedInvokeTimeoutMs.HasValue()) {
                    timedInvokeTimeoutMs.SetValue(10000);
                }
                request.commissioningTimeout = params.commissioningTimeout.unsignedShortValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)revokeCommissioningWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                        expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                   completion:(MTRStatusCompletion)completion
{
    [self revokeCommissioningWithParams:nil
                         expectedValues:expectedValues
                  expectedValueInterval:expectedValueIntervalMs
                             completion:completion];
}
- (void)revokeCommissioningWithParams:(MTRAdministratorCommissioningClusterRevokeCommissioningParams * _Nullable)params
                       expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                           completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeAdministratorCommissioningID,
                                     (unsigned int) MTRCommandIDTypeClusterAdministratorCommissioningCommandRevokeCommissioningID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                AdministratorCommissioning::Commands::RevokeCommissioning::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                if (!timedInvokeTimeoutMs.HasValue()) {
                    timedInvokeTimeoutMs.SetValue(10000);
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeWindowStatusWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeAdministratorCommissioningID)
                                        attributeID:@(MTRAttributeIDTypeClusterAdministratorCommissioningAttributeWindowStatusID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAdminFabricIndexWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeAdministratorCommissioningID)
                                     attributeID:@(MTRAttributeIDTypeClusterAdministratorCommissioningAttributeAdminFabricIndexID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAdminVendorIdWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeAdministratorCommissioningID)
                                        attributeID:@(MTRAttributeIDTypeClusterAdministratorCommissioningAttributeAdminVendorIdID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeAdministratorCommissioningID)
                        attributeID:@(MTRAttributeIDTypeClusterAdministratorCommissioningAttributeGeneratedCommandListID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeAdministratorCommissioningID)
                        attributeID:@(MTRAttributeIDTypeClusterAdministratorCommissioningAttributeAcceptedCommandListID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeAdministratorCommissioningID)
                                        attributeID:@(MTRAttributeIDTypeClusterAdministratorCommissioningAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeAdministratorCommissioningID)
                                        attributeID:@(MTRAttributeIDTypeClusterAdministratorCommissioningAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeAdministratorCommissioningID)
                                        attributeID:@(MTRAttributeIDTypeClusterAdministratorCommissioningAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeAdministratorCommissioningID)
                                        attributeID:@(MTRAttributeIDTypeClusterAdministratorCommissioningAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterAdministratorCommissioning (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

- (void)openCommissioningWindowWithParams:(MTRAdministratorCommissioningClusterOpenCommissioningWindowParams *)params
                           expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
                    expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                        completionHandler:(MTRStatusCompletion)completionHandler
{
    [self openCommissioningWindowWithParams:params
                             expectedValues:expectedDataValueDictionaries
                      expectedValueInterval:expectedValueIntervalMs
                                 completion:completionHandler];
}
- (void)openBasicCommissioningWindowWithParams:(MTRAdministratorCommissioningClusterOpenBasicCommissioningWindowParams *)params
                                expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
                         expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                             completionHandler:(MTRStatusCompletion)completionHandler
{
    [self openBasicCommissioningWindowWithParams:params
                                  expectedValues:expectedDataValueDictionaries
                           expectedValueInterval:expectedValueIntervalMs
                                      completion:completionHandler];
}
- (void)revokeCommissioningWithParams:(MTRAdministratorCommissioningClusterRevokeCommissioningParams * _Nullable)params
                       expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
                expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                    completionHandler:(MTRStatusCompletion)completionHandler
{
    [self revokeCommissioningWithParams:params
                         expectedValues:expectedDataValueDictionaries
                  expectedValueInterval:expectedValueIntervalMs
                             completion:completionHandler];
}
- (void)revokeCommissioningWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                        expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                            completionHandler:(MTRStatusCompletion)completionHandler
{
    [self revokeCommissioningWithParams:nil
                         expectedValues:expectedValues
                  expectedValueInterval:expectedValueIntervalMs
                      completionHandler:completionHandler];
}
@end

@implementation MTRClusterOperationalCredentials

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)attestationRequestWithParams:(MTROperationalCredentialsClusterAttestationRequestParams *)params
                      expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
               expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                          completion:(void (^)(MTROperationalCredentialsClusterAttestationResponseParams * _Nullable data,
                                         NSError * _Nullable error))completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeOperationalCredentialsID,
                                     (unsigned int) MTRCommandIDTypeClusterOperationalCredentialsCommandAttestationRequestID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTROperationalCredentialsClusterAttestationResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                OperationalCredentialsClusterAttestationResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTROperationalCredentialsClusterAttestationResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                OperationalCredentials::Commands::AttestationRequest::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.attestationNonce = [self asByteSpan:params.attestationNonce];

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)certificateChainRequestWithParams:(MTROperationalCredentialsClusterCertificateChainRequestParams *)params
                           expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                               completion:(void (^)(MTROperationalCredentialsClusterCertificateChainResponseParams * _Nullable data,
                                              NSError * _Nullable error))completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeOperationalCredentialsID,
                                     (unsigned int) MTRCommandIDTypeClusterOperationalCredentialsCommandCertificateChainRequestID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTROperationalCredentialsClusterCertificateChainResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                OperationalCredentialsClusterCertificateChainResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTROperationalCredentialsClusterCertificateChainResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                OperationalCredentials::Commands::CertificateChainRequest::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.certificateType = static_cast<std::remove_reference_t<decltype(request.certificateType)>>(
                    params.certificateType.unsignedCharValue);

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)CSRRequestWithParams:(MTROperationalCredentialsClusterCSRRequestParams *)params
              expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
       expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                  completion:(void (^)(MTROperationalCredentialsClusterCSRResponseParams * _Nullable data,
                                 NSError * _Nullable error))completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeOperationalCredentialsID,
                                     (unsigned int) MTRCommandIDTypeClusterOperationalCredentialsCommandCSRRequestID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTROperationalCredentialsClusterCSRResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                OperationalCredentialsClusterCSRResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTROperationalCredentialsClusterCSRResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                OperationalCredentials::Commands::CSRRequest::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.CSRNonce = [self asByteSpan:params.csrNonce];
                if (params.isForUpdateNOC != nil) {
                    auto & definedValue_0 = request.isForUpdateNOC.Emplace();
                    definedValue_0 = params.isForUpdateNOC.boolValue;
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)addNOCWithParams:(MTROperationalCredentialsClusterAddNOCParams *)params
           expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
               completion:(void (^)(MTROperationalCredentialsClusterNOCResponseParams * _Nullable data,
                              NSError * _Nullable error))completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeOperationalCredentialsID,
                                     (unsigned int) MTRCommandIDTypeClusterOperationalCredentialsCommandAddNOCID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTROperationalCredentialsClusterNOCResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                OperationalCredentialsClusterNOCResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTROperationalCredentialsClusterNOCResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                OperationalCredentials::Commands::AddNOC::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.NOCValue = [self asByteSpan:params.nocValue];
                if (params.icacValue != nil) {
                    auto & definedValue_0 = request.ICACValue.Emplace();
                    definedValue_0 = [self asByteSpan:params.icacValue];
                }
                request.IPKValue = [self asByteSpan:params.ipkValue];
                request.caseAdminSubject = params.caseAdminSubject.unsignedLongLongValue;
                request.adminVendorId = static_cast<std::remove_reference_t<decltype(request.adminVendorId)>>(
                    params.adminVendorId.unsignedShortValue);

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)updateNOCWithParams:(MTROperationalCredentialsClusterUpdateNOCParams *)params
             expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
      expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                 completion:(void (^)(MTROperationalCredentialsClusterNOCResponseParams * _Nullable data,
                                NSError * _Nullable error))completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeOperationalCredentialsID,
                                     (unsigned int) MTRCommandIDTypeClusterOperationalCredentialsCommandUpdateNOCID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTROperationalCredentialsClusterNOCResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                OperationalCredentialsClusterNOCResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTROperationalCredentialsClusterNOCResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                OperationalCredentials::Commands::UpdateNOC::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.NOCValue = [self asByteSpan:params.nocValue];
                if (params.icacValue != nil) {
                    auto & definedValue_0 = request.ICACValue.Emplace();
                    definedValue_0 = [self asByteSpan:params.icacValue];
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)updateFabricLabelWithParams:(MTROperationalCredentialsClusterUpdateFabricLabelParams *)params
                     expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
              expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                         completion:(void (^)(MTROperationalCredentialsClusterNOCResponseParams * _Nullable data,
                                        NSError * _Nullable error))completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeOperationalCredentialsID,
                                     (unsigned int) MTRCommandIDTypeClusterOperationalCredentialsCommandUpdateFabricLabelID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTROperationalCredentialsClusterNOCResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                OperationalCredentialsClusterNOCResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTROperationalCredentialsClusterNOCResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                OperationalCredentials::Commands::UpdateFabricLabel::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.label = [self asCharSpan:params.label];

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)removeFabricWithParams:(MTROperationalCredentialsClusterRemoveFabricParams *)params
                expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
         expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                    completion:(void (^)(MTROperationalCredentialsClusterNOCResponseParams * _Nullable data,
                                   NSError * _Nullable error))completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeOperationalCredentialsID,
                                     (unsigned int) MTRCommandIDTypeClusterOperationalCredentialsCommandRemoveFabricID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTROperationalCredentialsClusterNOCResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                OperationalCredentialsClusterNOCResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTROperationalCredentialsClusterNOCResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                OperationalCredentials::Commands::RemoveFabric::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.fabricIndex = params.fabricIndex.unsignedCharValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)addTrustedRootCertificateWithParams:(MTROperationalCredentialsClusterAddTrustedRootCertificateParams *)params
                             expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                      expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                 completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeOperationalCredentialsID,
                  (unsigned int) MTRCommandIDTypeClusterOperationalCredentialsCommandAddTrustedRootCertificateID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                OperationalCredentials::Commands::AddTrustedRootCertificate::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.rootCACertificate = [self asByteSpan:params.rootCACertificate];

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeNOCsWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOperationalCredentialsID)
                                        attributeID:@(MTRAttributeIDTypeClusterOperationalCredentialsAttributeNOCsID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFabricsWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOperationalCredentialsID)
                                        attributeID:@(MTRAttributeIDTypeClusterOperationalCredentialsAttributeFabricsID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeSupportedFabricsWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOperationalCredentialsID)
                                        attributeID:@(MTRAttributeIDTypeClusterOperationalCredentialsAttributeSupportedFabricsID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeCommissionedFabricsWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOperationalCredentialsID)
                                        attributeID:@(MTRAttributeIDTypeClusterOperationalCredentialsAttributeCommissionedFabricsID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeTrustedRootCertificatesWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeOperationalCredentialsID)
                        attributeID:@(MTRAttributeIDTypeClusterOperationalCredentialsAttributeTrustedRootCertificatesID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeCurrentFabricIndexWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOperationalCredentialsID)
                                        attributeID:@(MTRAttributeIDTypeClusterOperationalCredentialsAttributeCurrentFabricIndexID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeOperationalCredentialsID)
                                     attributeID:@(MTRAttributeIDTypeClusterOperationalCredentialsAttributeGeneratedCommandListID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOperationalCredentialsID)
                                        attributeID:@(MTRAttributeIDTypeClusterOperationalCredentialsAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOperationalCredentialsID)
                                        attributeID:@(MTRAttributeIDTypeClusterOperationalCredentialsAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOperationalCredentialsID)
                                        attributeID:@(MTRAttributeIDTypeClusterOperationalCredentialsAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOperationalCredentialsID)
                                        attributeID:@(MTRAttributeIDTypeClusterOperationalCredentialsAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOperationalCredentialsID)
                                        attributeID:@(MTRAttributeIDTypeClusterOperationalCredentialsAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterOperationalCredentials (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

- (void)attestationRequestWithParams:(MTROperationalCredentialsClusterAttestationRequestParams *)params
                      expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
               expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                   completionHandler:(void (^)(MTROperationalCredentialsClusterAttestationResponseParams * _Nullable data,
                                         NSError * _Nullable error))completionHandler
{
    [self attestationRequestWithParams:params
                        expectedValues:expectedDataValueDictionaries
                 expectedValueInterval:expectedValueIntervalMs
                            completion:^(MTROperationalCredentialsClusterAttestationResponseParams * _Nullable data,
                                NSError * _Nullable error) {
                                // Cast is safe because subclass does not add any selectors.
                                completionHandler(
                                    static_cast<MTROperationalCredentialsClusterAttestationResponseParams *>(data), error);
                            }];
}
- (void)certificateChainRequestWithParams:(MTROperationalCredentialsClusterCertificateChainRequestParams *)params
                           expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
                    expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                        completionHandler:(void (^)(MTROperationalCredentialsClusterCertificateChainResponseParams * _Nullable data,
                                              NSError * _Nullable error))completionHandler
{
    [self
        certificateChainRequestWithParams:params
                           expectedValues:expectedDataValueDictionaries
                    expectedValueInterval:expectedValueIntervalMs
                               completion:^(MTROperationalCredentialsClusterCertificateChainResponseParams * _Nullable data,
                                   NSError * _Nullable error) {
                                   // Cast is safe because subclass does not add any selectors.
                                   completionHandler(
                                       static_cast<MTROperationalCredentialsClusterCertificateChainResponseParams *>(data), error);
                               }];
}
- (void)CSRRequestWithParams:(MTROperationalCredentialsClusterCSRRequestParams *)params
              expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
       expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
           completionHandler:(void (^)(MTROperationalCredentialsClusterCSRResponseParams * _Nullable data,
                                 NSError * _Nullable error))completionHandler
{
    [self CSRRequestWithParams:params
                expectedValues:expectedDataValueDictionaries
         expectedValueInterval:expectedValueIntervalMs
                    completion:^(MTROperationalCredentialsClusterCSRResponseParams * _Nullable data, NSError * _Nullable error) {
                        // Cast is safe because subclass does not add any selectors.
                        completionHandler(static_cast<MTROperationalCredentialsClusterCSRResponseParams *>(data), error);
                    }];
}
- (void)addNOCWithParams:(MTROperationalCredentialsClusterAddNOCParams *)params
           expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
    expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
        completionHandler:(void (^)(MTROperationalCredentialsClusterNOCResponseParams * _Nullable data,
                              NSError * _Nullable error))completionHandler
{
    [self addNOCWithParams:params
               expectedValues:expectedDataValueDictionaries
        expectedValueInterval:expectedValueIntervalMs
                   completion:^(MTROperationalCredentialsClusterNOCResponseParams * _Nullable data, NSError * _Nullable error) {
                       // Cast is safe because subclass does not add any selectors.
                       completionHandler(static_cast<MTROperationalCredentialsClusterNOCResponseParams *>(data), error);
                   }];
}
- (void)updateNOCWithParams:(MTROperationalCredentialsClusterUpdateNOCParams *)params
             expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
      expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
          completionHandler:(void (^)(MTROperationalCredentialsClusterNOCResponseParams * _Nullable data,
                                NSError * _Nullable error))completionHandler
{
    [self updateNOCWithParams:params
               expectedValues:expectedDataValueDictionaries
        expectedValueInterval:expectedValueIntervalMs
                   completion:^(MTROperationalCredentialsClusterNOCResponseParams * _Nullable data, NSError * _Nullable error) {
                       // Cast is safe because subclass does not add any selectors.
                       completionHandler(static_cast<MTROperationalCredentialsClusterNOCResponseParams *>(data), error);
                   }];
}
- (void)updateFabricLabelWithParams:(MTROperationalCredentialsClusterUpdateFabricLabelParams *)params
                     expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
              expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                  completionHandler:(void (^)(MTROperationalCredentialsClusterNOCResponseParams * _Nullable data,
                                        NSError * _Nullable error))completionHandler
{
    [self updateFabricLabelWithParams:params
                       expectedValues:expectedDataValueDictionaries
                expectedValueInterval:expectedValueIntervalMs
                           completion:^(
                               MTROperationalCredentialsClusterNOCResponseParams * _Nullable data, NSError * _Nullable error) {
                               // Cast is safe because subclass does not add any selectors.
                               completionHandler(static_cast<MTROperationalCredentialsClusterNOCResponseParams *>(data), error);
                           }];
}
- (void)removeFabricWithParams:(MTROperationalCredentialsClusterRemoveFabricParams *)params
                expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
         expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
             completionHandler:(void (^)(MTROperationalCredentialsClusterNOCResponseParams * _Nullable data,
                                   NSError * _Nullable error))completionHandler
{
    [self removeFabricWithParams:params
                  expectedValues:expectedDataValueDictionaries
           expectedValueInterval:expectedValueIntervalMs
                      completion:^(MTROperationalCredentialsClusterNOCResponseParams * _Nullable data, NSError * _Nullable error) {
                          // Cast is safe because subclass does not add any selectors.
                          completionHandler(static_cast<MTROperationalCredentialsClusterNOCResponseParams *>(data), error);
                      }];
}
- (void)addTrustedRootCertificateWithParams:(MTROperationalCredentialsClusterAddTrustedRootCertificateParams *)params
                             expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
                      expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                          completionHandler:(MTRStatusCompletion)completionHandler
{
    [self addTrustedRootCertificateWithParams:params
                               expectedValues:expectedDataValueDictionaries
                        expectedValueInterval:expectedValueIntervalMs
                                   completion:completionHandler];
}
@end

@implementation MTRClusterGroupKeyManagement

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)keySetWriteWithParams:(MTRGroupKeyManagementClusterKeySetWriteParams *)params
               expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
        expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                   completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeGroupKeyManagementID,
                                     (unsigned int) MTRCommandIDTypeClusterGroupKeyManagementCommandKeySetWriteID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                GroupKeyManagement::Commands::KeySetWrite::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.groupKeySet.groupKeySetID = params.groupKeySet.groupKeySetID.unsignedShortValue;
                request.groupKeySet.groupKeySecurityPolicy
                    = static_cast<std::remove_reference_t<decltype(request.groupKeySet.groupKeySecurityPolicy)>>(
                        params.groupKeySet.groupKeySecurityPolicy.unsignedCharValue);
                if (params.groupKeySet.epochKey0 == nil) {
                    request.groupKeySet.epochKey0.SetNull();
                } else {
                    auto & nonNullValue_1 = request.groupKeySet.epochKey0.SetNonNull();
                    nonNullValue_1 = [self asByteSpan:params.groupKeySet.epochKey0];
                }
                if (params.groupKeySet.epochStartTime0 == nil) {
                    request.groupKeySet.epochStartTime0.SetNull();
                } else {
                    auto & nonNullValue_1 = request.groupKeySet.epochStartTime0.SetNonNull();
                    nonNullValue_1 = params.groupKeySet.epochStartTime0.unsignedLongLongValue;
                }
                if (params.groupKeySet.epochKey1 == nil) {
                    request.groupKeySet.epochKey1.SetNull();
                } else {
                    auto & nonNullValue_1 = request.groupKeySet.epochKey1.SetNonNull();
                    nonNullValue_1 = [self asByteSpan:params.groupKeySet.epochKey1];
                }
                if (params.groupKeySet.epochStartTime1 == nil) {
                    request.groupKeySet.epochStartTime1.SetNull();
                } else {
                    auto & nonNullValue_1 = request.groupKeySet.epochStartTime1.SetNonNull();
                    nonNullValue_1 = params.groupKeySet.epochStartTime1.unsignedLongLongValue;
                }
                if (params.groupKeySet.epochKey2 == nil) {
                    request.groupKeySet.epochKey2.SetNull();
                } else {
                    auto & nonNullValue_1 = request.groupKeySet.epochKey2.SetNonNull();
                    nonNullValue_1 = [self asByteSpan:params.groupKeySet.epochKey2];
                }
                if (params.groupKeySet.epochStartTime2 == nil) {
                    request.groupKeySet.epochStartTime2.SetNull();
                } else {
                    auto & nonNullValue_1 = request.groupKeySet.epochStartTime2.SetNonNull();
                    nonNullValue_1 = params.groupKeySet.epochStartTime2.unsignedLongLongValue;
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)keySetReadWithParams:(MTRGroupKeyManagementClusterKeySetReadParams *)params
              expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
       expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                  completion:(void (^)(MTRGroupKeyManagementClusterKeySetReadResponseParams * _Nullable data,
                                 NSError * _Nullable error))completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeGroupKeyManagementID,
                                     (unsigned int) MTRCommandIDTypeClusterGroupKeyManagementCommandKeySetReadID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRGroupKeyManagementClusterKeySetReadResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                GroupKeyManagementClusterKeySetReadResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRGroupKeyManagementClusterKeySetReadResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                GroupKeyManagement::Commands::KeySetRead::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.groupKeySetID = params.groupKeySetID.unsignedShortValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)keySetRemoveWithParams:(MTRGroupKeyManagementClusterKeySetRemoveParams *)params
                expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
         expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                    completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeGroupKeyManagementID,
                                     (unsigned int) MTRCommandIDTypeClusterGroupKeyManagementCommandKeySetRemoveID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                GroupKeyManagement::Commands::KeySetRemove::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.groupKeySetID = params.groupKeySetID.unsignedShortValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)keySetReadAllIndicesWithParams:(MTRGroupKeyManagementClusterKeySetReadAllIndicesParams *)params
                        expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                 expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                            completion:(void (^)(MTRGroupKeyManagementClusterKeySetReadAllIndicesResponseParams * _Nullable data,
                                           NSError * _Nullable error))completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeGroupKeyManagementID,
                                     (unsigned int) MTRCommandIDTypeClusterGroupKeyManagementCommandKeySetReadAllIndicesID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRGroupKeyManagementClusterKeySetReadAllIndicesResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                GroupKeyManagementClusterKeySetReadAllIndicesResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRGroupKeyManagementClusterKeySetReadAllIndicesResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                GroupKeyManagement::Commands::KeySetReadAllIndices::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                {
                    using ListType_0 = std::remove_reference_t<decltype(request.groupKeySetIDs)>;
                    using ListMemberType_0 = ListMemberTypeGetter<ListType_0>::Type;
                    if (params.groupKeySetIDs.count != 0) {
                        auto * listHolder_0 = new ListHolder<ListMemberType_0>(params.groupKeySetIDs.count);
                        if (listHolder_0 == nullptr || listHolder_0->mList == nullptr) {
                            return CHIP_ERROR_INVALID_ARGUMENT;
                        }
                        listFreer.add(listHolder_0);
                        for (size_t i_0 = 0; i_0 < params.groupKeySetIDs.count; ++i_0) {
                            if (![params.groupKeySetIDs[i_0] isKindOfClass:[NSNumber class]]) {
                                // Wrong kind of value.
                                return CHIP_ERROR_INVALID_ARGUMENT;
                            }
                            auto element_0 = (NSNumber *) params.groupKeySetIDs[i_0];
                            listHolder_0->mList[i_0] = element_0.unsignedShortValue;
                        }
                        request.groupKeySetIDs = ListType_0(listHolder_0->mList, params.groupKeySetIDs.count);
                    } else {
                        request.groupKeySetIDs = ListType_0();
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeGroupKeyMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGroupKeyManagementID)
                                        attributeID:@(MTRAttributeIDTypeClusterGroupKeyManagementAttributeGroupKeyMapID)
                                             params:params];
}

- (void)writeAttributeGroupKeyMapWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                     expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeGroupKeyMapWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeGroupKeyMapWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                     expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                    params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeGroupKeyManagementID)
                                  attributeID:@(MTRAttributeIDTypeClusterGroupKeyManagementAttributeGroupKeyMapID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeGroupTableWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGroupKeyManagementID)
                                        attributeID:@(MTRAttributeIDTypeClusterGroupKeyManagementAttributeGroupTableID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeMaxGroupsPerFabricWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGroupKeyManagementID)
                                        attributeID:@(MTRAttributeIDTypeClusterGroupKeyManagementAttributeMaxGroupsPerFabricID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeMaxGroupKeysPerFabricWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGroupKeyManagementID)
                                        attributeID:@(MTRAttributeIDTypeClusterGroupKeyManagementAttributeMaxGroupKeysPerFabricID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGroupKeyManagementID)
                                        attributeID:@(MTRAttributeIDTypeClusterGroupKeyManagementAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGroupKeyManagementID)
                                        attributeID:@(MTRAttributeIDTypeClusterGroupKeyManagementAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGroupKeyManagementID)
                                        attributeID:@(MTRAttributeIDTypeClusterGroupKeyManagementAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGroupKeyManagementID)
                                        attributeID:@(MTRAttributeIDTypeClusterGroupKeyManagementAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGroupKeyManagementID)
                                        attributeID:@(MTRAttributeIDTypeClusterGroupKeyManagementAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeGroupKeyManagementID)
                                        attributeID:@(MTRAttributeIDTypeClusterGroupKeyManagementAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterGroupKeyManagement (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

- (void)keySetWriteWithParams:(MTRGroupKeyManagementClusterKeySetWriteParams *)params
               expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
        expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
            completionHandler:(MTRStatusCompletion)completionHandler
{
    [self keySetWriteWithParams:params
                 expectedValues:expectedDataValueDictionaries
          expectedValueInterval:expectedValueIntervalMs
                     completion:completionHandler];
}
- (void)keySetReadWithParams:(MTRGroupKeyManagementClusterKeySetReadParams *)params
              expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
       expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
           completionHandler:(void (^)(MTRGroupKeyManagementClusterKeySetReadResponseParams * _Nullable data,
                                 NSError * _Nullable error))completionHandler
{
    [self keySetReadWithParams:params
                expectedValues:expectedDataValueDictionaries
         expectedValueInterval:expectedValueIntervalMs
                    completion:^(MTRGroupKeyManagementClusterKeySetReadResponseParams * _Nullable data, NSError * _Nullable error) {
                        // Cast is safe because subclass does not add any selectors.
                        completionHandler(static_cast<MTRGroupKeyManagementClusterKeySetReadResponseParams *>(data), error);
                    }];
}
- (void)keySetRemoveWithParams:(MTRGroupKeyManagementClusterKeySetRemoveParams *)params
                expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
         expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
             completionHandler:(MTRStatusCompletion)completionHandler
{
    [self keySetRemoveWithParams:params
                  expectedValues:expectedDataValueDictionaries
           expectedValueInterval:expectedValueIntervalMs
                      completion:completionHandler];
}
- (void)keySetReadAllIndicesWithParams:(MTRGroupKeyManagementClusterKeySetReadAllIndicesParams *)params
                        expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
                 expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
                     completionHandler:(void (^)(MTRGroupKeyManagementClusterKeySetReadAllIndicesResponseParams * _Nullable data,
                                           NSError * _Nullable error))completionHandler
{
    [self keySetReadAllIndicesWithParams:params
                          expectedValues:expectedDataValueDictionaries
                   expectedValueInterval:expectedValueIntervalMs
                              completion:^(MTRGroupKeyManagementClusterKeySetReadAllIndicesResponseParams * _Nullable data,
                                  NSError * _Nullable error) {
                                  // Cast is safe because subclass does not add any selectors.
                                  completionHandler(
                                      static_cast<MTRGroupKeyManagementClusterKeySetReadAllIndicesResponseParams *>(data), error);
                              }];
}
@end

@implementation MTRClusterFixedLabel

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (NSDictionary<NSString *, id> *)readAttributeLabelListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeFixedLabelID)
                                        attributeID:@(MTRAttributeIDTypeClusterFixedLabelAttributeLabelListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeFixedLabelID)
                                        attributeID:@(MTRAttributeIDTypeClusterFixedLabelAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeFixedLabelID)
                                        attributeID:@(MTRAttributeIDTypeClusterFixedLabelAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeFixedLabelID)
                                        attributeID:@(MTRAttributeIDTypeClusterFixedLabelAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeFixedLabelID)
                                        attributeID:@(MTRAttributeIDTypeClusterFixedLabelAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeFixedLabelID)
                                        attributeID:@(MTRAttributeIDTypeClusterFixedLabelAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeFixedLabelID)
                                        attributeID:@(MTRAttributeIDTypeClusterFixedLabelAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterFixedLabel (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

@end

@implementation MTRClusterUserLabel

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (NSDictionary<NSString *, id> *)readAttributeLabelListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeUserLabelID)
                                        attributeID:@(MTRAttributeIDTypeClusterUserLabelAttributeLabelListID)
                                             params:params];
}

- (void)writeAttributeLabelListWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                   expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeLabelListWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeLabelListWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                   expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                  params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeUserLabelID)
                                  attributeID:@(MTRAttributeIDTypeClusterUserLabelAttributeLabelListID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeUserLabelID)
                                        attributeID:@(MTRAttributeIDTypeClusterUserLabelAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeUserLabelID)
                                        attributeID:@(MTRAttributeIDTypeClusterUserLabelAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeUserLabelID)
                                        attributeID:@(MTRAttributeIDTypeClusterUserLabelAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeUserLabelID)
                                        attributeID:@(MTRAttributeIDTypeClusterUserLabelAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeUserLabelID)
                                        attributeID:@(MTRAttributeIDTypeClusterUserLabelAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeUserLabelID)
                                        attributeID:@(MTRAttributeIDTypeClusterUserLabelAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterUserLabel (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

@end

@implementation MTRClusterBooleanState

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (NSDictionary<NSString *, id> *)readAttributeStateValueWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBooleanStateID)
                                        attributeID:@(MTRAttributeIDTypeClusterBooleanStateAttributeStateValueID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBooleanStateID)
                                        attributeID:@(MTRAttributeIDTypeClusterBooleanStateAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBooleanStateID)
                                        attributeID:@(MTRAttributeIDTypeClusterBooleanStateAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBooleanStateID)
                                        attributeID:@(MTRAttributeIDTypeClusterBooleanStateAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBooleanStateID)
                                        attributeID:@(MTRAttributeIDTypeClusterBooleanStateAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBooleanStateID)
                                        attributeID:@(MTRAttributeIDTypeClusterBooleanStateAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeBooleanStateID)
                                        attributeID:@(MTRAttributeIDTypeClusterBooleanStateAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterBooleanState (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

@end

@implementation MTRClusterModeSelect

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)changeToModeWithParams:(MTRModeSelectClusterChangeToModeParams *)params
                expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
         expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                    completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeModeSelectID,
                                     (unsigned int) MTRCommandIDTypeClusterModeSelectCommandChangeToModeID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                ModeSelect::Commands::ChangeToMode::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.newMode = params.newMode.unsignedCharValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeDescriptionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeModeSelectID)
                                        attributeID:@(MTRAttributeIDTypeClusterModeSelectAttributeDescriptionID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeStandardNamespaceWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeModeSelectID)
                                        attributeID:@(MTRAttributeIDTypeClusterModeSelectAttributeStandardNamespaceID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeSupportedModesWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeModeSelectID)
                                        attributeID:@(MTRAttributeIDTypeClusterModeSelectAttributeSupportedModesID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeCurrentModeWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeModeSelectID)
                                        attributeID:@(MTRAttributeIDTypeClusterModeSelectAttributeCurrentModeID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeStartUpModeWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeModeSelectID)
                                        attributeID:@(MTRAttributeIDTypeClusterModeSelectAttributeStartUpModeID)
                                             params:params];
}

- (void)writeAttributeStartUpModeWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                     expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeStartUpModeWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeStartUpModeWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                     expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                    params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeModeSelectID)
                                  attributeID:@(MTRAttributeIDTypeClusterModeSelectAttributeStartUpModeID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeOnModeWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeModeSelectID)
                                        attributeID:@(MTRAttributeIDTypeClusterModeSelectAttributeOnModeID)
                                             params:params];
}

- (void)writeAttributeOnModeWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeOnModeWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeOnModeWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                               params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeModeSelectID)
                                  attributeID:@(MTRAttributeIDTypeClusterModeSelectAttributeOnModeID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeModeSelectID)
                                        attributeID:@(MTRAttributeIDTypeClusterModeSelectAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeModeSelectID)
                                        attributeID:@(MTRAttributeIDTypeClusterModeSelectAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeModeSelectID)
                                        attributeID:@(MTRAttributeIDTypeClusterModeSelectAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeModeSelectID)
                                        attributeID:@(MTRAttributeIDTypeClusterModeSelectAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeModeSelectID)
                                        attributeID:@(MTRAttributeIDTypeClusterModeSelectAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeModeSelectID)
                                        attributeID:@(MTRAttributeIDTypeClusterModeSelectAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterModeSelect (Deprecated)

- (instancetype)initWithDevice:(MTRDevice *)device endpoint:(uint16_t)endpoint queue:(dispatch_queue_t)queue
{
    return [self initWithDevice:device endpointID:@(endpoint) queue:queue];
}

- (void)changeToModeWithParams:(MTRModeSelectClusterChangeToModeParams *)params
                expectedValues:(NSArray<NSDictionary<NSString *, id> *> * _Nullable)expectedDataValueDictionaries
         expectedValueInterval:(NSNumber * _Nullable)expectedValueIntervalMs
             completionHandler:(MTRStatusCompletion)completionHandler
{
    [self changeToModeWithParams:params
                  expectedValues:expectedDataValueDictionaries
           expectedValueInterval:expectedValueIntervalMs
                      completion:completionHandler];
}
@end

@implementation MTRClusterTemperatureControl

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)setTemperatureWithParams:(MTRTemperatureControlClusterSetTemperatureParams * _Nullable)params
                  expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
           expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                      completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeTemperatureControlID,
                                     (unsigned int) MTRCommandIDTypeClusterTemperatureControlCommandSetTemperatureID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                TemperatureControl::Commands::SetTemperature::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                if (params != nil) {
                    if (params.targetTemperature != nil) {
                        auto & definedValue_0 = request.targetTemperature.Emplace();
                        definedValue_0 = params.targetTemperature.shortValue;
                    }
                    if (params.targetTemperatureLevel != nil) {
                        auto & definedValue_0 = request.targetTemperatureLevel.Emplace();
                        definedValue_0 = params.targetTemperatureLevel.unsignedCharValue;
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeTemperatureSetpointWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeTemperatureControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterTemperatureControlAttributeTemperatureSetpointID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeMinTemperatureWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeTemperatureControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterTemperatureControlAttributeMinTemperatureID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeMaxTemperatureWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeTemperatureControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterTemperatureControlAttributeMaxTemperatureID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeStepWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeTemperatureControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterTemperatureControlAttributeStepID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeCurrentTemperatureLevelIndexWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeTemperatureControlID)
                        attributeID:@(MTRAttributeIDTypeClusterTemperatureControlAttributeCurrentTemperatureLevelIndexID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeSupportedTemperatureLevelsWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeTemperatureControlID)
                                     attributeID:@(MTRAttributeIDTypeClusterTemperatureControlAttributeSupportedTemperatureLevelsID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeTemperatureControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterTemperatureControlAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeTemperatureControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterTemperatureControlAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeTemperatureControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterTemperatureControlAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeTemperatureControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterTemperatureControlAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeTemperatureControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterTemperatureControlAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeTemperatureControlID)
                                        attributeID:@(MTRAttributeIDTypeClusterTemperatureControlAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterRefrigeratorAlarm

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)resetWithParams:(MTRRefrigeratorAlarmClusterResetParams *)params
           expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
               completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeRefrigeratorAlarmID,
                                     (unsigned int) MTRCommandIDTypeClusterRefrigeratorAlarmCommandResetID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                RefrigeratorAlarm::Commands::Reset::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.alarms = static_cast<std::remove_reference_t<decltype(request.alarms)>>(params.alarms.unsignedIntValue);
                if (params.mask != nil) {
                    auto & definedValue_0 = request.mask.Emplace();
                    definedValue_0 = static_cast<std::remove_reference_t<decltype(definedValue_0)>>(params.mask.unsignedIntValue);
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeMaskWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeRefrigeratorAlarmID)
                                        attributeID:@(MTRAttributeIDTypeClusterRefrigeratorAlarmAttributeMaskID)
                                             params:params];
}

- (void)writeAttributeMaskWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
              expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeMaskWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeMaskWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
              expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                             params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeRefrigeratorAlarmID)
                                  attributeID:@(MTRAttributeIDTypeClusterRefrigeratorAlarmAttributeMaskID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeLatchWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeRefrigeratorAlarmID)
                                        attributeID:@(MTRAttributeIDTypeClusterRefrigeratorAlarmAttributeLatchID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeStateWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeRefrigeratorAlarmID)
                                        attributeID:@(MTRAttributeIDTypeClusterRefrigeratorAlarmAttributeStateID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeRefrigeratorAlarmID)
                                        attributeID:@(MTRAttributeIDTypeClusterRefrigeratorAlarmAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeRefrigeratorAlarmID)
                                        attributeID:@(MTRAttributeIDTypeClusterRefrigeratorAlarmAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeRefrigeratorAlarmID)
                                        attributeID:@(MTRAttributeIDTypeClusterRefrigeratorAlarmAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeRefrigeratorAlarmID)
                                        attributeID:@(MTRAttributeIDTypeClusterRefrigeratorAlarmAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeRefrigeratorAlarmID)
                                        attributeID:@(MTRAttributeIDTypeClusterRefrigeratorAlarmAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeRefrigeratorAlarmID)
                                        attributeID:@(MTRAttributeIDTypeClusterRefrigeratorAlarmAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterAirQuality

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (NSDictionary<NSString *, id> *)readAttributeAirQualityWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeAirQualityID)
                                        attributeID:@(MTRAttributeIDTypeClusterAirQualityAttributeAirQualityID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeAirQualityID)
                                        attributeID:@(MTRAttributeIDTypeClusterAirQualityAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeAirQualityID)
                                        attributeID:@(MTRAttributeIDTypeClusterAirQualityAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeAirQualityID)
                                        attributeID:@(MTRAttributeIDTypeClusterAirQualityAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeAirQualityID)
                                        attributeID:@(MTRAttributeIDTypeClusterAirQualityAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeAirQualityID)
                                        attributeID:@(MTRAttributeIDTypeClusterAirQualityAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeAirQualityID)
                                        attributeID:@(MTRAttributeIDTypeClusterAirQualityAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterSmokeCOAlarm

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)selfTestRequestWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                               completion:(MTRStatusCompletion)completion
{
    [self selfTestRequestWithParams:nil
                     expectedValues:expectedValues
              expectedValueInterval:expectedValueIntervalMs
                         completion:completion];
}
- (void)selfTestRequestWithParams:(MTRSmokeCOAlarmClusterSelfTestRequestParams * _Nullable)params
                   expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
            expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                       completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeSmokeCOAlarmID,
                                     (unsigned int) MTRCommandIDTypeClusterSmokeCOAlarmCommandSelfTestRequestID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                SmokeCoAlarm::Commands::SelfTestRequest::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeExpressedStateWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSmokeCOAlarmID)
                                        attributeID:@(MTRAttributeIDTypeClusterSmokeCOAlarmAttributeExpressedStateID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeSmokeStateWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSmokeCOAlarmID)
                                        attributeID:@(MTRAttributeIDTypeClusterSmokeCOAlarmAttributeSmokeStateID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeCOStateWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSmokeCOAlarmID)
                                        attributeID:@(MTRAttributeIDTypeClusterSmokeCOAlarmAttributeCOStateID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeBatteryAlertWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSmokeCOAlarmID)
                                        attributeID:@(MTRAttributeIDTypeClusterSmokeCOAlarmAttributeBatteryAlertID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeDeviceMutedWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSmokeCOAlarmID)
                                        attributeID:@(MTRAttributeIDTypeClusterSmokeCOAlarmAttributeDeviceMutedID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeTestInProgressWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSmokeCOAlarmID)
                                        attributeID:@(MTRAttributeIDTypeClusterSmokeCOAlarmAttributeTestInProgressID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeHardwareFaultAlertWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSmokeCOAlarmID)
                                        attributeID:@(MTRAttributeIDTypeClusterSmokeCOAlarmAttributeHardwareFaultAlertID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEndOfServiceAlertWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSmokeCOAlarmID)
                                        attributeID:@(MTRAttributeIDTypeClusterSmokeCOAlarmAttributeEndOfServiceAlertID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeInterconnectSmokeAlarmWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSmokeCOAlarmID)
                                        attributeID:@(MTRAttributeIDTypeClusterSmokeCOAlarmAttributeInterconnectSmokeAlarmID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeInterconnectCOAlarmWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSmokeCOAlarmID)
                                        attributeID:@(MTRAttributeIDTypeClusterSmokeCOAlarmAttributeInterconnectCOAlarmID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeContaminationStateWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSmokeCOAlarmID)
                                        attributeID:@(MTRAttributeIDTypeClusterSmokeCOAlarmAttributeContaminationStateID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeSensitivityLevelWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSmokeCOAlarmID)
                                        attributeID:@(MTRAttributeIDTypeClusterSmokeCOAlarmAttributeSensitivityLevelID)
                                             params:params];
}

- (void)writeAttributeSensitivityLevelWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                          expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeSensitivityLevelWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeSensitivityLevelWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                          expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                         params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeSmokeCOAlarmID)
                                  attributeID:@(MTRAttributeIDTypeClusterSmokeCOAlarmAttributeSensitivityLevelID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSmokeCOAlarmID)
                                        attributeID:@(MTRAttributeIDTypeClusterSmokeCOAlarmAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSmokeCOAlarmID)
                                        attributeID:@(MTRAttributeIDTypeClusterSmokeCOAlarmAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSmokeCOAlarmID)
                                        attributeID:@(MTRAttributeIDTypeClusterSmokeCOAlarmAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSmokeCOAlarmID)
                                        attributeID:@(MTRAttributeIDTypeClusterSmokeCOAlarmAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSmokeCOAlarmID)
                                        attributeID:@(MTRAttributeIDTypeClusterSmokeCOAlarmAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeSmokeCOAlarmID)
                                        attributeID:@(MTRAttributeIDTypeClusterSmokeCOAlarmAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterOperationalState

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)pauseWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
          expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                     completion:(void (^)(MTROperationalStateClusterOperationalCommandResponseParams * _Nullable data,
                                    NSError * _Nullable error))completion
{
    [self pauseWithParams:nil expectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs completion:completion];
}
- (void)pauseWithParams:(MTROperationalStateClusterPauseParams * _Nullable)params
           expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
               completion:(void (^)(MTROperationalStateClusterOperationalCommandResponseParams * _Nullable data,
                              NSError * _Nullable error))completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeOperationalStateID,
                                     (unsigned int) MTRCommandIDTypeClusterOperationalStateCommandPauseID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTROperationalStateClusterOperationalCommandResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                OperationalStateClusterOperationalCommandResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTROperationalStateClusterOperationalCommandResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                OperationalState::Commands::Pause::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)stopWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
         expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                    completion:(void (^)(MTROperationalStateClusterOperationalCommandResponseParams * _Nullable data,
                                   NSError * _Nullable error))completion
{
    [self stopWithParams:nil expectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs completion:completion];
}
- (void)stopWithParams:(MTROperationalStateClusterStopParams * _Nullable)params
           expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
               completion:(void (^)(MTROperationalStateClusterOperationalCommandResponseParams * _Nullable data,
                              NSError * _Nullable error))completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeOperationalStateID,
                                     (unsigned int) MTRCommandIDTypeClusterOperationalStateCommandStopID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTROperationalStateClusterOperationalCommandResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                OperationalStateClusterOperationalCommandResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTROperationalStateClusterOperationalCommandResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                OperationalState::Commands::Stop::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)startWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
          expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                     completion:(void (^)(MTROperationalStateClusterOperationalCommandResponseParams * _Nullable data,
                                    NSError * _Nullable error))completion
{
    [self startWithParams:nil expectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs completion:completion];
}
- (void)startWithParams:(MTROperationalStateClusterStartParams * _Nullable)params
           expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
               completion:(void (^)(MTROperationalStateClusterOperationalCommandResponseParams * _Nullable data,
                              NSError * _Nullable error))completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeOperationalStateID,
                                     (unsigned int) MTRCommandIDTypeClusterOperationalStateCommandStartID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTROperationalStateClusterOperationalCommandResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                OperationalStateClusterOperationalCommandResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTROperationalStateClusterOperationalCommandResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                OperationalState::Commands::Start::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)resumeWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
           expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                      completion:(void (^)(MTROperationalStateClusterOperationalCommandResponseParams * _Nullable data,
                                     NSError * _Nullable error))completion
{
    [self resumeWithParams:nil expectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs completion:completion];
}
- (void)resumeWithParams:(MTROperationalStateClusterResumeParams * _Nullable)params
           expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
               completion:(void (^)(MTROperationalStateClusterOperationalCommandResponseParams * _Nullable data,
                              NSError * _Nullable error))completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeOperationalStateID,
                                     (unsigned int) MTRCommandIDTypeClusterOperationalStateCommandResumeID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTROperationalStateClusterOperationalCommandResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                OperationalStateClusterOperationalCommandResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTROperationalStateClusterOperationalCommandResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                OperationalState::Commands::Resume::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributePhaseListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOperationalStateID)
                                        attributeID:@(MTRAttributeIDTypeClusterOperationalStateAttributePhaseListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeCurrentPhaseWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOperationalStateID)
                                        attributeID:@(MTRAttributeIDTypeClusterOperationalStateAttributeCurrentPhaseID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeCountdownTimeWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOperationalStateID)
                                        attributeID:@(MTRAttributeIDTypeClusterOperationalStateAttributeCountdownTimeID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeOperationalStateListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOperationalStateID)
                                        attributeID:@(MTRAttributeIDTypeClusterOperationalStateAttributeOperationalStateListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeOperationalStateWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOperationalStateID)
                                        attributeID:@(MTRAttributeIDTypeClusterOperationalStateAttributeOperationalStateID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeOperationalErrorWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOperationalStateID)
                                        attributeID:@(MTRAttributeIDTypeClusterOperationalStateAttributeOperationalErrorID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOperationalStateID)
                                        attributeID:@(MTRAttributeIDTypeClusterOperationalStateAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOperationalStateID)
                                        attributeID:@(MTRAttributeIDTypeClusterOperationalStateAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOperationalStateID)
                                        attributeID:@(MTRAttributeIDTypeClusterOperationalStateAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOperationalStateID)
                                        attributeID:@(MTRAttributeIDTypeClusterOperationalStateAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOperationalStateID)
                                        attributeID:@(MTRAttributeIDTypeClusterOperationalStateAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOperationalStateID)
                                        attributeID:@(MTRAttributeIDTypeClusterOperationalStateAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterHEPAFilterMonitoring

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)resetConditionWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                   expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                              completion:(MTRStatusCompletion)completion
{
    [self resetConditionWithParams:nil
                    expectedValues:expectedValues
             expectedValueInterval:expectedValueIntervalMs
                        completion:completion];
}
- (void)resetConditionWithParams:(MTRHEPAFilterMonitoringClusterResetConditionParams * _Nullable)params
                  expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
           expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                      completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeHEPAFilterMonitoringID,
                                     (unsigned int) MTRCommandIDTypeClusterHEPAFilterMonitoringCommandResetConditionID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                HepaFilterMonitoring::Commands::ResetCondition::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeConditionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeHEPAFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterHEPAFilterMonitoringAttributeConditionID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeDegradationDirectionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeHEPAFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterHEPAFilterMonitoringAttributeDegradationDirectionID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeChangeIndicationWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeHEPAFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterHEPAFilterMonitoringAttributeChangeIndicationID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeInPlaceIndicatorWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeHEPAFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterHEPAFilterMonitoringAttributeInPlaceIndicatorID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeHEPAFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterHEPAFilterMonitoringAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeHEPAFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterHEPAFilterMonitoringAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeHEPAFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterHEPAFilterMonitoringAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeHEPAFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterHEPAFilterMonitoringAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeHEPAFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterHEPAFilterMonitoringAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeHEPAFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterHEPAFilterMonitoringAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterActivatedCarbonFilterMonitoring

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)resetConditionWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                   expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                              completion:(MTRStatusCompletion)completion
{
    [self resetConditionWithParams:nil
                    expectedValues:expectedValues
             expectedValueInterval:expectedValueIntervalMs
                        completion:completion];
}
- (void)resetConditionWithParams:(MTRActivatedCarbonFilterMonitoringClusterResetConditionParams * _Nullable)params
                  expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
           expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                      completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeActivatedCarbonFilterMonitoringID,
                                     (unsigned int) MTRCommandIDTypeClusterActivatedCarbonFilterMonitoringCommandResetConditionID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                ActivatedCarbonFilterMonitoring::Commands::ResetCondition::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeConditionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeActivatedCarbonFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterActivatedCarbonFilterMonitoringAttributeConditionID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeDegradationDirectionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeActivatedCarbonFilterMonitoringID)
                        attributeID:@(MTRAttributeIDTypeClusterActivatedCarbonFilterMonitoringAttributeDegradationDirectionID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeChangeIndicationWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeActivatedCarbonFilterMonitoringID)
                        attributeID:@(MTRAttributeIDTypeClusterActivatedCarbonFilterMonitoringAttributeChangeIndicationID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeInPlaceIndicatorWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeActivatedCarbonFilterMonitoringID)
                        attributeID:@(MTRAttributeIDTypeClusterActivatedCarbonFilterMonitoringAttributeInPlaceIndicatorID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeActivatedCarbonFilterMonitoringID)
                        attributeID:@(MTRAttributeIDTypeClusterActivatedCarbonFilterMonitoringAttributeGeneratedCommandListID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeActivatedCarbonFilterMonitoringID)
                        attributeID:@(MTRAttributeIDTypeClusterActivatedCarbonFilterMonitoringAttributeAcceptedCommandListID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeActivatedCarbonFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterActivatedCarbonFilterMonitoringAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeActivatedCarbonFilterMonitoringID)
                                     attributeID:@(MTRAttributeIDTypeClusterActivatedCarbonFilterMonitoringAttributeAttributeListID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeActivatedCarbonFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterActivatedCarbonFilterMonitoringAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeActivatedCarbonFilterMonitoringID)
                        attributeID:@(MTRAttributeIDTypeClusterActivatedCarbonFilterMonitoringAttributeClusterRevisionID)
                             params:params];
}

@end

@implementation MTRClusterCeramicFilterMonitoring

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)resetConditionWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                   expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                              completion:(MTRStatusCompletion)completion
{
    [self resetConditionWithParams:nil
                    expectedValues:expectedValues
             expectedValueInterval:expectedValueIntervalMs
                        completion:completion];
}
- (void)resetConditionWithParams:(MTRCeramicFilterMonitoringClusterResetConditionParams * _Nullable)params
                  expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
           expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                      completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeCeramicFilterMonitoringID,
                                     (unsigned int) MTRCommandIDTypeClusterCeramicFilterMonitoringCommandResetConditionID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                CeramicFilterMonitoring::Commands::ResetCondition::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeConditionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeCeramicFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterCeramicFilterMonitoringAttributeConditionID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeDegradationDirectionWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeCeramicFilterMonitoringID)
                                     attributeID:@(MTRAttributeIDTypeClusterCeramicFilterMonitoringAttributeDegradationDirectionID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeChangeIndicationWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeCeramicFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterCeramicFilterMonitoringAttributeChangeIndicationID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeInPlaceIndicatorWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeCeramicFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterCeramicFilterMonitoringAttributeInPlaceIndicatorID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeCeramicFilterMonitoringID)
                                     attributeID:@(MTRAttributeIDTypeClusterCeramicFilterMonitoringAttributeGeneratedCommandListID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeCeramicFilterMonitoringID)
                                     attributeID:@(MTRAttributeIDTypeClusterCeramicFilterMonitoringAttributeAcceptedCommandListID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeCeramicFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterCeramicFilterMonitoringAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeCeramicFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterCeramicFilterMonitoringAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeCeramicFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterCeramicFilterMonitoringAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeCeramicFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterCeramicFilterMonitoringAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterElectrostaticFilterMonitoring

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)resetConditionWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                   expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                              completion:(MTRStatusCompletion)completion
{
    [self resetConditionWithParams:nil
                    expectedValues:expectedValues
             expectedValueInterval:expectedValueIntervalMs
                        completion:completion];
}
- (void)resetConditionWithParams:(MTRElectrostaticFilterMonitoringClusterResetConditionParams * _Nullable)params
                  expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
           expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                      completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeElectrostaticFilterMonitoringID,
                                     (unsigned int) MTRCommandIDTypeClusterElectrostaticFilterMonitoringCommandResetConditionID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                ElectrostaticFilterMonitoring::Commands::ResetCondition::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeConditionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeElectrostaticFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterElectrostaticFilterMonitoringAttributeConditionID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeDegradationDirectionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeElectrostaticFilterMonitoringID)
                        attributeID:@(MTRAttributeIDTypeClusterElectrostaticFilterMonitoringAttributeDegradationDirectionID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeChangeIndicationWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeElectrostaticFilterMonitoringID)
                        attributeID:@(MTRAttributeIDTypeClusterElectrostaticFilterMonitoringAttributeChangeIndicationID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeInPlaceIndicatorWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeElectrostaticFilterMonitoringID)
                        attributeID:@(MTRAttributeIDTypeClusterElectrostaticFilterMonitoringAttributeInPlaceIndicatorID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeElectrostaticFilterMonitoringID)
                        attributeID:@(MTRAttributeIDTypeClusterElectrostaticFilterMonitoringAttributeGeneratedCommandListID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeElectrostaticFilterMonitoringID)
                        attributeID:@(MTRAttributeIDTypeClusterElectrostaticFilterMonitoringAttributeAcceptedCommandListID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeElectrostaticFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterElectrostaticFilterMonitoringAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeElectrostaticFilterMonitoringID)
                                     attributeID:@(MTRAttributeIDTypeClusterElectrostaticFilterMonitoringAttributeAttributeListID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeElectrostaticFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterElectrostaticFilterMonitoringAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeElectrostaticFilterMonitoringID)
                                     attributeID:@(MTRAttributeIDTypeClusterElectrostaticFilterMonitoringAttributeClusterRevisionID)
                                          params:params];
}

@end

@implementation MTRClusterUVFilterMonitoring

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)resetConditionWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                   expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                              completion:(MTRStatusCompletion)completion
{
    [self resetConditionWithParams:nil
                    expectedValues:expectedValues
             expectedValueInterval:expectedValueIntervalMs
                        completion:completion];
}
- (void)resetConditionWithParams:(MTRUVFilterMonitoringClusterResetConditionParams * _Nullable)params
                  expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
           expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                      completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeUVFilterMonitoringID,
                                     (unsigned int) MTRCommandIDTypeClusterUVFilterMonitoringCommandResetConditionID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                UvFilterMonitoring::Commands::ResetCondition::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeConditionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeUVFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterUVFilterMonitoringAttributeConditionID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeDegradationDirectionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeUVFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterUVFilterMonitoringAttributeDegradationDirectionID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeChangeIndicationWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeUVFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterUVFilterMonitoringAttributeChangeIndicationID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeInPlaceIndicatorWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeUVFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterUVFilterMonitoringAttributeInPlaceIndicatorID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeUVFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterUVFilterMonitoringAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeUVFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterUVFilterMonitoringAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeUVFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterUVFilterMonitoringAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeUVFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterUVFilterMonitoringAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeUVFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterUVFilterMonitoringAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeUVFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterUVFilterMonitoringAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterIonizingFilterMonitoring

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)resetConditionWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                   expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                              completion:(MTRStatusCompletion)completion
{
    [self resetConditionWithParams:nil
                    expectedValues:expectedValues
             expectedValueInterval:expectedValueIntervalMs
                        completion:completion];
}
- (void)resetConditionWithParams:(MTRIonizingFilterMonitoringClusterResetConditionParams * _Nullable)params
                  expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
           expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                      completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeIonizingFilterMonitoringID,
                                     (unsigned int) MTRCommandIDTypeClusterIonizingFilterMonitoringCommandResetConditionID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                IonizingFilterMonitoring::Commands::ResetCondition::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeConditionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeIonizingFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterIonizingFilterMonitoringAttributeConditionID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeDegradationDirectionWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeIonizingFilterMonitoringID)
                                     attributeID:@(MTRAttributeIDTypeClusterIonizingFilterMonitoringAttributeDegradationDirectionID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeChangeIndicationWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeIonizingFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterIonizingFilterMonitoringAttributeChangeIndicationID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeInPlaceIndicatorWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeIonizingFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterIonizingFilterMonitoringAttributeInPlaceIndicatorID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeIonizingFilterMonitoringID)
                                     attributeID:@(MTRAttributeIDTypeClusterIonizingFilterMonitoringAttributeGeneratedCommandListID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeIonizingFilterMonitoringID)
                                     attributeID:@(MTRAttributeIDTypeClusterIonizingFilterMonitoringAttributeAcceptedCommandListID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeIonizingFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterIonizingFilterMonitoringAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeIonizingFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterIonizingFilterMonitoringAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeIonizingFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterIonizingFilterMonitoringAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeIonizingFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterIonizingFilterMonitoringAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterZeoliteFilterMonitoring

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)resetConditionWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                   expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                              completion:(MTRStatusCompletion)completion
{
    [self resetConditionWithParams:nil
                    expectedValues:expectedValues
             expectedValueInterval:expectedValueIntervalMs
                        completion:completion];
}
- (void)resetConditionWithParams:(MTRZeoliteFilterMonitoringClusterResetConditionParams * _Nullable)params
                  expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
           expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                      completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeZeoliteFilterMonitoringID,
                                     (unsigned int) MTRCommandIDTypeClusterZeoliteFilterMonitoringCommandResetConditionID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                ZeoliteFilterMonitoring::Commands::ResetCondition::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeConditionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeZeoliteFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterZeoliteFilterMonitoringAttributeConditionID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeDegradationDirectionWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeZeoliteFilterMonitoringID)
                                     attributeID:@(MTRAttributeIDTypeClusterZeoliteFilterMonitoringAttributeDegradationDirectionID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeChangeIndicationWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeZeoliteFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterZeoliteFilterMonitoringAttributeChangeIndicationID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeInPlaceIndicatorWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeZeoliteFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterZeoliteFilterMonitoringAttributeInPlaceIndicatorID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeZeoliteFilterMonitoringID)
                                     attributeID:@(MTRAttributeIDTypeClusterZeoliteFilterMonitoringAttributeGeneratedCommandListID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeZeoliteFilterMonitoringID)
                                     attributeID:@(MTRAttributeIDTypeClusterZeoliteFilterMonitoringAttributeAcceptedCommandListID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeZeoliteFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterZeoliteFilterMonitoringAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeZeoliteFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterZeoliteFilterMonitoringAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeZeoliteFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterZeoliteFilterMonitoringAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeZeoliteFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterZeoliteFilterMonitoringAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterOzoneFilterMonitoring

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)resetConditionWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                   expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                              completion:(MTRStatusCompletion)completion
{
    [self resetConditionWithParams:nil
                    expectedValues:expectedValues
             expectedValueInterval:expectedValueIntervalMs
                        completion:completion];
}
- (void)resetConditionWithParams:(MTROzoneFilterMonitoringClusterResetConditionParams * _Nullable)params
                  expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
           expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                      completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeOzoneFilterMonitoringID,
                                     (unsigned int) MTRCommandIDTypeClusterOzoneFilterMonitoringCommandResetConditionID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                OzoneFilterMonitoring::Commands::ResetCondition::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeConditionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOzoneFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterOzoneFilterMonitoringAttributeConditionID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeDegradationDirectionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOzoneFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterOzoneFilterMonitoringAttributeDegradationDirectionID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeChangeIndicationWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOzoneFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterOzoneFilterMonitoringAttributeChangeIndicationID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeInPlaceIndicatorWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOzoneFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterOzoneFilterMonitoringAttributeInPlaceIndicatorID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOzoneFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterOzoneFilterMonitoringAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOzoneFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterOzoneFilterMonitoringAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOzoneFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterOzoneFilterMonitoringAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOzoneFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterOzoneFilterMonitoringAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOzoneFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterOzoneFilterMonitoringAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeOzoneFilterMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterOzoneFilterMonitoringAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterWaterTankMonitoring

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)resetConditionWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                   expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                              completion:(MTRStatusCompletion)completion
{
    [self resetConditionWithParams:nil
                    expectedValues:expectedValues
             expectedValueInterval:expectedValueIntervalMs
                        completion:completion];
}
- (void)resetConditionWithParams:(MTRWaterTankMonitoringClusterResetConditionParams * _Nullable)params
                  expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
           expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                      completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeWaterTankMonitoringID,
                                     (unsigned int) MTRCommandIDTypeClusterWaterTankMonitoringCommandResetConditionID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                WaterTankMonitoring::Commands::ResetCondition::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeConditionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeWaterTankMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterWaterTankMonitoringAttributeConditionID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeDegradationDirectionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeWaterTankMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterWaterTankMonitoringAttributeDegradationDirectionID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeChangeIndicationWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeWaterTankMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterWaterTankMonitoringAttributeChangeIndicationID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeInPlaceIndicatorWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeWaterTankMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterWaterTankMonitoringAttributeInPlaceIndicatorID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeWaterTankMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterWaterTankMonitoringAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeWaterTankMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterWaterTankMonitoringAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeWaterTankMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterWaterTankMonitoringAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeWaterTankMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterWaterTankMonitoringAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeWaterTankMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterWaterTankMonitoringAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeWaterTankMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterWaterTankMonitoringAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterFuelTankMonitoring

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)resetConditionWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                   expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                              completion:(MTRStatusCompletion)completion
{
    [self resetConditionWithParams:nil
                    expectedValues:expectedValues
             expectedValueInterval:expectedValueIntervalMs
                        completion:completion];
}
- (void)resetConditionWithParams:(MTRFuelTankMonitoringClusterResetConditionParams * _Nullable)params
                  expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
           expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                      completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeFuelTankMonitoringID,
                                     (unsigned int) MTRCommandIDTypeClusterFuelTankMonitoringCommandResetConditionID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                FuelTankMonitoring::Commands::ResetCondition::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeConditionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeFuelTankMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterFuelTankMonitoringAttributeConditionID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeDegradationDirectionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeFuelTankMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterFuelTankMonitoringAttributeDegradationDirectionID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeChangeIndicationWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeFuelTankMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterFuelTankMonitoringAttributeChangeIndicationID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeInPlaceIndicatorWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeFuelTankMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterFuelTankMonitoringAttributeInPlaceIndicatorID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeFuelTankMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterFuelTankMonitoringAttributeGeneratedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeFuelTankMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterFuelTankMonitoringAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeFuelTankMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterFuelTankMonitoringAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeFuelTankMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterFuelTankMonitoringAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeFuelTankMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterFuelTankMonitoringAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeFuelTankMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterFuelTankMonitoringAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterInkCartridgeMonitoring

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)resetConditionWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                   expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                              completion:(MTRStatusCompletion)completion
{
    [self resetConditionWithParams:nil
                    expectedValues:expectedValues
             expectedValueInterval:expectedValueIntervalMs
                        completion:completion];
}
- (void)resetConditionWithParams:(MTRInkCartridgeMonitoringClusterResetConditionParams * _Nullable)params
                  expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
           expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                      completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeInkCartridgeMonitoringID,
                                     (unsigned int) MTRCommandIDTypeClusterInkCartridgeMonitoringCommandResetConditionID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                InkCartridgeMonitoring::Commands::ResetCondition::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeConditionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeInkCartridgeMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterInkCartridgeMonitoringAttributeConditionID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeDegradationDirectionWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeInkCartridgeMonitoringID)
                                     attributeID:@(MTRAttributeIDTypeClusterInkCartridgeMonitoringAttributeDegradationDirectionID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeChangeIndicationWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeInkCartridgeMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterInkCartridgeMonitoringAttributeChangeIndicationID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeInPlaceIndicatorWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeInkCartridgeMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterInkCartridgeMonitoringAttributeInPlaceIndicatorID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeInkCartridgeMonitoringID)
                                     attributeID:@(MTRAttributeIDTypeClusterInkCartridgeMonitoringAttributeGeneratedCommandListID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeInkCartridgeMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterInkCartridgeMonitoringAttributeAcceptedCommandListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeInkCartridgeMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterInkCartridgeMonitoringAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeInkCartridgeMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterInkCartridgeMonitoringAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeInkCartridgeMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterInkCartridgeMonitoringAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeInkCartridgeMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterInkCartridgeMonitoringAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterTonerCartridgeMonitoring

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)resetConditionWithExpectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                   expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                              completion:(MTRStatusCompletion)completion
{
    [self resetConditionWithParams:nil
                    expectedValues:expectedValues
             expectedValueInterval:expectedValueIntervalMs
                        completion:completion];
}
- (void)resetConditionWithParams:(MTRTonerCartridgeMonitoringClusterResetConditionParams * _Nullable)params
                  expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
           expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                      completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeTonerCartridgeMonitoringID,
                                     (unsigned int) MTRCommandIDTypeClusterTonerCartridgeMonitoringCommandResetConditionID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                TonerCartridgeMonitoring::Commands::ResetCondition::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeConditionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeTonerCartridgeMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterTonerCartridgeMonitoringAttributeConditionID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeDegradationDirectionWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeTonerCartridgeMonitoringID)
                                     attributeID:@(MTRAttributeIDTypeClusterTonerCartridgeMonitoringAttributeDegradationDirectionID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeChangeIndicationWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeTonerCartridgeMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterTonerCartridgeMonitoringAttributeChangeIndicationID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeInPlaceIndicatorWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeTonerCartridgeMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterTonerCartridgeMonitoringAttributeInPlaceIndicatorID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeGeneratedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeTonerCartridgeMonitoringID)
                                     attributeID:@(MTRAttributeIDTypeClusterTonerCartridgeMonitoringAttributeGeneratedCommandListID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAcceptedCommandListWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeTonerCartridgeMonitoringID)
                                     attributeID:@(MTRAttributeIDTypeClusterTonerCartridgeMonitoringAttributeAcceptedCommandListID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEventListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeTonerCartridgeMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterTonerCartridgeMonitoringAttributeEventListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeAttributeListWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeTonerCartridgeMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterTonerCartridgeMonitoringAttributeAttributeListID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeFeatureMapWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeTonerCartridgeMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterTonerCartridgeMonitoringAttributeFeatureMapID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeClusterRevisionWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeTonerCartridgeMonitoringID)
                                        attributeID:@(MTRAttributeIDTypeClusterTonerCartridgeMonitoringAttributeClusterRevisionID)
                                             params:params];
}

@end

@implementation MTRClusterDoorLock

- (instancetype)initWithDevice:(MTRDevice *)device endpointID:(NSNumber *)endpointID queue:(dispatch_queue_t)queue
{
    if (self = [super initWithQueue:queue]) {
        if (device == nil) {
            return nil;
        }

        _endpoint = [endpointID unsignedShortValue];
        _device = device;
    }
    return self;
}

- (void)lockDoorWithParams:(MTRDoorLockClusterLockDoorParams * _Nullable)params
            expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
     expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeDoorLockID, (unsigned int) MTRCommandIDTypeClusterDoorLockCommandLockDoorID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                DoorLock::Commands::LockDoor::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                if (!timedInvokeTimeoutMs.HasValue()) {
                    timedInvokeTimeoutMs.SetValue(10000);
                }
                if (params != nil) {
                    if (params.pinCode != nil) {
                        auto & definedValue_0 = request.PINCode.Emplace();
                        definedValue_0 = [self asByteSpan:params.pinCode];
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)unlockDoorWithParams:(MTRDoorLockClusterUnlockDoorParams * _Nullable)params
              expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
       expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                  completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeDoorLockID, (unsigned int) MTRCommandIDTypeClusterDoorLockCommandUnlockDoorID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                DoorLock::Commands::UnlockDoor::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                if (!timedInvokeTimeoutMs.HasValue()) {
                    timedInvokeTimeoutMs.SetValue(10000);
                }
                if (params != nil) {
                    if (params.pinCode != nil) {
                        auto & definedValue_0 = request.PINCode.Emplace();
                        definedValue_0 = [self asByteSpan:params.pinCode];
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)unlockWithTimeoutWithParams:(MTRDoorLockClusterUnlockWithTimeoutParams *)params
                     expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
              expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                         completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeDoorLockID,
                                     (unsigned int) MTRCommandIDTypeClusterDoorLockCommandUnlockWithTimeoutID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                DoorLock::Commands::UnlockWithTimeout::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                if (!timedInvokeTimeoutMs.HasValue()) {
                    timedInvokeTimeoutMs.SetValue(10000);
                }
                request.timeout = params.timeout.unsignedShortValue;
                if (params.pinCode != nil) {
                    auto & definedValue_0 = request.PINCode.Emplace();
                    definedValue_0 = [self asByteSpan:params.pinCode];
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)setWeekDayScheduleWithParams:(MTRDoorLockClusterSetWeekDayScheduleParams *)params
                      expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
               expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                          completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeDoorLockID,
                                     (unsigned int) MTRCommandIDTypeClusterDoorLockCommandSetWeekDayScheduleID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                DoorLock::Commands::SetWeekDaySchedule::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.weekDayIndex = params.weekDayIndex.unsignedCharValue;
                request.userIndex = params.userIndex.unsignedShortValue;
                request.daysMask
                    = static_cast<std::remove_reference_t<decltype(request.daysMask)>>(params.daysMask.unsignedCharValue);
                request.startHour = params.startHour.unsignedCharValue;
                request.startMinute = params.startMinute.unsignedCharValue;
                request.endHour = params.endHour.unsignedCharValue;
                request.endMinute = params.endMinute.unsignedCharValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)getWeekDayScheduleWithParams:(MTRDoorLockClusterGetWeekDayScheduleParams *)params
                      expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
               expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                          completion:(void (^)(MTRDoorLockClusterGetWeekDayScheduleResponseParams * _Nullable data,
                                         NSError * _Nullable error))completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeDoorLockID,
                                     (unsigned int) MTRCommandIDTypeClusterDoorLockCommandGetWeekDayScheduleID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRDoorLockClusterGetWeekDayScheduleResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                DoorLockClusterGetWeekDayScheduleResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRDoorLockClusterGetWeekDayScheduleResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                DoorLock::Commands::GetWeekDaySchedule::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.weekDayIndex = params.weekDayIndex.unsignedCharValue;
                request.userIndex = params.userIndex.unsignedShortValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)clearWeekDayScheduleWithParams:(MTRDoorLockClusterClearWeekDayScheduleParams *)params
                        expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                 expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                            completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeDoorLockID,
                                     (unsigned int) MTRCommandIDTypeClusterDoorLockCommandClearWeekDayScheduleID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                DoorLock::Commands::ClearWeekDaySchedule::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.weekDayIndex = params.weekDayIndex.unsignedCharValue;
                request.userIndex = params.userIndex.unsignedShortValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)setYearDayScheduleWithParams:(MTRDoorLockClusterSetYearDayScheduleParams *)params
                      expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
               expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                          completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeDoorLockID,
                                     (unsigned int) MTRCommandIDTypeClusterDoorLockCommandSetYearDayScheduleID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                DoorLock::Commands::SetYearDaySchedule::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.yearDayIndex = params.yearDayIndex.unsignedCharValue;
                request.userIndex = params.userIndex.unsignedShortValue;
                request.localStartTime = params.localStartTime.unsignedIntValue;
                request.localEndTime = params.localEndTime.unsignedIntValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)getYearDayScheduleWithParams:(MTRDoorLockClusterGetYearDayScheduleParams *)params
                      expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
               expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                          completion:(void (^)(MTRDoorLockClusterGetYearDayScheduleResponseParams * _Nullable data,
                                         NSError * _Nullable error))completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeDoorLockID,
                                     (unsigned int) MTRCommandIDTypeClusterDoorLockCommandGetYearDayScheduleID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRDoorLockClusterGetYearDayScheduleResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                DoorLockClusterGetYearDayScheduleResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRDoorLockClusterGetYearDayScheduleResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                DoorLock::Commands::GetYearDaySchedule::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.yearDayIndex = params.yearDayIndex.unsignedCharValue;
                request.userIndex = params.userIndex.unsignedShortValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)clearYearDayScheduleWithParams:(MTRDoorLockClusterClearYearDayScheduleParams *)params
                        expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                 expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                            completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeDoorLockID,
                                     (unsigned int) MTRCommandIDTypeClusterDoorLockCommandClearYearDayScheduleID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                DoorLock::Commands::ClearYearDaySchedule::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.yearDayIndex = params.yearDayIndex.unsignedCharValue;
                request.userIndex = params.userIndex.unsignedShortValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)setHolidayScheduleWithParams:(MTRDoorLockClusterSetHolidayScheduleParams *)params
                      expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
               expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                          completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeDoorLockID,
                                     (unsigned int) MTRCommandIDTypeClusterDoorLockCommandSetHolidayScheduleID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                DoorLock::Commands::SetHolidaySchedule::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.holidayIndex = params.holidayIndex.unsignedCharValue;
                request.localStartTime = params.localStartTime.unsignedIntValue;
                request.localEndTime = params.localEndTime.unsignedIntValue;
                request.operatingMode
                    = static_cast<std::remove_reference_t<decltype(request.operatingMode)>>(params.operatingMode.unsignedCharValue);

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)getHolidayScheduleWithParams:(MTRDoorLockClusterGetHolidayScheduleParams *)params
                      expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
               expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                          completion:(void (^)(MTRDoorLockClusterGetHolidayScheduleResponseParams * _Nullable data,
                                         NSError * _Nullable error))completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeDoorLockID,
                                     (unsigned int) MTRCommandIDTypeClusterDoorLockCommandGetHolidayScheduleID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRDoorLockClusterGetHolidayScheduleResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                DoorLockClusterGetHolidayScheduleResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRDoorLockClusterGetHolidayScheduleResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                DoorLock::Commands::GetHolidaySchedule::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.holidayIndex = params.holidayIndex.unsignedCharValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)clearHolidayScheduleWithParams:(MTRDoorLockClusterClearHolidayScheduleParams *)params
                        expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                 expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                            completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeDoorLockID,
                                     (unsigned int) MTRCommandIDTypeClusterDoorLockCommandClearHolidayScheduleID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                DoorLock::Commands::ClearHolidaySchedule::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.holidayIndex = params.holidayIndex.unsignedCharValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)setUserWithParams:(MTRDoorLockClusterSetUserParams *)params
           expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
               completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeDoorLockID, (unsigned int) MTRCommandIDTypeClusterDoorLockCommandSetUserID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                DoorLock::Commands::SetUser::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                if (!timedInvokeTimeoutMs.HasValue()) {
                    timedInvokeTimeoutMs.SetValue(10000);
                }
                request.operationType
                    = static_cast<std::remove_reference_t<decltype(request.operationType)>>(params.operationType.unsignedCharValue);
                request.userIndex = params.userIndex.unsignedShortValue;
                if (params.userName == nil) {
                    request.userName.SetNull();
                } else {
                    auto & nonNullValue_0 = request.userName.SetNonNull();
                    nonNullValue_0 = [self asCharSpan:params.userName];
                }
                if (params.userUniqueID == nil) {
                    request.userUniqueID.SetNull();
                } else {
                    auto & nonNullValue_0 = request.userUniqueID.SetNonNull();
                    nonNullValue_0 = params.userUniqueID.unsignedIntValue;
                }
                if (params.userStatus == nil) {
                    request.userStatus.SetNull();
                } else {
                    auto & nonNullValue_0 = request.userStatus.SetNonNull();
                    nonNullValue_0
                        = static_cast<std::remove_reference_t<decltype(nonNullValue_0)>>(params.userStatus.unsignedCharValue);
                }
                if (params.userType == nil) {
                    request.userType.SetNull();
                } else {
                    auto & nonNullValue_0 = request.userType.SetNonNull();
                    nonNullValue_0
                        = static_cast<std::remove_reference_t<decltype(nonNullValue_0)>>(params.userType.unsignedCharValue);
                }
                if (params.credentialRule == nil) {
                    request.credentialRule.SetNull();
                } else {
                    auto & nonNullValue_0 = request.credentialRule.SetNonNull();
                    nonNullValue_0
                        = static_cast<std::remove_reference_t<decltype(nonNullValue_0)>>(params.credentialRule.unsignedCharValue);
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)getUserWithParams:(MTRDoorLockClusterGetUserParams *)params
           expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
               completion:(void (^)(MTRDoorLockClusterGetUserResponseParams * _Nullable data, NSError * _Nullable error))completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeDoorLockID, (unsigned int) MTRCommandIDTypeClusterDoorLockCommandGetUserID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRDoorLockClusterGetUserResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                DoorLockClusterGetUserResponseCallbackType successCb, MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRDoorLockClusterGetUserResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                DoorLock::Commands::GetUser::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.userIndex = params.userIndex.unsignedShortValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)clearUserWithParams:(MTRDoorLockClusterClearUserParams *)params
             expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
      expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                 completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeDoorLockID, (unsigned int) MTRCommandIDTypeClusterDoorLockCommandClearUserID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                DoorLock::Commands::ClearUser::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                if (!timedInvokeTimeoutMs.HasValue()) {
                    timedInvokeTimeoutMs.SetValue(10000);
                }
                request.userIndex = params.userIndex.unsignedShortValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)setCredentialWithParams:(MTRDoorLockClusterSetCredentialParams *)params
                 expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
          expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                     completion:(void (^)(MTRDoorLockClusterSetCredentialResponseParams * _Nullable data,
                                    NSError * _Nullable error))completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeDoorLockID, (unsigned int) MTRCommandIDTypeClusterDoorLockCommandSetCredentialID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRDoorLockClusterSetCredentialResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                DoorLockClusterSetCredentialResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRDoorLockClusterSetCredentialResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                DoorLock::Commands::SetCredential::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                if (!timedInvokeTimeoutMs.HasValue()) {
                    timedInvokeTimeoutMs.SetValue(10000);
                }
                request.operationType
                    = static_cast<std::remove_reference_t<decltype(request.operationType)>>(params.operationType.unsignedCharValue);
                request.credential.credentialType
                    = static_cast<std::remove_reference_t<decltype(request.credential.credentialType)>>(
                        params.credential.credentialType.unsignedCharValue);
                request.credential.credentialIndex = params.credential.credentialIndex.unsignedShortValue;
                request.credentialData = [self asByteSpan:params.credentialData];
                if (params.userIndex == nil) {
                    request.userIndex.SetNull();
                } else {
                    auto & nonNullValue_0 = request.userIndex.SetNonNull();
                    nonNullValue_0 = params.userIndex.unsignedShortValue;
                }
                if (params.userStatus == nil) {
                    request.userStatus.SetNull();
                } else {
                    auto & nonNullValue_0 = request.userStatus.SetNonNull();
                    nonNullValue_0
                        = static_cast<std::remove_reference_t<decltype(nonNullValue_0)>>(params.userStatus.unsignedCharValue);
                }
                if (params.userType == nil) {
                    request.userType.SetNull();
                } else {
                    auto & nonNullValue_0 = request.userType.SetNonNull();
                    nonNullValue_0
                        = static_cast<std::remove_reference_t<decltype(nonNullValue_0)>>(params.userType.unsignedCharValue);
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)getCredentialStatusWithParams:(MTRDoorLockClusterGetCredentialStatusParams *)params
                       expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
                expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                           completion:(void (^)(MTRDoorLockClusterGetCredentialStatusResponseParams * _Nullable data,
                                          NSError * _Nullable error))completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeDoorLockID,
                                     (unsigned int) MTRCommandIDTypeClusterDoorLockCommandGetCredentialStatusID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRDoorLockClusterGetCredentialStatusResponseCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(value, error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session,
                DoorLockClusterGetCredentialStatusResponseCallbackType successCb, MTRErrorCallback failureCb,
                MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRDoorLockClusterGetCredentialStatusResponseCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                DoorLock::Commands::GetCredentialStatus::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                request.credential.credentialType
                    = static_cast<std::remove_reference_t<decltype(request.credential.credentialType)>>(
                        params.credential.credentialType.unsignedCharValue);
                request.credential.credentialIndex = params.credential.credentialIndex.unsignedShortValue;

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)clearCredentialWithParams:(MTRDoorLockClusterClearCredentialParams *)params
                   expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
            expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                       completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix = [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex,
                                     _endpoint, (unsigned int) MTRClusterIDTypeDoorLockID,
                                     (unsigned int) MTRCommandIDTypeClusterDoorLockCommandClearCredentialID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                DoorLock::Commands::ClearCredential::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                if (!timedInvokeTimeoutMs.HasValue()) {
                    timedInvokeTimeoutMs.SetValue(10000);
                }
                if (params.credential == nil) {
                    request.credential.SetNull();
                } else {
                    auto & nonNullValue_0 = request.credential.SetNonNull();
                    nonNullValue_0.credentialType = static_cast<std::remove_reference_t<decltype(nonNullValue_0.credentialType)>>(
                        params.credential.credentialType.unsignedCharValue);
                    nonNullValue_0.credentialIndex = params.credential.credentialIndex.unsignedShortValue;
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (void)unboltDoorWithParams:(MTRDoorLockClusterUnboltDoorParams * _Nullable)params
              expectedValues:(NSArray<NSDictionary<NSString *, id> *> *)expectedValues
       expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                  completion:(MTRStatusCompletion)completion
{
    NSString * logPrefix =
        [NSString stringWithFormat:@"MTRDevice command %u %u %u %u", self.device.deviceController.fabricIndex, _endpoint,
                  (unsigned int) MTRClusterIDTypeDoorLockID, (unsigned int) MTRCommandIDTypeClusterDoorLockCommandUnboltDoorID];
    // Make a copy of params before we go async.
    params = [params copy];
    NSNumber * timedInvokeTimeoutMsParam = params.timedInvokeTimeoutMs;
    if (timedInvokeTimeoutMsParam) {
        timedInvokeTimeoutMsParam = MTRClampedNumber(timedInvokeTimeoutMsParam, @(1), @(UINT16_MAX));
    }
    MTRAsyncCallbackQueueWorkItem * workItem = [[MTRAsyncCallbackQueueWorkItem alloc] initWithQueue:self.device.queue];
    MTRAsyncCallbackReadyHandler readyHandler = ^(MTRDevice * device, NSUInteger retryCount) {
        MTRClustersLogDequeue(logPrefix, self.device.asyncCallbackWorkQueue);
        MTRBaseDevice * baseDevice = [[MTRBaseDevice alloc] initWithNodeID:self.device.nodeID
                                                                controller:self.device.deviceController];
        auto * bridge = new MTRCommandSuccessCallbackBridge(
            self.device.queue,
            ^(id _Nullable value, NSError * _Nullable error) {
                MTRClustersLogCompletion(logPrefix, value, error);
                dispatch_async(self.callbackQueue, ^{
                    completion(error);
                });
                [workItem endWork];
            },
            ^(ExchangeManager & exchangeManager, const SessionHandle & session, CommandSuccessCallbackType successCb,
                MTRErrorCallback failureCb, MTRCallbackBridgeBase * bridge) {
                auto * typedBridge = static_cast<MTRCommandSuccessCallbackBridge *>(bridge);
                Optional<uint16_t> timedInvokeTimeoutMs;
                Optional<Timeout> invokeTimeout;
                ListFreer listFreer;
                DoorLock::Commands::UnboltDoor::Type request;
                if (timedInvokeTimeoutMsParam != nil) {
                    timedInvokeTimeoutMs.SetValue(timedInvokeTimeoutMsParam.unsignedShortValue);
                }
                if (params != nil) {
                    if (params.serverSideProcessingTimeout != nil) {
                        // Clamp to a number of seconds that will not overflow 32-bit
                        // int when converted to ms.
                        auto * serverSideProcessingTimeout
                            = MTRClampedNumber(params.serverSideProcessingTimeout, @(0), @(UINT16_MAX));
                        invokeTimeout.SetValue(Seconds16(serverSideProcessingTimeout.unsignedShortValue));
                    }
                }
                if (!timedInvokeTimeoutMs.HasValue()) {
                    timedInvokeTimeoutMs.SetValue(10000);
                }
                if (params != nil) {
                    if (params.pinCode != nil) {
                        auto & definedValue_0 = request.PINCode.Emplace();
                        definedValue_0 = [self asByteSpan:params.pinCode];
                    }
                }

                return MTRStartInvokeInteraction(typedBridge, request, exchangeManager, session, successCb, failureCb,
                    self->_endpoint, timedInvokeTimeoutMs, invokeTimeout);
            });
        std::move(*bridge).DispatchAction(baseDevice);
    };
    workItem.readyHandler = readyHandler;
    MTRClustersLogEnqueue(logPrefix, self.device.asyncCallbackWorkQueue);
    [self.device.asyncCallbackWorkQueue enqueueWorkItem:workItem];

    if (!expectedValueIntervalMs || ([expectedValueIntervalMs compare:@(0)] == NSOrderedAscending)) {
        expectedValues = nil;
    } else {
        expectedValueIntervalMs = MTRClampedNumber(expectedValueIntervalMs, @(1), @(UINT32_MAX));
    }
    if (expectedValues) {
        [self.device setExpectedValues:expectedValues expectedValueInterval:expectedValueIntervalMs];
    }
}

- (NSDictionary<NSString *, id> *)readAttributeLockStateWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDoorLockID)
                                        attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeLockStateID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeLockTypeWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDoorLockID)
                                        attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeLockTypeID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeActuatorEnabledWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDoorLockID)
                                        attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeActuatorEnabledID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeDoorStateWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDoorLockID)
                                        attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeDoorStateID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeDoorOpenEventsWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDoorLockID)
                                        attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeDoorOpenEventsID)
                                             params:params];
}

- (void)writeAttributeDoorOpenEventsWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                        expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeDoorOpenEventsWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeDoorOpenEventsWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                        expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                       params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeDoorLockID)
                                  attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeDoorOpenEventsID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeDoorClosedEventsWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDoorLockID)
                                        attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeDoorClosedEventsID)
                                             params:params];
}

- (void)writeAttributeDoorClosedEventsWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                          expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeDoorClosedEventsWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeDoorClosedEventsWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                          expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                         params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeDoorLockID)
                                  attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeDoorClosedEventsID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeOpenPeriodWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDoorLockID)
                                        attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeOpenPeriodID)
                                             params:params];
}

- (void)writeAttributeOpenPeriodWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeOpenPeriodWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeOpenPeriodWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                    expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                   params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeDoorLockID)
                                  attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeOpenPeriodID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeNumberOfTotalUsersSupportedWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDoorLockID)
                                        attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeNumberOfTotalUsersSupportedID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeNumberOfPINUsersSupportedWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDoorLockID)
                                        attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeNumberOfPINUsersSupportedID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeNumberOfRFIDUsersSupportedWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDoorLockID)
                                        attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeNumberOfRFIDUsersSupportedID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeNumberOfWeekDaySchedulesSupportedPerUserWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeDoorLockID)
                        attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeNumberOfWeekDaySchedulesSupportedPerUserID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeNumberOfYearDaySchedulesSupportedPerUserWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device
        readAttributeWithEndpointID:@(_endpoint)
                          clusterID:@(MTRClusterIDTypeDoorLockID)
                        attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeNumberOfYearDaySchedulesSupportedPerUserID)
                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeNumberOfHolidaySchedulesSupportedWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDoorLockID)
                                        attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeNumberOfHolidaySchedulesSupportedID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeMaxPINCodeLengthWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDoorLockID)
                                        attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeMaxPINCodeLengthID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeMinPINCodeLengthWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDoorLockID)
                                        attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeMinPINCodeLengthID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeMaxRFIDCodeLengthWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDoorLockID)
                                        attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeMaxRFIDCodeLengthID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeMinRFIDCodeLengthWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDoorLockID)
                                        attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeMinRFIDCodeLengthID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeCredentialRulesSupportWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDoorLockID)
                                        attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeCredentialRulesSupportID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeNumberOfCredentialsSupportedPerUserWithParams:(MTRReadParams * _Nullable)params
{
    return
        [self.device readAttributeWithEndpointID:@(_endpoint)
                                       clusterID:@(MTRClusterIDTypeDoorLockID)
                                     attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeNumberOfCredentialsSupportedPerUserID)
                                          params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeLanguageWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDoorLockID)
                                        attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeLanguageID)
                                             params:params];
}

- (void)writeAttributeLanguageWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                  expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeLanguageWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeLanguageWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                  expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                 params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeDoorLockID)
                                  attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeLanguageID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeLEDSettingsWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDoorLockID)
                                        attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeLEDSettingsID)
                                             params:params];
}

- (void)writeAttributeLEDSettingsWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                     expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeLEDSettingsWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeLEDSettingsWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                     expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                    params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeDoorLockID)
                                  attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeLEDSettingsID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeAutoRelockTimeWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDoorLockID)
                                        attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeAutoRelockTimeID)
                                             params:params];
}

- (void)writeAttributeAutoRelockTimeWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                        expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeAutoRelockTimeWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeAutoRelockTimeWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                        expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                       params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeDoorLockID)
                                  attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeAutoRelockTimeID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeSoundVolumeWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDoorLockID)
                                        attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeSoundVolumeID)
                                             params:params];
}

- (void)writeAttributeSoundVolumeWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                     expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeSoundVolumeWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeSoundVolumeWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                     expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                    params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeDoorLockID)
                                  attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeSoundVolumeID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeOperatingModeWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDoorLockID)
                                        attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeOperatingModeID)
                                             params:params];
}

- (void)writeAttributeOperatingModeWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                       expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeOperatingModeWithValue:dataValueDictionary expectedValueInterval:expectedValueIntervalMs params:nil];
}
- (void)writeAttributeOperatingModeWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                       expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                      params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeDoorLockID)
                                  attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeOperatingModeID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeSupportedOperatingModesWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDoorLockID)
                                        attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeSupportedOperatingModesID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeDefaultConfigurationRegisterWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDoorLockID)
                                        attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeDefaultConfigurationRegisterID)
                                             params:params];
}

- (NSDictionary<NSString *, id> *)readAttributeEnableLocalProgrammingWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDoorLockID)
                                        attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeEnableLocalProgrammingID)
                                             params:params];
}

- (void)writeAttributeEnableLocalProgrammingWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                                expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeEnableLocalProgrammingWithValue:dataValueDictionary
                                  expectedValueInterval:expectedValueIntervalMs
                                                 params:nil];
}
- (void)writeAttributeEnableLocalProgrammingWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                                expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                               params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeDoorLockID)
                                  attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeEnableLocalProgrammingID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeEnableOneTouchLockingWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDoorLockID)
                                        attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeEnableOneTouchLockingID)
                                             params:params];
}

- (void)writeAttributeEnableOneTouchLockingWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                               expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeEnableOneTouchLockingWithValue:dataValueDictionary
                                 expectedValueInterval:expectedValueIntervalMs
                                                params:nil];
}
- (void)writeAttributeEnableOneTouchLockingWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                               expectedValueInterval:(NSNumber *)expectedValueIntervalMs
                                              params:(MTRWriteParams * _Nullable)params
{
    NSNumber * timedWriteTimeout = params.timedWriteTimeout;

    [self.device writeAttributeWithEndpointID:@(_endpoint)
                                    clusterID:@(MTRClusterIDTypeDoorLockID)
                                  attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeEnableOneTouchLockingID)
                                        value:dataValueDictionary
                        expectedValueInterval:expectedValueIntervalMs
                            timedWriteTimeout:timedWriteTimeout];
}

- (NSDictionary<NSString *, id> *)readAttributeEnableInsideStatusLEDWithParams:(MTRReadParams * _Nullable)params
{
    return [self.device readAttributeWithEndpointID:@(_endpoint)
                                          clusterID:@(MTRClusterIDTypeDoorLockID)
                                        attributeID:@(MTRAttributeIDTypeClusterDoorLockAttributeEnableInsideStatusLEDID)
                                             params:params];
}

- (void)writeAttributeEnableInsideStatusLEDWithValue:(NSDictionary<NSString *, id> *)dataValueDictionary
                               expectedValueInterval:(NSNumber *)expectedValueIntervalMs
{
    [self writeAttributeEnableInsideStatusLEDWithVal